{
///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//                                                                           //
//                     Global Public Licence (GPL)                           //
//                                                                           //
//                                                                           //
//    This license described in this file overrides all other licenses       //
//    that might be specified in other files for this software.              //
//                                                                           //
//    This program is free software; you can redistribute it and/or          //
//    modify it under the terms of the GNU Lesser General Public             //
//    License as published by the Free Software Foundation; either           //
//    version 2.1 of the License, or (at your option) any later version.     //
//                                                                           //
//    This software is distributed in the hope that it will be useful,       //
//    but WITHOUT ANY WARRANTY; without even the implied warranty of         //
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      //
//    Library General Public License for more details.                       //
//                                                                           //
//    You should have received a copy of the GNU Lesser General Public       //
//    License along with this library (see COPYING.LIB); if not, write to    //
//    the Free Software Foundation :                                         //
//                         Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
}

{************     CPAS  version    *************}

{
        *** Modification(s) from major version ***


                  ----

                 NOTHING

                  ----

}

program GEN_PHASES_APPROXIMANTES;


%include               'satellite_env';         { get all the satellite related definitions }

const
  version   =   'V 1.0 A - 5-Nov-2010';         { Software Version }


  search_path   = './,HOME:/etc/,/usr/local/etc/';      { Search path for Gen_Approx program }
  help_file_spc =     'gen_approx.hlp';         { Help file specification }
  gap_ranfile   =     'gen_approx.ran';         { Random state file specification }

  pi            =    4.0*ARCTAN( 1.0 );         { The pi number }
  inrd          =             pi/180.0;         { Factor to convert degrees in radians }

  max_ze        =                  100;         { Maximum of Z to size the minimum Z specific exclusion distance }

  max_pos       =                   64;         { Maximum of atom position number }
  max_neigh     =                   32;         { Maximum of neigbouring atom index }
  max_ran       =                  768;         { Maximum of site for atom distribution (2*192) }
  max_poslist   =                   16;         { Maximum of position in a position list }
  max_atom      =                  999;         { Maximum number of atom in the unit cell }
  ope_max       =                   48;         { Maximum number of crystallographic group operator }
  max_anam      =                   11;         { Maximum size for an atom name }
  crystal_def   =                    2;         { Number of dot by Angstroem in the crystal array container }
  max_nei     =  max_atom*(max_atom-1);         { Maximum number of neigbouring definitions }

  gen_space_tmp =  'gen_space_out.tmp';         { Name of the output file created by GEN_SPACE Program }
  strdef_file   =  'gen_approx.struct';         { Default <case>.struct filename }
  strdef_type   =            '.struct';

  out_ener      =              1.0e+10;         { Impossible insert flag energy }

type
  sbyte         =                short_integer; { Idem to [BYTE] integer }
  sstring       =                 string( 62 ); { Define a short string type }
  posname       =         string( max_anam-3 ); { String type for position name }
  matrix        =   array[1..3,1..3] of   real; { Type for matrix and metric tensor }

  kw_symb       =   (   kw_group,               { * Define the input language keywords }
                        kw_cell,
                        kw_extnd,
                        kw_flat,
                        kw_axis,
                        kw_orig,
                        kw_title,
                        kw_mxdist,
                        kw_excdis,
                        kw_atmpos,
                        kw_putatm,
                        kw_ranini,
                        kw_end );

  elemty    =                      string( 2 ); { Define the element name type }

  pos_rec   = record
                pos_name:              posname; { Position name }
                pos_size,                       { Position order }
                pos_nuse,                       { Number of present atoms in the position }
                pos_offs:              integer  { Position offset in atm_tab array }
              end;

  atm_rec   = record
                atm_elem:               elemty; { The element }
                atm_posid,                      { Index of related position }
                atm_oneig, atm_nneig,           { Offset of site relative neighbouring table and there number }
                atm_zel:               integer; { Z electron number of atom }
                atm_ee, atm_rad, atm_elec,      { "Affinity" for this site, Atomic Radius and Electro-negativity }
                atm_x, atm_y, atm_z:      real  { Site coordinates }
              end;

  nei_rec   = record
                nei_dist:                 real; { Neighbouring site distance }
                nei_atm:               integer; { Index of neighbouring site in atm_tab }
                nei_ix, nei_iy, nei_iz:  sbyte  { Related lattice translation }
              end;

  ran_rec   = record
                ran_aid,                        { Atom index }
                ran_pid:               integer  { Position index }
              end;

  elm_tbty  = array[1..max_ze] of elemty;


const
  dir_table = cmd_smbtb[ 13, [ ORD( kw_group  ),  5, 'group'       ],
                             [ ORD( kw_cell   ),  4, 'cell'        ],
                             [ ORD( kw_extnd  ), 11, 'cell_extend' ],
                             [ ORD( kw_flat   ),  9, 'f_lattice'   ],
                             [ ORD( kw_axis   ),  4, 'axis'        ],
                             [ ORD( kw_orig   ),  6, 'origin'      ],
                             [ ORD( kw_title  ),  5, 'title'       ],
                             [ ORD( kw_mxdist ),  8, 'max_dist'    ],
                             [ ORD( kw_excdis ), 12, 'exclude_dist'],
                             [ ORD( kw_atmpos ),  8, 'position'    ],
                             [ ORD( kw_putatm ),  8, 'put_atom'    ],
                             [ ORD( kw_ranini ),  8, 'ran_init'    ],
                             [ ORD( kw_end    ),  3, 'end'         ]
                       ];


  elm_tab = elm_tbty[
     'H',                                                                                                 'He',
    'Li', 'Be',                                                              'B',  'C',  'N',  'O',  'F', 'Ne',
    'Na', 'Mg',                                                             'Al', 'Si',  'P',  'S', 'Cl', 'Ar',
     'K', 'Ca', 'Sc', 'Ti',  'V', 'Cr', 'Mn', 'Fe', 'Co', 'Ni', 'Cu', 'Zn', 'Ga', 'Ge', 'As', 'Se', 'Br', 'Kr',
    'Rb', 'Sr',  'Y', 'Zr', 'Nb', 'Mo', 'Tc', 'Ru', 'Rh', 'Pd', 'Ag', 'Cd', 'In', 'Sn', 'Sb', 'Te',  'I', 'Xe',
    'Cs', 'Ba', 'La', 'Ce', 'Pr', 'Nd', 'Pm', 'Sm', 'Eu', 'Gd', 'Tb', 'Dy', 'Ho', 'Er', 'Tm', 'Yb', 'Lu',
                      'Hf', 'Ta',  'W', 'Re', 'Os', 'Ir', 'Pt', 'Au', 'Hg', 'Tl', 'Pb', 'Bi', 'Po', 'At', 'Rn',
    'Fr', 'Ra', 'Ac', 'Th', 'Pa',  'U', 'Np', 'Pu', 'Am', 'Cm', 'Bk', 'Cf', 'Es', 'Fm'
  ];


var
  ran_ini,                                      { Flag for Random number generator init }
  fix_lat,                                      { Flag to force non primitive lattice }
  path_ok,                                      { Flag for Gen_Approx environmet directory found }
  distout,                                      { Flag for final distance output }
  verbose:            boolean := false;         { Flag for the verbose option }

  smsg,                                         { String to compose the message of ERRORs and WARNINGs }
  str_title,                                    { Wien-2K Work file title }
  gap_envdir,                                   { Environment directory Path, as determine by the searxch path and help file }
  gap_hlpfile,                                  { Complete help file Path }
  egs_ccmd,                                     { Command string to call gen_space program }
  egs_cmd0:                     string;         { Common part of all gen_space command line }

  egs_axis:            sstring :=   '';
  egs_group:           sstring := 'P1';

  egs_cell,
  egs_orig:             string :=   '';         { Returned Space Group name }

  egsb_group,                                   { Flag for specified input data }
  egsb_cell,
  egsb_orig,
  egsb_axis:          boolean := false;

  gap_int,                                      { Wild integer }
  grp_lat,                                      { Returned lattice }
  grp_inc,                                      { Lattice increment when fixed (else 1) }
  grp_mlt,                                      { Lattice multiplicity }
  grp_nbr,                                      { Returned Space Group number }
  grp_nsy,                                      { Returned Crystallographic system index number }
  grp_ori,                                      { Returned Axis orientation (for Monoclinic or orthorhombic system) }
  grp_ord:               integer := -1;         { Returned Space Group order }

  mxdist,  mxdist2,                             { Maximum of distance between atoms to examine it and its square }

  orgx, orgy, orgz,                             { Origine Choice of space group }
  fpaa, fpbb, fpcc,                             { Direct Final unit cell (multiple unit cell when cell_extend mode is used) }
  aa, bb, cc, al, be, ga:  real := 0.0;         { Direct unit cell }
  dten:                         matrix;         { Direct and reciprocal matrix tensor }

  cell_na, cell_nb, cell_nc, cell_abc,          { Cell extension multipliers }
  rnd_seek,                                     { Random generator seek number }
  npos,                                         { Number of position }
  posnb,                                        { Used size of pos_lst table }
  nnei,                                         { Filling levl of nei_tab }
  natm:                  integer :=  0;         { Total number of atom }

  { Statistic table for all position occupation }
  occ_tab:  array[1..max_pos,1..max_ze] of integer;

  exd_tab:  array[1..max_ze,1..max_ze] of real; { Minimum distance exculsion for specified Z element }

  nei_tab:     array[1..max_nei]   of  nei_rec; { Table of neighbouring distances link }
  ran_tab:     array[1..max_ran]   of  ran_rec; { Table for randomization of atom list }
  pos_tab:     array[1..max_pos]   of  pos_rec; { Position table }
  atm_tab:     array[1..max_atom]  of  atm_rec; { Atom table }

  pos_lst:    array[1..max_poslist] of integer; { Position index list table for put_atom statement management }

  inpfname,                                     { Name of input file }
  disfname,                                     { Distance output file }
  outfname:             string  :=          ''; { Name of Wien CASE study }

  dis_file,                                     { Distance output file }
  str_file:                               text; { Wien <cname>.struct file to create }




procedure EXECUTE_GEN_SPACE( in_var cmd: string );
{ Procedure to execute the MXD Gen_Space program.
  Used to get the metric tensor and the list of equivalent site
  of a specified site coordinates.
}
const
  mdnam = 'EXGS';

var
  ip, iu, ie:  integer;

begin
  ip := CREATE_PROCESS( 'MXDLIB:mxd_gen_space', cmd );
  if ip > 0 then
  begin
    iu := WAIT_PROCESS( ie, ip );
    if ie <> 0 then ERROR( mdnam, 'mxd_gen_space program return "Unknown Space Group"' )
  end
  else ERROR( mdnam, 'Cannot run the program "mxd_gen_space" (or gen_space)' )
end EXECUTE_GEN_SPACE;



procedure INIT;
{ Procedure to init all program data structure and
  manage all commande line arguments and options.
}
const

  mdnam = 'INIT';

type
  copt_typ = ( c_outf,
               c_inpf,
               c_disf,
               c_cell,
               c_extnd,
               c_flat,
               c_org,
               c_axis,
               c_title,
               c_mdist,
               c_exdis,
               c_ranin,
               c_verb,
               c_help );

const
  kwd_table = cmd_smbtb[ 28,
                [ ORD( c_outf  ), 2,   '-o'], [ ORD( c_outf  ),  4,         '-out'],
                [ ORD( c_inpf  ), 2,   '-i'], [ ORD( c_inpf  ),  4,         '-inp'],
                [ ORD( c_disf  ), 2,   '-d'], [ ORD( c_disf  ),  9,    '-distance'],
                [ ORD( c_cell  ), 2,   '-c'], [ ORD( c_cell  ),  5,        '-cell'],
                [ ORD( c_extnd ), 3,  '-ce'], [ ORD( c_extnd ), 12, '-cell_extend'],
                [ ORD( c_flat  ), 3,  '-fl'], [ ORD( c_flat  ),  9,    '-flattice'],
                [ ORD( c_org   ), 4, '-org'], [ ORD( c_org   ),  7,      '-origin'],
                [ ORD( c_axis  ), 5,'-axis'], [ ORD( c_axis  ), 10,   '-reference'],
                [ ORD( c_title ), 2,   '-t'], [ ORD( c_title ),  6,       '-title'],
                [ ORD( c_mdist ), 2,   '-d'], [ ORD( c_mdist ),  9,    '-max_dist'],
                [ ORD( c_exdis ), 3,  '-ed'], [ ORD( c_exdis ),  9,    '-exc_dist'],
                [ ORD( c_ranin ), 3,  '-ri'], [ ORD( c_ranin ),  9,    '-ran_init'],
                [ ORD( c_verb  ), 2,   '-v'], [ ORD( c_verb  ),  8,    ' -verbose'],
                [ ORD( c_help  ), 2,   '-h'], [ ORD( c_help  ),  5,        '-help']
  ];

var
  cd, ii, jj, ll, nv, np, z1, z2:      integer;
  sopt, sparm:                    string( 64 );
  rvl:                    array[1..12] of real;
  cvl:                                    real;



  procedure INIT_GET_VALUES( var nvl: integer; ll: integer; bfrac: boolean := false );
  const
    opttsz = 10;

    mdnam = 'INIT';


  var
    ij, jj, kp, num, den:      integer;
    snum:                 string( 14 );
    ch:                           char;

  begin
    nvl := 0;
    ij := 1;
    while (ij <= sparm.length) and (nvl < ll) do
    begin
      jj := INDEX( sparm, ':', nvl + 1 );
      kp := INDEX( sparm, ',', nvl + 1 );
      if kp > 0 then
        if (jj <= 0) or (jj > kp) then jj := kp;
      if jj <= 0 then jj := sparm.length + 1;
      nvl := nvl + 1;
      rvl[nvl] := 0.0;
      if jj > ij then
      begin
        if bfrac then
        begin
          snum := SUBSTR( sparm, ij, jj-ij+1 );
          kp := INDEX( snum, '/' );
          if kp > 1 then
          begin
            READV( snum, num:kp-1, ch, den:snum.length-kp );
            if den < 1 then den := 1;
            rvl[nvl] := num/den
          end
          else READV( sparm:ij, rvl[nvl]:jj-ij+1 )
        end
        else READV( sparm:ij, rvl[nvl]:jj-ij+1 )
      end;
      ij := jj + 1
    end
  end INIT_GET_VALUES;


begin { INIT }
  { Init the exclusion distance table }
  for i := 1 to max_ze do
    for j := 1 to max_ze do
      exd_tab[i,j] := 0.0;

  { Init the cell multipliers }
  cell_na := 1; cell_nb := 1; cell_nc := 1;

  { Start the command line parameter list parsing }
  np := 0;
  ii := 1;
  while ii < argc do
  begin
    sparm := argv[ii]^;
    if sparm.length > 0 then
      if sparm[1] = '-' then
      begin
        jj := INDEX( sparm, '=' );
        if jj > 0 then
        begin
          sopt  := SUBSTR( sparm, 1, jj - 1 );
          sparm := SUBSTR( sparm, jj + 1 )
        end
        else
        begin  sopt := sparm; sparm.length := 0  end;
        CHANGE_CASE( sopt );
        cd := IDE_SEARCH( kwd_table, sopt );
        if cd >= 0 then
        case copt_typ( cd ) of
          c_inpf: if sparm.length > 0 then inpfname := sparm;   { Select an input file }
          c_outf: if sparm.length > 0 then outfname := sparm;   { Select a Wien-2K output file }
          c_disf: begin
                    if sparm.length > 0 then disfname := sparm; { Select a distance output file }
                    distout := true
                  end;

          c_cell:
            begin
              INIT_GET_VALUES( nv, 6 );
              if (nv > 0) and (rvl[1] > 0.0) then
              begin
                aa := rvl[1]; bb := aa; cc := aa;
                al := 0.0; be := 0.0; ga := 0.0;
                if (nv >= 2) and (rvl[2] <> 0.0) then bb := rvl[2];
                if (nv >= 3) and (rvl[3] <> 0.0) then cc := rvl[3];
                if (nv >= 4) and (rvl[4] <> 0.0) then al := rvl[4];
                be := al; ga := al;
                if (nv >= 5) and (rvl[5] <> 0.0) then be := rvl[5];
                if (nv >= 6) and (rvl[6] <> 0.0) then ga := rvl[6];
                egsb_cell := true
              end
              else ERROR( mdnam, 'GEN_APPROX: Bad use of -cell option in the command line.' )
            end;

          c_extnd:
            begin
              INIT_GET_VALUES( nv, 3, true );
              if nv > 0 then
              begin
                cell_na := ROUND( rvl[1] );
                cell_nb := cell_na; cell_nc := cell_na;
                if nv > 1 then
                begin
                  cell_nb := ROUND( rvl[2] );
                  if nv > 2 then cell_nc := ROUND( rvl[3] )
                end
              end
            end;

          c_org:
            begin
              INIT_GET_VALUES( nv, 3, true );
              if nv > 0 then
              begin
                orgx := rvl[1];
                if nv >= 2 then orgy := rvl[2];
                if nv >= 3 then orgz := rvl[3];
                egsb_orig := true
              end
              else ERROR( mdnam, 'GEN_APPROX: Bad use of -origine option in the command line.' )
            end;

          c_axis: if sparm.length > 0 then begin  egs_axis := argv[ii]^; egsb_axis := true  end;

          c_title:if sparm.length > 0 then str_title := sparm;

          c_mdist:
            begin
              INIT_GET_VALUES( nv, 1, true );
              if nv >= 1 then mxdist := ABS( rvl[1] )
            end;

          c_exdis:
            begin
              INIT_GET_VALUES( nv, 3, true );
              if nv > 2 then
              begin
                z1 := ROUND( rvl[1] ); z2 := ROUND( rvl[2] ); cvl := rvl[3];
                if (z1 > 0) and (z1 <= max_ze) and
                   (z2 > 0) and (z2 <= max_ze) and
                   (cvl >= 0.0) and (cvl < 10.0) then
                begin  exd_tab[z1,z2] := cvl; exd_tab[z2,z1] := cvl  end
              end
              else ERROR( mdnam, 'GEN_APPROX: Bad number of parameters for the option -exc_dist (model: -exc_dist=<z1>,<z2>,<min_dist>)' )
            end;

          c_flat:  fix_lat := true;

          c_ranin: ran_ini := true;

          c_verb:  verbose := true;

          c_help: { Gen Space Help on standard output }
            if path_ok then
            begin
              WRITEV( inpfname, 'less ', gap_hlpfile );
              RUN_PROCESS( 'less', inpfname );
              WARNING( mdnam, 'GEN_APPROX: cannot run "less" program' );
              OPEN( str_file, gap_hlpfile, [read_file,error_file] );
              if iostatus = 0 then
              begin
                repeat
                  output^ := str_file^; GET( str_file ); PUT( output )
                until EOF( str_file );
                CLOSE( str_file )
              end
              else
                ERROR( mdnam, 'GEN_APPROX: Sorry, I can''t open the help file', help_file_spc );
              PASCAL_EXIT( 0 )
            end
            else ERROR( mdnam, 'GEN_APPROX: Sorry, Ican''t find the help file', help_file_spc );

        otherwise
          WRITELN;
          WRITELN( ' *** GEN_APPROX: Unimplemented Gen_Approx Option "', sopt, '" => Stop *** ' );
          WRITELN;
          PASCAl_EXIT( 2 )
        end
        else
        begin
          WRITELN;
          WRITELN( ' *** GEN_APPROX: Unknown Gen_Approx Option "', sopt, '" => Stop *** ' );
          WRITELN;
          PASCAl_EXIT( 2 )
        end
      end
      else
      begin
        case np of
          0: begin  egs_group := sparm; egsb_group := true  end;
          1: outfname := sparm;
        otherwise
          disfname := sparm
        end;
        np := np + 1
      end;
    ii := ii + 1
  end
end INIT;



function FIND_ELEMENT( in_var name: string ): integer;
{ Function to return the atomic Z number from the element Name.
}
const
  inmin = ORD( 'z' ) - ORD( 'Z' );

var
  el: string( 2 );
  ia: integer;

begin
  if (name.length > 0) and (name.length <= 2) then
  begin
    el.length := name.length;
    el[1] := name[1];
    if (el[1] >= 'a') and (el[1] <= 'z') then el[1] := CHR( ORD( el[1] ) - inmin );
    if el.length > 1 then
    begin
      el[2] := name[2];
      if (el[2] >= 'A') and (el[2] <= 'Z') then el[2] := CHR( ORD( el[2] ) + inmin )
    end;
    ia := 1;
    while ia <= max_ze do
    begin
    exit if el = elm_tab[ia];
      ia := ia + 1
    end;
    if ia <= max_ze then FIND_ELEMENT := ia
                    else FIND_ELEMENT :=  0
  end
  else FIND_ELEMENT := 0
end FIND_ELEMENT;



procedure GET_ZN_ELEMENT( in_var prt: string; var sn: elemty; var ze: integer );
{ Procedure to get an atomic element specification by
  chimical name or Z number value.
}
const
  mdnam = 'ZNEL';

var
  bin: boolean;

begin
  bin    := true;
  ze        := 0;
  sn.length := 0;
  INP_INQUIRE( prt );
  case inp_symb of
    sy_int:   begin
                ze := inp_int;
                if (ze > 0) and (ze <= max_ze) then sn := elm_tab[ze]
              end;
    sy_real:  begin
                if ABS( inp_real ) < FLOAT( maxint ) then ze := ROUND( inp_real )
                                                     else ze := 0;
                if (ze > 0) and (ze <= max_ze) then sn := elm_tab[ze]
              end;
    sy_ident,
    sy_str:   begin
                ze := FIND_ELEMENT( inp_str );
                if ze > 0 then sn := elm_tab[ze]
              end;
  otherwise
    ERROR( mdnam, 'Bad element specification' );
    bin := false
  end;
  if bin then INSYMBOL
end GET_ZN_ELEMENT;



procedure SET_Z_EXCL_DISTANCE;
{ Procedure to set a specific minimal interatomic distance.
}
var
  z1, z2:      integer;
  md:             real;
  sn1, sn2:     elemty;

begin
  INSYMBOL;
  GET_ZN_ELEMENT( ' Element 1 = ', sn1, z1 );
  if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
  GET_ZN_ELEMENT( ' Element 2 = ', sn2, z2 );
  if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
  INP_R_VALUE( ' Min Distance = ', md );
  if (md >= 0.0) and (md <= 10.0) then
    if (z1 > 0) and (z1 <= max_ze) and
       (z2 > 0) and (z2 <= max_ze) then begin
                                          exd_tab[z1,z2] := md;
                                          exd_tab[z2,z1] := md
(*
;WRITELN( ' Minimal distance between ', sn1, ' and ', sn2, ' is set to ', md:6:3, ' A.' )
*)
                                        end
end SET_Z_EXCL_DISTANCE;



procedure GET_FIRST_DATA;
{ Procedure to read and process all initial data statement.
}
const
  mdnam = 'GDAT';

begin { GET_FIRST_DATA }
  if inpfname.length > 0 then
  begin
    OPEN( inp_file, inpfname, [read_file,error_file] );
    if (iostatus <> 0) or EOF( inp_file ) then ERROR( mdnam, 'Cannot open the input file "', inpfname )
  end;

  { Flag the input from a data file }
  if inpfname.length > 0 then inp_file_flg := true;

  inp_csmbtb := dir_table"address;

  while (inp_symb <> sy_eof) and (npos < max_pos) do
  begin
    inp_prompt := ' Gen_Approx statement> ';
    INP_INQUIRE( '' );
(*
WRITELN( ' G : ', inp_symb, ' "', inp_str, '", cd = ', inp_code:0 );
*)
  exit if inp_symb = sy_eof;
    case inp_symb of
      sy_ident:
        if inp_code >= 0 then
        case kw_symb( inp_code ) of
          kw_group:
            begin
              inp_mdstr := true; inp_prompt2 := ' Space Group = '; INSYMBOL;
              if inp_symb <> sy_str then ERROR( mdnam, 'A space group name or number was expected' );
              egs_group := inp_str; egsb_group := true;;
              INSYMBOL;
              if verbose then
                WRITELN( ' Space Group "', egs_group, '" is set.' )
            end;

          kw_cell:
            begin
              INSYMBOL;
              INP_R_VALUE( ' A = ', aa );
              bb := aa; cc := aa;
              al := 0.0; be := 0.0; ga := 0.0;
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_R_VALUE( ' B = ', bb );
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_R_VALUE( ' C = ', cc );
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_R_VALUE( ' Alpha = ', al );
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_R_VALUE( ' Beta = ', be );
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_R_VALUE( ' Gamma = ', ga );
              if bb = 0.0 then bb := aa;
              if cc = 0.0 then cc := aa;
              if al <> 0.0 then
              begin
                if be = 0.0 then be := al;
                if ga = 0.0 then ga := al
              end;
              egsb_cell := true;
              if verbose then
                WRITELN( ' Cell ', aa:8:5, bb:8:5, cc:8:5, al:8:5, be:8:5, ga:8:5, ' is set.' )
            end;

          kw_extnd:
            begin
              INSYMBOL;
              INP_I_VALUE( ' n*a = ', cell_na );
              cell_nb := cell_na; cell_nc := cell_na;
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_I_VALUE( ' n*b = ', cell_nb );
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_I_VALUE( ' n*c = ', cell_nc )
            end;

          kw_flat:
            begin
              fix_lat := true; INSYMBOL;
              if verbose then
                WRITELN( ' The space group lattice will be forced for final structure.' )
            end;

          kw_axis:
            begin
              inp_mdstr := true; inp_prompt2 := ' Axis Permutation = '; INSYMBOL;
              if inp_symb <> sy_str then ERROR( mdnam, 'An axis permutation string was expected' );
              WRITEV( egs_axis, '-axis=', inp_str ); egsb_axis := true;
              INSYMBOL;
              if verbose then WRITELN( ' Axis = "', egs_axis, '" is set.' )
            end;

          kw_orig:
            begin
              orgx := 0.0; orgy := 0.0; orgz := 0.0;
              INSYMBOL;
              INP_R_VALUE( ' Tx = ', orgx );
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_R_VALUE( ' Ty = ', orgy );
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_R_VALUE( ' Tz = ', orgz );
              egsb_orig := true;
              if verbose then
                WRITELN( ' Set Origine at (', orgx:5:3, ',', orgy:5:3, ',', orgz:5:3, ').' )
            end;

          kw_title:
            begin
              inp_mdlin := true; inp_prompt2 := ' Title = '; INSYMBOL;
              str_title := inp_str;
              if inp_symb <> sy_str then ERROR( mdnam, 'A Title line for Wien2K was expected' );
              INSYMBOL;
              if verbose then WRITELN( ' Title = "', str_title, '" is set.' )
           end;

          kw_mxdist:
            begin
              INSYMBOL;
              INP_R_VALUE( ' Mx_Dist = ', mxdist );
              mxdist := ABS( mxdist );
              if verbose then WRITELN( ' Max_Dist = ', mxdist:8:4, ' is set.' )
            end;

          kw_ranini:
            begin
              INSYMBOL;
              ran_ini := true
            end;

          kw_excdis: SET_Z_EXCL_DISTANCE;

          kw_end,
          kw_atmpos,
          kw_putatm:
            begin
              if verbose then
              begin
                WRITELN( ' End of General First input process.' );
                WRITELN
              end;
              exit
            end;

        otherwise
          ERROR( mdnam, 'An implemented directive keyword was expected' );
        end
        else ERROR( mdnam, 'A Defined directive keyword was expected' );

      sy_eoln, sy_semicolon:
        INSYMBOL;

      sy_eof:
        begin
          WARNING( mdnam, ' Return to standard input' );
          inp_file_flg := false;
          CLOSE( inp_file );
          inp_symb := sy_eoln;
          INSYMBOL
        end;

    otherwise
      ERROR( mdnam, 'Input file syntax error.' )
    end
  end { of while };
end GET_FIRST_DATA;



procedure LOAD_RANDOM_STATE;
{ Procedure to load the Random Number Generator state
  in a file in the current directory if it is existing.
}
const
  mdnam = 'LDRA';

var
  fran:   text;
  bok: boolean;

begin
  if not ran_ini then
  begin
    OPEN( fran, gap_ranfile, [read_file,error_file] );
    bok := (iostatus = 0);
    if not bok then WARNING( mdnam, 'Cannot open (or find) the Random generator state file' )
  end
  else bok := false;
  if bok then
  begin
    READ( fran, rnd_seek );
    for i := 0 to 32 do READ( fran, random_state_table[i] );
    CLOSE( fran )
  end
end LOAD_RANDOM_STATE;



procedure SAVE_RANDOM_STATE;
{ Procedure to save the Random Number Generator state
  in a file in the current directory.
}
var
  fran: text;

begin
  OPEN( fran, gap_ranfile, [write_file,error_file] );
  if iostatus = 0 then
  begin
    WRITELN( fran, rnd_seek:16 );
    for i := 0 to 32 do WRITELN( fran, random_state_table[i]:16 );
    CLOSE( fran )
  end
  else WARNING( 'Cannot save the random generator status', gap_ranfile );
end SAVE_RANDOM_STATE;



procedure GET_SPACE_GROUP_TENSOR;
{ Procedure to get metric tensor, space group characteristics,
  and init all other sample creation computing.
}
const
  mdnam = 'GSPG';

var
  system: string( 14 );
  ij:          integer;

begin
  if not (egsb_group and egsb_cell) then
    ERROR( mdnam, ' A space group and a unit cell must be specified for Gen_Approx => Stop.' );

  if inpfname.length > 0 then
  begin
    if outfname.length = 0 then
    begin
      ij := INDEX( inpfname, '.', -1 );
      if ij = 0 then outfname := inpfname
                else outfname := SUBSTR( inpfname, 1, ij-1 );
      outfname := outfname||strdef_type
    end
  end
  else
    if outfname.length = 0 then outfname := strdef_file;

  WRITEV( egs_cell, '-cell=', aa:10:6, ',', bb:10:6, ',', cc:10:6, ',',
                              al:10:6, ',', be:10:6, ',', ga:10:6, ';' );
  NO_STRING_SPACE( egs_cell );

  { Force local angle value to be in ° - not Cosinus }
  if al < 1.0 then al := ARCCOS( al )/inrd;
  if be < 1.0 then be := ARCCOS( be )/inrd;
  if ga < 1.0 then ga := ARCCOS( ga )/inrd;

  if cell_na < 1 then cell_na := 1;
  if cell_nb < 1 then cell_nb := 1;
  if cell_nc < 1 then cell_nc := 1;
  cell_abc := cell_na*cell_nb*cell_nc;

  if fix_lat and (cell_abc > 1) then
  begin
    WARNING( mdnam, 'The f_lattice mode, incompatible with cell_extend specification, will be ignored' );
    fix_lat := false
  end;

  if egsb_orig then
  begin
    if orgx >= 1.0 then orgx := orgx - TRUNC( orgx )
                   else if orgx < -1.0 then orgx := orgx - TRUNC( orgx );
    if orgy >= 1.0 then orgy := orgy - TRUNC( orgy )
                   else if orgy < -1.0 then orgy := orgy - TRUNC( orgy );
    if orgz >= 1.0 then orgz := orgz - TRUNC( orgz )
                   else if orgz < -1.0 then orgz := orgz - TRUNC( orgz );
    WRITEV( egs_orig, '-origine=', orgx:8:6, ',', orgy:8:6, ',', orgz:8:6, ';' );
    NO_STRING_SPACE( egs_orig )
  end;

  { Perform the first call to gen_space to get the metric tensor }
  WRITEV( egs_cmd0, 'gen_space ', egs_group, ' -data ' , gen_space_tmp );
  if egsb_axis then WRITEV( egs_cmd0:false, ' ', egs_axis );
  if egsb_orig then WRITEV( egs_cmd0:false, ' ', egs_orig );
  WRITEV( egs_ccmd, egs_cmd0, ' -dten ', egs_cell );

(*
WRITELN( ' Execute "', egs_ccmd, '"' );
*)

  EXECUTE_GEN_SPACE( egs_ccmd );

  OPEN( str_file, gen_space_tmp, [read_file,error_file,del_file] );
  if (iostatus <> 0) or EOF( str_file ) then ERROR( mdnam, 'Cannot open the temporary space group file "', gen_space_tmp );
  READLN( str_file, system:13, egs_group:0:true );
  grp_lat := INDEX( 'PABCIFRH', egs_group[1] ) - 1;
  for i := 1 to 3 do  for j := 1 to 3 do  READ( str_file, dten[i,j] );
  READLN( str_file );
  CLOSE( str_file );

  if cell_abc > 1 then
  begin { Adapt the Tensor to the multiple unit cell }
    for i := 1 to 3 do
    begin
      dten[1,i] := cell_na*dten[1,i]; dten[i,1] := cell_na*dten[i,1];
      dten[2,i] := cell_nb*dten[2,i]; dten[i,2] := cell_nb*dten[i,2];
      dten[3,i] := cell_nc*dten[3,i]; dten[i,3] := cell_nc*dten[i,3]
    end
  end;
  fpaa := cell_na*aa; fpbb := cell_nb*bb; fpcc := cell_nc*cc;

  { Set the lattice multiplicity }
  case grp_lat of
    1, 2, 3, 4: { A, B, C, I } grp_mlt := 2;
    5: { F } grp_mlt := 4;
    7: { H } grp_mlt := 3;
  otherwise
    grp_mlt := 1
  end;
  grp_mlt := grp_mlt*cell_abc;
  if fix_lat then grp_inc := grp_mlt    { Set the group increment following the lattice mode. }
             else grp_inc :=   1;

  if bb = 0.0 then bb := aa;
  if cc = 0.0 then cc := aa;
  if al <> 0.0 then
  begin
    if be = 0.0 then be := al;
    if ga = 0.0 then ga := al
  end;

  if cell_abc > 1 then
    WRITELN( ' The Resulting Unit cell well be ', cell_na:0, '*A * ', cell_nb:0, '*B * ', cell_nc:0, '*C .' );

(*
WRITELN( ' Lattice code = ', grp_lat:0 );
WRITELN( ' Dtensor :' );
for i := 1 to 3 do
  WRITELN( '||':10:-1, dten[i,1]:10:6, ',',  dten[i,2]:10:6, ',',  dten[i,3]:10:6, ' ||' );
WRITELN;
*)

end GET_SPACE_GROUP_TENSOR;



procedure COMPUTE_DISTANCES;
{ Procedure to compute all inter-site distances.
}
const
  mdnam = 'CDIS';

var
  d2, dx, dy, dz,
  ex, ey, ez, t1, t2,
  xa, ya, za:             real;
  bndx, bndy, bndz:    integer;


  procedure SCAN_LSIZE;
  { Procedure to adjust the unit cell scan size to
    do not forget any site with a distance <= mxdist.
  }
  const
    mlat = 3;

  var
    mx, my, mz:        integer;
    d2, d2mx, t1, t2:     real;

  begin
    d2mx := 1.0e+10;
    bndx := 0; bndy := 0; bndz := 0;
    for nz := -mlat to mlat do
      if nz <> 0 then
      begin
        t2 := dten[3,3]*SQR( nz );
        mz := ABS( nz );
        for ny := -mlat to mlat do
          if ny <> 0 then
          begin
            t1 := ny*(dten[2,2]*ny + 2.0*dten[2,3]*nz) + t2;
            my := ABS( ny );
            for nx := -mlat to mlat do
              if nx <> 0 then
              begin
                d2 := nx*(dten[1,1]*nx + 2.0*(dten[3,1]*nz + dten[1,2]*ny)) + t1;
                if d2 <= mxdist2 then
                begin { For acceptable lattice node distance set the translation limits ... }
                  if mx > bndx then bndx := mx;
                  if my > bndy then bndy := my;
                  if mz > bndz then bndz := mz
                end
                else
                  { ... else, for a maximum indices, keep the minimum translation with 3*(a or b or c). }
                  if mx*my*my > 8 then
                    if d2mx > d2 then d2mx := d2
              end
          end
      end;
    bndx := bndx + 1; bndy := bndy + 1; bndz := bndz + 1;
    if mxdist2 > d2mx then
    begin  mxdist2 := d2mx; mxdist := SQRT( mxdist2 )  end
  end SCAN_LSIZE;


  procedure NEIGHBOURING_SORT( off, nb: integer );
  { Procedure to sort the neighbouring list by
    increasing distance.
  }
  var
    tmp: nei_rec;

  begin
    for i := off + 1 to off + nb - 1 do
      for j := i + 1 to off + nb do
        if nei_tab[i].nei_dist > nei_tab[j].nei_dist then
        begin
          tmp := nei_tab[i];
          nei_tab[i] := nei_tab[j];
          nei_tab[j] := tmp
        end
  end NEIGHBOURING_SORT;


begin { COMPUTE_DISTANCES }
  mxdist2 := SQR( mxdist );             { Set the square of maximum distance to up speed }
  SCAN_LSIZE;                           { Size the maximum of lattice translation }
  nnei := 0;                            { Init the neighbouring site table to empty }
  for ia := 1 to natm do                { Scan of all defined sites }
  with atm_tab[ia] do
  begin
    atm_oneig := nnei;                  { Set the distance table offset }
    xa := atm_x;                        { Keep the origine site coordinates }
    ya := atm_y;
    za := atm_z;
    for ja := 1 to natm do              { Loop to find all possible neighbouring sites }
      if ja <> ia then                  { Elliminate the site with itself distance }
      with atm_tab[ja] do
      begin
        dx := atm_x - xa;               { Keep the defined site distance vector }
        dy := atm_y - ya;
        dz := atm_z - za;
        for iz := -bndz to bndz do      { Scan to all distances by lattice translations }
        begin
          ez := dz + iz;
          t2 := dten[3,3]*SQR( ez );
          for iy := -bndy to bndy do
          begin
            ey := dy + iy;
            t1 := ey*(dten[2,2]*ey + 2.0*dten[2,3]*ez) + t2;
            for ix := -bndx to bndx do
            begin
              ex := dx + ix;
              d2 := ex*(dten[1,1]*ex + 2.0*(dten[3,1]*ez + dten[1,2]*ey)) + t1;
              if d2 <= mxdist2 then
              begin { We keep this distance as neighbouring site of the <ia> site }
                nnei := nnei + 1;
                with nei_tab[nnei] do
                begin
                  nei_dist := SQRT( d2 );       { Put the inter-site distance }
                  nei_atm  :=         ja;       { Put the extremity site index }
                  nei_ix := ix;                 { Put the lattice relative translation vector }
                  nei_iy := iy;
                  nei_iz := iz
                end
              end
            end
          end
        end
      end; { for ja := 1 to natm do if ja <> ia then with atm_tab[ja] do}
      atm_nneig := nnei - atm_oneig;
      NEIGHBOURING_SORT( atm_oneig, atm_nneig );
(*
;
with pos_tab[atm_posid] do
  WRITELN( ' ', atm_nneig:2, ' Neighbouring sites of the site # ', ia - pos_offs:3, ' Name = "', pos_name, '"' );
for j := 1 to atm_nneig do
  with nei_tab[atm_oneig+j], atm_tab[nei_atm] do
    WRITELN( '#':4:1, j:3, ' d = ', nei_dist:8:5, ' at (', atm_x:8:5, ',', atm_y:8:5, ',', atm_z:8:5,
             ') in cell (', nei_ix:2, ',', nei_iy:2, ',', nei_iz:2, ') of position "', pos_tab[atm_posid].pos_name, '"' );
*)

  end { for ia := 1 to natm do }
end COMPUTE_DISTANCES;



procedure SET_ATOM_SITE( natm: integer; ze: integer; rd, el: real; var elm: elemty );
{ Fill an crystallographic site with the specified data.
}
var
  xa, ya, za: real;

begin
  with atm_tab[natm] do
  begin
    atm_elem := elm;
    atm_zel := ze; atm_rad := rd; atm_elec := el;
    xa := atm_x; ya := atm_y; za := atm_z
  end
end SET_ATOM_SITE;



procedure SITE_ENERGY( ia, ze: integer; rd, el: real; var elm: elemty );
{ Evaluate the energy for one site by scanning of its neighbourings.
  <ia>    is the site number in the atom_tab list.
}
const
  mdnam = 'SENE';

var
  dm, te, sm, sr, md:     real;

begin
  te  :=           0.0;
  sm   :=     out_ener;
  dm   :=     out_ener;
  with atm_tab[ia] do
  begin
    for ja := 1 to atm_nneig do
      with nei_tab[atm_oneig+ja], atm_tab[nei_atm] do
      if atm_zel > 0 then
      begin
        sr := atm_rad + rd;
        md := exd_tab[ze,atm_zel];
        if md > sr then sr := md;
        if nei_dist > sr then
        begin
          if nei_dist <= mxdist then
          begin { Good distance to get in account the neighbouring atom }
            te := te - SQR( el - atm_elec )/SQR( nei_dist )
          end
        end
        else
        begin { Too small distance < sum of the atom radius }
          if sm > sr then sm := sr;             { Set the minimum of the sum of radius }
          if dm > nei_dist then dm := nei_dist; { Set the minimum of the distance }
          te := out_ener;                       { Set this site as not possible }
          exit
        end
      end;
    atm_ee := te;                               { Set the site "affinity/Energy" }

    if verbose then
      if te >= out_ener then
      begin
        with pos_tab[atm_posid] do
          WRITEV( smsg, 'In site # ', ia - pos_offs:3, ' at (',  atm_x:7:3, ',', atm_y:7:3, ',', atm_z:7:3,
                        ') of position "', pos_name, '". Too short distance to insert ',
                        atm_elem, ' d = ', dm:8:5, ' A < ', sm:8:4, ' A.' );
        WARNING( mdnam, smsg )
      end
  end
end SITE_ENERGY;



procedure INDEX_EXCHANGE( i, j: integer );
{ Exchange two site indices  in the site index array ran_tab.
}
var
  tmp: ran_rec;

begin
  tmp := ran_tab[i];
  ran_tab[i] := ran_tab[j];
  ran_tab[j] := tmp
end INDEX_EXCHANGE;



function ENERGY_SORT( np: integer ): integer;
{ Sort the position by increasing energy value.
  The double loop (N**2 process) is convenient because n always <= 192
  and the number of different ee values is probably small.
}
var
  ns: integer;
  vl:    real;

begin { ENERGY_SORT }
  for ii := 1 to np - 1 do
    for jj := ii + 1 to np do
      if atm_tab[ran_tab[jj].ran_aid].atm_ee < atm_tab[ran_tab[ii].ran_aid].atm_ee then INDEX_EXCHANGE( ii, jj );
  vl := atm_tab[ran_tab[1].ran_aid].atm_ee;
  ns := 2;
  while ns <= np do
  begin
  exit if atm_tab[ran_tab[ns].ran_aid].atm_ee > vl;
    ns := ns + 1;
  end;
  ENERGY_SORT := ns - 1
end ENERGY_SORT;



procedure GENERATE_ATOME( in_var sel: string; fr, ze, pop: integer; rd, el: real );
{ Procedure to select and fill a list of crystallographic site with a specified
  number of atomic element.
}
const
  mdnam = 'GEAT';

var
  ii, ip, na, ns, nu:  integer;
  ee:                     real;
  elm:                  elemty;

begin { GENERATE_ATOME }
  elm := SUBSTR( sel, 1, 2);

  na := 0;                              { Initialize the number of inserted atom(s) }
  nnei := 0;                            { Init the neigbouring count }
  while (fr > 0) and (pop > na) do      { Until all sites was tried or all atom are inserted are put in the crystal }
  begin
    for i := 1 to fr do                 { Compute the energy of each sites }
      SITE_ENERGY( ran_tab[i].ran_aid, ze, rd, el, elm );       { First one is checked when the lattice space group is preserved }
    ns := ENERGY_SORT( fr );            { Perform a sites energy sorting }

(*
for i := 1 to fr do
with ran_tab[i], pos_tab[pid], atm_tab[aid] do
WRITELN( ' Site # ', aid-poff:3, ' of position "', name, '"  -- E = ', ee:10:5 );
*)

    if atm_tab[ran_tab[1].ran_aid].atm_ee >= out_ener then fr := 0
    else
    begin
      { Select one site of this energy }
      if ns > 1 then ns := TRUNC( ns*RANDOM( rnd_seek ) ) + 1;
      with ran_tab[ns], pos_tab[ran_pid], atm_tab[ran_aid] do
      begin
        occ_tab[ran_pid,ze] := occ_tab[ran_pid,ze] + grp_inc;
        if verbose then
        begin
          WRITELN( ' ** Select site # ', ran_aid - pos_offs:3, ' of position "', pos_name:8,
                   '" at (', atm_x:8:5, ',', atm_y:8:5, ',', atm_z:8:5, ') for "', elm, '" with ', atm_nneig:0, ' neighbouring atoms.' );

          for i := 1 to atm_nneig do
            with  nei_tab[atm_oneig+i], atm_tab[nei_atm] do
              WRITELN( ' Neighbouring # ', i:3, ' ', atm_elem, ' ', atm_zel:3, ' of "', pos_tab[atm_posid].pos_name,
                       '" at (', atm_x:7:4, ',', atm_y:7:4, ',', atm_z:7:4, ') and d = ', nei_dist:8:5, ' A' );
        end;
        { Put the atom in the selected site }
        for ij := 0 to grp_inc - 1 do
          SET_ATOM_SITE( ran_aid + ij, ze, rd, el, elm );
        pos_nuse := pos_nuse + grp_inc  { Make the count of used sites in the position }
      end;

      { Elliminate it from the atom index list }
      for jj := ns to fr-1 do ran_tab[jj] := ran_tab[jj+1];
      na := na + 1
    end;
    fr := fr - 1
  end;
  if pop > na then
  begin
    if na > 0 then WRITEV( smsg, 'Only ', na:0, ' atom(s) have been inserted in the specified position' )
              else WRITEV( smsg, 'Any atom was inserted in the specified position' );
    if posnb > 1 then WRITEV( smsg:false, 's' )
                 else WRITEV( smsg:false, ' "', pos_tab[pos_lst[1]].pos_name, '"' );
    WARNING( mdnam, smsg )
  end
end GENERATE_ATOME;



procedure PUT_ATOMS( in_var sel: string; ze, pop: integer; rd, el: real );
{ Procedure to fill a list of crystallographic positions with a specified
  number of atomic element.
}
const
  mdnam = 'PATM';

var
  ii, ip, fr, sz:      integer;

begin
  fr := 0; sz := 0;
  for i := 1 to posnb do
    with pos_tab[pos_lst[i]] do
    begin
      sz := sz + pos_size;              { Summ the total number of sites }
      ii := pos_offs +  1;              { Get the index of the first site }
      for j := 1 to pos_size do
      begin
        with atm_tab[ii] do
        if atm_zel = 0.0 then           { For the free site only }
          if fr < max_ran  then
          begin
            fr := fr + 1;               { Allocate room in the Sort/randomize index table ... }
            ran_tab[fr].ran_aid := ii;  { ... and Fill it with atom and position index. }
            ran_tab[fr].ran_pid :=  pos_lst[i]
(*
;WRITELN( ' Examine site # ', ii:3, ' : (', xx:8:5, ',', yy:8:5, ',', zz:8:5, ')' )
*)
          end
          else
          begin
            WRITEV( smsg, 'Too many free sites to manage with several positions (max = ', max_ran:0, ')' );
            ERROR( mdnam, smsg )
          end;
        ii := ii + grp_inc              { Skip the not P lattice translation for the lattice driven position.  }
      end
    end;

  if pop <= 0 then pop := fr
  else
    if fr <= 0 then ERROR( mdnam, 'All specified position(s) are full' )
   else
    if pop > fr then
    begin
      WRITEV( smsg, 'Too many atom for the specified positions. They can hold(s) ',
                    fr:0, '/', sz:0, ' atoms.' );
      WARNING( mdnam, smsg );
      pop := fr
    end;

  GENERATE_ATOME( sel, fr, ze, pop, rd, el );

  WRITE( ' The position' );
  if posnb > 1 then WRITE( 's' );
  WRITELN( ' :' );
  for i := 1 to posnb do
  with pos_tab[pos_lst[i]] do
  begin
    WRITE( ' ':4, pos_name:8, ' with ', pos_nuse:0, '/', pos_size:0, ' occupied site' );
    if pos_nuse > 1 then WRITE( 's' );
    if i < posnb then WRITELN( ',' )
                 else WRITELN( '.' )
  end;

  WRITELN( ' with Elem = "', sel, '", Z = ', ze:2, ', pop = ', pop:0,
           ', R = ', rd:8:4, ', Elec = ', el:8:4 );
  WRITELN
end PUT_ATOMS;



procedure BUILD_POSITION( in_var namep: string; x, y, z: real );
{ Procedure to build crystallographic position structures.
}
const
  mdnam = 'BLDP';

var
  st:                   string;
  ip, np, mlt, off:    integer;
  ax, ay, az, yy, zz:     real;


  procedure GEN_SITE_PLACE_0( x, y, z: real; tx, ty, tz: integer );
  const
     mdnam = 'ASIT';

  begin
    { Put coordinate in the cell range [0.0, 1.9[ }
    if x >= 1.0 then x := x - TRUNC( x )
                else if x < 0.0 then x := x + 1.0 - TRUNC( x );
    if y >= 1.0 then y := y - TRUNC( y )
                else if y < 0.0 then y := y + 1.0 - TRUNC( y );
    if z >= 1.0 then z := z - TRUNC( z )
                else if z < 0.0 then z := z + 1.0 - TRUNC( z );

    x := x + tx; y := y + ty; z := z + tz;

    if natm < max_atom then natm := natm + 1
                       else ERROR( mdnam, ' Atom table overflow -- Too many atoms in the sunit cell.' );
    with atm_tab[natm] do
    begin
      atm_elem  :=        '  ';         { Set the element name to null }
      atm_posid :=        npos;         { Set the position index }
      atm_oneig :=           0;         { Init the neighbouring information }
      atm_nneig :=           0;
      atm_zel   :=           0;         { Init the element number of electron }
      atm_elec  :=         0.0;         { Init the electro-negativity/energie }
      atm_rad   :=         0.0;         { ... the atomic radius, ... }
      atm_x := x/cell_na;               { ... and set the atom coordinates (in the multiple cell ... }
      atm_y := y/cell_nb;               { ... when defined) }
      atm_z := z/cell_nc

(*
;WRITELN( ' Site # ', natm:4, ', Electronegativity = ', zel:4:1, ', R = ', rad,8:4, ', x,y,z = ', xx:8:5, yy:8:5, zz:8:5 );
*)
    end
  end GEN_SITE_PLACE_0;


  procedure GEN_SITE_PLACE_1( x, y, z: real; tx, ty, tz: integer );
  begin
    GEN_SITE_PLACE_0( ax, ay, az, tx, ty, tz );
    case grp_lat of
      1 { A }: GEN_SITE_PLACE_0( ax + 0.0, ay + 1/2, az + 1/2, tx, ty, tz );
      2 { B }: GEN_SITE_PLACE_0( ax + 1/2, ay + 0.0, az + 1/2, tx, ty, tz );
      3 { C }: GEN_SITE_PLACE_0( ax + 1/2, ay + 1/2, az + 0.0, tx, ty, tz );
      4 { I }: GEN_SITE_PLACE_0( ax + 1/2, ay + 1/2, az + 1/2, tx, ty, tz );
      5 { F }: begin
                 GEN_SITE_PLACE_0( ax + 0.0, ay + 1/2, az + 1/2, tx, ty, tz );
                 GEN_SITE_PLACE_0( ax + 1/2, ay + 0.0, az + 1/2, tx, ty, tz );
                 GEN_SITE_PLACE_0( ax + 1/2, ay + 1/2, az + 0.0, tx, ty, tz )
               end;
      7 { H }: begin
                 GEN_SITE_PLACE_0( ax + 2/3, ay + 1/3, az + 1/3, tx, ty, tz );
                 GEN_SITE_PLACE_0( ax + 1/3, ay + 2/3, az + 2/3, tx, ty, tz )
               end;
    otherwise
      { P and R : Nothing to do }
    end;
  end GEN_SITE_PLACE_1;


begin { BUILD_POSITION }
  ip := 1;
  while ip <= npos do
  begin
  exit if pos_tab[ip].pos_name = namep;
    ip := ip + 1
  end;
  if ip <= npos then
  begin
    WRITEV( smsg, ' The position "', namep, '" was already existing, we ignore this new definition.' );
    WARNING( mdnam, smsg );
    return
  end;

  if x >= 1.0 then x := x - TRUNC( x )
              else if x < 0.0 then x := x + 1.0 - TRUNC( x );
  if y >= 1.0 then y := y - TRUNC( y )
              else if y < 0.0 then y := y + 1.0 - TRUNC( y );
  if z >= 1.0 then z := z - TRUNC( z )
              else if z < 0.0 then z := z + 1.0 - TRUNC( z );

(*
WRITELN( ' Pos. Name "', namep, '" Lat_flg = ', lflg, ', x,y,z = [', x:8:5, ',', y:8:5, ',', z:8:5, ' ]' );
*)

  WRITEV( st, x:8:5, ',', y:8:5, ',', z:8:5, ';' );
  NO_STRING_SPACE( st );
  WRITEV( egs_ccmd, egs_cmd0, ' -pos=', st );

(*
WRITELN( ' Execute "', egs_ccmd, '"' );
*)

  EXECUTE_GEN_SPACE( egs_ccmd );

  OPEN( str_file, gen_space_tmp, [read_file,error_file{,del_file}] );
  if (iostatus <> 0) or EOF( str_file ) then ERROR( mdnam, 'Cannot open the temporary space group file "', gen_space_tmp );

  READLN( str_file );                   { Skip the first line with group specifications }
  READLN( str_file, np );               { Get the position order and skip its multiplicity }

  npos := npos + 1;                     { Define the new position }
  off  :=     natm;                     { Get the position offset }
  with pos_tab[npos] do                 { Fill the position record }
  begin
    pos_name :=      namep;             { Fill the new position record with the position name, ... }
    pos_size := np*grp_mlt;             { ... the position order, ... }
    pos_nuse :=          0;             { ... the number of atome presently in the position and ... }
    pos_offs :=        off              { ... the position offset. }
  end;

  for zen := 1 to max_ze do             { Initialize the contain table for this position }
    occ_tab[npos,zen] := 0;

  for ii := 1 to np do                  { Loop on all equivalent positions }
  begin
    READLN( str_file, ax, ay, az );     { Read the equivalent position coordinates }

    for ic := 0 to cell_nc - 1 do
      for ib := 0 to cell_nb - 1 do
        for ia := 0 to cell_na - 1 do
          GEN_SITE_PLACE_1( ax, ay, az, ia, ib, ic )
  end;
  CLOSE( str_file );

  if verbose then
  begin

    with pos_tab[npos] do
      WRITELN( ' Create position "', namep, '" of ', pos_size:0, ' sites (offset = ', pos_offs:0, ')' );
    WRITELN
  end
end BUILD_POSITION;



procedure BUILD_STRUCTURE;
{ Head procedure to read and execute the sample build statements.
}
const
  mdnam = 'BUIL';

var
  sel:                  elemty;
  sn, st:               string;
  fl, pop, ze:         integer;
  ee, pp, rd, x, y, z:    real;
  bput:                boolean;



  procedure ADD_NEW_POSITION;
  var
    ip, ij: integer;
    sname:  posname;

  begin
    sname := inp_str; CHANGE_CASE( sname );
    ip := 1;
    while ip <= npos do
    begin
      exit if pos_tab[ip].pos_name = sname;
      ip := ip + 1
    end;
    if ip > npos then
    begin
      WRITEV( smsg, 'The specified position "', sname, '" does not exist. We ignore this entry' );
      WARNING( mdnam, smsg );
      return
    end;
    ij := 1;
    while ij <= posnb do
    begin
    exit if pos_lst[ij] = ip;
      ij := ij + 1
    end;
    if ij <= posnb then
    begin
      WRITEV( smsg, 'The specified position "', sname, '" is already in the list. We ignore this entry' );
      WARNING( mdnam, smsg );
      return
    end;
    if posnb < max_poslist then
    begin
      posnb := posnb + 1;
      pos_lst[posnb] := ip
    end
    else
    begin
      WRITEV( smsg, 'Too many positions are specified (max = ', max_poslist:0, ')' );
      WARNING( mdnam, smsg )
    end
  end ADD_NEW_POSITION;



begin { BUILD_STRUCTURE }
  bput := false;

ET_STATE:
  while (inp_symb <> sy_eof) and (npos < max_pos) do
  begin
    inp_prompt := ' Gen_Approx Position/Put_Atom > ';
    INP_INQUIRE( '' );

  exit if inp_symb = sy_eof;
    case inp_symb of
      sy_ident:
        if inp_code >= 0 then
        case kw_symb( inp_code ) of
          kw_group,
          kw_cell,
          kw_extnd,
          kw_flat,
          kw_axis,
          kw_orig:
            begin
WRITELN( ' Code = ', inp_code );
              ERROR( mdnam, 'Illegal statement in this context (after position/put_atom/end)', inp_ident )
            end;

          kw_title:
            begin
              inp_mdlin := true; INP_INQUIRE( ' Title = ' );
              str_title := inp_str;
              if verbose then WRITELN( ' Title = "', str_title, '" is set.' )
           end;

(* ///
          kw_mxdist:
            begin
              INSYMBOL;
              INP_R_VALUE( ' Mx_Dist = ', mxdist );
              mxdist := ABS( mxdist );
              if verbose then WRITELN( ' Max_Dist = ', mxdist:8:4, ' is set.' )
            end;
/// *)

          kw_excdis: SET_Z_EXCL_DISTANCE;

          kw_atmpos:
            if not bput then
            begin
              inp_rsstr := true; inp_prompt2 := ' Position Name = '; INSYMBOL;
              if inp_symb <> sy_str then ERROR( mdnam, 'A New Position Name was expected' );
              sn := inp_str; CHANGE_CASE( sn ); INSYMBOL;
              fl := 0; x := 0.0; y := 0.0; z := 0.0;
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_R_VALUE( ' X = ', x );
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_R_VALUE( ' Y = ', y );
              if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
              INP_R_VALUE( ' Z = ', z );
              BUILD_POSITION( sn, x, y, z )
            end else ERROR( mdnam, 'Illegal position definition statement after a putatom statement' );

          kw_putatm:
            begin
              if not bput then
              begin
                COMPUTE_DISTANCES;      { Compute all inter-site distances }
                bput := true            { Lock any future position statement(s) }
              end;
              posnb := 0;
              inp_rsstr := true; inp_prompt2 := ' Position Name or (pos. name list) = '; INSYMBOL;
              if inp_symb = sy_lpar then
              begin
                repeat
                  inp_rsstr := true; inp_prompt2 := ' Position Name = '; INSYMBOL;
                  if posnb > 0 then
                    if (inp_symb = sy_comma) or (inp_symb = sy_colon) then
                    begin  inp_mdstr := true; INSYMBOL  end;
                  if inp_symb <> sy_rpar then
                  begin
                    if inp_symb <> sy_str then ERROR( mdnam, 'A New Position Name was expected in the list' );
                    ADD_NEW_POSITION
                  end
                until (inp_symb = sy_rpar) or (posnb >= max_poslist);
                if inp_symb <> sy_rpar then WARNING( mdnam, 'A ")" was expected to end the position list' )
              end
              else
              begin
                if inp_symb <> sy_str then ERROR( mdnam, 'A New Position Name was expected' );
                ADD_NEW_POSITION
              end;
              if posnb > 0 then
              begin
                INSYMBOL;
                GET_ZN_ELEMENT( ' Element = ', sel, ze );
                if (ze < 1) or (ze > max_ze) then ERROR( mdnam, 'Illegal atomic Z number (must be in range 1..100)' );
                if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
                INP_R_VALUE( ' Pop = ', pp );
                pop := ROUND( cell_abc*pp );
                if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
                INP_R_VALUE( ' R = ', rd );
                if (inp_symb = sy_comma) or (inp_symb = sy_colon) then INSYMBOL;
                INP_R_VALUE( ' Elec = ', ee );
                PUT_ATOMS( sel, ze, pop, rd, ee )
              end
            end;

          kw_end: exit;

        otherwise
          ERROR( mdnam, 'An implemented directive keyword was expected' );
        end
        else ERROR( mdnam, 'A Defined directive keyword was expected' );

      sy_eoln, sy_semicolon:
        INSYMBOL;

      sy_eof:
        begin
          WARNING( mdnam, ' Return to standard input' );
          inp_file_flg := false;
          CLOSE( inp_file );
          inp_symb := sy_eoln;
          INSYMBOL
        end;

    otherwise
      ERROR( mdnam, 'Input file syntax error.' )
    end
  end { of while };
end BUILD_STRUCTURE;



procedure OUTPUT_DISTANCE;
{ Procedure to produce a listing of used parameters,
  sample builting and inter-atomic distance list.
}
const
  mdnam = 'ODIS';

  procedure OUTPUT_DISTANCE1( var f: text );
  var
    bfrs:              boolean;
    ind:               integer;
    alp, bet, gam:        real;

  begin
    WRITELN( f );
    WRITELN( f, ' * * *  Gen_Approx ', version, ' to generate aproximative phases distance   * * *' );
    WRITELN( f );
    WRITELN( f );

    WRITELN( f, ' Unit cell :' );
    if al < 1.0 then alp := ARCCOS( al ) else alp := al;
    if be < 1.0 then bet := ARCCOS( be ) else bet := be;
    if ga < 1.0 then gam := ARCCOS( ga ) else gam := ga;
    WRITELN( f, 'A = ':10:1, aa:10:6, ' Ang., B = ', bb:10:6, ' Ang., C = ', cc:10:6,
                ' Ang., Alpha = ', alp:8:4, ' °, Beta = ', bet:8:4, ' °, Gamma = ', gam:8:4, ' °' );
    WRITELN( f );
    WRITE( f, ' The basic space group is "', egs_group, '"' );
    if egsb_orig then WRITE( f, ' with origin at ', orgx:8:6, ',', orgy:8:6, ',', orgz:8:6 );
    if egsb_axis then
    begin
      ind := INDEX( egs_axis, '=' );
      WRITE( f, ' with the axis permuttation ', SUBSTR( egs_axis, ind + 1 ) )
    end;

    bfrs := true;
    for z1 := 1 to max_ze do
      for z2 := z1 to max_ze do
        if exd_tab[z1,z2] > 0.0 then
        begin
          if bfrs then
          begin
            WRITELN( f );
            WRITELN( f );
            WRITELN( f, ' Minimum distances between atoms:' );
            WRITELN( f );
            bfrs := false
          end;
          WRITELN( f, '   Distance( ', elm_tab[z1], ', ', elm_tab[z2], ' ) >= ', exd_tab[z1,z2]:6:3 )
        end;

    WRITELN( f );
    WRITELN( f );
    WRITELN( f, ' * * * Defined Position(s) and atom contain * * *' );
    WRITELN( f );
    for ip := 1 to npos do
    with pos_tab[ip] do
      if pos_nuse > 0 then
      begin
        bfrs := true;
        WRITE( f, 'Position "':16:1, pos_name:8, '" with ', pos_nuse:3, '/', pos_size:3, ' atom' );
        if pos_nuse > 0 then WRITE( f, 's : ' );
        for zz := 1 to max_ze do
          if occ_tab[ip,zz] > 0 then
          begin
            if bfrs then bfrs := false
                    else WRITE( f, ', ' );
            WRITE( f, occ_tab[ip,zz]:3, ' ', elm_tab[zz]:2:1 )
          end;
        WRITELN( f, '.' )
      end;

    WRITELN( f );
    WRITELN( f );
    WRITELN( f, ' Resulting Inter-Atomic distance for "', str_title, '"' );
    WRITELN( f );
    for ip := 1 to npos do
    with pos_tab[ip] do
    begin
      WRITELN( f, ' Position "', pos_name, '"  with ', pos_nuse:0, '/', pos_size:0, ' atoms' );
      for ia := 1 to pos_size do
      with atm_tab[ia+pos_offs] do
        if atm_zel > 0.0 then
        begin
          WRITELN( f, '   From (', ia:3, ') at [', atm_x:7:4, ',', atm_y:7:4, ',', atm_z:7:4,
                      ']  with ', atm_elem:2, ' Z = ', atm_zel:3 );
          for jd := 1 to atm_nneig do
          with nei_tab[jd+atm_oneig], atm_tab[nei_atm], pos_tab[atm_posid] do
            if atm_zel > 0 then
            begin
              WRITELN( f, ' ':8, atm_elem:2, ' at [', atm_x:7:4, ',', atm_y:7:4, ',', atm_z:7:4,
                          '] of "', pos_name:6, '" # ', nei_atm - pos_offs:3,
                          ' with T = [', nei_ix:2, ',', nei_iy:2, ',', nei_iz:2, '] d = ', nei_dist:8:5, ' A' )
            end;
            WRITELN( f )
        end
    end;
    WRITELN( f )
  end OUTPUT_DISTANCE1;


begin { OUTPUT_DISTANCE }
  if disfname.length < 1 then
    OUTPUT_DISTANCE1( output )
  else
  begin
    OPEN( dis_file, disfname, [write_file,error_file] );
    if iostatus = 0 then OUTPUT_DISTANCE1( dis_file )
                    else begin
                           WRITEV( smsg, 'Open File Error # ', iostatus:0, ' => Cannot create the file ' );
                           WARNING( mdnam, smsg, disfname )
                         end
  end
end OUTPUT_DISTANCE;



procedure GEN_HEAD_STRUCT( nat: integer );
{
Fe40 Ni27 Zr33 -> Fe10 Ni6 Zr8
P   LATTICE,NONEQUIV.ATOMS: 25
MODE OF CALC=RELA unit=ang 
 13.245096 13.245096 13.245096 90.000000 90.000000 90.000000
}
const
  ang_bohr = 1.889725989;

var
  chl:            char;

begin
  if fix_lat then chl := egs_group[1]
             else chl := 'P';
  WRITELN( str_file, str_title );
  WRITELN( str_file, chl, '   LATTICE,NONEQUIV.ATOMS:', nat:3 );
  WRITELN( str_file, 'MODE OF CALC=RELA unit=ang' );
  WRITELN( str_file, fpaa*ang_bohr:10:6, fpbb*ang_bohr:10:6, fpcc*ang_bohr:10:6, al:10:6, be:10:6, ga:10:6 )
end GEN_HEAD_STRUCT;



procedure GEN_WATOM( num, iat: integer );
{ To generate the atom output for WIEN2K Software :

ATOM -21: X=0.62500000 Y=0.87500000 Z=0.87500000
          MULT= 1          ISPLIT= 8
Fe16d_13   NPT=  781  R0=0.00010000 RMT=    2.2100   Z: 26.0
LOCAL ROT MATRIX:    1.0000000 0.0000000 0.0000000
                     0.0000000 1.0000000 0.0000000
                     0.0000000 0.0000000 1.0000000
}
var
  iap: integer;

begin
  with atm_tab[iat], pos_tab[atm_posid] do
  begin
    iap := iat - pos_offs;
    WRITELN( str_file, 'ATOM', num:4, ': X=', atm_x:10:8, ' Y=', atm_y:10:8, ' Z=', atm_z:10:8 );
    WRITELN( str_file, '          MULT= 1          ISPLIT= 0' );
    WRITE( str_file, atm_elem:2, pos_name:3 );
    if iap < 100 then WRITE( str_file, '_', iap:2 )
                 else WRITE( str_file, iap:3 );
    WRITELN( str_file, '   NPT=  781  R0=0.00010000 RMT=    2.2100   Z:', atm_zel:3, '.0' );
    WRITELN( str_file, 'LOCAL ROT MATRIX:    1.0000000 0.0000000 0.0000000' );
    WRITELN( str_file, '                     0.0000000 1.0000000 0.0000000' );
    WRITELN( str_file, '                     0.0000000 0.0000000 1.0000000' )
  end
end GEN_WATOM;



procedure GEN_TRAILER_STRUCT;
begin
  WRITELN( str_file, '   0      NUMBER OF SYMMETRY OPERATIONS' )
end GEN_TRAILER_STRUCT;



procedure WRITE_STRUCT_FILE;
var
  sel:                  elemty;
  acp, nat, num:       integer;

begin
  { Create the output file }
  OPEN( str_file, outfname, [write_file,error_file] );
  if iostatus <> 0 then
  begin
    WRITELN( ' *** Gen_Approx Cannot Open the Output File "', outfname, '" => EXit.' );
    PASCAL_EXIT( 2 )
  end;

  nat := 0;
  acp := 1;
  while acp <= natm do
  begin
    if atm_tab[acp].atm_zel > 0 then nat := nat + 1;
    acp := acp + grp_inc
  end;

  GEN_HEAD_STRUCT( nat );

  sel := '  ';
  num := 0;
  acp := 1;
  for ii := 1 to npos do
  with pos_tab[ii] do
  begin
    acp := 1;
    while acp <= pos_size do
    begin
      with atm_tab[acp+pos_offs] do
        if atm_zel > 0 then
        begin
          num := num + 1;
          GEN_WATOM( num, acp+pos_offs )
        end;
      acp := acp + grp_inc
    end
  end;

  GEN_TRAILER_STRUCT;
  CLOSE( str_file );
end WRITE_STRUCT_FILE;



begin { GEN_PHASES_APPROXIMANTES }
  SEARCH_FILE( search_path, help_file_spc, 4 { For read }, gap_hlpfile, path_ok );
  if path_ok then
  begin
    gap_int := INDEX( gap_hlpfile, '/', -1 );
    if gap_int = 0 then gap_envdir.length := 0
                   else gap_envdir := SUBSTR( gap_hlpfile, 1, gap_int )
  end;
  INIT;                         { Get all the command line data }
  if verbose then WRITELN( ' Command file = "', inpfname, '".' );
  GET_FIRST_DATA;               { Get all preliminary DATA }
  LOAD_RANDOM_STATE;            { Load random generator status }
  GET_SPACE_GROUP_TENSOR;       { Get and check the space group, unit cell and get the metric tensor }
  BUILD_STRUCTURE;              { Build the appoximante structure following the user directives }
  SAVE_RANDOM_STATE;            { Save random generator status }
  if distout then
    OUTPUT_DISTANCE;            { Output the distance on the standard output }
  WRITE_STRUCT_FILE;            { Write the Wien2K structure file }
  WRITELN;
  WRITELN( ' Normal End.' )
end GEN_PHASES_APPROXIMANTES.
