{
        *****  CPAS Portable Shell External Environment  *******

*************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               G D A T A   (Get DATA for Diffraction)                  *
*                   using the CPAS SHELL Environment                    *
*               ( ILL Data Base Manager Source File )                   *
*                                                                       *
*                                                                       *
*                 Version  1.4-B  - - 15-Jun-2012                       *
*                                                                       *
*                                by                                     *
*                                                                       *
*                  Pierre Wolfers, Institut Neel                        *
*                                                                       *
*            CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166            *
*                                                                       *
*                       F 38042 GRENOBLE CEDEX 9                        *
*                                                                       *
*                             F R A N C E                               *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}
module GDATA_UTIL;


%include 'PASENV:cpas_d__sdir_env.pas'; { Get the Directory search Library environment }

%include 'GDASRC:gdata_type_env.pas';   { Get all usefull GDATA/LST/SRC definitions }


const
  illdat_linesz = 80;                   { Define the ILL Numor Data line size }

  gdata_nkey    =  8;                   { Maximum number of Data Base key }
  gdata_keys    = 24;                   { Size of a Data Base Key String }

  illdat_indexf =   'ill_index.data';   { ILL DATA Instrument Index File Name }
  illdat_sbindf = 'numor_index.data';   { ILL DATA Group Sub-Index Filename }

type

  { *** Define a ILL Numor file line *** }
  illdat_ltyp = packed array[1..illdat_linesz] of char;

  illdat_ptr  = ^illdat_rec := nil;     { Define a Numor Data Line Pointer }

  illdat_rec  = record                  { * Define a Numor Data Line Record }
    illdat_nxt,                         { Pointer to the next line }
    illdat_prv:         illdat_ptr;     { Pointer to the previous line }
    illdat_nbr:            integer;     { Line number }
    illdat_line:        illdat_ltyp     { the line character array }
  end;


  illdsc_typs = (     illdsc_field,     { Input field specification DIR }
                        illdsc_skp,     { Skip directive DIR }
                        illdsc_pos,     { Absolute position DIR }
                        illdsc_sea,     { Search DIR }
                        illdsc_par,     { Parameter DIR }
                        illdsc_dat,     { Data Organization main DIR }
                        illdsc_ddd,     { Data Dimension Definition DIR }
                        illdsc_dmd,     { Data Measure Definition DIR }
                        illdsc_whl,     { Data Get While Loop }
                        illdsc_seq,     { Gdata Sequence to execute }
                        illdsc_iwk      { Gdata work space number use }
               );

  illref_flgt = (     illref_flgln,     { Flag for Indirect line count }
                      illref_flgbc,     { Flag for Indirect position (in char) count }
                      illref_flgsz      { Flag for Indirect size count }
                );

  illref_flags = set of illref_flgt;    { Define the flag set for field reference }

  illsca_ptr  =        ^illsca_rec;     { Define the Structure for ILLDATA Base Scan }

  illdsc_ptr  =        ^illdsc_rec;     { Define an ILL Data Interpretor Record Pointer }

  illbad_ptr  =        ^illbad_rec;     { Define a bad cell management specifier pointer }

  illref_rec  = record                  { * Define the reference equivalence record - for indirect ref. }
    case boolean of
      false:( illdsc_rfc:  integer);    { Reference of an integer constant }
      true: ( illdsc_rfm:  idm_ptr)     { Reference to a macro expression }
  end;

  illdsc_rec  = record                  { * Define an ILL Data Interpretor Record = DIR }
    illdsc_nxt:         illdsc_ptr;     { Link to next DIR }
    case illdsc_typ: illdsc_typs of
      illdsc_field:(                    { *** Entry field specification *** }
          illdsc_fln,                   { Relative line specification # }
          illdsc_fbc,                   { Relative field (in char count) }
          illdsc_fsz:   illref_rec;     { Entry field size }
          illdsc_frf:   illref_flags    { Reference flag set }
        );
      illdsc_skp, illdsc_pos:(          { *** Abs. and Rel. Position DIR *** }
          illdsc_ife:      idm_ptr;     { Condition Macro expression  true if > 0.5 or nil }
          illdsc_lng,                   { Line number/count }
          illdsc_col:   illref_rec;     { column number/count }
          illdsc_spf:   illref_flags    { Reference flag set }
        );
      illdsc_sea:(                      { *** Search string DIR }
          illdsc_occ:   illref_rec;     { Occurence # if>=0 else line md. }
          illdsc_sst:      pstring;     { String to search }
          illdsc_sef:   illref_flags    { Reference flag set }
        );
      illdsc_par:(                      { *** Parameter definition DIR(s) *** }
          illdsc_pkn:    exp_kinds;     { Parameter kind }
          illdsc_pad:      ref_ptr;     { Address location }
          illdsc_pof,                   { For array: Offset, }
          illdsc_pcr,                   { ... current index and ... }
          illdsc_psz:      integer;     { ... size }
          illdsc_pfo:   illdsc_ptr      { Format to use in ILL DATA }
        );
      illdsc_dat:(                      { *** Data Organization DIR *** }
          illdsc_hdd,                   { Head list of Data Dim. Def. }
          illdsc_hdm:   illdsc_ptr;     { Head list of Data Mea. Def. }
          illdsc_ndi,                   { Number of dimension }
          illdsc_nme:      integer      { Number of measure (with sigma) }
        );
      illdsc_ddd:(                      { *** Data Dimension Definition *** }
          illdsc_ddn:   illdsc_ptr;     { Array reference }
          illdsc_ddi,                   { Initial value of coordinate }
          illdsc_dds:       double;     { Step of coordinate }
          illdsc_ddv,                   { Current value multiply }
          illdsc_ddl,                   { Size for this dimension }
          illdsc_ddt:      integer;     { Total array size }
          illdsc_sta,                   { Start expression macro parm }
          illdsc_end,                   { End value expression macro parm }
          illdsc_stp:      idm_ptr      { Step value expression macro parm }
        );
      illdsc_dmd:(                      { *** Data Measure Definition *** }
          illdsc_dmn,                   { Data Measure reference }
          illdsc_dme,                   { Sigma Measure reference }
          illdsc_csk,                   { Conditional skip pref. }
          illdsc_dmf,                   { Measure field specification }
          illdsc_dms:   illdsc_ptr;     { Sigma field specification or nil }
                 );
      illdsc_whl:(                      { Data While Loop to get any number of sub-set of data }
          illdsc_scd:      idm_ptr;     { Loop condition reference }
          illdsc_dal:   illdsc_ptr;     { Data loop body sequence }
        );
      illdsc_seq:(
          illdsc_msq:      idm_ptr      { *** Gdata Sequence to execute *** }
        );
      illdsc_iwk:(
          illdsc_iwrk:     integer      { *** Data set in the specified IW *** }
        )
  end;



  mseq_ptr = ^mseq_rec;

  mseq_rec = record
    mseq_nxt: mseq_ptr;
    mseq_txt: idm_ptr
  end;


var
  illdat_iniwork,                       { Initial Work Command String to Apply }
  illdat_endwork,                       { Final Work Command String }
  illdat_cmptyp,                        { Compressed file type (defaulted to '.Z') }
  illdat_uncompr0,
  illdat_uncompr1:      str_string;     { Uncompress command strings to use }

  illdat_compidx,                       { The compress data index (0 for not compressed) }
  illdat_corrnum,                       { Data correction factor numor }
  illdat_fgrp,                          { Field of group fact. for ILL Data subdirectory }
  illdat_ngrp:  integer     :=   0;     { Groupe factor for ILL Data subdirectory }

  illdat_rindex,                        { Define the ILL Data Root Index File Specification }
  illdat_numorf,                        { Define the ILL Data Current Numor File Specification }
  illdat_numtyp,                        { Define the current numor file type }
  illdat_tmp,                           { Define the ILL Data Temporary Directory }
  illdat_dir,                           { Define the Current ILL Data Directory (including the group specification) }
  illdat_rdir:   str_string :=  '';     { Define the Instrument ILL Data Root Directory }

  illdat_crch:  integer     :=   0;     { Define the current character count }
  illdat_curr,                          { Define the current numor stream line }
  illdat_hde,                           { Define the Numor Data Line head pointer }
  illdat_free:  illdat_ptr  := nil;     { ... and the related free list head }


  illdsc_last,
  illdsc_root:  illdsc_ptr  := nil;     { Root of ILL Data Interpretor record DIR }

  illcurr_num:  integer     :=   0;     { The current numor to get }

  illdat_mcf,                           { Master Catalogue File }
  illdat_scf:                 text;     { Sub Catalog File }




{*************************************************************************}
{***  L O C A L   S P A C E   D A T A   B A S E   D E F I N I T I O N  ***}
{*************************************************************************}


var
  bfull:       boolean  := false;       { Flag for array overflow during get data }


  gdb_inp,                              { Input Data Base Descriptor }
  gdb_out: [global] gdb_ptr := nil;     { Output Data Base Descriptor }

  gdb_inpb,                             { Input Data Base Binary flag }
  gdb_outb,                             { Output Data Base Binary flag }
  gdb_inpf,                             { Input Data Base flag }
  gdb_outf:    boolean  := false;       { Output Data Base flag }

  gdb_ifspc,                            { Input Binary Data Base File specification }
  gdb_ofspc:   string(255) := '';       { Output Binary Data Base File specification }

  gdb_inputf,                           { Data Base Input File }
  gdb_ouputf:  file of  gdbf_rec;       { Data Base Output File }

  gdb_outascf:              text;       { Database Output ASCII file }

  dat_seqhde,
  dat_seqlst,                           { *** List of proceed Gdata Sequence *** }
  dat_esqhde,
  dat_esqlst,                           { *** List of On Error (numor not found) Gdata Sequense *** }
  dat_fsqhde,
  dat_fsqlst:  mseq_ptr :=   nil;       { *** List of proceed Gdata End Sequence *** }

  dat_corr:    boolean  := false;       { Float to set/clear the correction count system }
  dat_cntmin:  gdreal   :=  1E-6;       { Minimum of count }

  gdat_iwrt:   integer  :=     0;       { Count of data set write to a Data Base }
  gdat_iwrk:   integer  :=     2;       { Work space number to get a new numor }

  gdat_nsum:   integer  :=     0;       { Count of summed data set }

  gdat_skeytb: array[1..gdata_nkey] of string( 2*gdata_keys );      { Table of search key strings }

  gdat_snum_min,                        { Minimum and maximum of numor in profil search }
  gdat_snum_max,
  gdat_skeynb: integer  :=     0;       { Number of used Search keys }






procedure SET_STRING_PARM( var p: pstring; in_var st: string );
{ To do a copy of the string st with size = capacity }
var
  i: integer;

begin
  if p <> nil then begin  DISPOSE( p ); p := nil  end;
  if st.length > 0 then
  begin
    i := st.length;
    while (i > 0) and (st[i] = ' ') do i := i - 1;
    if i > 0 then
    begin
      NEW( p, i );
      p^ := SUBSTR( st, 1, i )
    end
  end  
end SET_STRING_PARM;





{***************************************************************}
{****    I L L   D A T A   F O R M A T    R O U T I N E S   ****}
{***************************************************************}



procedure ILLNEW_FREF( var ref: illref_rec; var flgs: illref_flags; flg: illref_flags; def: integer := 0 );
const
  mdnam = 'IREF';

begin
  with ref do
    if sy_sym.sy = indirsign then       { The "^" character is gobble up by the NEW_MACRO_EXPR function }
    begin
      illdsc_rfm := NEW_MACRO_EXPR;     { Get the expression as a macro expression }
      { The stop character can be colon, comma or semicolon }
      flgs := flgs + flg                { ... and flags it }
    end
    else illdsc_rfc := GET_INTEXPR( def )
end ILLNEW_FREF;


procedure ILLNEW_FIELD( var pdsc: illdsc_ptr; bind: boolean := false );
{ To allocate and initialize an ILL data record descriptor }
begin
  NEW( pdsc, illdsc_field );
  with sy_sym, pdsc^ do
  begin
    illdsc_nxt := nil;
    illdsc_typ := illdsc_field;
    illdsc_frf := [];
    if bind then ILLNEW_FREF( illdsc_fln, illdsc_frf, [illref_flgln] )
            else illdsc_fln.illdsc_rfc := GET_INTEXPR( 0 );
    if sy = colon then
    begin
      INSYMBOL;
      if bind then ILLNEW_FREF( illdsc_fbc, illdsc_frf, [illref_flgbc] )
              else illdsc_fbc.illdsc_rfc := GET_INTEXPR( 0 );
      if sy = colon then
      begin
        INSYMBOL;
        if bind then ILLNEW_FREF( illdsc_fsz, illdsc_frf, [illref_flgsz] )
                else illdsc_fsz.illdsc_rfc := GET_INTEXPR( 0 )
      end
      else illdsc_fsz.illdsc_rfc := 0
    end
    else illdsc_fbc.illdsc_rfc := 0
  end
end ILLNEW_FIELD;



function ILLDATA_NEWPREF( bf: boolean; sk: set of exp_kinds; bind: boolean := false ): illdsc_ptr;
{ Set an array reference as a ILL Data Memory }
const
  mdnam = 'ILLP';

var
  rec:    exp_rec;
  p:   illdsc_ptr;

begin
  p := nil;
  GET_EXP_REFER( rec );                 { Get a variable reference }
  with rec, sy_sym do
  if exp_ref <> nil then
  begin
    NEW( p, illdsc_par );               { Allocate a ILL Descriptor }
    with p^ do
    begin
      exp_ref^.ide_lock := true;        { Lock cpsh variable for array size change }
      illdsc_nxt  :=        nil;
      illdsc_typ  := illdsc_par;
      illdsc_pkn  :=   exp_kind;        { Set the kind of parameter }
      illdsc_pof  :=  exp_shift;        { Set the default array index offset }
      illdsc_pcr  :=          0;
      if exp_adm <> nil then
        with exp_adm^ do
          illdsc_psz := idedim_stp*idedim_siz
      else
        illdsc_psz := 1;
      { Set as unsupported type when a type list is specified }
      if (sk <> []) and not (exp_kind in sk) then exp_kind := exp_valnull;
      case exp_kind of
        exp_valstr: illdsc_pad.ip := exp_ref;
        exp_telstr,
        exp_tabstr: illdsc_pad.ts := exp_aas;
        exp_valint: illdsc_pad.ip := exp_ref;
        exp_telint,
        exp_tabint: illdsc_pad.ti := exp_aai;
        exp_valflt: illdsc_pad.ip := exp_ref;
        exp_telflt,
        exp_tabflt: illdsc_pad.tf := exp_aaf;
      otherwise
        { Unsupported data parameter type }
        SRC_ERROR( mdnam, 603, e_severe )
      end;
      if bf and (sy = colon) then
      begin
        INSYMBOL;
        ILLNEW_FIELD( illdsc_pfo, bind )
      end
      else illdsc_pfo := nil
    end
  end;                                  { The error is already generated }
  ILLDATA_NEWPREF := p
end ILLDATA_NEWPREF;



[global]
procedure ILLDATA_FORMAT;
{ Use to proceed the ILL DATA base :
    Corresponding to the GDATA Specific Statement Block "illdata_spc ... end".
  This GDATA Statement must be used to define the Experimental Data Format
  to extract the Data from each ILL Numor data file.
}
const
  mdnam = 'ILLF';

type
  datkwdty = ( k_begwrk,                { To set a begin work system command }
               k_endwrk,                { To set a end work system command }
               k_uncpwrk,               { To set a uncompress system command }
               k_pathdef,               { To set the application paths }
               k_skip,                  { To perform a relative file positionning }
               k_pos,                   { To perform an absolute file positionning }
               k_search,                { To locate the specified string in data file }
               k_data,                  { To describe the data location and organisation }
               k_param,                 { To describe a data parameter and its format }
               k_while                  { To describe a while loop of data interpretation }
             );

  kwdid = record
    len: byte;
    nam: optid_name
  end;

var
  { Warning this table must be modified when the identifier size is changed }
  kwdtab: [static] array[datkwdty] of kwdid := (
  ( 10, 'begin_work     '),
  (  8, 'end_work       '),
  ( 14, 'uncompress_cmd '),
  ( 11, 'define_path    '),
  (  4, 'skip           '),
  (  8, 'position       '),
  (  6, 'search         '),
  (  4, 'data           '),
  (  5, 'param          '),
  (  5, 'while          ')
  );

  bf:                  boolean;
  dkwd:               datkwdty;
  st:               str_string;
  root_save,
  last_save,
  p, p1, p2:        illdsc_ptr;
  iflg, ii:            integer;


  function GEN_MOVE_NODE( bpos: boolean ): illdsc_ptr;
  var
    p: illdsc_ptr;

  begin
    if bpos then NEW( p, illdsc_pos )
            else NEW( p, illdsc_skp );
    with sy_sym, p^ do
    begin
      illdsc_nxt  := nil;
      if bpos then illdsc_typ := illdsc_pos
              else illdsc_typ := illdsc_skp;
      illdsc_ife  := nil;
      if sy = whensy then
      begin
        INSYMBOL;
        if sy = lparen then INSYMBOL
                       else SRC_ERROR( mdnam, 22, e_error );
        illdsc_ife := NEW_MACRO_EXPR;           { Get the condition expression }
        if sy = rparen then INSYMBOL
                       else SRC_ERROR( mdnam, 23, e_error )
      end;
      illdsc_spf  :=  [];
      ILLNEW_FREF( illdsc_lng, illdsc_spf, [illref_flgln], 1 );
      if sy = comma then
      begin
        INSYMBOL;
        ILLNEW_FREF( illdsc_col, illdsc_spf, [illref_flgbc], 1 )
      end
    end;
    GEN_MOVE_NODE := p
  end GEN_MOVE_NODE;



begin { ILLDATA_FORMAT }
  illdat_cmptyp  := '.Z,.gz';

  with src_control^ do src_insnb := src_insnb + 1;
  with sy_sym do
  loop
    repeat
      INSYMBOL;                                         { Gobble up keyword or separator }
    until sy <> semicolon;
  exit if (sy = peofsy) or (sy = eofsy) or (sy = endsy);
    if (sy = identsy) or (sy = whilesy) then
    begin
      bf := false;
      with sy_ident do
      for pkwd := datkwdty"first to datkwdty"last do
        with kwdtab[pkwd] do
        begin
          dkwd := pkwd;
          bf := (STR_MATCH( sy_ident, 0, nam, len ) = 0);
      exit if bf
        end;

      if bf then
      begin
        if sy <> whilesy then INSYMBOL;                 { Gobble up the option name except for while }
        case dkwd of

          k_begwrk:  GET_STREXPR( illdat_iniwork );     { Define a command sequence to execute at the begin of JOB }

          k_endwrk:  GET_STREXPR( illdat_endwork );     { Define a command sequence to execute on end of the JOB }

          k_uncpwrk:                                    { Define The uncompress command with optional scratch Path (for uncompressed numor) }
            begin
              GET_STREXPR( illdat_uncompr0 );
              ii := INDEX( illdat_uncompr0, '$' );      { The "$" Sign is a insertion point }
              illdat_uncompr1.length := 0;
              if ii > 0 then                            { When "$" is present ... }
              begin                                     { ... we cut the command in two part }
                illdat_uncompr1 := SUBSTR( illdat_uncompr0, ii + 1 );
                illdat_uncompr0[ii] := ' ';
                illdat_uncompr0.length := ii
              end else illdat_uncompr0 := illdat_uncompr0||' ';
              { The Command will be edited as "<first_part> <compressed_file> <second_part> <uncompressed_file>" }
              if sy = comma then
              begin
                INSYMBOL;
                GET_STREXPR( illdat_tmp );              { Get the location of temporary uncompressed files }
                if sy = comma then
                begin
                  INSYMBOL;
                  GET_STREXPR( illdat_cmptyp )          { Get the Compressed file file-type (if not the standard ".Z,.gz") }
                end
              end
            end;

          k_pathdef:                                    { Set the NUMOR Data Path }
            begin
              GET_STREXPR( illdat_dir );
              illdat_rdir := illdat_dir;
              if sy = comma then
              begin
                INSYMBOL;
                illdat_ngrp := GET_INTEXPR( 10000 );
                if sy = comma then INSYMBOL
                              else SRC_ERROR( mdnam, 29, e_error );
                illdat_fgrp := GET_INTEXPR( 1 )
              end;
              ii := INDEX( illdat_rdir, '/', -2 );
              if (ii > 0) and (illdat_ngrp > 1) then illdat_rdir.length := ii;
              illdat_rindex := illdat_rdir||illdat_indexf
            end;

          k_skip, k_pos:
            begin                                       { Perform a displacement in the NUMOR stream }
              p :=  GEN_MOVE_NODE( dkwd = k_pos );
              if illdsc_root = nil then illdsc_root := p
                                   else illdsc_last^.illdsc_nxt := p;
              illdsc_last :=   p;
            end;

          k_search:                                     { Search a string in the NUMOR stream }
            begin
              NEW( p, illdsc_sea );
              with p^ do
              begin
                illdsc_nxt := nil;
                if illdsc_root = nil then illdsc_root := p
                                     else illdsc_last^.illdsc_nxt := p;
                illdsc_last := p;
                illdsc_typ := illdsc_sea;
                GET_STREXPR( st );
                if st.length > 0 then
                begin
                  NEW( illdsc_sst, st.length );
                  illdsc_sst^ := st
                end
                else illdsc_sst := nil;
                illdsc_sef := [];
                if sy = comma then
                begin
                  INSYMBOL;
                  ILLNEW_FREF( illdsc_occ, illdsc_sef, [illref_flgln], -1 )
                end
                else illdsc_occ.illdsc_rfc := -1
              end
            end;

          k_param:                                      { Set the parameter setting structure }
            loop                                        { Loop on all specified parameter reference }
              p := ILLDATA_NEWPREF( true, [], true );   { Get the reference of gdata related object }
              if p <> nil then
              begin                                     { Queue it }
                if illdsc_root = nil then illdsc_root := p
                                     else illdsc_last^.illdsc_nxt := p;
                illdsc_last := p;
              end;
            exit if sy <> comma;                        { Exit of loop when finish }
              INSYMBOL
            end;

          k_data:                                       { Set the data organization structure }
            begin                                       { Create the Data DIR }
              NEW( p, illdsc_dat );
              with p^ do
              begin
                illdsc_nxt  := nil;
                illdsc_typ  := illdsc_dat;
                if illdsc_root = nil then illdsc_root := p
                                     else illdsc_last^.illdsc_nxt := p;
                illdsc_last :=   p;
                illdsc_hdd  := nil;                     { Assume no built-in coordinates }
                illdsc_hdm  := nil;                     { Assume no mesurement array to fill }
                illdsc_ndi  :=   0;                     { Init the built-in and mesurement counts }
                illdsc_nme  :=   0
              end;
              { Create the list of coordinates and sentinel }
              p1   := nil;
              if sy = lparen then                       { When some built-in coordinates must be generated, ... }
              begin
                INSYMBOL;
                loop                                    { ... we loop on all built-in coordinates definitions. }
                  NEW( p2, illdsc_ddd );                        { For Each of them, Create a Data Dimension Definition Record }
                  if p1 = nil then p^.illdsc_hdd := p2          { Link it in the Data Block list }
                              else p1^.illdsc_nxt := p2;
                  p1 := p2;
                  with p^ do illdsc_ndi := illdsc_ndi + 1;      { Update the built-in coordinate count }
                  with p2^ do
                  begin
                    illdsc_nxt := nil;
                    illdsc_typ := illdsc_ddd; 
                    { Get a single number, si       ngle number as array element or a (float or integer) array parameter only
                      - The number/array to fill reference }
                    illdsc_ddn := ILLDATA_NEWPREF( false, [exp_valint, exp_valflt,
                                                           exp_telint, exp_telflt,
                                                           exp_tabint, exp_tabflt] );
                    if sy <> becomes then                       { Get the kind of built-in filling }
                    begin
                      SRC_ERROR( mdnam, 32, e_error );
                      illdsc_sta := nil
                    end
                    else illdsc_sta := NEW_MACRO_EXPR;          { Get the start value macro definition }
                    illdsc_ddi := 0.0;
                    illdsc_stp := nil;                          { Assume the constant value mode }
                    illdsc_end := nil;
                    if sy = twodot then                         { For a definition loop }
                    begin
                      illdsc_end := NEW_MACRO_EXPR;             { Get the end value macro definition }
                      if (sy = identsy) and (sy_ident = 'by') then
                      begin
                        illdsc_stp := NEW_MACRO_EXPR;           { Get the step value macro expression }
                        illdsc_dds := 1.0;
                      end
                      else illdsc_dds := 1.0                    { Step by default }
                    end;
                    illdsc_ddl := 0                             {  }
                  end;
                exit if sy <> comma;
                  INSYMBOL
                end;
                if sy = rparen then INSYMBOL
                               else SRC_ERROR( mdnam, 24, e_error )
              end;
              { Now get the measure(s) specification(s) }
              p1 := nil;
              loop                                              { Loop on all measurement specifications }
                NEW( p2, illdsc_dmd );                          { Create a mesurement descriptor and ... }
                if p1 = nil then p^.illdsc_hdm := p2            { ... link it. }
                            else p1^.illdsc_nxt := p2;
                p1 := p2;
                { Update the count of values and sigmas }
                with p^ do illdsc_nme := illdsc_nme + 1;        { Update the measurment count }
                with p2^ do
                begin                                           { Initialize the measurement descriptor }
                  illdsc_nxt := nil;
                  illdsc_typ := illdsc_dmd;
                  illdsc_dmn := nil;
                  illdsc_dme := nil;
                  illdsc_csk := nil;
                  illdsc_dmf := nil;
                  illdsc_dms := nil;
                  { Get a float or integer array parameter only - The measure array to fill reference }
                  illdsc_dmn := ILLDATA_NEWPREF( false, [exp_tabint,exp_tabflt] );
                  if sy = colon then                            { When a sigma reference is used }
                  begin
                    INSYMBOL;
                    { Get a float or integer array parameter only - The sigma array to fill reference }
                    illdsc_dme := ILLDATA_NEWPREF( false, [exp_tabint,exp_tabflt] );
                    with p^ do illdsc_ndi := illdsc_ndi + 1
                  end;
                  if (sy = identsy) and (sy_ident = 'skip') then
                  begin
                    INSYMBOL;                                   { Gobble up the skip key word }
                    illdsc_csk :=  GEN_MOVE_NODE( false )       { Set the (condition) skip when required }
                  end;
                  if sy = lparen then                           { When some Numor Field are specified }
                  begin
                    INSYMBOL;
                    ILLNEW_FIELD( illdsc_dmf );
                    if sy = colon then                          { Can be also specified for the related sigma }
                    begin
                      INSYMBOL;
                      ILLNEW_FIELD( illdsc_dms )
                    end;
                    if sy = rparen then INSYMBOL
                                   else SRC_ERROR( mdnam, 24, e_error )
                  end
                end;
              exit if sy <> comma;                      { Exit of data specification when required }
                INSYMBOL
              end
            end;

          k_while:                                      { Set a while interpretation loop }
            begin
              NEW( p, illdsc_whl );                             { Create the while node }
              with p^ do
              begin
                illdsc_nxt := nil;                              { Initialize it with no next node }
                if illdsc_root = nil then illdsc_root := p      { Link it in the data interretration list }
                                     else illdsc_last^.illdsc_nxt := p;
                illdsc_last := p;
                illdsc_typ  := illdsc_whl;                      { Set the loop type node field }
                illdsc_scd  := NEW_MACRO_EXPR;                  { Get the expression as a macro expression }
                if sy <> dosy then SRC_ERROR( mdnam, 120, e_error )
                             { else INSYMBOL};
                root_save   := illdsc_root;                     { Save the directive list header }
                last_save   := illdsc_last;
                illdsc_root := nil;
                ILLDATA_FORMAT;                                 { Perform a recursive call for the directive sub_list }
                illdsc_dal  := illdsc_root;                     { Set the generated sub-list in the while directive node ... }
                illdsc_root := root_save;                       { and restore the directive list header }
                illdsc_last := last_save
              end
            end;

        otherwise
        end { case dkwd of }
      end { if bf then }
      else SRC_ERROR( mdnam, 601, e_severe )
    end { if sy = identsy }
    else
      if sy = sequencesy then
      begin                                             { Insertion of a GDATA Macro Statement Sequence }
        NEW( p, illdsc_seq );
        with p^ do
        begin
          illdsc_nxt := nil;
          if illdsc_root = nil then illdsc_root := p
                               else illdsc_last^.illdsc_nxt := p;
          illdsc_last :=          p;
          illdsc_typ  := illdsc_seq;
          illdsc_msq  := NEW_MACRO_LIST( endsy );
          INSYMBOL                                      { To gobble up the end of sequence }
        end
      end
      else SRC_ERROR( mdnam, 602, e_severe );
  exit if (sy = peofsy) or (sy = eofsy) or (sy = endsy);
    if sy <> semicolon then SRC_ERROR( mdnam, 21, e_error )
  end;
  with src_control^ do src_insnb := src_insnb - 1;
  if sy_sym.sy = endsy then INSYMBOL
end ILLDATA_FORMAT;






{**************************************************************************}
{****    I L L   D A T A   A C Q U I S I T I O N     R O U T I N E S   ****}
{**************************************************************************}



function  GET_NUMOR_DATA_STREAM( in_var fnam: string; bdelete: boolean := false ): integer;
const
  mdnam = 'GNDS';

var
  open_flags: flags_file := [read_file,error_file,case_ena_file];
  c_line:     str_string;
  p1, p2:     illdat_ptr;
  fnumor:           text;
  linenbr:       integer;

begin
  if bdelete then open_flags := open_flags + [del_file];
  OPEN( fnumor, fnam, open_flags );

  if iostatus = 0 then
  begin                                 { Open is a success }
    if illdat_hde <> nil then WRITELN( ' WARNING:  Un-Initialised Line Queue.' );
    p1 := nil;
    linenbr := 0;
    while not EOF( fnumor ) do
    begin
      READLN( fnumor, c_line );         { Read one Numor file line }
      linenbr := linenbr + 1;           { Establish a line count for macro process debugging }
      if c_line.length > 0 then
      begin
        if illdat_free = nil then NEW( p2 )
        else begin  p2 := illdat_free; illdat_free := p2^.illdat_nxt  end;
        if p1 = nil then illdat_hde := p2
                    else p1^.illdat_nxt := p2;
        with p2^ do
        begin
          illdat_nxt  :=           nil;
          illdat_prv  :=            p1;
          illdat_nbr  :=       linenbr;
          illdat_line :=        c_line
        end;
        p1 := p2
      end
    end;
    CLOSE( fnumor );                    { End on a success DATA STREAM Loading }
    GET_NUMOR_DATA_STREAM :=  0
  end
  else
  begin                                 { Failure on OPEN }
    SRC_ERROR( mdnam, 622, e_severe );
    GET_NUMOR_DATA_STREAM := -1
  end
end GET_NUMOR_DATA_STREAM;



function  GET_NUMOR_STREAM( num: integer ): integer;
{ To Load an ILL Numor Data Stream from the integer value of NUMOR }
const
  mdnam = 'GNUM';

var
  idir, ips, itb:              integer;
  scom, sfil, snum, styp:       string;
  fnd:                         boolean;

begin { To get the numor string }
  { Build the ILL DATA location specification }
  WRITEV( snum, num:-6 );                               { Get the file name as a string }
  if illdat_ngrp > 1 then
  begin                                                 { Get data with numor group subdirectory }
    idir := num div illdat_ngrp;                        { Deduce the NUMOR File Specification in the ILL DATA BASE }
    WRITEV( sfil, illdat_dir:illdat_dir.length - 1, idir:-illdat_fgrp,
                  illdat_dir[illdat_dir.length], snum )
  end
  else WRITEV( sfil, illdat_dir, num:-6 );              { Get data without numor group subdirectory - Special case }

  scom := SUBSTR( sfil, 1, sfil.length - 6 );
  SET_STRING_PARM( gd_sub_dir^.ide_str, scom );         { Set the Data Base Directory Reference }

  ierr  := 0;                                           { Assume no Error until shown otherwise }
  styp.length := 0;

  fnd :=  FILE_ACCESS_CHECK( sfil, 4 { Read Access }, [case_ena_file] );
  { When the file is found, we use the direct access from the data base }

  if not fnd then
  begin                                                 { Compressed NUMOR ? }
    ips := 0;
    fnd := false;
    repeat                                              { Loop on all compressed file types }
      ips := ips + 1;                                   { Skip any previous comma } 
      itb := ips;
      ips := INDEX( illdat_cmptyp, ',', ips );
      if ips = 0 then ips := illdat_cmptyp.length + 1;
      styp := SUBSTR( illdat_cmptyp, itb, ips - itb );  { Get one file type }
      sfil := sfil||styp;
      { See if the file is existing }
      fnd := FILE_ACCESS_CHECK( sfil, 4 { Read Access }, [case_ena_file] );
      { If the file does not exist, suppress the file type ... }
      if not fnd then sfil.length := sfil.length - styp.length
    until fnd or (ips > illdat_cmptyp.length);

    if fnd then
    begin                                               { OK - It is a Compressed NUMOR }
      illdat_numtyp := styp;                            { Keep a copy of numor type }
      WRITEV( scom, illdat_uncompr0, sfil, ' ', illdat_uncompr1, illdat_tmp, num:-6 );
      if not SYS_SPAWN( scom ) then                     { Perform the Uncompress task for the numor }
      begin { UNCOMPRESS ERROR }
        SRC_ERROR( mdnam, 621, e_fatal ); return -1
      end;
      WRITEV( sfil, illdat_tmp, num:-6 )
    end
  end;

  { Get the numor data stream wehen it is founded }
  if fnd then GET_NUMOR_STREAM := GET_NUMOR_DATA_STREAM( sfil, styp.length > 0 )
         else GET_NUMOR_STREAM := -1
end GET_NUMOR_STREAM;




function  GET_NUMOR_STREAM2( in_var n_fspc, tmp_fspc: string; bcompress: boolean := false ): integer;
{ Get One Numor Stream where :
    n_fspc is the complete NUMOR File specification.
}
const
  mdnam = 'GNU2';

var
  scom: str_string;

begin
  if bcompress then
  begin
    WRITEV( scom, illdat_uncompr0, n_fspc, ' ', illdat_uncompr1, tmp_fspc );
    if not SYS_SPAWN( scom ) then                       { Perform the Uncompress task for the numor }
    begin { UNCOMPRESS ERROR }
      SRC_ERROR( mdnam, 621, e_fatal ); return 1
    end
    else { Use the temporary NUMOR FILE with deletion on close time }
      GET_NUMOR_STREAM2 := GET_NUMOR_DATA_STREAM( tmp_fspc, true )
  end
  else { When no compressed file, Use directly the DATA BASE NUMOR FILE }
    GET_NUMOR_STREAM2 := GET_NUMOR_DATA_STREAM( n_fspc, false )
end GET_NUMOR_STREAM2;



function GET_FRMPARM( in_var ref: illref_rec; flgs, flg: illref_flags ): integer;
{ Return the value of ref reference (illref_rec following the not vanish value of set intersection flgs*flg }
begin
  if flgs*flg <> [] then GET_FRMPARM := GET_INT_VALUE( ref.illdsc_rfm )         { Macro Definition of value }
                    else GET_FRMPARM := ref.illdsc_rfc                          { Litteral Constant }
end GET_FRMPARM;



procedure FIELD_LOCATE( fld: illdsc_ptr; var ipos, f: integer; bl: boolean );
{ To Set the Current Read Pointer to a defined Position in the NUMOR Memory STREAM }
{ When bl is set to true, any end of line indice the continuation on the next line }
{ On return, illdat_curr^ is the line where is the read field,
             ipos         is the index of first character to read, and
             f            is the filed to use. }
const
  mdnam = 'FLOC';

var
  isl, ip, ivl, ifl: integer;

begin
  if fld <> nil then
  with fld^ do
  begin
    isl := GET_FRMPARM( illdsc_fln, illdsc_frf, [illref_flgln] );       { Get the relative line number (in line) }
    ivl := GET_FRMPARM( illdsc_fbc, illdsc_frf, [illref_flgbc] );       { Get the relative field position (in char) }
    ifl := GET_FRMPARM( illdsc_fsz, illdsc_frf, [illref_flgsz] );       { Get the field size (in char) }

    if bl and (illdat_crch < 0) then                    { When bl is set, we pass to next line after any EOLN }
    begin  isl := isl + 1; illdat_crch := 0  end;

    if ivl > 0 then ip := ivl - 1                       { Positive ivl is an absolute position inside the line, and ... }
               else ip := illdat_crch - ivl;            { ... negative value indicate a relative forward positionning }
    if ip >= illdat_linesz then                         { When the line size is reached, ... }
    begin                                               { ... we continue the position scan to the next line }
      isl := isl + ip div illdat_linesz;
      ip  := ip mod illdat_linesz
    end;
    { Get the line  to use }
    if isl > 0 then                                     { Get the line in forward direction }
      while (isl > 0) and (illdat_curr <> nil) do
      begin
        illdat_curr := illdat_curr^.illdat_nxt;
        isl := isl - 1
      end
    else                                                { Get the line in the backward direction }
      while (isl < 0) and (illdat_curr <> nil) do
      begin
        illdat_curr := illdat_curr^.illdat_prv;
        isl := isl + 1
      end;

    if illdat_curr <> nil then
    begin                                               { When the line is found }
      ipos := ip + 1;                                   { Get the position index in the line }
      f := illdat_linesz - ip;                          { Check for line overflow with the field size }
      if (ifl > 0) and (ifl < f) then f := ifl;
      illdat_crch := ip + f;                            { Update the current character position after the read field ... }
      if (illdat_crch >= illdat_linesz) and bl then illdat_crch := -1   { ... and set the EOLN value when required }
    end
    else
    begin
      SRC_ERROR( mdnam, 625, e_severe );                { Bad line specification }
      ipos := 1;
      f    := 0;
      illdat_curr := illdat_hde
    end
  end
end FIELD_LOCATE;



function EXTRACT_FLT( pf: illdsc_ptr ): gdreal;
{ To Extract a floating value from the numor stream.
  pf^ is the illdsc field descriptor to use.
}
const
  mdnam = 'EFLT';

var
  icp, isv: integer;
  res:      gdreal;
  stc: string( illdat_linesz );

begin
  if illdat_curr <> nil then
    if pf <> nil then
    begin { Read with a specified field descriptor }
      FIELD_LOCATE( pf, icp, isv, true );                               { Locate the filed to read, }
      stc := SUBSTR( illdat_curr^.illdat_line, icp, isv );              { ... extract it ... }
      READV( stc, res )                                                 { ... and read it. }
    end
    else
    begin { Read the floatting number in free format }
      READV( illdat_curr^.illdat_line:illdat_crch, res );               { Perform a direct read ... }
      if (illdat_crch < 0) or (illdat_crch >= illdat_linesz) then       { ... and update the current position }
      begin
        illdat_crch := 0; illdat_curr := illdat_curr^.illdat_nxt
      end
   end
  else
  begin
    SRC_ERROR( mdnam, 629, e_severe );
    illdat_curr := illdat_hde;
    res := 0.0
  end;
  EXTRACT_FLT := res
end EXTRACT_FLT;



procedure DEPOSIT_IPARM( p: illdsc_ptr; v: integer );
{ Deposite a Parameter integer Value (v) in a cpsh integer reference,
  p^.illdsc_pad^.ip^.ide_int for single integer,
  p^.illdsc_pad^.ti^.ide_itb[illdsc_pof] for integer array element. }
begin
  if p <> nil then
  with p^, illdsc_pad do
    if illdsc_pkn = exp_valint then ip^.ide_int := v
                               else ti^.ide_itb[illdsc_pof] := v
end DEPOSIT_IPARM;



procedure DEPOSIT_FPARM( p: illdsc_ptr; v: gdreal );
{ Deposite a Parameter floating Value in a cpsh floating reference,
  p^.illdsc_pad^.ip^.ide_flt for single integer,
  p^.illdsc_pad^.tf^.ide_ftb[illdsc_pof] for integer array element. }
begin
  if p <> nil then
  with p^, illdsc_pad do
    if illdsc_pkn = exp_valflt then ip^.ide_flt := v
                               else tf^.ide_ftb[illdsc_pof] := v
end DEPOSIT_FPARM;




procedure DEPOSIT_TABVAL( p: illdsc_ptr; v: gdreal );
{ Deposite an integer or floating Value in a cpsh array element reference,
  p^.illdsc_pad^.ti^.ide_itb[illdsc_pof+illdsc_pcr] for integer array element,
  p^.illdsc_pad^.tf^.ide_ftb[illdsc_pof+illdsc_pcr] for floating array element.
  The count illdsc_pcr is incremented. }
var
  i: integer;

begin
  if p <> nil then
  with p^ do
  if illdsc_pcr < illdsc_psz then                       { Check for full array }
  begin
    i := illdsc_pcr + illdsc_pof;
    case illdsc_pkn of
      exp_valint: illdsc_pad.ip^.ide_int    := ROUND( v );

      exp_valflt: illdsc_pad.ip^.ide_flt    :=          v;

      exp_telint: illdsc_pad.ti^.ide_itb[illdsc_pof] := ROUND( v );     { Integer Value in a GDATA integer array element }

      exp_telflt: illdsc_pad.tf^.ide_ftb[illdsc_pof] := v;              { Floating Value in a GDATA floating array element }

      exp_tabint: illdsc_pad.ti^.ide_itb[i] := ROUND( v );

      exp_tabflt: illdsc_pad.tf^.ide_ftb[i] :=          v

    otherwise
    end;
    illdsc_pcr := illdsc_pcr + 1
  end
  else
    bfull := true
end DEPOSIT_TABVAL;




function GET_TABVAL( p: illdsc_ptr; i: integer ): gdreal;
{ Get an integer or floating Value from a integer or floating cpsh array element reference }
begin
  if p <> nil then
  with p^ do
  if illdsc_pcr < illdsc_psz then
  begin
    case illdsc_pkn of
      exp_tabint: GET_TABVAL := illdsc_pad.ti^.ide_itb[i+illdsc_pof];
      exp_tabflt: GET_TABVAL := illdsc_pad.tf^.ide_ftb[i+illdsc_pof];
    otherwise
    end;
    i := i + 1
  end
end GET_TABVAL;




function ILLDAT_INCREMENT( p: illdsc_ptr ): boolean;
{ Increment Loop Routine }
var
  binc: boolean;

begin
  if p <> nil then
  with p^ do
  begin
    if illdsc_nxt <> nil then binc := ILLDAT_INCREMENT( illdsc_nxt )
                         else binc := true;
    if binc then
    begin
      illdsc_ddv := illdsc_ddv + 1;
      if illdsc_ddv > illdsc_ddl then illdsc_ddv := 0
                                 else binc := false
    end
  end;
  ILLDAT_INCREMENT := binc
end ILLDAT_INCREMENT;



procedure EXTRACT_NUMOR( pd: illdsc_ptr; var ierr: integer );
{ Procedure to Read a ILL NUMOR FILE and Extract all required data from the Memory stream }
const
  mdnam = 'ILLE';

var
  p:                    illdsc_ptr;
  brv, bok, bdo, bar:      boolean;
  ch:                         char;
  stc:               string( 255 );
  npt, isv, icp, iv:       integer;
  rv:                       gdreal;

label STOP_EXTRACT;


function  ERR_HANDLER( nerr: cc__int ): cc__int;
var
  isucc: integer;

begin
  ierr := nerr;
  isucc := 0;
  case nerr of
    51: { Numeric Read Error } isucc := 1;
  otherwise
  end;
  if isucc > 0 then goto STOP_EXTRACT;
  ERR_HANDLER := isucc
end ERR_HANDLER;




  procedure EXTRACT_MOVE( p: illdsc_ptr; babs: boolean );
  var
    bdo: boolean;

  begin
    with p^ do
    begin
      if illdsc_ife <> nil then bdo := GET_FLT_VALUE( illdsc_ife ) >= 0.5
                           else bdo := true;
      if bdo then
      begin
        isv := GET_FRMPARM( illdsc_lng, illdsc_spf, [illref_flgln] );
        if babs then
        begin
          illdat_curr := illdat_hde;
          isv := ABS( isv );
          if isv > 0 then isv := isv - 1; { First line number is 1 (not 0) }
          illdat_crch := 0
        end;
        { Goto specified line }
        if isv > 0 then
          while (isv > 0) and (illdat_curr <> nil) do
          begin
            illdat_curr := illdat_curr^.illdat_nxt; isv := isv - 1
          end
        else
          while (isv < 0) and (illdat_curr <> nil) do
          begin
            illdat_curr := illdat_curr^.illdat_prv; isv := isv + 1
          end;
        if illdat_curr = nil then
        begin
          SRC_ERROR( mdnam, 626, e_severe );
          illdat_curr := illdat_hde
        end
        else
        begin
          { Goto specified character }
          illdat_crch := illdat_crch + GET_FRMPARM( illdsc_col, illdsc_spf, [illref_flgbc] );
          if illdat_crch < 0 then illdat_crch := 0
          else
            if illdat_crch > illdat_linesz then illdat_crch := illdat_linesz
        end
      end
    end
  end EXTRACT_MOVE;



begin { EXTRACT_NUMOR }
  ESTABLISH( ERR_HANDLER );
  while pd <> nil do                    { Loop on all ILL DATA ITEMs }
  with pd^ do
  begin
    case illdsc_typ of
      illdsc_pos, { * Absolute Move in the Numor Stream Directive }
      illdsc_skp: { * Relative Move in the Numor Stream Directive }
        EXTRACT_MOVE( pd, illdsc_typ = illdsc_pos );

      illdsc_sea: { * Search a string directive }
        if illdsc_sst <> nil then
        begin
          if illdat_curr = nil then illdat_curr := illdat_hde;
          isv := GET_FRMPARM( illdsc_occ, illdsc_sef, [illref_flgln] );
          if isv > 0 then
          begin                         { Line by line search }
            loop
              illdat_curr := illdat_curr^.illdat_nxt;
            exit if illdat_curr = nil;
              icp := INDEX( illdat_curr^.illdat_line, illdsc_sst^ );
              if icp <> 0 then isv := isv - 1;
            exit if isv = 0
            end;
            illdat_crch := 0
          end
          else
          begin                         { Character by character search }
            icp := illdat_crch;
            repeat
              icp := icp + 1;
              icp := INDEX( illdat_curr^.illdat_line, illdsc_sst^, 1, icp );
              if icp = 0 then illdat_curr := illdat_curr^.illdat_nxt
                         else isv := isv + 1;
            until (illdat_curr = nil) or (isv = 0);
            illdat_crch := icp
          end;
          if illdat_curr = nil then illdat_curr := illdat_hde
        end;

      illdsc_par: { * Set parameter value Directive }
        begin
          FIELD_LOCATE( illdsc_pfo, icp, isv, false );  { Locate the read field, extract it }
          stc := SUBSTR( illdat_curr^.illdat_line, icp, isv );
          if debug_dat then WRITELN( 'Read Param = "', stc, '".' );
          with illdsc_pad do
            case illdsc_pkn of
              exp_valstr:                               { String Value in a GDATA string }
                SET_STRING_PARM( ip^.ide_str, stc );
              exp_valint:                               { Integer Value in a single GDATA integer number }
                begin
                  READV( stc, iv ); DEPOSIT_IPARM( pd, iv )
                end;
              exp_valflt:                               { Floating Value in a single GDATA float number }
                begin
                  READV( stc, rv ); DEPOSIT_FPARM( pd, rv )
                end;
              exp_telstr:                               { String Value in a GDATA string array element }
                SET_STRING_PARM( ts^.ide_stb[illdsc_pof], stc );
              exp_telint:                               { Integer Value in a GDATA integer array element }
                READV( stc, ti^.ide_itb[illdsc_pof] );
              exp_telflt:                               { Floating Value in a GDATA floating array element }
                READV( stc, tf^.ide_ftb[illdsc_pof] );
            otherwise
              { Not done for array }
            end
        end;

      illdsc_dat: { * First dimension Loop to compute the number of points }
        begin
          bfull :=  false;
          bok   :=   true;
          npt   :=      1;
          p := illdsc_hdd;                              { Get the head pointer of coordinates }
          bar   := p = nil;                             { Each array must be filledwhen no abscisse to generate }
          while p <> nil do
            with p^ do
            begin                                       { Loop to initialize the coordinates }
              illdsc_ddi :=  GET_FLT_VALUE( illdsc_sta );       { Get initial value of the current coordinate }
              if illdsc_end <> nil then                         { * For array filling with progressive value }
              begin                                             { For step by step array }
                rv := GET_FLT_VALUE( illdsc_end );              { Get final value of this coordinate }
                if illdsc_stp = nil then illdsc_dds := 1.0      { Set the coordinate step value }
                                    else illdsc_dds := GET_FLT_VALUE( illdsc_stp );
                illdsc_ddl := ROUND( (rv - illdsc_ddi)/illdsc_dds ) + 1;        { Set the dimension size }
              end
              else
              begin                                             { * For array filling with a constant value }
                illdsc_dds := 0.0;                              { Step value is set to zero  ... }
                illdsc_ddl := 1                                 { ... with only one value. }
              end;
              illdsc_ddv := 0;                                  { Init the step count }

              if illdsc_ddn <> nil then illdsc_ddn^.illdsc_pcr := 0     { Init the array index }
                                   else bok := false;
              npt := npt*illdsc_ddl;
              p := illdsc_nxt
            end;

          { Loop to initialize the indexies of each measure }
          p := illdsc_hdm;                                      { Get the head pointer of coordinates }
          while p <> nil do
            with p^ do
            begin                                               { Loop on each measure array }
              if illdsc_dmn <> nil then
              begin
                if bar then
                begin
                  npt := illdsc_dmn^.illdsc_psz;                { Get the array size (in elements) }
                  bar := false                                  { Only for the first array }
                end;
                illdsc_dmn^.illdsc_pcr := 0                     { Init the measure array index }
              end else bok := false;
              if illdsc_dme <> nil then                         { For the sigma array }
                illdsc_dme^.illdsc_pcr := 0;                    { Init the sigma array index }
              p := illdsc_nxt
            end;

          if bok then
          for ipt := 1 to npt do                                { Loop to scan each mesured point }
          begin
            if debug_dat then
            begin
              WRITE( ' * Data index = ', ipt:6 );
              if illdat_curr <> nil then with illdat_curr^ do WRITE( ' line # ', illdat_nbr:4 )
                                    else WRITELN
            end;
            p := illdsc_hdd;                                    { Get the head pointer of coordinates }
            while p <> nil do                                   { Loop on each array }
              with p^ do
              begin                                             { Loop on the coordinates }
                DEPOSIT_TABVAL( illdsc_ddn, illdsc_ddi + illdsc_ddv*illdsc_dds );       { Deposite the value }
                p := illdsc_nxt
              end;
            p := illdsc_hdm;                                    { Get the head pointer of coordinates }
            while p <> nil do                                   { Loop on the each mesure array }
              with p^ do
              begin
                if illdsc_csk <> nil then EXTRACT_MOVE( illdsc_csk, false );    { Perform a (conditional?) move if required }
                rv := EXTRACT_FLT( illdsc_dmf );
                if debug_dat then WRITE( ' Read Value = ', rv );
                DEPOSIT_TABVAL( illdsc_dmn, rv );
                if illdsc_dme <> nil then
                begin                                           { When a sigma is specified }
                  if illdsc_dms <> nil then rv := EXTRACT_FLT( illdsc_dms )     { Read with a specified field }
                                       else rv := SQRT( rv );                   { by default take the square root of count }
                  if debug_dat then WRITE( ' Sigma Value = ', rv );
                  DEPOSIT_TABVAL( illdsc_dme, rv )
                end;
                p := illdsc_nxt
              end;
            if debug_dat then WRITELN;
          exit if bfull;
            brv := ILLDAT_INCREMENT( illdsc_hdd )               { Increment Loop }
          end;
          if bfull then SRC_ERROR( mdnam, 628, e_error )
        end;

      illdsc_whl:
        while GET_INT_VALUE( illdsc_scd ) > 0 do
          EXTRACT_NUMOR( illdsc_dal, ierr );                    { Execute the directive sequence }

      illdsc_seq: EXECUTE_MACRO_CODE( illdsc_msq, endsy );

    otherwise
      SRC_ERROR( mdnam, 902, e_fatal )
    end;
    pd := illdsc_nxt
  end;
STOP_EXTRACT:
  REVERT
end EXTRACT_NUMOR;



[global]
procedure GET_ILLDATA( numor: integer; var ierr: integer );
{ Procedure to get one data numor from the ILL Data base,
  The Data are loaded in the identifier specified by a previous ill_dataspc block statement.
}
const
  mdnam = 'GILL';

var
  p:  illdat_ptr;
  i1:    integer;
  svsy:  sym_rec;

begin { GET_ILLDATA }
  svsy := sy_sym;
  gd_rq_numor^.ide_int := numor;                { Set the requested numor  in cpsh symbol }
  ierr := GET_NUMOR_STREAM( numor );            { Get the numor stream }
  if ierr = 0 then                              { When the Numor Flie Load is OK }
  begin
    if illdat_hde <> nil then
    begin { The numor is Loaded }
      illdat_curr := illdat_hde;                { Set the Stream scan to begin of Numor }
      illdat_crch := 0;
      EXTRACT_NUMOR( illdsc_root, ierr );       { Extract the relevant data information ... }
      illdat_free := illdat_hde;                { ... and free the input stream line for next }
      illdat_hde  := nil
    end
  end;
  sy_sym := svsy
end GET_ILLDATA;



[global]
function  GET_ILLDATA2( in_var n_fspc, t_fspc: string; bcompress: boolean := false ): integer;
{ Procedure to get one data numor from the ILL Data base,
  The Data are loaded in the identifier specified by a previous ill_dataspc block statement.
}
const
  mdnam = 'GILL';

var
  p:  illdat_ptr;
  i1:    integer;
  svsy:  sym_rec;
  ierr:  integer;

begin
  svsy := sy_sym;                               { Save the INSYMBOL Context -(du to possible Macro Sequence execution) }
  gd_rq_numor^.ide_int := 0;                    { Set the requested numor in cpsh symbol }
  ierr := GET_NUMOR_STREAM2( n_fspc, t_fspc, bcompress );       { Get the Numor Data Stream }
  if ierr = 0 then                              { When the Numor Flie Load is OK }
  begin
    if illdat_hde <> nil then
    begin { The numor is Loaded }
      illdat_curr := illdat_hde;                { Set the Stream scan to begin of Numor }
      illdat_crch := 0;
      EXTRACT_NUMOR( illdsc_root, ierr );       { Extract the relevant data information ... }
      illdat_free := illdat_hde;                { ... and free the input stream line for next }
      illdat_hde  := nil
    end
  end;
  sy_sym := svsy ;                              { Restore the INSYMBOL Context }
  GET_ILLDATA2 := ierr
end GET_ILLDATA2;





{************************************************************************************}
{****     I L L   D A T A   B A S E   I N D E X A T I O N     R O U T I N E S    ****}
{************************************************************************************}



procedure GDBSTR_GET( p: ^string; var s: string );
{ Copy a string }
begin
  if p = nil then s.length := 0
             else s := p^
end GDBSTR_GET;



procedure GDB_GETSTR( var st: string; p: gdbp_ptr );
{ Get a string Value from any object - with implicite conversion when required }
begin
  with p^ do
    case gdbp_kind of
      exp_valint: WRITEV( st, gdbp_addr.ip^.ide_int );
      exp_telint: WRITEV( st, gdbp_addr.ti^.ide_itb[gdbp_offs] );
      exp_valflt: WRITEV( st, gdbp_addr.ip^.ide_int );
      exp_telflt: WRITEV( st, gdbp_addr.tf^.ide_ftb[gdbp_offs] );
      exp_valstr: GDBSTR_GET( gdbp_addr.ip^.ide_str, st );
      exp_telstr: GDBSTR_GET( gdbp_addr.ts^.ide_stb[gdbp_offs], st );
    otherwise
      st.length := 0
    end;
end GDB_GETSTR;



procedure WRITE_NUMOR_CHAR( var findex: text; in_var n_fspc: string; num, nuf: integer; bcmp: boolean );
{ Procedure to write a numor related keys. }
var
  pw:             gdbp_ptr;
  ierr, ikey, iw:  integer;
  buf:             string( gdata_keys );
  bok:             boolean;

begin
  WRITE( findex, num:-nuf, ':', ORD( bcmp ):1 );        { Write the numor string and the compressed flag }
  if bcmp then WRITEV( buf, illdat_tmp, num:-6 );
  ierr := GET_ILLDATA2( n_fspc, buf, bcmp );            { Get the NUMOR Data }
  with gdb_out^ do
  begin
    for ikey := 1 to gdb_knb do
    with gdb_keytab[ikey] do
    begin
      if not (gdbkf_notinp in keyf) then
      begin
        pw := keyp;                                     { Get the key identifier reference }
        iw := 1;
        while iw < gdat_iwrk do                         { Loop to localize the get Work Space }
        begin  pw := pw^.gdbp_next; iw := iw + 1  end;

        with pw^ do
        begin
          GDB_GETSTR( buf, pw );                        { Get the string Key value }
          WRITE( findex, ' ', buf.length:0, ' ', buf )
        end
      end
    end
  end;
  WRITELN( findex )
end WRITE_NUMOR_CHAR;




procedure MAKE_SUB_INDEX( in_var sbdir: string; var min_numor, max_numor: integer );
{ Procedure to Create/Replace the sorted numor list present in
  the sbdir directory.
}
type
  numor_table( dim: integer ) = array[1..dim] of record
                                                   num: integer;
                                                   nuf, bcp: word_integer
                                                 end;

var
  scan_str, c_fspc, n_fspc, str0:     string;
  numdir_scan:               efb_ptr  := nil;   { Scan Profile for Subdirectory }
  i_nat, nv, ma, mi, mf, ip, jp, cp: integer;
  num_tab:                      ^numor_table;
  ch:                                   char;
  bok, bcompress:                    boolean;
  findex:                               text;

begin
  scan_str := sbdir||'/*';
  numdir_scan := FSPC_OPEN( scan_str );         { Init the Directory Scan }
  if numdir_scan <> nil then
  begin
    NEW( num_tab, illdat_ngrp );                { Allocate the Numor Table }
    for ii := 1 to illdat_ngrp do num_tab^[ii].num := 0;        { and set it as empty }
    cp        := 0;
    min_numor := 0;
    max_numor := 0;
    loop                                        { Loop to scan all Instrument Subdirectory }
      FSPC_SCAN( numdir_scan, c_fspc, i_nat );
    exit if i_nat < 0;                          { End Instrument Root Scan }
      if i_nat = 1 then                         { for regular file ... }
      begin
        str0 := SUBSTR( c_fspc, INDEX( c_fspc, '/', -1 ) + 1 );
        bok  := true;
        nv   := 0;
        ip   := INDEX( str0, '.' );             { Compressed numor ? => Set bcompress flag }
        if ip = 0 then begin  bcompress := false; ip := str0.length  end
                  else begin  bcompress :=  true; ip := ip - 1  end;
        jp := 0;                                { Get the Integer Filename value as NUMOR }
        while bok and jp < ip do
        begin
          jp := jp + 1;
          ch := str0[jp];
          if (ch >= '0') and (ch <= '9') then nv := nv*10 + ORD( ch ) - ORD( '0' )
                                         else bok := false
        end;
        if bok then                             { Only for a Numor File Name }
        with num_tab^[nv mod illdat_ngrp + 1] do
        begin
          num := nv;                            { It is a name of NUMOR FILE }
          nuf := jp;                            { Keep the Field size }
          bcp := ORD( bcompress );              { Keep the Compress Flag }
          cp := cp + 1                          { Update the count of numor }
        end
      end
    end;
    FSPC_CLOSE( numdir_scan );                  { Close the Scan of Directory }

    str0 := sbdir||'/numor_index.data';         { Create the NUMOR index File }
    OPEN( findex, str0, [error_file,write_file,case_ena_file] );
    if iostatus = 0 then
    begin                                       { for each existing NUMOR }
      mi := maxint; ma := - mi; mf := 0;
      for ii := 1 to illdat_ngrp do             { Write the related entry }
        with num_tab^[ii] do
          if num > 0 then
          begin
            WRITEV( n_fspc, sbdir, '/', num:-nuf );     { Set the NUMOR filename }
            if bcp > 0 then  n_fspc := n_fspc||illdat_cmptyp;
            WRITE_NUMOR_CHAR( findex, n_fspc, num, nuf, boolean( bcp ) );
            if mi > num then mi := num;         { update the NUMOR minimaxi }
            if ma < num then ma := num;
            if mf < nuf then mf := nuf          { and the the numor figure number }
          end;
      CLOSE( findex );
      min_numor := mi;
      max_numor := ma;
      WRITELN( ' The Numor "', str0, '" mini-maxi of numor are ', mi:-mf, '..', ma:-mf,
               ' and is ', cp:0, ' entries long.' )
    end
    else
    begin
      WRITELN( ' GDATA ERROR : Cannot Create a Numor Catalog (probably a protection problem),'  );
      WRITELN( ' ************** with the file specification "', str0 )
    end
  end;
  DISPOSE( num_tab );
end MAKE_SUB_INDEX;




function  MAKE_INDEX: boolean;
{ Create the index and sub-directory index for the current instrument }
const
  subd_rec_cap = 32;
  in_minor = ORD( 'a' ) - ORD( 'A' );

type
  subd_ptr = ^subd_ide;                 { * Define the pointer of sub-directory record }

  subd_ide( len: byte ) = record        { * Define the sub-directory record }
    subd_left,                          { Define the left and right link for alphabetic sort }
    subd_right:         subd_ptr;
    subd_min_numor,                     { Define the minimum and maximum of numor index }
    subd_max_numor:      integer;
    subd_fname:    string( len )        { Define the Filename of the sub-directory }
  end;

var
  pcurr, p1, p2,
  subd_root:        subd_ptr := nil;    { Sub-directory Tree head Pointer }
  subdir_scan:      efb_ptr  := nil;    { Scan Profile for Subdirectory }
  root_index:                  text;    { Root Index File }

  i, j, i_nat, i_cp:        integer;
  str0,
  i_fspc, c_fspc, c_inst:    string;
  ch:                          char;
  bok:                      boolean;


  procedure PUT_INDEX_ENTRY( p: subd_ptr );
  { Recursive sub-directory output to create a sorted sub-directory index }
  begin
    if p <> nil then
    with p^ do
    begin
      if subd_left  <> nil then PUT_INDEX_ENTRY( subd_left  );
      WRITELN( root_index, subd_min_numor:7, ' ', subd_max_numor:7, ' ', subd_fname );
      if subd_right <> nil then PUT_INDEX_ENTRY( subd_right )
    end;
    DISPOSE( p )
  end PUT_INDEX_ENTRY;



begin { MAKE_INDEX }
  bok := false;
  i := INDEX( illdat_rdir, '/', -2 );
  c_inst := SUBSTR( illdat_rdir, i + 1, INDEX( illdat_rindex, '/', -1 ) - i - 1 );
  for ii := 1 to c_inst.length do
  begin
    ch := c_inst[ii]; if (ch >= 'A') and (ch <= 'Z') then c_inst[ii] := CHR( ORD( ch ) + in_minor )
  end;
  i_fspc := illdat_rdir||c_inst||'*';
  subdir_scan := FSPC_OPEN( i_fspc );   { Init the Directory scan }
  if subdir_scan <> nil then
  loop                                  { Loop to scan all Instrument Subdirectory }
    FSPC_SCAN( subdir_scan, c_fspc, i_nat );
  exit if i_nat < 0;                    { End Instrument Root Scan }
    if i_nat = 0 then                   { If the entry is a directory, }
    begin                               { we prepare this directory scan }
      str0 := SUBSTR( c_fspc, INDEX( c_fspc, '/', -1 ) + 1 );
      NEW( pcurr, subd_rec_cap );       { Create the subdirectory entry, }
      with pcurr^ do
      begin
        subd_left  :=  nil;
        subd_right :=  nil;
        subd_fname := str0
      end;
      p1 := subd_root;
      if p1 <> nil then
      begin                             { and append it in the subdirectory tree }
        repeat
          p2 := p1                      { Keep the memory of the last p1 };
          with p1^ do
          begin
            i := STR_MATCH( p1^.subd_fname, str0 );
            if i < 0 then p1 := subd_right
                     else p1 := subd_left
          end;
        until p1 = nil;
        if i < 0 then p2^.subd_right := pcurr
                 else p2^.subd_left  := pcurr
      end
      else subd_root := pcurr;
      with pcurr^ do
        MAKE_SUB_INDEX( c_fspc, subd_min_numor, subd_max_numor )        { Create/Update the Sub-Index }
    end
  end;
  FSPC_CLOSE( subdir_scan );

  OPEN( root_index, illdat_rindex, [error_file,write_file,case_ena_file] );
  if iostatus = 0 then
  begin                                 { Write the master directory with sorted entry in increasing order }
    PUT_INDEX_ENTRY( subd_root );
    CLOSE( root_index );
    MAKE_INDEX := true
  end
  else MAKE_INDEX := false
end MAKE_INDEX;






function  ILL_DATA_INDEX( make_index_flg: boolean ): boolean;
var
 curr_dir: string;
  bok:     boolean;

begin
  { Check the read access of the ILL Data Root Index }
  if not FILE_ACCESS_CHECK( illdat_rindex, 4, [case_ena_file] ) or make_index_flg then
  begin
    WRITELN( ' GDATA Create or Update the ILL DATA Index.' );
    { No Access to the Root Index, We can try to create it }
    ILL_DATA_INDEX := MAKE_INDEX
  end
  else ILL_DATA_INDEX := true
end ILL_DATA_INDEX;





{************************************************************************************}
{****      I L L   D A T A   B A S E   S C A N   I N D E X   R O U T I N E S     ****}
{************************************************************************************}


procedure ILLDATA_SCAN_SEARCH( var p: $wild_pointer; b_load: boolean := true );
const
  mdnam     = 'SCAN';                   { SRC_ERROR Module Identifier }
  max_fname =     14;                   { Capacity of File_Name Strings }

type
  illsca_states = ( scan_error,         { Unrecoverable Error => Stop }
                    scan_closed,        { The Scan is in Closed State }
                    scan_masindex,      { The Scan is in Master Index File Open Phase }
                    scan_subindok       { The Scan is in Sub Index File Open Phase }
                  );

  illsca_rec  = record                  { * Define the scan context record }
    illsca_nu_min,                      { Minimum required Numor }
    illsca_nu_max:         integer;     { Maximum required Numor }
    illsca_mcf,                         { Master Index File }
    illsca_scf:               text;     { Sub-Index File }
    illsca_mflt:           flt_ptr;     { Master IndexFile Filter pointer }
    illsca_ktb: array[1..gdata_nkey] of flt_ptr;        { Table of filter for each key }
    illsca_sdir:       string(255);     { Current Path of Numor Files }
    illsca_status:   illsca_states      { Scan Status }
  end;

  illsca_ptr = ^illsca_rec;             { Define the Scan Pointer }


var
  psca:                 illsca_ptr;     { Pointer of the structure }
  nm_fname,                             { Numor Filename }
  si_fname:    string( max_fname );     { Sub-index Filename }
  sc_key:     string( gdata_keys );     { Current Key String }
  c_line,                               { Current input Line from an Index File }
  sb_fspc:                  string;     { Used to Prepare the Open of Files }
  c_num, lky, ierr,                     { Current Numor, Current Key Length, Error code }
  num_min, num_max:        integer;     { Minimum and Maximum of Numor in this Sub-Index File }
  c0, fc:                     char;     { Ignored Character, Compressed Flag }
  bsel:                    boolean;     { Flag for Numor Selection }



begin { ILLDATA_SCAN_SEARCH }
  if p = nil then
  begin                                 { It is the first call of the scan_search }
    { Get the Current ILL DATA Directory - that can be a wild specification without the root specification }
    sb_fspc := SUBSTR( illdat_dir, illdat_rdir.length + 1 );
    sb_fspc.length := sb_fspc.length - 1;      { Suppress the last "/" from this specification }
    sb_fspc := sb_fspc||'*';            { Append the wild specification of the Numor group }
    NEW( psca );                        { Allocate the Scan Data Structure }
    p := psca;
    with psca^ do
    begin                               { Initialise the Data Structure }
      { Open the Master Catalogue File }
      OPEN( illsca_mcf, illdat_rindex, [error_file,read_file,case_ena_file] );
      if iostatus <> 0 then
      begin                             { Error on Open of Master Index File Open }
        SRC_ERROR( mdnam, 651, e_severe );
        illsca_status := scan_error     { Flag the Error }
      end
      else illsca_status := scan_masindex;     { The master Index File is Opened }
      illsca_nu_min := gdat_snum_min;  { Set the Specified Numor Range }
      illsca_nu_max := gdat_snum_max;
      { We must create the basic sub-directory search profiles }
      illsca_mflt := FSPC_CREATE_FILTER( sb_fspc );
      { We must Create a additional filter for Each Specified Key }
      for ii := 1 to gdat_skeynb do
        illsca_ktb[ii] := FSPC_CREATE_FILTER( gdat_skeytb[ii] )
    end
  end else psca := p;

  bsel := false;
  with psca^ do
    repeat
      if illsca_status = scan_masindex then     { * Master Index Opened Phase }
        if EOF( illsca_mcf ) then
        begin                                   { It is the end of the Scan Process }
          CLOSE( illsca_mcf );                  { Close the Master Index File }
          for ii := 1 to gdat_skeynb do         { Free  all the Key Filters }
            FSPC_FREE_FILTER( illsca_ktb[ii] );
          FSPC_FREE_FILTER( illsca_mflt );      { Free the Sub-Index Filter }
          illsca_status := scan_closed          { Set the scan in the Close State }
        end
        else
        begin                                   { The Scan process examine an Master index Entry }
          { Get a line of ILLDATA Base Master Index }
          READLN( illsca_mcf, num_min, num_max, si_fname:0:true );
       {  READV( c_line, num_min, num_max, si_fname:0:true ); }
          if (num_min <= illsca_nu_max) and
             (num_max >= illsca_nu_min) and
             FSPC_MATCH( illsca_mflt, si_fname ) then
          begin                                 { This entry Match with the user requirement }
            { Form the NUMOR File Directory Specification }
            illsca_sdir := illdat_rdir||si_fname||'/';
            SET_STRING_PARM( gd_sub_dir^.ide_str, illsca_sdir );
            { Form the Sub-Index File Specification }
            sb_fspc := illsca_sdir||illdat_sbindf;
            { Open the sub-Index File  }
            OPEN( illsca_scf, sb_fspc, [error_file,read_file,case_ena_file] );
            if iostatus <> 0 then
            begin                               { Error on Open of Sub-Index File Open }
              SRC_ERROR( mdnam, 652, e_severe );
              for ii := 1 to gdat_skeynb do     { Free  all the Key Filters }
                FSPC_FREE_FILTER( illsca_ktb[ii] );
              FSPC_FREE_FILTER( illsca_mflt );  { Free the Sub-Index Filter }
              illsca_status := scan_error       { Flag the Error }
            end
            else illsca_status := scan_subindok
          end
        end;

      if illsca_status = scan_subindok then     { * Sub Index Opened Phase }
        if EOF( illsca_scf ) then
        begin                                   { It is the end of the Sub-Index File }
          CLOSE( illsca_scf );                  { Close the Sub-Index File }
          illsca_status := scan_masindex        { Continue to next Sub-Index }
        end
        else
        begin                                   { Read a Line of Sub-Index File OK }
          READ( illsca_scf, c_num:6, c0, fc );  { Get the Numor String and the Compressed Flag }
          if (c_num >= illsca_nu_min) and
             (c_num <= illsca_nu_max) then
          begin                                 { Numor To select for the Numor value }
            bsel := true;                       { Assume OK until shown otherwise }
            for ii := 1 to gdat_skeynb do       { For each Search key }
            begin
              READ( illsca_scf, lky, c0 );
              if lky = 0 then sc_key.length := 0
                         else READ( illsca_scf, sc_key:lky );
              { WRITE( ' Key # ', ii:2, ' = "', sc_key, '", model = "', gdat_skeytb[ii], '"' ); }
              if not FSPC_MATCH( illsca_ktb[ii], sc_key ) then
              begin  bsel := false; { WRITELN( ' Bad' ); } exit  end
              { else WRITELN( 'OK' ) }
            end;
            if bsel then
            begin                               { This Numor is OK to be Select }
              WRITEV( nm_fname, c_num:-6 );     { Build the Numor Filename }
              { Append the compressed type when it is a compressed file }
              illdat_numorf := illsca_sdir||nm_fname;
              if fc <> '0' then
              begin
                illdat_numorf := illdat_numorf||illdat_cmptyp;
                sb_fspc       := illdat_tmp||nm_fname
              end
              else sb_fspc.length := 0;
              ierr := GET_ILLDATA2( illdat_numorf, sb_fspc, fc <> '0' );
              if ierr > 0 then
              begin
                SRC_ERROR( mdnam, 623, e_severe );
                PASCAL_EXIT( 2 )
              end
            end
          end;
          READLN( illsca_scf )                  { Finish the Line read }
        end
    until (illsca_status < scan_masindex) or bsel;      { Repeat Loop }

  if psca^.illsca_status <= scan_closed then
  begin
    bsel := (psca^.illsca_status = scan_error);
    DISPOSE( psca );
    p := nil;
    if bsel then PASCAL_EXIT( 2 )
  end
end ILLDATA_SCAN_SEARCH;






{**************************************************************************}
{****   M E M O R Y  D A T A   B A S E   S E T U P   R O U T I N E S   ****}
{**************************************************************************}


procedure GDB_INIT( var gdb: gdb_ptr );
{ Allocation and Initialisation of a Get-Data-Base Descriptor }
begin
  if gdb = nil then
  begin
    NEW( gdb, gdata_nkey );
    with gdb^ do
    begin
      gdb_coef  :=   1.0;               { Set the normalization coefficient }
      gdb_tim   :=   nil;
      gdb_mon   :=   nil;               { Set to nil the monitor and tim entry parm. }
      gdb_ehde  :=   nil;               { Set to empty the related parameter list }
      gdb_elst  :=   nil;
      gdb_dhde  :=   nil;               { Set to empty the data array parameter list }
      gdb_dlst  :=   nil;
      gdb_knb   :=     0;               { Set to zero the number of key }
      gdb_dim   :=     0;               { Set to 0 the data dimension }
      gdb_ncol  :=     0;               { Set to 0 the column count }
      gdb_nrme  := false;               { Flag for normalisation enable/disable }
      gdb_nrmt  := false;               { Flag for normalisation by monitor }
      for i := 1 to gdb_nkey do
      with gdb_keytab[i] do
      begin
        keyf :=  [];                    { Set The not used Key Flag states }
        keyp := nil                     { Set to empty the key parameter table }
      end
    end
  end
end GDB_INIT;


procedure OUTADDR( p: $wild_pointer );
{ *** Procedure for Debugging *** }
var
  eq: record case boolean of
    false:( pp: $wild_pointer );
    true: ( iv: integer )
  end;

begin
  eq.pp := p;
  WRITELN( eq.iv:-10 );
end OUTADDR;


function GDB_NEWENTRY( gdb: gdb_ptr; gdbty: gdbe_ty; bdat: boolean ): gdbe_ptr;
{ Create a new Data Base Entry }
var
  p: gdbe_ptr;

begin
  NEW( p );
  with gdb_out^ do
    if bdat then begin
                   if gdb_dhde = nil then gdb_dhde := p
                                     else gdb_dlst^.gdbe_next := p;
                   gdb_dlst := p
                 end
            else begin
                   if gdb_ehde = nil then gdb_ehde := p
                                     else gdb_elst^.gdbe_next := p;
                   gdb_elst := p
                 end;
  with p^ do
  begin
    gdbe_next  := nil;
    gdbe_name  := nil;
    gdbe_rele  := nil;
    gdbe_plist := nil;
    gdbe_type  := gdbty
  end;
  GDB_NEWENTRY := p
end GDB_NEWENTRY;



function GDB_NEWPARM(   gdbe: gdbe_ptr; sk: set of exp_kinds ): gdbe_ptr;
{ Append a Data Base Parameter to the specified Data Base Entry }
const
  mdnam = 'GDBP';

var
  p, p1, p2: gdbp_ptr;
  nok:       boolean;
  rec:       exp_rec;

begin
  nok := false;
  GET_EXP_REFER( rec );                 { Get the object reference }
  if gdbe <> nil then
  with rec, sy_sym do
  if exp_ref <> nil then
  begin
    with gdbe^ do
      if gdbe_name = nil then
        gdbe_name := exp_ref^.ide_name;

    NEW( p );
    with p^ do
    begin
      exp_ref^.ide_lock := true;        { Lock for array size change }
      gdbp_next   := nil;               { Clear the next link }
      gdbp_kind   := exp_kind;          { Set the parameter kind }
      gdbp_offs   := exp_shift;         { Set the object offset }
      { Set the correct size of object (in element) }
      if exp_adm <> nil then
        with exp_adm^ do
          gdbp_size := idedim_stp*idedim_siz
      else
        gdbp_size := 1;
      { set the object address }
      case exp_kind of
        exp_valstr: gdbp_addr.ip := exp_ref;
        exp_telstr,
        exp_tabstr: gdbp_addr.ts := exp_aas;
        exp_valint: gdbp_addr.ip := exp_ref;
        exp_telint,
        exp_tabint: gdbp_addr.ti := exp_aai;
        exp_valflt: gdbp_addr.ip := exp_ref;
        exp_telflt,
        exp_tabflt: gdbp_addr.tf := exp_aaf;
      otherwise
        { Unsupported data parameter type }
        SRC_ERROR( mdnam, 603, e_severe )
      end;

      if sk <> [] and not (exp_kind in sk) then
      begin
        SRC_ERROR( mdnam, 504, e_severe );
         nok := true
      end
    end
  end;
  if nok then
  begin
    DISPOSE( p ); p := nil
  end
  else
  with gdbe^ do
  begin
    if gdbe_plist = nil then
      gdbe_plist := p
    else
    begin
      p1 := gdbe_plist;                 { Get the parameter list head }
      repeat                            { Find the end of parameter list }
        p2 := p1;
        p1 := p1^.gdbp_next
      until p1 = nil;
      p2^.gdbp_next := p                { Append the new parameter at the list }
    end
  end;
  GDB_NEWPARM := p
end GDB_NEWPARM;



function GDB_NEWLIST( gdbe: gdbe_ptr; sk: set of exp_kinds ): gdbp_ptr;
const
  mdnam = 'GDNL';

var
  pp, pm: gdbp_ptr;

begin
  pm := nil;
  with sy_sym do
  begin
    if sy = lparen then
    begin
      INSYMBOL;
      pm := GDB_NEWPARM( gdbe, sk );
      if pm <> nil then sk := [pm^.gdbp_kind];
      while sy = comma do
      begin
        INSYMBOL;
        pp := GDB_NEWPARM( gdbe, sk )
      end;
      if sy = rparen then INSYMBOL
                     else SRC_ERROR( mdnam, 24, e_error )
    end
    else pm := GDB_NEWPARM( gdbe, sk )
  end;
  GDB_NEWLIST := pm
end GDB_NEWLIST;



[global]
procedure DATABASE_FORMAT;
{ Use to define a Data Base organization :
  Corresponding to the GDATA Specific Statement Block "database_spc ... end".
  This GDATA Statement must be used to define the cpsh/gdata objects used
  to manage the database.
}
const
  mdnam = 'GDTB';

type
  datkwdty = ( k_key,                   { To define the Data Base used keys }
               k_keyinp,                { To define a key only for Intput }
               k_keyout,                { To define a Key only for Output }
               k_monitor,               { To define the monitor value link }
               k_time,                  { To define the time value link }
               k_param,                 { To define a parameter }
               k_flag,                  { To define a flag parameter array }
               k_xyz,                   { To define a coordinate parameter array }
               k_data                   { To describe the data location and organisation }
             );

  kwdid = record
    len: byte;
    nam: optid_name
  end;

var
  { Warning this table must be modified when the identifier size is changed }
  kwdtab: [static] array[datkwdty] of kwdid := (
  (  3, 'key            '),             { Any reference data key }
  (  7, 'key_inp        '),             { Any reference data key used only on input side }
  (  7, 'key_out        '),             { Any reference data key used only on output side }
  (  7, 'monitor        '),             { Monitor Reference }
  (  4, 'time           '),             { Time Reference }
  (  5, 'param          '),             { Experimental Condition Parameters }
  (  5, 'flags          '),             { Specific validity flags }
  (  3, 'xyz            '),             { Reference values as Theta/2_Theta/Omega/Chi/Phi ... }
  (  4, 'data           ')              { Count/Measure values }
  );

  bf:     boolean;
  dkwd:   datkwdty;
  id:     ide_ptr;
  pe, pd: gdbe_ptr;
  pp:     gdbp_ptr;
  gt:     gdbe_ty;

begin
  with sy_sym do
  begin
    GDB_INIT( gdb_out );
    with src_control^ do src_insnb := src_insnb + 1;
    with gdb_out^ do
    loop
      repeat
        INSYMBOL;                       { Gobble up the keyword or separator }
      until sy <> semicolon;
    exit if (sy = peofsy) or (sy = eofsy) or (sy = endsy);
      if sy = identsy then
      begin
        bf := false;
        with sy_ident do
        for pkwd := datkwdty"first to datkwdty"last do
          with kwdtab[pkwd] do
          begin
            dkwd := pkwd;
            bf := (STR_MATCH( sy_ident, 0, nam, len ) = 0);
        exit if bf
          end;

        if bf then
        begin
          INSYMBOL;
          case dkwd of
            k_keyinp,
            k_keyout,
            k_key:                      { To define a key for the indexed data base }
              begin
                pe := GDB_NEWENTRY( gdb_out, gdbe_dbkey, false );
                pp := GDB_NEWLIST( pe, [exp_valstr,exp_telstr] );
                if gdb_knb < gdb_nkey then
                begin
                  gdb_knb := gdb_knb + 1;
                  with gdb_keytab[gdb_knb] do
                  begin
                    keyp := pp;
                    if dkwd = k_keyinp then keyf := keyf + [gdbkf_notout];
                    if dkwd = k_keyout then keyf := keyf + [gdbkf_notinp]
                  end
                end
              end;

            k_monitor:                  { To define the monitor or time relation }
              begin
                gdb_mon := GDB_NEWENTRY( gdb_out, gdbe_monitor, false );
                pp      := GDB_NEWLIST( gdb_mon,
                           [exp_valflt,exp_valint,exp_telflt,exp_telint] )
              end;

            k_time:                     { To define the monitor or time relation }
              begin
                gdb_tim := GDB_NEWENTRY( gdb_out, gdbe_timer, false );
                pp      := GDB_NEWLIST( gdb_tim,
                           [exp_valflt,exp_valint,exp_telflt,exp_telint] )
              end;

            k_flag, k_xyz, k_data:
              loop
                case dkwd of
                  k_flag: gt := gdbe_flag;
                  k_xyz:  gt := gdbe_xyz;
                  k_data: gt := gdbe_adata;
                otherwise
                end;
                pe := GDB_NEWENTRY( gdb_out, gt, true );
                case dkwd of
                  k_flag: pp := GDB_NEWLIST( pe, [exp_tabint] );
                  k_xyz:
                    begin
                      pp := GDB_NEWLIST( pe, [exp_tabint,exp_tabflt] );
                      gdb_dim := gdb_dim + 1
                    end;
                  k_data:
                    begin
                      pp := GDB_NEWLIST( pe, [exp_tabint,exp_tabflt] );
                      pd := pe;
                      if sy = colon then
                      begin
                        if pp <> nil then gdb_ncol := gdb_ncol + 1;
                        INSYMBOL;
                        pe := GDB_NEWENTRY( gdb_out, gdbe_sigma, true );
                        pe^.gdbe_rele := pd;
                        pp := GDB_NEWLIST( pe, [exp_tabint,exp_tabflt] )
                      end;
                      if sy = colon then
                      begin
                        if pp <> nil then gdb_ncol := gdb_ncol + 1;
                        INSYMBOL;
                        pe := GDB_NEWENTRY( gdb_out, gdbe_rsigma, true );
                        pe^.gdbe_rele := pd;
                        pp := GDB_NEWLIST( pe, [exp_tabint,exp_tabflt] )
                      end
                    end;
                otherwise
                end;
                if pp <> nil then gdb_ncol := gdb_ncol + 1;
              exit if sy <> comma;
                INSYMBOL
              end;

            k_param:                    { Set a parameter }
              loop
                pe := GDB_NEWENTRY( gdb_out, gdbe_info, false );
                pp := GDB_NEWLIST( pe, [] );
              exit if sy <> comma;
                INSYMBOL
              end;

          otherwise
          end { case dkey of }
        end { if bf then }
        else SRC_ERROR( mdnam, 501, e_severe )
      end { if sy = identsy }
      else SRC_ERROR( mdnam, 502, e_severe );
    exit if (sy = peofsy) or (sy = eofsy) or (sy = endsy);
      if sy <> semicolon then SRC_ERROR( mdnam, 21, e_error )
    end;
    with src_control^ do src_insnb := src_insnb - 1;
    if sy = endsy then INSYMBOL
  end
end DATABASE_FORMAT;



[global]
procedure GDATA_SETTING;
{ Use to proceed with a data base (user data base or ILL data base):
    Corresponding to the GDATA Specific Statement Block "set_data ... end".
    Used to control the Data Extraction.
}
const
  mdnam = 'WRRF';

type
  setkwdty = ( k_gdb_out,               { To set a Data base output mode }
               k_gdb_not,               { To set the no data base output mode }
               k_pro_seq,               { To set a proceed sequence }
               k_oerr_seq,              { To set a On Error Data Sequence }
               k_end_seq,               { To set a close sequence }
               k_mon_nrm,               { To set the monitor normalisation mode }
               k_tim_nrm,               { To set the time normalisation mode }
               k_no_nrm,                { To set the no normalization mode }
               k_gdb_wrk                { To set the getting space number }
             );

  kwdid = record
    len: byte;
    nam: optid_name
  end;

var
  { Warning this table must be modified when the identifier size is changed }
  kwdtab: [static] array[setkwdty] of kwdid := (
  ( 15, 'database_output'),
  ( 14, 'no_auto_output '),
  (  4, 'main           '),
  (  8, 'on_error       '),
  (  7, 'on_exit        '),
  ( 12, 'monitor_norm   '),
  (  9, 'time_norm      '),
  (  7, 'no_norm        '),
  ( 10, 'work_space     ')
  );

  i, ierr:       integer;
  bf:            boolean;
  dkwd:          setkwdty;
  fspc, stn, st: str_string;
  id:            ide_ptr;
  pp:            gdbp_ptr;
  p:             mseq_ptr;


begin
  GDB_INIT( gdb_out );
  with src_control^ do src_insnb := src_insnb + 1;
  with sy_sym do
  loop
    repeat
      INSYMBOL;                         { Gobble up the keyword or separator }
    until sy <> semicolon;
  exit if (sy = peofsy) or (sy = eofsy) or (sy = endsy);
    if sy = identsy then
    begin
      bf := false;
      with sy_ident do
      for pkwd := setkwdty"first to setkwdty"last do
      with kwdtab[pkwd] do
      begin
        dkwd := pkwd;
        bf := (STR_MATCH( sy_ident, 0, nam, len ) = 0);
      exit if bf
      end;
      INSYMBOL;                         { Gobble up the key identifier }
      if bf then
      case dkwd of
        k_gdb_not:                      { To set the no output mode }
          begin
                                        { Not available now }
          end;

        k_gdb_out:                      { To set the Ascii Data base output }
          begin
            if sy = colon then INSYMBOL
                          else SRC_ERROR( mdnam, 31, e_error );
            if gdb_outf and (not gdb_outb) then CLOSE( gdb_outascf );
            GET_STREXPR( gdb_ofspc );
            gdb_outf := false;
            if gdb_ofspc.length > 0 then
            begin
              OPEN( gdb_outascf, gdb_ofspc, [error_file, write_file, case_ena_file] );
              if iostatus <> 0 then
                SRC_ERROR( mdnam, 712, e_severe )
              else
              begin
                gdb_outf := true; gdb_outb := false
              end
            end
          end;

        k_mon_nrm:                      { To set the monitor normalisation mode }
          with gdb_out^ do
          begin
            if sy = colon then
            begin
              INSYMBOL; gdb_coef := GET_FLTEXPR( gdb_coef );
              if gdb_coef <= 0.0 then
              begin
                SRC_ERROR( mdnam, 703, e_error ); gdb_coef := 1.0
              end
            end;
            gdb_nrmt := true; gdb_nrme := true
          end;

        k_tim_nrm:                      { To set the time normalisation mode }
          with gdb_out^ do
          begin
            if sy = colon then
            begin
              INSYMBOL; gdb_coef := GET_FLTEXPR( gdb_coef );
              if gdb_coef <= 0.0 then
              begin
                SRC_ERROR( mdnam, 703, e_error ); gdb_coef := 1.0
              end
            end;
            gdb_nrmt := false; gdb_nrme := true
          end;

        k_no_nrm:                       { To set the no normalization mode }
          with gdb_out^ do
          begin
            gdb_nrme := false
          end;

        k_gdb_wrk:                      { Set a work space number to locate the new data }
          begin
            gdat_iwrk := GET_INTEXPR( gdat_iwrk );
            if gdat_iwrk < 1 then
            begin
              SRC_ERROR( mdnam, 509, e_severe );
              gdat_iwrk := 1
            end
          end;

        k_pro_seq:                      { To set a proceed sequence }
          begin
            if sy <> sequencesy then SRC_ERROR( mdnam, 704, e_severe );
            NEW( p );
            if dat_seqhde = nil then dat_seqhde := p
                                else dat_seqlst^.mseq_nxt := p;
            dat_seqlst := p;
            with p^ do
            begin
              mseq_nxt := nil;
              mseq_txt := NEW_MACRO_LIST( endsy );
              INSYMBOL                  { To gobble up the end of sequence }
            end
          end;

        k_oerr_seq:                     { To set a on error proceed sequence }
          begin
            if sy <> sequencesy then SRC_ERROR( mdnam, 704, e_severe );
            NEW( p );
            if dat_esqhde = nil then dat_esqhde := p
                                else dat_esqlst^.mseq_nxt := p;
            dat_esqlst := p;
            with p^ do
            begin
              mseq_nxt := nil;
              mseq_txt := NEW_MACRO_LIST( endsy );
              INSYMBOL                  { To gobble up the end of sequence }
            end
          end;

        k_end_seq:                      { To set a end proceed sequence }
          begin
            if sy <> sequencesy then SRC_ERROR( mdnam, 704, e_severe );
            NEW( p );
            if dat_fsqhde = nil then dat_fsqhde := p
                                else dat_fsqlst^.mseq_nxt := p;
            dat_fsqlst := p;
            with p^ do
            begin
              mseq_nxt := nil;
              mseq_txt := NEW_MACRO_LIST( endsy );
              INSYMBOL                  { To gobble up the end of sequence }
            end
          end;
      otherwise
      end
      else SRC_ERROR( mdnam, 702, e_error )
    end else SRC_ERROR( mdnam, 701, e_error );
  exit if (sy = peofsy) or (sy = eofsy) or (sy = endsy);
    if sy <> semicolon then SRC_ERROR( mdnam, 21, e_error )
  end { loop };
  with src_control^ do src_insnb := src_insnb - 1;
  if sy_sym.sy = endsy then INSYMBOL
end GDATA_SETTING;



[global]
procedure GDATA_SHOW;
{
    Corresponding to the GDATA Specific Statement "show_data".
}
begin
  with gdb_out^ do
    if gdb_nrme then
    begin
      if gdb_nrmt then WRITE( ' The Monitor' )
                  else WRITE( ' The Time' );
      WRITE( ' normalization is enable with a factor of ' );
      WRITELN( gdb_coef:10:5, '.' )
    end
    else WRITELN( ' Automatic Normalization Disable.' )
end GDATA_SHOW;



procedure EXEC_PROC_SEQU( p: mseq_ptr := nil );
{ To execute a GDATA Statement Sequence.
}
begin
  if p = nil then p := dat_seqhde;
  while p <> nil do
  with p^ do
  begin
    EXECUTE_MACRO_CODE( mseq_txt, endsy );
    p := mseq_nxt
  end
end EXEC_PROC_SEQU;



function GDB_GETINT( p: gdbp_ptr ): integer;
{ To get an integer value from a Data Base Field }
var
  iv: integer;

begin
  with p^ do
    case gdbp_kind of
      exp_valint: iv := gdbp_addr.ip^.ide_int;
      exp_telint: iv := gdbp_addr.ti^.ide_itb[gdbp_offs];
      exp_valflt: iv := ROUND( gdbp_addr.ip^.ide_flt );
      exp_telflt: iv := ROUND( gdbp_addr.tf^.ide_ftb[gdbp_offs] );
      exp_valstr: with gdbp_addr.ip^ do
                    if ide_str <> nil then
                      if ide_str^.length > 0 then READV( ide_str^, iv )
                                             else iv := 0
                    else iv := 0;
      exp_telstr: with gdbp_addr.ts^ do
                    if ide_stb[gdbp_offs] <> nil then
                      if ide_stb[gdbp_offs]^.length > 0 then
                        READV( ide_stb[gdbp_offs]^, iv )
                      else iv := 0
                    else iv := 0;
    otherwise
      iv := 0
    end;
  GDB_GETINT := iv
end GDB_GETINT;


function GDB_GETFLT( p: gdbp_ptr ): gdreal;
{ To get a floating value from a Data Base Field }
var
  fv: gdreal;

begin
  with p^ do
    case gdbp_kind of
      exp_valint: fv := gdbp_addr.ip^.ide_int;
      exp_telint: fv := gdbp_addr.ti^.ide_itb[gdbp_offs];
      exp_valflt: fv := gdbp_addr.ip^.ide_int;
      exp_telflt: fv := gdbp_addr.tf^.ide_ftb[gdbp_offs];
      exp_valstr: with gdbp_addr.ip^ do
                    if ide_str <> nil then
                      if ide_str^.length > 0 then READV( ide_str^, fv )
                      else fv := 0.0
                    else fv := 0.0;
      exp_telstr: with gdbp_addr.ts^ do
                    if ide_stb[gdbp_offs] <> nil then
                      if ide_stb[gdbp_offs]^.length > 0 then
                        READV( ide_stb[gdbp_offs]^, fv )
                      else fv := 0.0
                    else fv := 0.0;
    otherwise
      fv := 0.0
    end;
  GDB_GETFLT := fv
end GDB_GETFLT;



procedure GDB_PUTINT( p: gdbp_ptr; iv: integer );
{ Deposite an integer value in a the data base specified object (with implicite coversion when required) }
var
  st: string( 64 );

begin
  with p^ do
    case gdbp_kind of
      exp_valint: gdbp_addr.ip^.ide_int := iv;
      exp_telint: gdbp_addr.ti^.ide_itb[gdbp_offs] := iv;
      exp_valflt: gdbp_addr.ip^.ide_int := iv;
      exp_telflt: gdbp_addr.tf^.ide_ftb[gdbp_offs] := iv;
      exp_valstr: with gdbp_addr.ip^ do
                  begin
                    if ide_str <> nil then DISPOSE( ide_str );
                    WRITEV( st, iv );
                    NEW( ide_str, st.length );
                    ide_str^ := st
                  end;
      exp_telstr: with gdbp_addr.ts^ do
                  begin
                    if ide_stb[gdbp_offs] <> nil then
                      DISPOSE( ide_stb[gdbp_offs] );
                    WRITEV( st, iv );
                    NEW( ide_stb[gdbp_offs], st.length );
                    ide_stb[gdbp_offs]^ := st
                  end;
    otherwise
    end;
end GDB_PUTINT;



procedure GDB_PUTFLT( p: gdbp_ptr; fv: gdreal );
{ Deposite a floating value in a the data base specified object (with implicite coversion when required) }
var
  st: string( 64 );

begin
  with p^ do
    case gdbp_kind of
      exp_valint: gdbp_addr.ip^.ide_int := ROUND( fv );
      exp_telint: gdbp_addr.ti^.ide_itb[gdbp_offs] := ROUND( fv );
      exp_valflt: gdbp_addr.ip^.ide_flt := fv;
      exp_telflt: gdbp_addr.tf^.ide_ftb[gdbp_offs] := fv;
      exp_valstr: with gdbp_addr.ip^ do
                  begin
                    if ide_str <> nil then DISPOSE( ide_str );
                    WRITEV( st, fv );
                    NEW( ide_str, st.length );
                    ide_str^ := st
                  end;
      exp_telstr: with gdbp_addr.ts^ do
                  begin
                    if ide_stb[gdbp_offs] <> nil then
                      DISPOSE( ide_stb[gdbp_offs] );
                    WRITEV( st, fv );
                    NEW( ide_stb[gdbp_offs], st.length );
                    ide_stb[gdbp_offs]^ := st
                  end;
    otherwise
    end;
end GDB_PUTFLT;



procedure GDBSTR_PUT( var p: ^string; var s: string );
{ Copy a string in a minimum string allocation }
begin
  if p <> nil then DISPOSE( p );
  if s.length > 0 then
  begin
    NEW( p, s.length ); p^ := s
  end
  else p := nil
end GDBSTR_PUT;



procedure GDB_PUTSTR( var st: string; p: gdbp_ptr );
{ Deposite a string value in a the data base specified object (with implicite coversion when required) }
begin
  with p^ do
    case gdbp_kind of
      exp_valint: WRITEV( st, gdbp_addr.ip^.ide_int );
      exp_telint: WRITEV( st, gdbp_addr.ti^.ide_itb[gdbp_offs] );
      exp_valflt: WRITEV( st, gdbp_addr.ip^.ide_int );
      exp_telflt: WRITEV( st, gdbp_addr.tf^.ide_ftb[gdbp_offs] );
      exp_valstr: GDBSTR_PUT( gdbp_addr.ip^.ide_str, st );
      exp_telstr: GDBSTR_PUT( gdbp_addr.ts^.ide_stb[gdbp_offs], st );
    otherwise
      st.length := 0
    end;
end GDB_PUTSTR;



procedure GDB_AII_OPE( op: arr_ope; var src, dst: gdbp_ptr;
                                             vd, vn: gdreal; bsq: boolean );
{ Copy of an Integer Array to an other Integer array procedure }
var
  size, srof, dsof: integer;
  srpi, dspi:       ^ide_arrint;

begin
  with src^ do
  begin
    srpi  := gdbp_addr.ti;
    srof  := gdbp_offs;
    size  := gdbp_size
  end;
  with dst^ do
  begin
    dspi  := gdbp_addr.ti;
    dsof  := gdbp_offs;
    if gdbp_size < size then size := gdbp_size
  end;

  with dspi^ do
  case op of
    ar_mov: if bsq then
              for ii := 1 to size do
              begin
                ide_itb[dsof] := SQR( srpi^.ide_itb[srof] );
                dsof := dsof + 1; srof := srof + 1
              end
            else
              for ii := 1 to size do
              begin
                ide_itb[dsof] := srpi^.ide_itb[srof];
                dsof := dsof + 1; srof := srof + 1
              end;
    ar_add: if bsq then
              for ii := 1 to size do
              begin
                ide_itb[dsof] := ide_itb[dsof] + SQR( srpi^.ide_itb[srof] );
                dsof := dsof + 1; srof := srof + 1
              end
            else
              for ii := 1 to size do
              begin
                ide_itb[dsof] := ide_itb[dsof] + srpi^.ide_itb[srof];
                dsof := dsof + 1; srof := srof + 1
              end;
    ar_sub: if bsq then
            begin
              vd := SQR( vd ); vn := SQR( vn );
              for ii := 1 to size do
              begin
                ide_itb[dsof] := ROUND( ide_itb[dsof]*vd +
                                        srpi^.ide_itb[srof]*vn );
                dsof := dsof + 1; srof := srof + 1
              end
            end
            else
              for ii := 1 to size do
              begin
                ide_itb[dsof] := ROUND( ide_itb[dsof]*vd -
                                        srpi^.ide_itb[srof]*vn );
                dsof := dsof + 1; srof := srof + 1
              end;

    ar_mul: for ii := 1 to size do
            begin
              ide_itb[dsof] := ide_itb[dsof] * srpi^.ide_itb[srof];
              dsof := dsof + 1; srof := srof + 1
            end;
  otherwise
  end
end GDB_AII_OPE;



procedure GDB_AIF_OPE( op: arr_ope; var src, dst: gdbp_ptr;
                                             vd, vn: gdreal; bsq: boolean );
{ Copy of an Integer Array to a Floating array procedure }
var
  size, srof, dsof: integer;
  srpi:             ^ide_arrint;
  dspf:             ^ide_arrflt;

begin
  with src^ do
  begin
    srpi  := gdbp_addr.ti;
    srof  := gdbp_offs;
    size  := gdbp_size
  end;
  with dst^ do
  begin
    dspf  := gdbp_addr.tf;
    dsof  := gdbp_offs;
    if gdbp_size < size then size := gdbp_size
  end;

  with dspf^ do
  case op of
    ar_mov: if bsq then
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := SQR( srpi^.ide_itb[srof] );
                dsof := dsof + 1; srof := srof + 1
              end
            else
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := srpi^.ide_itb[srof];
                dsof := dsof + 1; srof := srof + 1
              end;
    ar_add: if bsq then
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := ide_ftb[dsof] + SQR( srpi^.ide_itb[srof] );
                dsof := dsof + 1; srof := srof + 1
              end
            else
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := ide_ftb[dsof] + srpi^.ide_itb[srof];
                dsof := dsof + 1; srof := srof + 1
              end;
    ar_sub: if bsq then
            begin
              vd := SQR( vd ); vn := SQR( vn );
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := ide_ftb[dsof]*vd + srpi^.ide_itb[srof]*vn;
                dsof := dsof + 1; srof := srof + 1
              end
            end
            else
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := ide_ftb[dsof]*vd - srpi^.ide_itb[srof]*vn;
                dsof := dsof + 1; srof := srof + 1
              end;
  otherwise
  end
end GDB_AIF_OPE;



procedure GDB_AFI_OPE( op: arr_ope; var src, dst: gdbp_ptr;
                                             vd, vn: gdreal; bsq: boolean );
{ Copy of a Floating Array to an Integer array procedure }
var
  size, srof, dsof: integer;
  srpf:             ^ide_arrflt;
  dspi:             ^ide_arrint;

begin
  with src^ do
  begin
    srpf  := gdbp_addr.tf;
    srof  := gdbp_offs;
    size  := gdbp_size
  end;
  with dst^ do
  begin
    dspi  := gdbp_addr.ti;
    dsof  := gdbp_offs;
    if gdbp_size < size then size := gdbp_size
  end;

  with dspi^ do
  case op of
    ar_mov: if bsq then
              for ii := 1 to size do
              begin
                ide_itb[dsof] := ROUND( SQR( srpf^.ide_ftb[srof] ) );
                dsof := dsof + 1; srof := srof + 1
              end
            else
              for ii := 1 to size do
              begin
                ide_itb[dsof] := ROUND( srpf^.ide_ftb[srof] );
                dsof := dsof + 1; srof := srof + 1
              end;
    ar_add: if bsq then
              for ii := 1 to size do
              begin
                ide_itb[dsof] := ROUND( ide_itb[dsof] +
                                        SQR( srpf^.ide_ftb[srof] ) );
                dsof := dsof + 1; srof := srof + 1
              end
            else
              for ii := 1 to size do
              begin
                ide_itb[dsof] := ROUND( ide_itb[dsof]+srpf^.ide_ftb[srof] );
                dsof := dsof + 1; srof := srof + 1
              end;
    ar_sub: if bsq then
            begin
              vd := SQR( vd ); vn := SQR( vn );
              for ii := 1 to size do
              begin
                ide_itb[dsof] := ROUND( ide_itb[dsof]*vd +
                                        srpf^.ide_ftb[srof]*vn );
                dsof := dsof + 1; srof := srof + 1
              end
            end
            else
              for ii := 1 to size do
              begin
                ide_itb[dsof] := ROUND( ide_itb[dsof]*vd -
                                        srpf^.ide_ftb[srof]*vn );
                dsof := dsof + 1; srof := srof + 1
              end;
  otherwise
  end
end GDB_AFI_OPE;



procedure GDB_AFF_OPE( op: arr_ope; var src, dst: gdbp_ptr;
                                             vd, vn: gdreal; bsq: boolean );
{ Copy of a Floating Array to an other Floating array procedure }
var
  size, srof, dsof: integer;
  srpf, dspf:       ^ide_arrflt;

begin
  with src^ do
  begin
    srpf  := gdbp_addr.tf;
    srof  := gdbp_offs;
    size  := gdbp_size
  end;
  with dst^ do
  begin
    dspf  := gdbp_addr.tf;
    dsof  := gdbp_offs;
    if gdbp_size < size then size := gdbp_size
  end;

  with dspf^ do
  case op of
    ar_mov: if bsq then
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := SQR( srpf^.ide_ftb[srof] );
                dsof := dsof + 1; srof := srof + 1
              end
            else
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := srpf^.ide_ftb[srof];
                dsof := dsof + 1; srof := srof + 1
              end;
    ar_add: if bsq then
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := ide_ftb[dsof] + SQR( srpf^.ide_ftb[srof] );
                dsof := dsof + 1; srof := srof + 1
              end
            else
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := ide_ftb[dsof] + srpf^.ide_ftb[srof];
                dsof := dsof + 1; srof := srof + 1
              end;
    ar_sub: if bsq then
            begin
              vd := SQR( vd ); vn := SQR( vn );
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := SQRT( SQR( ide_ftb[dsof]*vd ) +
                                       SQR( srpf^.ide_ftb[srof]*vn ) );
                dsof := dsof + 1; srof := srof + 1
              end
            end
            else
              for ii := 1 to size do
              begin
                ide_ftb[dsof] := ide_ftb[dsof]*vd - srpf^.ide_ftb[srof]*vn;
                dsof := dsof + 1; srof := srof + 1
              end;
  otherwise
  end
end GDB_AFF_OPE;



procedure GDB_NRMI_OPE( var dst: gdbp_ptr; nrm: gdreal );
{ Normalisation of integer Array Procedure }
var
  size, dsof: integer;
  dspi:       ^ide_arrint;

begin
  with dst^ do
  begin
    dspi  := gdbp_addr.ti;
    dsof  := gdbp_offs;
    size  := gdbp_size
  end;

  with dspi^ do
  for ii := 1 to size do
  begin
    ide_itb[dsof] := ROUND( ide_itb[dsof]*nrm );
    dsof := dsof + 1
  end
end GDB_NRMI_OPE;



procedure GDB_NRMF_OPE( var dst: gdbp_ptr; nrm: gdreal );
{ Normalisation of Floating Array Procedure }
var
  size, dsof: integer;
  dspf:       ^ide_arrflt;

begin
  with dst^ do
  begin
    dspf  := gdbp_addr.tf;
    dsof  := gdbp_offs;
    size  := gdbp_size
  end;

  with dspf^ do
  for ii := 1 to size do
  begin
    ide_ftb[dsof] := ide_ftb[dsof]*nrm;
    dsof := dsof + 1
  end
end GDB_NRMF_OPE;



procedure GDB_KEY_APPEND( var sa: string; ch: char; var sb: string );
{ Append the characters SP,ch,SP and the Key in sb to the string sa }
var
  l, m: integer;
const
  mln = 120;

begin
  l := sa.length; m := sb.length;
  if sa[l] <> '*' then
  begin
    l := l + 1; sa[l] := ' ';
    if l+m+2 >= mln then
    begin
      l := l + 1; sa[l] := '*';
      l := l + 1; sa[l] := '*';
      l := l + 1; sa[l] := '*'
    end
    else
    begin
      l := l + 1; sa[l] :=  ch;
      l := l + 1; sa[l] := ' ';
      for i := 1 to m do
      begin  l := l + 1; sa[l] := sb[i]  end
    end;
    sa.length := l
  end
end GDB_KEY_APPEND;



procedure GDB_ADD_SCAN( ph: gdbe_ptr; isrc, idst: integer; bini: boolean );
{ Pattern addition Scan (summation) Procedure }
var
  curr_mon,
  curr_tim:    gdreal;
  st, st1:     string( 255 );
  op:          arr_ope;
  ps, pd:      gdbp_ptr;
  i, j:        integer;

begin
  if bini then op := ar_mov
          else op := ar_add;
  while ph <> nil do                   { Loop on all objects of one pattern }
  with ph^ do
  begin
    if gdbe_type = gdbe_rsigma then
      if gdbe_rele <> nil then ps := gdbe_rele^.gdbe_plist
                          else ps := nil
    else ps := gdbe_plist;
    i := isrc;
    while (i > 1) and (ps <> nil) do
    begin  ps := ps^.gdbp_next; i := i - 1  end;
    pd := gdbe_plist;
    j := idst;
    while (j > 1) and (pd <> nil) do
    begin  pd := pd^.gdbp_next; j := j - 1  end;
    if (ps <> nil) and (pd <> nil) then
    with pd^ do
    case gdbe_type of
      gdbe_monitor: begin
                     curr_mon := GDB_GETFLT( ps );
                     if not bini then curr_mon := curr_mon + GDB_GETFLT( pd );
                     GDB_PUTFLT( pd, curr_mon )
                   end;

      gdbe_timer:  begin
                     curr_tim := GDB_GETFLT( ps );
                     if not bini then curr_tim := curr_tim + GDB_GETFLT( pd );
                     GDB_PUTFLT( pd, curr_tim ) 
                   end;

      gdbe_dbkey:  if bini then
                   begin
                     GDB_GETSTR( st, ps ); GDB_PUTSTR( st, pd )
                   end
                   else
                   begin
                     GDB_GETSTR( st1, ps ); GDB_GETSTR( st, pd );
                     GDB_KEY_APPEND( st, '+', st1 );
                     GDB_PUTSTR( st, pd )
                   end;

      gdbe_info:   case gdbp_kind of
                     exp_valstr,
                     exp_telstr: begin
                                   GDB_GETSTR( st, ps ); GDB_PUTSTR( st, pd );
                                 end;
                     exp_valint,
                     exp_telint: GDB_PUTINT( pd, GDB_GETINT( ps ) );
                     exp_valflt,
                     exp_telflt: GDB_PUTFLT( pd, GDB_GETFLT( ps ) );
                   otherwise
                   end;

      gdbe_flag:   if bini then GDB_AII_OPE( ar_mov, ps, pd, 1.0, 1.0, false )
                           else GDB_AII_OPE( ar_mul, ps, pd, 1.0, 1.0, false );

      gdbe_xyz:    if bini then
                     if gdbp_kind = exp_tabint then
                       if ps^.gdbp_kind = exp_tabint then
                         GDB_AII_OPE( op, ps, pd, 1.0, 1.0, false )
                       else
                         GDB_AFI_OPE( op, ps, pd, 1.0, 1.0, false )
                     else
                       if ps^.gdbp_kind = exp_tabint then
                         GDB_AIF_OPE( op, ps, pd, 1.0, 1.0, false )
                       else
                         GDB_AFF_OPE( op, ps, pd, 1.0, 1.0, false );


      gdbe_adata:  begin
                     if gdbp_kind = exp_tabint then
                       if ps^.gdbp_kind = exp_tabint then
                         GDB_AII_OPE( op, ps, pd, 1.0, 1.0, false )
                       else
                         GDB_AFI_OPE( op, ps, pd, 1.0, 1.0, false )
                     else
                       if ps^.gdbp_kind = exp_tabint then
                         GDB_AIF_OPE( op, ps, pd, 1.0, 1.0, false )
                       else
                         GDB_AFF_OPE( op, ps, pd, 1.0, 1.0, false )
                   end;

      gdbe_rsigma,
      gdbe_sigma:  if gdbp_kind = exp_tabint then
                     if ps^.gdbp_kind = exp_tabint then
                       GDB_AII_OPE( op, ps, pd, 1.0, 1.0, true )
                     else
                       GDB_AFI_OPE( op, ps, pd, 1.0, 1.0, true )
                   else
                     if ps^.gdbp_kind = exp_tabint then
                       GDB_AIF_OPE( op, ps, pd, 1.0, 1.0, true )
                     else
                       GDB_AFF_OPE( op, ps, pd, 1.0, 1.0, true );

    otherwise
      { gdbe_invalid }
    end;
    ph := gdbe_next
  end
end GDB_ADD_SCAN;



procedure GDB_SUB_SCAN( ph: gdbe_ptr; isrc, idst: integer );
{ Pattern Subtract Scan Procedure }
var
  facd, facs:           [static] gdreal;
  curr_mon, mond, mons,
  curr_tim, timd, tims: gdreal;
  st, st1:              string( 255 );
  ps, pd:               gdbp_ptr;
  i, j:                 integer;

begin
  while ph <> nil do
  with ph^ do
  begin
    ps := gdbe_plist;
    i := isrc;
    while (i > 1) and (ps <> nil) do
    begin  ps := ps^.gdbp_next; i := i - 1  end;
    pd := gdbe_plist;
    j := idst;
    while (j > 1) and (pd <> nil) do
    begin  pd := pd^.gdbp_next; j := j - 1  end;

    if (ps <> nil) and (pd <> nil) then
    with pd^ do
    case gdbe_type of
      gdbe_monitor: begin
                     mons := GDB_GETFLT( ps ); mond := GDB_GETFLT( pd );
                     curr_mon := 1.0/( 1.0/mons + 1.0/mond );
                     GDB_PUTFLT( pd, curr_mon );
                     with gdb_out^ do
                     if gdb_nrme and gdb_nrmt then
                     begin
                       facd := curr_mon/mond;
                       facs := curr_mon/mons
                     end
                   end;

      gdbe_timer:  begin
                     tims := GDB_GETFLT( ps ); timd := GDB_GETFLT( pd );
                     curr_tim := 1.0/(1.0/tims + 1.0/timd );
                     GDB_PUTFLT( pd, curr_tim );
                     with gdb_out^ do
                     if gdb_nrme and not gdb_nrmt then
                     begin
                       facd := curr_tim/timd;
                       facs := curr_tim/tims
                     end
                   end;

      gdbe_dbkey:  begin
                     GDB_GETSTR( st1, ps ); GDB_GETSTR( st, pd );
                     GDB_KEY_APPEND( st, '-', st1 );
                     GDB_PUTSTR( st, pd )
                   end;

      gdbe_info:   case gdbp_kind of
                     exp_valstr,
                     exp_telstr: begin
                                   GDB_GETSTR( st, ps ); GDB_PUTSTR( st, pd );
                                 end;
                     exp_valint,
                     exp_telint: GDB_PUTINT( pd, GDB_GETINT( ps ) );
                     exp_valflt,
                     exp_telflt: GDB_PUTFLT( pd, GDB_GETFLT( ps ) );
                   otherwise
                   end;

      gdbe_flag:   GDB_AII_OPE( ar_mul, ps, pd, 1.0, 1.0, false );

      gdbe_adata:  if gdbp_kind = exp_tabint then
                     if ps^.gdbp_kind = exp_tabint then
                       GDB_AII_OPE( ar_sub, ps, pd, facd, facs, false )
                     else
                       GDB_AFI_OPE( ar_sub, ps, pd, facd, facs, false )
                   else
                     if ps^.gdbp_kind = exp_tabint then
                       GDB_AIF_OPE( ar_sub, ps, pd, facd, facs, false )
                     else
                       GDB_AFF_OPE( ar_sub, ps, pd, facd, facs, false );

      gdbe_rsigma,
      gdbe_sigma:  if gdbp_kind = exp_tabint then
                     if ps^.gdbp_kind = exp_tabint then
                       GDB_AII_OPE( ar_sub, ps, pd, facd, facs, true )
                     else
                       GDB_AFI_OPE( ar_sub, ps, pd, facd, facs, true )
                   else
                     if ps^.gdbp_kind = exp_tabint then
                       GDB_AIF_OPE( ar_sub, ps, pd, facd, facs, true )
                     else
                       GDB_AFF_OPE( ar_sub, ps, pd, facd, facs, true );

    otherwise
      { gdbe_xyz     }
      { gdbe_invalid }
    end;
    ph := gdbe_next
  end
end GDB_SUB_SCAN;




procedure GDB_NORM_RSIGMA( idst: integer );
{  }
var
  pe:            gdbe_ptr;
  pd, pda:       gdbp_ptr;
  i, o1, o2, sz: integer;
  nv, vv:        gdreal;

begin
  pe := gdb_out^.gdb_dhde;
  while pe <> nil do
  with pe^ do
  begin
    if gdbe_type = gdbe_rsigma then
      if gdbe_rele <> nil then pda := gdbe_rele^.gdbe_plist
                          else pda := nil
    else pda := nil;
    pd := gdbe_plist;
    i := idst;
    while (i > 1) and (pd <> nil) do
    begin
      pd := pd^.gdbp_next;
      if pda <> nil then pda := pda^.gdbp_next;
      i := i - 1
    end;

    if (pd <> nil) then
    with pd^ do
    begin
      sz := gdbp_size;
      o1 := gdbp_offs;
      case gdbe_type of
        gdbe_sigma:                     { Data measurment Sigma (squared sigma addition) }
          if gdbp_kind = exp_tabint then
            with gdbp_addr.ti^ do       { Integer type }
            for ii := 1 to sz do
            begin
              ide_itb[o1] := ROUND( SQRT( ide_itb[o1] ) );
              o1 := o1 + 1
            end
          else
            with gdbp_addr.tf^ do       { Floating type }
            for ii := 1 to sz do
            begin
              ide_ftb[o1] := SQRT( ide_ftb[o1] );
              o1 := o1 + 1
            end;

        gdbe_rsigma:                    { Data repartition Sigma Computing }
          if (pda <> nil) and (gdat_nsum > 1) then
          begin
            if sz > pda^.gdbp_size then sz := pda^.gdbp_size;
            o2 := pda^.gdbp_offs;
            nv := (gdat_nsum*gdat_nsum**2)/(gdat_nsum - 1);
            for ii := 1 to sz do
            begin
              if pda^.gdbp_kind = exp_tabint then
                vv := SQR( pda^.gdbp_addr.ti^.ide_itb[o2]/gdat_nsum )
              else
                vv := SQR( pda^.gdbp_addr.tf^.ide_ftb[o2]/gdat_nsum );
              if gdbp_kind = exp_tabint then
                with gdbp_addr.ti^ do
                  ide_itb[o1] :=
                     ROUND( SQRT( nv*(ide_itb[o1]/gdat_nsum - vv) ) )
              else
                with gdbp_addr.tf^ do
                  ide_ftb[o1] := SQRT( nv*(ide_ftb[o1]/gdat_nsum - vv) );
              o2 := o2 + 1;
              o1 := o1 + 1
            end;
            gd_rsig^.ide_int := 1
          end;

      otherwise
      end
    end;
    pe := gdbe_next
  end
end GDB_NORM_RSIGMA;



procedure GDB_ADD( isrc, idst: integer; bini: boolean );
{  }
begin
  with gdb_out^ do
  begin
    GDB_ADD_SCAN( gdb_ehde, isrc, idst, bini ); { Add for parameters }
    GDB_ADD_SCAN( gdb_dhde, isrc, idst, bini )  { Add for data }
  end
end GDB_ADD;



procedure GDB_SUB( isrc, idst: integer );
{  }
begin
  with gdb_out^ do
  begin
    GDB_SUB_SCAN( gdb_ehde, isrc, idst );
    GDB_SUB_SCAN( gdb_dhde, isrc, idst )
  end
end GDB_SUB;



procedure GDB_NRM_SCAN( idst: integer );
{ Normalisation Setting }
var
  ph:               gdbe_ptr;
  pmon, ptim:       gdbp_ptr;
  coef, cmon, ctim: gdreal;
  pd:               gdbp_ptr;
  i:                integer;

begin
  coef := 1.0;
  with gdb_out^ do
  if gdb_nrme then
  begin
    ph   := gdb_out^.gdb_dhde;
    pmon := gdb_mon^.gdbe_plist;
    ptim := gdb_tim^.gdbe_plist;
    i := idst;
    while (i > 1) and (pmon <> nil) and (ptim <> nil) do
    begin
      pmon := pmon^.gdbp_next;
      ptim := ptim^.gdbp_next;
      i := i - 1
    end;
    cmon := GDB_GETFLT( pmon );
    ctim := GDB_GETFLT( ptim );
    if gdb_nrmt then coef := cmon
                else coef := ctim;
    if (coef > 1.0) and (gdb_coef > 1.0) then
    begin
      coef := gdb_coef/coef;
      GDB_PUTFLT( pmon, cmon*coef );
      GDB_PUTFLT( ptim, ctim*coef )
    end
  end;

  if coef <> 1.0  then
  while ph <> nil do
  with ph^ do
  begin
    pd := gdbe_plist;
    i := idst;
    while (i > 1) and (pd <> nil) do
    begin  pd := pd^.gdbp_next; i := i - 1  end;
    if pd <> nil then
    with pd^ do
    case gdbe_type of
      gdbe_rsigma,
      gdbe_sigma,
      gdbe_adata:   if gdbp_kind = exp_tabint then GDB_NRMI_OPE( pd, coef )
                                              else GDB_NRMF_OPE( pd, coef );
    otherwise
    end;
    ph := gdbe_next
  end
end GDB_NRM_SCAN;



{***********************************************************}
{*********      Read Data Base File Procedures     *********}
{***********************************************************}


procedure GDB_READ_DATA( numor: integer; bkey: boolean; var ierr: integer );
{ To read a Data Base file with any key management }
begin

end GDB_READ_DATA;






{************************************************************}
{*********      Write Data Base File Procedures     *********}
{************************************************************}



procedure GDB_WRITE_RHD( icd, nz: integer; pn: pstring );
{ To Write an Ascii Data base file logical record header }
var
  sn: integer;

begin
  if pn <> nil then sn := pn^.length
               else sn := 0;
  WRITE( gdb_outascf, icd:4, sn:5, nz:5 );
  if sn > 0 then WRITE( gdb_outascf, ' ', pn^ );
  WRITELN( gdb_outascf )
end GDB_WRITE_RHD;



procedure GDB_WRITE_TAB( scd, icd, fcd: integer; pe: gdbe_ptr );
{ To Write in an Ascii Data Base file a Data Array record }
var
  i, ic, ia, ns: integer;
  ps:            pstring;

begin
  if pe <> nil then
  with pe^, gdbe_plist^, gdbp_addr do
  begin
    if gdbp_kind = exp_tabstr then icd := scd
    else
      if gdbp_kind = exp_tabflt then icd := fcd;
    GDB_WRITE_RHD( icd, gdbp_size, gdbe_name );
    i  :=         0;
    ia := gdbp_offs;
    case gdbp_kind of
      exp_tabstr: for ii := 1 to gdbp_size do
                  begin
                    ps := ts^.ide_stb[ia]; ia := ia + 1;
                    if ps = nil then ns := 0
                                else ns := ps^.length;
                    WRITELN( gdb_outascf, ns:5 );
                    if ps <> nil then WRITELN( gdb_outascf, ps^ )
                  end;
      exp_tabint: for ii := 1 to gdbp_size do
                  begin
                    i := i + 1;
                    WRITE( gdb_outascf, ' ', ti^.ide_itb[ia]:11 );
                    ia := ia + 1;
                    if i >= 5 then
                    begin  i := 0; WRITELN( gdb_outascf )  end
                  end;
      exp_tabflt: for ii := 1 to gdbp_size do
                  begin
                    i := i + 1;
                    WRITE( gdb_outascf, ' ', tf^.ide_ftb[ia]:-15 );
                    ia := ia + 1;
                    if i >= 5 then
                    begin  i := 0; WRITELN( gdb_outascf )  end
                  end;
    otherwise
    end;
    if i > 0 then WRITELN( gdb_outascf )
  end
end GDB_WRITE_TAB;



procedure GDB_WRITE_OLIST( pe: gdbe_ptr );
{ To Write an Ascii Data base file logical record header }
var
  st: string( 255 );
  p:  gdbp_ptr;

begin
  while pe <> nil do
  with pe^ do
  begin
    p := gdbe_plist;
    if p <> nil then
    with p^ do
    case gdbe_type of
      gdbe_dbkey:   begin
                      GDB_GETSTR( st, p );
                      GDB_WRITE_RHD( 2, st.length, gdbe_name );
                      WRITELN( gdb_outascf, ' ', st )
                    end;

      gdbe_monitor: begin
                      GDB_WRITE_RHD( 3, 1, gdbe_name );
                      WRITELN( gdb_outascf, GDB_GETFLT( p ):-16:8 )
                    end;

      gdbe_timer:   begin
                      GDB_WRITE_RHD( 4, 1, gdbe_name );
                      WRITELN( gdb_outascf, GDB_GETFLT( p ):-16:8 )
                    end;

      gdbe_info:    case gdbp_kind of
                      exp_valstr, exp_telstr:
                        begin
                          GDB_GETSTR( st, p );
                          GDB_WRITE_RHD( 5, st.length, gdbe_name );
                          WRITELN( gdb_outascf, ' ', st )
                        end;
                      exp_valint, exp_telint:
                        begin
                          GDB_WRITE_RHD( 6, 1, gdbe_name );
                          WRITELN( gdb_outascf, GDB_GETINT( p ):10 )
                        end;
                      exp_valflt, exp_telflt:
                        begin
                          GDB_WRITE_RHD( 7, 1, gdbe_name );
                          WRITELN( gdb_outascf, GDB_GETFLT( p ):-16:8 )
                        end;
                      exp_tabstr, exp_tabint, exp_tabflt:
                        GDB_WRITE_TAB(  8,  9, 10, pe );
                    otherwise
                    end;

      gdbe_flag:    GDB_WRITE_TAB( -1, 11, -1, pe );
      gdbe_xyz:     GDB_WRITE_TAB( -1, 12, 13, pe );
      gdbe_adata:   GDB_WRITE_TAB( -1, 14, 15, pe );
      gdbe_sigma:   GDB_WRITE_TAB( -1, 16, 17, pe );
      gdbe_rsigma:  if gd_rsig^.ide_int > 0 then
                      GDB_WRITE_TAB( -1, 18, 19, pe );
    otherwise { gdbe_invalid }
    end;
    pe := gdbe_next
  end
end GDB_WRITE_OLIST;



procedure GDB_WRITE_DATA;
{ Write Data Base form }
const
  mdnam = 'WRDB';

var
  pe:      gdbe_ptr;
  kc:       integer;
  st: string( 255 );

begin
  with gdb_out^ do
  begin
    { Write the logical Key Record }
    kc := 0;
    for ii := 1 to gdb_nkey do
      if not (gdbkf_notout in gdb_keytab[ii].keyf) then kc := kc + 1;

    WRITELN( gdb_outascf, 1:2, kc:2 );
    for ii := 1 to gdb_nkey do
    with gdb_keytab[ii] do
    if not (gdbkf_notout in keyf) then
    begin
      kc := kc + 1;
      GDB_GETSTR( st, keyp );
      if st.length > gdata_keys then
      begin
        for jj := 0 to 3 do
          st[gdata_keys - jj] := st[st.length - jj];
        st.length := gdata_keys
      end;
      WRITE( gdb_outascf, ' ', st:gdata_keys )
    end;
    WRITELN( gdb_outascf );

    GDB_WRITE_OLIST( gdb_ehde );
    GDB_WRITE_OLIST( gdb_dhde );

    { Write the end of logical data set }
    WRITELN( gdb_outascf, 0:2, 0:5 );
    gdat_iwrt := gdat_iwrt + 1
  end
end GDB_WRITE_DATA;




{*************************************************************}
{*********    General Proceed Statement Procedures   *********}
{*************************************************************}


function  SET_SCAN_MODE( num_min, num_max: integer ): boolean;
const
  mdnam = 'STSC';

var
  i:          integer;
  berr, bstp: boolean;

begin
  with sy_sym do
  begin
    berr   := false;
    gdat_snum_min := num_min;
    gdat_snum_max := num_max;
    gdat_skeynb   :=       0;
    INSYMBOL;                                   { Gobble up the keys symbol }
    if sy <> semicolon then
      GET_STREXPR( gdat_skeytb[1] );            { Get the string of a DATA BASE KEY }
    gdat_skeynb := 1;
    repeat
      i := INDEX( gdat_skeytb[gdat_skeynb], ':' );
      if i > 0 then
        if gdat_skeynb < gdata_nkey then
        begin
          gdat_skeytb[gdat_skeynb+1] := SUBSTR( gdat_skeytb[gdat_skeynb], i + 1 );
          gdat_skeytb[gdat_skeynb].length := i - 1;
          gdat_skeynb := gdat_skeynb + 1
        end
        else berr := true
    until (i = 0) or berr;
    { Input Key String(s) O.K. }
    if berr then
    begin
      SRC_ERROR( mdnam, 510, e_error );
      SET_SCAN_MODE := false
    end
  end;
  SET_SCAN_MODE := true
end SET_SCAN_MODE;



procedure GET_CATALOGUE;
{ Output a catalogue of data with optional specific data filter }
const
  mdnam = 'GCTL';

var
  i, num, numi:     integer;
  berr, brange:     boolean;
  psca:   illsca_ptr := nil;

begin
  with sy_sym do
  begin
    berr   := false;
    brange := false;
    gdat_snum_min :=      0;
    gdat_snum_max := maxint;
    gdat_skeynb   :=      0;
    if sy <> semicolon then
    loop
      if (sy = identsy) and (sy_ident = 'range') then
      begin
        if brange then SRC_ERROR( mdnam, 511, e_error )
                  else brange := true;
        INSYMBOL;                                               { Gobble up the range Word }
        gdat_snum_min := GET_INTEXPR( 0 );                      { Get the minimum Numor value }
        if gdat_snum_min > 0 then gdat_snum_max := gdat_snum_min;
        if sy = twodot then
        begin
          INSYMBOL;
          gdat_snum_max := GET_INTEXPR( gdat_snum_max )         { Get the Upper limit of Numor Range }
        end
      end
      else
      begin
        if gdat_skeynb < gdata_nkey then gdat_skeynb := gdat_skeynb + 1
                                    else berr := true;
        if not berr then GET_STREXPR( gdat_skeytb[gdat_skeynb] )        { Get the string of a DATA BASE KEY }
      end;
    exit if (sy <> comma) and ((sy <> identsy) or (sy_ident <> 'range'));
      if sy = comma then INSYMBOL                               { Gobble up comma and continue to the next Numor }
    end;
    if gdat_skeynb = 1 then
    begin { Try to split in more key with the colon separator }
      repeat
        i := INDEX( gdat_skeytb[gdat_skeynb], ':' );
        if i > 0 then
          if gdat_skeynb < gdata_nkey then
          begin
            gdat_skeytb[gdat_skeynb+1] := SUBSTR( gdat_skeytb[gdat_skeynb], i + 1 );
            gdat_skeytb[gdat_skeynb].length := i - 1;
            gdat_skeynb := gdat_skeynb + 1
          end
          else berr := true
      until (i = 0) or berr
    end;
    { Input Key String(s) O.K. }
    if berr then SRC_ERROR( mdnam, 510, e_error )
  end;

  if ILL_DATA_INDEX( false ) then
  loop                                  { Send a informative message to user }
    ILLDATA_SCAN_SEARCH( psca );        { Perform A Scan Search Step }
  exit if psca = nil;                   { Exit of the Loop on End of Scan }
  end;
  gdat_skeynb := 0                      { Clear the search key profile }
end GET_CATALOGUE;




procedure GET_NUMOR_LIST( badd: boolean; idst: integer; bout_ena: boolean );
{ To get a numor range (list on summation following badd) }
const
  mdnam = 'GNML';

var
  num, num1, num2, ierr:     integer;
  bpar, bini, bsca, bgdb:    boolean;
  psca:                   illsca_ptr;

begin
  bini :=  true;
  bsca := false;

  with sy_sym do
  begin
    if sy = lparen then                                 { When a left parenthesys is found, we flag and gobble up it }
    begin
      INSYMBOL; bpar := true
    end
    else bpar := false;

  main_loop:
    loop                                                { Main Loop on different Numor ranges }
      if (sy = identsy) and (sy_ident = 'from') then
      begin
        INSYMBOL;
        GET_STREXPR( gdb_ifspc );                       { Get the Data Base File Specification }
        bgdb := true                                    { Set flag to take data from the Local Data Base File }
      end
      else bgdb := false;                               { Take data from ILL Data Base }
      num1 := GET_INTEXPR( 0 );                         { Get the first Numor of the range }
      if sy = twodot then                               { For a range separator ... }
      begin
        INSYMBOL;                                       { ... gobble up it ... }
        num2 := GET_INTEXPR( num1 );                    { ... and get the last Numor of range }
      end else num2 := num1;                            { ... or for a unique Numor }
      if (num1 < 0) or (num2 < num1) then
      begin
        SRC_ERROR( mdnam, 751, e_severe );
        exit
      end;
      { Set the NUMOR Scan Aquisition mode if required }
      if (sy = identsy) and (sy_ident = 'keys') then
      begin
        bsca := SET_SCAN_MODE( num1, num2 );
        psca := nil
      end;
      { Get each numor in the range }
      num := num1;
      while num <= num2 do                              { For each numor in the range ... }
      begin
        if bgdb then GDB_READ_DATA( num, bsca, ierr )   { Get it from a file data base, ... }
        else
          if bsca then
          begin
            ILLDATA_SCAN_SEARCH( psca );                { ... from the ILL data base in search scan access }
            ierr := 0;
            if psca = nil then
            begin bsca := false; exit  end              { Nota: when the scan stops, the range scan is stopped }
          end
          else GET_ILLDATA( num, ierr );                { ... or from the ILL data base in direct access }
        if ierr <> 0 then
        begin                                           { When a numor does exist as specified (Local data-base or ILL data Base) }
          gd_errcd^.ide_int := ierr;                    { Put the error in the Sequence error identifier }
          EXEC_PROC_SEQU( dat_esqhde )                  { Execute the on error Sequence }
        end;
    exit main_loop if gd_errcd^.ide_int <> 0;
        if ierr = 0 then
        begin
          GDB_ADD( gdat_iwrk, idst, bini );             { Put or add the present Pattern/Numor to the Pattern Array  }
          if bini and badd then bini := false;          { For summation clear the init flag }
          if not badd then begin                        { For the List ... }
                             GDB_NORM_RSIGMA( idst );   { ... Normalize Pattern }
                             GDB_NRM_SCAN( idst );
                             EXEC_PROC_SEQU;
                             if bout_ena and gdb_outf then GDB_WRITE_DATA
                           end
                           else
                           begin
                             gdat_nsum := gdat_nsum + 1;
                             gd_ndat^.ide_int := gdat_nsum
                           end
        end
        else ierr := 0;                                 { Clear the error to can get the next numor }
        if not bsca then num := num + 1
      end { for };
    exit if sy <> comma;
      INSYMBOL                                          { Gobble up comma and continue to the next Numor }
    end { loop };
    if badd then GDB_NORM_RSIGMA( idst );
    if bpar then                               if (num2 > num1) then
              { When a left parenthesys was specified we must find a right one }
      if sy = rparen then INSYMBOL
                     else SRC_ERROR( mdnam, 23, e_error )
  end
end GET_NUMOR_LIST;



procedure GET_NUMOR_SUMM( bout_ena: boolean );
{ To perform a summ of NUMORs - Patterns }
begin
  gdat_nsum := 0;
  gd_ndat^.ide_int := 0;
  gd_rsig^.ide_int := 0;
  GET_NUMOR_LIST( true, 1, false );
  if (gdat_nsum > 0) and (gd_errcd^.ide_int = 0) then
  begin
    GDB_NRM_SCAN( 1 );
    EXEC_PROC_SEQU;
    if bout_ena and gdb_outf then GDB_WRITE_DATA
  end;
end GET_NUMOR_SUMM;



procedure GET_NUMOR_DIFF( bout_ena: boolean );
{ To perform a difference of NUMORs - Patterns }
var
  ncp: integer;

begin
  gdat_nsum := 0;
  gd_ndat^.ide_int := 0;
  gd_rsig^.ide_int := 0;
  GET_NUMOR_LIST( true, 1, false );
  with sy_sym do
    if sy = colon then
      INSYMBOL { A minus sign is expected }
    else
      SRC_ERROR( 'DIFF', 752, e_error );
  if (gdat_nsum > 0) and (gd_errcd^.ide_int = 0) then
  begin
    ncp := gdat_nsum;
    gdat_nsum := 0;
    gd_ndat^.ide_int := 0;
    gd_rsig^.ide_int := 0;
    GET_NUMOR_LIST( true, 2, false );
    if (gdat_nsum > 0) and (gd_errcd^.ide_int = 0) then
    begin
      GDB_SUB( 2, 1 );
      if ncp < gdat_nsum then ncp := gdat_nsum;
      gdat_nsum := ncp;
      gd_ndat^.ide_int := ncp;
      gd_rsig^.ide_int := 1
    end;
    GDB_NRM_SCAN( 1 );
    EXEC_PROC_SEQU;
    if bout_ena and gdb_outf then GDB_WRITE_DATA
  end
  else
  begin
    gdat_nsum := 0;
    gd_ndat^.ide_int := 0;
    gd_rsig^.ide_int := 0
  end
end GET_NUMOR_DIFF;



[global]
procedure GET_NUMOR_OPE;
{ Use to proceed a binary data base.
  Corresponding to the "proceed  ... " GDATA statement.
}
const
  mdnam = 'GOPE';


type
  setdirty = ( d_list,                  { To get a List of diagrams }
               d_summ,                  { To generate a summ of diagram }
               d_diff,                  { To generate a difference of diagram }
               d_keys,                  { To generate a catalog of data with a filter }
               d_mindex                 { To Force the index creation/update }
             );

  dirid = record
    len: byte;
    nam: optid_name
  end;

var
  { warning this table must be modified when the identifier size is changed }
  dirtab: [static] array[setdirty] of dirid := (
  (  4, 'list           '),             { Get a list of Patterns/Numors }
  (  4, 'summ           '),             { Get the summation of some Patterns/Numors }
  (  4, 'diff           '),             { Get the Difference of Patterns/Numors }
  (  4, 'keys           '),             { Get data Keys for create a catalog of data }
  ( 10, 'make_index     ')              { Force the ILL Data Base index creation }
  );

  ddir:                  setdirty;
  bf:                    boolean;


begin
  with sy_sym do
  begin
    INSYMBOL;                           { Gobble up the keyword }
    if sy = identsy then
    begin
      bf := false;
      with sy_ident do
      for pdir := setdirty"first to setdirty"last do
      with dirtab[pdir] do
      begin
        ddir := pdir;
        bf := (STR_MATCH( sy_ident, 0, nam, len ) = 0);
      exit if bf
      end;
      INSYMBOL;                         { Gobble up the dir identifier }
      if not bf then ddir := d_list
    end
    else
      ddir := d_list;

    { Proceed Now }
    case ddir of
      d_summ:  GET_NUMOR_SUMM( true );
      d_diff:  GET_NUMOR_DIFF( true );
      d_list:  GET_NUMOR_LIST( false, 1, true );
      d_keys:  GET_CATALOGUE;
      d_mindex: bf := ILL_DATA_INDEX( true )
    otherwise
    end
  end
end GET_NUMOR_OPE;



[global]
procedure SUMMARY;
{ Use to complete an operation.
  Corresponding to the specific function "data_summary".
}
var
  p: mseq_ptr;

begin
  { Perform the close sequences }
  p := dat_fsqhde;
  while p <> nil do
  with p^ do
  begin
    EXECUTE_MACRO_CODE( mseq_txt, endsy );
    p := mseq_nxt
  end;
  if gdat_iwrt > 0 then
    if gdat_iwrt > 1 then
      WRITELN( ' There are ', gdat_iwrt, ' Data sets written in Data Base.' )
    else
      WRITELN( ' There is 1 Data set written in Data Base.' );
  WRITELN
end SUMMARY;


end GDATA_UTIL.
