{
        *****  CPAS Portable Shell External Environment  *******

*************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               G D A T A   (Get DATA for Diffraction)                  *
*                   using the CPAS SHELL Environment                    *
*                     ( Plot Manager Source File )                      *
*                                                                       *
*                 Version  1.1-A  - - 01-Oct-2006                       *
*                                                                       *
*                                by                                     *
*                                                                       *
*            Pierre Wolfers, Laboratoire de Cristallographie            *
*                                                                       *
*            CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166            *
*                                                                       *
*                       F 38042 GRENOBLE CEDEX 9                        *
*                                                                       *
*                             F R A N C E                               *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}
module GDATA_DRAW;

%include 'PASENV:draw_defs';           { Get the DRAW Library Environment }
%include 'GDASRC:gdata_type_env.pas';  { Get all usefull GDATA/LST/SRC definitions }




  {**********************************************************************************
  *                                                                                 *
  *                        Global  Constants  Declarations                          *
  *                                                                                 *
  **********************************************************************************}


const
  npat_max  =       32;                                { Maximum number of pattern to select }

  icetemp   =   273.16;                                { Ice fusion temperature in Kelvin }


type
  functy = (                                           { * Program function definitions }
             fc_write,                                 { Function to write an output file }
             fc_3dplot,                                { Function to do a 3D plot }
             fc_plot,                                  { Function to do a plot }
             fc_ndia,                                  { Function to specify a diagram number }
             fc_light,                                 { Function to set the Light mode }
             fc_celsius,                               { Function to set celsius degrees for temperature }
             fc_label,                                 { Function for label on pattern plots }
             fc_dir                                    { Function Ctalogue/Directory }
           );

  data_point = record                                  { * Record definition for a pattern point }
    flag:  integer;                                    { Flag for point validity }
    theta,                                             { Theta angle }
    int:   real                                        { Intensity count }
  end;

  pat_ptr = ^pat_rec;                                  { Define a pointer of pattern }

  pat_rec( dim: integer ) = record                     { * Definition of Pattern record }
    next:       pat_ptr;                               { Pointer to next pattern }
    idnt:       integer;                               { Integer Identifier }
    remark,                                            { Remark on the measure }
    dateh,                                             { Pattern date and hour }
    comment:    string( 64 );                          { Pattern comment of 64 character max. }
    ctime,                                             { Counting time }
    lambda,                                            { Wave length }
    amin, amax,                                        { Mini-maxi on 2*theta }
    min,  max,                                         { Minimum and maximum of intensity }
    temp:       real;                                  { Pattern temperature }
    dat:        array[1..dim] of data_point            { The pattern dim points }
  end;





  {**********************************************************************************
  *                                                                                 *
  *                       Global  Variables  Declarations                           *
  *                                                                                 *
  **********************************************************************************}

var
  pat_sel,                                             { Pointer to the selected pattern }
  pat_first,                                           { Pointer to first and last pattern }
  pat_last:     pat_ptr := nil;

  funcs:        set of functy := [];                   { Set of function to execute }

  outty,                                               { Type of output file }
  ndiat,                                               { Number of pattern to select by temperature }
  ndia,                                                { Number of Pattern to Select by number }
  npat:         integer := 0;                          { Number of pattern }

  tbtem:     array[1..npat_max] of real;               { Table of Pattern temperature to select }
  tbnp:      array[1..npat_max] of integer;            { Table of Pattern number to select }

  tbpat:     array[1..2*npat_max] of pat_ptr;          { Table of Pattern pointer to select }
  tbknd:     array[1..2*npat_max] of boolean;          { Table of Marker mode }
  tbcol,                                               { Table of color }
  tbtyp:     array[1..2*npat_max] of integer;          { Table of Marker type }
  tbsiz:     array[1..2*npat_max] of    real;          { Table of Marker type }

  inp_name: string;

  dstatus,                                             { Return status for DRAW Graphic Library }
  irep:         Dint;                                  { Return value for DRAW$DIALOG Procedure }

  spcint,                                              { Intensity spacing for multi patterm plot }
  smin,  smax,                                         { Minimum and Maximum of Selected Pattern }
  tmin,  tmax,                                         { Minimum and Maximum of tempertypature }
  thmin, thmax,                                        { Minimum and Maximum of angle 2*theta }
  gmin,  gmax:            Dfloat := 0.0;               { General minimum and maximum }

  gdrw_x_plt_size, gdrw_y_plt_size,                    { Plot size (sheet/screen/paper) }
  gdrw_space_x, gdrw_space_y, gdrw_space_z,            { Plot space sizes (virtual - world coordinates) }
  gdrw_title,                                          { Plot Title }
  gdrw_x_axis_sz, gdrw_y_axis_sz, gdrw_z_axis_sz,      { Axis sizes }
  gdrw_x_box, gdrw_y_box, gdrw_z_box,                  { Box position in the plot space }
  gdrw_x_unit, gdrw_y_unit, gdrw_z_unit,               { Axis Label strings }
  gdrw_box_color,                                      { Box color (icolor index) }
  gdrw_box_model,                                      { Box model number }
  gdrw_box_thick,                                      { Box line thickness }
  gdrw_x_min, gdrw_x_max,                              { Mini-Maxi for each axis }
  gdrw_y_min, gdrw_y_max,
  gdrw_z_min, gdrw_z_max,
  gdrw_name,                                           { Window plot name }
  gdrw_status:                       ide_ptr;          { GDRW interface Status word }

  gdrw_init_flg: Dbool := false;                       { Flag for Initialize graphic }




  {**********************************************************************************
  *                                                                                 *
  *                               Drawing Procedures                                *
  *                                                                                 *
  **********************************************************************************}


[global]
procedure GDRW_INIT;
begin
end GDRW_INIT;




[global]
procedure GDRW_END;
begin
end GDRW_END;




[global]
procedure GDRW_CREATE_BOX;
begin
end GDRW_CREATE_BOX;





  {**********************************************************************************
  *                                                                                 *
  *                               Drawing Procedures                                *
  *                                                                                 *
  **********************************************************************************}



  procedure INIT_PLOT( flg3d: Dbool );
  {****************************************************}
  {          INITIALYSE THE DRAWING SYSTEM             }
  {****************************************************}
  { The external procedure

      DRAW$DEF_MENU( code: Dint; in_var name: [optional] string );

                  is a procedure to modify by insertion the DRAW$DIALOG
    standard menu setting.

    The code are :
      Menu_Close             Close the currently opened menu.
      Menu_Open              Open the next Standard Menu.
      Menu_Skip              Skip_the next standard (Menu or Entry).
      Menu_New    <Name>     Create and Open a new user menu of specified name.
      Menu_Entry  <Name>     Create a menu entry of specified name.
      Menu_Separ             Create a Separator menu Entry.

    The directives Menu_Entry and Menu_Separ allocate an integer identifier
    to the new menu elements beginning from 1. This identifier is returned by
    the procedure DRAW$DIALOG when the corresponding entry is selected by
    the mouse.
    The Menu_skip can be used with the form n*16+Menu_Skip where n is a skip number
    with the following specificities:
      If n < 0 the skip go to the end of current standard menu else,
      if n > 0 the result is to skip n entry in the current standard menu.
 
  }

  var
    unt: unit_type;
    pxx, pyy: Dfloat;

  begin
    if flg3d then
    begin
      DRAW$SET_MENU_SPC( Menu_Skip );                  { Skip the standard menu "File" }
      DRAW$SET_MENU_SPC( Menu_New,   '&Rotate' );      { Create a New menu "Rotate" }
      DRAW$SET_MENU_SPC( Menu_Entry, '&Omega' );       { Create a New menu entry "Omega" with id = 1 }
      DRAW$SET_MENU_SPC( Menu_Entry, '&Chi' );         { Create a New menu entry "Chi"   with id = 2 }
      DRAW$SET_MENU_SPC( Menu_Entry, '&Phi' );         { Create a New menu entry "Phi"   with id = 3 }
      DRAW$SET_MENU_SPC( Menu_Entry, '&Dist' );        { Create a New menu entry "Dist"  with id = 4 }
      DRAW$SET_MENU_SPC( Menu_Close );                 { Close the user menu "Rotate" }
      if fc_light in funcs then
      begin
        DRAW$SET_MENU_SPC( Menu_New, '&Light' );       { Create a New menu "Light" }
        DRAW$SET_MENU_SPC( Menu_Entry, '&L0 On/off' ); { Create a New menu entry "Light 0 On/off"   with id = 5 }
        DRAW$SET_MENU_SPC( Menu_Entry, '&L1 On/off' ); { Create a New menu entry "Light 1 On/Off"   with id = 6 }
        DRAW$SET_MENU_SPC( Menu_Close )                { Close the user menu "Rotate" }
      end;
      DRAW$SET_MENU_SPC( Menu_Close );                 { Close the main standard menu (barre menu) }
      if fc_light in funcs then
        Draw_Server_Mode := DrwSrv_Do_Lighting         { Require the Lighting mode }
    end;

    DRAW$INIT( pxx, pyy, unt, gdrw_name^.ide_str^ );   { Set the Window Owner Name }
    gdrw_x_plt_size^.ide_flt := pxx;
    gdrw_y_plt_size^.ide_flt := pyy;
    gdrw_init_flg := true                              { Init of DRAW is done }
  end INIT_PLOT;








  procedure MAKE_3D;
  const
    light0_position  =  Dpoint3[15.0, 30.0,  70.0 ];           { Position of the first light }
    light0_amb_color =  Dcolor[ 0.6, 0.6, 0.6, 1.0];           { Color for Ambiant light }
    light0_dif_color =  Dcolor[ 1.0, 1.0, 1.0, 1.0];           { Color for Diffuse light }
    light0_spe_color =  Dcolor[ 1.0, 1.0, 1.0, 1.0];           { Color for Specular light }
(*
    grid_color       =  Dcolor[ 0.9, 0.8, 0.6, 1.0];           { Color of lines use when lighting is disable }
    fill_color       =  Dcolor[ 0.3, 0.3, 0.7, 1.0];           { Filling color }
*)

  var
    dview,                                                     { View distance (cm) }
    ome, chi, phi,                                             { Eulerian Angles }
    xp, yp, zp:      Dfloat;                                   { 3D Picture size }

    bstatus:         Dbool;                                    { Status return word }

    bstlght0,
    bstlght1:        Dbool := true;

    lseg, iseg,                                                { Segments # for Boxe and Angles }
    aseg:            Dint;

    untx, unty, untz,                                          { Axis Unit strings }
    name,
    angmsg:          string( 255 ) := '';                      { Angle Message }


    procedure DISPLAY_SURFACE;
    { Procedure to display the 3D image }
    const
      mi2th =    1.0;                                          { Minimum 2*Theta Spacing }
      shz   =  0.001;                                          { Shift for Z lines }

    var
      x_axe, y_axe, z_axe, box: [static] Dint;
      bx_dir:    [static] array[1..5] of Dint;
      zsh, cmin, cmax, xx, yy, zz:     Dfloat;
      i, j, k, l, m:                  integer;
      p1, p2:                         pat_ptr;  
      ste:                       string( 16 );

      procedure DISPLAY_3D_0;
      { Display the surface as successive block in one surface }
      type
        tbraw( sz: integer ) = record
                                 tb: array[1..sz] of Dfloat
                               end;

      var
        p:       pat_ptr;
        sz:      integer;
        raw:     [static] ^tbraw;
        ver:     Dpoint3;
        i, nr:   integer;

      begin
        DRAW$FILL_ATTR( 2, 1 );                                { Set the Filling attribute mode }
        DRAW$OUT_MODE( 3 );                                    { Set the Filling mode }
        p := pat_first;
        if p <> nil then DRAW$SURFACE( npat, pat_first^.dim, 16 );
        nr := 0;
        while p <> nil do
        with p^ do
        begin
          if raw = nil then NEW( raw, dim*3 );
          i := 0;
          with raw^ do
          for ij := 1 to dim do
          with dat[ij] do
          begin
            i := i + 1; tb[i] := theta;
            i := i + 1; if fc_celsius in funcs then tb[i] := temp - icetemp
                                               else tb[i] := temp;
            i := i + 1; tb[i] :=   int;
          end;
          nr := nr + 1;
          DRAW$SEND_BLOCK( raw^.tb );
          p := next
        end;
        DRAW$COLOR( 1 )
      end  DISPLAY_3D_0;



      procedure DISPLAY_3D_1;
      { Display the surface as successive triangle_strips }
      var
        p1, p2:  pat_ptr;
        sz:      integer;
        ta, tb:   Dfloat;

      begin
        DRAW$FILL_ATTR( 2, 1 );                                { Set the Filling mode }
        DRAW$OUT_MODE( -5 {GL_TRIANGLE_STRIP} );               { Set the Triangle Strip mode }
        p1 := pat_first;                                       { Start from first pattern }
        p2 := p1^.next;                                        { Get the second pattern address }
        while p2 <> nil do
        begin { Loop on the patterns }
          if fc_celsius in funcs then
          begin
            ta := p1^.temp - icetemp;
            tb := p2^.temp - icetemp
          end
          else
          begin
            ta := p1^.temp;
            tb := p2^.temp
          end;
          sz := p1^.dim;
          if p2^.dim < p1^.dim then sz := p2^.dim;
          { Output the first point }
          with p1^, dat[1] do DRAW$PLOT3( theta, ta, int, false );
          for i := 1 to p1^.dim - 1 do                         { Loop on the pattern points }
          begin
           with p2^, dat[i] do DRAW$PLOT3( theta, tb, int, true );
           with p1^, dat[i + 1] do DRAW$PLOT3( theta, ta, int, true )
          end;
          with p2^, dat[dim] do DRAW$PLOT3( theta, tb, int, true );
          DRAW$PLOT3( 0.0, 0.0, 0.0, false );                  { Finish the Triangle Strip block }
          p1 := p2;                                            { Skip to the next patterns }
          p2 := p1^.next
        end
      end DISPLAY_3D_1;



      procedure DISPLAY_3D_2;
      { Display the surface as network of line_strips }
      var
        p:          pat_ptr;
        ts, te:     real;
        bs:         boolean;
        sz, ii, st: integer;

      begin
        DRAW$COLOR( 0.9, 0.8, 0.6 );
(*      DRAW$COLOR( grid_color ); *)
        DRAW$FILL_ATTR( 1, 1 );                                { Set the Hollow mode }
        { Plot each Pattern }
        p := pat_first;                                        { Start from first pattern }
        with p^ do
          ts := (dat[dim].theta - dat[1].theta)/(dim - 1);     { Compute the 2*theta step }
        st := TRUNC( mi2th/ts );
        if st*ts < mi2th then st := st + 1;                    { Get the2*theta spacing }
        sz := maxint;
        while p <> nil do
        with p^ do
        begin { Loop on the patterns }
          if dim < sz then sz := dim;
          if fc_celsius in funcs then te := temp - icetemp
                                 else te := temp;
          with dat[1] do
            DRAW$PLOT3( theta, te, int + zsh, false );
          for ij := 2 to dim do
            with dat[ij] do
              DRAW$PLOT3( theta, te, int + zsh, true );
          p := next
        end;
        { Plot for each theta the variation with temperature }
        ii := 1;
        while ii <= sz do
        begin
          bs :=    false;
          p := pat_first;
          while p <> nil do
          with p^, dat[ii] do
          begin
          if fc_celsius in funcs then te := temp - icetemp
                                 else te := temp;
            DRAW$PLOT3( theta, te, int + zsh, bs );
            bs := true;
            p := next
          end;
          ii := ii + st
        end;
        { Finish the Last curve }
        DRAW$PLOT3( 0.0, 0.0, 0.0, false )
      end DISPLAY_3D_2;


    begin { DISPLAY_SURFACE }

      if fc_light in funcs then                        { When the lighting mode is used }
      begin                                            { Define the Lights }
        lseg := DRAW$NEW_SEG( 2 );                     { Open/Create the light segement }
(*
        DRAW$LIGHT_ON_OFF( 1, 0 );                     { Eteint la lampe # 1 }
*)
        bstlght0 := true;
        bstlght1 := true;

        DRAW$LIGHT_ON_OFF( 0, 1 );                     { Allume la lampe # 0 }

        DRAW$LIGHT_DEFINE( 0, LIGHT_POSITION,       light0_position  );
        DRAW$LIGHT_DEFINE( 0, LIGHT_AMBIENT_COLOR,  light0_amb_color );
        DRAW$LIGHT_DEFINE( 0, LIGHT_DIFFUSE_COLOR,  light0_dif_color );
        DRAW$LIGHT_DEFINE( 0, LIGHT_SPECULAR_COLOR, light0_spe_color );

        DRAW$LIGHT_ON_OFF( 1, 1 );                     { Allume la lampe # 1 }

        DRAW$LIGHT_DEFINE( 1, LIGHT_POSITION,       Dpoint3[ 46.0,  0.0,  30.0] );
        DRAW$LIGHT_DEFINE( 1, LIGHT_AMBIENT_COLOR,  Dcolor[ 0.4, 0.4, 0.4, 1.0] );
        DRAW$LIGHT_DEFINE( 1, LIGHT_DIFFUSE_COLOR,  Dcolor[ 0.8, 0.8, 0.8, 1.0] );
        DRAW$LIGHT_DEFINE( 1, LIGHT_SPECULAR_COLOR, Dcolor[ 1.0, 1.0, 0.9, 1.0] );


        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_SPECULAR_COLOR, Dcolor[1.0, 1.0, 1.0, 0.55] );
        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_SHININESS, Dvalue[100.0] );
        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_EMISSION_COLOR, Dcolor[0.0, 0.0, 0.0, 1.0] );
        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_DIFFUSE_COLOR, Dcolor[1.0, 0.9, 0.2, 1.0] );

        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE, MAT_SPECULAR_COLOR, Dcolor[1.0, 1.0, 1.0, 0.55] );
        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE, MAT_SHININESS, Dvalue[100.0] );
        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE, MAT_EMISSION_COLOR, Dcolor[0.0, 0.0, 0.0, 1.0] );
        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE, MAT_DIFFUSE_COLOR, Dcolor[1.0, 0.9, 0.2, 1.0] );

        DRAW$COLOR( 1.0, 1.0, 0.5, 1.0 );
        DRAW$SPHERE( light0_position[1], light0_position[2], light0_position[3], 2.0, 32, 32, 4 );

        DRAW$SEG_END
      end;

      aseg := DRAW$NEW_SEG( 3 );                       { Open/Create the Angle segment }

      ome :=  90.0; chi :=  80.0; phi := -100.0;
      { Force the initial value of angle and display them }
      WRITEV( angmsg, ' Ome = ', ome:8:2, ', Chi = ', chi:8:3, ', Phi = ', phi:8:3 );
      DRAW$COLOR( 0.0, 0.0, 0.0, 1.0 );
      DRAW$STRING( -22.0, -17.5, 0.0, 1.0, angmsg );
      DRAW$SEG_END;                                    { Close the Angle Segment }

      iseg := DRAW$NEW_SEG( 4 );                       { Open/Create the Box/Plot Segment }
      { Create the Orientation Matrix of the Box/Plot }

      DRAW$ROTATE_SEG3( iseg, 0.0, 0.0, 0.0, ome, chi, phi );
      DRAW$SEG_ORIENTATION( iseg, 0.0, 0.0, 0.0 );

      { The flag parameter (last one) must be flag the axis characteristics as this :

        the value must be a set of this following constantes :

        *****   Axis Flags   *****

        Axis_Log       =       1; (* Axis is in Log10 Coordinate *)
        Axis_Left      =       2; (* Ticks and displayed value at left position ( else at right ) *)
        Axis_VOther    =       4; (* Value are in other side (than ticks) *)
        Axis_NFrs      =       8; (* Does not display the first Ticks *)
        Axis_NFVl      =      16; (* Does not display the first Value *)
        Axis_NLsT      =      32; (* Does not display the last Ticks *)
        Axis_NLVl      =      64; (* Does not display the last Value *)
        Axis_NZer      =     128; (* Does not display the Zero Ticks *)
        Axis_NZVl      =     256; (* Does not display the Zero Value *)
        Axis_Arrow     =     512; (* Put an arrow at the end of axis *)
        Axis_ArroF     =    1024; (* Fill mode for Axis_Arrow *)
        Axis_Tilte     =    2048; (* Axis character are tilted with ticks *)
        Axis_VPath     =    4096; (* String Path (2 bits) for Ticks Value *)
        Axis_Upath     =   16384; (* String Path (2 bits) for Axis Unit *)
        Axis_SpLim     =   65536; (* Axis Limit auto-adaptation when set *)
        Axis_VLeft     =  131072; (* Left Alignement for Unit String *)
        Axis_VRight    =  262144; (* Right Alignement for Unit String *)
        Axis_ULeft     =  524288; (* Left Alignement for Unit String *)
        Axis_URight    = 1048576; (* Right Alignement for Unit String *)
        Axis_UMiddle   = 2097152; (* Set Unit String at the Middle of Axis *)


        *****   Axis Plot Directives and Flags for Draw_Plot   *****

        Axis_Plot_X    =     1; (* Plot an X axis *)
        Axis_Plot_Y    =     2; (* Plot an Y axis *)
        Axis_Plot_Z    =     3; (* Plot a  Z axis *)
        Axis_PFlg_TVl  =     4; (* Ticks values and Unit String will be Displayed *)
        Axis_PFlg_TPrm =     8; (* The ticks left and right sides are permutted *)
        Axis_PFlg_TSym =    16; (* The Ticks must be symmetrized *)
        Axis_PFlg_XEnd =    32; (* Plot Axis with a shift of x axis size *)
        Axis_PFlg_YEnd =    64; (* Plot Axis with a shift of Y axis size *)
        Axis_PFlg_ZEnd =   128; (* Plot Axis with a shift of Z axis size *)
        Axis_PFlg_XPos =   256; (* Plot Axis at specified X shift *)
        Axis_PFlg_YPos =   512; (* Plot Axis at specified Y shift *)bs := (
        Axis_PFlg_ZPos =  1024; (* Plot Axis at specified Z shift *)
        Axis_Pflg_SymU =  2048; (* Apply a mU (mirror u) to tick direction *)
        Axis_Pflg_SymV =  4096; (* Apply a mV (mirror v) to tick direction *)
        Axis_Pflg_SymW =  8192; (* Apply a mW (mirror w) to tick direction *)
        Axis_Pflg_Rotv = 16384; (* Apply an additional rotation of 180 to value and unit strings *)
 
      }

      cmin := tmin; cmax := tmax;
      if fc_celsius in funcs then
      begin
        cmin := cmin - icetemp;
        cmax := cmax - icetemp;
        ste:= 'C'
      end
      else ste := 'K';
      ste := ste||' (Temperature)';

      zsh := (gmax - gmin)*shz;

      x_axe := DRAW$NEW_AXIS( Dpoint3[ 1.0, 0.0, 0.0],
                              Dpoint3[ 0.0, 1.0, 1.0],
                              40.0, thmin, thmax, -3,                                Axis_UMiddle );
      y_axe := DRAW$NEW_AXIS( Dpoint3[ 0.0, 1.0, 0.0],
                              Dpoint3[-1.0, 0.0, 1.0],
                              30.0, cmin,  cmax,  -3,  Axis_UMiddle + Axis_VPath*0 + Axis_UPath*0 );
      z_axe := DRAW$NEW_AXIS( Dpoint3[ 0.0, 0.0, 1.0],
                              Dpoint3[ 1.0, 1.0, 0.0],
                              28.0,  gmin, gmax,  -3,  Axis_Arrow   + Axis_ArroF  +  Axis_UMiddle );


      Draw$Set_Axis_Unit( x_axe, ' (2*Theta)' );
      Draw$Set_Axis_Unit( y_axe, ste );
      Draw$Set_Axis_Unit( z_axe, 'Cps (Intensity)' );

      Draw$Set_Axis_Value( x_axe, 2, 112,,,,, 4,  5, 2 );      { Set Value Font, Field and decimal }
      Draw$Set_Axis_Value( y_axe, 2, 112,,,,, 4,  5, 2 );      { Set Angle, Value Font, Field and decimal }
      Draw$Set_Axis_Value( y_axe, 4,   2,, 0.0 );              { Set Angle, fbs := (or Unit String }
      Draw$Set_Axis_Value( z_axe, 2, 112,,,,, 4,  5, 2 );      { Set Value Font, Field and decimal }

      bx_dir[1] := Axis_PFlg_TVl  +                    { Tick Values, }
                   Axis_Plot_X;                        { Plot X axis }
      bx_dir[2] := Axis_PFlg_TVl  +                    { Tick Values, }
                   Axis_PFlg_XEnd + Axis_Plot_Y;       { at end'X axis, Plot Y axis }
      bx_dir[3] := Axis_PFlg_TVl  + Axis_Pflg_Rotv +   { Tick Values, 180 String rotate, }
                   Axis_PFlg_SymW + Axis_PFlg_TPrm +   { mw symetry Ope., Left/right Permut, }
                   Axis_Plot_Y;                        { Plot Y axis }
      bx_dir[4] := Axis_PFlg_TVl  + Axis_Pflg_Rotv +   { Tick Values, 180 String rotate, }
                   Axis_PFlg_SymW + Axis_PFlg_TPrm +   { mw symetry Ope., Left/right Permut, }
                   Axis_PFlg_YEnd + Axis_Plot_X;       { at end'Y axis, Plot X axis }

      bx_dir[5] := Axis_PFlg_TVl  + Axis_Plot_Z;       { Tick Values, Plot Y axis }

      DRAW$COLOR( 1 );

      { Create the Axis Box }
      box := DRAW$NEW_BOX( Dpoint3[ -18.0, -20.0, -10.0], x_axe, y_axe, z_axe, bx_dir );

      DRAW$LINE_ATTR( 1, 2.0 );                        { Set continue line type }
      DRAW$PLOT_BOX( box );                            { Draw the Axis box }

      DRAW$COLOR( 1 );
      DRAW$OPEN_BOX( box );                            { Open the box to plot the curve }

      if fc_light in funcs then
      begin
        DRAW$COLOR( 0.55, 0.55, 0.70, 1.0 );
        DISPLAY_3D_0
      end
      else
      begin
        DRAW$COLOR( 0.3, 0.3, 0.7 );
(*      DRAW$COLOR( fill_color );    *)                { Set the surface color }
        DISPLAY_3D_1                                   { Form the Triangle fillings }
      end;

      if not (fc_light in funcs) then DISPLAY_3D_2;    { Form a grid of line }

      DRAW$CLOSE_BOX;                                  { Close the Axis box }
      DRAW$SEG_END;                                    { Close the boxe/plot segment }
      DRAW$VIEW( dview );
    end DISPLAY_SURFACE;


    procedure ROTATE_PLOT;
    var
      pmat, mat: array[1..3,1..3] of Dfloat;

    begin
      DRAW$UPDATE_SEG( aseg {, 0} );

      WRITEV( angmsg, ' Ome = ', ome:8:2, ', Chi = ', chi:8:3, ', Phi = ', phi:8:3 );
      DRAW$STRING( -22.0, -17.5, 0.0, 1.0, angmsg );

      DRAW$SEG_END;

      DRAW$VIEW( dview );
      DRAW$SEG_UNIT_TRANSF( iseg );
      DRAW$ROTATE_SEG3( iseg,  0.0,  0.0,  0.0, ome, chi, phi )
    end ROTATE_PLOT;


  begin { MAKE_3D }
    INIT_PLOT( true );
    DRAW$LIGHT_ON_OFF( 0, 1 );                         { Allume la lampe # 0 }
    DRAW$LIGHT_ON_OFF( 1, 1 );                         { Allume la lampe # 1 }
    bstatus := DRAW$PICTURE3( inp_name, 75.0, 50.0, 75.0, false, false );

    DISPLAY_SURFACE;

    repeat
      irep := DRAW$DIALOG( 1 );
      if irep < 0 then goto ET_Stop;
      case irep of
        1: { Euler Angle Omega Function }
          begin
            dstatus := DRAW$GET_VALUE( 'Eulerian Angle Omega', ome, -180.0, 180.0 );
            if dstatus <> 1 then ome :=  90.0;
            ROTATE_PLOT
          end;

        2: { Euler Angle chi Function }
          begin
            dstatus := DRAW$GET_VALUE( 'Eulerian Angle Chi', chi, -180.0, 180.0 );
            if dstatus <> 1 then chi :=  80.0;
            ROTATE_PLOT
          end;

        3: { Euler Angle Phi Function }
          begin
            dstatus := DRAW$GET_VALUE( 'Eulerian Angle Phi', phi, -180.0, 180.0 );
            if dstatus <> 1 then phi := -40.0;
            ROTATE_PLOT
          end;

        4: { View Distance }
          begin
            dstatus := DRAW$GET_VALUE( 'Distance of View', dview, 0.0, 200.0 );
            if dstatus <> 1 then dview := 0.0;
            ROTATE_PLOT
          end;

        5: { Light 0 On/off }
          begin   bstlght0 := not  bstlght0; DRAW$LIGHT_ON_OFF( 1, ORD( bstlght0 ) )  end;

        6: { Light 1 On/off }
          begin   bstlght1 := not  bstlght1; DRAW$LIGHT_ON_OFF( 1, ORD( bstlght1 ) )  end;

      end { case }
    until irep = 0;

  ET_Stop:
    { End of Graphic Session }
    DRAW$END
  end MAKE_3D;



  procedure MAKE_2D( nd: integer );
  const
    max_nbc = 2;

  var

    bstatus: boolean;
    i, j, k, irep, box: Dint;
    cmin, cmax,                                        { Minimum and Maximum for Intensities axis }
    shf, dlt:        Dfloat;

    lseg, iseg,                                        { Segments # for Boxe and Angles }
    dstatus:         Dint;                             { Entry number for DRAW$DIALOG Return }




    procedure PLOT_A_PATTERN( p: pat_ptr; shift: real );
    var
      slab: string( 32 );
      bs: boolean;
      xx, yy, zz, te: real;
      ir: integer;

    begin
      bs := false;
      with p^ do
        for ij := 1 to dim do
        begin
          with dat[ij] do
            DRAW$PLOT( theta, int + shift, bs );
          bs := true
        end;

      DRAW$RPLOT( 0.0, 0.0, false );
      if fc_label in funcs then
      with p^, dat[dim] do
      begin
        ir := DRAW$CONV_BOX( false, theta, int + shift, 0.0, xx, yy, zz  );
        te := temp;
        if fc_celsius in funcs then te := te - icetemp;
        WRITEV( slab, ' T = ', temp:6:2 );
        DRAW$STRING( xx + 1.0, yy + 0.5, 0.0, 0.4, slab )
      end
    end PLOT_A_PATTERN;


  begin { MAKE_2D }
    INIT_PLOT( false );
    bstatus := DRAW$PICTURE( inp_name, 28.0, 20.0, false, false );

    spcint := ABS( spcint );
    { Set the default value on incredible shift values }
    if (spcint > 2.0) or (spcint < 0.02) then spcint := 0.20;
    dlt := spcint*(smax - smin);
    cmin := smin; cmax := smax + dlt*(ndia - 1);
    if fc_celsius in funcs then
    begin  cmin := cmin - icetemp; cmax := cmax - icetemp  end;

    box := Draw$Easy_Box_2D( 2.0, 2.5, 24.0, 16.0, thmin, thmax, cmin, cmax, ' 2*theta', 'Counts', 2 );
    DRAW$COLOR( 1 );
    DRAW$PLOT_BOX( box );                              { Plot the Axis Box }
    DRAW$OPEN_BOX( box );

    shf := 0.0;
    for id := 1 to ndia do
    begin
      DRAW$COLOR( tbcol[id] );
      if tbknd[id] then
      begin { Marker Mode }
        DRAW$OUT_MODE( 2 );
        DRAW$MARKER_ATTR( tbtyp[id], tbsiz[id] )
      end
      else
      begin
        DRAW$OUT_MODE( 1 );
        DRAW$LINE_ATTR( tbtyp[id], tbsiz[id] )
      end;
      PLOT_A_PATTERN( tbpat[id], shf );
      shf := shf + dlt
    end;

    repeat
      irep := DRAW$DIALOG( 1 );
      if irep < 0 then goto ET_Stop;

    until irep = 0;

  ET_Stop:

    { End of Graphic Session }
    DRAW$END
  end MAKE_2D;





  {**********************************************************************************
  *                                                                                 *
  *                           Main Control Procedures                               *
  *                                                                                 *
  **********************************************************************************}


  [global]
  function  DRAW_PLOT_SETTING: integer;
  const
    mdnam = 'DRST';

  begin { DRAW_PLOT_SETTING }
   { POP_STREXPR( gdrw_name );                  { Get the Plot Title }

    DRAW_PLOT_SETTING := 0
  end DRAW_PLOT_SETTING;



  [global]
  function  DRAW_PLOT_2D: integer;
  const
    mdnam = 'DR2D';

  begin

    DRAW_PLOT_2D := 0
  end DRAW_PLOT_2D;


  [global]
  function  DRAW_PLOT_3D: integer;
  const
    mdnam = 'DR3D';

  begin

    DRAW_PLOT_3D := 0
  end DRAW_PLOT_3D;


  [global]
  procedure DRAW_PARM_SETUP( procedure DEF_ENTRY( in_var  nam: string;
                                                          knd: ide_kinds;
                                                          fnc: wennum;
                                                         npa1: integer;
                                                         npa2: integer := -1 ) );
  { Define all user/drawing system predefined parameters }

    function DEF_IPARAM( in_var nam: string; ini_val: integer ): ide_ptr;
    begin
      DEF_ENTRY( nam, ide_parint, blt_noop, 0 );
      sy_idenew^.ide_int := ini_val;
      DEF_IPARAM := sy_idenew
    end DEF_IPARAM;


    function DEF_FPARAM( in_var nam: string; ini_val: gdreal ): ide_ptr;
    begin
      DEF_ENTRY( nam, ide_parflt, blt_noop, 0 );
      sy_idenew^.ide_flt := ini_val;
      DEF_FPARAM := sy_idenew
    end DEF_FPARAM;


    function DEF_SPARAM( in_var nam, ini_val: string ): ide_ptr;
    begin
      DEF_ENTRY( nam, ide_parstr, blt_noop, 0 );
      if ini_val.length > 0 then begin
                                   NEW( sy_idenew^.ide_str, ini_val.length );
                                   sy_idenew^.ide_str^ := ini_val
                                 end
                                 else sy_idenew^.ide_str := nil;
      DEF_SPARAM := sy_idenew
    end DEF_SPARAM;

  begin { DRAW_PARM_SETUP }
    gdrw_name      := DEF_SPARAM(       'drw$wname',     'Gdata_Plot'  );
    gdrw_x_plt_size:= DEF_FPARAM(  'drw$x_plt_size',               0.0 );
    gdrw_y_plt_size:= DEF_FPARAM(  'drw$y_plt_size',               0.0 );
    gdrw_space_x   := DEF_FPARAM(     'drw$space_x',              20.0 );
    gdrw_space_y   := DEF_FPARAM(     'drw$space_x',              16.0 );
    gdrw_space_z   := DEF_FPARAM(     'drw$space_x',              16.0 );
    gdrw_title     := DEF_SPARAM(       'drw$title',                '' );
    gdrw_x_axis_sz := DEF_FPARAM( 'drw$x_axis_size',              24.0 );
    gdrw_y_axis_sz := DEF_FPARAM( 'drw$y_axis_size',              16.0 );
    gdrw_z_axis_sz := DEF_FPARAM( 'drw$z_axis_size',              16.0 );
    gdrw_x_box     := DEF_FPARAM(       'drw$x_box',               2.0 );
    gdrw_y_box     := DEF_FPARAM(       'drw$y_box',               2.0 );
    gdrw_z_box     := DEF_FPARAM(       'drw$z_box',               0.0 );
    gdrw_x_unit    := DEF_SPARAM(      'drw$x_unit',         '2*Theta' );
    gdrw_y_unit    := DEF_SPARAM(      'drw$y_unit',     'Intensities' );
    gdrw_z_unit    := DEF_SPARAM(      'drw$z_unit', 'Temperature (K)' );
    gdrw_box_color := DEF_IPARAM(   'drw$box_color',                 1 );
    gdrw_box_model := DEF_IPARAM(   'drw$box_model',                 1 );
    gdrw_box_thick := DEF_FPARAM(   'drw$box_thick',               1.0 );
    gdrw_x_min     := DEF_FPARAM(       'drw$x_min',               0.0 );
    gdrw_x_max     := DEF_FPARAM(       'drw$x_max',               0.0 );
    gdrw_y_min     := DEF_FPARAM(       'drw$y_min',               0.0 );
    gdrw_y_max     := DEF_FPARAM(       'drw$y_max',               0.0 );
    gdrw_z_min     := DEF_FPARAM(       'drw$z_min',               0.0 );
    gdrw_z_max     := DEF_FPARAM(       'drw$z_max',               0.0 );
    gdrw_status    := DEF_IPARAM(      'drw$status',                 0 );
  end DRAW_PARM_SETUP;


end GDATA_DRAW.

