{
*************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*                                                                       *
*          * * *    L I S P    I n t e r p r e t e r    * * *           *
*                                                                       *
*                                                                       *
*                ***  LISP SMALL ROUTINES MODULE   ***                  *
*                                                                       *
*       by :                                                            *
*                                                                       *
*           P. Wolfers                                                  *
*               c.n.r.s.,                                               *
*               Laboratoire de Cristallographie,                        *
*               B.P.  166 X   38042  Grenoble Cedex,                    *
*                                              FRANCE.                  *
*                                                                       *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************


/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////
}

{  Version 1.2-B (or Upper)  of  E - L I S P     System  }
{***********    CPAS  Version   **************}


{*********************************************************}
{**** Small LISP routines to handle the LISP elements ****}
{*********************************************************}

module LISP_ROUTINES( input, output );


%include 'LISPSRC:lisp_env';  { Get the Lisp Environment Definitions }



[global]
function NXT_PAR( var list: obj_ref ): obj_ref;
{ function to run along a list }
begin
  if list.typ = doublety then
  begin
    NXT_PAR := list.db^.car;
    list := list.db^.cdr
  end
  else
  begin
    NXT_PAR := list;
    list := obj_nil
  end
end NXT_PAR;



[global]
function NXD_PAR( var list: obj_ref ): obj_ref;
{ Variant of NXT_PAR To take off the list element }
var
  tmp: obj_ref;

begin
  tmp := list;
  if tmp.typ = doublety then
  begin
    NXD_PAR := tmp.db^.car;
    list := tmp.db^.cdr;
    tmp.db^.car := obj_nil;
    tmp.db^.cdr := dbl_free;
    dbl_free := tmp
  end
  else
  begin
    NXD_PAR := list;
    list := obj_nil
  end
end NXD_PAR;



[global]
function F_CONS( ob1, ob2: obj_ref ): obj_ref;
var
  p: obj_ref;
begin
  p := DOUBLET_ALLOC;
  with p.db^ do
  begin
    car := ob1; cdr := ob2
  end;
  F_CONS := p
end F_CONS;



[global]
function F_DBL_FREE( var stk: obj_ref ): obj_ref;
var
  obj, res: obj_ref;

begin
  if stk.typ = doublety then
  begin
    obj         := stk;
    res         := obj.db^.car;
    stk         := obj.db^.cdr;
    obj.db^.car := obj_nil;
    obj.db^.cdr := dbl_free;
    dbl_free    := obj
  end else res  := obj_nil;
  F_DBL_FREE := res
end F_DBL_FREE;



[global]
function F_CONS_INT( v: integer; lnk: obj_ref ): obj_ref;
var
  o: obj_ref;
begin
  o.flg := flg_def;
  o.typ := intty;
  o.int := v;
  F_CONS_INT := F_CONS( o, lnk )
end F_CONS_INT;



[global]
function F_CONS_STR( v: lisp_s_ptr; lnk: obj_ref ): obj_ref;
var
  o: obj_ref;
begin
  o := obj_nil;
  if v <> nil then
  begin
    o.typ    := strty;
    o.nam    := v
  end;
  F_CONS_STR := F_CONS( o, lnk )
end F_CONS_STR;



[global]
function F_CONS_FLT( v: lisp_real; lnk: obj_ref ): obj_ref;
var
  o: obj_ref;
begin
  o.flg := flg_def;
  o.typ := flty;
  o.flt := v;
  F_CONS_FLT := F_CONS( o, lnk )
end F_CONS_FLT;



[global]
function F_CONS_SFLT( v: single; lnk: obj_ref ): obj_ref;
var
  o: obj_ref;
begin
  o.flg := flg_def;
  o.typ := flty;
  o.flt := v;
  F_CONS_SFLT := F_CONS( o, lnk )
end F_CONS_SFLT;



[global]
function F_CONS_MEM( v: mem_ptr; lnk: obj_ref ): obj_ref;
var
  o: obj_ref;
begin
  o.flg    := flg_def;
  o.typ    := areatyp;
  o.mem    := v;
  F_CONS_MEM := F_CONS( o, lnk )
end F_CONS_MEM;



[global]
procedure GET_LISP_STR_REF( var r:       $wild_pointer;
                            var len:           integer;
                            var ref:            string;
                                obj:           obj_ref;
                                bref: boolean := false );
{ Routine to return the address of a first character of a string (in r formal).
  The string must be in the obj object.
  len is the returned length of the string,
  ref is a temporary string to use to store character or number string,
  When bref is true, for a string object (obj) the returned length is the string capacity
               else len is the actual length of the string.
}
begin
  r   := nil;
  len :=   0;
  with obj do
    case typ of
    nullty:  ;
    atomety: with at^ do
               if ats.nam <> nil then
               begin  r := ats.nam^.body"address; len := ats.nam^.length  end;
    strty:   if nam <> nil then
             begin
               r := nam^.body"address;
               if bref then len := nam^.capacity
                       else len := nam^.length
             end;
    charty:  begin
               ref[1] := ch;
               r := ref.body"address; len := 1
             end;
    intty:   begin
               WRITEV( ref, int:0 );
               r := ref.body"address; len := ref.length
             end;
  otherwise
    EXEC_ERROR( 'GRST', 55, e_severe )
  end
end GET_LISP_STR_REF;



[global]
function GET_LISP_STR_LEN( obj: obj_ref ): integer;
var
  len: integer;

begin
  len :=   0;
  with obj do
    case typ of
    nullty:  ;
    atomety: with at^ do
               if ats.nam <> nil then len := ats.nam^.length;
    strty:   if nam <> nil then len := nam^.length;
    charty:  len := 1;
  otherwise
    len := -1
  end;
  GET_LISP_STR_LEN := len
end GET_LISP_STR_LEN;



[global]
procedure GET_STRING( var res: string; obj: obj_ref; in_var sdef: string );
{ Get a LISP String value as a Pascal Standard String }
var
  l: integer;
  p: body_s_ptr;
  s: string( 16 );

begin
  GET_LISP_STR_REF( p, l, s, obj );
  if p = nil then
    res := sdef
  else
  begin
    if res.capacity < l then l := res.capacity;
    res.length := l;
    for i := 1 to l do  res.body[i] := p^[i]
  end
end GET_STRING;



[global]
function GET_LISP_STR( obj: obj_ref; in_var sdef: string ): obj_ref;
{ Get a LISP String value as a LISP String }
var
  l: integer;
  p: body_s_ptr;
  r: obj_ref;
  s: string( 16 );

begin
  GET_LISP_STR_REF( p, l, s, obj );
  r := obj_nuls;
  if p = nil then r.nam := NEW_LISP_STRINGV( sdef )
             else r.nam := NEW_LISP_STRINGV( p^, l );
  GET_LISP_STR := r
end GET_LISP_STR;



[global]
function STRING_INDEX( ob1, ob2: obj_ref;
                        nb: integer :=  0;
                        ip: integer := -1;
                        nc: boolean := false
                     ): integer;
{ General LISP Index Routines }
var
  sp1, sp2: body_s_ptr;
  st1, st2: string( 16 );
  l1, l2:   integer;

begin
  GET_LISP_STR_REF( sp1, l1, st1, ob1 );
  GET_LISP_STR_REF( sp2, l2, st2, ob2 );
  STRING_INDEX := LIB_REF_INDEX( sp1, l1, sp2, l2, nb, ip, nc )
end STRING_INDEX;



[global]
procedure STRING_SUBSTR( var res: obj_ref; obj: obj_ref; i, j: integer );
{ Perform a sub-string extraction from a given string, if res is a string
  reference then this string is used to store the STRING_SUBSTR result }
var
  p: body_s_ptr;
  l: integer;
  s: string( 16 );

begin
  GET_LISP_STR_REF( p, l, s, obj );
  if p <> nil then
  begin
    if i <= 0 then i := 1;
    if i <= l then
    begin
      p := p^[i]"address;        { Get the substring relative address }
      l := l - i + 1;            { Update the string size }
      if (j < 1) or (j > l) then j := l; { Check j values }
      if (res.typ <> strty) or (res.nam = nil) then
      begin { We must Create a new String }
        res := obj_nuls;
        res.nam := NEW_LISP_STRINGV( p^, j )
      end
      else
      with res.nam^ do
      begin { We use the given String }
        if j > capacity then j := capacity;
        length := j;
        for i := 1 to j do  body[i] := p^[i]
      end
    end
    else
      if (res.typ <> strty) or (res.nam = nil) then res := obj_nuls
                                               else res.nam^.length := 0
  end
  else
    if (res.typ <> strty) or (res.nam = nil) then res := obj_nuls
                                             else res.nam^.length := 0
end STRING_SUBSTR;



[global]
procedure STRING_CONCAT( var res: obj_ref; ll: obj_ref );
{ Perform a multi. string concatenation from a given string, if res is a string
  reference then this string is used to store the STRING_CONCAT result }
const
  max_concat = 64;

var
  pst:               body_s_ptr;
  i, j, len, n, siz: integer;
  scv:               string( 16 );
  stbl:              array[1..max_concat] of obj_ref;

begin
  len :=  0;
  n   :=  0;
  while (ll.typ = doublety) and (n < max_concat) do
  begin
    n := n + 1;
    stbl[n] := F_EVAL( NXT_PAR( ll ) );
    len := len + GET_LISP_STR_LEN( stbl[n] )
  end;
  { Now len is Required String Size }
  if (res.typ <> strty) or (res.nam = nil) then
  begin { We Create a new LISP String of computed size }
    res := obj_nuls;
    if len > 0 then res.nam := LISP_STRING_ALLOC( len );
  end;
  if res.nam <> nil then
  with res.nam^ do
  begin
    if capacity < len then len := capacity; { Adjust to allocated size }
    i := 1;
    j := 0;
    while (i <= n) and (j < capacity) do
    begin
      { Get the string reference }
      GET_LISP_STR_REF( pst, siz, scv, stbl[i] );
      if siz > 0 then
      begin
        if j + siz > capacity then siz := capacity - j;
        for k := 1 to siz do
        begin { Copy/Append the string }
          j := SUCC( j );
          body[j] := pst^[k]
        end
      end;
      i := SUCC( i )
    end;
    length := j
  end
end STRING_CONCAT;



[global]
function CHECK_FILE_SPC_TYPE( fnam: obj_ref ): boolean;
var
  pfn:      body_s_ptr;
  lfn, isp: integer;
  scv:      string( 32 );
  tbs:      array[1..32] of byte;

begin
  GET_LISP_STR_REF( pfn, lfn, scv, fnam );
  if lfn > 0 then
  begin
    STRING_LOOK_SEPAR( pfn, lfn, ':]/\.', tbs, isp );
    if isp > 1 then
      if tbs[isp-1] <> 5 then isp := 0
  end;
  CHECK_FILE_SPC_TYPE := (isp = 0)
end CHECK_FILE_SPC_TYPE;



[global]
function PARSE_FILE_SPC( fspc, lre: obj_ref ): obj_ref;
{ Routine to get SUBSTr parameters to extract from a file specification string <fspc> :
  a device specification,
  a directory path,
  a file name,
  a file type
  and possible file version number ().
  The return value is a list of ((<
}
var
  psp:                               body_s_ptr;
  i, iex, inm, lex, lnm, lph, lsp:      integer;
  csp:                    [static] string( 32 );
  tbs:                     array[1..64] of byte;

  procedure SET_PARM_ELEM( p, l: integer );
  begin
    if l <= 0 then
    begin
      SET_PARM_OBJ( obj_nil ); SET_PARM_OBJ( obj_nil )
    end
    else
    begin
      SET_PARM_INT( p ); SET_PARM_INT( l )
    end
  end SET_PARM_ELEM;


begin
  GET_LISP_STR_REF( psp, lsp, csp, fspc );
  if lsp > 0 then
  begin
    STRING_LOOK_SEPAR( psp, lsp, ':]/\.', tbs, i );
    lex := 0;
    lph := 0;

    if (i > 1) and (tbs[i-1] = 5) then  { "." for Extension Specified }
    begin
      iex := tbs[i];                    { The "." is the first char of the extension }
      lex := lsp + 1 - iex;
      i   := i - 2
    end
    else iex := lsp + 1;                { iex is the index of a virtual "." }

    if (i > 1) and (tbs[i-1] <= 4) then { "/" or "\" or ":" or "]" }
      { Separation between the file path and the file name }
      lph := tbs[i];                    { The file path is ended by the separator }

    inm := lph + 1;                     { The file name index follow the end of Path }
    lnm := iex - inm;                   { Extract the file name size }

    currobj := lre;              { Form the resulting list }
    SET_PARM_ELEM(   1, lph );   { Set the Path string SUBSTR parm }
    SET_PARM_ELEM( inm, lnm );   { Set the Name string SUBSTR parm }
    SET_PARM_ELEM( iex, lex )    { Set the extension string SUBSTR parm }
  end
  else lre := obj_nil;
  PARSE_FILE_SPC := lre
end PARSE_FILE_SPC;



[global]
procedure LST_PUT_LISP_STR( pst: lisp_s_ptr; l: integer := 0 );
begin
  if pst <> nil then
  with pst^ do
  begin
    if l <= 0 then l := length;
    for i := 1 to l do  LST_PUT_CHAR( body[i] )
  end
end LST_PUT_LISP_STR;



[global]
procedure STRING_CV_IS( var ob:    obj_ref;
                            iv:    integer;
                             f, b: integer := 0 );
var
  s: string( 32 );

begin
  WRITEV( s, iv:f:b );
  ob     := obj_nuls;
  ob.nam := NEW_LISP_STRINGV( s )
end STRING_CV_IS;



[global]
procedure STRING_CV_RS( var ob:        obj_ref;
                            rv:        lisp_real;
                             f, d, sg: integer := 0 );
var
  s: string( 64 );

begin
  WRITEV( s, rv:f:d:sg );
  ob     := obj_nuls;
  ob.nam := NEW_LISP_STRINGV( s )
end STRING_CV_RS;



[global]
function INTVAL( obj: obj_ref ): integer;
var
  iv: integer;
  st: [static] string( 64 );

begin
  case obj.typ of
    nullty, truety:
      if logint_mode then INTVAL := ORD( obj.typ = truety )
      else INTVAL := INTVAL( F_EXEC_ERROR( 'INTV', 51, e_error ) );
    intub,
    intsb,
    intuw,
    intsw,
    intty: INTVAL := obj.int;
    sflty,
    flty:  INTVAL := ROUND( obj.flt );
    charty,
    strty: begin
             GET_STRING( st, obj, '' ); READV( st, iv );
             INTVAL := iv
           end;
  otherwise
    INTVAL := INTVAL( F_EXEC_ERROR( 'INTV', 51, e_error ) )
  end
end INTVAL;



[global]
function INTVREC( r: rec_ptr; ad: integer; ty: obj_type ): integer;
begin
  case ty of
    intub: INTVREC := BLK_UB( r, ad );
    intsb: INTVREC := BLK_SB( r, ad );
    intuw: INTVREC := BLK_UW( r, ad );
    intsw: INTVREC := BLK_SW( r, ad );
    intty: INTVREC := BLK_LI( r, ad );
    sflty: INTVREC := ROUND( BLK_FL( r, ad ) );
    flty:  INTVREC := ROUND( BLK_DB( r, ad ) );
  otherwise
    INTVREC := INTVAL( F_EXEC_ERROR( 'INTR', 51, e_error ) )
  end
end INTVREC;



[global]
function FLTVAL( obj: obj_ref ): lisp_real;
var
  st: [static] string( 64 );
  rv: double;

begin
  case obj.typ of
    nullty, truety:
      if logint_mode then FLTVAL := ORD( obj.typ = truety )
      else FLTVAL := INTVAL( F_EXEC_ERROR( 'INTV', 51, e_error ) );
    intub,
    intsb,
    intuw,
    intsw,
    intty:  FLTVAL := obj.int;
    sflty,
    flty:   FLTVAL := obj.flt;
    charty,
    strty: begin
             GET_STRING( st, obj, '' ); READV( st, rv );
             FLTVAL := rv
           end;
  otherwise
    FLTVAL := FLTVAL( F_EXEC_ERROR( 'FLTV', 51, e_error ) )
  end
end FLTVAL;



[global]
function FLTVREC( r: rec_ptr; ad: integer; ty: obj_type ): lisp_real;
begin
  case ty of
    intub: FLTVREC := BLK_UB( r, ad );
    intsb: FLTVREC := BLK_SB( r, ad );
    intuw: FLTVREC := BLK_UW( r, ad );
    intsw: FLTVREC := BLK_SW( r, ad );
    intty: FLTVREC := BLK_LI( r, ad );
    sflty: FLTVREC := BLK_FL( r, ad );
    flty:  FLTVREC := BLK_DB( r, ad );
  otherwise
    FLTVREC := FLTVAL( F_EXEC_ERROR( 'INTR', 51, e_error ) )
  end
end FLTVREC;



[global]
function SFLTVREC( r: rec_ptr; ad: integer; ty: obj_type ): single;
begin
  case ty of
    intub: SFLTVREC := BLK_UB( r, ad );
    intsb: SFLTVREC := BLK_SB( r, ad );
    intuw: SFLTVREC := BLK_UW( r, ad );
    intsw: SFLTVREC := BLK_SW( r, ad );
    intty: SFLTVREC := BLK_LI( r, ad );
    sflty: SFLTVREC := BLK_FL( r, ad );
    flty:  SFLTVREC := BLK_DB( r, ad );
  otherwise
    SFLTVREC := FLTVAL( F_EXEC_ERROR( 'INTR', 51, e_error ) )
  end
end SFLTVREC;



[global]
procedure LIST_BLK_EVL( var ll, ob: obj_ref );
const
  mdnam = 'LIST';

begin
  ob :=  F_EVAL( NXT_PAR( ll ) );
  case ob.typ of
    doublety:  { QUEUE or LIST }
               with ob.db^ do
                 if (car.typ = doublety) or (car.typ = nullty) then ob := ob.db^.cdr;

    vectortyp: ;

  otherwise
    EXEC_ERROR( mdnam, 85, e_severe )
  end
end LIST_BLK_EVL;



[global]
procedure LIST_BLK_NEXT( var obj, cur: obj_ref; var idx: integer );
begin
  if obj.typ = vectortyp then
    if (idx < 0) or (idx >= cur.vect^.vect_size) then cur := obj_nil
    else obj := cur.vect^.vect_tab[idx]
  else
  begin
    obj := NXT_PAR( cur );
    if cur.typ <> doublety then cur := obj_nil
  end;
  idx := idx + 1;
  if (obj.typ <> areatyp) and (obj.typ <> mrecty) then
  begin
    obj := obj_nil;
    cur := obj_nil;
    idx := 0
  end
end LIST_BLK_NEXT;



[global]
procedure MEM_CHECK( obj: obj_ref; bt1: boolean );
begin
  if obj.typ <> areatyp then
    if bt1 or (obj.typ <> areatyp1) then EXEC_ERROR( 'MCHK', 84, e_fatal )
end MEM_CHECK;



[global]
procedure MEM_SET_TYPREF( var adr, obt: obj_ref; obp: obj_ref );
begin
  if obp.typ = doublety then
    with obp.db^ do
    begin
      if adr.mem <> nil then
        adr.int := adr.int + INTVAL( car )
      else
        EXEC_ERROR( 'MTRF', 83, e_fatal );
      obt := cdr;
      if obt.typ = nullty then obt := obj_nilp
    end
  else
    obt := adr
end MEM_SET_TYPREF;



[global]
function NUMEVL( var obj: obj_ref ): obj_ref;
var
  res: obj_ref;
  st: [static] string( 64 );

begin
  res := F_EVAL( NXT_PAR( obj ) );
  case res.typ of
    charty,
    strty: begin
             res := obj_zero;
             GET_STRING( st, res, '' ); READV( st, res.int )
           end;
    intub,
    intsb,
    intuw,
    intsw: res.typ := intty;
    sflty: res.typ := flty;
    intty, flty: ;
  otherwise
    NUMEVL := F_EXEC_ERROR( 'NUMB', 58, e_error )
  end;
  NUMEVL := res
end NUMEVL;



[global]
function INTEVL( var ll: obj_ref ): integer;
begin
  INTEVL := INTVAL( F_EVAL( NXT_PAR(ll) ) )
end INTEVL;



[global]
function FLTEVL( var ll: obj_ref ): lisp_real;
begin
  FLTEVL := FLTVAL( F_EVAL( NXT_PAR(ll) ) )
end FLTEVL;



[global]
function FFUNCT( id, exp: obj_ref; x: lisp_real ): lisp_real;
begin
  id.at^.val.flt := x;
  FFUNCT := FLTVAL( F_EVAL( exp ) )
end FFUNCT;



[global]
function INTEVLDEF( var ll: obj_ref; vdef: integer ): integer;
var
  ob: obj_ref;

begin
  if ll.typ = nullty then INTEVLDEF := vdef
  else
  begin
    ob := F_EVAL( NXT_PAR( ll ) );
    if ob.typ = nullty then
      INTEVLDEF := vdef
    else
      INTEVLDEF := INTVAL( ob )
  end
end INTEVLDEF;



[global]
function FLTEVLDEF( var ll: obj_ref; vdef: lisp_real ): lisp_real;
var
  ob: obj_ref;

begin
  if ll.typ = nullty then FLTEVLDEF := vdef
  else
  begin
    ob := F_EVAL( NXT_PAR( ll ) );
    if ob.typ = nullty then
      FLTEVLDEF := vdef
    else
      FLTEVLDEF := FLTVAL( ob )
  end
end FLTEVLDEF;



[global]
function SFLTEVL( var obj: obj_ref ): single;
begin
  SFLTEVL := FLTEVL( obj )
end SFLTEVL;



[global]
function SFLTEVLDEF( var obj: obj_ref; r: lisp_real ): single;
begin
  SFLTEVLDEF := FLTEVLDEF( obj, r )
end SFLTEVLDEF;



[global]
function RECEVL( var ll: obj_ref ): rec_ptr;
const
  mdnam = 'EVRC';

var
  ob: obj_ref;

begin
  ob := obj_nil;
  ob := F_EVAL( NXT_PAR( ll ) );
  case ob.typ of
    mrecty,
    areatyp: ;
  otherwise
    EXEC_ERROR( mdnam, 262, e_severe )
  end;
  RECEVL := ob.rec
end RECEVL;



[global]
function GET_ATOM( obj: obj_ref; absflg : boolean ): obj_ref;
const
  mdnam = 'GATM';
var
  res: obj_ref;

begin
  res := obj;
  if res.typ = doublety then EXEC_ERROR( mdnam, 53, e_severe );
  if absflg and (res.typ < atomety) then
  begin
    EXEC_ERROR( mdnam, 59, e_severe );
    res := und_atom
  end;
  GET_ATOM := res
end GET_ATOM;



[global]
procedure REC_EVL( var el: obj_ref; var bl: rec_ptr; var sz: integer );
const
  mdnam = 'REVL';

var
  fr: obj_ref;

begin
  fr  := GET_ATOM( NXT_PAR( el ), true ); { Get Record Atom }
  with fr.at^ do
  begin
    if val.typ <> mrecty then EXEC_ERROR( mdnam, 257, e_severe );
    if fncref.flg.k <> dre_funct then EXEC_ERROR( mdnam, 254, e_severe );
    bl := val.rec;                 { Get the record address }
    sz := fncref.mrd^.mrd_size     { Get the record size in byte }
  end
end REC_EVL;



[global]
function GET_LIST( var obj: obj_ref; flg: boolean ): obj_ref;
var
  res: obj_ref;

begin
  res := F_EVAL( NXT_PAR( obj ) );
  if (res.typ <> doublety) and ((res.typ <> nullty) or flg) then
    EXEC_ERROR( 'GLIS', 50, e_severe );
  GET_LIST := res
end GET_LIST;



[global]
function INSTALL_LISP_FUNC( var ll: obj_ref ): obj_ref;
begin
  INSTALL_LISP_FUNC := GET_LIST( ll, false );  { get the parameter list }
end INSTALL_LISP_FUNC;



[global]
function SET_RESULT_LIST( o: obj_ref; dim: integer ): obj_ref;
begin
  if o.typ <> doublety then
  begin
    o := obj_nil;
    while dim > 0 do
    begin  o := F_CONS( obj_nil, o ); dim := dim - 1  end
  end;
  currobj := o;
  SET_RESULT_LIST := o
end SET_RESULT_LIST;



[global]
procedure SET_PARM_OBJ( o: obj_ref );
begin
  if currobj.typ = doublety then
  with currobj.db^ do
  begin
    car := o;
    currobj := cdr
  end
end SET_PARM_OBJ;



[global]
procedure SET_PARM_STR( s: ^string );
begin
  if currobj.typ = doublety then
  with currobj.db^ do
  begin
    car := obj_nil;
    if s <> nil then
    with s^ do
    if length > 0 then
      if length = 1 then
      begin
        car.typ := charty; car.ch := body[1]
      end
      else
      begin
        car := obj_nuls;
        car.nam := NEW_LISP_STRINGV( s^ )
      end;
    currobj := cdr
  end
end SET_PARM_STR;



[global]
procedure SET_PARM_INT( v: integer );
begin
  if currobj.typ = doublety then
  with currobj.db^ do
  begin
    car.flg := flg_def;
    car.typ := intty;
    car.int := v;
    currobj := cdr
  end
end SET_PARM_INT;



[global]
procedure SET_PARM_BOOL( b: boolean );
begin
  if currobj.typ = doublety then
  with currobj.db^ do
  begin
    car := log_val[ b ];
    currobj := cdr
  end
end SET_PARM_BOOL;



[global]
procedure SET_PARM_FLT( v: lisp_real );
begin
  if currobj.typ = doublety then
  with currobj.db^ do
  begin
    car.flg := flg_def;
    car.typ := flty;
    car.flt := v;
    currobj := cdr
  end
end SET_PARM_FLT;



[global]
procedure SET_PARM_SFLT( v: single );
begin
  if currobj.typ = doublety then
  with currobj.db^ do
  begin
    car.flg := flg_def;
    car.typ := flty;
    car.flt := v;
    currobj := cdr
  end
end SET_PARM_SFLT;



[global]
function GET_VALFLAG( ob: obj_ref ): boolean;
begin
  with ob do
  case typ of
    nullty: GET_VALFLAG := false;
    intty:  if logint_mode then GET_VALFLAG := (int > 0)
                           else GET_VALFLAG := true;

    sflty,
    flty:   if logint_mode then GET_VALFLAG := (flt >= 0.5)
                           else GET_VALFLAG := true;

  otherwise
    GET_VALFLAG := true
  end
end GET_VALFLAG;



[global]
function GET_EVLFLAG( var ll: obj_ref ): boolean;
begin
  GET_EVLFLAG := GET_VALFLAG( F_EVAL( NXT_PAR( ll ) ) )
end GET_EVLFLAG;



[global]
function GET_FLAG( var ll: obj_ref ): boolean;
begin
  GET_FLAG := GET_VALFLAG( NXT_PAR( ll ) )
end GET_FLAG;



[global]
function GET_CHA( var ll: obj_ref; def: char := ' ' ): char;
var
  ob: obj_ref;

begin
  ob := NXT_PAR( ll );
  if ob.typ = charty then GET_CHA := ob.ch
                     else GET_CHA := def
end GET_CHA;



[global]
function GET_INT( var ll: obj_ref; def: integer := 0 ): integer;
var
  ob: obj_ref;

begin
  ob := NXT_PAR( ll );
  case ob.typ of
    intty: GET_INT := ob.int;
    sflty,
    flty:  GET_INT := ROUND( ob.flt );
  otherwise
    GET_INT := def
  end
end GET_INT;



[global]
function GET_REC( var ll: obj_ref ): obj_ref;
const
  mdnam = 'GREC';

var
  ob: obj_ref;

begin
  ob := NXT_PAR( ll );
  if ob.typ <> mrecty then EXEC_ERROR( mdnam, 261, e_severe );
  GET_REC := ob
end GET_REC;



[global]
function GET_FLT( var ll: obj_ref; def: lisp_real := 0.0 ): lisp_real;
var
  ob: obj_ref;

begin
  ob := NXT_PAR( ll );
  case ob.typ of
    intty: GET_FLT := ob.int;
    sflty,
    flty:  GET_FLT := ob.flt;
  otherwise
    GET_FLT := def
  end
end GET_FLT;



[global]
function GET_SFLT( var ll: obj_ref; def: single ): single;
var
  ob: obj_ref;

begin
  ob := NXT_PAR( ll );
  case ob.typ of
    intty: GET_SFLT := ob.int;
    sflty,
    flty:  GET_SFLT := ob.flt;
  otherwise
    GET_SFLT := def
  end
end GET_SFLT;



[global]
function GET_STRVAL( s: obj_ref ): obj_ref;
begin
  GET_STRVAL := GET_LISP_STR( s, '' )
end GET_STRVAL;



[global]
procedure GET_STR( var  s:   string;
                   var  ll:  obj_ref;
                   var  def: [READONLY] string );
var
  i, j: integer;
  st: [static] string( 255 );

begin
  GET_STRING( s, F_EVAL( NXT_PAR( ll ) ), def )
end GET_STR;



[global]
function STRING_MATCH( s1, s2: obj_ref ): integer;
var
  st1, st2: string( 255 );

begin
  GET_STRING( st1, s1, '' ); GET_STRING( st2, s2, '' );
  if s1.nam = nil then
    if s2.nam = nil then
      STRING_MATCH :=  0
    else
      STRING_MATCH := -1
  else
    if s2.nam = nil then
      STRING_MATCH :=  1
    else
      STRING_MATCH := STR_MATCH( st1, st2 )
end STRING_MATCH;



[global]
function FLOAT_MATCH_VAL( ob1, ob2: obj_ref ): integer;
var
  res: lisp_real;

begin { FLOAT_MATCH_VAL }
  res := FLTVAL( ob1 ) - FLTVAL( ob2 );
  if res = 0.0 then
    FLOAT_MATCH_VAL := 0
  else
    if res > 0.0 then
      FLOAT_MATCH_VAL :=  1
    else
      FLOAT_MATCH_VAL := -1
end FLOAT_MATCH_VAL;



end.
{  * * * *  End of Lisp Routine file  * * * *  }
