{
******************************************************************
*                                                                *
*                                                                *
*                                                                *
*                                                                *
*      * * *    L I S P    I n t e r p r e t e r    * * *        *
*                                                                *
*                                                                *
*                ***   MATHEMATIC MODULE  ***                    *
*                                                                *
*       by :                                                     *
*                                                                *
*           P. Wolfers                                           *
*               c.n.r.s.                                         *
*               Laboratoire de Cristallographie                  *
*               B.P.  166 X   38042  Grenoble Cedex              *
*                                       FRANCE.                  *
*                                                                *
******************************************************************
}


{  Version 1.2-B (or Upper)  of  E - L I S P     System  }
{***********    CPAS  Version   **************}


{
        *** modification(s) from major version ***


			----

		       nothing

			----

}
module LISP_MATH( Input, Output ); { input and output for user terminal }

{
	Module to save or restore a LISP environment.
}

%include 'lispsrc:lisp_env';   { Get the LISP Environment Definitions }


const
  joker    = 1.0E+30;
  defprec  = 1.0E-4;


var
  mth_ind: integer;
  mth_mem, mth_mem1, mth_mem2: lisp_real;





{ ***************************** }
{ *** Integration Functions *** }
{ ***************************** }

function IIPOWER( x, n: integer ): integer;
var
  r: integer;

begin
  r := 1;
  n := ABS( n );
  while n > 0 do
    if ODD( n ) then
    begin
      n := n - 1; r := r * x
    end
    else
    begin
      n := n div 2; x := SQR( x )
    end;
  IIPOWER := r
end IIPOWER;


[global]
procedure CREATE_FLT_ARRAY( fa: obj_ref; sz: integer );
var
  mr: mrd_ptr;
  rf: rfd_ptr;

begin
  { Build the coefficient table definition }
  NEW( mr );                       { Allocate a record descriptor }
  NEW( rf );                       { and a field descriptor }
  with rf^ do
  begin
    rfd_nxt := nil;                { No next field for an array }
    rfd_mrd := mr;                 { Reverse link to record the descriptor }
    rfd_atm := nil;                { No field descriptor for array }
    rfd_off := 0;                  { Offset is always 0 }
    rfd_dim := sz;                 { Set the numeric table size }
    rfd_typ.flg := flg_def;        { Set the LISP_FLOAT type (double) }
    rfd_typ.typ := flty;
    rfd_typ.flt := 0.0
  end;
  sz := sz*size_doub;              { Get the size in bytes }
  with fa.at^, mr^ do
  begin
    mrd_nxt      := mrd_alloc;     { Link with previously defined descr. }
    mrd_alloc    := mr;
    mrd_atm      := fa.at;         { Set the identifier link }
    mrd_rfdl     := rf;            { and the field descriptor link }
    mrd_size     := sz;            { Set the total size of record in byte }
    mrd_algn     := algn_doub;     { and the related alignement }
    fncref.flg.k := dre_funct;     { Set the atom as an array atom }
    fncref.typ   := mrdty;         { Set the descriptor record acc. fnc type }
    fncref.mrd   := mr;
    val.typ := mrecty;             { Set the allocation }
    { Set the array address in the descriptor }
    val.rec := NEW_RECORD_ALLOC( sz ) { Allocate the array }
  end
end CREATE_FLT_ARRAY;




[global]
procedure GET_REC_ARRAY( var lp: obj_ref;    rty: obj_type;
                         var bl: rec_ptr; var sz: integer   );
const
  mdnam = 'GRAR';
var
  rf: rfd_ptr;
  fa: obj_ref;

begin
  fa := GET_ATOM( NXT_PAR( lp ), true );  { Get the record array identifier }
  with fa.at^ do
  begin
    if (fncref.typ <> mrdty) or (val.typ <> mrecty) then
      EXEC_ERROR( mdnam, 999, e_severe );
    bl := val.rec;                        { Get the record pointer }
    rf := fncref.mrd^.mrd_rfdl
  end;
  with rf^ do
  begin
    if rfd_atm <> nil then EXEC_ERROR( mdnam, 999, e_severe );
    if rfd_typ.typ <> rty then EXEC_ERROR( mdnam, 999, e_severe );
    sz := rfd_dim
  end
end GET_REC_ARRAY;



[global]
function MTH_L_BESSELJ( lp: obj_ref ): obj_ref;
var
  re: obj_ref;

begin
  re.flg := flg_def;
  re.typ := flty;
  re.flt := 0.0;

  MTH_L_BESSELJ := re
end MTH_L_BESSELJ;





function MATH_GAMMALN( x: lisp_real ): lisp_real;
var
  i: integer;
  x1, t, s: lisp_real;
  ct: [static] array[1..6] of lisp_real := (
             76.18009172947146, -86.50532032941677,
             24.01409824083091, -1.231739572450155,
         0.1208650973866179E-2, -0.5395239384953E-5 );
begin
  x1 := x;
  t  := x1 + 5.5;
  t  := t - (x + 0.5)*LN( t );
  s  := 1.000000000190015E+00;
  for i := 1 to 6 do
  begin
    x1 := x1 + 1.0;
    s  := s  + ct[i]/x1
  end;
  MATH_GAMMALN := LN( 2.5066282746310005*s/x ) - t
end MATH_GAMMALN;




[global]
function MTH_L_GAMMA( lp: obj_ref ): obj_ref;
var
  x, y: lisp_real;
  re: obj_ref;

begin
  re.flg := flg_def;
  re.typ := flty;
  re.flt := EXP( MATH_GAMMALN( ABS( FLTEVL( lp ) ) ) );
  MTH_L_GAMMA := re
end MTH_L_GAMMA;



[global]
procedure MTH_L_NEXT_POINT( var ali: obj_ref;
                            var xp, yp: lisp_real;
                            var nc: integer );
var
  ob: obj_ref;
  br: boolean;

begin
  ob := NXT_PAR( ali );
  if ob.typ = doublety then
  with ob.db^ do
  begin
    xp := FLTVAL( car );
    yp := FLTVAL( cdr );
    nc := SUCC( nc )
  end
  else nc := - nc;
end MTH_L_NEXT_POINT;



[global]
function MTH_L_INTERPOL( lp: obj_ref ): obj_ref;
{ Perform an Interpolation of the A-list Table }
const
  mdnam = 'ITPO';

var
  re, al: obj_ref;
  xf, c1, c2, c3, r1, r2,
  xc, x1, x2, x3, x4, yc, y1, y2, y3, y4: lisp_real;
  n: integer;
  br: boolean;

begin { MTH_L_INTERPOL }
  re.flg := flg_def;
  re.typ := flty;
  re.flt := 0.0;
  al := GET_LIST( lp, true );
  xf := FLTEVL( lp );
  al := al.db^.cdr; { Skip the queue header }
  n  := 0;
  MTH_L_NEXT_POINT( al, x1, y1, n );
  MTH_L_NEXT_POINT( al, x2, y2, n );
  MTH_L_NEXT_POINT( al, x3, y3, n );
  MTH_L_NEXT_POINT( al, x4, y4, n );
  if n >= 4 then
  begin
    while (n > 0) and (xf > x3) do
    begin { Loop to search the points in the table }
      xc := x4; yc := y4;
      MTH_L_NEXT_POINT( al, x4, y4, n );
      if n > 0 then
      begin
        x1 := x2; y1 := y2;
        x2 := x3; y2 := y3;
        x3 := xc; y3 := yc
      end
    end;
    { Four points interpolation }
    r1 := (y1 - y2)/(x1 - x2);
    r2 := (r1 - (y1 - y3)/(x1 - x3))/(x2 - x3);
    c3 := (r2 - (r1 - (y1 - y4)/(x1 - x4))/(x2 - x4))/(x3 - x4);
    c2 := r2 - (x1 + x2 + x3)*c3;
    c1 := r1 - (x1 + x2)*c2 - (x1*x1 + x1*x2 + x2*x2)*c3;
    re.flt := y1 - x1*(c1 + x1*(c2 + x1*c3)) + xf*(c1 + xf*(c2 + xf*c3))
  end
  else
  case n of
    3: begin
         r1 := (y1 - y2)/(x1 - x2);
         c2 := (r1 - (y1 - y3)/(x1 - x3))/(x2 - x3);
         c1 := r1 - (x1 + x2)*c2;
         re.flt := y1 - x1*(c1 + x1*c2) + xf*(c1 + xf*c2)
       end;
    2: re.flt := y1 + (xf - x1)*(y1 - y2)/(x1 - x2);
    1: re.flt := y1;
  otherwise
  end;
  MTH_L_INTERPOL := re
end MTH_L_INTERPOL;




procedure INTGR_GAUSS_LEGENDRE( n:           integer;
                                eps, va, vb: lisp_real;
                                bl:          rec_ptr     );
var
  i, ip, j, jp: integer;
  e, c1, c2, c3, cder, xm, xl, x, w: lisp_real;

begin
  xm := 0.5*(vb + va);
  xl := 0.5*(vb - va);
  ip := 0;
  jp := 2*n - 1;
  for i := 1 to (n + 1) div 2 do
  begin
    x := COS( pi*(i - 0.25)/(n + 0.5) );
    repeat
      c1 := 1.0;
      c2 := 0.0;
      for j := 1 to n do
      begin
        c3 := c2; c2 := c1;    { Pn-1 -> pn-2, Pn -> Pn-1 }
        c1 := ((2.0*j - 1.0)*x*c2 - (j - 1.0)*c3)/j
      end;
      cder := n*(x*c1 - c2)/(SQR( x ) - 1.0);
      e := x;
      x := e - c1/cder
    until ABS( x - e ) <= eps;
    w := 2.0*xl/((1.0 - SQR( x ))*SQR( cder ));
    bl^.gt[ip] := w;
    bl^.gt[jp] := xm + xl*x;
    ip := ip + 1;
    jp := jp - 1;
    bl^.gt[ip] := xm - xl*x;
    bl^.gt[jp] := w;
    ip := ip + 1;
    jp := jp - 1
  end
end INTGR_GAUSS_LEGENDRE;




procedure INTGR_GAUSS_LAGUERRE( n:       integer;
                                al, eps: lisp_real;
                                bl:      rec_ptr     );
const
  mdnam = 'INTL';
  maxit = 500;

var
  i, ip, it, j: integer;
  c0, c1, c2, c3, pder, x, x1: lisp_real;

begin
  ip := 0;
  for i := 1 to n do
  begin
    case i of
      1: x := (1.0 + al)*(3.0 + 0.92*al)/(1.0 + 2.4*n + 1.8*al);
      2: x := x + (15.0 + 6.25*al)/(1.0 + 0.9*al + 2.5*n);
    otherwise
      c0 := i - 2;
      x  := x + ((1.0 + 2.55*c0)/(1.9*c0) + 1.26*c0*al/(1.0 + 3.5*c0))*
                (x - bl^.gt[ip - 2])/(1.0 + 0.3*al)
    end;
    it := 0;
    repeat
      c1 := 1.0;
      c2 := 0.0;
      for j := 1 to n do
      begin
        c3 := c2;
        c2 := c1;
        c1 := ((2*j - 1 + al - x)*c2 - (j - 1 + al)*c3)/j
      end;
      pder := (n*c1 - (n + al)*c2)/x;
      x1 := x;
      x  := x1 - c1/pder;
      it := it + 1
    until (it > maxit) or (ABS( x - x1 ) < eps);
    if it > maxit then EXEC_ERROR( mdnam, 602, e_error );
    bl^.gt[ip] := -EXP(  MATH_GAMMALN( al + n )
                       - MATH_GAMMALN( double( n ) ) )/(pder*n*c2);
    ip := ip + 1;
    bl^.gt[ip] := x;
    ip := ip + 1
  end
end INTGR_GAUSS_LAGUERRE;




procedure INTGR_GAUSS_HERMITE(  n:   integer;
                                ep:  lisp_real;
                                bl:  rec_ptr    );
const
  mdnam = 'INTH';
  maxit  = 500;
  uspir4 = 0.7511255444649425E+00;

var
  i, ip, it, j, jp: integer;
  c1, c2, c3, pder, x, x1: lisp_real;

begin
  ip := 0;
  jp := 2*n - 1;
  for i := 1 to (n + 1) div 2 do
  begin 
    case i of
      1: begin
           x1 := 2*n + 1;
           x  := SQRT( x1 ) - 1.85575*EXP( -1.6667*LN( x1 ) )
         end;

      2: begin
           x1 := n;
           x  := x - 1.14*EXP( 0.426*LN( x1 ) )/x
         end;

      3: x := 1.86*x - 0.86*bl^.gt[1]; { Compute 3-th root from the first one }

      4: x := 1.91*x - 0.91*bl^.gt[3]; { Compute 4-th root form the second one}

    otherwise
      x := 2.0*x - bl^.gt[ip - 2]      { Compute the i-th x from the i-2-th x }
    end;

    it := 0;
    repeat
      c1 := uspir4;
      c2 := 0.0;
      for j := 1 to n do
      begin
        c3 := c2;
        c2 := c1;
        c1 := j;
        c1 := x*SQRT( 2.0/c1 )*c2 - SQRT( (c1 - 1.0)/c1 )*c3
      end;
      pder := SQRT( 2.0*n )*c2;
      x1   := x;
      x    := x1 - c1/pder;
      it := it + 1
    until (ABS( x1 - x ) < ep) or (it > maxit);
    if it > maxit then EXEC_ERROR( mdnam, 602, e_error );
    pder := 2.0/SQR( pder );
    bl^.gt[ip] := pder;
    bl^.gt[jp] := x;
    ip := ip + 1;
    jp := jp - 1;
    bl^.gt[ip] := -x;
    bl^.gt[jp] := pder;
    ip := ip + 1;
    jp := jp - 1
  end
end INTGR_GAUSS_HERMITE;




[global]
function MTH_L_GAUSS_INTEGR_BLDTAB( lp: obj_ref ): obj_ref;
var
  ik, n: integer;
  al, va, vb, ep: lisp_real;
  re: obj_ref;
  bl: rec_ptr;

begin
  re := GET_ATOM( NXT_PAR( lp ), true );  { Get the array identifier }
  n  := INTEVLDEF( lp, 10);               { Get the required size }
  if n < 5 then n := 5;                   { Set a minimum of 5 coefficients }
  CREATE_FLT_ARRAY( re, 2*n );            { Create the coefficient array }
  bl := re.at^.val.rec;                   { Get the array address }
  ik := INTEVLDEF( lp, 0 );               { Get the kind of table }
  ep := FLTEVLDEF( lp, defprec );         { Get the required precision }
  case ik of
    1: { GAUSS LAGUERRE }
      begin
        al := FLTEVLDEF( lp, 1.0 );       { Get the alpha laguerre coef }
        INTGR_GAUSS_LAGUERRE( n, al, ep, bl )
      end;

    2: { GAUSS HERMITE }
      INTGR_GAUSS_HERMITE( n, ep, bl );

  otherwise
    { default to GAUSS LEGENDRE }
    va := FLTEVLDEF( lp, -joker ); { Get the low integration bound }
    vb := FLTEVLDEF( lp,  joker ); { Get the high integration bound }
    if vb <= va then vb := va + 1.0;
    INTGR_GAUSS_LEGENDRE( n, ep, va, vb, bl )
  end;
  MTH_L_GAUSS_INTEGR_BLDTAB := re
end MTH_L_GAUSS_INTEGR_BLDTAB;



[global]
function MTH_L_GAUSS_INTEGR( lp: obj_ref ): obj_ref;
const
  mdnam = 'MTHI';
  joker = 1.0E+30;

var
  i, ip, n: integer;
  w:        lisp_real;
  bl:       rec_ptr;
  re, id, sa, fn: obj_ref;

begin
  id := GET_ATOM( NXT_PAR( lp ), true );  { Get the summation identifier atom }
  sa := id.at^.val;                       { Save the identifier value }
  re.flg := flg_def;                      { Set the default flag }
  re.typ := flty;                         { Set the floatting type }
  re.flt := 0.0;                          { Set the zero value }
  id.at^.val := re;                       { Set Zero value at id }

  GET_REC_ARRAY( lp, flty, bl, n );       { Get the record array }
  n := n div 2;                           { Get the number of element }

  fn := NXT_PAR( lp );                    { Get function to integrate }

  { Integration Loop }
  ip := 0;
  if n > 5 then
  for i := 1 to n do
  begin
    w              := bl^.gt[ip];         { Get the weight }
    ip             := ip + 1;
    id.at^.val.flt := bl^.gt[ip];         { Set the integration var. value }
    ip             := ip + 1;
    re.flt := re.flt + w*FLTVAL( F_EVAL( fn ) ) { Compute the integral term }
  end;
  id.at^.val := sa;                       { Restore the integr. id. value }
  MTH_L_GAUSS_INTEGR := re
end MTH_L_GAUSS_INTEGR;

end.
