%pragma trace 1;
{
*************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*                                                                       *
*          * * *    L I S P    I n t e r p r e t e r    * * *           *
*                                                                       *
*                                                                       *
*              ***  LEAST SQUARES INTERFACE MODULE   ***                *
*                                                                       *
*       by :                                                            *
*                                                                       *
*           P. Wolfers                                                  *
*               c.n.r.s.,                                               *
*               Laboratoire de Cristallographie,                        *
*               B.P.  166 X   38042  Grenoble Cedex,                    *
*                                              FRANCE.                  *
*                                                                       *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************


/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////
}


{  Version 1.2-B (or Upper)  of  E - L I S P     System  }
{***********    CPAS  Version   **************}

{
        *** modification(s) from major version ***


			----

		       nothing

			----

}
module LISP__LSQ;


%include 'lispsrc:lisp_env';       { Get the LISP Environment Definitions }
%include 'lispsrc:lsq_global_ref'; { Get the LSQ Environment Def. }



type

  corr_ptr = ^corr_elem;

  corr_elem = record
    left, right:  corr_ptr;
    val:          lsq_real;
    va1, va2:     lsq_ptr
  end;




  { LSQ LISP list descriptors }

  lld_descr = record
    lld_rec_descr:  mrd_ptr;       { Record type descriptor }
    lld_rec_access: doublet_ptr    { Pointer of LISP doublet }
  end;

  lld_ptr = ^lld_descr;


  { Equivalence Record Between the Worlds LISP and LSQ }
  equ_type = record
    case integer of
     0:( dbl: doublet_ptr );
     1:( atm: atome_ptr   );
     2:( rec: rec_ptr     );
     3:( mem: mem_ptr     );
     4:( lsq: lsq_ptr     );
     5:( sta: statis_ptr  );
     6:( lld: lld_ptr     );
     7:( vec: vec_ptr     );
     8:( int: integer     );
     9:( flt: single      )
  end;




var
  { Translation Tables between LISP and LSQ Operator Node Kinds }

  { Without Derivation }
  cd_tab_n: array[f_acos_fnc..f_integr_fnc] of lsq_nodetypes := (
    { f_acos_fnc,    f_acosd_fnc,   f_asin_fnc,    f_asind_fnc    }
      nd_acos,       nd_adcos,      nd_asin,       nd_adsin,
    { f_atan_fnc,    f_atand_fnc,   f_cos_fnc,     f_cosd_fnc     }
      nd_atan,       nd_adtan,      nd_cos,        nd_dcos,
    { f_div_fnc,     f_exp_fnc,     f_float_fnc,   f_log_fnc      }
      nd_div,        nd_exp,        nd_round,      nd_log,
    { f_phase_fnc,   f_phased_fnc,  f_round_fnc,   f_sin_fnc      }
      nd_phase,      nd_dphase,     nd_round,      nd_sin,
    { f_sind_fnc,    f_sqrt_fnc,    f_tan_fnc,     f_tand_fnc     }
      nd_dsin,       nd_sqrt,       nd_tan,        nd_dtan,
    { f_trunc_fnc,   f_sinh_fnc,    f_cosh_fnc,    f_tanh_fnc     }
      nd_trunc,      nd_sinh,       nd_cosh,       nd_tanh,
    { f_asinh_fnc,   f_acosh_fnc,   f_atanh_fnc }
      nd_asinh,      nd_acosh,      nd_atanh,
    { f_bess1_fnc,   f_gamma_fnc }
      nd_bess1,      nd_gamma,
    { f_interpol_fnc,f_summ_fnc,    f_integr_fnc }
      nd_interpol,   nd_summ_loop,  nd_integr_loop
  );

  { With Derivation }
  cd_tab_d: array[f_acos_fnc..f_integr_fnc] of lsq_nodetypes := (
      nd_acos_d,     nd_adcos_d,    nd_asin_d,     nd_adsin_d,
      nd_atan_d,     nd_adtan_d,    nd_cos_d,      nd_dcos_d,
      nd_div_d,      nd_exp_d,      nd_round,      nd_log_d,
      nd_phase_d,    nd_dphase_d,   nd_round,      nd_sin_d,
      nd_dsin_d,     nd_sqrt_d,     nd_tan_d,      nd_dtan_d,
      nd_trunc,      nd_sinh_d,     nd_cosh_d,     nd_tanh_d,
      nd_asinh_d,    nd_acosh_d,    nd_atanh_d,
      nd_bess1_d,    nd_gamma_d,
      nd_interpol_d, nd_summ_loop_d,nd_integr_loop_d
  );



  err_nohandler: [external,volatile] boolean;

  deriv_enables,                   { Not Derivable Flag }
  deriv_flg:     boolean;          { Derivable Flag }

  identcp:       integer;          { Integer used a Identifier Assign }

  list_ref_fnc,                    { Flag function for list object reference }
  record_ref_fnc,                  { Flag function for refer record reference } 
  lisp_eval_fnc,                   { Flag function for force the LISP evaluation
                                     during the Least-Squares Process }

  usercons,                        { Constraint Notification Executable List }
  userfend,                        { End Of Fit Notification Executable List }
  uservarbl,                       { Variable Change Notif. Executable List }
  usercorr,                        { Correlation Matrix Notif. Exec. List }
  usersing,                        { Variable Singularity Notif Exec. List }
  usercyst,                        { User Cycle Start Notification List }
  usercyre,                        { User Cycle Solve Notification List }
  usercyen,                        { User Cycle End Notification List }
  usercost,                        { User Data Collection Start Notif. List }
  usercoen,                        { User Data Collection End Notif. List }
  usercoco,                        { User Data Coll. Contribution Notif. List }
  usercopa:     obj_ref;           { User Get Packet Notification List }





procedure SET_PARM_IDE( id: rec_ptr );
{ Procedure to set a LSQ object identifier field }
var
  eq: equ_type;
  ob: obj_ref;

begin
  eq.rec := id;
  if currobj.typ = doublety then
  with currobj.db^ do
  begin
    if id <> nil then car := eq.dbl^.car
                 else car := obj_nil;
    currobj := cdr
  end
end SET_PARM_IDE;



procedure SET_PARM_MEM( p: lsq_ptr );
{ Procedure to Set a Least-Squares Pointer in a LIST element }
var
  eq: equ_type;

begin
  eq.lsq := p;
  if currobj.typ = doublety then
  with currobj.db^ do
  begin
    if p <> nil then
    begin
      car.flg.f    := [];
      car.flg.k    := lsq_define;
      car.typ      := areatyp;
      car.mem      := eq.mem;
    end else  car  := obj_nil;
    currobj := cdr
  end
end SET_PARM_MEM;



procedure SET_PARM_STA( p: statis_ptr );
{ Procedure to set a Least-Squares Pointer in a LIST element }
var
  eq: equ_type;

begin
  eq.sta := p;
  if currobj.typ = doublety then
  with currobj.db^ do
  begin
    if p <> nil then
    begin
      car.flg.f    := [];
      car.flg.k    := lsq_define;
      car.typ      := mrecty;
      car.rec      := eq.rec;
    end else  car  := obj_nil;
    currobj := cdr
  end
end SET_PARM_STA;



procedure SET_PARM_ALM( p: rec_ptr );
{ Procedure to set a Least-Squares Returned LISP Record in a List element }
begin
  if currobj.typ = doublety then
  with currobj.db^ do
  begin
    if p <> nil then
    begin
      car.flg      := flg_def;
      car.typ      := mrecty;
      car.rec      := p
    end
    else
      car := obj_nil;
    currobj := cdr
  end
end SET_PARM_ALM;



function GET_AREA( var ll: obj_ref; flg: boolean ): lsq_ptr;
{ Routine to get a Least-Squares (LSQ) Pointer from a LISP effective List }
const
  mdnam = 'GLSQ';

var
  ob: obj_ref;
  eq: equ_type;

begin
  { Get the LSQ atom }
  ob := F_EVAL( NXT_PAR( ll ) );
  if (ob.typ = nullty) and not flg then
    eq.lsq := nil
  else
    if (ob.flg.k <> lsq_define) or
       (ob.typ <> areatyp) then
      EXEC_ERROR( mdnam, 1388, e_severe )
    else
      eq.mem := ob.mem;
  GET_AREA := eq.lsq { return the lsq_ptr pointer }
end GET_AREA;



function GET_LSQLIST( var ll: obj_ref; flg: boolean ): lsq_ptr;
{ Routine to Get a Least-Squares (LSQ) Pointer to a LSQ List. }
const
  mdnam = 'GLLS';

var
  ls: lsq_ptr;

begin
  ls := GET_AREA( ll, flg );
  { Get the LSQ atom }
  if ls <> nil then
    if ls^.lsq_ndty <> nd_list then EXEC_ERROR( mdnam, 1989, e_severe );
  GET_LSQLIST := ls
end GET_LSQLIST;



function GET_EVL_LSQ( var ll: obj_ref; nilflg: boolean ): lsq_ptr;
{ Routine to Evaluate a LSQ Pointer from a LISP effective List }
const
  mdnam = 'EVLS';

var
  eq: equ_type;
  ad: obj_ref;

begin
  ad := F_EVAL( NXT_PAR( ll ) );
  if nilflg and (ad.typ = nullty) then
    eq.lsq := nil
  else
    if (ad.typ = areatyp) and (ad.flg.k = lsq_define) then
      eq.mem := ad.mem
    else
      EXEC_ERROR( mdnam, 1388, e_severe );
  GET_EVL_LSQ := eq.lsq
end GET_EVL_LSQ;



function GEN_LISP_PTR( p: lsq_ptr ): obj_ref;
{ Routine to Generate a LISP LSQ Reference from a provided LSQ pointer }
var
  eq: equ_type;
  ob: obj_ref;

begin
  eq.lsq       := p;
  ob.flg.f     := [];
  ob.flg.k     := lsq_define;
  ob.typ       := areatyp;
  ob.mem       := eq.mem;
  GEN_LISP_PTR := ob
end GEN_LISP_PTR;



procedure NAME_LSQ_OBJ( var lp, atm: obj_ref; var r: rec_ptr );
{ To get and Parse the identification field of a LSQ object }
{ Two form of identification parameter :
   1/ <atome> :
         The atom is used as variable identifier,
         and the atom value is set as a LSQ object pointer.
   2/ ( <atome> <Lisp expr> ) :
         The value of atom is setas a LSQ object pointer,
         and the Lisp expression is used to generate a variable identification.
  If the specified atom is NIL, then it is not used.

  Example for the first mode  : at_scale
                           => generate at_scale as object ident. expr.
                           => SETQ at_scale <object reference>.
 
  Example for the second mode : (at_scale (ident_proc scale 35))
                           => Keep (ident_proc scale 35) as object ident. expr.
                           => SETQ at_scale <object reference>.

  The object reference is given by the name element of a (SYS_CALL 372 ...).

}
var
  lrf: obj_ref;
  eq:  equ_type;

begin
  { Get the attached atom }
  lrf := F_EVAL( NXT_PAR( lp ) );
  if lrf.typ = doublety then       { Get the attached atom - if exist }
    atm := GET_ATOM( NXT_PAR( lrf ), false )
  else
  begin
    atm := GET_ATOM( lrf, false );
    lrf := F_CONS( atm, obj_nil )
  end;
  eq.dbl := lrf.db;
  r      := eq.rec
end NAME_LSQ_OBJ;



function SET_PTR_VALUE( atm: obj_ref; p: lsq_ptr ): obj_ref;
{ If obj is an atom ptr then Set its Value to be Res else Set obj := res }
var
  eq: equ_type;
  ob: obj_ref;

begin
  eq.lsq   := p;
  ob.flg.f := [];
  ob.flg.k := lsq_define;
  ob.typ   := areatyp;
  ob.mem   := eq.mem;
  if atm.typ >= atomety then atm.at^.val := ob; { Set the local atom value }
  SET_PTR_VALUE := ob
end SET_PTR_VALUE;





    {*************************************}
    { *** Least-Squares User routines *** }
    {*************************************}


[global]
procedure LSQUSR_ERROR( err_nb,
                        sev: integer;
                        ptr: lsq_ptr );
{ LISP Signal Error routine for LSQ }
const
  mdnam = 'LSQE';

var
  sv: error_sev;

begin
  case sev of
    0: sv := e_success;
    1: sv := e_warning;
    2: sv := e_error;
    3: sv := e_severe
  otherwise
    sv := e_fatal
  end;
  EXEC_ERROR( mdnam, 1300 + err_nb, sv )
end LSQUSR_ERROR;



[global]
procedure LSQUSR_REFER_ID(    ptr: lsq_ptr );
{ Output (by F_PRINT use) a LSQ object identification as defined }
var
  eq: equ_type;
  ob: obj_ref;

begin
  ob := obj_nil;
  nctobj := 0;
  if ptr <> nil then
  begin
    with ptr^ do
    case lsq_ndty of
      nd_varbl:   eq.rec := var_name;
      nd_diablk:  eq.rec := blk_name;
      nd_list:    eq.rec := lis_name;
      nd_collect: eq.rec := coll_name;
      nd_parm:    eq.rec := par_name;
    otherwise
      return
    end;
    if eq.rec <> nil then ob := eq.dbl^.car
                     else ob := obj_nil
  end; 
  OUT_ATOM( ob );
  ob.typ := charty;
  ob.ch  :=  ' ';
  OUT_ATOM( ob )
end LSQUSR_REFER_ID;



[global]
procedure LSQUSR_SINGULARITY( cvar:        lsq_ptr;
                              nfail:       integer;
                              singovf_flg: boolean );
{ LISP signal singularity routine }
{ set parameter and call the LISP specified routine }
begin
  if usersing.typ = doublety then
  begin
    currobj := usersing.db^.cdr;
    SET_PARM_MEM( cvar );
    SET_PARM_INT( nfail );
    SET_PARM_BOOL( singovf_flg );
    currobj := F_EVAL( usersing )
  end
end LSQUSR_SINGULARITY;



function LSQUSR_FNC_PROC_COM( parm: rec_ptr ): obj_ref;
{ Common Code for LSQUSR_FUNCTION and LSQUSR_PROCEDURE }
var
  o1, o2, o3: obj_ref;
  re:         lsq_obj;
  eq:         equ_type;

begin
  eq.rec := parm;                  { Get the LISP function descriptor }
  o1 := eq.dbl^.car;               { Get LISP function or expression }
  o2 := eq.dbl^.cdr;               { Get LSQ parameter list }
  if (o1.typ = doublety) and
     (o2.typ = doublety) then
  begin { For a Legal Descriptor }
    currobj := o1.db^.cdr;
    { Set the effective list of the LISP call }
    while (currobj.typ = doublety) and (o2.typ = doublety) do
    begin
      o3 := NXT_PAR( o2 ); eq.mem := o3.mem;
      re := LSQ_GETREF( eq.lsq );  { Evaluate the effective LSQ parameter }
      case re.typ of
        0, 1, 2:
           SET_PARM_MEM( re.lp );  { Set a LSQ memory }
        3: SET_PARM_FLT( re.fv );  { Set a single float value }
        4: SET_PARM_INT( re.iv );  { Set am integer value }
      otherwise
        SET_PARM_FLT( 0.0 )        { Default to set a float value 0.0 }
      end
    end
  end;
  LSQUSR_FNC_PROC_COM := F_EVAL( o1 )  { Execute the LISP function }
end LSQUSR_FNC_PROC_COM;



[global]
function LSQUSR_FUNCTION( ident: integer;
                           parm: rec_ptr ): lsq_real;
{ Lisp do not Use the Ident Parameter Because <parm>
  is the LISP Expression to Evaluate }
begin
  LSQUSR_FUNCTION := FLTVAL( LSQUSR_FNC_PROC_COM( parm ) )
end LSQUSR_FUNCTION;



[global]
procedure LSQUSR_PROCEDURE( ident: integer;
                             parm: rec_ptr );
{ Lisp do not Use the Ident Parameter because <parm>
  is the LISP Expression to Evaluate }
begin
  currobj := LSQUSR_FNC_PROC_COM( parm )
end LSQUSR_PROCEDURE;



[global]
procedure LSQUSR_VARIABLE(    cvar:      lsq_ptr;
                              newval,
                              sig,
                              shift:     lsq_real;
                              lflg:      integer;
                              begin_flg,
                              final_flg: boolean );
{ Called by LSQ after each Variable Value Change }
begin
  if uservarbl.typ = doublety then
  begin
    currobj := uservarbl.db^.cdr;
    SET_PARM_MEM( cvar );
    SET_PARM_FLT( newval );
    SET_PARM_FLT( sig );
    SET_PARM_FLT( shift );
    SET_PARM_INT( lflg );
    SET_PARM_BOOL( begin_flg );
    SET_PARM_BOOL( final_flg );
    currobj := F_EVAL( uservarbl )
  end
end LSQUSR_VARIABLE;



[global]
procedure LSQUSR_CORRELATION( pblk: lsq_ptr;
                               dim: integer );
{ Called by LSQ after Matrix Correlation Computing }
begin
  if usercorr.typ = doublety then
  begin
    currobj := usercorr.db^.cdr;
    SET_PARM_MEM( pblk );
    SET_PARM_INT( dim );
    currobj := F_EVAL( usercorr )
  end
end LSQUSR_CORRELATION;



procedure BLT_CORREL_LIST( tree: corr_ptr; var obel: obj_ref );
{ Recursive Routine to Sort the correlation value by decreasing magnitude }
var
  obj: obj_ref;

begin { BLT_CORREL_LIST }
  if tree <> nil then
  begin
    with tree^ do
    begin
      { Perform result LISP setting for previous entry }
      if right <> nil then BLT_CORREL_LIST( right, obel );
      { Perform result LISP setting for current entry }
      if obel.typ = doublety then
      begin
        currobj := NXT_PAR( obel );
        SET_PARM_FLT( val );
        SET_PARM_MEM( va1 );
        SET_PARM_MEM( va2 )
      end;
      { Perform result LISP setting for next entry }
      if left <> nil then BLT_CORREL_LIST( left, obel )
    end;
    DISPOSE( tree )                { Free the temporary structure }
  end
end BLT_CORREL_LIST;



function GET_CORREL_LIST( in_var matrix:   vec_tab;
                                 liste:    obj_ref;
                                 vali:     lsq_ptr;
                                 minval:   lsq_real;
                                 dim:      integer   ): integer;
{ Procedure to Build a Sorted List of Correlation Factors }
var
  tree, p, p1, p_last: corr_ptr;
  i, j, k, l:          integer;
  v:                   lsq_real;
  sv:                  lsq_ptr;

begin { GET_CORREL_LIST }
  tree := nil;
  l := 0;
  k := 0;
  for i := 1 to dim do
  begin
    sv := vali;
    k := k + 1; { Skip the Diagonal Element }
    for j := i + 1 to dim do
    begin
      sv := sv^.var_nxtbl;
      k := k + 1;
      v := matrix[k];
      if ABS( v ) >= minval then
      begin
        l := l + 1;
        NEW( p );
        with p^ do
        begin
          left := nil; right := nil;
          val := v;
          va1 := vali;
          va2 := sv
        end;
        if tree = nil then
          tree := p
        else
        begin
          p1 := tree;
          while p1 <> nil do
          with p1^ do
          begin
            p_last := p1;
            if ABS( v ) < ABS( val ) then p1 := left
                                     else p1 := right
          end;
          with p_last^ do
            if ABS( v ) < ABS( val ) then left := p
                                     else right := p
        end
      end
    end;
    vali := vali^.var_nxtbl
  end;
  if l > 0 then BLT_CORREL_LIST( tree, liste );
  GET_CORREL_LIST := l
end GET_CORREL_LIST;




[global]
procedure LSQUSR_CYCL_START(     ncycl: integer;
                             final_flg: boolean  );
{ Procedure Called by LSQ at the Least-Squares Cycle Start Time }
begin
  if usercyst.typ = doublety then
  begin
    currobj := usercyst.db^.cdr;
    SET_PARM_INT( ncycl );
    SET_PARM_BOOL( final_flg );
    currobj := F_EVAL( usercyst )
  end
end LSQUSR_CYCL_START;



[global]
procedure LSQUSR_CYCL_RES(  ncycl: integer );
{ Procedure Called by LSQ at the Least-Squares Cycle Resolution Time }
begin
  if usercyre.typ = doublety then
  begin
    currobj := usercyre.db^.cdr;
    SET_PARM_INT( ncycl );
    currobj := F_EVAL( usercyre )
  end
end LSQUSR_CYCL_RES;



[global]
procedure LSQUSR_CYCL_END(  ncycl: integer );
{ Procedure Called by LSQ at the Least-Squares Cycle End Time }
begin
  if usercyen.typ = doublety then
  begin
    currobj := usercyen.db^.cdr;
    SET_PARM_INT( ncycl );
    currobj := F_EVAL( usercyen )
  end
end LSQUSR_CYCL_END;



[global]
procedure LSQUSR_COLL_START(  coll: lsq_ptr );
{ Procedure Called by LSQ at the Least-Squares Data Collection Scan Start Time }
begin
  if usercost.typ = doublety then
  begin
    currobj := usercost.db^.cdr;
    SET_PARM_MEM( coll );
    currobj := F_EVAL( usercost )
  end
end LSQUSR_COLL_START;



[global]
procedure LSQUSR_COLL_END(    coll: lsq_ptr );
{ Procedure Called by LSQ at the Least-Squares Data Collection Scan End Time }
begin
  if usercoen.typ = doublety then
  begin
    currobj := usercoen.db^.cdr;
    SET_PARM_MEM( coll );
    currobj := F_EVAL( usercoen )
  end
end LSQUSR_COLL_END;



[global]
procedure LSQUSR_COLL_CONTRIBUTION( coll:     lsq_ptr;
                                    ncontr,
                                    nobs:     integer;
                                    cndflg:   boolean );
{ Procedure Called by LSQ at the Least-Squares Data Collection Scan List Time }
begin
  if usercopa.typ = doublety then
  begin
    currobj := usercoco.db^.cdr;
    SET_PARM_MEM( coll );
    SET_PARM_INT( ncontr );
    SET_PARM_INT( nobs );
    SET_PARM_BOOL( cndflg );
    currobj := F_EVAL( usercoco )
  end
end LSQUSR_COLL_CONTRIBUTION;



[global]
procedure LSQUSR_COLL_PACKET( coll:     lsq_ptr;
                              nobs:     integer;
                              obs,
                              calc,
                              delta,
                              sigma,
                              weight,
                              delssig:  lsq_real;
                              regflg:   boolean );
{ Procedure Called by LSQ at the Least-Squares Data Collection Packet End Time }
begin
  if usercopa.typ = doublety then
  begin
    currobj := usercopa.db^.cdr;
    SET_PARM_MEM( coll );
    SET_PARM_INT( nobs );
    SET_PARM_FLT( obs );
    SET_PARM_FLT( calc );
    SET_PARM_FLT( delta );
    SET_PARM_FLT( sigma );
    SET_PARM_FLT( weight );
    SET_PARM_FLT( delssig );
    SET_PARM_BOOL( regflg );
    currobj := F_EVAL( usercopa )
  end
end LSQUSR_COLL_PACKET;



[global]
procedure LSQUSR_CONSTRAINT(  coll:        lsq_ptr;
                              obs,
                              calc,
                              delta,
                              sigma,
                              weight,
                              delssig:     lsq_real;
                              regflg:      boolean );
{ Procedure Called by LSQ at the Least-Squares Constraint Use Time }
begin
  if usercons.typ = doublety then
  begin
    currobj := usercons.db^.cdr;
    SET_PARM_MEM( coll );
    SET_PARM_FLT( obs );
    SET_PARM_FLT( calc );
    SET_PARM_FLT( delta );
    SET_PARM_FLT( sigma );
    SET_PARM_FLT( weight );
    SET_PARM_FLT( delssig );
    SET_PARM_BOOL( regflg );
    currobj := F_EVAL( usercons )
  end
end LSQUSR_CONSTRAINT;



[global]
procedure LSQUSR_END_FIT;
{ Procedure Called by LSQ at the Least-Squares Cycle End Time }
begin
    currobj := F_EVAL( userfend )
end LSQUSR_END_FIT;




{
*********************************************************
*                                                       *
*      Routines to Handle a Least-Squares List          *
*                                                       *
*********************************************************
}


[global]
function LSQUSR_LIST_OPEN( lis: lsq_ptr ): boolean;
{ Fonction to Open a LSQ_List by use of lis_descriptor field }
var
  eq: equ_type;
  fl: boolean;

begin
  fl := false;
  with lis^ do
  begin
    lis_current := nil;            { Set lis_current to nil before any read }
    eq.rec      := lis_descriptor; { The descriptor is always a doublet ptr. }
    if eq.lld <> nil then
    with eq.lld^ do
    if lld_rec_access <> nil then
    with lld_rec_access^ do
      case car.typ of
        mrecty:
          begin { *** Linked Block mode *** }
            car := cdr;            { Init the list scan }
            fl  := (car.rec <> nil)
          end;

        nullty, doublety:
          begin { *** QUEUE mode *** }
            car := cdr;            { Init the queue scan }
            fl  := (cdr.typ = doublety)  { Set the flag for the last que. elem.}
          end;

        intty:  { *** Vector mode *** }
          begin
            fl  := (cdr.vect <> nil);
            if fl then car.int := 0  { Set before the first elem. of vector}
            else car.int := -1
          end;

        truety: { *** Lisp CALL model list *** }
          begin                    { Used method for true record file }
            currobj := cdr.db^.cdr;  { Set in the first parameter place }
            SET_PARM_INT( 1 );     { Select Open User Function }
            fl := GET_VALFLAG( F_EVAL( cdr ) )
          end;

      otherwise
      end
  end;
  LSQUSR_LIST_OPEN := fl
end LSQUSR_LIST_OPEN;



[global]
function LSQUSR_LIST_NEXT( lis: lsq_ptr ): boolean;
{ Fonction to Read a LSQ_List by use of lis_descriptor field }
var
  eq:  equ_type;
  ref: obj_ref;
  flg: boolean;

begin
  ref := obj_nil;
  with lis^ do
  begin
    flg := true;
    lis_current := nil;
    eq.rec := lis_descriptor;
    if eq.lld <> nil then
    with eq.lld^ do
    if lld_rec_access <> nil then
      with lld_rec_access^ do
      case car.typ of
        mrecty:                    { *** Linked Block mode *** }
          begin
            ref     := car;
            car.rec := car.rec^.at[0]
          end;

        doublety:                  { *** QUEUE mode *** }
          begin
            ref := car.db^.car;
            car := car.db^.cdr
          end;

        intty:                     { *** Vector mode *** }
          if car.int >= 0 then
            if car.int <= cdr.vect^.vect_size then
            begin
              ref := cdr.vect^.vect_tab[car.int];
              car.int := car.int + 1;
            end
            else car.int := -1;

        truety:                    { *** LISP CALL model list *** }
          begin
            currobj := cdr.db^.cdr;
            SET_PARM_INT( 0 );     { Select Read User Function }
            ref := F_EVAL( cdr )
          end;

      otherwise
      end;

    if ref.typ = mrecty then lis_current := ref.rec
                        else flg := false
  end;
  LSQUSR_LIST_NEXT := flg
end LSQUSR_LIST_NEXT;



[global]
procedure LSQUSR_LIST_CLOSE( lis: lsq_ptr );
{ Function to Close a Least-Squares List }
var
  eq:  equ_type;
  ref: obj_ref;

begin
  with lis^ do
  begin
    lis_current := nil;
    eq.rec      := lis_descriptor;
    if eq.lld <> nil then
    with eq.lld^ do
    if lld_rec_access <> nil then
      with lld_rec_access^ do
      if car.typ = truety then     { LISP CALL model list }
      begin
        currobj := cdr.db^.cdr;
        SET_PARM_INT( -1 );        { Select close user function }
        ref     := F_EVAL( cdr )
      end;
  end
end LSQUSR_LIST_CLOSE;



function CORRELATION_GBL_FACTOR( var vect, matrix: vec_tab ): lsq_real;
var
  dim, i, j, k: integer;
  r, r1, rc:    lsq_real;

begin
  r := 0.0;
  dim := vect.v_size;
  if dim >= 2 then
  begin
    { Initialize the Sommators }
    for i := 1 to dim do  vect[i] := 0.0;
    k := 1;
    for i := 1 to dim - 1 do
    begin
      k := k + 1;
      r1 := 0.0;
      for j := i + 1 to dim do
      begin
        rc := SQR( matrix[k] );
        k := k + 1;
        vect[j] := vect[j] + rc;
        r1 := r1 + rc
      end;
      vect[i] := vect[i] + r1
    end;
    j := dim - 1;
    for i := 1 to dim do
    begin
      r1 := vect[i]/j;
      r := r + r1
    end;
    r := r/dim
  end;
  CORRELATION_GBL_FACTOR := r
end CORRELATION_GBL_FACTOR;



procedure TRANSL_VEXPR( obj: obj_ref ); forward;



procedure TRANSL_FEXPR( obj: obj_ref ); forward;



procedure TRANSL_CASE( pl: obj_ref );
{ To Translate a LISP Case Statement (v_case) to LSQ CASE one }
const
  mdnam = 'TRCA';

var
  ob, vec, oth:    obj_ref;
  eq:              eqv_type;
  p:               lsq_ptr;
  min, max, i, j:  integer;
  bcop:            boolean;

begin
  { Get the Case Vector Address }
  vec := NXT_PAR( pl );
  if vec.typ = vectortyp then
  begin
    { Translate the Selector Formula }
    TRANSL_FEXPR( NXT_PAR( pl ) );
    { Get the min value }
    min := GET_INT( pl, maxint );
    if min <> maxint then
    with vec.vect^ do
    begin
      { Get the max value }
      max    := min + vect_size - 1;
      { Get the otherwise formula }
      oth    := NXT_PAR( pl );
      TRANSL_VEXPR( oth );
      p      := LSQ_C_CASETABLE( min, max, LSQ_POP );
      { Get the case table pointer }
      with p^ do
      for i := 0 to max - min do
      begin
        if not TEST_EQ( vect_tab[i], oth ) then
        { For no Otherwise Code }
        begin { Doublet is Element of Vector }
          bcop := false;
          j := 0;
          while j < i do
          begin { Search for Multiple Use of a Unique Entry }
            if TEST_EQ( vect_tab[j], vect_tab[i] ) then
            begin
              case_table^[1 + j] := case_table^[1 + j];
              j := i;
              bcop := true
            end;
            j := j + 1
          end;
          if not bcop then { New Entry }
          begin
            TRANSL_VEXPR( vect_tab[i] );
            case_table^[1 + i] := LSQ_POP
          end
        end
      end
    end
  end
  else EXEC_ERROR( mdnam, 1351, e_severe )
end TRANSL_CASE;



procedure SETFNC_PARMLST( ob, pl: obj_ref );
{ To set the Effective Value List of Formal List LISP Function }
var
  eq:       equ_type;
  ob1, ob2: obj_ref;

begin
  ob1 := obj_nil;
  while pl.typ = doublety do
  begin
    ob2 := F_EVAL( NXT_PAR( pl ) );    { Get the additional parm list }
    TRANSL_FEXPR( ob2 );           { Translate it in lsq expr }
    eq.lsq       := LSQ_POP;       { Take the result }
    ob2.flg.f    := [];
    ob2.flg.k    := lsq_define;    { Send at a LSQ representation }
    ob2.typ      := areatyp;
    ob2.mem      := eq.mem;        { ... and set it as a lisp block ref. }
    ob2 := F_CONS( ob2, obj_nil ); { Build the parm list }
    if ob1.typ = nullty then
      ob.db^.cdr := ob2            { Link from the connection doublet }
    else
      ob1.db^.cdr := ob2;          { Link from the previous parameter }
    ob1 := ob2
  end
end SETFNC_PARMLST;



procedure TRANSL_LSQ_REFERENCE( ls: lsq_ptr; el: obj_ref );
const
  mdnam = 'TRRF';

var
  ad, idx: integer;
  mr:      mrd_ptr;
  rf:      rfd_ptr;
  fa:      obj_ref;
  eq:      equ_type;

begin  
  if ls <> nil then
  with ls^ do
  case lsq_ndty of
    nd_list: { List Record Reference }
      begin
        eq.rec := lis_descriptor;
        mr     := eq.lld^.lld_rec_descr;   { Get the record descriptor }
        if mr^.mrd_rfdl^.rfd_atm = nil then  { Array Reference }
          rf := mr^.mrd_rfdl
        else
        begin
          fa := GET_ATOM( NXT_PAR( el ), true );
          with fa.at^ do
            if val.typ = rfdty then rf := val.rfd
                               else EXEC_ERROR( mdnam, 258, e_severe );
        end;
        { Check field match with record definition }
        with rf^ do
        begin
          if rfd_mrd <> mr then EXEC_ERROR( mdnam, 256, e_severe );
          if el.typ = doublety then ad := INTEVL( el )
                               else ad := 1;
          if (ad < 1) or (ad > rfd_dim) then
          begin
            ad := 1;
            EXEC_ERROR( mdnam, 259, e_error )
          end;
          ad := ad + rfd_off - 1;  { Get field offset }
          LSQ_LSPUSH( ls );
          LSQ_IPUSH( ad );
          case rfd_typ.typ of
            intub:    LSQ_OPER( nd_ub_listref );
            intsb:    LSQ_OPER( nd_sb_listref );
            intuw:    LSQ_OPER( nd_uw_listref );
            intsw:    LSQ_OPER( nd_sw_listref );
            intty:    LSQ_OPER( nd_li_listref );
            sflty:    LSQ_OPER( nd_fl_listref );
            flty:     LSQ_OPER( nd_db_listref );
          otherwise
            LSQ_OPER( nd_listref ) { Expression case }
          end;
          mr := nil
        end
      end;

  otherwise
    EXEC_ERROR( mdnam, 1354, e_severe )
  end
  else EXEC_ERROR( mdnam, 1354, e_severe )
end TRANSL_LSQ_REFERENCE;



procedure TRANSL_EXPR( obj: obj_ref );
{ Translation of a LISP Expression to the LSQ Equivalent one }
const
  mdnam = 'TREX';

var
  save_flg, flg_flg:         boolean;
  i1, i2, n:                 integer;
  r:                         lsq_real;
  sav_point, eflist,
  calist, obty, memo, memo1: obj_ref;
  eq:                        equ_type;

begin { TRANSL_EXPR }
  sav_point  := curr_point;
  curr_point := obj;
  save_flg   := deriv_flg;         { Save the derivative form flag }
  deriv_flg  := false;             { Assume no derivation operator form }
  flg_flg    := false;             { Assume no atom flag use }

  case obj.typ of

    nullty: LSQ_NULL_PUSH;

    intty: begin  r := obj.int; LSQ_C_KONST( r )  end;

    sflty,
    flty: LSQ_C_KONST( obj.flt );  { Create a constante node }

    atomety: TRANSL_EXPR( obj.at^.val );

    areatyp:{ Possible Reference of Least-Squares Parameter, Variable or List }
      if obj.flg.k = lsq_define then
      begin { object defined by LISP$_LSQ }
        eq.mem := obj.mem;
        with eq.lsq^ do
        case lsq_ndty of
          nd_varbl, nd_parm, nd_index:
            begin
              LSQ_LSPUSH( eq.lsq );
              deriv_flg := true
            end;

          nd_list: LSQ_LSPUSH( eq.lsq );

        otherwise
          EXEC_ERROR( mdnam, 1352, e_severe )
        end
      end
      else
        EXEC_ERROR( mdnam, 1352, e_severe );

    doublety:
      begin { *** Doublet LISP Expression *** }
        eflist := obj.db^.cdr;     { Get the parameter list }
        calist := obj.db^.car;     { Get the function definition }
        if calist.typ >= atomety then
        with calist.at^ do
          if val.flg.k = lsq_define then
            calist := val
          else
            case fncref.flg.k of
              de__funct:
                begin
                  
                  flg_flg := true
                end;

              df__funct:
                begin

                  flg_flg := true
                end;

            otherwise
            end;

        if not flg_flg then
        case calist.typ of

          areatyp:   if calist.flg.k = lsq_define then
                     begin { Least-Squares Reference Function }
                       eq.mem := calist.mem;
                       TRANSL_LSQ_REFERENCE( eq.lsq, eflist )
                     end
                     else
                       EXEC_ERROR( mdnam, 1399, e_severe );

          atomety: begin           { User Defined Function }
                   end;

          quot_fnc:  { Quote function must be ignored here }
                     TRANSL_EXPR( NXT_PAR( eflist ) );

          eval_fnc:  TRANSL_EXPR( F_EVAL( NXT_PAR( eflist ) ) );

          doublety:  begin         { User LISP Function Call }
                       { Get the LISP Expression in a Connection Doublet }
                       calist := F_CONS( calist.db^.car, obj_nil );
                       SETFNC_PARMLST( calist, eflist );
                       eq.mem := calist.mem;
                       LSQ_IPUSH( 0 );     { Set the call index not used }
                       LSQ_LSPUSH( eq.lsq );
                       LSQ_OPER( nd_user_call )
                     end;

          and_fnc:   begin
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( nd_and )
                     end;

          or_fnc:    begin
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( nd_or )
                     end;

          null_fnc:  begin
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( nd_not )
                     end;

          if_fnc:    begin
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( nd_ifdir )
                     end;

          v_case_fnc: TRANSL_CASE( eflist );


          iadd_fnc:  { (+ ... ) }
                     begin
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( nd_add );
                       while eflist.typ <> nullty do
                       begin
                         TRANSL_EXPR( NXT_PAR( eflist ) );
                         LSQ_OPER( nd_add )
                       end
                     end;

          isub_fnc:  { (- ... ) }
                     begin
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( nd_sub );
                       while eflist.typ <> nullty do
                       begin
                         TRANSL_EXPR( NXT_PAR( eflist ) );
                         LSQ_OPER( nd_sub )
                       end
                     end;

          imul_fnc:  { (* ... ) }
                     begin         
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       if deriv_flg then
                         LSQ_OPER( nd_mul_d )
                       else
                         LSQ_OPER( nd_mul );
                       while eflist.typ <> nullty do
                       begin
                         TRANSL_EXPR( NXT_PAR( eflist ) );
                         if deriv_flg then
                           LSQ_OPER( nd_mul_d )
                         else
                           LSQ_OPER( nd_mul )
                       end
                     end;


          idiv_fnc:  { (I_DIV ... ) }
                     begin
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( nd_div );
                       LSQ_OPER( nd_trunc );
                       deriv_flg := false
                     end;

          imod_fnc:   { (MOD ... ) }
                     begin         
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( nd_mod );
                       deriv_flg := false
                     end;

          irem_fnc:   { (REM ... ) }
                     begin         
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( nd_rem );
                       deriv_flg := false
                     end;

          succ_fnc:  { (1+ ... ) }
                     begin
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       LSQ_C_KONST( 1.0 );
                       LSQ_OPER( nd_add )
                     end;

          pred_fnc:  { (1- ... ) }
                     begin
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       LSQ_C_KONST( -1.0 );
                       LSQ_OPER( nd_add )
                     end;

          iabs_fnc:
                     begin
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       if deriv_flg then
                         LSQ_OPER( nd_abs_d )
                       else
                         LSQ_OPER( nd_abs );
                     end;

          ineg_fnc:
                     begin
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( nd_neg )
                     end;

          f_div_fnc: { (/ ... ) }
                     begin
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       if deriv_flg then
                         LSQ_OPER( nd_div_d )
                       else
                         LSQ_OPER( nd_div );
                       while eflist.typ <> nullty do
                       begin
                         TRANSL_EXPR( NXT_PAR( eflist ) );
                         if deriv_flg then
                           LSQ_OPER( nd_div_d )
                         else
                           LSQ_OPER( nd_div )
                       end
                     end;

          ipow_fnc:  { (** ...) }
                     begin
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       eflist := NXT_PAR( eflist );
                       if eflist.typ = intty then
                       begin
                         LSQ_IPUSH( eflist.int );
                         if deriv_flg then
                           LSQ_OPER( nd_ipw_d )
                         else
                           LSQ_OPER( nd_ipw )
                       end
                       else
                       begin
                         TRANSL_EXPR( eflist );
                         if deriv_flg then
                           LSQ_OPER( nd_pow_d )
                         else
                           LSQ_OPER( nd_pow )
                       end
                     end;

        f_exp_fnc,   f_log_fnc,   f_sqrt_fnc,  f_sin_fnc,   f_sind_fnc,
        f_cos_fnc,   f_cosd_fnc,  f_tan_fnc,   f_tand_fnc,  f_asin_fnc,
        f_asind_fnc, f_acos_fnc,  f_acosd_fnc, f_atan_fnc,  f_atand_fnc,
        f_sinh_fnc,  f_cosh_fnc,  f_tanh_fnc,
        f_asinh_fnc, f_acosh_fnc, f_atanh_fnc:
                     begin
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       if deriv_flg then
                         LSQ_OPER( cd_tab_d[calist.typ] )
                       else
                         LSQ_OPER( cd_tab_n[calist.typ] )
                     end;

        f_phase_fnc, f_phased_fnc:
                     begin
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       if deriv_flg then
                         LSQ_OPER( cd_tab_d[calist.typ] )
                       else
                         LSQ_OPER( cd_tab_n[calist.typ] )
                     end;


        f_bess1_fnc: begin
                       calist := NXT_PAR( eflist );
                       if calist.typ <> intty then
                         EXEC_ERROR( mdnam, 1355, e_severe )
                       else
                         i1 := calist.int;
                       TRANSL_EXPR( NXT_PAR( eflist ) );
                       LSQ_IPUSH( i1 );
                       if deriv_flg then
                         LSQ_OPER( nd_bess1_d )
                       else
                         LSQ_OPER( nd_bess1 )
                     end;
 
    f_interpol_fnc:  begin
                       memo1 := GET_LIST( eflist, true ); { Get interp. list }
                       if memo1.typ = doublety then
                       begin
                         memo1 := memo1.db^.cdr; { Skip the QUEUE header }
                         memo  := memo1;
                         { Size the Interpolation A list }
                         n := 0;
                         while (memo.typ = doublety) do
                         begin  memo := memo.db^.cdr; n := SUCC( n )  end;
                         if n > 0 then
                         begin { Get the A list of values }
                           eq.rec := NEW_RECORD_ALLOC( n*2*size_doub );
                           { Build the Interpolation Table }
                           i1 := 0;
                           while (memo1.typ = doublety) do
                           with memo1.db^ do
                           begin
                             if car.typ = doublety then
                             with car.db^ do
                             begin
                               eq.rec^.gt[i1] := FLTVAL( car );
                               i1 := SUCC( i1 );
                               eq.rec^.gt[i1] := FLTVAL( cdr );
                               i1 := SUCC( i1 )
                             end;
                             memo1 := cdr
                           end;
                           LSQ_IPUSH( n );       { Push the table size }
                           LSQ_RECPUSH( eq.rec ) { and the table address }
                         end
                         else
                         begin
                           LSQ_IPUSH( 0 );
                           LSQ_RECPUSH( nil )
                         end
                       end;
                       TRANSL_EXPR( NXT_PAR( eflist ) ); { Get x expression }
                       if deriv_flg then LSQ_OPER( nd_interpol_d )
                                    else LSQ_OPER( nd_interpol )
                     end;

      f_integr_fnc:  begin
                       memo1 := GET_ATOM( NXT_PAR( eflist ), true );
                       with memo1.at^ do
                       begin
                         memo := val;      { Save the index atom value }
                         i1 := identcp; identcp := identcp + 1;
                         LSQ_C_LINDEX( i1 ); { Create the LSQ index }
                         { Save the index reference in the atom }
                         eq.lsq    := fit_idxlast;
                         val.flg.f := [];
                         val.flg.k := lsq_define;
                         val.typ   := areatyp;
                         val.mem   := eq.mem;
                         LSQ_LSPUSH( eq.lsq );
                         { Get Gauss list array ref. atom }
                         GET_REC_ARRAY( eflist, flty, eq.rec, n );
                         LSQ_IPUSH( n div 2 );
                         LSQ_RECPUSH( eq.rec );
                         TRANSL_EXPR( NXT_PAR( eflist ) ); { Get the expr. }
                         val := memo;        { Restore the index'atom value }
                         if deriv_flg then LSQ_OPER( cd_tab_d[calist.typ] )
                                      else LSQ_OPER( cd_tab_n[calist.typ] )
                       end
                     end; 

        f_summ_fnc:  begin
                       memo1 := GET_ATOM( NXT_PAR( eflist ), true );
                       with memo1.at^ do
                       begin
                         memo := val;  { Save the index atom value }
                         i1 := identcp; identcp := identcp + 1;
                         LSQ_C_LINDEX( i1 ); { Create the LSQ index }
                         { Save the index reference in the atom }
                         eq.lsq    := fit_idxlast;
                         val.flg.f := [];
                         val.flg.k := lsq_define;
                         val.typ   := areatyp;
                         val.mem   := eq.mem;
                         LSQ_LSPUSH( eq.lsq );
                         { Get the begin value }
                         TRANSL_FEXPR( NXT_PAR( eflist ) );
                         TRANSL_FEXPR( NXT_PAR( eflist ) ); { Get the end value }
                         TRANSL_FEXPR( NXT_PAR( eflist ) ); { Get the step }
                         TRANSL_EXPR( NXT_PAR( eflist ) );  { Get the expression }
                         val := memo;  { Restore the index'atom value }
                         if deriv_flg then LSQ_OPER( cd_tab_d[calist.typ] )
                                      else LSQ_OPER( cd_tab_n[calist.typ] )
                       end
                     end;
 

        f_float_fnc: TRANSL_FEXPR( NXT_PAR( eflist ) );

        f_round_fnc,
        f_trunc_fnc: begin
                       TRANSL_FEXPR( NXT_PAR( eflist ) );
                       LSQ_OPER( cd_tab_n[calist.typ] );
                       deriv_flg := false
                     end;


          ilt_fnc,
          ile_fnc,
          igt_fnc,
          ige_fnc,
          ieq_fnc,
          ine_fnc: begin
                     TRANSL_FEXPR( NXT_PAR( eflist ) );
                     TRANSL_FEXPR( NXT_PAR( eflist ) );
                     deriv_flg := false;
                     case calist.typ of
                       ilt_fnc: LSQ_OPER( nd_lt );
                       ile_fnc: LSQ_OPER( nd_le );
                       igt_fnc: LSQ_OPER( nd_gt );
                       ige_fnc: LSQ_OPER( nd_ge );
                       ieq_fnc: LSQ_OPER( nd_eq );
                       ine_fnc: LSQ_OPER( nd_ne );
                     end
                   end;

        otherwise
           EXEC_ERROR( mdnam, 1356, e_severe );
        end
      end
  otherwise
    EXEC_ERROR( mdnam, 1357, e_severe );
  end { case };
  deriv_flg  := deriv_flg or save_flg;
  curr_point := sav_point;
  if not deriv_enables then deriv_flg := false
end TRANSL_EXPR;



procedure TRANSL_VEXPR { ( obj: obj_ref ); was forward };
begin
  deriv_enables := true;
  deriv_flg := true;
  TRANSL_EXPR( obj )
end TRANSL_VEXPR;



procedure TRANSL_FEXPR { ( obj: obj_ref ) was forward };
var
  save_derena: boolean;

begin
  save_derena := deriv_enables;
  deriv_enables := false;
  TRANSL_EXPR( obj );
  deriv_enables := save_derena
end TRANSL_FEXPR;



procedure TRANSL_PROC ( proc: obj_ref ); forward;



procedure TRANSL_EXEC( obj: obj_ref ); forward;



[global]
function LISP$_LSQ( idx: integer; ll: obj_ref ): obj_ref;
const
  mdnam = 'LSQM';

var
  eq:                 equ_type;
  s1, s2, s3:         rec_ptr;
  o0, o1, o2, o3, re: obj_ref;
  mr:                 mrd_ptr;
  rf:                 rfd_ptr;
  lc, ls, l1, l2:     lsq_ptr;
  st:                 statis_ptr;
  r1, r2, r3:         lsq_real;
  i1, i2, i3, ix:     integer;
  tp:                 stkp_typ;
  mf:                 scaf_types;
  n1:                 lsq_nodetypes;
  f1, f2, f3, f4, f5: boolean;

begin { LISP$_LSQ }
  re := obj_nil;
  case idx of

    {****************************************}
    {*********  Activation entries  *********}
    {****************************************}

    1, { To Initialize the Least-Squares Loader System }
    5: { Change the LSQUSR LISP Procedure }
       begin
         { Get the User Change Variable Signal Procedure }
         uservarbl := INSTALL_LISP_FUNC( ll );

         { Get the User Correlation Procedure }
         usercorr  := INSTALL_LISP_FUNC( ll );

         { Get the User Singularity Signal Procedure }
         usersing  := INSTALL_LISP_FUNC( ll );

         { Get the Cycle Start Notification }
         usercyst := INSTALL_LISP_FUNC( ll );

         { Get the Cycle Solve Notification }
         usercyre := INSTALL_LISP_FUNC( ll );

         { Get the Cycle End Notification }
         usercyen := INSTALL_LISP_FUNC( ll );

         { Get the Collect Start Notification }
         usercost := INSTALL_LISP_FUNC( ll );

         { Get the Collect End Notification }
         usercoen := INSTALL_LISP_FUNC( ll );

         { Get the User Collect Packet Notification }
         usercoco := INSTALL_LISP_FUNC( ll );

         { Get the User Collect Packet Notification }
         usercopa := INSTALL_LISP_FUNC( ll );

         { Get the User Constraint Notification }
         usercons := INSTALL_LISP_FUNC( ll );

         { Get the User FIT End Notification }
         userfend := INSTALL_LISP_FUNC( ll );

         if idx = 1 then
         begin
           list_ref_fnc  := obj_nil;    { No flag functions defined }
           lisp_eval_fnc := obj_nil;
           identcp := 1;
           LSQ_INIT_LSQ
         end
       end;

    2: { Initialize the Least-Squares Kernel and Build the Run Graph. }
       begin
         f1 := LSQ_END( true );         { Get the Start Cycle Directive }
         if fit_nvarbl = 0 then LSQ_INIT_STRUCTURE
       end;

    3: { Activate the Least-Squares Run }
       begin
         f1 := LSQ_END( true );         { Get the Start Cycle Directive }
         if fit_nvarbl = 0 then LSQ_INIT_STRUCTURE;
         i1 := INTEVLDEF( ll, 1 );      { Get the number of cycle to perform }
         i2 := INTEVLDEF( ll, 1 );      { Get the first cycle ident. }
         i3 := INTEVLDEF( ll, 10 );     { Get the maximum of singul./cycle }
         r1 := FLTEVLDEF( ll, 0.0 );    { Get the minimal matrix pivot }
         f1 := GET_FLAG( ll );          { Get the suppress matrix comp. flag }
         f2 := GET_FLAG( ll );          { Get the no final cycle flag }
         f3 := GET_FLAG( ll );          { Get the correl. matrix comp. flag }
         f4 := GET_FLAG( ll );          { Get the no varbl chi2 depend flag }

         err_nohandler := true;         { Disable the LISP Condition Handler }
         LSQ_FIT( i1, i2, i3, r1, f1, f2, f3, f4 );
         err_nohandler := false         { Re-Enable the LISP Condition Handler }
       end;

    4: { Free the Run Graph to Edit the Least-Squares Structures }
       LSQ_FREE_STRUCTURE;



    9: { Define the Used Flag Function in the Following order :
          (SYS_CALL 9 <list_ref_fnc> <lisp_eval_fnc>)
       }
       begin
         o1 := F_EVAL( NXT_PAR( ll ) );
         if o1.typ <> nullty then list_ref_fnc := GET_ATOM( o1, true );
         o1 := F_EVAL( NXT_PAR( ll ) );
         if o1.typ <> nullty then lisp_eval_fnc := GET_ATOM( o1, true )
       end;

    {*********************************************}
    {*********  Build structure entries  *********}
    {*********************************************}


   10: { Create a List-Squares Expression }
       begin
         o1 := F_EVAL( NXT_PAR( ll ) ); { Get the LISP Expression }
         o2 := F_EVAL( NXT_PAR( ll ) ); { Get the Derivation Flag }
         if o2.typ <> nullty then
           TRANSL_VEXPR( o1 )           { Formula Translation for Derivable }
         else
           TRANSL_FEXPR( o1 );          { Formula Translation for not Deriv. }
         if GET_VALFLAG( F_EVAL( NXT_PAR( ll ) ) ) then
         begin
           eq.lsq := LSQ_POP;
           re.flg.f    := [];
           re.flg.k    := lsq_define;
           re.typ      := areatyp;
           re.mem      := eq.mem
         end
       end;



   11: { Create a Least-Squares Variable }
       begin
         NAME_LSQ_OBJ( ll, o1, s1 );    { Get the variable identifier }
         r1 := FLTEVLDEF( ll, 0.0 );    { Get the variable value }
         r2 := FLTEVLDEF( ll, 0.0 );    { Get the variable sigma }
         i1 := identcp;                 { Get a variable number }
         identcp := identcp + 1;
         ls := GET_EVL_LSQ( ll, true ); { Get a limit specification }
         if ls = nil then i2 := 0
         else
           with ls^ do
           if lsq_ndty = nd_limits then { The object must be limits }
             i2 := lim_id
           else
             EXEC_ERROR( mdnam, 1357, e_error );
           
         LSQ_C_VARBL( s1, r1, r2, i1, i2 ); { Create the Least-Squares var. }
         { set the attached atom value }
         re := SET_PTR_VALUE( o1, LSQ_S_VARBL( i1 ) )
       end;

   12: { Create a Least-Squares Parameter }
       begin
         if ll.db <> nil then
           if ll.db^.car.typ = intty then
           begin
             o1 := NXT_PAR( ll ); i2 := o1.int;
             f2 := ODD( i2 ); i2 := i2 div 2;
             f3 := ODD( i2 ); i2 := i2 div 2;
             f4 := ODD( i2 );
             f1 := i2 > 1;
             f5 := false
           end else f5 := true
         else f5 := true;
         NAME_LSQ_OBJ( ll, o1, s1 );
         i1 := identcp;
         identcp := identcp + 1;
         { Get the Parameter Definition Expression }
         o2 := F_EVAL( NXT_PAR( ll ) );
         if f5 then
         begin
           f1 := GET_FLAG( ll );        { Sum flag }
           f2 := GET_FLAG( ll );        { Derivation flag }
           f3 := GET_FLAG( ll );        { Sigma flag }
           f4 := GET_FLAG( ll )         { Cache flag }
         end;
         { Push the Parameter Definition Expression Translation }
         if f3 then f2 := true;         { Sigma flag => Derivation flag }
         if f2 then TRANSL_VEXPR( o2 )
               else TRANSL_FEXPR( o2 );
         LSQ_C_PARM( s1, i1, f1, f2, f3, f4 );
         re := SET_PTR_VALUE( o1, fit_parlast )
       end;

   13: { Create a List-Squares Variable Limits Block }
       begin
         NAME_LSQ_OBJ( ll, o1, s1 );    { Get the Limit Block Name }
         r1 := FLTEVLDEF( ll, -1E-20 );
         r2 := FLTEVLDEF( ll,  1E+20 );
         i1 := identcp;
         identcp := identcp + 1;
         LSQ_C_LIMITS( i1, r1, r2 );
         re := SET_PTR_VALUE( o1, fit_limlast )
       end;

   14: { Create a List-Squares Diagonal block }
       begin
         NAME_LSQ_OBJ( ll, o1, s1 );
         o2 := F_EVAL( NXT_PAR( ll ) ); { Get the Damping Factor Expr. }
         o3 := F_EVAL( NXT_PAR( ll ) ); { Get the Marquardt Factor Expr. }
         i1 := identcp;
         identcp := identcp + 1;
         TRANSL_FEXPR( o2 );            { Push transl. of damp. f. expr. }
         TRANSL_FEXPR( o3 );            { Push transl. of marquardt f. expr. }
         LSQ_C_DIABLK( s1, i1 );
         LSQ_OP_DIABLK( i1 );           { Set this diagonal block as opened }
         re := SET_PTR_VALUE( o1, fit_blklast )
       end;
    
   15: { Create a List-Squares List }
       begin
         NAME_LSQ_OBJ( ll, o1, s1 );    { Get the List Name (and attached atom)}
         o2 := GET_ATOM( NXT_PAR( ll ), true ); { Get a Record Type/Rec Atom }
         with o2.at^ do
           if fncref.typ <> mrdty then EXEC_ERROR( mdnam, 254, e_severe )
                                  else mr := fncref.mrd;

         o2 := F_EVAL( NXT_PAR( ll ) ); { Get the List Related Exec Sequence }
         i1 := identcp;                 { Allocate it an ident. # }
         identcp := identcp + 1;
         i2 := INTEVLDEF( ll,  0 );     { Get the Cache Size }
         i3 := INTEVLDEF( ll, -1 );     { Get the Cache Identifier Offset }
         LSQ_C_LIST( s1, i1, i2, i3 );
         { Set the List Atom Value }
         re := SET_PTR_VALUE( o1, fit_listlast );
         NEW( eq.lld );
         fit_listlast^.lis_descriptor := eq.rec;
         with eq.lld^ do
         begin { Set the Record Type Descriptor }
           lld_rec_descr   := mr;       { Set the Record Descriptor }
           lld_rec_access  := nil       { Set to no defined LISP Descriptor }
         end;

         { *** LSQ LIST related directive list generation *** }
         tp := stkp;
         TRANSL_PROC( o2 );             { Generate the EXEC Sequence }
         if stkp = tp then  { Take Conditions Expr as Additional Parameters }
         begin
           o1 := F_EVAL( NXT_PAR( ll ) );
           if o1.typ = nullty then LSQ_NULL_PUSH
                              else TRANSL_FEXPR( o1 );
           o1 := F_EVAL( NXT_PAR( ll ) );
           if o1.typ = nullty then LSQ_NULL_PUSH
                              else TRANSL_FEXPR( o1 )
         end
         else
           { at the Exec Sequence Ending the Stack Must be Filling with }
           { top level     : eligibility condition }
           { top level - 1 : absolute stop condition }
           { Push default stop cond and eligibility }
           while stkp < tp + 2 do LSQ_NULL_PUSH;

         f1 := LSQ_END( true )
       end;

   16: { Create a List-Squares Data Collection entry }
       begin
         NAME_LSQ_OBJ( ll, o1, s1 );    { Get the Coll. Atome (and att. atom)}
         { Get the related list areatyp reference }
         ls := GET_LSQLIST( ll, false );
         o2 := F_EVAL( NXT_PAR( ll ) ); { Get the Packet Related Exec Seq. }
         if ls <> nil then
           o3 := F_EVAL( NXT_PAR( ll ) ){ Get the End Related Exec Seq. }
         else
           o3 := obj_nil;               { Only one Sequence for Constraint }
         r1 := FLTEVLDEF( ll, 1E30 );   { Get the Optional Rejection Factor }
         LSQ_C_COLLECT( s1, ls );       { Create the Data Collection }
         lc := fit_colllast;            { Get the Data Collection Pointer }
         re := SET_PTR_VALUE( o1, lc ); { Set the Resulting LISP ref. }

         { *** LSQ DATA related directive list generation *** }
         tp := stkp;                    { Save the Initial Stack Level }
         LSQ_RPUSH( r1 );               { Push the Rejection Factor }
         TRANSL_PROC( o2 );             { Generate packet/constr EXEC sequence }

         if stkp = tp + 1 then
         begin  { When any value in the stack }
           o1 := F_EVAL( NXT_PAR( ll ) );
           if o1.typ = nullty then LSQ_NULL_PUSH
                              else TRANSL_FEXPR( o1 ); { For Weight }
           o1 := F_EVAL( NXT_PAR( ll ) );
           if o1.typ = nullty then LSQ_NULL_PUSH
                              else TRANSL_FEXPR( o1 ); { For Sigma }
           o1 := F_EVAL( NXT_PAR( ll ) );
           if o1.typ = nullty then LSQ_NULL_PUSH
                              else TRANSL_FEXPR( o1 ); { For Observed }
           o1 := F_EVAL( NXT_PAR( ll ) );
           if o1.typ = nullty then LSQ_NULL_PUSH
                              else TRANSL_VEXPR( o1 ); { For Computed }
           o1 := F_EVAL( NXT_PAR( ll ) );
           if o1.typ = nullty then LSQ_NULL_PUSH
                              else TRANSL_FEXPR( o1 ); { For Isent }
         end;
         { Values in the stack }
         { at the Exec Sequence Ending the Stack Must Be Filling With : }
         { top level     : sentinel expression }
         { top level - 1 : computed (must be a parameter) }
         { top level - 2 : observed }
         { top level - 3 : sigma }
         { top level - 4 : weight }
         { top level - 5 : reject }
         if stkp <> tp + 6 then
           EXEC_ERROR( mdnam, 1358, e_fatal )
         else
         begin
           f1 := (o3.typ = nullty );
           f1 := LSQ_END( f1 );
           if not f1 then
           begin
             TRANSL_PROC( o3 );
             f1 := LSQ_END( true )
           end
         end
       end;


   19: { Fit Related Lsq Directive List }
       begin
         o1 := F_EVAL( NXT_PAR( ll ) );
         o2 := F_EVAL( NXT_PAR( ll ) );
         NEW_CONTEXT( nil );            { Create the by "By cycle context" }
         TRANSL_PROC( o1 );             { Set the "By cycle" directive list }
         f1 := o2.typ <> nullty;
         f1 := LSQ_END( f1 );           { Return to base context }
         TRANSL_PROC( o2 );             { Set the "Final" directive list }
         f1 := LSQ_END( true )
       end;



    {********************************************}
    {*********   Exec Directive Entries *********}
    {********************************************}

   21: { Scan directive }
       begin
         { get the related list areatyp reference }
         ls  := GET_LSQLIST( ll, true );
         i1  := INTEVLDEF( ll, 0);          { get the scan mode }
         case i1 of
           1: mf := scaf_partial;
           2: mf := scaf_window;
         otherwise
           mf    := scaf_complete
         end;
         o1 := F_EVAL( NXT_PAR( ll ) );     { Get the packet related exec seq. }
         o2 := F_EVAL( NXT_PAR( ll ) );     { Get the scan end related exec seq. }
         LSQ_SCANDIR( ls, mf );             { Generate the Scan Directive }
         ls := define_scope[define_current - 1].last_dir;
         re := GEN_LISP_PTR( ls );

         { *** LSQ SCAN related directive list generation *** }
         tp := stkp;
         TRANSL_PROC( o1 );                 { generate packet the EXEC sequence }
         { at the exec sequence ending the stack must be filling with }
         { top level     : condition expr. }
         if stkp <= tp then
         begin
           o3 := F_EVAL( NXT_PAR( ll ) );
           if o3.typ = nullty then LSQ_NULL_PUSH { Push default condition }
                              else TRANSL_FEXPR( o3 ) { for condit }

         end;
         f1 := (o2.typ = nullty );
         f1 := LSQ_END( f1 );
         if not f1 then
         begin
           TRANSL_PROC( o2 );
           f1 := LSQ_END( true )
         end
       end; 


   22: { List Init directive }
       begin
         { get the related list areatyp reference }
         ls := GET_LSQLIST( ll, true );
         LSQ_LSPUSH( ls );
         LSQ_EXEC_DIR( nd_initlist )
       end;


   23: { If directive }
       begin
         o1 := F_EVAL( NXT_PAR( ll ) );
         o2 := F_EVAL( NXT_PAR( ll ) );
         o3 := F_EVAL( NXT_PAR( ll ) );
         TRANSL_FEXPR( o1 );
         LSQ_IF;
         TRANSL_PROC( o2 );
         if o3.typ <> nullty then
         begin
           LSQ_ELSE;
           TRANSL_PROC( o3 ) 
         end;
         LSQ_ENDIF
       end;

   24: { Case directive }
       begin
         { get the selector expression }
         TRANSL_FEXPR( F_EVAL( NXT_PAR( ll ) ) );
         { get entry sequence }
         o1 := F_EVAL( NXT_PAR( ll ) );
         LSQ_CASE;
         while o1.typ = doublety do
         { loop on each case sequence }
         begin
           o2 := NXT_PAR( o1 );             { get value label list }
           repeat
             LSQ_IPUSH( INTEVL( o2 ) )
           until o2.typ <> doublety;
           TRANSL_PROC( NXT_PAR( o1 ) );
           LSQ_WHEN;
         end;
         { get otherwise sequence }
         TRANSL_PROC( F_EVAL( NXT_PAR( ll ) ) );
         LSQ_ENDCASE
       end;


   25: { Store Directive }
       begin
         ls := GET_LSQLIST( ll, true );     { get the memory ref. }
         eq.rec := ls^.lis_descriptor;
         mr  := eq.lld^.lld_rec_descr;      { get the LSQ LIST record descriptor }
         if mr^.mrd_rfdl^.rfd_atm = nil then { array reference }
           rf := mr^.mrd_rfdl
         else
         begin { record field reference }
           o1  := GET_ATOM( NXT_PAR( ll ), true ); { get the field atom }
           with o1.at^ do
             if val.typ <> rfdty then EXEC_ERROR( mdnam, 1359, e_severe )
                                 else rf := val.rfd
         end;
         with rf^ do
         begin
           { check for list field }
           if rfd_mrd <> mr then EXEC_ERROR( mdnam, 256, e_severe );
           i1 := rfd_off;                   { get the field offset }
           { Index is allowed only for array with two or more elements }
           if rfd_dim > 1 then
           begin
             i2 := INTEVL( ll );            { get index for array }
             { check index value in allowed range }
             if (i2 < 1) or (i2 > rfd_dim) then
               EXEC_ERROR( mdnam, 259, e_severe );
             i1 := i1 + i2 - 1
           end;

           if ll.typ = doublety then
           begin { The value to load is directly specified here }
             o2 := F_EVAL( NXT_PAR( ll ) ); { get the LISP expression }
             TRANSL_FEXPR( o2 )             { translate the value to store }
           end;

           LSQ_LSPUSH( ls );                { push the list reference }
           LSQ_IPUSH( i1 );                 { push the offset }
           case rfd_typ.typ of              { select appropriate store fnc. }
             intub:  n1 := nd_ub_listref;
             intsb:  n1 := nd_sb_listref;
             intuw:  n1 := nd_uw_listref;
             intsw:  n1 := nd_sw_listref;
             intty:  n1 := nd_li_listref;
             sflty:  n1 := nd_fl_listref;
             flty:   n1 := nd_db_listref;
           otherwise
             EXEC_ERROR( mdnam, 1361, e_severe );
           end;
           LSQ_EXEC_DIR( n1 )
         end
       end;

   26: { User Lisp Procedure Call Directive }
       begin
         o1 := F_EVAL( NXT_PAR( ll ) );     { get the LISP expression }
         o3 := obj_nil;
         o1 := F_CONS( o1, obj_nil );       { allocate a connection doublet }
         SETFNC_PARMLST( o1, ll );          { handle the parameter list }
         LSQ_IPUSH( 0 );                    { set the call index not used }
         eq.mem := o1.mem;
         LSQ_LSPUSH( eq.lsq );
         LSQ_EXEC_DIR( nd_user_call )
       end;


    {*********************************************}
    {*********   Data management entries *********}
    {*********************************************}

   31: { Append List Data record }
       begin
         { Get the related list areatyp reference }
         ls := GET_LSQLIST( ll, true );
         with ls^ do
         begin
           o1 := F_EVAL( NXT_PAR( ll ) );
           { o1 can be memory block list structured as :
              a linked list of record,
              a simple LISP list,
              a QUEUE as build by the QUEUE_PUT LISP function.
              a LISP VECTOR,
              a user LISP function resulting record.
           }
           o2 := obj_nil;
           if o1.typ = doublety then
           with o1.db^ do
             if (car.typ = doublety) or
                ((car.typ = nullty) and (cdr.typ = nullty)) then
               o2 := o1;

           if o2.typ = nullty then
           begin
             o2 := F_CONS( obj_nil, o1 );
             with o2.db^ do
             case o1.typ of
               mrecty:                      { linked list of record }
                 car := cdr;                { set the initial car record pointer }

               doublety:                    { List (not queue), set the queue mode }
                 begin { Build a queue descriptor }
                   o3 := o1;
                   while o3.typ = doublety do
                   begin
                     car := o3;             { build the Queue last link }
                     o3 := o3.db^.cdr
                   end
                 end;


               vectortyp:                   { LISP vector }
                 begin
                   car.typ := intty;
                   car.int := -1
                 end;

               truety:                      { user function }
                 begin
                   car := o1;               { set the function code T }
                   cdr := GET_LIST( ll, true ) { set LISP call seq. }
                 end

             otherwise
               o1 := obj_nil
             end
           end;

           eq.rec := lis_descriptor;        { get the descriptor block }
           eq.lld^.lld_rec_access := o2.db;
           lis_current := nil               { init to closed state }
         end
       end;

   32: { Set a List Copy Data record }
       begin
         { get the related list areatyp reference }
         ls  := GET_LSQLIST( ll, true );    { get the list to load }
         l1  := GET_LSQLIST( ll, true );    { get the list to copy }
         with ls^ do
         begin
           lis_endflg     := l1^.lis_endflg;
           lis_current    := l1^.lis_current;
           lis_descriptor := l1^.lis_descriptor
         end
       end;

   33: { select or deselect a given data collection }
       begin
         ls  := GET_AREA( ll, true );       { get the data collection }
         with ls^ do
           if lsq_ndty = nd_collect then
             coll_enable := GET_EVLFLAG( ll )
           else
             EXEC_ERROR( mdnam, 1360, e_severe )
       end;



    {********************************************}
    {*********   Edit structure entries *********}
    {********************************************}

   41: { Open a diagonal block }
       begin
         { get the diagonal block areatyp reference }
         ls := GET_AREA( ll, false );
         if ls <> nil then
           if ls^.lsq_ndty = nd_diablk then opened_diablk := ls
                                       else EXEC_ERROR( mdnam, 1362, e_severe )
         else
           opened_diablk := fit_fixedvarblblk
       end;

   42: { Change the diagonal block variable attachement }
       begin
         ls := nil;
         repeat
           l1 := GET_AREA( ll, false );
           if l1 <> nil then
             if l1^.lsq_ndty = nd_diablk then { l1 -> A dia. Block }
             begin
               ls := l1; l1 := GET_AREA( ll, false )
             end;
           if l1 <> nil then
             if l1^.lsq_ndty = nd_varbl then LSQ_CH_DIABLK( l1, ls )
         until (ll.typ <> doublety)
       end;

   43: { Change a Parameter definition }
       begin
         ls := GET_AREA( ll, true );        { get the parameter reference }
         o1 := F_EVAL( NXT_PAR( ll ) );
         o2 := F_EVAL( NXT_PAR( ll ) );
         with ls^ do
           if lsq_ndty = nd_parm then
           begin
             { set the new parameter flags if required }
             if o2.typ <> nullty then
             begin
               f1 := GET_FLAG( o2 );        { sum flag }
               f2 := GET_FLAG( o2 );        { derivation flag }
               f3 := GET_FLAG( o2 );        { sigma flag }
               if f3 then f2 := true;       { sigma flag => derivation flag }
               LSQ_S_PARMFLAGS( ls, f1, f2, f3 )
             end
             else
               f2 := parf_derivate in par_flags;
             { set the new parameter definition expression if required }
             if o1.typ <> nullty then
             begin
               LSQ_FREE_EXPR( par_definition );
               if f2 then TRANSL_VEXPR( o1 )
                     else TRANSL_FEXPR( o1 );
               par_definition := LSQ_POP
             end
           end
           else  EXEC_ERROR( mdnam, 1363, e_severe )
       end;

     44: { To change a Variable Value (and sigma) }
       begin
         ls := GET_AREA( ll, true );
         with ls^ do
         case lsq_ndty of
           nd_varbl:
             begin
               var_value := FLTEVLDEF( ll, var_value );
               var_sigma := FLTEVLDEF( ll, var_sigma );
               l1 := GET_AREA( ll, false ); { Get a limit reference }
               if l1 <> nil then
                 if l1^.lsq_ndty = nd_limits then
                   var_limits := l1
             end;
           nd_limits:
             begin
               lim_low := FLTEVLDEF( ll, lim_low );
               lim_up  := FLTEVLDEF( ll, lim_up  )
             end
         otherwise
           EXEC_ERROR( mdnam, 1364, e_severe )
         end
       end;

     45: { To Fix Variables (= to put in Fixed Block) }
       while ll.typ = doublety do
       begin
         ls := GET_AREA( ll, true );
         if ls^.lsq_ndty = nd_varbl then LSQ_FIX_VARBL( ls )
       end;


     46: { To Unfix Variables (= to put in its previous block) }
       while ll.typ = doublety do
       begin
         ls := GET_AREA( ll, true );
         if ls^.lsq_ndty = nd_varbl then LSQ_UNFIX_VARBL( ls )
       end;


    {**********************************************}
    {*********   Get information entries  *********}
    {**********************************************}

     70: { To get the Least-Squares Object Type }
       begin
         ls := GET_AREA( ll,true );         { Get the parameter reference }
         case ls^.lsq_ndty of
           nd_varbl:   re.int := 1;
           nd_parm:    re.int := 2;
           nd_diablk:  re.int := 5;
           nd_limits:  re.int := 6;
           nd_list:    re.int := 3;
           nd_collect: re.int := 4;
         otherwise
         end;
         if re.int <> 0 then re.typ := intty
       end;

     71: { To get the main Least-Squares Parameters }
       begin
         re := F_EVAL( NXT_PAR( ll ) );     { Get the target list }
         if re.typ <> doublety then
           EXEC_ERROR( mdnam, 50, e_severe )
         else
         begin
           currobj := re;
           SET_PARM_INT( fit_ncycle );
           SET_PARM_INT( fit_nvarbl );
           SET_PARM_INT( fit_maxsing );
           SET_PARM_FLT( fit_mindiag );
           SET_PARM_MEM( fit_fixedvarblblk );
           SET_PARM_MEM( fit_limfirst );
           SET_PARM_MEM( fit_varfirst );
           SET_PARM_MEM( fit_parfirst );
           SET_PARM_MEM( fit_listfirst );
           SET_PARM_MEM( fit_collfirst )
         end
       end;

     72: { To get the Name and Last Value(s) of Variable or Parameter }
       begin
         ls := GET_AREA( ll, true );        { Get the parameter reference }
         re := F_EVAL( NXT_PAR( ll ) );     { Get the target list }
         if re.typ <> doublety then EXEC_ERROR( mdnam, 50, e_severe );
         currobj := re;
         with ls^ do
         case lsq_ndty of
           nd_varbl:
             begin
               SET_PARM_IDE( var_name );
               SET_PARM_FLT( var_value );
               SET_PARM_FLT( var_sigma );
               SET_PARM_INT( var_matind );
               SET_PARM_MEM( var_diablk );
               SET_PARM_MEM( var_limits )
             end;

           nd_parm:
             begin
               SET_PARM_IDE( par_name );
               SET_PARM_FLT( par_value );
               if parf_evalsigma in par_flags then SET_PARM_FLT( par_sigma )
                                              else SET_PARM_FLT( -1.0 )
             end;

           nd_diablk:
             begin
               SET_PARM_IDE( blk_name );
               SET_PARM_FLT( blk_effmarq );
               SET_PARM_FLT( blk_effdmp );
               if blk_vect_b <> nil then
                 SET_PARM_INT( blk_vect_b^.v_size )
               else
                 SET_PARM_INT( 0 );
               eq.vec := blk_vect_b;
               SET_PARM_MEM( eq.lsq );
               eq.vec := blk_vect_x;
               SET_PARM_MEM( eq.lsq );
               eq.vec := blk_matrix;
               SET_PARM_MEM( eq.lsq );
               SET_PARM_MEM( blk_frsvar )
             end;

           nd_limits:
             begin
               SET_PARM_FLT( lim_low );
               SET_PARM_FLT( lim_up  )
             end;


           nd_list:
             begin
               SET_PARM_IDE( lis_name );
               SET_PARM_ALM( lis_current );
               SET_PARM_ALM( lis_descriptor );
               SET_PARM_MEM( lis_parlist )
             end;

           nd_collect:
             begin
               SET_PARM_IDE( coll_name );
               SET_PARM_MEM( coll_list );
               SET_PARM_FLT( coll_reject );
               SET_PARM_STA( coll_statis )
             end;

         otherwise
           EXEC_ERROR( mdnam, 1365, e_severe )
         end
       end;

     73: { To get the statistic Least-Squares Parameters }
       begin
         ls := GET_AREA( ll, false );       { Get the record reference }
         re := F_EVAL( NXT_PAR( ll ) );     { Get the target list }
         currobj := re;
         if re.typ <> doublety then EXEC_ERROR( mdnam, 50, e_severe );
           if ls <> nil then
           with ls^ do
             if lsq_ndty = nd_collect then
               st := coll_statis
             else
               EXEC_ERROR( mdnam, 1366, e_error )
           else
             st := fit_statis;
           with st^ do
           begin
             SET_PARM_INT(  stat_obsnb   );
             SET_PARM_FLT( stat_usrchi2 );
             SET_PARM_FLT( stat_stdchi2 );
             SET_PARM_FLT( stat_sumstd  );
             SET_PARM_FLT( stat_surwsqr );
             SET_PARM_FLT( stat_sursqr  );
             SET_PARM_FLT( stat_surwabs );
             SET_PARM_FLT( stat_surabs  );
             SET_PARM_FLT( stat_suowsqr );
             SET_PARM_FLT( stat_suosqr  );
             SET_PARM_FLT( stat_suowabs );
             SET_PARM_FLT( stat_suoabs  )
           end
       end;

     74: { To get Corre. res. for Each Variable and for the Whole of Block }
       begin
         ls := GET_AREA( ll, false );       { Get the diagonal block reference }
         o1 := GET_LIST( ll, false );       { Get a target list or nil }
         if ls <> nil then
         with ls^ do
         begin
           if lsq_ndty <> nd_diablk then
             EXEC_ERROR( mdnam, 1362, e_severe )
           else
           if blk_matrix <> nil then
           begin
             re.typ := flty;
             re.flt := CORRELATION_GBL_FACTOR( blk_vect_b^, blk_matrix^ );
             if o1.typ = doublety then
             begin
               currobj := o1;
               for i1 := 1 to blk_vect_b^.v_size do
                 SET_PARM_FLT( blk_vect_b^[i1] )
             end
           end
         end
       end;

     75: { To get some correlation value(s) }
       begin
         ls := GET_AREA( ll, true );        { Get the diagonal block reference }
         r1 := FLTEVL( ll );                { Get the diagonal block limits }
         re  := F_EVAL( NXT_PAR( ll ) );    { Get the target list or nil }
         if re.typ <> doublety then EXEC_ERROR( mdnam, 50, e_severe );
         with ls^ do
         begin
           if lsq_ndty <> nd_diablk then EXEC_ERROR( mdnam, 1362, e_severe );
           if blk_matrix = nil then EXEC_ERROR( mdnam, 1367, e_severe );
           re.db^.car     := obj_zero;
           re.db^.car.int := GET_CORREL_LIST( blk_matrix^, re.db^.cdr,
                                              blk_frsvar,
                                              r1, blk_vect_b^.v_size )
         end
       end;

     76: { To get the list of the diagonal block }
       begin
         ls := GET_AREA( ll, false );
         re := F_EVAL( NXT_PAR( ll ) );     { Get the target list or nil }
         if re.typ <> doublety then EXEC_ERROR( mdnam, 50, e_severe );
         if ls <> nil then
         begin
           if ls^.lsq_ndty <> nd_diablk then
              EXEC_ERROR( mdnam, 1362, e_error );

           if ls = nil then ls := fit_fixedvarblblk
                       else ls := ls^.blk_next;
           currobj := re;
           while (ls <> nil) and (currobj.typ = doublety) do
           begin
             SET_PARM_MEM( ls );
             ls := ls^.blk_next
           end
         end
       end;


     77: { To get the list of the Variable for a Diagonal Block (or Fixed) }
       begin
         l1 := GET_AREA( ll, false );
         o1 := F_EVAL( NXT_PAR( ll ) );     { Get the target list or nil }
         if o1.typ <> doublety then EXEC_ERROR( mdnam, 50, e_severe );

         if l1 <> nil then
         with l1^ do
         case lsq_ndty of
           nd_varbl: { Get the next variables in the same block }
             begin
               l2 := var_nxtbl;
               l1 := var_diablk
             end;

           nd_diablk: { Get the variables in the specified block }
             l2 := blk_frsvar;

         otherwise
           EXEC_ERROR( mdnam, 1368, e_error )
         end
         else
         begin
           l2 := fit_fixedvarblblk^.blk_frsvar;
           l1 := fit_fixedvarblblk
         end;

         currobj := o1;
         i1 := 0;
         while l2 <> nil do
         begin { Scan the Diagonal Block list of Variable }
           SET_PARM_MEM( l2 );              { Insert addr. of each variable }
           i1 := i1 + 1;                    { and increment the variable count }
           l2 := l2^.var_nxtbl              { Skip to next variable }
         end;
         re.typ := intty; re.int := i1      { Send the Variable count }
       end;

    78: { To get the Complete List of Defined Variable }
       begin
         i1 := 0;
         { The given Object must be an Atom }
         o0 := GET_ATOM( NXT_PAR( ll ), true );
         with o0.at^ do
         if val.typ = doublety then re  := val
                               else val := obj_nil;
         o1 := re; { Init the Current List Pointer }
         o2 := obj_nil;
         { Get the first defined variable parameter }
         l1 := fit_varfirst;
         while l1 <> nil do
         begin
           i1 := i1 + 1;   { Update the Variable Count }
           { Create a LISP LSQ reference for this variable }
           eq.lsq   := l1;
           o3.flg.f := [];
           o3.flg.k := lsq_define;
           o3.typ   := areatyp;
           o3.mem   := eq.mem;
           if o1.typ = doublety then
           begin { Set a variable in the given list }
             o1.db^.car := o3;
             o2 := o1;
             o1 := o1.db^.cdr
           end
           else
           begin { Append a Variable to the previous list or create it }
             o3 := F_CONS( o3, obj_nil );   { Create the doublet }
             if re.typ = nullty then re := o3
                                else o2.db^.cdr := o3;
             o2 := o3
           end;
           l1 := l1^.var_next
         end;
         o0.at^.val := re;          { Set the resulting list in the atom }
         re := obj_zero;
         re.int := i1
       end;


    {************************************************}
    {*********   Get value/sigma dunction   *********}
    {************************************************}


     80: { To get the Name of a LSQ Parameter or Variable }
       begin
         o0 := F_EVAL( NXT_PAR( ll ) );     { Get the parameter value }
         re := obj_nil;
         eq.rec := nil;
         if (o0.flg.k = lsq_define) and (o0.typ = areatyp) then
         begin
           eq.mem := o0.mem;
           with eq.lsq^ do
           case lsq_ndty of
             nd_varbl:   eq.rec := var_name;
             nd_parm:    eq.rec := par_name;
             nd_diablk:  eq.rec := blk_name;
             nd_list:    eq.rec := lis_name;
             nd_collect: eq.rec := coll_name
           otherwise
           end;
           if eq.rec <> nil then re := eq.dbl^.car;
           { Evaluate LISP identification expression if required }
           if re.typ = doublety then re := F_EVAL( re );
           if re.typ >= atomety then re := re.at^.ats
         end
       end;


     81: { To get the Value of a LSQ expression }
       begin
         o0 := F_EVAL( NXT_PAR( ll ) );     { Get the parameter value }
         if (o0.flg.k = lsq_define) and (o0.typ = areatyp) then
         begin
           eq.mem := o0.mem;
           re.typ := flty;
           re.flt := LSQ_VALUE( eq.lsq )
         end
         else re := o0
       end;


     82: { To get the Sigma of a LSQ Parameter or Variable }
       begin
         re.typ := flty;
         re.flt := 0.0;
         o0 := F_EVAL( NXT_PAR( ll ) );     { Get the parameter value }
         if (o0.flg.k = lsq_define) and (o0.typ = areatyp) then
         begin
           eq.mem := o0.mem;
           with eq.lsq^ do
           case lsq_ndty of
             nd_varbl: re.flt := var_sigma;
             nd_parm:  if parf_evalsigma in par_flags then re.flt := par_sigma
                                                      else re.flt := -1.0
           otherwise
           end
         end
       end;


  otherwise
    EXEC_ERROR( mdnam, 1399, e_fatal )
  end { case };
  LISP$_LSQ := re
end LISP$_LSQ;



procedure TRANSL_EXEC { ( obj: obj_ref ); was forward };
var
  fok:                      boolean;
  idx:                      integer;
  stat, stl, ob0, ob1, ob2: obj_ref;

begin
  while obj.typ = doublety do
  with obj.db^ do
  begin
    stat := car;
    if stat.typ = doublety then
    begin
      stl := stat;
      ob0 := NXT_PAR( stl );
      case ob0.typ of
        intty: LISP$_LSQ( ob0.int mod 300, stl );

      otherwise
        ob0 := F_EVAL( stat )
      end
    end
    else ob0 := F_EVAL( stat );
    obj := cdr
  end
end TRANSL_EXEC;



procedure TRANSL_PROC { ( proc: obj_ref ); was forward };
const
  mdnam = 'TRSQ';

var
  slex, dlex, last_lex:                    char;
  isz, ifo:                                integer;
  body, dyn, sta, par, last_sta, last_dyn: obj_ref;
  fty:                                     obj_type;

begin
  if proc.typ >= atomety then               { Must be an atom reference }
    with proc.at^ do { This atom must be defined as a procedure function }
    if fncref.flg.k = dp__funct then
    begin
      body := fncref;                       { Get the procedure definition }

    end
    else { not supported use of procedure }
      EXEC_ERROR( mdnam, 3999, e_severe )
  else
    TRANSL_EXEC( proc )
end TRANSL_PROC;



end.
