/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               D R A W   -   S E R V E R   M O D U L E                 *
*                                                                       *
*              (Kernel to manage the bidirectional pipe)                *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                 Version V2.2 of the 20-Jan-2007                       *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/

/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/



/* include the environment SERVER */

#include <draw/draw_common.h>
#include <draw/sdrw_env.h>
#include <draw/sdrw_routines.h>

#include "Draw_VGL.h"
#include "Draw_AxisBox.h"


#define Menu_Max    32
#define Menu_ItmSz  64
#define Menu_BfSize Menu_Max*(Menu_ItmSz + 1)



/****  Local Variables  ****/

static eqv_if  equ;        /* An equivalent model for Sdrw routines */

static char * menu[Menu_Max + 1];    /* Pointer for strings */
static char   menu_buf[Menu_BfSize];
static int    menu_topn, menu_tops;

static char draw_pvw = 0;  /* Flag for DRAW$PIC_VIEW */




/* Debugging Output Update procedure */


void Draw_Fmsgupdate()
{
  fclose( fmsg );
  fmsg = fopen( "DRAW_ERROR.TXT", "a" );
}



void Draw_Error( char *msg, int* id )
{
  if (!fmsg) fmsg = fopen( "DRAW_ERROR.TXT", "a" );
  if (id) fprintf( fmsg, " *Draw Error: %s #%d.\n", msg, *id );
     else fprintf( fmsg, " *Draw Error: %s.\n", msg );
  Draw_Fmsgupdate();
}



void Draw_Exchange_Init( void )
{
  /* Send to User the Server capability word */
  Sdrw_Put_Int( SERVER_IDENT );                /* Send the Server Identity Number */
  Sdrw_Put_Int( SERVER_FUNCT );                /* Send the Server Capability Word */
  Sdrw_Put_String( "VGL_Draw_Server_2.3", 0 ); /* Send the Server name */
  Sdrw_Write();                                /* Write to Client  and wait for Answerd */
  Sdrw_Read();                                 /* Get the Client answerd */
  Drwgl_usr_request = Sdrw_Get_Int();          /* Get the User Requested Funct. */
  Drwgl_taskname = Sdrw_Get_String_Tmp();      /* Get the User task Name. */
}



void Draw_Extern_Init( void )
{ /* */
  Char  b0, b1, b2, b3;
  float x, y;

  status = Draw_Init( &x, &y, &b0 );
  /* Return the Server and Display Specifications to client Application. */
  Sdrw_Put_Float( x );       /* Send to Client the Display size ... */
  Sdrw_Put_Float( y );
  Sdrw_Put_Char( b0 );       /* and the unit Flag (1 = cm., 0 = Raster U. */
  Sdrw_Write();              /* Write the last Client Answerd */
  reqnxt          = 0;       /* Turn to User Task Command Mode */
  Drwgl_exitmd    = 0;       /* Set the None Defined Exit node mode */
}



void * Draw_Sequence_Exec( void * pp )
{ /* Main DRAW code Processor */
  Char  b0, b1, b2, b3, bupd;
  int   code, flg, i, i0, i1, i2, len, n, m;
  float r, v0, v1, v2, x, x0, x1, y, y0, y1, z, z0, z1;
  float *f1, *f2;
  float ftb[64];
  int   itb[16];
  Char  btb[16];
  Draw_Ptr p;
  Draw_Box_Ptr pb;

  static char s0[256], s1[256], s2[256];
 
  while (1)
  { /* Loop on Pipe User Program Requests */
    bupd = 0;                 /* Assume no update */
    len = Sdrw_Read();        /* Wait for Read Buffer from User Application. */
//  fprintf( fmsg, " Draw msg of %d bytes.\n", len );
    code = Sdrw_Get_Code();   /* Get the Draw Operator code. */
//  fprintf( fmsg, " Draw Code %d at %8.2f\n", code, Drwgl_timcnt );
//  Draw_Fmsgupdate();
    reqnxt = 0;
 
    Sdrw_read_lock();         /* Lock the User Directive Execution Graph. */

    switch(code) {            /* Dispatch to the related DRAW routine. */

    case cd_picture:
      Sdrw_Get_String( sv_title );
      x  = Sdrw_Get_Float();  /* Get the Required space */
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      b1 = Sdrw_Get_Char();
      b2 = Sdrw_Get_Char();
      b0 = Draw_Picture( sv_title, x, y, z, b1, b2 );
      bupd = draw_pvw;
      Sdrw_Put_Char( b0 );
    break;


    case cd_picture_scales:
      n = Sdrw_Get_Int();
      switch (n) {
        case -1: /* Sizes of the window Canvas in pixels */
          x = (float) dt_rx; y = (float) dt_ry;
        break;
        case -2: /* Sizes of the printer Canvas in pixels */
          x = (float) pt_rx; y = (float) pt_ry;
        break;
        case -3: /* Sizes of the window Canvas in Cm */
          x = ds_rx; y =    ds_ry;
        break;
        case -4: /* Sizes of the printer Canvas in Cm */
          x = ps_rx; y =    ps_ry;
        break;
        case -5: /* Sizes of the Screen in Cm */
          x = d_width; y =  d_height;
        break;
        case -6: /* Sizes of the Paper in Cm */
          x = p_width; y =  p_height;
        break;
        case  1: /* Scale in Pixel/Cm on the Screen */
          x = d_xscale; y = d_yscale;
        break;
        case  2: /* Scale in Pixel/Cm on the Printer */
          x = p_xscale; y = p_yscale;
        break;
        default:
          x = xy_scale; y = cv_scale;
        break;
      }
      fprintf( fmsg, " Get Scales %d -> (%8.2f %8.2f)\n", n, x, y );
      Draw_Fmsgupdate();
      Sdrw_Put_Float( x );
      Sdrw_Put_Float( y );
    break;

    case cd_pic_view:
      flg = Sdrw_Get_Char();
      draw_pvw = 1;
      bupd = draw_pvw;
      Sdrw_Put_Char( 0 );
    break;

    case cd_pic_reset:
      flg = Sdrw_Get_Char();
      draw_pvw = 0;
      Sdrw_Put_Char( 0 );
    break;

    case cd_tim_anim:
      i = Drwgl_time_period;
      i0 = Sdrw_Get_Int();
      if ((i0 >= 10) || (i0 <= 500)) Drwgl_period_set = i0;
      Sdrw_Put_Int( i );
    break;

    case cd_anim_stop:
      Drwgl_astpmod   = Sdrw_Get_Char();
      Drwgl_astpstate = 0;
      Sdrw_Put_Char( 0 );
    break;

    case cd_clipon:
      Draw_Gen_Box_Ref( 1 );
      Sdrw_Put_Char( 0 );
    break;

    case cd_clipoff:
      Draw_Gen_Box_Ref( 0 );
      Sdrw_Put_Char( 0 );
    break;

    case cd_org:
      x = Sdrw_Get_Float();
      y = Sdrw_Get_Float();
      z = Sdrw_Get_Float();
      i = Sdrw_Get_Int();
      if (i == 0)
      { /* Relative To previous Origine */
        orgx += x;
        orgy += y;
        orgz += z;
      } else { /* Absolute (as after Picture Call) Origine */
        orgx = org_orgx + x;
        orgy = org_orgy + y;
        orgz = org_orgz + z;
      }
      Sdrw_Put_Char( 0 );
    break;

    case cd_color:
      x  = Sdrw_Get_Float();
      if (sdrw_read_left) {
        y  = Sdrw_Get_Float();
        z  = Sdrw_Get_Float();
        v0 = (sdrw_read_left)?Sdrw_Get_Float():1.0;
      } else { y = 0.0; z = 0.0; v0 = 0.0; }
      Draw_Gen_Color( x, y, z, v0 );
      Sdrw_Put_Char( 0 );
    break;

    case cd_out_mode:
      draw_out_mode = Sdrw_Get_Int();
      Sdrw_Put_Char( 0 );
    break;

    case cd_mplot:
      n  = Sdrw_Get_Int();
      i0 = Sdrw_Get_Int();
//    fprintf( fmsg, "  Create/read Mplot %d, %d.\n", n, i0 );
//    Draw_Fmsgupdate();
      if (Draw_Opened_Box) Draw_Gen_SMplot( n, i0, NULL );
                      else Draw_Gen_Mplot( n, i0, NULL );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_circle:
      i0 = Sdrw_Get_Int();
      x  = Sdrw_Get_Float();
      y  = Sdrw_Get_Float();
      r  = Sdrw_Get_Float();
      v0 = Sdrw_Get_Float();
      v1 = Sdrw_Get_Float();
      v2 = Sdrw_Get_Float();
      if (!((i0&cdf_AB)||draw_mat_cnt)) { x += orgx; y += orgy; }
      Draw_Gen_Circle( x, y, r, v0, v1, v2 );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_string:
      i0 = Sdrw_Get_Char();
      x  = Sdrw_Get_Float();
      y  = Sdrw_Get_Float();
//    fprintf( fmsg, " => At (%f,%f) Draw String .\n", x, y );
      if (!((i0&cdf_AB)||draw_mat_cnt)) { x += orgx; y += orgy; }
      v0 = Sdrw_Get_Float();
      v1 = Sdrw_Get_Float();
      i  = Sdrw_Get_String( s0 );
//    fprintf( fmsg, " => At (%f,%f) %d:String %s .\n", x, y, i, s0 );
//    Draw_Fmsgupdate();
      Draw_Gen_Text( x, y, v0, v1, i, s0 );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_sphere:
      i0 = Sdrw_Get_Int();
      x  = Sdrw_Get_Float();
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      r  = Sdrw_Get_Float();
      i1 = Sdrw_Get_Int();
      i2 = Sdrw_Get_Int();
      if (!((i0&cdf_AB)||draw_mat_cnt)) { x += orgx; y += orgy; z += orgz; }
      Draw_Gen_Sphere( x, y, z, r, i1, i2, i0%cdf_AB );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_cylinder:
      i0 = Sdrw_Get_Int();
      for (i = 0; i < 8; i++) ftb[i] = Sdrw_Get_Float();
      i1 = Sdrw_Get_Int();
      i2 = Sdrw_Get_Int();
      if (!((i0&cdf_AB)||draw_mat_cnt)) { ftb[0] += orgx; ftb[1] += orgy; ftb[2] += orgz; }
      Draw_Gen_Cylinder( ftb, i1, i2, i0%cdf_AB );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_disk:
      i0 = Sdrw_Get_Int();
      for (i = 0; i < 9; i++) ftb[i] = Sdrw_Get_Float();
      i1 = Sdrw_Get_Int();
      i2 = Sdrw_Get_Int();
      if (!((i0&cdf_AB)||draw_mat_cnt)) { ftb[0] += orgx; ftb[1] += orgy; ftb[2] += orgz; }
      Draw_Gen_Disk( ftb, i1, i2, i0%cdf_AB );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_surface:
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      i2 = Sdrw_Get_Int();
      if (Draw_Opened_Box) Draw_Gen_MSurface( i1, i2, i0 );
                      else Draw_Gen_Surface( i1, i2, i0 );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_surface2:
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      i2 = Sdrw_Get_Int();
      x0 = Sdrw_Get_Float();
      y0 = Sdrw_Get_Float();
      x1 = Sdrw_Get_Float();
      y1 = Sdrw_Get_Float();
      if (Draw_Opened_Box) Draw_Gen_MSurface2( i1, i2, i0, x0, y0, x1, y1 );
                      else Draw_Gen_Surface2( i1, i2, i0, x0, y0, x1, y1 );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_polyhedral:
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      i2 = Sdrw_Get_Int();
      Draw_Gen_Polyhedral( i1, i2, i0 );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_view:
      Draw_Dist = Sdrw_Get_Float();
      if (Draw_Dist <= 0.1*draw_uzmax) Draw_Dist = 0.0;
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_wplane3: /* Define a plot plane for next drawing directive */
      for (i = 0; i < 12; i++) ftb[i] = Sdrw_Get_Float();
      Draw_Gen_Eplane( ftb );
      Sdrw_Put_Char( 0 );
    break;

    case cd_nplane3: /* Release a plot plane for next drawing directive */
      Draw_Gen_Eplane( NULL );
      Sdrw_Put_Char( 0 );
    break;

    case cd_alpha_set:
      Draw_Gen_BlendSet( Sdrw_Get_Int() );
      Sdrw_Put_Char( 0 );
    break;

    case cd_light_set:
      i0 = Sdrw_Get_Int() - 1;
      i1 = Sdrw_Get_Int();
      DrwGL_Light_OnOff( i0, i1 );
      bupd = draw_pvw;
      Sdrw_Put_Char( 0 );
    break;

    case cd_matp_light:
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      i = 0;
      while (sdrw_read_left&&(i < 4)) ftb[i++] = Sdrw_Get_Float();
      Draw_Gen_MatLight( i0, i1, i, ftb );
      Sdrw_Put_Char( 0 );
    break;

    case cd_light_def:
      i0 = Sdrw_Get_Int() - 1; /* DRAW Light # start from 1 and GL ones start from 0 */
      i1 = Sdrw_Get_Int();
      i = 0;
      while (sdrw_read_left&&(i < 4)) ftb[i++] = Sdrw_Get_Float();
      if ((i1 == 0)&&(!draw_mat_cnt)) { ftb[0] += orgx; ftb[0] += orgy; ftb[0] += orgz; }
      Draw_Gen_LightDef( i0, i1, i, ftb );
      bupd = draw_pvw;
      Sdrw_Put_Char( 0 ); 
    break;

    case cd_define_color:
      n  = Sdrw_Get_Int();
      x  = Sdrw_Get_Float();
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      v0 = (sdrw_read_left)?Sdrw_Get_Float():1.0;
      Draw_Set_Color_Def( n, x, y, z, v0 );
      bupd = draw_pvw;
      Sdrw_Put_Char( 0 );
    break;

    case cd_inquire_color:
      Draw_Get_Color_Def( Sdrw_Get_Int(), &x, &y, &z, &v0 );
      Sdrw_Put_Int( status );
      Sdrw_Put_Float( x );
      Sdrw_Put_Float( y );
      Sdrw_Put_Float( z );
      Sdrw_Put_Float( v0 );
    break;

    case cd_line_attr:
      n = Sdrw_Get_Int();
      r = Sdrw_Get_Float();
      Draw_Gen_Lattr( n, r );
      Sdrw_Put_Char( 0 );
    break;

    case cd_marker_attr:
      n = Sdrw_Get_Int();
      r = Sdrw_Get_Float();
      Draw_Gen_Mattr( n, r );
      Sdrw_Put_Char( 0 );
    break;

    case cd_fill_area_attr:
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      Draw_Gen_Fattr( i0, i1 );
      Sdrw_Put_Char( 0 );
    break;

    case cd_text_font:
      n  = Sdrw_Get_Int();
      i0 = Sdrw_Get_Int();
      Draw_Gen_Font( n, i0 );
      Sdrw_Put_Char( 0 );
    break;

    case cd_text_attr:
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      i2 = Sdrw_Get_Int();
      v0 = Sdrw_Get_Float();
      v1 = Sdrw_Get_Float();
      Draw_Gen_Tattr( i0, i1, i2, v0, v1 );
      Sdrw_Put_Char( 0 );
    break;

    case cd_gen_tmat:
      draw_mat_cnt++;
      i0 = Sdrw_Get_Char();
      i1 = Draw_Gen_TMat( i0 );
      if (i0) draw_mat_cnt++; else draw_mat_cnt--;
      Sdrw_Put_Int( i1 );
      break;

    case cd_init_trans_mat:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_TMat( n );  /* Get the Matrix Pointer */
      i  = Sdrw_Get_Char();
      if (p) DrwGL_Seg_Unit_Transform( p->tmat.trsf_mat );
        else Draw_Error( "Cannot Reset to Identity the Not Existing Matrix", &n );
      if (i) bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_move_mat:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_TMat( n );  /* Get the Matrix Pointer */
      x  = Sdrw_Get_Float();
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      if (p) DrwGL_Move_Seg( p->tmat.trsf_mat, x, y, z );
        else Draw_Error( "Cannot Change(Move) the Not Existing Matrix", &n );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_urotate_mat:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_TMat( n );  /* Get the Matrix Pointer */
      x  = Sdrw_Get_Float(); /* Do not translate from Origine, the */
      y  = Sdrw_Get_Float(); /* translation is performed from VGL base */
      z  = Sdrw_Get_Float();
      v0 = Sdrw_Get_Float();
      v1 = Sdrw_Get_Float();
      v2 = Sdrw_Get_Float();
      if (p) DrwGL_URot_Seg( p->tmat.trsf_mat, x, y, z, v0, v1, v2 );
        else Draw_Error( "Cannot Change(URot) the Not Existing Matrix", &n );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_rotate_mat:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_TMat( n );  /* Get the Matrix Pointer */
      x  = Sdrw_Get_Float(); /* Do not translate from Origine, the */
      y  = Sdrw_Get_Float(); /* translation is performed from VGL base */
      z  = Sdrw_Get_Float();
      v0 = Sdrw_Get_Float();
      v1 = Sdrw_Get_Float();
      v2 = Sdrw_Get_Float();
      if (p) DrwGL_Rotate_Seg( p->tmat.trsf_mat, x, y, z, v0, v1, v2 );
        else Draw_Error( "Cannot Change(Rotate) the Not Existing Matrix", &n );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_scale_mat:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_TMat( n );  /* Get the Matrix Pointer */
      x  = Sdrw_Get_Float(); /* Do not translate from Origine, the */
      y  = Sdrw_Get_Float(); /* translation is performed from VGL base */
      z  = Sdrw_Get_Float();
      x1 = Sdrw_Get_Float();
      y1 = Sdrw_Get_Float();
      z1 = Sdrw_Get_Float();
      if (p) DrwGL_Scale_Seg( p->tmat.trsf_mat, x, y, z, x1, y1, z1 );
        else Draw_Error( "Cannot Change(Scale) the Not Existing Matrix", &n );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_transform_mat:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_TMat( n );  /* Get the Matrix Pointer */
      x  = Sdrw_Get_Float();          /* Get the Translation */
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      x0 = Sdrw_Get_Float();          /* Get the fixed point */
      y0 = Sdrw_Get_Float(); /* Do not translate from Origine, the */
      z0 = Sdrw_Get_Float(); /* translation is performed from VGL base */
      v0 = Sdrw_Get_Float();          /* Get the rotation angles */
      v1 = Sdrw_Get_Float();
      v2 = Sdrw_Get_Float();
      x1 = Sdrw_Get_Float();          /* Get the x,y,z Scaling */
      y1 = Sdrw_Get_Float();
      z1 = Sdrw_Get_Float();
      if (p) DrwGL_Transf_Seg( p->tmat.trsf_mat, x, y, z, x0, y0, z0, v0, v1, v2, x1, y1, z1 );
        else Draw_Error( "Cannot Change(Transform) the Not Existing Matrix", &n );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_get_transf_mat:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_TMat( n );  /* Get the Matrix Pointer */
      if (p) {
        Sdrw_Put_Int( 12 );
        for (i1 = 0; i1 < 3; i1++ )
          for (i2 = 0; i2 < 4; i2++)
            Sdrw_Put_Float( p->seg.seg_mat[i1*4+i2] );
      } else {
        Draw_Error( "Cannot Get the Not Existing Matrix", &n );
        Sdrw_Put_Int( -1 );
      }
    break;

    case cd_put_transf_mat:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_TMat( n );  /* Get the Matrix Pointer */
      if (p) {
        for (i1 = 0; i1 < 3; i1++ )
          for (i2 = 0; i2 < 4; i2++)
            p->seg.seg_mat[i1*4+i2] = Sdrw_Get_Float();
        Sdrw_Put_Int( 0 );
      } else {
        Draw_Error( "Cannot Put the Not Existing Matrix", &n );
        Sdrw_Put_Int( -1 );
      }
    break;

    case cd_new_seg:
      draw_mat_cnt = 0;
      i0 = Sdrw_Get_Int();
      p = Draw_New_Segment( i0, 0 );
      Sdrw_Put_Int( p->seg.seg_ide );
//    fprintf( fmsg, " New Seg #%d\n", p->seg.seg_ide );
//    Draw_Fmsgupdate();
    break;

    case cd_seg_end:
      draw_mat_cnt = 0;
      draw_curseg = NULL;
      bupd = draw_pvw;
      Sdrw_Put_Char( 0 );
    break;

    case cd_del_seg:
      n = Sdrw_Get_Int();
      p = Draw_Locate_Seg( n );
      if (p) Draw_Destroye( p );
      bupd = draw_pvw;
      Sdrw_Put_Char( 0 );
    break;

    case cd_exec_seg:
      n = Sdrw_Get_Int();
      Draw_Gen_Exec_Seg( n );
      bupd = draw_pvw;
      Sdrw_Put_Char( 0 );
    break;

    case cd_update_seg:
      draw_mat_cnt = 0;
      i0 = Sdrw_Get_Int();
      p = Draw_New_Segment( i0, Sdrw_Get_Int() );
      if (p) Sdrw_Put_Int( p->seg.seg_ide );
        else Sdrw_Put_Int( -1 );
    break;

    case cd_seg_attr:
      n  = Sdrw_Get_Int();
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      i2 = Sdrw_Get_Int();
      r  = Sdrw_Get_Float();
      if (Draw_Set_Seg_Attr( n, i0, i1, i2, r )) bupd = draw_pvw;
      Sdrw_Put_Char( 0 );
    break;

    case cd_seg_picid:
      i0 = Sdrw_Get_Int();
      if (i0 > 0) { draw_curr_pid = i0; draw_curr_detf = 1; }
      if (i0 < 0) draw_curr_detf = 0;
      Sdrw_Put_Char( 0 );
    break;

    case cd_seg_pdetf:
      i0 = Sdrw_Get_Int();
      i1 = draw_curr_detf;
      if (i0>=0) draw_curr_detf = (i0 > 0);
      Sdrw_Put_Int( i1 );
    break;

    case cd_seg_inquire:
      if (draw_curseg) Sdrw_Put_Int( draw_curseg->seg.seg_ide );
                  else Sdrw_Put_Int( -1 );
    break;

    case cd_init_trans_seg:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_Seg( n );  /* Get the Segment Pointer */
      i  = Sdrw_Get_Char();
      if (p) DrwGL_Seg_Unit_Transform( p->seg.seg_mat );
        else Draw_Error( "Cannot Reset to Identity the Matrix of Not Existing Segment", &n );
      if (i) bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_move_seg:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_Seg( n );  /* Get the Segment Pointer */
      x  = Sdrw_Get_Float();
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      if (p) DrwGL_Move_Seg( p->seg.seg_mat, x, y, z );
        else Draw_Error( "Cannot Change(Move) the Matrix of Not Existing Segment", &n );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_urotate_seg:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_Seg( n );  /* Get the Segment Pointer */
      x  = Sdrw_Get_Float();
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      if (!draw_mat_cnt) { x += orgx; y += orgy; z += orgz; }      
      v0 = Sdrw_Get_Float();
      v1 = Sdrw_Get_Float();
      v2 = Sdrw_Get_Float();
      if (p) DrwGL_URot_Seg( p->seg.seg_mat, x, y, z, v0, v1, v2 );
        else Draw_Error( "Cannot Change(Urot) the Matrix of Not Existing Segment", &n );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_rotate_seg:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_Seg( n );  /* Get the Segment Pointer */
      x  = Sdrw_Get_Float();
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      if (!draw_mat_cnt) { x += orgx; y += orgy; z += orgz; }      
      v0 = Sdrw_Get_Float();
      v1 = Sdrw_Get_Float();
      v2 = Sdrw_Get_Float();
      if (p) DrwGL_Rotate_Seg( p->seg.seg_mat, x, y, z, v0, v1, v2 );
        else Draw_Error( "Cannot Change(Rotate) the Matrix of Not Existing Segment", &n );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_scale_seg:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_Seg( n );  /* Get the Segment Pointer */
      x  = Sdrw_Get_Float();
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      if (!draw_mat_cnt) { x += orgx; y += orgy; z += orgz; }      
      x1 = Sdrw_Get_Float();
      y1 = Sdrw_Get_Float();
      z1 = Sdrw_Get_Float();
      if (p) DrwGL_Scale_Seg( p->seg.seg_mat, x, y, z, x1, y1, z1 );
        else Draw_Error( "Cannot Change(Scale) the Matrix of Not Existing Segment", &n );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_transform_seg:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_Seg( n );  /* Get the Segment Pointer */
      x  = Sdrw_Get_Float();          /* Get the Translation */
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      x0 = Sdrw_Get_Float();          /* Get the fixed point */
      y0 = Sdrw_Get_Float();
      z0 = Sdrw_Get_Float();
      if (!draw_mat_cnt) { x0 += orgx; y0 += orgy; z0 += orgz; }      
      v0 = Sdrw_Get_Float();          /* Get the rotation angles */
      v1 = Sdrw_Get_Float();
      v2 = Sdrw_Get_Float();
      x1 = Sdrw_Get_Float();          /* Get the x,y,z Scaling */
      y1 = Sdrw_Get_Float();
      z1 = Sdrw_Get_Float();
      if (p) DrwGL_Transf_Seg( p->seg.seg_mat, x, y, z, x0, y0, z0, v0, v1, v2, x1, y1, z1 );
        else Draw_Error( "Cannot Change(Transform) the Matrix of Not Existing Segment", &n );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_get_transf_seg:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_Seg( n );  /* Get the Segment Pointer */
      if (p) {
        Sdrw_Put_Int( 12 );
        for (i1 = 0; i1 < 3; i1++ )
          for (i2 = 0; i2 < 4; i2++)
            Sdrw_Put_Float( p->seg.seg_mat[i2*4+i1] );
      } else {
        Draw_Error( "Cannot Get the Matrix of Not Existing Segment", &n );
        Sdrw_Put_Int( -1 );
      }
    break;

    case cd_put_transf_seg:
      n  = Sdrw_Get_Int();
      p  = Draw_Locate_Seg( n );  /* Get the Segment Pointer */
      if (p) {
        for (i1 = 0; i1 < 3; i1++ )
          for (i2 = 0; i2 < 4; i2++)
            p->seg.seg_mat[i2*4+i1] = Sdrw_Get_Float( );
        Sdrw_Put_Int( 0 );
      } else {
        Draw_Error( "Cannot Put the Matrix of Not Existing Segment", &n );
        Sdrw_Put_Int( -1 );
      }
    break;

    case cd_mat_orient:
    case cd_seg_orient:
      i0 = abs( Sdrw_Get_Int() );
      Drwgl_FixX  = Sdrw_Get_Float();
      Drwgl_FixY  = Sdrw_Get_Float();
      Drwgl_FixZ  = Sdrw_Get_Float();
      if (!draw_mat_cnt) { Drwgl_FixX += orgx; Drwgl_FixY += orgy; Drwgl_FixZ += orgz; }      
      Drwgl_viewflg = Sdrw_Get_Char();
      if (i0 > 0) {
        if (code == cd_mat_orient) {
          p = Draw_Locate_TMat( n );    /* Get the node pointer */
          if (p) Drwgl_ViewMat = p->tmat.trsf_mat;
        } else {
          p = Draw_Locate_Seg( n );     /* Get the Segment Matrix Pointer */
          if (p) Drwgl_ViewMat = p->seg.seg_mat;
        }
        /* if i1 is true, we must generate a user orinetation call Here */
        Drwgl_zomeview = 1;             /* Set implicite View change */
        if (Drwgl_ViewMat&&(Drwgl_viewflg&1)) {
          Drwgl_Input = drwgl_usr_view; /* Set The Request Kind */
          _Inp_Req_Display( &bupd );    /* Send the Request to VGL */
        } else Sdrw_Put_Int( 1 );
      } else {
        Drwgl_zomeview = 0;             /* Set implicite Zoom */
        Drwgl_ViewMat = NULL;
        Sdrw_Put_Int( 1 );
      }
    break;

    case cd_new_axis:
      for (i = 0; i < 6; i++) ftb[i] = Sdrw_Get_Float();
      x  = Sdrw_Get_Float();
      y  = Sdrw_Get_Float();
      z  = Sdrw_Get_Float();
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      Sdrw_Put_Int( Draw_Gen_Axis( ftb, ftb+3, x, y, z, i0, i1 ) );
    break;

    case cd_new_box:
      for (i = 0; i < 3; i++) ftb[i] = Sdrw_Get_Float();
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      i2 = Sdrw_Get_Int();
      n  = Sdrw_Get_Int();
      if (n > 16) n = 16;
      for (i = 0; i < n; i++) itb[i] = Sdrw_Get_Int();
      m = Sdrw_Get_Int();
      if (m > 0) {
        if (m > 48) m = 48;
        for (i = 0; i < m; i++) ftb[i+3] = Sdrw_Get_Float();      
        m = Draw_Gen_Box( ftb, i0, i1, i2, n, itb, m, ftb+3 );
      }
      else m = Draw_Gen_Box( ftb, i0, i1, i2, n, itb, 0, NULL );
      Sdrw_Put_Int( m );
    break;

    case cd_easy_box:
      x  = Sdrw_Get_Float();          /* Get the lower left corner of Box */
      y  = Sdrw_Get_Float();
      x0 = Sdrw_Get_Float();          /* Get the axis sizes */
      y0 = Sdrw_Get_Float();
      for (i = 0; i < 4; i++) ftb[i] = Sdrw_Get_Float(); /* Get XY mini_maxi */
      i0 = Sdrw_Get_String( Drwgl_strmsg ); /* Get the Units names */
      i1 = Sdrw_Get_String( Drwgl_strdef );
      m = Sdrw_Get_Int();
      i = Draw_Easy_Box_2D( x, y, x0, y0, ftb, Drwgl_strmsg, Drwgl_strdef, m );
      Sdrw_Put_Int( i );
    break;

    case cd_free_box:
      pb  = Draw_Locate_Box( Sdrw_Get_Int() );  /* Get the Segment Number */
      m   = Sdrw_Get_Char();
      Draw_Free_Box( pb, m );
      Sdrw_Put_Char( 0 );
    break;

    case cd_free_all_box:
      Draw_Free_Box_List();
      Sdrw_Put_Char( 0 );
    break;

    case cd_set_values:
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      Draw_Axis_Setv( Draw_Locate_Axis( i0 ), i1 );
      Sdrw_Put_Char( 0 );      
    break;

    case cd_get_values:
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      Draw_Axis_Getv( Draw_Locate_Axis( i0 ), i1 );
    break;

    case cd_box_info:
      i0 = Sdrw_Get_Int(); /* Get the Box ident */
      i1 = Sdrw_Get_Int(); /* Get the Box GEt Key */
      Draw_Box_Getv( Draw_Locate_Box( i0 ), i1 );
    break;

    case cd_plot_axis:
      i0 = Sdrw_Get_Int();
      i1 = Sdrw_Get_Int();
      /* Get the World Axis Origine */
      for (i = 0; i < 3; i++) ftb[i] = Sdrw_Get_Float();
      Draw_Plot_Axis( Draw_Locate_Axis( i0 ), ftb, i1, NULL );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_plot_box:
      i0 = Sdrw_Get_Int(); /* Get the Box ident */
      Draw_Plot_Box( Draw_Locate_Box( i0 ) );
      bupd = draw_pvw;
      Sdrw_Anim_acq( Drwgl_astpstate );
    break;

    case cd_open_box:
      pb  = Draw_Locate_Box( Sdrw_Get_Int() );  /* Get the Box Reference */
      Draw_Open_Box( pb );
      Sdrw_Put_Char( 0 );      
    break;

    case cd_close_box:
      Draw_Close_Box();
      Sdrw_Put_Char( 0 );      
    break;

    case cd_cm_to_box:
      if (Draw_Opened_Box) {
        x = Sdrw_Get_Float() + orgx;
        y = Sdrw_Get_Float() + orgy;
        z = Sdrw_Get_Float() + orgz;
        Draw_UnScale( ftb, ftb+1, ftb+2, x, y, z );
        Sdrw_Put_Int( 0 );
        Sdrw_Put_Float( ftb[0] );
        Sdrw_Put_Float( ftb[1] );
        Sdrw_Put_Float( ftb[2] );
      }
      else  Sdrw_Put_Int( -1 );
    break;


    case cd_box_to_cm:
      if (Draw_Opened_Box) {
        x = Sdrw_Get_Float();
        y = Sdrw_Get_Float();
        z = Sdrw_Get_Float();
        Draw_Scale3( x, y, z, ftb, 0 );
        Sdrw_Put_Int( 0 );
        Sdrw_Put_Float( ftb[0] - orgx );
        Sdrw_Put_Float( ftb[1] - orgy );
        Sdrw_Put_Float( ftb[2] - orgz );
      }
      else  Sdrw_Put_Int( -1 );
    break;

    case cd_continue:
      /* Send a 2D/3D Plot Data block to Client */
      Draw_Send_Points( Sdrw_Get_Char() );
    break;

    case cd_message:
      Sdrw_Get_String( Drwgl_strmsg );/* Set the Message String */
      Drwgl_Input = drwgl_message;    /* Set The Request Kind */
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
      Sdrw_Put_Int( 1 );              /* Return the success status */
    break;

    case cd_get_choice:
      n = Sdrw_Get_Int();
      if ((n < 1)||(n > Menu_Max))
      {
        Sdrw_Put_Int( -2 );
        break;
      }
      Sdrw_Get_String( Drwgl_strmsg );
      menu_tops = 0;
//    fprintf( fmsg, " *** Get Choice Request \"%s\".\n", Drwgl_strmsg );
//    Draw_Fmsgupdate();
      for (menu_topn = 0; menu_topn < n; menu_topn++)
      {
        menu[menu_topn] = menu_buf+menu_tops;
        len = Sdrw_Get_Char();
        if (len > Menu_ItmSz) len = Menu_ItmSz;
        while (len--)
          menu_buf[menu_tops++] = Sdrw_Get_Char();
        menu_buf[menu_tops++] = 0;
//      fprintf( fmsg, " Choice %2d / \"%s\"\n", menu_buf[menu_tops-1] );
//      Draw_Fmsgupdate();
      }
      menu[n] = NULL;
      Drwgl_strtab = menu;
      Drwgl_Input = drwgl_choice;     /* Set The Request Kind */
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_get_answerd:
      Sdrw_Get_String( Drwgl_strmsg );
      Drwgl_Input = drwgl_answerd;    /* Set The Request Kind */
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_get_string:
      i = Sdrw_Get_String( Drwgl_strmsg );
      n = Sdrw_Get_String( Drwgl_strdef );
      Drwgl_strcap = Sdrw_Get_Char();
      Drwgl_Input = drwgl_string;     /* Set The Request Kind */
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_get_file:
//    fprintf( fmsg, " *** Get File Request 0.\n" );
//    Draw_Fmsgupdate();
      Drwgl_newmd = Sdrw_Get_Char();  /* Get the Create New File Flag */
      n = Sdrw_Get_Char();            /* Get number of filter entries */
      Sdrw_Get_String( Drwgl_strmsg );/* Get the Title */
      menu_tops = 0;
//    fprintf( fmsg, " *** Get File Request 1.\n" );
//    Draw_Fmsgupdate();
      for (menu_topn = 0; menu_topn < n; menu_topn++)
      { /* Get each Filter table element */
        menu[menu_topn] = menu_buf+menu_tops;
        len = Sdrw_Get_Char();
        if (len > Menu_ItmSz) len = Menu_ItmSz;
        while (len--)
          menu_buf[menu_tops++] = Sdrw_Get_Char();
        menu_buf[menu_tops++] = 0;
      }
      menu[n] = NULL;
//    fprintf( fmsg, " *** Get File Request 2.\n" );
//    Draw_Fmsgupdate();
      Drwgl_strtab = menu;
      Drwgl_fidx = Sdrw_Get_Char();   /* Get the filter index */
      Sdrw_Get_String( Drwgl_strdef );/* Get the Default file spc. */
      Drwgl_strcap = Sdrw_Get_Char(); /* ... and the target string capacity */
      Drwgl_Input = drwgl_file;       /* Set The Request Kind */
//    fprintf( fmsg, " *** Get File Request.\n" );
//    Draw_Fmsgupdate();
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_get_position:
      Drwgl_Input = drwgl_position;   /* Set The Request Kind */
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_get_window:
      Drwgl_Input = drwgl_window;     /* Set The Request Kind */
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_get_value:
      Sdrw_Get_String( Drwgl_strmsg );
      Drwgl_valdef = Sdrw_Get_Float();
      Drwgl_valmin = Sdrw_Get_Float();
      Drwgl_valmax = Sdrw_Get_Float();
      Drwgl_Input  = drwgl_value;     /* Set The Request Kind */
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_get_stroke:
    case cd_get_line:
      Drwgl_gptmax = Sdrw_Get_Int();  /* Get Application Capacity */
      Drwgl_gptins = Sdrw_Get_Char(); /* Get the Plot insertion flag */
      Drwgl_gptcnt = 0;
      if (code == cd_get_line)             /* Set The Request Kind */
        Drwgl_Input  = drwgl_line;
      else
        Drwgl_Input  = drwgl_stroke;
//    fprintf( fmsg, " => Enter in Get_Line/Stroke_Req.\n" );
//    Draw_Fmsgupdate();
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_seg_pick:
      Drwgl_segment = Sdrw_Get_Int();
      Drwgl_ident   = Sdrw_Get_Int();
      Drwgl_Input   = drwgl_pick;     /* Set The Request Kind */
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_dialog:
      /* draw__dialog(); */
      Drwgl_exitmd  = Sdrw_Get_Int(); /* Get the Dialog Flag Word */
      Drwgl_Dstatus = 0;              /* Set the Initial Dialog Status */
      Drwgl_Input   = drwgl_dialog;   /* Set The Request Kind */
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
      Sdrw_Put_Int( Drwgl_Dstatus );  /* Return the account code to client */
    break;

    case cd_zoom:
      Drwgl_Input = drwgl_usr_zoom;   /* Set The Request Kind */
//    fprintf( fmsg, " => Enter in Zoom_Req.\n" );
//    Draw_Fmsgupdate();
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_scale:
      Drwgl_Input = drwgl_usr_scale;  /* Set The Request Kind */
//    fprintf( fmsg, " => Enter in Zoom_Req.\n" );
//    Draw_Fmsgupdate();
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_reload:
      Drwgl_Input = drwgl_reload;     /* Set The Request Kind */
//    fprintf( fmsg, " => Enter in Zoom_Req.\n" );
//    Draw_Fmsgupdate();
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_refresh:
      Drwgl_Input = drwgl_refresh;    /* Set The Request Kind */
//    fprintf( fmsg, " => Enter in Zoom_Req.\n" );
//    Draw_Fmsgupdate();
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_grid:
      Drwgl_Input = drwgl_usr_grid;   /* Set The Request Kind */
//    fprintf( fmsg, " => Enter in Zoom_Req.\n" );
//    Draw_Fmsgupdate();
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    case cd_print:
      Drwgl_Input = drwgl_usr_print;  /* Set The Request Kind */
//    fprintf( fmsg, " => Enter in Zoom_Req.\n" );
//    Draw_Fmsgupdate();
      _Inp_Req_Display( &bupd );      /* Send the Request to VGL */
    break;

    default:
      /* Normal Draw_End and Illegal or Unsupported Draw Code */
      i0 = (code == cd_end)?-1:-2;
      Draw_End();            /* Close All Draw Display. */
      Sdrw_Put_Int( i0 + 1 );/* Return the account code to client */
      Sdrw_Write();
      Sdrw_Close_Gate();     /* Used for debug in reverse mode. */
      /* Return the value -1 on Normal Draw_End call and ... */
      /* ... -2 on ERROR Draw_End call. */
      reqnxt = i0;
      goto Sequence_Exit;
    }

    Sdrw_read_unlock();      /* Lock the User Directive Execution Graph */

    if (bupd) DrwGL_Update_Display();
    Sdrw_Write();            /* Write the answerd to the user. */
    reqnxt = 0;              /* Return to continue pipe input */
  }

Sequence_Exit: /* Exit form proceed routine */

  Sdrw_read_unlock();        /* UnLock the User Directive Execution Graph */

  return NULL;               /* Requested by pthread management */
}

