/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               D R A W   -   S E R V E R   M O D U L E                 *
*                                                                       *
*              (Kernel to manage the bidirectional pipe)                *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/

/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/

/***********************************************************************/
/**                                                                   **/
/**           Echange DRAW_LIb <-> DRAW_SERVER Constantes             **/
/**                                                                   **/
/***********************************************************************/


#include <semaphore.h>
#include <pthread.h>

#include <draw/sdrw_env.h>
#include <draw/sdrw_routines.h>




#ifdef _PIPE_BIG
# ifdef VMS
#   define _Pipe( a , b, c ) pipe( a, b, c)
# else
#   define _Pipe( a , b, c ) _pipe( a, b, c)
# endif
#else
# define _Pipe( a , b, c ) pipe( a )
#endif



#ifdef _SDRW__REVERSE
# define SDRW_input  20
# define SDRW_output 21
#endif



/*********************************************************************/
/**                                                                 **/
/**      Echange DRAW_LIb <-> DRAW_SERVER Types Definitions         **/
/**                                                                 **/
/*********************************************************************/



typedef unsigned char Char;             /* Define the Unsigned char type Char */

typedef union {                         /* Define the Type Equiv. rec. type */
          struct { Char ch0, ch1, ch2, ch3; } ct;
                    int   i;
                    float f;
		} eqty;



/*********************************************************************/
/**                                                                 **/
/**          Exchange DRAW_SERVER <-> DRAW_LIB Variables            **/
/**                                                                 **/
/*********************************************************************/


static  int   s_inpipe,   s_outpipe,  /* Server pipe I/O channels */
              c_inpipe,   c_outpipe;  /* Client pipe I/O channels */

static  int   Sdrw_Insize,            /* Number of byte readden */
              Sdrw_Outsize,           /* Number of byte to write */
              Sdrw_Inread;            /* Index in the Read Pipe Buffer */

static  Char  Sdrw_Inp_Buf[BUF_SIZE], /* Buffer for Input from DRAW-LIB */
              Sdrw_Out_Buf[BUF_SIZE]; /* Buffer for output to DRAW-LIB */

static  int   Sdrw_Client_id;         /* Draw Client Process id. */

static  eqty  equ;                    /* Record for float<->int<->Char Conv. */

/* Mutex for critical sections */
//static  pthread_mutex_t sdrw_mutex = PTHREAD_MUTEX_INITIALIZER;

static  pthread_mutex_t sdrw_data_mutex; /* Mutex for VGL data managment */

static  pthread_t       sdrw_pthread; /* Pthread of pipe manager */

static  sem_t           sdrw_semcpx;  /* Semaphore for Request synchro. */



/*********************************************************************/
/**                                                                 **/
/**   Data Echange Routines for DRAW_LIB <-> DRAW_SERVER Dialogue   **/
/**                                                                 **/
/*********************************************************************/


void Sdrw_Open_Gate( int arc, char **arv )
{ /* Attach the DRAW-LIB pipe channels to the local channel */
  int ie;

#ifdef _SDRW__REVERSE

  int  cs_pipes[2], sc_pipes[2],  /* Original bi-directional pipe */
       status;

  if (_Pipe( cs_pipes, 0, BUF_SIZE ) == -1)
    {
      printf( " SDRW : - Client to Server Pipe allocation failed\n" );
      exit( 1 );
    }
  if (_Pipe( sc_pipes, 0, BUF_SIZE ) == -1)
    {
      printf( " SDRW : - Server to Client Pipe allocation failed\n" );
      exit( 2 );
    }

  s_inpipe  = cs_pipes[0];
  s_outpipe = sc_pipes[1];

  c_inpipe  = dup2( sc_pipes[0], SDRW_input );
  if (c_inpipe < 0) exit( 11 );

  c_outpipe = dup2( cs_pipes[1], SDRW_output );
  if (c_outpipe < 0) exit( 12 );


  status = fork();                            /* Create the Draw Client process */

  switch(status)                              /* Select the process */
  {
    case 0:                                   /* For Server process, we try ... */
      if ((status = execv( arv[1], arv+1 )) == -1)    /* ... to start it. */
        exit( 3 );                            /* Stop with message on Exec ERROR. */
    break;

    case -1:                                  /* Error return on fork */
      exit( 4 );
    break;

    default:                                  /* Start DRAW Server Success */
      close( c_inpipe );                      /* Free the unused Client pipe channel */
      close( c_outpipe );
    break;
  }

#else

  int chinp, chout, i;

  if (arc < 4) {
    fprintf( fmsg, " Illegal DRAW_SERVER Call (number of parameters < 4).\n" );
    for (i=0;i<arc;i++)
      fprintf( fmsg, " Parameter #%d = \"%s\".\n", i, arv[i] );
    fprintf( fmsg,
      " The normal call is <Server_Path> <in_pipe#> <out_pipe#> <User_pid> [<User_Label>]\n" );
    fclose( fmsg );
    exit( 2 );
  }

  chinp = atoi( arv[1] );                     /* Get the Channel number from ... */
  chout = atoi( arv[2] );                     /* ...the Parameter list */
  Sdrw_Client_id = atoi( arv[3] );            /* Get the Client Process id. */

  s_inpipe  = dup( chinp );                   /* Get the Channel number from ... */
  s_outpipe = dup( chout );                   /* ...the Parameter list */
  close( chinp );                             /* Close temporary channels */
  close( chout );

  /* Set the Asynchronous read mode to do not block the sreen reactivity */
  // fcntl( s_inpipe, F_SETFL, O_NONBLOCK|fcntl( s_inpipe, F_GETFL, 0 ) );

#endif

  ie = pthread_mutex_init( &sdrw_data_mutex, NULL );

  ie = sem_init( &sdrw_semcpx, 0, 0 );
  if (ie)
  {
    if (ie == ENOSYS)
      fprintf( fmsg, " ***Init Semaphore ENOSYS\n" );
    if (ie == 22)
      fprintf( fmsg, " ***Init Semaphore EINVAR\n" );
    Draw_Fmsgupdate();
    fclose( fmsg );
    exit ( 2 );
  }
}




/************************************************************************/

void Sdrw_Start_User_Pipe( void )
{ int ie;
  /* Start the pipe handler as a new thread */
  fprintf( fmsg, " *** User Pipe Thread Creation.\n" );
  Draw_Fmsgupdate();
  ie = pthread_create( &sdrw_pthread, NULL, Draw_Sequence_Exec, NULL );
  if (ie)
  {
    fprintf( fmsg, " *** Error on thread Creation: = %d:%d\n", ie, errno );
    Draw_Fmsgupdate();
  }
//else
//{
//  fprintf( fmsg, " *** User Pipe Thread Created.\n" );
//  Draw_Fmsgupdate();
//}
}



void Sdrw_Resume_User_Pipe( void )
{ int ie;
//fprintf( fmsg, " *** Post End Of Request (to continue Pipe Thread).\n" );
//Draw_Fmsgupdate();
  ie = sem_post( &sdrw_semcpx );
  if (ie)
  {
    fprintf( fmsg, " *** Post End Of Request err = %d:%d \n", ie, errno );
    Draw_Fmsgupdate();
  }
}



void Sdrw_Wait_For_Request( void )
{ int ie;
//fprintf( fmsg, " *** Wait (sem) for Request - Block pipe Thread.\n" );
//Draw_Fmsgupdate();

  ie = sem_wait( &sdrw_semcpx );
  if (ie)
  {
    fprintf( fmsg, " *** Wait for Request err = %d:%d\n", ie, errno );
    Draw_Fmsgupdate();
  }
//fprintf( fmsg, " *** End of Wait (sem) for Request, Continue in pipe Thread.\n" );
//Draw_Fmsgupdate();
}



void Sdrw_read_lock()
{
  int ie;

  ie = pthread_mutex_lock( &sdrw_data_mutex );
}


void Sdrw_read_unlock()
{
  int ie;

  ie = pthread_mutex_unlock( &sdrw_data_mutex );
}




/************************************************************************/

void Sdrw_Close_Gate()
{ /* Stop Server Connection and Exit */
  int status, cstatus;
  if ((status = wait( &cstatus )) == -1)
    exit( 10 );
}


/************************************************************************/

int Sdrw_Read()
{ /* Read a buffer from the DRAW Library */
  Sdrw_Insize = read( s_inpipe, Sdrw_Inp_Buf, BUF_SIZE );
  if (Sdrw_Insize < 0)
  {
    fprintf( fmsg, " SDRW: Read from Client failed.\n" );
    exit( 21 );
  }

//fprintf( fmsg, " SDRW Read %d bytes.\n", Sdrw_Insize );
//Draw_Fmsgupdate();

  Sdrw_Inread = 0;
  sdrw_read_left = Sdrw_Insize;
  return Sdrw_Insize;
}



/************************************************************************/

void Sdrw_Write()
{ /* Write a buffer to the DRAW Library */
  if (write( s_outpipe, Sdrw_Out_Buf, Sdrw_Outsize ) == -1)
  {
    fprintf( fmsg, " SDRW: Write to Client failed.\n" );
    exit( 22 );
  }
  Sdrw_Outsize = 0;
}



/************************************************************************/

int  Sdrw_Read_Block( Char* tb, int dim )
{ /* Read a User buffer from the DRAW Library */
  int n, i, t;

//fprintf( fmsg, " *** SDRW Read Block of %d bytes with a pipe buf size of %d bytes.\n", dim, BUF_SIZE );
//Draw_Fmsgupdate();

#ifndef  _PIPE_NO_CNTFLOW

    Sdrw_Put_Code( 0 );
    Sdrw_Write();

#endif

  /* The User has begun to send the Data Block (stored at tb, dim bytes length) */
  t = 0;
  while (t < dim) {

#ifdef  _PIPE_NO_CNTFLOW

    Sdrw_Put_Code( 0 );
    Sdrw_Write();

#endif

#ifdef _PIPE_NOBYPASS

    Sdrw_Read();
    n = Sdrw_Insize;

//  fprintf( fmsg, " SDRW Block: Read %d bytes.\n", n );
//  Draw_Fmsgupdate();

    for (i = 0; i < Sdrw_Insize; i++) tb[i] = Sdrw_Inp_Buf[i];

#else

    n = read( s_inpipe, tb, BUF_SIZE );
    if (n < 0)
    {
      fprintf( fmsg, " SDRW Block: D Read from Client failed.\n" );
      exit( 23 );
    }
//  fprintf( fmsg, " SDRW Block: D Read %d bytes.\n", n );
//  Draw_Fmsgupdate();


#endif

    t  += n;
    tb += n;

//  fprintf( fmsg, " SDRW Block: Left %d/%d bytes.\n", dim-t, dim );
//  Draw_Fmsgupdate();

  }
  return t;
}



/************************************************************************/

void Sdrw_Write_Block( Char* tb, int dim )
{ /* Write a User buffer to the DRAW Library */
  int i, n;

#ifndef _PIPE_NO_CNTFLO

    Sdrw_Read(); /* Read for synchro */

#endif

  /* The User must be ready to get the Data Block (address tb, dim bytes length) */
  while (dim) {
    n = (dim > BUF_SIZE)?BUF_SIZE:dim;

#ifdef _PIPE_NO_CNTFLO

    Sdrw_Read(); /* Read for synchro */

#endif

#ifdef _PIPE_NOBYPASS

    for (i = 0; i < n; i++) Sdrw_Out_Buf[i] = tb[i];
    Sdrw_Outsize = n;
    Sdrw_Write();

#else

    if (write( s_outpipe, tb, n ) == -1)
    {
      fprintf( fmsg, " SDRW Block: Write to Client failed.\n" );
      exit( 24 );
    }

#endif

    dim -= n;
    tb  += n;

  }
}



/************************************************************************/

int Sdrw_Dialog_Request()
{
  Sdrw_Write();
  return Sdrw_Read();
}



/************************************************************************/
/**          Input from and Output to the Graphic Server Routines      **/
/************************************************************************/

void Sdrw_Put_Char( Char ch )
{
  if (Sdrw_Outsize < BUF_SIZE)
    Sdrw_Out_Buf[Sdrw_Outsize++] = ch;
}



/*********************************************************************/

void Sdrw_Put_Int( int iv )
{
  equ.i = iv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
}



/*********************************************************************/

void Sdrw_Put_Int_Direct( int pos, int iv )
{
  int Save_Outsize;

  Save_Outsize = Sdrw_Outsize;
  Sdrw_Outsize = pos;

  equ.i = iv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
  Sdrw_Outsize = Save_Outsize;
}



/*********************************************************************/

void Sdrw_Put_Float( float fv )
{
  equ.f = fv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
}



/*********************************************************************/

void Sdrw_Put_String( char * s, int l )
{
  int i;

  if (l == 0) l = strlen( s );
  Sdrw_Put_Char( l );
  for ( i=0; i<l; i++)
  {
    Sdrw_Put_Char( s[i] );
  }
}



/*********************************************************************/

void Sdrw_Put_Code( int code )
{
  Sdrw_Outsize = 0;
  Sdrw_Put_Int( code );
}



/***************************************************************************/

Char Sdrw_Get_Char()
{
  char ch;

  if (Sdrw_Inread < Sdrw_Insize) {
    ch = Sdrw_Inp_Buf[Sdrw_Inread++];
    sdrw_read_left = Sdrw_Insize - Sdrw_Inread;
  }
  else
    ch = 0;
  return ch;
}



/*********************************************************************/

int Sdrw_Get_Int()
{
  equ.ct.ch0 = Sdrw_Get_Char();
  equ.ct.ch1 = Sdrw_Get_Char();
  equ.ct.ch2 = Sdrw_Get_Char();
  equ.ct.ch3 = Sdrw_Get_Char();
  return equ.i;
}



/*********************************************************************/

float Sdrw_Get_Float()
{
  equ.ct.ch0 = Sdrw_Get_Char();
  equ.ct.ch1 = Sdrw_Get_Char();
  equ.ct.ch2 = Sdrw_Get_Char();
  equ.ct.ch3 = Sdrw_Get_Char();
  return equ.f;
}



/*********************************************************************/

int Sdrw_Get_String( char * s )
{
  int i, l;

  l = Sdrw_Get_Char();

  for (i=0; i<l; i++)
    s[i] = Sdrw_Get_Char();
  s[i] = 0;                /* Append a SOH for C string handling */
  return l;
}


/*********************************************************************/

char * Sdrw_Get_String_Tmp()
{
  int i, l;
  char * s;

  l = Sdrw_Get_Char();
  s = (char *) malloc( l + 1 );

  for (i=0; i<l; i++)
    s[i] = Sdrw_Get_Char();
  s[i] = 0;                /* Append a SOH for C string handling */
  return s;
}


/*********************************************************************/

void Sdrw_String_Free_Tmp( char * s )
{
  if (s != NULL)
  {
    free( s );
    s = NULL;
  }
}


/*********************************************************************/

int Sdrw_Get_Code()
{
  Sdrw_Inread = 0;
  return Sdrw_Get_Int();
}


/*********************************************************************/

Char * Sdrw_Put_Address( int n )
{ Char * p;
  p = Sdrw_Out_Buf + Sdrw_Outsize;
  Sdrw_Outsize += n;
  return p;
}



/*********************************************************************/

Char * Sdrw_Get_Address( int n )
{ Char * p;
  p = Sdrw_Inp_Buf + Sdrw_Inread;
  Sdrw_Inread += n;
  return p;
}


/************************************************************************/

void Sdrw_Anim_acq( int isgn )
{
  int sig;

  Sdrw_Put_Char( isgn&255 );
//fprintf( fmsg, " Anim_acq %d %d %d\n", isgn, isgn&140, (isgn/16)&3 );
//Draw_Fmsgupdate();
  if (isgn&140) {
    switch ( (isgn/16)&3 ) {
      case 2: { sig = SIGUSR2; break; }
      case 3: { sig = SIGABRT; break; }
      default: sig = SIGUSR1;
    }
    kill( Sdrw_Client_id, sig );
  }
}



/*************************     E N D    ******************************/

