/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               D R A W   -   S E R V E R   M O D U L E                 *
*                                                                       *
*              (Kernel to manage the bidirectional pipe)                *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/

/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/

/***********************************************************************/
/**                                                                   **/
/**           Echange DRAW_LIb <-> DRAW_SERVER Constantes             **/
/**                                                                   **/
/***********************************************************************/


#include <draw/sdrw_null_env.h>



#ifdef _PIPE_BIG
# ifdef VMS
#   define _Pipe( a , b, c ) pipe( a, b, c)
# else
#   define _Pipe( a , b, c ) _pipe( a, b, c)
# endif
#else
# define _Pipe( a , b, c ) pipe( a )
#endif





#ifdef _SDRW__REVERSE
# define SDRW_input  20
# define SDRW_output 21
#endif



/*********************************************************************/
/**                                                                 **/
/**      Echange DRAW_LIb <-> DRAW_SERVER Types Definitions         **/
/**                                                                 **/
/*********************************************************************/



typedef unsigned char Char;             /* Define the Unsigned char type Char */

typedef union {                         /* Define the Type Equiv. rec. type */
          struct { Char ch0, ch1, ch2, ch3; } ct;
                    int   i;
                    float f;
		} eqty;



/*********************************************************************/
/**                                                                 **/
/**          Exchange DRAW_SERVER <-> DRAW_LIB Variables            **/
/**                                                                 **/
/*********************************************************************/


static  int   s_inpipe,   s_outpipe,  /* Server pipe I/O channels */
              c_inpipe,   c_outpipe;  /* Client pipe I/O channels */

static  int   Sdrw_Insize,            /* Number of byte readden */
              Sdrw_Outsize,           /* Number of byte to write */
              Sdrw_Inread;            /* Index in the Read Pipe Buffer */

static  Char  Sdrw_Inp_Buf[BUF_SIZE], /* Buffer for Input from DRAW-LIB */
              Sdrw_Out_Buf[BUF_SIZE]; /* Buffer for output to DRAW-LIB */

static  int   Sdrw_Client_id;         /* Draw Client Process id. */

static  eqty  equ;                    /* Record for float<->int<->Char Conv. */





/*********************************************************************/
/**                                                                 **/
/**   Data Echange Routines for DRAW_LIB <-> DRAW_SERVER Dialogue   **/
/**                                                                 **/
/*********************************************************************/


void Sdrw_Open_Gate( int arc, char **arv )
{ /* Attach the DRAW-LIB pipe channels to the local channel */
  int ie;

#ifdef _SDRW__REVERSE

  int  cs_pipes[2], sc_pipes[2],  /* Original bi-directional pipe */
       status;

  if (_Pipe( cs_pipes, 0, BUF_SIZE ) == -1)
    {
      printf( " SDRW : - Client to Server Pipe allocation failed\n" );
      exit( 1 );
    }
  if (_Pipe( sc_pipes, 0, BUF_SIZE ) == -1)
    {
      printf( " SDRW : - Server to Client Pipe allocation failed\n" );
      exit( 2 );
    }

  s_inpipe  = cs_pipes[0];
  s_outpipe = sc_pipes[1];
 
  c_inpipe  = dup2( sc_pipes[0], SDRW_input );
  if (c_inpipe < 0) exit( 11 );
  
  c_outpipe = dup2( cs_pipes[1], SDRW_output );
  if (c_outpipe < 0) exit( 12 );
  

  status = fork();                            /* Create the Draw Client process */

  switch(status)                              /* Select the process */
  {
    case 0:                                   /* For Server process, we try ... */
      if ((status = execv( arv[1], arv+1 )) == -1)    /* ... to start it. */
        exit( 3 );                            /* Stop with message on Exec ERROR. */
    break;

    case -1:                                  /* Error return on fork */
      exit( 4 );
    break;

    default:                                  /* Start DRAW Server Success */
      close( c_inpipe );                      /* Free the unused Client pipe channel */
      close( c_outpipe );
    break;
  }
  
#else

  int chinp, chout;

  chinp = atoi( arv[1] );                     /* Get the Channel number from ... */
  chout = atoi( arv[2] );                     /* ...the Parameter list */
  Sdrw_Client_id = atoi( arv[3] );            /* Get the Client Process id. */

  s_inpipe  = dup( chinp );                   /* Get the Channel number from ... */
  s_outpipe = dup( chout );                   /* ...the Parameter list */

  close( chinp );                             /* Close temporary channels */
  close( chout );

  /* Set the Asynchronous read mode to do not block the sreen reactivity */
  // fcntl( s_inpipe, F_SETFL, O_NONBLOCK|fcntl( s_inpipe, F_GETFL, 0 ) );

#endif

}




/************************************************************************/

void Draw_Start_User_Pipe( void )
{ int ie;
  /* Start the pipe handler as a new thread */
}



void Draw_Resume_User_Pipe( void )
{ int ie;
}



void Sdrw_Wait_For_Request( void )
{ int ie;
}



void Sdrw_read_lock( int id )
{
  int ie;

//fprintf( fmsg, " *** Mutex Read Lock from %d.\n", id );
//Draw_Fmsgupdate();
//ie = pthread_mutex_lock( &sdrw_data_mutex );
}


void Sdrw_read_unlock( int id )
{
  int ie;

//fprintf( fmsg, " *** Mutex Read Unlock from %d.\n", id );
//Draw_Fmsgupdate();
//ie = pthread_mutex_unlock( &sdrw_data_mutex );
//fprintf( fmsg, " *** Mutex Read  from %d, Unlock code = %d.\n", id, ie );
//Draw_Fmsgupdate();
}




/************************************************************************/

void Sdrw_Close_Gate()
{ /* Stop Server Connection and Exit */
  int status, cstatus;
  if ((status = wait( &cstatus )) == -1)
    exit( 10 );
}


/************************************************************************/

int Sdrw_Read()
{ /* Read a buffer from the DRAW Library */
  Sdrw_Insize = read( s_inpipe, Sdrw_Inp_Buf, BUF_SIZE );
  if (Sdrw_Insize < 0)
  {
    fprintf( fmsg, " SDRW: Read from Client failed.\n" );
    exit( 21 );
  }
  Sdrw_Inread = 0;
  return Sdrw_Insize;
}



/************************************************************************/

void Sdrw_Write()
{ /* Write a buffer to the DRAW Library */
  if (write( s_outpipe, Sdrw_Out_Buf, Sdrw_Outsize ) == -1)
  {
    fprintf( fmsg, " SDRW: Write to Client failed.\n" );
    exit( 22 );
  }
  Sdrw_Outsize = 0;
}



/************************************************************************/

int Sdrw_Dialog_Request()
{
  Sdrw_Write();
  return Sdrw_Read();
}



/************************************************************************/
/**          Input from and Output to the Graphic Server Routines      **/ 
/************************************************************************/

void Sdrw_Put_Char( Char ch )
{
  if (Sdrw_Outsize < BUF_SIZE)
    Sdrw_Out_Buf[Sdrw_Outsize++] = ch;
}



/*********************************************************************/

void Sdrw_Put_Int( int iv )
{
  equ.i = iv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
}



/*********************************************************************/

void Sdrw_Put_Int_Direct( int pos, int iv )
{
  int Save_Outsize;

  Save_Outsize = Sdrw_Outsize;
  Sdrw_Outsize = pos;

  equ.i = iv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
  Sdrw_Outsize = Save_Outsize;
}



/*********************************************************************/

void Sdrw_Put_Float( float fv )
{
  equ.f = fv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
}



/*********************************************************************/

void Sdrw_Put_String( char * s, int l )
{
  int i;

  if (l == 0) l = strlen( s );
  Sdrw_Put_Char( l );
  for ( i=0; i<l; i++)
  {
    Sdrw_Put_Char( s[i] );
  }
}



/*********************************************************************/

void Sdrw_Put_Code( int code )
{
  Sdrw_Outsize = 0;
  Sdrw_Put_Int( code );
}



/***************************************************************************/

Char Sdrw_Get_Char()
{
  if (Sdrw_Inread < Sdrw_Insize)
    return Sdrw_Inp_Buf[Sdrw_Inread++];
  else
    return 0;
}



/*********************************************************************/

int Sdrw_Get_Int()
{
  equ.ct.ch0 = Sdrw_Get_Char();
  equ.ct.ch1 = Sdrw_Get_Char();
  equ.ct.ch2 = Sdrw_Get_Char();
  equ.ct.ch3 = Sdrw_Get_Char();
  return equ.i;
}



/*********************************************************************/

float Sdrw_Get_Float()
{
  equ.ct.ch0 = Sdrw_Get_Char();
  equ.ct.ch1 = Sdrw_Get_Char();
  equ.ct.ch2 = Sdrw_Get_Char();
  equ.ct.ch3 = Sdrw_Get_Char();
  return equ.f;
}



/*********************************************************************/

int Sdrw_Get_String( char * s )
{
  int i, l;

  l = Sdrw_Get_Char();

  for (i=0; i<l; i++)
    s[i] = Sdrw_Get_Char();
  s[i] = 0;                /* Append a SOH for C string handling */
  return l;
}


/*********************************************************************/

char * Sdrw_Get_String_Tmp()
{
  int i, l;
  char * s;

  l = Sdrw_Get_Char();
  s = (char *) malloc( l + 1 );

  for (i=0; i<l; i++)
    s[i] = Sdrw_Get_Char();
  s[i] = 0;                /* Append a SOH for C string handling */
  return s;
}


/*********************************************************************/

void Sdrw_String_Free_Tmp( char * s )
{
  if (s != NULL)
  {
    free( s );
    s = NULL;
  }
}


/*********************************************************************/

int Sdrw_Get_Code()
{
  Sdrw_Inread = 0;
  return Sdrw_Get_Int();
}


/*********************************************************************/

Char * Sdrw_Put_Address( int n )
{ Char * p;
  p = Sdrw_Out_Buf + Sdrw_Outsize;
  Sdrw_Outsize += n;
  return p;
}



/*********************************************************************/

Char * Sdrw_Get_Address( int n )
{ Char * p;
  p = Sdrw_Inp_Buf + Sdrw_Inread;
  Sdrw_Inread += n;
  return p;
}


/*********************************************************************/

void Sdrw_Put_FBlock( float tb[], int dim )
{ /* Output to pipe the array tb[dim] */
  int i, j, r;

  j = 0;
  while (dim > 0) {
    /* Fill the Buffer */
    r = (BUF_SIZE - sizeof( int ) - Sdrw_Outsize)/sizeof( float );
    if (r > dim) r = dim;
    Sdrw_Put_Int( r );
    for (i = 0; i < r; i++) Sdrw_Put_Float( tb[j++] );
    dim -= r;
    if (dim > 0) {
      Sdrw_Dialog_Request();
      Sdrw_Put_Code( cd_continue );
    }
  }
}


/*********************************************************************/

int Sdrw_Get_FBlock( float tb[], int sz, int dim )
{ /* Input an array from pipe to the array tb[sz],
     the returned value is the dimension of array in float */

  int i, j, n, t;

  j = 0;
  t = 0;
  while (t < dim) {
    n = Sdrw_Get_Int();
    i = 0;
    while ((i < n)&&(i < sz)) {
      tb[j++] = Sdrw_Get_Float();
      i++;
    }
    t += n;
    if (t < dim) {  /* Get the next message */
      Sdrw_Put_Code( cd_continue );
      Sdrw_Dialog_Request();
    }
  }
  return t;
}


/*************************     E N D    ******************************/

