/***** DRAW Interface ROUTINES for OpenGL Graphic Libary *******/
/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               D R A W   -   S E R V E R   M O D U L E                 *
*                                                                       *
*           (Draw interface for V function and procedures)              *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/

/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/


#include "VGLdrw_cmdw.h"

#include "Draw_VGL.h"
#include "VGLdrw_gl2ps.h"


/* include the environment SERVER */

#include <v/vnotice.h>  // so we can use notice
#include <v/vfilesel.h> // for file selection
// #include <v/vutil.h>
// #include <v/vynreply.h>
// #include <v/vkeys.h>


// #include <process.h>



/***********************************************************************/
/*                                                                     */
/*               DRAW/GL PostScript Interface Routines                 */
/*                                                                     */
/***********************************************************************/


#include <v/vpen.h>	 // for a pen
#include <v/vbrush.h>    // for a brush


#include <v/vprinter.h>  // For Printer management.
#include <v/vprintdc.h>

#include <v/vcolor.h>    // To Define the Color table.



/* #define JPEG_VGEN_USE */

#ifdef JPEG_VGEN_USE

# include <jpeglib.h>     /* Load the JPEG File Definitions */

#endif


/***** Warning Compilation error was suppress by change long int *****/
/***** in int in the file (of cygwin) /usr/include/jmorecfg.h    *****/





static GLubyte * Pixel_Buffer = NULL; /* Pixel Buffer (unsigned char) */
static GLfloat * Deep_Buffer  = NULL; /* Deep Buffer for Local Z Buffer */

static int       ImgBuf_Size  =    0; /* Size of allocated Image Buffer */
static int       Image_LineSz =    0; /* BitMap line size in bytes (Round Up to *4) */

static int       Image_Size   =    0; /* Current size of the image */
static int       Image_Width  =    0; /* Current Width of the image */
static int       Image_High   =    0; /* Current high of the image */

static FILE *    fimg;                /* Image File Descriptor Pointer */



/***********************************************************************/
/*                     Get  Pixel Image  Routine                       */
/***********************************************************************/




void DrwGL_Get_Image( Char * image, int px, int py, int sx, int sy )
{
  /* Load the image from the current GL Buffer if the buffer image */
  /* (px,py) are the coordinate (pixel) of the lower-left corner   */
  /* of the area and (sx,sy) are the size (pixel) of the image to  */
  /* load. The Bit Map is in 24 bits/pixel mode.                   */

  int npix;

//fprintf( fmsg, " GEt_Image 0 oldsz = %d.\n", ImgBuf_Size );
//Draw_Fmsgupdate();

  glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );

  Image_LineSz = (sx*3 + 3)&(~3); /* Compute the Round Up BitMap Line size of *4 */

  npix = Image_LineSz*sy;         /* Total BitMap Size in bytes (with Round up) */
  py = dt_ry - py - sy;           /* Change to Screen Coordinates */
  
  if (npix != ImgBuf_Size) {
    if (!Pixel_Buffer) free( Pixel_Buffer );
    Pixel_Buffer = (Char *) malloc( npix /* For RGB */ );
    ImgBuf_Size  = npix;
  }
  Image_Width =   sx;
  Image_High  =   sy;
  Image_Size  =  sx*sy;

//fprintf( fmsg, " GEt_Image 2: %d*%d=%d at %d .\n", sx, sy, npix, (int) Pixel_Buffer );
//Draw_Fmsgupdate();

//glReadBuffer( GL_FRONT );
  glReadPixels( px, py, sx, sy, GL_RGB, GL_UNSIGNED_BYTE, (void *) Pixel_Buffer );

  fprintf( fmsg, " GEt_Image End.\n" );
  Draw_Fmsgupdate();

}




/***********************************************************************/
/*                     Basic  Display  Routines                        */
/***********************************************************************/




//=============================>>> wrInt16 <<<=============================
  static void wrInt16( int iv )
  {
    union { char  ct[2];
            short iv;
          } equ;

//    int c0, c1;

//  c0 = iv & 255; c1 = (iv >> 8) & 255;
//  putc( c0, fimg );  putc( c1, fimg );

    equ.iv = iv;
    putc( equ.ct[0], fimg ); putc( equ.ct[1], fimg );
  }

//=============================>>> wrInt32 <<<=============================
  static void wrInt32( int iv )
  {
    union { char  ct[4];
            int   iv;
          } equ;

    equ.iv = iv;
    putc( equ.ct[0], fimg ); putc( equ.ct[1], fimg );
    putc( equ.ct[2], fimg ); putc( equ.ct[3], fimg );
  }


//=============================>>> wrInt32 <<<=============================
  static void wrFlt32( float v )
  {
    union { float    v;
            char ch[4];
          } equ;

    equ.v = v;
    putc( equ.ch[0], fimg ); putc( equ.ch[1], fimg );
    putc( equ.ch[2], fimg ); putc( equ.ch[3], fimg );
  }



int Write_bmp_image( char * name, int prtflag )
{ /* Routine to Create a BMP File from the Pixel_buffer Image */

//static int       Image_Size;  /* Current size of the image */
//static int       Image_Width; /* Current Width of the image */
//static int       Image_High;  /* Current high of the image */

  int bmsz, Hres, npad, Vres, nCol, i, j, status;
  Char   * p;
  char com[255];

//fprintf( fmsg, " BMP_Image Start.\n" );
//Draw_Fmsgupdate();

  if (fimg = fopen( name, "wb" )) { /* Open is a success */

    bmsz = ImgBuf_Size;     /* Bit Map Size (with Round up to a multiple of 4) */
    npad = (Image_Width*3)&3; if (npad) npad = 4 - npad; /* Compute BMP Round UP */

    Hres = (int) (d_xscale*100.0 + 0.5);
    Vres = (int) (d_yscale*100.0 + 0.5);

    nCol = 0x1000000;       /* Set to 2**24 Colors. */

    /* The 14 First Bytes Are Write Now */
    putc( 'B', fimg );      /* Generate the BM BitMap type */
    putc( 'M', fimg );
    wrInt32( bmsz + 54 );   /* Bit Map file Size */
    wrInt16( 0 );           /* Reserved Dword */
    wrInt16( 0 );           /* Reserved Dword */
    wrInt32( 54 );          /* Offset of Beginning of data (no Color Map) */

    /* The 40 Bytes of Image are Write Now */
    wrInt32( 0x28 );        /* Write bitmap Info Header Size */
    wrInt32( Image_Width ); /* Write Image Width */
    wrInt32( Image_High );  /* Write Image High */
    wrInt16( 1 );           /* Write the Number of planes */
    wrInt16( 24 );          /* Write the number of bits by Pixels (always 24 Here) */
    wrInt32( 0 );           /* Specify NO Compression */
    wrInt32( bmsz );        /* Specify the total size of the bitmap data */
    wrInt32( Hres );        /* Specify the Horizontal resolution (pixels/meter) */
    wrInt32( Vres );        /* Specify the Vertical resolution (pixels/meter) */
    wrInt32( nCol );        /* Specify the total number of used Color */
    wrInt32( 0 );           /* Specify the total number of Important Color */

//  fprintf( fmsg, " bmsz = %d.\n", bmsz );
//  fprintf( fmsg, " W = %d, H = %d.\n", Image_Width, Image_High );
//  fprintf( fmsg, " WR = %d, HR = %d.\n", Hres, Vres );
//  fprintf( fmsg, " ncol = %d.\n", ncol );
//  Draw_Fmsgupdate();


    /* We do not generate a Color map */

    /* Now Write the RGB BitMap */
    p = Pixel_Buffer;
    for (j = 0; j < Image_High; j++) {
      p = Pixel_Buffer + j*Image_LineSz;
      for (i = 0; i < Image_Width; i++) {
        putc( p[2], fimg );
        putc( p[1], fimg );
        putc( p[0], fimg );
        p += 3;
      }
      /* Write pad byte */
      for (i = 0; i < npad; i++) putc( 0, fimg );
    }

    /* Close the BMP File */
    fclose( fimg );

//  status = fork();           /* Create the Draw Client process */
//
//  if (status == 0)           /* Select the process */
//    if ((status = execl( "mspaint", "mspaint", "Draw_Image.bmp" )) == -1) /* ... to start it. */
//    ; //  exit( 3 );         /* Stop with message on Exec ERROR. */

    if (prtflag) { /* When a BitMap print is required */
      sprintf( com, "mspaint /P %s\n", name );
      fprintf( fmsg, " Execute Command com %s\n", com );
      system( com );
    }

//  fprintf( fmsg, " BMP_Image End on Success.\n" );
//  Draw_Fmsgupdate();

    return 0;
  }

  fprintf( fmsg, " BMP_Image End on Open Error.\n" );
  Draw_Fmsgupdate();

  return -1; /* Error on Open */

}



#ifdef JPEG_VGEN_USE

int Write_jpeg_image( char * name )
{ /* Routine to Create a JPEG File from the Pixel_buffer Image */
  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr        jerr;
  JSAMPROW    p[1];
  int   row_stride;

  if (fimg = fopen( name, "wb" )) {
    cinfo.err = jpeg_std_error( &jerr );
    jpeg_create_compress( &cinfo );

    jpeg_stdio_dest( &cinfo, fimg );

    cinfo.image_width  = Image_Width; /* Set image size in pixel */
    cinfo.image_height = Image_High;
    cinfo.input_components = 3;       /* Set for true Color (24 bits) image */
    cinfo.in_color_space = JCS_RGB;

    /* Setting of Particular compression parameters */
    jpeg_set_defaults( &cinfo );

    jpeg_start_compress( &cinfo, TRUE );

    row_stride = Image_Width*4;
    while (cinfo.next_scanline < cinfo.image_height) {
      p[0] = &Pixel_Buffer[cinfo.next_scanline*row_stride];
      jpeg_write_scanlines( &cinfo, p, 1 );
    }

    jpeg_finish_compress( &cinfo );

    return 0;
  }
  else return -1;
}

#endif /* JPEG_VGEN_USE */




void Write_draw_segment( Draw_Ptr p )
{ /* Display a particular Segment */
  Draw_Ptr           q;
  float *          tab;
  int    cod, dim, flg;
  int    hitcnt, hitcd, i, j;
  static int Plot_Code[] = { DRW_POLYGON,        /*  3 Fill Area */
                             DRW_MARKERS,        /*  2 Markers */
                             DRW_LINE_STRIP,     /*  1 Lines */
                             DRW_POINTS,         /*  0  for  GL Codes */
                             DRW_LINES,          /* -1 */
                             DRW_LINE_STRIP,     /* -2, Use by draw_out 1 */
                             DRW_LINE_LOOP,      /* -3 */
                             DRW_TRIANGLES,      /* -4 */
                             DRW_TRIANGLE_STRIP, /* -5 */
                             DRW_TRIANGLE_FAN,   /* -6 */
                             DRW_QUADS,          /* -7 */
                             DRW_QUAD_STRIP,     /* -8 */
                             DRW_POLYGON         /* -9, Use by draw_out 3 */
                           };

  for (i = 0; i < 4; i++)
    for (j = 0; j < 4; j++)
      wrFlt32( p->seg.seg_mat[i,j] );


  q = p->seg.seg_fdir;
  while (q)
  { /* Loop on all Segment Directive */
    switch (q->seg.dcom.dir_knd) {
      case Draw_Segment:
      break;

      case Draw_Exec_Seg:
//      if (q->exeseg.exec_seg)
//        DrwGL_Display_Seg( q->exeseg.exec_seg, mode );
      break;

      case Draw_PP_Attr:
//      DrwGL_PP_Attrib( q->ppattr.ppa_code );
      break;

      case Draw_Ena_Plane:
//      DrwGL_Push_Plane( q );
      break;

      case Draw_Dis_Plane:
//      DrwGL_Pop_Plane();
      break;

      case Draw_PushTrfMat:
//      glPushMatrix();
//      glMultMatrixf( p->tmat.trsf_mat );
      break;

      case Draw_PopTrfMat:
//      glPopMatrix();
      break;

      case Draw_Box_Ref:
        DrwGL_Push_Bxrf( q );
      break;

      case Draw_Box_Pop:
        DrwGL_Pop_Bxrf();
      break;

      case Draw_Plot:
        tab = q->tab.plo_tab;
        dim = q->tab.plo_npt;
        flg = q->tab.plo_flg;
        cod = q->tab.plo_knd;
        if ((cod <= 3)&&(cod >= -9)) {
          cod = ((Plot_Code[3 - cod])<<3)|(flg&7);
          putc( 20, fimg ); putc( cod, fimg );
          wrInt16( dim );
          if (flg&cdf_3D) /* 3D Plot */
            for (i = 0; i < dim; i++) {
              wrFlt32( *(tab++) ); wrFlt32( *(tab++) ); wrFlt32( *(tab++) );
            }
          else /* 2D Plot */
            for (i = 0; i < dim; i++) {
              wrFlt32( *(tab++) ); wrFlt32( *(tab++) );
            }
        }
      break;

      case Draw_Text:
        putc( 21, fimg );
        wrFlt32( q->txt.txt_x ); wrFlt32( q->txt.txt_y );
        wrFlt32( q->txt.txt_high ); wrFlt32( q->txt.txt_ang );
        putc( q->txt.txt_len, fimg );
        for ( i = 0; i < q->txt.txt_len; i++ ) putc( q->txt.txt_str[i], fimg );
      break;

      case Draw_Circle:
//      putc( 21, fimg );
      break;

      case Draw_Sphere:
//      putc( 21, fimg );
      break;

      case Draw_Line_Attr:
        putc( 11, fimg );
        wrInt16( q->linattr.plia_kind );
        wrFlt32( q->linattr.plia_size );
      break;

      case Draw_Marker_Attr:
        putc( 12, fimg );
        wrInt16( q->mrkattr.mrka_kind );
        wrFlt32( q->mrkattr.mrka_size );
      break;

      case Draw_Fill_Attr:
        putc( 13, fimg );
        wrInt16( q->filattr.fila_kind );
        wrInt16( q->filattr.fila_styl );
      break;

      case Draw_Text_Attr:
        putc( 14, fimg );
        putc( q->txtattr.txta_haln, fimg );
        putc( q->txtattr.txta_valn, fimg );
        putc( q->txtattr.txta_path, fimg );
        wrFlt32( q->txtattr.txta_expf );
        wrFlt32( q->txtattr.txta_spcf );
      break;

      case Draw_Font:
        putc( 15, fimg );
        wrInt16( q->font.fnt_font ); wrInt16( q->font.fnt_prec );
      break;

      case Draw_Color:
        putc( 10, fimg );
        if (q->color.col_rgba[0] < 0.0) {
          i = (int) ( - q->color.col_rgba[0] + 0.5);
          putc( (int) (draw_srccoltb[i][0] + 0.5), fimg );
          putc( (int) (draw_srccoltb[i][1] + 0.5), fimg );
          putc( (int) (draw_srccoltb[i][2] + 0.5), fimg );
        }
        else
          for (j=0;j<4;j++)
            putc( (int) (q->color.col_rgba[j] + 0.5), fimg );
      break;

      default:
      break;
    }
    q = q->seg.dcom.dir_nxt;
  }
}




int Write_draw_image( char * name )
{ /* Routine to Create a Draw Binary File from the Draw Current Structure */
  Draw_Ptr p;
  int i, j;
  

  if (fimg = fopen( name, "wb" )) {

    putc( 1, fimg );          /* Put the View Space Informations */
    wrFlt32( dw_cxmin ); wrFlt32( dw_cxmax );
    wrFlt32( dw_cymin ); wrFlt32( dw_cymax );
    wrFlt32( draw_uzmin ); wrFlt32( draw_uzmax );
    for (i = 0; i < 4; i++)
      for (j = 0; j < 4; j++)
        wrFlt32( Drwgl_proj_wrd[i,j] );


    Drwgl_currbrf = NULL;

    p = draw_fsysseg;
    while (p) {
      Write_draw_segment( p );
      p = p->seg.seg_nxt;
    }

    putc( 2, fimg ); putc( 3, fimg );  /* Put the General 3D Clip Data */
    wrFlt32( draw_uxmin ); wrFlt32( draw_uxmax );
    wrFlt32( draw_uymin ); wrFlt32( draw_uymax );
    wrFlt32( draw_uzmin ); wrFlt32( draw_uzmax );


    p = draw_fsegm;
    while (p) {
      Write_draw_segment( p );
      p = p->seg.seg_nxt;
    }

    putc( 0, fimg );          /* Put the End of File mark */
    fclose( fimg );

    return 0;
  }
  else return -1;
}




int Write_ps_image( char * name, GLint kind, int prtflag )
{ /* Routine to Create a PS/EPS or PDF File from a GL FEEDBACK Buffer  */
  GLint state    = 0,
        viewport[4];
  GLfloat drap,
          prap;

# define STP_FDBSIZE (1024*1024)
# define INI_FDBSIZE STP_FDBSIZE

  GLint opt      = GL2PS_OCCLUSION_CULL |
                   GL2PS_BEST_ROOT |
                   GL2PS_DRAW_BACKGROUND,
        nbcol    = 0,         /* Triangle subdivision (default value) */
        status   = GL2PS_OVERFLOW,
        fdbsize  = INI_FDBSIZE - STP_FDBSIZE;

  fprintf( fmsg, " Prepare A PS/PDF File.\n" );
  Draw_Fmsgupdate();


  if (fimg = fopen( name, "wb" )) {

    if (ds_ry*ps_rx > ps_ry*ds_rx) { /* Hs/Ws > Hp/Wp */
      pw_cwsy = 0.5*ps_ry/xy_scale;
      pw_cwsx = pw_cwsy*ps_rx/ps_ry;
    } else {
      pw_cwsx = 0.5*ps_rx/xy_scale;
      pw_cwsy = pw_cwsx*ps_ry/ps_rx;
    }
//  pw_cwsx = 0.5*ps_rx/xy_scale;
//  pw_cwsy = 0.5*ps_ry/xy_scale;
//  pw_cwsx = 0.5*p_width*xy_scale;
//  pw_cwsy = 0.5*p_height*xy_scale;
    pw_cwpx = dw_cwpx; pw_cwpy = dw_cwpy;

//  pw_cwsx = dw_cwsx*ps_rx/ds_rx;
//  pw_cwsy = dw_cwsy*ps_ry/ds_ry;
    pw_cxmin = pw_cwpx - pw_cwsx; pw_cxmax = pw_cwpx + pw_cwsx;
    pw_cymin = pw_cwpy - pw_cwsy; pw_cymax = pw_cwpy + pw_cwsy;


    viewport[0] = (GLint) pp_rx;
    viewport[1] = (GLint) pp_ry;
    viewport[2] = (GLint) (pt_rx*1.25);
    viewport[3] = (GLint) (pt_ry*1.25);
/*
    viewport[0] = (GLint) 0;
    viewport[1] = (GLint) 0;
    viewport[2] = (GLint) pr_rx;
    viewport[3] = (GLint) pr_ry;
*/
    fprintf( fmsg, " GL2PS : xy_scale = %f, w = %f, h = %f.\n",
                   xy_scale, p_width, p_height );
    fprintf( fmsg, " GL2PS : ps_r = (%f,%f), ds_r = (%f,%f).\n",
                   ps_rx, ps_ry, ds_rx, ds_ry );
    fprintf( fmsg, " GL2PS : pw_cws = (%f,%f), pw_cwp = (%f,%f).\n",
                   pw_cwsx, pw_cwsy, pw_cwpx, pw_cwpy );
    fprintf( fmsg, " GL2PS : dw_cws = (%f,%f), dw_cwp = (%f,%f).\n",
                   dw_cwsx, dw_cwsy, dw_cwpx, dw_cwpy );
    fprintf( fmsg, " GL2PS : pw_cmin = (%f,%f), pw_cmax = (%f,%f).\n",
                   pw_cxmin, pw_cymin, pw_cxmax, pw_cymax );
    fprintf( fmsg, " GL2PS : dw_cmin = (%f,%f), dw_cmax = (%f,%f).\n",
                   dw_cxmin, dw_cymin, dw_cxmax, dw_cymax );
    fprintf( fmsg, " GL2PS : dt =(%d,%d), pt = (%d,%d).\n", dt_rx, dt_ry, pt_rx, pt_ry );
    fprintf( fmsg, " GL2PS : ViewPort = (%d,%d,%d,%d).\n",
                   viewport[0], viewport[1], viewport[2], viewport[3] );
    Draw_Fmsgupdate();

//  glGetIntegerv( GL_VIEWPORT, viewport );  

    fprintf( fmsg, " The file \"%s\" is created.\n", name );
    Draw_Fmsgupdate();

    if (dt_ry <= dt_rx) opt |= GL2PS_LANDSCAPE; 

    while (status == GL2PS_OVERFLOW) {
      fdbsize += STP_FDBSIZE;
      status = gl2psBeginPage( name, "VGL",
                               viewport,
                               kind,
                               GL2PS_BSP_SORT,
                               opt,
                               GL_RGBA, 0, NULL,
                               nbcol, nbcol, nbcol,
                               fdbsize,
                               fimg,
                               name );
 
     if (status != GL2PS_SUCCESS) {
        fprintf( fmsg, " gl2psBeginPage Error %d.\n", status );
        Draw_Fmsgupdate();
        return -1;
      }
      else
      {
        fprintf( fmsg, " The Resulted status of gl2psBeginPage is created %d.\n", status );
        Draw_Fmsgupdate();
      }
      DrwGL_Display( 1 );

      status = gl2psEndPage();
      fclose( fimg );
    }
    fprintf( fmsg, " GL2PS : We Have used %d K GLfloat with status %d.\n",
                   fdbsize/1024, status );
    Draw_Fmsgupdate();
    return (status != GL2PS_SUCCESS);
  }
  else return -1;

}








void VGL_Draw_ServerOGLCanvasPane::Draw_Save_Image( int px, int py, int sx, int sy )
{ /* Save the window (px, py, sx, sy) to an image file */
  static char * filter[] =    //
    { "*.bmp",     // 0 - Bit map File(s).
      "*.jpg",     // 1 - JPEG image file(s).
      "*.draw",    // 2 - Binary Draw Image File(s).
      "*.ps",      // 3 - Simple Postscript.
      "*.eps",     // 4 - Encapsulated Postscript.
      "*.pdf",     // 5 - PDF File.
      NULL
    };
  static char name[256] = "Draw_Image";
  static char * ftype;
  static int    fltidx = 0;
  static int i, len, lent, rp;

  vNoticeDialog fsnote( theApp );
  vFileSelect fsel( theApp );

  rp = fsel.FileSelectSave( "Image filename to create", name, 255, filter, fltidx );

  if (rp && *name) { /* A file name was obtained. */

    len  = strlen( name );
    lent = strlen( filter[fltidx] ) - 1;
    
    if (strcmp( name + len - lent, filter[fltidx] + 1 ) != 0) {
      /* Append an file type when it was not specified */
      strcat( name, filter[fltidx] + 1 );
      len = strlen( name );
    }

    fprintf( fmsg, " Image file name = %s\n", name );
    fprintf( fmsg, " Image Window Parameters = [%d,%d,%d,%d].\n", px, py, sx, sy );
    Draw_Fmsgupdate();


    /* Get the GL Screen Image in the Pixel Buffer */
    if (fltidx < 3) DrwGL_Get_Image( Pixel_Buffer, px, py, sx, sy );

    switch (fltidx) {
      case 0: /* BMP (24 bits/Pixel) Format */
        rp = Write_bmp_image( name, 0 ) < 0;
      break;

#ifdef JPEG_VGEN_USE
      case 1: /* JPG Format */

        rp = Write_jpeg_image( name ) < 0;

      break;
#endif

      case 2: /* Draw-Primitive Format */

        rp = Write_draw_image( name ) < 0;

      break;

      case 3: /* Draw-Primitive Format */
        rp = Write_ps_image( name, GL2PS_PS, 0 );
      break;

      case 4: /* Draw-Primitive Format */
        rp = Write_ps_image( name, GL2PS_EPS, 0 );
      break;

      case 5: /* Draw-Primitive Format */
        rp = Write_ps_image( name, GL2PS_PDF, 0 );
      break;



      default:
        rp = 0;
        fsnote.Notice( " Un-Implemented Image Format." );
    }
    if (rp) (void) fsnote.Notice( " Cannot Create the Image File." );
  }
}




#ifdef _NOPRINT

void VGL_Draw_ServerOGLCanvasPane::Draw_Print_Out( int iflg )
{ /* Send the Current Image to Line Printer */

  vPrintDC pdc;     // Create a printDCanvas object.
  vPrinter printer;

  vNoticeDialog prnote( theApp );

  Char   * p1, * p2;
  vColor * Print_Buffer, * pcolor;
  vPen   cpen;
  Char    cR, cG, cB;

  
  int npix, ndim, lx, ly, i, j;


  prnote.Notice( " Print Function Not implemented: Use Save to bmp file." );;

}



#else
#  ifdef _PRINT_OUT

void VGL_Draw_ServerOGLCanvasPane::Draw_Print_Out( int iflg )
{
  vPrintDC pdc;     // create a printDCanvas object.
  vPrinter printer;

  Char   * p1, * p2;
  vColor * Print_Buffer, * pcolor;
  vPen   cpen;
  Char    cR, cG, cB;

//vBrush pbrush( 0, 0, 0, vTransparent );
  
  int npix, ndim, lx, ly, i, j;

//  GLint viewport[4];
//  unsigned int buf_count, code, count, i, np, pr1, pr2, iseg, id;
//  GLfloat *fdb_buff, *cur_buff,
//          xx1, yy1, zz1, cR1, cG1, cB1, cA1,
//          xx2, yy2, zz2, cR2, cG2, cB2, cA2;

// static Char    * Pixel_Buffer = NULL; /* Pixel Buffer (unsigned char) */
// static GLfloat * Deep_Buffer  = NULL; /* Deep Buffer for Local Z Buffer */

  fprintf( fmsg, " Enter in Print Mode.\n" );
  Draw_Fmsgupdate();

//  PS_Draw_Init( "Draw_Out.ps" );

//  PS_Draw_Display();

  if (!printer.Setup( "Draw_Out.ps" ))  // Setup the printer.
    return;

  pdc.SetPrinter( printer );            // Change to the printer we setup.
  if (!pdc.BeginPrinting())             // print to Draw_Out.ps .
    return;

  // Have a valid Drawing Canvas.

  pdc.BeginPage();

//  lx = dt_rx;
//  ly = dt_ry;

  lx = pdc.GetPhysWidth() - 72;
  ly = pdc.GetPhysHeight() - 72;

// pdc.SetBrush( pbrush );
// pdc.DrawRectangle( 36, 36, lx, ly );

  if (lx > dt_rx) lx = dt_rx;
  if (ly > dt_ry) ly = dt_ry;

  npix = lx*ly;
  ndim = npix*3;

  fprintf( fmsg, " Print Pixel buffer of %d pixels (or %d Bytes).\n", npix, ndim );
  Draw_Fmsgupdate();

  Pixel_Buffer = (Char *) malloc( npix*3 /* For RGB */ );

  Print_Buffer = (vColor *) malloc( lx*sizeof( vColor ) );


//glReadBuffer( GL_FRONT );
  glReadPixels( 0, 0, lx, ly, GL_RGB, GL_UNSIGNED_BYTE, (void *) Pixel_Buffer );


  fprintf( fmsg, " Print Buffer values.\n" );

//  for (j = 0; j < ly/2; j++ ) {
//    p2 = p1 = Print_Buffer + j*lx;
//    for (i = 0; i < lx/2; i++) {
//      fprintf( fmsg, " (%d,%d,%d/", p2->r(), p2->g(), p2->b() );
//      p2->Set( 0, 0, 255 );
//      fprintf( fmsg, " /%d,%d,%d)", p2->r(), p2->g(), p2->b() );
//      p2++;
//    }
//    fprintf( fmsg, "\n" );
//  }

  Draw_Fmsgupdate();

  fprintf( fmsg, " Display Buffer values %d * %d.\n", lx, ly );
  Draw_Fmsgupdate();

  p1 = Pixel_Buffer; /* p1 -> First line of the image */
  j = 0;
  do {
    p2 = p1;
    pcolor = Print_Buffer;
    for (i = 0; i < lx; i++) {
      cR = *(p1++); cG =  *(p1++); cB = *(p1++);
      (pcolor++)->Set( cR, cG, cB );
      fprintf( fmsg, " (%d,%d,%d)/n", Print_Buffer[i].r(),
                                      Print_Buffer[i].g(),
                                      Print_Buffer[i].b() );
//    cpen.SetColor( *(pcolor++) );
//    pdc.DrawPoint( i, j );
    }

    pdc.DrawColorPoints( 36, j, lx, Print_Buffer );
//    p1 -= lx;
  } while (++j < ly);

//  Draw_Fmsgupdate();

//DrwGL_Display( 1 );                   // Perform the GL picture on the Printer.



  pdc.EndPage();

  pdc.EndPrinting();

//  PS_Draw_End();

  fprintf( fmsg, " Go Out of Print Mode.\n" );
  Draw_Fmsgupdate();

}


#else

void VGL_Draw_ServerOGLCanvasPane::Draw_Print_Out( int iflg )
{ /* Send the Current Image to Line Printer */

  static char name[128] = "Draw_Print_Out.bmp";
  int rp;

  vNoticeDialog prnote( theApp );

  /* Get the GL Screen Image in the Pixel Buffer */
  DrwGL_Get_Image( Pixel_Buffer, 0, 0, dt_rx, dt_ry );

  rp = Write_bmp_image( name, 1 ) < 0;

  if (rp) (void) prnote.Notice( " Cannot Create the Image File to Print." );

}


#  endif
#endif

