/********* DRAW Interface ROUTINES for OpenGL Graphic Libary ***********/
/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               D R A W   -   S E R V E R   M O D U L E                 *
*                                                                       *
*           (Draw interface for V function and procedures)              *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/



/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/



/* include the environment SERVER */

#define _VGL_DEF_GLOBAL 0

#include "Draw_VGL.h"

#include "VGLdrw_cmdw.h"
#include "VGLdrw_gl2ps.h"



/***********************************************************************/
/*                                                                     */
/*                    DRAW/GL Interface Routines                       */
/*                                                                     */
/***********************************************************************/



#define Hits_Bufsz             128
#define Hits_Preci              10

# define inrd (3.1415927/180.0)



/***********************************************************************/
/*                         Local  Variable                             */
/***********************************************************************/

 
typedef struct {  int   seg;
                  int   pid;
                  float prio;
                  float xmin, xmax,
                        ymin, ymax,
                        zmin, zmax;
                 } Hitst;


static Hitst     HitsHeap[Hits_Bufsz];
static int       HitsSp = -1;

static GLdouble  smix, smax, smiy, smay, smiz, smaz, /* Selection min max */
                 lmix, lmax, lmiy, lmay, lmiz, lmaz, /* Local Limits */    
                 hmix, hmax, hmiy, hmay, hmiz, hmaz; /* Echo Limits */

static GLdouble  sel_min, sel_max, tmix, tmax, tmiy, tmay,
                 Hitx, Hity, Hitz;

static int       Hits = 0;               /* Flag for Selection Hits */

static int       Tmat_Cnt = 0;           /* Count of Push Trsf. Matrix */

static GLfloat * Local_Mat_Ptr = NULL;   /* Local Matrix Pointer */

static GLfloat   svd_mat[16];            /* Saved Matrix for view input interaction */

static GLfloat   MinZ_Dist, MaxZ_Dist,   /* Minimum and Maximum Screen depth */
                 Zdepth, ZOrg;           /* Total Z Screen Depth size, Z Org */

static GLUquadricObj *QuaObj;            /* Common Quadric Structure Pointer */

/* GL Name of each Light */
static int    lightst[LIGHT_NUMBER];     /* Table of Light state */

static int    lighting   = 0;
static int    blending   = 0;

static int    gl2ps_flg  = 0;            /* Flag for GL2PS on/off (1/0) mode  */ 

static int    sup_pickid = 0;            /* shift of Pickid for exec seg */

static int    rst_view;                  /* Flag to restore initial view at end of View Orientation Request */


/***********************************************************************/
/*                        Call Back  Routine                           */
/***********************************************************************/

//void Drwgl_Error_CallBack( GLenum errcode )
//{
//  const GLubyte *errmsg;

//  errmsg = gluErrorString( errcode );
//  fprintf( fmsg, " GL/GLU Error \"%s\"\n", errmsg );
//  Draw_Fmsgupdate();
//}
 


/***********************************************************************/
/*                          Init  Routine                              */
/***********************************************************************/


void DrwGL_Init()
{ /* Initialize the Draw/GL Interface */
  int i, j, k, l;
  GLenum lig;

  /* Set Saved Color Tables */
  for (i=0;i<16;i++)
    for (j=0;j<4;j++)
    { draw_savsrcctb[i][j] = draw_srccoltb[i][j];
      draw_savlprctb[i][j] = draw_lprcoltb[i][j];
    }

  Drwgl_print        =       0;  /* Set The Screen output mode */
  Drwgl_flags        =       0;
  Drwgl_mactive      =       0;  /* Set mouse as inactive */

  Drwgl_astpmod      =       0;  /* Default to no Animation Stop Validation */
  Drwgl_astpstate    =       0;  /* Init the Stop Flag */

  Drwgl_unitx        =     " ";  /* Set the default unit cm */
  Drwgl_unity        =     " ";

  Drwgl_xusr_unt[0]  =       0;  /* Set the user units to undefined */
  Drwgl_yusr_unt[0]  =       0;

  Drwgl_inpmdflg     =       0;
  Drwgl_xlogflg      =       0;  /* Set Linear Coordinates */
  Drwgl_ylogflg      =       0;

  Drwgl_clipflg      =       0;  /* Set as No Pick Echo On. */
  Drwgl_segment      =       0;
  Drwgl_ident        =       0;
  Drwgl_pecho        =       0;

  Drwgl_mrk_kind     =      -1;  /* Set the default to POINTS for Marker */
  Drwgl_line_stipple =       0;  /* Init Line Stipple Flag */
  Drwgl_fill_stipple =       0;  /* Init Fill Area Stipple Flag */
  Drwgl_fill_front   =       0;  /* Init Flag for front face fill mode */
  Drwgl_fill_back    =       0;  /* Init Flag for back face fill mode */
  Drwgl_fill_Shift   =       0;  /* Init Flag for polygon depth shift */

  Drwgl_txt_haln     =       1;  /* Text Horizontal Alignment mode set to Left */
  Drwgl_txt_valn     =       1;  /* Text Vertical Alignment mode set to Base */
  Drwgl_txt_path     =       0;  /* Text Path set to Horizontal Left to Right */
  Drwgl_txt_expf     =     1.0;  /* Text Expanssion Factor set to 1.0 */
  Drwgl_txt_spcf     =     0.0;  /* Text Spacing set to Normal */

  Drwgl_txt_font     =    NULL;  /* Txet Font is undefined */
  Drwgl_txt_prec     =       0;

  Draw_Dist          =     0.0;  /* Set Without Perspective */

  Drwgl_ViewMat      =    NULL;  /* Initialize the view Matrix pointer */
  Drwgl_zomeview     =       0;  /* Set implicite user action as Zoom */

  glClearColor( draw_srccoltb[0][0], draw_srccoltb[0][1],
                draw_srccoltb[0][2], draw_srccoltb[0][3] );
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
  glEnable( GL_DEPTH_TEST);

  for (i = 0; i < LIGHT_NUMBER; i++)
    lightst[i] = Drwgl_light_entb[i] = 0;

  if (Drwgl_usr_request&SERVER_DO_LIGHTING) {
    lig = GL_LIGHT0;
    l = 0;
    glEnable( GL_LIGHTING );
    j = SERVER_ENA_LIGHT0;
    for (i = 0; i < LIGHT_NUMBER; i++) {
      Drwgl_light_entb[i] = (k = (Drwgl_usr_request&j)?1:0);
      if (k) { glEnable( lig ); l = 1;
//      fprintf( fmsg, " Set Light #%d\n", i );
//      Draw_Fmsgupdate();
      }
      j *= 2; lig++;
    }
    /* Always One Light is enabled ( the light 0 ) */
    if (!l) { glEnable( GL_LIGHT0 ); Drwgl_light_entb[0] = 1;
//    fprintf( fmsg, " Set Default Light #0\n" );
//    Draw_Fmsgupdate();
    }
    lighting   = 1;
  }

  glMatrixMode( GL_PROJECTION );
  glLoadIdentity();
  glViewport( 0, 0, s_dt_rx, s_dt_ry );
  glGetIntegerv( GL_VIEWPORT, Drwgl_wrdvport );
  glOrtho( (GLdouble) dw_cxmin, (GLdouble) dw_cxmax,
           (GLdouble) dw_cymin, (GLdouble) dw_cymax, 1.0, -1.0 );
  glGetDoublev( GL_PROJECTION_MATRIX, Drwgl_proj_wrd );


  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_mat_unit );
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_inpmat );

//gluQuadricCallback( QuaObj, GLU_ERROR, Drwgl_Error_CallBack );

//fprintf( fmsg, "  Init GL Viewport Size (%d*%d) .\n", s_dt_rx, s_dt_ry );
//fprintf( fmsg, "  Init GL Window Size (%f*%f) .\n", ds_rx, ds_ry );
//Draw_Fmsgupdate();

}



void DrwGL_Alpha_Set( int flagw, GLenum src, GLenum dst )
{ /* Enable or Disable Blending */
  if (flagw&DRWGL_BLD_ENA)  glEnable( GL_BLEND );
  if (flagw&DRWGL_BLD_DDIS) glDepthMask( GL_FALSE );
  if (flagw&DRWGL_BLD_DENA) glDepthMask( GL_TRUE );
  glBlendFunc( src, dst );    
  if (flagw&DRWGL_BLD_DIS)  glDisable( GL_BLEND );

//fprintf( fmsg, " Alpha Blend On:Off = %d:%d.\n",
//               flagw&DRWGL_BLD_ENA, flagw&DRWGL_BLD_DIS );
//fprintf( fmsg, " Alpha D Mask On:Off = %d:%d.\n",
//               flagw&DRWGL_BLD_DENA, flagw&DRWGL_BLD_DDIS );
//Draw_Fmsgupdate();
}



void DrwGL_Light_OnOff( int lnb, int OnOff )
{
  GLenum lig;

  lig = GL_LIGHT0;
  if ((lnb > 0)&&(lnb < LIGHT_NUMBER)) lig += lnb;
  if (OnOff >= 0) {
    OnOff = (OnOff)?1:0;
    if (OnOff != lightst[lnb]) {
      if (OnOff) glEnable( lig );
            else glDisable( lig );
      lightst[lnb] = OnOff;
    }
  }
//fprintf( fmsg, " GL Ligth #%d set state at %d\n", lnb, OnOff ), 
//Draw_Fmsgupdate();
}



void DrwGL_Set_Clear_Color()
{
  glClearColor( draw_srccoltb[0][0], draw_srccoltb[0][1],
                draw_srccoltb[0][2], draw_srccoltb[0][3] );
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
//fprintf( fmsg, "  Set Clear Color (%f,%f,%f,%f)\n",
//              draw_srccoltb[0][0], draw_srccoltb[0][1],
//              draw_srccoltb[0][2], draw_srccoltb[0][3] );
//Draw_Fmsgupdate();
}




/***********************************************************************/
/*                      Basic   Pick   Routines                        */
/***********************************************************************/


static void Drw_Local_Trans( GLfloat * m, float *x, float *y, float *z )
{ float wx, wy;

  wx = m[ 0]*(*x) + m[ 4]*(*y) + m[ 8]*(*z) + m[12];
  wy = m[ 1]*(*x) + m[ 5]*(*y) + m[ 9]*(*z) + m[13];
  *z = m[ 2]*(*x) + m[ 6]*(*y) + m[10]*(*z) + m[14];
  *x = wx; *y = wy;
}



static void DrwHVertex( float x, float y, float z )
{ GLdouble wx, wy, wz;
  int ret;

//fprintf( fmsg, " V3F point = (%5.1f,%5.1f,%5.1f)\n", x, y, z );

  /* Perform all Geometrical Transformations */

  if (gluProject( (GLdouble) x, (GLdouble) y, (GLdouble) z,
                   Drwgl_modlview, Drwgl_proj_wrd, Drwgl_wrdvport,
                   &wx, &wy, &wz ) == GL_TRUE) {
      /* Set for Echo Area */
      if (hmix > wx) hmix = wx; if (hmax < wx) hmax = wx;
      if (hmiy > wy) hmiy = wy; if (hmay < wy) hmay = wy;
      if (hmiz > wz) hmiz = wz; if (hmaz < wz) hmaz = wz;
      /* Set for Directive Hit */
      if ((wx >= smix)&&(wx <= smax)&&(wy >= smiy)&&(wy <= smay)) {
        Hits = 1;
//      Hitx = wx; Hity = wy; Hitz = wz;
//      fprintf( fmsg, " V3F Hits at (%4.0f,%4.0f,%4.0f).\n", wx, wy, wz );
      }
    }
//Draw_Fmsgupdate();
}





/***********************************************************************/
/*                       Basic  Plot  Routines                         */
/***********************************************************************/


void DrwGL_ColorRGB( Draw_RGBA rgba )
{ /* Set a Color by predefined index */
  if (lighting) glMaterialfv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, rgba );
           else glColor4fv( rgba );
//fprintf( fmsg, " GL RGBA Color [%f,%f,%f,%f]\n", rgba[0], rgba[1], rgba[2], rgba[3] );
//Draw_Fmsgupdate();
}




void DrwGL_Color( int idc )
{ /* Set a Color by predefined index */
//fprintf( fmsg, " GL Try to set Color # %d.\n", idc );
  if ((idc >= 0)&&(idc <= 15))
    if (Drwgl_print)
    {
//    glColor3f( draw_lprcoltb[idc][0], draw_lprcoltb[idc][1], draw_lprcoltb[idc][2] );
//    fprintf( fmsg, " GL Print Color [%f,%f,%f]\n",
//             draw_lprcoltb[idc][0], draw_lprcoltb[idc][1], draw_lprcoltb[idc][2] );
    }
    else
    {
      if (lighting)
        glMaterialfv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, draw_srccoltb[idc] );
      else
        glColor4fv( draw_srccoltb[idc] );
    }
//Draw_Fmsgupdate();
}




void DrwGL_Save_Chcntx( float sz )
{
  Drwgl_stipple_state = glIsEnabled( GL_LINE_STIPPLE );
  glGetFloatv( GL_LINE_WIDTH, &Drwgl_line_size );
  glDisable( GL_LINE_STIPPLE );
  glLineWidth( (GLfloat) sz );
  if (gl2ps_flg) {
    gl2psDisable( GL2PS_LINE_STIPPLE );
    gl2psLineWidth( (GLfloat) sz );
  }
}




void DrwGL_Restore_Chcntx()
{
  if (Drwgl_stipple_state) glEnable( GL_LINE_STIPPLE );
  glLineWidth( (GLfloat) Drwgl_line_size );
  if (gl2ps_flg) {
    gl2psEnable( GL2PS_LINE_STIPPLE );
    gl2psLineWidth( (GLfloat) Drwgl_line_size );
  }
}




void Show_Vwp(  char * nam, int * vw )
{
  fprintf( fmsg, "\n View Port %s = [%d,%d,%d,%d]\n",
                 nam, vw[0], vw[1], vw[2], vw[3] );
  fprintf( fmsg, "\n" );
  Draw_Fmsgupdate();
}




void Show_clplan(  char * nam, GLdouble * pl )
{
  fprintf( fmsg, "\n Clip Plane %s = [ %f, %f, %f, %f ]\n",
                 nam, pl[0], pl[1], pl[2], pl[3] );
  fprintf( fmsg, "\n" );
  Draw_Fmsgupdate();
}



void Show_Mat( char * nam, GLdouble * m )
{
  fprintf( fmsg, "\n Matrix %s\n", nam );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 0], m[ 4], m[ 8], m[12] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 1], m[ 5], m[ 9], m[13] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 2], m[ 6], m[10], m[14] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 3], m[ 7], m[11], m[15] );
  fprintf( fmsg, "\n" );
  Draw_Fmsgupdate();
}





void Show_Matf( char * nam, GLfloat * m )
{
  fprintf( fmsg, "\n Matrix %s\n", nam );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 0], m[ 4], m[ 8], m[12] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 1], m[ 5], m[ 9], m[13] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 2], m[ 6], m[10], m[14] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 3], m[ 7], m[11], m[15] );
  fprintf( fmsg, "\n" );
  Draw_Fmsgupdate();
}



void DrwGL_Line_Attr( int linetype, float sz )
{ /* Set Line Attribute */
  static GLushort pattern[] = { 0x3F3F,   /* Dashed */
                                0xAAAA,   /* Dotted */
                                0x5F5F,   /* Dashed_Dotted */
                                0x54A9,   /* Triple_Dot */
                                0x5050,   /* Double_Dot */
                                0x1010,   /* Spaced_Dot */
                                0x007F,   /* Spaced_Dash */
                                0x1C3F,   /* Long_Short */
                                0x3FFC,   /* Long_Dash */
                                0x55FF,   /* Dash_3_Dot */
                                0x57FF    /* Dash_2_Dot */
                              };
  static GLint factor[]     = { 2,   /* Dashed */
                                2,   /* Dotted */
                                2,   /* Dashed_Dotted */
                                2,   /* Triple_Dot */
                                2,   /* Double_Dot */
                                2,   /* Spaced_Dot */
                                1,   /* Spaced_Dash */
                                1,   /* Long_Short */
                                1,   /* Long_Dash */
                                2,   /* Dash_3_Dot */
                                2    /* Dash_2_Dot */
                              };
 GLint fact, patt;

  glLineWidth( (GLfloat) sz );
  if (gl2ps_flg) gl2psLineWidth( (GLfloat) sz );
  if ((linetype>1)&&(linetype<13)) {
    glLineStipple( factor[linetype-2], pattern[linetype-2] );
    Drwgl_line_stipple = 1;
  }
  else Drwgl_line_stipple = 0;
  Drwgl_AStack[Drwgl_AStkSp].lin_kind = linetype;
  Drwgl_AStack[Drwgl_AStkSp].lin_size = sz;
}




void DrwGL_Display_One_Marker( GLfloat x, GLfloat y, GLfloat z, int sel )
{ /* Display a particular Segment */
  int  i, bfill, nscan, msz;
  GLfloat sca, xx, yy;

  static GLbyte mrkt[] = {  20,   2, -20,   0,  20,   0, /*   0 -> Cross + */
                            20,   2,   0, -20,   0,  20, /*   6 */
                             1,   0,                     /*  12 */

                            20,   2, -20,   0,  20,   0, /*  14 -> Asterisk */
                            20,   2,   0, -20,   0,  20, /*  20 */
                            20,   2, -14, -14,  14,  14, /*  26 -> Diagonal Cross */
                            20,   2, -14,  14,  14, -14, /*  32 */
                             1,   0,                     /*  38 */

                           100, -37,-100,   0, -98, -17, /*  40 -> Small Circle */
                           -94, -34, -87, -50, -77, -64, /*  46 */
                           -64, -77, -50, -87, -34, -94, /*  52 */
                           -17, -98,   0,-100,  17, -98, /*  58 */
                            34, -94,  50, -87,  64, -77, /*  64 */
                            77, -64,  87, -50,  94, -34, /*  70 */
                            98, -17, 100,   0,  98,  17, /*  76 */
                            94,  34,  87,  50,  77,  64, /*  82 */
                            64,  77,  50,  87,  34,  94, /*  88 */
                            17,  98,   0, 100, -17,  98, /*  94 */
                           -34,  94, -50,  87, -64,  77, /* 100 */
                           -77,  64, -87,  50, -94,  34, /* 106 */
                           -98,  17,   1,   0,           /* 112 */

                            20,  -4, -20,   0,   0, -20, /* 116 -> Diamond */
                            20,   0,   0,  20,   1,   0, /* 122 */

                            20,  -3,   0,   0, -20,  20, /* 128 -> Hour Glass */
                            20,  20,  20,  -3,   0,   0, /* 134 */
                           -20, -20,  20, -20,   1,   0, /* 140 */

                            20,  -3,   0,   0, -20,  20, /* 146 -> Bowtie */
                           -20, -20,  20,  -3,   0,   0, /* 152 */
                            20, -20,  20,  20,   1,   0, /* 158 */

                           100,  -4, -71, -71,  71, -71, /* 164 -> Square */
                            71,  71, -71,  71,   1,   0, /* 170 */

                           100,  -3,   0,-100,  87,  50, /* 176 -> Triangle Down */
                           -87,  50,   1,   0,           /* 182 */

                           100,  -3,   0, 100,  87, -50, /* 186 -> Triangle Up */
                           -87, -50,   1,   0            /* 192 */

                         };

  static int mrkpt[] =   {   0, /* Code      2, Offset    0, -00- Cross + */
                            14, /* Code      3, Offset   14, -01- Asterisk */
                            40, /* Code      4, Offset   40, -02- Small Circle */
                            26, /* Code      5, Offset   26, -03- Diagonal Cross */
                          -116, /* Code -13/ 6, Offset -116, -04- Solid Diamond */
                           116, /* Code -12/ 7, Offset  116, -05- Diamond */
                          -128, /* Code -11/ 8, Offset -128, -06- Solid Hour Glass */
                           128, /* Code -10/ 9, Offset  128, -07- Hour Glass */
                          -146, /* Code - 9/10, Offset -146, -08- Solid Bowtie */
                           146, /* Code - 8/11, Offset  146, -09- Bowtie */
                          -164, /* Code - 7/12, Offset -164, -10- Solid Square */
                           164, /* Code - 6/13, Offset  164, -11- Square */
                          -176, /* Code - 5/14, Offset -176, -12- Solid Triangle Down */
                           176, /* Code - 4/15, Offset  176, -13- Triangle Down */
                          -186, /* Code - 3/16, Offset -186, -14- Solid Triangle Up */
                           186, /* Code - 2/17, Offset  186, -15- Triangle Up */
                           -40  /* Code - 1/18, Offset - 40, -16- Solid Circle */
                         };


  nscan = mrkpt[Drwgl_mrk_kind];
  if (nscan < 0) { nscan = - nscan; bfill = 1; }
            else bfill = 0;

  if (sel) {
    while (1) {
      sca = 0.1/(GLfloat) mrkt[nscan++];
      msz =        abs( mrkt[nscan++] );
    if (!msz) break;
      for (i = 0; i < msz; i++ ) {
        xx = Drwgl_mrk_size*((GLfloat) mrkt[nscan++])*sca + x;
        yy = Drwgl_mrk_size*((GLfloat) mrkt[nscan++])*sca + y;
        DrwHVertex( xx, yy, z );
      }
    }
  }
  else {
    while (1) {
      sca = 0.1/(GLfloat) mrkt[nscan++];
      msz =               mrkt[nscan++];
      if (bfill) glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );
            else glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );

    if (!msz) break;

      if (bfill) {
        glBegin( GL_POLYGON );
        msz = abs( msz );
      }
      else  if (msz > 0) glBegin( GL_LINE_STRIP );
                    else { msz = - msz; glBegin( GL_LINE_LOOP ); }

      for (i = 0; i < msz; i++ ) {
        xx = Drwgl_mrk_size*((GLfloat) mrkt[nscan++])*sca + x;
        yy = Drwgl_mrk_size*((GLfloat) mrkt[nscan++])*sca + y;
        glVertex3f( xx, yy, z );
      }
      glEnd();
    }
  }
}




void DrwGL_Display_Marker( int dim, int flag, float * tab, int sel )
{ /* Display a particular Segment */
  GLfloat x, y, z;
  int    i, nscan;

  if (sel) {
    hmix = hmiy = hmiz = 1e+30;
    hmax = hmay = hmaz = - hmix;
  }

  if (flag&cdf_3D)
  { /* 3D plot */
    for (i = 0; i < dim; i++)
    { x = *(tab++); y = *(tab++); z = *(tab++);
        if (Drwgl_currbrf) { /* 3D Graph Mode */
        }
        DrwGL_Display_One_Marker( (GLfloat) x, (GLfloat) y, (GLfloat) z, sel );
    }
  }
  else
  { /* 2D plot */
    for (i = 0; i < dim; i++)
    { x = *(tab++); y = *(tab++);
        if (Drwgl_currbrf) { /* 2D Graph Mode */
        }
        DrwGL_Display_One_Marker( (GLfloat) x, (GLfloat) y, (GLfloat) 0.0, sel );
    }
  }
}




void DrwGL_Marker_Attr( int typ, float sz )
{ /* Set Marker Attribute */
  if ((typ < 0)&&(typ >=-13)) typ = typ + 19;

  /* Set the Constant table index for the Marker */
  if ((typ <= 1)||(typ > 18 )) Drwgl_mrk_kind = -1;
                          else Drwgl_mrk_kind = typ - 2;
  Drwgl_mrk_size = sz;

  if (Drwgl_mrk_kind == -1) {
    glPointSize( (GLfloat) sz ); /* GL_POINTS Mode */
    if (gl2ps_flg) gl2psPointSize( (GLfloat) sz );
  }
  Drwgl_AStack[Drwgl_AStkSp].mrk_kind = typ;
  Drwgl_AStack[Drwgl_AStkSp].mrk_size = sz;

}



void DrwGL_Fill_Attr( int kind, int style )
{ /* Set Fill Area Attribute */

static GLint hatch_1[] = { 0x00000000, 0x00000000, 0x00000000, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x00000000, 0x00000000, 0x00000000,
                           0x00000000, 0x00000000, 0x00000000, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x00000000, 0x00000000, 0x00000000,
                           0x00000000, 0x00000000, 0x00000000, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x00000000, 0x00000000, 0x00000000,
                           0x00000000, 0x00000000, 0x00000000, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x00000000, 0x00000000, 0x00000000
                         };

static GLint hatch_2[] = { 0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818
                         };

static GLint hatch_3[] = { 0xC1C1C1C1, 0xE0E0E0E0, 0x70707070, 0x38383838,
                           0x1C1C1C1C, 0x0E0E0E0E, 0x07070707, 0x83838383,
                           0xC1C1C1C1, 0xE0E0E0E0, 0x70707070, 0x38383838,
                           0x1C1C1C1C, 0x0E0E0E0E, 0x07070707, 0x83838383,
                           0xC1C1C1C1, 0xE0E0E0E0, 0x70707070, 0x38383838,
                           0x1C1C1C1C, 0x0E0E0E0E, 0x07070707, 0x83838383,
                           0xC1C1C1C1, 0xE0E0E0E0, 0x70707070, 0x38383838,
                           0x1C1C1C1C, 0x0E0E0E0E, 0x07070707, 0x83838383
                         };

static GLint hatch_4[] = { 0x38383838, 0x70707070, 0xE0E0E0E0, 0xC1C1C1C1,
                           0x83838383, 0x07070707, 0x0E0E0E0E, 0x1C1C1C1C,
                           0x38383838, 0x70707070, 0xE0E0E0E0, 0xC1C1C1C1,
                           0x83838383, 0x07070707, 0x0E0E0E0E, 0x1C1C1C1C,
                           0x38383838, 0x70707070, 0xE0E0E0E0, 0xC1C1C1C1,
                           0x83838383, 0x07070707, 0x0E0E0E0E, 0x1C1C1C1C,
                           0x38383838, 0x70707070, 0xE0E0E0E0, 0xC1C1C1C1,
                           0x83838383, 0x07070707, 0x0E0E0E0E, 0x1C1C1C1C
                         };

static GLint hatch_5[] = { 0x18181818, 0x18181818, 0x18181818, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x18181818, 0x18181818, 0x18181818
                         };

static GLint hatch_6[] = { 0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818
                         };


static GLint * hatch_tab[6] = { hatch_1, hatch_2, hatch_3,
                                hatch_4, hatch_5, hatch_6
                              };

  int ishf = 0;

//fprintf( fmsg, " Fill Attr( %d, %d )\n", kind, style );
//Draw_Fmsgupdate();

  if (style&0xFF00) { /* Depth Shift Specification */
    Drwgl_fill_Shift = style>>8;
    glPolygonOffset( (GLfloat) Drwgl_fill_Shift, 1.0 );
    style = style&0xFF;
  }

  switch (kind) {
    case 0: /* Hollow/Solid Mode */
      Drwgl_fill_front = (style&1)?1:0;
      Drwgl_fill_back  = (style&2)?1:0;
      Drwgl_fill_stipple = 0;
    break;

    case 1: /* Hollow Mode => Style Ignored */
      Drwgl_fill_front   = 0;
      Drwgl_fill_back    = 0;
      Drwgl_fill_stipple = 0;
    break;

    case 2: /* Solid Mode => Style Ignored */
      Drwgl_fill_front   = 1;
      Drwgl_fill_back    = 1;
      if (style) {
        glPolygonOffset( 1.0, 1.0 );
      }
      Drwgl_fill_stipple = 0;
    break;

    case 3: /* Pattern Mode  => Hatched Mode */
    case 4: /* Hatched Mode */
      if ((style < 1)||(style > 6)) style = 1;
      Drwgl_fill_front   = 1;
      Drwgl_fill_back    = 1;
      glPolygonStipple( (GLubyte *) hatch_tab[style - 1] );
      Drwgl_fill_stipple = 1;
    break;

    default:
      Drwgl_fill_front   = 0;
      Drwgl_fill_back    = 0;
      Drwgl_fill_stipple = 0;
  }
  Drwgl_AStack[Drwgl_AStkSp].fil_kind = kind;
  Drwgl_AStack[Drwgl_AStkSp].fil_styl = style;
}




void DrwGL_text_Attr( int haln, int valn, int path, float expf, float spcf )
{ /* To set Text Attributs */
  Drwgl_AStack[Drwgl_AStkSp].txt_haln = Drwgl_txt_haln = haln;
  Drwgl_AStack[Drwgl_AStkSp].txt_valn = Drwgl_txt_valn = valn;
  Drwgl_AStack[Drwgl_AStkSp].txt_path = Drwgl_txt_path = path;
  Drwgl_AStack[Drwgl_AStkSp].txt_expf = Drwgl_txt_expf = expf;
  Drwgl_AStack[Drwgl_AStkSp].txt_spcf = Drwgl_txt_spcf = spcf;
}



void DrwGL_Font( int font, int prec )
{ /* To select a Font,
     Find a font in a Font List and Load it when not found */
  Draw_Font_Ptr p, q;

  p = Draw_first_font;
  q = NULL;

  while (p&&(!q))
    if (p->fnt_ident == font) q = p;
                         else p = p->fnt_next;

  if (!q) q = Draw_Load_Font( font );
  
  if (q)
  { /* Set the selected Font as Current */
    Drwgl_AStack[Drwgl_AStkSp].txt_font = Drwgl_txt_font =    q;
    Drwgl_AStack[Drwgl_AStkSp].txt_prec = Drwgl_txt_prec = prec;
  }
}





/***************************************************************************************
 *                                                                                     *
 *         Extracted from IBM XGKS Software and adapted to VGL Purpose                 *
 *              to plot any character string by using xgks fonts.                      *
 *                                                                                     *
 *     *** This Copyright is concerning only the following C/C++ function :  ***       *
 *                DrwGL_Display_Text     -  To Plot a string of character.             *
 *                                                                                     *
 *                                                                                     *
 *                                                                                     *
 *                     Copyright IBM Corporation 1989                                  *
 *                                                                                     *
 *                          All Rights Reserved                                        *
 *                                                                                     *
 * Permission to use, copy, modify, and distribute this software and its               *
 * documentation for any purpose and without fee is hereby granted,                    *
 * provided that the above copyright notice appear in all copies and that              *
 * both that copyright notice and this permission notice appear in                     *
 * supporting documentation, and that the name of IBM not be                           *
 * used in advertising or publicity pertaining to distribution of the                  *
 * software without specific, written prior permission.                                *
 *                                                                                     *
 * IBM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING                *
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL            *
 * IBM BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR                 *
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,                 *
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,              *
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS                 *
 * SOFTWARE.                                                                           *
 *                                                                                     *
 *                                                                                     *
 * University of Illinois at Urbana-Champaign                                          *
 * Department of Computer Science                                                      *
 * 1304 W. Springfield Ave.                                                            *
 * Urbana, IL   61801                                                                  *
 *                                                                                     *
 * (C) Copyright 1987, 1988 by The University of Illinois Board of Trustees.           *
 * All rights reserved.                                                                *
 *                                                                                     *
 * Tool: X 11 Graphical Kernel System                                                  *
 * Author: Gregory Scott Rogers                                                        *
 * Author: Sung Hsien Ching Kelvin                                                     *
 * Author: Yu Pan                                                                      *
 *                                                                                     *
 *                                                                                     *
 *                                                                                     *
 *                                                                                     *
 * Version changed for Draw Library.                                                   *
 * Author: Pierre Wolfers,                                                             *
 *         CNRS, Laboratoire de cristallographie,                                      *
 *         BP 166X                                                                     *
 *         F 38042 GRENOBLE CEDEX 9                                                    *
 *                                                                                     *
 ***************************************************************************************/



# define TXT_PATH_RIGHT   0
# define TXT_PATH_LEFT    1
# define TXT_PATH_UP      2
# define TXT_PATH_DOWN    3

# define TXT_HALN_NORMAL  1
# define TXT_HALN_LEFT    2
# define TXT_HALN_CENTRE  3
# define TXT_HALN_RIGHT   4

# define TXT_VALN_NORMAL  1
# define TXT_VALN_TOP     2
# define TXT_VALN_CAP     3
# define TXT_VALN_HALF    4
# define TXT_VALN_BASE    5
# define TXT_VALN_BOTTOM  6
# define TXT_ERR_CHAR    '.'

# define GET_VAL           *(wp++)
# define GET_XY( x, y )    x = *(wp++); y = *(wp++)



static void DrwGL_Display_Text( Draw_Ptr p, int sel )
{ /* Draw a Line of text */
  float   xc, yc, high, ang;
  int    len;
  char * str;
  int  i, il, ix, iy, npt;
  wptr wp, ws;
  GLfloat xx, yy, txt_size, txt_width, txt_xtrans, txt_ytrans;
  Draw_Font_Ptr pf;

  xc   = p->txt.txt_x;
  yc   = p->txt.txt_y;
  high = p->txt.txt_high;
  ang  = p->txt.txt_ang;
  len  = p->txt.txt_len;
  str  = p->txt.txt_str;

  pf = Drwgl_txt_font;


//fprintf( fmsg, " GL Text plot at (%f,%f), high = %f, ang = %f, len = %d, s = \"%s\" .\n",
//                 xc, yc, high, ang, len, str );
//Draw_Fmsgupdate();

  if (str)
  {
    if (sel) {
      hmix = hmiy = hmiz = 1e+30;
      hmax = hmay = hmaz = - hmix;
    } else
      DrwGL_Save_Chcntx( 1.0 );


    /* glMatrixMode( GL_MODELVIEW ); It is the normal state */

    glPushMatrix();
    glTranslatef( xc, yc, 0.0 );
    glRotatef( (GLfloat) ang, 0.0, 0.0, 1.0 );
    glScalef( Drwgl_txt_expf*high/((GLfloat)pf->fnt_nmszx),
                             high/((GLfloat)pf->fnt_nmszy), 1.0 );

    if (sel) glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

//  fprintf( fmsg, " Text Attr: haln = %d  valn = %d  path = %d\n",
//                 Drwgl_txt_haln, Drwgl_txt_valn, Drwgl_txt_path );
//  fprintf( fmsg, " exp = %f, spc = %f\n", Drwgl_txt_expf, Drwgl_txt_spcf );
//  Draw_Fmsgupdate();

    /* Find total length or height of the string to displayed */

    txt_size  =      0.0;
    txt_width = - 9999.0;

    for (i = 0; i < len; i++ ) { /* For each character of the string */
      il = pf->fnt_idx[(Char)str[i]];

//    fprintf( fmsg, " GL Text char %c # %d idx = %d\n", str[i], str[i], il );
//    Draw_Fmsgupdate();

      if (il < 0) { /* Set Error/Undefined Replacement Character */
        str[i] = TXT_ERR_CHAR;
        il = pf->fnt_idx[(Char)str[i]]; /* Set Dot for Unknown Character */
      }

      if (il >= 0)
      { /* Graphic character is available */
        wp = &(pf->fnt_grph[il]); /* Set the Graphic list ptr. */
        /* wp[0..3] are xmin, ymin, xmax ymax */
        switch (Drwgl_txt_path) {
          case TXT_PATH_LEFT:
          case TXT_PATH_RIGHT:
            txt_size += (GLfloat) wp[2] + pf->fnt_nmszx*Drwgl_txt_spcf;
            break;
          case TXT_PATH_UP:
          case TXT_PATH_DOWN:
            txt_size += (GLfloat) (wp[3] - wp[1]) + pf->fnt_nmszy*Drwgl_txt_spcf;
            if (wp[2] > txt_width) txt_width = (GLfloat) wp[2];
            break;
          default: ;
        }
      }
    }

    /* Translate the origin alignement point to first character position */

    il = pf->fnt_idx[(Char)str[0]];
    if (il >= 0) { /* Graphic character is available */
      wp = &(pf->fnt_grph[il]); /* Set the Graphic list ptr. */
      switch (Drwgl_txt_path) {
        default:
        case TXT_PATH_RIGHT:
          switch (Drwgl_txt_haln) {
            default:
            case TXT_HALN_NORMAL:
            case TXT_HALN_LEFT:    txt_xtrans =                  0.0; break;

            case TXT_HALN_CENTRE:  txt_xtrans =       - txt_size/2.0; break;

            case TXT_HALN_RIGHT:   txt_xtrans =           - txt_size; break;
          }
          switch (Drwgl_txt_valn) {
            case TXT_VALN_TOP:     txt_ytrans =        - pf->fnt_top; break;

            case TXT_VALN_CAP:     txt_ytrans =        - pf->fnt_cap; break;

            case TXT_VALN_HALF:    txt_ytrans =       - pf->fnt_half; break;

            default:
            case TXT_VALN_NORMAL:
            case TXT_VALN_BASE:    txt_ytrans =       - pf->fnt_base; break;

            case TXT_VALN_BOTTOM:  txt_ytrans =     - pf->fnt_bottom; break;
          }
          break;

        case TXT_PATH_LEFT:      
          switch (Drwgl_txt_haln) {
            case TXT_HALN_LEFT:    txt_xtrans = txt_size     - wp[2]; break;

            case TXT_HALN_CENTRE:  txt_xtrans = txt_size/2.0 - wp[2]; break;

            default:
            case TXT_HALN_NORMAL:
            case TXT_HALN_RIGHT:   txt_xtrans =              - wp[2]; break;
          }
          switch (Drwgl_txt_valn) {
            case TXT_VALN_TOP:     txt_ytrans =        - pf->fnt_top; break;

            case TXT_VALN_CAP:     txt_ytrans =        - pf->fnt_cap; break;

            case TXT_VALN_HALF:    txt_ytrans =       - pf->fnt_half; break;

            default:
            case TXT_VALN_NORMAL:
            case TXT_VALN_BASE:    txt_ytrans =       - pf->fnt_base; break;

            case TXT_VALN_BOTTOM:  txt_ytrans =     - pf->fnt_bottom; break;
          }
          break;

        case TXT_PATH_UP:
          switch (Drwgl_txt_haln) {
            default:
            case TXT_HALN_LEFT:    txt_xtrans =                  0.0; break;

            case TXT_HALN_NORMAL:
            case TXT_HALN_CENTRE:  txt_xtrans =      - txt_width/2.0; break;

            case TXT_HALN_RIGHT:   txt_xtrans =          - txt_width; break;
          }
          switch (Drwgl_txt_valn) {
            case TXT_VALN_TOP:     txt_ytrans =           - txt_size; break;

            case TXT_VALN_CAP:     txt_ytrans =           - txt_size
                                      - pf->fnt_nmszy*Drwgl_txt_spcf; break;

            case TXT_VALN_HALF:    txt_ytrans =       - txt_size/2.0; break;

            default:
            case TXT_VALN_NORMAL:
            case TXT_VALN_BASE:    txt_ytrans =                  0.0; break;

            case TXT_VALN_BOTTOM:  txt_ytrans =     - pf->fnt_bottom; break;
          }
          break;

        case TXT_PATH_DOWN:
          switch (Drwgl_txt_haln) {
            default:
            case TXT_HALN_LEFT:    txt_xtrans =                  0.0; break;

            case TXT_HALN_NORMAL:
            case TXT_HALN_CENTRE:  txt_xtrans =      - txt_width/2.0; break;

            case TXT_HALN_RIGHT:   txt_xtrans =          - txt_width; break;
          }
          switch (Drwgl_txt_valn) {
            default:
            case TXT_VALN_NORMAL:
            case TXT_VALN_TOP:     txt_ytrans =        - pf->fnt_top; break;

            case TXT_VALN_CAP:     txt_ytrans =        - pf->fnt_cap; break;

            case TXT_VALN_HALF:    txt_ytrans =  -txt_size/2.0-wp[3]; break;

            case TXT_VALN_BASE:    txt_ytrans =   - txt_size - wp[3]; break;

            case TXT_VALN_BOTTOM:  txt_ytrans =   - txt_size - wp[3] -
                                                      pf->fnt_bottom; break;
          }
          break;
      }

      /* Initial Position and set transformation */

      for (i = 0; i < len; i++ ) {  /* Loop to plot the character */
        il = Drwgl_txt_font->fnt_idx[(Char)str[i]];
        if (il >= 0) { /* Graphic character is available */
          ws = &(Drwgl_txt_font->fnt_grph[il]);
          wp = ws + 4;

//        fprintf( fmsg, " GL Text 2D Char \"%c\" .\n", str[i] );
//        fprintf( fmsg, " GL Text 2D Position (%f, %f)\n", txt_xtrans, txt_ytrans );
//        Draw_Fmsgupdate();

          while (npt = GET_VAL) { /* Execute each plot directive. */
            if (!sel) glBegin( GL_LINE_STRIP );
            while (npt--) {
              GET_XY( ix, iy );

              xx = ((GLfloat) ix) + txt_xtrans;
              yy = ((GLfloat) iy) + txt_ytrans;

              if (sel) DrwHVertex( xx, yy, 0.0 );
                  else glVertex2d( xx, yy );
            }
            if (!sel) glEnd();
          }

          /* Modify the translation to set for the next character */

          switch (Drwgl_txt_path) {
            default:
            case TXT_PATH_RIGHT: txt_xtrans += ws[2] + pf->fnt_nmszx*Drwgl_txt_spcf;
                                 break;

            case TXT_PATH_LEFT:  txt_xtrans -= ws[2] + pf->fnt_nmszx*Drwgl_txt_spcf;
                                 break;

            case TXT_PATH_UP:    txt_ytrans += ws[3] + pf->fnt_nmszy*Drwgl_txt_spcf;
                                 break;

            case TXT_PATH_DOWN:  txt_ytrans -= ws[3] + pf->fnt_nmszy*Drwgl_txt_spcf;
                                 break;
          }
        }
      }

    }

    glPopMatrix();
    if (sel) glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
        else DrwGL_Restore_Chcntx();
  }
}




/***********************************************************************/
/*                 Segment Transformations  Routines                   */
/***********************************************************************/


void DrwGL_Mul_Mat( GLfloat m1[16], GLfloat m2[16], GLfloat m3[16] )
{ int i, j, k;
  GLfloat r;

  for (i = 0; i < 4; i++)
    for (j = 0; j < 4; j++) {
      r = 0.0;
      for (k = 0; k < 4; k++)
        r += m1[4*i + k]*m2[4*k + j];
      m3[4*i + j] = r;
    }
}



void DrwGL_Seg_Unit_Transform( GLfloat* mat )
{ int i,j;

  if (mat)
    for (i = 0; i < 16; i++)
      mat[i] = Drwgl_mat_unit[i];
}




void DrwGL_Move_Seg( GLfloat* mat, float tx, float ty, float tz )
{
  static float m1[16] = { 1.0, 0.0, 0.0, 0.0,
                          0.0, 1.0, 0.0, 0.0,
                          0.0, 0.0, 1.0, 0.0,
                          0.0, 0.0, 0.0, 1.0 };
  static float m2[16];
  int i;

  if (mat) {
    for (i = 0; i < 16; i++) m2[i] = mat[i];
    m1[12] = tx; m1[13] = ty; m1[14] = tz;
    DrwGL_Mul_Mat( m1, m2, mat );
  }
}





void DrwGL_URot_Mat( GLfloat m[16], float  px, float  py, float  pz,
                                    float  th, float  ph, float  an )
// (X,Y,Z) Angles (th, ph, an) Based Rotation Operartor.
{
  static float cc, ux, uy, uz, sa, ca;
  int i;

//fprintf( fmsg, " URot Angle Seg th = %f, ph = %f, an = %f\n", th, ph, an );
//Draw_Fmsgupdate();

  th = inrd*th; ph = inrd*ph; an = inrd*an; cc = sin( th );
  ux = cc*cos( ph ); uy = cc*sin( ph ); uz = cos( th );
  ca = cos( an ); cc = 1.0 - ca; sa = sin( an );
  m[ 0] =  ux*ux*cc + ca;                        /* m11 */
  m[ 4] =  ux*uy*cc - uz*sa;                     /* m12 */
  m[ 8] =  ux*uz*cc + uy*sa;                     /* m13 */
  m[12] =  px - m[ 0]*px - m[ 4]*py - m[ 8]*pz;  /* m14 */
  m[ 1] =  uy*ux*cc + uz*sa;                     /* m21 */
  m[ 5] =  uy*uy*cc + ca;                        /* m22 */
  m[ 9] =  uy*uz*cc - ux*sa;                     /* m23 */
  m[13] =  py - m[ 1]*px - m[ 5]*py - m[ 9]*pz;  /* m24 */
  m[ 2] =  uz*ux*cc - uy*sa;                     /* m31 */
  m[ 6] =  uz*uy*cc + ux*sa;                     /* m32 */
  m[10] =  uz*uz*cc + ca;                        /* m33 */
  m[14] =  pz - m[ 2]*px - m[ 6]*py - m[10]*pz;  /* m34 */
  m[ 3] =  0.0;                                  /* m41 */
  m[ 7] =  0.0;                                  /* m42 */
  m[11] =  0.0;                                  /* m43 */
  m[15] =  1.0;                                  /* m44 */
}




void DrwGL_Rotate_Mat( float m[16], float  px, float  py, float  pz,
                                    float ome, float chi, float psi )
// Eulerian Angles (ome, chi, psi) Based Rotation Operartor.
{
  static float so, co, sc, cc, sp, cp;
  int i;

//fprintf( fmsg, " Rotate Angle Seg ome = %f, chi = %f, phi = %f\n", ome, chi, psi );
//Draw_Fmsgupdate();

  ome = inrd*ome; chi = inrd*chi; psi = inrd*psi;
  co = cos( ome ); cc = cos( chi ); cp = cos( psi );
  so = sin( ome ); sc = sin( chi ); sp = sin( psi );
  m[ 0] =  co*cc*cp - so*sp;                     /* m11 */
  m[ 4] = -co*cc*sp - so*cp;                     /* m12 */
  m[ 8] =  co*sc;                                /* m13 */
  m[12] =  px - m[ 0]*px - m[ 4]*py - m[ 8]*pz;  /* m14 */
  m[ 1] =  so*cc*cp + co*sp;                     /* m21 */
  m[ 5] = -so*cc*sp + co*cp;                     /* m22 */
  m[ 9] =  so*sc;                                /* m23 */
  m[13] =  py - m[ 1]*px - m[ 5]*py - m[ 9]*pz;  /* m24 */
  m[ 2] = -sc*cp;                                /* m31 */
  m[ 6] =  sc*sp;                                /* m32 */
  m[10] =  cc;                                   /* m33 */
  m[14] =  pz - m[ 2]*px - m[ 6]*py - m[10]*pz;  /* m34 */
  m[ 3] =  0.0;                                  /* m41 */
  m[ 7] =  0.0;                                  /* m42 */
  m[11] =  0.0;                                  /* m43 */
  m[15] =  1.0;                                  /* m44 */
}




void DrwGL_URot_Seg( GLfloat* mat, float  px, float  py, float  pz,
                                   float  th, float  ph, float  an )
{
  static float m1[16], m2[16];
  static float so, co, sc, cc, sp, cp, ux, uy, uz;
  int i;


  if (mat) {
    for (i = 0; i < 16; i++) m2[i] = mat[i];

    DrwGL_URot_Mat( m1, px, py, pz, th, ph, an );
    DrwGL_Mul_Mat( m1, m2, mat );
  }
}




void DrwGL_Rotate_Seg( GLfloat* mat, float  px, float  py, float  pz,
                                     float ome, float chi, float psi )
{
  static float m1[16], m2[16];
  static float so, co, sc, cc, sp, cp, ux, uy, uz;
  int i;


  if (mat) {
    for (i = 0; i < 16; i++) m2[i] = mat[i];

    DrwGL_Rotate_Mat( m1, px, py, pz, ome, chi, psi );
    DrwGL_Mul_Mat( m1, m2, mat );
  }
}




void DrwGL_Scale_Mat( float m[16], float px, float py, float pz,
                                   float sx, float sy, float sz )
{
  m[ 0] =   sx;               /* m11 */
  m[ 4] =  0.0;               /* m12 */
  m[ 8] =  0.0;               /* m13 */
  m[12] =  px*(1.0 - m[ 0]);  /* m14 */
  m[ 1] =  0.0;               /* m21 */
  m[ 5] =   sy;               /* m22 */
  m[ 9] =  0.0;               /* m23 */
  m[13] =  py*(1.0 - m[ 5]);  /* m24 */
  m[ 2] =  0.0;               /* m31 */
  m[ 6] =  0.0;               /* m32 */
  m[10] =   sz;               /* m33 */
  m[14] =  pz*(1.0 - m[10]);  /* m34 */
  m[ 3] =  0.0;               /* m41 */
  m[ 7] =  0.0;               /* m42 */
  m[11] =  0.0;               /* m43 */
  m[15] =  1.0;               /* m44 */
}




void DrwGL_Scale_Seg( GLfloat* mat, float px, float py, float pz,
                                    float sx, float sy, float sz )
{
  static float m1[16], m2[16];
  int i;

  if (mat) {
    for (i = 0; i < 16; i++) m2[i] = mat[i];

    DrwGL_Scale_Mat( m1, px, py, pz, sx, sy, sz );
    DrwGL_Mul_Mat( m1, m2, mat );
  }
}




void DrwGL_Transf_Seg( GLfloat* mat,
                       float  tx, float  ty, float  tz,
                       float  px, float  py, float  pz,
                       float ome, float chi, float psi,
                       float  sx, float  sy, float  sz )
{
  static float m1[16], m2[16];
  int i;

  if (mat) {
    for (i = 0; i < 16; i++) m2[i] = mat[i];

    DrwGL_Rotate_Mat( m1, px, py, pz, ome, chi, psi );
    DrwGL_Mul_Mat( m1, mat, m2 );
    DrwGL_Scale_Mat( m1, px, py, pz, sx, sy, sz );
    DrwGL_Mul_Mat( m1, m2, mat );
    mat[12] += tx;
    mat[13] += ty;
    mat[14] += tz;
  }
}




/***********************************************************************/
/*                     Basic  Display  Routines                        */
/***********************************************************************/



void DrwGL_Display_Plot( int code, int dim, int flag, float * tab, int sel )
{ /* Plot 2D/3D function */
  float x, y, z;
  int         i;
  static int wpdwn = 0;

//fprintf( fmsg, " GL ATTR: line Stipple %d, fill Stipple %d .\n",
//               Drwgl_line_stipple, Drwgl_fill_stipple );
//fprintf( fmsg, " GL ATTR: Fill Front %d, Fill Back %d .\n",
//               Drwgl_fill_front, Drwgl_fill_back );
//fprintf( fmsg, " GL_Plot: Line Cd = %d, Flg = %d, dim = %d\n",
//               code, flag, dim );
//Draw_Fmsgupdate();

  if (sel) {
//  fprintf( fmsg, " GL_Plotn Select.\n" );
//  Draw_Fmsgupdate();
    hmix = hmiy = hmiz = 1e+30;
    hmax = hmay = hmaz = - hmix;
    i = 0;
    while (i++ < dim) {
      x = *(tab++); y = *(tab++);
      z = (flag&cdf_3D)?*(tab++):0.0;
      DrwHVertex( x, y, z );
      if (flag&cdf_NRM) tab += 3;
    }
  }
  else
  {
    if (flag&cdf_ST) {
      if (wpdwn) glEnd();
      if (Drwgl_fill_front) glPolygonMode( GL_FRONT, GL_FILL );
                       else glPolygonMode( GL_FRONT, GL_LINE );
      if (Drwgl_fill_back)  glPolygonMode( GL_BACK, GL_FILL );
                       else glPolygonMode( GL_BACK, GL_LINE );
      if (Drwgl_line_stipple) glEnable( GL_LINE_STIPPLE );
      if (Drwgl_fill_stipple) glEnable( GL_POLYGON_STIPPLE );
      if (Drwgl_fill_Shift) glEnable( GL_POLYGON_OFFSET_FILL );
 
      if (gl2ps_flg) {
        if (Drwgl_line_stipple) gl2psEnable( GL2PS_LINE_STIPPLE );
//      if (Drwgl_fill_stipple) gl2psEnable( GL2PS_POLYGON_STIPPLE ); Not Impemented.
        if (Drwgl_fill_Shift) gl2psEnable( GL2PS_POLYGON_OFFSET_FILL );
      }

      glBegin( code );
//    fprintf( fmsg, " GL Plot Begin with code %d and size %d.\n", code, dim );
//    Draw_Fmsgupdate();
      wpdwn = 1;
    }

    if (flag&cdf_3D)
    { /* 3D plot */
      if (Drwgl_fill_front||Drwgl_fill_back) draw_3D_fill_flg = draw_3D_flg;
      for (i = 0; i < dim; i++)
      {
        if (flag&cdf_NRM) {
//        fprintf( fmsg, " GL Plot n#%d/ (%f,%f,%f).\n", i, tab[0], tab[1], tab[2] );
          glNormal3fv( tab ); tab += 3;
        }
//      fprintf( fmsg, " GL Plot v#%d/ (%f,%f,%f).\n", i, tab[0], tab[1], tab[2] );
        glVertex3fv( tab ); tab += 3;
//      Draw_Fmsgupdate();
      }
    }
    else
    { /* 2D plot */
      for (i = 0; i < dim; i++)
      {
        glVertex2fv( tab ); tab += 2;
      }
    }
    if (flag&cdf_ND)
    {
      glEnd();
//    fprintf( fmsg, " GL Plot end (code %d).\n", code );
//    Draw_Fmsgupdate();
      wpdwn = 0;
      if (Drwgl_line_stipple) glDisable( GL_LINE_STIPPLE );
      if (Drwgl_fill_stipple) glDisable( GL_POLYGON_STIPPLE );
      if (Drwgl_fill_Shift) glDisable( GL_POLYGON_OFFSET_FILL );

      if (gl2ps_flg) {
        if (Drwgl_line_stipple) gl2psDisable( GL2PS_LINE_STIPPLE );
//      if (Drwgl_fill_stipple) gl2psDisable( GL2PS_POLYGON_STIPPLE ); Not Impemented.
        if (Drwgl_fill_Shift) gl2psDisable( GL2PS_POLYGON_OFFSET_FILL );
      }
    }
  }
}




void DrwGL_UsrXYZ( GLdouble x, GLdouble y, GLdouble z, int ini )
{
  GLdouble u, v, w;

  if (gluProject( x, y, z, Drwgl_modlview, Drwgl_proj_wrd, Drwgl_wrdvport,
                  &u, &v, &w ) == GL_TRUE) {
    if (ini) {
      hmix = hmax = u; hmiy = hmay = v; hmiz = hmaz = w;
    } else {
      if (hmix > u) hmix = u; if (hmax < u) hmax = u;
      if (hmiy > v) hmiy = v; if (hmay < v) hmay = v;
      if (hmiz > w) hmiz = w; if (hmaz < w) hmaz = w;
    }
  }
  else {
    fprintf( fmsg, " UsrXYZ cannot Work with (%f,%f,%f)\n", x, y, z );
    Draw_Fmsgupdate();
  }
}



void DrwGL_Circle( Draw_Ptr q, int sel )
{
  GLdouble r, x, y, sa, ea, ac, an, ca;
  int i, n;

  r  = q->circ.circl_r;
  x  = q->circ.circl_x;
  y  = q->circ.circl_y;
  if (sel) {
    DrwGL_UsrXYZ( x - r, y - r, 0.0, 1 );
    DrwGL_UsrXYZ( x - r, y + r, 0.0, 0 );
    DrwGL_UsrXYZ( x + r, y + r, 0.0, 0 );
    DrwGL_UsrXYZ( x + r, y - r, 0.0, 0 );

    /* Set for Directive Hit */
    if ((hmix >= smax)&&(hmax <= smix)&&(hmiy >= smay)&&(hmay <= smiy)) {
      Hits = 1;
//    Hitx = (hmix + hmax)/2.0; Hity = (hmiy + hmay)/2.0; Hitz = (hmiz + hmaz)/2.0;
    }
  } else {
    sa = q->circ.circl_sa;
    ea = q->circ.circl_ea;
    ac = q->circ.circl_ac;
    if (ac < 1e-4) ac = 1e-4;
    n = 4; an = 3.14159275/n;
    do { n *= 2; an /= 2.0; } while ((1.0 - cos( an )) > ac);
    if ((sa < -180.0)||(sa >= 360.0)) sa = 0.0;
    if  ((ea <= sa)||(ea > 360.0)) ea = 0.0;
    n = (int) ((ea - sa)/an);

    if (Drwgl_fill_front) glPolygonMode( GL_FRONT, GL_FILL );
                     else glPolygonMode( GL_FRONT, GL_LINE );
    if (Drwgl_fill_back)  glPolygonMode( GL_BACK, GL_FILL );
                     else glPolygonMode( GL_BACK, GL_LINE );
    if (Drwgl_line_stipple) glEnable( GL_LINE_STIPPLE );
    if (Drwgl_fill_stipple) glEnable( GL_POLYGON_STIPPLE );
    if (gl2ps_flg) {
      if (Drwgl_line_stipple) gl2psEnable( GL2PS_LINE_STIPPLE );
//    if (Drwgl_fill_stipple) gl2psEnable( GL2PS_POLYGON_STIPPLE );
    }
    glBegin( q->circ.circl_kn );

    for (i = 0; i <=n; i++) {
      ca = sa + i*an;
      glVertex2d( x + r*cos( ca ), y + r*sin( ca ) );
    }
    if ((ea - ca) > 1e-4) glVertex2d( x + r*cos( ea ), y + r*sin( ea ) );

    glEnd();
    if (Drwgl_line_stipple) glDisable( GL_LINE_STIPPLE );
    if (Drwgl_fill_stipple) glDisable( GL_POLYGON_STIPPLE );
    if (gl2ps_flg) {
      if (Drwgl_line_stipple) gl2psDisable( GL2PS_LINE_STIPPLE );
//    if (Drwgl_fill_stipple) gl2psDisable( GL2PS_POLYGON_STIPPLE );
    }
  }
}



static void DrwGL_Set_Quadric_Mode( int code )
{
    QuaObj = gluNewQuadric();
    switch (code) {
      case 1: gluQuadricDrawStyle( QuaObj, GLU_LINE );
              gluQuadricNormals( QuaObj, GLU_NONE );
              break;
      case 2: gluQuadricDrawStyle( QuaObj, GLU_POINT );
              gluQuadricNormals( QuaObj, GLU_NONE );
              break;
      case 3: gluQuadricDrawStyle( QuaObj, GLU_FILL );
              gluQuadricNormals( QuaObj, GLU_FLAT );
              break;
      case 4: gluQuadricDrawStyle( QuaObj, GLU_FILL );
              gluQuadricNormals( QuaObj, GLU_SMOOTH );
              break;
      case 0:
      default: /* Standard Fill Mode */
        gluQuadricDrawStyle( QuaObj, GLU_FILL );
        gluQuadricNormals( QuaObj, GLU_NONE );
    }
}



static void DrwGL_Set_Obj_Orient( GLfloat tx, GLfloat ty, GLfloat tz, GLfloat th, GLfloat ph )
{
  glPushMatrix();
  glTranslatef( tx, ty, tz );
  glRotatef( ph, 0.0, 0.0, 1.0 );
  glRotatef( th, 0.0, 1.0, 0.0 );
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
}



void DrwGL_Sphere( Draw_Ptr q, int sel )
{
  GLdouble x, y, z, r;

  glPushMatrix();
  glTranslatef( (GLfloat) q->sphere.sphere_x,
                (GLfloat) q->sphere.sphere_y, (GLfloat) q->sphere.sphere_z );
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

  if (sel) {
    r = q->sphere.sphere_r;
    DrwGL_UsrXYZ( -r, -r, -r, 1 );
    DrwGL_UsrXYZ( -r, -r,  r, 0 );
    DrwGL_UsrXYZ(  r, -r,  r, 0 );
    DrwGL_UsrXYZ(  r, -r, -r, 0 );
    DrwGL_UsrXYZ(  r,  r, -r, 0 );
    DrwGL_UsrXYZ(  r,  r,  r, 0 );
    DrwGL_UsrXYZ( -r,  r,  r, 0 );
    DrwGL_UsrXYZ( -r,  r, -r, 0 );

    /* Set for Directive Hit */
    if ((smax >= hmix)&&(smix <= hmax)&&(smay >= hmiy)&&(smiy <= hmay)) {
      Hits = 1;
//    Hitx = q->sphere.sphere_x; Hity = q->sphere.sphere_y; Hitz = q->sphere.sphere_z;
    }
  } else {
      /* We Use the Quadric mode of GLU Library */

    DrwGL_Set_Quadric_Mode( q->sphere.sphere_k );

//  fprintf( fmsg, " Sphere # 1\n" );
//  Draw_Fmsgupdate();

    gluSphere( QuaObj, (GLfloat) q->sphere.sphere_r,
                         (GLint) q->sphere.sphere_px, (GLint) q->sphere.sphere_pz );
    gluDeleteQuadric( QuaObj );
  }
  glPopMatrix();
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
}



void DrwGL_Cylinder( Draw_Ptr q, int sel )
{
  GLdouble rb, rt, rh;

  DrwGL_Set_Obj_Orient(  q->cyl.cyl_x,  q->cyl.cyl_y, q->cyl.cyl_z,
                        q->cyl.cyl_th, q->cyl.cyl_ph );

  if (sel) {
    rb = q->cyl.cyl_rb; rt = q->cyl.cyl_rt;
    rh = q->cyl.cyl_lg;

    DrwGL_UsrXYZ( -rb, -rb, 0.0, 1 );
    DrwGL_UsrXYZ( -rb,  rb, 0.0, 0 );
    DrwGL_UsrXYZ(  rb,  rb, 0.0, 0 );
    DrwGL_UsrXYZ(  rb, -rb, 0.0, 0 );
    DrwGL_UsrXYZ( -rt, -rt,  rh, 0 );
    DrwGL_UsrXYZ( -rt,  rt,  rh, 0 );
    DrwGL_UsrXYZ(  rt,  rt,  rh, 0 );
    DrwGL_UsrXYZ(  rt, -rt,  rh, 0 );

    /* Set for Directive Hit */
    if ((smax >= hmix)&&(smix <= hmax)&&(smay >= hmiy)&&(smiy <= hmay)) {
      Hits = 1;
//    fprintf( fmsg, " Highlight Cylinder id %d:%d\n", q->seg.dcom.dir_pid, sup_pickid );
//    Draw_Fmsgupdate();
//    Hitx = (hmix + hmax)/2.0; Hity = (hmiy + hmay)/2.0; Hitz = (hmiz + hmaz)/2.0;
    }
  } else {
    /* We Use the Quadric mode of GLU Library */
    DrwGL_Set_Quadric_Mode( q->cyl.cyl_k );
    gluCylinder( QuaObj, (GLfloat) q->cyl.cyl_rb, (GLfloat) q->cyl.cyl_rt,
                         (GLfloat) q->cyl.cyl_lg,
                         (GLint) q->cyl.cyl_px, (GLint) q->cyl.cyl_pz );
    gluDeleteQuadric( QuaObj );
  }
  glPopMatrix();
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
}



void DrwGL_Disk( Draw_Ptr q, int sel )
{
  GLdouble re;

  DrwGL_Set_Obj_Orient(  q->disk.dsk_x,  q->disk.dsk_y,  q->disk.dsk_z,
                         q->disk.dsk_th, q->disk.dsk_ph );

  if (sel) {
    re = q->disk.dsk_re;
    DrwGL_UsrXYZ( -re, -re, 0.0, 1 );
    DrwGL_UsrXYZ( -re,  re, 0.0, 0 );
    DrwGL_UsrXYZ(  re,  re, 0.0, 0 );
    DrwGL_UsrXYZ(  re, -re, 0.0, 0 );
    /* Set for Directive Hit */
    if ((smax >= hmix)&&(smix <= hmax)&&(smay >= hmiy)&&(smiy <= hmay)) {
      Hits = 1;
//    fprintf( fmsg, " Highlight Disk id %d:%d\n",q->seg.dcom.dir_pid, sup_pickid );
//    Draw_Fmsgupdate();
//    Hitx = 0.0; Hity = 0.0; Hitz = 0.0;
    }
  } else {
      /* We Use the Quadric mode of GLU Library */
    DrwGL_Set_Quadric_Mode( q->disk.dsk_k );

    if ((q->disk.dsk_ea <= 0.0)||(q->disk.dsk_sa >= q->disk.dsk_ea))
      gluDisk( QuaObj, q->disk.dsk_ri, q->disk.dsk_re,
                       q->disk.dsk_pt, q->disk.dsk_pr );
    else
      gluPartialDisk( QuaObj, q->disk.dsk_ri, q->disk.dsk_re,
                              q->disk.dsk_pt, q->disk.dsk_pr,
                              q->disk.dsk_sa, q->disk.dsk_ea );

    gluDeleteQuadric( QuaObj );
  }
  glPopMatrix();
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
}




void Set_Clip_Plane_Equ( GLenum plane, float * equ )
{
  GLdouble gl_equ[4];
  int i;

  for (i = 0; i < 4; i++) gl_equ[i] = (GLdouble) equ[i];
  glClipPlane( plane, gl_equ );
}




void DrwGL_Push_Plane( Draw_Ptr p )
{ /* Push a new 2D Plot Plane reference */
  int              i;

  /* Transformation is handled by OpenGL */
  glPushMatrix();
  glMultMatrixf( p->eplane.epl_mat );
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

  Drwgl_segplcnt++;
  p->eplane.epl_prev = Drwgl_currplane;
  Drwgl_currplane = p;
}




void DrwGL_Pop_Plane()
{ /* Push a new 2D Plot Plane reference */
  if (Drwgl_currplane&&(Drwgl_segplcnt > 0)) {
    /* Transformation is handled by OpenGL */
    glPopMatrix();
    glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

    Drwgl_segplcnt--;
    Drwgl_currplane = Drwgl_currplane->eplane.epl_prev;
  }
}




void DrwGL_Push_Bxrf( Draw_Ptr p )
{ /* Push a new reference box */

  if (p->boxref.bxr_mod[1]) {
    Set_Clip_Plane_Equ( GL_CLIP_PLANE0, p->boxref.bxr_lim+0  );
    Set_Clip_Plane_Equ( GL_CLIP_PLANE1, p->boxref.bxr_lim+4  );
    glEnable( GL_CLIP_PLANE0 );
    glEnable( GL_CLIP_PLANE1 );
  }

  if (p->boxref.bxr_mod[2]) {
    Set_Clip_Plane_Equ( GL_CLIP_PLANE2, p->boxref.bxr_lim+8  );
    Set_Clip_Plane_Equ( GL_CLIP_PLANE3, p->boxref.bxr_lim+12 );
    glEnable( GL_CLIP_PLANE2 );
    glEnable( GL_CLIP_PLANE3 );
  }

  if (p->boxref.bxr_mod[3]) {
    Set_Clip_Plane_Equ( GL_CLIP_PLANE4, p->boxref.bxr_lim+16 );
    Set_Clip_Plane_Equ( GL_CLIP_PLANE5, p->boxref.bxr_lim+20 );
    glEnable( GL_CLIP_PLANE4 );
    glEnable( GL_CLIP_PLANE5 );
  }

  p->boxref.bxr_prev = Drwgl_currbrf,
  Drwgl_currbrf = p;

}




void DrwGL_Pop_Bxrf()
{ /* Pop a reference box of the box reference stack */
  Draw_Ptr p;


  p = Drwgl_currbrf;

  if (p) {
    Drwgl_currbrf = p->boxref.bxr_prev;

    if (Drwgl_currbrf) {
      if (p->boxref.bxr_mod[1]) {
        Set_Clip_Plane_Equ( GL_CLIP_PLANE0, Drwgl_currbrf->boxref.bxr_lim+0  );
        Set_Clip_Plane_Equ( GL_CLIP_PLANE1, Drwgl_currbrf->boxref.bxr_lim+4  );
      }
      if (p->boxref.bxr_mod[2]) {
        Set_Clip_Plane_Equ( GL_CLIP_PLANE2, Drwgl_currbrf->boxref.bxr_lim+8  );
        Set_Clip_Plane_Equ( GL_CLIP_PLANE3, Drwgl_currbrf->boxref.bxr_lim+12 );
      }
      if (p->boxref.bxr_mod[3]) {
        Set_Clip_Plane_Equ( GL_CLIP_PLANE4, Drwgl_currbrf->boxref.bxr_lim+16 );
        Set_Clip_Plane_Equ( GL_CLIP_PLANE5, Drwgl_currbrf->boxref.bxr_lim+20 );
      }
    }
    else
    {
      if (p->boxref.bxr_mod[1]) {
        glClipPlane( GL_CLIP_PLANE0, Drwgl_cliplane0 );
        glClipPlane( GL_CLIP_PLANE1, Drwgl_cliplane1 );
      }
      if (p->boxref.bxr_mod[2]) {
        glClipPlane( GL_CLIP_PLANE2, Drwgl_cliplane2 );
        glClipPlane( GL_CLIP_PLANE3, Drwgl_cliplane3 );
      }
      if (p->boxref.bxr_mod[3]) {
        glDisable( GL_CLIP_PLANE4 );
        glDisable( GL_CLIP_PLANE5 );
      }
    }
  }
}




void DrwGL_PP_Attrib( int code )
{
  drwgl_astkel* pck;

  if (code) { /* Push */
//  fprintf( fmsg, " Push on Stack from level %d\n", Drwgl_AStkSp );
//  Draw_Fmsgupdate();
    if (Drwgl_AStkSp < (ATTRSTKSZ-1)) {
      Drwgl_AStkSp++;
      Drwgl_AStack[Drwgl_AStkSp] = Drwgl_AStack[Drwgl_AStkSp-1];
      pck = &(Drwgl_AStack[Drwgl_AStkSp]);
      pck->txt_haln = Drwgl_txt_haln;
      pck->txt_valn = Drwgl_txt_valn;
      pck->txt_path = Drwgl_txt_path;
      pck->txt_expf = Drwgl_txt_expf;
      pck->txt_spcf = Drwgl_txt_spcf;
      pck->txt_font = Drwgl_txt_font;
      pck->txt_prec = Drwgl_txt_prec;
    } else {
      fprintf( fmsg, " DRAW_SERVER: Attibute STack overflow Error.\n" );
      Draw_Fmsgupdate();
    }
  } else { /* Pop */
//  fprintf( fmsg, " Pop on Stack from level %d\n", Drwgl_AStkSp );
//  Draw_Fmsgupdate();
    if (Drwgl_AStkSp > 0) { Drwgl_AStkSp--;
      pck = &(Drwgl_AStack[Drwgl_AStkSp]);
      DrwGL_Line_Attr( pck->lin_kind, pck->lin_size );
      DrwGL_Marker_Attr( pck->mrk_kind, pck->mrk_size );
      DrwGL_Fill_Attr( pck->fil_kind, pck->fil_styl );
      Drwgl_txt_haln = pck->txt_haln;
      Drwgl_txt_valn = pck->txt_valn;
      Drwgl_txt_path = pck->txt_path;
      Drwgl_txt_expf = pck->txt_expf;
      Drwgl_txt_spcf = pck->txt_spcf;
      Drwgl_txt_font = pck->txt_font;
      Drwgl_txt_prec = pck->txt_prec;
      if (pck->col_rgba[0] < 0.0) DrwGL_Color( (int) (- pck->col_rgba[0] + 0.5) );
                             else DrwGL_ColorRGB( pck->col_rgba );
    } else {
      fprintf( fmsg, " DRAW_SERVER: Attibute STack underflow Error.\n" );
      Draw_Fmsgupdate();
    }
  }
}



void DrwGL_Display_Seg( Draw_Ptr p, GLenum mode )
{ /* Display a particular Segment */
  Draw_Ptr      q;
  float *     tab;
  int    cod, dim, flg, i, sc, sel, ssl, save_segplcnt, svtmatcnt;
  static GLint Plot_Code[] = { GL_POINTS,         /*  0  for  GL Codes */
                               GL_LINES,          /* -1 */
                               GL_LINE_STRIP,     /* -2, Use by draw_out 1 */
                               GL_LINE_LOOP,      /* -3 */
                               GL_TRIANGLES,      /* -4 */
                               GL_TRIANGLE_STRIP, /* -5 */
                               GL_TRIANGLE_FAN,   /* -6 */
                               GL_QUADS,          /* -7 */
                               GL_QUAD_STRIP,     /* -8 */
                               GL_POLYGON         /* -9, Use by draw_out 3 */
                             };

  save_segplcnt = Drwgl_segplcnt;
  Drwgl_segplcnt = 0;

  glPushMatrix();
  glMultMatrixf( p->seg.seg_mat );
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
//Show_Matf( " Set Seg Matrix : \n", p->seg.seg_mat );
//Show_Mat( " Eff Seg Matrix : \n", Drwgl_modlview );

  sel = (mode == GL_SELECT);
  ssl = 1;

  svtmatcnt = Tmat_Cnt;

  q = p->seg.seg_fdir;
  while (q)
  { /* Loop on all Segment Directive */
//  fprintf( fmsg, " GL Out dir # %d, id = %d\n",
//                 q->seg.dcom.dir_knd, q->seg.dcom.dir_pid );
//  Draw_Fmsgupdate();
    if (sel) {
      Hits = 0;
      ssl = (q->seg.dcom.dir_pid&1);
//    fprintf( fmsg, " Selection Hits Search with %d.\n", ssl );
//    Draw_Fmsgupdate();
    }

    switch (q->seg.dcom.dir_knd) {
      case Draw_Segment:
      break;

      case Draw_Exec_Seg:
        i = q->exeseg.dcom.dir_pid>>1;
        sup_pickid += i;
        if (q->exeseg.exec_seg)
          DrwGL_Display_Seg( q->exeseg.exec_seg, mode );
        sup_pickid -= i;
      break;

      case Draw_PP_Attr:
        DrwGL_PP_Attrib( q->ppattr.ppa_code );
      break;

      case Draw_Ena_Plane:
        DrwGL_Push_Plane( q );
      break;

      case Draw_Dis_Plane:
        DrwGL_Pop_Plane();
      break;

      case Draw_Box_Ref:
        DrwGL_Push_Bxrf( q );
      break;

      case Draw_Box_Pop:
        DrwGL_Pop_Bxrf();
      break;

      case Draw_PushTrfMat:
        glPushMatrix();
        if (!Tmat_Cnt)
          glTranslatef( (GLfloat) orgx, (GLfloat) orgy, (GLfloat) orgz );
        glMultMatrixf( q->tmat.trsf_mat );
        glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
        Tmat_Cnt++;
      break;

      case Draw_PopTrfMat:
        glPopMatrix();
        glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
        Tmat_Cnt--;
      break;

      case Draw_Plot:
        tab = q->tab.plo_tab;
        dim = q->tab.plo_npt;
        flg = q->tab.plo_flg;
        cod = q->tab.plo_knd;
        if (ssl) /* Skip for UnDetect flag */
        switch (cod) {
          case 1: /* Line */
            if (dim > 0) DrwGL_Display_Plot( GL_LINE_STRIP, dim, flg, tab, sel );
                    else DrwGL_Display_Plot( GL_LINE_LOOP, -dim, flg, tab, sel );
          break;

          case 2: /* Marker */
            if (Drwgl_mrk_kind < 0)
              DrwGL_Display_Plot( GL_POINTS, dim, flg, tab, sel );
            else
              DrwGL_Display_Marker( abs( dim ), flg, tab, sel );
          break;

          case 3: /* Fill Area */
          case 4: /* Fill Area with Normals */
            DrwGL_Display_Plot( GL_POLYGON, dim, flg, tab, sel );
          break;

          default:
            if ((cod <= 0)&&(cod >= -9))
              DrwGL_Display_Plot( Plot_Code[-cod], dim, flg, tab, sel );
        }
      break;

      case Draw_Text:
        if (ssl) DrwGL_Display_Text( q, sel );
      break;

      case Draw_Circle:   if (ssl) DrwGL_Circle( q, sel );  break;
      case Draw_Sphere:   if (ssl) DrwGL_Sphere( q, sel );  break;
      case Draw_Cylinder: if (ssl) DrwGL_Cylinder( q,sel ); break;
      case Draw_Disk:     if (ssl) DrwGL_Disk( q, sel );    break;

      case Draw_Line_Attr:
        DrwGL_Line_Attr( q->linattr.plia_kind, q->linattr.plia_size );
      break;

      case Draw_Marker_Attr:
        DrwGL_Marker_Attr( q->mrkattr.mrka_kind, q->mrkattr.mrka_size );
      break;

      case Draw_Fill_Attr:
        DrwGL_Fill_Attr( q->filattr.fila_kind, q->filattr.fila_styl );
      break;

      case Draw_Text_Attr:
        DrwGL_text_Attr( q->txtattr.txta_haln, q->txtattr.txta_valn,
                         q->txtattr.txta_path, q->txtattr.txta_expf,
                         q->txtattr.txta_spcf );
      break;

      case Draw_Font:
        DrwGL_Font( q->font.fnt_font, q->font.fnt_prec );
      break;

      case Draw_Color:
        for (i=0;i<4;i++)
          Drwgl_AStack[Drwgl_AStkSp].col_rgba[i] = q->color.col_rgba[i];
        if (q->color.col_rgba[0] < 0.0)
          DrwGL_Color( (int) (- q->color.col_rgba[0] + 0.5) );
        else
          DrwGL_ColorRGB( q->color.col_rgba );
      break;

      case Draw_Blend_Set:
        DrwGL_Alpha_Set( q->blend.bld_flags,
                         q->blend.bld_src, q->blend.bld_dst );
      break;

      case Draw_Light_Def:
        glLightfv( q->light.lig_spc,
                   q->light.lig_kind, q->light.lig_prm );
//      fprintf( fmsg, " Set Light #%d, parameter #%d at (", q->light.lig_spc,
//                     q->light.lig_kind );
//      for (i=0;i<4;i++) fprintf( fmsg, " %f ", q->light.lig_prm[i] );
//      fprintf( fmsg, ").\n" );
//      Draw_Fmsgupdate();
      break;

      case Draw_Mat_Light_Prop:
        glMaterialfv( q->lmatp.lmat_face,
                      q->lmatp.lmat_kind, q->lmatp.lmat_prm );
//      fprintf( fmsg, " Set face #%d MatProp #%d at (", q->lmatp.lmat_face,
//                     q->lmatp.lmat_kind );
//      for (i=0;i<4;i++) fprintf( fmsg, " %f ", q->lmatp.lmat_prm[i] );
//      fprintf( fmsg, ").\n" );
//      Draw_Fmsgupdate();
      break;

      default:
      break;
    }


    if (sel&&Hits) { /* When Hit on select, Push the segment specifications */
      if (HitsSp < Hits_Bufsz-1) {
        HitsSp++;
        HitsHeap[HitsSp].seg  =  p->seg.seg_ide;
        HitsHeap[HitsSp].prio =  p->seg.seg_pri;
        HitsHeap[HitsSp].pid  =  (q->seg.dcom.dir_pid>>1) + sup_pickid;
        HitsHeap[HitsSp].xmin =  hmix;
        HitsHeap[HitsSp].ymin =  hmiy;
        HitsHeap[HitsSp].zmin =  hmiz;
        HitsHeap[HitsSp].xmax =  hmax;
        HitsHeap[HitsSp].ymax =  hmay;
        HitsHeap[HitsSp].zmax =  hmaz;
      }

//    fprintf( fmsg, " Hits # %d Seg # %d %f %d.\n", HitsSp, p->seg.seg_ide,
//                   p->seg.seg_pri, q->seg.dcom.dir_pid );
//    fprintf( fmsg, " Hits Echo Area [%f,%f,%f,%f]\n", hmix, hmiy, hmax, hmay );
//    Draw_Fmsgupdate();
      Hits = 0;
    }
//  else
//  if (sel&&(q->seg.dcom.dir_knd == Draw_Plot)) {
//    fprintf( fmsg, " No_Hits # %d Seg # %d %f %d.\n", HitsSp, p->seg.seg_ide,
//                   p->seg.seg_pri, q->seg.dcom.dir_pid );
//    fprintf( fmsg, " No_Hits Echo Area [%f,%f,%f,%f]\n", hmix, hmiy, hmax, hmay );
//    Draw_Fmsgupdate();      
//  }

//  fprintf( fmsg, " display Seg Loop.\n" );
//  Draw_Fmsgupdate();
    q = q->seg.dcom.dir_nxt;
  }

  while (Drwgl_segplcnt > 0) DrwGL_Pop_Plane();
  Drwgl_segplcnt = save_segplcnt;

  glPopMatrix();

  while (svtmatcnt < Tmat_Cnt--) glPopMatrix();

//fprintf( fmsg, " display Seg End.\n" );
//Draw_Fmsgupdate();
}




void Set_Clip_Lim( GLdouble tab[4], int cat, float val )
{
  int i;

  for (i = 0;i < 3; i++) tab[i] = 0.0;
  if (cat < 0)
  {
    cat = abs( cat );
    tab[cat - 1] =  1.0; tab[3] = - (GLdouble) val;
  }
  else
  {
    tab[cat - 1] = -1.0; tab[3] =   (GLdouble) val;
  }
}



void DrwGL_Display( int prflg )
{ /* Display the Whole of Draw Picture */
  Draw_Ptr p;
  GLdouble fovy, px, py, sx, sy;
  int Prsflg;

  Drwgl_currbrf   = NULL;
  Drwgl_currplane = NULL;
 
  Zdepth     = draw_uzmax - draw_uzmin;
  MinZ_Dist = fabs( draw_uzmin );
  if (Prsflg = (Draw_Dist > MinZ_Dist))
    MinZ_Dist = Draw_Dist; /* Perspective mode */
  MaxZ_Dist = MinZ_Dist + Zdepth;
  ZOrg      = (MaxZ_Dist + MinZ_Dist)/2.0;

  Drwgl_AStkSp    =    0;

  if (!Drwgl_clipflg)
  {
    Set_Clip_Lim( Drwgl_cliplane0, -1, draw_uxmin );
    Set_Clip_Lim( Drwgl_cliplane1,  1, draw_uxmax );
    Set_Clip_Lim( Drwgl_cliplane2, -2, draw_uymin );
    Set_Clip_Lim( Drwgl_cliplane3,  2, draw_uymax );
    Drwgl_clipflg = 1;
  }

  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

  glMatrixMode( GL_PROJECTION );
  glLoadIdentity();

//fprintf( fmsg, " *** Perspective at %f => %d ***\n", Draw_Dist, Prsflg );
//Draw_Fmsgupdate();

  p = draw_fsysseg;

  switch (prflg) {
    case  1: /* Printer output */
//    if (p) p = p->seg.seg_nxt; /* Skip the Sheet Contour */
      if (Prsflg) {
        fovy = atan( 2.0*pw_cwsy/ZOrg )/inrd;
        gluPerspective( fovy, pw_cwsx/pw_cwsy, MinZ_Dist, MaxZ_Dist );
        glTranslated( -pw_cwpx, -pw_cwpy, 0.0 );
      } else {
        glOrtho( (GLdouble) pw_cxmin, (GLdouble) pw_cxmax,
                 (GLdouble) pw_cymin, (GLdouble) pw_cymax,
                 MinZ_Dist, MaxZ_Dist );
      }
      glViewport( (GLint) pp_rx, (GLint) pp_ry, (GLint) pt_rx, (GLint) pt_ry );
      break;

    case  2: /* LG2PS Output */
      gl2ps_flg = 1; /* Enable The GL2PS Mode */
//    if (p) p = p->seg.seg_nxt; /* Skip the Sheet Contour */
    case  0: /* Screen Output */
      glGetIntegerv( GL_VIEWPORT, Drwgl_wrdvport );
      if (Prsflg) {
//      fprintf( fmsg, "  In Perspective.\n" );
//      Draw_Fmsgupdate();
        fovy = atan( 2.0*dw_cwsy/ZOrg )/inrd;
        gluPerspective( fovy, dw_cwsx/dw_cwsy, MinZ_Dist, MaxZ_Dist );
        glTranslated( -dw_cwpx, -dw_cwpy, 0.0 );
      } else {
//      fprintf( fmsg, "  In Parrallel.\n" );
//      Draw_Fmsgupdate();
        glOrtho( (GLdouble) dw_cxmin, (GLdouble) dw_cxmax,
                 (GLdouble) dw_cymin, (GLdouble) dw_cymax,
                 MinZ_Dist, MaxZ_Dist );
      }
      glViewport( 0, 0, dt_rx, dt_ry );
      break;
  }


  /*  OK to Redraw the Picture */

  glTranslatef( 0.0, 0.0, - ZOrg );

  glGetDoublev( GL_PROJECTION_MATRIX, Drwgl_proj_wrd );
  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

//  Show_clplan(  " Plane #0 ", Drwgl_cliplane0 );
//  Show_clplan(  " Plane #1 ", Drwgl_cliplane1 );
//  Show_clplan(  " Plane #2 ", Drwgl_cliplane2 );
//  Show_clplan(  " Plane #3 ", Drwgl_cliplane3 );

  glClipPlane( GL_CLIP_PLANE0, Drwgl_cliplane0 );
  glClipPlane( GL_CLIP_PLANE1, Drwgl_cliplane1 );
  glClipPlane( GL_CLIP_PLANE2, Drwgl_cliplane2 );
  glClipPlane( GL_CLIP_PLANE3, Drwgl_cliplane3 );

  glDisable( GL_CLIP_PLANE0 );
  glDisable( GL_CLIP_PLANE1 );
  glDisable( GL_CLIP_PLANE2 );
  glDisable( GL_CLIP_PLANE3 );

  /* Set all Plot Defaults */

  DrwGL_Line_Attr( 1, 1.0 );
  DrwGL_Marker_Attr( 1, 1.0 );
  DrwGL_Fill_Attr( 2, 1 );
  DrwGL_Color( 1 );
  if (lighting) { /* To force in all lighting case the color #1 */
    glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR, draw_srccoltb[1] );
    glMaterialfv( GL_FRONT_AND_BACK, GL_EMISSION, draw_srccoltb[1] );
  }

//  Show_Vwp( "World ViewPort", Drwgl_wrdvport );
//  Show_Mat( "World Projection", Drwgl_proj_wrd );
//  Show_Mat( "World Modelview", Drwgl_modlview );

  while (p) { /* Loop to display the System Segments */
    if (p->seg.seg_stat&DRWSEG_VISIBLE) DrwGL_Display_Seg( p, GL_RENDER );
    p = p->seg.seg_nxt;
  }

  glEnable( GL_CLIP_PLANE0 );
  glEnable( GL_CLIP_PLANE1 );
  glEnable( GL_CLIP_PLANE2 );
  glEnable( GL_CLIP_PLANE3 );

  p = draw_fsegm;
  if (p) {
    DrwGL_Line_Attr( 1, 1.0 );
    DrwGL_Marker_Attr( 1, 1.0 );
    DrwGL_Fill_Attr( 2, 1 );
    DrwGL_Color( 1 );

//  fprintf( fmsg, "  User Redraw Viewport GL (%d,%d,%d,%d).\n",
//                 d_vpxp, d_vpyp, d_vpsx, d_vpsy );
//  fprintf( fmsg, "**User Redraw Env. GL (%f,%f,%f,%f,%f,%f),\n",
//                 dw_cxmin, dw_cxmax, dw_cymin, dw_cymax,
//                 draw_uzmin, draw_uzmax );
//  fprintf( fmsg, "  User Redraw Z Env GL (%f, %f, %f, %f).\n",
//                 MinZ_Dist, MaxZ_Dist, Zdepth, ZOrg );
//  Draw_Fmsgupdate();

    while (p)
    { /* Loop to display the User Segments */
      if (p->seg.seg_stat&DRWSEG_VISIBLE) DrwGL_Display_Seg( p, GL_RENDER );
//                                else fprintf( fmsg, " Seg invisible!!\n" );
      p = p->seg.seg_nxt;
    }
  }


  /* glFlush(); */

  if (gl2ps_flg) { /* When GL2PS PostScript Output was active */
    gl2ps_flg = 0; /* Disable The GL2PS Mode */
  }

//fprintf( fmsg, "  End Redraw GL Environment.\n" );
//Draw_Fmsgupdate();
}




void DrwGL_Pick_Search( int xm, int ym )
{ /* To locate a Draw Segment as pointed by the mouse at (xm, ym). */
  GLint viewport[4];
  unsigned int    i, iseg, pid;
  GLuint       *ptr;
  Draw_Ptr p;
  GLdouble aspect, fovy;
  float    mix, miy, max, may, miz, maz, prior;

  ym = dt_ry - ym; /* Do Y Screen coordinate with lower left origine. */

  Drwgl_currbrf   = NULL;
  Drwgl_currplane = NULL;

  Drwgl_AStkSp    =    0;


//fprintf( fmsg, "  Pick Seg Search Hits for (%d,%d).\n", xm, ym );
//Draw_Fmsgupdate();

  p = draw_fsegm;
  if (p) { /* If a plot was performed with some segments */
    glViewport( 0, 0, dt_rx, dt_ry );
    glGetIntegerv( GL_VIEWPORT, Drwgl_wrdvport ); /* Get Current Viewport */

    smix = xm - Hits_Preci; smax = xm + Hits_Preci;
    smiy = ym - Hits_Preci; smay = ym + Hits_Preci;

    glMatrixMode( GL_PROJECTION );

    glLoadMatrixd( Drwgl_proj_wrd );

//  glLoadIdentity();

//  if (Draw_Dist > MinZ_Dist) {
//      fovy   = (360.0/3.1415927)*asin( (dw_cymax -  dw_cymin)/Draw_Dist );
//      aspect = (dw_cxmax - dw_cxmin)/(dw_cymax -  dw_cymin);
//      gluPerspective( (GLdouble) fovy, (GLdouble) aspect,
//                      MinZ_Dist, MaxZ_Dist );
//  } else {
//    glOrtho( (GLdouble) dw_cxmin, (GLdouble) dw_cxmax,
//             (GLdouble) dw_cymin, (GLdouble) dw_cymax,
//             MinZ_Dist, MaxZ_Dist );
//  }

//  glTranslatef( 0.0, 0.0, ZOrg );
//  glGetDoublev( GL_PROJECTION_MATRIX, Drwgl_proj_wrd );
    glMatrixMode( GL_MODELVIEW );
//  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
 
    /* Set all Plot Defaults */

    DrwGL_Line_Attr( 1, 1.0 );
    DrwGL_Marker_Attr( 1, 1.0 );
    DrwGL_Fill_Attr( 2, 1 );
    DrwGL_Color( 1 );

    Drwgl_segment = 0;
    Drwgl_ident   = 0;

    while (p)
    { /* Loop to display the System Segments */
      if ((p->seg.seg_stat&DRWSEG_DETECT) &&
          (p->seg.seg_stat&DRWSEG_VISIBLE)) {
//      fprintf( fmsg, " Pick Search Try Segment %d.\n", p->seg.seg_ide );
        DrwGL_Display_Seg( p, GL_SELECT );
      }
      // DRWSEG_HLIGHT for highlight on.
      p = p->seg.seg_nxt;
    }

    /* Now We can Examine the Hits Heap */
    prior = -1.0;
    while (HitsSp >= 0) {
      if (HitsHeap[HitsSp].prio >= prior)
      {
        iseg  = HitsHeap[HitsSp].seg;
        prior = HitsHeap[HitsSp].prio;
        pid   = HitsHeap[HitsSp].pid;
        mix   = HitsHeap[HitsSp].xmin;
        miy   = HitsHeap[HitsSp].ymin;
        miz   = HitsHeap[HitsSp].zmin;
        max   = HitsHeap[HitsSp].xmax;
        may   = HitsHeap[HitsSp].ymax;
        maz   = HitsHeap[HitsSp].zmax;
      }
      HitsSp--;
    }

    if (prior >= 0.0) {      
//    fprintf( fmsg, " Hits Area [%5.0f,%5.0f,%5.0f,%5.0f], dt_ry = %d,\n",
//                    mix, miy, max, may, dt_ry );
      Drwgl_pechw = (int)( max - mix + 0.5) + 2*Hits_Preci;
      Drwgl_pechh = (int)( may - miy + 0.5) + 2*Hits_Preci;
      Drwgl_pechx = (int)(mix + 0.5) - Hits_Preci;
      Drwgl_pechy = dt_ry - ((int)(may + 0.5) + Hits_Preci);
      Drwgl_segment = iseg; Drwgl_ident = pid;
//    fprintf( fmsg, " Hits of seg #%d (id=%d) with echo w [%d,%d:%d,%d]\n",
//             Drwgl_segment, Drwgl_ident,
//             Drwgl_pechx, Drwgl_pechy, Drwgl_pechw, Drwgl_pechh );
//    Draw_Fmsgupdate();
    }
  }
}




void DrwGL_Get_World_XY( int xsc, int ysc, Drwgl_Inptyp cd )
{
  static GLdouble  xx, yy, zz;

  ysc = dt_ry - ysc; /* Convert in real raster with Low Left Origine */

  /* Compute the User View Port Coordinates */
//  fprintf( fmsg, "  Screen x y = %d %d .\n", xsc, ysc );
//  Draw_Fmsgupdate();

  gluUnProject( (GLdouble) xsc, (GLdouble) ysc, 0.0,
                 Drwgl_mat_unit, Drwgl_proj_wrd, Drwgl_wrdvport,
                 &xx, &yy, &zz );

  Drwgl_xcurr = xx; Drwgl_ycurr = yy;
}




void DrwGL_Init_View_Matrix( int bt2 )
{
  int i;

  if (Drwgl_ViewMat) {
    Drwgl_xfirst = Drwgl_xcurr; Drwgl_yfirst = Drwgl_ycurr;
//  fprintf( fmsg, " Set New View Org 0 = (%f,%f).\n", Drwgl_xfirst, Drwgl_yfirst );
//  Draw_Fmsgupdate();
    /* Save the Initial View Matrix */
    for (i = 0; i<16;i++) svd_mat[i] = Drwgl_ViewMat[i];
  }
  rst_view = bt2; /* Save the View Restauration Flag */
}




void DrwGL_Change_View_Matrix()
{
  static GLfloat dx, dy, vv, nfx, nfy, sca, ang, cm, sm,
                 cm2, sm2, csm, cp, sp, co, so;
  static GLfloat m[16];
//static GLdouble md[16];
  int i;

  if (Drwgl_ViewMat) {
    vv = sqrt( Drwgl_xfirst*Drwgl_xfirst + Drwgl_yfirst*Drwgl_yfirst );
    if (vv < 2.0) {
      Drwgl_xfirst = Drwgl_xcurr; Drwgl_yfirst = Drwgl_ycurr;
    } else {
      nfx = Drwgl_xfirst/vv; nfy = Drwgl_yfirst/vv;
      dx = Drwgl_xcurr - Drwgl_xfirst; dy = Drwgl_ycurr - Drwgl_yfirst;
      /* Compute the angle of the axis og ang_p with X screen axis */
      ang = atan2( nfy, nfx );
//    fprintf( fmsg, " View  V = (%f,%f) Mu =% f, ", nfx, nfy, (ang/inrd) + 90.0 );
//    fprintf( fmsg, " View  D = (%f,%f), ", dx, dy );
      cm = - sin( ang ); sm = cos( ang );
      /* Compute the (dx,dy) projection on (xfirst,yfirst)  nd deduce the ang_p angle */
      sca = dx*nfx + dy*nfy;
      ang = atan2( sca, vv );
//    fprintf( fmsg, "sca %f, vv %f : Phi(U)  = %f, ", ang/inrd );
      cp = cos( ang ); sp = - sin( ang );
      /* Compute the perpandicular part of (dx,dy) and deduce the ang_n angle */
      dx -= sca*nfx; dy -= sca*nfy;
      sca = sqrt( dx*dx + dy*dy );
      if (nfy*dx - nfx*dy < 0.0) sca = - sca;
      ang = atan2( sca, vv );
      co = cos( ang ); so = sin( ang );
//    fprintf( fmsg, "Omega(Z) = %f .\n", ang/inrd );
//    Draw_Fmsgupdate();

      /* Build the Rot(ang_n,Z)*Rot(ang_p,p) rotation matrix (p in plane [X,Y]) */
      cm2 = cm*cm; sm2 = sm*sm;
      csm = cm*sm*(1.0 - cp);
      sca = cm2 + sm2*cp; sm2 = sm2 + cm2*cp; cm2 = sca;
      m[ 0] =  co*cm2 - so*csm; m[ 1] =  co*csm - so*sm2; m[ 2] = sp*(co*sm + so*cm);
      m[ 4] =  so*cm2 + co*csm; m[ 5] =  so*csm + co*sm2; m[ 6] = sp*(so*sm - co*cm);
      m[ 8] =          - sm*sp; m[ 9] =            cm*sp; m[10] =                 cp;
      m[ 3] = Drwgl_FixX - (m[ 0]*Drwgl_FixX + m[ 1]*Drwgl_FixY + m[ 2]*Drwgl_FixZ);
      m[ 7] = Drwgl_FixY - (m[ 4]*Drwgl_FixX + m[ 5]*Drwgl_FixY + m[ 6]*Drwgl_FixZ);
      m[11] = Drwgl_FixZ - (m[ 8]*Drwgl_FixX + m[ 9]*Drwgl_FixY + m[10]*Drwgl_FixZ);
      m[12] =        0.0; m[13] =       0.0; m[14] =       0.0; m[15] =         1.0;

//    for (i=0;i<16;i++) md[i] = m[i];
//    Show_Mat( " View Change Mat ", md );

      /* Multiply at left */
      DrwGL_Mul_Mat( svd_mat, m, Drwgl_ViewMat );
    }
  }
}



int DrwGl_Finish_View_Request()
{
  int i;

  if (Drwgl_ViewMat&&rst_view) {
    for (i = 0; i<16;i++) Drwgl_ViewMat[i] = svd_mat[i];
    return 1;
  }
  else
  return 0;
}




void DrwGL_Update_Display( void )
{ /* Update the Display */
  Drwgl_flags |= DRWSTATUS_UPDATE; /* Set the Picture Update Flag */
}




void DrwGL_Inp_Req_Display( Char * bupdate )
{ /* Update the Display if required */
  if (bupdate)
  { Drwgl_flags |= DRWSTATUS_UPDATE + DRWSTATUS_INPUT;
    bupdate = 0;
  }
  else Drwgl_flags |= DRWSTATUS_INPUT;
  /* ... and wait for Request Done */

  Sdrw_read_unlock(); /* UnLock the User Directive Execution Graph */
  Sdrw_Wait_For_Request();
}





/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/


