/*
 *		Copyright IBM Corporation 1989
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of IBM not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * IBM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * IBM BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 *
 * University of Illinois at Urbana-Champaign
 * Department of Computer Science
 * 1304 W. Springfield Ave.
 * Urbana, IL	61801
 *
 * (C) Copyright 1987, 1988 by The University of Illinois Board of Trustees.
 * All rights reserved.
 *
 * Tool: X 11 Graphical Kernel System
 * Author: Gregory Scott Rogers
 * Author: Sung Hsien Ching Kelvin
 * Author: Yu Pan
 *
 *
 *
 *
 * Version changed for Draw Library.
 * Author: Pierre Wolfers, 
 *         CNRS, Laboratoire de cristallographie,
 *         BP 166X
 *         F 38042 GRENOBLE CEDEX 9
 *
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>


/* The Next line must be before the include of Draw_Font.h */
#define _MAKE_FONT   20000  /* To force the maximum Font allocation */
                            /* Size of Temp. Tab. for Gr. Dsc. */


#include <draw/Draw_Font.h>                  /* Load Font Definitions */



static Draw_Font_Dsc   Font;            /* The Temporary Font Descriptor */

static int             min_x, min_y,    /* Minimaxi of current character */
                       max_x, max_y,
                       cname,           /* Character name */
                       margin,          /* Current Margin */
                       default_margin,  /* Default Margin */
                       grph_inxt;       /* Index of First Free Graphic Element */

static wptr            grph_next,       /* Pointer to First Free Graphic Element */
                       grph_cuch,       /* Pointer to curr. Graph. descr. of ch. */
                       grph_cupl;       /* Pointer to begin of curr. Plot dir. */

static char            inp_fspc[64],    /* Input Font File Specification */
                       out_fspc[64],    /* Output Draw/Font File Spc. */
                       lis_fspc[64];    /* Listing of Font File Spc. */

static FILE *          fnt_file;        /* Input Font File */
static FILE *          fnt_list;        /* Listing Font File */




#define GRPH_PUT( v )      *(grph_next++) = v; \
                           grph_inxt++
 
#define GRPH_DOT( x, y )   *(grph_next++) = x; \
                           *(grph_next++) = y; \
                           grph_inxt += 2




void clean_line( char * str )
{
  int i, len;

  if (str) {
    len = strlen( str );
    if (str[len - 1] = '\r') str[len -1] = 0;
  }
}



static void SetCapHalf()
{
  wptr pch;

  /* The Graphic Table Content is as following :
   *
   *   fnt_idx[<char>]  -> <ext_minx>, <ext_miny>,  ! Extend Size Minimum.
   *                       <ext_maxx>, <ext_maxy>,  ! Extend Size Maximum.
   *                       <plt_nsz1>,              ! First Pen Plot of nsz1 points.
   *                       <x_1>, <y_1>,            ! The nsz1 points.
   *                       .............
   *                       <x_nsz1>, <y_nsz1>
   *                       .............            ! Continue for nsz2 points
   *                       .............            !  ... and so on.
   *                       0                        ! End of Character Graphic.
   */ 

  if (Font.fnt_idx['A'] != -1) {
    pch = &(Font.fnt_grph[Font.fnt_idx['A']]);
    Font.fnt_cap = pch[3]; /* The Char First case is always the "S" value */
  }
  else
    Font.fnt_cap = Font.fnt_nmszy;

  if (Font.fnt_idx['a'] != -1) {
    pch = &(Font.fnt_grph[Font.fnt_idx['a']]);
    Font.fnt_half = pch[3];
  }
  else
    Font.fnt_half = Font.fnt_nmszy / 2;

  Font.fnt_base = 0;
}




static void MakeSpaceChar()
{
  wptr pchN;
  word maxx, maxy;
  int code;

  if (Font.fnt_idx['N'] != -1) {
    pchN = &(Font.fnt_grph[Font.fnt_idx['N']]);
  }
  else
    pchN = NULL;

  if (pchN) {
    maxx = pchN[2];
    maxy = pchN[3];
  }
  else
  {
    maxx = Font.fnt_nmszx;
    maxy = Font.fnt_nmszy;
  }

  Font.fnt_idx[' '] = grph_inxt;        /* Begin a SPACE charater */

  for (code = 1; code < 255; code++)  /* Make all invalid chars point to SPACE */
    if (Font.fnt_idx[code] == -1) Font.fnt_idx[code] = grph_inxt;

  GRPH_DOT(    0,    0 );  /* Extend x and y Minimums */
  GRPH_DOT( maxx, maxy );  /* Extend x and y Maximums */

  GRPH_PUT( 0 );           /* End of Graphic for space character */
}


static void WriteFont()
{
  int fd, size;

  size = (char *) grph_next - (char *) &Font;

  if ((fd = open( Font.fnt_name, O_CREAT | O_RDWR, 0644 )) < 0) {
    perror( "open Font file" );
    exit( 1 );
  }

  if (write( fd, (char *) &size, sizeof( int )) != sizeof( int )) {
    perror( "write size" );
    exit( 1 );
  }
  if (write( fd, (char *) &Font, size) != size ) {
    perror("write Font" );
    exit( 1 );
  }
  close( fd );
}


#if 0

Draw_Font_Ptr ReadFont( char *Fspc )
{
  int fd, size;
  Draw_Font_Ptr Fdsc;

  if ((fd = open( Fspc, O_RDONLY, 0644 )) < 0) {
    perror( "open" );
    exit( 1 );
  }
  
  if (read( fd, (char *) &size, sizeof( int ) ) != sizeof( int )) {
    perror( "read size" );
    exit( 1 );
  }

  if ((Fdsc = (Draw_Font_Ptr *) malloc( size ) == NULL) {
    perror("malloc failed");
    exit( 1 );
  }
  if (read(fd, (char *) Fdsc, size ) != size) {
    perror( "read Font" );
    exit( 1 );
  }
  close( fd );
  return Fdsc;
}

#endif


static void PrintFont( Draw_Font_Ptr F )
{
  int c, id, i, n, x, y;
  wptr cp;

  if (lis_fspc[0]) {
    fnt_list = fopen( lis_fspc, "w" );
    fprintf( fnt_list, " Font name = %s\n", F->fnt_name );
    fflush( fnt_list );

    fprintf( fnt_list, " Nominal sizes %4d * %4d\n",
                       F->fnt_nmszx, F->fnt_nmszx );
    fprintf( fnt_list, " Characters highs :\n" );
    fprintf( fnt_list, "   Char top at ...... %4d\n", F->fnt_top );
    fprintf( fnt_list, "   Char cap at ...... %4d\n", F->fnt_cap );
    fprintf( fnt_list, "   Char half at ..... %4d\n", F->fnt_half );
    fprintf( fnt_list, "   Char base at ..... %4d\n", F->fnt_base );
    fprintf( fnt_list, "   Char bottom at ... %4d\n\n", F->fnt_bottom );

    for (c = 0; c < 256; c++) {
      if ((id = F->fnt_idx[c]) != -1) {
        fprintf( fnt_list, " Character %c [%4d] : ", c, c );
        cp = &F->fnt_grph[id];
        fprintf( fnt_list, " Ext min and max : (%d,%d), (%d,%d);\n",
                           cp[0], cp[1], cp[2],  cp[3] );
        cp += 4;
        while (cp[0]) {
          n = *(cp++);
          fprintf( fnt_list, "   Plot[%4d] :", n );
          for (i = 1; i <= n; i++) {
            x = *(cp++); y = *(cp++);
            fprintf( fnt_list, " (%4d,%4d)", x, y );
            if (i < n) if (i%5 == 0) fprintf( fnt_list, ",\n               " );
                                else fprintf( fnt_list, "," );
                  else fprintf( fnt_list, ";\n" );
          }
        }
      }    
      fprintf( fnt_list, "\n" );
    }
    fclose( fnt_list );
  }
}



static void cklimits( int x, int y )
{
  if (x < min_x) min_x = x;
  if (x > max_x) max_x = x;
  if (y < min_y) min_y = y;
  if (y > max_y) max_y = y;

  if (y > Font.fnt_top) Font.fnt_top = y;
  if (y < Font.fnt_bottom) Font.fnt_bottom = y;
}


static void BeginChar( char ch1, char ch2 )
{
  margin = default_margin;

  min_x = min_y =  5000;
  max_y = max_y = -5000;

  if (((unsigned char) ch2) <= 32) cname = ch1;
           else cname = 128 + ch2;

  Font.fnt_idx[cname] = grph_inxt;
  
  grph_cuch = grph_next;   /* Save the Begin addr. of new Graphic */

  GRPH_DOT( 0, 0 );        /* Save room for x and y Minimums */
  GRPH_DOT( 0, 0 );        /* Save room for x and y Maximums */
}


static void FinishChar()
{
  grph_cuch[0] = min_x - margin;
  grph_cuch[1] = min_y - margin;
  grph_cuch[2] = max_x + margin;
  grph_cuch[3] = max_y + margin;

  GRPH_PUT( 0 );           /* End of Graphic for this character */

  if ((max_x - min_x) + 2*margin > Font.fnt_nmszx)
    Font.fnt_nmszx = (max_x - min_x) + 2*margin;
  if ((max_y - min_y) + 2*margin > Font.fnt_nmszy)
    Font.fnt_nmszy = (max_y - min_y) + 2*margin;
}


static void ReadVFont( int argc, char *argv[] )
{
  int i, j, penct, spacing, width, x, y;
  char s[128], *sp, ch1, ch2;

  /* Run model : make_font <input> <output> [<listing>] */

//  argv[1] = "../font/Draw_Font_R.src";
//  argv[2] = "../font/Draw_Font_1.drwfnt";
//  argv[3] = "../font/Draw_Font_1.lis";
//  argc = 4;

  if (argc >= 2) strncpy( inp_fspc, argv[1], 64 );
  else {
    printf( " Give the Source Input Font File Specification\n" );
    fgets( inp_fspc, 64, stdin );
    clean_line( inp_fspc );
  }

  if (argc >= 3) strncpy( out_fspc, argv[2], 64 );
  else {
    printf( " Give the Resulting Output Font File Specification\n" );
    fgets( out_fspc, 64, stdin );
    clean_line( out_fspc );
  }

  if (argc >= 4) strncpy( lis_fspc, argv[3], 64 );
            else lis_fspc[0] = 0;

  if ((fnt_file = fopen( inp_fspc, "r" )) == NULL) {
    fprintf( stderr, "cannot fopen(%s,\"r\")\n", inp_fspc );
    exit( 1 );
  }

  /* Set Each char as undefined */

  Font.fnt_next     =  NULL;       /* Null Font Link */
  Font.fnt_ident    =    -1;       /* Unassigned font number */

  strncpy( Font.fnt_name, out_fspc, 64 );

  for (i = 0; i < 256; i++) Font.fnt_idx[i] = -1;

  Font.fnt_nmszx    =     0;
  Font.fnt_nmszy    =     0;
  Font.fnt_top      = -9999;
  Font.fnt_cap      =     0;
  Font.fnt_half     =     0;
  Font.fnt_base     =     0;
  Font.fnt_bottom   =  9999;

  grph_next = &Font.fnt_grph[0];

  penct = 0;

  while (fgets( s, 80, fnt_file ) != NULL) {
    clean_line( s );
    sp = s;
    switch (*sp++) {
      case 'S': /* fprintf( stdout, "Special font\n" ); */
        break;

      case 'U':
        sscanf( sp, "%d", &spacing );
        default_margin = spacing/2;
        break;

      case 'u':
        sscanf( sp, "%d", &spacing );
        margin = spacing/2;
        break;

      case 'C':
        ch1 = sp[1]; ch2 = sp[2];
        /* fprintf( stdout, "Character name = %s\n", sp ); */
        BeginChar( ch1, ch2 );
        break;

      case 'w':
        sscanf( sp, "%d", &width );
        /* fprintf( stdout, "total width = %d\n", width ); */
        break;

      case 'm':
        if (penct > 1) {
          *grph_cupl = penct; /* Set the Prev. Plot count when exist. */
          /* printf( " m with previous plot of %d dot(s).\n", penct ); */
        }
        sscanf( sp, "%d%d", &x, &y );
        x += margin;
        y += margin;
        cklimits( x, y );
        penct = 1;         /* Set to flags the begin of a plot directive. */
        break;

      case 'n':
        if (penct == 1) {
          /* printf( " n with previous m (first) dot.\n" ); */
          grph_cupl = grph_next; /* Keep the location for the size of Plot */
          GRPH_PUT( 0 );     /* Make room for plot size */
          GRPH_DOT( x, y );  /* First dot of plot directive. */
        }
        sscanf( sp, "%d%d", &x, &y );
        x += margin;
        y += margin;
        cklimits( x, y );
        GRPH_DOT( x, y );    /* Graphic dot for this character */
        penct++;
        break;

      case 'E':
        /* fprintf( stdout, "End of that one...." ); */
        if (penct > 1) {
          *grph_cupl = penct; /* Set the Prev. Plot count when exist. */
          /* printf( " E with previous plot of %d dot(s).\n", penct ); */
        }
        penct = 0;        
        FinishChar();
        break;

      default:
	    /* fprintf( stdout, "%s\n", s ); */
        break;
    }
  }
}


main( int argc, char *argv[] )
{
  ReadVFont( argc, argv );

  SetCapHalf();

  MakeSpaceChar();

  PrintFont( (Draw_Font_Dsc *) &Font );

  WriteFont();

  return 0;
}
