/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*              D R A W   -   C L I E N T   L I B R A R Y                *
*                                                                       *
*               (Additional Control Graphic Directives)                 *
*                (For current 2D Axis Box Generation)                   *
*                                                                       *
*                                                                       *
*                                by                                     *
*                                                                       *
*            Pierre Wolfers, Laboratoire de Cristallographie            *
*                                                                       *
*            CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166            *
*                                                                       *
*                       F 38042 GRENOBLE CEDEX 9                        *
*                                                                       *
*                             F R A N C E                               *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/



/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/


#include <draw/draw_common.h>
#include <draw/draw_l4env.h>    /* load the DRAW L4 Library environment */

#define LOG_E 0.434294481903



void Draw_Axis_Inscale( float mi, float ma, int n,
                        float *inf, float *sup, float *delta, float *ef )
{ /**********************************************************************/
  /*                                                                    */
  /*    Procedure to get a set of suitable value for inf, sup and delta */
  /*    for an axis, from the min and max coordinate value and the      */
  /*    number of wanted intervals (n).                                 */
  /*    ef is the resulting filling efficience                          */
  /*                                                                    */
  /*                                                                    */
  /*    Draw_Inscale work on the User Coordinates, and not on their     */
  /*    Drawing Surface Coordinates (as Draw cm or paper coordinates).  */
  /*                                                                    */
  /**********************************************************************/

  int i, ip, i1;
  float lar, s, max, min;

  max = ma;
  min = mi;
  /* Permute min and max when required */
  if (max < min) { s   = max; max = min; min = s ; }
  lar = max - min;
  if (lar < 1.0e-10) lar = 1.0e-10;                /* Find the largest power of 10 under lar/n. */
  ip = floor( log( lar/n )*LOG_E );
  i1 = abs(ip);
  s  = 10.0;
  if (ip < 0) s = 1.0/s;
  *delta = 1.0;
  while (i1-- > 0) *delta = (*delta)*s;            /* Compute the suitable Delta for the axis */

  i = 0;
  do {                                             /* Loop on the Choice of delta */
    *inf = floor( min/(*delta)+0.00001 )*(*delta); /* inf and sup must be int. mult. of delta */
    *sup = (*inf) + (*delta)*n;
    if ( (max - (*sup))/lar > 1e-3 ) {             /* If delta is too low, we try an other delta */
      switch((i % 5)+1) {                          /* in the range of 10^ip * by (2, 2.5, 4, 5, 10) */
        case 1:
        case 5: *delta = (*delta)*2.0;  break;     /* Delta is 2Exxx or 10Exxx */

        case 2:
        case 4: *delta = (*delta)*1.25; break;     /* Delta is 2.5Exxx or 5Exxx */

        case 3: *delta = (*delta)*1.6;  break;     /* Delta is 4Exxx */

        default: break;
      } /*switch*/
      i++;                                         /* Increment i to try a new delta value */
    } /*if*/
  } while (((max - (*sup))/lar) > 1e-3);
  *ef = 100.0*lar/((*sup) - (*inf));               /* The efficience factor is given in percent */
} /* Draw_Axis_Inscale */



void DRW_API( axis_inscale )( float *mi, float *ma, int *n,
                              float *inf, float *sup, float *delta, float *ef )
{
  Draw_Axis_Inscale( *mi, *ma, *n, inf, sup, delta, ef );
}


/******************************************************************************/



void Draw_Axis_Setvscale( float  inf, float sup,  float  delta, float  rsiz,
                          float *fsiz, float *eff, float *scale )
{ /*************************************************************************/
  /*                                                                       */
  /*       ***  Procedure Axis_Setvscale to set the virtual scale .        */
  /*       ***  inf, sup and delta are the minimum, maximum and increment  */
  /*                of the coordinate, (as computed by Draw_Inscale),      */
  /*       ***  szp is the required size of this axis,                     */
  /*       ***  psup is the resulting maximal value of the axis,           */
  /*       ***  stp  is interval in paper coordinate,                      */
  /*       ***  eff is the efficience (real), and                          */
  /*       ***  scale is the final value to applied for the plot,          */
  /*                                                                       */
  /*                                                                       */
  /*   Draw_Setvscale work on the Delta scale on the Drawing support       */
  /*   to get a suitable interval between each axis tick in term of        */
  /*   length in cm (to get a easely readable scale on a measuring paper). */
  /*                                                                       */
  /*************************************************************************/

  int               ip;
  float sc, sd, sf, st;

  sc = rsiz/(sup - inf);                           /* Get the original Scale in cm/user_unit */
  sd = delta*sc;                                   /* Set the required user interval in cm */
  ip = floor( log( sd )*LOG_E + 0.00001 );         /* ipw = floor( LOG10( sd ) ) */

  if (ip < 0) sf = 0.1; else sf = 10.0;            /* Prepare sf to compute 10^ip */
  ip = abs( ip ); st = 1.0;
  while (ip > 0) { ip = ip - 1; st = st*sf; }      /* Now  st = (the largest power of 10) < the */
                                                   /* ... user interval on the paper. */
                                                   /* Select the appriopriate paper interval sc */

  switch((int)floor( (2.0*sd)/st + 0.00001 )) {    /* The selector is : floor( (2*sd)/sc) */
                                                   /* ... to scan the choices is (1, 2, 2.5, 4, 5 or 8)*st */
    case  4: st = st*2.0; break;                   /* The Interval (in Paper Unit) is 2.0Exxx cm */
    case  5:
    case  6:
    case  7: st = st*2.5; break;                   /* The Interval is 2.5Exxx cm */
    case  8:
    case  9: st = st*4.0; break;                   /* The Interval is 4.0Exxx cm */
    case 10:
    case 11:
    case 12:
    case 13:
    case 14:
    case 15: st = st*5.0; break;                   /* The Interval is 5.0Exxx cm */
    case 16:
    case 17:
    case 18:
    case 19: st = st*8.0; break;                   /* The Interval is 8.0Exxx cm */

    default: break;                                /* The interval is 1.0Exxx cm */
  } /*switch*/

  sc = st/delta;                                   /* Compute the scale */
  *fsiz  = (sup - inf)*sc;                         /* Set the resulting size of the axis */
  *eff   = st/sd;                                  /* Set the real efficience and the paper interval */
  *scale = sc;                                     /* Set the scale of paper transcription */
} /* Draw_Axis_Setvscale */



void DRW_API( axis_setvscale )( float *inf, float *sup, float *delta, float *rsiz,
                                float *fsiz, float *eff, float *scale )
{
  Draw_Axis_Setvscale( *inf, *sup, *delta, *rsiz, fsiz, eff, scale );
}




/******************************************************************************/

int Draw_Axis_AutoScale( float min, float max, float szp, int ani, int ana,
                         float *inf, float *sup, float *del, float *fsz, float *sca, float *eff )
{ /*************************************************************************/
  /*                                                                       */
  /*   ***  The autoscale function is used to find a suitable axis scale   */
  /*   ***  on a given Draw paper coordinate range.                        */
  /*                                                                       */
  /*   ***  The function result is the selected number of interval.        */
  /*                                                                       */
  /*   ***  ani and are the mini maxi on the number of ticks,              */
  /*   ***  min and max are the limits of user coordinate range,           */
  /*   ***  szp is the size of the axis in DRAW cm.                        */
  /*                                                                       */
  /*   ***  At return :                                                    */
  /*                                                                       */
  /*   ***    inf, sup and del are the user values of mininmum, maximum    */
  /*   ***    and increment after Rounding,                                */
  /*   ***    fsz is the final size of axis,                               */
  /*   ***    sca the final scale, and eff the related filling space       */
  /*   ***    efficience.                                                  */
  /*                                                                       */
  /*************************************************************************/

  int nt, nt0;
  float inf1, sup1, del1, ef1, ef2, fma, stp, sc1;

  *eff = 0.0;
  if ((ani > ana)||(ana <= 0)) {                   /* The default number in an interval of 1.0 cm. */
    ani = floor( szp + 0.5 ); ana = ani;
  }

  /* We try for each allowed interval number */
  for (nt0 = ani; nt <= ana; nt++) {
    /* We get appropriate axis display value in first */
    Draw_Inscale( min, max, nt0, &inf1, &sup1, &del1, &ef1 );
    /* Set the appropriate scale on the plot paper */
    Draw_Setvscale( inf1, sup1, del1, szp, &fma, &ef2, &sc1 );
    ef1 = ef1*ef2;                                 /* Get the resulting efficience in percent */
    if (ef1 > *eff) {
      *inf = inf1;                                 /* Load the low limit in user coordinate */
      *sup = sup1;                                 /* Load the high limit in user coordinate */
      *del = del1;                                 /* Load the increment in user coordinate */
      *fsz = fma;                                  /* Load the final size of the axis in cm. */
      *sca = sc1;                                  /* Load the scale in cm/user_unit */
      *eff = ef1;                                  /* Store any more efficient result */
      nt   = nt0;                                  /* Load the selected number of increment */
    } /* if */
  } /* for */
  return nt;
} /* Draw_Axis_Autoscale */



int DRW_API( axis_autoscale )( float *min, float *max, float *szp, int *ani, int *ana,
                               float *inf, float *sup, float *del, float *fsz, float *sca, float *eff )
{
  return Draw_Axis_AutoScale( *min, *max, *szp, *ani, *ana, inf, sup, del, fsz, sca, eff );
}




/******************************************************************************/

void Draw_Axis_Viewscale( float a_inf, float a_sup, float a_del,
                          float *a_vinf, float *a_vsup, float *a_vdel, float *a_sc, float *a_sh )
{ /* Draw_Axis_Viewscale */
  int i1, ip;
  float inf, sup, del, s;

  *a_sh = (float)0.0;
  inf = a_inf; sup = a_sup; del = a_del;           /* Assume no change */
  if (fabs( sup - inf ) < inf*100.0)               /* We must set a shift */
    *a_sh = 100.0*del*floor( inf/(100.0*del) );    /* Set shift value for displayed number along of axis */

  /* Compute the power of ten for multiplier */
  ip = floor( log( del )*LOG_E );                  /* sc power of ten */
  if ((ip > -3) &&  (ip <= 2)) *a_sc = 1.0;
  else {
    i1 = abs(ip); s = 10.0;
    if (ip < 0) s = 1.0/s;
    *a_sc = 1.0;
    while (i1 > 0) { i1 = i1 - 1; *a_sc = (*a_sc)*s; }
  } /* if */
  *a_vdel = del/(*a_sc);
  *a_vinf = (inf - (*a_sh))/(*a_sc);
  *a_vsup = (sup - (*a_sh))/(*a_sc);
} /* Draw_Viewscale */


void DRW_API( axis_viewscale )( float *a_inf, float *a_sup, float *a_del,
                                float *a_vinf, float *a_vsup, float *a_vdel, float *a_sc, float *a_sh )
{
  Draw_Axis_Viewscale( *a_inf, *a_sup, *a_del, a_vinf, a_vsup, a_vdel, a_sc, a_sh );
}

/******************************************************************************/
