/***** DRAW Interface ROUTINES for V Graphic Libary *******/
/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               D R A W   -   S E R V E R   M O D U L E                 *
*                                                                       *
*           (Draw interface for V function and procedures)              *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/

/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/



/* include the environment SERVER */

#include <draw/draw_common.h>
#include <draw/sdrw_env.h>
#include <draw/sdrw_routines.h>


#include <v/vnotice.h>  // for vNoticeDialog
// #include <v/vkeys.h>    // To map keys
// #include <v/vdc.h>      // To get the Drawing Definitions.

#include "sdrw_cnv.h"   // Get our Canvas Definitions.
#include "sdrw_cmdw.h"  // Our header.



/**************************************************************************/
/*                                                                        */
/*                       DRAW/V Local Variables                           */
/*                                                                        */
/**************************************************************************/



extern int    prmc;    /* Parameter Count. */
extern char** prmv;    /* Parameter List. */


int Draw_node_sizes[] = { sizeof( Draw_Segm_Dir ), /* Node Allocation ... */
                          sizeof( Draw_Clip_Dir ), /* ... Size Table */
                          sizeof( Draw_Poly_Dir ),
                          sizeof( Draw_Poly_Dir ),
                          sizeof( Draw_Poly_Dir ),
                          sizeof( Draw_Text_Dir ),
                          sizeof( Draw_Circle_Dir ),
                          sizeof( Draw_Ellipse_Dir ),
                          sizeof( Draw_LAttr_Dir ),
                          sizeof( Draw_MAttr_Dir ),
                          sizeof( Draw_FAttr_Dir ),
                          sizeof( Draw_SAttr_Dir )
                        };


static float  line_cx,  line_cy,   /* Line equation Coefficient for clipping */
              line_ct;
static Dpoint line_prv, line_cur;  /* ... and related Previous point */
static int    line_frs = 1,        /* 1/0 for First, or normal and last point */
              line_cnt = 0;        /* Output point count */






/**************************************************************************/
/*                                                                        */
/*                       DRAW/V Interface Routines                        */
/*                                                                        */
/**************************************************************************/





void vdrw_settransf( Draw_Window * w, Draw_W_Trans * t )
{ /**************************************************/
  /*                                                */
  /* Establish a Transformation Characteristic      */
  /* to form the device coordinates xd, yd from the */
  /* World coordinates xw, yw :                     */
  /*                                                */
  /*  The clipping limits are the window limits.    */
  /*                                                */
  /*   xd = (xw - trans->dfx)*trans->cfx;           */
  /*   yd = (yw - trans->dfy)*trans->cfy;           */
  /*                                                */
  /**************************************************/

  /* This Routine take in account the transformation of y axis orinetation. */
  /* The device x axis is increasing from left to right direction */
  /* The device y axis is increasing from up to down direction */
  /* The world axis origine is the lower left corner of the canvas */
  
  t->cfx  = ((float) (t->vpx2 - t->vpx1))/(w->winx2 - w->winx1);
  t->dfx  = t->vpx1/t->cfx - w->winx1;
  t->cfy  = ((float) (t->vpy1 - t->vpy2))/(w->winy2 - w->winy1);
  t->dfy  = t->vpy2/t->cfy - w->winy1;
}


static int line_cut( float x0, float x1, float xlim )
{
  return (x0 - xlim)*(xlim - x1) >= 0.0;
}


static void line_equation()
{ /* Compute the Segment line Equation */
  line_cy = line_cur.x - line_prv.x;
  line_cx = line_prv.y - line_cur.y;
  line_ct = line_cx*line_prv.x + line_cy*line_prv.y;
}


static void line_inter( int bxy, float val, Dpoint * pt )
{ /* Compute the intersection coordinates pt.     */
  /* bxy = 0/1 for val = x/y.                     */
  /* Warning: the intersection must be existing.  */
  if (bxy)
  { /* val is a y value */ 
    pt->y = val;
    pt->x = (line_ct - line_cy*val)/line_cx;
  }
  else
  {
    pt->x = val;
    pt->y = (line_ct - line_cx*val)/line_cy;
  }
}


void vdrw_outpoint( Dpoint* pt )
{ /* Output a point in World coordinates table */
  Dpoint* pts;

  if (pt)
  { /* Add a new point to plot in temporary table */
    line_cnt++;
    
  }
  else
  { /* Set the final sequence */
    /* Create the related poly node */
    pts = (Dpoint *) malloc( line_cnt*sizeof( Dpoint ) );

    line_cnt = 0;
  }
}



void vdrw_clip_compute( Dpoint* pt, Draw_Window* w )
{ /* For first point, called with pt == NULL */
  /* For normal points, called with *pt */
  static Dpoint rp, rp1;
  static int bprv, bcur, brp;

  if (line_frs)
  { /* Store the first point and test for position */
    line_prv.x = pt->x; line_prv.y = pt->y;
    bprv = ((w->winx1 <= line_prv.x)&&(w->winx2 >= line_prv.x)&&
            (w->winy1 <= line_prv.y)&&(w->winy2 >= line_prv.y));
    if (bprv) vdrw_outpoint( &line_prv );
    line_frs = 0;
  }
  else
  if (!pt)
  { /* Last point must be output */
    vdrw_outpoint( NULL );
    line_frs = 1; /* Next call is a first point */
  }
  else /* New point is specified */
  { /* Perform the Clipping Process */
    line_cur.x = pt->x; line_cur.y = pt->y;
    bcur = ((w->winx1 <= line_cur.x)&&(w->winx2 >= line_cur.x)&&
            (w->winy1 <= line_cur.y)&&(w->winy2 >= line_cur.y));
    if (bprv)
    { /* Previous point is in the window */
      if (bcur)
        vdrw_outpoint( &line_cur ); /* Current point is in the window */
      else
      { /* Current point is out of window */
        line_equation();
        brp = 0;
        /* Get the intersection with low or up x limit */
        if (line_cur.x < w->winx1)
        { line_inter( 0, w->winx1, &rp );
          brp = line_cut( w->winy1, w->winy2, rp.x ); 
        }
        else
        if (line_cur.x > w->winx2)
        { line_inter( 0, w->winx2, &rp );
          brp = line_cut( w->winy1, w->winy2, rp.x );
        }
        if (!brp) /* When the rp point is out of box we test with y limits */
          /* Get the intersection with low or up x limit */
          if (line_cur.y < w->winy1) line_inter( 1, w->winy1, &rp );
                                else line_inter( 0, w->winx2, &rp );
        /* Now rp is output */
        vdrw_outpoint( &rp );
        vdrw_outpoint( NULL ); /* Stop plot for Clipping */
      }
      /* The current point is set as the new previous point */
      line_prv.x = line_cur.x; line_prv.y = line_cur.y; 
      bprv = bcur;
    }
    else
    { /* Previous point is out of the window */
      line_equation();
      if (bcur)
      { /* The pen return in the window */
        /* Get the intersection with low or up x limit */
        if (line_cur.x < w->winx1)
        { line_inter( 0, w->winx1, &rp );
          brp = line_cut( w->winy1, w->winy2, rp.x ); 
        }
        else
        if (line_cur.x > w->winx2)
        { line_inter( 0, w->winx2, &rp );
          brp = line_cut( w->winy1, w->winy2, rp.x );
        }
        if (!brp) /* When the rp point is out of box we test with y limits */
          /* Get the intersection with low or up x limit */
          if (line_cur.y < w->winy1) line_inter( 1, w->winy1, &rp );
                                else line_inter( 0, w->winx2, &rp );
        /* Now rp is the new previous point to output */
        vdrw_outpoint( &line_prv );
        /* and the current point is the new previous one */
        line_prv.x = line_cur.x; line_prv.y = line_cur.y; bprv = 1;
      }
      else
      { /* The two points are out of the window */
        /* We look for the two intersection with the rectangle edge */
        brp = 0;
        if (line_cut( line_prv.x, line_cur.x, w->winx1 ))
        { line_inter( 0, w->winx1, &rp );
          if ((w->winy1 <= rp.y)&&(w->winy2 >= rp.y)) brp = 1;
        }
        if (line_cut( line_prv.x, line_cur.x, w->winx2 ))
        { line_inter( 0, w->winx2, &rp1 );
          if ((w->winy1 <= rp1.y)&&(w->winy2 >= rp1.y))
            if (brp) brp = 2;
                else { brp = 1; rp.x = rp1.x; rp.y = rp1.y; }
        }
        if (brp < 2)
        { if (line_cut( line_prv.y, line_cur.y, w->winy1 ))
          { line_inter( 0, w->winy1, &rp1 );
            if ((w->winx1 <= rp.x)&&(w->winx2 >= rp.x))
              if (brp) brp = 2;
                  else { brp = 1; rp.x = rp1.x; rp.y = rp1.y; }
          }
          if (brp == 1) line_inter( 0, w->winy2, &rp1 );
        }
        
        if (brp)
        { /* The segment has the part rp<->rp1 in the window */
          vdrw_outpoint( &rp ); vdrw_outpoint( &rp1 ); vdrw_outpoint( NULL );
        }
        line_prv.x = line_cur.x; line_prv.y = line_cur.y; /* bprv already 0 */
      }
    }
  }
}



Draw_Ptr vdrw_locate_seg( int idseg )
{ /* Locate a Segment by Identifier Number */
  int idx;
  Draw_Ptr p;
  
  idx = idseg%SEG_CACHE_SIZE;
  fprintf( fmsg, "  Look for Seg. # %d on cache # %d.\n", idseg, idx );
  Draw_Fmsgupdate();
  if (draw_sgtb[idx].seg_idnt != idseg)
  { p = draw_fsegm;
    while ((p != NULL)&&(p->seg.seg_ide != idseg))
      p = p->seg.seg_nxt;
    if (p != NULL)
    { draw_sgtb[idx].seg_ref  = p;
      draw_sgtb[idx].seg_idnt = idseg;
    }
  }
  else
    p = draw_sgtb[idx].seg_ref;
  return p;
}



Draw_Ptr vdrw_new_segment( int idseg, int bapd )
{ Draw_Ptr p, q;
  int i, j;
  
  if (idseg <= 0) idseg = draw_seg_count + 1;
  fprintf( fmsg, "  Create Seg. # %d in mode %d.\n", idseg, bapd );
  Draw_Fmsgupdate();
  if ((idseg <= draw_seg_count)||
      ((idseg>=1000000)&&(idseg<=1001000)))
  { p = vdrw_locate_seg( idseg );
    if (p)
    { /* A segment with this ident is existing */
      fprintf( fmsg, "  Seg is already existing => Clear it.\n" );
      Draw_Fmsgupdate();
      if (!bapd) /* Clear/update mode */
      { /* We clear the segment primitive */
        draw_curseg = p;
        vdrw_zero_seg( p );
      }
    }
  }
  else p = NULL;
  
  if (p == NULL)
  { /* Create a new Segment */
    fprintf( fmsg, "  Create New Seg.\n" );
    Draw_Fmsgupdate();
    p = (Draw_Ptr) malloc( Draw_node_sizes[Draw_Segment] );
    /* Initialize it */
    p->seg.dcom.dir_nxt = NULL;
    p->seg.dcom.dir_knd = Draw_Segment;
    p->seg.dcom.dir_pid = 0;
    draw_curr_pid = 1;
    p->seg.seg_ide  = idseg;
    p->seg.seg_fdir = NULL;
    p->seg.seg_ldir = NULL;
    p->seg.seg_stat = 1; /* Default is Visible */
    p->seg.seg_prv  = draw_lsegm;
    p->seg.seg_nxt  = NULL;
    for (i=0;i<3,i++;)
      for (j=0;j<4,j++;)
        p->seg.seg_mat[i][j] = (float) (i == j)?1.0:0.0;
    if ((idseg<1000000)||(idseg>1001000))
      if (draw_seg_count < idseg) draw_seg_count = idseg;
    /* Link it in the segment list */
    if (draw_fsegm != NULL) draw_lsegm->seg.seg_nxt = p;
                       else draw_fsegm = p;
    draw_lsegm = p;
    /* Put it in the Segment Cache */
    i = idseg%SEG_CACHE_SIZE;
    draw_sgtb[i].seg_ref  = p;
    draw_sgtb[i].seg_idnt = idseg;
  }
  draw_curseg = p;
  return p;
}



Draw_Ptr vdrw_new_node( Draw_Dir_Types ndty )
{ Draw_Ptr p, q;
  int i, j;

  if (ndty == Draw_Segment)
    /* Create a New Segment */
    p = vdrw_new_segment( 0, 0 );
  else
  { /* Create A Graphic Directive */
    if (!draw_curseg)
    { /* Allocate a new segment when no current segment defined */
      q = vdrw_new_segment( 0, 0 );
      draw_curseg = q;
    }
    else
      q = draw_curseg;
  
    /* Allocate the Requested Draw node */
    p = (Draw_Ptr) malloc( Draw_node_sizes[ndty] );

    /* Initialize the common part */
    p->seg.dcom.dir_nxt = NULL;
    p->seg.dcom.dir_prv = q->seg.seg_ldir;
    p->seg.dcom.dir_knd = ndty;
    p->seg.dcom.dir_pid = draw_curr_pid; /* Set the Pick Identifier */

    /* Link the directive in the Current Segment Directive List */
    if (q->seg.seg_ldir == NULL) q->seg.seg_fdir = p;
                            else q->seg.seg_ldir->seg.dcom.dir_nxt = p;
    q->seg.seg_ldir = p;
  }
  return p;
}





void vdrw_zero_seg( Draw_Ptr p )
{ /* Zero (or Re-Init) a segment */
  Draw_Ptr q, r;
  int i, j;
  
  if (p != NULL)
  {
    q = p->seg.seg_fdir;
    while (q != NULL) {
      r = q;
      q = q->seg.dcom.dir_nxt;
      vdrw_destroye( r );
    }
    /* The segment is now empty */
    p->seg.seg_fdir = NULL;
    p->seg.seg_ldir = NULL;
    /* The transformation matrix is reset to identity */
    for (i=0;i<3,i++;)
      for (j=0;j<4,j++;)
        p->seg.seg_mat[i][j] = (float) (i == j)?1.0:0.0;
  }
}



void vdrw_destroye( Draw_Ptr p )
{ /* Destroye and free any Draw Directive location */
  Draw_Ptr q, r;
  int        idx;

  if (p != NULL) {
    if (p->seg.dcom.dir_prv != NULL)
      /* Unlink from the previous directive */
      p->seg.dcom.dir_prv->seg.dcom.dir_nxt = p->seg.dcom.dir_nxt;
    else
      if (p->seg.dcom.dir_knd != Draw_Segment)
        draw_curseg->seg.seg_fdir = p->seg.dcom.dir_nxt;
    if (p->seg.dcom.dir_nxt != NULL)
      /* Unlink from the next directive */
      p->seg.dcom.dir_nxt->seg.dcom.dir_prv = p->seg.dcom.dir_prv;
    else
      if (p->seg.dcom.dir_knd != Draw_Segment)
        draw_curseg->seg.seg_ldir = p->seg.dcom.dir_prv;

    /* Perform the specific detroye tasks */
    switch (p->seg.dcom.dir_knd) {
      case Draw_Segment:
        /* Unlink the Segment from the segment list */
        if (p->seg.seg_prv = NULL)
          draw_fsegm = p->seg.seg_nxt;
        else
          p->seg.seg_prv->seg.seg_nxt = p->seg.seg_nxt;
        if (p->seg.seg_nxt = NULL)
          draw_lsegm = p->seg.seg_prv;
        else
          p->seg.seg_nxt->seg.seg_prv = p->seg.seg_prv;
        /* Take off the segment from the Segment cache when present */
        idx = p->seg.seg_ide%SEG_CACHE_SIZE;
        if (draw_sgtb[idx].seg_idnt == p->seg.seg_ide)
        { draw_sgtb[idx].seg_ref  = NULL;
          draw_sgtb[idx].seg_idnt = 0;
        }
        draw_curseg = NULL; /* Unselect the current segment */

        /* Destroye all internal directives */
        q = p->seg.seg_fdir;
        while (q != NULL) {
          r = q;
          q = q->seg.dcom.dir_nxt;
          draw_curseg = p;
          vdrw_destroye( r );
        }
      break;

      case Draw_Clip_Win:
        if (p->clip.clip_win)
        { /* It is a Clip enable directive */
          free( p->clip.clip_win ); /* Free the window def. */
          q = draw_curclip;
          r = NULL;
          while ((q != p)&&(q))
          { r = q;
            q = q->clip.clip_prv;
          }
          if (q == p)
          { /* Suppress the clip of active clip stack */
            if (r) r->clip.clip_prv = p->clip.clip_prv;
                  else draw_curclip = p->clip.clip_prv;
          }
        }
      break;
         
      case Draw_Polyline:
      case Draw_Polymarker:
      case Draw_Fill:
        if (p->tab.pol_tab != NULL) free( p->tab.pol_tab );
      break;

      case Draw_Text:
        if (p->txt.txt_str != NULL) free( p->txt.txt_str );
      break;
      
      case Draw_Circle:
      case Draw_Ellipse:
      break;
      
      case Draw_Line_Attr:
      case Draw_Marker_Attr:
      case Draw_Fill_Attr:
      case Draw_Seg_Attr:
      break;
    }
    free( p );
  }
}


void vdrw_draw_clear()
{ /* Free all Drawing structure for a new picture */
  while (draw_fsegm) vdrw_destroye( draw_fsegm );
  draw_fsegm     = NULL;
  draw_lsegm     = NULL;
  draw_curclip   = NULL;
  draw_winclip   = NULL;
  draw_curseg    = NULL;
  orgx           = 0.0; /* Set the Origine for External Contour. */
  orgy           = 0.0;
  cx_pen         = 0.0;
  cy_pen         = 0.0;
  Plot_Size      = 0;   /* State is Pen Up */
  draw_seg_count = 0;
}


