/***** DRAW Interface ROUTINES for Graphic Server *******/
/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               D R A W   -   S E R V E R   M O D U L E                 *
*                                                                       *
*           (Draw interface for V function and procedures)              *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/

/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/



/* include the environment SERVER */

#include <draw/sdrw_env.h>
#include <draw/sdrw_routines.h>


#include "Draw_VGL.h"
#include "Draw_AxisBox.h"




/**************************************************************************/
/*                                                                        */
/*                     DRAW Server Local Variables                        */
/*                                                                        */
/**************************************************************************/


typedef float * fptr;
typedef float v3d[3];

int Draw_node_sizes[] = { sizeof( Draw_Segm_Dir ),   /* Node Allocation ... */
                          sizeof( Draw_Plot_Dir ),   /* ... Size Table */
                          sizeof( Draw_Text_Dir ),
                          sizeof( Draw_Circle_Dir ),
                          sizeof( Draw_Sphere_Dir ),
                          sizeof( Draw_Cylinder_Dir ),
                          sizeof( Draw_Disk_Dir ),
                          sizeof( Draw_LAttr_Dir ),
                          sizeof( Draw_MAttr_Dir ),
                          sizeof( Draw_FAttr_Dir ),
                          sizeof( Draw_TAttr_Dir ),
                          sizeof( Draw_Font_Dir ),
                          sizeof( Draw_Color_Dir ),
                          sizeof( Draw_PPAttr_Dir ),
                          sizeof( Draw_EPlane_Dir ),
                          sizeof( Draw_DPlane_Dir ),
                          sizeof( Draw_Boxref_Dir ),
                          sizeof( Draw_Popbxr_Dir ),
                          sizeof( Draw_ExecSeg_Dir ),
                          sizeof( Draw_PushMat_Dir ),
                          sizeof( Draw_PopMat_Dir ),
                          sizeof( Draw_Blend_Dir ),
                          sizeof( Draw_LgtDf_Dir ),
                          sizeof( Draw_MLPrp_Dir )
                        };





/**************************************************************************/
/*                                                                        */
/*                    DRAW Server Interface Routines                      */
/*                                                                        */
/**************************************************************************/





Draw_Ptr Draw_Locate_Sys_Seg( int idseg )
{ /* Locate a System Segment by Identifier Number */
  Draw_Ptr p, q;

  p = NULL;
  q = draw_fsysseg;
  while (q && (!p)) 
  { /* Search the specified segment */
    if (q->seg.seg_ide == idseg) p = q;
                            else q = q->seg.seg_nxt;
  }
  return p;
}




Draw_Ptr Draw_New_Sys_Seg( int idseg, int bapd )
/* idseg must be the segment number required */
{ Draw_Ptr p, q;
  int         i;

  draw_usrseg = draw_curseg; /* Save the Current User Segment Ref. */
  p = Draw_Locate_Sys_Seg( idseg );
  if (p)
  { /* A segment with this ident is existing */
    if (!bapd) /* Clear/update mode */
    { /* We clear the segment primitive */
      draw_curseg = p;
      Draw_Zero_Seg( p );
    }
  }
  else
  { /* Create a New System Segment */
    p = (Draw_Ptr) malloc( Draw_node_sizes[Draw_Segment] );
    /* Initialize it */
    p->seg.dcom.dir_nxt = NULL; /* Never in Directive list => Always NULL */
    p->seg.dcom.dir_prv = NULL;
    p->seg.dcom.dir_knd = Draw_Segment;
    p->seg.dcom.dir_pid = 0;
    draw_curr_pid = 3; /* Detectable with pick_id = 1 */
    p->seg.seg_ide  = idseg;
    p->seg.seg_fdir = NULL;
    p->seg.seg_ldir = NULL;
    p->seg.seg_ref  = NULL;
    p->seg.seg_stat = DRWSEG_SYSTEM|DRWSEG_VISIBLE;
    p->seg.seg_pri  = 0.5;
    for (i = 0; i < 16; i++)
      p->seg.seg_mat[i] = (GLfloat) Drwgl_mat_unit[i];
    /* Link to the segment list */
    p->seg.seg_prv  = draw_lsysseg;
    p->seg.seg_nxt  = NULL;
    if (draw_lsysseg) draw_lsysseg->seg.seg_nxt = p;
                 else draw_fsysseg = p;
    draw_lsysseg = p;
  }
  draw_curseg = p;
  return p;
}




Draw_Ptr Draw_Locate_Seg( int idseg )
{ /* Locate a Segment by Identifier Number */
  int idx;
  Draw_Ptr p;

  if (idseg > 0 && idseg <= draw_seg_count) {
    idx = idseg%SEG_CACHE_SIZE;
    if (draw_sgtb[idx].ref_idnt != idseg) {
      p = draw_fsegm;
      while (p&&(p->seg.seg_ide != idseg))
        p = p->seg.seg_nxt;
      if (p) {
        draw_sgtb[idx].ref_ref  =     p;
        draw_sgtb[idx].ref_idnt = idseg;
      }
    }
    else
      p = draw_sgtb[idx].ref_ref;
  }
  else
    p = NULL;
  return p;
}




Draw_Ptr Draw_New_Segment( int idseg, int bapd )
{ Draw_Ptr p, q;
  int         i;

  if (idseg <= 0) idseg = draw_seg_count + 1;
//fprintf( fmsg, "  Create Seg. # %d (max =) %d) in mode %d.\n",
//               idseg, draw_seg_count, bapd );
//Draw_Fmsgupdate();
  if (idseg <= draw_seg_count)
  { p = Draw_Locate_Seg( idseg );
    if (p)
    { /* A segment with this ident is existing */
//    fprintf( fmsg, "  Seg is already existing => Clear it.\n" );
//    Draw_Fmsgupdate();
      if (!bapd) /* Clear/append mode */
      { /* We clear the segment primitive */
        draw_curseg = p;
        Draw_Zero_Seg( p );
      }
    }
  }
  else
    p = NULL;

  if (!p)
  { /* Create a new Segment */
//  fprintf( fmsg, "  Create New Seg.\n" );
//  Draw_Fmsgupdate();
    p = (Draw_Ptr) malloc( Draw_node_sizes[Draw_Segment] );
    /* Initialize it */
    p->seg.dcom.dir_nxt = NULL; /* Never in Directive list => Always NULL */
    p->seg.dcom.dir_prv = NULL;
    p->seg.dcom.dir_knd = Draw_Segment;
    p->seg.dcom.dir_pid = draw_curr_pid;
    p->seg.seg_ide  = idseg;
    p->seg.seg_fdir = NULL;
    p->seg.seg_ldir = NULL;
    p->seg.seg_ref  = NULL;
    p->seg.seg_stat = DRWSEG_VISIBLE; /* Default is Visible */
    p->seg.seg_pri  = 0.5;
    for (i = 0; i < 16; i++)
      p->seg.seg_mat[i] = (GLfloat) Drwgl_mat_unit[i];
    p->seg.seg_prv  = draw_lsegm;
    p->seg.seg_nxt  = NULL;
    if (draw_seg_count < idseg) draw_seg_count = idseg;
    /* Link it in the segment list */
    if (draw_fsegm) draw_lsegm->seg.seg_nxt = p;
               else draw_fsegm = p;
    draw_lsegm = p;
    /* Put it in the Segment Cache */
    i = idseg%SEG_CACHE_SIZE;
    draw_sgtb[i].ref_ref  = p;
    draw_sgtb[i].ref_idnt = idseg;
//  fprintf( fmsg, "  Create New Seg # %d.\n", idseg );
//  Draw_Fmsgupdate();
  }
  draw_curseg = p;
  return p;
}




Draw_Ptr Draw_New_Node( Draw_Dir_Types ndty )
{ Draw_Ptr p, q;
  int i, j;

  if (ndty == Draw_Segment)
    /* Create a New Segment */
    p = Draw_New_Segment( 0, 0 );
  else
  { /* Create A Graphic Directive */
    if (!draw_curseg)
    { /* Allocate a new segment when no current segment defined */
      q = Draw_New_Segment( 0, 0 );
      draw_curseg = q;
    }
    else
      q = draw_curseg;

    /* Allocate the Requested Draw node */
    p = (Draw_Ptr) malloc( Draw_node_sizes[ndty] );

    /* Initialize the common part */
    p->seg.dcom.dir_nxt = NULL;
    p->seg.dcom.dir_prv = q->seg.seg_ldir;
    p->seg.dcom.dir_knd = ndty;
    p->seg.dcom.dir_pid = draw_curr_pid*2 + /* Set the Pick Identifier */
                          draw_curr_detf;   /* ... and Detect Pick Flag */

    /* Link the directive in the Current Segment Directive List */
    if (q->seg.seg_ldir) q->seg.seg_ldir->seg.dcom.dir_nxt = p;
                    else q->seg.seg_fdir = p;
    q->seg.seg_ldir = p;
  }
//fprintf( fmsg, "  Create Node with pid = %d.\n", p->seg.dcom.dir_pid );
//Draw_Fmsgupdate();

  return p;
}



int Draw_Gen_TMat( int bpush )
{ Draw_Ptr p,  q;
  int      i, id;

  if (!bpush) {
    p = Draw_New_Node( Draw_PopTrfMat );
    return -1;
  }

  /* Push a New Matrix */
  p = Draw_New_Node( Draw_PushTrfMat );
  if (p) {
    p->tmat.trsf_ide = ++draw_trf_count;
    for (i = 0; i < 16; i++)
      p->tmat.trsf_mat[i] = (GLfloat) Drwgl_mat_unit[i];         

    /* Set all Links */
    p->tmat.trsf_nxt = NULL;
    if (draw_ltrmat) p->tmat.trsf_prv = draw_ltrmat;
                else { draw_ftrmat = p; p->tmat.trsf_prv = NULL; }
    draw_ltrmat = p;
    /* Put it in the Transformation Cache */
    i = draw_trf_count%TRF_CACHE_SIZE;
    draw_trtb[i].ref_ref  = p;
    draw_trtb[i].ref_idnt = draw_trf_count;
  }
  return draw_trf_count;
}



void Draw_Zero_Seg( Draw_Ptr p )
{ /* Zero (or Re-Init) a segment */
  Draw_Ptr q, r;
  int         i;

  if (p != NULL)
  {
//  fprintf( fmsg, " Zero the Seg # %d.\n", p->seg.seg_ide );
//  Draw_Fmsgupdate();
    q = p->seg.seg_fdir;
    while (q) {
      r = q;
      q = q->seg.dcom.dir_nxt;
      Draw_Destroye( r );
    }
    /* The segment is now empty */
    p->seg.seg_fdir = NULL;
    p->seg.seg_ldir = NULL;
    /* The transformation matrix is reset to identity */
    for (i = 0; i < 16; i++)
      p->seg.seg_mat[i] = (GLfloat) Drwgl_mat_unit[i];
  }
}




void Draw_Destroye( Draw_Ptr p )
{ /* Destroye and free any Draw Directive location */
  /* The directive to destroye must be in the current segment */
  Draw_Ptr       q,  r;
  int              idx;

  if (p) {
    if (p->seg.dcom.dir_prv)
      /* Unlink from the previous directive */
      p->seg.dcom.dir_prv->seg.dcom.dir_nxt = p->seg.dcom.dir_nxt;
    else
      if (p->seg.dcom.dir_knd != Draw_Segment)
        draw_curseg->seg.seg_fdir = p->seg.dcom.dir_nxt;

    if (p->seg.dcom.dir_nxt)
      /* Unlink from the next directive */
      p->seg.dcom.dir_nxt->seg.dcom.dir_prv = p->seg.dcom.dir_prv;
    else
      if (p->seg.dcom.dir_knd != Draw_Segment)
        draw_curseg->seg.seg_ldir = p->seg.dcom.dir_prv;

    /* Perform the specific detroye tasks */
    switch (p->seg.dcom.dir_knd) {
      case Draw_Segment:
//      fprintf( fmsg, " Destroye the Seg # %d.\n", p->seg.seg_ide );
//      Draw_Fmsgupdate();
        if (p->seg.seg_stat&DRWSEG_SYSTEM)
        { /* Destroye a System Segment */
          draw_curseg = draw_usrseg; /* Restore User Segment Context */
          if (p->seg.seg_prv) p->seg.seg_prv->seg.seg_nxt = p->seg.seg_nxt;
                         else draw_fsysseg = p->seg.seg_nxt;

          if (p->seg.seg_nxt) p->seg.seg_nxt->seg.seg_prv = p->seg.seg_prv;
                         else draw_lsysseg = p->seg.seg_prv;
        }
        else
        { /* Unlink the Segment from the segment list */
          if (p->seg.seg_prv) p->seg.seg_prv->seg.seg_nxt = p->seg.seg_nxt;
                         else draw_fsegm = p->seg.seg_nxt;

          if (p->seg.seg_nxt) p->seg.seg_nxt->seg.seg_prv = p->seg.seg_prv;
                         else draw_lsegm = p->seg.seg_prv;

          /* Take off the segment from the Segment cache when present */
          idx = p->seg.seg_ide%SEG_CACHE_SIZE;
          if (draw_sgtb[idx].ref_idnt == p->seg.seg_ide)
          { draw_sgtb[idx].ref_ref  = NULL;
            draw_sgtb[idx].ref_idnt = 0;
          }
        }
        /* Destroye any exec reference */
//      fprintf( fmsg, " Delete Seg. # %d Exec Ref.\n", p->seg.seg_id );
//      Draw_Fmsgupdate();
        q = p->seg.seg_ref;
        while (q) {
          r = q;
          q = q->exeseg.exec_lst;
          Draw_Destroye( r );
        }
        /* Destroye all internal directives */
        q = p->seg.seg_fdir;
        while (q) {
          r = q;
          q = q->seg.dcom.dir_nxt;
          draw_curseg = p;
          Draw_Destroye( r );
        }
        draw_curseg = NULL; /* Unselect the current segment */
      break;

      case Draw_Plot:
        if (p->tab.plo_tab) free( p->tab.plo_tab );
      break;

      case Draw_Text:
        if (p->txt.txt_str) free( p->txt.txt_str );
      break;

      case Draw_Exec_Seg:
        if (p->exeseg.exec_lst) {
          q = p->exeseg.exec_seg->seg.seg_ref;
          r = NULL;
          while (q&&(q!=p)) {
            r = q; q = q->exeseg.exec_lst;
          }
          if (q) /* Unlink from the segment list */
            if (r) r->exeseg.exec_lst = p->exeseg.exec_lst;
              else p->exeseg.exec_seg->seg.seg_ref = p->exeseg.exec_lst;
        }
      break;

      case Draw_PopTrfMat:
        if (p->tmat.trsf_mat) free( p->tmat.trsf_mat );
        if (p->tmat.trsf_prv) p->tmat.trsf_prv->tmat.trsf_nxt = p->tmat.trsf_nxt;
                           else draw_ftrmat = p->tmat.trsf_nxt;
        if (p->tmat.trsf_nxt) p->tmat.trsf_nxt->tmat.trsf_prv = p->tmat.trsf_prv;
                           else draw_ltrmat = p->tmat.trsf_prv;
        idx = p->tmat.trsf_ide%TRF_CACHE_SIZE;
        if (draw_trtb[idx].ref_idnt == p->tmat.trsf_ide)
        { draw_trtb[idx].ref_ref  = NULL;
          draw_trtb[idx].ref_idnt = 0;
        }
      break;

      default: ;
     }
    free( p );
  }
}




void Draw_Draw_Clear()
{ /* Free all Drawing structure for a new picture */
  int i;

  while (draw_fsysseg) Draw_Destroye( draw_fsysseg );
  while (draw_fsegm) Draw_Destroye( draw_fsegm );
  draw_fsegm       = NULL;
  draw_lsegm       = NULL;
  draw_fsysseg     = NULL;
  draw_lsysseg     = NULL;
  draw_curseg      = NULL;
  Draw_first_font  = NULL; /* Init the Loaded Font List */
  Draw_last_font   = NULL;
  Draw_Load_Font( 1 );     /* load the Basic Standard Font */
  org_orgx         = 0.0;  /* Set the Origine for External Contour. */
  org_orgy         = 0.0;
  org_orgz         = 0.0;
  orgx             = 0.0;
  orgy             = 0.0;
  orgz             = 0.0;
  cx_pen           = 0.0;
  cy_pen           = 0.0;
  cz_pen           = 0.0;
  Plot_Size        = 0;   /* State is Pen Up */
  Plot_Index       = 0;
  Plot_Flag        = 0;   /* No Large Plot in process */
  draw_seg_count   = 0;
  draw_ftrmat      = NULL;
  draw_ltrmat      = NULL;
  draw_trf_count   = 0;
  draw_out_mode    = 1;
  draw_mat_cnt     = 0;

  /* Re-Init The segment and matrix caches to empty */
  for (i=0;i<SEG_CACHE_SIZE;i++) {
    draw_sgtb[i].ref_ref  = NULL;
    draw_sgtb[i].ref_idnt =    0;
  }
  for (i=0;i<TRF_CACHE_SIZE;i++) {
    draw_trtb[i].ref_ref  = NULL;
    draw_trtb[i].ref_idnt =    0;
  }

  Drwgl_clipflg    = 0;   /* Force the clipping computing */
}




Draw_Ptr Draw_User_Append( Draw_Ptr p )
{ /* To Append the directive in the segment p to User List */
  Draw_Ptr q, r;

  if (p) { /* Use the current User Segment */
    r = p->seg.seg_fdir;
    if (Drwgl_gptins) {
      if (draw_usrseg) {
        draw_curseg = q = draw_usrseg;
        if (q->seg.seg_fdir) q->seg.seg_ldir->seg.dcom.dir_nxt = p->seg.seg_fdir;
                        else q->seg.seg_fdir = p->seg.seg_fdir;
      }  
      else { /* Create a new User segment when now segment was opened */
        draw_usrseg = q = Draw_New_Segment( 0, 0 );
        q->seg.seg_fdir = p->seg.seg_fdir;
      }
      q->seg.seg_ldir = p->seg.seg_ldir;
    }
    p->seg.seg_fdir = NULL;
    p->seg.seg_ldir = NULL;
    Draw_Destroye( p );
  }
  else r = NULL;

  return r;
}




Draw_Ptr Draw_Locate_TMat( int id )
{ /* Locate a Transformation ref. by Identifier Number */
  int      idx;
  Draw_Ptr   p;

  idx = id%TRF_CACHE_SIZE;
  if (draw_trtb[idx].ref_idnt != id)
  { p = draw_ftrmat;
    while (p&&(p->tmat.trsf_ide != id))
      p = p->tmat.trsf_nxt;
    if (p)
    { draw_trtb[idx].ref_ref  = p;
      draw_trtb[idx].ref_idnt = id;
    }
  }
  else
    p = draw_trtb[idx].ref_ref;
  return p;
}



static int Draw_Get_Point( fptr x, fptr y, fptr z )
{
  static Draw_Ptr    p = NULL;
  static int      pcnt =   -1;
  static int       f3D =    0;
  static int      lent =    0;

  if ((pcnt < 0)&&Drwgl_senddir) { /* For the First Call on one directive */
    p = Drwgl_senddir;
    while (p&&p->seg.dcom.dir_knd != Draw_Plot) {
      p = p->seg.dcom.dir_nxt;
      if (!Drwgl_gptins) free( Drwgl_senddir );
      Drwgl_senddir = p;
    }
    if (p) { /* We have locate a Plot Directive */
      f3D  = (p->tab.plo_flg&4); /* Advance for 3D/2D */
      lent =  p->tab.plo_npt*((f3D)?3:2);
      pcnt = 0;
    }
  }
  if (pcnt >= 0) {
    *x = p->tab.plo_tab[pcnt++]; /* Set current point x, y (, z) */
    *y = p->tab.plo_tab[pcnt++];
    if (f3D) *z = p->tab.plo_tab[pcnt++];
    if (pcnt >= lent) {
      pcnt = -1;
      p = p->seg.dcom.dir_nxt;
      /* We must destroye the Directive when not already Used */
      if (!Drwgl_gptins) free( Drwgl_senddir );
      Drwgl_senddir = p;
    }
  }
  /* When the Plot Directive List was empty */
  return f3D;
}



  
void Draw_Send_Points( int flg3D )
{ /* To send the Plot points to the User task from the list of the segment p */
  Draw_Ptr q;
  int bfcp, cnt, dim, f3D, i, kpck;
  float x, y, z, xc, yc, zc;

  bfcp = (BUF_SIZE - 2*sizeof( int ))/(2*sizeof( float ));

  if (Drwgl_senddir) {
    if (Drwgl_gptcnt > bfcp) { dim =         bfcp; kpck = 0; }
                        else { dim = Drwgl_gptcnt; kpck = 1; }
    if (flg3D) kpck |= 4;
    Sdrw_Put_Int( kpck );
    Sdrw_Put_Int( dim );
    for (i = 0; i < dim; i++) {
      f3D = Draw_Get_Point( &x, &y, &z );
      if (Draw_Opened_Box) {
        Draw_UnScale( &xc, &yc, &zc, x, y, z );
      } else {
        xc = x - orgx; yc = y - orgy; zc = z - orgz;
      }
      Sdrw_Put_Float( xc );
      Sdrw_Put_Float( yc );
      if (flg3D)
        if (f3D) Sdrw_Put_Float( zc );
            else Sdrw_Put_Float( 0.0 );
    }
    Drwgl_gptcnt -= dim;
  }
}




void Draw_Gen_Box_Ref( int bcreate )
{ /* Generate a Clip Box Reference On/Off condition */
  /* Get the data directly from the pipe */
  Draw_Ptr p, q;
  int i;

  if (bcreate) {
    p = Draw_New_Node( Draw_Box_Ref );

    p->boxref.bxr_prev = NULL;
    for (i = 0; i < 24; i++) p->boxref.bxr_lim[i] = Sdrw_Get_Float();
    for (i = 0; i <  3; i++) p->boxref.bxr_mod[i] = Sdrw_Get_Char();
  }
  else
    p = Draw_New_Node( Draw_Box_Pop );  
}



static int eq_vertex( float * v1, float * v2 )
{
  int i;

  for (i=0;i<3;i++)
    if (fabs( v1[i] - v2[i] ) < 1.0e-5 ) return 0;
  return 1;
}



static void nrm_prod( float * v1, float * v2, float * vp )
/* To Check the identity of two vertices */
{
  float nrm;

  vp[0] = v1[1]*v2[2] - v1[2]*v2[1]; nrm  = vp[0]*vp[0];
  vp[1] = v1[2]*v2[0] - v1[0]*v2[2]; nrm += vp[1]*vp[1];
  vp[2] = v1[0]*v2[1] - v1[1]*v2[0]; nrm += vp[2]*vp[2];
  if (nrm>1e-5) {
    nrm = sqrt( nrm );
    vp[0] /= nrm; vp[1] /= nrm; vp[2] /= nrm;
  }
}



void Draw_Gen_Normal( float * q, int cod, int dim )
/* To Generate the Normal from the points coordinates.
*/
{
  int  i, j, k, pa, pb, pc, pd, pe, pf, ne, np, siz;
  float rv, rv1, rv2;
  v3d vp, v1, v2, v3;

  siz = dim*6;
  pf = siz;
  pa = pf/2;
  /* Create the place for Normal vectors */
  while (pf >= 6) {
    q[--pf] = 0.0;             /* Set the Normal vector to [0.0,0.0,1.0] */
    q[--pf] = 0.0;
    q[--pf] = 1.0;
    q[--pf] = q[--pa];         /* Copy the vertex coordinates */
    q[--pf] = q[--pa];
    q[--pf] = q[--pa];
  }
  /* Specific code for each GL Kind of PLOT */
  pa = 0;
  switch (cod) {
    case -4: /* GL_TRIANGLES */
      /* Each triangle has one independant normal */
      ne = dim/3;              /* Number of triangles*/
      if (ne<0) break;
      for (k=0;k<ne;k++) {
        /* pa is the index of first vertex of the current triangle */
        pb = pa + 6;           /* Set pb as second vertex index */
        pc = pb + 6;           /* Set pc as the third vertex index */
        for (i=0;i<3;i++) {
          v1[i] = q[pb+i] - q[pa+i];
          v2[i] = q[pc+i] - q[pb+i];
        }
        nrm_prod( v1, v2, vp );
        /* The following line increment pa and store the normal vector for each vertex */
        for (j=0;j<3;j++) { pa +=3; for (i=0;i<3;i++) q[pa++] = vp[i]; }
      }
      break;

    case -6: /* GL_TRIANGLE_FAN */
      ne = dim - 3;      /* Number of triangles*/
      if (ne<=0) break;
      /* pa is the pivot vertex index of triangle fan */
      pb =  6; pc = 12; pd = 18;
      for (i=0;i<3;i++) {
        v1[i] = q[i+pb] - q[i+pa];
        v2[i] = q[i+pc] - q[i+pb];
      }
      nrm_prod( v1, v2, &q[pb] );
      /* Init the Average of normal for the Fan pivot */
      for (i=0;i<3;i++) vp[i] = q[pb+i];
      for (k=0;k<ne;k++) {     /* Loop on each triangle */
        for (i=0;i<3;i++) {
          v1[i] = q[pa+i] - q[pc+i];
          v2[i] = q[pd+i] - q[pb+i];
        }
        nrm_prod( v1, v2, &(q[pc+3]) );
        for (i=0;i<3;i++) vp[i] += q[pc+3+i];
        pb = pc; pc = pd; pd +=6;
      }
      /* Finish Fan with the pivot normal */
      for (i=0;i<3;i++) {
        v1[i] = q[pc+i] - q[pb+i];
        v2[i] = q[pa+i] - q[pc+i];
      }
      nrm_prod( v1, v2, &(q[pb+3]) );
      for (i=0;i<3;i++) vp[i] += q[pb+3+i];
      rv = sqrt( vp[0]*vp[0] + vp[1]*vp[1] + vp[2]*vp[2] );
      if (rv>1.0e-5) for (i=0;i<3;i++) q[pa+i] = vp[i]/rv;
      break;

    case -7: /* GL_QUADS */
      /* Each quad has one independant normal */
      ne = dim/4;              /* Number of quads */
      if (ne<0) break;
      for (k=0;k<ne;k++) {
        /* pa is the index of first vertex of the current quad */
        pb = pa + 6; pc = pb + 6; pd = pc + 6;
        for (i=0;i<3;i++) {
          v1[i] = q[pc+i] - q[pa+i];
          v2[i] = q[pd+i] - q[pb+i];
        }
        nrm_prod( v1, v2, vp );
        /* The following line increment pa and store the normal vector for each vertex */
        for (j=0;j<4;j++) { pa +=3; for (i=0;i<3;i++) q[pa++] = vp[i]; }
      }
      break;

    case -5: /* GL_TRIANGLE_STRIP */
    case -8: /* GL_QUAD_STRIP */
      ne = (dim - 4)/2;        /* Number of intermediary rung ladder */
      if (ne<=0) break;
      /* Set the initial rung of vertex ladder indices (pc,pd), (pe,pf) */
      pb =  6;                 /* First (pa,pb), ... */
      pc = 12; pd = 18;        /* ... second (pc,pd) ... */
      pe = 24; pf = 30;        /* ... and third (pe,pf) rung of vertex ladder */
      for (i=0;i<3;i++) {
        v1[i] = q[pb+i] - q[pa+i];
        v2[i] = q[pc+i] - q[pa+i];
        v3[i] = q[pd+i] - q[pb+i];
      }
      /* The following line increment pa and store the normal vectors for each vertex */
      nrm_prod( v1, v2, vp );
      pa +=3; for (i=0;i<3;i++) q[pa++] = vp[i]; /* Set first normal of first column vertex */
      nrm_prod( v1, v3, vp );
      pa +=3; for (i=0;i<3;i++) q[pa++] = vp[i]; /* Set first normal of second column vertex */
      for (k=0;k<ne;k++) {
        for (i=0;i<3;i++) {
          v1[i] = q[pd+i] - q[pc+i];
          v2[i] = q[pe+i] - q[pa+i];
          v3[i] = q[pf+i] - q[pb+i];
        }
        nrm_prod( v1, v2, vp );
        pa +=3; for (i=0;i<3;i++) q[pa++] = vp[i]; /* Set k-th normal of first column vertex */
        nrm_prod( v1, v3, vp );
        pa +=3; for (i=0;i<3;i++) q[pa++] = vp[i]; /* Set k-th normal of second column vertex */
        pb = pd; pc = pe; pd = pf;
        pe += 12; pf +=12;
      }
      for (i=0;i<3;i++) {
        v1[i] = q[pf+i] - q[pe+i];
        v2[i] = q[pe+i] - q[pc+i];
        v3[i] = q[pf+i] - q[pd+i];
      }
      nrm_prod( v1, v2, vp );
      pa +=3; for (i=0;i<3;i++) q[pa++] = vp[i]; /* Set last normal of first column vertex */
      nrm_prod( v1, v3, vp );
      pa +=3; for (i=0;i<3;i++) q[pa++] = vp[i]; /* Set last normal of second column vertex */
      break;

    case -9: /* GL_POLYGON */
      /* Complete Management here (the Polygone must be convex) */
      if (dim<3) break;
      ne = dim*6;
      for (i=0;i<3;i++) v3[i] = 0.0;
      pb =  6; pc = 12;
      for (k=0;k<dim;k++) {
        for (i=0;i<3;i++) {
          v1[i] = q[pb+i] - q[pa+i];
          v2[i] = q[pc+i] - q[pb+i];
        }
        nrm_prod( v1, v2, vp );
        for (i=0;i<3;i++) v3[i] += vp[i];
        pa = pb; pb = pc;
        pc += 6; if (pc >= ne) pc -= ne;
      }
      rv = sqrt( v3[0]*v3[0] +  v3[1]*v3[1] +  v3[2]*v3[2] );
      if (rv>1.0e-5) for (i=0;i<3;i++) v3[i] /= rv;
      pa = 0;
      for (k=0;k<dim;k++) { pa +=3; for (i=0;i<3;i++) q[pa++] = v3[i]; }
      break;

    default:
      break;
  }
}



void Draw_Gen_Mplot( int dim, int flg, fptr pt )
{ /* Generate a General Plot Request */
  Draw_Ptr     p;
  fptr         q;
  int   i, n, nm;

//fprintf( fmsg, "  Create Mplot %d, %d.\n", dim, flg );
//Draw_Fmsgupdate();

  /* Elliminate Normal flag when not appropriate */
  if (((draw_out_mode > -4)&&(draw_out_mode < 3))||!(flg&cdf_3D)) flg = flg&~(cdf_NRM|cdf_UNRM);

  if (flg&cdf_3D) { /* 3D */
    draw_3D_flg = 1;
    n = dim*3; nm = n;
    if ((!pt)&&(draw_out_mode > 3)) flg |= cdf_NRM;
    if (flg&(cdf_NRM|cdf_UNRM)) {
      nm *= 2;
      if (flg&cdf_UNRM) { flg |= cdf_NRM; n = nm; }
    }
  }
  else
    nm = n = dim*2; /* 2D */

  p = Draw_New_Node( Draw_Plot );
  p->tab.plo_knd = draw_out_mode;
  p->tab.plo_flg = flg;
  p->tab.plo_npt = dim;

  /* Allocate the required 3D plot Table */
  q = (float *) malloc( nm*sizeof( float ) );
  p->tab.plo_tab =   q;
  if (pt) /* Internal GPLOT or Surface */
    for (i = 0; i < n; i++) q[i] = pt[i];  
  else {
    Sdrw_Read_Block( (Char *) q, n*sizeof( float ) );
    if (flg&(cdf_NRM|cdf_UNRM) == cdf_NRM)
      Draw_Gen_Normal( p->tab.plo_tab, draw_out_mode,  dim );
    if (!((flg&cdf_AB)||draw_mat_cnt))
    { /* For Plot relative to current origine */
      i = 0;
      if (flg&cdf_3D) /* 3D */
        while (i < nm) {
          if (flg&cdf_NRM) i += 3;  /* Skip the Normal Componante (given or not) */
          q[i++] += orgx;
          q[i++] += orgy;
          q[i++] += orgz;
        }
      else  /* 2D */
        while (i < nm) {
          q[i++] += orgx;
          q[i++] += orgy;
        }
    }
  }
}




void Draw__G_Surface( int nraw, int ncol, int flg, fptr buf )
{ /* Generate a triangle strip plot node with an automatic normal generation when required */
  Draw_Ptr p;


  int     i, ir, ic, szr, sz;
  float nm[3], u1v[3], u2v[3], v1v[3], v2v[3];
  float nrm;
  fptr blk, p0, pp1, pp2, psr, p1p, p1n, p1r, p2p, p2n, p2l;
  static float defnrm[3] = { 0.0, 0.0, 1.0 };

  /* Management of Normal Generation Flag */
  flg |= cdf_3DSTND;
  if (draw_out_mode > 3) flg |= cdf_NRM;
  else
    if ((draw_out_mode < 3)&&(draw_out_mode > -4)) flg = flg&~cdf_NRM;

  szr = 3*ncol;                    /* Set the size of a raw in float unit in the input buffer */
  sz = (6*sizeof( float ))*ncol;   /* Set the size of the triangle strip in bytes */
  if (flg&cdf_NRM) sz *= 2;  /* Take in account the normal vector when required */

  pp1 = buf;                       /* Set the run pointer #1 to first raw */
  pp2 = buf + szr;                 /* Set the run pointer #1 to second raw */  
  if (flg&cdf_NRM) {
    p1p = pp1; p2p = pp2;          /* Set the previous and next raw vertex pointer of first raw */
    p1n = pp1 + 3;                 /* Set the next raw vertex pointer of first raw */
    p2n = pp2 + 3;                 /* Set the next raw vertex pointer of second raw */
    p1r = pp1;                     /* Set the right raw vertex pointer of first raw */
    p2l = pp2 + szr;               /* Set the left raw vertex pointer of second raw */
  }

  ir = 2;
  do { /* ir=raw count, Loop on each triangle raw */
//  fprintf( fmsg, " Raw #%d/%d\n", ir, nraw );
//  Draw_Fmsgupdate();
    blk = (float*) malloc( sz );
    p0 = blk;
    ic = 1;
    do { /* ic=column count on a triangle raw */
      if (flg&cdf_NRM) {
        for (i=0; i<3; i++) { /* Build the Basic vector to compute normal vector */
          u1v[i] = *(p1n++) - p1p[i]; u2v[i] = *(p2n++) - p2p[i];
          v1v[i] = pp2[i] - *(p1r++); v2v[i] = *(p2l++) - pp1[i];
        }
        psr = pp1;                 /* Save the begin of right raw for future right vertex */
        p1p = pp1; p2p = pp2;      /*current pointers becomes future previous ones */
      }
      if (flg&cdf_NRM) { /* Build the related normal vector when required */
        nm[0] = u1v[1]*v1v[2] - u1v[2]*v1v[1];
        nm[1] = u1v[2]*v1v[0] - u1v[0]*v1v[2];
        nm[2] = u1v[0]*v1v[1] - u1v[1]*v1v[0];
        nrm = sqrt( nm[0]*nm[0] + nm[1]*nm[1] + nm[2]*nm[2] );
        if (nrm > 1.0e-8) for (i=0; i<3; i++) *(p0++) = nm[i]/nrm;
                     else for (i=0; i<3; i++) *(p0++) = defnrm[i];
        if (ic < 5) {
//        fprintf( fmsg, "  Norm1[%d]/nrm = (%f,%f,%f).\n", ic, p0[-3], p0[-2], p0[-1] );
//        Draw_Fmsgupdate();
        }
      }
      for (i=0; i<3; i++) *(p0++) = *(pp1++); /* Copy the first raw vertex */
      if (ic < 5) {
//      fprintf( fmsg, "  xyz1[%d] = (%f,%f,%f).\n", ic, p0[-3], p0[-2], p0[-1] );
//      Draw_Fmsgupdate();
      }
      if (flg&cdf_NRM) { /* Build the related normal vector when required */
        nm[0] = u2v[1]*v2v[2] - u2v[2]*v2v[1];
        nm[1] = u2v[2]*v2v[0] - u2v[0]*v2v[2];
        nm[2] = u2v[0]*v2v[1] - u2v[1]*v2v[0];
        nrm = sqrt( nm[0]*nm[0] + nm[1]*nm[1] + nm[2]*nm[2] );
        if (nrm > 1.0e-8) for (i=0; i<3; i++) *(p0++) = nm[i]/nrm;
                     else for (i=0; i<3; i++) *(p0++) = defnrm[i];
        if (ic < 5) {
//        fprintf( fmsg, "  Norm2[%d]/nrm = (%f,%f,%f).\n", ic, p0[-3], p0[-2], p0[-1] );
//        Draw_Fmsgupdate();
        }
      }
      for (i=0; i<3; i++) *(p0++) = *(pp2++); /* Copy the second raw vertex */
      if (ic < 5) {
//      fprintf( fmsg, "  xyz2[%d] = (%f,%f,%f).\n", ic, p0[-3], p0[-2], p0[-1] );
//      Draw_Fmsgupdate();
      }
      ic += 1;
      if ((Plot_Flag&cdf_NRM)&&(ic == ncol)) { p1n = pp1; p2n = pp2; } /* Special for last raw vertex */
    } while (ic <= ncol);

    /* Now Create the Display Node */
    p = Draw_New_Node( Draw_Plot );
    p->tab.plo_knd =     -5; /* TRIANGLE STRIP */
    p->tab.plo_flg =    flg;
    p->tab.plo_npt = 2*ncol;
    p->tab.plo_tab =    blk;

    ir += 1;
//  fprintf( fmsg, "  Triangle strip # %d with flags = %d, size = %d, start at (%f, %f, %f).\n", ir, flg, 2*ncol, blk[0], blk[1], blk[2] );
//  Draw_Fmsgupdate();
    /* Update Normal Computing Vector Pointers  */
    if (flg&cdf_NRM) {
      p1p = pp1; p2p = pp2;
      p1n = pp1 + 3; p2n = pp2 + 3;
      p1r = psr;
      if (ir == nraw) p2l = pp2; else p2l = pp2 + szr;
    }
  } while (ir <= nraw);
  free( buf );
}



void Draw_Gen_Surface( int nraw, int ncol, int flg )
{ /* To create node(s) to perform Surface Plot with normal generation */
  int    ij, ndim, size, sz, sr;
  fptr                   p, buf;

  ndim = abs( nraw*ncol );
  sr   = abs( ncol)*3;
  sz   = sr*sizeof( float );
  size = abs( nraw )*sz;
  buf  = (float*) malloc( size );
  if ((ncol < 0)||(nraw < 0)) {
    ncol = abs( ncol ); nraw = abs( nraw ); 
    sz = ncol*3;
    p = buf;
    for (ij = 0; ij < nraw; ij++ ) {
//    fprintf( fmsg, "  Read a Block st # %d/%d od %d bytes.\n", ij, nraw, sz );
//    Draw_Fmsgupdate();
      Sdrw_Read_Block( (Char*) p, sz );
//    fprintf( fmsg, "  Read a Block nd \n" );
//    Draw_Fmsgupdate();
      p += sz;
    }
  } else Sdrw_Read_Block( (Char*) buf, size );

  if ((ncol <= 1)&&(nraw<= 1)) {
    free( buf );
    return;
  }

//fprintf( fmsg, "  Surface OK to Set Origine when required.\n" );
//Draw_Fmsgupdate();
  if (!((Plot_Flag&cdf_AB)||draw_mat_cnt)) /* Applied the origine when required */
    p = buf;
    for (ij = 0; ij < ndim; ij++) {
      *(p++) += orgx;
      *(p++) += orgy;
      *(p++) += orgz;
    }

  Draw__G_Surface( nraw, ncol, flg, buf );
}




void Draw_Gen_Surface2( int nraw, int ncol, int flg, float xo, float yo, float xs, float ys )
{
  int    ij, ik, ndim, size, sz, sr;
  fptr                   p, p0, buf;
  float                  xx, yy, zz;

  ndim = abs( nraw*ncol );
  sr   = abs( ncol);
  sz   = sr*sizeof( float );
  size = abs( nraw )*sz;
  buf  = (float*) malloc( size*3 );
  p = buf + 2*ndim;                 /* Fill the top of buffer with all z coordinates */
  if ((ncol < 0)||(nraw < 0)) {
    ncol = abs( ncol ); nraw = abs( nraw ); 
    sz = ncol*3;
    for (ij = 0; ij < nraw; ij++ ) {
//    fprintf( fmsg, "  Read a Block st # %d/%d od %d bytes.\n", ij, nraw, sz );
//    Draw_Fmsgupdate();
      Sdrw_Read_Block( (Char*) p, sz );
//    fprintf( fmsg, "  Read a Block nd \n" );
//    Draw_Fmsgupdate();
      p += sz;
    }
  } else Sdrw_Read_Block( (Char*) p, size );
  if ((ncol <= 1)&&(nraw<= 1)) {
    free( buf );
    return;
  }

//fprintf( fmsg, "  Surface OK to Set Origine when required.\n" );
//Draw_Fmsgupdate();
  p = buf; p0 = buf + 2*ndim;
  for (ij = 0; ij< nraw; ij++ ) {
    yy = yo + ij*ys;
    for (ik = 0; ik < ncol; ik++) {
      xx = xo + ik*xs; zz = *(p0++);
      if (!((Plot_Flag&cdf_AB)||draw_mat_cnt)) { /* Applied the origine when required */
        xx += orgx; yy += orgy; zz += orgz;
      }
      *(p++) = xx; *(p++) = yy; *(p++) = zz;
    }
  }

  Draw__G_Surface( nraw, ncol, flg, buf );
}




void Draw_Gen_Polyhedral( int nfc, int nvt, int flg )
{ /* To create node(s) to perform Polyhedral Surface Plot */
  int  fd, fe, i, j, k, l, nfv;
  int*   fc_tb, sv_mode;
  float* vt_tb;
  float nm, nx, ny, nz, x1, x2, y1, y2, z1, z2;

//fprintf( fmsg, "  Create Surface %d, %d, %d.\n", nfc, nvt, flg );
//fprintf( fmsg, "  out Mode = %d.\n", draw_out_mode );
//Draw_Fmsgupdate();

  if (nfc) {
    fc_tb = (int*)   malloc(   nfc*sizeof(   int ) );
    Sdrw_Read_Block( (Char*) fc_tb, nfc*sizeof( int ) );
  }
  if (nvt) {
    vt_tb = (float*) malloc( 3*nvt*sizeof( float ) );
    Sdrw_Read_Block( (Char*) vt_tb, 3*nvt*sizeof( float ) );
  }
  if (!(nfc&&nvt)) return;

  fd = 0;
  Plot_Flag  = cdf_3DSTND|flg;
  if (draw_out_mode > 3) Plot_Flag |= cdf_NRM;
                    else if (draw_out_mode > -4) Plot_Flag &= ~cdf_NRM;
  sv_mode = draw_out_mode;

//fprintf( fmsg, "  Flag Word  = (%d,%d) => %d (Normal=%d).\n", flg, draw_out_mode, Plot_Flag, cdf_NRM );
//Draw_Fmsgupdate();

  for (i = 0; i < nfc; i++ ) {
    /* Scan all faces */
    nfv = fc_tb[i];
    if (Plot_Flag&cdf_NRM) {
      /* Build the normal to the face */
      x1 = vt_tb[fd+3] - vt_tb[fd+0];
      y1 = vt_tb[fd+4] - vt_tb[fd+1];
      z1 = vt_tb[fd+5] - vt_tb[fd+2];
      x2 = vt_tb[fd+6] - vt_tb[fd+3];
      y2 = vt_tb[fd+7] - vt_tb[fd+4];
      z2 = vt_tb[fd+8] - vt_tb[fd+5];
      nx = y1*z2 - z1*y2;
      ny = z1*x2 - x1*z2;
      nz = x1*y2 - y1*x2;
      nm = sqrt( nx*nx + ny*ny + nz*nz );
      if (nm < 1.0E-7) goto Et_Err;
      nx /= nm; ny /= nm; nz /= nm;
//    fprintf( fmsg, "  Nonm[i]/nrm = (%f,%f,%f).\n", nx, ny, nz );
//    Draw_Fmsgupdate();
    }
    Plot_Size  = 0;
    l = 0;
    for (j = 0; j < nfv; j++) {
      /* Scan on all vertex of each face */
      if (Plot_Index <= (PLOT_TAB_LEN-6)) {
        if (Plot_Flag&cdf_NRM) {
          Plot_Tab[l++] = nx;
          Plot_Tab[l++] = ny;
          Plot_Tab[l++] = nz;
//        fprintf( fmsg, " Surface Normal of face #%d, v#%d (%f,%f,%f).\n",
//                       i, j, nx, ny, nz );
        }
        for (k = 0; k < 3; k++) Plot_Tab[l++] = vt_tb[fd++];
//      fprintf( fmsg, " Surface Vertex of face #%d, v#%d (%f,%f,%f).\n",
//                     i, j, Plot_Tab[l-3], Plot_Tab[l-2], Plot_Tab[l-1] );
//      Draw_Fmsgupdate();
        Plot_Size++;
      }
    }
//  fprintf( fmsg, "  Create Surface Polygone of %d vertex (cd = %d).\n",
//                 Plot_Size, Plot_Flag );
//  Draw_Fmsgupdate();
    draw_out_mode = -9; /* POLYGON */
    if (Draw_Opened_Box) Draw_Gen_SMplot( Plot_Size, Plot_Flag, Plot_Tab );
                    else Draw_Gen_Mplot( Plot_Size, Plot_Flag, Plot_Tab );
    Plot_Size  = 0;
  }
  draw_out_mode = sv_mode;
  Plot_Flag  = 0; 

Et_Err:
  if (fc_tb) free( fc_tb );
  if (vt_tb) free( vt_tb );
}



void Draw_Gen_Text( float x, float y, float ang, float high,
                    int len, char * txt )
{ /* Generate  a General Text Request */
  Draw_Ptr p;
  char *   q;
  int      i;

  p = Draw_New_Node( Draw_Text );
  p->txt.txt_x    = x;
  p->txt.txt_y    = y;
  p->txt.txt_high = high;
  p->txt.txt_ang  = ang;
  if (len < 0) len = strlen( txt );
  p->txt.txt_len  = len;
  q = (char *) malloc( (len+1)*sizeof( char ) );
  for (i = 0; i< len; i++) q[i] = txt[i];
  q[len] = 0;
  p->txt.txt_str  = q;

//fprintf( fmsg, " Draw Gen Text at (%f,%f), high = %f, ang = %f, len = %d, s = \"%s\" .\n",
//                 p->txt.txt_x, p->txt.txt_y,
//                 p->txt.txt_high, p->txt.txt_ang,
//                 p->txt.txt_len, p->txt.txt_str );
//Draw_Fmsgupdate();

}




void Draw_Gen_PP_Attr( int code )
{
  Draw_Ptr p;

  p = Draw_New_Node( Draw_PP_Attr );
  p->ppattr.ppa_code = code;
}




void Draw_Gen_Eplane( float * uplane )
{
  int      i;
  Draw_Ptr p;

  if (uplane) { /* Enable a 2D Draw_Plot Plane */
    p = Draw_New_Node( Draw_Ena_Plane );
    p->eplane.epl_prev = NULL;
    for (i = 0; i < 3; i++) p->eplane.epl_mat[i +  0] = uplane[i + 0];
    for (i = 0; i < 3; i++) p->eplane.epl_mat[i +  4] = uplane[i + 3];
    for (i = 0; i < 3; i++) p->eplane.epl_mat[i +  8] = uplane[i + 6];
    for (i = 0; i < 3; i++) p->eplane.epl_mat[i + 12] = uplane[i + 9];
    p->eplane.epl_mat[12] += orgx;
    p->eplane.epl_mat[13] += orgy;
    p->eplane.epl_mat[14] += orgz;
    p->eplane.epl_mat[ 3]  =  0.0;
    p->eplane.epl_mat[ 7]  =  0.0;
    p->eplane.epl_mat[11]  =  0.0;
    p->eplane.epl_mat[15]  =  1.0;
  } else { /* Disable a 2D Draw_Plot Plane */
    p = Draw_New_Node( Draw_Dis_Plane );
  }
}




void Draw_Gen_Circle( float x, float y, float r, float sa, float ea, float ac )
{ /* Generate a Circle Request */
  Draw_Ptr p;

  p = Draw_New_Node( Draw_Circle );
  p->circ.circl_x   = x;
  p->circ.circl_y   = y;
  p->circ.circl_r   = r;
  p->circ.circl_sa  = sa;
  p->circ.circl_ea  = ea;
  p->circ.circl_ac  = ac;
  p->circ.circl_li  = -1;
  p->circ.circl_kn  = draw_out_mode;
}




void Draw_Gen_Sphere( float x, float y, float z, float r,
                      int plg, int plt, int code )
{ /* Generate a Sphere Request */
  Draw_Ptr p;

//fprintf( fmsg, " Gen_Sphere at (%f,%f,%f):%f with %d,%d:%d\n", x, y, z, r,
//                 plg, plt, code );
//Draw_Fmsgupdate();

  p = Draw_New_Node( Draw_Sphere );
  p->sphere.sphere_x   = x;
  p->sphere.sphere_y   = y;
  p->sphere.sphere_z   = z;
  p->sphere.sphere_r   = r;
  p->sphere.sphere_px  = plg;
  p->sphere.sphere_pz  = plt;
  p->sphere.sphere_li  = -1;
  p->sphere.sphere_k   = code;
  draw_3D_flg = 1;
}




void Draw_Gen_Cylinder( float *fparms, int px, int pz, int code )
{ /* Generate a Cylinder Request */
  Draw_Ptr p;

  p = Draw_New_Node( Draw_Cylinder );
  p->cyl.cyl_x   = fparms[0];
  p->cyl.cyl_y   = fparms[1];
  p->cyl.cyl_z   = fparms[2];
  p->cyl.cyl_th  = fparms[3];
  p->cyl.cyl_ph  = fparms[4];
  p->cyl.cyl_lg  = fparms[5];
  p->cyl.cyl_rb  = fparms[6];
  p->cyl.cyl_rt  = fparms[7];
  p->cyl.cyl_px  = px;
  p->cyl.cyl_pz  = pz;
  p->cyl.cyl_li  = -1;
  p->cyl.cyl_k   = code;
  draw_3D_flg = 1;
}




void Draw_Gen_Disk( float *fparms, int pt, int pr, int code )
{ /* Generate a Disk Request */
  Draw_Ptr p;

  p = Draw_New_Node( Draw_Disk );
  p->disk.dsk_x  = fparms[0];
  p->disk.dsk_y  = fparms[1];
  p->disk.dsk_z  = fparms[2];
  p->disk.dsk_th = fparms[3];
  p->disk.dsk_ph = fparms[4];
  p->disk.dsk_ri = fparms[5];
  p->disk.dsk_re = fparms[6];
  p->disk.dsk_sa = fparms[7];
  p->disk.dsk_ea = fparms[8];
  p->disk.dsk_pt = pt;
  p->disk.dsk_pr = pr;
  p->disk.dsk_li = -1;
  p->disk.dsk_k  = code;
  draw_3D_flg = 1;
}




void Draw_Gen_Exec_Seg( int nseg )
{
  Draw_Ptr p, q;

  q = Draw_Locate_Seg( nseg );
  if (q) {
    p = Draw_New_Node( Draw_Exec_Seg );
    p->exeseg.exec_seg = q;
    p->exeseg.exec_lst = q->seg.seg_ref;
    q->seg.seg_ref = p;
  }
}




void Draw_Gen_Lattr( int knd, float sz )
{ /* Generate a Line_Attr Request */
  Draw_Ptr p;

  p = Draw_New_Node( Draw_Line_Attr );
  p->linattr.plia_kind = knd;
  p->linattr.plia_size = sz;
}




void Draw_Gen_Mattr( int knd, float sz )
{ /* Generate a Marker_Attr Request */
  Draw_Ptr p;

  p = Draw_New_Node( Draw_Marker_Attr );
  p->mrkattr.mrka_kind = knd;
  p->mrkattr.mrka_size = sz;
}




int Draw_Set_Seg_Attr( int seg_id, int det, int hli, int vis, float prior )
{ /* Change the specified segment Attribute */
  Draw_Ptr      p;
  int      sv_sta,
           nw_sta,
             iret;

  iret = 0;
  p =  Draw_Locate_Seg( seg_id );
  if (p)
  {
//  fprintf( fmsg, " Set Segment Attributes of Seg from # %d from %d\n",
//                 p->seg.seg_ide, p->seg.seg_stat );
//  Draw_Fmsgupdate();
    sv_sta = p->seg.seg_stat;
    if (det > 1) nw_sta = DRWSEG_DETECT;
            else if (det) nw_sta = sv_sta&DRWSEG_DETECT;
                     else nw_sta = 0;

    if (hli > 1) nw_sta = nw_sta|DRWSEG_HLIGHT;
            else if (hli == 1) nw_sta = nw_sta|(sv_sta&DRWSEG_HLIGHT);

    if (vis > 1) nw_sta  = nw_sta|DRWSEG_VISIBLE;
            else if (vis == 1) nw_sta = sv_sta|(sv_sta&DRWSEG_VISIBLE);
    p->seg.seg_stat = nw_sta;
    if (prior >= 0.0 && prior <= 1.0) p->seg.seg_pri  = prior;

//  fprintf( fmsg, " Set Segment Attributes of Seg # %d to %d\n",
//                 p->seg.seg_ide, p->seg.seg_stat );
//  Draw_Fmsgupdate();

    if (nw_sta&DRWSEG_VISIBLE)
      if (!(sv_sta&DRWSEG_VISIBLE))
        iret = 1; /* Generate a Plot Update */;
  }
  else
    Draw_Error( "Cannot Set Attribute of Not Existing Segment", &seg_id );

  return iret;
}




void Draw_Gen_Fattr( int knd, int stl )
{ /* Generate a Fill_Attr Request */
  Draw_Ptr p;

  p = Draw_New_Node( Draw_Fill_Attr );
  p->filattr.fila_kind = knd;
  p->filattr.fila_styl = stl;
}




void Draw_Gen_Tattr( int haln, int valn, int path, float expf, float spcf )
{ /* Generate a Text_Attr Request */
  Draw_Ptr p;

  p = Draw_New_Node( Draw_Text_Attr );
  if (haln == 0) haln = 1;
  if (valn == 0) valn = 1;
  if (haln == 0) haln = 1;
  if (path == 0) path = 1;
  p->txtattr.txta_haln = haln;
  p->txtattr.txta_valn = valn;
  p->txtattr.txta_path = path - 1;
  p->txtattr.txta_expf = expf;
  p->txtattr.txta_spcf = spcf;
}




void Draw_Gen_Font( int fnt, int prc )
{ /* Generate a Change Font Request */
  Draw_Ptr p;

  p = Draw_New_Node( Draw_Font );
  p->font.fnt_font = fnt;
  p->font.fnt_prec = prc;
}




void Draw_Gen_Color( float r, float v, float b, float a )
{ /* Generate a Change Color Request */
  Draw_Ptr p;
  int      i;

  p = Draw_New_Node( Draw_Color );
  p->color.col_rgba[0] = r;
  p->color.col_rgba[1] = v;
  p->color.col_rgba[2] = b;
  p->color.col_rgba[3] = a;
}




void Draw_Gen_BlendSet( int bmd )
{
  int isrc, idst, flagw;
  GLenum src, dst;
  GLenum ftb[11] = { GL_ZERO,                 /*  0 */
                     GL_ONE,
                     GL_DST_COLOR,            /*  2 */
                     GL_SRC_COLOR,
                     GL_ONE_MINUS_DST_COLOR,  /*  4 */
                     GL_ONE_MINUS_SRC_COLOR,
                     GL_SRC_ALPHA,            /*  6 */
                     GL_ONE_MINUS_SRC_ALPHA,
                     GL_DST_ALPHA,            /*  8 */
                     GL_ONE_MINUS_DST_ALPHA,
                     GL_SRC_ALPHA_SATURATE    /* 10 */
  };
  Draw_Ptr p;
  int      i;

//fprintf( fmsg, " ALPHA Set at = %d\n", bmd );
//Draw_Fmsgupdate();

  if (!(bmd&4096)) 
    switch (bmd) {
      case 1: /* Transparence in 3D */
        src = GL_SRC_ALPHA; dst = GL_ONE;
        flagw = DRWGL_BLD_ENA | DRWGL_BLD_DDIS;
      break;

      case 2: /* Transparence in 3D */
        src = GL_SRC_ALPHA; dst = GL_ONE_MINUS_SRC_ALPHA;
        flagw = DRWGL_BLD_ENA | DRWGL_BLD_DDIS;
      break;

      case 3: /* Single 2D Blending */
        src = GL_SRC_ALPHA; dst = GL_ONE_MINUS_SRC_ALPHA;
        flagw = DRWGL_BLD_ENA;
      break;

      case 4: /* 2D Multi part Supperposition */
        src = GL_SRC_ALPHA; dst = GL_ONE;
        flagw = DRWGL_BLD_ENA;
      break;

      default: /* Return to Normal mode */
        src = GL_ONE; dst = GL_ZERO;
        flagw = DRWGL_BLD_DENA | DRWGL_BLD_DIS;
    }
  else {
    bmd = - bmd;
    isrc = bmd&15; idst = (bmd/16)&15; flagw = (bmd/256)&15;
    src = ftb[isrc]; dst = ftb[idst];
  }

//fprintf( fmsg, " ALPHA Set give %d: %d->%d\n", flagw, src, dst );
//Draw_Fmsgupdate();

  p = Draw_New_Node( Draw_Blend_Set );
  p->blend.bld_flags = flagw;
  p->blend.bld_src   =   src;
  p->blend.bld_dst   =   dst;
}




void Draw_Gen_LightDef( int lnb, int kind, int nprm, float* prm )
{
  GLfloat par[4] = { 0.0, 0.0, 0.0, 1.0 };
  GLenum lig, knd;
  static GLenum pntb[10] = {
                      GL_POSITION,        GL_AMBIENT,       /* 0, 1 */
                      GL_DIFFUSE,         GL_SPECULAR,      /* 2, 3 */
                      GL_SPOT_DIRECTION,  GL_SPOT_EXPONENT, /* 4, 5 */
                      GL_SPOT_CUTOFF,                       /* 6    */
                      GL_CONSTANT_ATTENUATION,              /* 7    */
                      GL_LINEAR_ATTENUATION,                /* 8    */
                      GL_QUADRATIC_ATTENUATION              /* 9    */
                    };

  Draw_Ptr p;
  int      i;

  lig = GL_LIGHT0;
  if ((lnb > 0)&&(lnb < LIGHT_NUMBER)) lig += lnb;
  knd = ((kind >= 0)&&(kind < 10))?pntb[kind]:pntb[0];
  for (i = 0; i < nprm; i++) par[i] = prm[i];

//fprintf( fmsg, " Cnd Light%d Def. prop %d with value %d (%f,%f,%f,%f)\n",
//               lig-GL_LIGHT0,
//               kind, knd, par[0], par[1], par[2], par[3] );
//Draw_Fmsgupdate();

  p = Draw_New_Node( Draw_Light_Def );
  p->light.lig_spc  = lig;
  p->light.lig_kind = knd;
  for (i=0;i<nprm;i++)
    p->light.lig_prm[i]  = par[i];
}




void Draw_Gen_MatLight( int fcd, int kind, int nprm, float* prm )
{
  GLenum fce, knd;
  GLfloat eprm[4] = { 0.8, 0.8, 0.8, 1.0 }; /* Default Color Value */
  static GLenum pntb[6] = {
                      GL_AMBIENT_AND_DIFFUSE, GL_AMBIENT,  /* 0, 1 */
                      GL_DIFFUSE,             GL_SPECULAR, /* 2, 3 */
                      GL_SHININESS,           GL_EMISSION  /* 4, 5 */
                    };
  static GLenum fctb[3] = { GL_FRONT, GL_FRONT_AND_BACK, GL_BACK };
  Draw_Ptr p;
  int i;

  fce = ((fcd >= 0)&&(fcd < 3))?fctb[fcd]:fctb[1];
  knd = ((kind >= 0)&&(kind < 6))?pntb[kind]:pntb[0];
  for (i=0;i<nprm;i++) eprm[i] = prm[i];


//fprintf( fmsg, " Cnd MatProp. face %d, prop %d, value %d (%f,%f,%f,%f)\n",
//               fce, knd, nprm, prm[0], eprm[1], eprm[2], eprm[3] );
//Draw_Fmsgupdate();

  p = Draw_New_Node( Draw_Mat_Light_Prop );
  p->lmatp.lmat_face = fce;
  p->lmatp.lmat_kind = knd;
  for (i=0;i<nprm;i++) p->lmatp.lmat_prm[i]  = eprm[i];
}




void Draw_Get_Color_Def( int idc, float *r, float *g, float *b, float *a )
{
  if ((idc >= 0)&&(idc <= 15)) {
    *r = draw_srccoltb[idc][0];
    *g = draw_srccoltb[idc][1];
    *b = draw_srccoltb[idc][2];
    *a = draw_srccoltb[idc][3];
  }
}




void Draw_Set_Color_Def( int idc, float r, float g, float b, float a )
{
  if ((idc >= 0)&&(idc <= 15)) {
    draw_srccoltb[idc][0] = r;
    draw_srccoltb[idc][1] = g;
    draw_srccoltb[idc][2] = b;
    draw_srccoltb[idc][3] = a;
    if (!idc) DrwGL_Set_Clear_Color(); /* Change Clear Color when Color #0 is changed */
  }
}




void Draw_GPlots( void )
{
  Plot_Size     =   0;
  Plot_Index    =   0;
  Plot_Flag     =   0;
  cx_pen        = 0.0;
  cy_pen        = 0.0;
  cz_pen        = 0.0;
  draw_out_mode =   1;
}




void Draw_GPlot1( void )
{ /* To Put a first point directly in the Plot table */
  Plot_Tab[0] = cx_pen; 
  Plot_Tab[1] = cy_pen;
  Plot_Size   = 1;
  Plot_Index  = 2;
  Plot_Flag   = cdf_ST; /* Assume Pen set down at beginning */
}




void Draw_GPlot( float x, float y, int ipen )
{ /* To emulate a plotter pen movement in the server */
  /* To use only for small number of points (< 128) */
  if (ipen) /* A low pen movement is required */
    if (Plot_Size)
    { /* The pen was already in low state */
      if (Plot_Index >= (PLOT_TAB_LEN-1))
      { /* Table overflow */
        Draw_Gen_Mplot( Plot_Size, Plot_Flag, Plot_Tab );
        Plot_Tab[1] = Plot_Tab[--Plot_Index];
        Plot_Tab[0] = Plot_Tab[--Plot_Index];
        Plot_Size  = 1;
        Plot_Index = 2;
        Plot_Flag  = 0; /* The next Gen_Mplot is a continuation */
      }
      /* Append the new point to the table */
      Plot_Tab[Plot_Index++] = x;
      Plot_Tab[Plot_Index++] = y;
      Plot_Size++;
    }
    else 
    { /* The pen was up */
      Plot_Tab[0] = cx_pen;
      Plot_Tab[1] = cy_pen;
      Plot_Tab[2] = x;
      Plot_Tab[3] = y;
      Plot_Size   = 2;
      Plot_Index  = 4;
      Plot_Flag   = cdf_ST; /* Assume Pen set down */
    }
  else
  { /* An up pen movement is required */
    if (Plot_Size)
    { /* The pen move table is not empty */
      Draw_Gen_Mplot( Plot_Size, Plot_Flag|cdf_ND, Plot_Tab );
      Plot_Size  = 0;
      Plot_Index = 0;
      Plot_Flag  = 0;
    }
  }
  cx_pen = x;
  cy_pen = y;
}




void Draw_GPlot3( float x, float y, float z, int ipen )
{ /* To emulate a plotter pen movement in the server */
  /* To use only for small number of points (< 128) */

//fprintf( fmsg, " Gplot3 (%f,%f,%f):%4d.\n", x, y, z, ipen );
//fprintf( fmsg, " P Size  P Index = %d, %d\n", Plot_Size, Plot_Index );
//Draw_Fmsgupdate();

  if (ipen) /* A low pen movement is required */
    if (Plot_Size)
    { /* The pen was already in low state */
      if (Plot_Index >= (PLOT_TAB_LEN-2))
      { /* Table overflow */
        Draw_Gen_Mplot( Plot_Size, Plot_Flag, Plot_Tab );
        Plot_Tab[2] = Plot_Tab[--Plot_Index];
        Plot_Tab[1] = Plot_Tab[--Plot_Index];
        Plot_Tab[0] = Plot_Tab[--Plot_Index];
        Plot_Size  = 1;
        Plot_Index = 3;
        Plot_Flag  = cdf_3D; /* The next Gen_Mplot is a 3D continuation */
      }
      /* Append the new point to the table */
      Plot_Tab[Plot_Index++] = x;
      Plot_Tab[Plot_Index++] = y;
      Plot_Tab[Plot_Index++] = z;
      Plot_Size++;
    }
    else 
    { /* The pen was up */
      Plot_Tab[0] = cx_pen;
      Plot_Tab[1] = cy_pen;
      Plot_Tab[2] = cz_pen;
      Plot_Tab[3] = x;
      Plot_Tab[4] = y;
      Plot_Tab[5] = z;
      Plot_Size   = 2;
      Plot_Index  = 6;
      Plot_Flag   = cdf_3D|cdf_ST; /* Assume Pen set down */
    }
  else
  { /* An up pen movement is required */
    if (Plot_Size)
    { /* The pen move table is not empty */
      Draw_Gen_Mplot( Plot_Size, Plot_Flag|cdf_ND, Plot_Tab );
      Plot_Size  = 0;
      Plot_Index = 0;
      Plot_Flag  = 0;
    }
  }
  cx_pen = x;
  cy_pen = y;
  cz_pen = z;
}




void Draw_GPlot2D( Draw_Point pt, int ipen )
{ /* To emulate a plotter pen movement in the server */
  /* To use only for small number of points (< 128) */
  Draw_GPlot( pt[0], pt[1], ipen );
}



void Draw_GPlot3D( Draw_Point3 pt, int ipen )
{ /* To emulate a plotter pen movement in the server */
  /* To use only for small number of points */
  Draw_GPlot3( pt[0], pt[1], pt[2], ipen );
}



void Draw_GPlot_Rect( float x1, float y1, float x2, float y2 )
{ /* To Draw a rectangle. */
  int out_mode_save = draw_out_mode;
  float plt_tab[8];

  plt_tab[0] = x1; plt_tab[1] = y1;
  plt_tab[2] = x2; plt_tab[3] = y1;
  plt_tab[4] = x2; plt_tab[5] = y2;
  plt_tab[6] = x1; plt_tab[7] = y2;

  draw_out_mode = -3;
  Draw_Gen_Mplot( 4, cdf_STND, plt_tab );
  draw_out_mode = out_mode_save;
}




Draw_Ptr Draw_GPlot_Grid()
{
  Draw_Ptr p;
  int i, dxi, dxa, dyi, dya, pst, svmode;
  float xsta, ysta;

  svmode = draw_out_mode;
  draw_out_mode = -1; /* GL_LINES */

  xsta = 0.5*ws_rx/sxy_scale;
  ysta = 0.5*ws_ry/sxy_scale;

  dxi = (int) ceil( -xsta - orgx );  dxa = (int) floor(  xsta - orgx );
  dyi = (int) ceil( -ysta - orgy );  dya = (int) floor(  ysta - orgy );

  p = Draw_New_Sys_Seg( 5, 0 );

  Draw_Gen_Lattr( 3, 1.0 ); /* For all i%5 != 0 */
  pst = 0;
  for (i = dyi; i <= dya; i++) /* Plots Each Horizontal line */
    if (i%5) {
       Draw_GPlot( - xsta, i + orgy, pst ); pst = 1;
       Draw_GPlot(   xsta, i + orgy, 1 );
    }
  for (i = dxi; i <= dxa; i++) /* Plots Each Vertical line */
    if (i%5) {
       Draw_GPlot( i + orgx, - ysta, 1 );
       Draw_GPlot( i + orgx,   ysta, 1 );
    }
  Draw_GPlot( 0.0, 0.0, 0 );

  Draw_Gen_Lattr( 4, 1.0 ); /* For all i%5 == 0 */
  pst = 0;
  for (i = dyi; i <= dya; i++) /* Plots Each Horizontal line */
    if (!(i%5)) {
      Draw_GPlot( - xsta, i + orgy, pst ); pst = 1;
      Draw_GPlot(   xsta, i + orgy, 1 );
    }
  for (i = dxi; i <= dxa; i++) /* Plots Each Vertical line */
    if (!(i%5)) {
      Draw_GPlot( i + orgx, - ysta, 1 );
      Draw_GPlot( i + orgx,   ysta, 1 );
    }
  Draw_GPlot( 0.0, 0.0, 0 );

  Draw_Gen_Lattr( 1, 2.0 );
  Draw_GPlot( - xsta,   orgy, 0 );
  Draw_GPlot(   xsta,   orgy, 1 );
  Draw_GPlot(   orgx, - ysta, 1 );
  Draw_GPlot(   orgx,   ysta, 1 );
  Draw_GPlot(    0.0,    0.0, 0 );

  Draw_Gen_Lattr( 1, 1.0 );
  draw_curseg = NULL;
  draw_out_mode = svmode;
  return p;
}




Draw_Font_Ptr Draw_Load_Font( int font )
{ /* Load a Font from a File (via getenv name) */

  drwfnt_dir *pdir;     /* Pointer to the current graphic directive of readden Font */

  Draw_Font_Ptr p = NULL;  /* Pointer to the Loaded (DRAW_VGL) Font */
  
  char          fnt_fspc[128];   /* Logical File specification of the Font to Load */
  char *        fnt_lspc;        /* Special font specification */
  int           fnt_file, size, sz;
  GLfloat       xc, yc;
    

  /* Build the font file specification */
  sprintf( fnt_fspc, "DRAW_FONT_%d", font );
  if (fnt_lspc = getenv( fnt_fspc )) { /* A special Font definition is available */
    sprintf( fnt_fspc, "%s.Drwfnt", fnt_lspc ); /* Append the filetype ".Drwfnt" */
  }
  else /* We Use the standard DRAW_FONT logical and the filename   */
       /* ... "Draw_Font_<n>.Drwfnt" where <n> is the font number. */
    sprintf( fnt_fspc, "%sDraw_Font_%d.Drwfnt", getenv( "DRAW_FONT" ), font );
  
  if ((fnt_file = open( fnt_fspc, O_RDONLY, 0644 )) < 0) {
    fprintf( fmsg, " Font File \"%s\" Open Error.\n", fnt_fspc );
    Draw_Fmsgupdate();
    return NULL;
  }

//fprintf( fmsg, " Font File Open Successfull on %d .\n", fnt_file );
//Draw_Fmsgupdate();

  if (read( fnt_file, (char *) &size, sizeof( int ) ) != sizeof( int)) {
    fprintf( fmsg, " -> Error on Font Size Read.\n" );
    Draw_Fmsgupdate();    
    return NULL;
  }

  if (!(p = (Draw_Font_Ptr) malloc( size ))) {
    fprintf( fmsg, " -> Error on Font Allocation of %d bytes.\n", size );
    Draw_Fmsgupdate();    
    return NULL;
  }

  if ((sz = read( fnt_file, (char *) p, size )) != size) {
    fprintf( fmsg, " -> Error on Font Data Read of %d/%d bytes.\n", sz, size );
    Draw_Fmsgupdate();    
    return NULL;
  }

  close( fnt_file );

  // p->fnt_next  = NULL; /* Init the next link of font descriptor */
  p->fnt_ident = font; /* ... and set the font # identifier */

  if (!Draw_first_font) Draw_first_font = p;
                   else Draw_last_font->fnt_next = p;
  Draw_last_font = p;
  
  return p;
}





