/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*              D R A W   -   C L I E N T   L I B R A R Y                *
*                                                                       *
*              (Kernel to manage the bidirectional pipe)                *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/

/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/




/*********************************************************************/
/**                                                                 **/
/**          Echange DRAW_LIb <-> DRAW_SERVER Constantes            **/
/**                                                                 **/
/*********************************************************************/



#include <draw/draw_common.h>
#include <draw/sdrw_constantes.h>   /* Load the DRAW library Environment */


#ifdef _PIPE_BIG
# ifdef VMS
#   define _Pipe( a , b, c ) pipe( a, b, c)
# else
#   define _Pipe( a , b, c ) _pipe( a, b, c)
# endif
#else
# define _Pipe( a , b, c ) pipe( a )
#endif







/*********************************************************************/
/**                                                                 **/
/**      Echange DRAW_LIb <-> DRAW_SERVER Types Definitions         **/
/**                                                                 **/
/*********************************************************************/



typedef unsigned char Char;             /* Define the Unsigned char type Char */

typedef union {                         /* Define the Type Equiv. rec. type */
          struct { Char ch0, ch1, ch2, ch3; } ct;
                    int   i;
                    float f;
		} eqty;



/*********************************************************************/
/**                                                                 **/
/**          Echange DRAW_LIb <-> DRAW_SERVER Variables             **/
/**                                                                 **/
/*********************************************************************/

        int  Sdrw_Server_id;            /* Server Process identifier number (global object) */

static  int  c_inpipe,    c_outpipe,    /* Client Duplications */
             s_inpipe,    s_outpipe;    /* Server Duplications */

static  int  Sdrw_Insize,               /* Number of byte readden */
             Sdrw_Outsize,              /* Number of byte to write */
             Sdrw_Inread;               /* Index in the Read Pipe Buffer */

static Char  Sdrw_Inp_Buf[BUF_SIZE],    /* Buffer for Input from Server */
             Sdrw_Out_Buf[BUF_SIZE];    /* Buffer for output to server */

static  int  Sdrw_RT_mode;              /* Real time mode flag to Write to Server */

static  int  Sdrw_Server_Enable = 0;    /* Server Process Enable Flag */


static char *argu[8];                   /* Draw Server Argument Pointer Table */ 

static eqty equ;                        /* Record for float<->int<->Char Conv. */


#ifdef _SDRW__REVERSE
# define SDRW_input  20
# define SDRW_output 21
#endif




/*********************************************************************/
/**                                                                 **/
/**   Data Echange Routines for DRAW_LIB <-> DRAW_SERVER Dialogue   **/
/**                                                                 **/
/*********************************************************************/

int Sdrw_Open_Gate()
{ /* Create the bidirectional pipe */

#ifdef _SDRW__REVERSE

  c_inpipe  = dup( SDRW_input );
  c_outpipe = dup( SDRW_output );
  close( SDRW_input );
  close( SDRW_output );

#else
  int  cs_pipes[2], sc_pipes[2];        /* Original bi-directional pipe */

  if (_Pipe( cs_pipes, 0, BUF_SIZE ) == -1)
    {
      printf( " DRAW-LIB ERROR: - Client to Server Pipe allocation failed\n" );
      exit( 1 );
    }
  if (_Pipe( sc_pipes, 0, BUF_SIZE ) == -1)
    {
      printf( " DRAW-LIB ERROR: - Server to Client Pipe allocation failed\n" );
      exit( 1 );
    }
  /* Set the pipe file connections */
  c_inpipe  = sc_pipes[0];              /* Client input channel  */
  c_outpipe = cs_pipes[1];              /* Client output channel */
  s_outpipe = sc_pipes[1];              /* Server output channel */
  s_inpipe  = cs_pipes[0];              /* Server input channel  */

#endif

  Sdrw_RT_mode = 0;                     /* Init to wait mode (No RT mode) */
  Sdrw_Outsize = 0;                     /* Init Output message Length */
  return BUF_SIZE;                      /* Return the Pipe Buffer Size. */
}



/*********************************************************************/

static void Spawn_Server( Char *cmd )
{ /* To start the DRAW Server and establish the pipes with it */
#ifndef _SDRW__REVERSE

  int status, i;
  char * ExcErr = " DRAW-LIB ERROR: - Cannot exec the Server ""%s"" \n";

  /* Allocate room for Server Arguments */
  for (i=1; i<8; i++) argu[i] = (i<4)?((char *) malloc( 80 )):NULL;

  argu[0] = (char *) cmd;               /* ... and set the Server reference   */
  sprintf( argu[1],"%d", s_inpipe  );   /* Put the input and output pipe channels ... */
  sprintf( argu[2],"%d", s_outpipe );   /* ... as first and second Server arguments. */
  sprintf( argu[3],"%d", getpid() );    /* Put the Draw Client process id. */

  status = fork();                      /* Create the Server process */

  switch (status)                       /* Select the process */
  {
    case 0:                             /* For Server process, we try ... */
      if ((status = execv( (char *) cmd, argu )) == -1)  /* ... to start it. */
      {
        printf( ExcErr, *cmd );
        exit( 1 );                      /* Stop with message on Exec ERROR. */
      }
    break;

    case -1:                            /* Error return on fork */
      printf( " DRAW-LIB ERROR: Fork Error\n" );
      exit( 1 );
    break;

    default:                            /* Start DRAW Server Success */

      Sdrw_Server_id = status;          /* Keep the Server Process id. */

    break;
  }

#endif

  Sdrw_Insize  = 0;                     /* Init the pipe I/O buffers sizes/counts */
  Sdrw_Inread  = 0;
  Sdrw_Outsize = 0;
  Sdrw_Server_Enable = 1;               /* Flag as Server Ready */
}



/*********************************************************************/

int Sdrw_Establish_Connection()
{ /* Start the specified DRAW SERVER */
  /* Translate the logical server activation word */
char * command;

  if ((command = getenv( (char *) Sdrw_Out_Buf )) == NULL )
    return 0;                           /* No valid connection */
  else
  { /* Connection is ready... */
    /* ... activate the server by use of fork and exec fnc. */

    Spawn_Server( (Char *) command );
    return 1;                           /* Valid connection */
  }
}



/*********************************************************************/

void Sdrw_Connection_Release()
{ /* Stop Server Connection and Exit */
  int status, cstatus;
  if ((status = wait( &cstatus )) == -1)
  {
    printf( " DRAW-LIB ERROR: USER Wait failed\n" );
    exit( 1 );
  }
  Sdrw_Server_Enable = 0;               /* Flag as Server stopped */
}



/*********************************************************************/

int Sdrw_Read()
{ /* Read a buffer from the DRAW Server */
  if (!Sdrw_Server_Enable) {            /* If the Server is not Ready */
    printf( " DRAW-LIB ERROR: Server was not running.\n" );
    exit( 1 );
  }
  Sdrw_Insize = read( c_inpipe, Sdrw_Inp_Buf, BUF_SIZE );
  if (Sdrw_Insize<=0)
    printf( " DRAW-LIB ERROR: USER Pipe Read failed\n" );
  Sdrw_Inread = 0;
/*printf( " User Read %d bytes.\n", Sdrw_Insize );*/
  return Sdrw_Insize;
}



/*********************************************************************/

void Sdrw_Write()
{ /* Write a buffer to the DRAW Server and wait the Answerd */
  if (!Sdrw_Server_Enable) {            /* If the Server is not Ready */
    printf( " DRAW-LIB ERROR: Server was not running.\n" );
    exit( 1 );
  }
/*printf( " User Write %d bytes.\n", Sdrw_Outsize );*/
  if (write( c_outpipe, Sdrw_Out_Buf, Sdrw_Outsize ) == -1)
  {
    printf( " DRAW-LIB ERROR: Write to SERVER failed.\n" );
    exit( 1 );
  }
  Sdrw_Outsize = 0;
}



/************************************************************************/

int  Sdrw_Get_Code();
void Sdrw_Put_Code( int );



int  Sdrw_Read_Block( Char* tb, int dim )
{ /* Write a User buffer to the DRAW Library */
  int n, i, t;

#ifndef _PIPE_NO_CNTFLOW

    Sdrw_Put_Code( 0 );
    Sdrw_Write();

#endif

  /* The Server has begun to send the Data Block (stored at tb, dim bytes length) */
  t = 0;
  while (t < dim) {

#ifdef _PIPE_NO_CNTFLOW

    Sdrw_Put_Code( 0 );
    Sdrw_Write();

#endif

#ifdef _PIPE_NOBYPASS

    n = Sdrw_Insize;
    for (i = 0; i < Sdrw_Insize; i++) tb[i] = Sdrw_Inp_Buf[i];

#else

    n = read( c_inpipe, tb, BUF_SIZE );
    if (n < 0) {
      printf( " DRAW-LIB ERROR: USER Pipe Read Block failed.\n" );
      return n;
    }

#endif

    t  += n;
    tb += n;
    if (t < dim) {
      Sdrw_Put_Code( 0 );
      Sdrw_Write();
    }
  }
  return t;
}



/************************************************************************/

void Sdrw_Write_Block( Char* tb, int dim )
{ /* Write a User buffer to the DRAW Library */
  int i, n;

// printf( " Client_Write_Block of %d bytes.\n", dim );

#ifndef _PIPE_NO_CNTFLOW

    Sdrw_Read(); /* Read for synchro */

#endif

  /* The Server must be ready to get the Data Block (address tb, dim bytes length) */
  while (dim) {
    n = (dim > BUF_SIZE)?BUF_SIZE:dim;

//  printf( " Client_Write_Block record of %d/%d bytes.\n", n, dim );

#ifdef _PIPE_NO_CNTFLOW

    Sdrw_Read(); /* Read for synchro */

#endif

#ifdef _PIPE_NOBYPASS

    for (i = 0; i < n; i++) Sdrw_Out_Buf[i] = tb[i];
    Sdrw_Outsize = n;
    Sdrw_Write();

#else

    if (write( c_outpipe, tb, n ) == -1)
    { printf( " DRAW-LIB ERROR: Write Block to SERVER failed.\n" );
      exit( 1 );
    }

#endif

    dim -= n;
    tb  += n;

//  printf( " Client_Write_Block: %d left bytes.\n", dim );
  }
}



/*********************************************************************/

int Sdrw_Dialog_Request()
{
  Sdrw_Write();
  return Sdrw_Read();
}



/*********************************************************************/
/**        Input from and Output to the Graphic Server Routines     **/ 
/*********************************************************************/

void Sdrw_Put_Char( Char ch )
{
  if (Sdrw_Outsize < BUF_SIZE)
    Sdrw_Out_Buf[Sdrw_Outsize++] = ch;
}



/*********************************************************************/

void Sdrw_Put_Int( int iv )
{
  equ.i = iv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
}



/*********************************************************************/

void Sdrw_Put_Float( float fv )
{
  equ.f = fv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
}



/*********************************************************************/

void Sdrw_Put_String( Char * s, int l )
{
  int i;
  Sdrw_Put_Char( l );
  for ( i=0; i<l; i++)
  {
    Sdrw_Put_Char( s[i] );
  }
}



/*********************************************************************/

void Sdrw_Put_Code( int code )
{
  Sdrw_Outsize = 0;
  Sdrw_Put_Int( code );
}



/*********************************************************************/

Char Sdrw_Get_Char()
{
  if (Sdrw_Inread < Sdrw_Insize)
    return Sdrw_Inp_Buf[Sdrw_Inread++];
  else
    return 0;
}



/*********************************************************************/

int Sdrw_Get_Int()
{
  equ.ct.ch0 = Sdrw_Get_Char();
  equ.ct.ch1 = Sdrw_Get_Char();
  equ.ct.ch2 = Sdrw_Get_Char();
  equ.ct.ch3 = Sdrw_Get_Char();
  return equ.i;
}



/*********************************************************************/

float Sdrw_Get_Float()
{
  equ.ct.ch0 = Sdrw_Get_Char();
  equ.ct.ch1 = Sdrw_Get_Char();
  equ.ct.ch2 = Sdrw_Get_Char();
  equ.ct.ch3 = Sdrw_Get_Char();
  return equ.f;
}



/*********************************************************************/

int Sdrw_Get_String( Char* s )
{
  int i, l;

  l = Sdrw_Get_Char();
  for (i=0; i<l; i++)
    s[i] = Sdrw_Get_Char();
  s[l] = 0;
  return l;
}



int Sdrw_Get_String_SP( Char** p, int aflg )
{
  int i, l;
  Char*  s;

  l = Sdrw_Get_Char();
  if (aflg) {
    if (*p) free( (void*)*p );
    s = *p = (Char*) malloc( l );
  }
  for (i=0; i<l; i++)
    s[i] = Sdrw_Get_Char();
  s[l] = 0;
  return l;
}



/*********************************************************************/

int Sdrw_Get_Code()
{
  Sdrw_Inread = 0;
  return Sdrw_Get_Int();
}




/*********************************************************************/
/**   Block Input from and Output to the Graphic Server Routines    **/ 
/*********************************************************************/

void Sdrw_Put_FBlock( float tb[], int dim )
{ /* Output to pipe the array tb[dim] */
  int i, j, r;

  j = 0;
  while (dim > 0) {
    /* Fill the Buffer */
    r = (BUF_SIZE - sizeof( int ) - Sdrw_Outsize)/sizeof( float );
    if (r > dim) r = dim;
    Sdrw_Put_Int( r );
    for (i = 0; i < r; i++) Sdrw_Put_Float( tb[j++] );
    dim -= r;
    if (dim > 0) Sdrw_Dialog_Request();
  }
}



/*********************************************************************/

int Sdrw_Get_FBlock( float tb[], int dim )
{
  int i, j, n, t;

  j = 0;
  t = 0;
  while (t < dim) {
    n = Sdrw_Get_Int();
    for(i = 0; i < n; i++) tb[j++] = Sdrw_Get_Float();
    t += n;
    if (t < dim) { /* Get the next message */
      Sdrw_Put_Code( cd_continue );
      Sdrw_Dialog_Request();
    }
  }
  return t;
}




/*************************     E N D    ******************************/
