{
*************************************************************************
*                                                                       *
*                                                                       *
*                   *  C P A S  *  S Y S T E M  *                       *
*                                                                       *
*                                                                       *
*          * * *   S t a n d a r d   L i b r a r y   * * *              *
*                                                                       *
*                                                                       *
*                    ---  RUN-TIME KERNEL  ---                          *
*              ---  Main Part  of PASCAL RUN-TIME ---                   *
*               ---  Version 3.2-A1 -- 31/10/2017 ---                   *
*                                                                       *
*         by :                                                          *
*                                                                       *
*             P. Wolfers                                                *
*                 c.n.r.s.                                              *
*                 Institut Neel                                         *
*                 B.P.  166 X   38042  Grenoble Cedex                   *
*                                             FRANCE.                   *
*                                                                       *
*************************************************************************


/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////



*************************************************************************
*                                                                       *
*       Last revision of 15-JAN-2013 for CPASCAL Version 3.1-B 1        *
*                                                                       *
*************************************************************************
}



%pragma trace 0;       { Always here: called by PAS__ERROR to don't loop }
%pragma code_option c_code '#define BASIC_IO   0';
module PAS__STD;

const
  system_unix   = BOOLEAN( sys_system = systyp_unix );
  system_mosx   = BOOLEAN( sys_system = systyp_mosx );
  system_cygw   = BOOLEAN( sys_system = systyp_cygw );
  system_wind   = BOOLEAN( sys_system = systyp_wind );


%include 'PASSRC:cpas__ccdef.pas'{, list_on};

%include 'PASSRC:cpas__stddef.pas'{, list_on};


const
  log_dev_tty   = 'TT';                 { CPAS NAME for the current terminal device }
  log_dev_null  = 'NL';                 { CPAS NAME for the null device }

  unx_dev_tty   = '/dev/tty';           { Name of the terminal device for Unix like ... }
  wnt_dev_tty   = '/dev/tty';           { ... and for Windows OS }
  unx_dev_null  = '/dev/null';          { Name of the NULL device for Unix like ... }
  wnt_dev_null  = '/dev/null';          { ... and for Windows OS }

  heap_defsize  =     8192;             { Size of heap in quad word unit (size of double float (64 bits large = 8 bites) }
  heap_defincr  =     1024;             { Increment size in (8 bytes unit) }
  heap_minsize  =      128;             { Minimum allowed heap size }



type
  { Define the Class of file Procedure array variable }
  file_chtab = array[1..28] of cc__int; { File charac. table (for GET_FILE_INFO) }

  { Heap memory management specific types }
  heap_ptr              =         ^heap_descr; { Define the heap descriptor pointer }

  mark_ptr              =         ^mark_descr; { Define a MARK record pointer }

  heap_descr( heap_size: integer) = record     { * Define the heap descriptor }
    heap_htop:                        integer; { Current top allocation in this heap }
    heap_prev:                       heap_ptr; { Pointer to the previuos heap }
    heap_mark:                       mark_ptr;
    heap_tabl:   array[0..heap_size] of double { The heap array - The size in not used (do not compile with the range test switch }
  end;

  mark_descr = record                          { * Define a MARK record }
    mark_prev:                       mark_ptr; { Pointer to the previous MARK }
    mark_heap:                       heap_ptr; { Pointer of the Heap used before the MARK allocation }
    mark_ptop:                        integer  { Top allocation of the previous MARK }
  end;


var
  { *** WARNING these definitions are incompatible with use of std files *** }

  inp_descr: fild_rec := ( nil, nil, nil, nil, nil, nil,
                           nil,         { Filename pointer (string pointer) }
                           0,           { Input C descriptor }
                           1,           { File of char (text) }
                     def_buf_size,      { Size of the buffer }
                           0,           { Buffer count (Buffer is empty) }
                           0,           { Buffer decount for update read }
                           0,           { Index for update write }
                           0,           { Position in BUFFER UNIT for Map Buffer }
  [lio_inp,lio_txt,lio_std,lio_emp]     { File status }
                         );

  out_descr: fild_rec := ( nil, nil, nil, nil, nil, nil,
                           nil,         { Filename pointer (string pointer) }
                           1,           { Output C descriptor }
                           1,           { File of char (text) }
                     def_buf_size,      { Size of the buffer }
                           0,           { Buffer count (Buffer is empty) }
                           0,           { Buffer decount for update read }
                           0,           { Index for update write }
                           0,           { Position in BUFFER UNIT for Map Buffer }
        [lio_out,lio_txt,lio_std]       { File status }
                         );

  err_descr: fild_rec := ( nil, nil, nil, nil, nil, nil,
                           nil,         { Filename pointer (string pointer) }
                           2,           { stderr C descriptor }
                           1,           { File of char (text) }
                     def_buf_size,      { Size of the buffer }
                           0,           { Buffer count (Buffer is empty) }
                           0,           { Buffer decount for update read }
                           0,           { Index for update write }
                           0,           { Position in BUFFER UNIT for Map Buffer }
        [lio_out,lio_txt,lio_std]       { File status }
                         );


  str_descr: fild_rec := ( nil, nil, nil, nil, nil, nil,
                           nil,         { Filename pointer (string pointer) }
                          -1,           { string READV/WRITEV descriptor }
                           1,           { File of char (text) }
                           0,           { Size of the buffer }
                           0,           { Buffer count (Buffer is empty) }
                           0,           { Buffer decount for update re    curr_heap_ptr := base_heap_ptr;ad }
                           0,           { Index for update write }
                           0,           { Position in BUFFER UNIT for Map Buffer }
[lio_vir,lio_txt,lio_std,lio_est]       { File status }
                         );

  spc_arr: [static] packed array[1..4] of char := '    ';       { Cte. Character table for file^ access not undefined }

  curr_heap_ptr:     heap_ptr :=   nil; { Basic heap area pointer }
  curr_mark_ptr:     mark_ptr :=   nil; { The current heap pointer }


  [global 'PAS__first_descr'] fild_firstptr,
  [global 'PAS__last_descr']  fild_lastptr:  fild_ptr := nil;

  [global 'PAS__curr_iptr']   fild_c_inp,
  [global 'PAS__curr_optr']   fild_c_out:    fild_ptr := nil;

  [global 'PAS__f_input' ]    input_file,
  [global 'PAS__f_output']   output_file,
  [global 'PAS__f_error' ]   errmsg_file:    fild_ptr;


  [global 'PAS__argc']        sys_argc:      integer;
  [global 'PAS__argv']        sys_argv:      array[0..mxpar_process] of string_ptr;
  [global 'PAS__env']          sys_env:      $wild_pointer;

  [global 'PAS__iostatus']    io_errcode:    integer;
  [global 'PAS__fspc']        curr_fspc:     string;

  [global 'PAS__file_info']   tbfch:         file_chtab;        { Current File attribute Characteristic }

  [global 'PAS__heap_isiz']  heap_inisize:   integer :=  heap_defsize; { Initial heap size }
  [global 'PAS__heap_incr']  heap_increment: integer :=  heap_defincr; { Heap Allocation increment (default is 64k bytes) }



{ ** Basic I/O procedures to use with the standard files ** }




{ **** Basic Memory allocation:de-allocation **** }


{ MEM$$NEW is defined in cc__define.pas as PAS__MEM_ALLOC
  that is set as the C function malloc in cpas_defs.h }
[global 'PAS__NEW']
function PAS$$NEW( size: integer ): $wild_pointer;
begin
  PAS$$NEW := MEM$$ALLOC( size )
end PAS$$NEW;



{ MEM$$FREE is defined in cc__define.pas as PAS__MEM_FREE and
  PAS__MEM_FREE is set as the C function free in cpas_defs.h }
[global 'PAS__DISPOSE']
procedure PAS$$DISPOSE( var ptr: $wild_pointer );
begin
  MEM$$FREE( ptr );
  ptr := nil
end PAS$$DISPOSE;



[global 'PAS__HEAP_ALLOC']
function HEAP_ALLOC( size: integer ): $wild_pointer;
const
  aln_siz = double"size;
  aln_msk = aln_siz - 1;

var
  ia, it, sz: integer;


  procedure ALLOC_NEW_HEAP( sz: integer );
  var
    p: heap_ptr;

  begin
    if heap_increment < heap_minsize then heap_increment := heap_minsize;
    sz := sz + mark_descr"size;
    sz := ((sz + heap_increment - 1) div heap_increment)*heap_increment;
    sz := (sz + aln_msk) div aln_siz;
    NEW( p, sz );
    p^.heap_htop := 0;
    p^.heap_prev := curr_heap_ptr;
    p^.heap_mark := curr_mark_ptr;
    curr_heap_ptr := p
  end ALLOC_NEW_HEAP;


begin { HEAP_ALLOC }
  if curr_heap_ptr = nil then
  begin
    if heap_inisize < heap_minsize then heap_inisize := heap_minsize;
    NEW( curr_heap_ptr, heap_inisize ); { Allocation for initial heap }
    with curr_heap_ptr^ do
    begin
      heap_htop :=   0;
      heap_prev := nil;
      heap_mark := nil
    end;
    curr_mark_ptr   :=  nil
  end;

  ia := curr_heap_ptr^.heap_htop;
  sz := ((size + aln_msk) div aln_siz);
  it := ia + sz;
  if it >= curr_heap_ptr^.heap_size then
  begin
    ALLOC_NEW_HEAP( sz );
    it := sz; ia := 0
  end;
  with curr_heap_ptr^ do
  begin
    heap_htop := it;
    HEAP_ALLOC := POINTER_TO_W( heap_tabl[ia]"address )
  end
end HEAP_ALLOC;



{ Special routine for heap system of allocation }
[global 'PAS__MARK']
procedure PAS$$MARK;
var
  mrk: mark_ptr;
  hp1: heap_ptr;
  top:  integer;
  
begin
  if curr_heap_ptr <> nil then
  begin
    hp1 := curr_heap_ptr;
    top := curr_heap_ptr^.heap_htop
  end
  else
  begin
    hp1 := nil; top := 0
  end;
  mrk := HEAP_ALLOC( mrk^"size );
{ NEWH( mrk );                  { Allocate a MARK in the Heap - (A new heap is created if necessary) }
  with mrk^ do
  begin
    mark_prev  := curr_mark_ptr;
    if hp1 = nil then mark_heap := curr_heap_ptr
                 else mark_heap := hp1;
    mark_ptop  := top
  end;
  curr_mark_ptr := mrk
end PAS$$MARK;



[global 'PAS__RELEASE']
procedure PAS$$RELEASE;
var
  mrk:      mark_ptr;
  hp1, hp2: heap_ptr;
  top:       integer;

begin
  if (curr_heap_ptr <> nil) and (curr_mark_ptr <> nil) then
  begin
    with curr_mark_ptr^ do
    begin
      mrk := mark_prev;
      hp1 := mark_heap;
      top := mark_ptop
    end;
    if (hp1 <> nil) and (hp1 <> curr_heap_ptr) then
      while hp1 <> curr_heap_ptr do
      begin
        hp2 := curr_heap_ptr^.heap_prev;
        DISPOSE( curr_heap_ptr );
        curr_heap_ptr := hp2
      end;
    curr_heap_ptr^.heap_htop := top;
    curr_mark_ptr := mrk
  end
end PAS$$RELEASE;




{ **** String (virtual) Input/Output **** }


procedure VREAD_END( fp: fild_ptr );
var
  veq: record case boolean of
    false:( pr: $wild_pointer );
    true:(  pi: ^integer )
  end;

begin
  with str_descr do
  begin
    veq.pr := fild_rel;
    if fild_bufcnt > 0 then                     { String (File Buffer) Empty ? }
    begin
      { No just adjust the pointer when specified }
      if fild_rel <> nil then veq.pi^ := fild_bufsize - fild_bufcnt + 1
    end
    else
    begin                                       { Set the end of string => that can be rise error }
      fild_bufcnt := 0;                         { Set eof reached }
      fild_curr   := spc_arr"address;
      fild_state  := fild_state + [lio_eof,lio_eol,lio_est];
      if fild_rel <> nil then veq.pi^ := -1
    end
  end
end VREAD_END;



[global 'PAS__VREAD_INIT']
procedure VREAD_INIT( in_var str: packed array[sz: integer] of char; var ip: [optional] integer );
var
  ic: integer;

begin
  if ip"address <> nil then ic := ip            { Get the start index to use }
                       else ic :=  1;
  with str_descr do
  begin
    fild_curr    := str[ic]"address;            { Init the current character ptr. }
    fild_buf     := str"address;                { Set the string body address }
    fild_rel     := ip"address;                 { Set the index pointer }
    fild_pfname  := str"address;                { Set the string address }
    PAS__INS_PROC( fild_pro, VREAD_END );       { Install the end string procedure }
    fild_bufsize := sz;                         { Set the buffer length to string the length }
    fild_bufcnt  := sz - ic + 1;                { Init the character count down }
    fild_state   := [lio_inp,lio_vir,lio_txt,lio_std]   { Set the init state }
  end;
  fild_c_inp := str_descr"address               { Set as the current output stream }
end VREAD_INIT;



procedure TWRITE_END( fp: fild_ptr );
var
  veq: record case boolean of
    false:( pr: $wild_pointer );
    true:(  pi: ^integer )
  end;

begin
  with str_descr do
  begin
    veq.pr := fild_rel;
    if fild_rel <> nil then
      veq.pi^ :=  fild_bufcnt                   { Update the User Byte Count Variable }
    else
      for ii := fild_bufcnt + 1 to fild_bufsize do  fild_buf^[ii] := ' ';

    if lio_est in fild_state then PAS__ERROR( 54 )
                             else fild_state := fild_state + [lio_eof,lio_est]
  end
end TWRITE_END;



[global 'PAS__TWRITE_INIT']
procedure TWRITE_INIT( var str: packed array[sz: integer] of char; var  ip: [optional] integer );
var
  ic: integer;

begin
  if ip"address <> nil then ic := ip            { Get the start index to use }
                       else ic := 0;

  with str_descr do
  begin
    fild_curr    := str[ic+1]"address;          { Init the current character ptr. to first free byte in the array }
    fild_buf     := str"address;                { Set the Character Array address }
    fild_rel     := ip"address;                 { Set the index pointer }
    fild_pfname  := str"address;                { Set the string address }
    PAS__INS_PROC( fild_pro, TWRITE_END );      { Install the end string proc }
    fild_bufsize := sz;                         { Init the buffer/string size ... }
    fild_bufcnt  := ic;                         { ... and continue on the same string }
    fild_state   := [lio_out,lio_vir,lio_txt,lio_std]   { Set the init state }
  end;
  fild_c_out := str_descr"address               { Set as the current output stream }
end TWRITE_INIT;



procedure VWRITE_END( fp: fild_ptr );
begin
  with str_descr do
  begin
    fild_pfname^.length := fild_bufcnt;        { Set the string used size }
    if lio_est in fild_state then PAS__ERROR( 54 )
                             else fild_state := fild_state + [lio_eof,lio_est]
  end
end VWRITE_END;



[global 'PAS__VWRITE_INIT']
procedure VWRITE_INIT( var str: string; bini: boolean := true );
begin
  with str, str_descr do
  begin
    fild_buf      := body"address;              { Set the string body address }
    fild_pfname   := str"address;               { Set the string address }
    PAS__INS_PROC( fild_pro, VWRITE_END );      { Install the end string proc }
    fild_bufsize  := capacity;                  { Init the buffer/string size }
    if bini then
    begin
      fild_curr   := body"address;              { Init to the begin of string body }
      fild_bufcnt := 0                          { Init the character count ... }
    end
    else
    begin
      fild_curr   := body[length+1]"address;    { Set to the first free character }
      fild_bufcnt := length                     { and continue on the same string }
    end;
    fild_state   := [lio_out,lio_vir,lio_txt,lio_std]   { Set the init state }
  end;
  fild_c_out := str_descr"address               { Set as the current output stream }
end VWRITE_INIT;



[global]
procedure BUFFER_MAP( fp: fild_ptr; ibuf: integer );
var
  ir, nb: integer;

begin
  with fp^ do
  begin
    if lio_win in fild_state and fild_bufcnt > 0 then
    begin
      ir := CC__SEEK( fild_cfile, fild_curpos*fild_bufsize, cc__seek_set );
      nb := CC__WRITE( fild_cfile, fild_buf, fild_bufcnt );
      fild_state := fild_state + [lio_emp] - [lio_win]
    end;
    if ibuf >= 0 then
    begin
      fild_curpos := CC__SEEK( fild_cfile, ibuf*fild_bufsize, cc__seek_set )
                           div fild_bufsize;
      if lio_inp in fild_state then
        nb := CC__READ( fild_cfile, fild_buf, fild_bufsize )
      else
        nb := 0;
      fild_bufcnt := nb;
      if nb < 0 then PAS__ERROR( CC__ERROR )
      else
      begin
        if nb = 0 then fild_state := fild_state + [lio_eof]
                  else fild_state := fild_state - [lio_eof,lio_emp];
        fild_curr  := fild_buf^"address;
        fild_index := 0;
        if lio_out in fild_state then
          for i := nb to fild_bufsize-1 do fild_buf^[i] := char( 0 );
        fild_dcnt  := nb
      end
    end
  end
end BUFFER_MAP;



procedure UGET_ELEM( fp: fild_ptr );
begin
  with fp^ do
  begin
    fild_index := fild_index + fild_objsize;
    fild_dcnt  := fild_dcnt  - fild_objsize;
    if fild_dcnt <= 0 then                      { Buffer end or EOF }
      if fild_bufcnt < fild_bufsize then        { EOF is reached }
        fild_state := fild_state + [lio_eof]
      else
        BUFFER_MAP( fp, fild_curpos+1 )
    else
      INCR_PTR( fild_curr, fild_objsize )
  end
end UGET_ELEM;



procedure UPUT_ELEM( fp: fild_ptr );
begin
  with fp^ do
  begin
    fild_state := fild_state + [lio_win];
    fild_index := fild_index + fild_objsize;
    if fild_index > fild_bufcnt then
    begin
      fild_dcnt := fild_dcnt + fild_index - fild_bufcnt;
      fild_bufcnt := fild_index
    end;
    if fild_index >= fild_bufsize then
      BUFFER_MAP( fp, fild_curpos+1 )
    else
    begin
      INCR_PTR( fild_curr, fild_objsize );
      fild_dcnt := fild_dcnt - fild_objsize
    end
  end
end UPUT_ELEM;



[global 'PAS__SEEK']
procedure PAS$$SEEK( fp:  fild_ptr; irec: integer );
var
  ibf, ish, nr: integer;

begin
  if fp <> nil then
  with fp^ do
    if lio_ran in fild_state then
    begin
      irec := irec - 1;
      nr   := fild_bufsize div fild_objsize;
      ibf  := irec div nr;
      ish  := irec - ibf * nr;
      if ibf <> fild_curpos then BUFFER_MAP( fp, ibf );
      fild_index := fild_objsize * ish;
      if fild_index > fild_bufcnt then fild_bufcnt := fild_index;
      fild_dcnt := fild_bufcnt - fild_index;
      fild_curr := fild_buf^[fild_index]"address;
      if fild_dcnt <= 0 then fild_state := fild_state + [lio_eof]
                        else fild_state := fild_state - [lio_eof]
    end
    else PAS__ERROR( 101 )
  else PAS__ERROR( 100 )
end PAS$$SEEK;



[global 'PAS_put_buffer']
procedure PUT_BUFFER( fp: fild_ptr );
{ Put a buffer into a sequential file }
var
  sz, nc: integer;

begin
  with fp^ do
  begin
    nc := fild_bufcnt;
    { We write the buffer }
    if nc > 0 then
    begin
      sz := CC__WRITE( fild_cfile, fild_buf, nc );
      if sz = -1 then PAS__ERROR( CC__ERROR )
    end;
    fild_curr := fild_buf^"address;
    { Set PUT DONE flag and Clear buffer WRITE in anf EOF }
    fild_state  := fild_state - [lio_win, lio_eof];
    fild_bufcnt := 0
  end
end PUT_BUFFER;



procedure PUT_FILE_ELEM( fp: fild_ptr );
begin
  with fp^ do
    if lio_ran in fild_state then
      UPUT_ELEM( fp )                                   { Random Access File }
    else
    begin                                               { Sequential file }
      fild_bufcnt := fild_bufcnt + fild_objsize;
      if fild_bufcnt >= fild_bufsize then               { Buffer is full }
        if lio_vir in fild_state then PAS__EXE_PROC( fild_pro, fp )
                                 else PUT_BUFFER( fp )
      else
      begin                                             { Buffer not full }
        { For a true Text File }
        if (not (lio_vir in fild_state)) and (lio_txt in fild_state) and
           { ... on an interactive device or VMS O.S. }
           ([lio_pip,lio_tty,lio_std]*fild_state <> []) and
           { ... writing a LN character => Output the Buffer }
           (fild_curr^ = eoln_char) then PUT_BUFFER( fp )
        else
        begin                                           { Else we update the byte count and state }
          fild_state := fild_state + [lio_win];
          INCR_PTR( fild_curr, fild_objsize )
        end
      end
    end
end PUT_FILE_ELEM;



[global 'PAS__PUT']
procedure PUT_ELEM( fp: fild_ptr );
{ Standard PASCAL PUT procedure }
begin
  if fp <> nil then
    if lio_out in fp^.fild_state then PUT_FILE_ELEM( fp )       { Not an output file => error }
                                 else PAS__ERROR( 111 )
  else PAS__ERROR( 100 )                                { Not an opened file => error }
end PUT_ELEM;



procedure FLUSH_ASSOC_TERM( fp: fild_ptr );
{ Routine to flush the buffer on all output associated file,
  Used for any interactive associated files (TTY/PIPE/SOCKET...) }
var
  fd1: fild_ptr;

begin

  fd1 := fp^.fild_rel;
  while (fd1 <> nil) and (fd1 <> fp) do
  begin
    if [lio_win,lio_txt,lio_out] <= fd1^.fild_state then PUT_BUFFER( fd1 );
    fd1 := fd1^.fild_rel
  end
end FLUSH_ASSOC_TERM;



[global 'PAS_get_buffer']
procedure GET_BUFFER( fp: fild_ptr );
{ Get a new buffer for sequential file }
{ For the Interactive associated file, Flush any write associated buffer(s) }
var
  nc: integer;

begin
  with fp^ do
  begin
    { For related output file: Output any not empty buffer(s) }
    if not (lio_ran in fild_state) then
      if lio_txt in fild_state then
      begin { Associated output files on the same terminal must be flush }
        if fild_rel <> nil then FLUSH_ASSOC_TERM( fp )
      end
      else                                              { Sequential Update mode }
        if lio_win in fild_state then PUT_BUFFER( fp );

    { For terminal input file, Ignore any EOF }
    if lio_tty in fild_state then
      fild_state := fild_state - [lio_eof];             { Ignore any terminal EOF }

    { If EOF is always set, then ERROR code return }
    if lio_eof in fild_state then
      fild_bufcnt := -1                                 { Set eof override tentative error }
    else
    begin
      { We get a new buffer from the file }
      fild_curr  := fild_buf^"address;
      fild_state := fild_state - [lio_eol, lio_emp];    { Delete any eoln mark }

      { We fill the empty buffer }
      nc := CC__READ( fild_cfile, fild_buf, fild_bufsize );

      if nc < 0 then PAS__ERROR( CC__ERROR );

      if nc = 0 then { EOF seen }
        { EOF at begin of record => set EOF seen flag and, for text EOLN }
        if lio_txt in fild_state then
        begin
          fild_state := fild_state + [lio_eol,lio_eof];
          fild_curr^ := eoln_char ;
          nc         := 1
        end
        else                                            { Set EOF seen at end of buffer }
          fild_state := fild_state + [lio_eof];
      fild_bufcnt := nc;                                { Return the bytes number or -1 (for error) }
    end
  end
end GET_BUFFER;



procedure GET_FILE_ELEM( fp: fild_ptr );
{ Get a file element }
{ The bypass tentative of EOF is ignored on Terminal/Not on PIPE/SOCKET... }
var
  nc:  integer;

begin
  with fp^ do
    if lio_ran in fild_state then UGET_ELEM( fp )
    else
      if (lio_tty in fild_state) or not (lio_eof in fild_state) then
      begin { No file EOF reached or Terminal mode }
        nc := fild_bufcnt - fild_objsize;
        if nc > 0 then
        begin { One or more objects are present in the buffer }
              { The size of buffer must be a multiple of the object size }
          fild_bufcnt := nc;
          INCR_PTR( fild_curr, fild_objsize );
          if lio_txt in fild_state then                 { On a text file }
            if fild_curr^ = eoln_char then
            begin { Text file can be have an EOLN mark }
                  { End of line seen }
              fild_state := fild_state + [lio_eol];     { Set EOL flag }
              if lio_vir in fild_state then fild_curr^ := ' '
            end
            else
              fild_state := fild_state - [lio_eol];     { Clear EOLN flag }
        end
        else                                            { The buffer is empty (nc <= 0) }
          if lio_txt in fild_state then                 { On a text file only }
            if not (lio_lzy in fild_state) or           { When not lasy mode or ... }
               (lio_emp in fild_state) then             { ... lazy mode with empty state (UFB is set) }
            begin { Get a new buffer }
              fild_bufcnt := 0;                         { Set the Buffer as Empty }
              if lio_vir in fild_state then PAS__EXE_PROC( fild_pro, fp )
                                       else GET_BUFFER( fp );
              if fild_bufcnt > 0 then                   { Not empty buffer }
                if fild_curr^ = eoln_char then
                begin
                  fild_state := fild_state + [lio_eol]; { Set EOL flag }
                  fild_curr^ := ' '                     { Simule a space in place of eoln character for f^ access }
                end
                else
                  fild_state := fild_state - [lio_eol]; { Clear EOLN flag }
              if fild_bufcnt < 0 then PAS__ERROR( 104 ) { Error on EOF override }
            end
            else                                        { Empty buffer (UFB to set) }
            begin
              fild_bufcnt := 0;                         { Clear the buffer count }
              if nc = 0 then
                fild_curr := spc_arr"address;           { Give a space character by default }
              fild_state := fild_state + [lio_emp] - [lio_eol]; { Set UFB flag (empty buffer) }
            end
          else                                          { Binary file => Get a new buffer }
          begin
            if lio_vir in fild_state then PAS__EXE_PROC( fild_pro, fp )
                                     else GET_BUFFER( fp );
            if fild_bufcnt < 0 then PAS__ERROR( 104 );  { Try to skip it }
          end
      end
      else PAS__ERROR( 104 )                            { EOF Reached }
end GET_FILE_ELEM;



[global 'PAS__GET']
procedure GET_ELEM( fp: fild_ptr );
{ Standard PASCAL GET procedure }
begin
  if fp <> nil then
    if lio_inp in fp^.fild_state then GET_FILE_ELEM( fp )
                                 else PAS__ERROR( 105 )
  else PAS__ERROR( 100 )
end GET_ELEM;



procedure Pas_read_char( fp: fild_ptr; var ch: char );
begin
  if fp <> nil then
  with fp^ do
    if [lio_txt,lio_inp] <= fild_state then
    begin
      if lio_emp in fild_state then GET_FILE_ELEM( fp );
      ch := fild_curr^;
      GET_FILE_ELEM( fp )
    end
    else PAS__ERROR( 108 )
  else PAS__ERROR( 100 )
end Pas_read_char;



[global 'PAS__READ_CHAR']
procedure READ_CHAR( var ch: char );
begin
  Pas_read_char( fild_c_inp, ch )
end READ_CHAR;



[global 'PAS__NEXT_CHAR']
procedure NEXT_CHAR( fp: fild_ptr; var ch: char );
begin
  if fp <> nil then
  with fp^ do
    if [lio_txt,lio_inp] <= fild_state then
      if [lio_emp,lio_eol,lio_eof,lio_est]*fild_state <> [] then
        ch := CHR( 0 )
      else
        ch := fild_curr^
    else PAS__ERROR( 108 )
  else PAS__ERROR( 100 )
end NEXT_CHAR;



[global 'PAS__EOLN']
function TEST_EOLN( fp: fild_ptr ): boolean;
var
  flg: boolean;

begin
  if fp <> nil then
  with fp^ do
    if [lio_txt,lio_inp] <= fild_state then
    begin
      flg := (lio_eol in fild_state);
      if not (lio_eof in fild_state) then
        if lio_emp in fild_state then
        begin
          GET_FILE_ELEM( fp );
          flg := (lio_eol in fild_state)
        end
    end else  PAS__ERROR( 110 )
  else PAS__ERROR( 100 );
  TEST_EOLN := flg
end TEST_EOLN;



[global 'PAS__EOLN_INP']
function TEST_EOLN1: boolean;
begin
  TEST_EOLN1 := TEST_EOLN( input_file );
end TEST_EOLN1;



[global 'PAS__UFB']
function TEST_UFB( fp: fild_ptr ): boolean;
{ Test Undefined buffer pointer (or buffer empty) }
var
  flg: boolean;

begin
  if fp <> nil then
  with fp^ do
    if lio_inp in fild_state then flg := lio_emp in fild_state
                             else PAS__ERROR( 106 )
  else PAS__ERROR( 100 );
  TEST_UFB := flg
end TEST_UFB;



[global 'PAS__UFB_INP']
function TEST_UFB1: boolean;
begin
  TEST_UFB1 := TEST_UFB( input_file )
end TEST_UFB1;



[global 'PAS__EOF']
function TEST_EOF( fp: fild_ptr ): boolean;
var
  flg: boolean;

begin
  if fp <> nil then
  with fp^ do
    if lio_inp in fild_state then
    begin
      flg := (lio_eof in fild_state);
      if not flg then
        if [lio_emp,lio_txt] <= fild_state then
        begin
          GET_FILE_ELEM( fp );
          flg := (lio_eof in fild_state)
        end
    end else PAS__ERROR( 106 )
  else PAS__ERROR( 100 );
  TEST_EOF := flg
end TEST_EOF;



[global 'PAS__EOF_INP']
function TEST_EOF1: boolean;
begin
  TEST_EOF1 := TEST_EOF( input_file );
end TEST_EOF1;



[global 'PAS__CURR_EOLN']
function TEST_CURR_EOLN: boolean;
begin
  TEST_CURR_EOLN := TEST_EOLN( fild_c_inp )
end TEST_CURR_EOLN;



[global 'PAS__CURR_EOF']
function TEST_CURR_EOF: boolean;
begin
  TEST_CURR_EOF := TEST_EOF( fild_c_inp )
end TEST_CURR_EOF;



procedure Pas_read_eoln( fp: fild_ptr );
begin
  if fp <> nil then
  with fp^ do
    if [lio_txt,lio_inp] <= fild_state then
    begin
      { Load a buffer when it is empty }
      if lio_vir in fild_state then PAS__EXE_PROC( fild_pro, fp )
      else
      begin
        if lio_emp in fild_state then GET_BUFFER( fp );
        { Search for an EOLN mark }
        while not (lio_eol in fild_state) do GET_FILE_ELEM( fp );
        { Skip the current EOLN }
        if not (lio_eof in fild_state) then GET_FILE_ELEM( fp )
      end
    end
    else PAS__ERROR( 108 )
  else PAS__ERROR( 100 )
end Pas_read_eoln;



[global 'PAS__READ_EOLN']
procedure READ_EOLN;
begin
  if fild_c_inp <> nil then
  with fild_c_inp^ do
    if lio_bin in fild_state then
      if lio_vir in fild_state then PAS__EXE_PROC( fild_pro, fild_c_inp )
                               else GET_BUFFER( fild_c_inp )
    else PAS_read_eoln( fild_c_inp )
end READ_EOLN;



[global 'PAS__READ_BIN']
procedure READ_BIN( vp: buf_ptr );
var
  bp: ^char;

begin
  if fild_c_inp <> nil then
  with fild_c_inp^ do
    if [lio_inp, lio_bin] <= fild_state then
    begin
      { Input binary file }
      bp := fild_curr;
      for i := 0 to fild_objsize-1 do
      begin
        vp^[i] := bp^; INCR_PTR( bp, 1 )
      end;
      GET_FILE_ELEM( fild_c_inp )
    end
    else PAS__ERROR( 107 )
  else PAS__ERROR( 100 )
end READ_BIN;



procedure Pas_write_char( fp: fild_ptr; ch: char );
begin
  if fp <> nil then
  with fp^ do
    if [lio_txt,lio_out] <= fild_state then
    begin
      fild_curr^ := ch;
      PUT_FILE_ELEM( fp )
    end
    else PAS__ERROR( 114 )
  else PAS__ERROR( 100 )
end Pas_write_char;



[global 'PAS__WRITE_EOLN']
procedure WRITE_EOLN;
begin
  if fild_c_out <> nil then
  with fild_c_out^ do
    if (lio_bin in fild_state) or (fild_c_out = str_descr"address) then
      if lio_vir in fild_state then PAS__EXE_PROC( fild_pro, fild_c_out )
                               else PUT_BUFFER( fild_c_out )
    else PAS_write_char( fild_c_out, eoln_char )
end WRITE_EOLN;




[global 'PAS__WRITE_CHAR']
procedure WRITE_CHAR( ch: char );
begin
  Pas_write_char( fild_c_out, ch )
end WRITE_CHAR;



[global 'PAS__WRITE_BIN']
procedure WRITE_BIN( vp: buf_ptr );
var
  bp: ^char;

begin
  if fild_c_out <> nil then
  with fild_c_out^ do
    if [lio_out, lio_bin] <= fild_state then
    begin
      { Output binary file }
      bp := fild_curr;
      for i := 0 to fild_objsize-1 do
      begin
        bp^ := vp^[i]; INCR_PTR( bp, 1 )
      end;
      PUT_FILE_ELEM( fild_c_out )
    end
    else PAS__ERROR( 112 )
  else PAS__ERROR( 100 )
end WRITE_BIN;



procedure PAS_UNASSOCIATED_FILE( fp: fild_ptr );
var
  fd0, fd1: fild_ptr;

begin
  { Suppress the io relation for associated input/output files }
  fd1 := fp^.fild_rel;
  if (fd1 <> nil) and (fd1 <> fp) then
  begin { More associated files }
    while (fd1 <> nil) and (fd1 <> fp) do
    begin  fd0 := fd1; fd1 := fd1^.fild_rel  end;
    { fd0 -> previous of fp one's }
    fd0^.fild_rel := fp^.fild_rel;
    if fd0^.fild_rel = fd0 then fd0^.fild_rel := nil
  end;
  fp^.fild_rel := nil
end PAS_UNASSOCIATED_FILE;



[global 'PAS__file_close']
procedure FILE_CLOSE( var fp: fild_ptr; disp: flags_file );
var
  stn:   string;
  ierr: integer;

begin
  if fp <> nil then
  with fp^ do
  begin
    if (lio_out in fild_state) and (fild_bufcnt > 0) then
      { To write the objects in the PASCAL buffer }
      if lio_ran in fild_state then BUFFER_MAP( fp, -1 )
      else
      begin
        if [lio_txt,lio_tty] <= fild_state then
          PAS_write_char( fp, eoln_char );
        if lio_vir in fild_state then PAS__EXE_PROC( fild_pro, fp )
                                 else PUT_BUFFER( fp )
      end;

    fild_bufcnt := 0;

    if not (lio_std in fild_state) then
    begin { We must close the C file }
      if fild_rel <> nil then PAS_UNASSOCIATED_FILE( fp );
      if fild_cfile <> -1 then CC__CLOSE( fild_cfile );

      io_errcode := 0;

      if print_file in disp then fild_state := fild_state + [lio_pri];
      if del_file   in disp then fild_state := fild_state + [lio_del];

      if fild_state*[lio_pri,lio_blk] = [lio_pri,lio_blk] then
      begin
        { To perform a print or a print/del }
        if lio_del in fild_state then
        begin
          ierr := GET_LOGICAL( stn, 'CPAS_PRINT_AND_REMOVE' );
          if ierr <> 0 then io_errcode := 150
        end
        else
        begin
          ierr := GET_LOGICAL( stn, 'CPAS_PRINT' );
          if ierr <> 0 then io_errcode := 151
        end;
        if ierr = 0 then
        begin { Go to print process }
          stn := stn||' '||curr_fspc;
          ierr := CREATE_PROCESS( '', stn );
          if ierr > 0 then WAIT_PROCESS( ierr )
        end
      end
      else
        if lio_del in fild_state then 
          if CC__REMOVE( curr_fspc.body"address ) <> 0 then
            io_errcode := CC__ERROR;

      if (io_errcode <> 0) and not (error_file in disp) then
        PAS__ERROR( io_errcode );

      if fild_buf <> nil then
      begin
        PAS$$DISPOSE( fild_buf );
        fild_buf := nil
      end;
      if fild_pfname <> nil then
      begin
        PAS$$DISPOSE( fild_pfname );
        fild_pfname := nil
      end;
      PAS$$DISPOSE( fp );
      fp := nil
    end
  end
  else PAS__ERROR( 100 )
end FILE_CLOSE;



[global 'PAS__CLOSE']
procedure PAS$$CLOSE( var fp: fild_ptr; idisp: flags_file {?} );
{ Standard PASCAL CLOSE procedure }
{ Flush any output buffer and close the specified file }
{ The variable fp is the pascal file variable (a descriptor pointer) }
{ Call the FILE_CLOSE routine }

begin
  if fp <> nil then
  with fp^ do
  begin
    if not (lio_std in fild_state) then
    begin
      if fild_prv <> nil then fild_prv^.fild_nxt := fild_nxt;
      if fild_nxt <> nil then fild_nxt^.fild_prv := fild_prv;
      if fp = fild_lastptr then fild_lastptr := fild_prv
    end;

    FILE_CLOSE( fp, idisp );

    { Unselect any closed selected file }
    if fp = fild_c_inp then fild_c_inp  := input_file
    else
      if fp = fild_c_out then fild_c_out  := output_file;
    if not (lio_std in fild_state) then fp := nil
  end
  else
    PAS__ERROR( 100 );
end PAS$$CLOSE;



[global 'PAS__SET_FILEDEF']
function SET_FILEDEF( var str: string ): boolean;
var
  len:         integer;
  def:          string;
  sep:            char;
  bok, broot:  boolean;

begin
  bok := true;                         { Assume success }
  if str[1] <> '/' then broot := INDEX( str, ':' ) <> 0
                   else broot := true;
  if not broot then
  begin { Complete any incomplete path }
    def := GET_DEF_DIR;                                 { Get the default Path (pwd) }
    len := def.length;
    if len > 0 then
    begin
      if system_wind or system_cygw then
      begin
        if system_cygw then sep := '/'
                       else sep := '\';
        if (str[1] = '\') or (str[1] = '/') then        { Root without device }
        begin
          len := INDEX( def, ':' );                     { Keep the device only (in the default directory) }
          def.length := len
        end
        else
        if (def[len] <>'\') and (def[len] <> '/') then  { Else when no directory separator at end ... }
        begin
          len := len + 1;                               { ... we append it. }
          def := def||sep
        end
      end
      else
      begin
        if def[len] <> '/' then        { When No Unix final '/' ... }
        begin
          len := len + 1;              { ... we append it. }
          def := def||'/'
        end;
        sep := '/'
      end;

      if (str[1] = '.') and (str[2] = sep) then { The old reference was begining by a '.' }
      begin
        len := len + str.length - 2;   { Suppress the not usefull local reference ... }
        str := def||SUBSTR( str, 3 )   { ... append the local path to the default path }
      end
      else
      begin
        len := len + str.length;       { Compute the new file specification size }
        str := def||str                { Append the local path to the default path }
      end;
      if len > str.capacity then bok := false { String overflow }
    end 
    else bok := false                   { No default location }
  end;
  SET_FILEDEF := bok
end SET_FILEDEF;



[global 'PAS__SET_FILESPC']
function SET_FILESPC( var     trg: string;
                      in_var  src: string;
                             imod: flags_file ): boolean;

var
  ll, ls:  integer;
  ch:      char;
  buf:     string;
  fl:      boolean;

begin
  ll := 0;
  ls := 0;
  { Copy to suppress any end-space or control character }
  for i := 1 to src.length do
  begin
    ch := src[i];
  exit if ch = CHR( 0 );                { Stop on a null character }
    if (case_dis_file in imod)  then    { Change character to minor case when required }
      if (ch >= 'A') and (ch <= 'Z') then ch := CHR( ORD( ch ) + inmin );
    if ch = ' ' then
    begin { To elliminate the end space(s) }
      if ls = 0 then ls := i - 1        { If it is the first space of a space series }
    end
    else ls := 0;                       { else when it is not a space }
    if ch >= ' ' then
    begin
      ll := ll + 1;
      buf[ll] := ch
    end
  end;
  if ls = 0 then buf.length := ll       { Set the end of string }
            else buf.length := ls;      { Set the end of string without the end-space }

  fl := true; 
  if not (nolog_file in imod) then fl  := GET_PHYSIC_FSPC( trg, buf ) = 0
                              else trg := buf;
  if trg.length > 0 then                { No error }
  begin
    SET_FILEDEF( trg );
    { Append a null character for C use }
    if trg.length >= trg.capacity then
    begin
      trg.length := trg.capacity - 1;
      fl := false
    end;
    trg[trg.length+1] := char( 0 )
  end;
  SET_FILESPC := fl
end SET_FILESPC;



[global 'PAS__ASS_INTERACTIVE_FILES']
procedure ASSOCIATE_INTER_FILES( fd : fild_ptr ) ;
var
  fd0, fd1: fild_ptr;
  tty:      ^string;
  sl1, sl2: integer;

begin
  if fd <> nil then
  with fd^ do
  begin
    if [lio_tty,lio_pip]*fild_state <> [] then
    begin
      tty := fild_pfname;
      fd0 := nil;
      fd1 := fild_firstptr;             { Look for the same terminal/pipe device }
      while (fd1 <> nil) and (fd0 = nil) do
      begin
        if (fd1 <> fd) and ([lio_tty,lio_pip]*fd1^.fild_state <> []) then
          if fd1^.fild_pfname^ = tty^ then fd0 := fd1;
        fd1 := fd1^.fild_nxt
      end;
      if fd0 <> nil then
      begin { Two terminal/pipe with the same device access name }
            { Build the Open Interactive Specific File List }
        if fd0^.fild_rel = nil then fild_rel := fd0
                               else fild_rel := fd0^.fild_rel;
        fd0^.fild_rel := fd
      end
    end
  end
  else PAS__ERROR( 100 )
end ASSOCIATE_INTER_FILES;



[global 'PAS__FILE_KIND']
function FILE_KIND( imod: integer ): integer;
{ ** Fonction to get the kind of a file ** }
var
  ret: integer;

begin
  if TEST_ISDIR( imod ) <> 0 then ret := 0      { Directory }
  else
  if TEST_ISREG( imod ) <> 0 then ret := 1      { Regular File }
  else
  if TEST_ISLNK( imod ) <> 0 then ret := 2      { Symbolic Link }
  else
  if TEST_ISCHR( imod ) <> 0 then ret := 3      { Device in Character mode }
  else
  if TEST_ISBLK( imod ) <> 0 then ret := 4      { Device in Block mode }
  else
  if TEST_ISFIFO( imod )<> 0 then ret := 5      { FIFO/PIPE }
  else
  if TEST_ISSOCK( imod )<> 0 then ret := 6      { Socket }
  else ret := -1;
  FILE_KIND := ret
end FILE_KIND;



[global 'PAS__GF_INFO']
function  GF_INFO( in_var fst: string;
                      var tbi: array[ss: integer] of cc__int;
                     lnk, gmt: boolean := true;
                         umsk: flags_file := [] ): integer;
var
  ierr: integer;

begin
  if not SET_FILESPC( curr_fspc, fst, umsk ) then ierr := 121
  else
    if curr_fspc.length = 0 then ierr :=  201
    else
      ierr := CC_GET_FILE_INFO( curr_fspc.body"address, tbi, lnk, gmt );
  GF_INFO := ierr
end GF_INFO;



[global 'PAS__FSPC_GET_KIND']
function FSPC_GET_KIND( in_var spc: string;
                               lnk, gmt: boolean := true;
                               umsk: flags_file := [] ): integer;
{ ** Fonction to get the kind of a file form file specification ** }
{ Call GET_FILE_INFO (alias PAS__GF_INFO) and deduce the File Kind }
var
  ierr: integer;

begin
  ierr := GF_INFO( spc, tbfch, lnk, gmt, umsk );
  if ierr = 0 then FSPC_GET_KIND := FILE_KIND( tbfch[1] )       { Regular Ref. }
              else FSPC_GET_KIND := -10                         { Bad File/Dir. spc. }
end FSPC_GET_KIND;



function GET_KIND_OF_FILE( c_file: integer ): integer;
{ ** Fonction to get the kind of a file from file descriptor ** }
var
  ierr: integer;

begin
  ierr := CC_GET_DESCR_INFO( c_file, tbfch, false, false );
  if ierr = 0 then GET_KIND_OF_FILE := FILE_KIND( tbfch[1] )    { Regular Ref. }
              else GET_KIND_OF_FILE := -10                      { Bad File/Dir. spc. }
end GET_KIND_OF_FILE;



{ ** Procedure to init the file structure pointers ** }
procedure INIT_A_FILE( fd: fild_ptr; in_var pfspc: string );
type
  dvsty = packed array[0..127] of char;

var
  bf:     $wild_pointer;
  sz, ic: integer;
  btt:    boolean;

begin
  btt := false;
  with fd^ do
  begin
    { Insert the file in the list of opened files }
    if fild_firstptr = nil then fild_firstptr := fd
                           else fild_lastptr^.fild_nxt := fd;
    fild_nxt     := nil;
    fild_prv     := fild_lastptr;
    fild_lastptr := fd;

    { Set File characteristics }
    case GET_KIND_OF_FILE( fild_cfile ) of
      3: { Device in Character mode } fild_state := fild_state + [lio_lzy];
      4: { Device in Block     mode } fild_state := fild_state + [lio_blk];
      5: { Pipe/Fifo Device    mode } fild_state := fild_state + [lio_pip,lio_lzy];
      6: { Socket Device       mode } fild_state := fild_state + [lio_pip,lio_soc,lio_lzy];
    otherwise
    end;

    { Check for a terminal file }
    if [lio_pip,lio_soc]*fild_state = [] then
      if CC__ISATTY( fild_cfile ) <> 0 then
      begin
        fild_state := fild_state + [lio_tty,lio_lzy];
        { Try to link with another file on the same TTY when required }
        btt := true                     { btt = true/false for terminal/not_a_terminal } 
      end;

    { Set the Specified filename }
    NEW( fild_pfname, pfspc.length + 1 );
    with fild_pfname^ do
    begin
      length := pfspc.length;
      for i := 1 to length do  body[i] := pfspc.body[i];
      body[length+1] := CHR( 0 )
    end;
    bf           := MEM$$ALLOC( fild_bufsize );
    fild_curr    := bf;
    fild_buf     := bf;
    fild_rel     := nil;
    fild_bufcnt  := 0;
    fild_curpos  := 0;
    fild_index   := 0;
    fild_dcnt    := 0;

    if not (lio_lzy in fild_state) and (lio_inp in fild_state) then
      if lio_ran in fild_state then BUFFER_MAP( fd, 0 )
      else
        if lio_vir in fild_state then PAS__EXE_PROC( fild_pro, fd )
                                 else GET_BUFFER( fd )
  end;
  if btt then ASSOCIATE_INTER_FILES( fd )
end INIT_A_FILE;



[global 'PAS__OPEN']
procedure PAS$$OPEN( var      fp:     fild_ptr; { The Pascal file variable }
                         objsize:      integer; { size element file or 0 (text) }
                  in_var filespc:       string; { The filename specification }
                            umsk:   flags_file; { The flag I/O required mode }
                            ibuf,               { Size of buffer and ... }
                           iprot:      integer  { ...  file protection }
                   );
var
  dev, fil, ner:       integer;
  mope:              opeorinty;
  fnamp:            string_ptr;
  ioflg:             cc__setty;
  fstate:            lio_flags;
  bsuper:              boolean;

begin
  { if fp <> nil then PAS$$CLOSE( fp, 0 ); }    { A file was opened => Close it }

  { Set the user specified mode }
  bsuper       := false;
  ner          :=  0;
  ioflg        := [];
  fstate       := [];
  io_errcode   :=  0; { Until shown otherwise }

  if iprot < 0 then iprot := 7*64 + 5*8 + 5;    { Set o:rew,g:re,w:re prot. }

  if ner = 0 then
    if not SET_FILESPC( curr_fspc, filespc, umsk ) then ner := 121;

  if curr_fspc.length = 0 then ner :=  201;

  if ner = 0 then
  begin
    ner := CC_GET_FILE_SINF( curr_fspc.body"address, tbfch, 0, 0 );
    if ner = 0 then dev := FILE_KIND( tbfch[1] )
               else begin
                      ner := CC__ERROR;
                      if ner = 202 then { Assume creation of regular file when not existing ref. }
                      begin  ner := 0; dev := 1  end
                    end
  end;

  if ner = 0 then { OK when regular or not existing file }
  begin
    mope.sv := umsk * [read_file, write_file, append_file];

    case mope.iv of
      ifl_read: { READ }
        begin
          fstate := [lio_inp, lio_emp];
          if umsk*[new_file,unknown_file,old_file] = [] then umsk := umsk + [old_file];
          ioflg  := cc__o_rdonly
        end;

      ifl_write: { WRITE }
        begin
          fstate := [lio_out];
          if umsk*[new_file,unknown_file,old_file] = [] then umsk := umsk + [new_file];
          ioflg  := cc__o_wronly;
          bsuper := not (direct_file in umsk)
        end;

      ifl_update: { READ + WRITE }
        if objsize > 0 then
        begin
          fstate := [lio_inp, lio_out, lio_emp, lio_ran];
          if umsk*[new_file,unknown_file,old_file] = [] then umsk := umsk + [unknown_file];
          ioflg  := cc__o_rdwr
        end
        else ner := 120;

      ifl_append, ifl_append1: { APPEND }
        begin
          fstate := [lio_out];
          if umsk*[new_file,unknown_file,old_file] = [] then umsk := umsk + [unknown_file];
          ioflg  := cc__o_wronly + cc__o_append
        end;

    otherwise { OTHERS BITS = choice of READ/WRITE/OTHER by NEW/OLD/UNK }
      ner := 120;
    end
  end;

  if dev = 1 then
  begin { For regular file only }
    if exclude_file in umsk then ioflg := ioflg + cc__o_excl;
    if new_file in umsk then
      { Create a new version of the file }
      ioflg := ioflg + (cc__o_trunc + cc__o_creat)
    else
      if bsuper then                            { Sequential Write only mode => Supershed existing File }
        ioflg := ioflg + (cc__o_creat + cc__o_excl)
      else
        if unknown_file in umsk then ioflg := ioflg + cc__o_creat;

    if direct_file in umsk then
      if objsize > 0 then fstate := fstate + [lio_ran];

    if print_file in umsk then fstate := fstate + [lio_pri];
    if del_file in umsk then fstate := fstate + [lio_del]
  end;

  if ner <> 0 then
  begin
    io_errcode := ner;
    if not (error_file in umsk) then PAS__ERROR( ner )
  end
  else
  begin { OK To try open }
    if system_wind and text_file in umsk then
      if objsize <= 0 then ioflg := ioflg + cc_wnt__text
                      else ioflg := ioflg + cc_wnt__binary;

    fil := CC__OPEN(  curr_fspc.body"address, ioflg, iprot );

    { Check for write mode by supersheding of existing file }
    if bsuper and (fil < 0) then
    begin { Open error can be the result of pre-existing SEQ. Write file }
      ner := CC__ERROR;
      if (lio_out in fstate) and
         ((ner = 206 {privilege}) or (ner = 209 {file exists})) then
        if CC__REMOVE( curr_fspc.body"address ) = 0 then        { We try to delete the old file }
        begin { Delete is a success }
          { Try to open the new file }
          fil := CC__OPEN(  curr_fspc.body"address, ioflg+cc__o_creat, iprot );
          ner := 0
        end
    end;


    { Test for Success on Open file by C }
    if (fil = -1) then  { Open failed }
    begin { C Open has failed }
      ner := CC__ERROR;
      io_errcode := ner;                        { Put it in iostatus for user control }
      { User do not want manage the open error }
      if not (error_file in umsk) then PAS__ERROR( ner )
    end
  end;

  if ner = 0 then
  begin { Full OPEN success }
    if (objsize <= 0) then                      { it is a TEXT FILE }
    begin
      objsize := 1;
      if (ibuf < 1) then ibuf := def_buf_size;
      fstate := fstate + [lio_txt]
    end
    else                                        { it is binary file }
    begin
      if (ibuf < 1) then                        { default buffer allocation }
        if (objsize >= def_buf_size) then ibuf := 1
                                     else ibuf := (2*def_buf_size)/objsize;
      fstate := fstate + [lio_bin]
    end;

    { Allocate all file structures }
    NEW( fnamp, curr_fspc.length+1 );           { Create the filename string }
    fnamp^ := curr_fspc;
    NEW( fp );                                  { Create the PASCAL File Descriptor }

    with fp^ do
    begin
      fild_cfile   := fil;                      { Assign the C file descritor }
      fild_objsize := objsize;
      fild_bufsize := objsize*ibuf;             { fild_bufsize in bytes }
      fild_state   := fstate;                   { Set the file status }

      INIT_A_FILE( fp, curr_fspc )              { Complete the descriptor init. }
    end { with fp^ }
  end { if ner = 0 }
end PAS$$OPEN;



[global 'PAS__CREATE_PIPE']
procedure CREATE$$PIPE( var pipe : pipe_descr;
                               id: integer := 0;
                             ftxt: boolean := false );
var
  iv: integer;

begin
  if system_wind then
    if ftxt then iv := CC__PIPE( pipe"address, 512, cc_wnt__text )
            else iv := CC__PIPE( pipe"address, 512, cc_wnt__binary )
  else
    iv := CC__PIPE( pipe"address );

  if iv <> 0 then PAS__ERROR( CC__ERROR )
             else pipe.pipe_idnum := id
end CREATE$$PIPE;



[global 'PAS__OPEN_PIPE']
procedure OPEN$$PIPE( var      fp: fild_ptr;    { The file variable }
                          objsize: integer;     { size of file element }
                      var    pipe: pipe_descr;  { Pipe descriptor }
                             umsk: flags_file;     { The flag I/O required mode }
                             ibuf: integer );   { Size of buffer }

var
  fil, ner, iv, ichan: integer;
  mope:                opeorinty;
  fnamp:               string_ptr;
  fstate:              lio_flags;

begin
  { Set the user specified mode }
  ner          :=  0;
  fstate       := [];
  io_errcode   :=  0;                           { Until shown otherwise }

  mope.sv := umsk * [read_file, write_file];

  case mope.iv of
    ifl_read: { READ }
      begin
        fstate := [lio_inp, lio_emp];
        ichan  := pipe.pipe_input
      end;

    ifl_write: { WRITE }
      begin
        fstate := [lio_out];
        ichan  := pipe.pipe_output
      end;

  otherwise { OTHERS BITS = choice of READ/WRITE/OTHER by NEW/OLD/UNK }
    ner := 120;
  end;

  if ner <> 0 then PAS__ERROR( ner )
  else
  begin { OK To try open }
    WRITEV( curr_fspc, '_PIPE_', pipe.pipe_idnum:-4 );  { Build a PIPE name }
    if not (usc_file in umsk) then
    begin
      fil := CC__DUP( ichan );
      ichan := fil
    end
    else
      fil   := ichan;

    { Test for Success on Open file by C }
    if (fil = -1) then                                  { Open failed }
    begin { C Open has failed }
      ner := CC__ERROR;
      io_errcode := ner;                                { Put it in iostatus for user control }
      { User do not want manage the open error }
      if not (error_file in umsk) then PAS__ERROR( ner )
    end
  end;

  if ner = 0 then
  begin { Full OPEN success }
    if (objsize <= 0) then                              { It is a TEXT FILE }
    begin
      objsize := 1;
      if (ibuf < 1) then ibuf := def_buf_size;
      fstate := fstate + [lio_txt]
    end
    else                                                { It is binary file }
    begin
      if (ibuf < 1) then                                { Default buffer allocatcpas_printion }
        if (objsize >= def_buf_size) then ibuf := 1
                                     else ibuf := (2*def_buf_size)/objsize;
      fstate := fstate + [lio_bin]
    end;

    { Allocate all file structures }
    NEW( fnamp, curr_fspc.length+1 );                   { Create the filename string }
    fnamp^ := curr_fspc;
    NEW( fp );                                          { Create the PASCAL File Descriptor }

    with fp^ do
    begin
      fild_cfile   := fil;                              { Assign the C file descritor }
      fild_objsize := objsize;
      fild_bufsize := objsize*ibuf;                     { Set fild_bufsize in bytes }
      fild_state   := fstate;                           { Set the file status }
    end; { with fp^ }

    INIT_A_FILE( fp, fnamp^ )                           { Complete the decsriptor init. }

  end { if ner = 0 }
end OPEN$$PIPE;



[global 'PAS__ASSOCIATE_FILES']
procedure PAS_ASSOCIATE_FILES( fold, fnew: fild_ptr );
{ Use Only to associate PIPE or Socket files }
var
  fd0, fd1: fild_ptr;

begin
  { To associate PIPE/SOCKET... Interactive files }
  if (fold = nil) and (fnew = nil) then PAS__ERROR( 100 );

  if (fold <> nil) or (fnew <> nil) then
    if (fold <> nil) and (fnew <> nil) then
    begin
      if not ([lio_txt,lio_pip] <= fold^.fild_state) then PAS__ERROR( 501 );
      if not ([lio_txt,lio_pip] <= fnew^.fild_state) then PAS__ERROR( 502 );

      if fnew^.fild_rel = nil then
      begin { Only possible for a not associated file }
        if fold^.fild_rel = nil then fnew^.fild_rel := fold
                                else fnew^.fild_rel := fold^.fild_rel;
        fold^.fild_rel := fnew
      end
    end
    else
    begin { To break relation between associated files }
      if fold <> nil then fnew := fold;
      PAS_UNASSOCIATED_FILE( fnew )
    end
end PAS_ASSOCIATE_FILES;



[global 'PAS__TTY_FILE']
function TTY_FILE( fp: fild_ptr ): boolean;
begin
  if fp <> nil then TTY_FILE := (lio_tty in fp^.fild_state)
               else  PAS__ERROR( 100 )
end TTY_FILE;



[global 'PAS__TTY_CLREOF']
function TTY_CLREOF( fp: fild_ptr ): boolean;
begin
  if fp <> nil then
  with fp^ do
  begin
    TTY_CLREOF := (lio_eof in fild_state);
    if [lio_tty,lio_eof] <= fild_state then
      fild_state := fild_state - [lio_eof] + [lio_emp]
  end
  else  PAS__ERROR( 100 )
end TTY_CLREOF;



[global 'PAS__FILE_STATE']
function FILE_STATE( fp: fild_ptr ): lio_flags;
begin
  if fp <> nil then
    FILE_STATE := fp^.fild_state*[lio_inp, lio_out, lio_ran, lio_bin, lio_dir,
                                  lio_std, lio_lzy, lio_blk, lio_tty, lio_pip,
                                  lio_soc, lio_vir, lio_pri, lio_del, lio_eol,
                                  lio_est, lio_emp, lio_eof]   { FILE_STATE mask }
  else PAS__ERROR( 100 )
end FILE_STATE;



[global 'PAS__FILE_SPC']
function FILE_SPECIFICATION( fp: fild_ptr ): string;
begin
  if fp <> nil then
  with fp^ do
    if fild_pfname <> nil then return fild_pfname^
                          else return ''
end FILE_SPECIFICATION;


[global 'PAS__INIT']
procedure PAS$$INIT( parc: integer; var parv, penv: argv_tabty );
const
  devtt  = '/dev/tty';
  ldevtt = 'TT';
  devnl  = '/dev/null';
  ldevnl = 'NL';

var
  l:                  integer;
  empty_str: [static] string(2)  := '';
  fspc:                string;
  bnm:                boolean;

  procedure CHECK_DESCR( lun: cc__int; fmd: cc__setty; var sn: string );
  const
    iprt = 7*64 + 5*8 + 5;                      { Set o:rew,g:re,w:re file protection }

  type
    ch_tb = array[0..127] of char;

  var
    p:                  ^ch_tb;
    tfl, ii, ie:       integer;
    ch:                   char;

  begin
    ie := GET_KIND_OF_FILE( lun );              { Get the kind of file }
    
    p := CC__TTYNAME( lun );                    { Try to find a terminal name }
    if p <> nil then
    begin                                       { It is a terminal file descriptor }
      ii := 0;
      repeat                                    { We get the terminal name as the file path }
        ch := p^[ii];
      exit if ch = CHR( 0 );
        ii := ii + 1;
        sn.body[ii] := ch;
      until ii >= sn.capacity;
      sn.length := ii
    end
(*  /// To set as usablean a not opened I/O index ///
    else
      if cc__errno = cc__bad_fldsc then         { Bad file index (When file index is not closed/not_opened/uninitialzed) }
      begin
        { Get the System specific Null device Path Name }
        if system_unix or system_cygw then sn := unx_dev_null
                                      else sn := wnt_dev_null;
        sn.body[sn.length+1] := CHR( 0 );       { Must be have a Nul terminal for C routines }
        { Open the file as a standard file }
        tfl := CC__OPEN( sn.body"address, fmd, iprt );
        if tfl >= 0 then
        begin
          ii := CC__DUP2( lun, tfl );           { Force the lun to be changed as openned }
          CC__CLOSE( tfl )                      { Close the temporary file descriptor }
        end
      end
*)
  end CHECK_DESCR;



begin { PAS$$INIT }
  CC__TRAP_INIT;                        { Set up the pascal trap handler }

  { Setup the input, output and error standard files }
  input_file  := inp_descr"address;
  output_file := out_descr"address;
  errmsg_file := err_descr"address;

  { Define the device TT (as /dev/tty for unix like OS) when it is not already defined }
  if GET_LOGICAL( fspc, log_dev_tty ) <= 0 then
    if system_wind then SET_LOGICAL( log_dev_tty, wnt_dev_tty )
                   else SET_LOGICAL( log_dev_tty, unx_dev_tty );
  { Define the device NL (as /dev/null for unix like OS) when it is not already defined }
  if GET_LOGICAL( fspc, log_dev_null ) <= 0 then
    if system_wind then SET_LOGICAL( log_dev_null, wnt_dev_null )
                   else SET_LOGICAL( log_dev_null, unx_dev_null );
  with inp_descr do
  begin
    fild_cfile   := 0;
    fild_objsize := 1;
    fild_bufsize := def_buf_size;
    fild_state   := [lio_inp,lio_txt,lio_std,lio_emp];  { File status }
    fspc := 'SYS$INPUT';
    CHECK_DESCR( fild_cfile, cc__o_rdonly, fspc )
  end;
  INIT_A_FILE( inp_descr"address, fspc );

  with out_descr do
  begin
    fild_cfile   := 1;
    fild_objsize := 1;
    fild_bufsize := def_buf_size;
    fild_state   := [lio_out,lio_txt,lio_std];          { file status }
    fspc := 'SYS$OUTPUT';
    CHECK_DESCR( fild_cfile, cc__o_wronly, fspc )
   end;
  INIT_A_FILE( out_descr"address, fspc );

  with err_descr do
  begin
    fild_cfile   := 2;
    fild_objsize := 1;
    fild_bufsize := def_buf_size;
    fild_state   := [lio_out,lio_txt,lio_std];          { file status }
    fspc := 'SYS$ERROR';
    CHECK_DESCR( fild_cfile, cc__o_wronly, fspc )
  end;
  INIT_A_FILE( err_descr"address, fspc );

  with str_descr do
  begin
    fild_cfile   := -1;
    fild_objsize :=  1;
    fild_bufsize :=  0;
    fild_state   := [lio_vir,lio_txt,lio_std,lio_est]   { file status }
  end;

  { Set up the process system arguments }
  sys_argc := parc;
  for i := 0 to parc-1 do
  begin
    { Get the parameter length }
    l := 0;
    while (l < 255) and (parv[i]^[l] <> char( 0 ) ) do l := SUCC( l );

    if l > 0 then
    begin
      NEW( sys_argv[i], l );
      with sys_argv[i]^ do
      begin
        length := l;
        for j := 0 to l-1 do body[j+1] := parv[i]^[j]
      end
    end
    else sys_argv[i] := empty_str"address
  end;
  for i := parc to mxpar_process do sys_argv[i] := nil;
  sys_env := penv"address
end PAS$$INIT;



[global 'PAS__END']
procedure PAS$$END;
var
  p1, p2: fild_ptr;

begin
  p1 := fild_firstptr;
  while p1 <> nil do
  begin
    p2 := p1;
    p1 := p1^.fild_nxt;
    FILE_CLOSE( p2, [] )
  end
end PAS$$END;



[global 'PAS__EXIT']
procedure PAS$$EXIT( ncd: integer );
begin
  PAS$$END;
  CC__EXIT( ncd )
end PAS$$EXIT;



[global 'PAS__SYSTEM']
function SPAWN( in_var cmd: string ): boolean;
var
  buf: packed array[1..256] of char;

begin
  buf := cmd;
  buf[cmd.length+1] := char( 0 );
  SPAWN := (CC__SYSTEM( buf"address ) = 0)
end SPAWN;



[global 'PAS__READLINK']
function READ_LINK( in_var fst: string; var trg: string ): integer;
var
  ierr: integer;
  buf: packed array[1..256] of char;

begin
  for ii := 1 to fst.length do buf[ii] := fst[ii];
  buf[fst.length+1] := char( 0 );  
  ierr := CC__READLINK( buf"address, trg.body"address, trg.capacity );
  if ierr >= 0 then trg.length := ierr;
  READ_LINK := ierr
end READ_LINK;



[global 'PAS__FILE_ACCESS']
function FILE_ACCESS( in_var filespc: string;
                                iprv: integer := 0;
                                umsk: flags_file := [] ): boolean;
{ Check access for a file (default is the exist test),
    1 is the exec access test,
    2 is the write/delete access test,
    4 is the read access test.
}
var
  ner:   integer;
  acf: cc__setty;

begin
  { Set the user specified mode }
  if iprv > 0 then iprv := iprv mod 8;  { Ignore any high bit in access word }
  if iprv < 0 then iprv := 0;
  if iprv = 0 then acf := cc__f_ok
  else
  begin
    acf := [];
    if ODD( iprv ) then acf := cc__x_ok; iprv := iprv div 2;
    if ODD( iprv ) then acf := acf + cc__w_ok; iprv := iprv div 2;
    if ODD( iprv ) then acf := acf + cc__r_ok
  end;

  if not SET_FILESPC( curr_fspc, filespc, umsk ) then ner := 121
  else
    if curr_fspc.length = 0 then ner :=  201
    else
      if CC__ACCESS( curr_fspc.body"address, acf ) <> 0 then ner := CC__ERROR
                                                        else ner := 0;
  io_errcode := ner;
  FILE_ACCESS := (ner = 0)
end FILE_ACCESS;



[global 'PAS__FILE_RENAME']
function  FILE_RENAME( in_var  fold,  fnew: string;
                              oldmd, newmd: flags_file := [] ): boolean;
var
  ner:   integer;
  nfspc: string;

begin
  if not (SET_FILESPC( curr_fspc, fold, oldmd )
          and SET_FILESPC( nfspc, fnew, newmd )) then ner := 121
  else
    if (curr_fspc.length = 0) or (nfspc.length = 0) then ner :=  201
    else
      if CC__RENAME( curr_fspc.body"address, nfspc.body"address ) <> 0 then
        ner := CC__ERROR
      else
        ner := 0;
  FILE_RENAME := (ner = 0)
end FILE_RENAME;



[global 'PAS__FILE_REMOVE']
function  FILE_REMOVE( in_var fspc: string; md: flags_file := [] ): boolean;
var
  ner:   integer;

begin
  if not SET_FILESPC( curr_fspc, fspc, md ) then ner := 121
  else
    if curr_fspc.length = 0 then ner :=  201
    else
      if CC__REMOVE( curr_fspc.body"address ) <> 0 then ner := CC__ERROR
                                                   else ner := 0;
  io_errcode := ner;
  FILE_REMOVE := (ner = 0)
end FILE_REMOVE;



[global 'PAS__USER_UID']
function USER_UID: integer;
begin
  USER_UID := CC__GET_UID
end USER_UID;



[global 'PAS__USER_GID']
function USER_GID: integer;
begin
  USER_GID := CC__GET_GID
end USER_GID;



[global 'PAS__PATH_SEARCH']
procedure PATH_SEARCH(in_var paths,             { The PATH string }
                              fspc:     string; { The filename to search }
                              dsep: char :='/'; { The directory separator }
                       var fl_path:     string; { The resulting file path }
                       var  pthidx,             { The resulting path entry index }
                       dirsz:    integer  { The resulting path entry length }
                     );
{ Routine to search the file specified file (fspc), in the path (path arg.).
  On sucess, the complete file specification is returned (else a null string).
  Note 1: A file without read access is equivalent to a none existing file.
  Note 2: The PATH string length is limited to the maximum string capacity (presently 255 characters).
}
var
  ib, ie, n:   integer;
  dir, nam:     string;
  fnd:         boolean;

begin
  ib  :=     1;
  n   :=     1;
  fnd := false;
  while not fnd and (ib <> 0) do                                    { Loop on all PATH entries }
  begin
    ie := INDEX( paths, ';', n ); n := n + 1;
    if ie = 0 then                                                  { When we take the last path entry ... }
    begin  dir := SUBSTR( paths, ib ); ib := 0  end                 { ... we get it ... }
    else
    begin  dir := SUBSTR( paths, ib, ie - ib ); ib := ie + 1  end;  { ... else we get the current entry without the semicolon separator }
    if (dir[dir.length] <> dsep) and
       (dir[dir.length] <> ':' ) then                               { We complete the current entry with a directory separator if it is not present }
    begin  dir.length := dir.length + 1; dir[dir.length] := dsep  end;
    nam := dir||fspc;                                               { Build a possible complete setup file specification }
    fnd := FILE_ACCESS_CHECK( nam, 4 {Read access} )                { Test it as existing with free access to read }
  end;                                                              { End of search loop }

  if fnd then begin  pthidx := n - 1; fl_path := nam  end
         else begin  pthidx :=     0; fl_path :=  ''  end
end PATH_SEARCH;



[global 'PAS__GET_PARM_APPL']
function GET_PARM_APPL(    var tbp: cc__parprocess_tb;
                           var prg: string;
                        in_var cmd: string;
                               imd: integer := 0 ): integer;
const
  buflen = 1024;

var
  buf: [static] array[1..buflen] of char;
  i, j, len, np, sz: integer;
  prm:               string;
  bm:                boolean;
  md:             flags_file;

begin
  bm := true;
  case imd mod 8 of
    1: md :=        [];                 { Standard parameter management: logicals + case sensitive }
    2: md := [case_dis_file];           { Not case sensitive parameter management, minor case only }
    3: md := [nolog_file];              { No Logical management }
    4: md := [nolog_file,case_dis_file];{ Space suppression, case always minor }
  otherwise
    bm := false;
    md := [nolog_file]                  { No Parameter management }
  end;

  sz  := cmd.length;
  i   :=   1;
  len :=   0;
  np  :=   0;
  loop
    { Skip any Space or Control Character }
    while (i < sz) and (cmd[i] <= ' ') do  i := i + 1;
  exit if (i > sz) or (np >= mxpar_process);
    j := i + 1;
    { Locate the end of the parameter/program_name }
    while (j <= sz) and (cmd[j] > ' ') do  j := j + 1;
    { Copy each parameter to the buffer }
    prm := SUBSTR( cmd, i, j - i );
    if (np = 0) and (prg.capacity > 0) and (prg.length = 0) then prg := prm;
    i := j;
    if bm then
      if not SET_FILESPC( prm, prm, md ) then prm.length := 0;
    j := prm.length;
    if len + j + 1 >= buflen then goto ET_STOP;
    tbp[np] := buf[len+1]"address;      { Set the parameter string address }
    for l := 1 to j do
    begin  len := len + 1; buf[len] := prm[l]  end;
    len := len + 1; buf[len] := CHR( 0 );       { Append a null character }
    np := np + 1
  end { loop };

ET_STOP:
  tbp[np] := nil;                       { Set the final NULL pointer value for C }

(*
i := 0;
while tbp[i] <> nil do
begin
  j := 1;
  WRITE( i:2, ' "' );
  while tbp[i]^[j] <> CHR( 0 ) do
  begin  WRITE( tbp[i]^[j] ); j := j + 1  end;
  WRITELN( '"' );
  i := i + 1
end;
*)

  GET_PARM_APPL := np
end GET_PARM_APPL;



[global 'PAS__SET_PARM_APPL']
function SET_PARM_APPL(    var tbp: cc__parprocess_tb;
                           var prg: string;
                        in_var cmp: array[ll:integer] of string_ptr;
                               imd: integer := 0 ): integer;
const
  buflen = 1024;

var
  buf: [static] array[1..buflen] of char;
  i, j, len, np, sz: integer;
  prm:               string;
  bm:                boolean;
  md:             flags_file;

begin
  bm := true;
  case imd mod 8 of
    1: md :=        [];                 { Standard parameter management: logicals + case sensitive }
    2: md := [case_dis_file];           { Not case sensitive parameter management, minor case only }
    3: md := [nolog_file];              { No Logical management }
    4: md := [nolog_file,case_dis_file];{ Space suppression, case always minor }
  otherwise
    bm := false;
    md := [nolog_file]                  { No Parameter change }
  end;

  i   :=   1;
  len :=   1;
  np  :=   0;
  if (ll > 0) and (prg.capacity > 0) and (prg.length = 0) then prg := cmp[1]^;
  while (i <= ll) and (np < mxpar_process) and (len <= buflen) do
  begin
    if bm then
    begin
      if not SET_FILESPC( prm, cmp[i]^, md ) then prm.length := 0
    end else prm := cmp[i]^;
    j := prm.length;
    if len + j + 1 >= buflen then goto ET_STOP;
    tbp[np] := buf[len]"address;        { Set the parameter string address }
    for l := 1 to j do                  { Copy the parameter Characters }
      if len < buflen then
      begin  buf[len] := prm[l]; len := len + 1  end;
    buf[len] := CHR( 0 ); len := len + 1;       { Append a null character }
    i   :=  i + 1;
    np  := np + 1
  end { loop };

ET_STOP:
  tbp[np] := nil;                       { Set the final NULL pointer value for C }

(*
i := 0;
while tbp[i] <> nil do
begin
  j := 1;
  WRITE( i:2, ' "' );
  while tbp[i]^[j] <> CHR( 0 ) do
  begin  WRITE( tbp[i]^[j] ); j := j + 1  end;
  WRITELN( '"' );
  i := i + 1
end;
*)

  SET_PARM_APPL := np - 1
end SET_PARM_APPL;



[global 'PAS__IN_PATH']
function FIND_IN_PATH( in_var prog: string; in_var upath: string := '';
                       psep: char := ' '; acc: integer := 1 ): boolean;
var
  vpth:           cc__char_ptr;         { Pointer to the array of PATH Value }
  prg, str:             string;
  i, j, len:           integer;
  dir_sep:                char;
  bok:                 boolean;

begin
  if upath.length = 0 then len := PAS__GET_ENV( vpth, 'PATH' )
                      else begin  vpth := upath.body"address; len := upath.length  end;
  if psep <= ' ' then
    if not system_wind then psep := ':'
                       else psep := ';';
  if system_wind then dir_sep := '\'
                 else dir_sep := '/';
  if len > 0 then
  begin
    i := 1;
    bok := false;
    repeat
      while (i <= len) and (vpth^[i] = psep) do i := i + 1;
    exit if i > len;
      j := 1; str[1] := vpth^[i];
      i := i + 1;
      while (i <= len) and (vpth^[i] <> psep) do
      begin
        if j < str.capacity then j := j + 1; str[j] := vpth^[i];
        i := i + 1
      end;
      str.length := j;

      if (str[str.length] <> ':') and (str[str.length] <> dir_sep) then prg := str||dir_sep||prog
                                                                   else prg := str||prog;
      bok := FILE_ACCESS( prg, acc { Exec Access } );  { Exec when it is OK for required access }
    until bok or (i > len)              { Continue to Return Sequence when the required access is OK, or End of Path }
  end
  else bok := false;
  FIND_IN_PATH := bok
end FIND_IN_PATH;



[global 'PAS__RUN_PROC_0']
function RUN_PROC_0( in_var  prg: string;
                     in_var  tbp: cc__parprocess_tb;
                             pat: boolean := true;
                             env: $wild_pointer := nil ): boolean;
begin
  if (pat and FIND_IN_PATH( prg )) or FILE_ACCESS( prg, 1 { Exec Access } ) then
  begin { It is OK for task execution }
    PAS$$END; { Close all files }
    if env = nil then CC__EXECV( curr_fspc.body"address, tbp )
                 else CC__EXECVE( curr_fspc.body"address, tbp, env );
    PAS__ERROR( 998 );
    { Si O.K. pas d'execution ici }
    PAS$$END;                           { Flush Output Message }
    CC__EXIT( 1 )                       { Exit with Error status }
  end
  else RUN_PROC_0 := false
end RUN_PROC_0;


[global 'PAS__RUN_PROC_1']
function RUN_PROC_1( in_var prg, cmd: string;
                     imd: integer := 0; env: $wild_pointer := nil ): boolean;
var
  tbp: cc__parprocess_tb;
  pr0: string;

begin
  pr0 := prg;
  GET_PARM_APPL( tbp, pr0, cmd, imd );
  RUN_PROC_1 := RUN_PROC_0( pr0, tbp, (imd div 8) = 0, env )
end RUN_PROC_1;



[global 'PAS__RUN_PROC_2']
function RUN_PROC_2( in_var prg: string; in_var cmp: array[ll:integer] of string_ptr;
                     imd: integer := 0; env: $wild_pointer := nil ): boolean;
var
  tbp: cc__parprocess_tb;
  pr0: string;

begin
  pr0 := prg;
  SET_PARM_APPL( tbp, pr0, cmp, imd );
  RUN_PROC_2 := RUN_PROC_0( pr0, tbp, (imd div 8) = 0, env )
end RUN_PROC_2;



[global 'PAS__CREATE_PROC_0']
function CREATE_PROC_0( in_var  prg: string;
                        in_var  tbp: cc__parprocess_tb;
                                pat: boolean := true;
                                env: $wild_pointer := nil ): integer;
var
  stat, ist: integer;


begin
  if (pat and FIND_IN_PATH( prg )) or FILE_ACCESS( prg, 1 { Exec Access } ) then
  begin { It is OK for task execution }
    if system_cygw or system_wind then
    begin { For WINDOWS-NT with or without GCC compiler }
      if env = nil then
        stat := CC__SPAWNV( wnt_nowait, curr_fspc.body"address, tbp )
      else
        stat := CC__SPAWNVE( wnt_nowait, curr_fspc.body"address, tbp, env );
      if stat = -1 then PAS__ERROR( CC__ERROR )
    end
    else
    begin { For OPEN-VMS and UNIX }
      stat := CC__FORK;
      if stat = 0 then
      begin { we are in the children sequence }
        { Now we can exec the next program }
        if env = nil then ist := CC__EXECV( curr_fspc.body"address, tbp )
                     else ist := CC__EXECVE( curr_fspc.body"address, tbp, env );
        { En Unix, le fichier a executer ne peut etre lancer => PAS__ERROR }
        PAS__ERROR( 998 )
      end
      else
        if stat = -1 then PAS__ERROR( 999 )   { Cannot create the process }
        { stat is the fork result }
    end;
    CREATE_PROC_0 := stat
  end
  else CREATE_PROC_0 := -1
end CREATE_PROC_0;



[global 'PAS__CREATE_PROC_1']
function CREATE_PROC_1( in_var prg, cmd: string;
                        imd: integer := 0; env: $wild_pointer := nil ): integer;
var
  tbp: cc__parprocess_tb;
  pr0: string;
  bpa: boolean;

begin
  pr0 := prg;
  GET_PARM_APPL( tbp, pr0, cmd, imd );
  CREATE_PROC_1 := CREATE_PROC_0( pr0, tbp, (imd div 8) = 0, env )
end CREATE_PROC_1;



[global 'PAS__CREATE_PROC_2']
function CREATE_PROC_2( in_var prg: string; in_var cmp: array[ll:integer] of string_ptr;
                        imd: integer := 0; env: $wild_pointer := nil ): integer;
var
  tbp: cc__parprocess_tb;
  pr0: string;

begin
  pr0 := prg;
  SET_PARM_APPL( tbp, pr0, cmp, imd );
  CREATE_PROC_2 := CREATE_PROC_0( pr0, tbp, (imd div 8) = 0, env )
end CREATE_PROC_2;



[global 'PAS__WAIT_PROCESS']
function WAIT_PROCESS( var pro_sta: integer; proc_id: integer := 0 ): integer;
var
  iv, id: integer;

begin
  if system_wind then iv := CC__CWAIT( pro_sta, id, 0 )     { For WINDOWS-NT }
                 else iv := CC__WAIT( pro_sta );            { All UNIX like Systems }
  WAIT_PROCESS := iv
end WAIT_PROCESS;



[global 'PAS__CHANGE_DIR']
function CHANGE_DIR( in_var dir: string; md: flags_file := [] ): boolean;
var
  ner: integer;

begin
  ner := 0;
  if not SET_FILESPC( curr_fspc, dir, md ) then ner := 121
  else
    if curr_fspc.length = 0 then ner := 201
    else
      if CC__CHDIR( curr_fspc.body"address ) <> 0 then ner := CC__ERROR;
  CHANGE_DIR := (ner <> 0)
end CHANGE_DIR;



[global 'PAS__GET_DEF_DIR']
function GET_DEFDIR: string;
var
  pp: $wild_pointer;
  ii: integer;

begin
  GET_DEFDIR.length := 0;
  pp := CC__GETDIR( function.body[1]"address, function.capacity - 1 );
  if pp <> nil then
  begin
    ii := 1;
    while (ii < function.capacity) and
          (function.body[ii] <> CHR( 0 )) do ii := ii + 1;
    if ii > 0 then GET_DEFDIR.length := ii - 1
  end
end GET_DEFDIR;



[global 'PAS__DELAY']
function DELAY( sec: single ): single;
begin
  if sec > 0.0 then
    if SET_TIMER( sec, 0.0 ) = 0 then DELAY := WAIT_TIMER
                                 else DELAY := -1.0
  else DELAY := 1.0
end DELAY;


[global 'PAS__SLEEP']
function SLEEP( sec: integer ): integer;
begin
   SLEEP := ROUND( DELAY( single( sec ) ) )
end SLEEP;



[global 'PAS__RANGE']
function CHECK_RANGE( iv, low, high: integer ): integer;
begin
  if (iv < low) or (iv > high) then PAS__ERROR( 31 );
  CHECK_RANGE := iv
end CHECK_RANGE;



[global 'PAS__SET_LE']
function SET_LE( s1, s2: cc__setty ): boolean;
begin
  SET_LE := (s1 = (s1*s2))
end SET_LE;



[global 'PAS__Q_SQR']
function Q_SQR( i: long_integer ): long_integer;
begin
  Q_SQR := i*i
end Q_SQR;



[global 'PAS__I_SQR']
function I_SQR( i: integer ): integer;
begin
  I_SQR := i*i
end I_SQR;



[global 'PAS__Q_MOD']
function Q_MOD( iv, jv: long_integer ): long_integer;
var
  t1, t2: long_integer;

begin
  t2 := ABS( jv );
  t1 := iv rem jv;
  if t1 >= 0 then Q_MOD := t1
             else Q_MOD := t1 + t2
end Q_MOD;



[global 'PAS__I_MOD']
function I_MOD( iv, jv: integer ): integer;
var
  t1, t2: integer;

begin
  t2 := ABS( jv );
  t1 := iv rem jv;
  if t1 >= 0 then I_MOD := t1
             else I_MOD := t1 + t2
end I_MOD;



[global 'PAS__F_SQR']
function F_SQR( f: single ): single;
begin
  F_SQR := f*f
end F_SQR;



[global 'PAS__G_SQR']
function G_SQR( g: double ): double;
begin
  G_SQR := g*g
end G_SQR;



[global 'PAS__FQ_ROUND']
function FQ_ROUND( fv: single ): long_integer;
begin
  if fv > 0.0 then FQ_ROUND := TRUNC( fv + 0.5 )
              else FQ_ROUND := TRUNC( fv - 0.5 )
end FQ_ROUND;



[global 'PAS__GQ_ROUND']
function GQ_ROUND( fv: double ): long_integer;
begin
  if fv > 0.0 then GQ_ROUND := TRUNC( fv + 0.5 )
              else GQ_ROUND := TRUNC( fv - 0.5 )
end GQ_ROUND;


[global 'PAS__F_ROUND']
function F_ROUND( fv: single ): integer;
begin
  if fv > 0.0 then F_ROUND := TRUNC( fv + 0.5 )
              else F_ROUND := TRUNC( fv - 0.5 )
end F_ROUND;



[global 'PAS__G_ROUND']
function G_ROUND( fv: double ): integer;
begin
  if fv > 0.0 then G_ROUND := TRUNC( fv + 0.5 )
              else G_ROUND := TRUNC( fv - 0.5 )
end G_ROUND;


end.
