{
***********************************************************************
*                                                                       *
*                                                                       *
*                   *  C P A S  *  S Y S T E M  *                       *
*                                                                       *
*                                                                       *
*           * * *   S t a n d a r d   L i b r a r y   * * *             *
*                                                                       *
*                                                                       *
*                   ---  Numeric I/O Library  ---                       *
*                                                                       *
*              ---  Version  3.1-B6 -- 03/07/2022 ---                   *
*   by :                                                                *
*                                                                       *
*       P. Wolfers                                                      *
*         c.n.r.s.                                                      *
*         Institut Louis Neel                                           *
*         B.P.  166 X   38042  Grenoble Cedex                           *
*                           FRANCE.                                     *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////



*************************************************************************
*                                                                       *
*        Last revision of 28-APR-2005 for CPASCAL Version 1.9 L         *
*                                                                       *
*************************************************************************
}

%pragma trace 1;
module CPAS__RWNUM;

%include  'PASSRC:cpas__fenv_api_env';  { used to manage FP_NON and FP_INF exceptions when ... }
                                        { ... the floating value is not a number (or infinite }

const
  max_dig = 65;           { maximum of digit for a number }
  ln10    = 2.3025851249695E+00; { LN( 10.0 ) }
  ten = DOUBLE( 10 );     { Constant 10.0 in double }
  odt = 1.0/ten;          { Constant 0.1 in double }
  soh = CHR( 0 );         { NULL character SOH }


var
  [external 'PAS__curr_iptr']  curr_input: text; { The current input --- Patch  27/04/2005 }


[external 'PAS__ERROR'] procedure ERROR( nerr: integer ); external;

[external 'PAS__READ_EOLN'] procedure READ_CURRENT_EOLN; external;




(************   Integer/float  input procedures  **************)

[global]
procedure PAS__GET_DOUBLE( var fl: double; fld: integer := 0 );
type
  mdtyp = ( mdspace, mdsign, mdent, mdfrac, mdsexp, mdexp );

var
  val, fac:   double;
  iexp, idig: integer;
  bneg, bnex: boolean;
  md:         mdtyp;
  ch:         char;

begin
  val   := 0.0;
  fac   := odt;
  iexp  := 0;
  bneg  := false;
  bnex  := false;
  md    := mdspace;

  repeat
    if UFB( curr_input ) then GET( curr_input ); { --- Patch  27/04/2005 }
  exit if CURRENT_EOF or ((md > mdspace) and CURRENT_EOLN);    { --- Patch  05/04/2006 }

    if fld >= 0 then fld := fld - 1;
    ch := curr_input^;                           { --- Patch  27/04/2005 }
(*
    READ$OBJECT( ch );
*)
    case ch of
      soh..' ': exit if md > mdspace; { stop on space }

      '+', '-':
        case md of
          mdspace: begin  md := mdsign; if ch = '-' then bneg := true  end;
          mdsexp:  begin  md := mdexp;  if ch = '-' then bnex := true  end;
        otherwise
          exit
        end;

      '0', '1', '2', '3', '4', '5', '6', '7', '8', '9':
         begin
           idig := (ORD( ch ) - ORD( '0' ));
           case md of
             mdspace, mdsign, mdent:
                     begin
                       val := val*ten + idig;
                       if md < mdent then md := mdent
                     end;
             mdfrac: begin  val := val + fac*idig; fac := fac*odt  end;
             mdsexp, mdexp:
                     begin
                       iexp := iexp*10 + idig;
                       if md < mdexp then md := mdexp
                     end;
           otherwise
           end
         end;

      '.': if md < mdfrac then md := mdfrac else exit;

      'e', 'E', 'd', 'D':
         if md < mdsexp then
           begin  if md < mdent then val := 1.0; md := mdsexp;  end
         else exit;

    otherwise
      exit if md > mdspace;
      ERROR( 51 )
    end;
    GET( curr_input )                 { Skip to next character, Patch  27/04/2005 }
  until fld = 0;

  if md = mdspace then ERROR( 51 );   { Error on empty or not numeric string --- Patch 05/04:06 }

  { Modify val by the related power of ten if required }
  if iexp <> 0 then
  begin
    if bnex then iexp := -iexp;
    val := val*ten**iexp
  end;
  if bneg then val := -val;
  fl := val
end PAS__GET_DOUBLE;



[global 'PAS__READ_DBLE']
procedure READ_DOUBLE( var fl: double; f: integer := 0 );
begin
  PAS__GET_DOUBLE( fl, f )
end READ_DOUBLE;


[global 'PAS__READ_SNGL']
procedure READ_SINGLE( var fl: single; f: integer := 0 );
var db: double;
begin
  PAS__GET_DOUBLE( db, f );
  fl := SINGLE( db )
end READ_SINGLE;



[global 'PAS__READ_SL']
procedure READ_INT( var iv: integer; f: integer := 0 );
const
  dmaxint = DOUBLE( maxint );

var
  db: double;

begin
  PAS__GET_DOUBLE( db, f );
  iv := ROUND( db )
end READ_INT;


[global 'PAS__READ_UL']
procedure READ_UL( var iv: unsigned; f: integer := 0 );
var rv: integer;
begin
  READ_INT( rv, f ); iv := rv
end READ_UL;


[global 'PAS__READ_SW']
procedure READ_SW( var iv: short_integer; f: integer := 0 );
var rv: integer;
begin
  READ_INT( rv, f ); iv := rv
end READ_SW;


[global 'PAS__READ_UW']
procedure READ_UW( var iv: short_unsigned; f: integer := 0 );
var rv: integer;
begin
  READ_INT( rv, f ); iv := rv
end READ_UW;


[global 'PAS__READ_SB']
procedure READ_SB( var iv: byte_integer; f: integer := 0 );
var rv: integer;
begin
  READ_INT( rv, f ); iv := rv
end READ_SB;


[global 'PAS__READ_UB']
procedure READ_UB( var iv: byte_unsigned; f: integer := 0 );
var rv: integer;
begin
  READ_INT( rv, f ); iv := rv
end READ_UB;




(************   Output procedures  **************)

[global 'PAS__WRITE_UNS']
procedure WRITE_UNS( iv: unsigned; f, b: integer );
{ iv is the unsigned value, f the field and b the base }
var
  ditb: array[1..65] of char;
  fch: char;
  dig, i, j: integer;

begin
  fch := ' ';
  if iv = 0 then
  begin { For zero integer number }
    ditb[1] := '0'; j := 1
  end
  else
  begin { <> 0 number }
    if (b < 2) or (b > 16) then b := 10;
    j := 0;
    while iv <> 0 do
    begin
      j := j + 1;
      dig := iv rem b;
      if dig > 9 then
        ditb[j] := CHR( dig + ORD( 'A' ) - 10 )
      else
        ditb[j] := CHR( dig + ORD( '0' ) );
      iv := iv div b
    end
  end;
  if f = 0 then  f := j { we set the field in agreement of the number }
  else
  if f < 0 then
    begin { we set the field as positive with "0' at left }
      fch := '0'; f := - f
    end;
  i :=  f - j;    { get the number of digit at left }
  if i >= 0 then     { If the write is possible }
  begin
    for ii := 1 to i do WRITE$OBJECT( fch );
    while j > 0 do
    begin
      WRITE$OBJECT( ditb[j] ); j := j - 1
    end
  end
  else { Too small field }
    for ii := 1 to f do  WRITE$OBJECT( '*' )
end WRITE_UNS;




[global 'PAS__WRITE_INT']
procedure WRITE_INT( iv, f, b: integer );
{ iv is the integer value, f the field and b the base }
var
  ditb: array[1..65] of char;
  fch: char;
  dig, i, j: integer;
  bneg: boolean;

begin
  fch := ' ';
  if iv = 0 then
  begin { For zero integer number }
    ditb[1] := '0'; j := 1; bneg := false
  end
  else
  begin { <> 0 number }
    bneg := iv < 0; { Set number to be positive }
    if (b < 2) or (b > 16) then b := 10;
    j := 0;
    while iv <> 0 do
    begin
      j := j + 1;
      dig := ABS( iv rem b );
      if dig > 9 then
        ditb[j] := CHR( dig + ORD( 'A' ) - 10 )
      else
        ditb[j] := CHR( dig + ORD( '0' ) );
      iv := iv div b
    end
  end;
  if f = 0 then { we set the field in agreement of the number }
    f := j + ORD( bneg )
  else
  if f < 0 then
  begin { we set the field as positive with "0' at left }
    fch := '0'; f := - f
  end;
  i :=  f - j;    { get the number of digit at left }
  if bneg then i := i - 1;
  if i >= 0 then     { If the write is possible }
  begin
    if bneg and (fch = '0') then WRITE$OBJECT( '-' );
    for ii := 1 to i do WRITE$OBJECT( fch );
    if bneg and (fch = ' ') then WRITE$OBJECT( '-' );
    while j > 0 do
    begin
      WRITE$OBJECT( ditb[j] ); j := j - 1
    end
  end
  else { Too small field }
    for ii := 1 to f do  WRITE$OBJECT( '*' )
end WRITE_INT;




procedure PUT_DECIMAL( dv:   double;
                       ndig,
                       pent: integer;
                       bneg: boolean );
{ dv is a positive normalized number in the range [0.1,1.0[. or 0.0 }
{ ndig is the number of digit to write }
{ pent is the number of digit for the integer part }
{ bneg is the negative flag }
var
  dig:  integer;
  fdig: boolean;

begin
  fdig := true;                { Set to first character search flag }
  while ndig > 0 do
  begin { loop on all digits }
    dig := TRUNC( dv * ten );  { Take off the digit value and update the dv value ... }
    dv := dv*ten - dig;        { ... by a new multiply to avoid any rounding of dv by memory }
    if fdig then               { In first character search mode }
      if (dig = 0) and (pent > 1) then
        WRITE$OBJECT( ' ' )          { First character do not fall }
      else
      begin { First character is falling }
        fdig := false;         { Clear the first character search flag }
        if bneg then WRITE$OBJECT( '-' )
      end;

    { output the negative sign when required }
    if not fdig then WRITE$OBJECT( CHR( dig + ORD( '0' ) ) );
    ndig := ndig - 1;
    pent := pent - 1;
    { insert the period when required }
    if (pent = 0) and (ndig > 0) then WRITE$OBJECT( '.' )
  end
end PUT_DECIMAL;


procedure SIZE_DOUBLE( var dv: double;      { The float to normalize }
                       var iexp: integer;   { The resulting exponent }
                       var bneg: boolean;   { The negative flag }
                           bfix: boolean ); { The fiexd mode flag }
{ TO size and normalize a floatting number }
var
  dv1: double;

begin
  if dv < 0.0 then
  begin
    dv   := -dv;
    bneg := true
  end
  else bneg := false;
  if dv > 0.0 then
  begin
    iexp := 0; dv1 := dv;
    if dv > 0.0 then
      if dv >= 1.0 then
        while dv1 > double( 1.0 ) do  begin  dv1 := dv1/10.0; iexp := iexp + 1  end
      else
        while dv1 < double( 0.1 ) do  begin  dv1 := dv1*10.0; iexp := iexp - 1  end;
      
    {iexp := ROUND( LN( dv ) / ln10 + 0.5 );
    { Set the value in range [0.1..1.0[ or [0.0..1.0[ for fixed }
    {   if iexp >= 0 then dv1 := dv*ten**(-iexp)
                 else dv1 := dv/(ten**iexp);}
    { dv1 := dv/(ten**iexp); /// Not good, can implied some overflow as with for double"large }
    if dv1 >= 1.0 then
    begin
      dv1 := dv1*odt; iexp := iexp + 1
    end
    else
      if dv1 < odt then
      begin
        dv1 := dv1*ten; iexp := iexp - 1
      end;
    if bfix and (iexp <= 0) then dv := dv*odt
                            else dv := dv1
  end
  else iexp := 0
end SIZE_DOUBLE;


[global]
procedure PAS__PUT_E_DBLE( dv: double;       { The value to output }
                           fs,               { The filed size }
                           intsz,            { The wished integer part size }
                           dcsz,             { The wished decimal part size }
                           es:    integer ); { The wished exponent field size }
var
  iexp, ses, ef, i, ndg:  integer;
  bneg, eneg, bdec, bnsp: boolean;
  chexp, chsgn:           char;

begin
  es := ABS( es );
  { Check for correct parameter values }
  if dcsz = -1  then dcsz := 7 else dcsz := ABS( dcsz );
  if dcsz > 20  then dcsz := 20;
  if fs < 0 then begin  bnsp := true; fs := ABS( fs )  end
            else bnsp := false;
  if fs = 0 then fs := 22;
  if intsz < 1 then intsz := 1;
  { size the number to write and set it as positive number }
  SIZE_DOUBLE( dv, iexp, bneg, false );
  { iexp is the power of 10 to write on the form 0.d...E... }
  { dv is now in the range [0.1 1.0] } 
  { modify the exponant by the size of integer part }
  iexp := iexp - intsz;
  { select the sign character for the exponant }
  if iexp >= 0 then chexp := '+'
  else
  begin
    iexp := - iexp; chexp := '-'
  end;
  { determine the necessary exponant field size }
  if iexp < 10 then ses := 2 else
    if iexp < 100 then ses := 3 else ses := 4;
  { and set it when the user specifier is too small }
  if ses > es then es := ses;
  { compute the size of unused space where 2 for "E+" or "E-" }
  bdec := (dcsz > 0);
  ef := fs - dcsz - es - intsz - ORD( bneg ) - 2 - ORD( bdec );
  if ef < 0 then                   { not enouph room in the field }
  begin
    { Try to supress exponent sign }
    if (ef < 0) and (chexp = '+') then
    begin
      ef := ef + 1; chexp := ' '
    end;
    { If not enough, try to supress some exponent figures }
    while (es > ses) and (ef < 0) do
    begin
      es := es - 1; ef := ef + 1
    end;
    { if is not enough, try to supress some decimal figures }
    if (ef < 0) and (dcsz > 0) then { we can try to suppress some decimal digits }
    begin
      i := ef + dcsz;              { get the number of figure to suppress }
      if i > 0 then                { if some decimal are keep ... }
      begin
        dcsz := i; ef := 0         { set the new number of decimal }
      end
      else                         { when we have not enough space, but ... }
        if (i = -1) and bdec then  { ... the lack of just one character }
        begin
          dcsz := 0; ef := 0;      { we have just the good space without "." }
          bdec := false
        end
    end
  end;

  ndg := dcsz + intsz;
  dv := dv + 0.5*ten**(-ndg);      { We round up the number }
  if dv >= 1.0 then
  begin { We must change the floatting format parameters }
    dv := dv*odt;                  { The number must be always normalized }
    if chexp = '-' then            { and the exponant must be adapted }
    begin
      iexp := iexp - 1;            { < 0 then exponent magnitude is decreasing }
      if iexp = 0 then chexp := '+'
    end
    else
    begin
      iexp := iexp + 1;            { the exponent magnitude is increasing }
      if iexp >= 10 then           { The exp. field can be not enough large }
      begin { Adjust the exponant field in the positive case }
        if chexp = '+' then
        begin
          chexp := ' '; ef := ef + 1
        end;
        if (iexp = 10) and (es = 1) then
        begin  es := 2; ef := ef - 1  end
        else
          if (iexp = 100) and (es = 2) then
          begin  es := 3; ef := ef - 1  end
      end;

      if ef < 0 then
      begin { We must find room for one character }
        if ndg > 1 then
        begin
          ndg := ndg - 1; { Suppress one figure }
          if intsz <= ndg then
          begin { If no decimal are ouput we can suppress one decimal }
            if ndg = intsz then
            begin
              bdec := false; ef := ef + 1
            end
          end
          else
          begin { We must increment the exponent to win one figure }
            intsz := intsz - 1;
            iexp := iexp + 1
          end;
          ef := ef + 1
        end
      end
    end
  end;

  if ef < 0 then { it is impossible to write the number }
    for ii := 1 to fs do  WRITE$OBJECT( '*' )
  else
  begin { Output the number }
    if bnsp then ef := 0;
    if ef > 0 then
      for ii := 1 to ef do  WRITE$OBJECT( ' ' );   { Output the left space }
    { Output the number (integer and fractional part) }
    PUT_DECIMAL( dv, ndg, intsz, bneg );
    { Output the exponant character }
    WRITE$OBJECT( 'E' );
    { Output exponent sign when required }
    if chexp <> ' ' then WRITE$OBJECT( chexp );
    { Output the exponent with "0" left character }
    WRITE_INT( iexp, -es, 10 )
  end
end PAS__PUT_E_DBLE;


[global]
procedure PAS__PUT_F_DBLE( dv: double;       { the float number }
                           fs,               { the field size }
                           dcsz,             { the decimal part size }
                           dcmin: integer ); { minimum of figures }
var
  dv1:                         double;
  dc, ef, i, indig, iexp, ndg: integer;
  bneg, bdec:                  boolean;

begin
  { Check for correct parameter values }
  if dcsz = -1 then dcsz := 7
               else dcsz := ABS( dcsz );
  if dcsz > 20 then dcsz := 20;
  if fs < 0 then fs := ABS( fs );
  if fs = 0 then fs := 20;
  if dcmin < 0 then dcmin := ABS( dcmin );
             {/// else if dcmin < 1 then dcmin := 1; ///}
  { size the number to write and set it as positive number }
  dv1 := dv; { Keep a copy of the number to write }
  SIZE_DOUBLE( dv1, iexp, bneg, true );
  { select the number sign }
  { determine the number of digits for the integer part }
  if iexp > 1 then indig := iexp
              else indig := 1;
  bdec := (dcsz > 0);     { get if period is required }

  { get the unused space in the allocated character field as required }
  ef := fs - ORD( bneg ) - dcsz - indig - ORD( bdec );

  i  := dcsz + iexp;      { get the number of significative figures }
  dc := dcsz;
  if (iexp <= 0) and (i < dcmin) and (dcmin > 0) then
  begin { for the too small number with too many lost significative digits }
    dc := -iexp + dcmin;             { adjust the decimal part size }
    ef := fs - dc - 2 - ORD( bneg ); { left characters  with "0." }
    if not bdec and (dc > 0) then    { Add the "." when required }
    begin  ef := ef - 1; bdec := true  end
  end
  else
  begin { for not too small number, it can be too large }
    if ef < 0 then             { When we have not enough room in the field }
    begin
      i := ef + dc;            { we try to suppress some decimal digits }
      if i >= 0 then           { when it is possible to keep some fraction part }
      begin
        dc := i;               { ... we set the new number of decimal }
        if dc > 0 then ef := 0 { ... and supress the period when ... }
        else                   { the fractional part is suppressed }
        begin  bdec := false; ef := 1  end
      end
      else                     { else ... }
        if (i = -1) and bdec then { when we have a lack of just one character }
        begin
          dc := 0; ef := 0 ;   { ... the fraction part and "." are suppressed }
          bdec := false
        end
    end
  end;

  if ef >= 0 then
  begin
    ndg := indig + dc;
    dv1 := dv1 + 0.5*ten**(-ndg);
    if dv1 >= 1.0 then
    begin { We must adjust the dv value, and parameter }
      dv1 := dv1*odt; { Now dv is reset in the range [0.0..1.0[ }
      indig := indig + 1;
      ef := ef - 1; ndg := ndg + 1
    end
  end;

  { If we have always a lack of room in the field we try the E format }
  if ef < 0 then
    PAS__PUT_E_DBLE( dv, fs, 1, dcsz, 0 )
  else
  begin { ok for fixed }
    { Output the space of the unused field part }
    for ii := 1 to ef do WRITE$OBJECT( ' ' );
    { and now write it }
    PUT_DECIMAL( dv1, ndg, indig, bneg )
  end
end PAS__PUT_F_DBLE;



procedure WRITE_EXCEPT_NUMBER( cl: FloatClass; f: integer );
begin
  case cl of
    FP_Nan:   WRITE$OBJECT( 'Nan', f, 0 );
    FP_PInf:  WRITE$OBJECT( '+Inf', f, 0 );
    FP_NInf:  WRITE$OBJECT( '-Inf', f, 0 );
  otherwise
    WRITE$OBJECT( ' ??? ', f, 0 );
  end
end WRITE_EXCEPT_NUMBER;



[global 'PAS__WRITE_DBLE']
procedure WRITE_DBLE( fl: double; f, d, e: integer );
var
  cl: FloatClass;

begin
  PAS__FLOAT_TRAP;
  if f = 0 then f := -22;
  cl := PAS__FP_CLASSIFY( fl );
  if cl in [FP_PInf,FP_Ninf,FP_Nan] then
  begin  WRITE_EXCEPT_NUMBER( cl, f ); return  end;
  if (d < 0) or (f < 0)
  then PAS__PUT_E_DBLE( fl, f, 1, d, 3 )
  else PAS__PUT_F_DBLE( fl, f, d, e )
end WRITE_DBLE;


[global 'PAS__WRITE_SNGL']
procedure WRITE_SNGL( fl: single; f, d, e: integer );
var
  cl: FloatClass;

begin
  PAS__FLOAT_TRAP;
  if f = 0 then f := -16;
  cl := PAS__FP_CLASSIFY( fl );
  if cl in [FP_PInf,FP_Ninf,FP_Nan] then
  begin  WRITE_EXCEPT_NUMBER( cl, f ); return  end;
  if (d < 0) or (f < 0)
  then PAS__PUT_E_DBLE( DOUBLE( fl ), f, 1, d, 2 )
  else PAS__PUT_F_DBLE( DOUBLE( fl ), f, d, e )
end WRITE_SNGL;


end.
