/*
*************************************************************************
*                                                                       *
*                                                                       *
*                   *  C P A S  *  S Y S T E M  *                       *
*                                                                       *
*                                                                       *
*          * * *   S t a n d a r d   L i b r a r y   * * *              *
*                                                                       *
*                                                                       *
*         ---  RUN-TIME KERNEL Service Module (C-Part)  ---             *
*                                                                       *
*              ---  Version  2.2--A -- 30/06/2010 ---                   *
*                                                                       *
*         by :                                                          *
*                                                                       *
*             P. Wolfers                                                *
*                 c.n.r.s.                                              *
*                 Institut Neel                                         *
*                 B.P.  166 X   38042  Grenoble Cedex                   *
*                                             FRANCE.                   *
*                                                                       *
*************************************************************************


/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

*/

/* * * * * * * * * * * *     Version  2.2--A   * * * * * * * * * * * * */
/* * * * * * * * * * * * *     3O-Jun_2010   * * * * * * * * * * * * * */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/time.h>
#include <time.h>
#include <errno.h>

#ifdef VMS
#  include <stat.h>
#else
#  include <sys/stat.h>
#  include <unistd.h>
#endif





#ifndef CLOCKS_PER_SEC
#  ifndef CLK_TCK /* For HPUX */
#    define CLK_nticks 1000000 /* default */
#  else
#    if  defined(__alpha) && defined(__unix__)
#      define CLK_nticks 1000000 /* default */
#    else
#      define CLK_nticks CLK_TCK
#    endif
#  endif
#else
#  define CLK_nticks CLOCKS_PER_SEC
#endif



static int err_tab[] = {
       /* (VMS spc., error code,)..... */

#      ifdef EVMSERR
         EVMSERR, 200,         /* Not OWNER */
#      endif

#      ifdef ENOENT
         ENOENT,  202,         /* No such file or directory */
#      endif

#      ifdef EIO
         EIO,     203,         /* IO error */
#      endif

#      ifdef ENXIO
         ENXIO,   204,         /* No such device or address */
#      endif

#      ifdef EBADF
         EBADF,   205,         /* Bad file number */
#      endif

#      ifdef EACCES
         EACCES,  206,         /* Permission denied */
#      endif

#      ifdef ENOTBLK
         ENOTBLK, 207,         /* Block device required */
#      endif

#      ifdef EBUSY
         EBUSY,   208,         /* Mount devices busy */
#      endif

#      ifdef EEXIST
         EEXIST,  209,         /* File already exist */
#      endif

#      ifdef EXDEV
         EXDEV,   210,         /* Cross-device link */
#      endif

#      ifdef ENODEV
         ENODEV,  211,         /* No such device */
#      endif

#      ifdef ENOTDIR
         ENOTDIR, 212,         /* Not a directory */
#      endif

#      ifdef EISDIR
         EISDIR,  213,         /* Bad use of a directory */
#      endif

#      ifdef ENFILE
         ENFILE,  214,         /* File table overflow */
#      endif

#      ifdef EMFIL
         EMFIL,   215,         /* Too many open files */
#      endif

#      ifdef ENOTTY
         ENOTTY,  216,         /* Not a typewriter */
#      endif

#      ifdef ETXTBSY
         ETXTBSY, 217,         /* Text file busy */
#      endif

#      ifdef EFBIG
         EFBIG,   218,         /* File too big */
#      endif

#      ifdef ENOSPC
         ENOSPC,  219,         /* No space left on device */
#      endif

#      ifdef ESPIPE
         ESPIPE,  220,         /* Illegal seek */
#      endif

#      ifdef EROFS
         EROFS,   221,         /* Read only file system */
#      endif

#      ifdef EMLINK
         EMLINK,  222,         /* Too many links */
#      endif

#      ifdef EPIPE
         EPIPE,   223,         /* Broken pipe */
#      endif

#      ifdef EWOULDBLOCK
         EWOULDBLOCK, 224,     /* File I/O buffer are empty */
#      endif

#      ifdef ESRCH
         ESRCH,   301,         /* No such process */
#      endif

#      ifdef EINTR
         EINTR,   401,         /* Interrupt system call */
#      endif

         0,        10          /* UNKNOWN ERROR */
  };




int PAS__curtim[9];            /* current date */



typedef struct errhdl_rec *ptr_errhdl;

typedef int (*errhdl)( int );

struct errhdl_rec { ptr_errhdl prv;
                    errhdl     fnc;
                  };

static int ncnt = 0;           /* Error loop counter */

static ptr_errhdl errhdl_heap = NULL;



void  PAS__END();              /* Pascal Exit routine */

void  PAS__GEN_ERROR_MSG( int nerr ); /* Edit ERROR Messages procedure */



void PAS__ESTABLISH( errhdl f )
{
  ptr_errhdl p;

  p = (ptr_errhdl) malloc( sizeof( struct errhdl_rec ) );
  p->prv  = errhdl_heap;
  p->fnc  = f;
  errhdl_heap = p;
}



void PAS__REVERT()
{
  ptr_errhdl p;
  p = errhdl_heap;
  errhdl_heap = p->prv;
  free( p );
}



void PAS__ERROR( int nerr )
{ ptr_errhdl p;
  int        crd;
  /* Negative error code => Fatal, the program do not must continue */

  nerr = abs( nerr );

  if (ncnt>=10) exit(4);

  ncnt++;

  /* If some user error handlers are set => call it */
  p = errhdl_heap;
  crd = 0;
  while ((p != NULL) && (crd == 0))
  { crd = p->fnc( nerr );
    p = p->prv;
  }
  if (crd <= 0)
  { /* Standard action : message and exit */
    PAS__GEN_ERROR_MSG( abs( nerr ) );
    PAS__END();
    exit(2);
  }
  ncnt--;
}



int CC_ERROR()
{
  int i;

  i = 0;
  while ((err_tab[i++] != errno) && (err_tab[i-1])) i++;
  return err_tab[i];
}



double PAS__CLOCK()
{
  return 1000.0*((double)clock()/CLK_nticks);
}



void PAS__TIME_DECODE( time_t t, int *tb, char gmt )
{
  struct tm * bt;

  if (gmt) bt = gmtime ( &t ); /* Conv. time to broken-down UTC time */
   else bt = localtime ( &t ); /* Conv. time to broken-down local time */
  tb[0] = bt->tm_sec;          /* Seconds after the minute   (0-60) */
  tb[1] = bt->tm_min;          /* Minutes after the hour     (0-60) */
  tb[2] = bt->tm_hour;         /* Hours since midnight       (0-23) */
  tb[3] = bt->tm_mday;         /* Day of month               (1-31) */
  tb[4] = bt->tm_mon +     1;  /* Month of year              (1-12) */
  tb[5] = bt->tm_year + 1900;  /* Years since 1900                  */
  tb[6] = bt->tm_wday +    1;  /* Days since last Sunday      (1-7) */
  tb[7] = bt->tm_yday +    1;  /* Days since last January 1 (1-366) */
}


void PAS__N_TIME( int *tmtb, int ss, char gmt )
{
  time_t  t;
  struct tm * bt;

  if (ss >= 8)
  { t = time( (time_t) 0 );    /* Get the Universal Time (GMT) */
    PAS__TIME_DECODE( t, tmtb, gmt );  /* Set the Time in PASCAL Forme */
  }
}

static int GET_INFO( struct stat *buf, int *tbi, int ir, int ss, char lnk, char gmt )
{
  if (lnk&2) /* Device Channel Info */
    if (ss >= 10) {
      tbi[0] = (int) buf->st_dev;
      tbi[1] = (int) buf->st_ino;
      tbi[2] = (int) buf->st_mode;
      tbi[3] = (int) buf->st_nlink;
      tbi[4] = (int) buf->st_uid;
      tbi[5] = (int) buf->st_gid;
      tbi[6] = (int) buf->st_rdev;
      tbi[7] = (int) buf->st_size;
      tbi[8] = (int) buf->st_blksize;
      tbi[9] = (int) buf->st_blocks;
      return 0;
    } else return -2;
  else     /* File Info */
    if (ss >= 4) {
      tbi[0] = (int) buf->st_mode;
      tbi[1] = (int) buf->st_size;
      tbi[2] = (int) buf->st_uid;
      tbi[3] = (int) buf->st_gid;
      if (ss >= 28) {
        PAS__TIME_DECODE( buf->st_ctime, tbi+ 4, gmt );
        PAS__TIME_DECODE( buf->st_mtime, tbi+12, gmt );
        PAS__TIME_DECODE( buf->st_atime, tbi+20, gmt );
      }
      return 0;
    } else return -2;
}



int CC_Get_File_Info( char * fs, int *tbi, int ss, char lnk, char gmt )
{
  struct stat buf;
  int ir;

  ir = (lnk&1)?lstat( fs, &buf ):stat( fs, &buf );
  if (!ir) return GET_INFO( &buf, tbi, ir, ss, lnk, gmt );
      else return -1;          /* Bad lstat return */
}



int CC_Get_Descr_Info( int descr, int *tbi, int ss, char lnk, char gmt )
{
  struct stat buf;
  int ir;

  ir = fstat( descr, &buf );
  if (lnk&1) lnk = 3;
  if (!ir) return GET_INFO( &buf, tbi, ir, ss, lnk, gmt );
      else return -1;          /* Bad lstat return */
}



static struct itimerval new_c, old_c;

static int alarm_action( int icd )
{
  int ie;

  ie = getitimer( ITIMER_REAL, &old_c );
  if (!ie) {
    if (new_c.it_interval.tv_sec||new_c.it_interval.tv_usec)
      signal( SIGALRM, ( void ( * ) ( int ) ) alarm_action );
  }
}


int PAS__SET_TIMER( float t1, float t2 )
{
  int     ie;
  float tsec;

  signal( SIGALRM, ( void ( * ) ( int ) ) alarm_action );

  new_c.it_interval.tv_sec  = (int)(tsec = (int)t2);
  new_c.it_interval.tv_usec = (int)((t2 - tsec)*1000000);
  new_c.it_value.tv_sec     = (int)(tsec = (int)t1);
  new_c.it_value.tv_usec    = (int)((t1 - tsec)*1000000);

  ie = setitimer( ITIMER_REAL, &new_c, &old_c );
  return ie;
}


float PAS__WAIT_TIMER()
{
  int     ie;
  float tsec;

  ie = pause();
  ie = getitimer( ITIMER_REAL, &old_c );
  if (!ie)
    tsec = old_c.it_value.tv_sec + 1.0e-6*old_c.it_value.tv_usec;
  else
    tsec = -1.0;
  return tsec;
}


void PAS__ins_proc( void * *pp, void * pr )
{ *pp = pr; }

void PAS__exe_proc( void * pr( void * p ), void * pa )
{ (* pr)(pa); }


