{
*************************************************************************
*                                                                       *
*                                                                       *
*                   *  C P A S  *  S Y S T E M  *                       *
*                                                                       *
*                                                                       *
*          * * *   S t a n d a r d   L i b r a r y   * * *              *
*                                                                       *
*                                                                       *
*              ---  Source Read  Manager Library  ---                   *
*              ---  Version  2.2--A -- 30/06/2010 ---                   *
*                                                                       *
*         by :                                                          *
*                                                                       *
*             P. Wolfers                                                *
*                 c.n.r.s.                                              *
*                 Laboratoire de Cristallographie                       *
*                 B.P.  166 X   38042  Grenoble Cedex                   *
*                                             FRANCE.                   *
*                                                                       *
*************************************************************************


/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}

(*
[inherit ('LIB:BASIC_ENV_STR',                  { String manager def. }
          'LIB:BASIC_ENV_TXF',                  { Text file manager def. }
          'LIB:BASIC_ENV_LST',                  { List file manager def. }
          'LIB:BASIC_ENV_SRC')]                 { SRC file manager def. }
*)
module BASIC_SRC( input,output );               { Use input and output }




{ ****************************  Begin of PASCAL System Code Section  **************************** }

%pragma code_option (c_interface,               { To authorize the use of "standard" keyword }
    c_code '#define _FILE_DUPLICATE(fdst,fsrc) fdst = fsrc',
    c_code '#define _FILE_CLEAR(fdst) fdst = NULL'
  );


  procedure DUPLICATE_FILE( f1, f2: $wild_file ); standard '_FILE_DUPLICATE';
  procedure CLEAR_FILE( f: $wild_file ); standard '_FILE_CLEAR';

%pragma code_option noc_interface;              { To disable usage of "standard" keyword }

{ *****************************  End of PASCAL System Code Section  ***************************** }



{ Basic environment for source input operation }


%include 'pasenv:cpas_b__lst_env';
%include 'pasenv:cpas_b__err_env';



const
  src_maxerrline = 8;                           { Maximum repported error by line }

type
  { Source state mode flag definition }
  src_flagty = (src_blist,                      { List on if present }
                src_bmacroex,                   { Set for macro expantion listing memory }
                src_bphys,                      { Set physical output if present }
                src_bterminal,                  { Set if prompt is wanted }
                src_stdin,                      { Flag for the console mode (the Pascal file is input) }
                src_becho,                      { Set echo mode if present }
                src_echerr,                     { Set error echo on control terminal }
                src_delete,                     { Set if delete on close in required }
                src_toolong,                    { Set if present line is too long }
                src_macro,                      { Set if macro file }
                src_linemode,                   { Set if data mode is wanted }
                src_listbyline,                 { Set if line output on eoln reached }
                src_commentflg,                 { Set if only spaces or comments found in line }
                src_eolcomment,                 { Set for eoln => end of comment }
                src_eoln,                       { Set if end of line seen in data mode }
                src_eofrc,                      { Set if end of file reached (at end of last line) }
                src_eof,                        { Eof seen if set (but the last line can be no finished) }
                src_userflg1,                   { User flag 1, 2, 3 and 4 }
                src_userflg2,
                src_userflg3,
                src_userflg4
               );

  src_comty = ( src_pascomment,                 { Pascal comment handling }
                src_plicomment,                 { PL/1 and C comment handling }
                src_adacomment,                 { Ada comment handling }
                src_lispcomment,                { Lisp comment handling }
                src_nocomment);                 { Comment support disable }

  src_flagw     =    set of src_flagty;         { Flag word for source }

  src_ptr       =         ^src_context;         { Source context pointer definition }

  src_context = record                          { Source context definition }
    src_next,                                   { Link to next source }
    src_previous:              src_ptr;         { Previous source in command sequence }
    src_wchpt,                                  { Word pointer in src_cmdline }
    src_chidx,                                  { Character pointer in src_cmdline }
    src_insnb,                                  { Source instruction count }
    src_level,                                  { Source level for listing }
    src_lstmxlev,                               { Source listing level }
    src_errnb,                                  { Number of error in the current line }
    src_linenbr,                                { Line number in this file }
    src_frspos,                                 { First valid column to use }
    src_lstpos:                integer;         { Last valid column to use }
    src_lastchar,                               { Last valid character }
    src_fchcd,
    src_schcd:                    char;         { First and second character code }
    src_coltb: array[1..src_maxerrline]
                        of error_entry;         { Column table error entry }
    src_flags:               src_flagw;         { Source status flag set }
    src_commentty:           src_comty;         { Mode for comment handling }
    src_cmdline,                                { Commande line }
    src_prompt:                str_ptr;         { Prompt to use if terminal file }
    src_promptf,                                { Prompt file if not output }
    src_file:                     text          { Text file to use, if not input }
  end;


var
  def_prompt:  [global] str_ptr := nil;         { Default prompt }

  src_lstinsnb:       [global] integer;         { Last instruction number }

  src_control,                                  { Current source file pointer }
  src_cont_base,                                { Basis source file pointer }

  src_hde:            [global] src_ptr;         { Used source list header }

  src_listing:        [global] lst_ptr;         { Listing and error source file pointer }



{ Reference to the user procedure to get a new line of virtual file }
procedure VSR_NEXTSTRING;
external;



{********************************************************}
{***********    source managment routines    ************}
{********************************************************}



procedure SRC_R_INIT( p: src_ptr );
begin { SRC_R_INIT }
  with p^ do
  begin
    { *** src_next is never initialized here *** }
    src_previous       :=          nil;         { No previous source until showed otherwise }
    src_chidx          :=            0;         { Character pointer to begin of line }
    src_insnb          :=            0;         { Clear instruction count }
    src_level          :=            0;         { Clear source level }
    src_errnb          :=            0;         { Clear number of error in the current line }
    src_linenbr        :=            0;         { Clear line number in this file }
    src_frspos         :=            1;         { First valid column to use set to first column }
    src_fchcd          :=          ' ';         { Set the character codes to be spaces }
    src_schcd          :=          ' ';
    src_lastchar       :=     CHR( 1 );         { Set last character as an EOLN character }
    src_lstpos         :=          255;         { Last valid column to use at maximum }
    for i := 1 to src_maxerrline do
      src_coltb[i].err_msg := nil;
    { Set message table to empty }
    { Source status flag set }
    { Input used, and error echo }
    src_flags      := [src_bterminal, src_echerr, src_eoln];
    src_commentty  :=   src_pascomment;         { Default to pascal comment mode }
    src_cmdline^.length    :=        0;         { Commande line set to empty }
    { Prompt to use if terminal file }
    STR_COPY_LIM( src_prompt, def_prompt^, 64 );
    CLEAR_FILE( src_file );                     { Initialize the Pascal File Variable } 
    CLEAR_FILE( src_promptf )                   { Initialize the Prompt Pascal File Variable } 
  end
end SRC_R_INIT;



[global]
function SRC_ALLOCATE: src_ptr;
{ Procedure to allocate a source file,
  give a file context address at return time }
var
  ps: src_ptr;

begin { SRC_ALLOCATE }
  NEW( ps );
  with ps^ do
  begin
    src_next := src_hde                         { Link in used list };
    NEW( src_cmdline, 255 ); src_cmdline^.length := 0;
    NEW( src_prompt,  255 ); src_prompt^.length  := 0;
  end;
  SRC_R_INIT( ps );                             { Initialize the new source context }
  src_hde              :=           ps;         { Allocate this file }
  SRC_ALLOCATE         :=           ps          { Give the file context address }
end SRC_ALLOCATE;


[global]
procedure SRC_FREE( p: src_ptr );
{ Procedure to free an unused source input file }
var
  p1, p2: src_ptr;

begin { SRC_FREE }
  p2 := nil; 
  if p <> nil then
  begin
    p1 := src_hde;                              { Init for used file search }
    while (p1 <> p) and (p1 <> nil) do
    begin                                       { Localize the file to de-allocate }
      p2 := p1; p1 := p1^.src_next
    end
  end else p1 := nil;
  if p1 <> nil then                             { Found file }
  begin
    with p^ do
    begin
      DISPOSE( src_cmdline );
      if not (src_macro in src_flags) then
        DISPOSE( src_prompt );                  { Must be handle by user for macro mode }
      { Put out the file context from used list }
      if p2 = nil then src_hde := src_next
                  else p2^.src_next := src_next
    end;
    DISPOSE( p );
    p := nil
  end
end SRC_FREE;



[global]
procedure SRC_CLOSE( var p: src_ptr; bfree: boolean );
{ Close a source file, if bfree then the source variable is free }
var
  disp: flags_file;

begin { SRC_CLOSE }
  if p <> nil then
  with p^ do
  begin                                         { Close text file and delete if required }
    if src_delete in src_flags then disp := [del_file]
                               else disp := [];
    CLOSE( src_file, disp )                     { The free is always };
    if src_bterminal in src_flags then CLOSE( src_promptf )
  end;
  if not bfree then SRC_R_INIT( p )             { Reset input/output default }
               else SRC_FREE( p )
end SRC_CLOSE;



[global]
procedure SRC_OPEN( var       p: src_ptr; in_var fspc: string;
                        bdelete: boolean;    var ierr: integer );
{ Open a source file }
{   p is the pointer of source context, fspc is the file specification }
{   ierr is the return error code }
{ SRC_OPEN allocate the source context if p = nil 
  and close a previously opened file }
{ if bdelete is true the file is set for delete on close time }
var
  ballocated: boolean;

begin { SRC_OPEN }
  ierr := 0;
  if p = nil then
  begin  ballocated := false; p := SRC_ALLOCATE  end
  else                                          { Already allocated }
  begin
    ballocated := true;
    SRC_CLOSE( p, false )                       { Can be opened }
  end;
  with p^ do
  begin
    if fspc.length = 0 then
    begin
      DUPLICATE_FILE( src_file, input );        { We use the standard input descriptor }
      src_flags := src_flags + [src_stdin];
      iostatus := 0                             { Force the Success Open Status }
    end
    else OPEN( src_file, fspc, [read_file,error_file] + lst_spc_mode ); { Open the file }

    if iostatus = 0 then                        { Success }
    begin
      { Set terminal mode if open on terminal }
      src_flags := src_flags - [src_bterminal];
      if TTY_FILE(  src_file ) then
      begin
        if fspc.length = 0 then
        begin
          DUPLICATE_FILE( src_promptf, output );{ We use the standard input descriptor }
          iostatus := 0                         { Force the Success Open Status }
        end
        else OPEN( src_promptf, fspc, [write_file,error_file] );
        if iostatus <> 0 then
        begin  ierr := iostatus; CLOSE( src_file )  end
        else
          src_flags := src_flags + [src_bterminal]
      end;
      if not (src_bterminal in src_flags) then
        if bdelete then src_flags := src_flags + [src_delete]
    end
    else ierr := iostatus;
    if ierr <> 0 then
      if ballocated then SRC_R_INIT( p )
                    else SRC_FREE( p )
  end
end SRC_OPEN;



procedure SRC_OUT_TEXT_LINE( var f: text; var st: string; bterm: boolean );
{ To output a line of text to the Listing file.
}
var
  i, j, lim: integer;

begin { SRC_OUT_TEXT_LINE }
  WRITE( f, ' ':20 ); j := 20;
  with st, lst_current^ do
  begin
    if bterm then lim := 80
             else lim := lst_lnsize;
    for i := 1 to length do
    begin
      j := j + 1;
      if j > lim then
      begin
        WRITELN( f );
        if not bterm then lst_lncnt := SUCC( lst_lncnt );
        j := 20; WRITE( f, ' ':20 )
      end;
      WRITE( f, body[i] )
    end;
    WRITELN( f )
  end
end SRC_OUT_TEXT_LINE;


procedure SRC_OUTLINE( var f: text; bterm: boolean );
{ Basic procedure to output a listing.
}
var
  i, j, maxline: integer;

begin { SRC_OUTLINE }
  with src_control^ do
  begin
    if src_lstinsnb <> src_insnb then
    begin
      src_lstinsnb := src_insnb;
      WRITE( f, ' ', src_insnb:5 )
    end
    else WRITE( f, ' ':6 );                     { Output the statement number if changed }
    { Show level of source }
    WRITE( f, ' ', src_fchcd, src_schcd, ' ', src_level:3, ' ' );
    with src_cmdline^, lst_current^ do
    begin
      if bterm then maxline := 80
               else maxline := lst_lnsize;
      WRITE( f, src_linenbr:4, ' ':2 );
      if length > (maxline - 20) then
      begin
        j := 20;
        for i := 1 to length do
        begin
          j := j + 1;
          if j > maxline then
          begin
            WRITELN( f );
            if not bterm then lst_lncnt := SUCC( lst_lncnt );
            j := 20; WRITE( f, ' ':20 )
          end;
          WRITE( f, body[i] )
        end;
        WRITELN( f )
      end
      else  WRITELN( f, src_cmdline )
    end
  end
end SRC_OUTLINE;



{******************************************************}
{*********    Error managment procedures  *************}
{******************************************************}


procedure SRC_OUTERR_BANNER( var f: text; bterm: boolean );
{ Output error locator banner line on specified text file.
}
var
  sbanner: string( 255 );

begin { SRC_OUTERR_BANNER }
  with src_control^, lst_current^, sbanner do
  begin
    length := capacity;
    for i := 1 to length do body[i] := '_';     { Marked as visible line }
    { Mark in shadding the unusable column }
    { S_USE must be lower than lst_lnsize or 80 (if bterm is true ) }
    for i := 1 to src_frspos - 1 do body[i] := '\';
    for i := 1 to src_errnb do                  { For each error entry }
    with src_coltb[i] do
      body[err_pos] := CHR( ORD( '0' ) + i )    { Set error index };
    if src_cmdline^.length > src_lstpos then
    begin
      { Mark in shadding the unusable column }
      for i := src_lstpos+1 to src_cmdline^.length do body[i] :='\';
      length := src_cmdline^.length
    end
      else length := src_coltb[src_errnb].err_pos;
    { Now output banner line }
    SRC_OUT_TEXT_LINE( f, sbanner, bterm );
    if bterm then
    begin
      LST_NEWLINE;
      lst_lncnt := SUCC( lst_lncnt )
    end;
    WRITELN( f, ' %*+*+* error(s) was detected at line # ', src_linenbr:5 );
    WRITELN( f, ' in the file : "', FILE_SPECIFICATION( src_file ), '"' )
  end
end SRC_OUTERR_BANNER;



procedure SRC_OUT_ERROR( bmac: boolean := false );
begin { SRC_OUT_ERROR }
  with src_control^, lst_current^ do
  begin
    if (src_blist in src_flags) and (src_lstmxlev >= src_level) and
       (not (lstf_virtual in lst_flagsw)) then
    begin
      lst_lncnt := SUCC( lst_lncnt );           { Skip to next line without page skip }
      { Output banner locator line }
      SRC_OUTERR_BANNER( lst_file, false );
      for i := 1 to src_errnb do
      begin
        LST_NEWLINE;
        lst_lncnt := SUCC( lst_lncnt );         { To allocate two lines }
        ERR_DISPLAY( lst_file, i, src_coltb[i] )
      end;
      LST_NEWLINE; WRITELN( lst_file )
    end;
    if (src_echerr in src_flags) or (lstf_virtual in lst_flagsw) then
    begin
      if not bmac then SRC_OUTLINE( output, true );
      SRC_OUTERR_BANNER( output, true );        { Output banner locator line }
      for i := 1 to src_errnb do
        ERR_DISPLAY( output, i, src_coltb[i] );
      WRITELN
    end;
    { Free all error messages }
    for i := 1 to src_errnb do
      DISPOSE( src_coltb[i].err_msg )
  end;
end SRC_OUT_ERROR;



[global]
procedure SRC_OUT_MAC_ERROR( procedure OUT_MAC_LINE( var f: text; btt: boolean ) );
begin { SRC_OUT_MAC_ERROR }
  with src_control^, lst_current^ do
    if src_errnb > 0 then
    begin
      if (src_echerr in src_flags) or (lstf_virtual in lst_flagsw) then OUT_MAC_LINE( output, true );
      SRC_OUT_ERROR( true );
      src_errnb := 0
    end
end SRC_OUT_MAC_ERROR;



[global]
procedure SRC_ERROR( modulesy:   error_mdnam;
                     number:     integer;
                     severity:   error_sev );
{ To set an error in src processing.
  Must be called for all source errors }
var
  bok: boolean;

begin { SRC_ERROR }
  with src_control^ do
  begin
    { Set position of error in the src line limits range }
    if src_wchpt < src_frspos then src_wchpt := src_frspos
    else if src_wchpt > src_lstpos then src_wchpt := src_lstpos;

    if src_errnb >= src_maxerrline then         { Max error by line reached  ? }
    begin
      src_errnb := src_maxerrline; bok := false { Ignore this error }
    end
    else
    begin
      bok := true;
      for i := 1 to src_errnb do
        if src_wchpt = src_coltb[i].err_pos then bok := false;
      if bok then src_errnb := src_errnb + 1
    end;

    if bok then
    with src_coltb[src_errnb] do
    begin
      err_pos  := src_wchpt;                    { Set word position }
      if err_pos < 1 then err_pos := 1;
      err_code := number;                       { Set error number }
      err_sv   := severity;                     { Set severity }
      err_mdn  := modulesy;                     { Set the module name }
      ERR_GETMSG( src_coltb[src_errnb] )        { Set err_msg }
    end
    { Set the last error severity }
    else src_coltb[src_errnb].err_sv := severity;
    if not ((src_macro in src_flags) or TTY_FILE( src_file )) then
      if EOF( src_file ) then SRC_OUT_ERROR
  end;
  error_cnt[severity] := SUCC( error_cnt[severity] );
  if error_result < severity then error_result := severity
end SRC_ERROR;





{*******************************************************}
{*******   Input line management procedures  ***********}
{*******************************************************}


[global]
procedure SRC_GET_INPUTLINE( var f: text; var pstr: str_ptr );
{ Get a line from the specified file }
{ If btrm then terminal mode, else file mode }
{ Must not be used with a macro file }
{ Read the line with an auto-chage of tabulation character }
var
  ch: char;

begin { SRC_GET_INPUTLINE }
  if lst_current^.lst_currline^.length > 0 then LST_EOLN;

  if pstr = nil then NEW( pstr, 255 );
  with pstr^, src_control^ do
  begin
    length := 0;
    while not EOLN( f ) and not EOF( f ) do
    begin
      READ( f, ch );
      if ch = chr(9) then { ASCII TAB }
        repeat
          if length < 255 then length := length + 1;
          body[length] := ' '
        until (length >= 255) or (length mod 8 = 0)
      else
        if ch >= ' ' then
          if length < 255 then
          begin
            length := SUCC( length ); body[length] := ch
          end
          else
            src_flags := src_flags + [src_toolong]
    end;
    if EOF( f ) then src_flags := src_flags + [src_eof]
                else READLN( f );
    src_linenbr := SUCC( src_linenbr )
  end
end SRC_GET_INPUTLINE;



[global]
procedure SRC_PUT_PROMPT( pstr: str_ptr );
{ Generate a string output on the prompt file }
begin { SRC_PUT_PROMPT }
  if pstr <> nil then
    with src_control^ do
      if src_bterminal in src_flags then
      begin
        if boolean( sys_vms ) then WRITELN( src_promptf );
        WRITE( src_promptf, pstr^ )
      end
end SRC_PUT_PROMPT;



procedure SRC_GET_LINE;
{ Procedure to get a line from the control src file, internal use only }
{ Skip the unused column, generates the prompt echo }
{ Must not be used with a macro }
begin { SRC_GET_LINE }
  with src_control^ do
  begin
    src_chidx := src_frspos-1;                  { Set to first character in the future new line }
    src_errnb := 0;                             { Set to no error in this new line }
    { Get a line sequence with prompt or echo managment }
    if src_macro in src_flags then
    begin { Virtual macro input file case }
      VSR_NEXTSTRING;                           { To get a new input string }
      src_linenbr := SUCC( src_linenbr )
    end
    else
    begin { True input file }
      { Input from a terminal file (Associated to the prompt file) }
      if src_bterminal in src_flags then SRC_PUT_PROMPT( src_prompt );
      { Now get the input line }
      SRC_GET_INPUTLINE( src_file, src_cmdline );
      if src_becho in src_flags then            { Echo to generate }
      begin
        if src_bterminal in src_flags then LST_PUT_STRING( src_prompt^ );
        LST_PUT_STRING( src_cmdline^ );
        LST_EOLN
      end
    end;
    src_wchpt := src_frspos                     { Init word index for error message }
  end
end SRC_GET_LINE;



[global]
procedure SRC_END_OF_LINE;
{ Procedure to output a line with all appropriate error messages,
  to be called by src_inchar }
begin { SRC_END_OF_LINE }
  with src_control^, lst_current^ do
  begin
    if src_toolong in src_flags then
    begin
      { Push the too long error if possible }
      src_wchpt := src_lstpos - 1; SRC_ERROR( 'S_GL', 1, e_error );
      src_flags := src_flags - [src_toolong]
    end;
    if (src_blist in src_flags) and (src_lstmxlev >= src_level) then
    if not (lstf_virtual in lst_flagsw) then
    begin
      { Write the source line on listing }
      LST_NEWLINE;
      SRC_OUTLINE( lst_file, false )
    end;
    if src_errnb > 0 then SRC_OUT_ERROR         { Output errors };
  end
end SRC_END_OF_LINE;



function SRC_GET_CHAR: char;
var
  ch: char;

begin
  with src_control^, src_cmdline^ do
  begin
    while not (src_eof in src_flags)
          and (src_eoln in src_flags) do
    { An other source line is required }
    begin
      { Nothing to generate if first line of source file }
      if (src_linenbr > 0) and not (src_listbyline in src_flags) then
        SRC_END_OF_LINE;                        { Generate listing and error message }
      { Take off eoln for data mode };
      src_flags := src_flags - [src_eoln,src_listbyline];
      SRC_GET_LINE;                             { Get the next line }
      { Set comment flag at a begin of line }
      src_flags := src_flags + [src_commentflg]
    end;
    if not (src_eof in src_flags) then
      if (src_chidx >= src_lstpos) or (src_chidx >= length) then
      begin
        if (src_listbyline in src_flags) and (src_linenbr > 0) then
          SRC_END_OF_LINE;
        if [src_linemode,src_eolcomment] * src_flags <> [] then
          ch := CHR( 1 ) else ch := ' ';
        src_flags := src_flags + [src_eoln]
      end
      else
      begin
        src_chidx := SUCC( src_chidx ); ch := body[src_chidx]
      end
    else
    begin
      ch := chr(0);                             { End of file seen }
      src_flags := src_flags + [src_eofrc]
    end
  end;
  SRC_GET_CHAR := ch
end SRC_GET_CHAR;



function SKIP_COMMENT( stp: char; nlmd: boolean ): char;
var
  ch: char;
  savmode: src_comty;
  ct:   boolean;

begin { SKIP_COMMENT }
  ct := false;
  with src_control^ do
  begin
    { If no significant character in the line set the comment flag char. }
    if (src_commentflg in src_flags) and (src_fchcd = ' ') then
      src_fchcd := 'C';
    if nlmd then src_flags := src_flags + [src_eolcomment];
    savmode := src_commentty;
    src_commentty := src_nocomment;             { Do not proceed recursive comment }
    repeat
      repeat
        ch := SRC_GET_CHAR;
      until (ch = chr(0)) or (ch = stp) or (nlmd and (ch = CHR( 1 )));
      case savmode of
        src_pascomment:
          begin
            ct := ((stp = '}') and (ch = stp)) or (ch = chr(0));
            if (stp = '*') and (ch <> chr(0)) then
            begin
              if src_chidx < src_cmdline^.length then
              begin
                ct := ( src_cmdline^.body[src_chidx + 1] = ')' );
                if ct then ch := SRC_GET_CHAR
              end
            end
          end;

        src_plicomment:
          if ch <> CHR( 0 ) then
          begin
            if src_chidx < src_cmdline^.length then ch := SRC_GET_CHAR;
            ct := (ch = '/')
          end else ct := true;

        src_nocomment,                          { Must never seen }

        src_adacomment,
        src_lispcomment:  ct := true;           { Always a good stop comment }

      end;

    until ct;
    if nlmd then src_flags := src_flags - [src_eolcomment];
    src_commentty := savmode;
    { All comments are equivalent to one space }
    if ch <> CHR( 0 ) then
      if ((ch = CHR( 1 )) and not (src_linemode in src_flags))
         or (ch > CHR( 1 )) then
           ch := ' '
  end;
  SKIP_COMMENT := ch
end SKIP_COMMENT;



[global]
function SRC_INCHAR: char;
{ Can be used for real (terminal or other device) file or macro text,
  to get one character from input }
var
  ch: char;

begin { SRC_INCHAR }
  with src_control^, src_cmdline^ do
  begin
    ch := SRC_GET_CHAR;
    case src_commentty of

      src_pascomment:
        if ch = '{' then ch := skip_comment( '}', false ) else
          if src_chidx < length then
            if (ch = '(') and (body[src_chidx+1] = '*' ) then
            begin
              ch := SRC_GET_CHAR;
              ch := SKIP_COMMENT( '*', false )
            end;

      src_plicomment:
        if src_chidx < length then
          if (ch = '/') and (body[src_chidx+1] = '*' ) then
          begin
            ch := SRC_GET_CHAR;
            ch := SKIP_COMMENT( '*', false )
          end;

      src_lispcomment:
        if ch = ';' then ch := SKIP_COMMENT( ';', true );

      src_adacomment:
        if src_chidx < length then
          if (ch = '-') and (body[src_chidx+1] = '-' ) then
            ch := SKIP_COMMENT( CHR( 1 ), true );

      src_nocomment:
        { Nothing to do }

    end { Case };

    if src_commentflg in src_flags then
      if (ch <> ' ') and (ch > CHR( 1 )) then
      begin
        src_flags := src_flags - [src_commentflg];
        if src_fchcd = 'C' then src_fchcd := ' '
      end;

    src_lastchar := ch;
    SRC_INCHAR := ch
  end
end SRC_INCHAR;



[global]
function SRC_NEXT_CHAR: char;
begin
  with src_control^, src_cmdline^ do
    if (src_chidx >= length) or (src_chidx >= src_lstpos) then
      SRC_NEXT_CHAR := ' '
    else
      SRC_NEXT_CHAR := body[src_chidx+1]
end SRC_NEXT_CHAR;



[global]
function SRC_SEARCH_CHAR: char;
var
  b: boolean;
  c: char;
  i: integer;

begin
  with src_control^, src_cmdline^ do
  begin
    i := src_chidx;
    b := true;
    while b do
      if (i <= length) and (i<= src_lstpos) then
      begin
        c := body[i];
        if c = ' ' then i := i + 1
                   else b := false
      end
      else
      begin
        b := false;
        c := ' '
      end
  end;
  SRC_SEARCH_CHAR := c
end SRC_SEARCH_CHAR;



{*******************************************************}
{*********    common initialization routine    *********}
{*******************************************************}


[global]
procedure SRC_INIT( in_var def_prt, def_src: string );
{ Common command source initialization routine.
  Must be called after lst_txf_st_init procedure(s).
  this call open the default listing file as current listing,
  here we assume that this current file is the output error message file.
}
var
  ierr: integer := -1;

begin { SRC_INIT }
  { Set the default prompt }
  STR_COPY_LIM( def_prompt, def_prt, 64 );
  { Initialize error sub_system }
  for error_result := e_success to e_fatal do error_cnt[error_result] := 0;
  error_result := e_success;
  { Create a source for input/ouput managment }
  src_control := nil;                           { Allocate source }
  if def_src.length > 0 then                    { Some file specification given }
    SRC_OPEN( src_control, def_src, false, ierr );
  if ierr = 0 then
    with src_control^ do
    begin
      src_level    := 1;                        { File is at level one };
      src_lstmxlev := 1                         { Set last listing level to 1 }
    end
  else src_control := nil;
  src_hde       := src_control;
  src_cont_base := src_control
end SRC_INIT;

end.
