{    **************************************************************
     *                                                            *
     *                                                            *
     *              *  C P A S  *  S Y S T E M  *                 *
     *                                                            *
     *                                                            *
     *      * * *   S t a n d a r d   L i b r a r y   * * *       *
     *                                                            *
     *                                                            *
     *          ---  Large Set Management Library  ---            *
     *                                                            *
     *   by :                                                     *
     *                                                            *
     *       P. Wolfers                                           *
     *         c.n.r.s.                                           *
     *         Laboratoire de Cristallographie                    *
     *         B.P.  166 X   38042  Grenoble Cedex                *
     *                           FRANCE.                          *
     *                                                            *
     **************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}
%pragma trace 0;
module CPAS__LSETOPE;



(************   Large set Operators procedures  **************)

const
  max_card = 256;  { Maximum Cardinality Allowed }
  max_bset = max_card div 8;            { Maximum size (in Byte) of a Set }
  max_wset = max_card div integer"size; { Maximum size (in Long) of a Set }

type
  wenm = 0..integer"size*8-1; { if integer 4 octets ==> 0..31 }
  wset = set of wenm;

const
  wcard = wset"cardinality; { Long Word Card = 32 elem. }
  wlast = wenm"last;        { Long Word Set Element Last = 31 }
  wsize = wset"size;        { Long Word Set Size (4 bytes) }

type
  { Set of one Byte }
  eset = record case integer of
           0:( tb: array[1..wsize] of byte );
           1:( iv: integer );
           2:( sv: wset)
         end;

  lset  = array[1..max_wset] of eset;
  plset = ^lset;

  { Array of byte use as formal for Large Set Assignement }
  blset = array[1..max_bset] of byte;


var
  sd1, sd2: eset;  { Two sets of One word For Short_Set oper Large_Set }





[external 'PAS__ERROR'] procedure ERROR( nerr: integer ); external;


[global 'PAS__ASSIGN_LSET']
procedure LSET_ASSIGN(    var dst: blset; cad: integer;
                       in_var src: blset; szs: integer );
var
  szd: integer;
  msq: [static] array[1..7] of byte := (  1,  3,  7, 15, 31, 63,127);

begin
  { Build the Destination Size in byte(s) }
  szd := (cad + 7) div 8;
  { When the Destination and Source sizes are differents }
  if szs <> szd then
    if szs > szd then
    begin { Source Size is more large than destination'one }
      for i := szd + 1 to szs do
        if src[i] <> 0 then ERROR( 72 );
      szs := szd { Now we use only the common size }
    end
    else
      { When the destination is more large - Complet it by zero(s) }
      for i := szs + 1 to szd do  dst[i] := 0;

  { Copy all significant bytes of the destination set }
  for i := 1 to szs do  dst[i] := src[i];

  { Test for unsignificant bit in the last byte }
  cad := cad rem 8;   { Residual Cardinality for the last Byte }
  if cad <> 0 then    { Some final bits must be Null }
    if dst[szs] > msq[cad] then ERROR( 72 )
end LSET_ASSIGN;



[global 'PAS__SET_GENERATOR']
function LSET_GEN( lsetp: plset; card, lval: integer ): plset;
var
  nw, iw: integer;
  ib:     wenm;

begin
  { Get the number of wset in the large set }
  if (lval >= 0) and (lval <= card - 1) then
  begin { O.K. for large set generation }
    nw := (card + wlast) div wcard; { Get the number of elem. }
    iw := lval div wcard;           { Get the element number }
    ib := lval rem wcard;           { Get the bit number }
    for i := 1 to nw do
      with lsetp^[i] do
      if i = iw then sv := [ib]     { Set the appropriate bit of the element }
                else iv := 0        { Set to 0 all element bits }
  end
  else ERROR( 71 );                 { Cardinality Overflow }
  LSET_GEN := lsetp
end LSET_GEN;



procedure ENLARGE_SSET( var sd1: eset; var src: plset; var sz: integer );
begin
  for i := 1 to wsize do
    if i <= sz then sd1.tb[i] := src^[1].tb[i]
               else sd1.tb[i] := 0;
  sz  := wsize;
  src := sd1"address
end ENLARGE_SSET;


[global 'PAS__BIS_LSET']
function LSET_UNION( dst: plset;                  { Address of Ret }
                     s1:  plset; s1_sz:  integer; { Address and size of s1  }
                     s2:  plset; s2_sz:  integer ): plset;
var
  sz, sz1, sz2: integer;

begin
  { Extend the small set(s) to One Long Word }
  if s1_sz < wsize then ENLARGE_SSET( sd1, s1, s1_sz );
  if s2_sz < wsize then ENLARGE_SSET( sd2, s2, s2_sz );

  sz1 := s1_sz div wsize; { Get the size of s1 in UL }
  sz2 := s2_sz div wsize; { Get the size of s2 in UL }

  { Get the Common Set Size }
  if sz1 > sz2 then sz := sz2
               else sz := sz1;

  { Perform the UNION of the common Part }
  for i := 1 to sz do  dst^[i].sv := s1^[i].sv + s2^[i].sv;

  { ... and append the Remainder Part }
  if sz1 > sz then
    for i := sz+1 to sz1 do  dst^[i].iv := s1^[i].iv
  else
    for i := sz+1 to sz2 do  dst^[i].iv := s2^[i].iv;

  LSET_UNION := dst
end LSET_UNION;




[global 'PAS__BAND_LSET']
function LSET_INTER( dst: plset;
                     s1:  plset; s1_sz:  integer;
                     s2:  plset; s2_sz:  integer  ): plset;
var
  sz, szm, sz1, sz2: integer;

begin
  { Extend the small set(s) to One Long Word }
  if s1_sz < wsize then ENLARGE_SSET( sd1, s1, s1_sz );
  if s2_sz < wsize then ENLARGE_SSET( sd2, s2, s2_sz );

  sz1 := s1_sz div wsize; { Get the size of s1 in UL }
  sz2 := s2_sz div wsize; { Get the size of s2 in UL }

  { Get the common and Maximum Set sizes }
  if sz1 > sz2 then begin  sz := sz2; szm := sz1  end
               else begin  sz := sz1; szm := sz2  end;

  { Perform the intersection }
  for i := 1 to sz do  dst^[i].sv := s1^[i].sv * s2^[i].sv;

  { Complete with zero }
  for i := sz + 1 to szm do  dst^[i].iv := 0;

  LSET_INTER := dst
end LSET_INTER;


[global 'PAS__BIC_LSET']
function LSET_DIFF( dst: plset;
                    s1:  plset; s1_sz:  integer;
                    s2:  plset; s2_sz:  integer  ): plset;
var
  sz, szm, sz1, sz2: integer;

begin
  { Extend the small set(s) to One Long Word }
  if s1_sz < wsize then ENLARGE_SSET( sd1, s1, s1_sz );
  if s2_sz < wsize then ENLARGE_SSET( sd2, s2, s2_sz );

  sz1 := s1_sz div wsize; { Get the size of s1 in UL }
  sz2 := s2_sz div wsize; { Get the size of s2 in UL }

  { Get the Common Set Size }
  if sz1 > sz2 then sz := sz2
               else sz := sz1;

  { Perform the Extraction in Common Set Part }
  for i := 0 to sz do  dst^[i].sv := s1^[i].sv - s2^[i].sv;

  if sz1 > sz then
    { Complet the set part by the element of s1 if it is the more large }
    for i := sz+1 to sz1 do  dst^[i].iv := s1^[i].iv
  else
    if sz2 > sz then
      { Or by som [] if s2 is the largest }
      for i := sz+1 to sz2 do  dst^[i].iv := 0;

  LSET_DIFF := dst
end LSET_DIFF;


[global 'PAS__COM_LSET']
function LSET_COMPL( dst: plset;
                     src: plset; cad: integer   ): plset;
var
  sz: integer;
  mq: eset;

begin
  { Compute the set size in word }
  sz := (cad + wlast) div wcard;

  { Complement each word Set }
  for i := 1 to sz do  dst^[i].sv := - src^[i].sv;

  { We must cut the unsignificant bits in the last word }
  cad := cad rem wcard;       { Compute the last Word Cardinality }
  if cad > 0 then
  with dst^[sz] do
  begin
    if cad = 31 then mq.iv := maxint
                else mq.iv := 2**cad - 1;
    dst^[sz].sv := dst^[sz].sv * mq.sv
  end;

  LSET_COMPL := dst
end LSET_COMPL;



[global  'PAS__RINCL_LSET']
function LSET_GE( s1: plset; sz1: integer;
                  s2: plset; sz2: integer ): boolean;
{ Test of Inclusion of s2 in s1 }
var
  sz: integer;

begin
  if sz2 > sz1 then
  begin
    sz := sz1;
    for i := sz1 to sz2-1 do
      if s2^[i].iv <> 0 then return false
  end
  else sz := sz2;

  for i := 1 to sz do
    if not (s2^[i].sv <= s1^[i].sv) then return false;

  return true
end LSET_GE;


[global  'PAS__INCL_LSET']
function LSET_LE( s1: plset; sz1: integer;
                  s2: plset; sz2: integer ): boolean;
var
  sz: integer;

begin
  if sz1 > sz2 then
  begin
    sz := sz2;
    for i := sz2 to sz1-1 do
      if s1^[i].iv <> 0 then return false
  end
  else sz := sz1;

  for i := 0 to sz - 1 do
    if not (s1^[i].sv <= s2^[i].sv) then return false;
  return true
end LSET_LE;


[global  'PAS__NEQ_LSET']
function LSET_NE( s1: plset; sz1: integer;
                  s2: plset; sz2: integer ): boolean;
var
  sz: integer;

begin
  if sz1 <> sz2 then
    if sz1 > sz2 then
    begin
      sz := sz2;
      for i := sz2 to sz1 -1 do
        if s1^[i].iv <> 0 then return true
    end
    else
    begin
      sz := sz1;
      for i := sz1 to sz2 -1 do
        if s2^[i].iv <> 0 then return true
    end
  else sz := sz1;

  for i := 0 to sz - 1 do
    if s1^[i].iv <> s2^[i].iv then return true;

  return false
end LSET_NE;


[global  'PAS__EQU_LSET']
function LSET_EQ( s1: plset; sz1: integer;
                  s2: plset; sz2: integer ): boolean;
var
  sz: integer;

begin
  if sz1 <> sz2 then
    if sz1 > sz2 then
    begin
      sz := sz2;
      for i := sz2 to sz1 -1 do
        if s1^[i].iv <> 0 then return false
    end
    else
    begin
      sz := sz1;
      for i := sz1 to sz2 -1 do
        if s2^[i].iv <> 0 then return false
    end
  else sz := sz1;

  for i := 0 to sz - 1 do
    if s1^[i].iv <> s2^[i].iv then return false;

  return true
end LSET_EQ;


[global  'PAS__INOP_LSET']
function LSET_IN( elem : integer; s: plset; s_card: integer ): boolean;
begin
  if (elem < 0) or (elem >= s_card) then return false;
  return (elem rem wcard) in s^[elem div wcard].sv;
end LSET_IN;


end.
