program PLOT_PATTERN;

%include 'PASENV:draw_defs.pas' {, list_on};




  {**********************************************************************************
  *                                                                                 *
  *                        Global  Constants  Declarations                          *
  *                                                                                 *
  **********************************************************************************}


const
  npat_max  =            32;                                    { Maximum number of pattern to select }
  pi        = 4*ARCTAN( 1 );                                    { The pi number }
  inrd      =      pi/180.0;                                    { Cte to change from decimal degrees to radians }

  icetemp   =        273.16;                                    { Ice fusion temperature in Kelvin }




  {**********************************************************************************
  *                                                                                 *
  *                         Global  types  Declarations                             *
  *                                                                                 *
  **********************************************************************************}


type
  functy = (                                                    { * Program function definitions }
                        fc_write,                               { Function to write an output file }
                        fc_3dplot,                              { Function to do a 3D plot }
                        fc_plot,                                { Function to do a plot }
                        fc_ndia,                                { Function to specify a diagram number }
                        fc_light,                               { Function to set the Light mode }
                        fc_celsius,                             { Function to set celsius degrees for temperature }
                        fc_label,                               { Function for label on pattern plots }
                        fc_dir,                                 { Function Ctalogue/Directory }
                        fc_verbose                              { Flag for verbose run mode }
           );

  file_format = (                                               { * Pattern File format kind definitions }
                        pf_standard,                            { Future standard format }
                        pf_d1b,                                 { D1b/D20 file format }
                        pf_column,                              { Column format }
                        pf_xydata,                              { XY-DATA format }
                        pf_macinelrx,                           { X-Ray Inel on Macintosh format }
                        pf_null                                 { Null format for unspecified format }
                );

  data_point = record                                           { * Record definition for a pattern point }
    flag:  integer;                                             { Flag for point validity }
    theta,                                                      { Theta angle }
    int:   real                                                 { Intensity count }
  end;

  pat_ptr = ^pat_rec;                                           { Define a pointer of pattern }

  pat_rec( dim: integer ) = record                              { * Definition of Pattern record }
    next:                              pat_ptr;                 { Pointer to next pattern }
    idcnt,                                                      { Pattern index }
    numor:                             integer;                 { Integer Identifier NUMOR }
    dateh,                                                      { Pattern date and hour }
    sample,                                                     { sample on the measure }
    comment:                      string( 64 );                 { Pattern comment of 64 character max. }
    monitor,                                                    { Monitor count }
    cnttime,                                                    { Counting time }
    dtheta, omega, chi, phi,                                    { Initial characteristic angles }
    tr1, tr2, step,                                             { sample position X, Y values, theta step }
    t_set, t_reg, t_sample,                                     { Pattern temperature: Setting, on regulator, on sample }
    lambda,                                                     { Wave length (old rvp1 free parameter) }
    rvp2, rvp3, rvp4, rvp5,                                     { Free real parameters rvp1..rvp9 }
    rvp6, rvp7, rvp8, rvp9:               real;
    ista, iend,                                                 { Scan index limit number for internal use }
    ivp1, ivp2:                        integer;                 { Free integer parameters ivp1 and ivp2 }
    amin, amax,                                                 { Deduced Minimaxi on 2*theta }
    min,  max:                            real;                 { Deduced Minimum and maximum of intensity }
    dat:           array[1..dim] of data_point                  { The pattern dim points }
  end;




  {**********************************************************************************
  *                                                                                 *
  *                       Global  Variables  Declarations                           *
  *                                                                                 *
  **********************************************************************************}


var
  out_name,                                                     { Output file specification }
  inp_name:                    string   :=          '';         { Input file specification }

  pat_sel,                                                      { Pointer to the selected pattern }
  pat_first,                                                    { Pointer to first and last pattern }
  pat_last:                    pat_ptr  :=         nil;

  inp_frm,                                                      { input file format }
  out_frm:                     file_format  :=  pf_d1b;         { Output file format }

  funcs:                       set of functy    :=  [];         { Set of function to execute }

  ndiat,                                                        { Number of pattern to select by temperature }
  ndia,                                                         { Number of Pattern to Select by number }
  npat:                        integer  :=           0;         { Number of pattern }

  tbtem:               array[1..npat_max]   of    real;         { Table of Pattern temperature to select }
  tbnp:                array[1..npat_max]   of integer;         { Table of Pattern number to select }

  tbpat:               array[1..2*npat_max] of pat_ptr;         { Table of Pattern pointer to select }
  tbknd:               array[1..2*npat_max] of boolean;         { Table of Marker mode }
  tbcol,                                                        { Table of color }
  tbtyp:               array[1..2*npat_max] of integer;         { Table of Marker type }
  tbsiz:               array[1..2*npat_max] of    real;         { Table of Marker type }


  dstatus,                                                      { Return status for DRAW Graphic Library }
  irep:                                           Dint;         { Return value for DRAW$DIALOG Procedure }

  glambda,                                                      { External user specified Wave Length }
  g2thmin, g2thmax,                                             { External user specified 2theta Range }
  g2thstep,                                                     { External user specified 2theta step }

  gaxxsz, gaxysz, gaxzsz,                                       { User Prefered axis size (in cm) }

  spcint,                                                       { Intensity spacing for multi patterm plot }
  smin,  smax,                                                  { Minimum and Maximum of Selected Pattern }
  tmin,  tmax,                                                  { Minimum and Maximum of tempertypature }
  thmin, thmax,                                                 { Minimum and Maximum of angle 2*theta }
  gmin,  gmax,                                                  { General minimum and maximum }
  xs,    ys:                    Dfloat  :=         0.0;         { Paper size for draw }

  drw_init_flg:                 Dbool   :=       false;         { Flag for Initialize graphic }

  out,                                                          { Output for results }
  inp:                                            text;         { Input File }





  {**********************************************************************************
  *                                                                                 *
  *           Procedure and Function to manage the Pattern structures               *
  *                                                                                 *
  **********************************************************************************}





  procedure GET_PARAMETERS;
  { Procedure to get the input and output file specifications from the command arguments. }

  const
    opttab_sz = 36;
    opttab_sm = 14;
    opt_nval  = 64;
    frmtab_sz =  6;

  type
    opt_identy = ( opt_inpfrm,                                  { Input format option }
                   opt_outfrm,                                  { Output format option }
                   opt_outpat,                                  { Output pattern file specification }
                   opt_taxis,                                   { Select Celcius/Kelvin for the Temperature Axis }
                   opt_tempsel,                                 { For temperature pattern selection }
                   opt_3dplot,                                  { 3D Plot required }
                   opt_2dplot,                                  { 2D Plot required }
                   opt_axisz,                                   { 2D/3D axis size request }
                   opt_shift,                                   { 2D Pattern shift specification }
                   opt_npatt,                                   { Specify the Number of pattern to get }
                   opt_pcolor,                                  { Pattern Color Selection option }
                   opt_pltlin,                                  { Plot Pattern Line attributes option }
                   opt_pltmrk,                                  { Plot Pattern Marker attributes option }
                   opt_dir,                                     { Create a directory list }
                   opt_lambda,                                  { Specify a wave length }
                   opt_thlim,                                   { Specify Theta range }
                   opt_verbose,                                 { Verbose flag }
                   opt_help,                                    { Help display }
                   opt_not_opt                                  { Not an option == ERROR }
                 );

    opt_entry = record                                          { Define an option entry }
                  opt_iden: opt_identy;
                  opt_name: string( opttab_sm )
                end;

    opt_tabty = array[1..opttab_sz] of opt_entry;               { Define the option table type }

    frm_entry = record                                          { Define a file format entry }
                  frm_iden: file_format;
                  frm_name: string( opttab_sm )
                end;

    frm_names = array[1..frmtab_sz] of frm_entry;               { Define the file format table type }


  const
    opt_table = opt_tabty[ [opt_inpfrm,  '-if'     ], [opt_inpfrm,  '-inp_format'     ],
                           [opt_outfrm,  '-of'     ], [opt_outfrm,  '-out_format'     ],
                           [opt_outpat,  '-o'      ], [opt_outpat,  '-output'         ],
                           [opt_taxis,   '-tu'     ], [opt_taxis,   '-t_unit_axis'    ],
                           [opt_3dplot,  '-3d'     ], [opt_3dplot,  '-3dplot'         ],
                           [opt_2dplot,  '-2d'     ], [opt_2dplot,  '-2dplot'         ],
                           [opt_npatt,   '-ns'     ], [opt_npatt,   '-pn_select'      ],
                           [opt_tempsel, '-ts'     ], [opt_tempsel, '-pt_select'      ],
                           [opt_shift,   '-psh'    ], [opt_shift,   '-pattern_shift'  ],
                           [opt_pcolor,  '-pc'     ], [opt_pcolor,  '-pattern_color'  ],
                           [opt_pltlin,  '-lk'     ], [opt_pltlin,  '-line_kind'      ],
                           [opt_pltmrk,  '-mk'     ], [opt_pltmrk,  '-marker_kind'    ],
                           [opt_dir,     '-d'      ], [opt_dir,     '-directory'      ],
                           [opt_lambda,  '-w'      ], [opt_lambda,  '-wave_length'    ],
                           [opt_thlim,   '-th'     ], [opt_thlim,   '-theta_range'    ],
                           [opt_axisz,   '-ax'     ], [opt_axisz,   '-axis_sizes'     ],
                           [opt_verbose, '-v'      ], [opt_verbose, '-verbose'        ],
                           [opt_help,    '-h'      ], [opt_help,    '-help'           ]
                         ];

    frm_table = frm_names[ [ pf_standard,  'std'    ],
                           [ pf_d1b,       'd1b'    ],
                           [ pf_d1b,       'd20'    ],
                           [ pf_macinelrx, 'inelrx' ],
                           [ pf_xydata,    'xydata' ],
                           [ pf_column,    'column' ]
                         ];

    in_min = ORD( 'a' ) - ORD( 'A' );


  var
    nv, i, ii, iv, j, jj, ll, np:      integer;
    ch:                                   char;
    sopt, sparm:                  string( 64 );
    rvl:            array[1..opt_nval] of real;
    opt:                            opt_identy;
    frmf:                          file_format;
    helpf:                                text;
    bnop, bhelp, bm:                   boolean;


    procedure INIT_GET_VALUES( var nvl: integer; ll: integer; bfrac: boolean := false );
    const
      opttsz = 10;

    var
      ij, jj, kp, num, den:      integer;
      snum:                 string( 14 );
      ch:                           char;

    begin { INIT_GET_VALUES }
      nvl := 0;
      ij  := 1;
      while (ij <= sparm.length) and (nvl < ll) do
      begin
        jj := INDEX( sparm, ':', nvl + 1 );
        kp := INDEX( sparm, ',', nvl + 1 );
        if kp > 0 then
          if (jj <= 0) or (jj > kp) then jj := kp;
        if jj <= 0 then jj := sparm.length + 1;
        nvl := nvl + 1;
        rvl[nvl] := 0.0;
        if jj > ij then
        begin
          if bfrac then
          begin
            snum := SUBSTR( sparm, ij, jj-ij+1 );
            kp := INDEX( snum, '/' );
            if kp > 1 then
            begin
              READV( snum, num:kp-1, ch, den:snum.length-kp );
              if den < 1 then den := 1;
              rvl[nvl] := num/den
            end
            else READV( sparm:ij, rvl[nvl]:jj-ij+1 )
          end
          else READV( sparm:ij, rvl[nvl]:jj-ij+1 )
        end;
        ij := jj + 1
      end
    end INIT_GET_VALUES;


  begin { GET_PARAMETERS }
    bhelp := false;
    { Set the default mode }
    for ij := 1 to 2*npat_max do
    begin  tbknd[ij] := true; tbtyp[ij] := 1; tbsiz[ij] := 0.3  end;
    glambda    :=   0.0;
    g2thmin    :=  -1.0;
    g2thmax    :=  -1.0;
    g2thstep   :=   0.0;

    { *** Task Argument Loop *** }
    np := 0;
    ii := 1;
    while ii < argc do
    begin
      sparm := argv[ii]^;
      if sparm.length > 0 then
        if sparm[1] = '-' then                                  { It is an Option Name or abreviation }
        begin                                                   { Separes the Option Name and the Option Parameter List }
          jj := INDEX( sparm, '=' );
          if jj > 0 then
          begin
            sopt  := SUBSTR( sparm, 1, jj - 1 );
            if sopt.length > opttab_sm then sopt.length := opttab_sm;
            sparm := SUBSTR( sparm, jj + 1 )
          end
          else
          begin  sopt := sparm; sparm.length := 0  end;

          { Set option name in minor case }
          for ic := 1 to sopt.length do
            if (sopt[ic] >= 'A') and (sopt[ic] <= 'Z') then sopt[ic] := CHR( ORD( sopt[ic] ) + in_min );

          { Looking for this option name in the option table }
          opt := opt_not_opt;                                      { Assume help until shown otherwise }
          for ic := 1 to opttab_sz do
            with opt_table[ic] do
              if opt_name = sopt then opt := opt_iden;

          case opt of
            opt_inpfrm,
            opt_outfrm:
              if sparm.length > 0 then
              begin { Specify input/output File Format }
                for ic := 1 to sparm.length do
                  if (sparm[ic] >= 'A') and (sparm[ic] <= 'Z') then sparm[ic] := CHR( ORD( sparm[ic] ) + in_min );
                frmf := pf_null;
                for ic := 1 to frmtab_sz do
                  with frm_table[ic] do
                  begin
                    j := 1;
                    repeat
                    exit if frm_name[j] <> sparm[j];
                      j := j + 1
                    until j > frm_name.length;
                    if j > frm_name.length then
                      if frmf = pf_null then
                        frmf := frm_iden                         { First matching in format string search - OK }
                      else
                      begin                                     { Second matching - Ambigous format => ERROR }
                        WRITELN( ' *** PLOT_PATTERN Command error : Ambigous file format "', sparm, '. ***' );
                        PASCAL_EXIT( 2 )
                      end
                  end;
                if frmf = pf_null then
                begin
                  WRITELN( ' *** PLOT_PATTERN Command error : Unknown file format "', sparm, '. ***' );
                  PASCAL_EXIT( 2 )
                end;
                if opt = opt_inpfrm then inp_frm := frmf
                                    else out_frm := frmf
              end;

            opt_outpat:
              begin { Specify an output file }
                out_name := sparm;
                funcs := funcs + [fc_write]
              end;


            opt_2dplot:
              begin { 2D Plot request }
                funcs := funcs + [fc_plot];
                if (sparm[1] = 'l') or (sparm[1] = 'L') then funcs := funcs + [fc_label]
              end;

            opt_3dplot:
              begin { 3D Plot Request }
                funcs := funcs + [fc_3dplot];
                if (sparm[1] = 'l') or (sparm[1] = 'L') then funcs := funcs + [fc_light]
              end;

            opt_axisz:
              begin
                INIT_GET_VALUES( nv, 3 );
                if nv > 0 then gaxxsz := rvl[1];
                if nv > 1 then gaxysz := rvl[2];
                if nv > 2 then gaxzsz := rvl[3]
              end;

            opt_npatt:
              begin { Specify an sequence number pattern selection }
                INIT_GET_VALUES( nv, opt_nval );
                for ic := 1 to nv do  tbnp[ic] := ROUND( rvl[ic] );
                ndia := nv;
                funcs := funcs + [fc_ndia]
              end;

            opt_tempsel:
              begin { Specify a temperature pattern selection }
                i := 1; ndiat := 0;
                loop
                  ndiat := ndiat + 1;
                  READV( sparm:i, tbtem[ndiat] );
                exit if (sparm[i] <> ':') and (sparm[i] <> ',');
                  i := i + 1
                end;
                funcs := funcs + [fc_ndia]
              end;

            opt_shift:
              begin { Specify a 2D plot pattern shift }
                INIT_GET_VALUES( nv, 1 );
                if nv >= 1 then spcint := rvl[1]
              end;

            opt_pcolor:
              begin
                i := 1; j := 1;
                loop
                  READV( sparm:i, j );
                  if (j < 1) or (j > 2*npat_max) then j := 1;
                  tbcol[j] := 1;
                exit if sparm[i] <> '-';
                  i := i + 1;
                  READV( sparm:i, tbcol[j] );
                  if (tbcol[j] < 1) or (tbcol[j] > 8) then tbcol[j] := 1;
                exit if (sparm[i] <> ':') and (sparm[i] <> ',');
                  i := i + 1
                end
              end;

            opt_pltlin,
            opt_pltmrk:
              begin { Select kind of line or kind of marker for selected pattern }
                bm := (opt = opt_pltmrk);
                i := 1; j := 1;
                loop
                  READV( sparm:i, j );
                  if (j < 1) or (j > 2*npat_max) then j := 1;
                  tbknd[j] := bm;
                  tbtyp[j] := 1;
                exit if sparm[i] <> '-';
                  i := i + 1;
                  READV( sparm:i, tbtyp[j] );
                  if sparm[i] = '-' then
                  begin  i := i + 1; READV( sparm:i, tbsiz[j] )  end;
                exit if (sparm[i] <> ':') and (sparm[i] <> ',');
                  i := i + 1
                end
              end;

            opt_taxis: { Swap to Celcius or Kelvin display }
              if sparm[1] = 'c' then funcs := funcs + [fc_celsius]      { Set Celsius mode }
                                else funcs := funcs - [fc_celsius];     { Set Kelvin mode - the default) }

            opt_dir:            funcs := funcs + [fc_dir];      { Set the directory request }

            opt_lambda:
              begin { Specify a wave length  and optionaly a scan 2 theta step}
                INIT_GET_VALUES( nv, 1 );
                if nv > 0 then
                begin
                  glambda := rvl[1];
                  if nv > 1 then g2thstep := rvl[2]
                end
              end;

            opt_thlim:
              begin { Specify a display 2 theta range for 2D Plot }
                INIT_GET_VALUES( nv, 3 );
                if nv > 0 then
                begin
                  g2thmin := rvl[1];
                  if nv > 1 then g2thmax := rvl[2];
                end
              end;

            opt_verbose:        funcs := funcs + [fc_verbose];  { Set the verbose flag }

            opt_help:           bhelp           := true;        { Set the Help flag }

          otherwise
            WRITELN( ' *** PLOT_PATTERN Command error : Unknown option "', sopt, '. ***' );
            bhelp := true;
            goto ET_HELP
          end;
        end
        else
        begin { Not an option }
          np := np + 1;
          case np of
            1:  inp_name := sparm;                              { Set the Job Name and default for other files }
            2:  begin
                  out_name := sparm;                            { Set the output file specification ... }
                  funcs := funcs + [fc_write]                   { ... and the file output flag }
                end;
          otherwise
            WRITELN( ' *** PLOT_PATTERN Command warning : The file specification "', sparm, '"  will be ignored. ***' )
          end
        end;
      ii := ii + 1
    end { *** of Task Argument Loop *** };

ET_HELP:
    { Check for complet Requirements }
    if bhelp or (np < 1) then
    begin
      WRITELN( ' *** The PLOT_PATTERN command model is :' );
      WRITELN( ' ':4, 'plot_pattern <option_list> <input_file> [<output_file>]' );
      WRITELN;
      WRITELN( ' Where : <option_list> is a list of                                                               short option name' );
      WRITELN( '    -output=<output_file>          For an alternate specification of a pattern output file,                -o' );
      WRITELN( '    -inp_format=<iformat>          For a format specification of input file (default is d1b),              -if' );
      WRITELN( '    -out_format=<oformat>          For a format specification of output file (default is d1b),             -of' );
      WRITELN( '    -directory                     To get the directory of the file (Pattern characteristics),             -d' );
      WRITELN( '    -t_unit_axis={C|K}             The temperature unit in Celcius (C) or Kelvin (K) - the default,        -tu' );
      WRITELN( '    -3dplot[=Light]                To generate a 3D (temperature dep. plot) with optional Light,           -3d' );
      WRITELN( '    -2dplot[=Label]                To generate a 2D (temperature dep. plot) with opt. pattern label,       -2d' );
      WRITELN( '    -axis_sizes=<x>[:<y>[:<z>]]    To specify prefered axis size (in cm) for x,y (and z with 3D plot),     -ax' );
      WRITELN( '    -pn_select=<pn1>[:<pn2> ...]   To select 2D plot pattern(s) by order number,                           -ns' );
      WRITELN( '    -pt_select=<pt1>[:<pt2> ...]   To select 2D plot pattern(s) by temperature value <ptx> selection,      -pt' );
      WRITELN( '    -pattern_shift=<shift_val>     To set <shift_val> as the count shift between two successive Pattern,   -psh' );
      WRITELN( '    -pattern_color:<cb>-<cl>[:<cb>-<cl>...]                                                                -pc' );
      WRITELN( '                                   To set the color <cl> at the specified pattern <cb>.' );
      WRITELN( '    -marker_kind:<cb>-<nt>[-<sz>][:<cb>-<nt>[-<sz>]...]                                                    -mk' );
      WRITELN( '                                   To set the kind of Marker <nt> for the pattern <cb> with size <sz>.' );
      WRITELN( '    -line_kind:<cb>-<nt>[-<sz>][:<cb>-<nt>[-<sz>]...]                                                      -lk' );
      WRITELN( '                                   To set the kind of Line <nt> for the pattern <cb> with size <sz>.' );
      WRITELN( '    -wave_length=<lambda>,[tstep]  To force a wave length of <lambda> in Angstroem and                     -w' );
      WRITELN( '                                   an optional 2theta step (in decimal degrees),' );
      WRITELN( '    -theta_range=<tmin>[,<tmax>]   To set a displayed t2theta range to be [<tmin>..<tmax>].                -th' );
      WRITELN;
      WRITELN( ' The possible value for <iformat> and <oformat> are : inelxr, d1b, d20, xydata and column respectively for' );
      WRITELN( '        inelrx        X-ray data from Inel J. Marcus diffractometer (only for use with input as "-inp_format=inelrx"),' );
      WRITELN( '        D1B, D20      ILL neutron diffractometer format (D1B and D20 are same), It is the default for input and output,' );
      WRITELN( '        xydata        Standard XYDATA format that can be used only for output with "-out_format=xydata",' );
      WRITELN( '        column        Column format (Output is a file with number in column) to use with many plot and computing programs.' );
      WRITELN;
      WRITELN( ' The <pn1>, <pn2> ... values (integer numbers) are the ordering numbers of patterns in the input file.' );
      WRITELN( ' The <pt1>, <pt2> ... values (floating numbers) are the temperatures of patterns to select in the input file.' );
      WRITELN( ' The <shift_val>      value (floating number) is the shift (in count value) between to successive selected pattern to plot.' );
      WRITELN( ' The <cb>             values (integer numbers) are the sequence number of selected pattern to plot.' );
      WRITELN( ' The <cl>             values correspond to the color number as 0-white, 1-black, 2-red, 3-blue, 4-green, 5-yellow, 6-cyan, 7-magenta.' );
      WRITELN( ' The <nt>             values are corresponding to the numebr of marker or line kind (in agreement with the DRAW manual).' );
      WRITELN( ' The <sz>             values (floating numbers) are the thickness of lines or size of markers.' ); 
      WRITELN;
      if opt <> opt_not_opt then PASCAL_EXIT( 0 )
                            else PASCAL_EXIT( 2 )
    end
  end GET_PARAMETERS;




  procedure SKIP_INP_CHAR( nn: integer );
  var
    ch: char;

  begin
    for ii := 1 to nn do READ( inp, ch )
  end SKIP_INP_CHAR;



  procedure ELLIMINATE_CR( var st: string );
  const
    CR      = CHR( 13 );                                { ASCII Character 13 to elliminate }

  begin
    with st do
      if length > 0 then
        if body[length] = CR then length := length - 1
  end ELLIMINATE_CR;


  procedure GET_PATTERNS;
  { Procedure to read all the pattern in the input data file (D1B/D20 format) }
  const
    mimast       = 1.0E+10;                             { Value to init the minimum and maximum }

  var
    nda:               integer;



    procedure GET_STD_FORMAT;
(* pat_rec( dim: integer ) = record { * Definition of Pattern record }
    next:                              pat_ptr;                 { Pointer to next pattern }
    idcnt,                                                      { Pattern index }
    numor:                             integer;                 { Integer Identifier NUMOR }
    dateh,                                                      { Pattern date and hour }
    sample,                                                     { sample on the measure }
    comment:                      string( 64 );                 { Pattern comment of 64 character max. }
    monitor,                                                    { Monitor count }
    cnttime,                                                    { Counting time }
    dtheta, omega, chi, phi,                                    { Initial characteristic angles }
    tr1, tr2, step,                                             { sample position X, Y values, theta step }
    t_set, t_reg, t_sample,                                     { Pattern temperature: Setting, on regulator, on sample }
    lambda,                                                     { Wave length (old rvp1 free parameter) }
    rvp2, rvp3, rvp4, rvp5,                                     { Free real parameters rvp1..rvp9 }
    rvp6, rvp7, rvp8, rvp9:               real;
    ista, iend,                                                 { Scan index limit number for internal use }
    ivp1, ivp2:                        integer;                 { Free integer parameters ivp1 and ivp2 }
    amin, amax,                                                 { Deduced Minimaxi on 2*theta }
    min,  max:                            real;                 { Deduced Minimum and maximum of intensity }
    dat:           array[1..dim] of data_point                  { The pattern dim points }
  end;
*)
    begin
    end GET_STD_FORMAT;



    procedure GET_XYD_FORMAT;
    var
      p:                          pat_ptr;                      { Pointer to the new pattern record }
      line1, line2, line3, line4:  string;                      { String to keep the first four lines }
      i, len, nlupat:             integer;
      mi, ma:                        real;
      bs:                         boolean;
      sp:                            char;                      { Space character to read }

    begin
    end GET_XYD_FORMAT;



    procedure GET_D1B_FORMAT;
    var
      p:                          pat_ptr;                      { Pointer to the new pattern record }
      line1, line2, line3, line4:  string;                      { String to keep the first four lines }
      i, len, nlupat:             integer;
      mi, ma:                        real;
      bs:                         boolean;
      sp:                            char;                      { Space character to read }

    begin
      READLN( inp, nlupat );                                    { Ignore the first line (where the number of pattern can be specified) }
      while not EOF( inp ) do                                   { Loop on all pattern(s) }
      begin
        READLN( inp, line1 );
      exit if EOF( inp );
        READV( line1, line2:0:true );                           { Try to find the "-10000" of the end of data file }
      exit if line2 = '-10000';
        READLN( inp, line2 );
        READLN( inp, line3 ); READLN( inp, line4 );
        READV( line4, len );                                    { Get the size of pattern }
        NEW( p, len );                                          { Create the pattern record }
        with p^ do
        begin
          READV( line1, dateh:20, sp, sample:10, sp, comment ); { Get the date, sample name, and comment }
          READV( line2, idcnt, numor );                         { Get the pattern index and NUMOR }
          READV( line3, monitor, cnttime, dtheta,               { Get the monitor and counting time, ... }
                        omega,   chi,     phi,                  { ... Possible Eulerian angles, ... }
                        tr1,     tr2,     step,                 { ... X,Y sample position and 2theta step, ... }
                        t_set,   t_reg,   t_sample );           { ... Temperatures setting, on regulation and on sample. }
          READV( line4, len,     lambda,  rvp2,                 { Get the Size of pattern, wave Length and all free parameters }
                        rvp3,    rvp4,    rvp5,
                        rvp6,    rvp7,    rvp8,
                        rvp9,    ivp1,    ivp2 );

          amin := dtheta;                                       { Set the minimum and maximum of 2*theta }
          amax := dtheta + (dim - 1)*step;
          mi := mimast; ma := - mi;

          i := 0;
          ista := 1; iend := dim;                               { Set the default using }
          while i < dim do
          begin
            with dat[i + 1] do
            begin
              theta := dtheta + i*step;
              READ( inp, flag, int );                           { Read each point of the diagram }
              { Set the diagram minimum and maximum }
              if mi > int then mi := int
                          else if ma < int then ma := int
            end;
            i := i + 1
          end;
          { Set pattern minimaxi }
          min := mi; max := ma;
          { Update the general minimaxi }
          if tmin > t_set then tmin := t_set
                          else if tmax < t_set then tmax := t_set;
          if thmin > amin then thmin := amin;
          if thmax < amax then thmax := amax;
          if gmin > mi then gmin := mi;
          if gmax < ma then gmax := ma
        end;
        { Set the pattern in the pattern list }
        if pat_first = nil then pat_first := p
                           else pat_last^.next := p;
        pat_last := p;
        { Set this pattern in selection table if required }
        npat := npat + 1;
        bs := false;
        for ij := 1 to ndia do  if npat = tbnp[ij] then bs := true;
        for ij := 1 to ndiat do
          if fc_celsius in funcs then
          begin
            if ABS( p^.t_set - tbtem[ij] - icetemp ) < 1.0E-3 then bs := true
          end
          else if ABS( p^.t_set - tbtem[ij] ) < 1.0E-3 then bs := true;
        if bs then
        begin { Form the mini-maxi of selected diagrams }
          if smin > mi then smin := mi;
          if smax < ma then smax := ma;
          { Put the selected diagram in the selection table }
          nda := nda + 1; tbpat[nda] := p
        end;
        READLN( inp );                                      { Skip to end of current count line }
        READLN( inp, nlupat )                               { Read the End Pattern mark (should be -1000) }
      end
    end GET_D1B_FORMAT;



    procedure GET_INLRX_FORMAT;
    const
      inel_step        =  0.02;                         { Step angle between two consecutive points }
      inel_stth        =  0.00;                         { 2 Theta Start Angle }

    var
      p:                       pat_ptr;
      line:                     string;
      nc, i:                   integer;
      th, lb1, lb2, mi, ma:       real;
      bs:                      boolean;
      ch:                         char;

    begin
      repeat                                            { Loop on all pattern(s) }
        nc := 0;                                        { Initialize the number of point in one pattern }
        { Look for "RAW" word }
        repeat
          READLN( inp, line );
          if EOF( inp ) then nc := -1;
        until (nc < 0) or ((line[1] = 'R') and (line[2] = 'A') and (line[3] = 'W'));
      exit if nc < 0;
        READLN( inp, nc );                              { Read the size (in point) of the pattern }
        NEW( p, nc );                                   { Allocate  a pattern record for nc points }
        with p^ do
        begin                                           { Now we are filling the pattern record }
          next       :=          nil;                   { Default is to have no successor }
          idcnt      :=            0;                   { Set pattern index to null index value }
          numor      :=           -1;                   { No numor }
          monitor    :=            0;                   { No monitor }
          dtheta     :=    inel_stth;                   { Get initial 2 theta angle }
          omega      :=          0.0;                   { Set Eulrian Angle to 0.0 }
          chi        :=          0.0;
          phi        :=          0.0;
          tr1        :=          0.0;
          tr2        :=          0.0;
          step       :=    inel_stth;                   { Set the fixed theta step }
          rvp2       :=          0.0;                   { Init to zero all Free parameter fields }
          rvp3       :=          0.0;
          rvp4      :=           0.0;
          rvp5       :=          0.0;
          rvp6       :=          0.0;
          rvp7       :=          0.0;
          rvp8       :=          0.0;
          rvp9       :=          0.0;
          ivp1       :=            0;
          ivp2       :=            0;

          READLN( inp, cnttime );                       { Read the counting time }
          for ii := 1 to 7 do READLN( inp );            { Skip the next seven lines }
          READLN( inp, comment );                       { Read the Pattern comment }
          ELLIMINATE_CR( comment );
          READLN( inp, lb1 );                           { Read the Wave length for alpha1 and alpha 2 }
          READLN( inp, lb2 );
          lambda := (lb1 + lb2)/2.0;
          if glambda > 0.0 then lambda := glambda;      { Force the Lambda when it is externaly specified }
(*
          READLN( inp, dateh );                         { Get the date and hour of Measurement >>> Actually ignored }
          SKIP_INP_CHAR( 12 );
*)
          READLN( inp, ch, dateh );
          ELLIMINATE_CR( dateh );
          numor := npat + 1;                            { Set the integer identifier }
          SKIP_INP_CHAR( 12 );
          READLN( inp, sample );                        { Get sample field }
          ELLIMINATE_CR( sample );
          SKIP_INP_CHAR( 15 );                          { Skip 15 characters }
          READLN( inp, t_sample );                      { Get the temperature }
          t_sample := t_sample + icetemp;               { Convert it in Kelvin }
          t_set    := t_sample;                         { Default assignements }
          t_reg    := t_sample;
          READLN( inp );                                { Skip one line }
          READLN( inp );                                { Skip one line }
          repeat
            READLN( inp, line );
          until line[1] = '0';                          { Look for the begin of pattern Counts }
          i := 0;
          amin := dtheta;                               { Set the minimum and maximum of 2*theta }
          amax := dtheta + nc*step;
          mi := mimast; ma := - mi;
          ista := 1; iend := nc;
          while i < nc do
          begin
            with dat[i + 1] do
            begin
              flag := 1;                                { Set as a valid point }
              theta := dtheta + i*step;
              READLN( inp, int );                       { Read each point of the diagram }
              { Set the diagram minimum and maximum }
              if mi > int then mi := int
                          else if ma < int then ma := int
            end;
            i := i + 1
          end;
          { Set pattern minimaxi }
          min := mi; max := ma;
          { Update the general minimaxi }
          if tmin > t_set then tmin := t_set
                          else if tmax < t_set then tmax := t_set;
          if thmin > amin then thmin := amin;
          if thmax < amax then thmax := amax;
          if gmin > mi then gmin := mi;
          if gmax < ma then gmax := ma
        end;
        { Set the pattern in the pattern list }
        if pat_first = nil then pat_first := p
                           else pat_last^.next := p;
        pat_last := p;
        { Set this pattern in selection table if required }
        npat := npat + 1;
        bs := false;
        for ij := 1 to ndia do  if npat = tbnp[ij] then bs := true;
        for ij := 1 to ndiat do
          if fc_celsius in funcs then
          begin
            if ABS( p^.t_set - tbtem[ij] - icetemp ) < 1.0E-3 then bs := true
          end
          else if ABS( p^.t_set - tbtem[ij] ) < 1.0E-3 then bs := true;
        if bs then
        begin { Form the mini-maxi of selected diagrams }
          if smin > mi then smin := mi;
          if smax < ma then smax := ma;
          { Put the selected diagram in the selection table }
          nda := nda + 1; tbpat[nda] := p
        end
      until EOF( inp )
    end GET_INLRX_FORMAT;


  begin { GET_PATTERNS }
    { Initialize the Selected pattern array table }
    for i := 1 to npat_max do  tbpat[i] := nil;
    nda  := 0;                                                  { No selection }
    npat := 0;                                                  { No pattern in memory }

    RESET( inp, inp_name );                                     { Open the data file }

    npat  :=      0;                                            { Initialize the number of pattern }
    tmin  := mimast; tmax  := - tmin;                           { Initialize all minimaxi }
    thmin := mimast; thmax := -thmin;
    gmin  := mimast; gmax  := - gmin;
    smin  := mimast; smax  := - smin;

    case inp_frm of
(*
      pf_standard:   GET_STD_FORMAT;
*)

      pf_d1b:        GET_D1B_FORMAT;
(*
      pf_xydata:     GET_XYD_FORMAT;
*)
      pf_macinelrx:  GET_INLRX_FORMAT;

    otherwise
      WRITELN( ' *** PLOT_PATTERN Unsupported or unimplemented input format. ***' );
      PASCAL_EXIT( 2 )
    end;

    CLOSE( inp );                                               { Close the data file }

    WRITELN( ' PLOT PATTERN  Found ', npat:0, ' patterns in the file "', inp_name, '".' );
    if npat = 0 then
    begin
      WRITELN( ' PLOT PATTERN  Cannot work without Pattern and exit.' );
      PASCAL_EXIT( 4 )
    end
    else
    begin
      WRITELN( ' ':24, 'Global Statistics' );
      WRITELN( ' ':6, 'Minimum and maximum of 2*Theta Angle: ', thmin:8:3, ', ', thmax:8:3, ',' );
      WRITELN( ' ':6, 'Minimum and maximum of Temperature:   ', tmin:8:2,  ', ', tmax:8:2,  ',' );
      WRITELN( ' ':6, 'Minimum and maximum of intensities:   ', gmin:8:3,  ', ', gmax:8:3,  '.' );
      if npat = 1 then
      begin
        ndia := 1;
        tbpat[1] := pat_first;
        if fc_3dplot in funcs then funcs := funcs + [fc_plot,fc_ndia]
      end
      else
        if ndia = 0 then
          if fc_ndia in funcs then ndia :=  nda
                              else ndia := npat;
    end;
  end GET_PATTERNS;




  procedure OUT_CATALOGUE( bsel: boolean );
  var
    p: pat_ptr;
    ct:   char;
    tm:   real;

  begin
    WRITELN;
    if fc_celsius in funcs then ct := 'C'
                           else ct := 'K';
    if bsel then
    begin
      WRITELN( ' ':5, 'List of Selected Diffraction Pattern from the file "', inp_name, '" :' );
      for ij := 1 to ndia do
      with tbpat[ij]^ do
      begin
        if fc_celsius in funcs then tm := t_sample - icetemp
                               else tm := t_sample;
        WRITELN( ' #', numor:-4, ' /  T = ', tm:8:2, ct, ', t = ', cnttime:8:1, ', date/hour = ', dateh,
                                                         ', R = "', sample, '", C = "', comment, '"' )
      end
    end
    else
    begin
      WRITELN( ' ':5, 'List of Diffraction Pattern find in the file "', inp_name, '" :' );
      p := pat_first;
      while p <> nil do
      with p^ do
      begin
        if fc_celsius in funcs then tm := t_sample - icetemp
                               else tm := t_sample;
        WRITELN( ' #', numor:-4, ' /  T = ', tm:8:2, ct, ', t = ', cnttime:8:1, ', date/hour = ', dateh,
                                                         ', R = "', sample, '", C = "', comment, '"' );
        p := next
      end
    end;
    WRITELN
  end OUT_CATALOGUE;





  procedure WRITE_NORMALIZED_FILE;
  var
    p:                     pat_ptr;
    ij:                    integer;
    sn:            [static] string;
    bsingle, bemp:         boolean;

(*
    pf_standard,                            { Future standard format }
    pf_d1b,                                 { D1b/D20 file format }
    pf_column,                              { Column format }
    pf_xydata,                              { XY-DATA format }
    pf_macinelrx,                           { X-Ray Inel on Macintosh format }
    pf_null                                 { Null format for unspecified format }
*)

    procedure OUT_PATTERN( p: pat_ptr; nb: integer );
    var
      step: real;

    begin
      with p^ do
      begin
        case out_frm of
(* { The standard default file format in the future }
          pf_standard:
            begin
            end;
*)

          pf_column: { Column Mode }
            for ij := 1 to dim do
              with dat[ij] do
                WRITELN( out, ij:6, ' ', flag:4, theta:10:3, int:10:0, SQRT( int ):10:0 );

          pf_d1b: { D1B/D20 Mode }
            begin
              step := ROUND( (amax - amin)/(dim - 1) );
              WRITELN( out, dateh:20, sample:10, ' ', comment );
              WRITELN( out, nb:3, ' ', numor:6, ' ', lambda:8:4 );
              WRITELN( out, { Monitor } monitor:12:0, ' ', { Time }  cnttime:9:0, ' ',
                            { 2theta0 }   dtheta:8:3, ' ', { Omega }   omega:8:3, ' ',
                            { Chi }          chi:8:3, ' ', { Phi }       phi:8:3, ' ',
                                             tr1:8:3, ' ',               tr2:8:3, ' ', { Step }      step:8:3, ' ',
                                           t_set:8:3, ' ',             t_reg:8:3, ' ',           t_sample:8:3 );
              WRITELN( out, dim:4, ' ',   lambda:9:3, ' ', rvp2:9:3, ' ', rvp3:9:3, ' ',
                                            rvp4:9:3, ' ', rvp5:9:3, ' ', rvp6:9:3, ' ',
                                            rvp7:9:3, ' ', rvp8:9:3, ' ', rvp9:9:3, ' ',
                                            ivp1:6,   '   ', ivp2:6 );
             for idx := 1 to dim do
               with dat[idx] do
                 begin
                   WRITE( out, ' 1', int:8:0 );
                   if idx mod 10 = 0 then WRITELN( out )
                 end;
              if dim mod 10 <> 0 then WRITELN( out );
              WRITELN( out, -1000:10 )
            end;

          pf_xydata: { XY_DATA Mode }
            begin
              WRITELN( out, 'XYDATA     ', dateh:20, {$cycle} ' ':10, numor:10 );
              WRITELN( out, 'INTER  1.0  1.0  0  0.0' );
              WRITELN( out, {$title} t_sample:8:2 );
              WRITELN( out, comment, ' ', sample );
              WRITELN( out, { Monitor } monitor:12:0, ' ', { Time }  cnttime:9:0, ' ',
                                           t_set:8:3, ' ',             t_reg:8:3, ' ',           t_sample:8:3 );
              WRITELN( out, { Omega }      omega:8:3, ' ', { Chi }       chi:8:3, ' ', { Phi }        phi:8:3, ' ',
                            { Tr1 }          tr1:8:3, ' ', { Tr2 }       tr2:8:3, ' ' );
              for idx := 1 to dim do
                with dat[idx] do
                  if flag > 0 then WRITELN( out, theta:8:3, ' ', int:10:0, SQRT( int ):6:0 )
            end;

        otherwise
          WRITELN( ' *** PLOT_PATTERN Command error : The specified output file format is not currently implemented. ***' );
          PASCAL_EXIT( 2 )
        end
      end
    end OUT_PATTERN;


    function OUT_HEADER( in_var fname: string; nd: integer ): boolean;
    var
      br: boolean := false;

    begin
      case out_frm of
        pf_d1b:
          begin
            REWRITE( out, fname );
            WRITELN( out, nd:5 )
          end;

      otherwise
        br := true
      end;
      OUT_HEADER := br
    end OUT_HEADER;



    procedure OUT_TRAILER;
    begin
      case out_frm of
        pf_d1b:
          begin
            WRITELN( out, -10000:10 )
          end;

      otherwise
      end;
      CLOSE( out )
    end OUT_TRAILER;



  begin { WRITE_NORMALIZED_FILE }
    if ndia > 0 then                                    { One or more Pattern are selected in command line }
      if ndia > 1 then
      begin                                             { Many Pattern to output }
        bsingle := OUT_HEADER( out_name, ndia );        { bn flags when the format is only for one pattern }
        p := pat_first;
        ij := 0;
        repeat
          ij := ij + 1;
          with p^ do
          begin
            if bsingle then
            begin
              WRITEV( sn, out_name, '_', ij:-4 );
              bemp := OUT_HEADER( sn, ij )
            end;
            OUT_PATTERN( tbpat[ij], ij );
            if bsingle then OUT_TRAILER;
            p := next
          end
        until ij >= ndia;
        if not bsingle then OUT_TRAILER
      end
      else
      begin
        bemp := OUT_HEADER( out_name, 1 );
        OUT_PATTERN( tbpat[1], 1 );                     { Only One Pattern to Output }
        OUT_TRAILER
      end
    else                                                { No selected pattern in the command line }
      if npat > 1 then
      begin                                             { Many Pattern to output by default }
        bsingle := OUT_HEADER( out_name, ndia );        { bn flags when the format is only for one pattern }
        p := pat_first;
        ij := 0;
        repeat
          ij := ij + 1;
          with p^ do
          begin
            if bsingle then
            begin
              WRITEV( sn, out_name, '_', ij:-4 );
              bemp := OUT_HEADER( sn, ij )
            end;
            OUT_PATTERN( p, ij );
            if bsingle then OUT_TRAILER;
            p := next
          end
        until p = nil;
        if not bsingle then OUT_TRAILER
      end
      else                                              { One Pattern to Output }
      begin
        bemp := OUT_HEADER( out_name, 1 );
        OUT_PATTERN( pat_first, 1 );                    { Only one Pattern to output }
        OUT_TRAILER
      end
  end WRITE_NORMALIZED_FILE;







  {**********************************************************************************
  *                                                                                 *
  *                               Drawing Procedures                                *
  *                                                                                 *
  **********************************************************************************}



  procedure INIT_PLOT( flg3d: Dbool );
  {****************************************************}
  {          INITIALYSE THE DRAWING SYSTEM             }
  {****************************************************}
  { The external procedure

      DRAW$DEF_MENU( code: Dint; in_var name: [optional] string );

                  is a procedure to modify by insertion the DRAW$DIALOG
    standard menu setting.

    The code are :
      Menu_Close             Close the currently opened menu.
      Menu_Open              Open the next Standard Menu.
      Menu_Skip              Skip_the next standard (Menu or Entry).
      Menu_New    <Name>     Create and Open a new user menu of specified name.
      Menu_Entry  <Name>     Create a menu entry of specified name.
      Menu_Separ             Create a Separator menu Entry.

    The directives Menu_Entry and Menu_Separ allocate an integer identifier
    to the new menu elements beginning from 1. This identifier is returned by
    the procedure DRAW$DIALOG when the corresponding entry is selected by
    the mouse.
    The Menu_skip can be used with the form n*16+Menu_Skip where n is a skip number
    with the following specificities:
      If n < 0 the skip go to the end of current standard menu else,
      if n > 0 the result is to skip n entry in the current standard menu.
 
  }

  var
    unt: unit_type;

  begin
    if flg3d then
    begin
      DRAW$SET_MENU_SPC( Menu_Skip );                   { Skip the standard menu "File" }
      DRAW$SET_MENU_SPC( Menu_New, '&Rotate' );         { Create a New menu "Rotate" }
      DRAW$SET_MENU_SPC( Menu_Entry, '&Omega' );        { Create a New menu entry "Omega" with id = 1 }
      DRAW$SET_MENU_SPC( Menu_Entry, '&Chi' );          { Create a New menu entry "Chi"   with id = 2 }
      DRAW$SET_MENU_SPC( Menu_Entry, '&Phi' );          { Create a New menu entry "Phi"   with id = 3 }
      DRAW$SET_MENU_SPC( Menu_Entry, '&Dist' );         { Create a New menu entry "Dist"  with id = 4 }
      DRAW$SET_MENU_SPC( Menu_Close );                  { Close the user menu "Rotate" }
      if fc_light in funcs then
      begin
        DRAW$SET_MENU_SPC( Menu_New, '&Light' );        { Create a New menu "Light" }
        DRAW$SET_MENU_SPC( Menu_Entry, '&L0 On/off' );  { Create a New menu entry "Light 0 On/off"   with id = 5 }
        DRAW$SET_MENU_SPC( Menu_Entry, '&L1 On/off' );  { Create a New menu entry "Light 1 On/Off"   with id = 6 }
        DRAW$SET_MENU_SPC( Menu_Close )                 { Close the user menu "Rotate" }
      end;
      DRAW$SET_MENU_SPC( Menu_Close );                  { Close the main standard menu (barre menu) }
      if fc_light in funcs then
        Draw_Server_Mode := DrwSrv_Do_Lighting +        { Require the Lighting mode }
                            DrwSrv_Enable_Light0 +
                            DrwSrv_Enable_Light1;
    end;

    DRAW$INIT( xs, ys, unt, 'Read Pattern' );

    drw_init_flg := true                                { Init of DRAW is done }
  end INIT_PLOT;








  procedure MAKE_3D;
  const
    light0_amb_color =  Dcolor[ 0.6, 0.6, 0.6, 1.0];    { Color for Ambiant light }
    light0_dif_color =  Dcolor[ 1.0, 1.0, 1.0, 1.0];    { Color for Diffuse light }
    light0_spe_color =  Dcolor[ 1.0, 1.0, 1.0, 1.0];    { Color for Specular light }
(*
    grid_color       =  Dcolor[ 0.9, 0.8, 0.6, 1.0];    { Color of lines use when lighting is disable }
    fill_color       =  Dcolor[ 0.3, 0.3, 0.7, 1.0];    { Filling color }
*)
  type
    tbraw( sz: integer ) = record
                             tb: array[1..sz] of Dfloat
                           end;

  var
    dview,                                              { View distance (cm) }
    ome, chi, phi,                                      { Eulerian Angles }
    etmin, etmax,                                       { Mini-Maxi to use for 2theta axis }
    xp, yp, zp:                   Dfloat;               { 3D Picture size }

    bstatus:                       Dbool;               { Status return word }

    bstlght0,
    bstlght1:              Dbool := true;

    light0_position,                                    { Position of Lights }
    light1_position,
    box_position:                Dpoint3;               { Plot Origine }

    lseg, iseg,                                         { Segments # for Boxe and Angles }
    aseg, gdim:                     Dint;

    untx, unty, untz,                                   { Axis Unit strings }
    name,
    angmsg:          string( 255 ) := '';               { Angle Message }

    raw:                          ^tbraw;               { Raw array to send points }


    procedure DISPLAY_SIZE;
    const
      eps = 0.5E-5;

    var
      p:                              pat_ptr;
      minth0, maxth0, minth1, maxth1:  Dfloat;
      np0, np1, npa, ndg:             integer;

    begin

      if g2thmin > 0.0 then etmin := g2thmin*(1.0 - eps)
                       else etmin :=   thmin;
      if g2thmax > 0.0 then etmax := g2thmax*(1.0 + eps)
                       else etmax :=   thmax;

      minth0 := thmax;
      maxth0 := thmin;
      npa := 0;
      ndg := 0;
      p := pat_first;
      while p <> nil do
      with p^ do
      begin
        np1 := 0;
        ndg := ndg + 1;
        for ij := 1 to dim do
          with dat[ij] do
            if (theta >= etmin) and (theta <= etmax) then
              if np1 = 0 then
              begin
                np1  :=  1;
                ista := ij;
                iend := ij;
                minth1 := dat[1].theta;
                maxth1 := minth1
              end
              else
              begin
                np1 := np1 + 1;
                iend := ij;
                if theta > maxth1 then maxth1 := theta
                                  else if theta < minth1 then minth1 := theta
              end;
        if npa = 0 then
        begin
          npa := 1; np0 := np1; minth0 := minth1; maxth0 := maxth1
        end
        else
          if (ABS( minth1 - minth0 ) > ABS( step )/4.0) or
             (ABS( maxth1 - maxth0 ) > ABS( step )/4.0) or
             (np1 <> np1) then
          begin
            WRITELN( ' *** PLOT_PATTERN error : Not consistent pattern list for a 3D plot. ***' );
            WRITELN( ' ':26, 'The 2theta limits of each pattern must be close to the pattern step/4 and' );
            WRITELN( ' ':26, 'the number of point must be equals.' );
            PASCAL_EXIT( 2 )
          end;
        p := next
      end;
      if np0 < 10 then
      begin
        WRITELN( ' *** PLOT_PATTERN error : Too small Pattern for a 3D plot (', np0:0, ' dots: less long than 10 dots) . ***' );
        if np0 < 2 then
        begin
          WRITELN( ' ':26, 'Stop because can plot it with less than 2point/pattern.' );
          PASCAL_EXIT( 2 )
        end
      end;
      NEW( raw, np0*3 );
      gdim := np0
    end DISPLAY_SIZE;


    procedure DISPLAY_SURFACE;
    { Procedure to display the 3D image }
    const
      mi2th =    1.0;                                   { Minimum 2*Theta Spacing }
      shz   =  0.001;                                   { Shift for Z lines }

    var
      x_axe, y_axe, z_axe, box, patsize: [static] Dint;
      bx_dir:             [static] array[1..5] of Dint;
      zsh, cmin, cmax, xx, yy, zz:              Dfloat;
      i, j, k, l, m:                           integer;
      p1, p2:                                  pat_ptr;
      ste:                                string( 16 );

      procedure DISPLAY_3D_0;
      { Display the surface as successive block in one surface }
      var
        p:       pat_ptr;
        sz:      integer;
        ver:     Dpoint3;
        i, nr:   integer;

      begin
        DRAW$FILL_ATTR( 2, 1 );                         { Set the Filling attribute mode }
        DRAW$OUT_MODE( 3 );                             { Set the Filling mode }
        p := pat_first;
        if p <> nil then DRAW$SURFACE( npat, gdim, 16 );
        nr := 0;
        while p <> nil do
        with p^ do
        begin
          i := 0;
          with raw^ do
          for ij := ista to iend do
          with dat[ij] do
          begin
            i := i + 1; tb[i] := theta;
            i := i + 1; if fc_celsius in funcs then tb[i] := t_sample - icetemp
                                               else tb[i] := t_sample;
            i := i + 1; tb[i] :=   int;
          end;
          nr := nr + 1;
          DRAW$SEND_BLOCK( raw^.tb );
          p := next
        end;
        DRAW$COLOR( 1 )
      end  DISPLAY_3D_0;


      procedure DISPLAY_3D_1;
      { Display the surface as successive triangle_strips }
      var
        p1, p2:  pat_ptr;
        sz:      integer;
        ta, tb:   Dfloat;

      begin
        DRAW$FILL_ATTR( 2, 1 );                         { Set the Filling mode }
        DRAW$OUT_MODE( -5 {GL_TRIANGLE_STRIP} );        { Set the Triangle Strip mode }
        p1 := pat_first;                                { Start from first pattern }
        p2 := p1^.next;                                 { Get the second pattern address }
        while p2 <> nil do
        begin { Loop on the patterns }
          if fc_celsius in funcs then
          begin
            ta := p1^.t_sample - icetemp;
            tb := p2^.t_sample - icetemp
          end
          else
          begin
            ta := p1^.t_sample;
            tb := p2^.t_sample
          end;

          { Output the first point }
          with p1^, dat[ista] do DRAW$PLOT3( theta, ta, int, false );
          for i := p1^.ista to p1^.iend - 1 do          { Loop on the pattern points }
          begin
           with p2^, dat[i] do DRAW$PLOT3( theta, tb, int, true );
           with p1^, dat[i + 1] do DRAW$PLOT3( theta, ta, int, true )
          end;
          with p2^, dat[iend] do DRAW$PLOT3( theta, tb, int, true );
          DRAW$PLOT3( 0.0, 0.0, 0.0, false );           { Finish the Triangle Strip block }
          p1 := p2;                                     { Skip to the next patterns }
          p2 := p1^.next
        end
      end DISPLAY_3D_1;


      procedure DISPLAY_3D_2;
      { Display the surface as network of line_strips }
      var
        p:          pat_ptr;
        ts, te:     real;
        bs:         boolean;
        sz, ii, st: integer;

      begin
        DRAW$COLOR( 0.9, 0.8, 0.6 );
(*      DRAW$COLOR( grid_color ); *)
        DRAW$FILL_ATTR( 1, 1 );                         { Set the Hollow mode }
        { Plot each Pattern }
        p := pat_first;                                 { Start from first pattern }
        with p^ do
          ts := (dat[dim].theta - dat[1].theta)/(dim - 1);      { Compute the 2*theta step }
        st := TRUNC( mi2th/ts );
        if st*ts < mi2th then st := st + 1;             { Get the 2*theta spacing }
        while p <> nil do
        with p^ do
        begin { Loop on the patterns }
          if fc_celsius in funcs then te := t_sample - icetemp
                                 else te := t_sample;
          with dat[ista] do
            DRAW$PLOT3( theta, te, int + zsh, false );
          for ij := ista + 1 to iend do
            with dat[ij] do
              DRAW$PLOT3( theta, te, int + zsh, true );
          p := next
        end;
        { Plot for each theta the variation with temperature }
        ii := 0;
        while ii < gdim do
        begin
          bs :=    false;
          p := pat_first;
          while p <> nil do
          with p^, dat[ista + ii] do
          begin
          if fc_celsius in funcs then te := t_sample - icetemp
                                 else te := t_sample;
            DRAW$PLOT3( theta, te, int + zsh, bs );
            bs := true;
            p := next
          end;
          ii := ii + st
        end;
        { Finish the Last curve }
        DRAW$PLOT3( 0.0, 0.0, 0.0, false )
      end DISPLAY_3D_2;


    begin { DISPLAY_SURFACE }
      if fc_light in funcs then                         { When the lighting mode is used }
      begin                                             { Define the Lights }
        lseg := DRAW$NEW_SEG( 2 );                      { Open/Create the light segement }
(*
        DRAW$LIGHT_ON_OFF( 1, 0 );                      { Eteint la lampe # 1 }
*)
        bstlght0 := true;
        bstlght1 := true;

        DRAW$LIGHT_ON_OFF( 0, 1 );                      { Allume la lampe # 0 }

        DRAW$LIGHT_DEFINE( 0, LIGHT_POSITION,       light0_position  );
        DRAW$LIGHT_DEFINE( 0, LIGHT_AMBIENT_COLOR,  light0_amb_color );
        DRAW$LIGHT_DEFINE( 0, LIGHT_DIFFUSE_COLOR,  light0_dif_color );
        DRAW$LIGHT_DEFINE( 0, LIGHT_SPECULAR_COLOR, light0_spe_color );

        DRAW$LIGHT_ON_OFF( 1, 1 );                      { Allume la lampe # 1 }

        DRAW$LIGHT_DEFINE( 1, LIGHT_POSITION,       light1_position );
        DRAW$LIGHT_DEFINE( 1, LIGHT_AMBIENT_COLOR,  Dcolor[ 0.1, 0.1, 0.1, 1.0] );
        DRAW$LIGHT_DEFINE( 1, LIGHT_DIFFUSE_COLOR,  Dcolor[ 0.7, 0.7, 0.4, 1.0] );
        DRAW$LIGHT_DEFINE( 1, LIGHT_SPECULAR_COLOR, Dcolor[ 0.8, 0.8, 0.5, 1.0] );


        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_SPECULAR_COLOR, Dcolor[1.0, 1.0, 1.0, 0.55] );
        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_SHININESS,      Dvalue[100.0] );
        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_EMISSION_COLOR, Dcolor[0.0, 0.0, 0.0, 1.0] );
        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_DIFFUSE_COLOR,  Dcolor[1.0, 0.9, 0.2, 1.0] );

        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE,  MAT_SPECULAR_COLOR, Dcolor[1.0, 1.0, 1.0, 0.55] );
        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE,  MAT_SHININESS,      Dvalue[100.0] );
        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE,  MAT_EMISSION_COLOR, Dcolor[0.0, 0.0, 0.0, 1.0] );
        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE,  MAT_DIFFUSE_COLOR,  Dcolor[1.0, 0.9, 0.2, 1.0] );

        DRAW$COLOR( 1.0, 1.0, 0.5, 1.0 );
        DRAW$SPHERE( light0_position[1], light0_position[2], light0_position[3], 2.0, 32, 32, 4 );

        DRAW$SEG_END
      end;

      aseg := DRAW$NEW_SEG( 3 );                        { Open/Create the Angle segment }

      ome :=  90.0; chi :=  80.0; phi := -100.0;
      { Force the initial value of angle and display them }
      WRITEV( angmsg, ' Ome = ', ome:8:2, ', Chi = ', chi:8:3, ', Phi = ', phi:8:3 );
      DRAW$COLOR( 0.0, 0.0, 0.0, 1.0 );
      DRAW$STRING( -22.0, -17.5, 0.0, 1.0, angmsg );
      DRAW$SEG_END;                                     { Close the Angle Segment }

      iseg := DRAW$NEW_SEG( 4 );                        { Open/Create the Box/Plot Segment }
      { Create the Orientation Matrix of the Box/Plot }

      DRAW$ROTATE_SEG3( iseg, 0.0, 0.0, 0.0, ome, chi, phi );
      DRAW$SEG_ORIENTATION( iseg, 0.0, 0.0, 0.0 );

      { The flag parameter (last one) must be flag the axis characteristics as this :

        the value must be a set of this following constantes :

        *****   Axis Flags   *****

        Axis_Log       =       1; (* Axis is in Log10 Coordinate *)
        Axis_Left      =       2; (* Ticks and displayed value at left position ( else at right ) *)
        Axis_VOther    =       4; (* Value are in other side (than ticks) *)
        Axis_NFrs      =       8; (* Does not display the first Ticks *)
        Axis_NFVl      =      16; (* Does not display the first Value *)
        Axis_NLsT      =      32; (* Does not display the last Ticks *)
        Axis_NLVl      =      64; (* Does not display the last Value *)
        Axis_NZer      =     128; (* Does not display the Zero Ticks *)
        Axis_NZVl      =     256; (* Does not display the Zero Value *)
        Axis_Arrow     =     512; (* Put an arrow at the end of axis *)
        Axis_ArroF     =    1024; (* Fill mode for Axis_Arrow *)
        Axis_Tilte     =    2048; (* Axis character are tilted with ticks *)
        Axis_VPath     =    4096; (* String Path (2 bits) for Ticks Value *)
        Axis_Upath     =   16384; (* String Path (2 bits) for Axis Unit *)
        Axis_SpLim     =   65536; (* Axis Limit auto-adaptation when set *)
        Axis_VLeft     =  131072; (* Left Alignement for Unit String *)
        Axis_VRight    =  262144; (* Right Alignement for Unit String *)
        Axis_ULeft     =  524288; (* Left Alignement for Unit String *)
        Axis_URight    = 1048576; (* Right Alignement for Unit String *)
        Axis_UMiddle   = 2097152; (* Set Unit String at the Middle of Axis *)


        *****   Axis Plot Directives and Flags for Draw_Plot   *****

        Axis_Plot_X    =     1; (* Plot an X axis *)
        Axis_Plot_Y    =     2; (* Plot an Y axis *)
        Axis_Plot_Z    =     3; (* Plot a  Z axis *)
        Axis_PFlg_TVl  =     4; (* Ticks values and Unit String will be Displayed *)
        Axis_PFlg_TPrm =     8; (* The ticks left and right sides are permutted *)
        Axis_PFlg_TSym =    16; (* The Ticks must be symmetrized *)
        Axis_PFlg_XEnd =    32; (* Plot Axis with a shift of x axis size *)
        Axis_PFlg_YEnd =    64; (* Plot Axis with a shift of Y axis size *)
        Axis_PFlg_ZEnd =   128; (* Plot Axis with a shift of Z axis size *)
        Axis_PFlg_XPos =   256; (* Plot Axis at specified X shift *)
        Axis_PFlg_YPos =   512; (* Plot Axis at specified Y shift *)bs := (
        Axis_PFlg_ZPos =  1024; (* Plot Axis at specified Z shift *)
        Axis_Pflg_SymU =  2048; (* Apply a mU (mirror u) to tick direction *)
        Axis_Pflg_SymV =  4096; (* Apply a mV (mirror v) to tick direction *)
        Axis_Pflg_SymW =  8192; (* Apply a mW (mirror w) to tick direction *)
        Axis_Pflg_Rotv = 16384; (* Apply an additional rotation of 180 to value and unit strings *)
 
      }

      cmin := tmin; cmax := tmax;
      if fc_celsius in funcs then
      begin
        cmin := cmin - icetemp;
        cmax := cmax - icetemp;
        ste:= 'C'
      end
      else ste := 'K';
      ste := ste||' (Temperature)';

      zsh := (gmax - gmin)*shz;

      x_axe := DRAW$NEW_AXIS( Dpoint3[ 1.0, 0.0, 0.0],
                              Dpoint3[ 0.0, 1.0, 1.0],
                              gaxxsz, etmin, etmax, -3,                                Axis_UMiddle );
      y_axe := DRAW$NEW_AXIS( Dpoint3[ 0.0, 1.0, 0.0],
                              Dpoint3[-1.0, 0.0, 1.0],
                              gaxysz, cmin,  cmax,  -3,  Axis_UMiddle + Axis_VPath*0 + Axis_UPath*0 );
      z_axe := DRAW$NEW_AXIS( Dpoint3[ 0.0, 0.0, 1.0],
                              Dpoint3[ 1.0, 1.0, 0.0],
                              gaxzsz,  gmin, gmax,  -3,  Axis_Arrow   + Axis_ArroF  +  Axis_UMiddle );


      Draw$Set_Axis_Unit( x_axe, ' (2*Theta)' );
      Draw$Set_Axis_Unit( y_axe, ste );
      Draw$Set_Axis_Unit( z_axe, 'Cps (Intensity)' );

      Draw$Set_Axis_Value( x_axe, 2, 112,,,,, 4,  5, 2 );       { Set Value Font, Field and decimal }
      Draw$Set_Axis_Value( y_axe, 2, 112,,,,, 4,  5, 2 );       { Set Angle, Value Font, Field and decimal }
      Draw$Set_Axis_Value( y_axe, 4,   2,, 0.0 );               { Set Angle, fbs := (or Unit String }
      Draw$Set_Axis_Value( z_axe, 2, 112,,,,, 4,  5, 2 );       { Set Value Font, Field and decimal }

      bx_dir[1] := Axis_PFlg_TVl  +                     { Tick Values, }
                   Axis_Plot_X;                         { Plot X axis }
      bx_dir[2] := Axis_PFlg_TVl  +                     { Tick Values, }
                   Axis_PFlg_XEnd + Axis_Plot_Y;        { at end'X axis, Plot Y axis }
      bx_dir[3] := Axis_PFlg_TVl  + Axis_Pflg_Rotv +    { Tick Values, 180 String rotate, }
                   Axis_PFlg_SymW + Axis_PFlg_TPrm +    { mw symetry Ope., Left/right Permut, }
                   Axis_Plot_Y;                         { Plot Y axis }
      bx_dir[4] := Axis_PFlg_TVl  + Axis_Pflg_Rotv +    { Tick Values, 180 String rotate, }
                   Axis_PFlg_SymW + Axis_PFlg_TPrm +    { mw symetry Ope., Left/right Permut, }
                   Axis_PFlg_YEnd + Axis_Plot_X;        { at end'Y axis, Plot X axis }

      bx_dir[5] := Axis_PFlg_TVl  + Axis_Plot_Z;        { Tick Values, Plot Y axis }

      DRAW$COLOR( 1 );

      { Create the Axis Box }
      box := DRAW$NEW_BOX( box_position, x_axe, y_axe, z_axe, bx_dir );

      DRAW$LINE_ATTR( 1, 2.0 );                         { Set continue line type }
      DRAW$PLOT_BOX( box );                             { Draw the Axis box }

      DRAW$COLOR( 1 );
      DRAW$OPEN_BOX( box );                             { Open the box to plot the curve }

      if fc_light in funcs then
      begin
        DRAW$COLOR( 0.55, 0.55, 0.70, 1.0 );
        DISPLAY_3D_0
      end
      else
      begin
        DRAW$COLOR( 0.3, 0.3, 0.7 );
(*      DRAW$COLOR( fill_color );    *)                 { Set the surface color }
        DISPLAY_3D_1                                    { Form the Triangle fillings }
      end;

      if not (fc_light in funcs) then DISPLAY_3D_2;     { Form a grid of line }

      DRAW$CLOSE_BOX;                                   { Close the Axis box }
      DRAW$SEG_END;                                     { Close the boxe/plot segment }
      DRAW$VIEW( dview );
    end DISPLAY_SURFACE;


    procedure ROTATE_PLOT;
    var
      pmat, mat: array[1..3,1..3] of Dfloat;

    begin
      DRAW$UPDATE_SEG( aseg {, 0} );

      WRITEV( angmsg, ' Ome = ', ome:8:2, ', Chi = ', chi:8:3, ', Phi = ', phi:8:3 );
      DRAW$STRING( -22.0, -17.5, 0.0, 1.0, angmsg );

      DRAW$SEG_END;

      DRAW$VIEW( dview );
      DRAW$SEG_UNIT_TRANSF( iseg );
      DRAW$ROTATE_SEG3( iseg,  0.0,  0.0,  0.0, ome, chi, phi )
    end ROTATE_PLOT;


  begin { MAKE_3D }
    DISPLAY_SIZE;                                       { Sizes the surface to plot }

    INIT_PLOT( true );

{   DRAW$LIGHT_ON_OFF( 0, 1 );                          { Allume la lampe # 0 }
{   DRAW$LIGHT_ON_OFF( 1, 1 );                          { Allume la lampe # 1 }

    { Set the axis sizes when not user specified }
    if (gaxxsz < 5.0) or (gaxxsz > 300.0) then gaxxsz := 40.0;
    if (gaxysz < 5.0) or (gaxysz > 300.0) then gaxysz := 40.0;
    if (gaxzsz < 5.0) or (gaxzsz > 300.0) then gaxzsz := 28.0;

    box_position[1]    :=  -0.5*gaxxsz;
    box_position[2]    :=  -0.5*gaxysz;
    box_position[3]    :=  -0.5*gaxzsz;

    light0_position[1] :=  0.35*gaxxsz;                 { Position of Lights }
    light0_position[2] :=  0.55*gaxysz;
    light0_position[3] :=  1.40*gaxzsz;

    light1_position[1] := -1.10*gaxxsz;
    light1_position[2] :=  0.10*gaxysz;
    light1_position[3] :=  1.00*gaxzsz;

    bstatus := DRAW$PICTURE3( inp_name, gaxxsz*2.0, gaxysz*2.0, gaxzsz*2.0, false, false );

    DISPLAY_SURFACE;

    repeat
      irep := DRAW$DIALOG( 1 );
      if irep < 0 then goto ET_Stop;
      case irep of
        1: { Euler Angle Omega Function }
          begin
            dstatus := DRAW$GET_VALUE( 'Eulerian Angle Omega', ome, -180.0, 180.0 );
            if dstatus <> 1 then ome :=  90.0;
            ROTATE_PLOT
          end;

        2: { Euler Angle chi Function }
          begin
            dstatus := DRAW$GET_VALUE( 'Eulerian Angle Chi', chi, -180.0, 180.0 );
            if dstatus <> 1 then chi :=  80.0;
            ROTATE_PLOT
          end;

        3: { Euler Angle Phi Function }
          begin
            dstatus := DRAW$GET_VALUE( 'Eulerian Angle Phi', phi, -180.0, 180.0 );
            if dstatus <> 1 then phi := -40.0;
            ROTATE_PLOT
          end;

        4: { View Distance }
          begin
            dstatus := DRAW$GET_VALUE( 'Distance of View', dview, 0.0, 200.0 );
            if dstatus <> 1 then dview := 0.0;
            ROTATE_PLOT
          end;

        5: { Light 0 On/off }
          begin  bstlght0 := not  bstlght0; DRAW$LIGHT_ON_OFF( 0, ORD( bstlght0 ) )  end;

        6: { Light 1 On/off }
          begin  bstlght1 := not  bstlght1; DRAW$LIGHT_ON_OFF( 1, ORD( bstlght1 ) )  end;

      otherwise
      end { case }
    until irep = 0;

  ET_Stop:
    { End of Graphic Session }
    DRAW$END;
    DISPOSE( raw )
  end MAKE_3D;



  procedure MAKE_2D( nd: integer );
  const
    max_nbc = 2;

  var

    bstatus:           boolean;
    i, j, k, irep, box:   Dint;
    cmin,  cmax,                                        { Minimum and Maximum for Intensities axis }
    etmin, etmax,                                       { Effective range of theta to display }
    shf,     dlt:       Dfloat;

    lseg, iseg,                                         { Segments # for Boxe and Angles }
    dstatus:              Dint;                         { Entry number for DRAW$DIALOG Return }




    procedure PLOT_A_PATTERN( p: pat_ptr; shift: real );
    var
      slab:           string( 32 );
      bs:                  boolean;
      xx, yy, zz, te:         real;
      ir, il:              integer;

    begin
      il :=     0;                                      { No default last point }
      bs := false;                                      { Looking for start point }
      with p^ do
        for ij := 1 to dim do                           { Loop on the pattern points }
          with dat[ij] do
            if (theta >= etmin) and (theta <= etmax) then
            begin                                       { When the point is in the theta range, ... }
              DRAW$PLOT( theta, int + shift, bs );      { ... we plot it ... }
              if bs then il :=   ij;                    { ... and il keep the last valid point (except for the first one). }
              bs := true                                { Set the pen down state }
            end;

      DRAW$RPLOT( 0.0, 0.0, false );                    { Finish the pattern plot. }

      if (fc_label in funcs) and (il > 0) then          { Add plot pattern label whene required }
      with p^, dat[il] do
      begin
        ir := DRAW$CONV_BOX( false, theta, int + shift, 0.0, xx, yy, zz  );
        te := t_sample;
        if fc_celsius in funcs then te := te - icetemp;
        WRITEV( slab, ' T = ', t_sample:6:2 );
        DRAW$STRING( xx + 1.0, yy + 0.5, 0.0, 0.4, slab )
      end
    end PLOT_A_PATTERN;


  begin { MAKE_2D }
    INIT_PLOT( false );
    if (gaxxsz < 5.0) or (gaxxsz > 300.0) then gaxxsz := 24.0;
    if (gaxysz < 5.0) or (gaxysz > 300.0) then gaxysz := 16.0;
    bstatus := DRAW$PICTURE( inp_name, gaxxsz + 4.0, gaxysz + 4.0, false, false );

    spcint := ABS( spcint );
    { Set the default value on incredible shift values }
    if (spcint > 2.0) or (spcint < 0.0) then spcint := 0.20;
    dlt := spcint*(smax - smin);
    cmin := smin; cmax := smax + dlt*(ndia - 1);
    if fc_celsius in funcs then
    begin  cmin := cmin - icetemp; cmax := cmax - icetemp  end;

    if g2thmin > 0.0 then etmin := g2thmin
                     else etmin :=   thmin;
    if g2thmax > 0.0 then etmax := g2thmax
                     else etmax :=   thmax;

    box := Draw$Easy_Box_2D( 2.0, 2.5, gaxxsz, gaxysz, etmin, etmax, cmin, cmax, ' 2*theta', 'Counts', 2 );
    DRAW$COLOR( 1 );
    DRAW$PLOT_BOX( box );                               { Plot the Axis Box }
    DRAW$OPEN_BOX( box );

    shf := 0.0;
    for id := 1 to ndia do
    begin
      DRAW$COLOR( tbcol[id] );
      if tbknd[id] then
      begin { Marker Mode }
        DRAW$OUT_MODE( 2 );
        DRAW$MARKER_ATTR( tbtyp[id], tbsiz[id] )
      end
      else
      begin
        DRAW$OUT_MODE( 1 );
        DRAW$LINE_ATTR( tbtyp[id], tbsiz[id] )
      end;
      PLOT_A_PATTERN( tbpat[id], shf );
      shf := shf + dlt
    end;

    repeat
      irep := DRAW$DIALOG( 1 );
      if irep < 0 then goto ET_Stop;
    until irep = 0;

  ET_Stop:

    { End of Graphic Session }
    DRAW$END
  end MAKE_2D;



begin { Min - READ_PATTERN }
  { Initialize the program and get the specified files and options }
  GET_PARAMETERS;

  { Get All Diffraction Patterns }
  GET_PATTERNS;

  { Do a 3D Plot when required }
  if (fc_3dplot in funcs) and (npat > 1) then MAKE_3D;

  { Do a 2D Plot when required }
  if [fc_ndia,fc_plot]*funcs = [fc_ndia,fc_plot] and (ndia > 0) then MAKE_2D( ndia );

  if fc_dir in funcs then OUT_CATALOGUE( ndia <> 0  );

  { Create teh Output when required }
  if fc_write in funcs then WRITE_NORMALIZED_FILE;

  WRITELN( ' Normal End.' )
end PLOT_PATTERN.
 
