/***** DRAW ROUTINES of Level L1 (Control and Basic Functions) *******/
/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*          F L T K / D R A W   -   S E R V E R   M O D U L E            *
*                                                                       *
*           Version  1.4-C for Draw Library Level V 2.4 A               *
*                                                                       *
*      (Draw interface for FLTK/OPENGL function and procedures)         *
*                                                                       *
*                                                                       *
*               (Basic Draw functions and procedures)                   *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/

/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/



/* Force allocation of all Global variables with this module. */
#define _DEF_GLOBAL

/* Get All Global server definitions. */
#include "Server_GL.h"



 /*************************************************************************
 *                                                                        *
 *                        DRAW Local Variables                            *
 *                                                                        *
 *************************************************************************/





 /*************************************************************************
 *                                                                        *
 *                        DRAW  Services  Routines                        *
 *                                                                        *
 *************************************************************************/


void Draw_Error( const char* msg, const char* id )
{
  char smsg[256];
  static char hdm[] = " * DRAW_SERVER ERROR : ";
  int    i = 0,
         j = 0;

  while (hdm[i] != 0) { smsg[i] = hdm[i]; i++; }
  if (msg) while (msg[j] != 0) { smsg[i] = msg[j]; i++; j++; }
  smsg[i++] = '\n';
  smsg[i] = 0;
  if (!fmsg) fmsg = fopen( "DRAW_ERROR.TXT", "a" );
  fprintf( fmsg, smsg, id );
  fflush( fmsg );
}





 /*************************************************************************
 *                                                                        *
 *                    DRAW L1 Init and Screen  Routines                   *
 *                                                                        *
 *************************************************************************/

void Sdrw_Pre_Init()
{ /* Initialize the Draw/GL Interface */
  int i;

  dr_rx = 600; dr_ry = 440;             /* Set default resolution (OpenGL Window Size in pixel). */

  fmsg = fopen( "DRAW_ERROR.TXT", "w" );

  /* Init Color Tables */
  for (i=0;i<15;i++)
  { draw_srccoltb[i][0] = draw_defcoltb[i][0];
    draw_srccoltb[i][1] = draw_defcoltb[i][1];
    draw_srccoltb[i][2] = draw_defcoltb[i][2];
    draw_srccoltb[i][3] = draw_defcoltb[i][3];
  }

  /* Init The segment and matrix caches to empty */
  for (i=0;i<SEG_CACHE_SIZE;i++) {
    draw_sgtb[i].ref_ref  = NULL;
    draw_sgtb[i].ref_idnt =    0;
  }
  for (i=0;i<TRF_CACHE_SIZE;i++) {
    draw_trtb[i].ref_ref  = NULL;
    draw_trtb[i].ref_idnt =    0;
  }
  for (i=0;i<16;i++)
    Drwgl_mat_unit[i] = (i%5==0)?1.0:0.0;

  /* Set a clean state for predefined font tables */
  for (i=0;i<SIZE_DEF_FONT;i++) {
    draw_fontnamtb[i] = 0; /* No font file-name */
    draw_fontstyle[i] = 0; /* Set an undefined font style */
  }
  draw_fontpath = 0; /* and no Defined font file path */
}



/**************************************************************************/

static void Get_pwd()
{
  int             i, n;
#if !defined( _WIN32) || defined( __CYGWIN__ )
  char           * pwd;
#endif

  Drwgl_curpat[0] = 0;
#if !defined( _WIN32) || defined( __CYGWIN__ )
  /* For Unix like system we get directly the PWD */
  pwd = getenv( "PWD" );
  if (pwd&&pwd[0]) { strcpy( Drwgl_curpat, pwd ); n = strlen( pwd ); }
              else n = 0;
#else
  /* For the windows system we get the PWD and adapt the streing to windows coding */
  /* When the User task is a Cygwin program, the PWD must be modified (/C/xxx -> C:/xxx) */
  n = GetEnvironmentVariable( "PWD", Drwgl_curpat, DRWSTRBUF_LENGTH-1 );
  if (n>0) {
    if (Drwgl_curpat[0] == '/') {
      i = 1;
      while ((i<n)&&(Drwgl_curpat[i] != '/')) { Drwgl_curpat[i-1] = Drwgl_curpat[i]; i++; }
      Drwgl_curpat[i-1] = ':';
    }
  } else n = 0;
#endif
  /* Append a directory separator at the end of path when not present */
  if (n>0)
    if ((Drwgl_curpat[n-1] != '/')&&(Drwgl_curpat[n-1] != '\\')) Drwgl_curpat[n++] = '/';
}



void Sdrw_Init( void )
{
  if (d_id&&(dr_rx > 0)&&(dr_ry > 0))
  { /* The Screen Work Station is Defined. */
    draw_noscalech     =     2;         /* Lock the Scale change during a window size change (init mode) */
    fprintf( fmsg, " Screen Display Setup:\n" );
    fprintf( fmsg, " Resolution = %d * %d\n", dr_rx, dr_ry );

    if (d_width > 0.0)
    { /* A metric is defined for the Screen Graphic Work Station. */
      d_unit = centimeter;
      if (d_height <= 0.0) d_height = (((float)dr_ry)/((float)dr_rx))*d_width;
    } else {
      d_unit   = raster;
      d_width  = (float) dr_rx;
      d_height = (float) dr_ry;
    }
    if ((ds_rx >= d_width )||(ds_rx <= 0.0)) ds_rx = 0.8*d_width;
    if ((ds_ry >= d_height)||(ds_ry <= 0.0)) ds_ry = 0.8*d_height;

    ds_rx = floor( ds_rx );                     /* Round Down W. Size */
    ds_ry = floor( ds_ry );
    d_xscale = dr_rx/d_width;                   /* Scales for X and Y in Pixel/cm. */
    d_yscale = dr_ry/d_height;
    dt_rx = (int) (ds_rx*d_xscale + 0.5);       /* Window Size in Pixel (integer) */
    dt_ry = (int) (ds_ry*d_yscale + 0.5);
    dp_rx = (int) (d_borders_x*d_xscale + 0.5); /* Window origine(position on screen) in Pixel */
    dp_ry = dr_ry - dt_ry - ((int) (d_borders_y*d_yscale - 0.5));     /* Screen Pixel org at left top */
    fprintf( fmsg, "  Screen Size (cm) = %f * %f\n", d_width, d_height );
    fprintf( fmsg, "  Window Size (cm) = %f * %f\n", ds_rx, ds_ry );
    fprintf( fmsg, "  Window Size (pixels) = %d * %d\n", dt_rx, dt_ry );
    fprintf( fmsg, "  => %f * %f pixels/cm.\n", d_xscale, d_yscale );
  }
  else
  { /* No defined Display => Set a default Screen Display */
    draw_noscalech = 0;                 /* The initial Scales are set here */
    d_id  = 1;                          /* Force a Defaulted Screen */
    dr_rx = 640; dr_ry = 480;           /* Select a default resolution */
    ds_rx = (float) dr_rx;
    ds_ry = (float) dr_ry;

    d_height = sqrt( ds_rx*ds_rx + ds_ry*ds_ry );               /* Diagonal size in pixels */
    d_xscale = d_height/(14.0*2.54);    /* Scale for a 14" Screen */
    d_yscale = d_xscale;
    d_width  = dr_rx/d_xscale;
    d_height = dr_ry/d_yscale;
    ds_rx = ceil( 0.8*d_width );
    ds_ry = ceil( 0.8*d_height );
    dt_rx = (int) (ds_rx*d_xscale);
    dt_ry = (int) (ds_ry*d_yscale);
    dp_rx = (int) (d_borders_x*d_xscale + 0.5);
    dp_ry =  dr_ry - dt_ry - ((int) (d_borders_y*d_yscale - 0.5)); /* GL Pixel org at left top */
    fprintf( fmsg, "\n*** Data set with the assumption of 14\" diagonal screen with the 640*480 resolution. ***\n\n" );
    fprintf( fmsg, "  Defaulted Screen Size (cm?) = %f * %f\n", d_width, d_height );
    fprintf( fmsg, "  Defaulted Window Size (cm?) = %f * %f\n", ds_rx, ds_ry );
    fprintf( fmsg, "  Defaulted Window Size (pixels) = %d * %d\n", dt_rx, dt_ry );
    fprintf( fmsg, "  => %f * %f pixels/cm?.\n", d_xscale, d_yscale );
    d_unit  = centimeter;
  }

  s_dt_rx  =  dt_rx; s_dt_ry  =  dt_ry; /* Save the Window sizes (pixels) */

  dw_ixmin = dw_sxmin = dw_cxmin = - (
                dw_ixmax = dw_sxmax = dw_cxmax = ds_rx*0.5);
  dw_iymin = dw_symin = dw_cymin = - (
                dw_iymax = dw_symax = dw_cymax = ds_ry*0.5);

//fprintf( fmsg, "  dw_cmin = (%f,%f), dw_cmax = (%f,%f).\n",
//              dw_cxmin, dw_cymin, dw_cxmax, dw_cymax );

  emarginx    =  1.0;                   /* Set the Default Picture Margins (cm) or (cm?) */
  emarginy    =  1.0;

  Draw_pic_sx = 10.0; /* Init the Picture X,Y Picture Size */
  Draw_pic_sy = 10.0;
  Draw_pic_yx =  1.0; /* Init the Picture Y/X ratio */

  dw_swsx = dw_cwsx = 10.0;
  dw_swsy = dw_cwsy = 10.0;

  /* Initialize the Drawing Structure to Empty state */

  d_mult_linesize      =   1.0;         /* Init the Line thickness multiplicator */

  draw_root            =  NULL;
  draw_fsegm           =  NULL;
  draw_lsegm           =  NULL;
  draw_fsysseg         =  NULL;
  draw_lsysseg         =  NULL;
  draw_curseg          =  NULL;
  draw_seg_count       =     0;
  draw_ftrmat          =  NULL;
  draw_ltrmat          =  NULL;
  draw_trf_count       =     0;

  draw_mat_cnt         =     0;

  draw_lock_flag       =   - 1;         /* Init the lock flag */

  draw_curr_detf       =     1;         /* Init the Pick_Id bit to detectable */
  draw_curr_pid        =     3;         /* Init The Pick_Identifier to 1 and detectable */

  Draw_Dist            =   0.0;         /* Set Without Perspective */
  Draw_ViewMat         =  NULL;         /* Initialize the view Matrix pointer */

  Draw_Qfirst          =  NULL;         /* Init the Init Request Queue. */
  Draw_Qlast           =  NULL;

  g_flags = 1;

  DrwGL_Font_Support_Init();            /* Init the Font Support */

  Get_pwd();                            /* Get the current pwd (in the current O.S. environment) */

  fflush( fmsg );
}




/**************************************************************************/


void Draw_End()
{
  if (g_flags)
  {
    g_flags = 0;
  }
}



/**************************************************************************/


int Draw_Init( float *px, float *py, Char *unit )
{ /* Init the  GL - DRAW_SERVER.
     In Draw Server, The Insymbol step is performed
     during the Task init step.
  */

  *px      =     ds_rx;                 /* Transmit the Setup Window sizes and unit */
  *py      =     ds_ry;
  *unit    =    d_unit;
  draw_lock_flag       =     0;         /* Set the lock flag in the inited State */
  draw_3D_init         =     0;         /* Set Default to 2D Init */
  draw_3D_flg          =     0;         /* Set Default to 2D plot */
  draw_3D_fill_flg     =     0;         /* Init the 3D Fill flag */

//fprintf( fmsg, " Draw_Init -> %f * %f unit: %d .\n", ds_rx, ds_ry, d_unit );
//fflush( fmsg );

  return 1;
}




//**********************************************************************

void Draw_Ini_Sys_Seg()
/**********************************************************************
 *                                                                    *
 *              SUBROUTINE TO DRAW THE PICTURE FRAMES                 *
 *                                                                    *
 *                                                                    *
 *********************************************************************/
{
  int     slen;
  float y1, y2;

  /*******************  Plot the Sheet Contour ***********************/

  Draw_New_Sys_Seg( 1, 0 );
  Draw_Gen_Lattr( 3, 1.0 );                     /* Small point Line, Size = 1 */
  Draw_GPlot_Rect( dw_ixmin, dw_iymin, dw_ixmax, dw_iymax );
  draw_curseg = NULL;

  /******************  Plot the Picture Contour **********************/

  Draw_New_Sys_Seg( 2, 0 );
//Show_Matf( "Sys seg #2", draw_curseg->seg.seg_mat );
  Draw_Gen_Lattr( 1, 2.0 );                     /* Line, Size = 2 . */
  Draw_GPlot_Rect( draw_cxmin, draw_cymin, draw_cxmax, draw_cymax );
fprintf( fmsg, " Call select Font\n" ); fflush( fmsg );
  Draw_Gen_SelectFont( 0 );                           /* Set font # 0 */
fprintf( fmsg, " Select Font OK\n" ); fflush( fmsg );

  if (draw_pic_emarg)
  { /* When a Margin is Required. */
    Draw_GPlot_Rect( draw_uxmin, draw_uymin, draw_uxmax, draw_uymax );
    slen = strlen( draw_pic_title );
    if (slen > 0)
    { /*  Plot the Title  */
      Draw_New_Sys_Seg( 3, 0 );
//    Show_Matf( "Sys seg #3", draw_curseg->seg.seg_mat );
      y1 = 0.50*(draw_cymax + draw_uymax);
      y2 = 0.65*(draw_cymax - draw_uymax);
      /* Set the Mode Center+Middle_of_line+Normal_Path. */
      Draw_Gen_Tattr( 3, 4, 0, 1.0, 0.0, 0.0 );
      Draw_Gen_Text( 0.0, y1, 0.0, y2, slen, draw_pic_title );
      Draw_Gen_Tattr( 1, 1, 0, 1.0, 0.0, 0.0 );
    }
  }

  Draw_Gen_Lattr( 1, 1.0 );
  draw_curseg = NULL;
}






/**************************************************************************/

int Draw_Picture( char *title, float px, float py, float pz,
                  Char bscale, Char bmarg )
/**********************************************************************
 *                                                                    *
 *      SUBROUTINE TO SET A PICTURE PLOT AREA.                        *
 *                                                                    *
 *  Init a picture in DRAW/V system.                                  *
 *                                                                    *
 *    - px, py (and pz) are the required Draw Canvas size in cm.      *
 *                                                                    *
 *    - title is the title of the picture.                            *
 *                                                                    *
 *    - bscale is true to get a strict scale when possible.           *
 *       (Possible with metric work station only).                    *
 *                                                                    *
 *    - bmarg is true if the user want keep a margin around his       *
 *      picture (where the title is located).                         *
 *                                                                    *
 *  The returned value time bscale is true when the effective scale   *
 *  resulting scale of picture is 1.0 and false otherwise.            *
 *                                                                    *
 *  N.B. :    DRAW_PICTURE must be called after DRAW_INIT.            *
 *                                                                    *
 *                                                                    *
 **********************************************************************/
{
  int Flg2D;

//fprintf( fmsg, "  Draw Picture with %f %f %d %d .\n", px, py, bscale, bmarg );
//fflush( fmsg );

  /************ Get picture area in world unit ********************/
  Draw_pic_xx = floor( fabs(px) + 0.5 );        /* Round the Picture Sizes to nearest cm. */
  Draw_pic_yy = floor( fabs(py) + 0.5 );
  Draw_pic_zz = floor( fabs(pz) + 0.5 );        /* And Picture Deep */

  draw_3D_init         =     0;                 /* Assume 2D Init until shown otherwise */
  draw_lock_flag       =     0;                 /* Set the lock flag in the free state */
  b_grid               =     0;                 /* Init the Grid Flag. */

  /*************  Compute the User Display Size  ******************/

  Draw_pic_sx = Draw_pic_xx;
  Draw_pic_sy = Draw_pic_yy;

  if (bmarg) {                                  /* Allocate a user margin. */
    if (bscale&1) {                             /* User dimension in Cm. */
      emarginx         =           1.0;
      emarginy         =           1.0;
    } else {                                    /* User dimension in Arbitrary unit. */
      emarginx     =   0.1*Draw_pic_xx;
      emarginy     =   0.1*Draw_pic_yy;
    }
    Draw_pic_sx        +=     emarginx;         /* Increase the Used size by the margin. */
    Draw_pic_sy        +=     emarginy;
  }
  draw_pic_emarg       =         bmarg;
  draw_pic_title       =         title;

  /***********  Compute the Scale of Picture to use all the Graphical Space  ************/

  Draw_pic_yx  =       Draw_pic_sy/Draw_pic_sx;

  if (Draw_pic_yx >= ds_ry/ds_rx) ixy_scale = ds_ry/Draw_pic_sy;/* Adjust it with the Required size Ratio. */
                             else ixy_scale = ds_rx/Draw_pic_sx;

  /* Here xy_scale is the maximum possible scale in Cm_Station/Cm_User. */
  if (bscale)                                                   /* when the unit scale is required, ... */
  { /* A Scale of 1.0 is required by the user */
    if ((ixy_scale < 1.0)||(ixy_scale > 5.0)) bscale     =   0; /* When it is not ok/possible, we signal it */
                                         else ixy_scale  = 1.0; /* when it is possible, we force scale to 1.0 */
  }


//fprintf( fmsg, "  Picture Scale = %f (Output Cm)/(User Cm),\n", sxy_scale );
//fprintf( fmsg, "  Picture Sizes = (%f * %f), ryx = %f.\n",
//               Draw_pic_sx, Draw_pic_sy, Draw_pic_yx );
//fflush( fmsg );


  /***********  Compute the Useable Display Size  *****************/

  /* Set the Screen View xy_scale Scale */
  cxy_scale   =  zxy_scale   =  sxy_scale   =   ixy_scale;      /* Set Current scale values (for zoom reload). */

  efcxy_scale =  efzxy_scale =  efsxy_scale = efixy_scale = 1.0/ixy_scale;      /* Scale in user_cm/station_cm to optimize. */

  /* Set the Window Sizes, positions and minimaxi in User cm - The origine is at the picture center. */
  dw_cwsx = dw_zwsx = dw_swsx = dw_iwsx = 0.5*ds_rx*efixy_scale;
  dw_cwsy = dw_zwsy = dw_swsy = dw_iwsy = 0.5*ds_ry*efixy_scale;
  dw_cwpx = dw_zwpx = dw_swpx = dw_iwpx = 0.0;
  dw_cwpy = dw_zwpy = dw_swpy = dw_iwpy = 0.0;

  dw_cxmin = dw_zxmin= dw_sxmin = dw_ixmin = - (dw_cxmax = dw_zxmax = dw_sxmax = dw_ixmax = dw_iwsx);
  dw_cymin = dw_zymin= dw_symin = dw_iymin = - (dw_cymax = dw_zymax = dw_symax = dw_iymax = dw_iwsy);

  fprintf( fmsg, "  Screen Work Space Minimaxi in User Cm [%f,%f,%f,%f]\n",
                    dw_ixmin, dw_ixmax, dw_iymin, dw_iymax );
  fflush( fmsg );


  /*******   Compute the User World Frame Limits (User Cm)   ********/

  draw_iwxmi = draw_swxmi = draw_zwxmi = draw_cxmin = - (
                       draw_iwxma = draw_swxma = draw_zwxma = draw_cxmax = 0.5*Draw_pic_sx);
  draw_iwymi = draw_swymi = draw_zwymi = draw_cymin = - (
                       draw_iwyma = draw_swyma = draw_zwyma = draw_cymax = 0.5*Draw_pic_sy);


  /********   Compute the User World Limits (for Clipping)   ********/

  if (Draw_pic_zz <= 0.05*(Draw_pic_xx + Draw_pic_yy))
  { /* 2D Mode */
    Draw_pic_zz = (Draw_pic_xx + Draw_pic_yy)*0.5;
    Flg2D = 1;
  }
  else Flg2D = 0;

  draw_suxmi = draw_uxmin = - (draw_suxma = draw_uxmax = 0.5*Draw_pic_xx);
  draw_suymi = draw_uymin = - (draw_suyma = draw_uymax = 0.5*Draw_pic_yy);
  draw_suzmi = draw_uzmin = - (draw_suzma = draw_uzmax = 0.5*Draw_pic_zz);



  /*****************  Initialize the Plot System  ********************/

  Draw_Draw_Clear();                            /* Init The Draw Plot System */
  Draw_GPlots();                                /* Init the virtual plotter */

  Draw_Ini_Sys_Seg();                           /* Plot all Picture Frames */


  /********** Initialyse the Plotting System **************************/

  if (Flg2D)
  { /* Set The Standard Origine at Lower Left Corner. */
    org_orgx = draw_uxmin;
    org_orgy = draw_uymin;
  } else {
    org_orgx =        0.0;
    org_orgy =        0.0;
    draw_3D_init   =    1;
  }
  orgx = org_orgx;
  orgy = org_orgy;
  orgz = org_orgz;

  draw_out_mode = 1;
  draw_curr_pid = 1;


  /*************** Set device coordinate system window ***************/

  draw_lock_flag       =     1;                 /* Set the lock flag in the Active State */
  return bscale;
}




/**************************************************************************/
