// ****************************************************************************
// *                                                                          *
// *                                                                          *
// *                                                                          *
// *            F L T K / D R A W   -   S E R V E R   M O D U L E             *
// *                                                                          *
// *             Version  1.4-C for Draw Library Level V 2.4 A                *
// *                                                                          *
// *        (Draw interface for FLTK/OPENGL function and procedures)          *
// *                                                                          *
// *                                                                          *
// *               (Kernel to manage the bidirectional pipe)                  *
// *                                                                          *
// *                                 by                                       *
// *                                                                          *
// *           Pierre Wolfers, Laboratoire de Cristallographie                *
// *                                                                          *
// *           CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166                *
// *                                                                          *
// *                      F 38042 GRENOBLE CEDEX 9                            *
// *                                                                          *
// *                             F R A N C E                                  *
// *                                                                          *
// *                                                                          *
// *                                                                          *
// ****************************************************************************

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////




/***********************************************************************/
/**                                                                   **/
/**           Echange DRAW_LIb <-> DRAW_SERVER Constantes             **/
/**                                                                   **/
/***********************************************************************/




#include <stdio.h>
#include <errno.h>
#include <math.h>
#include <signal.h>

#include <draw/draw_constantes.h>

#include "Server_Env.h"





/*********************************************************************/
/**                                                                 **/
/**      Echange DRAW_LIb <-> DRAW_SERVER Types Definitions         **/
/**                                                                 **/
/*********************************************************************/



typedef unsigned char Char;                     // Define the Unsigned char type Char.

typedef union {                                 // Define the Type Equiv. rec. type.
                  struct { Char ch0, ch1, ch2, ch3; } ct;
                  int   i;
                  float f;
              } eqty;



/*********************************************************************/
/**                                                                 **/
/**          Exchange DRAW_SERVER <-> DRAW_LIB Variables            **/
/**                                                                 **/
/*********************************************************************/


static int    Sdrw_Insize,                      // Number of byte readden.
              Sdrw_Outsize,                     // Number of byte to write.
              Sdrw_Inread;                      // Index in the Read Pipe Buffer.

static Char   Sdrw_Inp_Buf[BUF_SIZE],           // Buffer for Input from DRAW-LIB.
              Sdrw_Out_Buf[BUF_SIZE];           // Buffer for output to DRAW-LIB.

int           Sdrw_Client_id,                   // Draw Client Process id (must be extern).
              Sdrw_HalfDuplex;                  // Half Duplex Block Flag (must be extern).

static  eqty  equ;                              // Record for float<->int<->Char Conv.



int  Sdrw_Read_Blk(  Char*, int );

void Sdrw_Write_Blk( Char*, int );



/*********************************************************************/
/**                                                                 **/
/**   Data Echange Routines for DRAW_LIB <-> DRAW_SERVER Dialogue   **/
/**                                                                 **/
/*********************************************************************/


int Sdrw_Read()
{ /* Read a buffer from the DRAW Library */

  Sdrw_Insize = Sdrw_Read_Blk( Sdrw_Inp_Buf, BUF_SIZE );
  Sdrw_Inread = 0;
  sdrw_read_left = Sdrw_Insize;
  return Sdrw_Insize;
}



/************************************************************************/

void Sdrw_Write()
{ /* Write a buffer to the DRAW Library */
  Sdrw_Write_Blk( Sdrw_Out_Buf, Sdrw_Outsize );
  Sdrw_Outsize = 0;
}



/************************************************************************/


int  Sdrw_Read_Block( Char* tb, int dim )
{ /* Read a User buffer from the DRAW Library */
  int n, t;

  if (!Sdrw_HalfDuplex) { Sdrw_Put_Code( 0 ); Sdrw_Write(); }  /* Write synchro to start block transaction */

  /* The User has begun to send the Data Block (stored at tb, dim bytes length) */
  t = 0;
  while (t < dim) {
    if (Sdrw_HalfDuplex) { Sdrw_Put_Code( 0 ); Sdrw_Write(); } /* Write synchro for each Data Packet */
    n = Sdrw_Read_Blk( tb, dim - t );                          /* Read Directly in the data array */
    if (n < 0)
    { fprintf( fmsg, " SDRW Block: D Read from Client failed.\n" );
      fflush( fmsg );
      exit( 2 );
    }
    t  += n;                                                   /* Update the bytes counts */
    tb += n;
  }
  return t;
}



/************************************************************************/

void Sdrw_Write_Block( Char* tb, int dim )
{ /* Write a User buffer to the DRAW Library */
  int n;

  if (!Sdrw_HalfDuplex) Sdrw_Read();    /* Read for synchro, acknowledge before to send all the data */

  /* The User must be ready to get the Data Block (address tb, dim bytes length) */
  while (dim) {
    n = (dim > BUF_SIZE)?BUF_SIZE:dim;  /* Determine the size of the data packet */
    if (Sdrw_HalfDuplex) Sdrw_Read();   /* Read for synchro, acknowledge before to send each data packet */
    Sdrw_Write_Blk( tb, n );            /* Send the data packet directly */
    dim -= n;                           /* Update the data byte count */
    tb  += n;
  }
}



/************************************************************************/

int Sdrw_Dialog_Request()
{
  Sdrw_Write();
  return Sdrw_Read();
}



/************************************************************************/
/**          Input from and Output to the Graphic Server Routines      **/
/************************************************************************/

void Sdrw_Put_Char( Char ch )
{
  if (Sdrw_Outsize < BUF_SIZE)
    Sdrw_Out_Buf[Sdrw_Outsize++] = ch;
}



/*********************************************************************/

void Sdrw_Put_Int( int iv )
{
  equ.i = iv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
}



/*********************************************************************/

void Sdrw_Put_Int_Direct( int pos, int iv )
{
  int Save_Outsize;

  Save_Outsize = Sdrw_Outsize;
  Sdrw_Outsize = pos;

  equ.i = iv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
  Sdrw_Outsize = Save_Outsize;
}



/*********************************************************************/

void Sdrw_Put_Float( float fv )
{
  equ.f = fv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
}



/*********************************************************************/

void Sdrw_Put_String( const char * s, int l )
{
  int i;

  if (l == 0) l = strlen( s );
  Sdrw_Put_Char( l );
  for ( i=0; i<l; i++)
  {
    Sdrw_Put_Char( s[i] );
  }
}



/*********************************************************************/

void Sdrw_Put_Code( int code )
{
  Sdrw_Outsize = 0;
  Sdrw_Put_Int( code );
}



/***************************************************************************/

Char* Sdrw_Get_Buffer()
{
  Char* p;
  int   i;

  p = (Char*) malloc( Sdrw_Insize );
  for (i=0;i<Sdrw_Insize;i++) p[i] = Sdrw_Inp_Buf[i];
  return p;
}



/***************************************************************************/

Char Sdrw_Get_Char()
{
  char ch;

  if (Sdrw_Inread < Sdrw_Insize) {
    ch = Sdrw_Inp_Buf[Sdrw_Inread++];
    sdrw_read_left = Sdrw_Insize - Sdrw_Inread;
  }
  else
    ch = 0;
  return ch;
}



/*********************************************************************/

int Sdrw_Get_Int()
{
  equ.ct.ch0 = Sdrw_Get_Char();
  equ.ct.ch1 = Sdrw_Get_Char();
  equ.ct.ch2 = Sdrw_Get_Char();
  equ.ct.ch3 = Sdrw_Get_Char();
  return equ.i;
}



/*********************************************************************/

float Sdrw_Get_Float()
{
  equ.ct.ch0 = Sdrw_Get_Char();
  equ.ct.ch1 = Sdrw_Get_Char();
  equ.ct.ch2 = Sdrw_Get_Char();
  equ.ct.ch3 = Sdrw_Get_Char();
  return equ.f;
}



/*********************************************************************/

int Sdrw_Get_String( char * s, int sz )
{
  int i, l, n;

  n = l = Sdrw_Get_Char();
  if (n >= sz) n = sz - 1;
  for (i=0; i<n; i++) s[i] = Sdrw_Get_Char();
  while (n<l) { s[i-1] = Sdrw_Get_Char(); n++; }
  s[i] = 0;                /* Append a SOH for C string handling */
  return l;
}


/*********************************************************************/

char * Sdrw_Get_String_Tmp()
{
  int i, l;
  char * s;

  l = Sdrw_Get_Char();
  s = (char *) malloc( l + 1 );

  for (i=0; i<l; i++)
    s[i] = Sdrw_Get_Char();
  s[i] = 0;                /* Append a SOH for C string handling */
  return s;
}


/*********************************************************************/

void Sdrw_String_Free_Tmp( char * s )
{
  if (s != NULL)
  {
    free( s );
    s = NULL;
  }
}


/*********************************************************************/

int Sdrw_Get_Code()
{
  Sdrw_Inread = 0;
  return Sdrw_Get_Int();
}


/*********************************************************************/

Char * Sdrw_Put_Address( int n )
{ Char * p;
  p = Sdrw_Out_Buf + Sdrw_Outsize;
  Sdrw_Outsize += n;
  return p;
}



/*********************************************************************/

Char * Sdrw_Get_Address( int n )
{ Char * p;
  p = Sdrw_Inp_Buf + Sdrw_Inread;
  Sdrw_Inread += n;
  return p;
}


/************************************************************************/

void Sdrw_Anim_acq( int isgn )
{

  Sdrw_Put_Char( isgn&255 );
//fprintf( fmsg, " Anim_acq %d %d %d\n", isgn, isgn&140, (isgn/16)&3 );
//fflush( fmsg );
#if !defined( _WIN32 ) || defined ( _CYGWIN_ )
  int sig;
  if (isgn&0140) {
    switch ( (isgn/16)&3 ) {
      case 2: { sig = SIGUSR2; break; }
      case 3: { sig = SIGABRT; break; }
      default:  sig = SIGUSR1;
    }
    kill( Sdrw_Client_id, sig );
  }
#endif
}



/*************************     E N D    ******************************/
