// ****************************************************************************
// *                                                                          *
// *                                                                          *
// *                                                                          *
// *           F L T K  /  D R A W   -   S E R V E R   M O D U L E            *
// *                                                                          *
// *             Version  1.4-C for Draw Library Level V 2.4 A                *
// *                                                                          *
// *           (Draw interface for FLTK function and procedures)              *
// *                                                                          *
// *                                                                          *
// *    D R A W   -   S E R V E R   C O N N E C T I O N   M A N A G E R       *
// *                                                                          *
// *          (Kernel to Open/close the bidirectional IO Channel)             *
// *                                                                          *
// *                                                                          *
// *           (Draw interface for FLTK function and procedures)              *
// *                                                                          *
// *                                 by                                       *
// *                                                                          *
// *           Pierre Wolfers, Laboratoire de Cristallographie                *
// *                                                                          *
// *           CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166                *
// *                                                                          *
// *                      F 38042 GRENOBLE CEDEX 9                            *
// *                                                                          *
// *                             F R A N C E                                  *
// *                                                                          *
// *                                                                          *
// *                                                                          *
// ****************************************************************************

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


 /******************************************************************
 *                                                                 *
 *       Socket routines for Unix and for Windows environment.     *
 *                                                                 *
 ******************************************************************/

#include <stdio.h>
#include <errno.h>
#include <fltk/Threads.h>
#include <fltk/run.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>


#include <draw/draw_constantes.h>
#include "Server_Env.h"


typedef unsigned char          Char;

static  int              inch, ioch;            // Input and output channel used for client connection.

extern  int           Sdrw_Client_id,           // Draw Client Process id.
                     Sdrw_HalfDuplex;           // Half Duplex Block Flag.




#if !defined( _WIN32 ) || defined( __CYGWIN__ )
/*
 *       For any Unix like Operating Systems.
 *
*/

# include <sys/types.h>
# include <sys/socket.h>
# include <netinet/in.h>
# include <netdb.h>

static struct sockaddr_in srv_addr;

static int Cli_Sock_Open( int port, const char* host )
{
  char dfhost[] = "127.0.0.1";
  struct hostent *server;

  if ((ioch = socket( AF_INET, SOCK_STREAM, 0 )) < 0) return -2;
  if (!host) host = dfhost;
  if (!(server = gethostbyname( host ))) return -6;
  bzero( (char*) &srv_addr, sizeof( srv_addr ) );
  srv_addr.sin_family = AF_INET;
  bcopy( (char*)server->h_addr, (char*)&srv_addr.sin_addr.s_addr, server->h_length );
  srv_addr.sin_port   = htons( port );
  if (connect( ioch, (struct sockaddr *)&srv_addr, sizeof( srv_addr ) ) != 0) return -7;
  return 0;
}



static void Cli_Sock_Close( void )
{
  close( ioch );
}



inline int Cli_Sock_Read( char* buf, int buflen )
{
  return read( ioch, buf, buflen );
}


inline int Cli_Sock_Write( char* buf, int buflen )
{
  return write( ioch, buf, buflen );
}






#else
/*
 *       For Windows Operating Systems.
 *
*/

# include <winsock2.h>


static WSADATA                wsaData;
static SOCKET                    sock;
static struct sockaddr_in server_addr;

# define Do_Exit( ierr ) { WSACleanup(); return ierr; }


static int Cli_Sock_Open( int port, char* host )
{
  int    addr_len = sizeof( server_addr );
  HOSTENT *server;

  WSAStartup( 0x0101, &wsaData );

  if ((sock = socket( AF_INET, SOCK_STREAM, 0 )) < 0) Do_Exit( -2 );
  memset( &server_addr, 0, sizeof( server_addr ) );
  server_addr.sin_family = AF_INET;
  server_addr.sin_port   = htons( port );
  if (host) {
    if (!(server = gethostbyname( host ))) return -6;
    /* server_addr.sin_addr.s_addr = inet_addr( server->h_addr_list[0] }; */
    server_addr.sin_addr.S_un.S_un_b.s_b1 = (unsigned char)server->h_addr_list[0][0];
    server_addr.sin_addr.S_un.S_un_b.s_b2 = (unsigned char)server->h_addr_list[0][1];
    server_addr.sin_addr.S_un.S_un_b.s_b3 = (unsigned char)server->h_addr_list[0][2];
    server_addr.sin_addr.S_un.S_un_b.s_b4 = (unsigned char)server->h_addr_list[0][3];
  }
  else
    server_addr.sin_addr.s_addr = htonl( 0x7F000001 );
  if (connect( sock, (struct sockaddr*)&server_addr, addr_len) != 0) Do_Exit( -3 );
  return 0;
}



static void Cli_Sock_Close( void )
{
  closesocket( sock );
  WSACleanup();
}



inline int Cli_Sock_Read( char* buf, int buflen )
{
  int r = recv( sock, buf, buflen, 0 );
  if (r == SOCKET_ERROR) errno = WSAGetLastError();
  return r;
}


inline int Cli_Sock_Write( char* buf, int buflen )
{
  int r = send( sock, buf, buflen, 0 );
  if (r == SOCKET_ERROR) errno = WSAGetLastError();
  return r;
}



#endif



 /************************************************************************
 *                                                                       *
 *                    Common Connection Manager Part.                    *
 *                                                                       *
 ************************************************************************/



void Sdrw_Open_Gate( int arc, char **arv )
{ /* Attach the DRAW-LIB pipe channels to the local channel */
  int ie;
  static int chinp, chout, i;

  fprintf( fmsg, " Command %s %s %s %s\n", arv[0], arv[1], arv[2], arv[3] );
  fflush( fmsg );

  if (arc < 4) {
    fprintf( fmsg, " Illegal DRAW_SERVER Call (number of parameters < 4).\n" );
    for (i=0;i<arc;i++)
      fprintf( fmsg, " Parameter #%d = \"%s\".\n", i, arv[i] );
    fprintf( fmsg,
      " The normal call is <Server_Path> <in_pipe#> <out_pipe#> <User_pid> [<User_Label>]\n" );
    fclose( fmsg );
    exit( 2 );
  }

  chinp = atoi( arv[1] );                     /* Get the Channel number from ... */
  chout = atoi( arv[2] );                     /* ...the Parameter list */
  Sdrw_Client_id = atoi( arv[3] );            /* Get the Client Process id */

  Sdrw_HalfDuplex = 1;                        /* Half Duplex is the default (use flow control message for block io). */

#ifndef _PIPE_CNTFLOW                         /* When full duplex (no block control flow) is possible */

  if (chinp>=0) Sdrw_HalfDuplex = 0;          /* The Full Duplex mode is enabled */

#endif

  if (chinp<0) {                              /* Input Channel < 0 is used to flag the socket use */
    fprintf( fmsg, " Draw_Server uses the socket port # %d for data exchange.\n", chout );
    fflush( fmsg );
    if (ie = Cli_Sock_Open( chout, "127.0.0.1" )) {
      fprintf( fmsg, " errno = %d\n", errno );
      fprintf( fmsg, " Draw Server Cannot open the Client created socket on port %d on error %d.\n", chout, ie );
      fclose( fmsg );
      exit( 2 );
    }
    inch = -1;                                /* value to flag the socket use */
                                              /* ioch is set by Cli_Sock_Open */
  } else {                                    /* We use the bi directional pipe */

#ifndef _PIPE_CNTFLOW                         /* When full duplex (no block control flow) is possible */
    Sdrw_HalfDuplex = 0;                      /* The Full Duplex mode is enabled */
#endif

    fprintf( fmsg, " Draw_Server uses the a pipe for data exchange in %s duplex mode.\n", (Sdrw_HalfDuplex)?"half":"full" );
    fflush( fmsg );

    inch = dup( chinp );                      /* Get the Channel number from ... */
    ioch = dup( chout );                      /* ...the Parameter list */
    close( chinp );                           /* Close temporary channels */
    close( chout );
  }
}




/************************************************************************/

void Sdrw_Close_Gate()
{ /* Stop Server Connection */
//int status, cstatus;

  if (inch>=0) { close( inch ); close( ioch ); }
          else Cli_Sock_Close();

//if ((status = wait( &cstatus )) == -1)
/*
    if ((status = fltk::wait()) == -1)
    exit( 10 );
*/
}




int Sdrw_Read_Blk( Char* buf, int buflen )
{ /* Read a buffer from the DRAW Library */
  int sz;

  if (inch>0) sz = read( inch, (char*)buf, buflen );
         else sz = Cli_Sock_Read( (char*)buf, buflen );
  if (sz < 0) {
    fprintf( fmsg, " SDRW: Read_Blk from Client failed with err # %d .\n", errno );
    fprintf( fmsg, " msg = \"%s\".\n", strerror( errno ) );
    fflush( fmsg );
    exit( 21 );
  }
  return sz;
}



void Sdrw_Write_Blk( Char* buf, int bufsize )
{ /* Write a buffer to the DRAW Library */
  int ie;

  if (inch>0) ie = write( ioch, (char*)buf, bufsize );
         else ie = Cli_Sock_Write( (char*)buf, bufsize );
  if (ie == -1) {
    fprintf( fmsg, " SDRW: Write_Blk to Client failed with err # %d .\n", errno );
    fprintf( fmsg, " msg = \"%s\".\n", strerror( errno ) );
    fflush( fmsg );
    exit( 2 );
  }
}


/*************************     E N D    ******************************/
