/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*              D R A W   -   C L I E N T   L I B R A R Y                *
*                                                                       *
*                    Draw Library Version V 2.4 A                       *
*                                                                       *
*              (Kernel to manage the bidirectional pipe)                *
*                                                                       *
*                     Draw Library Version V 2.4                        *
*                                                                       *
*                           08-Oct-2009                                 *
*                                                                       *
*                               by                                      *
*                                                                       *
*                 Pierre Wolfers, Institut Neel                         *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/

/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/




/*********************************************************************/
/**                                                                 **/
/**          Echange DRAW_LIb <-> DRAW_SERVER Constantes            **/
/**                                                                 **/
/*********************************************************************/

# include <stdio.h>
# include <sys/types.h>
# include <sys/socket.h>
# include <netinet/in.h>
# include <string.h>
# include <netdb.h>
# include <unistd.h>
# include <errno.h>

#if (!defined( _WIN32 )) || defined( __CYGWIN__ )
#  define PATH_SEPAR   ':'
#  define DIR_SEPAR    '/'
#  ifdef __CYGWIN__
#    define DEF_PATH   "/C/Program Files/Draw_Env/bin:/C/WINDOWS/system32:/C/WINDOWS"
#  else
#    define DEF_PATH   "/bin:/usr/bin:/usr/local/bin"
#  endif
#else
#  define PATH_SEPAR   ';'
#  define DIR_SEPAR    '\\'
#  define DEF_PATH   "C:\\Program Files\\Draw_Env\\bin;C:\\WINDOWS\\system32;C:\\WINDOWS"
#endif


#define MAX_PATH    256


#include <draw/draw_common.h>
#include <draw/draw_constantes.h>   /* Load the DRAW library Environment */


#ifdef _PIPE_BIG
# ifdef VMS
#   define _Pipe( a , b, c ) pipe( a, b, c)
# else
#   define _Pipe( a , b, c ) _pipe( a, b, c)
# endif
#else
# define _Pipe( a , b, c ) pipe( a )
#endif







/*********************************************************************/
/**                                                                 **/
/**      Echange DRAW_LIb <-> DRAW_SERVER Types Definitions         **/
/**                                                                 **/
/*********************************************************************/



typedef unsigned char Char;             /* Define the Unsigned char type Char */

typedef union {                         /* Define the Type Equiv. rec. type */
          struct { Char ch0, ch1, ch2, ch3; } ct;
                    int   i;
                    float f;
		} eqty;



/*********************************************************************/
/**                                                                 **/
/**          Echange DRAW_LIb <-> DRAW_SERVER Variables             **/
/**                                                                 **/
/*********************************************************************/

        int          Sdrw_Server_id;    /* Server Process identifier number (global object) */
static  int         Sdrw_HalfDuplex;    /* Flag for half/full Duplex in PIPE/SOCKET Exchange */

static  int  c_inpipe,    c_outpipe,    /* Client Duplications */
             s_inpipe,    s_outpipe;    /* Server Duplications */

static  int  port,   sock,   iosock;    /* Socket port and socket channels */

static  struct sockaddr_in srv_addr, cli_addr;

static  int             Sdrw_Insize,    /* Number of byte readden */
                       Sdrw_Outsize,    /* Number of byte to write */
                        Sdrw_Inread;    /* Index in the Read Pipe Buffer */

static Char  Sdrw_Inp_Buf[BUF_SIZE],    /* Buffer for Input from Server */
             Sdrw_Out_Buf[BUF_SIZE];    /* Buffer for output to server */

static  int            Sdrw_RT_mode;    /* Real time mode flag to Write to Server */

static  int  Sdrw_Server_Enable = 0;    /* Server Process Enable Flag */

static char                 *s_port;    /* String pointer for socket port number */

static char                *argu[8];    /* Draw Server Argument Pointer Table */ 

static eqty equ;                        /* Record for float<->int<->Char Conv. */


#ifdef _SDRW__REVERSE
# define SDRW_input  20
# define SDRW_output 21
#endif




/*********************************************************************/
/**                                                                 **/
/**   Data Echange Routines for DRAW_LIB <-> DRAW_SERVER Dialogue   **/
/**                                                                 **/
/*********************************************************************/

int Sdrw_Open_Gate()
{ /* Create the bidirectional pipe or socket server */

#ifdef _SDRW__REVERSE

  c_inpipe  = dup( SDRW_input );
  c_outpipe = dup( SDRW_output );
  close( SDRW_input );
  close( SDRW_output );

#else

/* # if (defined( _WIN32)) && !defined(__CYGWIN__) */
# ifdef _WIN32 
/* Always use Win32 SOcket with Windows operating system */
  int name_length;


printf( " Work in the native WIN32 mode\n" );

   /* Native Windows mode : Use an Half-Duplex Socket */
  s_port = getenv( "DRAW_SOCKET_PORT" );
  if (s_port&&(s_port[0])) port = atoi( s_port );       /* Get port number value ... */
                      else port = 0;                    /* ... or set the automatic port allocation */

  if ((sock = socket( AF_INET, SOCK_STREAM, 0 )) < 0) { /* Create the connection socket */
    perror( "DRAW-LIB ERROR: Socket establish on socket #-2: " );
    printf( " Server err -2: %d\n", errno );
    exit( 1 );
  }
  bzero( (char*) &srv_addr, sizeof( srv_addr ) );
  srv_addr.sin_family      =       AF_INET;
  srv_addr.sin_port        = htons( port );
  srv_addr.sin_addr.s_addr =    INADDR_ANY;
  if (bind( sock, (struct sockaddr*) &srv_addr, sizeof( srv_addr ) ) < 0) {
    perror( "DRAW-LIB ERROR: Socket establish err on bind #-3: " );
    /* printf( " Binding err -3: %d\n", errno ); */
    exit( 1 );
  }
  if (!port) { /* Get the port number when it is automaticaly allocated */
    name_length = sizeof(srv_addr);
    getsockname( sock, (struct sockaddr *)&srv_addr, &name_length );
    port = htons( srv_addr.sin_port );
    s_port = malloc( 10 ); sprintf( s_port, "%d", port );
  }
  if (listen( sock, 5 ) < 0) {
    close( sock );
    perror( "DRAW-LIB ERROR: Socket establish err on listen #-4: " );
    /* printf( " Listen err -4:%d\n", errno ); */
    exit( 1 );
  }

  /* Now we are ready to accept a connection */
  /* We can start the Draw Server Program */

  

# else

  int cs_pipes[2], sc_pipes[2];                         /* Original bi-directional pipe */

  port = 0;                                             /* Flag the pipe mode for next routine */

  /* Unix like Operating system : with the bi-directionnal pipe use */

  if (_Pipe( cs_pipes, 0, BUF_SIZE ) == -1) {
    perror( "DRAW-LIB ERROR: - Client to Server Pipe allocation failed" );
    /* printf( " DRAW-LIB ERROR: - Client to Server Pipe allocation failed\n" ); */
    exit( 1 );
  }
  if (_Pipe( sc_pipes, 0, BUF_SIZE ) == -1) {
    perror( "DRAW-LIB ERROR: - Server to Server Pipe allocation failed" );
    /* printf( " DRAW-LIB ERROR: - Server to Client Pipe allocation failed\n" ); */
    exit( 1 );
  }
  /* Set the pipe file connections */
  c_inpipe  = sc_pipes[0];            /* Client input channel  */
  c_outpipe = cs_pipes[1];            /* Client output channel */
  s_outpipe = sc_pipes[1];            /* Server output channel */
  s_inpipe  = cs_pipes[0];            /* Server input channel  */

# endif

#endif

  Sdrw_RT_mode = 0;                     /* Init to wait mode (No RT mode) */
  Sdrw_Outsize = 0;                     /* Init Output message Length */
  return BUF_SIZE;                      /* Return the Pipe Buffer Size. */
}



/*********************************************************************/

static char * Search_In_Sys_Path( char * name )
{
  static char    buf[MAX_PATH];
  char           * path, * fnm;
  int              i, j, k, re;


//printf( " DEF_PATH = \"%s\"\n Path Separators = \'%c\', \'%c\'\n", DEF_PATH, DIR_SEPAR, PATH_SEPAR );


  if (re = access( name, X_OK )) {              /* Use the PATH when the direct use of name is not a success */
    path = DEF_PATH;                            /* Look on the standard DRAW path before */
    while (path) {
      re = 0;
      i  = 0;
      do {
        fnm = buf;
        j = k = 0;
        while (path[i]&&(path[i] != PATH_SEPAR)) buf[j++] = path[i++]; /* Get one path entry (separator is ";") */
        if (path[i] == PATH_SEPAR) i++;                 /* Skip the separator for the next path list entry */
        if (buf[j-1] != DIR_SEPAR) buf[j++] =DIR_SEPAR; /* Append a directory separator when it is not present. */
        while (name[k]) buf[j++] = name[k++];   /* Append the file name to the directory comming from the path */
        buf[j] = 0;                             /* Append the final null character */
        re = access( buf, X_OK );               /* Check for execute access */
      } while (re&&path[i]);
      if (re)
        if (path == DEF_PATH) path = getenv( "PATH" ); /* When not found, we try with the Windows PATH */
                         else path = (char *) 0;
      else path = (char *) 0;
    }
  } else fnm = name;                            /* We use the server in the User Program Directory */

//printf( " Find \"%s\" with x access of %d\n", fnm, re );

  if (!re) return fnm;
      else return   0;
}



/*********************************************************************/

static int Spawn_Server( Char *fnm, Char *s_port )
{ /* To start the DRAW Server and establish the pipes with it */
  int     ierr;

  ierr = 1;                                     /* Assume Success until shown otherwise. */

#ifndef _SDRW__REVERSE

  int status, i, cli_len;
  /* char * ExcErr = " DRAW-LIB ERROR: - Cannot exec the Server ""%s"" \n"; */

  /* Allocate room for Server Arguments */
  for (i=1; i<8; i++) argu[i] = (i<4)?((char *) malloc( 80 )):NULL;

  if (s_port) port = atoi( (char*) s_port );

  Sdrw_HalfDuplex = 1;                          /* Half Duplex is the default (use flow control message for block io). */

# ifndef _PIPE_CNTFLOW                          /* When full duplex (no block control flow) is possible */

  if (!port) Sdrw_HalfDuplex = 0;               /* The Full Duplex mode is enabled */

# endif

  if (fnm&&fnm[0]) {                            /* When to get the Draw Server Complete Path is a success, ... */
    argu[0] = (char *) fnm;                     /* ... set the Server reference. */
    if (port) {                                 /* When we use a socket connection */
      strcpy( argu[1], "-1" );                  /* Put the server flag for socket use ... */
      strcpy( argu[2], (char*)s_port );         /* ... and the port number as second parameter. */
    } else {                                    /* When we use the bidirectionnal pipe */
      sprintf( argu[1],"%d", s_inpipe  );       /* Put the input and output pipe channels ... */
      sprintf( argu[2],"%d", s_outpipe );       /* ... as first and second Server arguments. */
    }
    sprintf( argu[3],"%d", getpid() );          /* Put the Draw Client process id. */

    /*
     *       For any Unix like Operating Systems.
     *
     */

    status = fork();                            /* Create the Server process */
    switch (status) {                           /* Select the process */
      case 0:                                   /* For Server process, we try ... */
        if ((status = execv( (char *) fnm, argu )) == -1) {     /* ... to start it. */
          perror( "DRAW-LIB INIT ERROR: Server activation fails on execv #-5" );
          /* printf( ExcErr, *cmd ); */
          exit( 1 );                            /* Stop with message on Exec ERROR. */
        }
      break;

      case -1:                                  /* Error return on fork */
        perror( "DRAW-LIB INIT ERROR: Server activation fails on fork #-5" );
        /* printf( " DRAW-LIB INIT ERROR: Fork Error\n" ); */
        exit( 1 );
      break;

      default:                                  /* Start DRAW Server Success */
        Sdrw_Server_id = status;                /* Keep the Server Process id. */
      break;
    }

    if (port) {                                 /* We can accept the port connection of Draw Server */
      cli_len = sizeof( cli_addr );
      if ((iosock = accept( sock, (struct sockaddr*) &cli_addr, (socklen_t*)&cli_len )) <= 0) {
        close( sock );
        perror( "DRAW-LIB INIT ERROR: Socket establish err on accept #-5" );
        /* printf( " Socket establish err on accept #-5: %d\n", errno ); */
        exit( 1 );
      }
      c_inpipe = c_outpipe = iosock;
    }
  } else ierr = 0;

#endif

/* printf( " Opened Socket Io Channel in=%d, out=%d\n", c_inpipe, c_outpipe ); */

  Sdrw_Insize  = 0;                     /* Init the pipe I/O buffers sizes/counts */
  Sdrw_Inread  = 0;
  Sdrw_Outsize = 0;
  Sdrw_Server_Enable = 1;               /* Flag as Server Ready */

  return ierr;
}



/*********************************************************************/

int Sdrw_Establish_Connection( char * envv, char * srv_nam )
{ /* Start the specified DRAW SERVER */
  /* Translate the logical server activation word */
  char * prg_nam;

  prg_nam = getenv( envv );
  if (!(prg_nam&&prg_nam[0])) prg_nam = srv_nam;
  prg_nam = Search_In_Sys_Path( prg_nam );
  return Spawn_Server( (Char *) prg_nam, (Char *) s_port );
}



/*********************************************************************/

void Sdrw_Connection_Release()
{ /* Stop Server Connection and Exit */
  int status, cstatus;
  if ((status = wait( &cstatus )) == -1)
  {
    printf( " DRAW-LIB ERROR: USER Wait failed\n" );
    exit( 1 );
  }
  Sdrw_Server_Enable = 0;               /* Flag as Server stopped */
}



/*********************************************************************/

int Sdrw_Read()
{ /* Read a buffer from the DRAW Server */
  if (!Sdrw_Server_Enable) {            /* If the Server is not Ready */
    printf( " DRAW-LIB ERROR: Server was not running.\n" );
    exit( 1 );
  }
  Sdrw_Insize = read( c_inpipe, Sdrw_Inp_Buf, BUF_SIZE );
  if (Sdrw_Insize<=0) {
    printf( " DRAW-LIB ERROR: USER Pipe Read failed\n" );
    exit( 1 );
  }
  Sdrw_Inread = 0;
  /* printf( " User Read %d bytes.\n", Sdrw_Insize ); */
  return Sdrw_Insize;
}



/*********************************************************************/

void Sdrw_Write()
{ /* Write a buffer to the DRAW Server and wait the Answerd */
  if (!Sdrw_Server_Enable) {            /* If the Server is not Ready */
    printf( " DRAW-LIB ERROR: Server was not running.\n" );
    exit( 1 );
  }
/*printf( " User Write %d bytes.\n", Sdrw_Outsize );*/
  if (write( c_outpipe, Sdrw_Out_Buf, Sdrw_Outsize ) == -1)
  {
    printf( " DRAW-LIB ERROR: Write to SERVER failed.\n" );
    exit( 1 );
  }
  Sdrw_Outsize = 0;
}



/************************************************************************/

int  Sdrw_Get_Code();
void Sdrw_Put_Code( int );



int  Sdrw_Read_Block( Char* tb, int dim )
{ /* Read a Block from the DRAW Library */
  int n, i, t;

  /* For the full duplex mode, we send only the initial read request */
  if (!Sdrw_HalfDuplex) { Sdrw_Put_Code( 0 ); Sdrw_Write(); }
  /* The Server has begun to send the Data Block (stored at tb, dim bytes length) */
  t = 0;
  while (t < dim) {
    /* For Half Duplex, we send a flow control request before to read each buffer */
    if (Sdrw_HalfDuplex) { Sdrw_Put_Code( 0 ); Sdrw_Write(); }
    n = read( c_inpipe, tb, BUF_SIZE ); /* Read the data packet directly in the target array */
    if (n < 0) {
      printf( " DRAW-LIB ERROR: USER Pipe Read Block failed.\n" );
      return n;
    }
    t  += n;                            /* Update the byte counts. */
    tb += n;
  }
  return t;
}



/************************************************************************/

void Sdrw_Write_Block( Char* tb, int dim )
{ /* Write a User buffer to the DRAW Library */
  int i, n;

  if (!Sdrw_HalfDuplex) Sdrw_Read();          /* Read for synchro at the begin of block transaction */
  /* The Server must be ready to get the Data Block (address tb, dim bytes length) */
  while (dim) {
    n = (dim > BUF_SIZE)?BUF_SIZE:dim;
    if (Sdrw_HalfDuplex) Sdrw_Read();        /* Read for synchro to get each data packet */
    if (write( c_outpipe, tb, n ) == -1)     /* Send Directly the data packet */
    { printf( " DRAW-LIB ERROR: Write Block to SERVER failed.\n" );
      exit( 1 );
    }
    dim -= n;                                /* Update the bytes counts */
    tb  += n;
  }
}



/*********************************************************************/

int Sdrw_Dialog_Request()
{
  Sdrw_Write();
  return Sdrw_Read();
}



/*********************************************************************/
/**        Input from and Output to the Graphic Server Routines     **/ 
/*********************************************************************/

void Sdrw_Put_Char( Char ch )
{
  if (Sdrw_Outsize < BUF_SIZE)
    Sdrw_Out_Buf[Sdrw_Outsize++] = ch;
}



/*********************************************************************/

void Sdrw_Put_Int( int iv )
{
  equ.i = iv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
}



/*********************************************************************/

void Sdrw_Put_Float( float fv )
{
  equ.f = fv;
  Sdrw_Put_Char( equ.ct.ch0 );
  Sdrw_Put_Char( equ.ct.ch1 );
  Sdrw_Put_Char( equ.ct.ch2 );
  Sdrw_Put_Char( equ.ct.ch3 );
}



/*********************************************************************/

void Sdrw_Put_String( Char * s, int l )
{
  int i;
  Sdrw_Put_Char( l );
  for ( i=0; i<l; i++)
  {
    Sdrw_Put_Char( s[i] );
  }
}



/*********************************************************************/

void Sdrw_Put_Code( int code )
{
  Sdrw_Outsize = 0;
  Sdrw_Put_Int( code );
}



/*********************************************************************/

Char Sdrw_Get_Char()
{
  if (Sdrw_Inread < Sdrw_Insize)
    return Sdrw_Inp_Buf[Sdrw_Inread++];
  else
    return 0;
}



/*********************************************************************/

int Sdrw_Get_Int()
{
  equ.ct.ch0 = Sdrw_Get_Char();
  equ.ct.ch1 = Sdrw_Get_Char();
  equ.ct.ch2 = Sdrw_Get_Char();
  equ.ct.ch3 = Sdrw_Get_Char();
  return equ.i;
}



/*********************************************************************/

float Sdrw_Get_Float()
{
  equ.ct.ch0 = Sdrw_Get_Char();
  equ.ct.ch1 = Sdrw_Get_Char();
  equ.ct.ch2 = Sdrw_Get_Char();
  equ.ct.ch3 = Sdrw_Get_Char();
  return equ.f;
}



/*********************************************************************/

int Sdrw_Get_String( char* s )
{
  int i, l;

  l = Sdrw_Get_Char();
  for (i=0; i<l; i++)
    s[i] = Sdrw_Get_Char();
  s[l] = 0;
  return l;
}



int Sdrw_Get_String_Lim( char* s, int n )
{
  int i, l, m;
  char      c;

  m = l = Sdrw_Get_Char();
  if (n-1 < l) m= n-1;
  for (i=0; i<m; i++) s[i] = Sdrw_Get_Char();
  if (n<l) for (; i<l; i++) c = Sdrw_Get_Char();
  s[m] = 0;
  return m;
}



int Sdrw_Get_String_SP( char** p, int aflg )
{
  int i, l;
  Char*  s;

  l = Sdrw_Get_Char();
  if (aflg) {
    if (*p) free( (void*)*p );
    s = *p = (Char*) malloc( l );
  }
  for (i=0; i<l; i++)
    s[i] = Sdrw_Get_Char();
  s[l] = 0;
  return l;
}



/*********************************************************************/

int Sdrw_Get_Code()
{
  Sdrw_Inread = 0;
  return Sdrw_Get_Int();
}




/*********************************************************************/
/**   Block Input from and Output to the Graphic Server Routines    **/ 
/*********************************************************************/

void Sdrw_Put_FBlock( float tb[], int dim )
{ /* Output to pipe the array tb[dim] */
  int i, j, r;

  j = 0;
  while (dim > 0) {
    /* Fill the Buffer */
    r = (BUF_SIZE - sizeof( int ) - Sdrw_Outsize)/sizeof( float );
    if (r > dim) r = dim;
    Sdrw_Put_Int( r );
    for (i = 0; i < r; i++) Sdrw_Put_Float( tb[j++] );
    dim -= r;
    if (dim > 0) Sdrw_Dialog_Request();
  }
}



/*********************************************************************/

int Sdrw_Get_FBlock( float tb[], int dim )
{
  int i, j, n, t;

  j = 0;
  t = 0;
  while (t < dim) {
    n = Sdrw_Get_Int();
    for(i = 0; i < n; i++) tb[j++] = Sdrw_Get_Float();
    t += n;
    if (t < dim) { /* Get the next message */
      Sdrw_Put_Code( cd_continue );
      Sdrw_Dialog_Request();
    }
  }
  return t;
}




/*************************     E N D    ******************************/
