//********* DRAW Interface ROUTINES for OpenGL Graphic Libary ***********

// *************************************************************************
// *                                                                       *
// *                                                                       *
// *                                                                       *
// *          F L T K / D R A W   -   S E R V E R   M O D U L E            *
// *                                                                       *
// *            Version  1.4-C for Draw Library Level V 2.4 A              *
// *                                                                       *
// *           (Draw interface for FLTK/OPENGL/Font Manager )              *
// *                                                                       *
// *               * * * FTGL based Font Manager * * *                     *
// *                                                                       *
// *                               by                                      *
// *                                                                       *
// *                 Pierre Wolfers, Institut Néel                         *
// *                                                                       *
// *          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
// *                                                                       *
// *                     F 38042 GRENOBLE CEDEX 9                          *
// *                                                                       *
// *                           F R A N C E                                 *
// *                                                                       *
// *                                                                       *
// *                                                                       *
// *************************************************************************



/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////


/* Debug mode Undefined => no, Define => yes. */
/*
#define _No_GL_Error
//#define _GL_Debug 0
#ifdef _GL_Debug
#  define _GL_FullDebug 0
#endif
*/



#include "Server_GL.h"

#include <FTGL/ftgl.h>

// To debug the String place managment the next definition must be activate.
#define FRAME_PLOT



// ***********************************************************************
// *                                                                     *
// *     DRAW/GL/FTGL Interface Routines to Set and Plots Characters     *
// *                                                                     *
// ***********************************************************************


#ifndef word

# define word short	/* MACHINE DEPENDENT */

#endif

// Define Font table allocation unit (in font pointer number).
#define  FONT_ALLOC_UNIT       32
#define  MAX_FONT_NUMBER      128
// MAX_FONT_NUMBER must be greater than SIZE_DEF_FONT.



typedef struct Font_Dsc *  Draw_Font_Ptr;                // Define a pointer of font.


#define MAX_FONTS             64
#define DEF_DEPTH            0.3


typedef struct Font_Dsc { /* Font Descriptor */
                 int            fnt_style;               // Font style.
                 char*          fnt_fname;               // Font File Path.
                 FTFont*        fnt_ftgld;               // Font descriptor used by FTGL.
               } Draw_Font_Dsc;


static  Draw_Font_Ptr        * font_table;               // Pointer of Font pointer table.
static  Draw_Font_Ptr           curr_font;               // Pointer of Current Font record.
static  int                    font_tsize;               // Current size of font table.

static  FTSimpleLayout            slayout;
static  FTBBox                       tbox;
static  FTFont*                       fnt;               // Current FTGL font.
static  FTPoint  spacing( 0.0, 0.0, 0.0 );               // Spacing vector.
static  FTPoint              user_spacing;
static  FTBBox                       bbox;

static  int             hflg       =    1;               // Horizontal mode flag.

static  int                    valn, haln;               // Current Vertical and Horizontal alignements.


#if 0
static FTGL::TextAlignment halntb[] =
       { FTGL::ALIGN_LEFT, FTGL::ALIGN_LEFT, FTGL::ALIGN_CENTER, // Table of FTGL value to map frm Draw Values.
         FTGL::ALIGN_RIGHT, FTGL::ALIGN_JUSTIFY};
#endif



// *************    Use FTGL FONT    ************



static void Usr_To_Scr( float ux, float uy, float uz, float *sx, float *sy, float *sz )
{ // Convert the Point User coordinates (ux,uy,uz) to Screen coordinates (sx,sy,sz).
  GLdouble xsx, ysy, zsz;

  if (gluProject( (GLdouble) ux, (GLdouble) uy, (GLdouble) uz,
                  Drwgl_modlview, Drwgl_proj_wrd, Drwgl_wrdvport, &xsx, &ysy, &zsz ) == GL_TRUE) {
    *sx = xsx; *sy = ysy; *sz = zsz;
  } else {
    fprintf( fmsg, " Usr_To_Src Error : gluProject error with User Coordinates (%f, %f, %f).\n", ux, uy, uz );
    fflush( fmsg );
    *sx = 0.0; *sy = 0.0, *sz = 0.0;
  }
}



static void Scr_To_Usr( float sx, float sy, float sz, float *ux, float *uy, float *uz )
{ // Convert the Point Screen coordinates (ux,uy,uz) to User coordinates (sx,sy,sz).
  GLdouble xux, yuy, zuz;

  if (gluUnProject( (GLdouble) sx, (GLdouble) sy, (GLdouble) sz,
                    Drwgl_modlview, Drwgl_proj_wrd, Drwgl_wrdvport, &xux, &yuy, &zuz ) == GL_TRUE) {
    *ux = xux; *uy = yuy; *uz = zuz;
  } else {
    fprintf( fmsg, " Scr_To_Usr Error : gluUnProject error with Screen Coordinates (%f, %f, %f).\n", sx, sy, sz );
    fflush( fmsg );
    *ux = 0.0; *uy = 0.0, *uz = 0.0;
  }
}



static void DrwGL_ScrXYZ( GLdouble x, GLdouble y, GLdouble z, int ini )
{
  if (ini) {
    Drwh_mix = Drwh_max = x; Drwh_miy = Drwh_may = y; Drwh_miz = Drwh_maz = z;
  } else {
    if (Drwh_mix > x) Drwh_mix = x; if (Drwh_max < x) Drwh_max = x;
    if (Drwh_miy > y) Drwh_miy = y; if (Drwh_may < y) Drwh_may = y;
    if (Drwh_miz > z) Drwh_miz = z; if (Drwh_maz < z) Drwh_maz = z;
  }
}



#if 0
static void Enable_Font( Draw_Font_Ptr pf, float hc, float *rp, float *asc, float *dsc, float *lnh )
{
  float x1, y1, z1, x2, y2, z2;
  int                      ifc;

  fnt  =  pf->fnt_ftgld;

  // Get the size of one window pixel in the user space coordinates.
  Scr_To_Usr( 0.0, 0.0, 0.0, &x1, &y1, &z1 );
  Scr_To_Usr( 1.0, 0.0, 0.0, &x2, &y2, &z2 );
  x2 -= x1; y2 -= y1; z2 -= z1;
  ifc = (int)(hc/sqrt( x2*x2 + y2*y2 + z2*z2 ) + 0.5); // Get the character high in pixel.
  if (ifc < 1) ifc = 1;
  fnt->FaceSize( ifc );        // Set the required font size.
  *rp = hc/ifc;                // Keep the Scale Ratio between the user space and the Face (Glyphe) space (in user_unit/(1/72")).
  *asc =    fnt->Ascender();   // Get the geometrical font parameters for this font size (in 1/72").
  *dsc =   fnt->Descender();
  *lnh =  fnt->LineHeight();
}
#endif



static void Display_Text_Basic( float xc, float yc, float zc, float hc, float ha, char * str, int len, int sel )
{ // Draw a Line of text.
  Draw_Font_Ptr                                     pf;
  int                                              ifc;
  float         x1, y1, z1, x2, y2, z2, rp, vv, dx, dy,
               txt_width, txt_high, txt_depth, txt_asc,
			   txt_dsc, txt_lnh;

  if (str&&fnt) {
    dx = dy = 0.0;
//
// Here The Path is supposed to be always TXT_PATH_RIGHT
//

//  fprintf( fmsg, " GL Text plot at (%f,%f), high = %f, ang = %f, len = %d, s = \"%s\" .\n",
//                 xc, yc, hc, ha, len, str );
//  fflush( fmsg );

    pf = curr_font;                     // Get the Font descriptor address in a local variable for optimization.

    // Get the size of one window pixel in the user space coordinates.
    Scr_To_Usr( 0.0, 0.0, 0.0, &x1, &y1, &z1 );
    Scr_To_Usr( 1.0, 0.0, 0.0, &x2, &y2, &z2 );
    x2 -= x1; y2 -= y1; z2 -= z1;
    ifc = (int)(hc/sqrt( x2*x2 + y2*y2 + z2*z2 ) + 0.5); // Get the character high in pixel.
    if (ifc < 1) ifc = 1;
    fnt->FaceSize( ifc );               // Set the required font size.
    rp = hc/ifc;                        // Keep the Scale Ratio between the user space and the Face (Glyphe) space (in user_unit/(1/72")).
    txt_asc =    fnt->Ascender();       // Get the geometrical font parameters for this font size (in 1/72").
    txt_dsc =   fnt->Descender();
    txt_lnh =  fnt->LineHeight();
    spacing = user_spacing*(1.0/rp);

/*
fprintf( fmsg, " OK 1\n" );
fflush( fmsg );
*/

//  if (pf == font_table[2]) {
//    fprintf( fmsg, " dx, dy = %F, %f\n", x2, y2 );
//    fprintf( fmsg, " ifc = %4d, hc = %8.3f, asc/hc = %8.5f, dsc/hc = %8.5f, lnh/hc = %8.5f\n", ifc, hc, txt_asc/ifc, txt_dsc/ifc, txt_lnh/ifc );
//    fflush( fmsg );
//  }

    // In the current space, 1/72" of font is corresponding to 1 window Pixel
    // to optimize the rendering of glyphes (when the rotations are not using - possible with vectorial fonts).

/*
fprintf( fmsg, " OK 2\n" );
fflush( fmsg );
*/

    if ((rp >0.0)&&(pf->fnt_style == FTGL_EXTRUDE)) {
      vv = Drwgl_Attr->txt_dept/rp;
      if (vv <= 0.0) vv = DEF_DEPTH*fnt->FaceSize();
      fnt->Depth( vv );                 // Set the Character depth for extruded font.
    }

    // Get the text box coordinates in the font space reference (1/72").
    // Get Box sise in (1/72").

/*
fprintf( fmsg, " OK 3\n" );
fflush( fmsg );
*/

    bbox = fnt->BBox( str, -1, FTPoint(), spacing );

/*
fprintf( fmsg, " OK 4\n" );
fflush( fmsg );
*/

    txt_width = bbox.Upper().Xf() - bbox.Lower().Xf();
    txt_high  = bbox.Upper().Yf() - bbox.Lower().Yf();
    txt_depth = bbox.Upper().Zf() - bbox.Lower().Zf();

/*
fprintf( fmsg, " OK 5\n" );
fflush( fmsg );
*/

    if (pf->fnt_style <= FTGL_PIXMAP) { // Pixel Fonts :
      Usr_To_Scr( xc, yc, zc, &x1, &y1, &z1 );

/*
fprintf( fmsg, " OK PIXMAP \n" );
fflush( fmsg );
*/

      // Adjust the string origine with the user alignement requirements.
      switch (haln) {
        case TXT_HALN_RIGHT:  x1 -=     txt_width; break;
        case TXT_HALN_CENTRE: x1 -= 0.5*txt_width; break;
        default: /* TXT_HALN_NORMAL, TXT_HALN_LEFT, TXT_HALN_JUSTIFY */ break;
      }
      switch (valn) {
        case TXT_VALN_TOP:    y1 -=      txt_high; break;
        case TXT_VALN_CAP:    y1 -=       txt_asc; break;
        case TXT_VALN_HALF:   y1 -= 0.5*(txt_asc+txt_dsc); break;
        case TXT_VALN_BOTTOM: y1 -=       txt_dsc; break;
        default: /* TXT_VALN_NORMAL, TXT_VALN_BASE */
        break;
      }
    } else {                                    // For Vectorial Fonts :
      txt_width *= /* Drwgl_txt_expf* */ rp;
      txt_high  *= rp; txt_depth *= rp;
      txt_asc  *= rp; txt_dsc  *= rp; txt_lnh  *= rp;

/*
fprintf( fmsg, " OK VECTORIAL \n" );
fflush( fmsg );
*/

      // Adjust the string origine with the user alignement requirements.
      switch (haln) {
        case TXT_HALN_RIGHT:  dx =     - txt_width; break;
        case TXT_HALN_CENTRE: dx = -0.5*txt_width; break;
        default: /* TXT_HALN_NORMAL, TXT_HALN_LEFT, TXT_HALN_JUSTIFY */
          dx = 0.0;
        break;
      }
      switch (valn) {
        case TXT_VALN_TOP:    dy =      -txt_high; break;
        case TXT_VALN_CAP:    dy =       -txt_asc; break;
        case TXT_VALN_HALF:   dy = -0.5*(txt_asc+txt_dsc); break;
        case TXT_VALN_BOTTOM: dy =      -txt_dsc; break;
        default: /* TXT_VALN_NORMAL, TXT_VALN_BASE */
          dy = 0.0;
        break;
      }
    }

//  fprintf( fmsg, " TBOX = ( %6.3f * %6.3f * %6.3f ), FaceSize = %d => LineHeight = %6.3f (1/72\"),\n",
//                 txt_width, txt_high, txt_depth, fnt->FaceSize(), txt_lnh );
//  fprintf( fmsg, " Asc = %6.3f, Dsc = %6.3f, RP = %f\n", txt_asc, txt_dsc, rp );
//  fprintf( fmsg, " At (%f,%f) with angle %f the string \"%s\".\n", xc, yc, ha, str );
//  fflush( fmsg );


    if (sel) { // Selection Mode.
      if (pf->fnt_style <= FTGL_PIXMAP) {       // Pixel Fonts :
        x2 = x1 + txt_width; y2 = y1 + txt_high;        // Set the box finish vertex.
        DrwGL_ScrXYZ( (GLdouble) x1, (GLdouble) y1, (GLdouble) z1, (hflg>0)?1:0 );
        DrwGL_ScrXYZ( (GLdouble) x2, (GLdouble) y1, (GLdouble) z1, 0 );
        DrwGL_ScrXYZ( (GLdouble) x2, (GLdouble) y2, (GLdouble) z1, 0 );
        DrwGL_ScrXYZ( (GLdouble) x1, (GLdouble) y2, (GLdouble) z1, 0 );
      } else {                                  // For Vectorial Fonts :
        glPushMatrix();
        glTranslatef( xc, yc, 0.0 );
        glRotatef( (GLfloat) ha, 0.0, 0.0, 1.0 );
        glTranslatef( dx, dy, 0.0 );
        glScalef( Drwgl_Attr->txt_expf, 1.0, 1.0 );
        glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
        DrwGL_UsrXYZ( (GLdouble)       0.0, (GLdouble)      0.0, (GLdouble) 0.0, (hflg>0)?1:0 );
        DrwGL_UsrXYZ( (GLdouble) txt_width, (GLdouble)      0.0, (GLdouble) 0.0, 0 );
        DrwGL_UsrXYZ( (GLdouble) txt_width, (GLdouble) txt_high, (GLdouble) 0.0, 0 );
        DrwGL_UsrXYZ( (GLdouble)       0.0, (GLdouble) txt_high, (GLdouble) 0.0, 0 );
        if (txt_depth > 0.0) {
          DrwGL_UsrXYZ( (GLdouble)       0.0, (GLdouble)      0.0, (GLdouble) txt_depth, 0 );
          DrwGL_UsrXYZ( (GLdouble) txt_width, (GLdouble)      0.0, (GLdouble) txt_depth, 0 );
          DrwGL_UsrXYZ( (GLdouble) txt_width, (GLdouble) txt_high, (GLdouble) txt_depth, 0 );
          DrwGL_UsrXYZ( (GLdouble)       0.0, (GLdouble) txt_high, (GLdouble) txt_depth, 0 );
        }
      }
      if (hflg > 1) hflg = 0;

//    fprintf( fmsg, " Pick area (cm) = (%f,%f,%f,%f)\n", xc, yc, x2, y2 );
//    fprintf( fmsg, " Pick area (pixel) = (%f,%f,%f,%f)\n", Drwh_mix, Drwh_miy, Drwh_max, Drwh_may );
//    fflush( fmsg );

      if (pf->fnt_style > FTGL_PIXMAP) {
        glPopMatrix();
        glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
      }
      if ((Drws_max >= Drwh_mix)&&(Drws_mix <= Drwh_max)&&
          (Drws_may >= Drwh_miy)&&(Drws_miy <= Drwh_may))  Hits = 1;
    } else { // Render Mode
      if (pf->fnt_style <= FTGL_PIXMAP) { // BITMAP and PIXMAP.
        Scr_To_Usr( x1, y1, z1, &xc, &yc, &zc );
        glRasterPos3d( xc, yc, zc );
        fnt->Render( str, -1, FTPoint(), spacing, FTGL::RENDER_ALL );
      } else { // VECTORIAL.
        glPushMatrix();
        glTranslatef( xc, yc, 0.0 );
        glRotatef( (GLfloat) ha, 0.0, 0.0, 1.0 );
        glTranslatef( dx, dy, 0.0 );
//      glScalef( rp, rp, rp );
        glScalef( Drwgl_Attr->txt_expf*rp, rp, rp );
        fnt->Render( str, -1, FTPoint(), spacing, FTGL::RENDER_ALL );
        glPopMatrix();
      }
    }


//  fprintf( fmsg, " Text Attr: haln = %d  valn = %d  path = %d\n",
//                 haln, valn, Drwgl_Attr->txt_path );
//  fprintf( fmsg, " exp = %f, spc = %f\n", Drwgl_Attr->txt_expf, Drwgl_Attr->txt_spcf );
//  fflush( fmsg );

  }
}



static char* reverse_string( char * str, int len )
{
  int             i, j;
  static char buf[256];

  if (len>=256) len = 255;
  i =   0;
  j = len;
  while (i<len) buf[--j] = str[i++];
  buf[len] = 0;
  return buf;
}



static int Get_String_Box( char * str, int len, float hc, float * box, float * boy, float * sca )
{
  int   ifc;
  float  rp;

  if (fnt&&str) {                       // OK to process.

    // Get the lowest resolution to compute the FaceSize: esc is in Screen_Pixels/Screen_Cm.
    rp = (d_xscale > d_yscale)?d_yscale:d_xscale;
    ifc = (int)(hc*rp + 0.5);           // Get the character high in pixel (Rounded to nearest integer).
    if (ifc < 1) ifc = 1;               // And set to minimum of one.
    fnt->FaceSize( ifc );               // Set the required font FaceSize.
    fnt->Depth( 1.0 );                  // Set Depth for Extrude fonts.

    rp = hc/ifc;                        // Keep the Scale Ratio between the user space and the Face (Glyphe) space (in user_unit/(1/72")).
    // Get the text box coordinates in the font space reference (1/72").

    bbox = fnt->BBox( str );

//  fnt->BBox( str, x1, y1, z1, x2, y2, z2 );   // Get the String Box coordinates.

//  printf( " Box of (%f,%f)\n", x2 - x1, y2 - y1 );

    *box = rp*(bbox.Upper().Xf() - bbox.Lower().Xf());  // Return the width and height of the box and ...
    *boy = rp*(bbox.Upper().Yf() - bbox.Lower().Yf());
    if (sca) *sca = rp;
    return 0;                           // ... the success return code.
  } else return -1;
}




static void Display_Text_Exotic( float xc, float yc, float zc, float hc, float ha, char * str, int len, int sel )
{
  char                           * tmp;
  static         char sch[] = { 0, 0 };
  int                           ii, ir;
  float as, ds, bx, by, dy, rp, sp, yy;

  if (str&&str[0]) {
    sp  = Drwgl_Attr->txt_spcf;                 // Get the User spacified spacing.
    tmp = str;

    switch (Drwgl_Attr->txt_path) {
      case TXT_PATH_LEFT:
        tmp = reverse_string( str, len );
        user_spacing = FTPoint( sp, 0.0, 0.0 );
        Display_Text_Basic( xc, yc, zc, hc, ha, tmp, len, sel );
      break;

      case TXT_PATH_UP:
        tmp = reverse_string( str, len );

      case TXT_PATH_DOWN:
        ir = Get_String_Box( str, len, hc, &bx, &by, &rp );     // Get the standard box for the string.
        user_spacing = FTPoint();               // Force for no horizontal spacing.
        dy = by + sp;                           // Compute thevertical spacing ...
        bx /= len; by = len*dy;                 // ... and the sizeof box.
        as = rp*fnt->Ascender();
        ds = rp*fnt->Descender();

        switch (haln) {
          case TXT_HALN_RIGHT:  xc -=     bx; break;
          case TXT_HALN_CENTRE: xc -= 0.5*bx; break;
          default: /* TXT_HALN_NORMAL, TXT_HALN_LEFT, TXT_HALN_JUSTIFY */ break;
        }
        switch (valn) {
          case TXT_VALN_TOP:
          case TXT_VALN_CAP:    yc -=     as; break;
          case TXT_VALN_HALF:   yc += -0.5*(as + ds) + 0.5*(len-1)*dy; break;
          case TXT_VALN_BOTTOM: yc +=  ds -(len-1)*dy; break;
          default: /* TXT_VALN_NORMAL, TXT_VALN_BASE */
          break;
        }

        hflg = 2;
        yy = yc;
        for (ii=0; ii<len; ii++) {
          sch[0] = tmp[ii];
          Display_Text_Basic( xc, yy, zc, hc, ha, sch, 1, sel );
          yy -= dy;
        }
        if (sel)
          if ((Drws_max >= Drwh_mix)&&(Drws_mix <= Drwh_max)&&
              (Drws_may >= Drwh_miy)&&(Drws_miy <= Drwh_may))  Hits = 1;
      break;

      default:
        ;
      break;
    }
  }
}



void DrwGL_eval_string_box()
{
  float       box, boy;
  int               ir;
  Draw_Font_Ptr     pf;

  ir = -1;
  if ((Drwgl_ifont >= 0)&&(Drwgl_ifont < font_tsize))
    if (pf = font_table[Drwgl_ifont]) {
      fnt = pf->fnt_ftgld;                      // A previously installed font was present get the FTFont address.
      ir = Get_String_Box( Drwgl_strdef, Drwgl_gptmax, Drwgl_valdef, &box, &boy, 0 );
    }
  Sdrw_Put_Int( ir );
  Sdrw_Put_Float( box );
  Sdrw_Put_Float( boy );
}



void DrwGL_Display_Text( Draw_Ptr p, int sel )
{ // Draw a Line of text.
  float ha, hc, xc, yc, zc;
  int                  len;
  char               * str;

//int i1, i2;  // Should be long long int on 64 bits machine.
//float f1, f2;
//i1 = (int)(&Drwgl_Attr->txt_dept); f1 = Drwgl_Attr->txt_dept;
//i2 = (int)(&Drwgl_Attr->txt_expf); f2 = Drwgl_Attr->txt_expf;
//printf( " FtMan : Dept = %f, addr = %d, test = %f, addr = %d\n", f1, i1, f2, i2 );
//fprintf( fmsg, " with Haln = %d, Valn = %d, Path = %d\n",
//               Drwgl_Attr->txt_haln, Drwgl_Attr->txt_valn, Drwgl_Attr->txt_path );
//fflush( fmsg );

  curr_font = (Draw_Font_Ptr) Drwgl_Attr->txt_font;     // Get the current font record address.

  if (p&&curr_font) {
    xc     =      p->txt.txt_x;                         // Get all the string drawing parameters.
    yc     =      p->txt.txt_y;
    zc     =               0.0;
    hc     =   p->txt.txt_high;
    ha     =    p->txt.txt_ang;
    len    =    p->txt.txt_len;
    str    =    p->txt.txt_str;
    valn   = Drwgl_Attr->txt_valn;
    haln   = Drwgl_Attr->txt_haln;
    fnt    = curr_font->fnt_ftgld;
    if (fnt) {
      if (Drwgl_Attr->txt_path == TXT_PATH_RIGHT) {
        user_spacing = FTPoint( Drwgl_Attr->txt_spcf, 0.0, 0.0 );
        Display_Text_Basic( xc, yc, 0.0, hc, ha, str, len, sel );
      }
      else Display_Text_Exotic( xc, yc, 0.0, hc, ha, str, len, sel );
    }
  }
}



void DrwGL_Display_Text_Box( Draw_Ptr p, int sel )
{
  Draw_Font_Ptr                                     pf;
  int                               hal, len, ifc, jfc;
  char                                           * txt;
  float   x1, y1, z1, x2, y2, z2, xc, yc, zc, rp, high,
    box, boy, txt_zshf, txt_width, txt_high, txt_depth;


  pf   = (Draw_Font_Ptr) Drwgl_Attr->txt_font;  // Get the Font Descriptor address.

  if (p&&pf) {
    hal  =     p->tbox.txb_hal;
    xc   =       p->tbox.txb_x;
    yc   =       p->tbox.txb_y;
    zc   =                 0.0;
    box  =     p->tbox.txb_box;
    boy  =     p->tbox.txb_boy;
    high =    p->tbox.txb_high;
    len  =     p->tbox.txb_len;
    txt  =     p->tbox.txb_txt;
    fnt    =     pf->fnt_ftgld;
    valn   = Drwgl_Attr->txt_valn;
    haln   = Drwgl_Attr->txt_haln;

//fprintf( fmsg, " GL Text plot at (%f,%f), high = %f, len = %d, s = \"%s\" .\n", xc, yc, high, len, txt );
//fflush( fmsg );

  } else txt = 0;

  if (txt) {
	txt_depth = 0.0;
    // Get the size of one window pixel in the user space coordinates.
    Scr_To_Usr( 0.0, 0.0, 0.0, &x1, &y1, &z1 );
    Scr_To_Usr( 1.0, 0.0, 0.0, &x2, &y2, &z2 );
    x2 -= x1; y2 -= y1; z2 -= z1;
    ifc = (int)(high/sqrt( x2*x2 + y2*y2 + z2*z2 ) + 0.5);      // Get the character high in pixel.
    if (ifc < 1) ifc = 1;
    slayout.SetFont( fnt );                     // Attach the font to layout.
    fnt->FaceSize( ifc );                       // Set the required FaceSize.
    rp = high/ifc;                              // Keep the Scale Ratio between the user space and the Face (Glyphe) space (in user_unit/(1/72")).
    if (rp<1.0E-4) rp = 1.0E-4;
    if (pf->fnt_style == FTGL_EXTRUDE) {        // Set the Font Depth for extruded font.
      txt_depth = Drwgl_Attr->txt_dept/rp;
      fnt->Depth( txt_depth );
    }
    slayout.SetLineLength( box/rp );            // Set the User specified line length.
    slayout.SetLineSpacing( 1.0 );              // Set the Line spacing.
    bbox = slayout.BBox( txt );                 // Get the Resulting Box sizes.
    txt_high  = bbox.Upper().Yf() - bbox.Lower().Yf();
    if (boy <= 0.0) boy = txt_high;
    if (txt_high > boy/rp) {                    // When the box is too small, we adapt the character size.
      if (rp<1.0E-4) rp = 1.0E-4;
      jfc = (int)(ifc*boy/(rp*txt_high) + 0.5); // Adapt the FaceSize.
      if (jfc < 1) jfc = 1;
      fnt->FaceSize( jfc );                     // Set the new required FaceSize.
      rp = (rp*jfc)/ifc;                        // Compute the new scale (in User_Unit/(1/72").
      if (pf->fnt_style == FTGL_EXTRUDE) {      // Re-Set the Font Depth for extruded font with new scale.
        txt_depth = Drwgl_Attr->txt_dept/rp;
        fnt->Depth( txt_depth );
      }
//    slayout.SetFont( fnt );                   // Attach the font to layout.
      slayout.SetLineLength( box/rp );          // Set the User specified line length.
      bbox = slayout.BBox( txt );               // Get the Resulting Box sizes. ?
      txt_high = bbox.Upper().Yf() - bbox.Lower().Yf(); // Get the new height of the box.
    }
    txt_width = bbox.Upper().Xf() - bbox.Lower().Xf(); // Get the Width of the box.


    // In the current space, 1/72" of font is corresponding to 1 window Pixel
    // to optimize the rendering of glyphes (when the rotations are not using - possible with vectorial fonts).


    if (pf->fnt_style <= FTGL_PIXMAP) {         // Pixel Fonts :
      Usr_To_Scr( xc, yc, zc, &x1, &y1, &z1 );  // Convert (xc,yc,zc) to Screen Coordinates.
      // Adjust the string origine with the user alignement requirements.
      txt_zshf = 0.5*boy/rp;
      switch (haln) {
        case TXT_HALN_RIGHT:  x1 -=     txt_width; break;
        case TXT_HALN_CENTRE: x1 -= 0.5*txt_width; break;
        default: /* TXT_HALN_NORMAL, TXT_HALN_LEFT, TXT_HALN_JUSTIFY */ break;
      }
      switch (valn) {
        case TXT_VALN_CAP:
        case TXT_VALN_TOP:  y1 -=  txt_zshf; break;
        case TXT_VALN_HALF: break;
        default: /* TXT_BOTTOM, TXT_VALN_NORMAL, TXT_VALN_BASE */
          y1 += txt_zshf;
        break;
      }
    } else {                                    // For Vectorial Fonts :
        txt_width *= /* Drwgl_txt_expf* */ rp;
        txt_high  *= rp; txt_depth *= rp;
        txt_zshf = 0.5*boy;
        // Adjust the string origine with the user alignement requirements.
        switch (haln) {
          case TXT_HALN_RIGHT:  xc -=     txt_width; break;
          case TXT_HALN_CENTRE: xc -= 0.5*txt_width; break;
          default: /* TXT_HALN_NORMAL, TXT_HALN_LEFT, TXT_HALN_JUSTIFY */ break;
        }
        switch (valn) {
          case TXT_VALN_CAP:
          case TXT_VALN_TOP:  yc -= txt_zshf; break;
          case TXT_VALN_HALF: break;
          default: /*  TXT_BOTTOM, TXT_VALN_NORMAL, TXT_VALN_BASE */
            yc += txt_zshf;
          break;
        }
      }

    switch (hal) {
      case TXT_HALN_CENTRE:  slayout.SetAlignment( FTGL::ALIGN_CENTER );  break;
      case TXT_HALN_RIGHT:   slayout.SetAlignment( FTGL::ALIGN_RIGHT );   break;
      case TXT_HALN_JUSTIFY: slayout.SetAlignment( FTGL::ALIGN_JUSTIFY ); break;
      default: /* TXT_HALN_NORMAL, TXT_HALN_LEFT */
        slayout.SetAlignment( FTGL::ALIGN_LEFT );
      break;
    }

//  fprintf( fmsg, " TBOX = ( %f * %f * %f ), FaceSize = %d,\n",
//                 txt_width, txt_high, txt_depth, ifc );
//  fprintf( fmsg, " At (%f,%f) the string \"%s\".\n", xc, yc, txt );
//  fflush( fmsg );


    if (sel) { // Selection Mode.
      if (pf->fnt_style <= FTGL_PIXMAP) {       // Pixel Fonts :
        x2 = x1 + txt_width;                    // Set the box finish vertex.
        y2 = y1 + txt_zshf; y1 -= txt_zshf;
        DrwGL_ScrXYZ( (GLdouble) x1, (GLdouble) y1, (GLdouble) z1, 1 );
        DrwGL_ScrXYZ( (GLdouble) x2, (GLdouble) y1, (GLdouble) z1, 0 );
        DrwGL_ScrXYZ( (GLdouble) x2, (GLdouble) y2, (GLdouble) z1, 0 );
        DrwGL_ScrXYZ( (GLdouble) x1, (GLdouble) y2, (GLdouble) z1, 0 );
      } else {                                  // For Vectorial Fonts :
        glPushMatrix();
        glTranslatef( xc, yc - txt_zshf, 0.0 );
        glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
        DrwGL_UsrXYZ( (GLdouble)       0.0, (GLdouble) 0.0, (GLdouble) 0.0, 1 );
        DrwGL_UsrXYZ( (GLdouble) txt_width, (GLdouble) 0.0, (GLdouble) 0.0, 0 );
        DrwGL_UsrXYZ( (GLdouble) txt_width, (GLdouble) boy, (GLdouble) 0.0, 0 );
        DrwGL_UsrXYZ( (GLdouble)       0.0, (GLdouble) boy, (GLdouble) 0.0, 0 );
        if (txt_depth != 0.0) {
          DrwGL_UsrXYZ( (GLdouble)       0.0, (GLdouble) 0.0, (GLdouble) txt_depth, 0 );
          DrwGL_UsrXYZ( (GLdouble) txt_width, (GLdouble) 0.0, (GLdouble) txt_depth, 0 );
          DrwGL_UsrXYZ( (GLdouble) txt_width, (GLdouble) boy, (GLdouble) txt_depth, 0 );
          DrwGL_UsrXYZ( (GLdouble)       0.0, (GLdouble) boy, (GLdouble) txt_depth, 0 );
        }
        glPopMatrix();
        glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
      }
//    fprintf( fmsg, " Pick area (cm) = (%f,%f,%f,%f)\n", xc, yc, x2, y2 );
//    fprintf( fmsg, " Pick area (pixel) = (%f,%f,%f,%f)\n", Drwh_mix, Drwh_miy, Drwh_max, Drwh_may );
//    fflush( fmsg );

      if ((Drws_max >= Drwh_mix)&&(Drws_mix <= Drwh_max)&&
          (Drws_may >= Drwh_miy)&&(Drws_miy <= Drwh_may))  Hits = 1;

    } else { // Render Mode

      if (pf->fnt_style <= FTGL_PIXMAP) {
//      y1 += txt_high;
        Scr_To_Usr( x1, y1, z1, &xc, &yc, &zc );
        glRasterPos3f( xc, yc, zc );
        slayout.Render( txt );
      } else {
        glPushMatrix();
        glTranslatef( xc, yc, 0.0 );
        glScalef( rp, rp, rp );
        slayout.Render( txt );
        glPopMatrix();
      }
    }

//  fprintf( fmsg, " Text Attr: haln = %d  valn = %d  path = %d\n",
//                 Drwgl_txt_haln, Drwgl_txt_valn, Drwgl_txt_path );
//  fprintf( fmsg, " exp = %f, spc = %f\n", Drwgl_txt_expf, Drwgl_txt_spcf );
//  fflush( fmsg );

  }
}



static int Draw_Set_TT_Font( char * name, int style, int ifnt )
{ // Set the specified True Type font in the specified mode to be used.
  //
  // name     is the file specification of the True-type_2 font file.
  //
  //            When the name is not begining by "/", "\" or "." the font file is search in the
  //            font specific path. The default path is specified by the parameter FONT_PATH of
  //            the server setting file
  //
  // style    is the type of use for the specified font as this :
  //
  //            Two mode to get Pixel Map Fonts :
  //                FTGL_BITMAP   or  1  to use as a BITMAP font, 1 bit for each Pixel.
  //                FTGL_PIXMAP   or  2  to use as a PIXMAP font, 3 bytes for each Pixel,
  //
  //            Two mode to get Vectorial Fonts that can be oriented by matrix and segment processing :
  //                FTGL_OUTLINE  or  3  to use as a wire font,
  //                FTGL_POLYGON  or  4  to use as font with surface.
  //                FTGL_EXTRUDE  or  5  to use as font with thickness,
  //                FTGL_TEXTURE  or  6  to generate each glyphe (character) as a texture.
  //
  // All other values of style are equivalent to the default FTGL_DEF_STYLE = FTGL_POLYGON.
  //
  char        buf[256];
  char           * fnm;
  static   FTFont* fnt;
  Draw_Font_Ptr      p;

  if (p = font_table[ifnt]) {                   // A previously installed font was present.
    if (p->fnt_ftgld) p->fnt_ftgld->~FTFont();
    if (p->fnt_fname) free( p->fnt_fname );
    free( p );
    font_table[ifnt] = 0;
    p = 0;
  }

  if (draw_fontpath&&draw_fontpath[0]&&name&&name[0]) {
    if (Sdrw_Search_In_Path( draw_fontpath, name, buf ) < 0) {
      fprintf( fmsg, " Cannot find the font File \"%s\" in the path \"%s\".\n", name, draw_fontpath );
      fflush( fmsg );
      return -1;
    }
  } else {
    fprintf( fmsg, " Undefined Font Fath or Font Name.\n" );
    fflush( fmsg );
    return -1;
  }

//sprintf( buf, "%s/%s", draw_fontpath, name );

  if ((style < FTGL_BITMAP)||(style > FTGL_TEXTURE)) style = FTGL_DEF_STYLE;

  switch (style) {
    case FTGL_BITMAP:  fnt = new FTBitmapFont(  buf ); break;
    case FTGL_PIXMAP:  fnt = new FTPixmapFont(  buf ); break;
    case FTGL_OUTLINE: fnt = new FTOutlineFont( buf ); break;
    case FTGL_POLYGON: fnt = new FTPolygonFont( buf ); break;
    case FTGL_EXTRUDE: fnt = new FTExtrudeFont( buf ); break;
    case FTGL_TEXTURE: fnt = new FTTextureFont( buf ); break;
  }

  if (fnt->Error()) {
    fprintf( fmsg, " Failed to open font \"%s\"\n", buf );
    fprintf( fmsg, " --> Font \"%s\" Not Loaded **** .\n", buf );
    fflush( fmsg );
    fnt->~FTFont();
    return -3;
  } else {
    fnt->FaceSize( 72 );
    fnt->Depth( 24.0 );
    fnt->CharMap(ft_encoding_unicode);          // Handle the Unicode character (UTF8).

    fprintf( fmsg, " --> Font #%d of name \"%s\" is Loaded in mode %d\n", ifnt, buf, style );
    fflush( fmsg );

    p = (Draw_Font_Ptr) malloc( sizeof( Draw_Font_Dsc ) );

    fnm = (char*)malloc( strlen( buf ) + 1 );   // Make a local copy of the Font File path.
    strcpy( fnm, buf );

    p->fnt_style       =         style;         // Keep the font style.
    p->fnt_fname       =           fnm;         // Set the Font File Path.
    p->fnt_ftgld       =           fnt;         // Set the FTGL font descriptor pointer */
    font_table[ifnt]   =             p;         // Set the Font pointer in the installed font table.
    return ifnt;
  }
}



static void Extend_Font_Table( int nreq )
{ // Extend the font table to the required size.
  Draw_Font_Ptr * p;
  int       ii, nsz;

  nsz = FONT_ALLOC_UNIT*( (nreq + FONT_ALLOC_UNIT-1)/FONT_ALLOC_UNIT);  // Get the minimum sie in portion of FONT_ALLOC_UNIT elements.
  p = (Draw_Font_Ptr *) malloc( nsz*sizeof( Draw_Font_Ptr ) );          // Allocate the new Font Table.
  for(ii=0; ii<font_tsize; ii++) p[ii] = font_table[ii];                // Copy the old Font Table to the new one.
  for(ii=font_tsize; ii<nsz; ii++) p[ii] = 0;                           // Complete by NULL pointers.
  free( font_table );                                   // free the old Font Table and ...
  font_table = p;                                       // ... install the new one.
  font_tsize = nsz;
}



static int Draw_Load_Predefined_Font( int iftn )
{ /* Load a Predefined Font from its File (via getenv name) */
  int ir;

  if ((iftn < 0)||(iftn >= SIZE_DEF_FONT)) return -1;   // Return on illegal Font Number.

  if (iftn >= font_tsize) Extend_Font_Table( iftn );    // Extend the font table to the required size.

  ir = Draw_Set_TT_Font( draw_fontnamtb[iftn], draw_fontstyle[iftn], iftn );    // Install the Predefined Font.
  return ir;
}



int Draw_Check_Font( int iftn )
{ // To verify the availability of the specified font.
  int rldf;

  rldf = iftn&FNT_RELOAD;
  iftn &= ~FNT_RELOAD;

  if (iftn < 0) return -1;

  if (iftn >= font_tsize) return -1;                    // Return on illegal Font Number.
  if (iftn >= SIZE_DEF_FONT) {                          // For user setting Font.
    if (font_table[iftn]) return iftn;                  // When the font is installed, it is OK.
                     else return -1;                    // Font not installed => No action.
  } else {                                              // For initial setting fonts.
    if ((!rldf)&&font_table[iftn]) return iftn;         // Font is ready to be selected and must not be reloaded.
    return Draw_Load_Predefined_Font( iftn );           // else return the success of Font Load and Install.
  }
}



int Draw_Install_Font( int iftn, int style, int len, char* name )
{ // To make the true-type font file <name> available as the font # ftnb
  // in the stype specified by style.

  iftn &= ~FNT_RELOAD;                                  // Suppress the reload bit.

  if ((iftn < 0)||(iftn >= MAX_FONT_NUMBER)) return -1; // Return on illegal Font Number.

  if (name&&name[0]) {
    if (iftn >= font_tsize) Extend_Font_Table( iftn );  // Extend the font table to the required size if necessary.
    return Draw_Set_TT_Font( name, style, iftn );       // Install the Requested Font.
  } else { // No name but, can reload a standard font.
    if (iftn < SIZE_DEF_FONT) return Draw_Load_Predefined_Font( iftn );
                         else return -1;
  }
}



void DrwGL_Font_Selection( int iftn )
{ // To make the specified font as the current font Executed by DrawGL module.
  if ((iftn >= 0)&&(iftn < font_tsize))
    Drwgl_Attr->txt_font = (void *) font_table[iftn];   // Try to select the user requested font.
  if (!(Drwgl_Attr->txt_font))
    Drwgl_Attr->txt_font = (void *) font_table[0];      // On selection failure, select the # 0 font (if defined).

//fprintf( fmsg, " Select font # %d\n", iftn );
//fflush( fmsg );
}




int Draw_Font_Info( int ifnt, int * stl, char ** fnm )
{ // To return the Font filename and the font style.
  if ((ifnt>=0)&&(ifnt<font_tsize)&&font_table[ifnt]) {
    *stl = font_table[ifnt]->fnt_style;                 // Return the font style.
    *fnm = font_table[ifnt]->fnt_fname;                 // Return the font file name.
    return ifnt;
  } else {                                              // One negative Font # we return ...
    *stl =            -1;                               // ... an invalid style value and ...
    *fnm = draw_fontpath;                               // ... the font path (from the Server Setup file).
	return -1;
  }
}



void DrwGL_Font_Support_Init( void )
{ // To initialize the FTGL font support.
  int i;
/*
  Draw_first_font = 0;
  Draw_last_font  = 0;
  Draw_curr_font  = 0;
*/
  // Initialize the font table.
  font_table = (Draw_Font_Ptr *) malloc( FONT_ALLOC_UNIT*sizeof( Draw_Font_Ptr ) );
  font_tsize = FONT_ALLOC_UNIT;
  for(i=0; i<FONT_ALLOC_UNIT; i++) font_table[i] = 0;

  fprintf( fmsg, " Font Path = \"%s\"\n", draw_fontpath );
  for (i=0; i<8; i++) fprintf( fmsg, "   Font #%d name is \"%s\"\n", i, draw_fontnamtb[i] );
  fflush( fmsg );

}


