/********* DRAW Interface ROUTINES for OpenGL Graphic Libary ***********/
/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*          F L T K / D R A W   -   S E R V E R   M O D U L E            *
*                                                                       *
*           Version  1.4-C for Draw Library Level V 2.4 A               *
*                                                                       *
*      (Draw interface for FLTK/OPENGL function and procedures)         *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/



/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/


/* debug mode Undefined => no, Define => yes */

#define _No_GL_Error
//#define _GL_Debug 0
#ifdef _GL_Debug
#  define _GL_FullDebug 0
#endif

/* include the environment SERVER */

#define _DRAWGL_DEF_GLOBAL 0


#include "Server_GL.h"



/***********************************************************************/
/*                                                                     */
/*                    DRAW/GL Interface Routines                       */
/*                                                                     */
/***********************************************************************/



#define Hits_Bufsz             128
#define Hits_Preci              10

# define inrd (3.1415927/180.0)



/***********************************************************************/
/*                         Local  Variable                             */
/***********************************************************************/

 
typedef struct {  int   seg;
                  int   pid;
                  float prio;
                  float xmin, xmax,
                        ymin, ymax,
                        zmin, zmax;
                 } Hitst;


static Hitst     HitsHeap[Hits_Bufsz];
static int       HitsSp = -1;

static int       Tmat_Cnt = 0;           /* Count of Push Trsf. Matrix */

static GLfloat   svd_mat[16];            /* Saved Matrix for view input interaction */

static GLfloat   MinZ_Dist, MaxZ_Dist,   /* Minimum and Maximum Screen depth */
                 Zdepth,         Zorg;   /* Total Z Screen Depth size, Z Org */

static GLUquadricObj *QuaObj;            /* Common Quadric Structure Pointer */

/* GL Name of each Light */
static int       lightst[LIGHT_NUMBER];  /* Table of Light state */

static int       lighting   = 0;

static int       sup_pickid = 0;         /* shift of Pickid for exec seg */

static int       rst_view;               /* Flag to restore initial view at end of View Orientation Request */


static GLfloat def_lmod_two_side[1]     = { 0.0 };
static GLfloat def_lmod_ambient[4]      = { 0.2, 0.2, 0.2, 1.0 };
static GLfloat def_lmod_loc_view[1]     = { 0.0 };

static GLfloat lmod_two_side[1];
static GLfloat lmod_ambient[4];
static GLfloat lmod_loc_view[1];

static GLint Plot_Code[] = { GL_POINTS,         /*  0  for  GL Codes */
                             GL_LINES,          /* -1 */
                             GL_LINE_STRIP,     /* -2, Use by draw_out 1 */
                             GL_LINE_LOOP,      /* -3 */
                             GL_TRIANGLES,      /* -4 */
                             GL_TRIANGLE_STRIP, /* -5 */
                             GL_TRIANGLE_FAN,   /* -6 */
                             GL_QUADS,          /* -7 */
                             GL_QUAD_STRIP,     /* -8 */
                             GL_POLYGON         /* -9, Use by draw_out 3 */
                           };



/***********************************************************************/
/*                 Call Back and GL error  Routines                    */
/***********************************************************************/

//void Drwgl_Error_CallBack( GLenum errcode )
void Drwgl_Error_CallBack()
{
  GLenum   code = glGetError();
  const GLubyte        *errmsg;

  errmsg = gluErrorString( code );
  fprintf( fmsg, " Draw Server : *** GLU Error on Quadric : %s\n", errmsg );
  fflush( fmsg );
}



void DrwGL_Error( const char* pid )
{
  GLenum                  code;
  const GLubyte        *errmsg;

  while ((code = glGetError()) != GL_NO_ERROR) {
    errmsg = gluErrorString( code );
    if (pid) fprintf( fmsg, " Draw Server %s : *** GL Error : %s\n", pid, errmsg );
        else fprintf( fmsg, " Draw Server : *** GL Error : %s\n", errmsg );
    fflush( fmsg );
  }
}



/***********************************************************************/
/*                          Init  Routine                              */
/***********************************************************************/

void DrwGL_Prev_Init()
{ /* First step Initialization of Draw/GL Interface */
  int i, j;

  /* Set Saved Color Tables */
  for (i=0;i<16;i++)
    for (j=0;j<4;j++)
      draw_savsrcctb[i][j] = draw_srccoltb[i][j];

  Drwgl_print        =       0;  /* Set The Screen output mode */
  Drwgl_mactive      =       0;  /* Set mouse as inactive */

  Drwgl_astpmod      =       0;  /* Default to no Animation Stop Validation */
  Drwgl_astpstate    =       0;  /* Init the Stop Flag */

  Drwgl_unitx    = (char*) " ";  /* Set the default unit cm */
  Drwgl_unity    = (char*) " ";

  Drwgl_xusr_unt[0]  =       0;  /* Set the user units to undefined */
  Drwgl_yusr_unt[0]  =       0;

  Drwgl_inpmdflg     =       0;
  Drwgl_xlogflg      =       0;  /* Set Linear Coordinates */
  Drwgl_ylogflg      =       0;

  Drwgl_clipflg      =       0;  /* Set as No Pick Echo On. */
  Drwgl_segment      =       0;
  Drwgl_ident        =       0;

  /* Initialize all attributs */

  Drwgl_Attr->lin_type  =    1;  /* Line Type */
  Drwgl_Attr->mrk_type  =   -1;  /* Marker Type */
  Drwgl_Attr->fil_type  =    1;  /* Fill Type */
  Drwgl_Attr->fil_styl  =    0;  /* Fill style */
  Drwgl_Attr->txt_haln  =    1;  /* Text Horizontal Alignment */
  Drwgl_Attr->txt_valn  =    1;  /* Text Vertical Alignment */
  Drwgl_Attr->txt_path  =    1;  /* Text Path */
  Drwgl_Attr->txt_font  =    0;  /* CFont to use, (No font loaded at start time) */
  Drwgl_Attr->lgh_stat  =    0;  /* By Default, Lighting is Off */
  Drwgl_Attr->lin_size  =  1.0;  /* Line size */
  Drwgl_Attr->mrk_size  =  1.0;  /* Marker size */
  Drwgl_Attr->txt_expf  =  1.0;  /* Character width expanssion factor */
  Drwgl_Attr->txt_spcf  =  0.0;  /* Character Spacing modificator */
  Drwgl_Attr->txt_dept  =  0.5;  /* Extruded Character depth */
  for(i=0; i<3; i++) Drwgl_Attr->col_rgba[i] = draw_srccoltb[1][i]; /* Set default color # 1 (std color #1 is black) */


  Drwgl_line_stipple =       0;  /* Init Line Stipple Flag */
  Drwgl_fill_stipple =       0;  /* Init Fill Area Stipple Flag */
  Drwgl_fill_front   =       0;  /* Init Flag for front face fill mode */
  Drwgl_fill_back    =       0;  /* Init Flag for back face fill mode */
  Drwgl_fill_Shift   =       0;  /* Init Flag for polygon depth shift */

  Drwgl_zoomview     =       0;  /* Set implicite user action as Zoom */
}



void DrwGL_Init()
{ /* Second step Initialization of the Draw/GL Interface */
  GLfloat def_black[] = { 0.0, 0.0, 0.0, 1.0 };
  GLfloat def_white[] = { 1.0, 1.0, 1.0, 1.0 };

  int i, j, k, l;
  GLenum lig;

  glClearColor( draw_srccoltb[0][0], draw_srccoltb[0][1],
                draw_srccoltb[0][2], draw_srccoltb[0][3] );
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
  glEnable( GL_DEPTH_TEST);

  lmod_two_side[0]     = def_lmod_two_side[0];
  lmod_ambient[0]      = def_lmod_ambient[0];
  lmod_ambient[1]      = def_lmod_ambient[1];
  lmod_ambient[2]      = def_lmod_ambient[2];
  lmod_ambient[3]      = def_lmod_ambient[3];
  lmod_loc_view[0]     = def_lmod_loc_view[0];

  for (i = 0; i < LIGHT_NUMBER; i++)
    lightst[i] = Drwgl_light_entb[i] = 0;
  if (Drwgl_usr_request&SERVER_DO_LIGHTING) {
    lig = GL_LIGHT0;
    l = 0;
    glEnable( GL_LIGHTING );
    Drwgl_Attr->lgh_stat = 1;
    glLightModelfv( GL_LIGHT_MODEL_AMBIENT, lmod_ambient );
    glLightModelfv( GL_LIGHT_MODEL_LOCAL_VIEWER, lmod_loc_view );
    glLightModelfv( GL_LIGHT_MODEL_TWO_SIDE, lmod_two_side );

    j = SERVER_ENA_LIGHT0;
    /* Always One Light is enabled ( the light 0 ) */
    /* Each array Drwgl_light_entb[light#] flag the light that must be On or Off */
    /* Each array lightst[light#] flag the light that is used */

    for (i = 0; i < LIGHT_NUMBER; i++) {
      Drwgl_light_entb[i] = lightst[i] = (k = (Drwgl_usr_request&j||(!i))?1:0);
      if (k||(i==0)) {
        glEnable( lig );
        glLightfv( lig, GL_AMBIENT, def_black );
        glLightfv( lig, GL_SPECULAR, def_black );
        glLightfv( lig, GL_DIFFUSE, (lig!=GL_LIGHT0)?def_black:def_white );
      }
      j *= 2; lig++;
    }
    lighting   = 1;
  }
  glMatrixMode( GL_PROJECTION );
  glLoadIdentity();
  glViewport( 0, 0, s_dt_rx, s_dt_ry );
  glGetIntegerv( GL_VIEWPORT, Drwgl_wrdvport );
  glOrtho( (GLdouble) dw_cxmin, (GLdouble) dw_cxmax,
           (GLdouble) dw_cymin, (GLdouble) dw_cymax, 1.0, -1.0 );
  glGetDoublev( GL_PROJECTION_MATRIX, Drwgl_proj_wrd );


  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_mat_unit );
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_inpmat );

  QuaObj = gluNewQuadric();
  gluQuadricCallback( QuaObj, GLU_ERROR, (_GLUfuncptr) Drwgl_Error_CallBack );

//fprintf( fmsg, "  Init GL Viewport Size (%d*%d) .\n", s_dt_rx, s_dt_ry );
//fprintf( fmsg, "  Init GL Window Size (%f*%f) .\n", ds_rx, ds_ry );
//fflush( fmsg );
}



void DrwGL_Alpha_Set( int flagw, GLenum src, GLenum dst )
{ /* Enable or Disable Blending */
  if (flagw&DRWGL_BLD_ENA)  glEnable( GL_BLEND );
  if (flagw&DRWGL_BLD_DDIS) glDepthMask( GL_FALSE );
  if (flagw&DRWGL_BLD_DENA) glDepthMask( GL_TRUE );
  glBlendFunc( src, dst );
  if (flagw&DRWGL_BLD_DIS)  glDisable( GL_BLEND );

//fprintf( fmsg, " Alpha Blend On:Off = %d:%d.\n",
//               flagw&DRWGL_BLD_ENA, flagw&DRWGL_BLD_DIS );
//fprintf( fmsg, " Alpha D Mask On:Off = %d:%d.\n",
//               flagw&DRWGL_BLD_DENA, flagw&DRWGL_BLD_DDIS );
//fflush( fmsg );
}




/***********************************************************************/
/*                      Basic   Pick   Routines                        */
/***********************************************************************/


#if 0
static void Drw_Local_Trans( GLfloat * m, float *x, float *y, float *z )
{ float wx, wy;

  wx = m[ 0]*(*x) + m[ 4]*(*y) + m[ 8]*(*z) + m[12];
  wy = m[ 1]*(*x) + m[ 5]*(*y) + m[ 9]*(*z) + m[13];
  *z = m[ 2]*(*x) + m[ 6]*(*y) + m[10]*(*z) + m[14];
  *x = wx; *y = wy;
}
#endif



void DrwHVertex( float x, float y, float z )
{
  GLdouble wx, wy, wz;

//fprintf( fmsg, " V3F point = (%5.1f,%5.1f,%5.1f)\n", x, y, z );

  /* Perform all Geometrical Transformations */

  if (gluProject( (GLdouble) x, (GLdouble) y, (GLdouble) z,
                   Drwgl_modlview, Drwgl_proj_wrd, Drwgl_wrdvport,
                   &wx, &wy, &wz ) == GL_TRUE) {
      /* Set for Echo Area */
      if (Drwh_mix > wx) Drwh_mix = wx; if (Drwh_max < wx) Drwh_max = wx;
      if (Drwh_miy > wy) Drwh_miy = wy; if (Drwh_may < wy) Drwh_may = wy;
      if (Drwh_miz > wz) Drwh_miz = wz; if (Drwh_maz < wz) Drwh_maz = wz;
      /* Set for Directive Hit */
      if ((wx >= Drws_mix)&&(wx <= Drws_max)&&(wy >= Drws_miy)&&(wy <= Drws_may)) {
        Hits = 1;
//      Hitx = wx; Hity = wy; Hitz = wz;
//      fprintf( fmsg, " V3F Hits at (%4.0f,%4.0f,%4.0f).\n", wx, wy, wz );
      }
    }
//fflush( fmsg );
}





/***********************************************************************/
/*                       Basic  Plot  Routines                         */
/***********************************************************************/


void DrwGL_ColorRGB( Draw_RGBA rgba )
{ /* Set a Color by predefined index */
  int i;

  if (lighting) glMaterialfv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, rgba );
           else glColor4fv( rgba );
  for (i=0;i<4;i++) Drwgl_Attr->col_rgba[i] = rgba[i];
}




void DrwGL_Save_Chcntx( float sz )
{
  Drwgl_stipple_state = glIsEnabled( GL_LINE_STIPPLE );
  glGetFloatv( GL_LINE_WIDTH, &(Drwgl_Attr->lin_size) );
  glDisable( GL_LINE_STIPPLE );
  glLineWidth( (GLfloat) sz*d_mult_linesize );
}




void DrwGL_Restore_Chcntx()
{
  if (Drwgl_stipple_state) glEnable( GL_LINE_STIPPLE );
  glLineWidth( (GLfloat) Drwgl_Attr->lin_size*d_mult_linesize );
}




void Show_Vwp( const char * nam, const int * vw )
{
  fprintf( fmsg, "\n ViewPort %s = [%d,%d,%d,%d]\n",
                 nam, vw[0], vw[1], vw[2], vw[3] );
  fprintf( fmsg, "\n" );
  fflush( fmsg );
}




void Show_clplan(  const char * nam, const GLdouble * pl )
{
  fprintf( fmsg, "\n Clip Plane %s = [ %f, %f, %f, %f ]\n",
                 nam, pl[0], pl[1], pl[2], pl[3] );
  fprintf( fmsg, "\n" );
  fflush( fmsg );
}



void Show_Mat( const char * nam, const GLdouble * m )
{
  fprintf( fmsg, "\n Matrix %s\n", nam );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 0], m[ 4], m[ 8], m[12] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 1], m[ 5], m[ 9], m[13] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 2], m[ 6], m[10], m[14] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 3], m[ 7], m[11], m[15] );
  fprintf( fmsg, "\n" );
  fflush( fmsg );
}





void Show_Matf( const char * nam, const GLfloat * m )
{
  fprintf( fmsg, "\n Matrix %s\n", nam );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 0], m[ 4], m[ 8], m[12] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 1], m[ 5], m[ 9], m[13] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 2], m[ 6], m[10], m[14] );
  fprintf( fmsg, "  [ %f, %f, %f, %f ]\n", m[ 3], m[ 7], m[11], m[15] );
  fprintf( fmsg, "\n" );
  fflush( fmsg );
}



void DrwGL_Line_Attr( int linetype, float sz )
{ /* Set Line Attribute */
  static GLushort pattern[] = { 0x3F3F,   /* Dashed */
                                0xAAAA,   /* Dotted */
                                0x5F5F,   /* Dashed_Dotted */
                                0x54A9,   /* Triple_Dot */
                                0x5050,   /* Double_Dot */
                                0x1010,   /* Spaced_Dot */
                                0x007F,   /* Spaced_Dash */
                                0x1C3F,   /* Long_Short */
                                0x3FFC,   /* Long_Dash */
                                0x55FF,   /* Dash_3_Dot */
                                0x57FF    /* Dash_2_Dot */
                              };
  static GLint factor[]     = { 2,   /* Dashed */
                                2,   /* Dotted */
                                2,   /* Dashed_Dotted */
                                2,   /* Triple_Dot */
                                2,   /* Double_Dot */
                                2,   /* Spaced_Dot */
                                1,   /* Spaced_Dash */
                                1,   /* Long_Short */
                                1,   /* Long_Dash */
                                2,   /* Dash_3_Dot */
                                2    /* Dash_2_Dot */
                              };

  if ((linetype>1)&&(linetype<13)) {
    glLineStipple( factor[linetype-2], pattern[linetype-2] );
//  fprintf( fmsg, " Line type = %d,%d\n", factor[linetype-2], pattern[linetype-2] ); fflush( fmsg );
    Drwgl_line_stipple = 1;
  }
  else Drwgl_line_stipple = 0;
  glLineWidth( (GLfloat) sz*d_mult_linesize );
  Drwgl_Attr->lin_type = linetype;
  Drwgl_Attr->lin_size =   sz;
}




void DrwGL_Display_One_Marker( GLfloat x, GLfloat y, GLfloat z, int sel )
{ /* Display a particular Segment */
  int  i, bfill, nscan, msz;
  GLfloat mrksize, sca, xx, yy;

  static GLbyte mrkt[] = {  20,   2, -20,   0,  20,   0, /*   0 -> Cross + */
                            20,   2,   0, -20,   0,  20, /*   6 */
                             1,   0,                     /*  12 */

                            20,   2, -20,   0,  20,   0, /*  14 -> Asterisk */
                            20,   2,   0, -20,   0,  20, /*  20 */
                            20,   2, -14, -14,  14,  14, /*  26 -> Diagonal Cross */
                            20,   2, -14,  14,  14, -14, /*  32 */
                             1,   0,                     /*  38 */

                           100, -37,-100,   0, -98, -17, /*  40 -> Small Circle */
                           -94, -34, -87, -50, -77, -64, /*  46 */
                           -64, -77, -50, -87, -34, -94, /*  52 */
                           -17, -98,   0,-100,  17, -98, /*  58 */
                            34, -94,  50, -87,  64, -77, /*  64 */
                            77, -64,  87, -50,  94, -34, /*  70 */
                            98, -17, 100,   0,  98,  17, /*  76 */
                            94,  34,  87,  50,  77,  64, /*  82 */
                            64,  77,  50,  87,  34,  94, /*  88 */
                            17,  98,   0, 100, -17,  98, /*  94 */
                           -34,  94, -50,  87, -64,  77, /* 100 */
                           -77,  64, -87,  50, -94,  34, /* 106 */
                           -98,  17,   1,   0,           /* 112 */

                            20,  -4, -20,   0,   0, -20, /* 116 -> Diamond */
                            20,   0,   0,  20,   1,   0, /* 122 */

                            20,  -3,   0,   0, -20,  20, /* 128 -> Hour Glass */
                            20,  20,  20,  -3,   0,   0, /* 134 */
                           -20, -20,  20, -20,   1,   0, /* 140 */

                            20,  -3,   0,   0, -20,  20, /* 146 -> Bowtie */
                           -20, -20,  20,  -3,   0,   0, /* 152 */
                            20, -20,  20,  20,   1,   0, /* 158 */

                           100,  -4, -71, -71,  71, -71, /* 164 -> Square */
                            71,  71, -71,  71,   1,   0, /* 170 */

                           100,  -3,   0,-100,  87,  50, /* 176 -> Triangle Down */
                           -87,  50,   1,   0,           /* 182 */

                           100,  -3,   0, 100,  87, -50, /* 186 -> Triangle Up */
                           -87, -50,   1,   0            /* 192 */

                         };

  static int mrkpt[] =   {   0, /* Code      2, Offset    0, -00- Cross + */
                            14, /* Code      3, Offset   14, -01- Asterisk */
                            40, /* Code      4, Offset   40, -02- Small Circle */
                            26, /* Code      5, Offset   26, -03- Diagonal Cross */
                          -116, /* Code -13/ 6, Offset -116, -04- Solid Diamond */
                           116, /* Code -12/ 7, Offset  116, -05- Diamond */
                          -128, /* Code -11/ 8, Offset -128, -06- Solid Hour Glass */
                           128, /* Code -10/ 9, Offset  128, -07- Hour Glass */
                          -146, /* Code - 9/10, Offset -146, -08- Solid Bowtie */
                           146, /* Code - 8/11, Offset  146, -09- Bowtie */
                          -164, /* Code - 7/12, Offset -164, -10- Solid Square */
                           164, /* Code - 6/13, Offset  164, -11- Square */
                          -176, /* Code - 5/14, Offset -176, -12- Solid Triangle Down */
                           176, /* Code - 4/15, Offset  176, -13- Triangle Down */
                          -186, /* Code - 3/16, Offset -186, -14- Solid Triangle Up */
                           186, /* Code - 2/17, Offset  186, -15- Triangle Up */
                           -40  /* Code - 1/18, Offset - 40, -16- Solid Circle */
                         };


  nscan = mrkpt[Drwgl_Attr->mrk_type];
  if (nscan < 0) { nscan = - nscan; bfill = 1; }
            else bfill = 0;

  mrksize = Drwgl_Attr->mrk_size*d_mult_linesize;

  if (sel) {
    while (1) {
      sca = 0.1/(GLfloat) mrkt[nscan++];
      msz =        abs( mrkt[nscan++] );
    if (!msz) break;
      for (i = 0; i < msz; i++ ) {
        xx = mrksize*((GLfloat) mrkt[nscan++])*sca + x;
        yy = mrksize*((GLfloat) mrkt[nscan++])*sca + y;
        DrwHVertex( xx, yy, z );
      }
    }
  }
  else {
    while (1) {
      sca = 0.1/(GLfloat) mrkt[nscan++];
      msz =               mrkt[nscan++];
      if (bfill) glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );
            else glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );

    if (!msz) break;

      if (bfill) {
        glBegin( GL_POLYGON );
        msz = abs( msz );
      }
      else  if (msz > 0) glBegin( GL_LINE_STRIP );
                    else { msz = - msz; glBegin( GL_LINE_LOOP ); }

      for (i = 0; i < msz; i++ ) {
        xx = mrksize*((GLfloat) mrkt[nscan++])*sca + x;
        yy = mrksize*((GLfloat) mrkt[nscan++])*sca + y;
        glVertex3f( xx, yy, z );
      }
      glEnd();
    }
  }
}




void DrwGL_Display_Marker( int dim, int flag, float * tab, int sel )
{ /* Display a particular Segment */
  GLfloat x, y, z;
  int           i;

  if (sel) {
    Drwh_mix = Drwh_miy = Drwh_miz = 1e+30;
    Drwh_max = Drwh_may = Drwh_maz = - Drwh_mix;
  }

  if (flag&cdf_3D)
  { /* 3D plot */
    for (i = 0; i < dim; i++)
    { x = *(tab++); y = *(tab++); z = *(tab++);
        if (Drwgl_currbrf) { /* 3D Graph Mode */
        }
        DrwGL_Display_One_Marker( (GLfloat) x, (GLfloat) y, (GLfloat) z, sel );
    }
  }
  else
  { /* 2D plot */
    for (i = 0; i < dim; i++)
    { x = *(tab++); y = *(tab++);
        if (Drwgl_currbrf) { /* 2D Graph Mode */
        }
        DrwGL_Display_One_Marker( (GLfloat) x, (GLfloat) y, (GLfloat) 0.0, sel );
    }
  }
}




void DrwGL_Marker_Attr( int typ, float sz )
{ /* Set Marker Attribute */
  if ((typ < 0)&&(typ >=-13)) typ = typ + 19;

  /* Set the Constant table index for the Marker */
  if ((typ <= 1)||(typ > 18 )) typ = -1;
                          else typ = typ - 2;

  if (typ == -1) {
    glPointSize( (GLfloat) sz*d_mult_linesize ); /* GL_POINTS Mode */
  }
  Drwgl_Attr->mrk_type = typ;
  Drwgl_Attr->mrk_size = sz;

}



void DrwGL_Fill_Attr( int type, int style )
{ /* Set Fill Area Attribute */

static GLint hatch_1[] = { 0x00000000, 0x00000000, 0x00000000, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x00000000, 0x00000000, 0x00000000,
                           0x00000000, 0x00000000, 0x00000000, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x00000000, 0x00000000, 0x00000000,
                           0x00000000, 0x00000000, 0x00000000, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x00000000, 0x00000000, 0x00000000,
                           0x00000000, 0x00000000, 0x00000000, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x00000000, 0x00000000, 0x00000000
                         };

static GLint hatch_2[] = { 0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0x18181818
                         };

static GLint hatch_3[] = { 0xC1C1C1C1, 0xE0E0E0E0, 0x70707070, 0x38383838,
                           0x1C1C1C1C, 0x0E0E0E0E, 0x07070707, 0x83838383,
                           0xC1C1C1C1, 0xE0E0E0E0, 0x70707070, 0x38383838,
                           0x1C1C1C1C, 0x0E0E0E0E, 0x07070707, 0x83838383,
                           0xC1C1C1C1, 0xE0E0E0E0, 0x70707070, 0x38383838,
                           0x1C1C1C1C, 0x0E0E0E0E, 0x07070707, 0x83838383,
                           0xC1C1C1C1, 0xE0E0E0E0, 0x70707070, 0x38383838,
                           0x1C1C1C1C, 0x0E0E0E0E, 0x07070707, 0x83838383
                         };

static GLint hatch_4[] = { 0x38383838, 0x70707070, 0xE0E0E0E0, 0xC1C1C1C1,
                           0x83838383, 0x07070707, 0x0E0E0E0E, 0x1C1C1C1C,
                           0x38383838, 0x70707070, 0xE0E0E0E0, 0xC1C1C1C1,
                           0x83838383, 0x07070707, 0x0E0E0E0E, 0x1C1C1C1C,
                           0x38383838, 0x70707070, 0xE0E0E0E0, 0xC1C1C1C1,
                           0x83838383, 0x07070707, 0x0E0E0E0E, 0x1C1C1C1C,
                           0x38383838, 0x70707070, 0xE0E0E0E0, 0xC1C1C1C1,
                           0x83838383, 0x07070707, 0x0E0E0E0E, 0x1C1C1C1C
                         };

static GLint hatch_5[] = { 0x18181818, 0x18181818, 0x18181818, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x18181818, 0x18181818, 0x18181818,
                           0x18181818, 0x18181818, 0x18181818, 0xFFFFFFFF,
                           0xFFFFFFFF, 0x18181818, 0x18181818, 0x18181818
                         };

static GLint hatch_6[] = { 0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818,
                           0x81818181, 0x42424242, 0x24242424, 0x18181818
                         };


static GLint * hatch_tab[6] = { hatch_1, hatch_2, hatch_3,
                                hatch_4, hatch_5, hatch_6
                              };


//fprintf( fmsg, " Fill Attr( %d, %d )\n", kind, style );
//fflush( fmsg );

  if (style&0xFF00) { /* Depth Shift Specification */
    Drwgl_fill_Shift = style>>8;
    glPolygonOffset( (GLfloat) Drwgl_fill_Shift, 1.0 );
    style = style&0xFF;
  }

  switch (type) {
    case 0: /* Hollow/Solid Mode */
      Drwgl_fill_front = (style&1)?1:0;
      Drwgl_fill_back  = (style&2)?1:0;
      Drwgl_fill_stipple = 0;
    break;

    case 1: /* Hollow Mode => Style Ignored */
      Drwgl_fill_front   = 0;
      Drwgl_fill_back    = 0;
      Drwgl_fill_stipple = 0;
    break;

    case 2: /* Solid Mode => Style Ignored */
      Drwgl_fill_front   = 1;
      Drwgl_fill_back    = 1;
      if (style) {
        glPolygonOffset( 1.0, 1.0 );
      }
      Drwgl_fill_stipple = 0;
    break;

    case 3: /* Pattern Mode  => Hatched Mode */
    case 4: /* Hatched Mode */
      if ((style < 1)||(style > 6)) style = 1;
      Drwgl_fill_front   = 1;
      Drwgl_fill_back    = 1;
      glPolygonStipple( (GLubyte *) hatch_tab[style - 1] );
      Drwgl_fill_stipple = 1;
    break;

    default:
      Drwgl_fill_front   = 0;
      Drwgl_fill_back    = 0;
      Drwgl_fill_stipple = 0;
  }
  Drwgl_Attr->fil_type =  type;
  Drwgl_Attr->fil_styl = style;
}



void DrwGL_Text_Attr( int haln, int valn, int path, float expf, float spcf, float dept )
{ /* To set Text Attributs */
  Drwgl_Attr->txt_haln = haln;
  Drwgl_Attr->txt_valn = valn;
  Drwgl_Attr->txt_path = path;
  Drwgl_Attr->txt_expf = expf;
  Drwgl_Attr->txt_spcf = spcf;
  Drwgl_Attr->txt_dept = dept;
}





/***********************************************************************/
/*                 Segment Transformations  Routines                   */
/***********************************************************************/


void DrwGL_Mul_Mat( GLfloat m1[16], GLfloat m2[16], GLfloat m3[16] )
{ int i, j, k;
  GLfloat r;

  for (i = 0; i < 4; i++)
    for (j = 0; j < 4; j++) {
      r = 0.0;
      for (k = 0; k < 4; k++)
        r += m1[4*i + k]*m2[4*k + j];
      m3[4*i + j] = r;
    }
}



void DrwGL_Seg_Unit_Transform( GLfloat* mat )
{ int i;

  if (mat)
    for (i = 0; i < 16; i++)
      mat[i] = Drwgl_mat_unit[i];
}




void DrwGL_Move_Seg( GLfloat* mat, float tx, float ty, float tz )
{
  static float m1[16] = { 1.0, 0.0, 0.0, 0.0,
                          0.0, 1.0, 0.0, 0.0,
                          0.0, 0.0, 1.0, 0.0,
                          0.0, 0.0, 0.0, 1.0 };
  static float m2[16];
  int i;

  if (mat) {
    for (i = 0; i < 16; i++) m2[i] = mat[i];
    m1[12] = tx; m1[13] = ty; m1[14] = tz;
    DrwGL_Mul_Mat( m1, m2, mat );
  }
}





void DrwGL_URot_Mat( GLfloat m[16], float  px, float  py, float  pz,
                                    float  th, float  ph, float  an )
// (X,Y,Z) Angles (th, ph, an) Based Rotation Operartor.
{
  static float cc, ux, uy, uz, sa, ca;

//fprintf( fmsg, " URot Angle Seg th = %f, ph = %f, an = %f\n", th, ph, an );
//fflush( fmsg );

  th = inrd*th; ph = inrd*ph; an = inrd*an; cc = sin( th );
  ux = cc*cos( ph ); uy = cc*sin( ph ); uz = cos( th );
  ca = cos( an ); cc = 1.0 - ca; sa = sin( an );
  m[ 0] =  ux*ux*cc + ca;                        /* m11 */
  m[ 4] =  ux*uy*cc - uz*sa;                     /* m12 */
  m[ 8] =  ux*uz*cc + uy*sa;                     /* m13 */
  m[12] =  px - m[ 0]*px - m[ 4]*py - m[ 8]*pz;  /* m14 */
  m[ 1] =  uy*ux*cc + uz*sa;                     /* m21 */
  m[ 5] =  uy*uy*cc + ca;                        /* m22 */
  m[ 9] =  uy*uz*cc - ux*sa;                     /* m23 */
  m[13] =  py - m[ 1]*px - m[ 5]*py - m[ 9]*pz;  /* m24 */
  m[ 2] =  uz*ux*cc - uy*sa;                     /* m31 */
  m[ 6] =  uz*uy*cc + ux*sa;                     /* m32 */
  m[10] =  uz*uz*cc + ca;                        /* m33 */
  m[14] =  pz - m[ 2]*px - m[ 6]*py - m[10]*pz;  /* m34 */
  m[ 3] =  0.0;                                  /* m41 */
  m[ 7] =  0.0;                                  /* m42 */
  m[11] =  0.0;                                  /* m43 */
  m[15] =  1.0;                                  /* m44 */
}




void DrwGL_Rotate_Mat( float m[16], float  px, float  py, float  pz,
                                    float ome, float chi, float psi )
// Eulerian Angles (ome, chi, psi) Based Rotation Operartor.
{
  static float so, co, sc, cc, sp, cp;

//fprintf( fmsg, " Rotate Angle Seg ome = %f, chi = %f, phi = %f\n", ome, chi, psi );
//fflush( fmsg );

  ome = inrd*ome; chi = inrd*chi; psi = inrd*psi;
  co = cos( ome ); cc = cos( chi ); cp = cos( psi );
  so = sin( ome ); sc = sin( chi ); sp = sin( psi );
  m[ 0] =  co*cc*cp - so*sp;                     /* m11 */
  m[ 4] = -co*cc*sp - so*cp;                     /* m12 */
  m[ 8] =  co*sc;                                /* m13 */
  m[12] =  px - m[ 0]*px - m[ 4]*py - m[ 8]*pz;  /* m14 */
  m[ 1] =  so*cc*cp + co*sp;                     /* m21 */
  m[ 5] = -so*cc*sp + co*cp;                     /* m22 */
  m[ 9] =  so*sc;                                /* m23 */
  m[13] =  py - m[ 1]*px - m[ 5]*py - m[ 9]*pz;  /* m24 */
  m[ 2] = -sc*cp;                                /* m31 */
  m[ 6] =  sc*sp;                                /* m32 */
  m[10] =  cc;                                   /* m33 */
  m[14] =  pz - m[ 2]*px - m[ 6]*py - m[10]*pz;  /* m34 */
  m[ 3] =  0.0;                                  /* m41 */
  m[ 7] =  0.0;                                  /* m42 */
  m[11] =  0.0;                                  /* m43 */
  m[15] =  1.0;                                  /* m44 */
}




void DrwGL_URot_Seg( GLfloat* mat, float  px, float  py, float  pz,
                                   float  th, float  ph, float  an )
{
  static float m1[16], m2[16];
  int i;

  if (mat) {
    for (i = 0; i < 16; i++) m2[i] = mat[i];

    DrwGL_URot_Mat( m1, px, py, pz, th, ph, an );
    DrwGL_Mul_Mat( m1, m2, mat );
  }
}




void DrwGL_Rotate_Seg( GLfloat* mat, float  px, float  py, float  pz,
                                     float ome, float chi, float psi )
{
  static float m1[16], m2[16];
  int i;

  if (mat) {
    for (i = 0; i < 16; i++) m2[i] = mat[i];

    DrwGL_Rotate_Mat( m1, px, py, pz, ome, chi, psi );
    DrwGL_Mul_Mat( m1, m2, mat );
  }
}




void DrwGL_Scale_Mat( float m[16], float px, float py, float pz,
                                   float sx, float sy, float sz )
{
  m[ 0] =   sx;               /* m11 */
  m[ 4] =  0.0;               /* m12 */
  m[ 8] =  0.0;               /* m13 */
  m[12] =  px*(1.0 - m[ 0]);  /* m14 */
  m[ 1] =  0.0;               /* m21 */
  m[ 5] =   sy;               /* m22 */
  m[ 9] =  0.0;               /* m23 */
  m[13] =  py*(1.0 - m[ 5]);  /* m24 */
  m[ 2] =  0.0;               /* m31 */
  m[ 6] =  0.0;               /* m32 */
  m[10] =   sz;               /* m33 */
  m[14] =  pz*(1.0 - m[10]);  /* m34 */
  m[ 3] =  0.0;               /* m41 */
  m[ 7] =  0.0;               /* m42 */
  m[11] =  0.0;               /* m43 */
  m[15] =  1.0;               /* m44 */
}




void DrwGL_Scale_Seg( GLfloat* mat, float px, float py, float pz,
                                    float sx, float sy, float sz )
{
  static float m1[16], m2[16];
  int i;

  if (mat) {
    for (i = 0; i < 16; i++) m2[i] = mat[i];

    DrwGL_Scale_Mat( m1, px, py, pz, sx, sy, sz );
    DrwGL_Mul_Mat( m1, m2, mat );
  }
}




void DrwGL_Transf_Seg( GLfloat* mat,
                       float  tx, float  ty, float  tz,
                       float  px, float  py, float  pz,
                       float ome, float chi, float psi,
                       float  sx, float  sy, float  sz )
{
  static float m1[16], m2[16];
  int i;

  if (mat) {
    for (i = 0; i < 16; i++) m2[i] = mat[i];

    DrwGL_Rotate_Mat( m1, px, py, pz, ome, chi, psi );
    DrwGL_Mul_Mat( m1, mat, m2 );
    DrwGL_Scale_Mat( m1, px, py, pz, sx, sy, sz );
    DrwGL_Mul_Mat( m1, m2, mat );
    mat[12] += tx;
    mat[13] += ty;
    mat[14] += tz;
  }
}
  



/***********************************************************************/
/*                     Basic  Display  Routines                        */
/***********************************************************************/



void DrwGL_Display_Plot( GLint code, GLint dim, GLint flag, GLfloat * tab, GLint sel )
{ /* Plot 2D/3D function */
  GLfloat x, y, z;
  GLint         i;
  static GLint wpdwn = 0;

//fprintf( fmsg, " GL ATTR: line Stipple %d, fill Stipple %d .\n",
//               Drwgl_line_stipple, Drwgl_fill_stipple );
//fprintf( fmsg, " GL ATTR: Fill Front %d, Fill Back %d .\n",
//               Drwgl_fill_front, Drwgl_fill_back );
//fprintf( fmsg, " GL_Plot: Line Cd = %d, Flg = %d, dim = %d, sel = %d\n",
//               code, flag, dim, sel );
//fflush( fmsg );

  if (sel) {
//  fprintf( fmsg, " GL_Plotn Select.\n" );
//  fflush( fmsg );
    Drwh_mix = Drwh_miy = Drwh_miz = 1e+30;
    Drwh_max = Drwh_may = Drwh_maz = - Drwh_mix;
    i = 0;
    while (i++ < dim) {
      x = *(tab++); y = *(tab++);
      z = (flag&cdf_3D)?*(tab++):0.0;
      DrwHVertex( x, y, z );
      if (flag&cdf_NRM) tab += 3;
    }
  }
  else
  {
    if (Drwgl_line_stipple) { int ip, ir;
      glGetIntegerv( GL_LINE_STIPPLE_PATTERN, &ip ); glGetIntegerv( GL_LINE_STIPPLE_REPEAT, &ir );
//    fprintf( fmsg, " plot with Line_stipple pattern = %d.\n", ip );
//    fprintf( fmsg, " plot with Line_stipple repeat = %d.\n", ir );
//    fflush( fmsg );
    }
    if (flag&cdf_ST) {
      if (wpdwn) {
        fprintf( fmsg, " No previous plot_end while a new plot_�start is required\n" );
        fflush( fmsg );
        glEnd();
      }
      if (Drwgl_fill_front) glPolygonMode( GL_FRONT, GL_FILL );
                       else glPolygonMode( GL_FRONT, GL_LINE );
      if (Drwgl_fill_back)  glPolygonMode( GL_BACK, GL_FILL );
                       else glPolygonMode( GL_BACK, GL_LINE );
      if (Drwgl_line_stipple) glEnable( GL_LINE_STIPPLE );
      if (Drwgl_fill_stipple) glEnable( GL_POLYGON_STIPPLE );
      if (Drwgl_fill_Shift) glEnable( GL_POLYGON_OFFSET_FILL );

#     ifdef _GL_FullDebug
        DrwGL_Error( "Before Exec glBegin" );
#     endif

      glBegin( code );
//    fprintf( fmsg, " GL Plot Begin with code %d and size %d.\n", code, dim );
//    fflush( fmsg );
      wpdwn = 1;
    }

    if (flag&cdf_3D)
    { /* 3D plot */
      if (Drwgl_fill_front||Drwgl_fill_back) draw_3D_fill_flg = draw_3D_flg;
      for (i = 0; i < dim; i++)
      {
        if (flag&cdf_NRM) {
//        fprintf( fmsg, " GL Plot n#%d/ (%f,%f,%f).\n", i, tab[0], tab[1], tab[2] );
          glNormal3fv( tab ); tab += 3;
        }
//      fprintf( fmsg, " GL Plot v#%d/ (%f,%f,%f).\n", i, tab[0], tab[1], tab[2] );
        glVertex3fv( tab ); tab += 3;
//      fflush( fmsg );
      }
    }
    else
    { /* 2D plot */
//    GLdouble scx, scy, scz;
      for (i = 0; i < dim; i++)
      {
//      gluProject( (GLdouble) tab[0], (GLdouble) tab[1], 0.0,
//               Drwgl_mat_unit, Drwgl_proj_wrd, Drwgl_wrdvport,
//               &scx, &scy, &scz );
//      fprintf( fmsg, " GL Plot v#%d/ (%f,%f) to (%f,%f,%f).\n", i, tab[0], tab[1], scx, scy, scz );
//      fprintf( fmsg, " GL Plot v#%d/ (%f,%f).\n", i, tab[0], tab[1] );
//      fflush( fmsg );
        glVertex2fv( tab ); tab += 2;
      }
    }
    if (flag&cdf_ND)
    {
      glEnd();

#     ifdef _GL_FullDebug
        DrwGL_Error( "Exec glEnd" );
#     endif

//    fprintf( fmsg, " GL Plot end (code %d).\n", code );
//    fflush( fmsg );

      wpdwn = 0;
      if (Drwgl_line_stipple) glDisable( GL_LINE_STIPPLE );
      if (Drwgl_fill_stipple) glDisable( GL_POLYGON_STIPPLE );
      if (Drwgl_fill_Shift) glDisable( GL_POLYGON_OFFSET_FILL );
    }
  }
}




void DrwGL_UsrXYZ( GLdouble x, GLdouble y, GLdouble z, int ini )
{
  GLdouble u, v, w;

  if (gluProject( x, y, z, Drwgl_modlview, Drwgl_proj_wrd, Drwgl_wrdvport,
                  &u, &v, &w ) == GL_TRUE) {
    if (ini) {
      Drwh_mix = Drwh_max = u; Drwh_miy = Drwh_may = v; Drwh_miz = Drwh_maz = w;
    } else {
      if (Drwh_mix > u) Drwh_mix = u; if (Drwh_max < u) Drwh_max = u;
      if (Drwh_miy > v) Drwh_miy = v; if (Drwh_may < v) Drwh_may = v;
      if (Drwh_miz > w) Drwh_miz = w; if (Drwh_maz < w) Drwh_maz = w;
    }
  }
  else {
    fprintf( fmsg, " UsrXYZ cannot Work with (%f,%f,%f)\n", x, y, z );
    fflush( fmsg );
  }
}



void DrwGL_Circle( Draw_Ptr q, int sel )
{
  GLdouble r, x, y, sa, ea, ac, an, ca;
  int   i, n, kn;
  GLint       lk;

  r  = q->circ.circl_r;
  x  = q->circ.circl_x;
  y  = q->circ.circl_y;
  if (sel) {
    DrwGL_UsrXYZ( x - r, y - r, 0.0, 1 );
    DrwGL_UsrXYZ( x - r, y + r, 0.0, 0 );
    DrwGL_UsrXYZ( x + r, y + r, 0.0, 0 );
    DrwGL_UsrXYZ( x + r, y - r, 0.0, 0 );

    /* Set for Directive Hit */
    if ((Drwh_mix >= Drws_max)&&(Drwh_max <= Drws_mix)&&
        (Drwh_miy >= Drws_may)&&(Drwh_may <= Drws_miy)) {
      Hits = 1;
//    Hitx = (Drwh_mix + Drwh_max)/2.0;
//    Hity = (Drwh_miy + Drwh_may)/2.0;
//    Hitz = (Drwh_miz + Drwh_maz)/2.0;
    }
  } else {
	ca = 0.0;
    sa = q->circ.circl_sa;
    ea = q->circ.circl_ea;
    ac = q->circ.circl_ac;
    kn = q->circ.circl_kn;

    if (ac < 1e-4) ac = 1e-4;
    n = 4; an = (180.0*inrd)/n;
    do { n *= 2; an /= 2.0; } while ((1.0 - cos( an )) > ac);
    if ((sa < -180.0)||(sa >= 360.0)) sa = 0.0;
    if  ((ea <= sa)||(ea > 360.0)) ea = 0.0;
    if ( fabs(sa - ea) < 1e-4) ea = 360.0;
    sa *= inrd; ea *= inrd;
    n = (int) ((ea - sa)/an);
    if (Drwgl_fill_front) glPolygonMode( GL_FRONT, GL_FILL );
                     else glPolygonMode( GL_FRONT, GL_LINE );
    if (Drwgl_fill_back)  glPolygonMode( GL_BACK, GL_FILL );
                     else glPolygonMode( GL_BACK, GL_LINE );
    if (Drwgl_line_stipple) glEnable( GL_LINE_STIPPLE );
    if (Drwgl_fill_stipple) glEnable( GL_POLYGON_STIPPLE );
    if (Drwgl_fill_Shift) glEnable( GL_POLYGON_OFFSET_FILL );

     if ((kn <= 0)&&(kn >= -9)) lk = Plot_Code[-kn];
                             else if (kn > 2) lk = GL_POLYGON;
                                         else lk = GL_LINE_STRIP;

    glBegin( lk );

    for (i = 0; i <=n; i++) {
      ca = sa + i*an;
      glVertex2d( x + r*cos( ca ), y + r*sin( ca ) );
    }
    if ((ea - ca) > 1e-4) glVertex2d( x + r*cos( ea ), y + r*sin( ea ) );

    glEnd();
    if (Drwgl_line_stipple) glDisable( GL_LINE_STIPPLE );
    if (Drwgl_fill_stipple) glDisable( GL_POLYGON_STIPPLE );
    if (Drwgl_fill_Shift) glDisable( GL_POLYGON_OFFSET_FILL );
  }
}



static void DrwGL_Set_Quadric_Mode( int code )
{
    switch (code) {
      case 1: gluQuadricDrawStyle( QuaObj, GLU_LINE );
              gluQuadricNormals( QuaObj, GLU_NONE );
              break;
      case 2: gluQuadricDrawStyle( QuaObj, GLU_POINT );
              gluQuadricNormals( QuaObj, GLU_NONE );
              break;
      case 3: gluQuadricDrawStyle( QuaObj, GLU_FILL );
              gluQuadricNormals( QuaObj, GLU_FLAT );
              break;
      case 4: gluQuadricDrawStyle( QuaObj, GLU_FILL );
              gluQuadricNormals( QuaObj, GLU_SMOOTH );
              break;
      case 0:
      default: /* Standard Fill Mode */
        gluQuadricDrawStyle( QuaObj, GLU_FILL );
        gluQuadricNormals( QuaObj, GLU_NONE );
    }
}



static void DrwGL_Set_Obj_Orient( GLfloat tx, GLfloat ty, GLfloat tz, GLfloat th, GLfloat ph )
{
  glPushMatrix();
#   ifdef _GL_FullDebug
      DrwGL_Error( "Obj_Orient PushMatrix" );
#   endif

  glTranslatef( tx, ty, tz );
#   ifdef _GL_FullDebug
      DrwGL_Error( "Obj_Orient Translate" );
#   endif

  glRotatef( ph, 0.0, 0.0, 1.0 );
#   ifdef _GL_FullDebug
      DrwGL_Error( "Obj_Orient Rotate ph" );
#   endif

  glRotatef( th, 0.0, 1.0, 0.0 );
#   ifdef _GL_FullDebug
      DrwGL_Error( "Obj_Orient Rotate th" );
#   endif

  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
#   ifdef _GL_FullDebug
      DrwGL_Error( "Obj_Orient GetDouble" );
#   endif

}



void DrwGL_Sphere( Draw_Ptr q, int sel )
{
  GLdouble r;


#   ifdef _GL_FullDebug
      DrwGL_Error( "Before exec Sphere 1" );
#   endif

  glPushMatrix();

#   ifdef _GL_FullDebug
      DrwGL_Error( "Sphere 1 PushMatrix" );
#   endif

  glTranslatef( (GLfloat) q->sphere.sphere_x,
                (GLfloat) q->sphere.sphere_y, (GLfloat) q->sphere.sphere_z );

#   ifdef _GL_FullDebug
      DrwGL_Error( "Sphere 1 Transatef" );
#   endif

  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

#   ifdef _GL_FullDebug
      DrwGL_Error( "Sphere 1 GetDouble" );
#   endif



  if (sel) {
    r = q->sphere.sphere_r;
    DrwGL_UsrXYZ( -r, -r, -r, 1 );
    DrwGL_UsrXYZ( -r, -r,  r, 0 );
    DrwGL_UsrXYZ(  r, -r,  r, 0 );
    DrwGL_UsrXYZ(  r, -r, -r, 0 );
    DrwGL_UsrXYZ(  r,  r, -r, 0 );
    DrwGL_UsrXYZ(  r,  r,  r, 0 );
    DrwGL_UsrXYZ( -r,  r,  r, 0 );
    DrwGL_UsrXYZ( -r,  r, -r, 0 );

    /* Set for Directive Hit */
    if ((Drws_max >= Drwh_mix)&&(Drws_mix <= Drwh_max)&&
        (Drws_may >= Drwh_miy)&&(Drws_miy <= Drwh_may)) {
      Hits = 1;
//    Hitx = q->sphere.sphere_x; Hity = q->sphere.sphere_y; Hitz = q->sphere.sphere_z;
    }
  } else {
      /* We Use the Quadric mode of GLU Library */

    DrwGL_Set_Quadric_Mode( q->sphere.sphere_k );

//  fprintf( fmsg, " Sphere # 1\n" );
//  fflush( fmsg );

#   ifdef _GL_FullDebug
      DrwGL_Error( "Before exec Sphere 2" );
#   endif

    gluSphere( QuaObj, (GLfloat) q->sphere.sphere_r,
                         (GLint) q->sphere.sphere_px, (GLint) q->sphere.sphere_pz );
//  gluDeleteQuadric( q->sphere.quad );
  }
  glPopMatrix();

#   ifdef _GL_FullDebug
      DrwGL_Error( "Sphere 2 PopMatrix" );
#   endif

  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

#   ifdef _GL_FullDebug
      DrwGL_Error( "Sphere 2 GetDouble" );
#   endif

}



void DrwGL_Cylinder( Draw_Ptr q, int sel )
{
  GLdouble rb, rt, rh;


#   ifdef _GL_FullDebug
      DrwGL_Error( "Before exec Cylinder 1" );
#   endif

  DrwGL_Set_Obj_Orient(  q->cyl.cyl_x,  q->cyl.cyl_y, q->cyl.cyl_z,
                        q->cyl.cyl_th, q->cyl.cyl_ph );

  if (sel) {
    rb = q->cyl.cyl_rb; rt = q->cyl.cyl_rt;
    rh = q->cyl.cyl_lg;

    DrwGL_UsrXYZ( -rb, -rb, 0.0, 1 );
    DrwGL_UsrXYZ( -rb,  rb, 0.0, 0 );
    DrwGL_UsrXYZ(  rb,  rb, 0.0, 0 );
    DrwGL_UsrXYZ(  rb, -rb, 0.0, 0 );
    DrwGL_UsrXYZ( -rt, -rt,  rh, 0 );
    DrwGL_UsrXYZ( -rt,  rt,  rh, 0 );
    DrwGL_UsrXYZ(  rt,  rt,  rh, 0 );
    DrwGL_UsrXYZ(  rt, -rt,  rh, 0 );

    /* Set for Directive Hit */
    if ((Drws_max >= Drwh_mix)&&(Drws_mix <= Drwh_max)&&
        (Drws_may >= Drwh_miy)&&(Drws_miy <= Drwh_may)) {
      Hits = 1;
//    fprintf( fmsg, " Highlight Cylinder id %d:%d\n", q->seg.dcom.dir_pid, sup_pickid );
//    fflush( fmsg );
//    Hitx = (Drwh_mix + Drwh_max)/2.0;
//    Hity = (Drwh_miy + Drwh_may)/2.0;*
//    Hitz = (Drwh_miz + Drwh_maz)/2.0;
    }
  } else {
    /* We Use the Quadric mode of GLU Library */
    DrwGL_Set_Quadric_Mode( q->cyl.cyl_k );
    gluCylinder( QuaObj, (GLfloat) q->cyl.cyl_rb, (GLfloat) q->cyl.cyl_rt,
                         (GLfloat) q->cyl.cyl_lg,
                           (GLint) q->cyl.cyl_px, (GLint) q->cyl.cyl_pz );
    // gluDeleteQuadric( q->cyl.quad );
  }
  glPopMatrix();

#   ifdef _GL_FullDebug
      DrwGL_Error( "Cylinder 2 PopMatrix" );
#   endif

  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

#   ifdef _GL_FullDebug
      DrwGL_Error( "Cylinder 2 GetDouble" );
#   endif

}



void DrwGL_Disk( Draw_Ptr q, int sel )
{
  GLdouble re;

  DrwGL_Set_Obj_Orient(  q->disk.dsk_x,  q->disk.dsk_y,  q->disk.dsk_z,
                         q->disk.dsk_th, q->disk.dsk_ph );


#   ifdef _GL_FullDebug
      DrwGL_Error( "Before exec Disk 1" );
#   endif

  if (sel) {
    re = q->disk.dsk_re;
    DrwGL_UsrXYZ( -re, -re, 0.0, 1 );
    DrwGL_UsrXYZ( -re,  re, 0.0, 0 );
    DrwGL_UsrXYZ(  re,  re, 0.0, 0 );
    DrwGL_UsrXYZ(  re, -re, 0.0, 0 );
    /* Set for Directive Hit */
    if ((Drws_max >= Drwh_mix)&&(Drws_mix <= Drwh_max)&&
        (Drws_may >= Drwh_miy)&&(Drws_miy <= Drwh_may)) {
      Hits = 1;
//    fprintf( fmsg, " Highlight Disk id %d:%d\n",q->seg.dcom.dir_pid, sup_pickid );
//    fflush( fmsg );
//    Hitx = 0.0; Hity = 0.0; Hitz = 0.0;
    }
  } else {
      /* We Use the Quadric mode of GLU Library */
    DrwGL_Set_Quadric_Mode( q->disk.dsk_k );

    if ((q->disk.dsk_ea <= 0.0)||(q->disk.dsk_sa >= q->disk.dsk_ea))
      gluDisk( QuaObj, q->disk.dsk_ri, q->disk.dsk_re,
                       q->disk.dsk_pt, q->disk.dsk_pr );
    else
      gluPartialDisk( QuaObj, q->disk.dsk_ri, q->disk.dsk_re,
                              q->disk.dsk_pt, q->disk.dsk_pr,
                              q->disk.dsk_sa, q->disk.dsk_ea );

    // gluDeleteQuadric( q->disk.quad );
  }
  glPopMatrix();

#   ifdef _GL_FullDebug
      DrwGL_Error( "Disk 2 PopMatrix" );
#   endif

  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

#   ifdef _GL_FullDebug
      DrwGL_Error( "Disk 2 GetDouble" );
#   endif

}




void Set_Clip_Plane_Equ( GLenum plane, float * equ )
{
  GLdouble gl_equ[4];
  int i;

  for (i = 0; i < 4; i++) gl_equ[i] = (GLdouble) equ[i];
  glClipPlane( plane, gl_equ );
}




void DrwGL_Push_Plane( Draw_Ptr p )
{ /* Push a new 2D Plot Plane reference */

  /* Transformation is handled by OpenGL */
  glPushMatrix();
  glMultMatrixf( p->eplane.epl_mat );
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

  Drwgl_segplcnt++;
  p->eplane.epl_prev = Drwgl_currplane;
  Drwgl_currplane = p;
}




void DrwGL_Pop_Plane()
{ /* Push a new 2D Plot Plane reference */
  if (Drwgl_currplane&&(Drwgl_segplcnt > 0)) {
    /* Transformation is handled by OpenGL */
    glPopMatrix();
    glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

    Drwgl_segplcnt--;
    Drwgl_currplane = Drwgl_currplane->eplane.epl_prev;
  }
}




void DrwGL_Push_Bxrf( Draw_Ptr p )
{ /* Push a new reference box */

  if (p->boxref.bxr_mod[1]) {
    Set_Clip_Plane_Equ( GL_CLIP_PLANE0, p->boxref.bxr_lim+0  );
    Set_Clip_Plane_Equ( GL_CLIP_PLANE1, p->boxref.bxr_lim+4  );
    glEnable( GL_CLIP_PLANE0 );
    glEnable( GL_CLIP_PLANE1 );
  }

  if (p->boxref.bxr_mod[2]) {
    Set_Clip_Plane_Equ( GL_CLIP_PLANE2, p->boxref.bxr_lim+8  );
    Set_Clip_Plane_Equ( GL_CLIP_PLANE3, p->boxref.bxr_lim+12 );
    glEnable( GL_CLIP_PLANE2 );
    glEnable( GL_CLIP_PLANE3 );
  }

  if (p->boxref.bxr_mod[3]) {
    Set_Clip_Plane_Equ( GL_CLIP_PLANE4, p->boxref.bxr_lim+16 );
    Set_Clip_Plane_Equ( GL_CLIP_PLANE5, p->boxref.bxr_lim+20 );
    glEnable( GL_CLIP_PLANE4 );
    glEnable( GL_CLIP_PLANE5 );
  }

  p->boxref.bxr_prev = Drwgl_currbrf,
  Drwgl_currbrf = p;

}




void DrwGL_Pop_Bxrf()
{ /* Pop a reference box of the box reference stack */
  Draw_Ptr p;


  p = Drwgl_currbrf;

  if (p) {
    Drwgl_currbrf = p->boxref.bxr_prev;

    if (Drwgl_currbrf) {
      if (p->boxref.bxr_mod[1]) {
        Set_Clip_Plane_Equ( GL_CLIP_PLANE0, Drwgl_currbrf->boxref.bxr_lim+0  );
        Set_Clip_Plane_Equ( GL_CLIP_PLANE1, Drwgl_currbrf->boxref.bxr_lim+4  );
      }
      if (p->boxref.bxr_mod[2]) {
        Set_Clip_Plane_Equ( GL_CLIP_PLANE2, Drwgl_currbrf->boxref.bxr_lim+8  );
        Set_Clip_Plane_Equ( GL_CLIP_PLANE3, Drwgl_currbrf->boxref.bxr_lim+12 );
      }
      if (p->boxref.bxr_mod[3]) {
        Set_Clip_Plane_Equ( GL_CLIP_PLANE4, Drwgl_currbrf->boxref.bxr_lim+16 );
        Set_Clip_Plane_Equ( GL_CLIP_PLANE5, Drwgl_currbrf->boxref.bxr_lim+20 );
      }
    }
    else
    {
      if (p->boxref.bxr_mod[1]) {
        glClipPlane( GL_CLIP_PLANE0, Drwgl_cliplane0 );
        glClipPlane( GL_CLIP_PLANE1, Drwgl_cliplane1 );
      }
      if (p->boxref.bxr_mod[2]) {
        glClipPlane( GL_CLIP_PLANE2, Drwgl_cliplane2 );
        glClipPlane( GL_CLIP_PLANE3, Drwgl_cliplane3 );
      }
      if (p->boxref.bxr_mod[3]) {
        glDisable( GL_CLIP_PLANE4 );
        glDisable( GL_CLIP_PLANE5 );
      }
    }
  }
}




void DrwGL_PP_Attrib( int code )
{
  if (code) { /* Push */
//  fprintf( fmsg, " Push on Stack from level %d\n", Drwgl_AStkSp );
//  fflush( fmsg );
    if (Drwgl_AStkSp < (ATTRSTKSZ-1)) {
      Drwgl_AStkSp++; Drwgl_Attr++;
      Drwgl_AStack[Drwgl_AStkSp] = Drwgl_AStack[Drwgl_AStkSp-1];
      /*
      pck = &(Drwgl_AStack[Drwgl_AStkSp]);
      pck->txt_haln = Drwgl_txt_haln;
      pck->txt_valn = Drwgl_txt_valn;
      pck->txt_path = Drwgl_txt_path;
      pck->txt_expf = Drwgl_txt_expf;
      pck->txt_spcf = Drwgl_txt_spcf;drwgl_astkel
      pck->txt_font =  Drw_txt_font;
      pck->txt_prec = Drwgl_txt_prec;
      pck->txt_dept = Drwgl_txt_dept;
      */
    } else {
      fprintf( fmsg, " DRAW_SERVER: Attibute Stack Overflow Error.\n" );
      fflush( fmsg );
    }
  } else { /* Pop */
//  fprintf( fmsg, " Pop on Stack from level %d\n", Drwgl_AStkSp );
//  fflush( fmsg );
    if (Drwgl_AStkSp > 0) {
      Drwgl_AStkSp--; Drwgl_Attr--;
      DrwGL_Line_Attr( Drwgl_Attr->lin_type, Drwgl_Attr->lin_size );
      DrwGL_Marker_Attr( Drwgl_Attr->mrk_type, Drwgl_Attr->mrk_size );
      DrwGL_Fill_Attr( Drwgl_Attr->fil_type, Drwgl_Attr->fil_styl );
      DrwGL_Text_Attr( Drwgl_Attr->txt_haln, Drwgl_Attr->txt_valn, Drwgl_Attr->txt_path,
                       Drwgl_Attr->txt_expf, Drwgl_Attr->txt_spcf, Drwgl_Attr->txt_dept );

      if (lighting) glMaterialfv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, Drwgl_Attr->col_rgba );
               else glColor4fv( Drwgl_Attr->col_rgba );
      if (Drwgl_Attr->lgh_stat) glEnable( GL_LIGHTING );
                           else glDisable( GL_LIGHTING );
    } else {
      fprintf( fmsg, " DRAW_SERVER: Attibute Stack Underflow Error.\n" );
      fflush( fmsg );
    }
  }
}



void DrwGL_Display_Seg( Draw_Ptr p, GLenum mode )
{ /* Display a particular Segment */
  Draw_Ptr      q;
  float *     tab;
  int     cod, dim, flg, i, sel, ssl, save_segplcnt, svtmatcnt;
  
  save_segplcnt = Drwgl_segplcnt;
  Drwgl_segplcnt = 0;

//fprintf( fmsg, " Start Display of segment # %d\n", p->seg.seg_ide );
//fflush( fmsg );

  glPushMatrix();
  glMultMatrixf( p->seg.seg_mat );
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
//Show_Matf( " Set Seg Matrix : \n", p->seg.seg_mat );
//Show_Mat( " Eff Seg Matrix : \n", Drwgl_modlview );

  sel = (mode == GL_SELECT);
  ssl = 1;

  svtmatcnt = Tmat_Cnt;

  q = p->seg.seg_fdir;
  while (q)
  { /* Loop on all Segment Directive */
//fprintf( fmsg, " Loop continue\n" ); fflush( fmsg );
//fprintf( fmsg, " GL Out dir # %d, id = %d\n",
//               q->seg.dcom.dir_knd, q->seg.dcom.dir_pid );
//fflush( fmsg );
    if (sel) {
      Hits = 0;
      ssl = (q->seg.dcom.dir_pid&1);
//    fprintf( fmsg, " Selection Hits Search with %d.\n", ssl );
//    fflush( fmsg );
    }

    switch (q->seg.dcom.dir_knd) {
      case Draw_Segment:
      break;

      case Draw_Exec_Seg:
        i = q->exeseg.dcom.dir_pid>>1;
        sup_pickid += i;
        if (q->exeseg.exec_seg)
          DrwGL_Display_Seg( q->exeseg.exec_seg, mode );
        sup_pickid -= i;
      break;

      case Draw_PP_Attr:
        DrwGL_PP_Attrib( q->ppattr.ppa_code );
      break;

      case Draw_Ena_Plane:
        DrwGL_Push_Plane( q );
      break;

      case Draw_Dis_Plane:
        DrwGL_Pop_Plane();
      break;

      case Draw_Box_Ref:
        DrwGL_Push_Bxrf( q );
      break;

      case Draw_Box_Pop:
        DrwGL_Pop_Bxrf();
      break;

      case Draw_PushTrfMat:
        glPushMatrix();
        if (!Tmat_Cnt)
          glTranslatef( (GLfloat) orgx, (GLfloat) orgy, (GLfloat) orgz );
        glMultMatrixf( q->tmat.trsf_mat );
        glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
        Tmat_Cnt++;
      break;

      case Draw_PopTrfMat:
        glPopMatrix();
        glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );
        Tmat_Cnt--;
      break;

      case Draw_Plot:
        tab = q->tab.plo_tab;
        dim = q->tab.plo_npt;
        flg = q->tab.plo_flg;
        cod = q->tab.plo_knd;
        if (ssl) /* Skip for UnDetect flag */
//      fprintf( fmsg, " Plot Dir. with cod = %d, dim = %d, flg = %d.\n", cod, dim, flg ); fflush( fmsg );
        switch (cod) {
          case 1: /* Line */
            if (dim > 0) DrwGL_Display_Plot( GL_LINE_STRIP, dim, flg, tab, sel );
                    else DrwGL_Display_Plot( GL_LINE_LOOP, -dim, flg, tab, sel );
          break;

          case 2: /* Marker */
            if (Drwgl_Attr->mrk_type < 0)
              DrwGL_Display_Plot( GL_POINTS, dim, flg, tab, sel );
            else
              DrwGL_Display_Marker( abs( dim ), flg, tab, sel );
          break;

          case 3: /* Fill Area */
          case 4: /* Fill Area with Normals */
            DrwGL_Display_Plot( GL_POLYGON, dim, flg, tab, sel );
          break;

          default:
            if ((cod <= 0)&&(cod >= -9))
              DrwGL_Display_Plot( Plot_Code[-cod], dim, flg, tab, sel );
        }
      break;

      case Draw_Text:
        if (ssl) DrwGL_Display_Text( q, sel );
      break;

      case Draw_TBox:
        if (ssl) DrwGL_Display_Text_Box( q, sel );
      break;

      case Draw_Circle:   if (ssl) DrwGL_Circle( q, sel );  break;
      case Draw_Sphere:   if (ssl) DrwGL_Sphere( q, sel );  break;
      case Draw_Cylinder: if (ssl) DrwGL_Cylinder( q,sel ); break;
      case Draw_Disk:     if (ssl) DrwGL_Disk( q, sel );    break;

      case Draw_Line_Attr:
        DrwGL_Line_Attr( q->linattr.plia_kind, q->linattr.plia_size );
      break;

      case Draw_Marker_Attr:
        DrwGL_Marker_Attr( q->mrkattr.mrka_kind, q->mrkattr.mrka_size );
      break;

      case Draw_Fill_Attr:
        DrwGL_Fill_Attr( q->filattr.fila_kind, q->filattr.fila_styl );
      break;

      case Draw_Text_Attr:
        DrwGL_Text_Attr( q->txtattr.txta_haln, q->txtattr.txta_valn,
                         q->txtattr.txta_path, q->txtattr.txta_expf,
                         q->txtattr.txta_spcf, q->txtattr.txta_dept );
      break;

      case Draw_Font:
        DrwGL_Font_Selection( q->font.fnt_font );
      break;

      case Draw_Color:
        DrwGL_ColorRGB( q->color.col_rgba );
      break;

      case Draw_Blend_Set:
        DrwGL_Alpha_Set( q->blend.bld_flags,
                         q->blend.bld_src, q->blend.bld_dst );
      break;

      case Draw_Light_Def:
        if (lightst[q->light.lig_spc-GL_LIGHT0]) glLightfv( q->light.lig_spc,
                                                            q->light.lig_kind, q->light.lig_prm );
//      GLfloat pos[4];
//      glGetLightfv( q->light.lig_spc, q->light.lig_kind, pos );
//      fprintf( fmsg, " Light #%d prop %d set at (%f,%f,%f,%f)\n", q->light.lig_spc-GL_LIGHT0, q->light.lig_kind,
//                     q->light.lig_prm[0], q->light.lig_prm[1], q->light.lig_prm[2], q->light.lig_prm[3] );
//      fprintf( fmsg, " Light #%d prop %d put at (%f,%f,%f,%f)\n", q->light.lig_spc-GL_LIGHT0, q->light.lig_kind,
//                     pos[0], pos[1], pos[2], pos[3] );
//      fflush( fmsg );

      break;

      case Draw_Mat_Light_Prop:
        glMaterialfv( q->lmatp.lmat_face,
                      q->lmatp.lmat_kind, q->lmatp.lmat_prm );
      break;

      case Draw_Light_OnOff:
        if (q->lig01.lgs_sta) glEnable( GL_LIGHTING );
                         else glDisable( GL_LIGHTING );
        Drwgl_Attr->lgh_stat = q->lig01.lgs_sta;
      break;

      default:
		fprintf( fmsg, " *** Fatal Error: Illegal Internal Graphic Code %n ***\n",
		               q->seg.dcom.dir_knd );
		fflush( fmsg );
		exit( 4 );
      break;
    }

#   ifdef _GL_Debug
      static char st[64];
      sprintf( st, " Code = %d\n", q->seg.dcom.dir_knd );
      DrwGL_Error( st );
#   endif

    if (sel&&Hits) { /* When Hit on select, Push the segment specifications */
      if (HitsSp < Hits_Bufsz-1) {
        HitsSp++;
        HitsHeap[HitsSp].seg  =  p->seg.seg_ide;
        HitsHeap[HitsSp].prio =  p->seg.seg_pri;
        HitsHeap[HitsSp].pid  =  (q->seg.dcom.dir_pid>>1) + sup_pickid;
        HitsHeap[HitsSp].xmin =  Drwh_mix;
        HitsHeap[HitsSp].ymin =  Drwh_miy;
        HitsHeap[HitsSp].zmin =  Drwh_miz;
        HitsHeap[HitsSp].xmax =  Drwh_max;
        HitsHeap[HitsSp].ymax =  Drwh_may;
        HitsHeap[HitsSp].zmax =  Drwh_maz;
      }

//    fprintf( fmsg, " Hits # %d Seg # %d %f %d.\n", HitsSp, p->seg.seg_ide,
//                   p->seg.seg_pri, q->seg.dcom.dir_pid );
//    fprintf( fmsg, " Hits Echo Area [%f,%f,%f,%f]\n", Drwh_mix, Drwh_miy, Drwh_max, Drwh_may );
//    fflush( fmsg );
      Hits = 0;
    }
//  else
//  if (sel&&(q->seg.dcom.dir_knd == Draw_Plot)) {
//    fprintf( fmsg, " No_Hits # %d Seg # %d %f %d.\n", HitsSp, p->seg.seg_ide,
//                   p->seg.seg_pri, q->seg.dcom.dir_pid );
//    fprintf( fmsg, " No_Hits Echo Area [%f,%f,%f,%f]\n", Drwh_mix, Drwh_miy, Drwh_max, Drwh_may );
//    fflush( fmsg );
//  }
    q = q->seg.dcom.dir_nxt;
  }

  while (Drwgl_segplcnt > 0) DrwGL_Pop_Plane();
  Drwgl_segplcnt = save_segplcnt;

  while (svtmatcnt < Tmat_Cnt--) glPopMatrix(); /* Pop each active identpendant matrix. */
  glPopMatrix();                                /* Pop the segment matrix. */

//fprintf( fmsg, " display Seg End.\n" );
//fflush( fmsg );
}




void Set_Clip_Lim( GLdouble tab[4], int cat, float val )
{
  int i;

  for (i = 0;i < 3; i++) tab[i] = 0.0;
  if (cat < 0)
  {
    cat = abs( cat );
    tab[cat - 1] =  1.0; tab[3] = - (GLdouble) val;
  }
  else
  {
    tab[cat - 1] = -1.0; tab[3] =   (GLdouble) val;
  }
}



void DrwGL_Display()
{ /* Display the Whole of Draw Picture */
  Draw_Ptr      p;
  GLdouble   fovy;
  int      Prsflg;

  Drwgl_currbrf   = NULL;
  Drwgl_currplane = NULL;

  Zdepth     = draw_uzmax - draw_uzmin;
  Zorg       = (draw_uzmax + draw_uzmin)*0.5;
  MinZ_Dist  = fabs( draw_uzmin );              /* Compute minimum distance for perspective */
  if (Prsflg = (Draw_Dist > MinZ_Dist)) {       /* Set the Perspective mode */
    MinZ_Dist  = Draw_Dist - 0.5*Zdepth;        /* Warning MinZ is the Z for near plane coordinate */
    MaxZ_Dist  = Draw_Dist + 0.5*Zdepth;        /* Warning MaxZ is the Z for far plane coordinate */
    Zorg       = (MaxZ_Dist + MinZ_Dist)*0.5;   /* Force the origine to be relative to the observer */
  }

//fprintf( fmsg, " ************ Start a display\n" );
//fflush( fmsg );

  Drwgl_AStkSp    =    0;

  if (Drwgl_ClrColor) {                             // Change Clear Color when Color #0 is changed.
    glClearColor( draw_srccoltb[0][0], draw_srccoltb[0][1],
                  draw_srccoltb[0][2], draw_srccoltb[0][3] );
    Drwgl_ClrColor = 0;
  }


  if (lighting)                                     // Set the required commutation(s) on lights.
    for (int light=0;light<LIGHT_NUMBER;light++)
      if (Drwgl_light_entb[light]!=lightst[light])
        if (Drwgl_light_entb[light]) { glEnable( light + GL_LIGHT0 ); 
                                       lightst[light] = 1;
                                       // printf( " L #%d on\n", light );
                                     }
                                else { glDisable( light + GL_LIGHT0 );
                                       lightst[light] = 0;
                                       // printf( " L #%d off\n", light );
                                     }

  if (!Drwgl_clipflg)
  {
    Set_Clip_Lim( Drwgl_cliplane0, -1, draw_uxmin );
    Set_Clip_Lim( Drwgl_cliplane1,  1, draw_uxmax );
    Set_Clip_Lim( Drwgl_cliplane2, -2, draw_uymin );
    Set_Clip_Lim( Drwgl_cliplane3,  2, draw_uymax );
    Drwgl_clipflg = 1;
  }

  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

  glViewport( 0, 0, dt_rx, dt_ry );
  glMatrixMode( GL_PROJECTION );
  glLoadIdentity();

  if (Drwgl_tile) {                     // A tile partitionning must be apply.
//  glTranslated( (GLdouble) Drwgl_xshift, (GLdouble) Drwgl_yshift, 0.0 );
//  glScaled( (GLdouble) d_hc_resolution, (GLdouble) d_hc_resolution, 1.0 );

    glOrtho( Drwgl_gxmin, Drwgl_gxmax, Drwgl_gymin, Drwgl_gymax,  1.0, -1.0 );

//  glGetDoublev( GL_PROJECTION_MATRIX, Drwgl_proj_wrd );
//  fprintf( fmsg, "\n Tile at (%f,%f)\n", -Drwgl_xshift, -Drwgl_yshift );
//  Show_Mat( "Resulting Projection Matrix", Drwgl_proj_wrd );
  }

//fprintf( fmsg, " *** Perspective at %f => %d ***\n", Draw_Dist, Prsflg );
//fflush( fmsg );

  if (Prsflg) {
//  fprintf( fmsg, "  In Perspective.\n" );
//  fflush( fmsg );
    fovy = atan( 2.0*dw_cwsy/Zorg )/inrd;
    gluPerspective( fovy, dw_cwsx/dw_cwsy, MinZ_Dist, MaxZ_Dist );
    glTranslated( -dw_cwpx, -dw_cwpy, 0.0 );
  } else {
//  fprintf( fmsg, "  In Parallel on ViewPort [%d,%d,%d,%d].\n", 0, 0, dt_rx, dt_ry );
//  fflush( fmsg );
    glOrtho( (GLdouble) dw_cxmin, (GLdouble) dw_cxmax,
             (GLdouble) dw_cymin, (GLdouble) dw_cymax,
             (GLdouble) draw_uzmin, (GLdouble) draw_uzmax );    // Warning: The positive Z direction go out of screen to the user direction
  }

  /* OK to Redraw the Picture */

  glTranslatef( 0.0, 0.0, - Zorg );             // Put the Z origine at the middle of Z range.

  glGetIntegerv( GL_VIEWPORT, Drwgl_wrdvport );
  glGetDoublev( GL_PROJECTION_MATRIX, Drwgl_proj_wrd );
  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

/*
  glClipPlane( GL_CLIP_PLANE0, Drwgl_cliplane0 );
  glClipPlane( GL_CLIP_PLANE1, Drwgl_cliplane1 );
  glClipPlane( GL_CLIP_PLANE2, Drwgl_cliplane2 );
  glClipPlane( GL_CLIP_PLANE3, Drwgl_cliplane3 );

  glDisable( GL_CLIP_PLANE0 );
  glDisable( GL_CLIP_PLANE1 );
  glDisable( GL_CLIP_PLANE2 );
  glDisable( GL_CLIP_PLANE3 );
*/
  /* Set all Plot Defaults */

  DrwGL_Line_Attr( 1, 1.0 );
  DrwGL_Marker_Attr( 1, 1.0 );
  DrwGL_Fill_Attr( 2, 1 );
  DrwGL_ColorRGB( draw_srccoltb[1] );

  if (lighting) { /* To force in all lighting case the color #1 */
    glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR, draw_srccoltb[1] );
    glMaterialfv( GL_FRONT_AND_BACK, GL_EMISSION, draw_srccoltb[1] );
  }

//Show_Vwp( "for World", Drwgl_wrdvport );
//Show_Mat( "World Projection", Drwgl_proj_wrd );
//Show_Mat( "World Modelview", Drwgl_modlview );

//fprintf( fmsg, " *** Start display of system segments ***\n" );
//fflush( fmsg );

  p = draw_fsysseg;

  while (p) { /* Loop to display the System Segments */
    if (p->seg.seg_stat&DRWSEG_VISIBLE) DrwGL_Display_Seg( p, GL_RENDER );
    p = p->seg.seg_nxt;
  }

  glClipPlane( GL_CLIP_PLANE0, Drwgl_cliplane0 );
  glClipPlane( GL_CLIP_PLANE1, Drwgl_cliplane1 );
  glClipPlane( GL_CLIP_PLANE2, Drwgl_cliplane2 );
  glClipPlane( GL_CLIP_PLANE3, Drwgl_cliplane3 );

//  Show_clplan(  " Plane #0 ", Drwgl_cliplane0 );
//  Show_clplan(  " Plane #1 ", Drwgl_cliplane1 );
//  Show_clplan(  " Plane #2 ", Drwgl_cliplane2 );
//  Show_clplan(  " Plane #3 ", Drwgl_cliplane3 );

/*
  glEnable( GL_CLIP_PLANE0 );
  glEnable( GL_CLIP_PLANE1 );
  glEnable( GL_CLIP_PLANE2 );
  glEnable( GL_CLIP_PLANE3 );
*/

  p = draw_fsegm;
  if (p) {
    DrwGL_Line_Attr( 1, 1.0 );
    DrwGL_Marker_Attr( 1, 1.0 );
    DrwGL_Fill_Attr( 2, 1 );
    DrwGL_ColorRGB( draw_srccoltb[1] );

//	fprintf( fmsg, "**User Redraw Env. GL (%f,%f,%f,%f,%f,%f),\n",
//				   dw_cxmin, dw_cxmax, dw_cymin, dw_cymax,
//                 draw_uzmin, draw_uzmax );
//  fprintf( fmsg, "  User Redraw Z Env GL (%f, %f, %f, %f).\n",
//                   MinZ_Dist, MaxZ_Dist, Zdepth, Zorg );
//    fflush( fmsg );

//fprintf( fmsg, " *** Start display of user segments ***\n" );
//fflush( fmsg );

  while (p)
    { /* Loop to display the User Segments */
       if (p->seg.seg_stat&DRWSEG_VISIBLE) DrwGL_Display_Seg( p, GL_RENDER );
//                                else fprintf( fmsg, " Seg invisible!!\n" );
      p = p->seg.seg_nxt;
    }
  }


  /* glFlush(); */

#ifndef _No_GL_Error
  DrwGL_Error(" End_Cycle ");                                // Output any GL/GLU Error.
#endif

//fprintf( fmsg, " *** End Redraw GL Environment ***\n" );
//fflush( fmsg );
}




void DrwGL_Pick_Search( int xm, int ym )
{ /* To locate a Draw Segment as pointed by the mouse at (xm, ym). */
  unsigned int  iseg = 0, pid = 0;
  Draw_Ptr                      p;
  float    mix, miy, max, may, miz, maz, prior;

  mix = miy = max = may = 0.0;
  
  ym = dt_ry - ym - 1; /* Do Y Screen coordinate with lower left origine. */

  Drwgl_currbrf   = NULL;
  Drwgl_currplane = NULL;

  Drwgl_AStkSp    =    0;


//fprintf( fmsg, "  Pick Seg Search Hits for (%d,%d).\n", xm, ym );
//fflush( fmsg );

  p = draw_fsegm;
  if (p) { /* If a plot was performed with some segments */
    glViewport( 0, 0, dt_rx, dt_ry );
    glGetIntegerv( GL_VIEWPORT, Drwgl_wrdvport ); /* Get Current Viewport */

    Drws_mix = xm - Hits_Preci; Drws_max = xm + Hits_Preci;
    Drws_miy = ym - Hits_Preci; Drws_may = ym + Hits_Preci;

    glMatrixMode( GL_PROJECTION );

    glLoadMatrixd( Drwgl_proj_wrd );

//  glLoadIdentity();

//  if (Draw_Dist > MinZ_Dist) {
//    GLdouble aspect, fovy;
//      fovy   = (360.0/3.1415927)*asin( (dw_cymax -  dw_cymin)/Draw_Dist );
//      aspect = (dw_cxmax - dw_cxmin)/(dw_cymax -  dw_cymin);
//      gluPerspective( (GLdouble) fovy, (GLdouble) aspect,
//                      MinZ_Dist, MaxZ_Dist );
//  } else {
//    glOrtho( (GLdouble) dw_cxmin, (GLdouble) dw_cxmax,
//             (GLdouble) dw_cymin, (GLdouble) dw_cymax,
//             MinZ_Dist, MaxZ_Dist );
//  }

//  glTranslatef( 0.0, 0.0, Zorg );
//  glGetDoublev( GL_PROJECTION_MATRIX, Drwgl_proj_wrd );
    glMatrixMode( GL_MODELVIEW );
//  glGetDoublev( GL_MODELVIEW_MATRIX, Drwgl_modlview );

    /* Set all Plot Defaults */

    DrwGL_Line_Attr( 1, 1.0 );
    DrwGL_Marker_Attr( 1, 1.0 );
    DrwGL_Fill_Attr( 2, 1 );
    DrwGL_ColorRGB( draw_srccoltb[1] );

    Drwgl_segment = 0;
    Drwgl_ident   = 0;

    while (p)
    { /* Loop to display the System Segments */
      if ((p->seg.seg_stat&DRWSEG_DETECT) &&
          (p->seg.seg_stat&DRWSEG_VISIBLE)) {
//      fprintf( fmsg, " Pick Search Try Segment %d.\n", p->seg.seg_ide );
        DrwGL_Display_Seg( p, GL_SELECT );
      }
      // DRWSEG_HLIGHT for highlight on.
      p = p->seg.seg_nxt;
    }

    /* Now We can Examine the Hits Heap */
    prior = -1.0;
    while (HitsSp >= 0) {
      if (HitsHeap[HitsSp].prio >= prior)
      {
        iseg  = HitsHeap[HitsSp].seg;
        prior = HitsHeap[HitsSp].prio;
        pid   = HitsHeap[HitsSp].pid;
        mix   = HitsHeap[HitsSp].xmin;
        miy   = HitsHeap[HitsSp].ymin;
        miz   = HitsHeap[HitsSp].zmin;
        max   = HitsHeap[HitsSp].xmax;
        may   = HitsHeap[HitsSp].ymax;
        maz   = HitsHeap[HitsSp].zmax;
      }
      HitsSp--;
    }

    if (prior >= 0.0) {
//    fprintf( fmsg, " Hits Area [%5.0f,%5.0f,%5.0f,%5.0f], dt_ry = %d,\n",
//                    mix, miy, max, may, dt_ry );
      Drwgl_pechw = (int)( max - mix + 0.5) + 2*Hits_Preci;
      Drwgl_pechh = (int)( may - miy + 0.5) + 2*Hits_Preci;
      Drwgl_pechx = (int)(mix + 0.5) - Hits_Preci;
      Drwgl_pechy = dt_ry - ((int)(may - 0.5) + Hits_Preci);
      Drwgl_segment = iseg; Drwgl_ident = pid;
//    fprintf( fmsg, " Hits of seg #%d (id=%d) with echo w [%d,%d:%d,%d]\n",
//             Drwgl_segment, Drwgl_ident,
//             Drwgl_pechx, Drwgl_pechy, Drwgl_pechw, Drwgl_pechh );
//    fflush( fmsg );
    }
  }
}




void DrwGL_Get_World_XY( int xsc, int ysc )
{
  static GLdouble  xx, yy, zz;

  ysc = dt_ry - ysc - 1; /* Convert in real raster with Low Left Origine */

  /* Compute the User View Port Coordinates */
//  fprintf( fmsg, "  Screen x y = %d %d .\n", xsc, ysc );
//  fflush( fmsg );

  gluUnProject( (GLdouble) xsc, (GLdouble) ysc, 0.0,
                 Drwgl_mat_unit, Drwgl_proj_wrd, Drwgl_wrdvport,
                 &xx, &yy, &zz );

  Drwgl_xcurr = xx; Drwgl_ycurr = yy;
}




void DrwGL_Init_View_Matrix( int bt2 )
{
  int i;

  if (Drwgl_ViewMat) {
    Drwgl_xfirst = Drwgl_xcurr; Drwgl_yfirst = Drwgl_ycurr;
//  fprintf( fmsg, " Set New View Org 0 = (%f,%f).\n", Drwgl_xfirst, Drwgl_yfirst );
//  fflush( fmsg );
    /* Save the Initial View Matrix */
    for (i = 0; i<16;i++) svd_mat[i] = Drwgl_ViewMat[i];
  }
  rst_view = bt2; /* Save the View Restauration Flag */
}




void DrwGL_Change_View_Matrix()
{
  static GLfloat dx, dy, vv, nfx, nfy, sca, ang, cm, sm,
                 cm2, sm2, csm, cp, sp, co, so;
  static GLfloat m[16];
//static GLdouble md[16];

  if (Drwgl_ViewMat) {
    /* Get the first point distance (with center origine) */
    vv = sqrt( Drwgl_xfirst*Drwgl_xfirst + Drwgl_yfirst*Drwgl_yfirst );
    if (vv < 2.0) {
      Drwgl_xfirst = Drwgl_xcurr; Drwgl_yfirst = Drwgl_ycurr;
    } else {
      /* Get the direction as a normalized vector A */
      nfx = Drwgl_xfirst/vv; nfy = Drwgl_yfirst/vv;
      /* Get the displacment vector D */
      dx = Drwgl_xcurr - Drwgl_xfirst; dy = Drwgl_ycurr - Drwgl_yfirst;
      /* Compute the angle ANG_A of the axis A with X screen axis */
      ang = atan2( nfy, nfx );
//    fprintf( fmsg, " View  V = (%f,%f) Mu =% f, ", nfx, nfy, (ang/inrd) + 90.0 );
//    fprintf( fmsg, " View  D = (%f,%f), ", dx, dy );
      cm = - sin( ang ); sm = cos( ang );
      /* Compute the D projection on A and deduce the ANG_P angle */
      sca = dx*nfx + dy*nfy;
      ang = sca/vv;
//    fprintf( fmsg, "sca %f, vv %f : Phi(U)  = %f, ", ang/inrd );
      cp = cos( ang ); sp = - sin( ang );
      /* Compute the A perpandicular part of D and deduce the ANG_N angle */
      dx -= sca*nfx; dy -= sca*nfy;
      sca = sqrt( dx*dx + dy*dy );
      if (nfy*dx - nfx*dy < 0.0) sca = - sca;
      ang = sca/vv;
      co = cos( ang ); so = sin( ang );
//    fprintf( fmsg, "Omega(Z) = %f .\n", ang/inrd );
//    fflush( fmsg );

      /* Build the Rot(ang_n,Z)*Rot(ang_p,p) rotation matrix (p in plane [X,Y]) */
      cm2 = cm*cm; sm2 = sm*sm;
      csm = cm*sm*(1.0 - cp);
      sca = cm2 + sm2*cp; sm2 = sm2 + cm2*cp; cm2 = sca;
      m[ 0] =  co*cm2 - so*csm; m[ 1] =  co*csm - so*sm2; m[ 2] = sp*(co*sm + so*cm);
      m[ 4] =  so*cm2 + co*csm; m[ 5] =  so*csm + co*sm2; m[ 6] = sp*(so*sm - co*cm);
      m[ 8] =          - sm*sp; m[ 9] =            cm*sp; m[10] =                 cp;
      m[12] = Drwgl_FixX - (m[ 0]*Drwgl_FixX + m[ 4]*Drwgl_FixY + m[ 8]*Drwgl_FixZ);
      m[13] = Drwgl_FixY - (m[ 1]*Drwgl_FixX + m[ 5]*Drwgl_FixY + m[ 9]*Drwgl_FixZ);
      m[14] = Drwgl_FixZ - (m[ 2]*Drwgl_FixX + m[ 6]*Drwgl_FixY + m[10]*Drwgl_FixZ);
      m[ 3] =        0.0; m[ 7] =       0.0; m[11] =       0.0; m[15] =         1.0;

//    for (i=0;i<16;i++) md[i] = m[i];
//    Show_Mat( " View Change Mat ", md );

      /* Multiply at left */
      DrwGL_Mul_Mat( svd_mat, m, Drwgl_ViewMat );
    }
  }
}



void DrwGL_Change_Posit_Matrix()
{
  static GLfloat m[16] = { 1.0, 0.0, 0.0, 0.0,
                           0.0, 1.0, 0.0, 0.0,
                           0.0, 0.0, 1.0, 0.0,
                           0.0, 0.0, 0.0, 1.0 };

  if (Drwgl_ViewMat) {
    /* Get the first point distance (with center origine) */
    m[12] = Drwgl_xcurr - Drwgl_xfirst;
    m[13] = Drwgl_ycurr - Drwgl_yfirst;

    /* Multiply at left */
    DrwGL_Mul_Mat( svd_mat, m, Drwgl_ViewMat );
  }
}



int DrwGl_Finish_View_Request()
{
  int i;

  if (Drwgl_ViewMat&&rst_view) {
    for (i = 0; i<16;i++) Drwgl_ViewMat[i] = svd_mat[i];
    return 1;
  }
  else
  return 0;
}



void DrwGL_Update_Display( void )
{ /* Update the Display */
  fprintf( fmsg, " Wait for Refresh sequence\n" );
  fflush( fmsg );
  Drwgl_flags |= DRWSTATUS_UPDATE; /* Set the Picture Update Flag */
  // ... and wait for Request Done.
}




void DrwGL_Inp_Req_Display( Char * bupdate )
{ // Set the DrawGL Flags when required.void          DrwGL_Color( int );

  if (*bupdate)
  { Drwgl_flags |= DRWSTATUS_UPDATE + DRWSTATUS_INPUT;
    *bupdate = 0;
  }
  else Drwgl_flags |= DRWSTATUS_INPUT;
  // ... and wait for Request Done.
  DrwSrv_Wait_For_Req();                        // Wait for and of Request.
}





/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/


