/***** DRAW Interface ROUTINES for Graphic Server *******/
/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*          F L T K / D R A W   -   S E R V E R   M O D U L E            *
*                                                                       *
*           Version  1.4-C for Draw Library Level V 2.4 A               *
*                                                                       *
*      (Draw interface for FLTK/OPENGL function and procedures)         *
*                                                                       *
*              L4 Part : 2D/3D Axis/boxs Management                     *
*                                                                       *
*                               by                                      *
*                                                                       *
*          Pierre Wolfers, Laboratoire de Cristallographie              *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/

/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/



/* include the environment SERVER */

//#include <draw/sdrw_env.h>
//#include <draw/sdrw_routines.h>


#include "Server_DrawGL.h"




/***********************************************************************/
/**                                                                   **/
/**           C O N S T A N T E S    D E F I N I T I O N S            **/
/**                                                                   **/
/***********************************************************************/


//#  include <draw/draw_constantes.h>   /* Draw Server Constantes definitions */


#define  _DRAWLIB_L4BASE_  0            /* Set the L4 Base Mode of L4 Declaration */


#include "Server_AxisBox.h"


#define _LINMAX_ 10
#define _LOGMAX_  5

#define EPS       1.0E-5

#define LOG_E 0.434294481903



typedef  struct list_entr { /* Log List Entry Definition */
                  int     list_per;     /* Log Ticks Sequence Period */
                  float * list_lis;     /* List of Tick Factors */
                } list_entry;

typedef  struct tck_descr { /* Tick Descriptor Record Definition */
                  Char  tckd_freq,      /* Tick Frequency */
                        tckd_shift,     /* Tick Shift */
                        tckd_kind;      /* Tick Kind */
                  float tckd_size;      /* Tick size */
                } tckd_rec;

typedef  struct tcktb_rec { /* Ticks Table Set Record Definition */
                  float     tcktb_px,   /* X and Y Ticks Value Shift */
                            tcktb_py,
                            tcktb_del;  /* Default Delta Interval */
                  int       tcktb_len;  /* Number of Ticks to use */
                  tckd_rec* tcktb_tab;  /* Ticks Table pointer */
                } tcktb_rec;






/**************************************************************************/
/*                                                                        */
/*                     DRAW Server Local Variables                        */
/*                                                                        */
/**************************************************************************/


/*    * * * *  Ticks tables for each ticks model  * * * *    */

          /*    * * *  Ticks model table  * * *     */
/*
     Each ticks model is described by one line with a set of three integers
      and one real : { <tf>, <sh>, <kn>, <sz> } for each kind of ticks.

     These four numbers are respectively:
        The tick frequency <tf>   is the number of repetition of the tick
                                  before pass to the following tick set.

        The tick shift     <sh>   is the shift in tick number to select this ticks.
                                  the tick set is selected when <nt> mod <tf> == <sh>
                                  with <nt> that denots the current tick number from 0
                                  to last tick on the axis.

        The tick kind      <kn>   is the code of the tick kind as:
                                     0 - for single left tick (its the default),
                                     1 - for single right tick,
                                     2 - for centered single tick,
                                     3 - x,y,z aligned cross for 3D axis,
                                     4 - x,y,z diagonal cross for 3D axis.

        the tick size      <sz>   is the size of each tick in cm (when the initial work space
                                  tranformation in keep).

     It is possible to describe a maximum of four (4) tick set. The tick set selection
     is performed from the first to the last.

     For example, the following table ltck_md3[] define a ticks model with three kind of ticks
     similar the a regular cm graduation :
        The first set defines a tick of 0.6 cm each ten (10) graduations (or mm?),
        the second one defines a tick of 0.4 cm each five (5) graduations (or mm?),
        and the last one defines a tick of 0.2 cm for each other graduation because
        <nt> mod 1 = 0 (=<sh>) always. The first set that satisfies the relation
        <nt> mod <tf> == <sh> is selected.

*/

static tckd_rec ltck_md0[] = { { 1, 0, 1, 0.2 } };
static tckd_rec ltck_md1[] = { { 2, 0, 1, 0.4 }, { 1, 0, 1, 0.2 } };
static tckd_rec ltck_md2[] = { { 5, 0, 1, 0.6 }, { 1, 0, 1, 0.2 } };
static tckd_rec ltck_md3[] = { {10, 0, 1, 0.6 }, { 5, 0, 1, 0.4 }, { 1, 0, 1, 0.2 } };
static tckd_rec ltck_md4[] = { {10, 0, 1, 0.5 }, { 1, 0, 1, 0.2 } };
static tckd_rec ltck_md5[] = { { 4, 0, 1, 0.5 }, { 1, 0, 1, 0.2 } };
static tckd_rec ltck_md6[] = { {20, 0, 1, 0.6 }, { 4, 0, 1, 0.4 }, { 1, 0, 1, 0.2 } };
static tckd_rec ltck_md7[] = { {20, 0, 1, 0.8 }, {10, 0, 1, 0.5 },
                               { 5, 0, 1, 0.4 }, { 1, 0, 1, 0.2 } };
static tckd_rec ltck_md8[] = { { 9, 0, 1, 0.4 }, { 1, 0, 1, 0.2 } };
static tckd_rec ltck_md9[] = { {18, 0, 1, 0.6 }, { 9, 0, 1, 0.4 }, { 1, 0, 1, 0.2 } };

static tckd_rec ltck_mda[] = { { 3, 0, 1, 0.4 }, { 1, 0, 1, 0.2 } };
static tckd_rec ltck_mdb[] = { { 9, 0, 1, 0.6 }, { 9, 4, 1, 0.4 }, { 1, 0, 1, 0.2 } };


/*    * * *       Ticks model descriptors       * * *      */

/*
     The model descriptor give three real, one integer and a pointer to the ticks model
     respectively denoted <px>, <py>, <delta>, <nmd> and <ptr>.

     <ptr>        is the reference to the ticks model.

     <px>, <py>   is the shift vector componantes to place the string of the tick values.
                  Only the first tick set is used for graduation values.

     <delta>      The standard size of tick interval.

     <len>        is the number of tick set in the model (in the range [1..4].
*/

/*    * * *   Linear ticks model descriptors    * * *      */

static tcktb_rec list_tck_lin[_LINMAX_] = {
         {-0.5, 0.5, 2.5, 1, ltck_md0 },
         {-0.5, 0.7, 1.0, 2, ltck_md1 },
         {-0.5, 0.8, 1.0, 2, ltck_md2 },
         {-0.5, 0.9, 1.0, 3, ltck_md3 },
         {-0.5, 0.8, 1.0, 2, ltck_md4 },
         {-0.5, 0.8, 1.0, 2, ltck_md5 },
         {-0.5, 0.9, 1.0, 3, ltck_md6 },
         {-0.9, 0.9, 0.1, 4, ltck_md7 },
         {-0.5, 0.8, 1.0, 2, ltck_md8 },
         {-0.5, 0.9, 1.0, 3, ltck_md9 }
       };

/*    * * *   Logarithmic ticks model descriptors    * * *     */

static tcktb_rec list_tck_log[_LOGMAX_] = {
         {-0.5, 0.9, 0.0, 3, ltck_mdb },
         {-0.5, 0.8, 0.0, 2, ltck_md2 },
         {-0.5, 0.7, 0.0, 2, ltck_mda },
         {-0.5, 0.7, 0.0, 2, ltck_md1 },
         {-0.5, 0.5, 0.0, 1, ltck_md0 }
       };



static float list_log0[] = { 1.0,  2.0,  3.0,  4.0,  5.0,  6.0,  7.0,  8.0,  9.0 };
static float list_log1[] = { 1.0,  2.0,  3.0,  5.0,  7.5 };
static float list_log2[] = { 1.0,  2.5,  5.0 };
static float list_log3[] = { 1.0,  5.0 };
static float list_log4[] = { 1.0 };


static list_entry log_list[_LOGMAX_] = {
   {  9, list_log0 },
   {  5, list_log1 },
   {  3, list_log2 },
   {  2, list_log3 },
   {  1, list_log4 }
};


static float* open_dirmat;              /* Direct and reciprocal ... */
static float* open_recmat;              /* ... matrix transf. for opened Box */

static int    open_xlogf,               /* Log Coord flag for each dimension */
              open_ylogf,
              open_zlogf;





/**************************************************************************/
/*                                                                        */
/*                    DRAW Server Axis/Box Routines                       */
/*                                                                        */
/**************************************************************************/



int Mod( int i, int j )
{ int r;
  r = i%j;
  return (r<0)?j+r:r;
}




static void Draw__NormV( Draw_Point3 vect )
{
  float n;

  n = sqrt( vect[0]*vect[0] + vect[1]*vect[1] + vect[2]*vect[2] );
  if (n > 1.0e-6) {
    vect[0] /= n;
    vect[1] /= n;
    vect[2] /= n;
  }
}



/*
static void Sho_VGL_Matrix( float* m )
{
  int i;

  for (i = 0; i < 3; i++)
    fprintf( fmsg, " [ %16f  %16f  %16f  %16f ]\n",
                   m[i + 0], m[i + 3], m[i + 6], m[i + 9] );
  fflush( fmsg );
}
*/


static void Draw__RMul_Vect( float* r, float* m, float* v )
{
  r[0] = m[0]*v[0] + m[3]*v[1] + m[6]*v[2];
  r[1] = m[1]*v[0] + m[4]*v[1] + m[7]*v[2];
  r[2] = m[2]*v[0] + m[5]*v[1] + m[8]*v[2];
}



/*
static void Draw__Mul_Matrix( float* m1, float* m2, float* m3 )
{
  m3[ 0] = m1[ 0]*m2[ 0] +  m1[ 3]*m2[ 1] + m1[ 6]*m2[ 2];
  m3[ 3] = m1[ 0]*m2[ 3] +  m1[ 3]*m2[ 4] + m1[ 6]*m2[ 5];
  m3[ 6] = m1[ 0]*m2[ 6] +  m1[ 3]*m2[ 7] + m1[ 6]*m2[ 8];
  m3[ 9] = m1[ 0]*m2[ 9] +  m1[ 3]*m2[10] + m1[ 6]*m2[11] + m1[ 9];

  m3[ 1] = m1[ 1]*m2[ 0] +  m1[ 4]*m2[ 1] + m1[ 7]*m2[ 2];
  m3[ 4] = m1[ 1]*m2[ 3] +  m1[ 4]*m2[ 4] + m1[ 7]*m2[ 5];
  m3[ 7] = m1[ 1]*m2[ 6] +  m1[ 4]*m2[ 7] + m1[ 7]*m2[ 8];
  m3[10] = m1[ 1]*m2[ 9] +  m1[ 4]*m2[10] + m1[ 7]*m2[11] + m1[10];

  m3[ 2] = m1[ 2]*m2[ 0] +  m1[ 5]*m2[ 1] + m1[ 8]*m2[ 2];
  m3[ 5] = m1[ 2]*m2[ 3] +  m1[ 5]*m2[ 4] + m1[ 8]*m2[ 5];
  m3[ 8] = m1[ 2]*m2[ 6] +  m1[ 5]*m2[ 7] + m1[ 8]*m2[ 8];
  m3[11] = m1[ 2]*m2[ 9] +  m1[ 5]*m2[10] + m1[ 8]*m2[11] + m1[11];
}
*/


static void Draw__InvMat( float* m, float* n )
{
  float d;

  d = m[ 0]*(m[ 4]*m[ 8] - m[ 7]*m[ 5]) +
      m[ 3]*(m[ 7]*m[ 2] - m[ 1]*m[ 8]) +
      m[ 6]*(m[ 1]*m[ 5] - m[ 4]*m[ 2]);

  if (d != 0.0) {
    n[ 0] = (m[ 4]*m[ 8] - m[ 7]*m[ 5])/d;
    n[ 3] = (m[ 5]*m[ 6] - m[ 8]*m[ 3])/d;
    n[ 6] = (m[ 3]*m[ 7] - m[ 6]*m[ 4])/d;
    n[ 1] = (m[ 7]*m[ 2] - m[ 1]*m[ 8])/d;
    n[ 4] = (m[ 8]*m[ 0] - m[ 2]*m[ 6])/d;
    n[ 7] = (m[ 6]*m[ 1] - m[ 0]*m[ 7])/d;
    n[ 2] = (m[ 1]*m[ 5] - m[ 4]*m[ 2])/d;
    n[ 5] = (m[ 2]*m[ 3] - m[ 5]*m[ 0])/d;
    n[ 8] = (m[ 0]*m[ 4] - m[ 3]*m[ 1])/d;

    n[ 9] = - (n[ 0]*m[ 9] + n[ 3]*m[10] + n[ 6]*m[11]);
    n[10] = - (n[ 1]*m[ 9] + n[ 4]*m[10] + n[ 7]*m[11]);
    n[11] = - (n[ 2]*m[ 9] + n[ 5]*m[10] + n[ 8]*m[11]);
  }
}



static void Draw__Set_DirMat( Draw_Box_Ptr p )
{
  int i;
  float sca, orx, ory, orz;
  float* m;
  Draw_Axis_Ptr u, v, w;

  u = p->box_axis[0];
  v = p->box_axis[1];
  w = p->box_axis[2];

  m = p->box_dirmat;

  for (i = 0; i < 12; i++) m[i] = (i%4 ==0)?1.0:0.0;

  /* Get the physical coordinate values at axis origine and related axis scale */
  if (u) {
    /* Get the Scale/Positioning Info. */
    sca = u->axis_scale;
    orx = u->axis_evinf;
    /* Set the u rotation part of the matrix */
    for (i = 0; i < 3; i++) m[i + 0] = u->axis_vct[i]*sca;
  } else orx = 0.0;

  if (v) {
    /* Get the Scale/Positioning Info. */
    sca = v->axis_scale;
    ory   = v->axis_evinf;
    /* Set the v rotation part of the matrix */
    for (i = 0; i < 3; i++) m[i + 3] = v->axis_vct[i]*sca;
  } else ory = 0.0;

  if (w) {
    /* Get the Scale/Positioning Info. */
    sca = w->axis_scale;
    orz   = w->axis_evinf;
    /* Set the w rotation part of the matrix */
    for (i = 0; i < 3; i++) m[i + 6] = w->axis_vct[i]*sca;
  } else orz = 0.0;

  /* Set the Translation part of the scale matrix part */
  m[ 9] = p->box_org[0] - (m[ 0]*orx + m[ 3]*ory + m[ 6]*orz);
  m[10] = p->box_org[1] - (m[ 1]*orx + m[ 4]*ory + m[ 7]*orz);
  m[11] = p->box_org[2] - (m[ 2]*orx + m[ 5]*ory + m[ 8]*orz);
  if (!draw_mat_cnt) { m[ 9] += orgx; m[10] += orgy; m[11] += orgz; }      
}





static void Draw__Axis_Plane( Draw_Point3 org,
                              Draw_Point3 vct,
                              Draw_Point3 upd,
                              float* mat, float* mst )
{
  Draw_Point3 axy, axz;
  int   i;

  axz[0] = vct[1]*upd[2] - vct[2]*upd[1];
  axz[1] = vct[2]*upd[0] - vct[0]*upd[2];
  axz[2] = vct[0]*upd[1] - vct[1]*upd[0];
  Draw__NormV( axz );
  axy[0] = axz[1]*vct[2] - axz[2]*vct[1];
  axy[1] = axz[2]*vct[0] - axz[0]*vct[2];
  axy[2] = axz[0]*vct[1] - axz[1]*vct[0];
  for (i = 0; i < 3; i++) {
    mst[i + 0] = mat[i + 0] = vct[i];
    mst[i + 3] = axy[i]; mat[i + 3] = upd[i];
    mst[i + 6] = mat[i + 6] = axz[i];
    mst[i + 9] = mat[i + 9] = org[i];
  }

//fprintf( fmsg, " Axis Coordinate matrix.\n" );
//Sho_VGL_Matrix( mat );
//fprintf( fmsg, " Axis String matrix.\n" );
//fflush( fmsg );
//Sho_VGL_Matrix( mst );
}



static void Draw__Set_Clip_Eq( Draw_Point3    v,
                               Draw_Point3    w,
                               Draw_Point3 linf,
                               Draw_Point3 lsup,
                               float *        e )
{ /* To Set a Clipping Plane Equation */
  float n[3], nrm;

  n[0] = v[1]*w[2] - v[2]*w[1]; nrm  = n[0]*n[0];
  n[1] = v[2]*w[0] - v[0]*w[2]; nrm += n[1]*n[1];
  n[2] = v[0]*w[1] - v[1]*w[0]; nrm += n[2]*n[2];

  if (nrm > 0.0) {
    e[4] = -(e[0] = n[0]/nrm);
    e[5] = -(e[1] = n[1]/nrm);
    e[6] = -(e[2] = n[2]/nrm);
    e[3] = - (n[0]*linf[0] + n[1]*linf[1] + n[2]*linf[2] );
    e[7] =   (n[0]*lsup[0] + n[1]*lsup[1] + n[2]*lsup[2] );
  }
}




void Draw_Scale( float x, float y, float * tb, int ip ) 
{ /* 2D Direct coordinates transformation Physical Coord. -> Cm 3D space */
  if (Draw_Opened_Box) {
    if (open_xlogf&&(x>0.0)) x = log10( x );
    if (open_ylogf&&(y>0.0)) y = log10( y );
    tb[ip+0] = open_dirmat[ 0]*x + open_dirmat[ 3]*y + open_dirmat[ 9];
    tb[ip+1] = open_dirmat[ 1]*x + open_dirmat[ 4]*y + open_dirmat[10];
    tb[ip+2] = open_dirmat[ 2]*x + open_dirmat[ 5]*y + open_dirmat[11];
  }
}



void Draw_Scale3( float  x, float  y, float  z, float * tb, int ip )
{ /* 3D Direct coordinates transformation Physical Coord. -> Cm 3D space */
  if (Draw_Opened_Box) {
    if (open_xlogf&&(x>0.0)) x = log10( x );
    if (open_ylogf&&(y>0.0)) y = log10( y );
    if (open_zlogf&&(z>0.0)) z = log10( z );
    tb[ip+0] = open_dirmat[ 0]*x + open_dirmat[ 3]*y + open_dirmat[ 6]*z + open_dirmat[ 9];
    tb[ip+1] = open_dirmat[ 1]*x + open_dirmat[ 4]*y + open_dirmat[ 7]*z + open_dirmat[10];
    tb[ip+2] = open_dirmat[ 2]*x + open_dirmat[ 5]*y + open_dirmat[ 8]*z + open_dirmat[11];
  }
}




void Draw_ScaleNormal( float  x, float  y, float  z, float * tb, int ip )
{ /* 3D Direct Normal componante transformation Physical Coord. -> Cm 3D space */
  float nx, ny, nz, nm;
  if (Draw_Opened_Box) {
    if (open_xlogf&&(x>0.0)) x = log10( x );
    if (open_ylogf&&(y>0.0)) y = log10( y );
    if (open_zlogf&&(z>0.0)) z = log10( z );
    nx = open_dirmat[ 0]*x + open_dirmat[ 3]*y + open_dirmat[ 6]*z + open_dirmat[ 9];
    ny = open_dirmat[ 1]*x + open_dirmat[ 4]*y + open_dirmat[ 7]*z + open_dirmat[10];
    nz = open_dirmat[ 2]*x + open_dirmat[ 5]*y + open_dirmat[ 8]*z + open_dirmat[11];
    nm = nx*nx + ny*ny + nz*nz;
    if (nm > 1.e-6) { /* Normalized vector */
      nm = sqrt( nm );
      nx /= nm; ny /= ny; nz /=nz;
    }
    tb[ip+0] = nx;
    tb[ip+1] = ny;
    tb[ip+2] = nz;
  }
}




void Draw_UnScale( float  *x, float  *y, float *z, float xi, float yi, float zi )
{ /* 3D Direct coordinates transf. from Cm 2D space -> Physical Coord. 3D */
  float xx, yy, zz;

  if (Draw_Opened_Box&&open_recmat) {
    /* Transformation */
    xx = open_recmat[ 0]*xi + open_recmat[ 3]*yi + open_recmat[ 6]*zi + open_recmat[ 9];
    yy = open_recmat[ 1]*xi + open_recmat[ 4]*yi + open_recmat[ 7]*zi + open_recmat[10];
    zz = open_recmat[ 2]*xi + open_recmat[ 5]*yi + open_recmat[ 8]*zi + open_recmat[11];

    if (open_xlogf&&(xx<30.0)) xx = pow( 10.0, xx );
    if (open_ylogf&&(yy<30.0)) yy = pow( 10.0, yy );
    if (open_zlogf&&(zz<30.0)) zz = pow( 10.0, zz );

    *x = xx; *y = yy; *z = zz;

  } else {
    /* 2D simulation */
    *x = xi; *y = yi; *z = 0.0;

  }
}




Draw_Axis_Ptr Draw_Locate_Axis( int id )
{
  Draw_Axis_Ptr p;

  p = Draw_First_Axis;    
  while (p)
    if (p->axis_ident == id) return p;
                        else p = p->axis_next;
  fprintf( fmsg, " Undefined Axis # %d\n", id );
  fflush( fmsg );
  return NULL;
}




Draw_Box_Ptr  Draw_Locate_Box( int id )
{
  Draw_Box_Ptr p;

  p = Draw_First_Box;
  while (p)
    if (p->box_ident == id) break;
                        else p = p->box_next;
  if (!p) {
    fprintf( fmsg, " Undefined Box # %d\n", id );
    fflush( fmsg );
  }
  return p;
}




/************************************************************************/
/*                                                                      */
/*                   DRAW Server Axis Easy Routines                     */
/*                                                                      */
/************************************************************************/




void Draw_Axis_Inscale( float mi, float ma, int n,
                        float *inf, float *sup, float *delta, float *ef )
{ /**********************************************************************/
  /*                                                                    */
  /*    Procedure to get a set of suitable value for inf, sup and delta */
  /*    for an axis, from the min and max coordinate value and the      */
  /*    number of wanted intervals (n).                                 */
  /*    ef is the resulting filling efficience                          */
  /*                                                                    */
  /*                                                                    */
  /*    Draw_Inscale work on the User Coordinates, and not on their     */
  /*    Drawing Surface Coordinates (as Draw cm or paper coordinates).  */
  /*                                                                    */
  /**********************************************************************/

  int i, ip, i1;
  float lar, s, max, min;

  max = ma;
  min = mi;
  /* Permute min and max when required */
  if (max < min) { s   = max; max = min; min = s ; }
  lar = max - min;
  if (lar < 1.0e-10) lar = 1.0e-10;                /* Find the largest power of 10 under lar/n. */
  ip = (int)floor( log( lar/n )*LOG_E );
  i1 = abs(ip);
  s  = 10.0;
  if (ip < 0) s = 1.0/s;
  *delta = 1.0;
  while (i1-- > 0) *delta = (*delta)*s;            /* Compute the suitable Delta for the axis */

  i = 0;
  do {                                             /* Loop on the Choice of delta */
    *inf = floor( min/(*delta)+0.00001 )*(*delta); /* inf and sup must be int. mult. of delta */
    *sup = (*inf) + (*delta)*n;
    if ( (max - (*sup))/lar > 1e-3 ) {             /* If delta is too low, we try an other delta */
      switch((i % 5)+1) {                          /* in the range of 10^ip * by (2, 2.5, 4, 5, 10) */
        case 1:
        case 5: *delta = (*delta)*2.0;  break;     /* Delta is 2Exxx or 10Exxx */

        case 2:
        case 4: *delta = (*delta)*1.25; break;     /* Delta is 2.5Exxx or 5Exxx */

        case 3: *delta = (*delta)*1.6;  break;     /* Delta is 4Exxx */

        default: break;
      } /*switch*/
      i++;                                         /* Increment i to try a new delta value */
    } /*if*/
  } while (((max - (*sup))/lar) > 1e-3);
  *ef = 100.0*lar/((*sup) - (*inf));               /* The efficience factor is given in percent */
} /* Draw_Axis_Inscale */




void Draw_Axis_Setvscale( float  inf, float sup,  float  delta, float  rsiz,
                          float *fsiz, float *eff, float *scale )
{ /*************************************************************************/
  /*                                                                       */
  /*       ***  Procedure Axis_Setvscale to set the virtual scale .        */
  /*       ***  inf, sup and delta are the minimum, maximum and increment  */
  /*                of the coordinate, (as computed by Draw_Inscale),      */
  /*       ***  szp is the required size of this axis,                     */
  /*       ***  psup is the resulting maximal value of the axis,           */
  /*       ***  stp  is interval in paper coordinate,                      */
  /*       ***  eff is the efficience (real), and                          */
  /*       ***  scale is the final value to applied for the plot,          */
  /*                                                                       */
  /*                                                                       */
  /*   Draw_Setvscale work on the Delta scale on the Drawing support       */
  /*   to get a suitable interval between each axis tick in term of        */
  /*   length in cm (to get a easely readable scale on a measuring paper). */
  /*                                                                       */
  /*************************************************************************/

  int               ip;
  float sc, sd, sf, st;

  sc = rsiz/(sup - inf);                           /* Get the original Scale in cm/user_unit */
  sd = delta*sc;                                   /* Set the required user interval in cm */
  ip = (int)floor( log( sd )*LOG_E + 0.00001 );    /* ipw = floor( LOG10( sd ) ) */

  if (ip < 0) sf = 0.1; else sf = 10.0;            /* Prepare sf to compute 10^ip */
  ip = abs( ip ); st = 1.0;
  while (ip > 0) { ip = ip - 1; st = st*sf; }      /* Now  st = (the largest power of 10) < the */
                                                   /* ... user interval on the paper. */
                                                   /* Select the appriopriate paper interval sc */

  switch((int)floor( (2.0*sd)/st + 0.00001 )) {    /* The selector is : floor( (2*sd)/sc) */
                                                   /* ... to scan the choices is (1, 2, 2.5, 4, 5 or 8)*st */
    case  4: st = st*2.0; break;                   /* The Interval (in Paper Unit) is 2.0Exxx cm */
    case  5:
    case  6:
    case  7: st = st*2.5; break;                   /* The Interval is 2.5Exxx cm */
    case  8:
    case  9: st = st*4.0; break;                   /* The Interval is 4.0Exxx cm */
    case 10:
    case 11:
    case 12:
    case 13:
    case 14:
    case 15: st = st*5.0; break;                   /* The Interval is 5.0Exxx cm */
    case 16:
    case 17:
    case 18:
    case 19: st = st*8.0; break;                   /* The Interval is 8.0Exxx cm */

    default: break;                                /* The interval is 1.0Exxx cm */
  } /*switch*/

  sc = st/delta;                                   /* Compute the scale */
  *fsiz  = (sup - inf)*sc;                         /* Set the resulting size of the axis */
  *eff   = st/sd;                                  /* Set the real efficience and the paper interval */
  *scale = sc;                                     /* Set the scale of paper transcription */
} /* Draw_Axis_Setvscale */




float Draw_Axis_AutoScale( Draw_Axis_Ptr ax, int ani, int ana )
{ /*************************************************************************/
  /*                                                                       */
  /*   ***  The autoscale function is used to find a suitable axis scale   */
  /*   ***  on a given Draw paper coordinate range.                        */
  /*                                                                       */
  /*   ***  The function result is the selected number of interval.        */
  /*                                                                       */
  /*   ***  ax is the axis structure pointer,                              */
  /*   ***  ani and ana are the mini maxi on the number of ticks,          */
  /*                                                                       */
  /*   ***  At return the axis value are reset to optimal values and       */
  /*   ***  the resulting efficience is returned.                          */
  /*                                                                       */
  /*************************************************************************/

  int nt;
  float min, max, szp, inf1, sup1, del1, eff, ef1, ef2, fma, sc1;

  min = ax->axis_vinf;
  max = ax->axis_vsup;
  szp = ax->axis_length;

  eff = 0.0;
  if ((ani > ana)||(ana <= 0)) {                   /* The default number in an interval of 1.0 cm. */
    ani = (int) floor( szp + 0.5 ); ana = ani;
  }

  /* We try for each allowed interval number */
  for (nt = ani; nt <= ana; nt++) {
    /* We get appropriate axis display value in first */
    Draw_Axis_Inscale( min, max, nt, &inf1, &sup1, &del1, &ef1 );
    /* Set the appropriate scale on the plot paper */
    Draw_Axis_Setvscale( inf1, sup1, del1, szp, &fma, &ef2, &sc1 );
    ef1 = ef1*ef2;                                 /* Get the resulting efficience in percent */
    if (ef1 > eff) {
      ax->axis_vinf   = inf1;                      /* Load the low limit in user coordinate */
      ax->axis_vsup   = sup1;                      /* Load the high limit in user coordinate */
      ax->axis_vincr  = del1;                      /* Load the increment in user coordinate */
      ax->axis_length = fma;                       /* Load the final size of the axis in cm. */
      ax->axis_scale  = sc1;                       /* Load the scale in cm/user_unit */
      ax->axis_tcknbr =  nt;                       /* Load the selected number of increment */
      eff = ef1;                                   /* Store any more efficient result */
    } /* if */
  } /* for */
  return eff;
} /* Draw_Axis_Autoscale */




/**************************************************************************/
/*                                                                        */
/*                  DRAW Server Build Axis/Box Routines                   */
/*                                                                        */
/**************************************************************************/



static void Draw__Set_Model_Ticks( Draw_Axis_Ptr p, int *ntc )
{ /* Routine called to set an axis model when it is required. */
  /* This call is imperative for a logarithmic axis.          */
  /* *ntc must be <= 0 as the - <ticks_model_number> */

  int    i, n, n1, n2, ns, itck;
  tckd_rec*    ptck;
  tcktb_rec*   pttb;
  float         eff;
  int      auto_flg;

//fprintf( fmsg, " SET_MODEL_TICKS : Axis Set Model Ticks ntc=%d.\n", *ntc );
//fflush( fmsg );

  n = *ntc;
  auto_flg = 0;                                /* Assume automatic flag off until shown otherwise */
  if (n == 0)
    if (p->axis_flags&AX_LOG) n = 1;
    else
    { /* The automatic linear axis limit adaptation is required  */
      ns = (int) (p->axis_length + 0.5);       /* Set the minimum increment at 1 cm */
      if (ns < 5) { n1 = 2; n2 = 4; }          /* Set default minimum and maximum of tick intervals */
             else { n2 = ns; n1 = n2 / 2; }
      eff = Draw_Axis_AutoScale( p, n1, n2 );  /* Determine the best number of ticks */
      *ntc = p->axis_tcknbr;                   /* Get the best number of ticks */
      ns = ns/p->axis_tcknbr;                  /* Get the rounded size of axis step */
      auto_flg = 1;                            /* Set the AutoScale setup Mode */
      if ((p->axis_tcknbr < 6)&&(ns > 2))
        n = 1;                                 /* Select the single tick kind mode */
      else {                                   /* Select a double tick model 1 or 2 */
        if (p->axis_tcknbr < 10) { n = 2; p->axis_unps[1] -= 0.2; }
                            else { n = 3; p->axis_unps[1] -= 0.3; }
      }
    }
  else  n = abs( n );                          /* Convert to positive tick model index */

  /* When the Ticks Auto Mode is set */
  if (p->axis_flags&AX_LOG) {                  /* For Log axis */
    if (n > _LOGMAX_) n = _LOGMAX_;            /* Control User specified axis ticks model index limit */
    *ntc = 0;                                  /* Force Log specific tick number and ... */
    auto_flg = 1;                              /* ... set automatic flag on */
    pttb = &(list_tck_log[n-1]);               /* Get the LOG ticks models address */
  } else {                                     /* For linear axis */
    if (n > _LINMAX_) n = _LINMAX_;            /* Control User specified axis ticks model index limit */
    pttb = &(list_tck_lin[n-1]);               /* Get the Linear ticks model descriptor address */
  }

  p->axis_vaps[0] = - pttb->tcktb_px;          /* Get the position shift vector for the graduation values  */
  p->axis_vaps[1] = - pttb->tcktb_py;
  p->axis_unps[1] = 2*p->axis_vaps[1];         /* Set the default Value font size */

  itck = pttb->tcktb_len;                      /* Get the number of used ticks */
  ptck = pttb->tcktb_tab;                      /* Get the tick model reference */

//fprintf( fmsg, " SET_MODEL_TICKS :  itck =%d, L = %f, D = %f.\n", itck, p->axis_length, pttb->tcktb_del );
//fflush( fmsg );

  if (!auto_flg) *ntc = (int) (fabs( p->axis_length/pttb->tcktb_del ) + 0.5);

  /* Now we set the Tick tables */
  for (i = 0; i < 4; i++)
    if (i < itck) {
      p->axis_tfre[i]   = ptck[i].tckd_freq;   /* Set the tick frequency, */
      p->axis_tshf[i]   = ptck[i].tckd_shift;  /* ... the tick shift, */
      p->axis_tknd[i]   = ptck[i].tckd_kind;   /* ... the tick kind, */
      p->axis_tcklen[i] = ptck[i].tckd_size;   /* ... and the tick size */
    } else {
      p->axis_tfre[i]   =   0;                 /* Disable the other possible ticks */
      p->axis_tshf[i]   =   0;
      p->axis_tknd[i]   =   0;
      p->axis_tcklen[i] = 0.0;
    }
}




Draw_Axis_Ptr Draw_New_Axis( Draw_Point3 vct,
                             Draw_Point3 upd,
                             float       len,
                             float       inf,
                             float       sup,
                             int         ntc,
                             int         flg )
{ /* The value of flg is a bit set as this :
       1 = AX_LOG     - Axis is in Log10 Coordinate,
       2 = AX_LEFT    - Ticks and displayed value at left position ( else at right ),
       8 = AX_NFRS    - Does not display the first Ticks,
      16 = AX_NFVL    - Does not display the first Value,
      32 = AX_NLST    - Does not display the last Ticks,
      64 = AX_NLVL    - Does not display the last Value,
     128 = AX_NZER    - Does not display the Zero Ticks,
     256 = AX_NZVL    - Does not display the Zero Value,
     512 = AX_ARRO    - Put an arrow at the end of axis,
    1024 = AX_ARRF    - Fill arrow at end of axis,
    2048 = AX_TILT    - Use Tilted Char for Ticks value,
    4096 = AX_VPATH   - String Value Path (First of two bits),
   16384 = AX_UPATH   - String Unit Path (First of two bits),
   65536 = AX_SPLIM   - Flag to enable the Coordinate simplification,
  131072 = AX_VLEFT   - Flags to set the Value string position,
  262144 = AX_VRIGHT
  524288 = AX_ULEFT   - Flag to set the Unit string positions, place -> Middle of axis if required.
 1048576 = AX_URIGHT
 2097152 = AX_UMIDLE

  */


  Draw_Axis_Ptr    p;
  int              i;
  float          del;
 
  /* Allocate a new Axis record */
  p = (Draw_Axis_Ptr) malloc( sizeof( Draw_Axis ) );
  if (Draw_First_Axis) Draw_Last_Axis->axis_next = p;
                  else Draw_First_Axis = p;
  Draw_Last_Axis = p;
  p->axis_next   = NULL;
  p->axis_ident  = ++Draw_Curr_AIde;   /* Set the Axis Identifier Number */
  p->axis_ticktb = NULL;

  for (i = 0; i < 3; i++) {
    p->axis_vct[i] = vct[i];           /* Set the Axis direction vector */
    p->axis_upd[i] = upd[i];           /* set the Axis ticks direction vector */
  }

  if (sup < inf) { del = inf; inf = sup; sup = del; }

//fprintf( fmsg, " NEW_AXIS: Flag=%d, ntck=%d, minmaxi=[%f,%f], size=%f.\n",
//               flg, ntc, inf, sup, len );
//fflush( fmsg );

  /* Set Default Linear mod with Ticks for First value, Numbers 0 and at Right */
  p->axis_ticktb    = NULL;            /* No allocated Tick table */
  p->axis_flags     =  flg;            /* Set the required mode */
  p->axis_length    =  len;            /* Set the Length (cm) of the axis */
  p->axis_vinf      =  inf;            /* Set the minimum and maximum ... */
  p->axis_vsup      =  sup;            /* ... of physical value along axis */
  p->axis_vhigh     =  0.4;            /* Defaulted at 0.4 cm for Value char high */
  p->axis_vangl     =  0.0;            /* Set The Def. Str. Orientation */
  p->axis_vaps[0]   =  0.0;            /* Set the values strings shift */
  p->axis_vaps[1]   = -0.5;
  p->axis_uhigh     =  0.6;            /* Defaulted at 0.6 cm for Unit char high */
  p->axis_uangl     =  0.0;            /* Set The Def. Str. Orientation */
  p->axis_unps[0]   =  len;            /* Set the units string position */
  p->axis_unps[1]   = -1.2;
  p->axis_scale     =  1.0;            /* Set the Axis Scale &t 1.0 */
  p->axis_eff       =  0.0;            /* Clear the efficience */
  p->axis_evinf     =  0.0;
  p->axis_evsup     =  0.0;
  p->axis_vfont     =    1;            /* Set a defaulted font number to use ... */
  p->axis_ufont     =    1;            /* ... for Values and Unit */
  p->axis_frstck    =    0;
  p->axis_lsttck    =    0;
  p->axis_unit      = NULL;            /* Set the Unit string to be empty */
  p->axis_eunit     = NULL;            /* Set the effective Unit string to be empty */

  if ((ntc <= 0)||(flg&AX_LOG)) Draw__Set_Model_Ticks( p, &ntc );
  else {
    p->axis_tcklen[0] =  0.2;          /* By default define two tick lengths */
    p->axis_tcklen[1] =  0.4;
    p->axis_tcklen[2] =  0.0;
    p->axis_tcklen[3] =  0.0;
    p->axis_tfre[0]   =    1;          /* Set the first ticks #1 frequency at One */
    p->axis_tshf[0]   =    0;          /* Set the First ticks Shift */
    p->axis_tknd[0]   =    1;          /* Single dash ticks */
    for (i = 1; i < 4; i++) {
      p->axis_tfre[i] =    0;          /* Disable the other ticks */
      p->axis_tshf[i] =    0;          /* Set the other ticks Shift and kinds */
      p->axis_tknd[i] =    0;
    }
  }
  p->axis_tcknbr    =  ntc;            /* Set required ticks number */
  p->axis_field     =    6;            /* Set default Value Field */
  p->axis_decm      =    2;            /* Set default Value Decimal Field */
  p->axis_refcnt    =    0;            /* Set as not referenced axis */

//fprintf( fmsg, " Create Axis # %d.\n", p->axis_ident );
//fflush( fmsg );

  return p;
} /* Draw_New_Axis */




int Draw_Gen_Axis( Draw_Point3 u_axis,
                   Draw_Point3 v_axis,
                   float       length,
                   float         vinf,
                   float         vsup,
                   int            ntc,
                   int          flags
                 )
{ /* Create a New Axis Record and return its Identifier Number */

  Draw_Axis_Ptr p;

  p = Draw_New_Axis( u_axis, v_axis, length, vinf, vsup, ntc, flags );

  if (p) return p->axis_ident;
    else return 0;
} /* Draw_Gen_Axis */




static void Draw__Free_Axis( Draw_Axis_Ptr p, Char axflg )
{
  Draw_Axis_Ptr q1, q2;
 
  q1 = NULL;
  q2 = Draw_First_Axis;
  if (p&&q2)
    if (((--(p->axis_refcnt)) <= 0)&&axflg) {
      while (q2&&(q2 != p)) { /* Look for previous of p */
        q1 = q2; q2 = q2->axis_next;
      }
      if (q2 == p) {
        /* When found, put out of axis list */
        if (q1) q1->axis_next = p->axis_next;
           else Draw_First_Axis = p->axis_next;
        if (p == Draw_Last_Axis) {
          Draw_Last_Axis = q1;
          Draw_Curr_AIde = p->axis_ident - 1;
        }
        if (Draw_Curr_Axis == p) Draw_Curr_Axis = NULL;
        if (p->axis_ticktb) free( p->axis_ticktb );
        if (p->axis_unit) free( p->axis_unit );
        if (p->axis_eunit) free( p->axis_eunit );
        free( p );
      }
    }
} /* Draw__Free_Axis */




Draw_Box_Ptr Draw_New_Box( Draw_Point3   org, /* The Box Origin Coordinates */
                           Draw_Axis_Ptr   u, /* The X Axis Reference */
                           Draw_Axis_Ptr   v, /* The Y Axis Reference */
                           Draw_Axis_Ptr   w, /* The Z Axis Reference */
                           int          nflg, /* Number of Axis Directives */
                           int         flg[], /* Box Directive Table */
                           int           nsh, /* Box Dir. Shift table size */
                           float      shtb[]  /* Box Dir. Shift table */
                         )
{
  Draw_Point3  psdw = { 0.0, 0.0, 1.0 },
                   axif;
  Draw_Box_Ptr    p = 0;
  int              i, j;

  if (u&&v) {
    p = (Draw_Box_Ptr) malloc( sizeof( Draw_Box ) );
    if (Draw_First_Box) Draw_Last_Box->box_next = p;
                   else Draw_First_Box = p;
    Draw_Last_Box  = p;
    p->box_next    = NULL;
    p->box_ident   = ++Draw_Curr_BIde; /* Set the Box Identifier Number */
    p->box_status  = 0;             /* Set the Box to Not Plotted Status */  

    /* Set the axis reference and update each axis ref. count */
    p->box_axis[0] = u; u->axis_refcnt++;
    p->box_axis[1] = v; v->axis_refcnt++;
    p->box_axis[2] = w; if (w) w->axis_refcnt++;

    /* Set the origine of box */
    for (i = 0; i < 3; i++) p->box_org[i] = org[i];

    /* Set the axis directive flags table */
    if (nflg > 16) nflg = 16;
    for (i = 0; i < nflg; i++) p->box_drtb[i] = flg[i];
    p->box_drtbsz = nflg;
    /* Set the axis directive shift table when required */
    if (nsh) {
      if (nsh > 48) nsh = 48;
      for (i = 0; i < nsh; i++) p->box_drshf[i] = shtb[i];
    }
    p->box_drshtbsz = nsh;



//  fprintf( fmsg, " Box Directive table # %d.\n", p->box_ident );
//  for (i = 0; i < nflg; i++) fprintf( fmsg, " %d ", p->box_drtb[i] );
//  fprintf( fmsg, "\n" );
//  fflush( fmsg );

    /* Initialize the Transformation Matrix(s) */
    for (i = 0; i < 3; i++)
      for (j = 0; j < 4; j++) {
        p->box_dirmat[3*i+j] = (i == j)?1.0:0.0;
        p->box_recmat[3*i+j] = (i == j)?1.0:0.0;
      }

    /* Compute the Box Clip plane equations */
    axif[0] = org[0] + u->axis_length;
    axif[1] = org[1];
    axif[2] = org[2];
    if (w) { /* 3D Mode */
      Draw__Set_Clip_Eq( v->axis_vct, w->axis_vct, org, axif, p->box_clip +  0 );
      axif[0] = org[0];
      axif[1] = org[1] + v->axis_length;
      Draw__Set_Clip_Eq( w->axis_vct, u->axis_vct, org, axif, p->box_clip +  8 );
      axif[1] = org[1];
      axif[2] = org[2] + w->axis_length;
      Draw__Set_Clip_Eq( u->axis_vct, v->axis_vct, org, axif, p->box_clip + 12 );
    } else { /* 2D Mode */
      Draw__Set_Clip_Eq( v->axis_vct, psdw, org, axif, p->box_clip + 0 );
      axif[0] = org[0];
      axif[1] = org[1] + v->axis_length;
      Draw__Set_Clip_Eq( psdw, u->axis_vct, org, axif, p->box_clip + 8 );
    }

    /* Set the Initial Current Pen Position */
    for (i = 0; i < 3; i++)
      p->box_currp[i] = (p->box_axis[i])?p->box_axis[i]->axis_vinf:0.0;
    /* Set the Curve reference list to the empty state */
    p->box_refirst = p->box_reflast = NULL;
  }

//fprintf( fmsg, " Create Box # %d.\n", p->box_ident );
//fflush( fmsg );

  return p;
} /* Draw_New_Box */




int Draw_Gen_Box( Draw_Point3    org, /* The Box Origin Coordinates */
                  int            idx, /* Index of the X Axis */
                  int            idy, /* Index of the Y Axis */
                  int            idz, /* Index of the Z Axis */
                  int           nflg, /* Number of Box Directives */
                  int          flg[], /* Box Directive Table */
                  int            nsh, /* Box Dir. Shift table size */
                  float       shtb[]  /* Box Dir. Shift table */
                )
{ /* Create a New Box Record and return its Identifier Number */
  Draw_Box_Ptr p;

  p = Draw_New_Box( org, Draw_Locate_Axis( idx ),
                         Draw_Locate_Axis( idy ),
                         (idz)?Draw_Locate_Axis( idz ):NULL,
                         nflg, flg, nsh, shtb );

  if (p) return p->box_ident;
    else return 0;
} /* Draw_Gen_Box */




int Draw_Easy_Box_2D( float    orgx,  /* The Box Origin Coordinates */
                      float    orgy,
                      float    aszx,  /* Sizes of the axis */
                      float    aszy,
                      float *  mxyz,  /* Minimaxi of coordinates */
                      char  *  untx,  /* Unit string for x and y */
                      char  *  unty,
                      int      nsch   /* Schema type of coordinates */
                    )
{ /* Easy system to create a 2D axis box from a standard model */
  Draw_Point3   p1, p2;
  Draw_Axis_Ptr px, py;
  Draw_Box_Ptr      pb;
  int       box_dir[4];
  float     box_shf[4];
  int         ndr, nsh;

  /* Create the X axis with ticks and values at right */
  p1[0] = 1.0; p1[1] = 0.0; p1[2] = 0.0;
  p2[0] = 0.0; p2[1] = 1.0; p2[2] = 0.0;
  px = Draw_New_Axis( p1, p2, aszx, mxyz[0], mxyz[1], 0, AX_URIGHT+AX_SPLIM );
  px->axis_unit = (char*) malloc( strlen( untx ) + 1 );
  strcpy( px->axis_unit, untx );

//fprintf( fmsg, " Axis X created with id = %d, '%s'.\n",
//               px->axis_ident, px->axis_unit );
//fflush( fmsg );

//fprintf( fmsg, " Create the Y axis of model %d.\n", nsch );
//fflush( fmsg );

  /* Create the Y axis with ticks and values at left */
  p1[0] =-1.0;
  py = Draw_New_Axis( p2, p1, aszy, mxyz[2], mxyz[3], 0, AX_URIGHT+AX_SPLIM );
  py->axis_unit = (char*) malloc( strlen( unty ) + 1 );
  strcpy( py->axis_unit, unty );

//fprintf( fmsg, " Axis Y created with id = %d, '%s'..\n",
//               py->axis_ident, py->axis_unit );
//fflush( fmsg );

  switch (nsch) {
    case 1:  /* Box with of axis crossing to lower-left corner */
      box_dir[0] = BX_VALU + 1;                /* Plot X axis with ticks and values */
      box_dir[1] = BX_VALU + BX_TRVM + 2;      /* Plot Y axis with ticks at left side */
      ndr = 2;
      nsh = 0;
     break;

    case 2:  /* Box with of axis crossing to origine */
      box_dir[0] = BX_YSHPOS + BX_TSYM +       /* Plot a Y shifted X axis with ... */
                   BX_VALU           + 1;      /* symetrised ticks with and values */
      box_dir[1] = BX_XSHPOS + BX_TSYM +       /* Plot a X shifted Y axis with ... */
                   BX_VALU + BX_TRVM + 2;      /* symetrised ticks with and values */
      ndr = 2;
      box_shf[0] = 0.0;                        /* Shift To Y = 0.0 value for X axis */
      box_shf[1] = 0.0;                        /* Shift To X = 0.0 value for Y axis */
      nsh = 2;
    break;

    default: /* Default to rectangular box */
      box_dir[0] = BX_VALU           + 1;      /* Plot X axis with ticks and values */
      box_dir[1] = BX_VALU + BX_TRVM + 2;      /* ... and also Y axis at left side */
      box_dir[2] = BX_YSHEND+BX_TRVM + 1;      /* Plot other box edge without ... */
      box_dir[3] = BX_XSHEND         + 2;      /* ... ticks values */
      ndr = 4; nsh = 0;
    break;
  }

  p1[0] = orgx; p1[1] = orgy;
  pb = Draw_New_Box( p1, px, py, NULL, ndr, box_dir, nsh, box_shf ); 

//fprintf( fmsg, " Easy Box created with id = %i.\n", pb->box_ident );
//fflush( fmsg );

  if (pb) return pb->box_ident;
     else return 0;
}




void Draw_Free_Box( Draw_Box_Ptr p, Char axflg )
{
  Draw_Box_Ptr  p1, p2;
  Draw_Axis_Ptr q;
  int i;

  if (Draw_Last_Box&&p) {
    p1 = NULL;
    p2 = Draw_First_Box;
    while (p2&&(p2 != p)) {
      p1 = p2; p2 = p1->box_next;
    }
    if (p2 == p) { /* The Box is located in the box list. */
      /* We suppress this Box of the box list */
      if (p1) p1->box_next = p2->box_next;
         else Draw_First_Box = p2->box_next;
      if (p2 == Draw_Last_Box) Draw_Last_Box = p1;
      /* Suppress the related box axis. */
      for (i = 0; i < 3; i++)
        if (q = p->box_axis[i]) Draw__Free_Axis( q, axflg );
      /* Free the box record */
      free( p );
    }
  }
} /* Draw_Free_Box */




void Draw_Free_Box_List()
{
  Draw_Box_Ptr  p;
  Draw_Axis_Ptr q;
  int i;

  if (Draw_Last_Box) {
    while (Draw_First_Box) {
      p = Draw_First_Box;
      Draw_First_Box = p->box_next;

      for (i = 0; i < 3; i++)
        if (q = p->box_axis[i]) Draw__Free_Axis( q, 1 );

      free( p );
    }
    Draw_Last_Box  = NULL;
    Draw_First_Box = NULL;
    Draw_Curr_AIde = 0;
    Draw_Curr_BIde = 0;
  }
  Draw_Curr_Box = NULL;
} /* Draw_Free_Box_List */





/**************************************************************************/
/*                                                                        */
/*                 DRAW Server Axis/Box Get/set Routines                  */
/*                                                                        */
/**************************************************************************/



void Draw_Axis_Setv( Draw_Axis_Ptr p, int cd )
{ /* To set an Axis parameter value (See from User task) */
  int   i, j, k, n;

//fprintf( fmsg, " Axis Setv code # %d.\n", cd );
//fflush( fmsg );

  if (p) { /* The Box is existing */
    p->axis_eff = 0.0; /* To force a new tick/value Computing */

    switch (cd) {
      case 0: /* Flags */
        p->axis_flags = Sdrw_Get_Int();
      break;

      case 1: /* Number of ticks and Driven Table of ticks */
        n = Sdrw_Get_Int();
        k = Sdrw_Get_Int();
//      fprintf( fmsg, " Set Tick  %d, %d.\n", n, k );
//      fflush( fmsg );
        if (k < 0) {
          k = -k;
          Draw__Set_Model_Ticks( p, &k );
          if (n != 0) p->axis_tcknbr = (n > 0)?n:k;
        } else {
          if (n > 0) p->axis_tcknbr = n;
          n = k&7; if (n > 4) n = 4;
          if (n > 0) {
            for (i = 0; i < n; i++) {
              if (k&16) { j =  Sdrw_Get_Int();
                          p->axis_tfre[i] = j&0xff;
                          p->axis_tshf[i] = (j>>8)&0xff;
                        }
              if (k&32) p->axis_tknd[i]   = Sdrw_Get_Int();
              if (k&64) p->axis_tcklen[i] = Sdrw_Get_Float();
            }
          }
        }
//      fprintf( fmsg, " Set Tick key %d.\n", k );
//      for (i = 0; i < 4; i++)
//        fprintf( fmsg, " Set Tick as %d, %d, %f.\n",
//                       p->axis_tfre[i], p->axis_tknd[i], p->axis_tcklen[i] );
//      fflush( fmsg );
      break;

      case 2: /* Tick Value String Attributs */
        n = Sdrw_Get_Int();
        if (n& 1) p->axis_vhigh   =   Sdrw_Get_Float();
        if (n& 2) p->axis_vangl   =   Sdrw_Get_Float();
        if (n& 4) p->axis_vaps[0] = - Sdrw_Get_Float();
        if (n& 8) p->axis_vaps[1] = - Sdrw_Get_Float();
        if (n&16) p->axis_vfont   =   Sdrw_Get_Int();
        if (n&32) p->axis_field   =   Sdrw_Get_Int();
        if (n&64) p->axis_decm    =   Sdrw_Get_Int();
//      fprintf( fmsg, " Set Tick V at %f, %f, %f, %f, %d, %d, %d.\n",
//                     p->axis_vhigh,   p->axis_vangl,
//                     p->axis_vaps[0], p->axis_vaps[1],
//                     p->axis_vfont,   p->axis_field,   p->axis_decm );
//      fflush( fmsg );
      break;

      case 3: /* Unit String */
        n = Sdrw_Get_Char();
        if (p->axis_unit) free( p->axis_unit );   p->axis_unit = NULL;
        if (p->axis_eunit) free( p->axis_eunit ); p->axis_eunit = NULL;
        if (n > 0) {
          p->axis_unit = (char*) malloc( n + 1 );
          for (i = 0; i < n; i++) p->axis_unit[i] = Sdrw_Get_Char();
          p->axis_unit[n] = 0;
        }
//      fprintf( fmsg, " Set Unit String of %d length : %s.\n", n, p->axis_unit );
//      fflush( fmsg );
      break;

      case 4: /* Unit String Attributs */
        n = Sdrw_Get_Int();
        if (n& 1) p->axis_uhigh   =   Sdrw_Get_Float();
        if (n& 2) p->axis_uangl   =   Sdrw_Get_Float();
        if (n& 4) p->axis_unps[0] = - Sdrw_Get_Float();
        if (n& 8) p->axis_unps[1] = - Sdrw_Get_Float();
        if (n&16) p->axis_ufont   =   Sdrw_Get_Int();
      break;

      case 5: /* Axis Length, Inf, Sup */
        n = Sdrw_Get_Int();
        if (n& 1) p->axis_length = Sdrw_Get_Float();
        if (n& 2) p->axis_vinf   = Sdrw_Get_Float();
        if (n& 4) p->axis_vsup   = Sdrw_Get_Float();
      break;

      default: ;
    }
  }
}




void Draw_Axis_Getv( Draw_Axis_Ptr p, int cd )
{  /* To get an Axis parameter value  (See from User task) */
  int   i, n;

//fprintf( fmsg, " Draw_Axis_Getv Code=%d\n", cd );
//fflush( fmsg );

  if (p) { /* The Box is existing */
    switch (cd) {
      case 0: /* Flags */
        Sdrw_Put_Int( p->axis_flags );
      break;

      case 1: /* Number of ticks and Driven Table of ticks */
        n = Sdrw_Get_Int(); if (n > 4) n = 4;
        Sdrw_Put_Int( p->axis_tcknbr );
        Sdrw_Put_Int( n );
        for (i = 0; i < n; i++)
          Sdrw_Put_Int( p->axis_tfre[i] + (p->axis_tshf[i]<<8) );
        for (i = 0; i < n; i++) Sdrw_Put_Int( p->axis_tknd[i] );
        for (i = 0; i < n; i++) Sdrw_Put_Float( p->axis_tcklen[i] );
      break;

      case 2: /* Tick Value String Attributs */
        Sdrw_Put_Float( p->axis_vhigh );
        Sdrw_Put_Float( p->axis_vangl );
        Sdrw_Put_Float( - p->axis_vaps[0] );
        Sdrw_Put_Float( - p->axis_vaps[1] );
        Sdrw_Put_Int( p->axis_vfont );
        Sdrw_Put_Int( p->axis_field );
        Sdrw_Put_Int( p->axis_decm );
      break;

      case 3: /* Unit String */
        Sdrw_Put_String( p->axis_unit, 0 );
      break;

      case 4: /* Unit String Attributs */
        Sdrw_Put_Float( p->axis_uhigh );
        Sdrw_Put_Float( p->axis_uangl );
        Sdrw_Put_Float( - p->axis_unps[0] );
        Sdrw_Put_Float( - p->axis_unps[1] );
        Sdrw_Put_Int( p->axis_ufont );
      break;

      case 5: /* Axis Length, Inf, Sup */
        Sdrw_Put_Float( p->axis_length );
        Sdrw_Put_Float( p->axis_vinf );
        Sdrw_Put_Float( p->axis_vsup );
      break;

      case 6: /* Axis Scale, Efficience, Inftck Val., Suptck Val.
                 and first and lastr ticks numbers  */
        Sdrw_Put_Float( p->axis_scale );
        Sdrw_Put_Float( p->axis_eff );
        if (p->axis_ticktb) {
          Sdrw_Put_Float( p->axis_ticktb[0] );
          i = p->axis_lsttck - p->axis_frstck;
          Sdrw_Put_Float( p->axis_ticktb[2*i] );
          Sdrw_Put_Int( p->axis_frstck );
          Sdrw_Put_Int( p->axis_lsttck );
        } else {
          Sdrw_Put_Float( 0.0 );
          Sdrw_Put_Float( 0.0 );
          Sdrw_Put_Int( 0 );
          Sdrw_Put_Int( 0 );
        }
      break;

      default: Sdrw_Put_Int( -1 );
    }
  }
  else Sdrw_Put_Int( -1 );
}



void Draw_Box_Getv( Draw_Box_Ptr p, int cd )
{ /* To get the box information */
  int i;

  if (p) { /* The Box is existing */
    switch (cd) {
      case 0: /* Get Axis identifiers */
        Sdrw_Put_Int( 3 );
        for (i = 0; i < 3; i++)
          if (p->box_axis[i]) Sdrw_Put_Int( p->box_axis[i]->axis_ident );
                         else Sdrw_Put_Int( 0 );
      break;

      case 1: /* Get Box Lower Left Corner Coordinates */
        Sdrw_Put_Int( 3 );
        for (i = 0; i < 3; i++) Sdrw_Put_Float( p->box_org[i] );
      break;

      case 2: /* Get Box Directive table */
        Sdrw_Put_Int( p->box_drtbsz );
        for (i = 0; i < p->box_drtbsz; i++) Sdrw_Put_Int( p->box_drtb[i] );
      break;

      case 3: /* Get Box Shift table */
        Sdrw_Put_Int( p->box_drshtbsz );
        for (i = 0; i < p->box_drshtbsz; i++) Sdrw_Put_Float( p->box_drshf[i] );
      break;

      default:  Sdrw_Put_Int( -1 );
    }  
  }
  else Sdrw_Put_Int( -1 );
}



void Draw_Enable_Axis( Draw_Axis_Ptr p )
{ /* Enable the axis record definition with its present
     parameter set to be ready for Plot by creation of
     ticks table. Two mode are managed: Linear and Log10 */
  int   c_t, flg, f_t,  ii,  ij, l_t, nsh, per, pr1;
  float del, fct, fml, inf, len, sca, sup, vsh, vsc;
  float   *  mti,   *  ptb;

  if (p&&(p->axis_eff <= 0.1)) { /* When an axis is specified */

    /* Normalize the two vectors */
    Draw__NormV( p->axis_vct ); Draw__NormV( p->axis_upd );

    flg = p->axis_flags;            /* Get the Axis mode */
    nsh = p->axis_tcknbr;           /* Get the required Total Ticks Number */
    inf = p->axis_vinf;             /* Get origine physical value */
    sup = p->axis_vsup;             /* Get final physical value */
    len = p->axis_length;           /* Get the Axis Length (in Cm) */
    ptb = p->axis_ticktb;           /* Get the Axis ticks table address */

    if ((flg&AX_LOG)&&(inf<=0.0)) {
      flg &= ~AX_LOG;               /* Disable Log10 mode when inf <= 0 */
      p->axis_flags = flg;
    }

//  fprintf( fmsg, " Flag=%d, ntck=%d, minmaxi=[%f,%f], size=%f.\n",
//                 flg, nsh, inf, sup, len );
//  fflush( fmsg );

    if (p->axis_eunit != NULL)
      if (p->axis_eunit != p->axis_unit) free( p->axis_eunit );

    if (flg&AX_LOG) { /* Log 10 Mode */
      /* No User specification => Default the Tick Sequence to 0 Seq. # */
      if (nsh < 0) nsh = abs( nsh );
      if (nsh >= _LOGMAX_) nsh = _LOGMAX_ - 1;
      per = log_list[nsh].list_per; /* Get the ticks sequence period */
      mti = log_list[nsh].list_lis; /* Get the multiplicator table index */
      pr1 = per - 1;

//    fprintf( fmsg, " nsh = %d, per = %d, mti[0] = %f.\n", nsh, per, mti[0] );
//    fflush( fmsg );

      /* Compute the first tick Number */
      sca = log10( inf );           /* Look for the inf log base */
      f_t = (inf < 1.0)?(int)(sca - (1.0 - EPS)):(int)(sca + EPS);
      fct = pow( 10.0, f_t );       /* Get the Lower Nearest power of ten */
      ii  = 0;
      while ((ii < pr1)&&(mti[ii]*fct < inf-EPS)) ii++;
      f_t = f_t*per + ii;           /* Set the first tick number */
      inf = mti[ii]*fct;            /* Adjust the inf value */

//    fprintf( fmsg, " sca = %f, fct = %f, f_t = %d, inf = %f.\n",
//                   sca, fct, f_t, inf );
//    fflush( fmsg );

      /* Compute the last tick Number */
      sca = log10( sup );           /* Look for the sup log base */
      l_t = (sup < 1.0)?(int)(sca - (1.0 - EPS)):(int)(sca + EPS);
      fct = pow( 10.0, l_t );       /* Get the Upper Nearest power of ten */
      ii  = 0;
      while ((ii < pr1)&&(mti[ii]*fct < sup-EPS)) ii++;
      l_t = l_t*per + ii;           /* Set the last tick number */
      sup = mti[ii]*fct;            /* Adjust the sup value */

//    fprintf( fmsg, " sca = %f, fct = %f, l_t = %d, sup = %f.\n",
//                   sca, fct, l_t, sup );
//    fflush( fmsg );

      /* Create the ticks table */
      if (ptb) free( ptb );         /* Free any previously existing ticks table */
      ptb = (float *) malloc( sizeof( float )*(l_t - f_t + 1)*2 );
      p->axis_ticktb = ptb;

      /* Fill the ticks table */
      c_t = f_t;                    /* Initialize the current ticks count */
      ii  = Mod(c_t++, per );       /* Find the modulo of c_t */
      fct  = 1.0/mti[ii++];         /* Set Origine factor */
      sca  = len/log10( sup/inf );  /* Set the Scale Factor */
      ij   =   0;
      ptb[ij++] = inf;              /* Put the Initial Tick Value */
      ptb[ij++] = 0.0;              /* Put the Initial Tick Position */
      while (c_t++ < l_t) {         /* Loop on all Ticks */
        if (ii >= per) { ii = 0; fct *= 10.0; }
        fml = fct*mti[ii++];        /* Get the vinf factor */
        /* Give the translation in vct unit */
        ptb[ij++] = inf*fml;        /* Put the Tick Value */
        ptb[ij++] = sca*log10( fml ); /* Put the Tick Position */

//      fprintf( fmsg, " fct = %f, fml = %f, t_val = %f, t_pos = %f.\n",
//                       fct, fml, ptb[ij-2], ptb[ij-1] );
//      fflush( fmsg );

      }
      ptb[ij++] = sup;              /* Put the Last Tick Value */
      ptb[ij++] = len;              /* Put the Last Tick Position */

      inf = log10( inf );
      sup = log10( sup );
      p->axis_scale = sca;          /* Set the scale value, (log10/cm) */
      p->axis_evinf = inf;          /* ... minimaxi of values (log10), and */
      p->axis_evsup = sup;
      /* Set the Axis Efficience */
      p->axis_eff   = log10( p->axis_vsup/p->axis_vinf )/(sup - inf);

//    fprintf( fmsg, " Scale = %f, inf = %f, sup = %f.\n",
//                   sca, inf, sup );
//    fflush( fmsg );

    } else { /* Linear Mode */
      /* Supply for Negative or null number of ticks */
      if (nsh <= 0) nsh = ((int) (len/2.5));
      del = (sup - inf)/nsh;        /* Get the interval in physical units */

      if (flg&AX_SPLIM) { /* For automatic adaptation of axis range */
        if (fabs( sup - inf ) < inf*100.0)          /* We must set a shift */
          vsh = 100.0*del*floor( inf/(100.0*del) ); /* Set shift value for displayed number along of axis */
        else vsh = 0.0;

        /* Compute the Value Power of Ten for the Value Multiplier */
        ij = (int) floor( log( del )*LOG_E );       /* vsc power of ten */
        if ((ij > -3) &&  (ij <= 2)) vsc = 1.0;
        else {
          ii = abs( ij ); fct = 10.0;
          if (ij < 0) fct = 1.0/fct;
          vsc = 1.0;
          while (ii > 0) { ii = ii - 1; vsc = vsc*fct; }
        } /* if */

        ii  = strlen( p->axis_unit );

        if (vsc != 1.0)
          if (vsh != 0.0) {         /* Form : (<unit_str> - <shift>)*<scale> */
            p->axis_eunit = (char*) malloc( ii + 27 );
            sprintf( p->axis_eunit, "(%s - %10g)*%10g", p->axis_unit, vsh, vsc );
          } else {                    /* Form : <unit_str>*<scale> */
            p->axis_eunit = (char*) malloc( ii + 12 );
            sprintf( p->axis_eunit, "%s*%10g", p->axis_unit, vsc );
          }
        else
          if (vsh != 0.0) {         /* Form : <unit_str> - <shift> */
            p->axis_eunit = (char*) malloc( ii + 14 );
            sprintf( p->axis_eunit, "%s - %10g", p->axis_unit, vsh );
          } else
            p->axis_eunit = p->axis_unit;
      } else { vsc = 1.0; vsh = 0.0; p->axis_eunit = NULL; }

      sca = log10( del );           /* Get the Lower Nearest Power of Ten */
      ii  = (sca < 0.0)?(int)(sca - (1.0 - EPS)):(int)(sca + EPS);
      sca = pow( 10.0, ii-2 );      /* ... divide by 100 */
      del = ((int)(del/sca + 0.5))*sca;/* Round Down to 1% of original value */
      f_t = (int)(inf/del);         /* Round Down inf in del interval */
      if (inf < f_t*del - EPS) f_t--;
      inf = f_t*del;
      l_t = (int)(sup/del);         /* Round Up sup in del interval */
      if (sup > l_t*del + EPS) l_t++;
      sup = l_t*del;
      sca = len/(sup - inf);        /* Set the Scale Factor */
      p->axis_scale = sca;
      p->axis_evinf = inf;          /* ... and minimaxi of values */
      p->axis_evsup = sup;

//    fprintf( fmsg, " del=%f, ticks %d to %d, inf=%f, sup=%f, sca=%f.\n",
//                   del, f_t, l_t, inf, sup, sca );
//    fflush( fmsg );

      /* Create the ticks table */
      if (ptb) free( ptb );
      ptb = (float *) malloc( sizeof( float )*(l_t - f_t + 1)*2 );
      p->axis_ticktb = ptb;

      ij  =   0;
      fct = (inf - vsh)/vsc;        /* Get the displayed minimum ticks value */
      ptb[ij++] = fct;              /* Put the Initial Tick Value */
      ptb[ij++] = 0.0;              /* Put the Initial Tick Position */
      c_t = f_t + 1;                /* Initialize the current ticks count */
      while (c_t < l_t) {           /* Loop on all Ticks */
        fml = del*((c_t++) - f_t);  /* Get the coord. in Physical Units */
        ptb[ij++] = fct + fml/vsc;  /* Put the Tick Value */
        ptb[ij++] = sca*fml;        /* Put the Tick Position */
      }
      ptb[ij++] = (sup - vsh)/vsc;  /* Put the Last Tick Value */
      ptb[ij++] = len;              /* Put the Last Tick Position */

      /* Set the Axis Efficience */
      p->axis_eff = (p->axis_vsup - p->axis_vinf)/(sup - inf);
    } /* else of if (flg&AX_LOG)  */

    if ((p->axis_eunit == NULL)&&(p->axis_unit != NULL)) {
      p->axis_eunit = (char*) malloc( strlen( p->axis_unit ) + 1 );
      strcpy( p->axis_eunit, p->axis_unit );
    }

//  fprintf( fmsg, " Ticks Table created.\n" );
//  fflush( fmsg );

    p->axis_frstck  = f_t;          /* Set the First and Last ... */
    p->axis_lsttck  = l_t;          /* ... Ticks Numbers */
  }
}




static void Draw__TckMrk(
                  Draw_Axis_Ptr      p,  /* Specify the axis descriptor */
                         int       tck,  /* The current Ticks Number */
                         int       flg,  /* Left/Right/Symmetric ticks flags */
                         float    tckp   /* Position of ticks along the axis */
                       )
{
  int i, knd;
  float tckln;

  if (p) {
    /* We look for the appropriate ticks class number */
    i = 0;
    while ((i < 4)&&p->axis_tfre[i])
      if (Mod( tck, p->axis_tfre[i] ) == p->axis_tshf[i]) break;
                                                     else i++;

    if (i < 4) { /* When a Ticks Class Number is found */
      tckln = - fabs( p->axis_tcklen[i] );
      knd   = p->axis_tknd[i];
      if (flg&BX_TRVM) tckln = - tckln;
                  else if ((flg&BX_TSYM)&&(knd < 2)) knd = 2;

//    fprintf( fmsg, " Ticks # %d: Ax=%f, Class=%d, len=%f, kind=%d.\n",
//                   tck, tckp, i, tckln, knd );
//    fflush( fmsg );

      /* The current plot mode must be -1 (LINES) */
      switch (knd) {
        case  1:
          /* Single tick dash at right */
          Draw_GPlot3( tckp,    0.0,   0.0, 1 );
          Draw_GPlot3( tckp,  tckln,   0.0, 1 );
          break;

        case  2:
          /* Single tick dash at center */
          Draw_GPlot3( tckp,  tckln,   0.0, 1 );
          Draw_GPlot3( tckp, -tckln,   0.0, 1 );
          break;

        case  3:
          /* Aligned Cross tick */
          Draw_GPlot3( tckp,  tckln,    0.0, 1 );
          Draw_GPlot3( tckp, -tckln,    0.0, 1 );
          Draw_GPlot3( tckp,    0.0,  tckln, 1 );
          Draw_GPlot3( tckp,    0.0, -tckln, 1 );
          break;

        case  4:
          /* Diagonal Cross tick */
          tckln /= 1.4142;
          Draw_GPlot3( tckp,  tckln,  tckln, 1 );
          Draw_GPlot3( tckp, -tckln, -tckln, 1 );
          Draw_GPlot3( tckp,  tckln, -tckln, 1 );
          Draw_GPlot3( tckp, -tckln,  tckln, 1 );
          break;

        case  0:
        default:
          /* Single tick dash at left */
          Draw_GPlot3( tckp,    0.0,    0.0, 1 );
          Draw_GPlot3( tckp, -tckln,    0.0, 1 );
      }
    }
  }
} /* Draw__TckMrk */




void Draw__TckVal( Draw_Axis_Ptr      p,  /* Specify the axis descriptor */
                          int       tck,  /* The current Ticks Number */
                          int       flg,  /* Left/Right/Sym. flags */
                          float    tckp,  /* Position of ticks along the axis */
                          float    tckv,  /* Value to display with the ticks */
                          float    angl   /* Angle of strings */
                 )
{
  int     ip, f, d;
  float xstr, ystr;
  char  strval[32];

  if (p)
    if (Mod( tck, p->axis_tfre[0] ) == p->axis_tshf[0]) {
      /* The Ticks Values are Displayed only for the Main Ticks */ 
      xstr = p->axis_vaps[0] + tckp;
      ystr = p->axis_vaps[1];
      if (flg&BX_TRVM) ystr = - ystr;
      if (p->axis_flags&AX_LEFT) ystr = - ystr;
      if (p->axis_flags&AX_VOTH) ystr = - ystr;

      ip = 0;
      f = p->axis_field;
      d = p->axis_decm;

//    fprintf( fmsg, " Value:%d:%d = %f at %f,%f.\n", f, d, tckv, xstr, ystr );
//    fflush( fmsg );

      if (f > 0)
        if (d >= 0) Draw_Write_Fix( strval, &ip, (double) tckv, f, d, 0 );
               else Draw_Write_Float( strval, &ip, (double) tckv, f, 1, - d, 1 );
      else
        Draw_Write_Int( strval, &ip, (int) tckv, - f );

      strval[ip] = 0;

//    fprintf( fmsg, " The string(%d) \"%s\" will be plotted at %f,%f.\n",
//                   ip, strval, xstr, ystr );
//    fflush( fmsg );

      Draw_Gen_Text( xstr, ystr, angl, p->axis_vhigh, -1, strval );

    }
} /* Draw__TckVal */




void Draw_Plot_Axis( Draw_Axis_Ptr p, Draw_Point3 org, int fvl, Draw_Box_Ptr pbx )
{ /* Plot the axis p, with origine at org,
     fvl is a set of bits,
     When fvl is a bit set as this:
           4 = BX_VALU    - Ticks values and Unit String will be Displayed.
           8 = BX_TRVM    - The ticks left and right sides are permutted,
          16 = BX_TSYM    - The Ticks must be symmetrized.
     The value of p->axis_flag is a bit set as this :
           1 = AX_LOG     - Axis is in Log10 Coordinate,
           2 = AX_LEFT    - Ticks and displayed value at left position ( else at right ),
           4 = AX_VOTH    - Value are on other side than ticks,
           8 = AX_NFRS    - Does not display the first Ticks,
          16 = AX_NFVL    - Does not display the first Value,
          32 = AX_NLST    - Does not display the last Ticks,
          64 = AX_NLVL    - Does not display the last Value,
         128 = AX_NZER    - Does not display the Zero Ticks,
         256 = AX_NZVL    - Does not display the Zero Value,
         512 = AX_ARRO    - Put an arrow at the end of axis,
        1024 = AX_ARROF   - Fill arrow triangle,
        2048 = AX_TILT    - Values and Unit written with tilted characters,
       12288 = AXM_VPATH  - Mask for Path of value strings,
       49152 = AXM_UPATH  - Mask for path of unit string.
      131072 = AX_VLEFT   - Flag for left alignement of Ticks Values,
      262144 = AX_VRIGHT  - Flag for right alignement of Ticks Values,
      524288 = AX_ULEFT   - Flag for left alignement of Unit String,
     1048576 = AX_URIGHT  - Flag for right alignement of Unit String,
     2097152 = AX_UMIDLE  - Flag for Unit String Located to midlle of Axis.
  */

  int   c_t,  cp, f_t, flg,  ii,  ij, ih, l_t, srv, sout_mode;
  float ang, inf, len, sup,  xx,  yy;
  float   *  ptb;
  float mc[12], ms[12];             /* Plane reference definition */
  Draw_Point3  up, u1;              /* Temporary vectors for ticks */

  if (p) { /* When an axis is specified */

//  fprintf( fmsg, "Enable Axis # %d.\n", p->axis_ident );
//  fflush( fmsg );

    /* Enable Axis when not already done */
    if (p->axis_eff <= 0.1) Draw_Enable_Axis( p );

//  fprintf( fmsg, "Get Axis information.\n" );
//  fflush( fmsg );

    f_t = p->axis_frstck;           /* Get the first ticks number */
    l_t = p->axis_lsttck;           /* Get the last ticks number */
    inf = p->axis_vinf;             /* Get origine physical value */
    sup = p->axis_vsup;             /* Get final physical value */
    len = p->axis_length;           /* Get final physical value */
    flg = p->axis_flags;            /* Get the set of flags */
    ptb = p->axis_ticktb;           /* Get the ticks table address */

    srv = fvl&(BX_TRVM|BX_TSYM);    /* Set the left/right reverse flag */


//  fprintf( fmsg, "Ticks from %d to %d,\n", f_t, l_t );
//  fprintf( fmsg, "Axis form %f to %f with length = %f,\n", inf, sup, len );
//  fprintf( fmsg, "Axis flags words are a:%d, b:%d.\n", flg, fvl );
//  fprintf( fmsg, "Value Output mode = %d.%d\n",
//                 p->axis_field, p->axis_decm );
//  fflush( fmsg );

    sout_mode = draw_out_mode;      /* Save the current Out Mode */

//  fprintf( fmsg, "Set Axis plane reference.\n" );
//  fflush( fmsg );

    if (pbx)
      if (pbx->box_axis[3]) { /* 3D box only */
        Draw__RMul_Vect( u1, pbx->box_recmat, p->axis_upd );
//      fprintf( fmsg, "Get Up Tick Vect (%f,%f,%f).\n", u1[0], u1[1], u1[2] );
        if (fvl&BX_TSYMU) u1[0] = - u1[0];
        if (fvl&BX_TSYMV) u1[1] = - u1[1];
        if (fvl&BX_TSYMW) u1[2] = - u1[2];
//      fprintf( fmsg, "Set Up Tick Vect (%f,%f,%f).\n", u1[0], u1[1], u1[2] );
//      fflush( fmsg );
        Draw__RMul_Vect( up, pbx->box_dirmat, u1 );
      }
      else
        for (ii = 0; ii < 3; ii++) up[ii] = p->axis_upd[ii];

    Draw__Axis_Plane( org, p->axis_vct, up, mc, ms );
    Draw_Gen_Eplane( mc );             /* Push the Matrix to plot the axis in its local coordinates */

    Draw_GPlots();

    draw_out_mode = -1;                /* Set the Lines Mode */
    Draw_GPlot3( 0.0, 0.0, 0.0, 0 );   /* Go to the begin of axis and plot the axis */
    Draw_GPlot3( len, 0.0, 0.0, 1 );

//  fprintf( fmsg, "Plot Axis ticks.\n" );
//  fflush( fmsg );

    /* Generate the begin axis ticks when required */
    if (!(flg&AX_NFRS)) Draw__TckMrk( p, f_t, srv, ptb[1] );

//  fprintf( fmsg, "Plot the Axis segments # (%d..%d).\n", f_t, l_t );
//  fflush( fmsg );

    ij = 3;
    for (c_t = f_t + 1; c_t < l_t; c_t++) {
      if (c_t||!(flg&AX_NZER)) Draw__TckMrk( p, c_t, srv, ptb[ij] );
      ij += 2;
    }

//  fprintf( fmsg, "Plot extremity of axis.\n" );
//  fflush( fmsg );

    if (flg&(AX_NLST|AX_ARRO)) {
      Draw_GPlot3( 0.0, 0.0, 0.0, 0 );
      if (flg&AX_ARRO) {

        draw_out_mode = (flg&AX_ARRF)?3:-3;     /* Set Line Loop/Fill Mode */
        Draw_GPlot3( len,       0.2, 0.0, 0 );
        Draw_GPlot3( len + 0.6, 0.0, 0.0, 1 );
        Draw_GPlot3( len,      -0.2, 0.0, 1 );
        Draw_GPlot3( len,       0.0, 0.0, 0 );
      }
    } else {
      Draw__TckMrk( p, c_t, srv, ptb[ij] );
      Draw_GPlot3( 0.0, 0.0, 0.0, 0 );
    }

    if (fvl&BX_VALU) { /* For a master Axis */
      if (!(flg&AX_TILT)) {
        Draw_Gen_Eplane( NULL );
        Draw_Gen_Eplane( ms );
      }


//    fprintf( fmsg, "Plot the Axis Unit String.\n" );
//    fflush( fmsg );

      if (p->axis_eunit) { /* Put the Unit Name String */
        cp = ((flg/AX_UPATH)&3) + 1;
        ih = (flg&AX_URIGHT)?4:((flg&AX_ULEFT)?2:3);
        Draw_Gen_Tattr( ih, 4, cp, 1.0, 0.0, 0.0 );
        xx = p->axis_unps[0];
        if (flg&AX_UMIDLE) xx /=2.0;
                         else xx -= 0.1*p->axis_length;
        yy = - p->axis_unps[1];
        if (srv&BX_TRVM) yy = - yy;
        if (flg&AX_LEFT) yy = - yy;
        if (flg&AX_VOTH) yy = - yy;
        Draw_Gen_SelectFont( p->axis_ufont );

//     fprintf( fmsg, " Plot Unit of an axis at (%f,%f) attr = (%d) <%s>.\n", xx, yy, ih, p->axis_eunit );
//     fflush( fmsg );

        ang = p->axis_uangl;
        if (fvl&BX_ROTUV) ang += 180.0;
//      fprintf( fmsg, "fvl = %d, Value angle = %f.\n", fvl, ang );
//      fflush( fmsg );

        Draw_Gen_Text( xx, -yy, ang, p->axis_uhigh, -1, p->axis_eunit );
      }

      ang = p->axis_vangl;
      if (fvl&BX_ROTUV) ang += 180.0;

      cp = ((flg/AX_VPATH)&3) + 1;

      ih = (flg&AX_VRIGHT)?4:((flg&AX_VLEFT)?2:3);
      Draw_Gen_Tattr( ih, 4, cp, 1.0, 0.0, 0.0 );
      Draw_Gen_SelectFont( p->axis_vfont );

//    fprintf( fmsg, "Plot the Axis Unit 3.\n" );
//    fflush( fmsg );

      if (!(flg&AX_NLVL)) Draw__TckVal( p, l_t, srv, ptb[ij], ptb[ij-1], ang );
      ij -= 3;

//    fprintf( fmsg, "Plot the Axis Ticks Values.\n" );
//    fflush( fmsg );

      for (c_t = l_t - 1; c_t > f_t; c_t--) {
           if (c_t||!(flg&AX_NZVL)) Draw__TckVal( p, c_t, srv, ptb[ij+1], ptb[ij], ang );
        ij -=2;
      }

      if (!(flg&AX_NFVL)) Draw__TckVal( p, f_t, srv, ptb[ij+1], ptb[ij], ang );

      Draw_Gen_Tattr( 1, 1, 0, 1.0, 0.0, 0.0 );
    }

    Draw_Gen_Eplane( NULL );
    draw_out_mode = sout_mode;      /* Restore the current Out Mode */

//  fprintf( fmsg, " Return to Normal User Space.\n" );
//  fflush( fmsg );
  }
} /* Draw_Plot_Axis */




void Draw_Plot_Box( Draw_Box_Ptr p )
{ /* Plot the Referenced Box */
  Draw_Point3    org;
  float*           m;
  float   xo, yo, zo;
  Draw_Axis_Ptr  ax, axs[3];
  int   i, iax, idr, drf, tsz, nf, nfs;

  if (p) {
    if (p&&(!p->box_status)) {        /* Locking for multi-plot of a box */
      tsz = p->box_drtbsz;            /* Get the Plot Directive Table Size */
      nfs = p->box_drshtbsz;          /* And the related Shift table (if def.) */

//    fprintf( fmsg, " Plot Box.\n" );
//    fflush( fmsg );

      /* Get the Axis References */
      for (i = 0; i < 3; i++) {
        axs[i] = p->box_axis[i];
        if (axs[i])  Draw_Enable_Axis( axs[i] );
      }

      /* Build the Box related transformation Matrixs */
      Draw__Set_DirMat( p );
      Draw__InvMat( p->box_dirmat, p->box_recmat );
//    fprintf( fmsg, " Box direct matrix.\n" );
//    fflush( fmsg );
//    Sho_VGL_Matrix( p->box_dirmat );
//    fprintf( fmsg, " Box Reciprocal matrix.\n" );
//    fflush( fmsg );
//    Sho_VGL_Matrix( p->box_recmat );
      m = p->box_dirmat;

      Draw_Gen_PP_Attr( 1 );          /* Save attribute */

      /* Loop on all box to plot all axis directives */
      nf = 0;
      for (idr = 0; idr < tsz; idr++) {
        drf = p->box_drtb[idr];
//      fprintf( fmsg, " Box Directive elem # %d is %d.\n", idr, drf );
//      fflush( fmsg );
        iax = drf&3;                  /* Extract the axis specification bits */
        ax = (iax)?axs[iax - 1]:NULL; /* Select the related Axis */
        if (ax) {                     /* Execute a plot for this Axis */
          /* Set the origine of axis to plot */
          if (axs[0]) { /* Possible X axis shift (in X axis value) */
            if (drf&BX_XSHPOS) {
              xo = p->box_drshf[nf++]; /* User Specified X Shift */
              if (axs[0]->axis_flags&AX_LOG) xo = (xo>0.0)?log10( xo ):0.0;
            } else /* Possible X axis length shift */
              if (drf&BX_XSHEND) xo = axs[0]->axis_evsup;
                            else xo = axs[0]->axis_evinf;
          }
          else xo = 0.0;

          if (axs[1]) { /* Possible Y axis shift (in Y axis value) */
            if (drf&BX_YSHPOS) {
              yo = p->box_drshf[nf++]; /* User Specified Y Shift */
              if (axs[1]->axis_flags&AX_LOG) yo = (yo>0.0)?log10( yo ):0.0;
            } else /* Possible Y axis length shift */
              if (drf&BX_YSHEND) yo = axs[1]->axis_evsup;
                            else yo = axs[1]->axis_evinf;
          }
          else yo = 0.0;

          if (axs[2]) { /* Possible Z axis shift (in Z axis value) */
            if (drf&BX_ZSHPOS) {
              zo = p->box_drshf[nf++]; /* User Specified Z Shift */
              if (axs[2]->axis_flags&AX_LOG) zo = (zo>0.0)?log10( zo ):0.0;
            } else /* Possible Z axis length shift */
              if (drf&BX_ZSHEND) zo = axs[2]->axis_evsup;
                            else zo = axs[2]->axis_evinf;
          }
          else zo = 0.0;

          org[0] = m[0]*xo + m[3]*yo + m[6]*zo + m[ 9];
          org[1] = m[1]*xo + m[4]*yo + m[7]*zo + m[10];
          org[2] = m[2]*xo + m[5]*yo + m[8]*zo + m[11];
          if (!draw_mat_cnt) { org[0] -= orgx; org[1] -= orgy; org[2] -= orgz; }

          /* Plot the Axis */
          Draw_Plot_Axis( ax, org, drf, p );

//        fprintf( fmsg, " Axis #%d Plotted at (%f,%f,%f) -> (%f,%f,%f).\n", idr,
//                       xo, yo, zo, org[0], org[1], org[2] );
//        fflush( fmsg );
        }
      }

      p->box_status = 1;              /* Set the Already Plotted Status */
      Draw_Gen_PP_Attr( 0 );          /* Restore Attribute */

//    fprintf( fmsg, " Return From Plot Box.\n" );
//    fflush( fmsg );
    }
  }
}




void Draw_Open_Box( Draw_Box_Ptr p )
{ /* Select the referenced Box to put a curve */
  Draw_Axis_Ptr q;

  Draw_Opened_Box = p;
  if (Draw_Opened_Box) {
    open_dirmat = p->box_dirmat;
    open_recmat = p->box_recmat;
    if (q = p->box_axis[0]) open_xlogf = (q->axis_flags&AX_LOG)?1:0;
                       else open_xlogf = 0;
    if (q = p->box_axis[1]) open_ylogf = (q->axis_flags&AX_LOG)?1:0;
                       else open_ylogf = 0;
    if (q = p->box_axis[2]) open_zlogf = (q->axis_flags&AX_LOG)?1:0;
                       else open_zlogf = 0;
  }
}




void Draw_Close_Box()
{
  Draw_Opened_Box = NULL;
  open_dirmat     = NULL;
  open_recmat     = NULL;
}



void Draw_Update_Box( Draw_Box_Ptr p )
{
}



void Draw_Gen_SMplot( int dim, int flg, float * pt )
{
  Draw_Ptr        p;
  float         * q;
  float  xx, yy, zz;
  int   i, m, n, nm;

  if (Draw_Opened_Box) {

//  fprintf( fmsg, " SMPLOT dim=%d, flg=%d.\n", dim, flg );
//  fflush( fmsg );

    /* Elliminate Normal flag when not appropriate */
    if (((draw_out_mode > -4)&&(draw_out_mode < 3))||!(flg&cdf_3D)) flg = flg&~(cdf_NRM|cdf_UNRM);

    draw_3D_flg = 1;                   /* ... (Always in 3D) */
    n = dim*3; nm = n;
    if ((!pt)&&(draw_out_mode > 3)) flg |= cdf_NRM;
    if (flg&(cdf_NRM|cdf_UNRM)) {
      nm *= 2;
      if (flg&cdf_UNRM) { flg |= cdf_NRM; n = nm; }
    }

    /* Generate a General Plot Request */
    p = Draw_New_Node( Draw_Plot );
    p->tab.plo_knd = draw_out_mode;
    p->tab.plo_flg =    flg|cdf_3D;
    p->tab.plo_npt =           dim;
    p->tab.plo_box =          NULL;

//  fprintf( fmsg, " SMPLOT Draw_Plot Node allocated.\n" );
//  fflush( fmsg );

    /* Allocate the required 3D Plot Table */
    q = (float *) malloc( nm*sizeof( float ) );

//  fprintf( fmsg, " SMPLOT Draw_Plot Table allocated (%d float).\n", nm );
//  fflush( fmsg );


    p->tab.plo_tab =   q;
    m  = dim*((flg&cdf_3D)?3:2);    /* Compute the data table size */
    if (flg&cdf_UNRM) m *= 2;       /* When the normal are given by the user */

    if (pt) i = 0;                  /* For Data array we use direct access */
    else { /* Get the Data point (2D or 3D) */
      /* The 2D/3D vertex coordinates */
      i  = n - m;                   /* Get the Origine of input Table */
      pt = q+i;                     /* Put 2D Data in high array index part */
	  Sdrw_Read_Block( (Char *) pt, m*sizeof( float ) );
    }

//  fprintf( fmsg, " SMPLOT Read Done n=%d, m=%d.\n", n, m );
//  fflush( fmsg );

    i = 0;
    while (i < n) { /* Loop on all points */
      if (flg&cdf_3D) { /* 3D */
        xx = *(pt++);               /* Origine was add before */
        yy = *(pt++);
        zz = *(pt++);
        Draw_Scale3( xx, yy, zz, q, i );
        if (flg&cdf_UNRM) {
          xx = *(pt++);             /* Get the normal vector */
          yy = *(pt++);
          zz = *(pt++);
          i += 3;
          Draw_ScaleNormal( xx, yy, zz, q, i );
        }
      } else { /* 2D */
        xx = *(pt++);               /* Origine was add before */
        yy = *(pt++);
        Draw_Scale( xx, yy, q, i ) ;

//      fprintf( fmsg, " 2D SMPLOT %d/ (%f,%f) => (%f,%f,%f).\n", i,
//                     xx, yy, q[i], q[i+1], q[i+2] );
//      fflush( fmsg );
      }
      i += 3;
    }
//  Generate the normal when required.
    if (flg&(cdf_NRM|cdf_UNRM) == cdf_NRM)
      Draw_Gen_Normal( p->tab.plo_tab, draw_out_mode,  dim );

    Draw_Link_Node( p );
  }
}



void Draw_Gen_SMprogplot( float x, float y )
{
}



void Draw_Gen_MSurface( int nraw, int ncol, int flg )
{
  int   ij, ndim, size, sz, sr;
  float              *p,  *buf;
  float             xx, yy, zz;

//fprintf( fmsg, "  Surface st # %d  %d  %d.\n", nraw, ncol, flg );
//fflush( fmsg );
  ndim = abs( nraw*ncol );
  sr   = abs( ncol )*3;
  sz   = sr*sizeof( float );
  size = abs( nraw )*sz;
  buf  = (float*) malloc( size );
  if ((ncol < 0)||(nraw < 0)) {
    ncol = abs( ncol ); nraw = abs( nraw );
    p = buf;
    for (ij = 0; ij < nraw; ij++ ) {
//    fprintf( fmsg, "  Read a Block st # %d/%d od %d bytes.\n", ij+1, nraw, sz );
//    fflush( fmsg );
      Sdrw_Read_Block( (Char*) p, sz );
//    fprintf( fmsg, "  Read a Block nd of %d bytes.\n", sz );
//    for (ik = 0; ik < 5; ik++) fprintf( fmsg, " RD XYZ[%d] = (%f, %f, %f)\n", ik, p[(ik*3)+0], p[(ik*3)+1], p[(ik*3)+2] );
//    fflush( fmsg );
      p += sr;
    }
  } else Sdrw_Read_Block( (Char*) buf, size );

  /* Convert the box coordinates to cm of display */
  if (Draw_Opened_Box&&(ncol > 1)&&(nraw > 1)) {
//  fprintf( fmsg, "  Surface OK to Scale.\n" );
//  fflush( fmsg );
    p = buf;
    for (ij = 0; ij < ndim; ij++) {
      xx = p[0]; yy = p[1]; zz = p[2];
//    if (ij%ncol < 5) {
//      fprintf( fmsg, "To Sc XYZ[%d,%d] = (%f,%f,%f).\n", ij/ncol, ij%ncol, xx, yy, zz );
//      fflush( fmsg );
//    }
      Draw_Scale3( xx, yy, zz, p, 0 );
//    if (ij%ncol < 5) {
//      fprintf( fmsg, " In scale XYZ[%d,%d] = (%f,%f,%f).\n", ij/ncol, ij%ncol, p[0], p[1], p[2] );
//      fflush( fmsg );
//    }
      p += 3;
    }
    Draw__G_Surface( nraw, ncol, flg, buf );
  }
  else { free( buf ); return; }
}



void Draw_Gen_MSurface2( int nraw, int ncol, int flg, float xo, float yo, float xs, float ys )
{
  int   ij, ik, ndim, size, sz, sr;
  float             *p, *p0,  *buf;
  float                 xx, yy, zz;

//fprintf( fmsg, "  Surface st # %d  %d  %d.\n", nraw, ncol, flg );
//fflush( fmsg );
  ndim = abs( nraw*ncol );
  sr   = abs( ncol );
  sz   = sr*sizeof( float );
  size = abs( nraw )*sz;
  buf  = (float*) malloc( size*3 );
  p = buf + 2*ndim;                 /* Fill the top of buffer with all z coordinates */
  if ((ncol < 0)||(nraw < 0)) {
    ncol = abs( ncol ); nraw = abs( nraw );
    for (ij = 0; ij < nraw; ij++ ) {
//    fprintf( fmsg, "  Read a Block st # %d/%d od %d bytes.\n", ij+1, nraw, sz );
//    fflush( fmsg );
      Sdrw_Read_Block( (Char*) p, sz );
//    fprintf( fmsg, "  Read a Block nd of %d bytes.\n", sz );
//    for (ik = 0; ik < 5; ik++) fprintf( fmsg, " RD Z[%d] = %f\n", ik, p[ik] );
//    fflush( fmsg );
      p += sz;
    }
  } else Sdrw_Read_Block( (Char*) p, size );

  /* Convert the box coordinates to cm of display */
  /* Convert the box coordinates to cm of display */
  if (Draw_Opened_Box&&(ncol > 1)&&(nraw > 1)) {
//  fprintf( fmsg, "  Surface OK to Scale.\n" );
//  fflush( fmsg );
    p = buf; p0 = buf + 2*ndim;
    for (ij = 0; ij< nraw; ij++ ) {
      yy = yo + ij*ys;
      for (ik = 0; ik < ncol; ik++) {
        xx = xo + ik*xs;
        zz = *(p0++);
//      if (ij%ncol < 5) {
//        fprintf( fmsg, "To Sc XYZ[%d,%d] = (%f,%f,%f).\n", ij/ncol, ij%ncol, xx, yy, zz );
//        fflush( fmsg );
//      }
        Draw_Scale3( xx, yy, zz, p, 0 );
//      if (ij%ncol < 5) {
//        fprintf( fmsg, " In scale XYZ[%d,%d] = (%f,%f,%f).\n", ij/ncol, ij%ncol, p[0], p[1], p[2] );
//        fflush( fmsg );
//      }
        p += 3;
      }
    }

    Draw__G_Surface( nraw, ncol, flg, buf );
  }
  else { free( buf ); return; }
}
