/************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               D R A W   -   C L I E N T   L I B R A R Y               *
*                                                                       *
*                 (Pannel Control Graphic Directives)                   *
*                       (Multi-Language Support)                        *
*                                                                       *
*                     Draw Library Version V 2.4                        *
*                                                                       *
*                            15-Mar-2009                                *
*                                                                       *
*                                by                                     *
*                                                                       *
*                   Pierre Wolfers,  Institut Neel                      *
*                                                                       *
*           CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166             *
*                                                                       *
*                      F 38042 GRENOBLE CEDEX 9                         *
*                                                                       *
*                            F R A N C E                                *
*                                                                       *
*                                                                       *
*                                                                       *
************************************************************************/


/*///////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
///////////////////////////////////////////////////////////////////////*/




#include <draw/draw_common.h>
#include <draw/draw_env.h>      /* Load the DRAW library Environment */



typedef struct pnl_item   *itm_ptr;

typedef struct pnl_item   {
                            itm_ptr itm_nxt;    /* Next item pointer */
                            int     itm_knd;    /* Code the kind of item */
                            void*   itm_val;    /* Link to item value */
                            Char*   itm_len;    /* Length of string */
                            int     itm_cap;    /* String capacity */
                          } itm_rec;

typedef struct pnl_record *pnl_ptr;

typedef struct pnl_record {
                            pnl_ptr pnl_nxt;    /* Link to next pannel record */
                            int     pnl_ide;    /* Pannel Identifier */
                            itm_ptr pnl_frs,    /* Pannel item list head */
                                    pnl_lst;
                          } pnl_rec;

static pnl_ptr fpannel = 0;
static pnl_ptr lpannel = 0;

extern int     Pannel_Init;                     /* Flag to force pannel init execution (Set by Draw_Init) */


void Draw_Pannel_Align( int ide, int aln )
{
  Sdrw_Put_Code( cd_pannel_attr );              /* Put the Attribut Pannel Code, ... */
  Sdrw_Put_Int( 0 );                            /*... the attribute code, ... */
  Sdrw_Put_Int( ide );                          /* ... the Pannel identifier */
  Sdrw_Put_Int( aln );                          /* Put the alignement flag word */
  Sdrw_Server_Request();                        /* Send the request */
}



void Draw_Pannel_Fsize( int ide, int lbf, float sz )
{
  Sdrw_Put_Code( cd_pannel_attr );              /* Put the  Attribut Pannel Code, ... */
  Sdrw_Put_Int( lbf?1:2 );                      /*... the Change Font size code, ... */
  Sdrw_Put_Int( ide );                          /* ... the Pannel identifier */
  Sdrw_Put_Float( sz );                         /* Put the requested float size */
  Sdrw_Server_Request();                        /* Send the request */
}



void Draw_Pannel_Font( int ide, int lbf, char* nam, int len, int opt )
{
  Sdrw_Put_Code( cd_pannel_attr );              /* Put the  Attribut Pannel Code, ... */
  Sdrw_Put_Int( lbf?3:4 );                      /*... the Change Font code, ... */
  Sdrw_Put_Int( ide );                          /* ... the Pannel identifier */
  Sdrw_Put_String( nam, len );                  /* Put the Font name string */
  Sdrw_Put_Int( opt );                          /* Put the Font modifier () */
  Sdrw_Server_Request();                        /* Send the request */
}



static pnl_ptr Pannel_Locate( int ide )
{
  pnl_ptr p;

  p = fpannel;                                  /* Scan the list to find the specified Pannel*/
  while (p&&(p->pnl_ide != ide)) p = p->pnl_nxt;
  return p;
}



void Draw_Pannel_Delete( int ide )
{
  pnl_ptr p0, p1;
  itm_ptr q1, q2;

  p0 = 0;
  p1 = fpannel;                                 /* Locate the specified Pannel ... */
  while (p1&&(p1->pnl_ide != ide)) {            /* ... in pannel queue where p0 ... */
    p0 = p1; p1 = p1->pnl_nxt;                  /* ... is the previous pannel pointer or NULL. */
  }
  if (p1) {                                     /* When this pannel is localized */
    if (p0) p0->pnl_nxt = p1->pnl_nxt;          /* Remove the pannel from the queue */
       else fpannel = p1->pnl_nxt;
    if (p1 == lpannel) lpannel = p0;

    q1 = p1->pnl_frs;                           /* Get the pointer of first pannel item */
    while (q1) {                                /* Loop on all pannel item descriptor */
      q2 = q1;                                  /* Keep the item descriptor address to free */
      q1 = q1->itm_nxt;                         /* Pass the loop pointer to next item (or NULL) */
      free( q2 );                               /* Free the item descriptor */
    }
    free( p1 );                                 /* Free the pannel descriptor record */
    Sdrw_Put_Code( cd_pannel_del );             /* Emit the Pannel destroy code, ... */
    Sdrw_Put_Int( ide );                        /* ... the pannel identifier ... */
    Sdrw_Server_Request();                      /* ... and send request */
  }
}



void Local_Pannel_Init()
{ /* To destroye any old pannel after a Draw server Restart */
  pnl_ptr    p1, p2;
  itm_ptr    q1, q2;

  p1 = fpannel;                                 /* Loop one all existing pannel descriptors */
  while (p1) {
    p2 = p1; p1 = p1->pnl_nxt;
    q1 = p2->pnl_frs;
    while (q1) {                                /* Loop on all entries of a each pannel */
      q2 = q1; q1 = q1->itm_nxt;
      free( q2 );
    }
    free( p2 );
  }
  fpannel = 0;                                  /* Clear the pannel queue header to the empty state */
  lpannel = 0;
  Pannel_Init = 0;                              /* Clear the Init flag. */
}



int  Draw_Pannel_Create( float w, float h, char* title, int len )
{
  int ide;
  pnl_ptr p;

  if (Pannel_Init) Local_Pannel_Init();         /* Perform a Pannel Local Queue free and re-init on each DRAW_INit Call */
  Sdrw_Put_Code( cd_pannel_create );            /* Emit the Create Pannel Code, ... */
  Sdrw_Put_Float( w );                          /* ... the pannel sizes, ... */
  Sdrw_Put_Float( h );
  Sdrw_Put_String( title, len );                /* ... and the pannel title */
  Sdrw_Server_Request();                        /* Send the request to the graphic server */
  ide = Sdrw_Get_Int();                         /* Get the pannel identifier */
  if (ide > 0) {
    p = (pnl_ptr) malloc( sizeof( pnl_rec ) );  /* Allocate a pannel descriptor */
    p->pnl_nxt =   0;                           /* Init the next link */
    p->pnl_ide = ide;                           /* Set the pannel identifier */
    p->pnl_frs =   0;                           /* Init the pannel item list */
    p->pnl_lst =   0;
  }
  if (lpannel) lpannel->pnl_nxt = p;            /* Put the new pannel in the pannel list */
          else fpannel = p;
  lpannel = p;
  return ide;                                   /* Return the pannel identifier */
}


static itm_ptr Send_Item_Creation( int ide, float x, float y, float w, float h, char* lab, int len )
{
  pnl_ptr p;
  itm_ptr q;

  if (p = Pannel_Locate( ide )) {               /* Scan the list to find the specified Pannel*/
    q =(itm_ptr) malloc( sizeof( itm_rec ) );   /* Allocate the item record */
    q->itm_nxt =   0;                           /* Init the item link */
    if (p->pnl_lst) p->pnl_lst->itm_nxt = q;    /* link the new item in the item list of the pannel */
               else p->pnl_frs = q;
    p->pnl_lst = q;
    Sdrw_Put_Code( cd_pannel_add );             /* Emit the Pannel item addition code, ... */
    Sdrw_Put_Int( ide );                        /* ... the pannel identifier, ... */
    Sdrw_Put_Float( x );                        /* ... the item position and size, ... */
    Sdrw_Put_Float( y );
    Sdrw_Put_Float( w );
    Sdrw_Put_Float( h );
    Sdrw_Put_String( lab, len );                /* ... and the item label */
    return q;
  } else return 0;
}



int  Draw_Bool_To_Pannel( int ide, float x, float y, float w, float h, char* lab, int len,
                          char* bv )
{
  itm_ptr q;

  if (q = Send_Item_Creation( ide, x, y, w, h, lab, len )) {    /* Send the common item data */
    q->itm_knd = ITM_B;                         /* Set the kind Boolean Item */
    q->itm_val = (void*) bv;                    /* Keep the boolean pointer */
    Sdrw_Put_Char( ITM_B );                     /* Emit the Boolean kind */
    Sdrw_Put_Char( *bv );                       /* Send the initial boolean value */
    Sdrw_Server_Request();
    return Sdrw_Get_Int();                      /* Return the pannel found status */
  } else return -10;
}



int  Draw_Enum_To_Pannel( int ide, float x, float y, float w, float h, char* lab, int len,
                          int *ev, int sz, char* st[], int lt[], int min )
{
  int i, ll;
  itm_ptr q;

  ll = 0;
  for (i=0; i<sz; i++) if (ll<lt[i]) ll = lt[i];
  if (q = Send_Item_Creation( ide, x, y, w, h, lab, len )) {    /* Send the common item data */
    q->itm_knd = ITM_E;                         /* Set the kind Enum Item */
    q->itm_val = (void*) ev;                    /* Keep the Enum pointer */
    Sdrw_Put_Char( ITM_E );                     /* Emit the Enum kind */
    Sdrw_Put_Int( *ev );                        /* Emit the initial enum value, ... */
    Sdrw_Put_Int( min );                        /* Emit the minimum value */
    Sdrw_Put_Int( sz );                         /* ... the size of choice, ... */
    for (i=0; i<sz; i++) Sdrw_Put_String( st[i], lt[i] );       /* ... and each string. */
    Sdrw_Server_Request();
    return Sdrw_Get_Int();                      /* Return the pannel found status */
  } else return -10;
}



int  Draw_Str_To_Pannel( int ide, float x, float y, float w, float h, char* lab, int len,
                         Char* sv, Char* siz, int cap, Char fed )
{
  itm_ptr q;

  if (q = Send_Item_Creation( ide, x, y, w, h, lab, len )) {    /* Send the common item data */
    q->itm_knd = (fed)?ITM_S:ITM_STR;           /* Set the kind String Item */
    q->itm_val = (void*) sv;                    /* Keep the string pointer */
    if (cap>255) cap = 255;
    if (siz[0]>=cap) siz[0] = cap - 1;          /* Limit of string size to capacity */
    q->itm_len =        siz;                    /* Keep the string length address */
    q->itm_cap =        cap;                    /* keep the capacity value */
    Sdrw_Put_Char( q->itm_knd );                /* Emit the String kind */
    Sdrw_Put_String( sv, siz[0] );              /* Emit the initial string value ... */
    Sdrw_Put_Int( cap );                        /* ... and the string capacity */
    Sdrw_Server_Request();
    return Sdrw_Get_Int();                      /* Return the pannel found status */
  } else return -10;
}



int  Draw_Int_To_Pannel( int ide, float x, float y, float w, float h, char* lab, int len,
                         int *iv, int min, int max )
{
  itm_ptr q;

  if (q = Send_Item_Creation( ide, x, y, w, h, lab, len )) {    /* Send the common item data */
    q->itm_knd = ITM_I;                         /* Set the kind Integer Item */
    q->itm_val = (void*) iv;                    /* Keep the integer pointer */
    Sdrw_Put_Char( ITM_I );                     /* Emit the Integer kind */
    Sdrw_Put_Int( *iv );                        /* Emit the initial integer value, ... */
    Sdrw_Put_Int( min );                        /* ... the minimum value ... */
    Sdrw_Put_Int( max );                        /* ... and the maximum value */
    Sdrw_Server_Request();
    return Sdrw_Get_Int();                      /* Return the pannel found status */
  } else return -10;
}



int Draw_Flt_To_Pannel( int ide, float x, float y, float w, float h, char* lab, int len,
                        float *fv, float min, float max , float step )
{
  itm_ptr q;

  if (q = Send_Item_Creation( ide, x, y, w, h, lab, len )) {    /* Send the common item data */
    q->itm_knd = ITM_F;                         /* Set the kind Float Item */
    q->itm_val = (void*) fv;                    /* Keep the floating pointer */
    Sdrw_Put_Char( ITM_F );                     /* Emit the Float kind */
    Sdrw_Put_Float( *fv );                      /* Emit the initial floating value, ... */
    Sdrw_Put_Float( min );                      /* ... the minimum value, ... */
    Sdrw_Put_Float( max );                      /* ... the maximum value ... */
    Sdrw_Put_Float( step );                     /* ... and the step value */
    Sdrw_Server_Request();
    return Sdrw_Get_Int();                      /* Return the pannel found status */
  } else return -10;
}



int  Draw_Text_To_Pannel( int ide, float x, float y, float w, float h, char* lab, int len,
                          char* txt, int siz )
{
  itm_ptr q;

  if (q = Send_Item_Creation( ide, x, y, w, h, lab, len )) {    /* Send the common item data */
    q->itm_knd =  ITM_TXT;                      /* Set the kind Text Display Item */
    q->itm_val =        0;                      /* Keep the floating pointer */
    Sdrw_Put_Char( ITM_TXT );                   /* Emit the Text Display kind */
    Sdrw_Put_Int( siz );                        /* Send the text size */
    Sdrw_Write();                               /* Send the head message */
    Sdrw_Write_Block( (Char*) txt, siz );       /* Send the text */
    Sdrw_Read();                                /* Wait server for answerd */
    return Sdrw_Get_Int();                      /* Return the pannel found status */
  } else return -10;
}



int  Draw_Frame_To_Pannel( int ide, float x, float y, float w, float h, char* lab, int len, float flw, int stl )
{
  itm_ptr q;

  if (q = Send_Item_Creation( ide, x, y, w, h, lab, len )) {    /* Send the common item data */
    q->itm_knd =  ITM_FRM;                      /* Set the kind Label Display Item */
    q->itm_val =        0;                      /* Keep the floating pointer */
    Sdrw_Put_Char( ITM_FRM );                   /* Emit the Label kind */
    Sdrw_Put_Float( flw );                      /* Send the Frame Line width ..; */
    Sdrw_Put_Int( stl );                        /* ... and the Frame style */
    Sdrw_Server_Request();
    return Sdrw_Get_Int();                      /* Return the pannel found status */
  } else return -10;
}



int  Draw_Pannel_Change( int ide, int itm )
{
  pnl_ptr    p;
  itm_ptr    q;
  int   ij, ll;

  if (p = Pannel_Locate( ide )) {
    q = p->pnl_frs;                             /* Get the first item reference */
    ij = 0;
    while (q&&(itm != ++ij)) q = q->itm_nxt;    /* Locate the specified item */
    if (q&&(q->itm_knd != ITM_TXT)) {
      Sdrw_Put_Code( cd_pannel_change );        /* Emit the Pannel Enable/Disable code, ... */
      Sdrw_Put_Int( ide );                      /* ... the pannel identifier ... */
      Sdrw_Put_Int( itm );                      /* ... the item identifier in the pannel, ... */
      switch (q->itm_knd) {
        case ITM_B: Sdrw_Put_Char( ((Char*)(q->itm_val))[0] );  /* Set the Boolean value */
                    break;

        case ITM_I:
        case ITM_E: Sdrw_Put_Int( ((int*)(q->itm_val))[0] );    /* Set the integer or literal/enum value */
                    break;

        case ITM_STR:
        case ITM_S: if (q->itm_len) ll = q->itm_len[0];
                               else ll = q->itm_cap;
                    Sdrw_Put_String( ((Char*)(q->itm_val)),ll );/* Set the string value */
                    break;

        case ITM_F: Sdrw_Put_Float( ((float*)(q->itm_val))[0] );/* Set the floating value */
                    break;

     default: ;
      }
      if (q->itm_knd != ITM_TXT) Sdrw_Server_Request(); /* Send request except for text beacause it is already done. */
      ij = Sdrw_Get_Int();                      /* Get the resulting code */
      Draw_AnmSSt = Sdrw_Get_Char();            /* Get Run/animation Stop Flag */
      return ij;                                /* Return the resulting code */
    } else return -2;
  } else return -1;
}



int  Draw_Pannel_New_Text( int ide, int itm, char* txt, int siz )
{
  pnl_ptr    p;
  itm_ptr    q;
  int       ij;

  if (p = Pannel_Locate( ide )) {
    q = p->pnl_frs;                             /* Get the first item reference */
    ij = 0;
    while (q&&(itm != ++ij)) q = q->itm_nxt;    /* Locate the specified item */
    if (q&&(q->itm_knd == ITM_TXT)) {
      Sdrw_Put_Code( cd_pannel_change );        /* Emit the Pannel Enable/Disable code, ... */
      Sdrw_Put_Int( ide );                      /* ... the pannel identifier ... */
      Sdrw_Put_Int( itm );                      /* ... the item identifier in the pannel, ... */
      Sdrw_Put_Int( siz );                      /* Send the new text size */
      Sdrw_Write();                             /* Send the head message */
      Sdrw_Write_Block( (Char*) txt, siz );     /* Send the text */
      Sdrw_Read();                              /* Wait server for answerd */
      ij = Sdrw_Get_Int();                      /* Get the resulting code */
      Draw_AnmSSt = Sdrw_Get_Char();            /* Get Run/animation Stop Flag */
      return ij;                                /* Return the resulting code */
    } else return -2;
  } else return -1;
}



int  Draw_Pannel_EnaDis( int ide, int itm, Char flg )
{
  pnl_ptr    p;
  itm_ptr    q;
  int       ij;

  if (p = Pannel_Locate( ide )) {
    q = p->pnl_frs;                             /* Get the first item reference */
    ij = 0;
    while (q&&(itm != ++ij)) q = q->itm_nxt;    /* Locate the specified item */
    if (q) {
      Sdrw_Put_Code( cd_pannel_enadis );        /* Emit the Pannel Enable/Disable code, ... */
      Sdrw_Put_Int( ide );                      /* ... the pannel identifier ... */
      Sdrw_Put_Int( itm );                      /* ... the item identifier in the pannel, ... */
      Sdrw_Put_Char( flg );                     /* ... the FLag (true/false for Enable/Disable) ... */
      Sdrw_Server_Request();                    /* ... and send request */
      return Sdrw_Get_Int();                    /* Return the resulting code */
    } else return -2;
  } else return -1;
}



void Draw_Pannel_Open( int ide )
{
  pnl_ptr p;

  if (p = Pannel_Locate( ide )) {
    Sdrw_Put_Code( cd_pannel_open );            /* Emit the Pannel open code, ... */
    Sdrw_Put_Int( ide );                        /* ... the pannel identifier ... */
    Sdrw_Server_Request();                      /* ... and send request */
  }
}



void Draw_Pannel_Close( int ide )
{
  pnl_ptr p;

  if (p = Pannel_Locate( ide )) {
    Sdrw_Put_Code( cd_pannel_close );           /* Emit the Pannel close code, ... */
    Sdrw_Put_Int( ide );                        /* ... the pannel identifier ... */
    Sdrw_Server_Request();                      /* ... and send request */
  }
}



int  Draw_Pannel_Get( int ide )
{ /* To set the user program variables with the Draw Server pannel #ide values */
  pnl_ptr    p;
  itm_ptr    q;
  int  flg, ll;

  if (p = Pannel_Locate( ide )) {
    Sdrw_Put_Code( cd_pannel_get );             /* Emit the Pannel Get code, ... */
    Sdrw_Put_Int( ide );                        /* ... the pannel identifier ... */
    Sdrw_Server_Request();                      /* ... and send request */
    flg = Sdrw_Get_Int();                       /* Get the pannel change flag (or -1) for not existing pannel */
    if (flg < 0) return -11;                    /* Stop on erruer for not existing pannel in the Server *** bad error *** . */
    q = p->pnl_frs;
    while (q) {                                 /* Loop on all pannel item. */
      switch (q->itm_knd) {
        case ITM_B: ((Char*)q->itm_val)[0] = Sdrw_Get_Char();   /* Set the Boolean value */
                    break;

        case ITM_I:
        case ITM_E: ((int*)q->itm_val)[0] = Sdrw_Get_Int();     /* Set the integer or literal/enum value */
                    break;

        case ITM_S: ll = Sdrw_Get_String( (Char*)q->itm_val );  /* Set the string value */
                    if (q->itm_len) q->itm_len[0] = ll;         /* Set the Resulting string length */
                    break;

        case ITM_F: ((float*)q->itm_val)[0] = Sdrw_Get_Float(); /* Set the floating value */
                    break;

        default: ;
      }
      q = q->itm_nxt;
    }
  } else return -10;                            /* Not existing pannel - Normal Error */
}



int  Draw_Pannel_Put( int ide )
{ /* To set Draw Server pannel #ide values with the user program variables */
  pnl_ptr p;
  itm_ptr q;
  int    ll;

  if (p = Pannel_Locate( ide )) {
    Sdrw_Put_Code( cd_pannel_put );             /* Emit the Pannel Put code, ... */
    Sdrw_Put_Int( ide );                        /* ... the pannel identifier ... */
/*  Sdrw_Server_Request();                      /* ... and send request */
    q = p->pnl_frs;
    while (q) {                                 /* Loop on all pannel item. */
      switch (q->itm_knd) {
        case ITM_B: Sdrw_Put_Char( ((Char*)(q->itm_val))[0] );  /* Set the Boolean value */
                    break;

        case ITM_I:
        case ITM_E: Sdrw_Put_Int( ((int*)(q->itm_val))[0] );    /* Set the integer or literal/enum value */
                    break;

        case ITM_S: if (q->itm_len) ll = q->itm_len[0];
                               else ll = q->itm_cap;
                    Sdrw_Put_String( ((Char*)(q->itm_val)), ll );       /* Set the string value */
                    break;

        case ITM_F: Sdrw_Put_Float( ((float*)(q->itm_val))[0] );        /* Set the floating value */
                    break;

        default: ;
      }
      q = q->itm_nxt;
    }
    Sdrw_Server_Request();                      /* ... and send request */
    ll = Sdrw_Get_Int();                        /* Get the resulting code */
    Draw_AnmSSt = Sdrw_Get_Char();              /* Get Run/animation Stop Flag */
    return ll;                                  /* Return the resulting code */
  }
}



int  Draw_Pannel_Request( int ide )
{
  pnl_ptr p;

  if (p = Pannel_Locate( ide )) {
    Sdrw_Put_Code( cd_pannel_req );             /* Emit the Pannel Request code, ... */
    Sdrw_Put_Int( ide );                        /* ... the pannel identifier ... */
    Sdrw_Server_Request();                      /* ... and send request */
    return Sdrw_Get_Int();                      /* Return the final interaction status */
  } else return -10;                            /* Send the not found pannel code. */
}
