{
*************************************************************************
*                                                                       *
*                                                                       *
*                      *   D R A W   S Y S T E M  *                     *
*                                                                       *
*                                                                       *
*                 * * *   C A L I B R A T I O N   * * *                 *
*                                                                       *
*                                                                       *
*              ---  Version  1.4-C -- 15/03/2009 ---                    *
*                                                                       *
*                                                                       *
*             by :                                                      *
*                                                                       *
*               P. Wolfers                                              *
*                   C.N.R.S.                                            *
*                   Institut Néel                                       *
*                   B.P.  166 X   38042  Grenoble Cedex 9               *
*                                          FRANCE.                      *
*                                                                       *
*************************************************************************



/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This program is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This software is distributed in the hope that it will be useful,    //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////




   Program for DRAW Calibration.

   Read modify and Write the current DRAW8SERVER_SETTING file.

  N.B. :
     Can work only on Unix-Like system.

}
%pragma trace 1;
program DRAW_CALIBRATION;

%include 'DRAWENV:draw_defs';                   { Get the DRAW definitions for C-PASCAL }

const
  VERSION       =                   'V1.0';     { Version of this program }
  VDATE         =            '15-MAR-2009';     { falling date of this version }

  PATH_SEP      =                      ';';     { Define the path field separator character (CPAS Specific PATH Separator -- Not Unix like) }
  DIR_SEP       =                      '/';     { Define the Standard Unix-like Directory separator }
  LNM_SEP       =                      ':';     { Define the Logicam name separator (CPAS Specific PATH Separator -- Not Unix like) }

  MAX_STR       =                      255;     { Maximum size for line, string identifier ... }
  MAX_IDE       =                       30;     { Maximum size to compare identifier }

  LOG_NAME      =    'DRAW_SERVER_SETTING';     { Logical Name to use an other Draw Server Setting file }
  SEARCH_PATH   =  '.;/etc;/usr/local/etc';     { Path to search the default setting file }
{ SEARCH_PATH   =    '/etc;/usr/local/etc';     { Path to search the default setting file }
  DEF_SETFILE   =   'draw_server.draw_setting'; { Default name for the Draw server Setting file }

  CNTOPE  = 'Error : Cannot open input file "'; { Main part of error message for cannot open input file error. }

  DEF_XRESOL    =                     1024;     { Default to a standard and frequently found resolution }
  DEF_YRESOL    =                      768;

  N_SCR_WIDTH   =              'SCR_WIDTH';
  N_SRC_HEIGHT  =             'SCR_HEIGHT';
  N_SRC_RESOL   =         'SCR_RESOLUTION';
  N_WIN_SIZES   =              'WIN_SIZES';
  N_WIN_POS     =           'WIN_POSITION';

type
  ide_code = (                                  { * Define the Known Identifier code }
                 idc_width,                     { Width of screen id.(cm) }
                 idc_height,                    { Height of screen id.(cm) }
                 idc_resol,                     { Screen resolution for x and y id (raster unit) }
                 idc_winsize,                   { Window size id (in cm) }
                 idc_winpos,                    { Window position id --- if managed by the server --- }
                 idc_noop                       { Noop id }
             );

  setf_ptr   = ^setf_rec;                       { * Define the Setting File Record Pointer }

  setf_rec   = record                           { * Define the Setting File Record }
                 next:        setf_rec;         { Links for file line, and managed file line }
                 line:         ^string;         { The related line }
                 idcd:        ide_code          { Index of related managed identifer or idc_noop }
               end;

  symbol = (      identsy,    stringsy,         { * Define Symbol that can find in the Setting file }
                    intsy,       fltsy,
                  commasy,   semicolsy,
                    addsy,       subsy,
                    mulsy,       divsy,
                    powsy,
                  equalsy, separatorsy,
                  eolnsy,        eofsy,
                  undefsy
           );

  ide_string = STRING( MAX_IDE );               { * Define the Identifier string type }

  ide_rec    = record                           { * Define the Known Identifier to handle record }
                 ide_name: ide_string;          { Name of identifier }
                 ide_line:   setf_ptr;          { Related line record pointer, ... }
                 ide_lcnt:    integer;          { ... related number of line change. }
               end;

var
  msg,                                          { Current string to format message }
  file_path,                                    { Path of the Draw Server Setting file }
  out_path,                                     { Path of the output file }
  dsp_string,                                   { string to display in the DRAW/GL Window }
  curr_line:         string( MAX_STR );         { Current read line }

  curr_ide:                 ide_string;         { Current identifier }

  crpline,                                      { Current line record pointer }
  fline, lline:        setf_ptr := nil;         { First and last setting record pointers }

  rdflg, wrflg:         Dbool := false;         { Flag for read and write access to Draw setting File }

  fset:                           text;         { Server Setting Pascal file variable }

  sy:                           symbol;         { Last symbol kind read by INSYMBOL }
  sy_err:                      boolean;         { Local INSYMBOL error flag }
  sy_gblerr:                   boolean;         { Global INSYMBOL error flag }
  sy_code:                    ide_code;         { Code for last found identifier }
  sy_linenb,                                    { Line count }
  sy_errline,                                   { Line count for the last syntax error }
  sy_pos,                                       { Position of syntax unit in the line }
  sy_eolgln,                                    { Position of end of Logical line in the current line }
  sy_ip:                       integer;         { Index in input line }
  sy_line:                    setf_ptr;         { pointer to the syntax unit line }
  sy_sep,                                       { Find separator }
  sy_ch, sy_cm:                   char;         { Last character seen }
  sy_string:         string( MAX_STR );         { Identifier name or string seen by INSYMBOL }
  intval:                         Dint;         { Integer and/or floatting value }
  fltval:                       Dfloat;
  rmaxint:            double := maxint;         { Maximum of integer in floatting }


  idm_tbl: array[ide_code] of ide_rec := [      { * Define the table of managed identifier during the calibration operation }
                [      N_SCR_WIDTH, nil, 0 ],   { idc_width   }
                [     N_SRC_HEIGHT, nil, 0 ],   { idc_height  }
                [      N_SRC_RESOL, nil, 0 ],   { idc_resol   }
                [      N_WIN_SIZES, nil, 0 ],   { idc_winsize }
                [        N_WIN_POS, nil, 0 ],   { idc_winpos  }
                [              ' ', nil, 0 ] ]; { idc_noop    }


  Server_On:            Dbool := false;         { Flag : When true,the Draw_server is running }

  unit:                      unit_type;         { Unit set as raster or Cm }

  iflt,
  status,                                       { Resulting status of user Interaction }
  sv_s_xreso, sv_s_yreso,                       { Saved values of original Screen resolution }
  src_xresol, src_yresol:         Dint;         { Screen resolution (raster unit) }

  sv_s_width, sv_s_heigh,                       { Saved values of original screen sizes }
  sv_w_width, sv_w_heigh,                       { Saved values of original window sizes }
   sv_w_posx,  sv_w_posy,                       { Saved values of original window positions }
   src_width, src_height,                       { Sizes of screen (cm) }
   win_width, win_height,                       { Size of window (cm) }
    win_posx,   win_posy,                       { Position of window (cm) }

       dt_rx,      dt_ry,                       { Size of the GL Window in Pixels }
       ds_rx,      ds_ry,                       { Size of the GL Window in Centimeters }
   cxy_scale,  sxy_scale,                       { Get The actual DRAW Scale user/surface and CV on Screen }
    d_xscale,   d_yscale,                       { Get The actual DRAW X and y Scales in pixel/(Pseudo-Cm) }

      charsz,                                   { Character size to use }
      vx, vy,    xx,  yy,
        xorg,       yorg,                       { Origine coordinates at the window center }
     x_paper,    y_paper:       Dfloat;         { Available drawing area sizes }

  fltr: Choice_List(2,15 ):=[ '*.draw_setting', { Filter for Draw Setting file selection }
                              '*.*'
                            ];



{ *****************************************************************************
  ***                                                                       ***
  ***      Procedures and/or functions to read a setting file model.        ***
  ***                                                                       ***
  *****************************************************************************
}


procedure MSG_ERROR( in_var msg: string );
var
  nerr: [static] integer;

begin
  sy_err       := true;
  sy_gblerr    := true;
  if sy_errline <> sy_linenb then
  begin
    nerr := 0;
    WRITELN( err, ' Syntax Error(s) in the Draw Setting File "', file_path, '" at line :' );
    WRITELN( err, ' ', sy_linenb:5, ' ':4, curr_line );
    sy_errline := sy_linenb
  end;
  nerr := nerr + 1;
  WRITELN( err, ' ':9+sy_pos, nerr:0 );
  WRITELN( err, ' ', nerr:3, ' with the message: ', msg );
  WRITELN( err );
end MSG_ERROR;



procedure INSYMBOL;
const
  DEL = CHR( 127 );
  NUL = CHR(   0 );
  TAB = CHR(   9 );
  TEN =       10.0;
  ONE =        0.0;

type
  chcat = (   alpha,  digit, period,  comma,semicol,  colon,   addo,   subo,
               mulo,   divo,   powo,   quot,   lpar,   rpar,   equo,    oth,
            spacech,  baseo,   clto,   cgto,   ando,comment,   lbra,   rbra,
             eolnch,  eofch
          );

[static]
var
  attch: array[' '..DEL] of chcat :=
            { ' '       '!'       '"'       '#'       '$'       '%'       '&'      '''' }
           [ spacech,  comment,  quot,     baseo,    alpha,    baseo,    ando,     quot,
            { '('       ')'       '*'       '+'       ','       '-'       '.'       '/' }
             lpar,     rpar,     mulo,     addo,     comma,    subo,     period,   divo,
            { '0'       '1'       '2'       '3'       '4'       '5'       '6'       '7' }
             digit,    digit,    digit,    digit,    digit,    digit,    digit,    digit,
            { '8'       '9'       ':'       ';'       '<'       '='       '>'       '?' }
             digit,    digit,    colon,    semicol,  clto,     equo,     cgto,     oth,
            { '@'       'A'       'B'       'C'       'D'       'E'       'F'       'G' }
             oth,      alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
            { 'H'       'I'       'J'       'K'       'L'       'M'       'N'       'O' }
             alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
            { 'P'       'Q'       'R'       'S'       'T'       'U'       'V'       'W' }
             alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
            { 'X'       'Y'       'Z'       '['       '\'       ']'       '^'       '_' }
             alpha,    alpha,    alpha,    lbra,     oth,      rbra,     powo,     alpha,
            { '`'       'a'       'b'       'c'       'd'       'e'       'f'       'g' }
             alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
            { 'h'       'i'       'j'       'k'       'l'       'm'       'n'       'o' }
             alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
            { 'p'       'q'       'r'       's'       't'       'u'       'v'       'w' }
             alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
            { 'x'       'y'       'z'      '{'       '|'      '}{'       '~'       '?'  }
             alpha,    alpha,    alpha, comment,      oth,      oth,      oth,      oth
           ];

  categ:               chcat := eolnch;         { Current character category }
  beg_line,                                     { Flag for line begin }
  beg_lglflg, end_lglflg,                       { Begin and End of logical line flag }
  binstr, bint:       boolean := false;
  beg_lgp, end_lgp:   integer :=     0;         { Begin and End of logical line }

  ivl, i1, i, j, base:         integer;
  rdig, rexp, rfac, rval:       double;


  procedure DEB_OUT( r: char );
  begin
    WRITE( ' ', r:2, ' ch = "' );
    if (sy_ch < ' ') or (sy_ch >= CHR( 127 )) then WRITE( '\', ORD( sy_ch ):-3:8, '\' )
                                              else WRITE( sy_ch );
    WRITE( '" categ = "' );
    if (categ < categ"first) or (categ > categ"last) then WRITE( '\', ORD( categ ):0, '\' )
                                                     else WRITE( categ );
    WRITELN( '".' )
  end DEB_OUT;



  function  GET_SRCLINE: integer;
  begin
    if EOF( fset ) then GET_SRCLINE := -1
    else
    begin
      READLN( fset, curr_line );
      GET_SRCLINE := curr_line.length;
      beg_line := true
    end
  end GET_SRCLINE;



  procedure CREATE_MEMLINE;
  var
    p:        setf_ptr;
    len:       integer;
    bf:        boolean;

  begin
    if beg_lgp < 1 then beg_lgp := 1;
    if end_lgp = 0 then len := curr_line.length + 1 - beg_lgp
                   else len := end_lgp - beg_lgp;
(*
WRITELN( ' CREATE_MEMLINE from ', beg_lgp:4, ' to ', end_lgp:4, ' = (', len:0, ') "', SUBSTR( curr_line, beg_lgp, len ), '"' );
*)
    bf := true;
    if bf and (beg_lgp = 1) then
    begin { Skip any head file comment beginnig by "!**" }
      if (curr_line[1] <> '!') or
         (curr_line[2] <> '*') or
         (curr_line[3] <> '*') then bf := false
    end else bf := false;

    if not bf then
    begin
      NEW( p );                                 { Create the line record and link it in the line list }
      if fline = nil then fline := p
                     else lline^.next := p;
      lline := p;
      with p^ do
      begin
        next := nil;                            { Init the line record fields }
        line := nil;
        idcd := idc_noop;
        if len > 0 then                         { Suppress any trailing space }
          while (len > 0) and (curr_line[beg_lgp+len-1] <= ' ') do len := len - 1;
        if len > 0 then                         { When the line is not empty }
        begin
          NEW( line, len );                     { Create the logical line string }
          for ii := 1 to len do                 { Copy the string character ... }
            line^[ii] := curr_line[beg_lgp+ii-1];
          line^.length := len;                  { ... and set the string length. }
          beg_lgp := end_lgp                    { Set the begin position to the previuos end position }
        end
      end
    end;
    beg_lglflg := false;
    end_lglflg := false
  end CREATE_MEMLINE;



  procedure GETDCHAR;
  var
    len: [static] integer := 0;

  begin
    if sy_ip <= 0 then len := GET_SRCLINE;      { On any end of line we try to get a new line }
    if len > 0 then
    begin                                       { When a new line was readden }
      if sy_ip < curr_line.length then
      begin
        sy_ip := sy_ip + 1;                     { We get one character }
        sy_ch := curr_line[sy_ip];
        if (sy_ch < ' ') and (sy_ch <> TAB) then sy_ch := ' ';  { We change in space any control character except TAB }
        sy_cm := sy_ch;                         { We get the Capital letter copy of this character }
        if (sy_cm >= 'a') and (sy_cm <= 'z') then sy_cm := CHR( ORD( sy_cm ) - 32 )
        else if sy_cm < ' ' then sy_cm := ' '   { TAB is equivalent to a space and Not ASCII character as DEL (=127) }
                            else if sy_cm > CHR( 127 ) then sy_cm := CHR( 127 );
        categ := attch[sy_cm]                   { We get the character category }
      end else begin                            { When we reached the end of line ... }
        if beg_lglflg then                      { when a logical line was running, create it now }
        begin end_lgp := 0; CREATE_MEMLINE  end;
        sy_linenb := sy_linenb + 1;             { Increment the line number }
        sy_ch := ' '; sy_cm := ' ';             { ... we emit a space with the EOLN category and ... }
        sy_ip := 0; categ := eolnch             { ... prepare the call of a next line. }
      end
    end
    else
    begin                                       { When the new line was empty or we reached the End of File }
      sy_ch := ' '; sy_cm := ' '; sy_ip := 0;   { Set as a space and empty line }
      if len = 0 then
      begin                                     { Empty line <EOLN> }
        if beg_lglflg then                      { when a logical line was running, create it now }
        begin end_lgp := 0; CREATE_MEMLINE  end;
        sy_linenb := sy_linenb + 1;             { Increment the line number }
        categ := eolnch                         { Create an empty line Copy }
      end
      else categ := eofch                       { End Of File <EOF> }
    end
  end GETDCHAR;


  procedure SKIP_COMMENT( cs: char );
  begin
    if beg_line then
    begin                                       { For a beginning of line with a comment ... }
      beg_lglflg := true; beg_lgp := 1;         { Set as a record line to create }
      end_lglflg := true; beg_line := false
    end;
    GETDCHAR;
    while (sy_ch <> cs) and (categ <> eolnch) and (categ <> eofch) do GETDCHAR;
    if sy_ch = cs then GETDCHAR
  end SKIP_COMMENT;



  procedure NXTCH;
  begin
    GETDCHAR;
    if not binstr then
      if sy_ch = '{' then SKIP_COMMENT( '}' )
                     else if sy_ch = '!' then SKIP_COMMENT( NUL )
  end NXTCH;



begin { INSYMBOL }
  binstr := false;
  while categ = spacech do NXTCH;
  sy      := undefsy;
  sy_line := crpline;
  sy_pos  :=   sy_ip;

  if end_lglflg then
  begin { Update the logical end of line to include the final comment }
    end_lgp := sy_ip;
    CREATE_MEMLINE
  end;

  if not beg_lglflg then
  begin
    beg_lgp    := sy_ip;
    beg_lglflg :=  true
  end;

(*
DEB_OUT( 'I' );
*)
  case categ of
    digit, period:
      begin
        rval :=    0.0;
        rexp :=    TEN;
        rfac :=    ONE;
        sy   :=  intsy;
        while categ = digit do
        begin
          rdig := ORD( sy_ch ) - ORD( '0' );
          NXTCH;
          rval := rval*TEN + rdig
        end;
        if categ = period then
        begin
          NXTCH;
          sy := fltsy;
          while categ = digit do
          begin
            rdig := ORD( sy_ch ) - ORD( '0' );
            NXTCH;
            rfac := rfac/TEN;
            rval := rval + rfac*rdig
          end
        end;
        if (sy_cm = 'e') or (sy_cm = 'E') then
        begin
          sy := fltsy;
          NXTCH;
          if (sy_ch = '+') or (sy_ch = '-') then
          begin
            if sy_ch = '-' then rexp := ONE/rexp;
            NXTCH
          end;
          ivl := 0;
          while categ = digit do
          begin
            ivl := ivl*10 + (ORD( sy_ch ) - ORD( '0' ));
            NXTCH
          end;
          if ivl > 38 then
          begin
            ivl := 0;
            MSG_ERROR( 'Too large floatting number exposant' )
          end;
          rfac := ONE;
          while ivl <> 0 do
          begin
            if ODD( ivl ) then
            begin  ivl := ivl - 1; rfac := rfac*rexp  end
            else
            begin  ivl := ivl div 2; rexp := SQR( rexp )  end
          end;
          rval := rval*rfac
        end;
        if (sy = intsy) and (rval <= rmaxint) and
           (rval >= -(rmaxint+1.0)) then intval := ROUND( rval )
                                    else sy := fltsy;
        fltval := rval
      end;

    alpha: { Identifier }
      begin
        sy := identsy;
        sy_string.length := 0;
        ivl := 0;
        while (categ = alpha) or (categ = digit) or (categ = period) do
        begin
          if ivl < MAX_STR then ivl := ivl + 1
                           else MSG_ERROR( 'Too long identifier' );
          sy_string[ivl] := sy_cm;
          NXTCH
        end;
        sy_string.length := ivl;
        sy_code := ide_code"first;
        repeat
        exit if sy_string = idm_tbl[sy_code].ide_name;
          sy_code := SUCC( sy_code )
        until sy_code = idc_noop
      end;

    baseo: { Integer in specific base }
      begin
        sy := intsy;
        if sy_ch = '#' then
        begin
          NXTCH;
          base := 0;
          while categ = digit do
          begin
            base := base*10 + (ORD( sy_ch ) - ORD( '0' ));
            NXTCH
          end;
          if sy_ch <> '#' then MSG_ERROR( 'Illegal numeric base specification' )
        end
        else
        if sy_ch = '%' then
        begin
          base := 10;
          NXTCH;
          case sy_ch of
            'x', 'X', 'h', 'H': base := 16;
            'd', 'D':           base := 10;
            'o', 'O':           base :=  8;
            'b', 'B':           base :=  2;
          otherwise
            { base = 10 }
          end
        end;
        if not sy_err then
        begin
          NXTCH;
          ivl := 0;
          while sy_ch > ' ' do
          begin
            if (sy_cm >= '0') and (sy_cm <= '9') then j := ORD( sy_cm ) - ORD( '0' )
            else
              if (sy_cm >= 'A') and (sy_cm <= 'F') then j := ORD( sy_cm ) - ORD( 'A' ) + 10;
            if j < base then begin  ivl := ivl*base + j; NXTCH  end
                        else MSG_ERROR( 'Illegal digit for the specified base' )
          end;
          intval := ivl;
          fltval := ivl
        end
      end;

    addo:    { '+' } begin  sy := addsy; NXTCH  end;
    subo:    { '-' } begin  sy := subsy; NXTCH  end;
    mulo:    { '*' } begin  sy := mulsy; NXTCH  end;
    divo:    { '/' } begin  sy := divsy; NXTCH  end;
    powo:    { '^' } begin  sy := powsy; NXTCH  end;

    equo:    { '=' } begin  sy := equalsy; NXTCH  end;

    comma:   { ',' } begin  sy := commasy; NXTCH  end;

    semicol: { '=' } begin
                       sy := semicolsy;  NXTCH; { Set the semicolon symbol }
                       end_lglflg  :=     true; { Set the flag to end-of-statment comment search }
                       end_lgp     :=    sy_ip  { Set the current end-of-statement position }
                     end;

    quot:    { String }
      begin
        bint   := true;
        ivl    :=    0;
        binstr := true;
        sy := stringsy;
        sy_string.length := 0;
        while bint and (ivl < MAX_STR) do
        begin
          NXTCH;
          if sy_ch = NUL then bint := false;
          if sy_ch = '''' then
          begin
            NXTCH;
            if sy_ch = '''' then begin  ivl := ivl + 1; sy_string[ivl] := sy_ch  end
                            else bint := false
          end
          else begin  ivl := ivl + 1; sy_string[ivl] := sy_ch  end;
          if ivl = MAX_STR then
          begin
            ivl := ivl - 1;
            MSG_ERROR( 'Too long string' )
          end
        end;
        binstr := false;
        sy_string.length := ivl
      end;

    eolnch: begin  sy := eolnsy; NXTCH;  end;

    eofch:  sy := eofsy;

  otherwise
    sy := separatorsy; sy_sep := sy_ch;
    NXTCH
  end;

  if sy <> eolnsy then beg_line := false;       { Clear the line begin flag except at the begin of new line }

(*
;WRITE( ' sy = ', sy );
case sy of
  identsy,
  stringsy: WRITELN( ' (', sy_string.length:0, ') "', sy_string, '"' );
  intsy:    WRITELN( ' = ', intval:0 );
  fltsy:    WRITELN( ' = ', fltval );
  separatorsy: WRITELN( ' ''', sy_sep, ''' ', ORD( sy_sep ):4:8  );
otherwise
  WRITELN
end*)
end INSYMBOL;



procedure SKIP_SYMBOL( stp: symbol; bskp: boolean );
begin
  while (sy <> stp) and (sy <> eofsy) do INSYMBOL;
  if bskp then INSYMBOL
end SKIP_SYMBOL;



procedure CHECK_AND_SKIP( sp: symbol );
var
  msg: string( 128 );

begin
  while sy = eolnsy do INSYMBOL;
  if sy = sp then INSYMBOL
             else begin
                    WRITEV( msg, 'The symbol ', sp, ' was expected' );
                    MSG_ERROR( msg )
                  end
end CHECK_AND_SKIP;



procedure GET_FLOAT( var fv: Dfloat );
var
  neg: boolean := false;
  val: Dfloat;

begin
  while sy = eolnsy do INSYMBOL;
  if (sy = addsy) or (sy = subsy) then
  begin  if sy = subsy then neg := true; INSYMBOL  end;
  while sy = eolnsy do INSYMBOL;
  if (sy = intsy) or (sy = fltsy) then begin  val := fltval; INSYMBOL  end
                                  else begin  val := 0.0; MSG_ERROR( 'A floatting number was expected' )  end;
  if neg then val := -val;
  fv := val
end GET_FLOAT;



procedure GET_INTEGER( var iv: Dint );
var
  neg: boolean := false;
  val: Dint;

begin
  val := 0;
  while sy = eolnsy do INSYMBOL;
  if (sy = addsy) or (sy = subsy) then
  begin  if sy = subsy then neg := true; INSYMBOL  end;
  while sy = eolnsy do INSYMBOL;
  if sy = intsy then begin  val := intval; if neg then val := -val  end
  else
  if sy = fltsy then
  begin
    if neg then fltval := - fltval;
    if (fltval >= (-rmaxint - 1)) and (fltval <= rmaxint) then val := ROUND( fltval )
    else MSG_ERROR( 'Too large number to be an integer' )
  end
  else MSG_ERROR( 'An integer number was expected' );
  if not sy_err then INSYMBOL;
  iv := val
end GET_INTEGER;



procedure GET_STRING( var st: string );
begin
  while sy = eolnsy do INSYMBOL;
  if sy = stringsy then begin  st := sy_string; INSYMBOL  end
                   else MSG_ERROR( 'A string was expected' );
end GET_STRING;



procedure LOAD_SETTING;
var
  p:          setf_ptr;
  ident:  string( 30 );
  ip, ll, ns:     Dint;
  bf:            Dbool;
  svcd:       ide_code;

begin { LOAD_SETTING }
  RESET( fset, file_path );                     { Open the input file }
  bf :=     true;
  sy :=   eolnsy;
  sy_ch   := ' ';
  sy_errline:= 0;
  sy_linenb := 1;                               { Initialize the Line Count }
  sy_gblerr := false;                           { Initialize the global error flag }

  repeat
    sy_err := false;                            { Initialize the local (to each statement) error flag }
    while sy = eolnsy do INSYMBOL;
    if sy = identsy then
      if sy_code = idc_noop then SKIP_SYMBOL( semicolsy, true )
      else                                      { We must manage this identifier }
      with idm_tbl[sy_code] do
      begin
        svcd := sy_code;                        { Save the code identifier }
        ide_line :=      lline;                 { Set the Line pointer To previous line record or nil }
        ide_lcnt :=  sy_linenb;                 { Save the start line number }
        INSYMBOL;                               { Gobble up the known identifier }
        case sy_code of
          idc_width:    { Width of screen id.(cm) }
            begin
              CHECK_AND_SKIP( equalsy );
              if not sy_err then GET_FLOAT( src_width );
              sv_s_width := src_width;
            end;
          idc_height:   { Height of screen id.(cm) }
            begin
              CHECK_AND_SKIP( equalsy );
              if not sy_err then GET_FLOAT( src_height );
              sv_s_heigh := src_height
            end;
          idc_resol:    { Screen resolution for x and y id (raster unit) }
            begin
              CHECK_AND_SKIP( equalsy ); if sy_err then goto ET_SKIP;
              GET_INTEGER( src_xresol ); if sy_err then goto ET_SKIP;
              CHECK_AND_SKIP( commasy ); if sy_err then goto ET_SKIP;
              GET_INTEGER( src_yresol );
              sv_s_xreso := src_xresol; sv_s_yreso := src_yresol
            end;
          idc_winsize:  { Window size id (in cm) }
            begin
              CHECK_AND_SKIP( equalsy ); if sy_err then goto ET_SKIP;
              GET_FLOAT( win_width    ); if sy_err then goto ET_SKIP;
              CHECK_AND_SKIP( commasy ); if sy_err then goto ET_SKIP;
              GET_FLOAT( win_height   );
              sv_w_width := win_width; sv_w_heigh := win_height
            end;
          idc_winpos:   { Window position id --- if managed by the server --- }
            begin
              CHECK_AND_SKIP( equalsy ); if sy_err then goto ET_SKIP;
              GET_FLOAT( win_posx     ); if sy_err then goto ET_SKIP;
              CHECK_AND_SKIP( commasy ); if sy_err then goto ET_SKIP;
              GET_FLOAT( win_posy     );
              sv_w_posx := win_posx; sv_w_posy := win_posy
            end;
        otherwise
          SKIP_SYMBOL( semicolsy, false )
        end;
        CHECK_AND_SKIP( semicolsy );
        { Here, INSYMBOL must have created the line record of the statement. }
        if ide_line = nil then ide_line := fline{ Set ide_line as the pointer of the statment line record }
                          else ide_line := ide_line^.next;
(*
if ide_line = nil then begin WRITELN( ' Line record not created => STOP.' ); PASCAL_EXIT( 2 )  end;
*)
        ide_line^.idcd := svcd;                 { Set the istatment index in the line record }
        ide_lcnt := sy_linenb - ide_lcnt        { Get the number of Line numbers change }
      end;
  ET_SKIP:
    if sy_err then SKIP_SYMBOL( semicolsy, true )
  until sy = eofsy;
  CLOSE( fset )
end LOAD_SETTING;





{ *****************************************************************************
  ***                                                                       ***
  ***      Procedure(s) and/or function(s) create a new setting file        ***
  ***                                                                       ***
  *****************************************************************************
}


function  OUTPUT_SETTING( in_var out_path: string ): boolean;
var
  p:                  setf_ptr;
  ident:  string( 30 );
  n:              Dint;
  bok:                 boolean;
  sdate, stime:   string( 14 );
  msg:       string( MAX_STR );
  out:                    text;

begin
  OPEN( out, out_path, [write_file, error_file] );
  if iostatus <> 0 then
  begin
    WRITEV( msg, 'Cannot Create the Output file "', out_path, '"' );
    DRAW$MESSAGE( msg );
    bok := false
  end
  else
  begin
    n := 0;
    DATE( sdate ); TIME( stime );
    WRITELN( out, '!**' );
    WRITELN( out, '!**   ---   Draw Setting for FLTK/FTGL Draw Server   ---' );
    WRITELN( out, '!**' );
    WRITELN( out, '!**   Calibration program Version ', VERSION, ' of ', VDATE );
    WRITELN( out, '!**   by P. Wolfers CNRS, Institut Neel, Grenoble FRANCE' );
    WRITELN( out, '!**   Calibration program run the ', sdate, ' at ', stime, '.' );
    WRITELN( out, '!**   The Calibration program modified the lines "! *" flagged comment.' );
    WRITELN( out, '!**' );

    p := fline;
    while p <> nil do
    with p^ do
    begin
      if idcd = idc_noop then
        if line <> nil then WRITELN( out, line^ )
                       else WRITELN( out )
      else
        with idm_tbl[idcd] do
          if ide_line = p then                  { To suppress any duplicate statement, we keep the last one only }
            case idcd of
              idc_width:
                WRITELN( out,  N_SCR_WIDTH:16, '=':2, src_width:18:2,  ';':4, '! * Screen Width (cm).' );
              idc_height:
                WRITELN( out, N_SRC_HEIGHT:16, '=':2, src_height:18:2, ';':4, '! * Screen Height (cm).' );
              idc_resol:
                WRITELN( out,  N_SRC_RESOL:16, '=':2, src_xresol:8,    ',':2, src_yresol:8, ';':4,
                              '! * Screen Resolution (raster units).' );
              idc_winsize:
                WRITELN( out,  N_WIN_SIZES:16, '=':2, win_width:8:2,   ',':2, win_height:8:2, ';':4, '! * Window Sizes (cm).' );
              idc_winpos:
                WRITELN( out,    N_WIN_POS:16, '=':2, win_posx:8:2,    ',':2, win_posy:8:2,   ';':4, '! * Window Position (cm).' );

            otherwise
            end;
      p := next
    end;
    CLOSE( out );
    bok := true
  end;
  OUTPUT_SETTING := bok
end OUTPUT_SETTING;





{ *****************************************************************************
  ***                                                                       ***
  ***       Procedures and/or functions to manage the draw interface        ***
  ***                                                                       ***
  ***                                and                                    ***
  ***                                                                       ***
  ***                 to perform the calibration process.                   ***
  ***                                                                       ***
  *****************************************************************************
}



procedure DRAW_BEGIN;
begin
  if Server_On then DRAW$END
               else Server_On := true;
  DRAW$INIT( x_paper, y_paper, unit, 'Draw Calibration' );
  DRAW$PICTURE( '', x_paper, y_paper, false, false );
  DRAW$PIC_VIEW( 0 );
  xorg := x_paper/2.0; yorg := y_paper/2.0;     { Get the Display Canvas Centre coordinates }
  DRAW$ORG( xorg, yorg, 0 );                    { Origine at centre }
  DRAW$COLOR( 1 );                              { Color is Black }
  DRAW$TEXT_FONT( 3, 1 );                       { Set a Nice Font }
  DRAW$TEXT_ATTR( 3, 4, 1, 1.0, 0.0 );          { Text is centered on x and y }
  DRAW$LINE_ATTR( 1, 1.0 )                      { Continue Line with thickness of 1.0 }
end DRAW_BEGIN;



procedure STOP_ALL;
begin
  if Server_On then DRAW$END;
  WRITELN( ' New_Calibration Anomalous End.' );
  PASCAL_EXIT( 2 )
end STOP_ALL;



procedure DRAW_INQUIRE_INT( in_var Title: string; var ix, iy: Dint );
var
  smsg: [static] string( 64 );
  status: Dint;
  pannel: Dint;

begin
  pannel := DRAW$PANNEL_CREATE( 12.0, 4.0, Title );
  DRAW$PANNEL_ATTR( pannel,  true, 'HELVETICA', 3 );    { Set Label Character Font in Bold }
  DRAW$PANNEL_ATTR( pannel, false, 'HELVETICA', 0 );    { Set Text/value Character Font }
  DRAW$PANNEL_ATTR( pannel, Drw_Align_Left );
  DRAW$PANNEL_ATTR( pannel, false, 12.0 );              { Set Text/value Character Size }
  DRAW$PANNEL_ATTR( pannel,  true, 12.0 );              { Set Label Character Size }
  DRAW$PANNEL_ADD( pannel, 2.0, 2.0, 3.0, 0.6, 'X (p)', ix, 200, 5000 );
  DRAW$PANNEL_ADD( pannel, 8.0, 2.0, 3.0, 0.6, 'Y (p)', iy, 200, 4000 );

  repeat
    status := DRAW$PANNEL_REQUEST( pannel );
    DRAW$PANNEL_GET( pannel );
    if status < 0 then STOP_ALL;
    WRITEV( smsg, 'The read values was (', ix:8, ', ', iy:8, '). That is Correct?' );
    status := DRAW$GET_ANSWERD( smsg )
  until status = 1;
  DRAW$PANNEL_DELETE( pannel )
end DRAW_INQUIRE_INT;




procedure DRAW_INQUIRE_FLOAT( in_var Title: string; var xx, yy: Dfloat );
var
  smsg: [static] string( 64 );
  status: Dint;
  pannel: Dint;

begin
  pannel := DRAW$PANNEL_CREATE( 12.0, 4.0, Title );
  DRAW$PANNEL_ATTR( pannel,  true, 'HELVETICA', 3 );    { Set Label Character Font in Bold }
  DRAW$PANNEL_ATTR( pannel, false, 'HELVETICA', 0 );    { Set Text/value Character Font }
  DRAW$PANNEL_ATTR( pannel, Drw_Align_Left );
  DRAW$PANNEL_ATTR( pannel, false, 12.0 );              { Set Text/value Character Size }
  DRAW$PANNEL_ATTR( pannel,  true, 12.0 );              { Set Label Character Size }
  DRAW$PANNEL_ADD( pannel, 2.0, 2.0, 3.0, 0.6, 'X (cm)', xx, 5.0, 100.0, 0.01 );
  DRAW$PANNEL_ADD( pannel, 8.0, 2.0, 3.0, 0.6, 'Y (cm)', yy, 5.0, 200.0, 0.01 );

  repeat
    status := DRAW$PANNEL_REQUEST( pannel );
    DRAW$PANNEL_GET( pannel );
    if status < 0 then STOP_ALL;
    WRITEV( smsg, 'The read values was (', xx:10:2, ', ', yy:10:2, '). That is Correct?' );
    status := DRAW$GET_ANSWERD( smsg )
  until status = 1;
  DRAW$PANNEL_DELETE( pannel )
end DRAW_INQUIRE_FLOAT;








{ *****************************************************************************
  ***                                                                       ***
  ***    Procedures and/or function to manage the input and output files    ***
  ***                                                                       ***
  ***                             references                                ***
  ***                                                                       ***
  *****************************************************************************
}


procedure CHECK_ACCESS;
begin
  rdflg := FILE_ACCESS_CHECK( file_path, 4 { File Read access test } );
  wrflg := FILE_ACCESS_CHECK( file_path, 2 { File Write access test } )
end CHECK_ACCESS;



procedure USR_REQUEST_FOR_INPUT;
var
  nam: [static] string( MAX_STR );
  ifl, ist:         Dint := 0;

begin
  file_path := DEF_SETFILE;                     { When the default access is bad, reset the default name }
  repeat
    ist := DRAW$SELECT_FILE( 'Input Draw Server Setting File', fltr, file_path, nam, ifl, 0 );
    if ist <= 0 then STOP_ALL;
    file_path := nam;
    CHECK_ACCESS;
    if not rdflg then
    begin
      WRITEV( msg, CNTOPE, file_path, '"' );
      DRAW$MESSAGE( msg )
    end
  until rdflg
end USR_REQUEST_FOR_INPUT;



procedure LOCATE_SETTING_FILE;
var
  i0, i1, ir, ll: Dint;
  bk:            Dbool;

begin
  rdflg := false;
  wrflg := false;
  file_path.length := 0;
  ll := GET_LOGICAL( file_path, LOG_NAME );
  if ll < 0  then
  begin { The Logical is not defined }
    i0 := 1;
    ll := SEARCH_PATH.length;
    bk := false;
    repeat
      { Skip any previous path field separator(s) }
      while (i0 <= ll) and (SEARCH_PATH[i0] = PATH_SEP) do i0 := I0 + 1;
    exit if i0 > ll;
      i1 := 1;
      file_path[1] := SEARCH_PATH[i0];
      i0 := i0 + 1;
      { Get each Path Field character - stop on end-of-path or on a path-separator }
      while (i0 <= ll) and (SEARCH_PATH[i0] <> PATH_SEP) do
      begin
        if i1 < file_path.capacity then i1 := i1 + 1;
        file_path[i1] := SEARCH_PATH[i0];
        i0 := i0 + 1
      end;
      { When the field is not finished by directory separator we append ones }
      if (file_path[i1] <> LNM_SEP) and (file_path[i1] <> DIR_SEP) then
      begin  i1 := i1 + 1; file_path[i1] := DIR_SEP  end;
      { We append the File name to search }
      for ii := 1 to DEF_SETFILE.length do
      begin  i1 := i1 + 1; file_path[i1] := DEF_SETFILE[ii]  end;
      { and we set the file path string length }
      file_path.length := i1;
      bk := FILE_ACCESS_CHECK( file_path, 0 { File existing test } );
      { Loop until founded or on end of path }
    until bk or (i0 > ll);
    if bk then CHECK_ACCESS { Check for read and write access }
  end
  else CHECK_ACCESS
end LOCATE_SETTING_FILE;



procedure INIT_SETTING_ACCESS;
var
  ifl, ist:       Dint;

begin
  ifl := 0;
  if argc > 1 then
  begin
    file_path := argv[1]^;                      { Get the first command parameter as the setting file to read ... }
    CHECK_ACCESS;                               { ... and check for read/write access }
    WRITEV( msg, CNTOPE, file_path, '"' );
    if not rdflg then DRAW$MESSAGE( msg );      { Send the Warning message to the user }
    LOCATE_SETTING_FILE;                        { Get the default specification }
    if not rdflg then USR_REQUEST_FOR_INPUT     { Ask to user for a readable file on Access failure }
  end
  else
  begin
    LOCATE_SETTING_FILE;                        { Try to find the standard draw setting file }
    if not rdflg then USR_REQUEST_FOR_INPUT     { Ask to user for a readable file on Access failure }
  end
end INIT_SETTING_ACCESS;



begin { Main Program }
  DRAW_BEGIN;                                   { First init of Draw system }
  INIT_SETTING_ACCESS;                          { Locate the input setting file }
  LOAD_SETTING;                                 { Load the setting file in memory }

  if sy_gblerr then
  begin
    DRAW$MESSAGE( 'Syntax Error(s) detected in the Draw Server Setting file. See Detail in the standard error file.' );
    STOP_ALL
  end;

  src_xresol := DEF_XRESOL;
  src_yresol := DEF_YRESOL;

  DRAW_INQUIRE_INT( 'Please Enter the Known Resolution of your Screen', src_xresol, src_yresol );

  src_width  :=  src_xresol/30.0;               { Force a known unit mode ... }
  src_height :=  src_yresol/30.0;
  win_width  :=  0.60*src_width;                { ... with a not too large window (in raster unit }
  win_height :=  0.60*src_height;
  win_posx   :=   1.0; win_posy   := 1.0;

  OUTPUT_SETTING( 'tmp.draw_setting' );
  SET_LOGICAL( 'DRAW_SERVER_SETTING', 'tmp.draw_setting', 1 );  { Clear the previously defined Settings access }
  DRAW_BEGIN;                                   { Restart the Draw system }

  DRAW$STRING_DISPLAY( 'Increase the Window as possible to get its accurate sizes, and clic anywhere inside' );

  DRAW$MESSAGE( 'Increase the Window as possible and clic inside' );

  (* DRAW$MEssage( 'Increase the Window as possible to get its accurate sizes, and clic anywhere inside' ); *)
  status := DRAW$GET_POSITION( xx, yy );

  DRAW$STRING_DISPLAY( '' );

  { Plot the Reticle }
  DRAW$COLOR( 0.4, 0.4, 0.6 );
  DRAW$LINE_ATTR( 1, 2.0 );
  DRAW$OUT_MODE( 1 );
  xx := xorg*0.99999; yy := yorg*0.99999;
  charsz := 0.08*yorg;
  DRAW$PLOT( -0.95*xx, -0.75*yy, false );
  DRAW$PLOT(      -xx, -0.80*yy,  true );
  DRAW$PLOT( -0.95*xx, -0.85*yy,  true );
  DRAW$PLOT(      -xx, -0.80*yy, false );
  DRAW$PLOT(       xx, -0.80*yy,  true );
  DRAW$PLOT(  0.95*xx, -0.85*yy, false );
  DRAW$PLOT(       xx, -0.80*yy,  true );
  DRAW$PLOT(  0.95*xx, -0.75*yy,  true );

  DRAW$PLOT(  0.75*xx, -0.95*yy, false );
  DRAW$PLOT(  0.80*xx,      -yy,  true );
  DRAW$PLOT(  0.85*xx, -0.95*yy,  true );
  DRAW$PLOT(  0.80*xx,      -yy, false );
  DRAW$PLOT(  0.80*xx,       yy,  true );
  DRAW$PLOT(  0.85*xx,  0.95*yy, false );
  DRAW$PLOT(  0.80*xx,       yy,  true );
  DRAW$PLOT(  0.75*xx,  0.95*yy,  true );
  DRAW$PLOT(      0.0,      0.0, false );

  DRAW$COLOR( 3 );
(*                                 { Write in Blue }
  WRITEV( dsp_string, 'Temporary GL Window Size = (', x_paper:8:2, ' * ', y_paper:8:2, ') cm^2.' );
  DRAW$STRING( -xx*0.1, yy*0.80, 0.0, charsz, dsp_string );

  WRITEV( dsp_string, 'Picture Size = (', x_paper:8:2, ' * ', y_paper:8:2, ') pixel^2.' );
  DRAW$STRING( -xx*0.1, yy*0.60, 0.0, charsz, dsp_string );
*)
  DRAW$STRING( -xx*0.1, yy*0.30, 0.0, charsz, 'You Must Measure the real Dimensions of the external' );
  DRAW$STRING( -xx*0.1, yy*0.15, 0.0, charsz, 'frame (The Blue Grey Arrow Line Lengths)' );

  DRAW$COLOR( 1 );

  { Code for Draw_Picture_Scales :
     1 =>  d_xscale,  d_yscale, # Screen(Scr) Scales for x and y in Pixel/cm,
    -1 =>     dt_rx,     dt_ry, # Size of Screen GL Window in pixels,
    -2 =>     ds_rx,     ds_ry, # Size of Screen GL Window in Cm with border,
    -3 =>   d_wight,   d_hight, # Size of Screen in Cm,
     0 or other
       => cxy_scale, sxy_scale, # Current and saved scales in User_cm/Screen_cm.
  }

  DRAW$PICTURE_SCALES( -1, dt_rx, dt_ry );      { Get The actual Window size in raster units (pixels) }
  WRITELN( ' dt_rx, dt_ry = ', dt_rx, dt_ry );

  DRAW$PICTURE_SCALES( -2, ds_rx, ds_ry );      { Get The actual Window size in (pseudo) centimeters }
  WRITELN( ' ds_rx, ds_ry = ', ds_rx, ds_ry );

  DRAW$PICTURE_SCALES(  0, cxy_scale, sxy_scale );{ Get The actual DRAW Scale user/surface and CV on Screen }
  WRITELN( ' cxy_scale, sxy_scale = ', cxy_scale, sxy_scale );


  xx := sv_w_width; yy := sv_w_heigh;
(* xx := 31.17; yy := 20.00; *)
  DRAW_INQUIRE_FLOAT( 'Please, Enter the Measured Size (X and Y) in cm of the arrows.', xx, yy );
(*
  WRITELN( ' x_paper, y_paper (cm?) = ', x_paper:8:2, y_paper:8:2 );
  WRITELN( '   ds_rx,   ds_ry (cm?) = ', ds_rx:8:2, ds_ry:8:2 );
*)
  ds_rx := xx*(ds_rx/(x_paper*cxy_scale));                  { Compute the true window sizes from the mesures and ... }
  ds_ry := yy*(ds_ry/(y_paper*cxy_scale));
(*
  WRITELN( '   ds_rx,   ds_ry (cm)  = ', ds_rx:8:2, ds_ry:8:2 );
*)
  src_width  := ds_rx*(src_xresol/dt_rx);       { ... deduce the real sizes of the screen }
  src_height := ds_ry*(src_yresol/dt_ry);
(*
  WRITELN( ' Screen sizes (r.u.)    = ', src_xresol:8, src_yresol:8 );
  WRITELN( ' Screen sizes (cm)      = ', src_width:8:2, src_height:8:2 );
*)
  win_width  := sv_w_width; win_height := sv_w_heigh;

  DRAW_INQUIRE_FLOAT( 'Please, Enter the Desired sizes of the window.', win_width, win_height );

  DRAW_INQUIRE_FLOAT( 'Please, Enter the Desired Position of the window (low left corner).', win_posx, win_posy );

  OUTPUT_SETTING( 'tmp.draw_setting' );
  DRAW_BEGIN;                                   { Restart the Draw system using the new calibration }

  DRAW$OUT_MODE( 1 );
  DRAW$COLOR( 3 );
  DRAW$LINE_ATTR( 1, 2.0 );
  DRAW$CIRCLE( 0.0, 0.0, 5.0 );

  DRAW$TEXT_FONT( 7 ); DRAW$COLOR( 0.8, 0.4, 0.2 );
  DRAW$TEXT_ATTR( TXT_HALN_CENTRE, TXT_VALN_HALF, TXT_PATH_RIGHT (* , 1.0 {char x-scale}, 0.0 {spacing}, 0.0{depth} *) );
  DRAW$TEXT_IN_BOX( 0.0, 0.0, xorg, yorg,
                    'Your calibration is done, we hope that you are satisfied.', TXT_HALN_JUSTIFY,, 1.5 );

  (* SLEEP( 3.0 ); *)
  DRAW$STRING_DISPLAY( 'Dialog is called to verify the scaling, Continue by Resume (Escape key) or Quit  (CTRL Q key)' );

  status := DRAW$DIALOG( 1 );
  if status < 0 then STOP_ALL;

  DRAW$STRING_DISPLAY( '' );

ET_SELOUT:
  iflt := 1;
  status := DRAW$SELECT_FILE( 'Define the Draw Server Setting File to create or replace', fltr, file_path, out_path, iflt, 1);
  if status < 0 then STOP_ALL;

  if out_path = file_path then
    if not FILE_RENAME( file_path, file_path||'_old_saved' ) then
    begin
      DRAW$MESSAGE( ' The input (old) setting file cannot be rename, Please give an other output filename.' );
      goto ET_SELOUT
    end;

  OUTPUT_SETTING( out_path );

  if Server_On then DRAW$END;                   { Normal End of New_Calibration program }
  WRITELN( ' New_Calibration Normal End with the creation of file "', out_path, '".' )
end DRAW_CALIBRATION.
