{%pragma trace 1;}
(*

             *****  CPAS Portable Shell  *******

*************************************************************************
*                                                                       *
*                                                                       *
*                                                                       *
*               C P A S   P O R T A B L E   S H E L L                   *
*                                                                       *
*                ( Common Part of any interpreter )                     *
*                                                                       *
*         by :                                                          *
*                                                                       *
*             P. Wolfers                                                *
*                 www.pierre.wolfers.fr                                 *
*                                             FRANCE.                   *
*                                                                       *
*                                                                       *
*        Portable CPAS SHELL -  Version V1.7B -  31-JUL-2019            *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

****************************************************************************************
*                                                                                      *
*      Model of use in a main program file :                                           *
*                                                                                      *
*                                                                                      *
*  program Cpas_Shell;                                                                 *
*    { Portable Shell Program }                                                        *
*                                                                                      *
*    %include 'PASSRC:cpsh_base_env.pas';   { Get the common shell definitions }       *
*                                                                                      *
*                                                                                      *
*                                                                                      *
*                                                                                      *
*                                                                                      *
*  { The following specific definitions must be keep the same identifier names }       *
*  const                                                                               *
*    cpsh_heading= 'CPAS-SHell V1.3-A of the 03-NOV-2009';                             *
*    cpsh_deflst = 'nl:';             { Define the default listing file }              *
*    cpsh_errfile= 'cpsh.err';        { Define the source error message file }         *
*    cpsh_prompt = ' cpsh>';          { Define the Shell prompt }                      *
*    cpsh_defsrc = 'cpsh_env.std';    { Define the Initial Script file }               *
*                                                                                      *
*                                                                                      *
*    { Define the path to find the setup start file(s) for each O.S. }                 *                                                                                                                                      *
*    cpsh_wind_path = 'CPSH_ETC:;/C/cygwin/usr/local/etc/;./;';                        *
*    cpsh_unix_path = 'CPSH_ETC:;/usr/local/etc;./';                                   *
*    cpsh_vms_path  = 'CPSH_ETC:;/CPAS$ENV:;';                                         *
*                                                                                      *
*                                                                                      *
*  type                               { Define the specific types }                    *
*    spc_fnc = ( activate, deactivate, status );                                       *
*                                                                                      *
*  { Define the Specific procedure here, local global or external }                    *
*                                                                                      *
*                                                                                      *
*                                                                                      *
*  {{  *** List of functions (sample) to use to execute a specific statmenet }}        *
*  procedure INSYMBOL;                { To get a new symbol from the input stream }    *
*                                     { The INSYMBOL result is in the record sy_sym }  *
*  procedure SRC_ERROR( modulesy: error_mdnam; { Four Characters issuer code }         *
*                       number:   integer;     { Error number in cpsh_errfile file }   *
*                       severity: error_sev;   { Severity: e_warning, e_error, ... }   *
*                                              { ... e_severe or e_fatal }             *
*  procedure SRC_ERROR_S( modulesy: error_mdnam; { To use to insert one or two ... }   *
*                         number:   integer;     { ... strings (id1, id2) in the ... } *
*                         severity: error_sev;   { ... error message at flagged "`" }  *
*                         var id1, id2: [optional] string );     { ... char place. }   *
*  procedure SKIP_SYMBOL( sy: symbol );        { To skip the text until the next sy }  *
*  function  IDE_SEARCH( berr: boolean ): ide_ptr; { To find the INSYMBOled ident. }   *
*                                              { if berr=false and not exist => nil }  *
*  function  IDE_NEW( knd: ide_kinds; ilvl: integer := 0 ): ide_ptr;                   *
*          { To create the identifier sy_ident with the specified class in             *
*            the current display level tree. If it is already present, then it is not  *
*            created and an error message is edited.}                                  *
*  function  GET_INTEXPR( iv: integer ): integer;  { To get an integer Expr. value }   *
*  function  GET_FLTEXPR( rv: shreal ): shreal;    { To get an floatting Expr. value } *
*  procedure GET_STREXPR( var st: string );        { Get a string value as result }    *
*  procedure GET_NUMEXPR( var bflt: boolean;       { Get an undefined numeric expr. }  *
*                         var   iv: integer;       { Integer value if not bflt }       *
*                         var   rv:  shreal );     { floatting value if blt }          *
*  procedure GET_EXP_VALUE( var knd: exp_kinds;    { Get an expression value, knd is } *
*                           var  iv: integer;      { its kind, iv the integer value }  *
*                           var  rv: shreal;       { if knd = exp_valint, rv its flt } *
*                           var  st: string );     { val. if knd = exp_valflt and st } *
*                                                  { if knd = exp_valstr }             *
*  procedure GET_EXP_REFER( var rec: exp_rec );    { Procedure to get an ident. ref. } *
*                                                                                      *
*                                                                                      *
*                                                                                      *
*  [global]                                                                            *
*  procedure SET_SPECIFIC_SYMBOL( procedure SET_KWD( in_var kname: string;             *
*                                                             skw: symbol;             *
*                                                            opkw: wennum;             *
*                                                             ske: symbol:=nothing) ); *
*  begin { To Define all specific Key-Words }                                          *
*    SET _KWD( 'spckwd_1', specificsy, 0 { integer or ennumerated identifier );        *
*                     .  .  .                                                          *
*    SET _KWD( 'spckwd_n', specificsy, n )                                             *
*  end SET_SPECIFIC_SYMBOL;                                                            *
*                                                                                      *
*  The symbol specificsy can be replaced by its synonym "specific0sy", the additional  *
*  symbol "specific1sy" can be use to define 256 other keywords (512 as total).        *
*                                                                                      *
*                                                                                      *
*                                                                                      *
*  {{  *** List of functions to use to POP or PUSH value from the operational Stack }} *
*    function EXP_GETKINDS( ish: integer ): exp_kinds;  { To get parmam kind }         *
*    procedure POP_EXP_REFER( var rec: exp_rec );  { To get a reference of param }     *
*    procedure POP_EXP_VALUE( var knd: exp_kinds; var iv: integer; { To get an ... }   *
*                                                 var rv: shreal;  { expression value }*
*                                                 var st: string ); { of any type }    *
*    procedure POP_NUMEXPR( var bflt: boolean;     { To get a numeric value }          *
*                           var   iv: integer;     { if bflt, value was a float in rv }*
*                           var   rv:  shreal );   { else integer in iv }              *
*    function  POP_INTEXPR( iv: integer ): integer;{ Get an integer value of param. }  *
*    function  POP_FLTEXPR( iv: shreal ): shreal;  { Get a floatting value of param. } *
*    procedure POP_STREXPR( var st: string );      { Get a string value of param }     *
*    procedure EXP_PUTINT( iv: integer ): integer; { Put an integer value as result }  *
*    procedure EXP_PUTFLT( iv: shreal ): shreal;   { Put a floatting value as result } *
*    procedure EXP_PUTSTR( in_var st: string );    { Put a string value as result }    *
*  }}                                                                                  *
*                                                                                      *
*  [global]                                                                            *
*  procedure DEFINE_SPECIFIC_ENTRY( procedure DEF_ENTRY( in_var nam: string;           *
*                                                               knd: ide_kinds;        *
*                                                               fnc: wennum;           *
*                                                               npa1: integer;         *
*                                                               npa2: integer := -1 ); *
*  begin { To Define all specific built_in entries }                                   *
*    DEF_ENTRY(   'activate', ide_functsp, 0,  1, 4 );   { Funct. with 1 to 4 params } *
*    DEF_ENTRY( 'deactivate', ide_functsp, 0,  1 );      { Funct. with 1 parameter }   *
*                     .  .  .                                                          *
*    DEF_ENTRY(  'predef_v1', ide_parint, blt_noop, 0 ); { Predefined integer Varbl. } *
*    pdf_v1 := sy_idenew;                                { Save the identifier addr.}  *
*    sy_idenew^.ide_int := 0;                            { Init the value to 0 }       *
*                     .  .  .                                                          *
*    DEF_ENTRY(  'predef_vn', ide_parflt, blt_noop, 0 ); { Predefined float Varbl. }   *
*    pdf_v1 := sy_idenew;                                { Save the identifier addr.}  *
*    sy_idenew^.ide_flt := 10.0;                         { Init the value to 10.0 }    *
*                     .  .  .                                                          *
*    DEF_ENTRY( 'predef_stb', ide_tabstr, blt_noop, 0 ); { Predef. array of string }   *
*    pdf_vst := sy_idenew;                               { Save the identifier addr.}  *
*    with sy_idenew^ do                  { To Create the array ... }                   *
*    begin                               { init it to dim = [3] }                      *
*      NEW( ide_adm );                   { allocate an array descriptor }              *
*      with ide_adm^ do                  { and fill it for ... }                       *
*      begin                                                                           *
*        idedim_nxt := nil;              { Only one dimension ... }                    *
*        idedim_stp :=   1;              { ... with a step of one string (1 dim.) }    *
*        idedim_siz :=   3;              { ... and the size of 3 elements }            *
*      end;                                                                            *
*      NEW( ide_aas, 3 );                { Allocate the 3 string descriptors }         *
*      with ide_aas^ do                  { ... and fill it }                           *
*      begin                                                                           *
*        NEW( ide_stb[1], 5 );           { Put 'abcde' in the first string ... }       *
*        for i := 2 to 3 do                                                            *
*          ide_stb[i] := nil             { and '' the the other strings }              *
*      end                                                                             *
*    end;                                                                              *
*                     .  .  .                                                          *
*  end DEFINE_SPECIFIC_ENTRY;                                                          *
*                                                                                      *
*                                                                                      *
*                                                                                      *
*  [global]                                                                            *
*  procedure MANAGE_SPECIFIC_SYMBOL( spc_sym: wennu );                                 *
*  begin                                                                               *
*    case spc_sym of                                                                   *
*      0 { spc_symbol_0 }:                                                             *
*               .  .  .                                                                *
*      n { spc_symbol_n }:                                                             *
*               .  .  .                                                                *
*    end                                                                               *
*  end MANAGE_SPECIFIC_SYMBOL;                                                         *
*                                                                                      *
*                                                                                      *
*                                                                                      *
*  [global]                                                                            *
*  procedure EXECUTE_SPECIFIC_ENTRY( npa: integer; fnc: wennum );                      *
*  { npa = number of effective parameter(s), fnc = the function specification }        *
*  var                                                                                 *
*    s1, s2: string( 128 );                                                            *
*    i1, i2: integer;                                                                  *
*    f1, f2: real;                                                                     *
*                                                                                      *
*  begin                                                                               *
*    case spc_fnc( fnc ) of                                                            *
*      activate: begin { must have One parameter, can be have 3 optional parameters }  *
*          POP_STREXPR( s1 ); { Get the function name to activate }                    *
*          if npa > 1 then i1 := POP_INTEXPR( 0 )  { get P2 int. val. - 0 def. val. }  *
*                     else i1 := 0;                                                    *
*          if npa > 2 then i2 := POP_INTEXPR( -1 ) { get P3 int. val. - 1 def. val. }  *
*                     else i2 := -1;                                                   *
*          if npa > 3 then f1 := POP_FLTEXPR( 0.0 ){ get P4 flt. val. 0.0 def. val. }  *
*                     else f1 := 0.0;                                                  *
*                     .  .  . < The computing for activate function >   .  .  .        *
*          EXP_PUTINT( <integer_returned_code> )                                       *
*        end;                                                                          *
*                     .  .  .                                                          *
*      deactivate: begin { Always one string parameter }                               *
*          POP_STREXPR( s1 ); { Get the function name to deactivate }                  *
*                     .  .  . < The computing for deactivate function >   .  .  .      *
*          EXP_PUTINT( <integer_returned_code> )                                       *
*        end;                                                                          *
*                     .  .  .                                                          *
*      status:   begin { always to parameters }                                        *
*          POP_STREXPR( s1 ); { Get the function name to get status }                  *
*                     .  .  . < The computing for status function >   .  .  .          *
*          EXP_PUTSTR( <returned_string> )                                             *
*        end                                                                           *
*    end                                                                               *
*  end EXECUTE_SPECIFIC_ENTRY;                                                         *
*                                                                                      *
*                                                                                      *
*                                                                                      *
*  %chaine 'PASSRC:cpsh_base.pas';  { Continue to interpreter main part }              *
*                                                                                      *
*                                                                                      *
*                                                                                      *
****************************************************************************************
*)



{ ******************************************************************** }
{ ***********     Types to get and parse a CPSH command   ************ }
{ ******************************************************************** }

const
  in_min = ORD( 'a' ) - ORD( 'A' );


type
  iocnt_opmode = ( iocnt_close,
                   iocnt_input,
                   iocnt_output
                 );

  iocnt_rec = record
    iocnt_mode: iocnt_opmode;
    iocnt_file: text
  end;



      {*******************************}
      { keyword information Structure }
      {*******************************}

  keyword_ptr = ^keyword;        { keyword pointer definition }

  keyword = record
    name:          pstring;      { Address of identifier string }
    leftp, rightp: keyword_ptr;  { Pointers to build binary tree }
    symb:          sym_rec;      { Keyword Syntaxe Definition }
    endksy:        symbol        { Related End Keyword or nothing }
  end;




{ ******************************************************************** }
{ ***********  Variables to get and parse a SHELL command  *********** }
{ ******************************************************************** }

var

  system_name: string := %SYSTEM;{ Get the target system name }

  keyword_tree:  keyword_ptr;    { keyword definition tree }

  udc_int,                       { Undeclared Integer }
  udc_ident:  ide_ptr;           { Undeclared Identifier }

  path_index:  integer :=   0;   { General path search index }

  pasenv_path,                   { Path to the current pasenv (that can be in installation process run) }
  inienv_path,                   { Path to the current base files (as *env.std) }
  curr_ident:  pstring := nil;   { Current identifier pointer for Search }

  idm_space,                     { Flag for space already out }
  idm_outmacro: boolean := false;{ Flag to generate a macro stream }

  idm_newmac:  idm_apt := nil;   { Macro in build }
  idm_actstk,                    { Stack pointer of active macro code }
  idm_tmphde,                    { Stack pointer of temporary macro parm }
  idm_defstk:  idm_ptr := nil;   { Stack pointer of Defined Macro Identifiers }

  exp_stkp:    integer := 0;                   { Current stack pointer level }
  exp_stk: array[1..max_stk] of exp_rec;       { Expression stack }
  exp_res:     [global] exp_rec;               { Current expression }

  sym_iof: array[1..max_lun] of iocnt_rec;     { User text file }

  furf_hde,
  furf_lst:    ide_ptr := nil;   { Final User Referencable Identifier list Header }


{ ****************************************************************************** }
{ ***  Global Variables of Cpas Shell environment declared with init values  *** }
{ ****************************************************************************** }


[global] var

  sy_ival:              integer; { INSYMBOL: Last readden integer number }
  sy_rval:               shreal; { INSYMBOL: Last readden floatting number }

  sy_noexec,                     { INSYMBOL: To flag the no exec mode }
  sy_nomacrflg:         boolean; { INSYMBOL: Flag to disable the mac. parm replace }
  sy_macro:             idm_ptr; { INSYMBOL: Last readen macro symbol pointer }

  sy_maclst,                     { INSYMBOL: Macro Expenssion Source Line }
  sy_string:         str_string; { INSYMBOL: Last readden string }
  sy_ident:          ide_string; { INSYMBOL: Last readden identifier }
  sy_cmin,                       { INSYMBOL: Last readden character in major }
  sy_ch:                   char; { INSYMBOL: Last readden character }
  sy_sym:               sym_rec; { INSYMBOL: Symbol found }
  sy_idenew:            ide_ptr; { IDE_NEW:  Pointer to the last created symbol }
  sy_endsy,                      { INSYMBOL: related End Symbol or nothing }
  lastsymb:              symbol; { Last symbol in source }

  sy_init_mod: boolean := true;  { Program init flag mode }

  symc_stk:     symc_ptr := nil; { Symbol context pointer }

  debug_macsrc,                  { Debug Macro Source }
  debug_mac,                     { Debug of macro }
  debug_sym,                     { Debug on Input syntax unit flag }
  debug_exp,                     { Debug on Expression element flag }
  debug_dat:   boolean := false; { Debug on data (for external ressources) }

  { Display tables for identifiers }
  last_ident,                    { Current last identifier table }
  disptab:          cpsh_dsptab;

  curr_disp,                     { Current top of lex display }
  curr_idlex:           integer; { Current identifier lex }

  ierr:                 integer; { General error code }

  data_mode,                     { Macro mode flag for insymbol }
  fatal_error:          boolean; { General flag for fatal error }

  env_path_search: string := PASCAL_PATH_ENV;{ System related environment path search }

  io_winfo,                      { Wait process out information }
  io_libpath,                    { Path to *env.std and *.err files }
  io_err,                        { Open File Error variable }
  io_count,                      { Array input count identifier pointer }
  io_eoln,                       { EOLN flag identifier pointer }
  io_eof:               ide_ptr; { EOF flag identifer pointer }







     {************************************************}
     { Title and Subtitle Listing of source managment }
     {************************************************}


%pragma trace 0;
procedure TRACING( var f: text ); external 'PAS__BACK_TRACING';

procedure TRACE( in_var s: string );
begin
  WRITELN;
  WRITELN( ' *** TRACING "', s, '" :' );
  TRACING( output )
end TRACE;
%pragma trace 1;




procedure LISTING_SET_TITLE;
var
  st: string( 255 );

begin
  st := ' The Current Command File is "' ||
        FILE_SPECIFICATION( src_control^.src_file );
  st := st || '".';
  with lst_current^ do
  begin
    if lst_title <> nil then DISPOSE( lst_title );
    NEW( lst_title, st.length );
    lst_title^ := st
  end
end LISTING_SET_TITLE;


procedure LISTING_SET_SBTTL( in_var sbttl: string );
begin
  with lst_current^ do
  begin
    if lst_sbttl <> nil then DISPOSE( lst_sbttl );
    NEW( lst_sbttl, sbttl.length );
    lst_sbttl^ := sbttl
  end
end LISTING_SET_SBTTL;





     {*****************************************}
     { Keyword Language Tree Creation routines }
     {*****************************************}


procedure SET_KEYWORD_TREE;

  procedure SETKEYWORD( in_var kname: string; skw: symbol; opkw: wennum; ske: symbol := nothing );
  var
    m, i:      integer;
    p, p1, p2: keyword_ptr;
    nm:        ide_string;

  begin { SETKEYWORD }
    NEW( p );
    with p^ do
    begin
      NEW( name, kname.length );
      name^ := kname;
      leftp := nil; rightp := nil;
      symb.sy   := skw;
      symb.op   := opkw;
      endksy    := ske
    end;
    p1 := keyword_tree;
    if p1 <> nil then
    begin
      repeat
        p2 := p1 { keep the memory of the last p1 };
        with p1^ do
        begin
          m := STR_MATCH( p^.name^, name^ );
          if m > 0 then  p1 := rightp else p1 := leftp
        end;
      until p1 = nil;
      if m > 0 then p2^.rightp := p else p2^.leftp := p
    end
    else keyword_tree := p
  end { SETKEYWORD };


begin { SET_KEYWORD_TREE }
  keyword_tree := nil;

  {**** set keywords symbol operators ****}

  SETKEYWORD(          'or',     lgorop,     or_op );
  SETKEYWORD(         'xor',     lgorop,    xor_op );
  SETKEYWORD(         'and',    lgandop,    and_op );
  SETKEYWORD(         'not',      notop,    not_op );
  SETKEYWORD(         'div',      mulop,   idiv_op );
  SETKEYWORD(         'mod',      mulop,   imod_op );
  SETKEYWORD(         'rem',      mulop,   irem_op );

        {**** set keywords symbol ****}

  SETKEYWORD(     'integer',       intsy,    no_op );
  SETKEYWORD(       'float',     floatsy,    no_op );
  SETKEYWORD(      'string',    stringsy,    no_op );
  SETKEYWORD(        'enum',      enumsy,    no_op );

  SETKEYWORD(    'sequence',  sequencesy,    no_op, endsy );
  SETKEYWORD(        'case',      casesy,    no_op, endsy );
  SETKEYWORD(        'when',      whensy,   stp_op );
  SETKEYWORD(       'other',     othersy,    no_op );
  SETKEYWORD(       'begin',     beginsy,    no_op, endsy );
  SETKEYWORD(         'end',       endsy,   stp_op );
  SETKEYWORD(          'if',        ifsy,    no_op, endsy );
  SETKEYWORD(        'then',      thensy,    no_op );
  SETKEYWORD(        'else',      elsesy,   stp_op );
  SETKEYWORD(       'while',     whilesy,    no_op );
  SETKEYWORD(      'repeat',    repeatsy,    no_op, untilsy );
  SETKEYWORD(       'until',     untilsy,   stp_op );
  SETKEYWORD(        'loop',      loopsy,    no_op, endsy );
  SETKEYWORD(          'do',        dosy,    no_op, endsy );
  SETKEYWORD(       'macro',     macrosy,    no_op, endsy );
  SETKEYWORD(       'purge',     purgesy,    no_op );

  SETKEYWORD(         'for',       forsy,    no_op, endsy );
  SETKEYWORD(          'to',        tosy,    no_op );
  SETKEYWORD(      'downto',    downtosy,    no_op );

  SETKEYWORD(  'open_input', open_inpsy,     no_op );
  SETKEYWORD( 'open_output', open_outsy,     no_op );
  SETKEYWORD( 'open_append', open_appsy,     no_op );
  SETKEYWORD(       'close',    closesy,     no_op );
  SETKEYWORD(        'read',     readsy,     no_op );
  SETKEYWORD(       'write',    writesy,     no_op );
  SETKEYWORD(     'writeln',  writelnsy,     no_op );
  SETKEYWORD(    'writemsg', writemsgsy,     no_op );
  SETKEYWORD(     'display',  displaysy,     no_op );
  SETKEYWORD(       'reply',    replysy,     no_op );

  SETKEYWORD(      'chaine',    chainesy,    no_op );
  SETKEYWORD(     'include',   includesy,    no_op );
  SETKEYWORD(      'pragma',    pragmasy,    no_op );


        {**** set application specific symbols ****}

  SET_SPECIFIC_SYMBOL( SETKEYWORD );

        {**** set micellious symbol ****}

  SETKEYWORD(         'eof',      peofsy,   stp_op );


        {**** end of the keyword list ****}

end { SET_KEYWORD_TREE };



     {*******************************************}
     { INSYMBOL group (Lexical parsing) Routines }
     {*******************************************}




[global]
procedure SRC_ERROR_S( modulesy:   error_mdnam;
                       number:     integer;
                       severity:   error_sev;
                       var id1, id2: [optional] string );
var
  smb: string( 32 );

begin
  if id1"address <> nil then
  with id1 do
  begin
    smb.length := length;
    for i := 1 to length do  smb[i] := body[i];
    ERR_PUT_SYMBOL( smb )
  end;
  if id2"address <> nil then
  with id2 do
  begin
    smb.length := length;
    for i := 1 to length do  smb[i] := body[i];
    ERR_PUT_SYMBOL( smb )
  end;
  SRC_ERROR( modulesy, number, severity )
end SRC_ERROR_S;



procedure SAVE_SYM_CNTX( svall: boolean := false );
var
  p: symc_ptr;

begin
  NEW( p );
  with p^ do
  begin
    symc_prv   := symc_stk;
    symc_ch    := sy_ch;
    symc_cmin  := sy_cmin;
    symc_svall := svall;
    if svall then
    begin
      symc_sym  := sy_sym;
      symc_ival := sy_ival;
      symc_rval := sy_rval;
      case sy_sym.sy of
        identsy:  begin
                    NEW( symc_string, sy_ident.length );
                    symc_string^ := sy_ident
                  end;
        stringconst: begin
                    NEW( symc_string, sy_string.length );
                    symc_string^ := sy_string
                  end
      otherwise
      end
    end
  end;
  symc_stk  := p
end SAVE_SYM_CNTX;



procedure RESTORE_SYM_CNTX;
var
  p: symc_ptr;

begin
  p := symc_stk;
  with p^ do
  begin
    symc_stk := symc_prv;
    sy_ch    := symc_ch;
    sy_cmin  := symc_cmin;
    if symc_svall then
    begin
      sy_sym   := symc_sym;
      sy_ival  := symc_ival;
      sy_rval  := symc_rval;
      case sy_sym.sy of
        identsy:  begin
                    sy_ident := symc_string^;
                    DISPOSE( symc_string )
                  end;
        stringconst: begin
                    sy_string := symc_string^;
                    DISPOSE( symc_string )
                  end;
      otherwise
      end
    end
  end;
  DISPOSE( p )
end RESTORE_SYM_CNTX;



procedure GEN_MACRO_CODE;
var
  p: idm_apt;

begin
  if (sy_ch <> ' ') or not idm_space then
  begin { A character must be put in the macro code }
    if sy_ch <= ' ' then idm_space := true
                    else idm_space := false;
    if idm_newmac = nil then
    begin
      NEW( idm_newmac, idm_mac_all );
      with idm_newmac^ do
      begin
        idm_use  :=   1;
        idm_ctb[1] := sy_ch
      end
    end
    else
      if idm_newmac^.idm_use >= idm_newmac^.idm_size then
      begin { Too large macro we must extend the macro code array }
        p := idm_newmac;
        NEW( idm_newmac, p^.idm_size + idm_mac_all );
        with idm_newmac^ do
        begin { We must extend the macro record }
          idm_use  := p^.idm_use + 1;
          for i := 1 to p^.idm_use do idm_ctb[i] := p^.idm_ctb[i];
          idm_ctb[idm_use] := sy_ch
        end;
        DISPOSE( p )
      end
      else
      with idm_newmac^ do
      begin
        idm_use := idm_use + 1;
        idm_ctb[idm_use] := sy_ch
      end
  end
end GEN_MACRO_CODE;



function SEARCH_MACRO( knd: idm_kinds ): idm_ptr;
var
  p: idm_ptr;

begin
  p := idm_defstk;
  while p <> nil do
  with p^ do
  begin
    if (knd = idm_undef) or (knd = idm_kind) then
      if (idm_name <> nil) and not idm_run then
        if STR_MATCH( sy_ident, idm_name^ ) = 0 then exit;
    p := idm_prv
  end;
  SEARCH_MACRO := p
end SEARCH_MACRO;




[global]
procedure ACTIVE_MACRO_CODE( p: idm_ptr; svall: boolean := false );
begin
  if p <> nil then
  with p^ do
  begin
    idm_nch    := 1;               { Set the macro index to begin of code }
    SAVE_SYM_CNTX( svall );        { Save the current INSYMBOL context }
    idm_run    := true;            { Set the run flag }
    idm_cntx   := idm_actstk;      { Push it in the active stack }
    idm_actstk := p;
    sy_ch      := ' '              { Set sy_ch for next INSYMBOL call }
  end
end ACTIVE_MACRO_CODE;



procedure OUT_MACRO_LINE;

  procedure OUT_MACRO_LINE_TEXT( var f: text; bterm: boolean );
  var
    mxl: integer;

  begin
    with sy_maclst, lst_current^ do
    begin
      if not bterm then LST_NEWLINE;
      WRITE( f, ' ':7, 'ME':13 );
      mxl := lst_lnsize - 20;
      if mxl >= length then
        WRITELN( f, sy_maclst )
      else
      begin
        WRITELN( f, sy_maclst:mxl );
        if not bterm then LST_NEWLINE;
        sy_maclst := SUBSTR( sy_maclst, mxl + 1 );
        WRITELN( f, ' ':7, 'ME_next':13, sy_maclst )
      end
    end
  end OUT_MACRO_LINE_TEXT;

begin
  with src_control^, lst_current^ do
  begin
    OUT_MACRO_LINE_TEXT( lst_file, false );
    if src_errnb > 0 then SRC_OUT_MAC_ERROR( OUT_MACRO_LINE_TEXT );
    sy_maclst.length := 0
  end
end OUT_MACRO_LINE;



[global]
procedure RET_OF_MACRO_CODE( p: idm_ptr );
begin
  if p = idm_actstk then
  with p^ do
  begin
    if debug_mac then          { When required ... }
      OUT_MACRO_LINE;          { ... output the macro code }
    RESTORE_SYM_CNTX;          { Restore the INSYMBOL context }
    idm_run    := false;       { Clear the run flag }
    idm_actstk := idm_cntx     { Set the old context }
  end
end RET_OF_MACRO_CODE;


procedure FREE_TEMP_MAC;
var
  p: idm_ptr;

begin
  p := idm_actstk;
  if p <> nil then
  begin
    with p^ do
    begin
      if debug_sym and (idm_tab <> nil) then
        with idm_tab^, lst_current^ do
          WRITELN( lst_file, ' Free the Temporary_Mac_Temp ', ' "', idm_ctb:idm_use, '".' );
      idm_actstk := idm_cntx;
      idm_tmphde := idm_parl;
      if idm_tab <> nil then DISPOSE( idm_tab )
    end;
    DISPOSE( p )
  end
end FREE_TEMP_MAC;





     {***************************************************}
     { Identifier group (identifier management) Routines }
     {***************************************************}


[global]
function  LEVEL_SEARCH( fp: ide_ptr ): ide_ptr;
{ search the sy_ident identifier in s specified identifier display tree }
var
  p:     ide_ptr;
  i:     integer;
  found: boolean;

begin { LEVEL_SEARCH }
  p     := fp;
  found := false;
  while not found and (p <> nil) do
  with p^ do
  begin
    if curr_ident = nil then i := STR_MATCH( ide_name^, sy_ident )
                        else i := STR_MATCH( ide_name^, curr_ident^ );
    if i = 0 then found := true
             else if i > 0 then p := p^.ide_left
                           else p := p^.ide_right
  end;
  LEVEL_SEARCH := p
end LEVEL_SEARCH;




[global]
function  IDE_SEARCH( berr: boolean ): ide_ptr;
{ Search the sy_ident identifier in the current identifier scope.
  If it is founded in an another class then the result is nil.
  If the identifier is not existing then nil is returned.
  If berr and "not found" then an error message is generated.}

const
  mdnam = 'IDES';

var
  ilvl: integer;
  p:    ide_ptr;

begin { IDE_SEARCH }
  curr_idlex := curr_disp; { set to predefined identifier lex level }
  repeat
    p := LEVEL_SEARCH( disptab[curr_idlex] );
    if p = nil then curr_idlex := PRED( curr_idlex )
  until (p <> nil) or (curr_idlex < 0);
  if berr and (p = nil) then
  begin
    SRC_ERROR_S( mdnam, 104, e_severe, sy_ident );
    p := udc_ident
  end;
  IDE_SEARCH := p
end IDE_SEARCH;




[global]
function IDE_NEW( knd: ide_kinds; ilvl: integer := 0 ): ide_ptr;
{ To create the identifier sy_ident with the specified class in
  the current display level tree. If it is already present, then it is not
  created and an error message is edited.}
const
  mdnam = 'NEWI';

var
  p, p1, p2: ide_ptr;
  i, ndisp: integer;
  errps, lleft, twdcl: boolean;

begin { IDE_NEW }
  if ilvl = 0 then ilvl := curr_disp;
  twdcl := false;
  { Look for place in the tree and for previous declaration }
  p2  := disptab[ilvl];
  p1  := nil;
  if p2 <> nil then
  repeat
    p1 := p2;
    with p2^ do
    begin
      i := STR_MATCH( ide_name^, sy_ident );
      if i = 0 then twdcl := true { Identifier is already existing }
               else if i < 0 then begin
                                    p2 := p2^.ide_right; lleft := false
                                  end
                             else begin
                                    p2 := p2^.ide_left;  lleft := true
                                  end
    end
  until (p2 = nil) or twdcl;

  { If twdcl then the specified identifier is existing ... }
  { else p1 is nil (tree is empty) or must be used to attach the new ident. }
  if twdcl then begin
                  SRC_ERROR_S( mdnam, 101, e_error, sy_ident ); p := p2
                end
           else begin { New identifier to create }
                  NEW( p );
                  with p^ do
                  begin
                    ide_kind := knd;
                    { Now set the identifier pointers }
                    NEW( ide_name, sy_ident.length );
                    ide_name^ := sy_ident;
                    ide_nrlnk := nil; ide_prlnk := nil;
                    ide_nxt   := nil; ide_left  := nil; ide_right := nil;
                    ide_adm   := nil;
                    ide_ronly := false;
                    ide_lock  := false;
                    { Now Initialize the identifier value }
                    case knd of
                      ide_functsp,
                      ide_funct:  ide_fnc := blt_noop;
                      ide_parstr: ide_str := nil;
                      ide_parint: ide_int := 0;
                      ide_parflt: ide_flt := 0.0;
                      ide_tabstr: ide_aas := nil;
                      ide_tabint: ide_aai := nil;
                      ide_tabflt: ide_aaf := nil;
                    otherwise
                    end
                  end;
                  { Now attach the new identifier to the identifier tree }
                  if p1 = nil then disptab[ilvl] := p
                              else if lleft then p1^.ide_left  := p
                                            else p1^.ide_right := p;
                  if last_ident[ilvl] <> nil then
                    last_ident[ilvl]^.ide_nxt := p;
                  last_ident[ilvl] := p
                end;
  sy_idenew := p;
  IDE_NEW := p
end IDE_NEW;



[global]
procedure USR_IDE_APPEND( p: ide_ptr );
begin
  if p <> nil then
  with p^ do
    if ide_prlnk = nil then { It is not already in the list }
    begin
      if furf_hde <> nil then
      begin
        furf_lst^.ide_nrlnk := p; ide_prlnk := furf_lst
      end
      else
      begin { The first in the list has its prlnk -> on itself to flag it }
        furf_hde := p; ide_prlnk := p
      end;
      furf_lst := p
    end
end USR_IDE_APPEND;



[global]
procedure USR_IDE_REMOVE( p: ide_ptr );
begin
  if p <> nil then
  with p^ do
    if ide_prlnk <> nil then
    begin { It is really in the list }
      { Adjust the next pointer string }
      if furf_hde = p then
        if furf_lst = p then
        begin { It is the unique id in the list = first + last }
          furf_lst := nil; furf_hde := nil
        end
        else
        begin { it the first in the list }
          furf_hde := ide_nrlnk;
          furf_hde^.ide_prlnk := furf_hde
        end
      else { It is not the first in the list }
        if furf_lst = p then
        begin { It is the last in the list }
          ide_prlnk^.ide_nrlnk := nil; furf_lst := ide_prlnk
        end
        else
        begin { it is not the last and not the first in the list }
          ide_prlnk^.ide_nrlnk := ide_nrlnk;
          ide_nrlnk^.ide_prlnk := ide_prlnk
        end;
      ide_prlnk := nil; ide_nrlnk := nil
    end
end USR_IDE_REMOVE;



[global]
function USR_IDE_LOCATE( in_var id_name: string ): ide_ptr;
var
  p: ide_ptr;

begin
  p := furf_hde;
  while p <> nil do
  begin
  exit if STR_MATCH( p^.ide_name^, id_name ) = 0;
    p := p^.ide_nrlnk
  end;
  USR_IDE_LOCATE := p
end USR_IDE_LOCATE;




function IDE_SEARCH_FROM_NAMEID( ip: ide_ptr ): ide_ptr;
{ Load a given identifier in sy_ident and re-search it }
{ use to get internal access of user defined generic of standard procedure/function }
var
  i, sz: integer;
  ir:    ide_ptr;

begin { IDE_SEARCH_FROM_NAMEID }
  ir := nil;
  if ip <> nil then
  with ip^ do
  if ide_name <> nil then
  begin
    curr_ident := ip^.ide_name;
    ir := IDE_SEARCH( true );
    curr_ident := nil
  end;
  IDE_SEARCH_FROM_NAMEID := ir
end IDE_SEARCH_FROM_NAMEID;



[global]
procedure DISPLAY_NEW;
const
  mdnam = 'DISN';

begin
  if curr_disp >= max_display then SRC_ERROR( mdnam, 4, e_severe )
  else
  begin
    curr_disp := SUCC( curr_disp );
    disptab[curr_disp]    := nil;
    last_ident[curr_disp] := nil
  end
end DISPLAY_NEW;



[global]
procedure DISPLAY_FREE;
const
  mdnam = 'DISF';

var
  p, p1: ide_ptr;

begin
  if curr_disp < 0 then SRC_ERROR( mdnam, 5, e_severe )
  else
  begin
    last_ident[curr_disp]    := nil;
    p := disptab[curr_disp];
    while p <> nil do
    begin
      if p^.ide_prlnk <> nil then USR_IDE_REMOVE( p );
      p1 := p^.ide_nxt;
      DISPOSE( p );
      p := p1
    end;
    curr_disp := PRED( curr_disp )
  end
end DISPLAY_FREE;



procedure DEFINE_BUILTIN;
const
   sysnam_unix = 'unix';
   sysnam_mosx = 'mac-os-x';
   sysnam_cygw = 'windows-Cygwin';
   sysnam_wind = 'windows';
   sysnam_unkn = 'unknown_os';

var
  st: ide_string;
  ip: ide_ptr;

  procedure PREDEFINE( in_var nam: string;
                              knd: ide_kinds;
                              fnc: ide_bltfnc;
                              npa1: integer; npa2: integer := -1 );
  var
    ids: pstring;

  begin
    sy_ident := nam;
    ip := IDE_NEW( knd );        { Create the predefined identifier }
    { For a Builtin Function set the appropriate specification value }
    with ip^ do
    begin
      ide_kind := knd;
      if (knd = ide_funct) or (knd = ide_functsp) then
      begin
        ide_fnc := fnc; ide_nparm := npa1;
        if npa2 < 0 then ide_nparmax := npa1
                    else ide_nparmax := npa2
      end
    end
  end PREDEFINE;

begin { DEFINE_BUILTIN }
  curr_disp := -1;               { Initialize the display to empty }
  DISPLAY_NEW;                   { Create the predefined display }
  PREDEFINE(         'abs',  ide_funct,     blt_abs,  1 );
  PREDEFINE(       'round',  ide_funct,   blt_round,  1 );
  PREDEFINE(       'trunc',  ide_funct,   blt_trunc,  1 );

  PREDEFINE(    'instring',  ide_funct,  blt_string,  1, 3 );
  PREDEFINE(      'substr',  ide_funct,  blt_substr,  1, 3 );
  PREDEFINE(       'index',  ide_funct,  blt_nindex,  2, 3 );
  PREDEFINE(      'length',  ide_funct, blt_slength,  1 );
  PREDEFINE(     'setcase',  ide_funct, blt_setcase,  1, 2 );
  PREDEFINE(  'check_char',  ide_funct, blt_checkch,  2, 3 );
  PREDEFINE('numeric_string',ide_funct, blt_checknst, 1 );
  PREDEFINE(     'element',  ide_funct,blt_selement,  1, 5 );

  PREDEFINE(    'file_spc',  ide_funct, blt_filespc,  1, 2 );
  PREDEFINE('file_spc_com',  ide_funct, blt_filspcm,  1, 3 );

  PREDEFINE(        'time',  ide_funct,    blt_time,  0 );
  PREDEFINE(        'date',  ide_funct,    blt_date,  0 );
  PREDEFINE(     'def_dir',  ide_funct,   blt_dfdir,  0 );
  PREDEFINE(     'get_env',  ide_funct,  blt_getenv,  1 );
  PREDEFINE(     'set_env',  ide_funct,  blt_setenv,  1, 2 );
  PREDEFINE(    'get_path',  ide_funct, blt_getpath,  2, 3 );
  PREDEFINE(    'set_path',  ide_funct, blt_setpath,  2, 3 );
  PREDEFINE(       'spawn',  ide_funct,   blt_spawn,  0, 1 );
  PREDEFINE( 'run_program',  ide_funct,     blt_run,  1, 2 );
  PREDEFINE('create_process',ide_funct,    blt_exec,  1, 2 );
  PREDEFINE('wait_for_exit', ide_funct,    blt_wait,  1 );
  PREDEFINE(       'chdir',  ide_funct,   blt_chdir,  1, 2 );
  PREDEFINE(  'file_exist',  ide_funct, blt_f_exist,  1, 2 );
  PREDEFINE( 'file_delete',  ide_funct, blt_f_delete, 1, 2 );
  PREDEFINE( 'file_rename',  ide_funct, blt_f_rename, 2, 4 );

  PREDEFINE(     'defined',  ide_funct, blt_id_exist, 1 );
  PREDEFINE( 'mac_defined',  ide_funct, blt_mid_exist,1 );

  PREDEFINE( 'insert_ident', ide_funct, blt_fidl_insert, 1 );
  PREDEFINE( 'remove_ident', ide_funct, blt_fidl_remove, 1 );
  PREDEFINE(   'substitute', ide_funct, blt_fidl_replace, 2 );
  PREDEFINE(  'sup_comment', ide_funct, blt_supcomment, 1, 3 );
  PREDEFINE(        'exit',  ide_funct,     blt_exit, 0, 1 );


  PREDEFINE(        'sqrt',  ide_funct,    blt_sqrt,  1 );

  PREDEFINE(         'sin',  ide_funct,     blt_sin,  1 );
  PREDEFINE(         'cos',  ide_funct,     blt_cos,  1 );
  PREDEFINE(         'tan',  ide_funct,     blt_tan,  1 );
  PREDEFINE(        'asin',  ide_funct,    blt_asin,  1 );
  PREDEFINE(        'acos',  ide_funct,    blt_acos,  1 );
  PREDEFINE(        'atan',  ide_funct,    blt_atan,  1, 2 );

  PREDEFINE(        'sind',  ide_funct,    blt_sind,  1 );
  PREDEFINE(        'cosd',  ide_funct,    blt_cosd,  1 );
  PREDEFINE(        'tand',  ide_funct,    blt_tand,  1 );
  PREDEFINE(       'asind',  ide_funct,   blt_asind,  1 );
  PREDEFINE(       'acosd',  ide_funct,   blt_acosd,  1 );
  PREDEFINE(       'atand',  ide_funct,   blt_atand,  1, 2 );

  PREDEFINE(          'ln',  ide_funct,      blt_ln,  1 );
  PREDEFINE(         'exp',  ide_funct,     blt_exp,  1 );


  PREDEFINE(   'array_dim',  ide_funct,   blt_ardim,  1, 2 );
  PREDEFINE( 'array_chdim',  ide_funct, blt_chardim,  2, 9 );

  PREDEFINE('array_search',  ide_funct,blt_arsearch,  2, 3 );

  PREDEFINE(   'array_add',  ide_funct,   blt_aradd,  2 );
  PREDEFINE(   'array_sub',  ide_funct,   blt_arsub,  2 );
  PREDEFINE(   'array_mul',  ide_funct,   blt_armul,  2 );
  PREDEFINE(   'array_div',  ide_funct,   blt_ardiv,  2 );

  PREDEFINE('$sys_setupdir', ide_parstr,    blt_noop,  0 );
  io_libpath  := ip;
  io_libpath^.ide_str := inienv_path;

  { Define the current CPASCAL/CPSH system library }
  PREDEFINE(  '$sys_pasenv', ide_parstr, blt_noop,  0 );
  ip^.ide_str := pasenv_path;


  PREDEFINE('$sys_command', ide_parstr,    blt_noop,  0 );
  with ip^ do
    if argv[0]^.length > 0 then
    begin
      NEW( ide_str, argv[0]^.length );
      ide_str^ := argv[0]^
    end else ide_str := nil;

  PREDEFINE(  '$sys_nparm', ide_parint,    blt_noop,  0 );
  ip^.ide_int := argc - 1;

  PREDEFINE(   '$sys_parm', ide_tabstr,    blt_noop,  0 );
  with ip^ do
  begin
    NEW( ide_adm );
    with ide_adm^ do
    begin
      idedim_nxt := nil;
      idedim_stp :=   1;
      idedim_siz :=  16;
    end;
    NEW( ide_aas, 16 );
    with ide_aas^ do
      for i := 1 to 16 do
      begin
        ide_stb[i] := nil;
        if i < argc then
          if argv[i]^.length > 0 then
          begin
            NEW( ide_stb[i], argv[i]^.length );
            ide_stb[i]^ := argv[i]^
          end
      end
  end;

  { Define the System Identification Variable }
  PREDEFINE(  '$sys_system', ide_parstr,   blt_noop,  0 );
  with ip^ do
    case sys_system of
      systyp_unix: begin  NEW( ide_str, sysnam_unix.length ); ide_str^ := sysnam_unix  end;
      systyp_mosx: begin  NEW( ide_str, sysnam_mosx.length ); ide_str^ := sysnam_mosx  end;
      systyp_cygw: begin  NEW( ide_str, sysnam_cygw.length ); ide_str^ := sysnam_cygw  end;
      systyp_wind: begin  NEW( ide_str, sysnam_wind.length ); ide_str^ := sysnam_wind  end;
    otherwise
      NEW( ide_str, sysnam_unkn.length ); ide_str^ := sysnam_unkn
    end;

  { Define the CPascal PATH variable }
  PREDEFINE(  '$sys_env_path', ide_parstr,   blt_noop,  0 );
  ip^.ide_str := env_path_search"address;

  { Define the current system kit-builder name }
  PREDEFINE(  '$system_name', ide_parstr,   blt_noop,  0 );
  ip^.ide_str := system_name"address;

  { Define the IO error variable }
  PREDEFINE(  '$cpsh_version', ide_parstr,  blt_noop,  0 );
  with ip^ do
  begin
    NEW( ide_str, cpsh_heading.length );
    ide_str^ := cpsh_heading
  end;

  { Define the IO error variable }
  PREDEFINE(  '$sys_ioerror', ide_parint,    blt_noop,  0 );
  io_err := ip;
  ip^.ide_int := 0;

  { Define the End of file variable }
  PREDEFINE(  '$sys_eof', ide_parint,    blt_noop,  0 );
  io_eof := ip;
  ip^.ide_int := 0;

  { Define the End of line variable }
  PREDEFINE(  '$sys_eoln', ide_parint,    blt_noop,  0 );
  io_eoln := ip;
  ip^.ide_int := 0;

  { Define the wait_info integer variable }
  PREDEFINE(  '$sys_winfo', ide_parint,    blt_noop,  0 );
  io_winfo := ip;
  ip^.ide_int := 0;

  { Define the Read Count variable }
  PREDEFINE(  '$sys_iocount', ide_parint,    blt_noop,  0 );
  io_count := ip;
  ip^.ide_int := 0;

  PREDEFINE(  '.UNDECLARED.', ide_parint,    blt_noop,  0 );
  udc_ident := ip;

  DEFINE_SPECIFIC_ENTRY( PREDEFINE ); { Define the specific built-in function }

  DISPLAY_NEW                    { Create the first user display }
end DEFINE_BUILTIN;




{ procedure GET_STREXPR( var st: string ); forward; /Implied: Already declared external }

[global]
procedure INSYMBOL;
const
  mdnam = 'INSY';

  ten   =   10.0;
  one   =    1.0;

type
  chartype = (nul, ctl, oth, dig, let, quo,
              db0, db1, db2, db3,
              eos, eol, eom, idi,
              s00, s01, s02, s03, s04, s05, s06, s07,
              s08, s09, s10, s11, s12, s13, s14, s15, s16);

  chartabtype = array[CHR( 0 )..CHR( 127 )] of chartype;

  chartoktab = array[s00..s15] of sym_rec;

var
  chartab: [static] chartabtype := (
         { 0    1    2    3    4    5    6    7 }
  { 000 } nul, eol, eom, ctl, eos, ctl, ctl, ctl,  {*NUL,*SOH, STX, ETX, EOT, ENQ, ACK, BEL  } { *EOT (eos) used as End of Stream }
  { 010 } ctl, ctl, ctl, ctl, ctl, ctl, ctl, ctl,  {  BS, TAB,  LF,  VT,  FF,  CR,  SO,  SI  } { *SOH (eol) used as End of Line }
  { 020 } ctl, ctl, ctl, ctl, ctl, ctl, ctl, ctl,  { DLE, DC1, DC2, DC3, DC4, NAK, SYN, ETB  } { *NUL (nul) always skipped }
  { 030 } ctl, ctl, ctl, ctl, ctl, ctl, ctl, ctl,  { CAN,  EM, SUB, ESC,  FS,  GS,  RS,  US  }
  { 040 } oth, s00, s01, let, let, let, s02, quo,  { space  !    "    #    $    %    &    '  }
  { 050 } s03, s04, s05, s06, s07, s08, db3, s09,  {   (    )    *    +    ,    -    .    /  }
  { 060 } dig, dig, dig, dig, dig, dig, dig, dig,  {   0    1    2    3    4    5    6    7  }
  { 070 } dig, dig, db0, s10, db1, s11, db2, oth,  {   8    9    :    ;    <    =    >    ?  }
  { 100 } let, let, let, let, let, let, let, let,  {   @    A    B    C    D    E    F    G  }
  { 110 } let, let, let, let, let, let, let, let,  {   H    I    J    K    L    M    N    O  }
  { 120 } let, let, let, let, let, let, let, let,  {   P    Q    R    S    T    U    V    W  }
  { 130 } let, let, let, s12, s13, s14, s15, let,  {   X    Y    Z    [    \    ]    ^    _  }
  { 140 } oth, let, let, let, let, let, let, let,  {   `    a    b    c    d    e    f    g  }
  { 150 } let, let, let, let, let, let, let, let,  {   h    i    j    k    l    m    n    o  }
  { 160 } let, let, let, let, let, let, let, let,  {   p    q    r    s    t    u    v    w  }
  { 170 } let, let, let, oth, s00, oth, oth, ctl); (*  x    y    z    {    |    }    ~  DEL *)

  chartok: [static] chartoktab := (
    (lgorop,    or_op ), {s00: '!' = '|'}
    (attrsign,  no_op ), {s01: '"'}
    (lgandop,   and_op), {s02: '&'}
    (lparen,    no_op ), {s03: '('}
    (rparen,    no_op ), {s04: ')'}
    (mulop,     mul_op), {s05: '*'}
    (addop,     add_op), {s06: '+'}
    (comma,     no_op ), {s07: ','}
    (addop,     sub_op), {s08: '-'}
    (mulop,     div_op), {s09: '/'}
    (semicolon, stp_op), {s10: ';'}
    (relop,     eq_op ), {s11: '='}
    (lbrack,    no_op ), {s12: '['}
    (notop,     not_op), {s13: '\'}
    (rbrack,    no_op ), {s14: ']'}
    (indirsign, no_op )  {s15: '^'}
    );
var
  ivl, i, iprec, j, k, n, scale, radix: integer;
  rdig, rexp, rfac, rval: shreal;
  pch: char;
  getnuchar, maxstr, found, sign, bint, bline_enabled: boolean;
  pkw: keyword_ptr;
  idp: ide_ptr;


  procedure NEXTCH;
  var
    beoln: boolean;

  begin { NEXTCH }
    if idm_actstk <> nil then
    with idm_actstk^, idm_tab^ do
    begin
      if idm_nch > idm_use then { Use of macro code }
        case idm_kind of
          idm_parm, idm_temp:
            begin  { Call return of macro param code }
              idm_run    := false;
              RESTORE_SYM_CNTX;
              if idm_kind = idm_temp then FREE_TEMP_MAC
                                     else idm_actstk := idm_cntx
            end;
        otherwise
          sy_ch := CHAR( 2 )
        end
      else
      begin
        sy_ch := idm_ctb[idm_nch];
        idm_nch := idm_nch + 1
      end;
      if debug_mac then
      with sy_maclst do
      begin { build the macro expanssion list line }
        if length >= capacity then OUT_MACRO_LINE;
        length := length + 1;
        if sy_ch >= ' ' then body[length] := sy_ch
                        else body[length] := ' '
      end
    end
    else sy_ch := SRC_INCHAR;
    if idm_outmacro then GEN_MACRO_CODE;
    sy_cmin := sy_ch;
    if (sy_ch >= 'A') and (sy_ch <= 'Z') then
      sy_cmin := CHR( ORD( sy_ch ) + in_min )
  end NEXTCH;


  function CHAR_NEXT: char;
  begin { CHAR_NEXT }
    if idm_actstk <> nil then
    with idm_actstk^, idm_tab^ do
      if idm_nch > idm_use then CHAR_NEXT := CHR( 2 )
                           else CHAR_NEXT := idm_ctb[idm_nch]
    else CHAR_NEXT := SRC_NEXT_CHAR;
  end CHAR_NEXT;


  procedure PUT_CHAR;
  begin { PUT_CHAR }
    if not maxstr then
    if k < str_maxsize then
    begin
      sy_string.body[k] := sy_ch; k := SUCC(k)
    end
    else
    begin
      SRC_ERROR( mdnam, 12, e_error );
      maxstr := true
    end;
  end PUT_CHAR;



begin {INSYMBOL}

ET_READ: { Label for continue analyse by INSYMBOL after an indirection }

  while (sy_ch = ' ') or (sy_ch = src_ch_null) do NEXTCH;
  getnuchar := true;
  with src_control^ do
    { Validate the position error pointer on interpretation position }
    if (idm_actstk <> nil) and debug_mac then src_wchpt := sy_maclst.length
                                         else src_wchpt := src_chidx;

  with sy_sym do
  case chartab[sy_ch] of
    dig, db3: { number can be begin by digit or period }
      begin
        sy := period; {assume '.' until shown otherwise}
        op := no_op;
        iprec := 0;
        rval  := 0.0; rexp := ten; rfac := one;
        while chartab[sy_ch] = dig do
        begin
          sy := intconst; { it is a number }
          rdig := ORD( sy_ch ) - ORD( '0' );
          NEXTCH;
          rval := rval*ten + rdig;
          iprec := iprec + 1
        end;
        if sy_ch = '.' then
        begin
          if CHAR_NEXT = '.' then { ".." is following }
            if sy = period then
            begin { our syntax unit }
              NEXTCH;
              sy := twodot
            end
            else getnuchar := false { for the next syntax unit }
          else
          begin { decimal period }
            NEXTCH;
            if chartab[sy_ch] <> dig then
              getnuchar := false
            else
            begin
              sy := doubleconst;
              while chartab[sy_ch] = dig do
              begin
                rdig := ORD( sy_ch ) - ORD( '0' );
                NEXTCH;
                rfac := rfac / ten;
                rval := rval + rfac*rdig;
                iprec := iprec + 1
              end
            end
          end
        end;

        if (sy <> period) and (sy <> twodot) then
        begin
          if sy_cmin = 'e' then
          begin
            sy := doubleconst;
            NEXTCH;
            if (sy_ch = '+') or (sy_ch = '-') then
            begin
              if sy_ch = '-' then rexp := one/rexp;
              NEXTCH
            end;
            ivl := 0;
            while chartab[sy_ch] = dig do
            begin
              ivl := ivl*10 + (ORD( sy_ch ) - ORD( '0' ));
              NEXTCH
            end;
            if ivl > max_dblpow then
            begin
              SRC_ERROR( mdnam, 11, e_error );
              ivl := max_dblpow; rval := one
            end
            else
              { for two large exponante we force the double precision }
              if ivl > max_fltpow then iprec := max_single + 1;
            rfac := one;
            while ivl <> 0 do
            begin
              if ODD( ivl ) then
              begin
                ivl := ivl - 1;
                rfac := rfac*rexp
              end
              else
              begin
                ivl := ivl div 2;
                rexp := SQR( rexp )
              end
            end;
            rval := rval*rfac
          end;
          if sy = intconst then
            if (rval <= unsmax) and (rval >= intmin) then
            begin { set unsigned value in integer equivalent }
              if rval > intmax then sy_ival := TRUNC( unsmax - rval ) + 1
                               else sy_ival := TRUNC( rval );
              sy_rval := sy_ival;
            end
            else
            begin
              sy_ival := 0;
              if iprec > max_single then sy := doubleconst
                                    else sy := singleconst
            end
          else
           if iprec <= max_single then sy := singleconst;

          sy_rval := rval;
          getnuchar := false
        end
      end;

    s01: { Macro replacing character '"' }
      if idm_outmacro or sy_noexec then
      begin { Put it in macro text only }
        NEXTCH;
        INSYMBOL;
        if sy = lparen then
        begin
          k := 1;
          repeat
            INSYMBOL;
            if sy = lparen then k := k + 1
                           else if sy = rparen then k := k - 1
          until (k = 0) or (sy = peofsy) or (sy = eomcsy);
          if sy <> rparen then SRC_ERROR( mdnam, 23, e_severe )
        end
        else
          if sy <> identsy then SRC_ERROR( mdnam, 315, e_severe )
      end
      else
      begin { Simule a macro parameter with a string }
        NEXTCH;                   { Gobble up '"' }
        INSYMBOL;                 { Get the string begining symbol }
        if sy = lparen then
        begin
          INSYMBOL;
          GET_STREXPR( sy_string ); { Get the Macro text String }
          if sy <> rparen then SRC_ERROR( mdnam, 23, e_severe )
        end
        else
        begin
          sy_string.length := 0;
          if sy = identsy then
          begin
            idp := IDE_SEARCH( true );
            if idp <> nil then
            with idp^ do
              if ide_kind = ide_parstr then
                if ide_str <> nil then sy_string := ide_str^
          end
        end;
        if sy_string.length > 0 then
        begin
          NEW( sy_macro );        { Build a Temporary Unnamed Macro Parameter }
          with sy_macro^ do
          begin
            idm_name   := nil;
            idm_parl   := idm_tmphde;
            idm_cntx   := nil;
            idm_nxt    := nil;
            idm_prv    := nil;
            idm_run    := false;
            idm_kind   := idm_temp;
            NEW( idm_tab, sy_string.length );
            with idm_tab^ do
            begin
              idm_use := sy_string.length;
              for ii := 1 to sy_string.length do idm_ctb[ii] := sy_string[ii];
              ACTIVE_MACRO_CODE( sy_macro );
              if debug_sym then
                WRITELN( lst_current^.lst_file, ' Temporary_Mac_Temp read at ', idm_nch:3,
                         ' "', idm_ctb:idm_use, '".' )
            end
          end;
          idm_tmphde := sy_macro; { Push it in the temporary stack }
          goto ET_READ
        end
        else
        begin
          if debug_sym then WRITELN( lst_current^.lst_file, ' Temporary_Mac_Temp = nil.');
          SRC_ERROR( mdnam, 316, e_error )
        end;
        getnuchar := false { Do'nt get the next character - it's already done }
      end;

    let: { keyword or identifier }
      with sy_ident do
      begin
        k := sy_maclst.length - 1;
        if idm_outmacro and (idm_newmac <> nil) then n := idm_newmac^.idm_use - 1;
        length := 0;
        repeat
          if length < ide_maxsize then
          begin
            { Map to lower case in keywords and identifiers }
            length := SUCC( length );
            body[length] := sy_cmin
          end;
          NEXTCH
        until (chartab[sy_ch] <> let) and (chartab[sy_ch] <> dig);
        { now search for known keyword }
        pkw := keyword_tree;

        repeat
          with pkw^ do
          begin
            i := STR_MATCH( sy_ident, name^ );
            if i <> 0 then
              if i > 0 then pkw := rightp else pkw := leftp
          end
        until (i = 0) or (pkw = nil);
        if pkw = nil then { it is an identifier or a macro identifier }
        begin
          sy_macro := SEARCH_MACRO( idm_undef );
          sy := identsy; op := no_op;
          if (sy_macro <> nil) and not sy_nomacrflg then
          begin
            { a Macro symbol is found }
            if sy_macro^.idm_kind = idm_parm then
            with sy_macro^, lst_current^ do
            begin { It is a macro formal parameter }
              { We supress the parameter name from the Macro expenssion Listing }
              if k >= 0 then sy_maclst.length := k
                        else sy_maclst.length := 0;
              if idm_outmacro then
                { We suppress it also it from the macro code to build }
                if n >= 0 then idm_newmac^.idm_use := n
                          else idm_newmac^.idm_use := 0;
              if debug_sym then
                if idm_tab <> nil then
                with idm_tab^ do
                  WRITELN( lst_file, ' Mac_Parm read at ', idm_nch:3,
                           ' "', idm_ctb:idm_use, '".' )
                else
                  WRITELN( lst_file, ' Mac_Parm = nil.');

              if idm_tab <> nil then ACTIVE_MACRO_CODE( sy_macro );
              INSYMBOL
            end
          end
        end
        else
        begin { it is a known keyword }
          sy_sym   := pkw^.symb;
          sy_endsy := pkw^.endksy
        end;
        getnuchar := false
      end;

    quo: { quote }
      begin
        op := no_op;
        { set line mode to ignore any end of line or comment mark }
        with src_control^ do
        begin
          bline_enabled := (src_linemode in src_flags);
          src_commentty := src_nocomment;
          src_flags := src_flags + [src_linemode]
        end;
        sy := stringconst;
        k := 1; maxstr := false;
        bint := true;
        while bint do
        begin
          NEXTCH;
          while chartab[sy_ch] = eol do NEXTCH; { skip the end of line }
          if chartab[sy_ch] = eos then bint := false;
          if sy_ch = '''' then
          begin
            src_control^.src_commentty := src_pascomment;
            NEXTCH;
            if sy_ch = '''' then
            begin
              src_control^.src_commentty := src_nocomment;
              PUT_CHAR
            end
            else bint := false
          end
          else PUT_CHAR
        end;
        sy_string.length := PRED(k);
        getnuchar := false;
        with src_control^ do
          if not bline_enabled then
          begin
            src_flags := src_flags - [src_linemode];
            if chartab[sy_ch] = eol then sy_ch := ' '
          end
      end;

    db0: {':' or ':='}
      begin
        NEXTCH;
        op := no_op;
        if sy_ch = '=' then sy := becomes
        else
        begin
          sy        := colon;
          getnuchar := false
        end
      end;

    db1: {'<' or '<=' or '<>'}
      begin
        NEXTCH;
        sy := relop;
        if sy_ch = '=' then op := le_op
        else
        if sy_ch = '>' then op := ne_op
        else
        begin
          op        := lt_op;
          getnuchar := false
        end
      end;

    db2: {'>' or '>='}
      begin
        NEXTCH;
        sy := relop;
        if sy_ch = '=' then op := ge_op
        else
        begin
          op        := gt_op;
          getnuchar := false
        end
      end;

    s00: { '!','|' = "logical or" or '!!','||' = "concatenation" }
      begin
        pch := sy_ch;
        NEXTCH;
        if sy_ch = pch then
        begin
          sy := addop;
          op := concat_op
        end
        else
        begin
          sy := lgorop;
          op := or_op;
          getnuchar := false
        end
      end;

    s02, s03, s04, s06, s07,
    s08, s09, s10, s12, s13, s14, s15:
      sy_sym := chartok[chartab[sy_ch]];

    s05: { '*' look at power '**' }
      begin
        NEXTCH;
        if sy_ch = '*' then
        begin
          sy := powop;
          op := pow_op
        end
        else
        begin
          sy_sym    := chartok[s05];
          getnuchar := false
        end
      end;

    s11: { '=>' = "implicate" }
      begin
        NEXTCH;
        if sy_ch = '>' then
        begin
          sy := implic;
          op := no_op
        end
        else
        begin
          sy_sym    := chartok[s11];
          getnuchar := false
        end
      end;

    eos: { end_of_file }
      begin
        sy := peofsy;
        op := stp_op
      end;

    eom: { end of macro }
      begin
        sy := eomcsy;
        op := stp_op;
        getnuchar := false;
        sy_ch := ' '
      end;

    eol: { end_of_line }
      begin
        sy := eolnsy;
        op := no_op;
        getnuchar := not data_mode;
        sy_ch := ' '
      end;

  otherwise
    sy := nothing;
    op := no_op;
    getnuchar := false;
    if sy_ch <> ' ' then SRC_ERROR( mdnam, 13, e_error );
    sy_ch := ' '
  end { case chartab };

  if debug_mac and (idm_actstk <> nil) then { We are in macro execution mode }
    if (sy_maclst.length >= 80) or (sy_sym.sy = semicolon) then OUT_MACRO_LINE;

  if debug_sym then
  with sy_sym, lst_current^ do
  begin
    WRITE( lst_file, ' D_symbol: ', sy );
    case sy of
      identsy:     WRITELN( lst_file, ' "', sy_ident, '".' );
      singleconst,
      doubleconst: WRITELN( lst_file, ' ', sy_rval );
      intconst:    WRITELN( lst_file, ' ', sy_ival );
      stringconst: WRITELN( lst_file, ' "', sy_string, '".' );
      unaop, powop, mulop, addop,
      relop, notop, lgandop, lgorop:
                   WRITELN( lst_file, ' ', op );
    otherwise
      WRITELN( lst_file )
    end
  end;
  if getnuchar then NEXTCH
end INSYMBOL;



[global]
procedure SKIP_SYMBOL( tosymbol: symbol );
const
  mdnam = 'SKPS';

var
  bif, sve: boolean;
  lsy:      symbol;

begin { SKIP_SYMBOL }
  sve := sy_noexec;
  sy_noexec := true;
  bif := (tosymbol = elsesy);
  with sy_sym, src_control^ do
  while (sy <> tosymbol) and (sy <> eofsy) and
        (sy <> peofsy) and (sy <> endsy) and
        (sy <> eomcsy) do
  begin
    INSYMBOL;
    case sy of
      ifsy:
        begin
          src_insnb := src_insnb + 1;
          SKIP_SYMBOL( elsesy );
          if sy = elsesy then SKIP_SYMBOL( endsy );
          if sy <> endsy then SRC_ERROR( mdnam, 108, e_severe );
          src_insnb := src_insnb - 1;
          sy := nothing
        end;
      beginsy, loopsy, dosy, casesy, macrosy,
      sequencesy:
        begin
          src_insnb := src_insnb + 1;
          SKIP_SYMBOL( endsy );
          src_insnb := src_insnb - 1;
          if sy <> endsy then SRC_ERROR( mdnam, 108, e_severe )
                         else sy := nothing
        end;
      lparen:
        begin
          SKIP_SYMBOL( rparen );
          if sy <> rparen then SRC_ERROR( mdnam, 23, e_severe )
                          else sy := nothing
        end;
      lbrack:
        begin
          SKIP_SYMBOL( rbrack );
          if sy <> rbrack then SRC_ERROR( mdnam, 26, e_severe )
                          else sy := nothing
        end;
      repeatsy:
        begin
          src_insnb := src_insnb + 1;
          SKIP_SYMBOL( untilsy );
          src_insnb := src_insnb - 1;
          if sy <> untilsy then SRC_ERROR( mdnam, 123, e_severe )
                           else sy := nothing
        end;
      elsesy:
        if not bif then SRC_ERROR( mdnam, 124, e_severe );
      specificsy:
        if sy_endsy <> nothing then
        begin
          lsy := sy_endsy;
          src_insnb := src_insnb + 1;
          SKIP_SYMBOL( lsy );
          src_insnb := src_insnb - 1;
          if sy <> lsy then SRC_ERROR( mdnam, 125, e_severe )
                       else sy := nothing
        end
    otherwise
    end
  end;
  sy_noexec := sve
end { SKIP_SYMBOL };






     {************************************************}
     { EXPRESSION group (expression parsing) Routines }
     {************************************************}


{ ***  Routine to get an expression value *** }

[global]
procedure USR_IDE_SUBSTITUTE( in_var src: string; var dst: string; ch: char );
var
  ip: ide_ptr;
  i, j, k, ls, ld: integer;
  c1, c2, cc: char;
  bf: boolean;
  sna: [static] string(255);

begin
  bf := false;
  ld := dst.capacity;
  ls := src.length;
  i  := 1;
  j  := 0;
  loop
  exit if i > ls;
    c1 := src[i]; i := i + 1;
    if not bf then
      if c1 = ch then
      begin { the flag character is found }
        if i > ls then c2 := ch
                  else c2 := src[i];
        if c2 <> ch then
        begin  { Active the substitution mode }
          k := 0;
          bf := true
        end
        else
        begin { ch is managed as a normal character }
          i := i + 1;
          if j < ld then begin j := j + 1; dst[j] := c1 end
        end
      end
      else
      begin { Normal character }
        if j < ld then begin j := j + 1; dst[j] := c1 end
      end
    else
    begin { Get identifier name mode }
      if c1 = ch then
      begin
        bf := false;
        sna.length := k;
        ip := USR_IDE_LOCATE( sna );
        if ip <> nil then
        with ip^ do
        begin
          case ide_kind of
            ide_parstr: if ide_str <> nil then sna := ide_str^
                                          else sna.length := 0;
            ide_parint: WRITEV( sna, ide_int:0 );
            ide_parflt: WRITEV( sna, ide_flt );
          otherwise
          end;
          for k := 1 to sna.length do
            if j < ld then begin j := j + 1; dst[j] := sna[k] end
        end
      end
      else
      begin
        { All identifier name must be in minor case }
        if (c1 >= 'A') and (c1 <= 'Z') then c1 := CHR( ORD( c1 ) + in_min );
        if k < sna.capacity then begin k := k + 1; sna[k] := c1 end
      end
    end
  end;
  dst.length := j
end USR_IDE_SUBSTITUTE;


[global]
procedure USR_S_ELEMENT(    var dst: string;  { The extracted element string }
                         in_var src: string;  { The string }
                                iel: integer; { The element number starting from 0 }
                                csp,          { The separator character }
                                cst: char;    { The string character (or space) }
                                bss: boolean := false ); { the string expanding flag }
var
  b_s, e_s, b_e, e_e, i, j, ls, ld: integer;
  cc, cf: char;
  bst, bel: boolean;

begin
  ls  := src.length;
  ld  := dst.capacity;
  bel := false;
  bst := false;
  i   := 1;
  b_e := 0; b_s := 0;
  e_e := 0; e_s := 0;
  if csp < ' ' then csp := ' ';
  if cst < ' ' then cst := ' ';
  if cst = csp then cst := ' ';

  while (i <= ls) and (iel >= 0) do
  begin
    cc := src[i]; if cc < ' ' then cc := ' ';
    if cst > ' ' then { The string separator is defined }
      if bst then
      begin { We are in a string }
        if cc = cst then
        begin { We have a string initiator character }
          if i <= ls then cf := src[i+1]
                     else cf := cst;
          if cc = cf then i := i + 1 { We skip the string initiator inside the string }
                     else begin  bst := false; e_s := i  end { End of string }
        end
      end
      else { We are not in a string but we can enter in a string now }
        if cc = cst then { we have find a string begining }
        begin
          if not bel then begin  bel := true; b_e := i  end;
          b_s := i; bst := true
        end;

    if not bst then { we are not in a string }
      if bel then
      begin { We are in an element }
        if cc = csp then
        begin  bel := false; e_e := i - 1; iel := iel - 1  end
      end
      else
      begin { We are not in an element but we can enter in now }
        if cc <> csp then { We have find the begin of an element }
        begin  bel := true; b_e := i  end
        else
        begin { we have an empty element when the separator is not a space }
          if (csp <> ' ') and (cc = csp) then
          begin  b_e := i; e_e := i - 1; iel := iel - 1  end
        end
      end;
    i := i + 1;
  end;
  if bel then { Unterminated element on end of string }
  begin  e_e := ls; bel := false; iel := iel - 1  end;

  j := 0;
  if iel < 0 then
  begin { We have found the desired element }
    { Now we can load the element in the destination string }
    bst := false;
    if bss then
      for k := b_e to e_e do
      begin
        if j < ld then begin  j := j + 1; dst[j] := src[k]  end
      end
    else
    begin
      bel := false;
      for k := b_e to e_e do
      begin
        cc := src[k];
        if (cc = cst) and (cst <> ' ') then
          if bst then
          begin  bel := true; bst := false  end
          else
          begin
            if bel then
            begin
              bel := false;
              if j < ld then begin  j := j + 1; dst[j] := cc  end
            end;
            bst := true
          end
        else
        begin
          bel := false;
          if j < ld then begin  j := j + 1; dst[j] := cc  end
        end
      end
    end
  end;
  dst.length := j
end USR_S_ELEMENT;




[global]
procedure USR_SUPPRESS_COMMENT( in_var src: string; var dst: string; cc, cs: char );
var
  bst: boolean;
  ii, jj, sz: integer;
  ch: char;

begin
  if cc <= ' ' then cc := '!';
  if cs = cc then cs := '"';
  if cs <= ' ' then
  begin
    ii := INDEX( src, cc );
    if ii = 0 then dst := src
              else dst := SUBSTR( src, 1, ii - 1 )
  end
  else
  begin
    bst := false;
    sz  := dst.capacity;
    if sz > src.length then sz := src.length;
    ii  := 0; jj := 0;
    while (ii < sz) do
    begin
      ii := ii + 1;
      ch := src[ii];
      if ch = cs then bst := not bst;
    exit if (not bst) and (ch = cc);
      jj := jj + 1;
      dst[jj] := ch;
    end;
    dst.length := jj
  end
end USR_SUPPRESS_COMMENT;



[global]
function  USR_NUMERIC_STRING( in_var str: string ): boolean;
type
  mdtyp = ( mdspace, mdsign, mdent, mdfrac, mdsexp, mdexp );

var
  ip: integer;
  br: boolean;
  md:   mdtyp;
  ch:    char;

begin
  md := mdspace;
  ip := 1;
  while (md <= mdspace) and (ip <= str.length) do
  begin
    ch := str[ip];
    case ch of
      SOH..' ':
        exit if md > mdspace; { stop on space }

      '+', '-':
        case md of
          mdspace: md := mdsign;
          mdsexp:  md := mdexp;
        otherwise
          exit
        end;

      '0'..'9':
        case md of
          mdspace, mdsign, mdent:
            if md < mdent then md := mdent;
          mdsexp, mdexp:
            if md < mdexp then md := mdexp;
        otherwise
        end;

      '.':
        if md < mdfrac then md := mdfrac
                       else exit;

      'e', 'E', 'd', 'D':
        if md < mdsexp then md := mdsexp
                       else exit;

    otherwise
      exit if md > mdspace;
    end;
    ip := ip + 1                        { Skip to next character }
  end { while };
  case md of
    mdent, mdfrac, mdexp:
      USR_NUMERIC_STRING := true;
  otherwise
    USR_NUMERIC_STRING := false
  end
end USR_NUMERIC_STRING;




{ Special Definitions for PATH Variable Access (too long for normal string) }
{ Used to replace a long and dynamic string management }

function  USR_GETPATH( sp: char; in_var st: string; var tbrf: ide_arrstr ): integer;
const
  mxseptb = 256;

type
  buffer_typ  = packed array[short_unsigned] of char;

  buf_ptr = ^buffer_typ;

var
  bf: buf_ptr;
  ie, ip, ist, isz, len: integer;
  ch: char;
  tp: array[1..mxseptb] of byte;
  sb: string( 255 );
  bsov: boolean;

  function  GET_ENV_ARRAY( var ptr: buf_ptr; in_var src: string ): integer;
  external 'PAS__GET_ENV_ARRAY';


begin { USR_GETPATH }
  isz := GET_ENV_ARRAY( bf, st );
  if isz > 0 then
  with tbrf do
  begin
    ist := 0;
    ip  := 0;
    len := 0;
    bsov := false;
    while (ip < isz) and (ist < tbrf.ide_all) do
    begin
      ch := bf^[ip];
      len := len + 1;
      ip  := ip  + 1;
      sb[len] := ch;
      if (ch = sp) or (len = sb.capacity) or (ip = isz) then
      begin
        sb.length := len;
        ist := ist + 1;
        if ide_stb[ist] <> nil then DISPOSE( ide_stb[ist] );
        NEW( ide_stb[ist], sb.length );
        ide_stb[ist]^ := sb;
        sb.length := 0; len := 0;
        if (ch <> sp) and (ip <> isz) then bsov := true
      end
    end;
    if (ist = ide_all) and (ip < isz) then ist := ide_all + 1
    else
      for ii := ist + 1 to ide_all do
        if ide_stb[ii] <> nil then DISPOSE( ide_stb[ii] )
  end
  else ist := -1;
  if bsov then ist := -ist;
  USR_GETPATH := ist
end USR_GETPATH;



function  USR_SETPATH( in_var st: string; i: integer; var tbrf: ide_arrstr ): integer;
const
  mxsz = 16384;

type
  chtb = array[1..mxsz] of char;

var
  bf: ^chtb;
  ie, ip, isz: integer;

  function  SET_ENV_ARRAY( in_var log: string;
                              var val: array[dim: integer] of char; sz, ovr: integer ): integer;
  external 'PAS__SET_ENV_ARRAY';

begin
  NEW( bf );
  with tbrf do
  begin
    ip := 0;
    for ii := 1 to ide_all do
    begin
      if ide_stb[ii] <> nil then
        with ide_stb[ii]^ do
          for jj := 1 to length do
            if ip < mxsz then
            begin  ip := ip + 1; bf^[ip] := body[jj]  end
    end;
    if ip < mxsz then
    begin
    { ip := ip + 1; bf^[ip] := CHR( 0 );       { Append a null character to the end of new path }
      ie := SET_ENV_ARRAY( st, bf^, ip, 1 )    { Always allow to supershed }
    end
    else ie := -99
  end;
  DISPOSE( bf );
  USR_SETPATH := ie;
end USR_SETPATH;



procedure EXPRESSION; forward;


procedure POP_EXPRESSION;
{ Routine to get an expression, the result is located in the record exp_res }
const
  mdnam = 'PEXP';

begin
  if exp_stkp > 0 then
  begin { *** Get the expression result *** }
    exp_res  := exp_stk[exp_stkp];
    exp_stkp := exp_stkp - 1
  end
  else
  begin
    SRC_ERROR( mdnam, 7, e_severe );
    exp_res.exp_kind := exp_valnull
  end;
end POP_EXPRESSION;


[global]
procedure GET_EXPRESSION;
{ Routine to get an expression, the result is located in the record exp_res }
const
  mdnam = 'GEXP';

var
  sp: integer;

begin
  sp := exp_stkp;            { Save the stack pointer }
  EXPRESSION;                { Manage the current expression }
  POP_EXPRESSION;            { Pop the resulting expression from the stack }
  if sp <> exp_stkp then
  begin
    SRC_ERROR( mdnam, 9, e_error );
    exp_stkp := sp           { Restore the stack pointer }
  end
end GET_EXPRESSION;



[global]
function EXP_GETKINDS( ish: integer ): exp_kinds;
const
  mdnam = 'EXGK';

var
  kn: exp_kinds;

begin
  kn  := exp_valnull;
  ish := exp_stkp - ish;
  if ish > 0 then kn := exp_stk[ish].exp_kind
             else SRC_ERROR( mdnam, 8, e_severe );
  EXP_GETKINDS := kn
end EXP_GETKINDS;


{ ***  Routines to get an expression value of a predefined type *** }

procedure CHECK_LVALUE_REF;
begin
  if debug_exp then
  with lst_current^, exp_res do
  begin
    WRITE( lst_file, ' Lvalue POP : ' );
    if exp_ref = nil then WRITE( lst_file, '<Not a Identifier Ref.>' )
    else
      with exp_ref^ do
        if ide_name <> nil then WRITE( lst_file, '"',ide_name, '"' )
                           else WRITE( lst_file, '<Identifier Without Name>' );
    WRITELN( lst_file )
  end;
  if exp_res.exp_ref = nil then SRC_ERROR( 'EXRF', 111, e_severe )
end CHECK_LVALUE_REF;



[global]
procedure POP_EXP_REFER( var rec: exp_rec );
begin
  POP_EXPRESSION;
  CHECK_LVALUE_REF;
  rec := exp_res
end POP_EXP_REFER;


[global]
procedure GET_EXP_REFER( var rec: exp_rec );
begin
  GET_EXPRESSION;
  CHECK_LVALUE_REF;
  rec := exp_res
end GET_EXP_REFER;


[global]
procedure POP_EXP_VALUE( var knd: exp_kinds; var iv: integer;
                                             var rv: shreal;
                                             var st: string );
{ To pop a number value }
const
  mdnam = 'EXVL';

begin
  POP_EXPRESSION;
  with exp_res do
  case exp_kind of
    exp_telstr,
    exp_valstr: begin
                  if exp_kind = exp_telstr then
                  begin
                    if exp_aas <> nil then
                      exp_str := exp_aas^.ide_stb[exp_shift]
                    else
                      exp_str := nil
                  end;
                  knd := exp_valstr;
                  if exp_str <> nil then
                  begin
                    st := exp_str^;
                    if exp_ref = nil then
                    begin { Free any temporary string }
                      DISPOSE( exp_str ); exp_str := nil
                    end
                  end else st.length := 0
                end;

    exp_telint,
    exp_valint: begin
                  if exp_kind = exp_telint then
                    if exp_aai <> nil then
                      exp_int := exp_aai^.ide_itb[exp_shift]
                    else
                      exp_int := 0;
                  knd := exp_valint;
                  iv  := exp_int
                end;

    exp_telflt,
    exp_valflt: begin
                  if exp_kind = exp_telflt then
                    if exp_aaf <> nil then
                      exp_flt := exp_aaf^.ide_ftb[exp_shift]
                    else
                      exp_flt := 0.0;
                  knd := exp_valflt;
                  rv  := exp_flt
                end;

  otherwise
    { Illegal array use or null expression }
    if exp_kind <> exp_valnull then SRC_ERROR( mdnam, 112, e_severe );
    knd := exp_valnull
  end;

  if debug_exp then
  with lst_current^ do
  begin
    WRITE( lst_file, ' POP : ', ORD( knd ) );
    case knd of
      exp_valint:      WRITE( lst_file, iv );
      exp_valflt:      WRITE( lst_file, rv );
      exp_valstr:      WRITE( lst_file, '"', st, '"' );
    otherwise
    end;
    WRITELN( lst_file )
  end
end POP_EXP_VALUE;



[global]
procedure GET_EXP_VALUE( var knd: exp_kinds; var iv: integer;
                                             var rv: shreal;
                                             var st: string );
{ To get a value }
const
  mdnam = 'EXVL';

var
  sp: integer;

begin
  sp := exp_stkp;            { Save the stack pointer }
  EXPRESSION;                { Manage the current expression }
  POP_EXP_VALUE( knd, iv, rv, st );
  if sp <> exp_stkp then
  begin
    SRC_ERROR( mdnam, 9, e_error );
    exp_stkp := sp           { Restore the stack pointer }
  end
end GET_EXP_VALUE;




[global]
procedure POP_NUMEXPR( var bflt: boolean;
                       var   iv: integer;
                       var   rv:  shreal );
{ To pop a number value }
var
  kn: exp_kinds;
  st: str_string;

begin
  POP_EXP_VALUE( kn, iv, rv, st );
  case kn of
    exp_valstr: begin
                  bflt := false;
                  if st.length > 0 then READV( st, iv )
                                   else iv := 0
                end;
    exp_valint: bflt := false;
    exp_valflt: bflt := true;
  otherwise
  end
end POP_NUMEXPR;



[global]
procedure GET_NUMEXPR( var bflt: boolean;
                       var   iv: integer;
                       var   rv:  shreal );
{ To get a number value }
var
  kn: exp_kinds;
  st: str_string;

begin
  GET_EXP_VALUE( kn, iv, rv, st );
  case kn of
    exp_valstr: begin
                  bflt := false;
                  if st.length > 0 then READV( st, iv )
                                   else iv := 0
                end;
    exp_valint: bflt := false;
    exp_valflt: bflt := true;
  otherwise
  end
end GET_NUMEXPR;



[global]
function POP_INTEXPR( iv: integer ): integer;
{ To pop an integer value }
var
  kn: exp_kinds;
  st: str_string;
  rv: shreal;

begin
  POP_EXP_VALUE( kn, iv, rv, st );
  case kn of
    exp_valstr: if st.length > 0 then READV( st, iv )
                                 else iv := 0;
    exp_valflt: iv := ROUND( rv );
    exp_valint: ;
  otherwise
  end;
  POP_INTEXPR := iv
end POP_INTEXPR;



[global]
function GET_INTEXPR( iv: integer ): integer;
{ To get an integer value }
var
  kn: exp_kinds;
  st: str_string;
  rv: shreal;

begin
  GET_EXP_VALUE( kn, iv, rv, st );
  case kn of
    exp_valstr: if st.length > 0 then READV( st, iv )
                                 else iv := 0;
    exp_valflt: iv := ROUND( rv );
    exp_valint: ;
  otherwise
  end;
  GET_INTEXPR := iv
end GET_INTEXPR;



[global]
function POP_FLTEXPR( rv: shreal ): shreal;
{ To pop a floatting value }
var
  kn: exp_kinds;
  st: str_string;
  iv: integer;

begin
  POP_EXP_VALUE( kn, iv, rv, st );
  case kn of
    exp_valstr: if st.length > 0 then READV( st, rv )
                                 else rv := 0.0;
    exp_valflt: ;
    exp_valint: rv := iv;
  otherwise
  end;
  POP_FLTEXPR := rv
end POP_FLTEXPR;




[global]
function GET_FLTEXPR( rv: shreal ): shreal;
{ To get a floatting value }
var
  kn: exp_kinds;
  st: str_string;
  iv: integer;

begin
  GET_EXP_VALUE( kn, iv, rv, st );
  case kn of
    exp_valstr: if st.length > 0 then READV( st, rv )
                                 else rv := 0.0;
    exp_valflt: ;
    exp_valint: rv := iv;
  otherwise
  end;
  GET_FLTEXPR := rv
end GET_FLTEXPR;



[global]
procedure POP_STREXPR( var st: string );
{ To pop a string value }
var
  kn: exp_kinds;
  iv: integer;
  rv: shreal;

begin
  POP_EXP_VALUE( kn, iv, rv, st );
  case kn of
    exp_valstr: ;
    exp_valflt: WRITEV( st, rv );
    exp_valint: WRITEV( st, iv );
  otherwise
  end
end POP_STREXPR;


[global]
procedure GET_STREXPR( var st: string );
{ To get a string value }
var
  kn: exp_kinds;
  iv: integer;
  rv: shreal;

begin
  GET_EXP_VALUE( kn, iv, rv, st );
  case kn of
    exp_valstr: ;
    exp_valflt: WRITEV( st, rv );
    exp_valint: WRITEV( st, iv );
  otherwise
  end
end GET_STREXPR;




procedure EXP_PUTVALUE(       knd: exp_kinds;
                               iv: integer;
                               rv: shreal;
                               st: pstring );
const
  mdnam = 'EXPP';

begin
  if exp_stkp < max_stk then exp_stkp := SUCC( exp_stkp )
                        else SRC_ERROR( mdnam, 6, e_severe );
  with sy_sym, exp_stk[exp_stkp] do
  begin
    exp_ref   := nil;
    exp_adm   := nil;
    exp_edim  := 1;
    exp_shift := 0;
    exp_kind  := knd;
    case knd of
      exp_valint: exp_int := iv;
      exp_valflt: exp_flt := rv;
      exp_valstr: if st <> nil then
                    if st^.length > 0 then
                    begin
                      NEW( exp_str, st^.length );
                      exp_str^ := st^
                    end else exp_str := nil
                  else exp_str := nil;
    otherwise
    end
  end;
  if debug_exp then
  with lst_current^ do
  begin
    WRITE( lst_file, ' PUSH : ', ORD( knd ) );
    case knd of
      exp_valint:      WRITE( lst_file, iv );
      exp_valflt:      WRITE( lst_file, rv );
      exp_valstr:      WRITE( lst_file, '"', st, '"' );
    otherwise
    end;
    WRITELN( lst_file )
  end
end EXP_PUTVALUE;


[global]
procedure EXP_PUTINT( iv: integer );
begin
  EXP_PUTVALUE( exp_valint, iv, 0.0, nil )
end EXP_PUTINT;


[global]
procedure EXP_PUTFLT( rv: shreal );
begin
  EXP_PUTVALUE( exp_valflt, 0, rv, nil )
end EXP_PUTFLT;


[global]
procedure EXP_PUTSTR( in_var st: string );
begin
  EXP_PUTVALUE( exp_valstr, 0, 0.0, st"address )
end EXP_PUTSTR;



procedure ARRSCA_SS_OPE( op: arr_ope; var src, dst: exp_rec );
var
  idst: integer;

begin
  with dst do
  begin
    idst := exp_shift;
    if exp_aas <> nil then
    with exp_aas^ do
    { case op of }
      { ar_mov: }
              for ii := 1 to dst.exp_edim do
              begin
                if ide_stb[idst] <> nil then DISPOSE( ide_stb[idst] );
                if src.exp_str <> nil then
                  if src.exp_ref = nil then
                    { Temporary string begin permanent }
                    ide_stb[idst] := src.exp_str
                  else
                  begin
                    { Create a string copy }
                    NEW( ide_stb[idst], src.exp_str^.length );
                    ide_stb[idst]^ := src.exp_str^
                  end
                else ide_stb[idst] := nil;
                idst := idst + 1
              end;
    { end }
  end
end ARRSCA_SS_OPE;


procedure ARRSCA_II_OPE( op: arr_ope; vsrc: integer; var dst: exp_rec );
var
  idst: integer;

begin
  with dst do
  begin
    idst := exp_shift;
    if exp_aai <> nil then
    with exp_aai^ do
    case op of
      ar_mov: for ii := 1 to exp_edim do
              begin
                ide_itb[idst] := vsrc; idst := idst + 1
              end;
      ar_add: for ii := 1 to exp_edim do
              begin
                ide_itb[idst] := ide_itb[idst] + vsrc; idst := idst + 1
              end;
      ar_sub: for ii := 1 to exp_edim do
              begin
                ide_itb[idst] := ide_itb[idst] - vsrc; idst := idst + 1
              end;
      ar_mul: for ii := 1 to exp_edim do
              begin
                ide_itb[idst] := ide_itb[idst] * vsrc; idst := idst + 1
              end;
      ar_div: for ii := 1 to exp_edim do
              begin
                ide_itb[idst] := ide_itb[idst] div vsrc; idst := idst + 1
              end;
    otherwise
    end
  end
end ARRSCA_II_OPE;


procedure ARRSCA_FF_OPE( op: arr_ope; vsrc: shreal; var dst: exp_rec );
var
  idst: integer;

begin
  with dst do
  begin
    idst := exp_shift;
    if exp_aaf <> nil then
    with exp_aaf^ do
    case op of
      ar_mov: for ii := 1 to exp_edim do
              begin
                ide_ftb[idst] := vsrc; idst := idst + 1
              end;
      ar_add: for ii := 1 to exp_edim do
              begin
                ide_ftb[idst] := ide_ftb[idst] + vsrc; idst := idst + 1
              end;
      ar_sub: for ii := 1 to exp_edim do
              begin
                ide_ftb[idst] := ide_ftb[idst] - vsrc; idst := idst + 1
              end;
      ar_mul: for ii := 1 to exp_edim do
              begin
                ide_ftb[idst] := ide_ftb[idst] * vsrc; idst := idst + 1
              end;
      ar_div: for ii := 1 to exp_edim do
              begin
                ide_ftb[idst] := ide_ftb[idst] / vsrc; idst := idst + 1
              end;
    otherwise
    end
  end
end ARRSCA_FF_OPE;


procedure ARRAY_DEF_MATCH( src, dst: exp_rec;
                           var siz: integer; var flg: boolean );
const
  mdnam = 'ARMA';

var
  p1, p2: idedim_ptr;
  ok: boolean;

begin
  ok := true;
  p1 := src.exp_adm;
  p2 := dst.exp_adm;
  if src.exp_edim = dst.exp_edim then
    while (p1 <> nil) and (p2 <> nil) and ok do
    begin
      ok := (p1^.idedim_stp = p2^.idedim_stp) and
            (p1^.idedim_siz = p2^.idedim_siz);
      p1 := p1^.idedim_nxt;
      p2 := p2^.idedim_nxt
    end
  else ok := false;
  flg := ok and (p1 = p2);
  if not flg then SRC_ERROR( mdnam, 78, e_severe );
  siz := dst.exp_edim
end ARRAY_DEF_MATCH;


procedure ARRAY_SS_OPE( op: arr_ope; var src, dst: exp_rec );
var
  size, isrc, idst: integer;
  psrc:             ^ide_arrstr;
  bok:              boolean;

begin
  ARRAY_DEF_MATCH( src, dst, size, bok );
  if bok and (src.exp_aas <> nil) and (dst.exp_aas <> nil) then
  with dst.exp_aas^, src do
  begin
    psrc := exp_aas;
    isrc := exp_shift;
    idst := dst.exp_shift;
    { case op of }
      { ar_mov: }
         for ii := 1 to size do
         begin
           if ide_stb[isrc] <> nil then DISPOSE( ide_stb[idst] );
           if psrc^.ide_stb[isrc] <> nil then
           begin
             NEW( ide_stb[idst], psrc^.ide_stb[isrc]^.length );
             ide_stb[idst]^ := psrc^.ide_stb[isrc]^
           end
           else ide_stb[idst] := nil;
           idst := idst + 1; isrc := isrc + 1
         end;
    { end }
  end
end ARRAY_SS_OPE;


procedure ARRAY_II_OPE( op: arr_ope; var src, dst: exp_rec );
var
  size, isrc, idst: integer;
  psrc:             ^ide_arrint;
  bok:              boolean;

begin
  ARRAY_DEF_MATCH( src, dst, size, bok );
  if bok and (src.exp_aai <> nil) and (dst.exp_aai <> nil) then
  with dst.exp_aai^, src do
  begin
    psrc := exp_aai;
    isrc := exp_shift;
    idst := dst.exp_shift;
    case op of
      ar_mov: for ii := 1 to size do
              begin
                ide_itb[idst] := psrc^.ide_itb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_add: for ii := 1 to size do
              begin
                ide_itb[idst] := ide_itb[idst] + psrc^.ide_itb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_sub: for ii := 1 to size do
              begin
                ide_itb[idst] := ide_itb[idst] - psrc^.ide_itb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_mul: for ii := 1 to size do
              begin
                ide_itb[idst] := ide_itb[idst] * psrc^.ide_itb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_div: for ii := 1 to size do
              begin
                ide_itb[idst] := ide_itb[idst] div psrc^.ide_itb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
    otherwise
    end
  end
end ARRAY_II_OPE;


procedure ARRAY_IF_OPE( op: arr_ope; var src, dst: exp_rec );
var
  size, isrc, idst: integer;
  psrc:             ^ide_arrflt;
  bok:              boolean;

begin
  ARRAY_DEF_MATCH( src, dst, size, bok );
  if bok and (src.exp_aaf <> nil) and (dst.exp_aai <> nil) then
  with dst.exp_aai^, src do
  begin
    psrc := exp_aaf;
    isrc := exp_shift;
    idst := dst.exp_shift;
    case op of
      ar_mov: for ii := 1 to size do
              begin
                ide_itb[idst] := ROUND( psrc^.ide_ftb[isrc] );
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_add: for ii := 1 to size do
              begin
                ide_itb[idst] := ide_itb[idst] + ROUND( psrc^.ide_ftb[isrc] );
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_sub: for ii := 1 to size do
              begin
                ide_itb[idst] := ide_itb[idst] - ROUND( psrc^.ide_ftb[isrc] );
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_mul: for ii := 1 to size do
              begin
                ide_itb[idst] := ide_itb[idst] * ROUND( psrc^.ide_ftb[isrc] );
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_div: for ii := 1 to size do
              begin
                ide_itb[idst] := ide_itb[idst] div ROUND( psrc^.ide_ftb[isrc] );
                idst := idst + 1; isrc := isrc + 1
              end;
    otherwise
    end
  end
end ARRAY_IF_OPE;


procedure ARRAY_FI_OPE( op: arr_ope; var src, dst: exp_rec );
var
  size, isrc, idst: integer;
  psrc:             ^ide_arrint;
  bok:              boolean;

begin
  ARRAY_DEF_MATCH( src, dst, size, bok );
  if bok and (src.exp_aai <> nil) and (dst.exp_aaf <> nil) then
  with dst.exp_aaf^, src do
  begin
    psrc := exp_aai;
    isrc := exp_shift;
    idst := dst.exp_shift;
    case op of
      ar_mov: for ii := 1 to size do
              begin
                ide_ftb[idst] := psrc^.ide_itb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_add: for ii := 1 to size do
              begin
                ide_ftb[idst] := ide_ftb[idst] + psrc^.ide_itb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_sub: for ii := 1 to size do
              begin
                ide_ftb[idst] := ide_ftb[idst] - psrc^.ide_itb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_mul: for ii := 1 to size do
              begin
                ide_ftb[idst] := ide_ftb[idst] * psrc^.ide_itb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_div: for ii := 1 to size do
              begin
                ide_ftb[idst] := ide_ftb[idst] / psrc^.ide_itb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
    otherwise
    end
  end
end ARRAY_FI_OPE;



procedure ARRAY_FF_OPE( op: arr_ope; var src, dst: exp_rec );
var
  size, isrc, idst: integer;
  psrc:             ^ide_arrflt;
  bok:              boolean;

begin
  ARRAY_DEF_MATCH( src, dst, size, bok );
  if bok and (src.exp_aaf <> nil) and (dst.exp_aaf <> nil) then
  with dst.exp_aaf^, src do
  begin
    psrc := exp_aaf;
    isrc := exp_shift;
    idst := dst.exp_shift;
    case op of
      ar_mov: for ii := 1 to size do
              begin
                ide_ftb[idst] := psrc^.ide_ftb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_add: for ii := 1 to size do
              begin
                ide_ftb[idst] := ide_ftb[idst] + psrc^.ide_ftb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_sub: for ii := 1 to size do
              begin
                ide_ftb[idst] := ide_ftb[idst] - psrc^.ide_ftb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_mul: for ii := 1 to size do
              begin
                ide_ftb[idst] := ide_ftb[idst] * psrc^.ide_ftb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
      ar_div: for ii := 1 to size do
              begin
                ide_ftb[idst] := ide_ftb[idst] / psrc^.ide_ftb[isrc];
                idst := idst + 1; isrc := isrc + 1
              end;
    otherwise
    end
  end
end ARRAY_FF_OPE;



procedure CREATE_NEW_ARR_DEF( var pn: idedim_ptr; var  sz:     integer;
                                  po: idedim_ptr;      np, nc: integer );
{ procedure to create array dimension structures form
  the np parameters in expression stack and copy the old array structure
  for the undefined dimension }
const
  mdnam = 'ARDF';

var
  pl, pc: idedim_ptr;

begin
  if po <> nil then
  begin
    NEW( pn );                        { Create the new array definition block }
    with pn^ do
    begin
      idedim_nxt := nil;
      idedim_siz := po^.idedim_siz;   { set the old size  as default }
      if po^.idedim_nxt <> nil then   { array of array }
      begin
        CREATE_NEW_ARR_DEF( idedim_nxt, idedim_stp, po^.idedim_nxt, np, nc+1 );
        if np >= nc then              { if a dimension is given }
        begin
          idedim_siz := POP_INTEXPR( 0 ); { Get it }
          if idedim_siz <= 0 then     { and if not > 0 => error and set at 1 }
          begin
            SRC_ERROR( mdnam, 75, e_severe ); idedim_siz := 1
          end
        end
      end
      else
      begin                           { array of scalar or top sub-array }
        idedim_nxt := nil;
        idedim_stp := 1;
        if np > nc then
        begin { ** Too many specified dimension error ** }
          SRC_ERROR( mdnam, 114, e_severe );
          while np > nc do
          begin { Take off the overflow of parameters }
            exp_stkp := exp_stkp - 1;
            np := np - 1
          end
        end;
        if np = nc then               { All dimension was specified }
        begin                         { get the specified dimension }
          idedim_siz := POP_INTEXPR( 0 ); { Get it }
          if idedim_siz <= 0 then     { and if not > 0 => error and set at 1 }
          begin
            SRC_ERROR( mdnam, 75, e_severe ); idedim_siz := 1
          end
        end
      end;
      sz := idedim_stp*idedim_siz     { Return the step of outer dimension }
    end
  end
end CREATE_NEW_ARR_DEF;



procedure EMODIF_ARRAY( np: integer );
const
  mdnam = 'ARMD';

type
  e_r = record case integer of
    1:( ps: ^ide_arrstr );
    2:( pi: ^ide_arrint );
    3:( pf: ^ide_arrflt )
  end;

var
  dat_new: [static]      e_r;
  size, iold, inew, sps: integer;
  pold, pnew:            idedim_ptr;
  ip:                    ide_ptr;


  procedure COPY_ELEM( knd: ide_kinds; pn, po: idedim_ptr );
  var
    sz: integer;
  begin
    with pn^, ip^, dat_new do
    begin
      if po^.idedim_siz > idedim_siz then sz := idedim_siz
                                     else sz := po^.idedim_siz;
      if idedim_nxt = nil then
        { For the array of scalar (or last sub-array) }
        { Copy all the common element and init the new others }
        case knd of
          ide_tabstr: with ps^ do
                      begin
                        for ii := 1 to sz do
                        begin
                          iold := iold + 1; inew := inew + 1;
                          ide_stb[inew] := ide_aas^.ide_stb[iold]
                        end;
                        for ii := sz + 1 to idedim_siz do
                        begin
                          inew := inew + 1;
                          ide_stb[inew] := nil
                        end
                      end;
          ide_tabint: with pi^ do
                      begin
                        for ii := 1 to sz do
                        begin
                          iold := iold + 1; inew := inew + 1;
                          ide_itb[inew] := ide_aai^.ide_itb[iold]
                        end;
                        for ii := sz + 1 to idedim_siz do
                        begin
                          inew := inew + 1;
                          ide_itb[inew] := 0
                        end
                      end;
          ide_tabflt: with pf^ do
                      begin
                        for ii := 1 to sz do
                        begin
                          iold := iold + 1; inew := inew + 1;
                          ide_ftb[inew] := ide_aaf^.ide_ftb[iold]
                        end;
                        for ii := sz + 1 to idedim_siz do
                        begin
                          inew := inew + 1;
                          ide_ftb[inew] := 0.0
                        end
                      end;
        otherwise
        end { case knd of }
      else
      begin { For an array of array }
        { For the previously existing sub-array elements }
        for ii := 1 to sz do  COPY_ELEM( knd, idedim_nxt, po^.idedim_nxt );
        { For the new sub sub-array elements }
        for ii := sz + 1 to idedim_siz do
        case knd of
          ide_tabstr: with ps^ do
                        for jj := 1 to idedim_stp do
                        begin
                          inew := inew + 1;
                          ide_stb[inew] := nil
                        end;
          ide_tabint: with pi^ do
                        for jj := 1 to idedim_stp do
                        begin
                          inew := inew + 1;
                          ide_itb[inew] := 0
                        end;
          ide_tabflt: with pf^ do
                        for jj := 1 to idedim_stp do
                        begin
                          inew := inew + 1;
                          ide_ftb[inew] := 0.0
                        end;
        otherwise
        end
      end
    end
  end COPY_ELEM;


begin { EMODIF_ARRAY }
  ip := nil;
  sps := exp_stkp - np;
  if exp_stkp < np then
    SRC_ERROR( mdnam, 7, e_fatal ) { expression stack underflow }
  else
  with exp_stk[sps + 1] do
  begin
    if exp_ref = nil then
      SRC_ERROR( mdnam, 111, e_severe )  { not a space reference error }
    else
    case exp_kind of
      exp_tabstr,
      exp_tabint,
      exp_tabflt: ip := exp_ref;
    otherwise { Not an array error }
      SRC_ERROR( mdnam, 113, e_severe )
    end
  end;
  if ip <> nil then
  with ip^ do
  if ide_lock then SRC_ERROR( mdnam, 79, e_error )
  else
  begin
    CREATE_NEW_ARR_DEF( pnew, size, ide_adm, np - 1, 1 );
    if pnew <> nil then
    with dat_new, pnew^ do
    begin
      { allocate the new data space }
      size := idedim_stp*idedim_siz;
      case ide_kind of
        ide_tabstr: NEW( ps, size );
        ide_tabint: NEW( pi, size );
        ide_tabflt: NEW( pf, size );
      otherwise
      end;
      { Initialize the data indexes }
      iold := 0; inew := 0;
      { Copy the already existing element and initialize the new ones }
      COPY_ELEM( ide_kind, pnew, ide_adm );
      { free the old data space and set the new one }
      case ide_kind of
        ide_tabstr: begin  DISPOSE( ide_aas ); ide_aas := ps  end;
        ide_tabint: begin  DISPOSE( ide_aai ); ide_aai := pi  end;
        ide_tabflt: begin  DISPOSE( ide_aaf ); ide_aaf := pf  end;
      otherwise
      end;
      { set the new array structure definition and free the old one }
      pold := ide_adm;
      ide_adm := pnew;
      while pold <> nil do
      begin
        pnew := pold;
        pold := pold^.idedim_nxt;
        DISPOSE( pnew )
      end
    end
  end;
  exp_stkp := sps
end EMODIF_ARRAY;


function IN_FILEMODE( iv: integer ): flags_file;
var
  imd: flags_file;

begin
  if ODD( iv ) then imd := [case_ena_file]
               else imd := [];
  if ODD( iv div 2 ) then imd := imd + [nolog_file];
  IN_FILEMODE := imd
end IN_FILEMODE;



procedure EXP_CALLBLT( npa: integer; fnc: ide_bltfnc );
{ To perform the builtin functions }
const
  mdnam = 'BLTF';

var
  iv, i, j, k, l, r: integer;
  rv:                shreal;
  bf, bs:            boolean;
  st, ss:            str_string;
  ftst:              text;
  rec, rec1:         exp_rec;
  knd:               exp_kinds;
  arf:               arr_ope;
  ip:                ide_ptr;
  ch, cc:            char;
  imd, jmd:          flags_file;

begin
  { The parameters are put in the stack }
  case fnc of
    blt_abs:     begin POP_NUMEXPR( bf, iv, rv );
                   if bf then EXP_PUTFLT( ABS( rv ) )
                         else EXP_PUTINT( ABS( iv ) )
                 end;
    blt_round:   EXP_PUTINT( ROUND( POP_FLTEXPR( 0.0 ) ) );
    blt_trunc:   EXP_PUTINT( TRUNC( POP_FLTEXPR( 0.0 ) ) );
    blt_sqrt:    EXP_PUTFLT( SQRT( POP_FLTEXPR( 0.0 ) ) );
    blt_sin:     EXP_PUTFLT( SIN( POP_FLTEXPR( 0.0 ) ) );
    blt_cos:     EXP_PUTFLT( COS( POP_FLTEXPR( 0.0 ) ) );
    blt_tan:     EXP_PUTFLT( TAN( POP_FLTEXPR( 0.0 ) ) );
    blt_asin:    EXP_PUTFLT( ARCSIN( POP_FLTEXPR( 0.0 ) ) );
    blt_acos:    EXP_PUTFLT( ARCCOS( POP_FLTEXPR( 0.0 ) ) );

    blt_atan:    if npa = 2 then
                 begin
                   rv := POP_FLTEXPR( 0.0);
                   EXP_PUTFLT( ARCTAN( POP_FLTEXPR( 1.0 ), rv ) )
                 end
                 else EXP_PUTFLT( ARCTAN( POP_FLTEXPR( 0.0 ) ) );

    blt_sind:    EXP_PUTFLT( SIN( POP_FLTEXPR( 0.0 )*inrd ) );
    blt_cosd:    EXP_PUTFLT( COS( POP_FLTEXPR( 0.0 )*inrd ) );
    blt_tand:    EXP_PUTFLT( TAN( POP_FLTEXPR( 0.0 )*inrd ) );
    blt_asind:   EXP_PUTFLT( ARCSIN( POP_FLTEXPR( 0.0 ) )/inrd );
    blt_acosd:   EXP_PUTFLT( ARCCOS( POP_FLTEXPR( 0.0 ) )/inrd );

    blt_atand:   if npa = 2 then
                 begin
                   rv := POP_FLTEXPR( 0.0);
                   EXP_PUTFLT( ARCTAN( POP_FLTEXPR( 1.0 ), rv )/inrd )
                 end
                 else EXP_PUTFLT( ARCTAN( POP_FLTEXPR( 0.0 ) )/inrd );

    blt_ln:      EXP_PUTFLT( LN( POP_FLTEXPR( 1.0 ) ) );
    blt_exp:     EXP_PUTFLT( EXP( POP_FLTEXPR( 0.0 ) ) );

    blt_filespc,
    blt_filspcm: begin
                  if npa > 1 then i := POP_INTEXPR( 1 ) else i := 0;
                  POP_STREXPR( ss );
                  imd := IN_FILEMODE( i );
                  if SET_FILE_SPECIFICATION( st, ss, imd ) then j := 0
                                                           else j := 121;
                  io_err^.ide_int := j;
                  if (fnc = blt_filspcm) and (j = 0) and (st.length > 0) then
                  begin
                    j := 0;
                    for ii := 1 to st.length do
                    begin
                      if st[ii] <= ' ' then begin  j := j + 1; if j <= ss.capacity then ss[j] := '\'  end;
                      j := j + 1; if j <= ss.capacity then ss[j] := st[ii]
                    end;

                    if j > ss.capacity then begin  j := ss.capacity; io_err^.ide_int := -1  end;
                    ss.length := j;
                    EXP_PUTSTR( ss )
                  end else EXP_PUTSTR( st )
                end;

    blt_string: begin
                  if npa > 2 then j := POP_INTEXPR( 0 ) else j := 0;
                  if npa > 1 then i := POP_INTEXPR( 0 ) else i := 0;
                  POP_NUMEXPR( bf, iv, rv );
                  if bf then WRITEV( st, rv:i:j )
                        else WRITEV( st, iv:i:j );
                  EXP_PUTSTR( st )
                end;

    blt_substr: begin
                  if npa > 2 then j := POP_INTEXPR( 0 ) else j := 0;
                  if npa > 1 then i := POP_INTEXPR( 0 ) else i := 0;
                  POP_STREXPR( st );
                  EXP_PUTSTR( SUBSTR( st, i, j ) )
                end;

    blt_nindex: begin
                  if npa > 2 then i := POP_INTEXPR( 0 )
                             else i := 1;
                  POP_STREXPR( ss );
                  POP_STREXPR( st );
                  EXP_PUTINT( INDEX( st, ss, i ) )
                end;

    blt_slength: begin
                  POP_STREXPR( st );
                  EXP_PUTINT( st.length )
                end;

    blt_setcase: begin
                  POP_STREXPR( st );
                  if npa > 1 then i := POP_INTEXPR( 0 ) else i := 0;
                  if i > 0 then { in maj }
                    for ii := i to st.length do
                    begin
                      if (st[ii] >= 'a') and (st[ii] <= 'z') then
                                     st[ii] := CHR( ORD( st[ii] ) - 32 )
                    end
                  else { in min }
                    for ii := i to st.length do
                    begin
                      if (st[ii] >= 'A') and (st[ii] <= 'Z') then
                                     st[ii] := CHR( ORD( st[ii] ) + 32 )
                    end;
                  EXP_PUTSTR( st )
                end;

    blt_checkch: begin
                  if npa > 2 then k := POP_INTEXPR( 0 )
                             else k := 0;
                  POP_STREXPR( ss );
                  POP_STREXPR( st );
                  iv := 0;
                  i  := 1;
                  while (i <= st.length) and (iv = 0) do
                  begin
                    ch := st[i];
                    if k > 0 then
                      if (ch >='A') and (ch <= 'Z') then ch := CHR( ORD( ch ) + 32 );
                    j := 1;
                    while (j <= ss.length) and (iv = 0) do
                    begin
                      cc := ss[j];
                      if k > 0 then
                        if (cc >='A') and (cc <= 'Z') then cc := CHR( ORD( cc ) + 32 );
                      if cc = ch then iv := j;
                      j := j + 1
                    end;
                    if iv = 0 then i := i + 1
                  end;
                  EXP_PUTINT( iv )
                end;

    blt_checknst: begin
                  POP_STREXPR( st );
                  EXP_PUTINT( ORD( USR_NUMERIC_STRING( st ) ) )
                end;

    blt_selement: begin
                  ch := ' '; cc := '"';
                  if npa > 4 then bs := POP_INTEXPR( 0 ) > 0
                             else bs := false;
                  if npa > 3 then
                  begin  POP_STREXPR( ss ); if ss.length > 0 then cc := ss[1]  end;
                  if npa > 2 then
                  begin  POP_STREXPR( ss ); if ss.length > 0 then ch := ss[1]  end;
                  if npa > 1 then iv := POP_INTEXPR( 1 ) else iv := 1;
                  POP_STREXPR( ss );
                  USR_S_ELEMENT( st, ss, iv, ch, cc, bs );
                  EXP_PUTSTR( st )
                end;

    blt_supcomment: begin
                  cc := '"'; ch := '!';
                  if npa > 2 then
                  begin  POP_STREXPR( ss ); if ss.length > 0 then cc := ss[1]  end;
                  if npa > 1 then
                  begin  POP_STREXPR( ss ); if ss.length > 0 then ch := ss[1]  end;
                  POP_STREXPR( ss );
                  USR_SUPPRESS_COMMENT( ss, st, ch, cc );
                  EXP_PUTSTR( st )
                end;

    blt_fidl_replace: begin
                  POP_STREXPR( ss );
                  if ss.length > 0 then ch := ss[1]
                                   else ch := '"';
                  POP_STREXPR( ss );
                  USR_IDE_SUBSTITUTE( ss, st, ch );
                  EXP_PUTSTR( st )
                end;

    blt_getenv: begin
                  POP_STREXPR( st );
                  i := GET_LOGICAL( ss, st );
                  if i <> 0 then ss := '';
                  EXP_PUTSTR( ss )
                end;

    blt_setenv: begin
                  if npa > 1 then POP_STREXPR( ss )
                             else ss := '';
                  POP_STREXPR( st );
                  EXP_PUTINT( ORD( SET_LOGICAL( st, ss ) = 0 ) )
                end;

    blt_getpath: begin
                  ch := ':';
                  if npa > 2 then
                  begin
                    POP_STREXPR( ss );
                    if ss.length > 0 then ch := ss[1]
                  end;
                  POP_STREXPR( st );           { Get the logical name string }
                  POP_EXP_REFER( rec );        { Get the string array reference }
                  i := -1;                     { Assume bad array error until shown otherwise }
                  if rec.exp_kind <> exp_tabstr then SRC_ERROR( mdnam, 999, e_severe )
                                                else i := USR_GETPATH( ch, st, rec.exp_aas^ );
                  EXP_PUTINT( i )
                end;

    blt_setpath: begin
                  if npa > 2 then i := POP_INTEXPR( 0 )
                             else i := 0;
                  POP_STREXPR( st );
                  POP_EXP_REFER( rec );
                  i := -1;                     { Assume bad array error until shown otherwise }
                  if rec.exp_kind <> exp_tabstr then SRC_ERROR( mdnam, 999, e_severe )
                                                else i := USR_SETPATH( st, i, rec.exp_aas^ );
                  EXP_PUTINT( i )
                end;

    blt_spawn:  begin
                  if npa > 0 then POP_STREXPR( st ) else st := '';
                  EXP_PUTINT( ORD( SYS_SPAWN( st ) ) )
                end;

    blt_run:    begin
                  if npa > 1 then i := POP_INTEXPR( 0 ) else i := 0;
                  POP_STREXPR( st );
                  RUN_PROCESS( '', st, i );
                  EXP_PUTINT( -1 )
                end;

    blt_exec:   begin
                  if npa > 1 then i := POP_INTEXPR( 0 ) else i := 0;
                  POP_STREXPR( st );
                  j := CREATE_PROCESS( '', st, i );
                  EXP_PUTINT( j )
                end;

    blt_wait:   EXP_PUTINT( WAIT_PROCESS( io_winfo^.ide_int, POP_INTEXPR( 0 ) ) );

    blt_exit:   begin
                  if npa > 0 then i := POP_INTEXPR( 0 )
                             else i := 1;
                   PASCAL_EXIT( i )
                end;

    blt_time:   begin  TIME( st ); EXP_PUTSTR( st )  end;
    blt_date:   begin  DATE( st ); EXP_PUTSTR( st )  end;
    blt_dfdir:  EXP_PUTSTR( GET_DEF_DIR );

    blt_chdir:  begin
                  if npa > 1 then i := POP_INTEXPR( 1 ) else i := 0;
                  POP_STREXPR( st );
                  imd := IN_FILEMODE( i );
                  EXP_PUTINT( ORD( CHANGE_DIRECTORY( st, imd ) ) )
                end;

    blt_f_exist: begin
                  if npa > 1 then i := POP_INTEXPR( 4 ) else i := 4; { Read access }
                  POP_STREXPR( st );
                  if FILE_ACCESS_CHECK( st, i, [case_ena_file] ) then EXP_PUTINT( 1 )
                                                                 else EXP_PUTINT( 0 )
                end;

    blt_f_delete: begin
                  if npa > 1 then i := POP_INTEXPR( 1 ) else i := 0;
                  POP_STREXPR( st );
                  imd := IN_FILEMODE( i );
                  if FILE_REMOVE( st, imd ) then EXP_PUTINT( 1 )
                                            else EXP_PUTINT( 0 )
                end;

    blt_f_rename: begin
                  if npa > 3 then j := POP_INTEXPR( 1 ) else j := 0;
                  jmd := IN_FILEMODE( j );
                  if npa > 2 then i := POP_INTEXPR( 1 ) else i := 0;
                  imd := IN_FILEMODE( i );
                  POP_STREXPR( ss );
                  POP_STREXPR( st );
                  if FILE_RENAME( st, ss, imd, jmd ) then EXP_PUTINT( 1 )
                                                     else EXP_PUTINT( 0 )
                end;

    blt_ardim:  begin { Get an array dimension }
                  if npa > 1 then j := POP_INTEXPR( 0 ) else j := 0;
                  POP_EXP_REFER( rec );
                  i := -1;
                  with rec do
                  if (j > 0) and (exp_adm <> nil) then
                  begin
                    repeat
                      i := exp_adm^.idedim_siz;
                      exp_adm := exp_adm^.idedim_nxt;
                      j := j - 1
                    until (exp_adm = nil) or (j <= 0);
                  end
                  else i := exp_edim;
                  EXP_PUTINT( i )
                end;

    blt_arsearch:
                with rec do
                begin { Search an element in a one dimension array }
                  if npa > 2 then j := POP_INTEXPR( 0 ) else j := 0;
                  if j > 0 then j := 1 else if j < 0 then j := -1;
                  knd := EXP_GETKINDS( 1 );
                  { To following the type of array }
                  case knd of
                    exp_tabint: iv := POP_INTEXPR( 0 );
                    exp_tabflt: rv := POP_FLTEXPR( 0.0 );
                    exp_tabstr: POP_STREXPR( st );
                  otherwise
                    SRC_ERROR( mdnam, 113, e_error )
                  end;
                  { Get the array reference }
                  POP_EXP_REFER( rec );
                  r := 0;
                  i := 0;
                  if exp_adm <> nil then
                  with exp_adm^ do
                  if idedim_nxt = nil then
                  begin
                    i := exp_shift;
                    l := exp_edim + exp_shift - 1;
                    case exp_kind of
                      exp_tabint:
                        if exp_aai <> nil then
                        with exp_aai^ do
                        repeat
                          case j of
                           -1: if ide_itb[i] < iv then r := i;
                            0: if ide_itb[i] = iv then r := i;
                            1: if ide_itb[i] > iv then r := i;
                          otherwise
                          end;
                        exit if r <> 0;
                          i := i + 1;
                        until i > l;

                      exp_tabflt:
                        if exp_aaf <> nil then
                        with exp_aaf^ do
                        repeat
                          case j of
                           -1: if ide_ftb[i] < rv then r := i;
                            0: if ide_ftb[i] = rv then r := i;
                            1: if ide_ftb[i] > rv then r := i;
                          otherwise
                          end;
                        exit if r <> 0;
                          i := i + 1;
                        until i > l;

                      exp_tabstr:
                        if exp_aas <> nil then
                        with exp_aas^ do
                        repeat
                          if ide_stb[i] = nil then
                            if st.length = 0 then k :=  0
                                             else k := -1
                          else
                            k := STR_MATCH( ide_stb[i]^, st );
                          case j of
                           -1: if k < 0 then r := i;
                            0: if k = 0 then r := i;
                            1: if k > 0 then r := i;
                          otherwise
                          end;
                        exit if r <> 0;
                          i := i + 1;
                        until i > l;

                    otherwise
                    end
                  end;
                  EXP_PUTINT( r )
                end;

    blt_chardim: begin
                  EMODIF_ARRAY( npa );
                  EXP_PUTINT( 0 )
                end;

    blt_aradd, blt_arsub, blt_armul, blt_ardiv:
                begin
                  case fnc of
                    blt_aradd: arf := ar_add;
                    blt_arsub: arf := ar_sub;
                    blt_armul: arf := ar_mul;
                    blt_ardiv: arf := ar_div;
                  otherwise
                  end;
                  POP_EXPRESSION;
                  rec1 := exp_res;
                  POP_EXP_REFER( rec );
                  i := 0; { Assume no error until shown otherwise }
                  with rec do
                  begin
                    case exp_kind of
                      exp_tabint: { Integer Array Destination }
                        if exp_aai <> nil then
                        case rec1.exp_kind of
                          exp_valint: ARRSCA_II_OPE( arf, rec1.exp_int, rec );

                          exp_valflt:
                            ARRSCA_II_OPE( arf, ROUND( rec1.exp_flt ), rec );

                          exp_telint:
                            if rec1.exp_aai <> nil then
                            with rec1.exp_aai^ do
                              ARRSCA_II_OPE( arf, ide_itb[exp_shift], rec );

                          exp_telflt:
                            if rec1.exp_aaf <> nil then
                            with rec1.exp_aaf^ do
                              ARRSCA_II_OPE( arf,
                                             ROUND( ide_ftb[exp_shift] ), rec );

                          exp_tabint: ARRAY_II_OPE( arf, rec1, rec );

                          exp_tabflt: ARRAY_IF_OPE( arf, rec1, rec );

                        otherwise
                        end;

                      exp_tabflt: { Floatting Array Destination }
                        case rec1.exp_kind of
                          exp_valint:
                            ARRSCA_FF_OPE( arf, SHREAL( rec1.exp_int ), rec );

                          exp_valflt: ARRSCA_FF_OPE( arf, rec1.exp_flt, rec );

                          exp_telint:
                            if rec1.exp_aai <> nil then
                            with rec1.exp_aai^ do
                              ARRSCA_FF_OPE( arf,
                                             SHREAL( ide_itb[exp_shift]), rec );

                          exp_telflt:
                            if rec1.exp_aaf <> nil then
                            with rec1.exp_aaf^ do
                              ARRSCA_FF_OPE( arf, ide_ftb[exp_shift], rec );

                          exp_tabint: ARRAY_FI_OPE( arf, rec1, rec );

                          exp_tabflt: ARRAY_FF_OPE( arf, rec1, rec );
                        otherwise
                          i := -1
                        end;
                    otherwise
                      i := -1
                    end;
                    if i <> 0 then SRC_ERROR( mdnam, 115, e_severe )
                  end;
                  EXP_PUTINT( 0 )
                end;

  otherwise
    SRC_ERROR( mdnam, 901, e_severe )
  end
end EXP_CALLBLT;



procedure EXP_ARRAY_REF( exp_elem: exp_kinds );
{ To manipulate the array reference in the stack }
const
  mdnam = 'ARRF';

var
  idx, jdx: integer;

begin
  with sy_sym, exp_stk[exp_stkp] do
  if exp_adm <> nil then                 { For correct array definition }
  with exp_adm^ do
  begin
    INSYMBOL;                            { Gobble up the array identifier }
    idx := GET_INTEXPR( 0 );             { Get the specified index }
    if (idx <= 0) or (idx > idedim_siz) then
    begin                                { If the index is out of range }
      SRC_ERROR( mdnam, 71, e_error );   { Error ! }
      idx := 1
    end;
    { Set the shift of object in the array }
    exp_shift := exp_shift + (idx - 1)*idedim_stp;
    exp_adm   := idedim_nxt;             { set as subarray or scalar ref. }
    if sy = comma then EXP_ARRAY_REF( exp_elem ) { more internal dimension }
    else                                 { last index, a sub-array ref. ? }
      if sy = twodot then
      begin { sub-array reference }
        INSYMBOL;
        jdx := GET_INTEXPR( idx );       { Get the upper index bound }
        if (jdx < idx) or (jdx > idedim_siz) then
        begin { upper bound < lower bound or upper bound greater than max. }
          SRC_ERROR( mdnam, 80, e_error );
          jdx := idedim_siz - idx + 1
        end;
        exp_edim := idedim_stp*(jdx - idx + 1);
        if (exp_adm = nil) and (jdx = idx) then exp_kind := exp_elem
      end
      else
      begin
        exp_edim := idedim_stp;
        if exp_adm = nil then exp_kind := exp_elem { Array Element Reference }
      end
  end
  else
  begin { Not a good array }
    SRC_ERROR( mdnam, 82, e_error );
    SKIP_SYMBOL( rbrack )
  end
end EXP_ARRAY_REF;



procedure EXP_ENTRY;
const
  mdnam = 'EXPE';

var
  ip:       ide_ptr;
  bin:      boolean;
  npa, idw: integer;

begin
  bin := true;
  if exp_stkp < max_stk then exp_stkp := SUCC( exp_stkp )
                        else SRC_ERROR( mdnam, 6, e_severe );
  with sy_sym, exp_stk[exp_stkp]  do
  case sy of
    identsy: begin
               ip := IDE_SEARCH( true ); { Search the identifier }
               if ip <> nil then
               with ip^ do
               case ide_kind of
                 ide_funct: (* blt_id_exist / blt_mid_exist *)
                   case ide_fnc of
                     blt_fidl_insert,
                     blt_fidl_remove:
                       begin
                         INSYMBOL;           { Gobble up the function identifier }
                         bin := false;
                         if sy = lparen then sy := comma
                                        else SRC_ERROR( mdnam, 22, e_error );
                         while sy = comma do
                         begin
                           INSYMBOL;         { Gobble up the separator ',' or '(' }
                           ip := IDE_SEARCH( true ); { Locate the identifier }
                           if ide_fnc = blt_fidl_insert then USR_IDE_APPEND( ip )
                                                        else USR_IDE_REMOVE( ip );
                           INSYMBOL          { Gobble up the parameter identifier }
                         end;
                         exp_ref   := nil;
                         exp_adm   := nil;
                         exp_shift :=   0;
                         exp_edim  :=   1;
                         exp_kind  := exp_valint;
                         exp_int   :=   1;
                         if sy <> rparen then SRC_ERROR( mdnam, 23, e_error )
                                         else INSYMBOL
                       end;

                     blt_id_exist,
                     blt_mid_exist:
                       begin
                         INSYMBOL;           { Gobble up the function identifier }
                         bin := false;
                         sy_nomacrflg := (ide_fnc = blt_mid_exist);
                         if sy = lparen then INSYMBOL
                                        else SRC_ERROR( mdnam, 22, e_error );
                         if sy = identsy then
                         begin
                           exp_ref   := nil;
                           exp_adm   := nil;
                           exp_shift :=   0;
                           exp_edim  :=   1;
                           exp_kind  := exp_valint;
                           exp_int   :=   0;
                           if sy_nomacrflg and sy_macro <> nil then
                             case sy_macro^.idm_kind of
                               idm_macro: exp_int := 20;
                               idm_parm:  exp_int := 21;
                               idm_list:  exp_int := 22;
                               idm_temp:  exp_int := 23;
                             otherwise
                               exp_int := -2
                             end
                           else
                           begin
                             ip := IDE_SEARCH( false ); { Search the identifier }
                             if ip <> nil then
                             case ip^.ide_kind of
                               ide_parstr: exp_int := 1;
                               ide_parint: exp_int := 2;
                               ide_parflt: exp_int := 3;
                               ide_tabstr: exp_int := 4;
                               ide_tabint: exp_int := 5;
                               ide_tabflt: exp_int := 6;
                             otherwise
                               exp_int := -1
                             end
                           end
                         end
                         else
                           SRC_ERROR( mdnam, 122, e_warning );
                         sy_nomacrflg := false;
                         INSYMBOL;
                         if sy <> rparen then SRC_ERROR( mdnam, 23, e_error )
                                         else INSYMBOL
                     end;

                   otherwise { Builtin function call }
                     INSYMBOL;           { Gobble up the identifier }
                     exp_stkp := PRED( exp_stkp );
                     npa := 0;
                     bin := false;
                     if sy = lparen then
                     begin
                       sy := comma;
                       while sy = comma do
                       begin
                         INSYMBOL; { Gobble up "(" or "," }
                         if (sy <> comma) and (sy <> rparen) then
                         begin
                           EXPRESSION; npa := SUCC( npa )
                         end
                       end;
                       if sy <> rparen then SRC_ERROR( mdnam, 23, e_error )
                                       else INSYMBOL
                     end;
                     if (npa < ide_nparm) or (npa > ide_nparmax) then
                       SRC_ERROR( mdnam, 41, e_severe );
                     EXP_CALLBLT( npa, ide_fnc )
                   end;

                 ide_functsp:
                   begin { as ide_funct except the call of EXECUTE_SPECIFIC_ENTRY }
                     INSYMBOL;           { Gobble up the identifier }
                     exp_stkp := PRED( exp_stkp );
                     npa := 0;
                     bin := false;
                     if sy = lparen then
                     begin
                       sy := comma;
                       while sy = comma do
                       begin
                         INSYMBOL; { Gobble up "(" or "," }
                         if (sy <> comma) and (sy <> rparen) then
                         begin
                           EXPRESSION; npa := SUCC( npa )
                         end
                       end;
                       if sy <> rparen then SRC_ERROR( mdnam, 23, e_error )
                                       else INSYMBOL
                     end;
                     if (npa < ide_nparm) or (npa > ide_nparmax) then
                       SRC_ERROR( mdnam, 41, e_severe );
                     EXECUTE_SPECIFIC_ENTRY( npa, ide_fnc )
                   end;

                 ide_parstr:
                   begin
                     exp_ref   :=  ip;
                     exp_adm   := nil;
                     exp_shift :=   0;
                     exp_edim  :=   1;
                     exp_kind  := exp_valstr;
                     exp_str   := ide_str
                   end;

                 ide_parint:
                   begin
                     exp_ref   :=  ip;
                     exp_adm   := nil;
                     exp_shift :=   0;
                     exp_edim  :=   1;
                     exp_kind  := exp_valint;
                     exp_int   := ide_int
                   end;

                 ide_parflt:
                   begin
                     exp_ref   :=  ip;
                     exp_adm   := nil;
                     exp_shift :=   0;
                     exp_edim  :=   1;
                     exp_kind  := exp_valflt;
                     exp_flt   := ide_flt
                   end;

                 ide_tabstr:
                   if ide_aas <> nil then
                   with ide_aas^ do
                   begin
                     exp_ref   :=         ip;
                     exp_shift :=          1;
                     exp_adm   :=    ide_adm;
                     exp_kind  := exp_tabstr;
                     if ide_adm <> nil then
                     with ide_adm^ do
                       exp_edim  := idedim_stp*idedim_siz;
                     exp_aas   :=    ide_aas;
                     INSYMBOL;
                     if sy = lbrack then
                     begin
                       EXP_ARRAY_REF( exp_telstr );
                       if sy <> rbrack then SRC_ERROR( mdnam, 26, e_error )
                     end else bin := false
                   end
                   else SRC_ERROR( mdnam, 801, e_severe );

                 ide_tabint:
                   if ide_aai <> nil then
                   with ide_aai^ do
                   begin
                     exp_ref   :=         ip;
                     exp_shift :=          1;
                     exp_adm   :=    ide_adm;
                     exp_kind  := exp_tabint;
                     exp_aai   :=    ide_aai;
                     if ide_adm <> nil then
                     with ide_adm^ do
                       exp_edim  := idedim_stp*idedim_siz;
                     INSYMBOL;
                     if sy = lbrack then
                     begin
                       EXP_ARRAY_REF( exp_telint );
                       if sy <> rbrack then SRC_ERROR( mdnam, 26, e_error )
                     end else bin := false
                   end
                   else SRC_ERROR( mdnam, 802, e_severe );

                 ide_tabflt:
                   if ide_aaf <> nil then
                   with ide_aaf^ do
                   begin
                     exp_ref   :=         ip;
                     exp_shift :=          1;
                     exp_adm   :=    ide_adm;
                     exp_kind  := exp_tabflt;
                     if ide_adm <> nil then
                     with ide_adm^ do
                       exp_edim  := idedim_stp*idedim_siz;
                     exp_aaf   :=    ide_aaf;
                     INSYMBOL;
                     if sy = lbrack then
                     begin
                       EXP_ARRAY_REF( exp_telflt );
                       if sy <> rbrack then SRC_ERROR( mdnam, 26, e_error )
                     end else bin := false
                   end
                   else SRC_ERROR( mdnam, 803, e_severe );

               otherwise
               end { case ide_kind of }
      end;

    intconst:    begin
                   exp_ref   := nil;
                   exp_adm   := nil;
                   exp_shift :=   0;
                   exp_edim  :=   1;
                   exp_kind  := exp_valint;
                   exp_int   := sy_ival
                 end;

    singleconst,
    doubleconst: begin
                   exp_ref   := nil;
                   exp_adm   := nil;
                   exp_shift :=   0;
                   exp_edim  :=   1;
                   exp_kind  := exp_valflt;
                   exp_flt   := sy_rval
                 end;

    stringconst: begin
                   exp_ref   := nil;
                   exp_adm   := nil;
                   exp_shift :=   0;
                   exp_edim  :=   1;
                   exp_kind  := exp_valstr;
                   if sy_string.length > 0 then
                   begin
                     NEW( exp_str, sy_string.length );
                     exp_str^ := sy_string
                   end else exp_str := nil;
                 end;

  otherwise
    SRC_ERROR( mdnam, 81, e_error )
  end;
  if debug_exp then
  with exp_stk[exp_stkp], lst_current^ do
  begin
    WRITE( lst_file, ' Entry_push : ', exp_kind, ORD( exp_kind ):6, ' ' );
    case exp_kind of
      exp_valint:      WRITE( lst_file, exp_int );
      exp_valflt:      WRITE( lst_file, exp_flt );
      exp_valstr:      if exp_str <> nil then WRITE( lst_file, '"', exp_str^, '"' );
    otherwise
    end;
    WRITELN( lst_file )
  end;
  if bin then INSYMBOL
end EXP_ENTRY;



procedure EXPRESSION; { Was forward }
const
  mdnam = 'EXPR';

var
  bi, bj:  boolean;
  ops: operator;

  procedure EXP_AND;
  var
    i, j:  integer;

    procedure EXP_NOT;

      procedure EXP_REL;
      var
        ops: operator;
        kn1, kn2: exp_kinds;
        st1, st2: string( 255 );
        rv: shreal;
        iv: integer;

        procedure EXP_ADD;
        var
          ops: operator;
          iv, iw:  integer;
          rv, rw:  shreal;
          bf, bg:  boolean;
          st1, st2: string( 255 );

          procedure EXP_MUL;
          var
            ops: operator;
            iv, iw:  integer;
            rv, rw:  shreal;
            bf, bg:  boolean;

            procedure EXP_POW;
            var
              ops: operator;
              iv, iw:  integer;
              rv, rw:  shreal;
              bf, bg:  boolean;

              procedure EXP_UNA;
              var
                ops: operator;
                iv:  integer;
                rv:  shreal;
                bf:  boolean;
                st:  string( 255 );

              begin { EXP_UNA }
                with sy_sym do
                begin
                  if (sy = addop) and ((op = sub_op) or (op = add_op)) then
                    sy := unaop;
                  if sy = unaop then
                  begin
                    ops := op;
                    INSYMBOL;                  { Gobble up the operator }
                    EXP_UNA;                   { Get the parameter }
                    if ops = sub_op then
                    begin
                      POP_NUMEXPR( bf, iv, rv );
                      if bf then EXP_PUTFLT( -rv )
                            else EXP_PUTINT( -iv )
                    end
                  end
                  else
                  if sy = lparen then
                  begin
                    INSYMBOL;
                    EXPRESSION;
                    if sy = rparen then INSYMBOL
                                   else SRC_ERROR( mdnam, 23, e_severe )
                  end
                  else EXP_ENTRY               { Get the object }
                end
              end EXP_UNA;

            begin { EXP_POW }
              EXP_UNA;
              with sy_sym do
                if sy = powop then             { For any power class operator }
                begin
                  ops := op;                   { Save the operator }
                  INSYMBOL;                    { Gobble up it }
                  EXP_UNA;                     { Get the second operand }
                  bg := false; iw := 1;        { Set default value }
                  POP_NUMEXPR( bg, iw, rw );
                  bf := false; iv := 1;        { Set default value }
                  POP_NUMEXPR( bf, iv, rv );
                  if ops = pow_op then
                    if bg then                 { i**r or r**r }
                      if bf then { r**r } EXP_PUTFLT( rv**rw )
                            else { i**r } EXP_PUTFLT( iv**rw )
                    else
                      if bf then { r**i } EXP_PUTFLT( rv**iw )
                            else { i**i } EXP_PUTINT( iv**iw )
                end
            end EXP_POW;

          begin { EXP_MUL }
            EXP_POW;
            with sy_sym do
            while sy = mulop do                { For any power class operator }
            begin
              ops := op;                       { Save the operator }
              INSYMBOL;                        { Gobble up it }
              EXP_POW;                         { Get the second operand }
              case ops of
                mul_op:
                  begin
                    bg := false; iw := 1;      { Set default value }
                    POP_NUMEXPR( bg, iw, rw );
                    bf := false; iv := 1;      { Set default value }
                    POP_NUMEXPR( bf, iv, rv );
                    if bg then if bf then EXP_PUTFLT( rv*rw )
                                     else EXP_PUTFLT( iv*rw )
                          else if bf then EXP_PUTFLT( rv*iw )
                                     else EXP_PUTINT( iv*iw )
                  end;
                div_op:
                  begin
                    rv := POP_FLTEXPR( 1.0 );
                    EXP_PUTFLT( POP_FLTEXPR( 0.0 )/rv )
                  end;
                idiv_op:
                  begin
                    iv := POP_INTEXPR( 1 );
                    EXP_PUTINT( POP_INTEXPR( 0 ) div iv )
                  end;
                imod_op:
                  begin
                    iv := POP_INTEXPR( 1 );
                    EXP_PUTINT( POP_INTEXPR( 0 ) mod iv )
                  end;
                irem_op:
                  begin
                    iv := POP_INTEXPR( 1 );
                    EXP_PUTINT( POP_INTEXPR( 0 ) rem iv )
                  end;
              otherwise
              end
            end
          end EXP_MUL;

        begin { EXP_ADD }
          EXP_MUL;
          with sy_sym do
          while sy = addop do          { For any power class operator }
          begin
            ops := op;                 { Save the operator }
            INSYMBOL;                  { Gobble up it }
            EXP_MUL;                   { Get the second operand }
            if ops = concat_op then
            begin
              POP_STREXPR( st2 ); POP_STREXPR( st1 );
              EXP_PUTSTR( st1||st2 )
            end
            else
            begin
              bg := false; iw := 1;    { Set default value }
              POP_NUMEXPR( bg, iw, rw );
              bf := false; iv := 1;    { Set default value }
              POP_NUMEXPR( bf, iv, rv );
              if ops = add_op then
                if bg then if bf then EXP_PUTFLT( rv+rw )
                                 else EXP_PUTFLT( iv+rw )
                      else if bf then EXP_PUTFLT( rv+iw )
                                 else EXP_PUTINT( iv+iw )
              else
                if bg then if bf then EXP_PUTFLT( rv-rw )
                                 else EXP_PUTFLT( iv-rw )
                      else if bf then EXP_PUTFLT( rv-iw )
                                 else EXP_PUTINT( iv-iw )
            end
          end
        end EXP_ADD;

      begin { EXP_REL }
        EXP_ADD;
        with sy_sym do
        while sy = relop do
        begin
          ops := op;                   { Get the definition list head }
          INSYMBOL;
          EXP_ADD;
          kn1 := EXP_GETKINDS(  1 );
          kn2 := EXP_GETKINDS(  0 );
          case kn1 of
            exp_valstr: kn2 := exp_valstr;
            exp_valflt: if kn2 = exp_valstr then kn1 := exp_valstr
                                            else kn2 := exp_valflt;
            exp_valint: case kn2 of
                          exp_valstr: kn1 := exp_valstr;
                          exp_valflt: kn1 := exp_valflt;
                        otherwise
                          kn2 := exp_valint
                        end;
          otherwise
            if kn2 = exp_valnull then kn2 := exp_valint;
            kn1 := kn2
          end;
          case kn1 of
            exp_valstr:
              begin
                POP_STREXPR( st2 ); POP_STREXPR( st1 );
                rv := STR_MATCH( st1, st2 )
              end;
            exp_valflt:
              begin
                rv := POP_FLTEXPR( 0.0 );
                rv := POP_FLTEXPR( 0.0 ) - rv
              end;
            exp_valint:
              begin
                rv := POP_INTEXPR( 0 );
                rv := POP_INTEXPR( 0 ) - rv
              end
          otherwise
            rv := 0.0
          end;
          case ops of
            lt_op: iv := ORD( rv  < 0.0 );
            le_op: iv := ORD( rv <= 0.0 );
            ge_op: iv := ORD( rv >= 0.0 );
            gt_op: iv := ORD( rv  > 0.0 );
            eq_op: iv := ORD( rv  = 0.0 );
            ne_op: iv := ORD( rv <> 0.0 );
          otherwise
          end;
          EXP_PUTINT( iv )
        end;
      end EXP_REL;

    begin { EXP_NOT }
      with sy_sym do
        if sy = notop then
        begin
          INSYMBOL;
          EXP_NOT;
          EXP_PUTINT( ORD( POP_INTEXPR( 0 ) <= 0) )
        end
        else EXP_REL;
    end EXP_NOT;

  begin { EXP_AND }
    EXP_NOT;
    with sy_sym do
    while sy = lgandop do
    begin
      INSYMBOL;
      EXP_NOT;
      i := POP_INTEXPR( 0 ); j := POP_INTEXPR( 0 );
      EXP_PUTINT( ORD( (i > 0) and (j > 0) ) )
    end
  end EXP_AND;

begin { EXPRESSION }
  EXP_AND;
  with sy_sym do
  while sy = lgorop do
  begin
    ops := op;
    INSYMBOL;
    EXP_AND;
    bi := (POP_INTEXPR( 0 ) > 0); bj := (POP_INTEXPR( 0 ) > 0);
    if ops = or_op then EXP_PUTINT( ORD( bi or bj ) )
                   else EXP_PUTINT( ORD( bi <> bj ) )
  end;
  if debug_exp then
  with exp_stk[exp_stkp], lst_current^ do
  begin
    WRITE( lst_file, ' EXP : ', ORD( exp_kind ), ' ' );
    case exp_kind of
      exp_valint:      WRITE( lst_file, exp_int );
      exp_valflt:      WRITE( lst_file, exp_flt );
      exp_valstr:      if exp_str <> nil then WRITE( lst_file, '"', exp_str^, '"' );
    otherwise
    end;
    WRITELN( lst_file )
  end
end EXPRESSION;






     {**************************************}
     { MACRO CODE GENERATION group Routines }
     {**************************************}


procedure GENERATE_MACRO_PARM( pa: idm_ptr; bpar: boolean; fch: char );
const
  mdnam = 'GMPA';

var
  ll:  integer;
  chs: char;


  procedure MAC_EXPR( blist: boolean );
  var
    icnt: integer;
    bpa:  boolean;

  begin
    bpa  := false;
    icnt :=     0;
    with sy_sym do
    loop
      case sy of
        comma, colon, twodot:
          if (not blist) and (icnt = 0) then exit;
        identsy, intconst, singleconst, doubleconst, stringconst:
          if icnt = 0 then
            if bpa then exit
                   else bpa := true;
        unaop, notop:
          if (icnt = 0) and bpa then exit;
        powop, mulop, addop, relop, lgandop, lgorop:
          if icnt = 0 then bpa := false;
        lbrack, lparen:
          icnt := icnt + 1;
        rbrack, rparen:
          begin
            icnt := icnt - 1;
            if icnt < 0 then exit
          end
      otherwise
        exit
      end;
      if idm_newmac <> nil then ll := idm_newmac^.idm_use - 1;
      INSYMBOL
    end
  end MAC_EXPR;


begin { GENERATE_MACRO_PARM }
  idm_space    := true;
  idm_outmacro := true;
  ll := 0;
  if pa <> nil then
  with sy_sym, pa^, lst_current^ do
  begin
    if idm_newmac <> nil then
      idm_newmac^.idm_use := 0;
    if bpar then
    begin
      chs := sy_ch;   sy_ch := '(';
      GEN_MACRO_CODE; sy_ch := chs
    end;
    GEN_MACRO_CODE; { Put the first character }
    if sy <> semicolon then INSYMBOL; { Get the first syntax unit }
    if (sy <> comma) and (sy <> colon) and (sy <> semicolon) and
       (sy <> rparen) and (sy <> eofsy) then
    begin
      MAC_EXPR( false );

      if debug_macsrc then
        if idm_name = nil then
          if idm_newmac <> nil then
          with idm_newmac^ do
            WRITELN( lst_file, ' Internal Parameter (', ll:-3, ') = "', idm_ctb:idm_use, '"' )
          else
            WRITELN( lst_file, ' Internal Empty Parameter.' )
        else
          if idm_newmac <> nil then
          with idm_newmac^ do
            WRITELN( lst_file, ' Parameter "', pa^.idm_name^, '" (', ll:-3,
                     ') = "', idm_ctb:idm_use, '"' )
          else
            WRITELN( lst_file, ' Empty Parameter "', pa^.idm_name^, '"' );

      if idm_newmac <> nil then idm_newmac^.idm_use := ll;
      chs := sy_ch;
      if bpar then
      begin
        sy_ch := ')'; GEN_MACRO_CODE
      end;
      idm_space := false; sy_ch := ' '; GEN_MACRO_CODE;
      if fch <> ' ' then
      begin
        sy_ch := fch; GEN_MACRO_CODE
      end;
      sy_ch := chs;
      if idm_newmac <> nil then
      begin { Now we adjust the parameter location }
        if idm_tab <> nil then DISPOSE( idm_tab );
        if ll > 0 then
        begin
          ll := idm_newmac^.idm_use;
          NEW( idm_tab, ll );
          with idm_tab^ do
          begin
            idm_use := ll;
            for i := 1 to ll do idm_ctb[i] := idm_newmac^.idm_ctb[i];
            if debug_sym then
            begin
              WRITELN( lst_file, ' M_Parm code loaded :' );
              WRITELN( lst_file, ' "', idm_ctb: ll, '".' )
            end
          end;
          idm_nch := 1
        end;
        idm_newmac^.idm_use := 0
      end
    end
  end;
  idm_outmacro := false
end GENERATE_MACRO_PARM;



[global]
function NEW_MACRO_EXPR: idm_ptr;
{ Create a unnamed macro parameter }
{ The created macro parameter is not put in the defined stack }
var
  p: idm_ptr;

begin
  NEW( p );
  with p^ do
  begin
    idm_name := nil;
    idm_parl := nil;
    idm_cntx := nil;
    idm_nxt  := nil;
    idm_prv  := nil;
    idm_tab  := nil;
    idm_run  := false;
    idm_kind := idm_parm;
    GENERATE_MACRO_PARM( p, false, ';' )
  end;
  NEW_MACRO_EXPR := p
end NEW_MACRO_EXPR;



[global]
procedure PURGE_MACRO_EXPR( var p: idm_ptr );
begin
  if p <> nil then
  begin
    with p^ do
    begin
      if idm_prv <> nil then idm_prv^.idm_nxt := idm_nxt;
      if idm_nxt <> nil then idm_nxt^.idm_prv := idm_prv;
      if idm_tab <> nil then DISPOSE( idm_tab )
    end;
    DISPOSE( p );
    p := nil
  end
end PURGE_MACRO_EXPR;



[global]
function GET_INT_VALUE( pa: idm_ptr ): integer;
const
  mdnam = 'IVAL';

var
  res: integer;

begin
  res := 0;
  with pa^ do
  if idm_kind <> idm_parm then SRC_ERROR( mdnam, 311, e_severe )
  else
  begin
    ACTIVE_MACRO_CODE( pa );
    INSYMBOL;
    res := GET_INTEXPR( 0 );
    RET_OF_MACRO_CODE( pa )
  end;
  GET_INT_VALUE := res
end GET_INT_VALUE;



[global]
function GET_FLT_VALUE( pa: idm_ptr ): shreal;
const
  mdnam = 'FVAL';

var
  res: shreal;

begin
  res := 0.0;
  with pa^ do
  if idm_kind <> idm_parm then SRC_ERROR( mdnam, 312, e_severe )
  else
  begin
    ACTIVE_MACRO_CODE( pa );
    INSYMBOL;
    res := GET_FLTEXPR( 0.0 );
    RET_OF_MACRO_CODE( pa )
  end;
  GET_FLT_VALUE := res
end GET_FLT_VALUE;



[global]
procedure GET_STR_VALUE( var st: string; pa: idm_ptr );
const
  mdnam = 'SVAL';

begin
  with pa^ do
  if idm_kind <> idm_parm then SRC_ERROR( mdnam, 313, e_severe )
  else
  begin
    ACTIVE_MACRO_CODE( pa );
    INSYMBOL;
    GET_STREXPR( st );
    RET_OF_MACRO_CODE( pa )
  end
end GET_STR_VALUE;



function GEN_MACRO_LIST( mcmd: symbol ): idm_apt;
var
  p:  idm_apt;
  iv: integer;
  sv_noexec: boolean;

begin
  sv_noexec    := sy_noexec;
  sy_noexec    := true;
  idm_space    := false;
  idm_outmacro := true;
  with sy_sym, src_control^ do
  if sy <> eofsy then
  begin
    src_insnb := src_insnb + 1;
    if sy_ch > ' ' then GEN_MACRO_CODE;
    SKIP_SYMBOL( mcmd );
    if mcmd = untilsy then
    begin
       INSYMBOL;
       iv := GET_INTEXPR( 1 )
    end;
    { We create a length adapted macro code array }
    if idm_newmac <> nil then
    with idm_newmac^ do
    if idm_use > 0 then
    begin
      NEW( p, idm_use );
      p^.idm_use := idm_use;
      for i := 1 to idm_use do  p^.idm_ctb[i] := idm_ctb[i];
      idm_use := 0
    end
    else p := nil;
    src_insnb := src_insnb - 1
  end;
  idm_outmacro := false;
  sy_noexec := sv_noexec;
  GEN_MACRO_LIST := p
end GEN_MACRO_LIST;



[global]
function NEW_MACRO_LIST( mcmd: symbol ): idm_ptr;
{ Create a unnamed macro list }
{ The created macro parameter is not put in the defined stack }
var
  p: idm_ptr;

begin
  NEW( p );
  with p^ do
  begin
    idm_name := nil;
    idm_parl := nil;
    idm_cntx := nil;
    idm_nxt  := nil;
    idm_kind := idm_list;
    idm_run  := false;
    idm_tab  := GEN_MACRO_LIST( mcmd )
  end;
  NEW_MACRO_LIST := p
end NEW_MACRO_LIST;





     {************************************************}
     { EXECUTION MANAGER                     Routines }
     {************************************************}



procedure PRAGMA_STATE( p_src: src_ptr );
const
  mdnam = 'PRAG';

type
  optionsty = ( opt_liston,   opt_listoff,
                opt_echoon,   opt_echooff,
                opt_listlvl,  opt_format,
                opt_sh_sym,   opt_sh_exp,
                opt_sh_mac,   opt_macsrc,
                opt_sh_dat
              );

  optid = record
    len: byte;
    nam: optid_name
  end;

var
  { warning this table must be modified when the identifier size is changed }
  opttab: [static] array[optionsty] of optid := (
  ( 7, 'list_on        '),  { List on }
  ( 8, 'list_off       '),  { List off }
  ( 7, 'echo_on        '),  { Echo on }
  ( 8, 'echo_off       '),  { No echo }
  ( 7, 'listlvl        '),  { Listing source level }
  ( 6, 'format         '),  { Listing source format }
  (11, 'show_syntax    '),  { Listing of readen syntax unit by INSYMBOL }
  ( 9, 'show_expr      '),  { Listing of readen syntax unit by INSYMBOL }
  (10, 'show_macro     '),  { Listing of macro expension }
  (13, 'show_macrosrc  '),  { Listing of macro source at macro creation }
  ( 9, 'show_data      ')   { Listing of macro expension }
  );

  i, ipa, iln, ierr:    integer;
  plist:                lst_ptr := nil;
  save_status:          src_flagw;
  bf:                   boolean;
  option:               optionsty;
  fspc:                 str_string;

  procedure PRAGMA_OPT_FLAG( var bflg: boolean );
  begin
    with sy_sym, p_src^ do
      if sy = colon then
      begin  INSYMBOL; bflg := (GET_INTEXPR( 1 ) > 0)  end
      else bflg := not bflg;
  end PRAGMA_OPT_FLAG;


begin { PRAGMA_STATE }
  with sy_sym, p_src^ do
  begin
    sy := comma;              { Simule a comma }
    save_status := src_flags;
    while sy = comma do
    begin
      INSYMBOL;
      if (sy <> identsy) and
         (sy <> specific0sy) and
         (sy <> specific1sy) then SRC_ERROR( mdnam, 201, e_severe )
      else
      begin
        bf := false;
        with sy_ident do
        for popt := optionsty"first to optionsty"last do
          with opttab[popt] do
          begin
            option := popt;
            bf := (STR_MATCH( sy_ident, 0, nam, len ) = 0);
        exit if bf
          end;

        INSYMBOL;                     { gobble up the option name }
        if bf then
        case option of
          opt_liston:
            begin { List_on }
              if sy = colon then
              with lst_current^ do
              begin
                INSYMBOL;
                GET_STREXPR( fspc );
                ipa := -1; iln := -1;
                if sy = colon then
                begin
                  INSYMBOL;
                  ipa := GET_INTEXPR( ipa );
                  if sy = colon then
                  begin
                    INSYMBOL;
                    iln := GET_INTEXPR( iln )
                  end
                end;
                { flush output if some output line is begining }
                if lst_currline^.length > 0 then LST_EOLN;
                LST_OPEN( lst_current, fspc, lst_heading^,
                          lst_title^, lst_sbttl^,
                          iln,        { 80/132 for tty/file in char./line }
                          ipa,        { tty/no => dis./ena. page managment }
                          false,      { new version mode }
                          false,      { no print on close time }
                          false,      { no append }
                          ierr );

                if ierr <> 0 then
                  if not sy_init_mod then
                    { if error lst -> terminal }
                    SRC_ERROR( mdnam, 211, e_error )
                  else
                    save_status := save_status - [src_blist]
                else
                begin
                  if src_lstmxlev < src_level then src_lstmxlev := src_level;
                  save_status := save_status + [src_blist]
                end
              end
              else
              begin
                if src_lstmxlev < src_level then src_lstmxlev := src_level;
                save_status := save_status + [src_blist]
              end
            end;

          opt_listoff: save_status := save_status - [src_blist]; { List_off }
          opt_echoon:  save_status := save_status + [src_becho]; { Echo_on }
          opt_echooff: save_status := save_status - [src_becho]; { Echo_off }

          opt_listlvl: if sy = colon then
                       begin { Listlvl }
                         INSYMBOL;
                         src_lstmxlev := GET_INTEXPR( 1 )
                       end else src_lstmxlev := 1;

          opt_format: { Format }
              if sy = colon then
              begin
                INSYMBOL;
                src_frspos := GET_INTEXPR( 1 );
                if sy = colon then
                begin
                  INSYMBOL;
                  src_lstpos := GET_INTEXPR( str_maxsize )
                end
              end;

          opt_sh_sym: PRAGMA_OPT_FLAG( debug_sym ); { INSYMBOL Debug }
          opt_sh_exp: PRAGMA_OPT_FLAG( debug_exp ); { EXPRESSION Debug }
          opt_sh_mac: PRAGMA_OPT_FLAG( debug_mac ); { Macro Debug }
          opt_sh_dat: PRAGMA_OPT_FLAG( debug_dat ); { Macro Data Debug }
          opt_macsrc: PRAGMA_OPT_FLAG( debug_macsrc ); { Macro source debug }

        otherwise { no legal option }
        end { case }
        else
          SRC_ERROR( mdnam, 202, e_warning)
      end
    end;
    src_flags := save_status
  end
end PRAGMA_STATE;



[global]
procedure INCLUDE_STATE( bincl: boolean );
const
  mdnam = 'INCL';

var
  src_p:       src_ptr;
  fname, fext: str_string;
  i, ierr:     integer;
  fspc:        str_string;

begin { INCLUDE_STATE }
  INSYMBOL;
  with sy_sym do
  begin
    GET_STREXPR( fspc );
    { By default of file type we set the ".cpsh" file type }
    if fspc.length > 0 then
    begin
      i := INDEX( fspc, '.', -1 );
      if i = 0 then fspc := fspc || '.cpsh'
    end;
    src_p := SRC_FILE_OPEN( fspc );     { Open this file }
    io_err^.ide_int := src_openerr;
    if src_openerr = 0 { No open error } then
    begin
      with src_p^ do
      begin
        src_previous := src_control;    { link the file to the previous src. }
        src_level := src_control^.src_level;
        if bincl then src_level := src_level + 1;
        src_lstmxlev := src_control^.src_lstmxlev;
        src_insnb := src_control^.src_insnb;
        src_frspos := src_control^.src_frspos;
        src_lstpos := src_control^.src_lstpos;
        src_flags := src_flags +
          src_control^.src_flags * [src_blist,src_bmacroex,src_bphys,
                                    src_becho,src_echerr]
      end;
      if sy = comma then PRAGMA_STATE( src_p ); { look for %PRAGMA options }

      if bincl then SAVE_SYM_CNTX( true )
               else SRC_END_OF_LINE;

      src_control := src_p; { switch to new source file }
      LISTING_SET_TITLE;
      sy_ch := ' ';         { init the read character }
      sy    := semicolon;   { force to unsignificant value }
      if not bincl then     { chaine macro STATEMENT }
      begin
        with src_control^ do
        begin
          src_p := src_previous;
          src_previous := src_p^.src_previous { get the true previous src}
        end;
        SRC_CLOSE( src_p, true ) { elliminate the old source file context }
      end
    end else
    begin
      if not sy_init_mod then
        SRC_ERROR( mdnam, 212, e_severe ); { cannot open the file }
      SKIP_SYMBOL( semicolon )
    end
  end
end INCLUDE_STATE;


[global]
procedure ENDFILE_STATE;
var
  src_p: src_ptr;

begin { ENDFILE_STATE }
  with src_control^ do
  begin
    if src_previous <> nil then
    begin { return to a previous source file }
      src_p := src_control;
      src_control := src_p^.src_previous;
      SRC_CLOSE( src_p, true );  { close end elliminate old src context }
      RESTORE_SYM_CNTX           { Restore the context with symbol in saved varbl. }
    end
    else
      sy_sym.sy := eofsy { It is the external end of file }
  end;
  LISTING_SET_TITLE
end ENDFILE_STATE;




[global]
procedure EXECUTE_MACRO_CODE( pmc: idm_ptr; stopper: symbol );
var
  s: sym_rec;

begin
  if pmc <> nil then
  begin
    s := sy_sym;
    ACTIVE_MACRO_CODE( pmc );
    STATELIST( endsy );
    RET_OF_MACRO_CODE( pmc );
    sy_sym := s;
  end
end EXECUTE_MACRO_CODE;




procedure REPEAT_STATE;
const
  mdnam = 'REAP';

var
  pr: idm_rec;
  ir: integer;
  c1, c2: char;

begin
  with sy_sym, pr do
  begin
    idm_name := nil;
    idm_parl := nil;
    idm_nxt  := nil;
    idm_prv  := nil;
    idm_kind := idm_list;
    idm_run  := false;
    sy       := nothing; { To force to ignore the present repeat }
    idm_tab  := GEN_MACRO_LIST( untilsy );
    idm_cntx := idm_actstk;
    c1 := sy_ch; c2 := sy_cmin;
    if sy <> eofsy then
    begin
      repeat
        idm_actstk := pr"address;
        idm_run    := true;
        sy_ch := ' ';
        idm_nch := 1; { Exec from the begining }
        STATELIST( untilsy );
        INSYMBOL;
        ir := GET_INTEXPR( 1 )
      until ir > 0
    end
    else
      SRC_ERROR( mdnam, 303, e_severe );
    { Restore the previous context }
    sy_ch := c1; sy_cmin := c2;
    sy := semicolon;
    idm_actstk := idm_cntx;
    DISPOSE( idm_tab )
  end
end REPEAT_STATE;



procedure WHILE_STATE;
const
  mdnam = 'WHIL';

var
  pr:     idm_rec;
  ir:     integer;
  c1, c2: char;
  wex:    idm_ptr;

begin
  with sy_sym, pr do
  begin
    wex := NEW_MACRO_EXPR;
    if sy <> dosy then SRC_ERROR( mdnam, 120, e_error );
    idm_name := nil;
    idm_parl := nil;
    idm_nxt  := nil;
    idm_prv  := nil;
    idm_run  := false;
    idm_kind := idm_list;
    idm_tab  := GEN_MACRO_LIST( endsy );
    idm_cntx := idm_actstk;
    c1 := sy_ch; c2 := sy_cmin;
    if sy <> eofsy then
      while GET_INT_VALUE( wex ) > 0 do
      begin
        idm_actstk := pr"address;
        idm_run    := true;
        sy_ch := ' ';
        idm_nch := 1; { Exec from the begining }
        STATELIST( endsy );
        INSYMBOL
      end
    else
      SRC_ERROR( mdnam, 303, e_severe );
    { Restore the previous context }
    idm_actstk := idm_cntx;
    sy_ch := c1; sy_cmin := c2;
    INSYMBOL;
    DISPOSE( idm_tab );
    PURGE_MACRO_EXPR( wex )
  end
end WHILE_STATE;



procedure FOR_STATE;
const
  mdnam = 'FORL';

var
  pr:         idm_rec;
  ir, id, ie: integer;
  c1, c2:     char;
  ip:         ide_ptr;

begin
  INSYMBOL;
  with sy_sym, pr do
  if sy = identsy then
  begin
    DISPLAY_NEW;
    ip := IDE_NEW( ide_parint, curr_disp );
    if ip <> nil then
    with ip^ do
    begin
      INSYMBOL;
      if sy = becomes then INSYMBOL
                      else SRC_ERROR( mdnam, 32, e_error );
      id := GET_INTEXPR( 1 );
      case sy of
            tosy: begin  ir :=  1; INSYMBOL  end;
        downtosy: begin  ir := -1; INSYMBOL  end;
      otherwise
        SRC_ERROR( mdnam, 119, e_error )
      end;
      ie := GET_INTEXPR( id );
      if sy <> dosy then SRC_ERROR( mdnam, 120, e_error );
      if ir > 0 then ie := ie - id + 1
                else ie := id - ie + 1;
      ide_int := id;

      idm_name := nil;
      idm_parl := nil;
      idm_nxt  := nil;
      idm_prv  := nil;
      idm_run  := false;
      idm_kind := idm_list;
      idm_tab  := GEN_MACRO_LIST( endsy );
      idm_cntx := idm_actstk;
      c1 := sy_ch; c2 := sy_cmin;
      if sy <> eofsy then
      for ii := 1 to ie do
      begin
        idm_actstk := pr"address;
        idm_run    := true;
        sy_ch := ' ';
        idm_nch := 1; { Exec from the begining }
        STATELIST( endsy );
        ide_int := ide_int + ir
      end
      else
        SRC_ERROR( mdnam, 303, e_severe );
      { Restore the previous context }
      idm_actstk := idm_cntx;
      sy_ch := c1; sy_cmin := c2;
      INSYMBOL;
      DISPOSE( idm_tab )
    end
    else
    begin
      SKIP_SYMBOL( endsy );
      INSYMBOL
    end;
    DISPLAY_FREE
  end
  else
  begin
    SRC_ERROR( mdnam, 102, e_severe );
    SKIP_SYMBOL( endsy );
    INSYMBOL
  end
end FOR_STATE;



procedure MACRO_STATE;
const
  mdnam = 'MACR';

var
  pm, pp, psv, ph, pl: idm_ptr;
i: integer := 0;
begin
  with sy_sym do
  begin
    INSYMBOL;
    if sy <> identsy then
    begin
      SRC_ERROR( mdnam, 304, e_severe );
      SKIP_SYMBOL( endsy ); INSYMBOL
    end
    else
    begin
      pm := SEARCH_MACRO( idm_macro );
      if pm <> nil then
      begin { Cannot create two macros with the same name }
        SRC_ERROR_S( mdnam, 305, e_severe, sy_ident );
        SKIP_SYMBOL( endsy ); INSYMBOL;
        pm := nil
      end
      else
      begin
        NEW( pm );
        with pm^ do
        begin
          NEW( idm_name, sy_ident.length );
          idm_name^ := sy_ident;
          idm_cntx  := nil;
          idm_nxt   := nil;
          idm_prv   := idm_defstk;
          idm_run   := false;
          idm_kind  := idm_macro;
          idm_parl  := nil;
          idm_nch   := 0;
          INSYMBOL; { Skip in parameter list }
          { Mask all present macro definition }
          ph := nil;
          pl := nil;
          idm_defstk := nil;
          while sy = identsy do
          begin
            pp := SEARCH_MACRO( idm_parm );
            if pp = nil then
            begin
              NEW( pp );
              if pl = nil then ph := pp
                          else pl^.idm_parl := pp;
              with pp^ do
              begin
                NEW( idm_name, sy_ident.length );
                idm_name^ := sy_ident;
                idm_parl  := nil;
                idm_cntx  := nil;
                idm_nxt   := nil;
                idm_prv   :=  pl;
                idm_run   := false;
                idm_kind  := idm_parm;
                idm_nch   :=   0;
                idm_tab   := nil
              end;
              pl := pp
            end
            else SRC_ERROR_S( mdnam, 306, e_severe, sy_ident );
            INSYMBOL;
          exit if sy <> comma;
            INSYMBOL
          end;
          if sy <> semicolon then  SRC_ERROR( mdnam, 21, e_error );
          idm_parl := ph;        { Attach the parameter list to the macro bloc }
          idm_defstk := idm_prv; { Restore the previous macro def. context }
          { Get the macro text }
          idm_tab := GEN_MACRO_LIST( endsy );
          { Set the macro in the macro def. symbol list }
          if idm_defstk <> nil then idm_defstk^.idm_nxt := pm;
          idm_defstk := pm;
          INSYMBOL
        end
      end
    end
  end
end MACRO_STATE;



procedure PURGE_STATE;
const
  mdnam = 'PRGM';

var
  pm, pp: idm_ptr;

begin
  with sy_sym do
  loop
    INSYMBOL;
  exit if (sy = semicolon) or (sy = eofsy) or
          (sy = endsy) or (sy = untilsy);
    if sy = identsy then
    begin
      pm := SEARCH_MACRO( idm_macro );
      if pm = nil then SRC_ERROR_S( mdnam, 307, e_error, sy_ident )
      else
      begin
        with pm^ do
        { Check of do not purge an active macro }
        if idm_run then SRC_ERROR_S( mdnam, 308, e_severe, sy_ident )
        else
        begin
          { Supress the macro of the macro definition list }
          if idm_nxt = nil then idm_defstk := idm_prv        { Last in the macro list }
                           else idm_nxt^.idm_prv := idm_prv; { not first in the macro list }
          if idm_prv <> nil then idm_prv^.idm_nxt := idm_nxt;

          { Free the identifier name }
          DISPOSE( idm_name );
          while idm_parl <> nil do
          begin
            pp := idm_parl;
            idm_parl := idm_parl^.idm_parl;
            with pp^ do
            begin
              DISPOSE( idm_name );
              if idm_tab <> nil then DISPOSE( idm_tab )
            end;
            DISPOSE( pp )
          end;
          { Free the code }
          if idm_tab <> nil then DISPOSE( idm_tab )
        end;
        DISPOSE( pm )
      end
    end
    else SRC_ERROR( mdnam, 309, e_error );
    INSYMBOL;
    if sy <> comma then SRC_ERROR( mdnam, 37, e_error )
  end
end PURGE_STATE;



procedure CALL_MACRO;
const
  mdnam = 'CALM';

var
  pm, pp, pa, pc: idm_ptr;
  svch, svcm: char;

begin
  with sy_sym, lst_current^ do
  begin
    pm := sy_macro;
    with pm^ do
    begin { It is a defined macro }
      { Set all formal of macro as defined with the given effective values }
      pp := idm_parl;
      pa := nil;
      while pp <> nil do
      with pp^ do
      begin
        { Get the effective parameter value }
        GENERATE_MACRO_PARM( pp, false, ' ' );
        if debug_macsrc then
        with pp^do
        begin
          WRITELN( lst_file, ' Macro Parm "', idm_name^, '" :' );
          WRITE( lst_file, ' ' );
          if idm_tab <> nil then
          with idm_tab^do
            for ij := 1 to idm_use do WRITE( lst_file, idm_ctb[ij] )
          else WRITE( lst_file, '.nil.') ;
          WRITELN( lst_file )
        end;
        pa := pp;
      exit if sy <> comma;
        pp := idm_parl
      end;
      { Link the parameter string to the pmacro stack }
      pp := idm_parl;
      if pp <> nil then
      begin
        idm_defstk^.idm_nxt := pp;
        pp^.idm_prv := idm_defstk;
        idm_defstk  := pa
      end;
      if debug_macsrc then
      begin
        WRITELN( lst_file, ' Macro Stack include :' );
        pc := idm_defstk;
        while pc <> nil do
        begin
          WRITELN( lst_file, ' ::: "', pc^.idm_name^, '",' );
          pc := pc^.idm_prv
        end;
        WRITELN( lst_file, ' Macro Code of "', idm_name^, '" :' );
        WRITE( lst_file, ' "' );
        with idm_tab^do
          for ij := 1 to idm_use do WRITE( lst_file, idm_ctb[ij] );
        WRITELN( lst_file, '"' )
      end;

      ACTIVE_MACRO_CODE( pm, true );
      STATELIST( endsy );
      RET_OF_MACRO_CODE( pm );

      { Unlink the macro parameters }
      if pp <> nil then
      with pp^ do
      begin
        idm_prv^.idm_nxt := pa^.idm_nxt;
        if pa^.idm_nxt <> nil then pa^.idm_nxt^.idm_prv := idm_prv
                              else idm_defstk := idm_prv
      end;
      while pp <> nil do
      with pp^ do
      begin
        if idm_tab <> nil then begin
                                 DISPOSE( idm_tab );
                                 idm_tab := nil
                               end;
        idm_nch  := 0;
        pp := idm_parl
      end;
    end
  end
end CALL_MACRO;



procedure CREATE_NEW_ARR( var p: idedim_ptr; var sz: integer );
{ procedure to create array dimension structures }
const
 mdnam = 'ARCR';

begin
  INSYMBOL;                           { Gobble up "," or "[" }
  NEW( p );
  with sy_sym, p^ do
  begin
    idedim_siz := GET_INTEXPR( 0 );   { Get the required array dimension }
    if idedim_siz <= 0 then
    begin  SRC_ERROR( mdnam, 75, e_severe ); idedim_siz := 1  end;
    if sy = comma then CREATE_NEW_ARR( idedim_nxt, idedim_stp )
                  else begin
                         idedim_nxt := nil;
                         idedim_stp :=   1
                       end;
    sz := idedim_stp*idedim_siz
  end
end CREATE_NEW_ARR;


procedure DECLARE_NEW_ENNUMS;
const
  mdnam = 'DCLE';

var
  ilvl, icnt:      integer;
  p, p1:           ide_ptr;


begin
  with sy_sym do
  begin
    INSYMBOL;                           { Gobble up the enum keyword }
    if sy = intconst then
    begin
      ilvl := sy_ival;
      if ilvl <= 0 then ilvl := 1
                   else if ilvl > curr_disp then ilvl := curr_disp;
      INSYMBOL
    end else ilvl := curr_disp;
    if sy <> lparen then SRC_ERROR( mdnam, 22, e_error );
    icnt := 0;                          { Initialize the value counter }
    repeat
      INSYMBOL;                         { Gobble up the separator "(" or "," }
      if sy = identsy then
      begin
        p := IDE_NEW(ide_parint,ilvl ); { Create the new integer identifier }
        INSYMBOL;                       { Gobble up the identifier }
        with p^ do
        begin
          if sy = becomes then
          begin
            INSYMBOL;                   { Gobble up ":=" }
            icnt := GET_INTEXPR(icnt);  { Get the user specified value }
          end;
          ide_int    :=   icnt;         { Set the indentifier value }
          ide_ronly  :=   true;         { Set the read only characteristic of the identifier }
          icnt := icnt + 1              { Increment the default indentifier value }
        end
      end
      else
      begin
        SRC_ERROR( mdnam, 102, e_severe );
        INSYMBOL
      end
    until sy <> comma;
    if sy <> rparen then SRC_ERROR( mdnam, 22, e_error )
  end
end DECLARE_NEW_ENNUMS;



procedure DECLARE_NEW_IDENT( sym: symbol );
{ Create a new identifier when specified }
const
  mdnam = 'DCLI';

var
  k:               ide_kinds;
  p, p1:           ide_ptr;
  pdm:             idedim_ptr;
  s:               str_string;
  ival,
  ilvl, isz, idim: integer;
  rval:            shreal;

begin
  case sym of
    enumsy,
    intsy:    k := ide_parint;
    floatsy:  k := ide_parflt;
    stringsy: k := ide_parstr;
  otherwise
  end;
  with sy_sym do
  begin
    repeat
      INSYMBOL;                         { Gobble up the separator or keyword }
      if sy = intconst then
      begin
        ilvl := sy_ival;
        if ilvl <= 0 then ilvl := 1
                     else if ilvl > curr_disp then ilvl := curr_disp;
        INSYMBOL
      end else ilvl := curr_disp;
      if sy = identsy then
      begin
        p := IDE_NEW( k, ilvl );        { Create the new identifier }
        INSYMBOL;                       { Gobble up the identifier }
        if sy = lbrack then
        begin
          CREATE_NEW_ARR( pdm, isz );   { Create the array description records}
          with p^ do
            case k of
              ide_parstr: begin
                            ide_kind := ide_tabstr;
                            ide_adm  := pdm;
                            NEW( ide_aas, isz );
                            with ide_aas^ do
                            for ii := 1 to isz do  ide_stb[ii] := nil
                          end;
              ide_parint: begin
                            ide_kind := ide_tabint;
                            ide_adm  := pdm;
                            NEW( ide_aai, isz );
                            with ide_aai^ do
                            for ii := 1 to isz do  ide_itb[ii] := 0
                          end;
              ide_parflt: begin
                            ide_kind := ide_tabflt;
                            ide_adm  := pdm;
                            NEW( ide_aaf, isz );
                            with ide_aaf^ do
                            for ii := 1 to isz do  ide_ftb[ii] := 0.0
                          end;
            otherwise
            end;
          if sy = rbrack then INSYMBOL
                         else SRC_ERROR( mdnam, 26, e_error )
        end;
        if sy = becomes then            { An initial value can be specified }
        with p^ do
        begin
          INSYMBOL;
          case ide_kind of
            ide_parint: ide_int := GET_INTEXPR( 0 );
            ide_parflt: ide_flt := GET_FLTEXPR( 0.0 );
            ide_parstr: begin
                          GET_STREXPR( s );
                          if s.length > 0 then
                          begin  NEW( ide_str, s.length ); ide_str^ := s  end
                          else ide_str := nil
                        end;
            ide_tabint: with ide_aai^ do
                        if sy = lparen then
                        begin
                          sy := comma;
                          for ii := 1 to ide_all do
                          begin
                            if sy = comma then INSYMBOL
                                          else SRC_ERROR( mdnam, 29, e_error );
                            ide_itb[ii] := GET_INTEXPR( 0 )
                          end;
                          if  sy = rparen then INSYMBOL
                                          else SRC_ERROR( mdnam, 23, e_error )
                        end
                        else
                        begin
                          ival := GET_INTEXPR( 0 );
                          for ii := 1 to ide_all do ide_itb[ii] := ival
                        end;
            ide_tabflt: with ide_aaf^ do
                        if sy = lparen then
                        begin
                          sy := comma;
                          for ii := 1 to ide_all do
                          begin
                            if sy = comma then INSYMBOL
                                          else SRC_ERROR( mdnam, 29, e_error );
                            ide_ftb[ii] := GET_FLTEXPR( 0.0 )
                          end;
                          if sy = rparen then INSYMBOL
                                         else SRC_ERROR( mdnam, 23, e_error )
                        end
                        else
                        begin
                          rval := GET_FLTEXPR( 0.0 );
                          for ii := 1 to ide_all do ide_ftb[ii] := rval
                        end;
            ide_tabstr: with ide_aas^ do
                        if sy = lparen then
                        begin
                          sy := comma;
                          for ii := 1 to ide_all do
                          begin
                            if sy = comma then INSYMBOL
                                          else SRC_ERROR( mdnam, 29, e_error );
                            GET_STREXPR( s );
                            if s.length > 0 then
                            begin
                              NEW( ide_stb[ii], s.length );
                              ide_stb[ii]^ := s
                            end
                          end;
                          if  sy = rparen then INSYMBOL
                                          else SRC_ERROR( mdnam, 23, e_error )
                        end
                        else
                        begin
                          GET_STREXPR( s );
                          if s.length > 0 then
                            for ii := 1 to ide_all do
                            begin
                              NEW( ide_stb[ii], s.length );
                              ide_stb[ii]^ := s
                            end
                        end;
          otherwise
          end
        end
      end
      else
      begin
        SRC_ERROR( mdnam, 102, e_severe );
        INSYMBOL
      end
    until sy <> comma;
  end
end DECLARE_NEW_IDENT;



procedure ASSIGNEMENT;
const
  mdnam = 'ASSI';

var
  st:                  str_string;
  ind, iv, iw, fr, ls: integer;
  vl:                  shreal;
  rec:                 exp_rec;

begin
  GET_EXP_REFER( rec );
  with sy_sym, rec do
  if (exp_ref <> nil) and (exp_kind <> exp_valnull) then
  begin
    fr := exp_shift;
    if exp_adm <> nil then
      with exp_adm^ do
        ls := idedim_stp*idedim_siz + fr - 1
    else ls := fr;

    if sy = becomes then INSYMBOL
                    else SRC_ERROR( mdnam, 32, e_error );

    with exp_ref^ do
    case exp_kind of
      exp_valint: begin
                    iv := GET_INTEXPR( ide_int );
                    if ide_ronly then SRC_ERROR_S( mdnam, 131, e_error, ide_name^ )
                                 else ide_int := iv
                  end;
      exp_valflt: begin
                    vl := GET_FLTEXPR( ide_flt );
                    if ide_ronly then SRC_ERROR_S( mdnam, 131, e_error, ide_name^ )
                                 else ide_flt := vl
                  end;
      exp_valstr: begin
                    GET_STREXPR( st );
                    if ide_ronly then SRC_ERROR_S( mdnam, 131, e_error, ide_name^ )
                    else
                    begin
                      if ide_str <> nil then DISPOSE( ide_str );
                      if st.length > 0 then
                      begin
                        NEW( ide_str, st.length );
                        ide_str^ := st
                      end else ide_str := nil
                    end
                  end;

      exp_telint: if exp_aai <> nil then
                    exp_aai^.ide_itb[exp_shift] := GET_INTEXPR( 0 );
      exp_telflt: if exp_aaf <> nil then
                    exp_aaf^.ide_ftb[exp_shift] := GET_FLTEXPR( 0.0 );
      exp_telstr: if exp_aas <> nil then
                  with exp_aas^ do
                  begin
                    GET_STREXPR( st );
                    if ide_stb[exp_shift] <> nil then
                      DISPOSE( ide_stb[exp_shift] );
                    if st.length > 0 then
                    begin
                      NEW( ide_stb[exp_shift], st.length );
                      ide_stb[exp_shift]^ := st
                    end else ide_stb[exp_shift] := nil
                  end;
      exp_tabint: begin
                    GET_EXPRESSION;
                    case exp_res.exp_kind of
                      exp_valint:
                        ARRSCA_II_OPE( ar_mov, exp_res.exp_int, rec );
                      exp_valflt:
                        ARRSCA_II_OPE( ar_mov, ROUND( exp_res.exp_flt ), rec );
                      exp_tabint: ARRAY_II_OPE( ar_mov, exp_res, rec );
                      exp_tabflt: ARRAY_IF_OPE( ar_mov, exp_res, rec );
                    otherwise
                      SRC_ERROR( mdnam, 116, e_severe )
                    end
                  end;
      exp_tabflt: begin
                    GET_EXPRESSION;
                    case exp_res.exp_kind of
                      exp_valint:
                        ARRSCA_FF_OPE( ar_mov, shreal( exp_res.exp_int ), rec );
                      exp_valflt:
                        ARRSCA_FF_OPE( ar_mov, exp_res.exp_flt, rec );
                      exp_tabint: ARRAY_FI_OPE( ar_mov, exp_res, rec );
                      exp_tabflt: ARRAY_FF_OPE( ar_mov, exp_res, rec );
                    otherwise
                      SRC_ERROR( mdnam, 117, e_severe )
                    end
                  end;
      exp_tabstr: begin
                    GET_EXPRESSION;
                    case exp_res.exp_kind of
                      exp_valstr:
                        ARRSCA_SS_OPE( ar_mov, exp_res, rec );
                      exp_tabstr:
                        ARRAY_SS_OPE( ar_mov, exp_res, rec );
                    otherwise
                      SRC_ERROR( mdnam, 118, e_severe )
                    end
                  end;

    otherwise
      SRC_ERROR_S( mdnam, 105, e_severe, sy_ident );
      INSYMBOL
    end
  end
end ASSIGNEMENT;



procedure OPENFILE_STATE( sym: symbol );
{ To open a text read/write file }
const
  mdnam = 'OPEN';

var
  lun: integer;
  fspc:  str_string;

begin
  with sy_sym do
  begin
    INSYMBOL;
    lun := GET_INTEXPR( 1 );
    if (lun < 0) or (lun > max_lun) then
    begin
      SRC_ERROR( mdnam, 77, e_severe );  SKIP_SYMBOL( semicolon )
    end
    else
    with sym_iof[lun] do
    begin
      if sy = colon then INSYMBOL
                    else SRC_ERROR( mdnam, 31, e_error );
      GET_STREXPR( fspc );
      if iocnt_mode <> iocnt_close then CLOSE( iocnt_file );
      case sym of
        open_inpsy: begin
                     OPEN( iocnt_file, fspc, [read_file,error_file,case_ena_file] );
                     if iostatus = 0 then iocnt_mode := iocnt_input
                                     else iocnt_mode := iocnt_close;

                   end;
        open_outsy,
        open_appsy: begin
                     if sym = open_appsy then
                       OPEN( iocnt_file, fspc, [append_file,error_file,case_ena_file] )
                     else
                       OPEN( iocnt_file, fspc, [write_file,error_file,case_ena_file] );
                     if iostatus = 0 then iocnt_mode := iocnt_output
                                     else iocnt_mode := iocnt_close
                   end
      otherwise
      end
    end
  end;
  io_err^.ide_int := ORD( iostatus <> 0 )
end OPENFILE_STATE;



procedure CLOSEFILE_STATE;
{ to close a text read/write file }
var
  lun: integer;

begin
  INSYMBOL;
  lun := GET_INTEXPR( 1 );
  if (lun >= 0) and (lun <= max_lun) then
    with sym_iof[lun] do
    begin
      if iocnt_mode <> iocnt_close then CLOSE( sym_iof[lun].iocnt_file );
      iocnt_mode := iocnt_close
    end
    else SRC_ERROR( 'CLOS', 77, e_severe )
end CLOSEFILE_STATE;



procedure GET_IO_FORMAT( nb: integer; var i1, i2, i3: integer );
begin
  with sy_sym do
  if sy = colon then
  begin
    INSYMBOL;
    i1 := GET_INTEXPR( i1 );
    if (nb > 1) and (sy = colon) then
    begin
      INSYMBOL;
      i2 := GET_INTEXPR( i2 );
      if (nb > 2) and (sy = colon) then
      begin
        INSYMBOL;
        i3 := GET_INTEXPR( i3 )
      end
    end
  end
end GET_IO_FORMAT;



procedure READ_VALUES( sym: symbol );
{ To get a variable from terminal or input file : string or number(s) }
const
  mdnam = 'RPLY';

var
  st:                   str_string;
  lun, nel, fs, dc, ls: integer;
  rec:                  exp_rec;
  fv:                   shreal;
  iv:                   integer;
  sv:                   pstring;

  function FREAD_INT( var f: text; fl: integer ): integer;
  var
    iv: integer;

  begin
    iv := 0;
    if EOLN( f ) and not EOF( f ) then READLN( f );
    if not EOF( f ) then READ( f, iv:fl );
    if EOF( f ) then
    begin
      io_eof^.ide_int  := 1;
      io_eoln^.ide_int := 1
    end
    else
    begin
      io_eof^.ide_int := 0;
      io_eoln^.ide_int := ORD( EOLN( f ) );
      TTY_CLR_EOF( f )
    end;
    FREAD_INT := iv
  end FREAD_INT;


  function FREAD_FLT( var f: text; fl: integer ): shreal;
  var
    fv: shreal;

  begin
    fv := 0.0;
    if EOLN( f ) and not EOF( f ) then READLN( f );
    if not EOF( f ) then READ( f, fv:fl );
    if EOF( f ) then
    begin
      io_eof^.ide_int  := 1;
      io_eoln^.ide_int := 1
    end
    else
    begin
      io_eof^.ide_int := 0;
      io_eoln^.ide_int := ORD( EOLN( f ) );
      TTY_CLR_EOF( f )
    end;
    FREAD_FLT := fv
  end FREAD_FLT;


  procedure FREAD_STR( var f: text; var sv: pstring; fl, dc: integer );
  const
    TAB = CHR( 9 );

  var
    st: str_string;
    il: integer;
    beof, beoln: boolean;
    ch: char;

  begin
    st.length := 0;
    if not UFB( f ) then
      if EOLN( f ) and not EOF( f ) then READLN( f );
    if not (EOLN( f ) or EOF( f )) then
    begin
      if sv <> nil then DISPOSE( sv );
      READ( f, st:fl:(dc > 0) );
      { size the string by supress any trailing space(s) }
      for ii := st.length downto 1 do
        if st[ii] > ' ' then
        begin  st.length := ii; exit  end;
      if st.length > 0 then
      { Now suppress any control character except space and TAB }
      il := 0;
      for ii := 1 to st.length do
        if (st[ii] >= ' ') or (st[ii] = TAB) then
        begin  il := il + 1; st[il] := st[ii]  end;
      st.length := il;
      { Create the final string }
      if st.length > 0 then
      begin
        NEW( sv, il );
        sv^ := st
      end else sv := nil;
    end;
    if EOF( f ) then
    begin
      io_eof^.ide_int  := 1;
      io_eoln^.ide_int := 1;
      if TTY_FILE( f ) then TTY_CLR_EOF( f )
    end
    else
    begin
      io_eof^.ide_int := 0;
      io_eoln^.ide_int := ORD( EOLN( f ) );
    end
 end FREAD_STR;


begin { READ_VALUES }
  lun := 1;
  INSYMBOL;
  if sym = readsy then
  begin
    lun := GET_INTEXPR( lun );
    if (lun < 0) or (lun > max_lun) then
    begin
      SRC_ERROR( mdnam, 77, e_severe );
      lun := 1
    end
    else
      if sym_iof[lun].iocnt_mode <> iocnt_input then
      begin
        SRC_ERROR( mdnam, 401, e_severe );
        lun := 1
      end;
    if sy_sym.sy <> colon then SRC_ERROR( mdnam, 31, e_error )
                          else INSYMBOL
  end;
  if lun >= 0 then
  with sy_sym, sym_iof[lun] do
  begin
    if op <> stp_op then
    loop
      GET_EXP_REFER( rec );       { Get the reference to read }
      fs := 0; dc := 0; ls := 0;
      GET_IO_FORMAT( 2, fs, dc, ls );
      with rec do
      begin
        nel := exp_shift;
        ls  := exp_edim + exp_shift;
        if exp_ref <> nil then
        with exp_ref^ do
        case exp_kind of
          exp_valint:
            begin
              if sym = readsy then iv := FREAD_INT( iocnt_file, fs )
                              else iv := FREAD_INT( input, fs );
              if ide_ronly then SRC_ERROR_S( mdnam, 131, e_error, ide_name^ )
                           else ide_int := iv
            end;
          exp_valflt:
            begin
              if sym = readsy then fv := FREAD_FLT( iocnt_file, fs )
                              else fv := FREAD_FLT( input, fs );
              if ide_ronly then SRC_ERROR_S( mdnam, 131, e_error, ide_name^ )
                           else ide_flt := fv
            end;
          exp_valstr:
            if ide_ronly then
            begin
              sv := nil;
              if sym = readsy then FREAD_STR( iocnt_file, sv, fs, dc )
                              else FREAD_STR( input, sv, fs, dc );
              if sv <> nil then DISPOSE( sv );
              SRC_ERROR_S( mdnam, 131, e_error, ide_name^ )
            end
            else
              if sym = readsy then FREAD_STR( iocnt_file, ide_str, fs, dc )
                              else FREAD_STR( input, ide_str, fs, dc );
          exp_telint:
            if exp_aai <> nil then
            with exp_aai^ do
            if sym = readsy then ide_itb[exp_shift] := FREAD_INT( iocnt_file, fs )
                            else ide_itb[exp_shift] := FREAD_INT( input, fs );
          exp_telflt:
            if exp_aaf <> nil then
            with exp_aaf^ do
            if sym = readsy then ide_ftb[exp_shift] := FREAD_FLT( iocnt_file, fs )
                            else ide_ftb[exp_shift] := FREAD_FLT( input, fs );
          exp_telstr:
            if exp_aas <> nil then
            with exp_aas^ do
              if sym = readsy then
                FREAD_STR( iocnt_file, ide_stb[exp_shift], fs, dc )
              else
                FREAD_STR( input, ide_stb[exp_shift], fs, dc );
          exp_tabint:
            if exp_aai <> nil then
            with exp_aai^ do
            begin
              while (nel < ls) and (io_eof^.ide_int = 0) do
              begin
                if sym = readsy then ide_itb[nel] := FREAD_INT( iocnt_file, fs )
                                else ide_itb[nel] := FREAD_INT( input, fs );
                nel := nel + 1
              end;
              if io_eof^.ide_int > 0 then nel := nel - 1;
              io_count^.ide_int := nel - 1
            end;

          exp_tabflt:
            if exp_aaf <> nil then
            with exp_aaf^ do
            begin
              while (nel < ls) and (io_eof^.ide_int = 0) do
              begin
                if sym = readsy then ide_ftb[nel] := FREAD_FLT( iocnt_file, fs )
                                else ide_ftb[nel] := FREAD_FLT( input, fs );
                nel := nel + 1
              end;
              if io_eof^.ide_int > 0 then nel := nel - 1;
              io_count^.ide_int := nel - 1
            end;

          exp_tabstr:
            if exp_aas <> nil then
            with exp_aas^ do
            begin
              while (nel < ls) and (io_eof^.ide_int = 0) do
              begin
                if sym = readsy then FREAD_STR( iocnt_file, ide_stb[nel], fs, dc )
                                else FREAD_STR( input, ide_stb[nel], fs, dc );
                nel := nel + 1
              end;
              if io_eof^.ide_int > 0 then nel := nel - 1;
              io_count^.ide_int := nel - 1
            end;
        otherwise
          SRC_ERROR_S( mdnam, 403, e_severe,  )
        end
      end;
    exit if sy <> comma;
      INSYMBOL
    end;
    if sym = readsy then begin  if not EOF( iocnt_file ) then READLN( iocnt_file )  end
                    else begin  if not EOF then READLN  end
  end
end READ_VALUES;



procedure WRITE_VALUES( sym: symbol );
const
  mdnam = 'DSPL';

var
  lun, iv, fs, dc, npt, ncr: integer;
  rv:                        shreal;
  st:                        str_string;
  bndl:                      boolean;
  rec:                       exp_rec;

begin  { WRITE_VALUES }
  lun := 0;
  INSYMBOL;
  if (sym <> writemsgsy) and (sym <> displaysy) then
  begin
    lun := GET_INTEXPR( 1 );  { Get the I/O Lun value }
    if (lun < 0) or (lun > max_lun) then
    begin
      SRC_ERROR( mdnam, 77, e_severe );
      lun := -1
    end
    else
      if sym_iof[lun].iocnt_mode <> iocnt_output then
      begin
        SRC_ERROR( mdnam, 401, e_severe );
        lun := -1
      end;
    if sy_sym.sy <> colon then SRC_ERROR( mdnam, 31, e_error )
                          else INSYMBOL
  end;
  if lun >= 0 then
  with sy_sym, rec do
  begin
    if op <> stp_op then
    loop
      bndl := true;
      GET_EXPRESSION;
      rec := exp_res;
      case exp_kind of { If function of the expression type }
        exp_valint: begin
                      iv := exp_int;
                      fs := 10; dc := 0;
                      GET_IO_FORMAT( 2, fs, dc, iv );
                      WRITEV( st, iv:fs:dc )
                    end;
        exp_valflt: begin
                      rv := exp_flt;
                      fs := 12; dc := 6;
                      GET_IO_FORMAT( 2, fs, dc, iv );
                      WRITEV( st, rv:fs:dc )
                    end;
        exp_valstr: begin
                      fs := 0; dc := 0;
                      GET_IO_FORMAT( 2, fs, dc, iv );
                      if exp_str <> nil then
                      begin
                        WRITEV( st, exp_str^:fs:dc );
                        if exp_ref = nil then
                        begin
                          DISPOSE( exp_str );
                          exp_str := nil
                        end
                      end
                      else
                        if fs > 0 then WRITEV( st, ' ':fs ) else st := ''
                    end;
        exp_telint: if exp_aai <> nil then
                    with exp_aai^ do
                    begin
                      fs := 10; dc := 0;
                      GET_IO_FORMAT( 2, fs, dc, iv );
                      if fs = 0 then fs := 10;
                      WRITEV( st, ide_itb[exp_shift]:fs:dc )
                    end;
        exp_tabint: if exp_aai <> nil then
                    with exp_aai^ do
                    begin
                      bndl := false;
                      iv := 4; fs := 10; dc := 0;
                      GET_IO_FORMAT( 3, iv, fs, dc );
                      if iv <= 0 then iv := 4;
                      st.length := 0;
                      with sym_iof[lun] do
                      for ii := 1 to exp_edim do
                      begin
                        WRITEV( st:false, ide_itb[exp_shift]:fs:dc );
                        exp_shift := exp_shift + 1;
                        if (ii mod iv = 0) or (ii = exp_edim) then
                        begin
                          case sym of
                            displaysy:  WRITELN( st );
                            writemsgsy: begin
                                          LST_PUT_STRING( st );
                                          LST_EOLN
                                        end;
                            writelnsy, writesy: WRITELN( iocnt_file, st );
                          otherwise
                          end;
                          st.length := 0
                        end
                      end
                    end;
        exp_telflt: if exp_aaf <> nil then
                    with exp_aaf^ do
                    begin
                      fs :=12; dc := 6;
                      GET_IO_FORMAT( 2, fs, dc, iv );
                      WRITEV( st, ide_ftb[exp_shift]:fs:dc )
                    end;
        exp_tabflt: if exp_aaf <> nil then
                    with exp_aaf^ do
                    begin
                      bndl := false;
                      iv := 4; fs := 12; dc := 6;
                      GET_IO_FORMAT( 3, iv, fs, dc );
                      if iv <= 0 then iv := 4;
                      st.length := 0;
                      with sym_iof[lun] do
                      for ii := 1 to exp_edim do
                      begin
                        WRITEV( st:false, ide_ftb[exp_shift]:fs:dc );
                        exp_shift := exp_shift + 1;
                        if (ii mod iv = 0) or (ii = exp_edim) then
                        begin
                          case sym of
                            displaysy:  WRITELN( st );
                            writemsgsy: begin
                                          LST_PUT_STRING( st );
                                          LST_EOLN
                                        end;
                            writelnsy, writesy: WRITELN( iocnt_file, st );
                          otherwise
                          end;
                          st.length := 0
                        end
                      end
                    end;
        exp_telstr: if exp_aas <> nil then
                    with exp_aas^ do
                    begin
                      fs := 0; dc := 0;
                      GET_IO_FORMAT( 2, fs, dc, iv );
                      if ide_stb[exp_shift] = nil then
                        if fs > 0 then WRITEV( st, ' ':fs )
                                  else st.length := 0
                      else
                        WRITEV( st, ide_stb[exp_shift]:fs:dc )
                    end;
        exp_tabstr: if exp_aas <> nil then
                    with exp_aas^ do
                    begin
                      bndl := false;
                      iv := 1; fs := 0; dc := 0;
                      GET_IO_FORMAT( 3, iv, fs, dc );
                      if iv <= 0 then iv := 1;
                      st.length := 0;
                      with sym_iof[lun] do
                      for ii := 1 to exp_edim do
                      begin
                        if ide_stb[exp_shift] = nil then
                        begin
                          if fs > 0 then WRITEV( st:false, ' ':fs )
                        end
                        else
                          WRITEV( st:false, ide_stb[exp_shift]:fs:dc );
                        exp_shift := exp_shift + 1;
                        if (ii mod iv = 0) or (ii = exp_edim) then
                        begin
                          case sym of
                            displaysy:  WRITELN( st );
                            writemsgsy: begin
                                          LST_PUT_STRING( st );
                                          LST_EOLN
                                        end;
                            writelnsy, writesy: WRITELN( iocnt_file, st );
                          otherwise
                          end;
                          st.length := 0
                        end
                      end
                    end;

      otherwise
        SRC_ERROR( mdnam, 402, e_severe )
      end;

      case sym of
        displaysy:   WRITE( st );
        writemsgsy:  LST_PUT_STRING( st );
        writelnsy,
        writesy:     WRITE( sym_iof[lun].iocnt_file, st );
      otherwise
      end;
    exit if sy <> comma;
      INSYMBOL
    end;

    if bndl then
    case sym of
      displaysy: if sy = replysy then READ_VALUES( replysy )
                                 else WRITELN;
      writemsgsy: LST_EOLN;
      writelnsy:  WRITELN( sym_iof[lun].iocnt_file );
    otherwise
    end
  end
end WRITE_VALUES;



procedure CASE_STATE;
const
  mdnam = 'CASE';

var
  iv, jv, kv: integer;
  bg, be: boolean;

begin
  with sy_sym, src_control^ do
  begin
    bg := false;
    be := false;
    INSYMBOL;                    { Gobble up the case keyword }
    src_insnb := src_insnb + 1;
    iv := GET_INTEXPR( 0 );      { Get the selector value }
    if sy <> whensy then SRC_ERROR( mdnam, 121, e_error )
                    else INSYMBOL;
    loop
      if sy <> othersy  then
      begin
        loop
          jv := GET_INTEXPR( iv );
          if sy = twodot then
          begin
            INSYMBOL;
            kv := GET_INTEXPR( jv )
          end else kv := jv;
          if (iv >= jv) and (iv <= kv) then bg := true;
        exit if sy <> comma;
          INSYMBOL
        end;
        if sy <> colon then SRC_ERROR( mdnam, 31, e_error );
        if bg then
        begin
          STATELIST( whensy );
          if sy <> endsy then SKIP_SYMBOL( endsy );
          be := true
        end
        else
        begin
          src_insnb := src_insnb + 1;
          SKIP_SYMBOL( whensy );
          src_insnb := src_insnb - 1
        end
      end
      else
      begin { Other case }
        STATELIST( othersy );
        be := true
      end;
    exit if (sy <> whensy) or be;
      INSYMBOL
    end;
    if sy <> endsy then
      if be then
        SRC_ERROR( mdnam, 108, e_error )
      else
      begin
        src_insnb := src_insnb + 1;
        SKIP_SYMBOL( endsy );
        src_insnb := src_insnb - 1
      end
    else INSYMBOL;
    src_insnb := src_insnb - 1
  end
end CASE_STATE;



procedure IF_STATE;
const
  mdnam = 'IFST';

var
  iv: integer;

begin
  with sy_sym, src_control^ do
  begin
    INSYMBOL;                    { Gobble up the if keyword }
    iv := GET_INTEXPR( 1 );      { Get the condition value }
    if sy <> thensy then SRC_ERROR( mdnam, 107, e_error );
    if iv > 0 then STATELIST( elsesy )
              else
              begin
                src_insnb := src_insnb + 1;
                SKIP_SYMBOL( elsesy );
                src_insnb := src_insnb - 1
              end;
    if sy = elsesy then
    begin
      if iv > 0 then
                begin
                  src_insnb := src_insnb + 1;
                  SKIP_SYMBOL( endsy );
                  src_insnb := src_insnb - 1
                end
                else STATELIST( endsy )
    end;
    if sy = endsy then INSYMBOL
                  else SRC_ERROR( mdnam, 108, e_error )
  end
end IF_STATE;



procedure BLOCK_STATE;
begin
  DISPLAY_NEW;                   { Create a new identifier display level }
  STATELIST( endsy );            { Perform the block statements }
  DISPLAY_FREE;                  { Destroy the display }
  if sy_sym.sy = endsy then
    INSYMBOL                     { Gobble up the end keyword }
  else
    SRC_ERROR( 'BLOC', 108, e_severe )
end BLOCK_STATE;



procedure STATEMENT;
{ Procedure to execute one SHELL statement }
const
  mdnam = 'STAT';

var
 stat_sy: symbol;
 ide:     ide_ptr;


begin
  with sy_sym do
  begin
    stat_sy := sy;
    case stat_sy of
      endsy, elsesy, untilsy: ;
      chainesy, includesy:
        INCLUDE_STATE( stat_sy = includesy );
      peofsy:        ENDFILE_STATE;
      pragmasy:      PRAGMA_STATE( src_control );

      intsy,    { Integer parameter definition }
      floatsy,  { Floatting parameter definition }
      stringsy: { String parameter definition }
                     DECLARE_NEW_IDENT( stat_sy );

      enumsy:        DECLARE_NEW_ENNUMS;

      beginsy:       BLOCK_STATE;
       casesy:       CASE_STATE;
         ifsy:       IF_STATE;

      whilesy:       WHILE_STATE;
      repeatsy:      REPEAT_STATE;
      loopsy: ;
      forsy:         FOR_STATE;

      writesy, writelnsy, writemsgsy, displaysy:
                     WRITE_VALUES( stat_sy );
      readsy, replysy:
                     READ_VALUES( stat_sy );
      open_inpsy, open_outsy:
                     OPENFILE_STATE( stat_sy );
      closesy:       CLOSEFILE_STATE;

      macrosy:       MACRO_STATE;
      purgesy:       PURGE_STATE;

      specificsy:    MANAGE_SPECIFIC_SYMBOL( op );

      identsy: if sy_macro = nil then
               begin
                 ide := IDE_SEARCH( false );
                 if ide = nil then
                 begin
                   SRC_ERROR_S( mdnam, 104, e_severe, sy_ident );
                   SKIP_SYMBOL( semicolon );
                   ide := udc_ident
                 end
                 else
                   case ide^.ide_kind of
                     ide_funct, ide_functsp: GET_EXPRESSION
                   otherwise
                     ASSIGNEMENT
                   end
               end
               else
                 if sy_macro^.idm_kind = idm_macro then CALL_MACRO
                 else SRC_ERROR_S( mdnam, 109, e_severe, sy_ident );

      semicolon, nothing:  ; { Not a statement }

    otherwise
      SRC_ERROR( mdnam, 110, e_error );
      SKIP_SYMBOL( semicolon )
    end;
    case sy of
      eofsy, eomcsy, semicolon,
      untilsy, endsy, elsesy: ;
    otherwise
      INSYMBOL
    end
  end
end STATEMENT;



[global]
procedure STATELIST( stopper: symbol ); { was external }
begin
  with sy_sym do
  begin
    with src_control^ do src_insnb := src_insnb + 1;
    INSYMBOL;
    if stopper = othersy then
    begin { Tolerate a colon after the other keyword in a case statement }
      stopper := endsy;
      if sy = colon then INSYMBOL
    end;
    if not (fatal_error or (sy = stopper) or
             (sy = endsy) or (sy = eofsy)) then
    repeat
      if sy <> semicolon then STATEMENT;
      while sy = peofsy do ENDFILE_STATE;
    exit if fatal_error or (sy = stopper) or (sy = endsy) or (sy = eofsy);
      if sy = semicolon then INSYMBOL
                        else SRC_ERROR( 'STLI', 21, e_error )
    until fatal_error or (sy = stopper) or
         (sy = endsy) or (sy = eofsy) or (sy = elsesy) or (sy = eomcsy);
    with src_control^ do src_insnb := src_insnb - 1
  end
end STATELIST;



[global]
function FILE_EXIST_CHECK( in_var fspc: string ): boolean;
begin
  FILE_EXIST_CHECK := FILE_ACCESS_CHECK( fspc, 4 { Read Access }, [case_ena_file] )
end FILE_EXIST_CHECK;



[global]
procedure FILE_PATH_SEARCH( in_var path, fnam: string; var dpath, fpath: string );
var
  dir:          string;
  ib, ie, n:   integer;
  fnd:         boolean;

begin
  ib  :=     1;
  n   :=     1;
  fnd := false;
  while not fnd and (ib <> 0) do
  begin
    ie := INDEX( path, ';', n ); n := n + 1;
    if ie = 0 then
    begin  dir := SUBSTR( path, ib ); ib := 0  end
    else
    begin  dir := SUBSTR( path, ib, ie - ib ); ib := ie + 1  end;
    if dir[dir.length] <> '/' then begin  dir.length := dir.length + 1; dir[dir.length] := '/'  end;
    fpath := dir||fnam;
    fnd := FILE_ACCESS_CHECK( fpath, 4 {Read access}, [case_ena_file] )
  end;
  if not fnd then
  begin
    dpath.length := 0; fpath := fnam; path_index := 0
  end
  else
  begin
    fpath := current_filename; { Get the complete file Path build by FILE_ACCESS_CHECK (CPAS__STD) }
    dpath := SUBSTR( fpath, 1, fpath.length - fnam.length ); { faster than SET_FILE_SPECIFICATION( dpath, dir ) }
    path_index := n            { Keep the path index value }
  end
end FILE_PATH_SEARCH;




begin { Main - Cpas_Shell }
  {  *** Initialize the Source/Listing/Error Context ***  }
  { Initialize the LST sub-system }
  data_mode   := false;
  fatal_error := false;
  LST_G_INIT( cpsh_deflst, cpsh_heading, ierr );
  if ierr = 0 then
  begin
    env_path_search := PASCAL_PATH_ENV;

    FILE_PATH_SEARCH( env_path_search, cpsh_defsrc, sy_maclst, sy_string );

    if sy_maclst.length > 0 then
    begin { Set the $sys_cpasenvlib predefined symbol value (set by temporary pointer }
      NEW( inienv_path, sy_maclst.length );
      inienv_path^ := sy_maclst;
      if path_index > 1 then
      begin
	sy_maclst.length := INDEX( sy_maclst, '/', -3 );
	sy_maclst := sy_maclst||'cpas/'
      end;
      NEW( pasenv_path, sy_maclst.length );
      pasenv_path^ := sy_maclst
    end
    else
    begin
      NEW( inienv_path, 2 ); inienv_path^ := './';
      NEW( pasenv_path, 2 ); pasenv_path^ := './'
    end;

    sy_maclst := sy_maclst||cpsh_errfile;

    ERR_INIT( sy_maclst );                  { Initialize the ERR sub-system }
    SRC_INIT( cpsh_prompt, sy_string );     { Initialize the SRC sub-system }

    if src_control = nil then
      SRC_INIT( cpsh_prompt, 'tt:' );       { Try to open tt: as source file }
    if src_control <> nil then
    begin
      SET_KEYWORD_TREE;                     { Create the Keyword tree }
      DEFINE_BUILTIN;                       { Create the pre-declared tree }
      { Init the IO file lun status }
      for i := 1 to max_lun do
        sym_iof[i].iocnt_mode := iocnt_close;

      sy_ch := ' ';                         { Initialize the nextch process }
      sy_maclst := '';                      { Init the Macro source to empty state }
      sy_nomacrflg := false;
      sy_noexec := false;
      sy_sym.sy := nothing;
      STATELIST( endsy );                   { Execute init statements }
      INSYMBOL;                             { Purge file now }
      src_control^.src_insnb := 0;          { Force the base statement level }
      if not fatal_error then
      begin
        sy_init_mod := false;               { Set the user mode }
        sy_ch := ' ';                       { Initialize the nextch process }
        sy_sym.sy := nothing;
        STATELIST( endsy );                 { Execute user statements }
        if fatal_error then WRITELN( cpsh_name, ' Stop after a fatal ERROR.' )
                       else WRITELN
      end
    end
    else
      WRITELN( cpsh_name, ' cannot open the primary source file.' )
  end
  else
  begin
    WRITELN( cpsh_name, ' cannot open the default Listing file.' );
    WRITELN
  end
end {Cpas_Shell_Base}.
