{
*************************************************************************
*                                                                       *
*                                                                       *
*                       *  P A S  *  S Y S T E M                        *
*                                                                       *
*                                                                       *
*                    * * *   C o m p i l e r    * * *                   *
*                                                                       *
*                                                                       *
*         ---     PASCAL II - C   GENERATOR   MODULE     ---            *
*                                                                       *
*              ---  Version  3.1-B5-3 -- 30/06/2024 ---                 *
*                                                                       *
*           by :                                                        *
*                                                                       *
*               www.pierre.wolfers.fr                                   *
*               P. Wolfers                                              *
*                                          FRANCE.                      *
*                                                                       *
*************************************************************************



/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This program is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This software is distributed in the hope that it will be useful,    //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////


}


{************     CPAS  version    *************}


 
  {**************************************************}
  {*******            Program  Head            ******}
  {**************************************************}


(*
[inherit(     'pasenv:cpas_b__src',
              'lib:pas_env')]          { Use kernel tree definitions }
*)
module PAS_PASS2( input, output );     { input and output for user terminal }


{ *** Include basic compiler environment *** }
%include 'passrc:pcmp_env';


  {**************************************************}
  {*******          const declarations         ******}
  {**************************************************}

const
  maxcoderec     = 132;
  maxcodespace   =  80;
  maxidentsize   =  31;
  sch            = ORD( 'A' ) - ORD( 'a' );

  mdnam = 'GN_C';                      { Generic Error identifier for the C code generator }

  p_ptr_cd = 0;
  p_cha_cd = 1;
  p_wrd_cd = 2;
  p_int_cd = 3;
  p_sng_cd = 4;
  p_dbl_cd = 5;

  bsw_w =  0;                          { Byte Swap table Offset for Word Integer - and 16 bits pointer }
  bsw_l =  2;                          { Byte Swap table Offset for Long Integer - and 32 bits pointer }
  bsw_q =  6;                          { Byte Swap table Offset for Quad Integer - and 64 bits pointer }
  bsw_f = 14;                          { Byte Swap table Offset for 32 bits Single Precision Floating }
  bsw_g = 18;                          { Byte Swap table Offset for 64 bits Double Precision Floating }

type

  acc_kinds = ( acc_formal,            { C formal access }
                acc_pascal,            { Use pascal Blk direct access }
                acc_varbl,             { Variable reference set }
                acc_other              { Special access }
              );


  cop_kinds = ( cop_una,               { Unary operator }
                cop_unb,               { Unary on boolean operand }
                cop_aft,               { Unary post ope. }
                cop_bin,               { Binary operator }
                cop_bib,               { Binary on boolean operand }
                cop_fn1,               { One arg. function }
                cop_fn2,               { Two arg. function }
                cop_fn3                { Three arg. function }
             );

  cop_rec = record
              cop_prio: byte;          { C operator priority max = 0, min = 15 }
              cop_kind: cop_kinds;     { C operator kind }
              cop_lnam: byte;          { C code string length }
              cop_snam: packed array[1..8] of char     { C code name string }
            end;   


  cmac_ptr = ^cmac_rec;                { C macro definition pointer }

  cmac_rec = record                    { C macro record definition }
               cmac_nxt: cmac_ptr;     { C macro link pointer }
               cmac_str: str_ptr       { C macro string definition }
             end;



  { *** C code line text in memory *** }

  str_rec = packed array[0..255] of char;

  ccod_ptr = ^ccod_rec;

  ccod_rec = record
    ccod_nxt: ccod_ptr;                { Link to next C code line }
    ccod_str: str_rec                  { Line of generated C code }
  end;

  ctyp_ptr = ^ctyp_rec;                { Define the local C type record pointer }

  ctyp_rec = record                    { Define the local C type record }
    ctyp_nxt: ctyp_ptr;                { Link to next one }
    ctyp_ide:  str_ptr                 { C typedef string }
  end;


var

  ccod_free,                           { List of free ccod_rec }
  ccod_hde,                            { First C code record pointer }
  ccod_lst: [static] ccod_ptr := nil;  { Last C code record pointer }



  cop_table: [static] array[pcod_not..pcod_gatanh] of cop_rec := (

    {*** logical operators ***}

    ( 13, cop_unb, 1, '!       ' ),    { not }
    (  4, cop_bib, 2, '&&      ' ),    { and }
    (  3, cop_bib, 2, '||      ' ),    { or  }
    (  6, cop_bin, 1, '^       ' ),    { xor }

    { *** Set operators *** }

    ( 13, cop_una, 1, '~       ' ),    { Bit Complement }
    (  7, cop_bin, 1, '&       ' ),    { Bit and }
    (  6, cop_bin, 1, '^       ' ),    { Bit xor }
    (  7, cop_bin, 2, '&~      ' ),    { Bit Clear }
    (  5, cop_bin, 1, '|       ' ),    { Bit Set }
    (  7, cop_bin, 1, '&       ' ),    { Bit Test }
    ( 10, cop_una, 3, '1<<     ' ),    { Simple Set generator }
    (  5, cop_fn1, 6, '_S_GEN  ' ),    { Large  Set generator }
    (  0, cop_fn1, 7, '_S_ADEL ' ),    { Large  Set add element }
    (  8, cop_fn2, 7, '_SET_LE ' ),    { Set Test LT }
    (  8, cop_fn2, 7, '_SET_LE ' ),    { Set Test LE }
    (  8, cop_fn2, 7, '_SET_LE ' ),    { Set Test GE }
    (  8, cop_fn2, 7, '_SET_LE ' ),    { Set Test GT }
    (  8, cop_bib, 2, '==      ' ),    { Set Test EQ }
    (  8, cop_bib, 2, '!=      ' ),    { Set Test NE }
    ( 10, cop_bin, 7, '<< !=0 ?' ),    { in operator }

    { ***** Numeric operators ***** }

    (  8, cop_bib, 2, '==      ' ),    { EQ operator }
    (  8, cop_bib, 2, '!=      ' ),    { NE operator }

    { ------------  For unsigned integer ------------- }
    (  9, cop_bib, 1, '<       ' ),    { LT operator }
    (  9, cop_bib, 2, '<=      ' ),    { LE operator }
    (  9, cop_bib, 2, '>=      ' ),    { GE operator }
    (  9, cop_bib, 1, '>       ' ),    { GT operator }

    { ------------  For signed integer ------------- }
    (  9, cop_bib, 1, '<       ' ),    { LT operator }
    (  9, cop_bib, 2, '<=      ' ),    { LE operator }
    (  9, cop_bib, 2, '>=      ' ),    { GE operator }
    (  9, cop_bib, 1, '>       ' ),    { GT operator }

    { ------------  For block in memory ------------- }
    (  0, cop_fn2, 7, '_BLK_EQ ' ),    { EQ block operator }
    (  0, cop_fn2, 7, '_BLK_NE ' ),    { NE block operator }

    {*** Other basic literal operators ***}

    ( 13, cop_aft, 7, '++      ' ),    { INC }
    ( 13, cop_aft, 7, '--      ' ),    { DEC }

    ( 10, cop_fn2, 8, '_BIT_ASH' ),    { Bit arithmetic shift }
    ( 10, cop_fn2, 8, '_BIT_LSH' ),    { Bit logical shift }
    ( 10, cop_fn2, 8, '_BIT_ROT' ),    { Bit rotation }

    ( 11, cop_aft, 2, '+1      ' ),    { SUCC( iv ) }
    ( 11, cop_aft, 3, ' -1      ' ),   { PRED( iv ) }
    (  7, cop_aft, 2, '&1      ' ),    { ODD( iv )  }

    {*** Integer operators ***}

    ( 13, cop_una, 2, ' -      ' ),    { Unary - }
    (  0, cop_fn1, 3, 'abs     ' ),    { ABS( int ) }
    (  0, cop_fn1, 6, '_I_SQR  ' ),    { SQR( int ) }

    ( 11, cop_bin, 1, '+       ' ),    { int + int }
    ( 11, cop_bin, 1, '-       ' ),    { int - int }
    ( 12, cop_bin, 1, '*       ' ),    { int * int }
    ( 12, cop_bin, 1, '/       ' ),    { int / int }
    (  0, cop_fn2, 6, '_I_MOD  ' ),    { int mod int }
    ( 12, cop_bin, 1, '%       ' ),    { int rem int }

    {*** single float operators ***}

    (  9, cop_bib, 1, '<       ' ),    { LT operator }
    (  9, cop_bib, 2, '<=      ' ),    { LE operator }
    (  9, cop_bib, 2, '>=      ' ),    { GE operator }
    (  9, cop_bib, 1, '>       ' ),    { GT operator }
    (  8, cop_bib, 2, '==      ' ),    { EQ operator }
    (  8, cop_bib, 2, '!=      ' ),    { NE operator }

    ( 13, cop_una, 2, ' -      ' ),    { Unary - }
    (  0, cop_fn1, 4, 'fabs    ' ),    { ABS( single ) }
    (  0, cop_fn1, 6, '_F_SQR  ' ),    { SQR( single ) }
    ( 13, cop_una, 7, '(float) ' ),    { Cv. int -> single }
    ( 13, cop_una, 8, '_F_ROUND' ),    { ROUND( single ) }
    (  0, cop_fn1, 5, '(int)   ' ),    { TRUNC( single ) }

    ( 11, cop_bin, 1, '+       ' ),    { flt + flt }
    ( 11, cop_bin, 1, '-       ' ),    { flt - flt }
    ( 12, cop_bin, 1, '*       ' ),    { flt * flt }
    ( 12, cop_bin, 1, '/       ' ),    { flt / flt }

    {*** Double Float Operators ***}

    (  9, cop_bib, 1, '<       ' ),    { LT operator }
    (  9, cop_bib, 2, '<=      ' ),    { LE operator }
    (  9, cop_bib, 2, '>=      ' ),    { GE operator }
    (  9, cop_bib, 1, '>       ' ),    { GT operator }
    (  8, cop_bib, 2, '==      ' ),    { EQ operator }
    (  8, cop_bib, 2, '!=      ' ),    { NE operator }

    ( 13, cop_una, 1, '-       ' ),    { Unary - }
    (  0, cop_fn1, 4, 'fabs    ' ),    { ABS( double ) }
    (  0, cop_fn1, 6, '_G_SQR  ' ),    { SQR( double ) }
    ( 13, cop_una, 8, '(double)' ),    { Cv. int -> double }
    ( 13, cop_una, 8, '_G_ROUND' ),    { ROUND( double ) }
    (  0, cop_fn1, 5, '(int)   ' ),    { TRUNC( double ) }

    ( 13, cop_una, 8, '(double)' ),    { single -> double }
    ( 13, cop_una, 7, '(float) ' ),    { double -> single }

    ( 11, cop_bin, 1, '+       ' ),    { dbl + dbl }
    ( 11, cop_bin, 1, '-       ' ),    { dbl - dbl }
    ( 12, cop_bin, 1, '*       ' ),    { dbl * dbl }
    ( 12, cop_bin, 1, '/       ' ),    { dbl / dbl }

    {*** Other Operators ***}

    (  0, cop_fn2, 8, '_IIPOWER' ),    { int**int }
    (  0, cop_fn2, 3, 'pow     ' ),    { single**single }
    (  0, cop_fn2, 8, '_IFPOWER' ),    { single**int }

    (  0, cop_fn1, 4, 'sqrt    ' ),    { SQRT( single ) }

    (  0, cop_fn1, 3, 'sin     ' ),    { SIN( single ) }
    (  0, cop_fn1, 3, 'cos     ' ),    { COS( single ) }
    (  0, cop_fn1, 3, 'tan     ' ),    { TAN( single ) }

    (  0, cop_fn1, 4, 'asin    ' ),    { ASIN( single ) }
    (  0, cop_fn1, 4, 'acos    ' ),    { ACOS( single ) }
    (  0, cop_fn1, 4, 'atan    ' ),    { ATAN( single ) }

    (  0, cop_fn2, 5, 'atan2   ' ),    { PHASE( single, single ) }
    (  0, cop_fn1, 3, 'exp     ' ),    { EXP( single ) }
    (  0, cop_fn1, 3, 'log     ' ),    { LN( single ) }

    (  0, cop_fn1, 4, '_SHF    ' ),    { SINH( single ) }
    (  0, cop_fn1, 4, '_CHF    ' ),    { COSH( single ) }
    (  0, cop_fn1, 4, '_THF    ' ),    { TANH( single ) }

    (  0, cop_fn1, 5, '_ASHF   ' ),    { ARGSINH( single ) }
    (  0, cop_fn1, 5, '_ACHF   ' ),    { ARGCOSH( single ) }
    (  0, cop_fn1, 5, '_ATHF   ' ),    { ARGTANH( single ) }

    (  0, cop_fn2, 5, 'pow     ' ),    { double**double }
    (  0, cop_fn2, 8, '_IGPOWER' ),    { double**int }

    (  0, cop_fn1, 4, 'sqrt    ' ),    { SQRT( double ) }

    (  0, cop_fn1, 3, 'sin     ' ),    { SIN( double ) }
    (  0, cop_fn1, 3, 'cos     ' ),    { COS( double ) }
    (  0, cop_fn1, 3, 'tan     ' ),    { TAN( double ) }

    (  0, cop_fn1, 4, 'asin    ' ),    { ASIN( double ) }
    (  0, cop_fn1, 4, 'acos    ' ),    { ACOS( double ) }
    (  0, cop_fn1, 4, 'atan    ' ),    { ATAN( double ) }

    (  0, cop_fn2, 5, 'atan2   ' ),    { PHASE( double, double ) }
    (  0, cop_fn1, 3, 'exp     ' ),    { EXP( double ) }
    (  0, cop_fn1, 3, 'log     ' ),    { LN( double ) }

    (  0, cop_fn1, 4, '_SHG    ' ),    { SINH( double ) }
    (  0, cop_fn1, 4, '_CHG    ' ),    { COSH( double ) }
    (  0, cop_fn1, 4, '_THG    ' ),    { TANH( double ) }

    (  0, cop_fn1, 5, '_ASHG   ' ),    { ARGSINH( double ) }
    (  0, cop_fn1, 5, '_ACHG   ' ),    { ARGCOSH( double ) }
    (  0, cop_fn1, 5, '_ATHG   ' )     { ARGTANH( double ) }
  );


  SP_code: array[p_ptr_cd..p_dbl_cd] of packed array[1..8] of char := (
       'Adr__ptr',
       'Adr__cha',
       'Adr__wrd',
       'Adr__int',
       'Adr__sng',
       'Adr__dbl'
      );


  in_int: record
          case boolean of
            false:( i: integer );
            true:(  p: lgt_ptr )
          end;

  in_hexa: record
           case integer of
             0:(bt: array[0..7] of ubyte);
             1:(ub0, ub1, ub2, ub3, ub4, ub5, ub6, ub7: ubyte );
             2:(sb0, sb1, sb2, sb3, sb4, sb5, sb6, sb7: sbyte );
             3:(uw0, uw1, uw2, uw3: uword );
             4:(sw0, sw1, sw2, sw3: uword );
             5:(ui0, ui1: unsigned );
             6:(li0, li1: integer );
             7:(lgi: integer );
             8:(lgu: unsigned );
             9:(fl0, fl1: single );
            10:(dbl: double );
           end;


  module_gsta_size,                    { Static size for the module }
  module_gsti_size,                    { Static with init values size for the volume }
  module_gdat_size,                    { Data size for the module }

  module_sta_size,                     { Static size for the module }
  module_sti_size,                     { Static with init values size for the volume }
  module_dat_size:    integer := 0;    { Data size for the module }

  proc_intlex_count,                   { Internal Access Lex count for Lex Identifier }
  ctyp_count,                          { Internal Access Count for formal entry type }
  curr_dyn_lex,                        { Current Dynamic lex }
  proc_dyn_size:      integer := 0;    { Dynamic size to allocate for local routine }

  ctyp_first,                          { ctyp_rec list header (first and last pointer) }
  ctyp_last:   ctyp_ptr := nil;



  { Code_rec: array[1..maxcoderec] of byte; }
  code_len:            integer;

  bcase,                               { Flag for case change after a "$" character }
  bcomma,                              { Flag for comma to insert for list out }
  bsemicolon,                          { Flag for semicolon not inserted in list out }
  bspace:  boolean  :=   false;        { Flag for space insertion in C code }

  buffer: string( 255 )  := '';        { The C object file buffer }

  icolnbr,                             { Current column number }
  imaxcol,                             { Required maximum column number }
  icolumn,                             { Identation number for C code }
  procedure_lex,                       { Current Procedure/function lex level }
  dyn_local_top,
  dyn_local_base,
  tmp_loc_count: integer :=  0;

  setreg_i1,                           { Flag for integer register 1 }
  setreg_i2,                           { Flag for integer register 2 }
  setreg_v,                            { Flag for Pointer Register }
  setreg_f,                            { Flag for float register }
  setreg_g:            boolean;        { Flag for double register }

  global_labenv:       boolean;        { Global label environment flag }

  global_labhde:       ide_ptr;        { List of global label }

  labenv_adr,                          { Address of current label environment }
  label_cnt,                           { Count for label }
  for_var_cnt,                         { For Variable count }
  for_var_max: integer :=    0;        { Current maximum count of For Variable }


  curr_block:          lgt_ptr;        { Current block pointer for exit statement }

  return_label,                        { If True => A return label must be used }
  end_c_block,                         { If TRUE => the semicolon is suppressed }
  objf_open:  boolean := false;        { To flag the open object file success }
  objf_ok:             boolean;        { Boolean to flag the object file open success }

  obj_file:               text;        { Object file in C }


  cenv_fil:    str_ptr :=  nil;        { C environment file specification }

  { Tables to drive permutation of bytes for various data type } 
  { Word, longword and quadword format table, to drive the byte permuttations }
  { Each byte give the correct place of a byte for a given type }
  { 00..01 for  16 bits (2 bytes) word     (integer/pointer) }
  { 02..05 for  32 bits (4 bytes) longword (integer/pointer) }
  { 06..13 for  64 bits (8 bytes) quadword (integer/pointer) }
  { 14..17 for  32 bits (4 bytes) longword (single)  }
  { 18..25 for  64 bits (8 bytes) quadword (double)  }

  cmp_swapt,                           { Swap Byte Table from source to target }
  cmp_trgta,                           { Address order of bytes for target }
  cmp_srcta: array[0..25] of integer;  { Address order of bytes for source }

  cmp_pascal_cdef: array[0..10] of str_ptr;    { C type definition for PASCAL }

  { Used format for integer and floatting point }
  cmp_bytesmd,                         { Option flag for float format modifier }
  cmp_lparop:          boolean;        { Option to set ( <>, ... ) operator usage }

  cmac_hde,                            { Macro head pointer }
  cmac_lst:  cmac_ptr  :=  nil;        { Macro last pointer }



[global]
function  BINARY_OBJF_OPEN: integer;
var
  i, ierr: integer;

begin
  OPEN( obj_file, pas_obj^, [write_file,error_file,case_ena_file] );
  objf_open := (iostatus = 0);
  cmp_objf  := objf_open;
  bspace    := false;
  icolumn   := 0;
  BINARY_OBJF_OPEN := iostatus
end BINARY_OBJF_OPEN;



[global]
procedure BINARY_OBJF_CLOSE;
begin
  if objf_open = true then
  begin
    CLOSE( obj_file );
    objf_open := false
  end;
  if pas_obj <> nil then
  begin
    DISPOSE( pas_obj );
    pas_obj := nil
  end
end BINARY_OBJF_CLOSE;



procedure OUT_EOLN;
var
  i: integer;

begin
  WRITELN( obj_file, buffer );
  buffer.length := 0;
  bspace := false
end OUT_EOLN;



procedure OUT_CEOL;
begin
  if buffer.length > 0 then OUT_EOLN
end OUT_CEOL;



procedure OUT_BLINE;
var
  i: integer;

begin
  i := icolumn mod maxcodespace;
  if i > 0 then WRITEV( buffer:false, ' ':i )
end OUT_BLINE;



procedure OUT_CHAR( ch: char );
var
  i: integer;

begin
  if buffer.length = 0 then OUT_BLINE;
  WRITEV( buffer:false, ch )
end OUT_CHAR;



procedure OUT_SPACE;
begin
  if buffer.length >= maxcodespace then OUT_EOLN
                                   else OUT_CHAR( ' ' );
  bspace := false
end OUT_SPACE;



procedure OUT_SEPAR( ch: char );
begin
  OUT_CHAR( ch );
  if buffer.length >= maxcodespace then OUT_EOLN
                                   else bspace := false
end OUT_SEPAR;



procedure OUT_BEGIN;
begin
  if bspace then OUT_SPACE;
  OUT_CHAR( '{' );
  icolumn := icolumn + 2;
  bspace := false
end OUT_BEGIN;



procedure OUT_END;
begin
  OUT_CEOL;
  icolumn := icolumn - 2;
  if icolumn < 0 then icolumn := 0;
  OUT_CHAR( '}' );
  OUT_EOLN;
  end_c_block := true;
  bspace := false
end OUT_END;



procedure OUT_SEMICOLON;
begin
  bsemicolon := false;
  OUT_CHAR( ';' );
  OUT_EOLN
end OUT_SEMICOLON;



procedure OUT_PASTR( in_var str: string );
begin
  if buffer.length >= maxcodespace then OUT_EOLN;
  if buffer.length = 0 then OUT_BLINE
                       else if bspace then OUT_SPACE;
  WRITEV( buffer:false, str );
  bspace := true
end OUT_PASTR;



procedure OUT_PASCH( ch: char );
begin
  if buffer.length >= maxcodespace then OUT_EOLN;
  if buffer.length = 0 then OUT_BLINE
                       else if bspace then OUT_SPACE;
  WRITEV( buffer:false, ch );
  bspace := true
end OUT_PASCH;



procedure OUT_CHARR( var cha: packed array[$sz: integer] of char );
begin
  if buffer.length >= maxcodespace then OUT_EOLN;
  if buffer.length = 0 then OUT_BLINE
                       else if bspace then OUT_SPACE;
  WRITEV( buffer:false, cha );
  bspace := true
end OUT_CHARR;



procedure OUT_STR( in_var str: string );
begin
  if buffer.length >= maxcodespace then OUT_EOLN;
  if buffer.length = 0 then OUT_BLINE
                       else if bspace then OUT_SPACE;
  WRITEV( buffer:false, str );
  bspace := false
end OUT_STR;



procedure OUT_CHR( ch: char );
begin
  if buffer.length >= maxcodespace then OUT_EOLN;
  if buffer.length = 0 then OUT_BLINE
                       else if bspace then OUT_SPACE;
  WRITEV( buffer:false, ch );
  bspace := false
end OUT_CHR;



procedure OUT_PSTR( sp: str_ptr );
var
  i: integer;

begin
  if sp <> nil then
  with sp^ do
  if length > 0 then
  begin
    for i := 1 to length do OUT_CHAR( body[i] );
    OUT_EOLN
  end
end OUT_PSTR;



procedure OUT_SREG( ty: typ_ptr );
begin
  if ty <> nil then
  case ty^.typ_form of
    form_nil, form_pointer:        OUT_STR( 'Rgv' );
    form_range, form_set, form_wset,
    form_char, form_lit, form_int: OUT_STR( 'Rgi' );
    form_single:                   OUT_STR( 'Rgf' );
    form_double:                   OUT_STR( 'Rgg' );
  otherwise
  end;
end OUT_SREG;



procedure OUT_OPER( pcod: pcod_codes );
var
  i: integer;

begin
  with cop_table[pcod] do
  if cop_lnam > 0 then
  begin
    if (cop_kind >= cop_fn1) and bspace then OUT_SPACE else bspace := false;
    if cop_snam[1] = '_' then OUT_STR( 'PAS_' );
    for i := 1 to cop_lnam do
      OUT_CHAR( cop_snam[i] );
    if cop_kind >= cop_fn1 then bspace := true
  end;
  if buffer.length >= maxcodespace then OUT_EOLN
end OUT_OPER;



procedure OUT_IDCH( ch: char );
begin
  if bcase then
  begin
    if (ch >= 'a') and (ch <= 'z') then ch := CHR( ORD( ch ) + sch )
    else
    if (ch >= 'A') and (ch <= 'Z') then ch := CHR( ORD( ch ) - sch );
    OUT_CHAR( ch );
    bcase := false
  end
  else
    if ch = '$' then
    begin
      OUT_CHAR( '_' );
      bcase := true
    end
    else OUT_CHAR( ch )
end OUT_IDCH;



procedure OUT_IDENT( ide: nam_ptr );
var
  i: integer;

begin
  bcase := false;
  if buffer.length >= maxcodespace then OUT_EOLN;
  if buffer.length = 0 then OUT_BLINE
                       else if bspace then OUT_SPACE;

  if ide <> nil then
  with ide^ do
    if l > maxidentsize then
    begin
      for i := 1 to maxidentsize - 4 do OUT_IDCH( s[i] );
      for i := l-3 to l do OUT_IDCH( s[i] );
    end
    else
      for i := 1 to l do  OUT_IDCH( s[i] );
  bspace := true
end OUT_IDENT;



procedure OUT_FULL_IDENT( ide: nam_ptr );
var
  i: integer;

begin
  bcase := false;
  if buffer.length >= maxcodespace then OUT_EOLN;
  if buffer.length = 0 then OUT_BLINE
                       else if bspace then OUT_SPACE;

  if ide <> nil then
  with ide^ do
    for i := 1 to l do  OUT_IDCH( s[i] );
  bspace := true
end OUT_FULL_IDENT;



procedure OUT_F_IDENT( ide: nam_ptr );
var
  i: integer;

begin
  if buffer.length >= maxcodespace then OUT_EOLN
                                   else if bspace then OUT_SPACE;
  bcase := false;
  if ide <> nil then
  begin
    OUT_CHAR( 'F' );
    OUT_CHAR( '_' );
    with ide^ do
      if l > maxidentsize - 2 then
      begin
        for i := 1 to maxidentsize - 6 do OUT_IDCH( s[i] );
        for i := l-3 to l do OUT_IDCH( s[i] );
      end
      else
        for i := 1 to l do OUT_IDCH( s[i] )
  end;
  bspace := true
end OUT_F_IDENT;



procedure OUT_MODULE_CHAR( ch: char );
begin
  if (ch >= 'a') and (ch <= 'z') then ch := CHR( ORD( ch ) + sch )
  else if ch = '$' then ch := '_';
  OUT_CHAR( ch )
end OUT_MODULE_CHAR;



procedure OUT_MODULE_IDENT( ide: nam_ptr );
var
  i, len: integer;
  ch: char;

begin
  bcase := false;
  if buffer.length >= maxcodespace then OUT_EOLN
                                  else if bspace then OUT_SPACE;
  with pas_main^.pro_stdname^ do
    if l > 11 then
    begin
      len := 11;
      for i := 1 to 7 do    OUT_MODULE_CHAR( s[i] );
      for i := l-3 to l do  OUT_MODULE_CHAR( s[i] )
    end
    else
    begin
      len := l;
      for i := 1 to l do    OUT_MODULE_CHAR( s[i] )
    end;

  OUT_CHAR( '_' );

  if ide <> nil then
  with ide^ do
    if l+len < maxidentsize then
      for i := 1 to l do OUT_IDCH( s[i] )
    else
    begin
      len := maxidentsize - len - 5;
      for i := 1 to len do OUT_IDCH( s[i] );
      for i := l-3 to l do OUT_IDCH( s[i] )
    end
end OUT_MODULE_IDENT;



procedure OUT_INT( iv: integer );
begin
  if bspace then OUT_SPACE;
  WRITEV( buffer:false, iv:0 );
  bspace := true
end OUT_INT;



procedure OUT_INT1( iv: integer );
begin
  WRITEV( buffer:false, iv:0 );
  bspace := false
end OUT_INT1;



procedure OUT_LABEL( il: integer; bsm: boolean );
begin
  OUT_CEOL;
  WRITEV( buffer:false, 'L_l_', il:0, ':' );
  if bsm then WRITEV( buffer:false, ' ;' );
  OUT_EOLN
end OUT_LABEL;



procedure OUT_OFFSET( iv: integer );
begin
  if iv > 0 then
  begin
    OUT_SEPAR( '+' ); OUT_INT( iv )
  end
  else
  if iv < 0 then
  begin
    OUT_SEPAR( '-' ); OUT_INT( -iv )
  end
end OUT_OFFSET;



procedure OUT_SINGLEV( fv: double );
begin
  if buffer.length >= maxcodespace then OUT_EOLN
                                   else if bspace then OUT_SPACE;
  { in Floatting and Without Space }
  WRITEV( buffer:false, SINGLE( fv ):-16:6:3 );
  { /// STR_PUT_FLOAT( buffer, fv, 16, 1, 7, 3 ); }
  bspace := true
end OUT_SINGLEV;



procedure OUT_DOUBLEV( fv: double );
begin
  if buffer.length >= maxcodespace then OUT_EOLN
                                   else if bspace then OUT_SPACE;
  WRITEV( buffer:false, fv:-24:15:3 ); { in Floatting and Without Space char }
  { /// STR_PUT_FLOAT( buffer, fv, 24, 1, 15, 3 ); }
  bspace := true
end OUT_DOUBLEV;



procedure OUT_DIGIT( dig: integer );
var
  dig_tab: array[0..9] of char := ('0', '1', '2', '3', '4', 
                                   '5', '6', '7', '8', '9');
begin
  OUT_CHAR( dig_tab[dig] )
end OUT_DIGIT;



procedure GET_C_TYPE( var s: string; ty: typ_ptr; ct: char := 'V' );
begin
  if ty <> nil then
  with ty^ do
    if typ_simple then
    case typ_form of
      form_char:  s := 'C';

      form_ennum,
      form_wlit,
      form_set,
      form_wset,
      form_lit: case typ_size of
                  1:   s := 'UB';
                  2:   s := 'UW';
                  3,4: s := 'UL';
                otherwise
                end;

      form_int: case typ_size of
                  1:   if typ_unsigned then s := 'UB'
                                       else s := 'SB';
                  2:   if typ_unsigned then s := 'UW'
                                       else s := 'SW';
                  3,4: if typ_unsigned then s := 'UL'
                                       else s := 'SL';
                  5,6,7,8:
                       if typ_unsigned then s := 'UQ'
                                       else s := 'SQ';
                otherwise
                end;

      form_range:  GET_C_TYPE( s, typ_parent );

      form_single: s := 'F';
      form_double: s := 'G';
    otherwise
      s := 'V';
    end
    else s := ct
end GET_C_TYPE;



procedure OUT_C_TYPE( ty: typ_ptr );
var
  s: string( 2 );

begin
  GET_C_TYPE( s, ty, 'S' );
  OUT_PASTR( s );
end OUT_C_TYPE;



procedure OUT_CREG_TYPE( p: all_ptr );
begin
  bspace := false;
  if p <> nil then
  with p^ do
    if all_typ <> nil then OUT_C_TYPE( all_typ )
end OUT_CREG_TYPE;



procedure OUT_LEX_SPC( lex: integer );
begin
  OUT_PASTR( 'Lx' );
  OUT_CHAR( '[' );
  OUT_INT1( lex );
  OUT_CHAR( ']' );
  bspace := false
end OUT_LEX_SPC;



procedure OUT_DISP_SPC( disp: integer; bad: boolean );
begin
  if bad then
  begin
    if disp <> 0 then
    begin
      if disp > 0 then OUT_SEPAR( '+' );
      OUT_INT( disp )
    end
  end
  else
  begin
    OUT_SEPAR( '[' ); OUT_INT( disp ); OUT_SEPAR( ']' )
  end;
  bspace := false
end OUT_DISP_SPC;



procedure OUT_IMMED_CTE( val: val_ptr; bad: boolean );
{ Output a literal value on the C source, if bad then out an address reference }
begin
  with val^ do
  if val_all = nil then
  begin
    if bad then OUT_CHR( '&' );
    bspace := false;
    case val_kind of
      form_char, form_lit,
      form_int,  form_range: OUT_INT( val_ival );
      form_set,  form_wset:  OUT_INT( val_set.siv );
      form_single:  begin  bspace := false; OUT_SINGLEV( val_rval )  end;
      form_double:  begin  bspace := false; OUT_DOUBLEV( val_rval )  end;
      form_pointer, form_nil:     OUT_STR( 'NULL' );
      form_wwset: OUT_CHR( '0' );
    otherwise
      SRC_ERROR( mdnam, 998, e_fatal )
    end
  end
end OUT_IMMED_CTE;



procedure OUT_M_BYTE( iv: integer );
begin
  if bcomma then OUT_CHAR( ',' );
  if icolnbr >= imaxcol then
  begin
    icolnbr := 0; OUT_EOLN
  end
  else if bspace then OUT_SPACE;
  WRITEV( buffer:false, iv:3 );
  icolnbr := icolnbr + 1;
  bcomma  := true;
  bspace  := true
end OUT_M_BYTE;



procedure OUT_M_WORD( iv: integer );
begin
  in_hexa.uw0 := iv;
  if cmp_bytesmd then
  begin
    OUT_M_BYTE( in_hexa.bt[cmp_swapt[bsw_w+0]] );
    OUT_M_BYTE( in_hexa.bt[cmp_swapt[bsw_w+1]] )
  end
  else
  begin
    OUT_M_BYTE( in_hexa.ub0 );
    OUT_M_BYTE( in_hexa.ub1 )
  end
end OUT_M_WORD;



procedure OUT_M_LONG( iv: integer );
var
  i: integer;

begin
  in_hexa.li0 := iv;
  if cmp_bytesmd then
    for i := 0 to 3 do  OUT_M_BYTE( in_hexa.bt[cmp_swapt[bsw_l+i]] )
  else
    for i := 0 to 3 do  OUT_M_BYTE( in_hexa.bt[i] )
end OUT_M_LONG;



procedure OUT_M_QUAD( iv: integer );
var
  i: integer;

begin
  in_hexa.li0 := iv;
  if cmp_bytesmd then
    for i := 0 to 7 do  OUT_M_BYTE( in_hexa.bt[cmp_swapt[bsw_q+i]] )
  else
    for i := 0 to 7 do  OUT_M_BYTE( in_hexa.bt[i] )
end OUT_M_QUAD;



procedure OUT_M_INT( iv, size: integer );
begin
  case size of
    1: OUT_M_BYTE( iv );
    2: OUT_M_WORD( iv );
    3,4:
       OUT_M_LONG( iv );
    5,6,7,8:
       OUT_M_QUAD( iv );
  otherwise
  end
end OUT_M_INT;



procedure OUT_M_SINGLE( fv: double );
var
  i: integer;

begin
  in_hexa.fl0 := fv;
  if cmp_bytesmd then
    for i := 0 to 3 do  OUT_M_BYTE( in_hexa.bt[cmp_swapt[bsw_f+i]] )
  else
    for i := 0 to 3 do  OUT_M_BYTE( in_hexa.bt[i] )
end OUT_M_SINGLE;



procedure OUT_M_DOUBLE( fv: double );
var
  i: integer;

begin
  in_hexa.dbl := fv;
  if cmp_bytesmd then
    for i := 0 to 7 do  OUT_M_BYTE( in_hexa.bt[cmp_swapt[bsw_g+i]] )
  else
    for i := 0 to 7 do  OUT_M_BYTE( in_hexa.bt[i] )
end OUT_M_DOUBLE;



procedure OUT_M_PAD( nb: integer );
begin
  while nb > 0 do
  begin
    OUT_M_BYTE( 0 );
    nb := nb - 1
  end
end OUT_M_PAD;



procedure OUT_PROC_IDENT( pr: pro_ptr; icall: boolean );
begin
  with pr^ do
    case pro_pkind of
      pro_standard, pro_global, pro_external, pro_main, pro_package:
        OUT_IDENT( pro_stdname );

      pro_decl, pro_inline:
        OUT_MODULE_IDENT( pro_stdname );

      pro_formal:
        begin
          if icall then
          begin  bspace := false; OUT_PASTR( '(*' )  end;
          OUT_F_IDENT( pro_stdname );
          if icall then OUT_SEPAR( ')' )
        end;

    otherwise
      OUT_PASTR( '***ILLEGAL***' )
    end
end OUT_PROC_IDENT;



procedure GEN_SCAN( head: lgt_ptr; prior: integer; blvl: boolean ); forward;



procedure GEN_CALL( head: lgt_ptr; bindir: boolean ); forward;
(*  --- To recall the offset names for the swap byte tables ---
  bsw_w =  0;                  { Byte Swap table Offset for Word Integer - and 16 bits pointer }
  bsw_l =  2;                  { Byte Swap table Offset for Long Integer - and 32 bits pointer }
  bsw_q =  6;                  { Byte Swap table Offset for Quad Integer - and 64 bits pointer }
  bsw_f = 14;                  { Byte Swap table Offset for 32 bits Single Precision Floating }
  bsw_g = 18;                  { Byte Swap table Offset for 64 bits Double Precision Floating }
*)



procedure INIT_SWAP_TABLES;
{ Prepare the Swap table with the Default of the Runing Machine.
  We suppose that the bytes order is the same for each double precision
  floating number long words and for a long integer number (32 bits) and also that
  the long word order ios the same for 64 bits Integer/pointer and 64 bits Floating number.
  The default byte ordering of the target computer is supposed to be the same of the
  current computer used for the compilation.
}
var
  iv:  integer;
  brv: boolean;

begin
  in_hexa.li0 :=   0;          { Double Precision (64 bits) Floating point Number Long word Order Test ... }
  in_hexa.li1 :=   0;          { ... by set to zero the two long words and put the floating value 1.0 after }
  in_hexa.dbl := 1.0;          { 1.0 can have only the exponant set to non zero value (in excess encoding) }
  brv := (in_hexa.li1 = 0);    { Set the reverse order flag when the exponant is not in the last word }
  { Hypothese of no swap for 64 bits }

  { Enable bytesmd correction }
   cmp_bytesmd := true;
  { Set the default swap table }
  { 16 bits integer/pointer }
  in_hexa.uw0 := 0 + 256;      { Ok for 16 bits word }
  for i := 0 to 1 do
  begin
    cmp_srcta[bsw_w+i] := in_hexa.bt[i];
    cmp_trgta[bsw_w+i] := in_hexa.bt[i];
    cmp_swapt[bsw_w+i] := i
  end;
  in_hexa.li0 := 0 + 256*(1 + 256*(2 + 256*3 ));       { Ok for 32 bits low long word }
  in_hexa.li1 := 4 + 256*(5 + 256*(6 + 256*7 ));       { Ok for 32 bits high long word }
  { For 32 bits integer/pointer }
  for i := 0 to 3 do
  begin
    cmp_srcta[bsw_l+i] := in_hexa.bt[i];
    cmp_trgta[bsw_l+i] := in_hexa.bt[i];
    cmp_swapt[bsw_l+i] := i
  end;
  { 64 bits integer/pointer }
  { This test is presently not supported }
  { But the constant pointer (of 64 bits) are never used in a CPAS }
  for i := 0 to 7 do
  begin
    cmp_srcta[bsw_q+i] := in_hexa.bt[i];
    cmp_trgta[bsw_q+i] := in_hexa.bt[i];
    cmp_swapt[bsw_q+i] := i
  end;
  { For 32 bits float }
  for i := 0 to 3 do
  begin
    cmp_srcta[bsw_f+i] := in_hexa.bt[i];
    cmp_trgta[bsw_f+i] := in_hexa.bt[i];
    cmp_swapt[bsw_f+i] := i
  end;
  { 64 bits double }
  for i := 0 to 7 do
  begin
    cmp_srcta[bsw_g+i] := in_hexa.bt[i];
    cmp_trgta[bsw_g+i] := in_hexa.bt[i];
    cmp_swapt[bsw_g+i] := i
  end;
  if brv then  { Permute the Two long words for 64 bits integer/pointer and double }
               { We suppose the same long word order for floating and quad integer }
    for i := 0 to 3 do
    begin
      iv := cmp_srcta[bsw_q+i]; cmp_srcta[bsw_q+i] := cmp_srcta[bsw_q+4+i]; cmp_srcta[bsw_q+4+i] := iv;
      iv := cmp_trgta[bsw_q+i]; cmp_trgta[bsw_q+i] := cmp_trgta[bsw_q+4+i]; cmp_trgta[bsw_q+4+i] := iv;
      iv := cmp_srcta[bsw_g+i]; cmp_srcta[bsw_g+i] := cmp_srcta[bsw_g+4+i]; cmp_srcta[bsw_g+4+i] := iv;
      iv := cmp_trgta[bsw_g+i]; cmp_trgta[bsw_g+i] := cmp_trgta[bsw_g+4+i]; cmp_trgta[bsw_g+4+i] := iv
    end
end INIT_SWAP_TABLES;



[global]
procedure DATA_FORMAT_SET( ty: typ_ptr );
const
  mdnam = 'DFRM';

var
  i, j, k, l: integer;
  tmp: array[0..7] of integer;
  cid: str_ptr;

begin
  with sy_sym, ty^ do
  begin
    INSYMBOL;
    if sy = stringconst then   { C type specified }
    begin
      cid := nil;
      NEW( cid, sy_string.length );
      cid^ := sy_string;
      case typ_form of
        form_char:   cmp_pascal_cdef[0]  := cid;
        form_int: case typ_size of
                    1:       if typ_unsigned then cmp_pascal_cdef[2] := cid
                                             else cmp_pascal_cdef[1] := cid;
                    2:       if typ_unsigned then cmp_pascal_cdef[4] := cid
                                             else cmp_pascal_cdef[3] := cid;
                    3,4:     if typ_unsigned then cmp_pascal_cdef[6] := cid
                                             else cmp_pascal_cdef[5] := cid;
                    5,6,7,8: if typ_unsigned then cmp_pascal_cdef[8] := cid
                                             else cmp_pascal_cdef[7] := cid;
                  otherwise
                  end;
        form_single: cmp_pascal_cdef[ 9] := cid;
        form_double: cmp_pascal_cdef[10] := cid;
      otherwise
        SRC_ERROR( mdnam, 999, e_fatal )
      end;
      INSYMBOL;
      if sy = colon then INSYMBOL
    end;
    { Get the byte succession ordering }
    if sy = lparen then
    begin
      { For first use Init the swap tables }
      if not cmp_bytesmd then INIT_SWAP_TABLES;
      case typ_form of
        form_char, form_lit, form_int, form_pointer:
          case typ_size of
            2:       begin  i :=  bsw_w; l := 2  end;
            3,4:     begin  i :=  bsw_l; l := 4  end;
            5,6,7,8: begin  i :=  bsw_q; l := 8  end;
          otherwise
            i := -1
          end;
        form_single: begin  i := bsw_f; l := 4  end;
        form_double: begin  i := bsw_g; l := 8  end;
      otherwise
        SRC_ERROR( mdnam, 993, e_fatal );
        SKIP_SYMBOL( semicolon );
        return
      end;
      { Now get and set the type related swap type }
      j := 0;
      repeat
        { Fill the target table }
        INSYMBOL;
        if sy <> intconst then SRC_ERROR( mdnam, 33, e_severe )
        else
          if (sy_ival < 0) or (sy_ival >= l) then
          begin
            SRC_ERROR( mdnam, 37, e_severe );
            sy_ival := 0
          end
          else cmp_trgta[i+j] := sy_ival;
        INSYMBOL;      { Gobble up the number }
        j := j + 1
      until (sy <> comma) or (j = l);
      if sy = rparen then INSYMBOL
                     else SRC_ERROR( mdnam, 23, e_error );

      if sy = colon then
      begin
        INSYMBOL;
        if sy = lparen then INSYMBOL
                       else SRC_ERROR( mdnam, 22, e_error );
        j := 0;
        repeat
          { Fill the specified source table itb }
          INSYMBOL;
          if sy <> intconst then SRC_ERROR( mdnam, 33, e_severe )
          else
            if (sy_ival < 0) or (sy_ival >= l) then
            begin
              SRC_ERROR( mdnam, 37, e_severe );
              sy_ival := 0
            end
            else
            begin
              cmp_srcta[i+j] := sy_ival;       { Set the source table ... }
              tmp[sy_ival]   := j              { and the inverse source permutation table }
            end;
          INSYMBOL;                            { Gobble up the number }
          j := j + 1
        until (sy <> comma) or (j = l);
        if sy = rparen then INSYMBOL
                       else SRC_ERROR( mdnam, 23, e_error )
      end
      else                                     { Default source mode }
        for j := 0 to l - 1 do
          tmp[cmp_srcta[i+j]] := j;            { Rebuild the special permuttation table }

      { Build the Swap byte transformation table }
      for j := 0 to l - 1 do
        cmp_swapt[i+j] := cmp_trgta[i+tmp[j]]  { Swap table = trg*(src^(-1)) }
    end
  end
end DATA_FORMAT_SET;



[global]
procedure GENERATION_SETTING;
{ Called during initialization setting to force any specific type setting }
var
  i:   integer;
  path:    string;

begin
  cmp_bytesmd   := false;                       { Set default byte exchange to No Byte Exchange }
  { Locate the cpas_defs.h header file for C compilation }
  i := GET_LOGICAL( sy_string, 'CPAS_C_HEADER' );
  if (i < 0) or (sy_string.length = 0) then
  begin
    case cpas_envdir_idx of
      5, { Main standard Root /etc/cpas for the system }
      4: { /usr/local/etc/cpas or equivalent as defined by C }
        sy_string := '<cpas_defs.h>';

      3, { CPAS_Root in Home directory }
      2: { User defined CPAS_Root }
        begin
          i := INDEX( cpas_env_dir, '/etc/cpas', -1 );
          if i = 0 then sy_string := cpas_env_dir||'cpas_defs.h'
                   else if i = 1 then sy_string := '<cpas_defs.h>'
                                 else sy_string := SUBSTR( cpas_env_dir, 1, i )||'include/cpas_defs.h'
        end;

    otherwise
      sy_string := cpas_env_dir||'cpas_defs.h'
    end;
    { Translate the path without logical reference }
    if INDEX( sy_string, ':' ) > 0 then SET_FILE_SPECIFICATION( path, sy_string, [case_ena_file] )
                                   else path := sy_string;
    if (path[1] <> '>') and (path[1] <> '<') then
      if not FILE_ACCESS_CHECK( path, 4 {Read access} ) then
      begin
        WRITELN;
        WRITELN( ' *** PCMP-CPAS-COMPILER (GENC) ERROR: Cannot open the CPAS C header file "', sy_string, '"' );
        WRITELN;
        PASCAL_EXIT( 2 )
      end
    end;

  NEW( cenv_fil, path.length );
  cenv_fil^ := path;
  for i := 0 to 10 do  cmp_pascal_cdef[i] := nil;
  global_labenv := false                       { Assume no global label environment }
end GENERATION_SETTING;



[global]
procedure GENERATION_PRAGMA;
{ Called by %pragma to set any code generator option }
const
  ncopt = 8;

var
  { Warning this table must be modified when the identifier size is changed }
  optcnm: [static] array[1..ncopt] of id_name := (
    (13,'c_environment  '),    { 1--C environment specification file }
    (11,'c_interface    '),    { 2--C To set the C interf. mode in the user code }
    (13,'noc_interface  '),    { 3--C To set the C interf. mode in the user code }
    (11,'parenthesys    '),    { 4--C Enable use of ( ..., ... ) C operator }
    (13,'noparenthesys  '),    { 5--C Disable use of ( ..., ... ) C operator }
    ( 7,'dynamic        '),    { 6--C Enable the Dynamic variable size manager }
    ( 9,'nodynamic      '),    { 7--C Disable the Dynamic variable size manager }
    ( 6,'c_code         ')     { 8--C to set a C headline }
  );

  stm: cmac_ptr;

  i, j, k:   integer;
  bf, swflg, parflg: boolean;

begin
  with sy_sym do
  begin
    parflg := (sy = lparen);
    repeat
      if parflg then INSYMBOL_SRC;
      if sy <> identsy then SRC_ERROR( mdnam, 38, e_error )
      else
      begin
        i  := 1;
        bf := false;
        repeat
          if MATCH( sy_ident, optcnm[i] ) <> 0 then i := i + 1
                                               else bf := true
        until (i > ncopt) or bf;
        if bf then
        begin
          INSYMBOL_SRC;
          case i of
            1: if sy = stringconst then
               begin                           { C environment }
                 { for logical get the translation }
                 if cenv_fil <> nil then DISPOSE( cenv_fil );
                 NEW( cenv_fil, sy_string.length );
                 cenv_fil^ := sy_string;
                 INSYMBOL_SRC;                 { Gobble up the object file name }
               end else SRC_ERROR( mdnam, 904, e_fatal );

            2, 3: sy_init_mod := (i = 2);

            4, 5: cmp_lparop  := (i = 4);

            6, 7: cmp_dynamic := (i = 6);

            8: if sy = stringconst then
               begin                           { C head line for macro definition or other }
                 NEW( stm );
                 with stm^ do
                 begin
                   cmac_nxt := nil;
                   cmac_str := nil;
                   NEW( cmac_str, sy_string.length );
                   cmac_str^ := sy_string
                 end;
                 if cmac_hde = nil then cmac_hde := stm
                                   else cmac_lst^.cmac_nxt := stm;
                 cmac_lst := stm;
                 INSYMBOL_SRC
               end else SRC_ERROR( mdnam, 904, e_fatal );

          otherwise
          end
        end
        else
          SRC_ERROR( mdnam, 7001, e_error )
      end
    until (sy <> comma) or not parflg;
    if parflg then if sy = rparen then INSYMBOL_SRC
                                  else SRC_ERROR( mdnam, 23, e_fatal )
  end
end GENERATION_PRAGMA;





{**************************************************************************}
{**************************************************************************}
{***************  C   Code  Generation  P A S S - 1   *********************}
{**************************************************************************}
{**************************************************************************}





procedure GEN_FENTRY_CTYP( pr: pro_ptr );
{ Procedure to generate the related C type of the given Pascal entry (C function).
}
const
  mdnam = 'GENF';

var
  ctp:      ctyp_ptr;
  str: string( 255 );
  snm:  string( 16 );


  procedure GEN_C_ARG_LIST( var st: string; pr: pro_ptr );
  var
    idf:     ide_ptr;
    bfr:     boolean;
    sty: string( 2 );

  begin
    bfr :=           false;
    idf := pr^.pro_parmlst;
    if pr^.pro_typ <> nil then
      if not pr^.pro_typ^.typ_simple then idf := idf^.ide_nxt; { Skip the return value arg. when it is a function of complex type }
    while idf <> nil do
    with idf^ do
    begin
      if bfr then WRITEV( str:false, ',' )
             else bfr := true;
      if ide_class = cla_fentry then
      begin                                            { The formal argument is also a formal entry }
        GEN_FENTRY_CTYP( idf^.ide_entry );             { Recursive call for formal entry inside formal entry arg. list }
        with ide_entry^.pro_stdname^ do
        WRITEV( st:false, s:l )                        { Put the deduced formal C type }
      end
      else
      begin
        GET_C_TYPE( sty, ide_typ );                    { Get the formal arg. type }
        WRITEV( st:false, sty )                        { Standard formal argument }
      end;
      idf := ide_nxt
    end;
  end GEN_C_ARG_LIST;


begin { GEN_FENTRY_CTYP }
  str.length := 0;
  if pr <> nil then
  with pr^ do
    if pro_stdname = nil then                          { Do only One time for each defined fentry type record }
    begin
      WRITEV( snm, 'Re', ctyp_count:0 );               { Build the type name ... }
      ctyp_count := ctyp_count + 1;
      NEW( pro_stdname );
      for ii := 1 to snm.length do                     { ... and put it in the formal stdname }
        pro_stdname^.s[ii] := snm[ii];
      pro_stdname^.l := snm.length;
      if pro_typ <> nil then GET_C_TYPE( str, pro_typ )
                        else str := 'void';
      WRITEV( str:false, ' (*', snm, ')(' );           { Forms the string "<C-rtype> (* <entry_type>) (" }
      GEN_C_ARG_LIST( str, pr );                       { Append the argument list }
      WRITEV( str:false, ')' );                        { Append the final ")" }

      NEW( ctp );                                      { Create the C type record }
      with ctp^ do
      begin
        ctyp_nxt := nil;                               { Init the next link }
        NEW( ctyp_ide, str.length );                   { Set the C type definition }
        ctyp_ide^ := str
      end;
      if ctyp_first = nil then ctyp_first := ctp       { Append the record to the C type record list }
                          else ctyp_last^.ctyp_nxt := ctp;
      ctyp_last := ctp
    end
end GEN_FENTRY_CTYP;



procedure GEN_ALL_FENTRY_CTYPE( pr: pro_ptr );
var
  ty: typ_ptr;
  ds: integer;
  ow: pro_ptr;

begin
  if pr <> nil then
  begin
(*
    with pr^ do
      if pro_stdname <> nil then
        with pro_stdname^ do
          WRITELN( ' G_A_FENTRY for Entry "', s:l, '"' );
*)
    ds := curr_disp;
    while (lex_ident_tree[ds].disp_owner = pr) and (ds > 0) do
    begin
      ty := lex_ident_tree[ds].disp_typ_hde;

      while ty <> nil do
      with ty^ do
      begin
(*
        WRITELN( '    Type_form = ', typ_form );
*)
        if typ_form = form_fentry then GEN_FENTRY_CTYP( typ_entry );
        ty := typ_nxt
      end;
(*
      WRITELN( ' ---------- ' );
*)
      ds := ds - 1
    end
  end
end GEN_ALL_FENTRY_CTYPE;

    

function SIZE_DATA_PART( var size: integer; vp: val_ptr ): integer;
{ To size a data (const) definition }
var
  bsz:           boolean;
  iad, jad, isz: integer;
  vp1:           val_ptr;

begin
  bsz := true;
  with vp^ do
  begin
    { Set the start address (with alignement, using the type info.) }
    iad := IDE_TYP_ALIGN( size, val_typ );
    case val_kind of
      form_string: { Out as string:  val_string }
        begin
          size := iad + val_size;              { Update the size by the cte. size }
          if val_typ <> nil then
            if val_typ^.typ_form = form_record then
              { Add the room for the descriptor when required }
              size := size + stri_stroffset;
          bsz := false                         { The allocation is done }
        end;

      form_record,
      form_array:  { Out as an array or a record: val_lst^ list }
        begin
          vp1  := val_lst;
          isz  := iad;
          while vp1 <> nil do
          begin { For each element of record or array }
            { We update isz by the size of element }
            jad := SIZE_DATA_PART( isz, vp1 );
            if val_kind = form_array then
              size := IDE_TYP_ALIGN( isz, val_typ );
            vp1 := vp1^.val_next
          end;
          { Note : The size of the record or array is given by
                   the val_rec record of the record or array. }
          size := iad + val_size;
          bsz := false
        end;

      form_lset,
      form_wlset:
        begin
          size := iad + val_typ^.typ_size;
          bsz := false
        end;

      form_null:
        begin
          size := iad + val_size;
          bsz  := false
        end;

    otherwise
    end;
    { when it is not already done set the new data space size }
    if bsz then size := iad + val_typ^.typ_size
  end;
  SIZE_DATA_PART := iad
end SIZE_DATA_PART;



procedure SIZE_DATA( pr: pro_ptr );
{ To SIZE the data (const) program section }
var
  p: all_ptr;

begin
  p := all_fdata;
  { Erase the compiler value list header for the next module }
  all_fdata := nil;
  all_ldata := nil;
  while p <> nil do
  with p^ do
  if all_psect = -1 then
  begin { Local data }
    all_disp := SIZE_DATA_PART( module_dat_size, all_cte );
    if vald_first = nil then vald_first := all_cte
                        else vald_last^.val_next := all_cte;
    vald_last := all_cte;
    p := all_nxt
  end
  else
  begin { Global obj of an environment }
    all_disp := SIZE_DATA_PART( module_gdat_size, all_cte );
    if vagd_first = nil then vagd_first := all_cte
                        else vagd_last^.val_next := all_cte;
    vagd_last := all_cte;
    p := all_nxt
  end
end SIZE_DATA;



procedure SIZE_STATIC( pr: pro_ptr );
{ To SIZE the static (const) program sections (with initial value or not) }
var
  p, q:              all_ptr;
  breg, bnide, bidh: boolean;

begin
  p := all_fstatic; { Take the head list of static declarations }
  { Erase the compiler static list header for the next module }
  all_fstatic := nil;
  while p <> nil do
  begin
    with p^ do
    begin
      q := all_nxt;
      all_nxt := nil;
      bnide := true;
      if all_ide <> nil then
      with all_ide^ do
      begin { The related identifier must be defined }
(* Probablement inutil maintenant grace aux modification faites par appel de la
 * nouvelle procedure GEN_ALL_FENTRY_CTYPE (étape GENERATE_CODE_P1) *)
        if ide_typ <> nil then
        with ide_typ^ do { For each entry pointer type we define the attached C type }
          if typ_form = form_fentry then GEN_FENTRY_CTYP( typ_entry );

        if (ide_inival <> nil) { and (not (var_initialized in ide_vacc)) } then
          if all_psect = -1 then
          begin { Static variable Section with initialization }
            all_disp := SIZE_DATA_PART( module_sti_size, all_cte );
            { Build the constant value list }
            if alsi_first = nil then alsi_first := p
                                else alsi_last^.all_nxt := p;
            alsi_last := p;
            bnide := false
          end
          else
          begin { Static global variable Section with initialization }
            all_disp := SIZE_DATA_PART( module_gsti_size, all_cte );
            { build the constant value list }
            if algi_first = nil then algi_first := p
                                else algi_last^.all_nxt := p;
            algi_last := p;
            bnide := false
          end
      end;

      if bnide then
      begin { Static variable without initialization }
        breg := false;
        if ((all_kind = var_decl) or (all_kind = var_tmp)) and
           (not (var_intaccess in all_acc)) then
          if all_ide <> nil then
          with all_ide^ do
          begin { Object with identifier }
            if (not cmp_genenv) or (var_hidden in ide_vacc) then
              if ide_typ <> nil then
                if ide_typ^.typ_simple then breg := true
          end
          else { for var_tmp }
            if all_first_u <> nil then
            with all_first_u^ do
              if lgt_typ <> nil then
                if lgt_typ^.typ_simple then breg := true;

        if breg then
        with pr^ do
        begin { A Register is used }
          if pro_fdyn_all = nil then pro_fdyn_all := p
                                else pro_ldyn_all^.all_nxt := p;
          pro_ldyn_all := p
        end
        else
        begin
          if all_ide <> nil then if all_psect = -1 then bidh := true
                                                   else bidh := false
                            else bidh := true;
          if bidh then
          begin { ALlocation in Local static Section }
            ALL_TYP_ALLOCATE( module_sta_size, p );
            { Build the pure static variable list }
            if alls_first = nil then alls_first := p
                                else alls_last^.all_nxt := p;
            alls_last := p
          end
          else
          begin { Allocation in Global (environment) Section }
            ALL_TYP_ALLOCATE( module_gsta_size, p );
            { Build the pure static variable list }
            if algs_first = nil then algs_first := p
                                else algs_last^.all_nxt := p;
            algs_last := p
          end
        end
      end
    end;
    p := q
  end
end SIZE_STATIC;



procedure SIZE_FORMAL( pr: pro_ptr );
{ To SIZE the formal argument(s) of the compiled entry (procedure/function) }
var
  ip:    ide_ptr;
  pl:    all_ptr;
  bcopy: boolean;

begin
  with pr^ do
  begin
    ip := pro_parmlst;
    if ip <> nil then
    begin
      repeat
        bcopy := false;                        { Assume use of register possible }
        with ip^ do
          if ide_class = cla_fentry then       { For formal entry argument }
          begin
            if  (ide_f_all <> nil) and (ide_entry <> nil) then
            begin
              if prf_intaccess in ide_entry^.pro_flags then { When an internal reference:call is used, ... }
              with ide_f_all^ do
              begin                            { We must use a local automatic copy }
                all_acc := all_acc + [var_intaccess];
                GEN_FENTRY_CTYP( ide_entry );
                if pro_fdyn_all = nil then pro_fdyn_all := ide_f_all
                                      else pro_ldyn_all^.all_nxt := ide_f_all;
                pro_ldyn_all := ide_f_all
              end
            end
          end
          else                                 { For cla_varbl formal argument }
          begin
            if ide_all <> nil then
            begin
              if var_intaccess in ide_vacc then
                bcopy := true                  { Always set in auto class if internal access }
              else
                if ide_vkind = var_vformal then
                  if ide_typ <> nil then
                    if not ide_typ^.typ_simple then bcopy := true;
              if bcopy then
              begin                            { Use a local automatic variable copy }
                if pro_fdyn_all = nil then pro_fdyn_all := ide_all
                                      else pro_ldyn_all^.all_nxt := ide_all;
                pro_ldyn_all := ide_all
              end
            end
          end;
        ip := ip^.ide_nxt
      until ip = nil
    end
  end
end SIZE_FORMAL;



procedure SIZE_AUTOMATIC( pr: pro_ptr );
{ To SIZE the automatic variable procedure/function }
var
  p, q, pf, pl, pft, plt: all_ptr;
  breg: boolean;

begin
  pf := nil; pft := nil;
  pl := nil; plt := nil;
  with pr^ do
  begin
    (*
    if ireg_flag then
    begin  pro_flags := pro_flags + [prf_cdg_02]; ireg_flag := false  end;
    if freg_flag then
    begin  pro_flags := pro_flags + [prf_cdg_03]; freg_flag := false  end;
    if greg_flag then
    begin  pro_flags := pro_flags + [prf_cdg_04]; greg_flag := false  end;
    *)
    pro_dyn_size := 0;
    pro_intacc   := false;                     { Assume no internal access until shown otherwise }
    pro_reglist  := nil;                       { and no register use }
    p := pro_fdyn_all;
    while p <> nil do                          { Loop on all automatic/tmp allocation }
    begin
      with p^ do
      begin
        q := all_nxt;
        all_nxt := nil;
        breg := true;
        if var_intaccess in all_acc then
        begin                                  { Set the internal access and allocate lex ident number }
          if not pro_intacc  then
          begin                                { Allocate a new lex pointer in the module }
            pro_intacc := true;
            pro_envidx := proc_intlex_count;
            proc_intlex_count := proc_intlex_count + 1
          end;
          all_lexid := pro_envidx;
          breg := false
        end
        else
          if all_ide <> nil then
          with all_ide^ do
          begin                                { The related identifier must be defined and used }
            if ide_typ <> nil then
              if not ide_typ^.typ_simple then breg := false
          end
          else
            if all_first_u <> nil then
            with all_first_u^ do
              if lgt_typ <> nil then
                if not lgt_typ^.typ_simple then breg := false;

        if all_typ <> nil then
        with all_typ^ do { For each entry pointer type we define the attached C type }
          if typ_form = form_fentry then GEN_FENTRY_CTYP( typ_entry );


        if breg then                           { Temporary - no identifier => register }
        begin
          if pft = nil then pft := p else plt^.all_nxt := p;
          plt := p
        end
        else
          { True location to allocate }
          case all_kind of
            var_pformal,
            var_formal:
              begin { * var_formal with internal access use a local pointer }
                all_disp := IDE_TYP_ALIGN( pro_dyn_size, typ_std[form_nil] );
                pro_dyn_size := all_disp + ABS( fptr_size )
              end;

            var_vformal:
              if all_ide <> nil then
              with all_ide^ do
              if ide_typ <> nil then
              with ide_typ^ do
              begin { var_vformal with a copy to perform }
                if typ_size > 0 then
                begin
                  all_disp := IDE_TYP_ALIGN( pro_dyn_size, ide_typ );
                  pro_dyn_size := all_disp + ABS( ide_typ^.typ_size )
                end
                else
                begin { dynamic size with a register use }
                  all_disp := IDE_TYP_ALIGN( pro_dyn_size, typ_std[form_nil] );
                  pro_dyn_size := all_disp + ABS( fptr_size )
                end;
                if pf = nil then pf := p else pl^.all_nxt := p;
                pl := p
              end;

          otherwise
            { Allocation for the object }
            ALL_TYP_ALLOCATE( pro_dyn_size, p );
            { build the pure automatic variable list }
            if pf = nil then pf := p else pl^.all_nxt := p;
            pl := p
          end
      end;
      p := q
    end;
    { Reset the procedure/function allocation headers }
    pro_fdyn_all := pf;
    pro_ldyn_all := pl;
    pro_reglist  := pft;
    pro_dyn_size := IDE_TYP_ALIGN( pro_dyn_size, typ_std[form_double] )
  end
  (*
  ; with pr^ do
    if pro_stdname <> nil then with pro_stdname^ do
      WRITELN( ' *** GENC Compute a dyn size of ', pro_dyn_size:0, ' for entry "', s:l, '"' );
  *)
end SIZE_AUTOMATIC;



procedure SETUP_LABEL( pr: pro_ptr );
{ To allocate a label number for each local label }
{ and an environment pointer for label internal access when required }
var
  p1, p2:      ide_ptr;
  lbcnt, iadr: integer;
  gbllab:      boolean;

begin
  gbllab := false;
  lbcnt  :=     0;
  if pr <> nil then
  begin
    p1 := pr^.pro_labelhde;
    while p1 <> nil do
    with p1^ do
    begin                                      { Loop on all existing label }
      if [lab_defined,lab_refer] <= ide_labflg then
      begin                                    { For any defined, referenced }
        if lab_inref in ide_labflg then gbllab := true;
        if (ide_lablgt <> nil) and             { For a good label definition and ... }
           (ide_labadr < 0) then               { ... without adress number. }
          if ide_lablgt <> nil then            { To prevent a possible syntax error result }
          begin
            p2 := ide_lablgt^.lgt_lide;        { Get the synonymous label head list }
            if p2 <> nil then                  { To prevent other possible syntax error result }
              if p2^.ide_labadr >= 0 then
                ide_labadr := p2^.ide_labadr   { When the head synonymous list is OK we copy the ... } 
              else
              begin                            { ... address else we give a new ... }
                iadr  :=         lbcnt;        { ... label Number/address to the head ... }
                lbcnt := SUCC( lbcnt );        { ... synonymous list label and, if it is not ... }
                p2^.ide_labadr := iadr;        { ... the same, to our current label }
                if p2 <> p1 then ide_labadr := iadr
              end
          end
      end;
      p1 := ide_labnxt                         { Loop on the next label }
    end
  end;

  with pr^ do
  begin
    pro_labelcnt := lbcnt;
    if gbllab then                             { To flag the label environment use }
      if pro_lex = 0 then                      { Main/module program => use the static schema }
        global_labenv := true                  { and set flag to create a global label env. }
      else
      begin { * Internal routine => allocate dyn. env. ptr. to use the lex acc. }
        if not pro_intacc  then
        begin                                  { Allocate a new lex pointer in the module }
          pro_intacc := true;
          pro_envidx := proc_intlex_count;
          proc_intlex_count := proc_intlex_count + 1
        end;
        pro_labelenv := IDE_TYP_ALIGN( pro_dyn_size, typ_std[form_nil] );
        pro_dyn_size := pro_labelenv + ABS( fptr_size )
      end
  end
end SETUP_LABEL;



[global]
procedure GENERATE_CODE_P1( proc: pro_ptr );
{ Pass 1 of code Generation :
  Set all static_inited/ data allocation list.

  *** Called by CMP_PASS2 after the optimizer step ***
}
begin
  { Must be performed in this order }
  SIZE_DATA( proc );
  SIZE_STATIC( proc );
  GEN_ALL_FENTRY_CTYPE( proc );
  SIZE_FORMAL( proc );
  SIZE_AUTOMATIC( proc );
  SETUP_LABEL( proc )
end GENERATE_CODE_P1;




{**************************************************************************}
{**************************************************************************}
{***************  C   Code  Generation  P A S S - 2   *********************}
{***************                                      *********************}
{***************      Declaration     Generation      *********************}
{**************************************************************************}
{**************************************************************************}


procedure GENERATE_MODULE_HEADING;
{ Pass 2 of code Generation: generation of Module Heading. }
var
  psm: cmac_ptr;
  i, j: integer;
  c: char;
  frst: boolean;

begin
  objf_ok := false;
  frst    := true;
  if cmp_objf and not objf_open then
  begin
    i := BINARY_OBJF_OPEN;
    if i <> 0 then SRC_ERROR( mdnam, 888, e_fatal )
    else
    begin
      { Output the CPAS Banner }
      OUT_PASTR( '/*' ); OUT_PASTR( pas_page_head ); OUT_EOLN;
      OUT_PASTR( '  ***  PASCAL II implementation in C. *** */' );
      { Generate the Standard CPAS C environment reference }
      OUT_EOLN;
      if cmac_hde <> nil then
      begin
        OUT_EOLN;
        OUT_PASTR( '/* Set specific Macro setting. */' );
        OUT_EOLN;

        psm := cmac_hde;
        while psm <> nil do
        with psm^ do
        begin
          OUT_PSTR( cmac_str );
          psm := cmac_nxt
        end
      end;
      for i := 0 to 10 do
        if cmp_pascal_cdef[i] <> nil then
        with cmp_pascal_cdef[i]^ do
        if length > 0 then
        begin
          if frst then
          begin
            OUT_EOLN;
            OUT_PASTR( '/* Define the PASCAL standard types. */' );
            OUT_EOLN;
            frst := false
          end;
          OUT_STR( 'typedef ' );
          for j := 1 to length do  OUT_CHAR( body[j] );
          case i of
            0: OUT_STR( '  C;' );
            1: OUT_STR( ' SB;' );
            2: OUT_STR( ' UB;' );
            3: OUT_STR( ' SW;' );
            4: OUT_STR( ' UW;' );
            5: OUT_STR( ' SL;' );
            6: OUT_STR( ' UL;' );
            7: OUT_STR( ' SQ;' );
            8: OUT_STR( ' UQ;' );
            9: OUT_STR( '  F;' );
           10: OUT_STR( '  G;' );
          end;
          OUT_EOLN
        end;
      OUT_EOLN;
      OUT_PASTR( '/* Include the PAS environment file. */' );
      OUT_EOLN;
      OUT_PASTR( '#include ' );
      with cenv_fil^ do
      begin
        c := body[1];
        if (c <> '"') and (c <> '<') then OUT_CHAR( '"' );
        for i := 1 to length do  OUT_CHAR( body[i] );
        if (c <> '"') and (c <> '<') then OUT_CHAR( '"' )
      end;
      OUT_EOLN;

(* ---///---
      if pas_main^.pro_pkind = pro_main then
        OUT_PASTR( 'P Lx[max_lex];' )
      else
        OUT_PASTR( 'extern P Lx[max_lex];' );
      OUT_EOLN;
   ---///--- *)

      if proc_intlex_count > 0 then
      begin { Create the Lex pointer table for the module when required }
        OUT_PASTR( 'static P Lx[' );
        OUT_INT1( proc_intlex_count ); bspace := false;
        OUT_PASTR( '];' );
        OUT_EOLN
      end;

      if cmp_genenv then
      with pro_last^ do
      begin { Environment mode }
        if module_gdat_size > 0 then
        begin
          OUT_PASTR( '#define Rd0' ); OUT_MODULE_IDENT( nil );
          bspace := false; OUT_STR( 'Data' ); OUT_EOLN
        end;
        if module_gsti_size > 0 then
        begin
          OUT_PASTR( '#define Ri0' ); OUT_MODULE_IDENT( nil );
          bspace := false; OUT_STR( 'Ista' ); OUT_EOLN
        end;
        if module_gsta_size > 0 then
        begin
          OUT_PASTR( '#define Rs0' ); OUT_MODULE_IDENT( nil );
          bspace := false; OUT_STR( 'Stat' ); OUT_EOLN
        end
      end;
      OUT_EOLN;
      objf_ok := true
    end
  end
end GENERATE_MODULE_HEADING;



procedure GENERATE_DATA_PART( var base: integer; vp: val_ptr );
{ To generate the data (const) program section }
var
  bsz:                 boolean;
  i, j, iad, jad, isz: integer;
  vp1:                 val_ptr;

begin
  bsz := true;
  with vp^ do
  begin
    { Set the start alignement }
    iad := IDE_TYP_ALIGN( base, val_typ );
    if base < iad then
    begin
      OUT_M_PAD( iad - base );
      base := iad
    end;
    with val_typ^ do
    case val_kind of
      form_char:
        begin
          j := typ_std[form_char]^.typ_size;
          OUT_M_INT( val_ival, j );
          if j < typ_size then OUT_M_PAD( typ_size - j )
        end;

      form_lit, form_wlit, form_ennum,
      form_int, form_range: OUT_M_INT( val_ival, typ_size );

      form_nil:    OUT_M_PAD( typ_size );

      form_single: OUT_M_SINGLE( val_rval );

      form_double: OUT_M_DOUBLE( val_rval );

      form_wset,
      form_wwset,
      form_set:    OUT_M_INT( val_set.siv, typ_size );

      form_lset,
      form_wlset:
        begin
          j := typ_std[form_wset]^.typ_size;
          if val_sar <> nil then
            for i := 0 to val_size - 1 do
              OUT_M_INT( val_sar^[i].siv, j );
          OUT_M_PAD( val_typ^.typ_size - val_size*j );

        end;

      form_string: { Out as string:  val_string }
        begin
          j := val_size;
          if val_str <> nil then
          begin
            with val_str^ do if j > length then j := length
          end
          else j := 0;
          if typ_form = form_record then
          begin { Fill the string descriptor }
            { set the allocated size and related pad byte(s) }
            OUT_M_INT( val_size, typ_parmlst^.ide_typ^.typ_size );
            OUT_M_PAD( stri_descrsz - typ_parmlst^.ide_typ^.typ_size );
            OUT_M_INT( j, stri_lengthsz )
          end;
          { *** The string character are assumed to be byte aligned *** }
          if val_str <> nil then
            with val_str^ do
              for i := 1 to j do  OUT_M_BYTE( ORD( body[i] ) );
          if j < val_size then OUT_M_PAD( val_size - j );
          base := iad + val_size;
          if typ_form = form_record then base := base + stri_stroffset;
          bsz := false { the allocation is done }
        end;

      form_conf,
      form_record,
      form_array:  { out as an array or a record: val_lst^ list }
        begin
          vp1 := val_lst;
          while vp1 <> nil do
          begin
            iad := base;
            GENERATE_DATA_PART( base, vp1 );
            if val_kind <> form_record then
            begin
              { For each array element }
              iad := iad + vp1^.val_size;
              if base < iad then
              begin
                OUT_M_PAD( iad - base );
                base := iad
              end
            end;
            vp1 := vp1^.val_next
          end;
          bsz := false
        end;

      form_null:
        begin
          base := base + val_size;
          OUT_M_PAD( val_size );
          bsz := false
        end;

    otherwise
    end;
    { When it is not already done set the new data space size }
    if bsz then base := iad + val_typ^.typ_size
  end
end GENERATE_DATA_PART;



procedure GENERATE_DATA;
{ To generate the data (const) program section }
var
 topdat: integer;
 p:      val_ptr;

begin
  topdat := 0;
  p := vagd_first;
  if p <> nil then
  begin
    OUT_PASTR( '/* Define the environment data section. */' );
    OUT_EOLN;
    OUT_PASTR( 'static Blk( Rd0,' );
    OUT_INT( module_gdat_size );
    OUT_PASTR( ') = {{' );
    OUT_EOLN;
    bcomma  := false;
    icolnbr := 0;
    imaxcol := 16;
    repeat
      GENERATE_DATA_PART( topdat, p );
      p := p^.val_next
    until p = nil;
    OUT_M_PAD( module_gdat_size - topdat );
    OUT_EOLN;
    OUT_PASTR( '}}' ); OUT_SEMICOLON;
    OUT_EOLN
  end;
  p := vald_first;
  if p <> nil then
  begin
    OUT_PASTR( '/* Define the local data section. */' );
    OUT_EOLN;
    OUT_PASTR( 'static Blk( Rd,' );
    OUT_INT( module_dat_size );
    OUT_PASTR( ') = {{' );
    OUT_EOLN;
    bcomma  := false;
    icolnbr := 0;
    imaxcol := 16;
    repeat
      GENERATE_DATA_PART( topdat, p );
      p := p^.val_next
    until p = nil;
    OUT_M_PAD( module_dat_size - topdat );
    OUT_EOLN;
    OUT_PASTR( '}}' ); OUT_SEMICOLON;
    OUT_EOLN
  end
end GENERATE_DATA;



procedure GENERATE_STATIC_INIT;
{ To generate the data (const) program section }
var
 topsti: integer;
 p:      all_ptr;

begin
  topsti := 0;
  p := algi_first;
  { For global initialized static }
  if p <> nil then
  begin
    OUT_PASTR( '/* Define the environment static Initialized section. */' );
    OUT_EOLN;
    OUT_PASTR( 'Blk( Ri0,' );
    OUT_INT( module_gsti_size );
    OUT_PASTR( ') = {{' );
    OUT_EOLN;
    bcomma  := false;
    icolnbr := 0;
    imaxcol := 16;
    repeat
      GENERATE_DATA_PART( topsti, p^.all_cte );
      p := p^.all_nxt
    until p = nil;
    OUT_M_PAD( module_gsti_size - topsti );
    OUT_EOLN;
    OUT_PASTR( '}}' ); OUT_SEMICOLON;
    OUT_EOLN
  end;
  { For initialized static }
  p := alsi_first;
  if p <> nil then
  begin
    OUT_PASTR( '/* Define the local static Initialized section. */' );
    OUT_EOLN;
    OUT_PASTR( 'static Blk( Ri,' );
    OUT_INT( module_sti_size );
    OUT_PASTR( ') = {{' );
    OUT_EOLN;
    bcomma  := false;
    icolnbr := 0;
    imaxcol := 16;
    repeat
      GENERATE_DATA_PART( topsti, p^.all_cte );
      p := p^.all_nxt
    until p = nil;
    OUT_M_PAD( module_sti_size - topsti );
    OUT_EOLN;
    OUT_PASTR( '}}' ); OUT_SEMICOLON;
    OUT_EOLN
  end
end GENERATE_STATIC_INIT;



procedure GENERATE_STATIC;
{ To generate the data (const) program section }
var
 iad: integer;

begin
  if module_gsta_size > 0 then
  begin
    OUT_PASTR( '/* Define the environment static section. */' ); OUT_EOLN;
    OUT_PASTR( 'Blk( Rs0,' );
    OUT_INT( module_gsta_size ); OUT_PASTR( ');' ); OUT_EOLN;
    OUT_EOLN
  end;
  if module_sta_size > 0 then
  begin
    OUT_PASTR( '/* Define the local static section. */' ); OUT_EOLN;
    OUT_PASTR( 'static Blk( Rs,' );
    OUT_INT( module_sta_size ); OUT_PASTR( ');' ); OUT_EOLN;
    OUT_EOLN
  end
end GENERATE_STATIC;



procedure GENERATE_GLOBAL_CTYP;
{ Generate the global C-type definitions }
var
  p: ctyp_ptr;

begin
  p := ctyp_first;
  if p <> nil then
  begin
    OUT_PASTR( '/* Global C type definitions */' ); OUT_EOLN;
    while p <> nil do
    with p^ do
    begin
      OUT_PASTR( 'typedef ' ); OUT_PASTR( ctyp_ide^ ); OUT_SEMICOLON;
      p := ctyp_nxt
    end;
    OUT_EOLN
  end
end GENERATE_GLOBAL_CTYP;



procedure GENERATE_AUTOMATIC( pr: pro_ptr );
{ To generate the data (const) program section }
var
 p:   all_ptr;
 irg: integer;

begin
  with pr^ do
  begin
    if pro_stk_size > 0 then
    begin { * For dynamic stack }
      pro_stk_size := IDE_TYP_ALIGN( pro_stk_size, typ_std[form_double] );
      OUT_PASTR( '/* Define the dynamic stack space. */' ); OUT_EOLN;
      OUT_PASTR( 'Blk( Rk,' );
      OUT_INT( pro_stk_size ); OUT_PASTR( ');' ); OUT_EOLN;
      OUT_EOLN
    end;

    if pro_dyn_size > 0 then
    begin { * For dynamic structured type }
      OUT_PASTR( '/* Define the automatic variable space. */' ); OUT_EOLN;
      OUT_PASTR( 'Blk( Ra,' );
      OUT_INT( pro_dyn_size ); OUT_PASTR( ');' ); OUT_EOLN;
      OUT_EOLN
    end;
    p := pro_reglist;
    irg := 0;
    if p <> nil then
    begin
      OUT_PASTR( '/* Local variable stored in C variable */' ); OUT_EOLN;
      repeat
        with p^ do
        begin
          if all_kind = var_tmp then OUT_PASTR( 'register ' );
          if all_ide <> nil then OUT_C_TYPE( all_ide^.ide_typ )
                            else OUT_CREG_TYPE( p );
          OUT_STR( 'Rv' );
          OUT_INT( irg ); OUT_CHAR( ';' ); OUT_EOLN;
          all_disp := irg; irg := irg + 1;
          all_acc  := all_acc + [var_register];
          p := all_nxt
        end;
      until p = nil
    end
  end
end GENERATE_AUTOMATIC;



procedure GEN_GLOBAL_SYMBOL_DECL( id: ide_ptr; bext, bval: boolean );
{ Creates C extern or global declaration(s) }
var
  isz, base: integer;

begin
  if id <> nil then
  with id^ do
  if ide_typ <> nil then
  with ide_typ^ do
  begin
    if bext then
    begin
      bval := false;
      OUT_PASTR( 'extern' )
    end;
    if typ_simple then
    begin
      OUT_C_TYPE( ide_typ ); OUT_IDENT( ide_extnam );
      if bval then { We generate an initial value }
      begin
        OUT_PASCH( '=' );
        with ide_inival^ do
        case typ_form of
          form_char: if val_ival <= 127 then OUT_INT( val_ival )
                                        else OUT_INT( 256 - val_ival );
          form_lit,
          form_int: OUT_INT( val_ival );
          form_set,
          form_wset,
          form_wwset:  OUT_INT( val_set.siv );
          form_single: OUT_SINGLEV( val_rval );
          form_double: OUT_DOUBLEV( val_rval );
          form_file,
          form_wfile,
          form_pointer,
          form_nil:    OUT_PASTR( 'NULL' );
        otherwise
        end
      end
    end
    else
    begin
      isz := IDE_TYP_ALIGN( ABS( typ_size ), typ_std[form_double] );
      OUT_STR( 'Blk(' ); OUT_IDENT( ide_extnam ); OUT_CHAR( ',' );
      OUT_INT( isz ); OUT_CHR( ')' );
      if bval and not bext then
      begin
        base := 0;
        OUT_STR( ' = {{' );
        bcomma  := false;
        icolnbr := buffer.length;
        imaxcol := 8;
        GENERATE_DATA_PART( base, ide_inival );
        OUT_M_PAD( isz - base ); OUT_PASTR( '}}' )
      end
    end;
    OUT_SEMICOLON
  end
end GEN_GLOBAL_SYMBOL_DECL;



procedure GENERATE_EXTERN_REF;
{ Generate the External variable reference }
var
  p: all_ptr;

begin
  p := all_fextern;
  if p <> nil then
  begin
    OUT_PASTR( '/* Declaration of external variables. */' ); OUT_EOLN;
    repeat
      if p^.all_kind <> var_standard then
        GEN_GLOBAL_SYMBOL_DECL( p^.all_ide, true, false );
      p := p^.all_nxt
    until p = nil;
    OUT_EOLN
  end
end GENERATE_EXTERN_REF;



procedure GENERATE_GLOBAL_DEF;
{ Generate the Global variable definition }
var
  p: all_ptr;

begin
  p := all_fglobal;
  if p <> nil then
  begin
    OUT_PASTR( '/* Declaration of global variables. */' ); OUT_EOLN;
    repeat
      GEN_GLOBAL_SYMBOL_DECL( p^.all_ide, false, p^.all_cte <> nil );
      p := p^.all_nxt
    until p = nil
  end;
  OUT_EOLN
end GENERATE_GLOBAL_DEF;



procedure GENERATE_FORMAL_DEF( fl: ide_ptr );
{ Generate a C procedure/function formal argument list }
var
  id: integer;
  bf: boolean;

begin
  bf := true;
  id := -1;
  OUT_SEPAR( '(' );
  while fl <> nil do
  with fl^ do
  begin
    if bf then bf := false
          else begin  bspace := false; OUT_STR( ', ' )  end;
    case ide_class of
      cla_fentry:
        if ide_entry <> nil then
        with ide_entry^ do
        begin { Formal procedure/function }
          if pro_typ <> nil then
            if pro_typ^.typ_simple then OUT_C_TYPE( pro_typ )
                                   else OUT_CHR( 'V' )
          else OUT_PASTR( 'void' );
          OUT_F_IDENT( ide_name );
          GENERATE_FORMAL_DEF( pro_parmlst )
        end;

      cla_varbl:
        begin
          case ide_vkind of
            var_result:  OUT_PASTR( 'V Ret' );
            var_formal:
              begin
                if ide_typ^.typ_simple then begin
                                              OUT_C_TYPE( ide_typ );
                                              OUT_CHR( '*' )
                                            end
                                       else OUT_PASCH( 'V' );
                OUT_F_IDENT( ide_name )
              end;
            var_refer,
            var_vformal:
              begin
                if ide_typ^.typ_simple then OUT_C_TYPE( ide_typ )
                                       else OUT_PASCH( 'V' );
                OUT_F_IDENT( ide_name )
              end;
          otherwise
          end;

          { Insert the attached attribute parameter }
          if var_card  in ide_vacc then
          begin { Formal with the size attribute }
            id := id + 1; bspace := false;
            OUT_STR( ', SL FSz_' ); OUT_INT ( id )
          end;
          if var_size  in ide_vacc then
          begin { Formal with the size attribute }
            id := id + 1; bspace := false;
            OUT_STR( ', SL FSz_' ); OUT_INT ( id )
          end;
          if var_image in ide_vacc then
          begin { Formal with the size attribute }
            id := id + 1; bspace := false;
            OUT_STR( ', V FIm_' ); OUT_INT ( id )
          end;
          if var_named in ide_vacc then
          begin { Formal with the name attribute }
            id := id + 1; bspace := false;
            OUT_STR( ', V FId_' ); OUT_INT ( id )
          end
        end;

    otherwise
    end;
    fl := ide_nxt
  end;
  OUT_SEPAR( ')' )
end GENERATE_FORMAL_DEF;



procedure GENERATE_ENTRY( pr: pro_ptr );
{ Generate a C procedure/function prototype }
begin
  with pr^ do
  begin
(*  if pro_pkind in [pro_decl] then OUT_PASTR( 'static');  *)
    if pro_typ <> nil then                     { Function }
      if pro_typ^.typ_simple then OUT_C_TYPE( pro_typ )
                             else OUT_PASCH( 'V' )
    else
      OUT_PASTR( 'void' );                     { Procedure }

    OUT_PROC_IDENT( pr, false );
    GENERATE_FORMAL_DEF( pro_parmlst );
  end
end GENERATE_ENTRY;



procedure GENERATE_PROTOTYPE;
{ To generate all used prototype for the module }
var
  p: pro_ptr;
  f: boolean;

begin
  f := true;
  p := pro_d_first;
  while p <> nil do
  begin
    with p^ do
    case pro_pkind of
      pro_decl, pro_external, pro_forward, pro_global:
        if prf_callfw in pro_flags then
        begin
          if f then
          begin
            OUT_PASTR( '/* Function/procedure prototypes. */' ); OUT_EOLN;
            f := false
          end;
          GENERATE_ENTRY( p );
          OUT_SEMICOLON
        end;

    otherwise
    end;
    p := p^.pro_link
  end
end GENERATE_PROTOTYPE;



procedure GENERATE_GLOBAL_OBJ;
{ Pass 2 of code Generation: generation of global/static/data structure. }
var
  i: integer;

begin
  { Set the alignements }
  module_dat_size  := IDE_TYP_ALIGN(  module_dat_size, typ_std[form_double] );
  module_sti_size  := IDE_TYP_ALIGN(  module_sti_size, typ_std[form_double] );
  module_sta_size  := IDE_TYP_ALIGN(  module_sta_size, typ_std[form_double] );
  module_gdat_size := IDE_TYP_ALIGN( module_gdat_size, typ_std[form_double] );
  module_gsti_size := IDE_TYP_ALIGN( module_gsti_size, typ_std[form_double] );
  module_gsta_size := IDE_TYP_ALIGN( module_gsta_size, typ_std[form_double] );
  { Generate all common data }
  GENERATE_DATA;               { Generate the DATA    section }
  GENERATE_STATIC_INIT;        { Generate the STATICI section }
  GENERATE_STATIC;             { Generate the STATIC  section }
  { Generate a global label environment when required }
  if global_labenv then
  begin
    OUT_STR( 'static sigjmp_buf G_lb_env;' ); OUT_EOLN; OUT_EOLN
  end;
  GENERATE_GLOBAL_CTYP;        { Generate the Global C type definitions }
  GENERATE_EXTERN_REF;         { Generate the External variable reference }
  GENERATE_GLOBAL_DEF;         { Generate the Global variable definition }
  GENERATE_PROTOTYPE           { Generate the procedure(s) prototype(s) }
end GENERATE_GLOBAL_OBJ;




procedure SIZE_REGISTER( nd: lgt_ptr );
var
  nd1: lgt_ptr;
  clb: ide_ptr;

begin
  nd1 := nd;
  while nd1 <> nil do
  with nd1^ do
  begin
    if lgt_lbl in lgt_status then
    if lgt_lide <> nil then
    begin  { *** An internaly referenced label cannot be a reference node *** }
      clb := lgt_lide;
      while clb <> nil do
      with clb^ do
      begin
        if lab_inref in ide_labflg then         { Global label }
        begin
          ide_lablnk    := global_labhde;
          global_labhde := clb
        end;
        clb := ide_labsyn
      end
    end;
      
    case lgt_kind of
      lgt_ctlflow:
        case lgt_stm of
          stm_for:
            begin
              for_var_cnt := for_var_cnt + 1;
              if for_var_max < for_var_cnt then for_var_max := for_var_cnt;
              SIZE_REGISTER( lgt_parmlst );    { Look for other stm_for node }
              for_var_cnt := for_var_cnt - 1
            end;

          stm_exit:
            if lgt_parmlst <> nil then SIZE_REGISTER( lgt_parmlst^.lgt_nxt );

          stm_return:
            begin
              return_label := true;
              if lgt_parmlst <> nil then SIZE_REGISTER( lgt_parmlst )
            end;

          stm_goto, { Local goto }
          stm_jump: { Not local goto }
            if lgt_lab <> nil then
            with lgt_lab^ do
              if ide_lablgt <> nil then
              with ide_lablgt^ do
                lgt_status := lgt_status + [lgt_lbl];

        otherwise
          SIZE_REGISTER( lgt_parmlst );
        end;

      lgt_codep:
        begin
          case lgt_pcode of
            pcod_isqr,
            pcod_setlt, pcod_setle, pcod_setge, pcod_setgt: setreg_i1 := true;
            pcod_imod: begin  setreg_i1 := true; setreg_i2 := true  end;
            pcod_cvfi, pcod_fsqr: setreg_f  := true;
            pcod_cvgi, pcod_gsqr: setreg_g  := true;
          otherwise
          end; 
          SIZE_REGISTER( lgt_parmlst )
        end;

      lgt_agregat: SIZE_REGISTER( lgt_parmlst );

      lgt_result, lgt_specific, lgt_const,
      lgt_varbl, lgt_proref, lgt_eproref, lgt_null: ;    { Nothing to do }

    otherwise
      SIZE_REGISTER( lgt_parmlst )
    end;
    nd1 := lgt_nxt
  end;

end SIZE_REGISTER;



function SIZE_FORMAL_REG( pf: ide_ptr ): boolean;
var
  bres: boolean;

begin
  while pf <> nil do
  begin
    with pf^ do
      if (ide_vkind = var_vformal) and (ide_typ <> nil) then
      with ide_typ^ do
        if (not typ_simple) and (typ_size < 0) then
        begin
          setreg_i1 := true;
          bres := true
        end;
    pf := pf^.ide_nxt
  end;
  SIZE_FORMAL_REG := bres
end SIZE_FORMAL_REG;



{**************************************************************************}
{**************************************************************************}
{***************  C   Code  Generation  P A S S - 2   *********************}
{***************                                      *********************}
{***************    Procedure   Code  Generation      *********************}
{**************************************************************************}
{**************************************************************************}



procedure GEN_TYP_CONV( ty:  typ_ptr; var incr:  integer;
                        bad: boolean; var bsptr: boolean );
{ Generate a variable refer type conversion }
begin
  bsptr := false;
  incr :=  1;
  OUT_SEPAR( '(' );
  if ty <> nil then
  with ty^ do
    if typ_simple then
    begin
      incr := typ_size;
      case typ_form of
        form_char:   OUT_CHR( 'C' );
        form_range,
        form_lit,
        form_int: case typ_size of
                    1:   if typ_unsigned then OUT_STR( 'UB' )
                                         else OUT_STR( 'SB' );
                    2:   if typ_unsigned then OUT_STR( 'UW' )
                                         else OUT_STR( 'SW' );
                    3,4: if typ_unsigned then OUT_STR( 'UL' )
                                         else OUT_STR( 'SL' );
                    5,6,7,8:
                         if typ_unsigned then OUT_STR( 'UQ' )
                                         else OUT_STR( 'SQ' );
                         
                  otherwise
                  end;
        form_set,
        form_wset,
        form_wwset:case typ_size of
                    1:   OUT_STR( 'UB' );
                    2:   OUT_STR( 'UW' );
                    3,4: OUT_STR( 'UL' );
                   otherwise
                   end;
        form_single: OUT_CHR( 'F' );
        form_double: OUT_CHR( 'G' );
        form_fentry,
        form_file,
        form_wfile,
        form_pointer,
        form_nil:    OUT_CHR( 'V' );
      otherwise
        bsptr := true
      end
    end
    else bsptr := true
  else bsptr := true;
  if bsptr then OUT_CHR( 'S' );
  if bad then OUT_STR( ' *)' ) else OUT_SEPAR( ')' )
end GEN_TYP_CONV;



procedure GEN_V_TYPREF( ty: typ_ptr; var incr: integer; var bsptr: boolean );
{ Generate a variable refer type specification }
begin
  bsptr := false;
  incr :=  1;
  if ty <> nil then
  with ty^ do
    if typ_simple then
    begin
      incr := typ_size;
      case typ_form of
        form_char:   OUT_CHR( 'c' );
        form_range,
        form_lit,
        form_int: case typ_size of
                    1:   if typ_unsigned then OUT_STR( 'ub' )
                                         else OUT_STR( 'sb' );
                    2:   if typ_unsigned then OUT_STR( 'uw' )
                                         else OUT_STR( 'sw' );
                    3,4: if typ_unsigned then OUT_STR( 'ul' )
                                         else OUT_STR( 'sl' );
                    5,6,7,8:
                         if typ_unsigned then OUT_STR( 'uq' )
                                         else OUT_STR( 'sq' );
                        
                  otherwise
                  end;
        form_set,
        form_wset,
        form_wwset:case typ_size of
                    1:   OUT_STR( 'ub' );
                    2:   OUT_STR( 'uw' );
                    3,4: OUT_STR( 'ul' );
                   otherwise
                   end;
        form_single: OUT_CHR( 'f' );
        form_double: OUT_CHR( 'g' );
        form_fentry,
        form_pointer,
        form_file,
        form_wfile,
        form_nil:    OUT_CHR( 'v' );
      otherwise
        bsptr := true
      end
    end
    else bsptr := true
  else bsptr := true;
  if bsptr then OUT_CHR( 's' )
end GEN_V_TYPREF;



function CHECK_FORMAL_COPY( fp: ide_ptr ): boolean;
var
  bcop: boolean;

begin
  bcop := false;
  with fp^ do
    case ide_class of
      cla_varbl:
        if var_intaccess in ide_vacc then bcop := true
        else
          if ide_vkind = var_vformal then
            if ide_typ <> nil then
              if not ide_typ^.typ_simple then bcop := true;
      cla_fentry:
        if ide_entry <> nil then
          if prf_intaccess in ide_entry^.pro_flags then bcop := true;

    otherwise
    end;
  CHECK_FORMAL_COPY := bcop
end CHECK_FORMAL_COPY;



procedure GEN_NEXT( var lgt: lgt_ptr; prior: integer; blvl: boolean );
begin
  if lgt <> nil then
  begin
    GEN_SCAN( lgt, prior, blvl );
    lgt := lgt^.lgt_nxt
  end
end GEN_NEXT;



function SET_VINDEX( lgs: lgt_ptr; incr: integer ): boolean;
{ Set the modified size by the incr (alignement divisor) }
var
  lgh, lgl, l0, l1: lgt_ptr;
  iv, jv: integer;
  br, bs: boolean;


  function SET_VILIST( lgt: lgt_ptr; incr: integer; bndv: boolean := false ): boolean;
  { Check for divisibility (by incr) of element size list }
  { Set the vilist Queue (Header:lgh,lgl) as the divisible by incr size part }
  { lgt is an array element size expression }
  var
    lg, l1, l2: lgt_ptr;
    iv, jv, kv: integer;
    br:         boolean;

  begin
    br := false;
    if (lgt <> nil) and (incr > 1) then
    with lgt^ do
    case lgt_kind of
      lgt_const:
        if lgt_cte <> nil then
          if (lgt_cte^.val_ival mod incr = 0) or bndv then
          begin
            if lgh = nil then lgh := lgt
                         else lgl^.lgt_nct := lgt;
            lgl := lgt;
            br := true
          end;

      lgt_codep:
        begin
          l1 := lgt_parmlst;
          if l1 <> nil then l2 := l1^.lgt_nxt
                       else l2 := nil;
          case lgt_pcode of
            pcod_iadd, pcod_isub:
              if l2 <> nil then
              begin
                br := SET_VILIST( l2, incr );
                if br then
                begin
                  br := SET_VILIST( l1, incr );
                  if br then
                  begin
                    if l2^.lgt_kind = lgt_const then l2^.lgt_nxt := lgt
                  end
                  else
                    lgh := l2^.lgt_nct
                end
              end;
            pcod_imul:
              begin
                br := SET_VILIST( l2, incr );
                if br then
                  if l2^.lgt_kind = lgt_const then l2^.lgt_nxt := lgt
              end;
            pcod_bic:
              with l2^ do
                if (lgt_kind = lgt_const) and (lgt_cte <> nil) then
                begin { Stop the scan here and change the tree }
                  iv := lgt_cte^.val_ival + 1;
                  jv := -1; kv := 2;
                  while kv < incr do begin  kv := kv*2; jv := jv - 1  end;
                  { incr cannot greater than iv that is always > 1 }
                  if iv < incr then
                  begin { Insert the new node >> }
                    LGT_NEW_COPY( lgt, lg );
                    lgt^.lgt_pcode  := pcod_lsh;
                    lgt^.lgt_parmlst := lg;
                    lg^.lgt_nxt := LGT_NEW_ECONST( int_typ, jv )
                  end
                  else
                  begin { Replace the bic by a shift }
                    lgt^.lgt_pcode  := pcod_lsh;
                    VAL_FREE( lgt_cte );
                    VAL_NEW( lgt_cte, int_typ );
                    lgt_cte^.val_ival := jv
                  end;
                  br := true
                end else br := false
          otherwise
          end
        end;
    otherwise
    end;
    SET_VILIST := br
  end SET_VILIST;


begin { SET_VINDEX }
  lgh := nil;
  br := SET_VILIST( lgs, incr );
  if lgh <> nil then
  repeat
    l0 := lgh;
    with l0^ do
    begin
      bs := false;
      iv := lgt_cte^.val_ival div incr;
      if lgt_nxt <> nil then
      begin
        case lgt_nxt^.lgt_pcode of
          pcod_imul: if iv = 1 then bs := true;
          pcod_iadd,
          pcod_isub: if iv = 0 then bs := true;
        otherwise
        end;
        if bs then
        begin
          l1 := lgt_nxt^.lgt_nxt;              { Save the next link of the operator }
          lgt_nxt^ := lgt_nxt^.lgt_parmlst^;   { Replace oper. by the reference }
          lgt_nxt^.lgt_nxt := l1               { Restore the next link }
        end
        else
        begin
          VAL_FREE( lgt_cte );
          VAL_NEW( lgt_cte, int_typ );
          lgt_cte^.val_ival := iv
        end;
        lgh := lgt_nct
      end;
      if bs then LGT_FREE( l0 )
    end
  until lgh = nil;
  SET_VINDEX := br
end SET_VINDEX;



function CHECK_DIRECT( lgt: lgt_ptr ): boolean;
var
  br: boolean;

begin
  br := false;
  if lgt <> nil then
  with lgt^ do
  if lgt_typ <> nil then
  if lgt_typ^.typ_simple then
  case lgt_kind of
    lgt_const: br := true;
    lgt_varbl:
      if lgt_alloc <> nil then
      with lgt_alloc^ do
      case all_kind of
        var_vformal:
          if not CHECK_FORMAL_COPY( all_ide ) then br := true;

        var_decl,
        var_tmp: if var_register in all_acc then br := true;

        var_global,
        var_standard,
        var_external: br := true;

      otherwise
      end;

    lgt_icall, lgt_call: ;
    lgt_iproref, lgt_eproref, lgt_proref: ;

  otherwise
  end;
  CHECK_DIRECT := br
end CHECK_DIRECT;



function GEN_MOTHER_REFER(       lgt:  lgt_ptr;        { Variable/constant node }
                                 ty:   typ_ptr;        { The type to use }
                                 bad,                  { true for address required }
                                 bex:  boolean;        { true if S address mode }
                                 ind:  integer;        { > 0 if in index }
                            var idisp,                 { Resulting displacement }
                                iincr: integer         { resulting increment }
                         ): integer;                   { sub index flag }
{ Generate the mother variable/constant reference }
var
  ty1:                                 typ_ptr;
  lgta, lgti, lgts:                    lgt_ptr;
  anam:                                nam_ptr;
  vlex, i, iidsp, ijncr, indl, indr:   integer;
  bspt:                                boolean;
  acck:                              acc_kinds;

begin { GEN_MOTHER_REFER }
  iincr := 1;
  indl  := 0;
  if bex then bad := true;
  if lgt <> nil then
  with lgt^ do
    if lgt_kind = lgt_null then lgt := lgt_parmlst;

  if lgt <> nil then
  with lgt^ do
  begin
    case lgt_kind of
      lgt_null,                                { * * * 1.9 L Patch * * * }
      lgt_offset:
        begin
          idisp := idisp + lgt_disp;
          indl  := GEN_MOTHER_REFER( lgt_parmlst, ty, bad, bex, ind, idisp, iincr );
          acck  := acc_other
        end;

      lgt_iproref:
        begin
        end;

      lgt_refer:
        if lgt_parmlst <> nil then
        begin
          OUT_STR( 'PAS__Ptr'); GEN_V_TYPREF( lgt_typ, iincr, bspt );
          OUT_STR( '(Rk.' );    GEN_V_TYPREF( lgt_typ, iincr, bspt );
          if lgt_disp <> 0 then
          begin
            OUT_CHAR( '+' ); OUT_INT( lgt_disp div iincr )
          end;
          OUT_SEPAR( ',' );
          GEN_SCAN( lgt_parmlst, 0, true ); OUT_SEPAR( ')' );
          acck := acc_other
        end;

      lgt_indir:
        if lgt_parmlst <> nil then
        begin
          iidsp  := 0;                         { Generate the pointer value access }
          { We put a cast pointer conversion to the required type }
          if not bex then
          begin
            OUT_SEPAR( '(' );
            GEN_TYP_CONV( ty, iincr, true, bspt )
          end;
          ty1 := lgt_parmlst^.lgt_typ;
          if ty1 <> nil then
          with ty1^ do
            if (typ_form = form_file) or
               (typ_form = form_wfile) then ty1 := typ_std[form_nil];
          indr   := GEN_MOTHER_REFER( lgt_parmlst, ty1,
                                      false, false, 0, iidsp, ijncr );
          if not bex then OUT_SEPAR( ')' );
          { The result is a required type pointer }
          idisp  := idisp + lgt_disp;
          if ind = 0 then OUT_DISP_SPC( idisp div iincr, bad );
          acck   := acc_other
        end;

      lgt_index:
        if lgt_typlnk <> nil then              { Get the array type }
        with lgt_typlnk^ do
        begin
          lgta := lgt_parmlst;
          { Get the array reference }
          if bad and (ind = 0) then OUT_SEPAR( '(' );
          indl := GEN_MOTHER_REFER( lgta, ty, bad, false,
                                    ind + 1, idisp, iincr ) + 1;
          if lgta <> nil then lgti := lgta^.lgt_nxt;   { Get Index Expression }
          if lgti <> nil then lgts := lgti^.lgt_nxt;   { Get Element Size Expression }

          idisp := idisp + lgt_disp;

          if indl = 1 then                     { For the first (or main) index }
          begin
            if bad then OUT_SEPAR( '+' )
                   else OUT_SEPAR( '[' );
            { If the element type is not the target with correct element size }
          end
          else OUT_SEPAR( '+' );               { For other index }

          GEN_SCAN( lgti, 12, true );          { Generate the index value }

          if lgts <> nil then                  { With the element Size }
          with lgts^ do
            if lgt_kind = lgt_const then
            begin { Element Size is a Constante }
              i := 1;
              if lgt_cte <> nil then
                i := lgt_cte^.val_ival div iincr;
              if i > 1 then
              begin                            { Multiply only is > 1 }
                OUT_SEPAR( '*' ); OUT_INT( i )
              end
            end
            else
            begin { Element Size is Variable/Dynamic }
              OUT_SEPAR( '*' );
              i := cop_table[pcod_imul].cop_prio;
              if (iincr <= 1) or (not ty^.typ_simple) then
                { For not simple element or iincr < 2 => Compute the El. Size }
                GEN_SCAN( lgts, i, true )
              else { For simple types of size > 1, Set the increment and Separate the cte part }
                if SET_VINDEX( lgts, iincr ) then
                  GEN_SCAN( lgts, i, true )
                else SRC_ERROR( mdnam, 998, e_fatal )
            end;

          if ind = 0 then
          begin { The more external index }
            if idisp > 0 then
            begin
              OUT_SEPAR( '+' ); OUT_INT( idisp div iincr )
            end
            else
            if idisp < 0 then
            begin
              OUT_SEPAR( '-' ); OUT_INT( (- idisp) div iincr )
            end;
            if bad then OUT_SEPAR( ')' ) else OUT_SEPAR( ']' )
          end;
          acck := acc_other
        end;

      lgt_icall,
      lgt_call:
        if lgt_typ <> nil then
        with lgt_typ^ do
        begin
          if bad and typ_simple then
          { Call always returns an address for the complex object type }
          begin
            if bex then begin  OUT_STR( '(V)' ); iincr := 1  end;
            { //// Not implemented now //// }
          end;
          GEN_CALL( lgt, lgt_kind = lgt_icall );
          acck := acc_other
        end;

      lgt_varbl:
        if lgt_alloc <> nil then
        with lgt_alloc^ do
        begin
          vlex := all_lex;
          case all_kind of
            var_result:  { *** For function result by reference *** }
              if all_ide <> nil then
              with all_ide^ do
              begin { Use the C formal access }
                idisp := idisp + lgt_disp;
                if ide_typ <> nil then
                with ide_typ^ do
                  if typ_simple then
                  begin
                    if bex then OUT_STR( '((V) ')
                           else if not bad then OUT_STR( '(*' );
                    OUT_STR( 'Ret' );
                    if bex or (not bad) then OUT_CHAR( ')' )
                  end
                  else { Formal of Complex Type }
                    if bex then
                    begin
                      if ty^.typ_simple then OUT_STR( '(V)' );
                      OUT_STR( 'Ret' );
                      if ind = 0 then OUT_DISP_SPC( idisp, true )
                    end
                    else
                    begin
                      { Output the cast operator if required }
                      OUT_SEPAR( '(' );
                      { Output offset when required }
                      GEN_TYP_CONV( ty, iincr, true, bspt );
                      OUT_STR( 'Ret)' );
                      { Output offset when required }
                      if ind = 0 then OUT_DISP_SPC( idisp div iincr, bad )
                    end;
                acck := acc_other
              end;

            var_formal:  { *** For Formal by reference *** }
              if all_ide <> nil then
              with all_ide^ do
              begin
                if var_intaccess in ide_vacc then
                { For formal by reference with internal access }
                begin { Use the local or lex tab access }
                  if vlex = curr_dyn_lex then
                    OUT_STR( 'Ra.p' )
                  else
                  begin
                    OUT_LEX_SPC( all_lexid ); OUT_STR( '.p' )
                  end;
                  { Set addresse value }
                  OUT_DISP_SPC( all_disp div fptr_size, false );
                  OUT_CHR( '.' );
                  idisp := idisp + lgt_disp;
                  acck  := acc_pascal          { it is a pascal access }
                end
                else
                begin { Use the C formal access }
                  { Get the formal C identifier }
                  idisp := idisp + lgt_disp;
                  if ide_typ <> nil then
                  with ide_typ^ do
                    if typ_simple then
                    begin { simple type of formal or S pointer required }
                      if bex then OUT_STR( '((V) ')
                             else if not bad then OUT_STR( '(*' );
                      OUT_F_IDENT( ide_name ); bspace := false;
                      if bex or (not bad) then OUT_CHAR( ')' )
                    end
                    else { Formal of complex Type }
                      if bex then
                      begin
                        if ty^.typ_simple then OUT_STR( '(V)' );
                        OUT_F_IDENT( ide_name );
                        if ind = 0 then OUT_DISP_SPC( idisp, true );
                      end
                      else
                      begin  
                        { Output the cast operator if required }
                        OUT_SEPAR( '(' );
                        GEN_TYP_CONV( ty, iincr, true, bspt );
                        OUT_F_IDENT( ide_name );
                        { Output offset when required }
                        OUT_SEPAR( ')' );
                        if ind = 0 then OUT_DISP_SPC( idisp div iincr, bad );
                      end;
                  acck := acc_other
                end
              end;

            var_vformal: { *** For Formal by value *** }
              if all_ide <> nil then
              with all_ide^ do
              begin
                if CHECK_FORMAL_COPY( all_ide ) then
                { For formal by value with internal access or of complex type }
                begin { Use the local or lex tab access }
                  if ide_typ^.typ_size < 0 then
                  begin { Dynamic mode => indirect access }
                    if vlex = curr_dyn_lex then
                      OUT_STR( 'Ra.p' )
                    else
                    begin
                      OUT_LEX_SPC( all_lexid ); OUT_STR( '.p' )
                    end;
                    { Set addresse value }
                    OUT_DISP_SPC( all_disp div fptr_size, false );
                    OUT_CHR( '.' );
                    idisp := idisp + lgt_disp
                  end
                  else
                  begin
                    if vlex = curr_dyn_lex then
                      OUT_STR( 'Ra.' )
                    else
                    begin
                      OUT_LEX_SPC( all_lexid ); OUT_CHR( '.' )
                    end;
                    bspace := false;
                    { Get the local space offset }
                    idisp := idisp + all_disp + lgt_disp
                  end;
                  acck  := acc_pascal          { it is a pascal access }
                end
                else
                begin { For object without internal access of a simple type
                        we use the normal C parameter mode }
                  anam := ide_name; 
                  acck := acc_formal           { it is a C formal access }
                end
              end;

            var_tmp, var_decl: { *** Temporary or local automatic variable *** }
              if var_register in all_acc then
              begin { *** Access by C variable *** }
                if bad then
                begin
                  if bex then OUT_STR( '(V)' );
                  OUT_CHR( '&' )
                end;
                OUT_STR( 'Rv' ); OUT_INT( all_disp );
                acck := acc_other
              end
              else
              begin
                if vlex <= 1 then              { for global lex }
                begin
                  if all_ide <> nil then       { if there is a identifier }
                  with all_ide^ do                  begin  
                    if ide_inival <> nil then  { if a static value is given }
                      OUT_STR( 'Ri' )          { Use the istatic access 'Ri' }
                    else OUT_STR( 'Rs' );       { Else use the static ... }
                    if all_psect >= 0 then OUT_INT1( all_psect );
                    OUT_CHR( '.' )
                  end
                  else OUT_STR( 'Rs.' )        { ... access 'Rs' }
                end
                else                           { ELSE for iother lex ... }
                  if vlex = curr_dyn_lex then  {  ... with the local lex ... }
                    OUT_STR( 'Ra.' )           { Use the local access 'Ra' }
                  else                         { Use the lex tab access }
                  begin
                    OUT_LEX_SPC( all_lexid ); OUT_CHR( '.' )
                  end;
                { Get the area displacement }
                idisp := idisp + lgt_disp + all_disp;
                bspace := false;
                acck  := acc_pascal            { it is a pascal access }
              end;

            var_imported,
            var_static:
              begin { *** For static variable *** }
                if all_ide <> nil then
                with all_ide^ do
                  if ide_inival <> nil then    { if a static value is given }
                    OUT_STR( 'Ri' )            { Use the istatic access 'Ri' )
                  else OUT_STR( 'Rs' )         { Else use the static ... }
                else OUT_STR( 'Rs' );          { ... access 'Rs' }
                if all_psect >= 0 then OUT_INT1( all_psect );
                OUT_CHR( '.' );
                { Get the area displacement }
                idisp := idisp + lgt_disp + all_disp;
                acck  := acc_pascal            { it is a pascal access }
              end;

            var_global,
            var_standard,
            var_external: { *** For Global and External variable *** }
              if all_ide <> nil then
              with all_ide^ do
              begin
                anam := ide_extnam;            { Get the global symbol name }
                if ide_typ <> nil then
                  if ide_typ^.typ_simple then  { For simple Global type }
                    acck  := acc_varbl         { Use C global variable }
                  else
                    acck  := acc_pascal        { Else use the global ... }
                else acck  := acc_pascal;      { ... Pascal access }
                if acck = acc_pascal then
                begin
                  OUT_IDENT( ide_extnam );     { Output the external name }
                  bspace := false;
                  OUT_CHR( '.' );
                  idisp := idisp + lgt_disp    { Get the related offset }
                end
              end;

          otherwise
          end;
        end;

      lgt_const:
        if lgt_cte <> nil then
        with lgt_cte^ do
          if val_all = nil then                { For immediate constante }
          begin
            if bex then begin  OUT_STR( '(V)' ); iincr := 1  end
            else
              if ty <> nil then
                if ty^.typ_form = form_single then OUT_STR( '(F)' );
            OUT_IMMED_CTE( lgt_cte, bad );
            acck  := acc_other                 { Set no action mode }
          end
          else
          begin
            OUT_STR( 'Rd' );                   { Set data access mode 'Rd' }
            if val_psect >= 0 then OUT_INT1( val_psect );
            OUT_CHR( '.' );
            { Get the data offset }
            idisp := idisp + val_all^.all_disp + lgt_disp;
            acck  := acc_pascal                { Use the pascal access }
          end;

    otherwise
    end;

    case acck of
      acc_formal,
      acc_varbl:  begin { *** C access *** }
                    if bad then
                    begin
                      if bex then begin  OUT_STR( '(V)' ); iincr := 1  end;
                      OUT_CHR( '&' )
                    end;
                    if acck = acc_varbl then
                      OUT_IDENT( anam )        { Output the global name }
                    else
                      OUT_F_IDENT( anam );     { Output the C name }
                  end;

      acc_pascal: begin { *** Pascal access *** }
                    if bex then OUT_CHR( 's' )
                           else GEN_V_TYPREF( ty, iincr, bspt );
                    if ind = 0 then OUT_DISP_SPC( idisp div iincr, bad )
                  end;

    otherwise
    end
  end;
  GEN_MOTHER_REFER := indl
end GEN_MOTHER_REFER;



procedure GEN_SET_EFFECTIVE( nd: lgt_ptr; id: ide_ptr );
{ Set the effective argument in agreement with the
  procedure/function formal argument list,
  nd -> effective expression, id -> formal argument identifier.
}
var
  card, idsp, indll, iincr: integer;

begin
  idsp := 0;
  with nd^, id^ do
  begin
    case ide_class of
      cla_fentry:                              { The formal argument is a formal entry }
        if lgt_pro <> nil then
        with lgt_pro^ do
          if pro_pkind = pro_formal then       { The effective argument is also a formal entry }
            if (prf_intaccess in pro_flags) and (pro_lex <> curr_dyn_lex) then
            begin                              { With an internal access copy to use }
            end
            else OUT_F_IDENT( ide_name )       { }
          else OUT_PROC_IDENT( lgt_pro, false );       { The effective argument is a generic entry }

      cla_varbl:
        begin
          case ide_vkind of
            var_result,
            var_formal:
              if lgt_kind = lgt_empty then OUT_PASTR( 'NULL' )
              else
                if ide_typ <> nil then
                begin
                  case ide_typ^.typ_form of
                    form_lset, form_wlset:
                      if lgt_typ^.typ_simple then OUT_STR( '(V)' );
                  otherwise
                  end;
                  indll := GEN_MOTHER_REFER( nd, lgt_typ, true,
                                             (not ide_typ^.typ_simple),
                                             0, idsp, iincr )
                end;

            var_vformal:
              if ide_typ <> nil then
                if ide_typ^.typ_simple then
                  GEN_SCAN( nd, 0, true )
                else
                  indll := GEN_MOTHER_REFER( nd, lgt_typ, true, true, 0, idsp, iincr )
              else
                GEN_SCAN( nd, 0, true );
          otherwise
          end;
          if lgt_typ <> nil then
          with lgt_typ^ do
          begin
            if var_card in ide_vacc then
              case typ_form of
                form_set, form_lset, form_wset, form_wlset, form_wwset:
                  begin { Append a cardinality parameter }
                    OUT_CHAR( ',' );
                    card := typ_cardinality;
                    if card < 0 then card := 0;
                    OUT_INT( card )
                  end;
              otherwise
                OUT_PASTR( ', 0' )
              end;

            if var_size in ide_vacc then
            begin
              OUT_CHAR( ',' );
              if typ_size >= 0 then OUT_INT( typ_size )
                               else OUT_INT( -1 )
            end;

            if var_image in ide_vacc then
              case typ_form of
                form_lit:
                  if typ_idetab <> nil then
                  begin { table has always a complex (not simple) type }
                    OUT_CHAR( ',' ); idsp := 0;
                    indll := GEN_MOTHER_REFER( typ_idetab,
                                               typ_idetab^.lgt_typ, true,
                                               true, 0, idsp, iincr )
                  end
                  else OUT_PASTR( ', NULL' );

                form_set, form_lset:
                  if typ_seltype <> nil then
                  with typ_seltype^ do
                    if typ_form in [form_int,form_char,form_lit] then
                      if typ_idetab <> nil then
                      begin { table has always a complex (not simple) type }
                        OUT_CHAR( ',' ); idsp := 0;
                        indll := GEN_MOTHER_REFER( typ_idetab,
                                                   typ_idetab^.lgt_typ, true,
                                                   true, 0, idsp, iincr )
                      end
                      else OUT_PASTR( ', NULL' )
                    else OUT_PASTR( ', NULL' )
                  else OUT_PASTR( ', NULL' );

              otherwise
                OUT_PASTR( ', NULL' )
              end
          end
        end;

    otherwise
    end
  end
end GEN_SET_EFFECTIVE;


(*
procedure GEN_ENTRY_PROTOTYPE( pr: pro_ptr; bf: boolean := false );
var
  id: ide_ptr;
  b1: boolean;

begin
  with pr^ do
  begin
    id := pro_parmlst;
    b1 :=       false;
    OUT_CHAR( '(' );
    while id <> nil do
    begin { We must give effective parameter references }
      if not b1 then OUT_CHAR( ',' );
      (*GEN_SET_EFFECTIVE( lgt, id );*)(*
      b1 := false;
      id  := id^.ide_nxt
    end;
    OUT_CHAR( ')' )
  end
end GEN_ENTRY_PROTOTYPE;
*)


procedure GEN_ENTRY_REF( lgt: lgt_ptr );
begin
  if lgt <> nil then
  with lgt^, lgt_pro^  do
  begin
    if pro_pkind = pro_formal then             { For formal entry }
      with pro_f_all^, all_ide^ do
      begin
        if var_intaccess in all_acc then       { Formal entry with internal access }
        begin                                  { Use the local or lex tab access }
          OUT_STR( '((' );
          OUT_IDENT( pro_stdname ); bspace := false;
          OUT_STR( ')(' );
          if ide_lex = curr_dyn_lex then       { Select local mode for local use }
            OUT_STR( 'Ra.v' )
          else
          begin
            OUT_LEX_SPC( all_lexid ); OUT_STR( '.v' )
          end;
          { Set addresse value }
          OUT_DISP_SPC( all_disp div fptr_size, false );
          OUT_CHAR( ')' )
        end
        else
        begin
          OUT_STR( '(*' );
          OUT_F_IDENT( ide_name )              { Use directly the formal identifier name }
        end;
        bspace := false; OUT_CHR( ')' )
      end
    else OUT_PROC_IDENT( lgt_pro, false )       { For generic use directly the standard generic name }
  end
end GEN_ENTRY_REF;



procedure GEN_CALL { ( head: lgt_ptr; bindir: boolean ) was forward };
const
  mdnam = 'GECA';

var
  id:    ide_ptr;
  lgt:   lgt_ptr;
  first: boolean;

begin
  with head^ do
  if lgt_pro <> nil then
  with lgt_pro^ do
    if pro_pkind = pro_inline then
      SRC_ERROR( mdnam, 998, e_fatal )
    else
    begin                                      { pro_decl or other true call }
      lgt := lgt_parmlst;
      if bindir then
      begin
        OUT_STR( '(*(' );
        OUT_IDENT( pro_stdname ); bspace := false;
        OUT_CHR( ')' );
        GEN_SCAN( lgt, 13, true );             { Evaluate the entry address expression (13 = C prior of cast ope. }
        bspace := false; OUT_CHR( ')' );
        lgt := lgt^.lgt_nxt
      end
      else GEN_ENTRY_REF( head );
      id  := pro_parmlst;
      first := true;
      { if only one effective parameter, do not put colon }
      bspace := false;
      OUT_CHAR( '(' );
      while (lgt <> nil) and (id <> nil) do
      begin
        { We must give effective parameter references }
        if not first then OUT_CHAR( ',' );
        GEN_SET_EFFECTIVE( lgt, id );
        first := false;
        id  := id^.ide_nxt;
        lgt := lgt^.lgt_nxt
      end;
      OUT_CHAR( ')' )
    end
end GEN_CALL;



procedure GEN_ADDRESS( nd: lgt_ptr; bex: boolean );
{ bex = true for the lgt_address node only }
var
  idisp, indll, iincr: integer;
  bsmd: boolean;

begin
  idisp := 0;
  if nd <> nil then
  with nd^ do
  if lgt_typ <> nil then
  with lgt_typ^ do
  begin
    if typ_simple then
    begin
      bsmd := false;
      if bex and (lgt_kind = lgt_varbl) then
        if lgt_alloc <> nil then
        case lgt_alloc^.all_kind of
          var_result, var_formal: bsmd := true;
        otherwise
        end
    end
    else bsmd := true;
    indll := GEN_MOTHER_REFER( nd, lgt_typ, true, bsmd, 0, idisp, iincr )
  end
end GEN_ADDRESS;



procedure GEN_VALUE( nd: lgt_ptr );
var
  idisp, indll, iincr: integer;

eq: record case boolean of false:( s: lgt_states); true:( i:integer) end;

begin
  idisp := 0;
  if nd <> nil then
  with nd^ do
  if lgt_typ <> nil then
    indll := GEN_MOTHER_REFER( nd, lgt_typ, false, (not lgt_typ^.typ_simple),
                               0, idisp, iincr )
end GEN_VALUE;



procedure GEN_STATEMENT( head: lgt_ptr );
begin
  if head <> nil then
    if head^.lgt_kind <> lgt_null then
    begin
      bsemicolon := true;
      GEN_SCAN( head, 0, false );
      if bsemicolon then OUT_SEMICOLON
    end
end GEN_STATEMENT;



procedure GEN_BLOCK( head: lgt_ptr );
var
  lgt: lgt_ptr;

begin
  if head <> nil then
  with head^ do
    if lgt_kind = lgt_ctlflow then
    case lgt_stm of
      stm_return:
        if lgt_parmlst <> nil then
        begin
          OUT_BEGIN; OUT_EOLN;
          GEN_STATEMENT( head );
          OUT_END
        end
        else
          GEN_SCAN( head, 0, false );

      stm_sequence, stm_parallel:
        begin
          OUT_BEGIN; OUT_EOLN;
          lgt := lgt_parmlst;
          while lgt <> nil do
          begin
            if lgt^.lgt_kind <> lgt_null then GEN_STATEMENT( lgt );
            lgt := lgt^.lgt_nxt
          end;
          OUT_END
        end;

      stm_if:
        begin
          OUT_BEGIN; OUT_EOLN;
          GEN_STATEMENT( head );
          OUT_END
        end;

    otherwise
      GEN_STATEMENT( head )
    end
    else GEN_SCAN( head, 0, false );
  if bsemicolon then OUT_SEMICOLON;
  OUT_CEOL
end GEN_BLOCK;



procedure GEN_SCAN_CASE( plst: lgt_ptr; blvl: boolean );
var
  i, w_number, sz: integer;
  lgt1, oth, sel: lgt_ptr;

begin
  with plst^, lgt_cte^ do
  if lgt_nxt <> nil then
  begin
    sz  := val_size;                           { Get the size of the case table }
    oth := lgt_nxt;                            { Get the other index }
    sel := oth^.lgt_nxt;                       { Get the selector expr. }
    if sel <> nil then
    begin
      lgt1 := sel^.lgt_nxt;
      OUT_STR( 'switch (' );
      GEN_SCAN( sel, 0, true );                { Generate the selector expression }
      OUT_SEPAR( ')' ); OUT_CHAR( ' ' ); OUT_BEGIN;
      w_number := -1;                          { Initialize the work count }
      while lgt1 <> nil do
      begin                                    { Loop for each work }
        w_number := w_number + 1;              { Update the work count }
        for i := 0 to sz - 1 do                { Look for this work number ... }
        if val_tab^.lw[i] = w_number then      { ... in the case table }
        begin                                  { For a found selector value ... }
          OUT_CEOL;                            { ... put the C case specification }
          OUT_STR( 'case ' );
          OUT_INT( i + lgt_disp );
          OUT_SEPAR( ':' )
        end;
        GEN_STATEMENT( lgt1 );                 { Generate the body of the work }
        OUT_STR( 'break;' );                   { Append the break statement }
        lgt1 := lgt1^.lgt_nxt
      end {while};
      bsemicolon := false;
      with oth^ do
      if ((lgt_kind = lgt_null) and (lgt_spc in lgt_status)) or
         (lgt_kind <> lgt_null) then
      begin
        OUT_CEOL;
        OUT_STR( 'default:');
        GEN_STATEMENT( oth );                  { Put the otherwise seq. with label }
        OUT_STR( 'break;' )                    { Append the break statement }
      end {if};
      OUT_CEOL;
      OUT_END
    end {if}
  end {if}
end GEN_SCAN_CASE;



procedure GEN_CNTXREF( lgt: lgt_ptr; ifld: integer );
begin
  if lgt <> nil then
  with lgt^ do
  case ifld of
    0: { Previous pointer access }
      begin
        lgt_typ := typ_std[form_nil];
        GEN_VALUE( lgt );
        lgt_typ := cntx_typ
      end;

    1: { Procedure name pointer access }
      begin
        lgt_typ := typ_std[form_nil];
        lgt_disp := lgt_disp + cntx_procname;
        GEN_VALUE( lgt );
        lgt_disp := lgt_disp - cntx_procname;
        lgt_typ := cntx_typ
      end;

    2: { Source file name pointer access }
      begin
        lgt_typ := typ_std[form_nil];
        lgt_disp := lgt_disp + cntx_srcfname;
        GEN_VALUE( lgt );
        lgt_disp := lgt_disp - cntx_srcfname;
        lgt_typ := cntx_typ
      end;

    3: { Source line number access }
      begin
        lgt_typ := typ_std[form_int];
        lgt_disp := lgt_disp + cntx_linenbr;
        GEN_VALUE( lgt );
        lgt_disp := lgt_disp - cntx_linenbr;
        lgt_typ := cntx_typ
      end;

  otherwise
    GEN_ADDRESS( lgt, false )
  end
end GEN_CNTXREF;



procedure GEN_LINENUMBER( lgt: lgt_ptr );
begin
  with lgt^ do
  begin
    GEN_CNTXREF( lgt_parmlst, 3 ); OUT_SEPAR( '=' );
    OUT_INT( lgt_disp ); bspace := false;
    OUT_STR( '; /* line # ' ); OUT_INT( lgt_disp ); OUT_STR( '*/' );
    bsemicolon := false;
    OUT_EOLN
  end
end GEN_LINENUMBER;


(*
procedure GEN_LABEL_LIST( id: ide_ptr );
begin
  repeat
    OUT_LABEL( id^.ide_labadr, false );
    id := id^.ide_labsyn
  until id = nil
end GEN_LABEL_LIST;
*)


procedure GEN_SCAN {( head: lgt_ptr; prior: integer; blvl: boolean )
                    was forward };
{ head -> the lgt record,
  prior   The context C operator priority,
  blvl    true/false for value/(test or not value) result required.
}
const
  mdnam = 'GNSC';

var
  bbeg, bi, bs, first, bref:        boolean;
  i, ind, indll, siz:               integer;
  sav_curr_block, lgt1, lgt2, lgt3: lgt_ptr;
  id:                               ide_ptr;

begin { GEN_SCAN }
  lgt3 := nil;
  if head <> nil then
  with head^ do
  begin
    if (lgt_lbl in lgt_status) and (lgt_lide <> nil) and (lgt_kind <> lgt_srcinfo) then
      OUT_LABEL( lgt_lide^.ide_labadr, false );

    case lgt_kind of
      lgt_ctlflow:
        begin
          case lgt_stm of
            stm_case:
              begin
                sav_curr_block := curr_block;
                curr_block := head;
                GEN_SCAN_CASE( lgt_parmlst, blvl );
                curr_block := sav_curr_block
              end;

            stm_goto:
              if lgt_lab <> nil then
              begin
                OUT_STR( 'goto L_l_' ); OUT_INT( lgt_lab^.ide_labadr );
                OUT_SEMICOLON
              end;

            stm_jump:
              if lgt_lab <> nil then
              begin
                OUT_STR( 'siglongjmp(' );
                if lgt_lab <> nil then
                with lgt_lab^, ide_owner^ do
                begin
                  if ide_lex <= 1 then OUT_STR( 'G_lb_env,' )
                  else
                  begin { For not global env, use the lex access }
                    OUT_STR( '(void *)' );
                    OUT_LEX_SPC( pro_envidx ); OUT_STR( '.v[' );
                    OUT_INT( pro_labelenv div fptr_size ); bspace := false;
                    OUT_STR( '],')
                  end;
                  OUT_INT( ide_labadr + 1 );
                end;
                bspace := false; OUT_CHAR( ')' ); bspace := true
              end;

            stm_if:
              begin
                lgt1 := lgt_parmlst;           { lgt1 = expression}
                lgt2 := lgt1^.lgt_nxt;         { lgt2 = then}
                if lgt2 <> nil then 
                begin
                  lgt3 := lgt2^.lgt_nxt;       { lgt3 = else}
                  if lgt2^.lgt_kind = lgt_null then lgt2 := nil
                end;
                if lgt3 <> nil then
                  if lgt3^.lgt_kind = lgt_null then lgt3 := nil;
                if (lgt2 <> nil) or (lgt3 <> nil) then
                begin
                  OUT_PASTR( 'if ' );
                  OUT_SEPAR( '(' );
                  if lgt2 = nil then
                  begin
                    OUT_SEPAR( '!' );
                    OUT_SEPAR( '(' );
                    GEN_SCAN( lgt1, 0, false );
                    OUT_SEPAR( ')' );
                    OUT_SPACE;
                    lgt2 := lgt3; lgt3 := nil
                  end
                  else GEN_SCAN( lgt1, 0, false );
                  OUT_SEPAR( ')' );
                  OUT_SPACE;
                  bsemicolon := true;
                  GEN_BLOCK( lgt2 );
                  if lgt3 <> nil then
                  begin
                    OUT_PASTR( 'else' );
                    bsemicolon := true;
                    GEN_BLOCK( lgt3 )
                  end
                end
              end;

            stm_for:
              begin
                sav_curr_block := curr_block;
                curr_block := head;
                lgt_status := lgt_status - [lgt_cdg_3];        { Clear the label flg }
                lgt1 := lgt_parmlst;
                OUT_STR( 'for( Rf' );   OUT_INT( for_var_max );
                OUT_SEPAR( '=' );       GEN_NEXT( lgt1, 0, true );
                OUT_SEPAR( ';' );       OUT_STR( 'Rf' );
                OUT_INT( for_var_max ); bspace := false;
                OUT_STR( '>0;Rf' );     OUT_INT( for_var_max );
                bspace := false;        OUT_PASTR( '-- )' );
                for_var_max := for_var_max - 1;
                if lgt1 <> nil then
                begin
                  bi := false;
                  if lgt1^.lgt_nxt <> nil then bi := true;
                  with lgt1^ do
                  if (lgt_kind = lgt_ctlflow) and
                     ((lgt_stm = stm_sequence) or
                      (lgt_stm = stm_return  ) or
                      (lgt_stm = stm_parallel)) then bi := true;

                  if bi then
                  begin 
                    OUT_BEGIN; OUT_EOLN;
                    while lgt1 <> nil do
                    begin  GEN_STATEMENT( lgt1 ); lgt1 := lgt1^.lgt_nxt  end;
                    OUT_END;
                  end
                  else 
                  begin
                    OUT_EOLN;
                    GEN_STATEMENT( lgt1 )
                  end
                end
                else OUT_SEMICOLON;
                if lgt_cdg_3 in lgt_status then OUT_LABEL( lgt_disp, true );
                curr_block := sav_curr_block;
                for_var_max := for_var_max + 1
              end;


            stm_loop:
              begin
                sav_curr_block := curr_block;
                curr_block := head;
                lgt_status := lgt_status - [lgt_cdg_3];        { Clear the label flg }
                lgt1 := lgt_parmlst;
                if lgt1^.lgt_kind <> lgt_null then
                begin { A direct stop loop condition is specified }
                  OUT_STR( 'while (' );
                  GEN_SCAN( lgt1, 0, false );
                  OUT_SEPAR( ')' )
                end
                else
                  OUT_STR( 'while (1)' );
                lgt1 := lgt1^.lgt_nxt;                         { Get the first loop statement }
                if lgt1 <> nil then
                begin
                  bi := false;
                  if lgt1^.lgt_nxt <> nil then bi := true;
                  with lgt1^ do
                  if (lgt_kind = lgt_ctlflow) and
                     ((lgt_stm = stm_sequence) or
                      (lgt_stm = stm_return) or
                      (lgt_stm = stm_parallel)) then bi := true;

                  if bi then
                  begin 
                    OUT_BEGIN; OUT_EOLN;
                    while lgt1 <> nil do
                    begin  GEN_STATEMENT( lgt1 ); lgt1 := lgt1^.lgt_nxt  end;
                    OUT_END
                  end
                  else
                  begin
                    OUT_EOLN;
                    GEN_STATEMENT( lgt1 )
                  end
                end
                else OUT_SEMICOLON;
                curr_block := sav_curr_block;
                if lgt_cdg_3 in lgt_status then OUT_LABEL( lgt_disp, true )
              end;


            stm_sequence, stm_parallel:
              begin
                lgt1 := lgt_parmlst;
                while lgt1 <> nil do
                begin
                  GEN_STATEMENT( lgt1 );
                  lgt1 := lgt1^.lgt_nxt
                end
              end;

            stm_exit: { only for the condition }
              begin
                lgt1 := nil;
                if lgt_parmlst <> nil then
                begin
                  lgt1 := lgt_parmlst^.lgt_nxt;
                  lgt2 := lgt_parmlst^.lgt_parmlst;
                  if lgt2 = curr_block then lgt2 := nil;
                end;
                if lgt1 <> nil then
                begin
                  OUT_PASTR( 'if (' );
                  bspace := false;
                  GEN_SCAN( lgt1, 0, false );
                  OUT_CHAR( ')' );
                  bspace := true
                end; 
                if lgt2 <> nil then
                with lgt2^ do
                begin
                  if not (lgt_cdg_3 in lgt_status) then
                  begin
                    lgt_disp   := label_cnt;
                    label_cnt  := label_cnt + 1;
                    lgt_status := lgt_status + [lgt_cdg_3]     { Set the label flg }
                  end;
                  OUT_STR( 'goto L_l_' ); OUT_INT( lgt_disp )
                end
                else OUT_STR( ' break' );
                OUT_SEMICOLON
              end;

            stm_return:
              begin
                if lgt_parmlst <> nil then
                  GEN_SCAN( lgt_parmlst, 0, false );
                OUT_PASTR( 'goto Ret_Label;' ); bsemicolon := false;
                OUT_EOLN
              end;

          otherwise
          end
        end;

      lgt_specific:
        begin
          OUT_CHARR( SP_code[lgt_icode] );
          OUT_CHAR( '(' );
          OUT_CHAR( '&' );
          bspace := false;
          GEN_SCAN( lgt_parmlst, 0, true );
          OUT_SEPAR( ',' );
          GEN_SCAN( lgt_parmlst^.lgt_nxt, 0, true );
          OUT_CHAR( ')' ); OUT_SEMICOLON
        end;

      lgt_codep:
        case lgt_pcode of
          pcod_noop:
            if (lgt_parmlst <> nil) and (lgt_typ <> nil) then
              if lgt_parmlst^.lgt_typ <> nil then
                if lgt_parmlst^.lgt_typ^.typ_size = lgt_typ^.typ_size then
                  GEN_SCAN( lgt_parmlst, prior, true )
                else                           { Size conversion }
                begin
                  if prior >= 13 then OUT_SEPAR( '(' );
                  GEN_TYP_CONV( lgt_typ, i, false, bref );
                  GEN_SCAN( lgt_parmlst, 13, true );   { 13 = cast op. prior }
                  if prior >= 13 then OUT_SEPAR( ')' )
                end;

          pcod_store:
            if lgt_parmlst <> nil then
            begin
              OUT_STR( 'memcpy(' );
              GEN_ADDRESS( lgt_parmlst, false );
              with lgt_parmlst^ do
              begin
                OUT_SEPAR( ',' );
                GEN_ADDRESS( lgt_nxt, false );
                OUT_SEPAR( ',' );
                with lgt_typ^ do
                  if typ_size > 0 then OUT_INT( typ_size )
                                  else GEN_SCAN( typ_comp_size, 0, true )
              end;
              OUT_SEPAR( ')' ); OUT_SEMICOLON
            end;

          pcod_range:
            begin
              OUT_STR( 'PAS__RANGE(' );
              lgt1 := lgt_parmlst;
              GEN_NEXT( lgt1, 0, true );
              OUT_SEPAR( ',' );
              GEN_NEXT( lgt1, 0, true );
              OUT_SEPAR( ',' );
              GEN_NEXT( lgt1, 0, true );
              OUT_SEPAR( ')' )
            end;

          pcod__genstrl1:
            begin
              OUT_STR( 'PAS__CHAR_IN_STRING(' );
              lgt1 := lgt_parmlst;
              GEN_NEXT( lgt1, 0, true );
              OUT_SEPAR( ',' );
              GEN_NEXT( lgt1, 0, true );
              OUT_SEPAR( ')' )
            end;

          pcod__eq, pcod__ne:
            if lgt_parmlst <> nil then
            with cop_table[pcod_eq] do
            begin
              lgt1 := lgt_parmlst;
              if prior >= cop_prio then   OUT_SEPAR( '(' );
              OUT_STR( 'memcmp(' );
              GEN_NEXT( lgt1, 0, true ); OUT_SEPAR( ',' );
              GEN_NEXT( lgt1, 0, true ); OUT_SEPAR( ',' );
              if lgt_parmlst^.lgt_typ <> nil then
              with lgt_parmlst^.lgt_typ^ do
                if typ_size > 0 then OUT_INT( typ_size )
                                else GEN_SCAN( typ_comp_size, 0, true );
              bspace := false;
              if lgt_pcode = pcod__eq then OUT_STR( ')==0' )
                                      else OUT_STR( ')!=0' );
              if prior >= cop_prio then   OUT_SEPAR( ')' )
            end;

          pcod_istore, pcod_fstore, pcod_gstore:
            if lgt_parmlst <> nil then
            begin
              GEN_SCAN( lgt_parmlst, 0, false );
              OUT_SEPAR( '=' );
              GEN_SCAN( lgt_parmlst^.lgt_nxt, 0, true );
              OUT_SEMICOLON
            end;

          pcod_inc, pcod_dec:
            begin
              GEN_SCAN( lgt_parmlst, 14, false );
              bspace := false;
              if lgt_pcode = pcod_inc then OUT_STR( '++;' )
                                      else OUT_STR( '--;' );
              bsemicolon := false;
              OUT_EOLN
            end;

          pcod_isqr,
          pcod_fsqr,
          pcod_gsqr:
            if CHECK_DIRECT( lgt_parmlst ) then
            with cop_table[pcod_imul] do
            begin
              if prior >= cop_prio then OUT_SEPAR( '(' );
              GEN_SCAN( lgt_parmlst, cop_prio, true ); OUT_SEPAR( '*' );
              GEN_SCAN( lgt_parmlst, cop_prio, true );
              if prior >= cop_prio then OUT_SEPAR( ')' )
            end
            else
              if cmp_lparop then
              begin { Use the ( ..., ... ) C operator }
                OUT_SEPAR( '(' ); OUT_SREG( lgt_typ ); OUT_SEPAR( '=' );
                GEN_SCAN( lgt_parmlst, 0, true ); OUT_SEPAR( ',' );
                OUT_SREG( lgt_typ ); OUT_SEPAR( '*' ); OUT_SREG( lgt_typ );
                OUT_SEPAR( ')' )
              end
              else
              begin { Use an external routine }
                OUT_OPER( lgt_pcode );            OUT_SEPAR( '(' );
                GEN_SCAN( lgt_parmlst, 0, true ); OUT_SEPAR( ')' )
              end;

          pcod_imod:
            begin
              lgt1 := lgt_parmlst;
              if cmp_lparop then
              begin
                OUT_SEPAR( '(' ); OUT_STR( 'Rgi=' );
                GEN_NEXT( lgt1, 12, true ); bspace := false;
                if lgt1 <> nil then
                  with lgt1^ do
                  if lgt_kind = lgt_const then
                  begin
                    OUT_SEPAR( '%');
                    if lgt_cte <> nil then i := ABS( lgt_cte^.val_ival )
                                      else i := 2;
                    OUT_INT( i ); bspace := false;
                    OUT_STR( ',Rgi>=0?Rgi:Rgi+' ); OUT_INT( i );
                    OUT_SEPAR( ')' )
                  end
                  else
                  begin
                    OUT_STR( '%(Rgj=abs(' );
                    GEN_SCAN( lgt1, 0, true ); bspace := false;
                    OUT_STR( ')),Rgi>=0?Rgi:Rgi+Rgj)' )
                  end
              end
              else
              begin { Use an external function }
                OUT_OPER( lgt_pcode );     OUT_SEPAR( '(' );
                GEN_NEXT( lgt1, 0, true ); OUT_SEPAR( ',' );
                GEN_SCAN( lgt1, 0, true ); OUT_SEPAR( ')' )
              end
            end;

          pcod_cvfi,
          pcod_cvgi:
            if lgt_parmlst <> nil then
            if CHECK_DIRECT( lgt_parmlst ) then
            with cop_table[pcod_iadd] do
            begin
              OUT_SEPAR( '(' ); GEN_SCAN( lgt_parmlst, 0, true );
              bspace := false; OUT_STR( '>0.0?(SL)(' );
              GEN_SCAN( lgt_parmlst, cop_prio, true ); bspace := false;
              OUT_STR( '+0.5):(SL)(' );
              GEN_SCAN( lgt_parmlst, cop_prio, true ); bspace := false;
              OUT_STR( '-0.5))' )
            end
            else
              if cmp_lparop then
              begin
                OUT_SEPAR( '(' ); OUT_SREG( lgt_parmlst^.lgt_typ );
                OUT_SEPAR( '=' );
                GEN_SCAN( lgt_parmlst, 0, true ); OUT_SEPAR( ',' );
                OUT_SREG( lgt_parmlst^.lgt_typ ); OUT_STR( '>0.0?(SL)(' );
                OUT_SREG( lgt_parmlst^.lgt_typ ); OUT_STR( '+0.5):(SL)(' );
                OUT_SREG( lgt_parmlst^.lgt_typ ); OUT_STR( '-0.5))' )
              end
              else
              begin { Use an external routine }
                OUT_OPER( lgt_pcode );            OUT_SEPAR( '(' );
                GEN_SCAN( lgt_parmlst, 0, true ); OUT_SEPAR( ')' )
              end;

          pcod_lsh:
            with cop_table[pcod_bic] do
            if lgt_parmlst <> nil then
            begin
              lgt1 := lgt_parmlst^.lgt_nxt;
              if lgt1 <> nil then
                if (lgt1^.lgt_kind = lgt_const) and (lgt1^.lgt_cte <> nil) then
                with lgt1^.lgt_cte^ do
                begin
                  if prior >= cop_prio then OUT_SEPAR( '(' );
                  GEN_SCAN( lgt_parmlst, 0, true );
                  bspace := false;
                  if val_ival > 0 then OUT_STR( '<<' )
                                  else OUT_STR( '>>' );
                  OUT_INT( ABS( val_ival ) );
                  if prior >= cop_prio then OUT_SEPAR( ')' )
                end
                else
                begin
                  OUT_OPER( lgt_pcode );     OUT_SEPAR( '(' );
                  GEN_SCAN( lgt_parmlst, 0, true ); OUT_SEPAR( ',' );
                  GEN_SCAN( lgt1, 0, true ); OUT_SEPAR( ')' );
                end
            end;

          pcod_bic:
            with cop_table[pcod_bic] do
            begin
              lgt1 := lgt_parmlst;
              if prior >= cop_prio then OUT_SEPAR( '(' );
              GEN_NEXT( lgt1, cop_prio, true );
              OUT_OPER( lgt_pcode );
              GEN_SCAN( lgt1, cop_table[pcod_com].cop_prio, true );
              if prior >= cop_prio then OUT_SEPAR( ')' )
            end;

          pcod_com:
            if (lgt_parmlst <> nil) and (lgt_typ <> nil) then
            begin
              i := lgt_typ^.typ_cardinality;
              if i < dst_setw then
              { We must perform a complement of valid bits only }
              with cop_table[pcod_xor] do
              begin
                if prior >= cop_prio then OUT_SEPAR( '(' );
                GEN_SCAN( lgt_parmlst, cop_prio, true );
                OUT_SEPAR( '^' );
                if i = dst_seta - 1 then OUT_INT( maxint )
                                    else OUT_INT( 2**i - 1 );
                if prior >= cop_prio then OUT_SEPAR( ')' )
              end
              else  
              with cop_table[pcod_com] do
              begin
                if prior >= cop_prio then OUT_SEPAR( '(' );
                OUT_SEPAR( '~' );
                GEN_SCAN( lgt_parmlst, cop_prio, true );
                if prior >= cop_prio then OUT_SEPAR( ')' )
              end
            end;

          pcod_bit:
            begin
              lgt1 := lgt_parmlst;
              with cop_table[pcod_bit] do
              if blvl then
              begin { To give a logical value }
                if prior >= cop_table[pcod_ne].cop_prio then OUT_SEPAR( '(' );
                OUT_SEPAR( '(' );
                GEN_NEXT( lgt1, cop_prio, true );
                OUT_SEPAR( '&' );
                GEN_SCAN( lgt1, cop_prio, true ); bspace := false;
                OUT_STR( ')!=0' );
                if prior >= cop_table[pcod_ne].cop_prio then OUT_SEPAR( ')' )
              end
              else
              begin { To perform a test }
                if prior >= cop_prio then OUT_SEPAR( '(' );
                GEN_NEXT( lgt1, cop_prio, true );
                OUT_SEPAR( '&' );
                GEN_SCAN( lgt1, cop_prio, true );
                if prior >= cop_prio then OUT_SEPAR( ')' )
              end
            end;

          pcod_inset:
            begin
              lgt1 := lgt_parmlst;
              if blvl then
              with cop_table[pcod_ne] do
              begin
                if prior >= cop_prio then OUT_SEPAR( '(' );
                bspace := false; OUT_STR( '(1<<' );
                GEN_NEXT( lgt1, cop_table[pcod_setgen].cop_prio, true );
                OUT_SEPAR( '&' );
                GEN_SCAN( lgt1, cop_table[pcod_bit].cop_prio, true );
                OUT_STR( ')!=0' );
                if prior >= cop_prio then OUT_SEPAR( ')' )
              end
              else
              with cop_table[pcod_bit] do
              begin
                if prior >= cop_prio then OUT_SEPAR( '(' );
                bspace := false; OUT_STR( '1<<' );
                GEN_NEXT( lgt1, cop_table[pcod_setgen].cop_prio, true );
                OUT_SEPAR( '&' );
                GEN_SCAN( lgt1, cop_prio, true );
                if prior >= cop_prio then OUT_SEPAR( ')' )
              end
            end;

          pcod_setgt, pcod_setge,
          pcod_setlt, pcod_setle:
            with cop_table[pcod_setlt] do
            begin
              lgt1 := lgt_parmlst; bspace := false;
              if lgt1 <> nil then lgt2 := lgt1^.lgt_nxt;
              if lgt_pcode >= pcod_setge then 
              begin  lgt3 := lgt2; lgt2 := lgt1; lgt1 := lgt3  end;
              if CHECK_DIRECT( lgt1 ) then
              begin
                if prior >= cop_prio then OUT_SEPAR( '(' );
                GEN_SCAN( lgt1, cop_prio, true );
                bspace := false; OUT_STR( '==(' );
                GEN_SCAN( lgt1, cop_table[pcod_band].cop_prio, true );
                OUT_SEPAR( '&' ); 
                GEN_SCAN( lgt2, cop_table[pcod_band].cop_prio, true );
                OUT_SEPAR( ')' );
                if prior >= cop_prio then OUT_SEPAR( ')' )
              end
              else
              if cmp_lparop then
              begin
                OUT_STR( '(Rgi=' ); GEN_SCAN( lgt1, 0, true );
                bspace := false; OUT_STR( ',Rgi==(Rgi&' );
                GEN_SCAN( lgt2, cop_table[pcod_band].cop_prio, true );
                bspace := false; OUT_STR( '))' )
              end
              else
              begin { Use an external function }
                OUT_OPER( lgt_pcode );       OUT_SEPAR( '(' );
                GEN_SCAN( lgt1, 0, true );   OUT_SEPAR( ',' );
                GEN_SCAN( lgt2, 0, true );   OUT_SEPAR( ')' )
              end
            end;

        otherwise
          lgt1 := lgt_parmlst;
          with cop_table[lgt_pcode] do
            case cop_kind of
              cop_una, { Prefix unary operator }
              cop_unb: { Prefix unary operator on boolean operator }
                begin 
                  if prior >= cop_prio then OUT_SEPAR( '(' );
                  OUT_OPER( lgt_pcode );
                  GEN_SCAN( lgt1, cop_prio, cop_kind = cop_una );
                  if prior >= cop_prio then OUT_SEPAR( ')' )
                end;

              cop_aft: { Suffix unary operator }
                begin 
                  if prior >= cop_prio then OUT_SEPAR( '(' );
                  GEN_SCAN( lgt1, cop_prio, true );
                  OUT_OPER( lgt_pcode );
                  if prior >= cop_prio then OUT_SEPAR( ')' )
                end;

              cop_bin, { Binary operator } 
              cop_bib: { Binary operator on boolean operator } 
                begin 
                  if prior >= cop_prio then OUT_SEPAR( '(' );
                  GEN_NEXT( lgt1, cop_prio, cop_kind = cop_bin );
                  OUT_OPER( lgt_pcode );
                  GEN_SCAN( lgt1, cop_prio, cop_kind = cop_bin );
                  if prior >= cop_prio then OUT_SEPAR( ')' )
                end;

              cop_fn1:
                begin 
                  OUT_OPER( lgt_pcode );     OUT_SEPAR( '(' );
                  GEN_SCAN( lgt1, 0, true ); OUT_SEPAR( ')' )
                end;

              cop_fn2:
                begin 
                  OUT_OPER( lgt_pcode );     OUT_SEPAR( '(' );
                  GEN_NEXT( lgt1, 0, true ); OUT_SEPAR( ',' );
                  GEN_SCAN( lgt1, 0, true ); OUT_SEPAR( ')' )
                end;

              cop_fn3:
                begin 
                  OUT_OPER( lgt_pcode );     OUT_SEPAR( '(' );
                  GEN_NEXT( lgt1, 0, true ); OUT_SEPAR( ',' );
                  GEN_NEXT( lgt1, 0, true ); OUT_SEPAR( ',' );
                  GEN_SCAN( lgt1, 0, true ); OUT_SEPAR( ')' )
                end;

            otherwise
            end
        end;

      lgt_icall,
      lgt_call:
        begin
          GEN_CALL( head, lgt_kind = lgt_icall );
          { lgt_cas signal an assignement ":=" lgt statement and lgt_nil a procedure call }
          if (lgt_cas in lgt_status) or (lgt_typ = nil) then OUT_SEMICOLON
        end;

      lgt_iproref:
        with lgt_pro^ do
        begin
          OUT_CHR( '(' );
          OUT_IDENT( pro_stdname ); bspace := false;
          OUT_CHR( ')' );
          GEN_SCAN( lgt_parmlst, 13, true )    { Evaluate the entry address expression (13 = C prior of cast ope. }
        end;

      lgt_eproref,
      lgt_proref:
        with lgt_pro^ do
        begin
          if lgt_kind = lgt_eproref then OUT_STR( '(V)' );
          case pro_pkind of
            pro_formal:
              GEN_ENTRY_REF( head );
(*            with pro_f_all^ do
                (* GEN_REFERENCE( all_lex, all_disp, typ_std[form_nil] ); *)

            pro_standard: OUT_IDENT( pro_stdname );

          otherwise
            OUT_PROC_IDENT( lgt_pro, false )
          end
        end;

      lgt_result: OUT_PASTR( 'Ret' );          { Reference to current funct. result }

      lgt_null: if lgt_parmlst <> nil then GEN_SCAN( lgt_parmlst, prior, blvl )
                                        else
                                          if lgt_lide <> nil then OUT_SEMICOLON;

      lgt_refer:
        begin
          OUT_STR( 'PAS_Ptr'); GEN_V_TYPREF( lgt_typ, i, bref );
          OUT_STR( '(Rk.' );   GEN_V_TYPREF( lgt_typ, i, bref );
          if lgt_disp <> 0 then
          begin
            OUT_CHAR( '+' ); OUT_INT( lgt_disp div i )
          end;
          OUT_SEPAR( ',' ); GEN_SCAN( lgt_parmlst, 0, true ); OUT_SEPAR( ')' )
        end;

      lgt_offset,
      lgt_index,
      lgt_indir,
      lgt_varbl,
      lgt_const: GEN_VALUE( head );

      lgt_address: GEN_ADDRESS( lgt_parmlst, true );

      lgt_srvcall:
        begin
          OUT_STR( ' Srv_' ); OUT_INT( lgt_srvfunc^.srv_ide ); OUT_CHAR( '(' );
          lgt1 := lgt_parmlst;
          while lgt1 <> nil do
          begin  GEN_NEXT( lgt1, 0, true ); if lgt1 <> nil then OUT_CHAR( ',' )  end;
          OUT_CHAR( ')' );
          if (lgt_cas in lgt_status) or (lgt_typ = nil) then OUT_SEMICOLON
        end;

      lgt_srvret:
        begin
          OUT_STR( 'return' );
          if lgt_parmlst <> nil then begin  OUT_CHAR( ' ' ); GEN_SCAN( lgt_parmlst, 0, true ) end;
          OUT_SEMICOLON
        end;

      lgt_srvref:
        begin
          if lgt_icode < 0 then OUT_STR( 'Rp' )
                           else OUT_STR( 'Rv' );
          OUT_INT( ABS( lgt_icode ) )
        end;

      lgt_srcinfo:
        if lgt_parmlst <> nil then
        begin { Source information node }
          lgt1 := lgt_parmlst^.lgt_nxt;

          case lgt_icode of
            0, 1, 2: { New_Line, New_line(forced), NEW_line_and_context }
              begin { parm: cntxv [, srcn ] }
                { Generate the Label when required }
                if (lgt_lbl in lgt_status) and (lgt_lide <> nil) then
                  OUT_LABEL( lgt_lide^.ide_labadr, false );
                if lgt_icode = 2 then
                begin
                  OUT_STR( 'PAS__curr_cntx=' ); GEN_CNTXREF( lgt_parmlst, -1 );
                  OUT_SEMICOLON
                end;

                if lgt1 <> nil then
                begin
                  GEN_CNTXREF( lgt_parmlst, 2 ); OUT_SEPAR( '=' );
                  GEN_NEXT( lgt1, 0, true );     OUT_SEMICOLON
                end;
                GEN_LINENUMBER( head );
                if cmp_debug then
                begin
                  OUT_STR( 'PAS__DEBUG_CHECK(0); ' ); OUT_EOLN
                end
              end;

            3, 5: { Call_Tracing, Init_Tracing }
              begin { parm: cntxv, procn [, srcn ] }
                { Save old context in current one's }
                GEN_CNTXREF( lgt_parmlst, 0 );
                OUT_SEPAR( '=' ); OUT_STR( 'PAS__curr_cntx;' ); OUT_EOLN;
                { Set the new context as current one's }
                OUT_STR( 'PAS__curr_cntx=' ); GEN_CNTXREF( lgt_parmlst, -1 );
                OUT_SEMICOLON;
                { Set the procedure name }
                GEN_CNTXREF( lgt_parmlst, 1 );
                OUT_SEPAR( '=' );
                GEN_NEXT( lgt1, 0, true ); OUT_SEMICOLON;
                { Set source file name }
                GEN_CNTXREF( lgt_parmlst, 2 ); OUT_SEPAR( '=' );
                if lgt1 = nil then OUT_STR( 'NULL' )
                              else GEN_NEXT( lgt1, 0, true );
                OUT_SEMICOLON;
                { Set the line number }
                if (lgt_lbl in lgt_status) and (lgt_lide <> nil) then
                  OUT_LABEL( lgt_lide^.ide_labadr, false );
                GEN_LINENUMBER( head );
                if cmp_debug then
                begin
                  OUT_STR( 'PAS__DEBUG_CHECK(1); ' ); OUT_EOLN
                end
              end;

            4, 6: { Return_Tracing, Exit_tracing }
              { parm: cntxv }
              begin { Restore the previous context as current one's }
                if (lgt_lbl in lgt_status) and (lgt_lide <> nil) then
                  OUT_LABEL( lgt_lide^.ide_labadr, false );
                if cmp_debug then
                begin
                  GEN_LINENUMBER( head );
                  OUT_STR( 'PAS__DEBUG_CHECK(-1); ' ); OUT_EOLN
                end;
                { We put the return label when required (use of return stat.) }
                if return_label then
                begin
                  return_label := false;
                  OUT_PASTR( 'Ret_Label: ;' ); OUT_EOLN
                end;

                OUT_STR( 'PAS__curr_cntx=' ); GEN_CNTXREF( lgt_parmlst, 0 );
                OUT_SEMICOLON
              end;
          otherwise
          end
        end;

      lgt_dynall:
        { Nothing to do, The managment is performed by GEN_FORMAL_SETTING
          and FREE_DYNAMIC procedures }

    otherwise
    end
  end
end GEN_SCAN;



function GEN_FORMAL_SETTING( proc: pro_ptr ): boolean;
{ To generate any formal argument loading when required }
var
  iincr:      integer;
  fp:         ide_ptr;
  bspt, bres: boolean;

begin
  bres := false;
  fp := proc^.pro_parmlst;
  while fp <> nil do
  with fp^ do
  begin { Loop on all formal argument }
    case ide_class of
      cla_fentry:
        if ide_f_all <> nil then
        with ide_f_all^ do
        begin
          if CHECK_FORMAL_COPY( fp ) then
          begin { Copy the formal entry pointer - that is the address of effective }
            OUT_STR( 'Ra.v[' );
            OUT_INT( all_disp div fptr_size ); bspace := false;
            OUT_STR( ']=(V)' );
            OUT_F_IDENT( ide_name );
            OUT_SEMICOLON;
            all_acc := all_acc + [var_copied]
          end
        end;

      cla_varbl:
        if (ide_all <> nil) and (ide_vkind <> var_result) then
        with ide_all^ do
        begin
          if CHECK_FORMAL_COPY( fp ) then
          begin
            if ide_vkind = var_formal then
            begin { Copy the formal - that is the address of effective }
              OUT_STR( 'Ra.v[' );
              OUT_INT( all_disp div fptr_size ); bspace := false;
              OUT_STR( ']=' );
              if ide_typ <> nil then
                if ide_typ^.typ_simple then OUT_STR( '(V)' );
              OUT_F_IDENT( ide_name );
              OUT_SEMICOLON
            end
            else
            with ide_typ^ do
              if typ_simple then
              begin
                OUT_STR( 'Ra.' ); GEN_V_TYPREF( ide_typ, iincr, bspt );
                OUT_DISP_SPC( all_disp div iincr, false );
                OUT_SEPAR( '=' );
                OUT_F_IDENT( ide_name ); OUT_SEMICOLON
              end
              else
              begin { Copy the formal by value }
                if typ_size < 0 then
                begin { Dynamic size => indirect by simulation of var_formal }
                  { Simule a formal parameter to get the indirection }
                  if ide_all <> nil then ide_all^.all_kind := var_formal;
                  OUT_STR( 'Ra.v[' );
                  OUT_INT( all_disp div fptr_size ); bspace := false;
                  OUT_STR( ']=PAS__MEM_ALLOC((Rgi=' );
                  bspt := var_intaccess in ide_vacc;
                  if bspt then ide_vacc := ide_vacc - [var_intaccess];
                  GEN_SCAN( all_size_exp, 0, true ); bspace := false;
                  { and simule for next access an internal access }
                  ide_vacc := ide_vacc + [var_intaccess];
                  OUT_STR( '));' ); OUT_EOLN;
                  OUT_STR( 'memcpy( Ra.v[' );
                  OUT_INT( all_disp div fptr_size ); bspace := false;
                  OUT_STR( '],' ); OUT_F_IDENT( ide_name ); bspace := false;
                  OUT_STR( ',Rgi)' ); OUT_SEMICOLON;
                  bres := true
                end
                else
                begin { Static size }
                  OUT_STR( 'memcpy( Ra.s+' );
                  OUT_INT( all_disp );     OUT_SEPAR( ',' );
                  OUT_F_IDENT( ide_name ); OUT_SEPAR( ',' );
                  OUT_INT( typ_size );
                  OUT_SEPAR( ')' ); OUT_SEMICOLON
                end
              end;
            all_acc := all_acc + [var_copied]
          end
        end;

    otherwise
    end;
    fp := ide_nxt
  end;
  GEN_FORMAL_SETTING := bres
end GEN_FORMAL_SETTING;



procedure FREE_DYNAMIC( pf: ide_ptr );
{ Free all dynamic allocation for large by value formal }
begin
  while pf <> nil do
  with pf^ do
  begin
    if ide_class = cla_varbl then
      if (ide_all <> nil) and (ide_vkind = var_vformal) then
      with ide_all^ do
        if all_size_exp <> nil then
        begin
          OUT_STR( 'PAS__MEM_FREE(Ra.v[' );
          OUT_INT( all_disp div fptr_size ); bspace := false;
          OUT_STR( ']);' ); OUT_EOLN
        end;
    pf := ide_nxt
  end
end FREE_DYNAMIC;



procedure SET_GBLLAB_SWITCH( pr: pro_ptr );
var
 ip1, ip2: ide_ptr;
 icnt: integer;

begin
  with pr^ do
  begin
    if pro_labelenv <> -1 then                 { If it is not a global environment }
    begin { Set the automatic pointer for the lex access }
      OUT_STR( 'Ra.v[' );
      OUT_INT( pro_labelenv div fptr_size );
      bspace := false; OUT_STR( ']=(V)L_lb_env;' );
      OUT_EOLN
    end;
    ip1  := global_labhde;
    icnt := 1;
    with ip1^ do
      if ide_lablnk <> nil then
        if ide_lablnk^.ide_lablnk = nil then icnt := 2
                                        else icnt := 3
      else icnt := 1;

    if icnt > 1 then                           { Large number of global label }
    begin
      OUT_STR( 'switch (sigsetjmp(' );
      if pro_lex = 0 then OUT_STR( 'G_lb_env, 1 ))' )
                     else OUT_STR( 'L_lb_env, 1 ))' );
      OUT_BEGIN; OUT_EOLN;
      while ip1 <> nil do
      with ip1^ do
      begin
        OUT_STR( 'case ' ); OUT_INT( ide_labadr + 1 ); bspace := false;
        OUT_STR( ': goto L_l_' ); OUT_INT( ide_labadr ); OUT_SEMICOLON;
        ip1 := ide_lablnk
      end;
      OUT_STR( 'default: ;' ); OUT_EOLN;
      OUT_END
    end
    else
    begin { Only one global label }
      OUT_STR( 'if (sigsetjmp(' );
      if pro_lex = 0 then OUT_STR( 'G_lb_env, 1 ))' )
                     else OUT_STR( 'L_lb_env, 1 ))' );
      OUT_STR( ' goto L_l_' ); OUT_INT( ip1^.ide_labadr ); OUT_SEMICOLON
    end;
    OUT_EOLN
  end
end SET_GBLLAB_SWITCH;



procedure GENERATE_ROUTINE( srv: srv_ptr );
var
  prm: svp_ptr;
  i: integer;

begin
  OUT_EOLN;
  with srv^ do
  begin
    if srv_typ <> nil then { function }
      if srv_typ^.typ_simple then OUT_C_TYPE( srv_typ )
                             else OUT_PASCH( 'V' )
    else OUT_PASTR( 'void' );
    OUT_STR( ' Srv_' ); OUT_INT( srv_ide );
    OUT_CHAR( '(' );
    prm := srv_prm;
    i := 1;
    while prm <> nil do
    with prm^ do
    begin
      OUT_C_TYPE( svp_typ ); OUT_STR( ' Rp' ); OUT_INT( i );
      i := i + 1; prm := svp_nxt;
      if prm <> nil then OUT_CHAR( ',' )
    end;
    OUT_CHAR( ')' ); OUT_EOLN;
    OUT_BEGIN;
    i := 0;
    prm := srv_lva;
    while prm <> nil do
    with prm^ do
    begin
      OUT_C_TYPE( svp_typ ); OUT_STR( ' Rv' ); OUT_INT( i ); OUT_SEMICOLON;
      i := i + 1; prm := svp_nxt
    end;
    GEN_SCAN( srv_cod, 0, false );
    OUT_END;
    OUT_EOLN
  end
end GENERATE_ROUTINE;



procedure GENERATE_CODE( proc: pro_ptr );
{ Pass 2 of code Generation: generation of C function/procedure body }
var
  i:                    integer;
  lgt1:                 lgt_ptr;
  loc:                  all_ptr;
  fp:                   ide_ptr;
  bdynf, blocal, first: boolean ;

begin
  if objf_open then
  with proc^ do
  if pro_pkind <> pro_inline then
  begin
    blocal := (pro_fdyn_all <> nil);
    loc := nil;
    { Generate a C procedure head }
    OUT_EOLN; OUT_EOLN;
    OUT_PASTR( '/* Procedure/Function : ' );
    if proc^.pro_geneide <> nil then
    with proc^.pro_geneide^ do
    begin
      OUT_CHAR( '"' ); OUT_FULL_IDENT( ide_name ); OUT_PASTR( '" ' )
    end
    else OUT_PROC_IDENT( proc, false );
    OUT_PASTR( '*/' ); OUT_EOLN;

    if pro_pkind = pro_main then OUT_PASTR( 'int main( int argc, C * argv[], C * env[] )' )
                            else GENERATE_ENTRY( proc );
    OUT_BEGIN; OUT_EOLN;

    end_c_block := false;

    { Declare a local label environment when required }
    if pro_labelenv <> -1 then
    begin
      labenv_adr := pro_labelenv;
      OUT_STR( 'sigjmp_buf L_lb_env;' ); OUT_EOLN
    end;

    { Declare the dynamic space and Blk when used }

    GENERATE_AUTOMATIC( proc );

    { Look for register(s) used in code }
    global_labhde := nil;         { List of global label }
    label_cnt   :=  pro_labelcnt; { Set the local label count }

    for_var_cnt := -1;
    for_var_max := -1;

    setreg_i1   := false;
    setreg_i2   := false;
    setreg_v    := false;
    setreg_f    := false;
    setreg_g    := false;

    return_label := false;

    SIZE_REGISTER( pro_lgt );
    bdynf := SIZE_FORMAL_REG( pro_parmlst );

    if setreg_i1 then begin  OUT_STR( 'register int Rgi;' ); OUT_EOLN  end;
    if setreg_i2 then begin  OUT_STR( 'register int Rgj;' ); OUT_EOLN  end;
    if setreg_f  then begin  OUT_STR( 'register F Rgf;'   ); OUT_EOLN  end;
    if setreg_g  then begin  OUT_STR( 'register G Rgg;'   ); OUT_EOLN  end;

    if for_var_max >= 0 then
    begin
      for_var_cnt := for_var_max;
      OUT_PASTR( 'register int' );
      repeat
        OUT_PASTR( 'Rf' ); bspace := false;
        OUT_INT( for_var_cnt );
        if for_var_cnt > 0 then OUT_CHAR( ',' );
        for_var_cnt := for_var_cnt - 1
      until for_var_cnt < 0;
      OUT_SEMICOLON
    end;

    if pro_typ <> nil then                     { Function }
      if pro_typ^.typ_simple then
      begin
        OUT_PASTR( 'register' );               { Limit use to local access }
        OUT_C_TYPE( pro_typ );
        OUT_PASTR( 'Ret;' );
        OUT_EOLN
      end;

    if pro_intacc then                         { Save the call lex link and set it in lex table }
    begin
      OUT_STR( 'register P Lx_Sv;' ); OUT_EOLN;
      OUT_STR( 'Lx_Sv=' ); OUT_LEX_SPC( pro_envidx ); OUT_SEMICOLON;
      OUT_LEX_SPC( pro_envidx ); OUT_PASTR( '.s=Ra.s' ); OUT_SEMICOLON
    end;

    curr_dyn_lex := pro_lex + 1;               { Set the current variable lex }

    OUT_EOLN;
    OUT_PASTR( '/* Code of procedure/function */' );
    OUT_EOLN;

    if pro_pkind = pro_main then
    begin
      OUT_PASTR( 'PAS__INIT( argc, argv, env );' ); OUT_EOLN
    end
    else
      if pro_parmlst <> nil then GEN_FORMAL_SETTING( proc );

    { Must be always after any PAS__INIT or GEN_FORMAL SETTING
      for the Stack integrity when the dynamic allocation is performed
      on the stack }
    if global_labhde <> nil then SET_GBLLAB_SWITCH( proc );

    { Now generate the procedure body }
    GEN_SCAN( pro_lgt, 0, false );
    OUT_CEOL;

    { *** At the end of code *** }

    { We put the return label when required (use of return stat.) }
    if return_label then
    begin { Only when was not down by the tracing }
      OUT_PASTR( 'Ret_Label: ;' ); OUT_EOLN
    end;

    { Add the freeing of any dynamic locations }
    if bdynf then FREE_DYNAMIC( pro_parmlst );

    { Add now the end of main or procedure/function return statements }
    if pro_pkind = pro_main then
    begin
      OUT_PASTR( 'PAS__EXIT( 0 );' );
      OUT_EOLN
    end
    else
    begin
      if pro_intacc then                       { Restore the old lex table link }
      begin
        OUT_LEX_SPC( pro_envidx ); bspace := false;
        OUT_STR( '=Lx_Sv' ); OUT_SEMICOLON
      end;
      if pro_typ <> nil then                   { If it's a function: we generate a C return }
      begin
        OUT_PASTR( 'return(Ret);' );
        OUT_EOLN
      end
    end;

    icolumn := 0;
    OUT_END
  end
end GENERATE_CODE;



[global]
procedure GENERATE_CODE_P2;
{ Pass 2 of code Generation:
   generation of global/static/data structure,
   generation of each C function/procedure body.

   *** Called by CMP_COMPILE on the end of module/program compilation,  ***
   *** Called on compilation success, after any procedure/function      ***
   *** GENERATE_CODE_P1 phasis. The entry queue head is pro_first.      ***
}
var
  pcurr: pro_ptr;
  srvcr: srv_ptr;

begin
  GENERATE_MODULE_HEADING;
  if objf_ok then
  begin
    GENERATE_GLOBAL_OBJ;

    srvcr := srv_first;
    if srvcr <> nil then
    begin
      OUT_EOLN;
      OUT_EOLN;
      OUT_PASTR( ' /* Services Routines */' );
      OUT_EOLN;
      OUT_EOLN;
      while srvcr <> nil do
      with srvcr^ do
      begin
        if srv_ide > 0 then GENERATE_ROUTINE( srvcr );
        srvcr := srv_nxt
      end;
      OUT_EOLN
    end;

    pcurr := pro_first;
    while pcurr <> nil do
    with pcurr^ do
    begin
      if (pro_pkind <> pro_package) or (pro_lgt <> nil) then
        GENERATE_CODE( pcurr );
      pcurr := pro_next
    end;
  end
end GENERATE_CODE_P2;



[global]
procedure GENERATE_START;
{ This procedure is normaly used to complet any informations
  for the main module setting }
begin
  { Empty procedure for C generator }
end GENERATE_START;


end PAS_PASS2.
