{
*************************************************************************
*                                                                       *
*                                                                       *
*                       *  P A S  *  S Y S T E M                        *
*                                                                       *
*                                                                       *
*                    * * *   C o m p i l e r    * * *                   *
*                                                                       *
*                                                                       *
*            ---   Source INSYMBOL Lexical parsing   ---                *
*                                                                       *
*              ---  Version 3.1-B5-2 -- 31/05/2024 ---                  *
*                                                                       *
*           by :                                                        *
*                                                                       *
*             P. Wolfers                                                *
*             www.pierre.wolfers.fr                                     *
*                                             FRANCE.                   *
*                                                                       *
*************************************************************************


/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This program is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This software is distributed in the hope that it will be useful,    //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////


}


{************     CPAS  version    *************}


{
    *** modification(s) from major version ***


                       ---

          Macro statement implementation

                       ---

}


  {**************************************************}
  {*******            Program  Head            ******}
  {**************************************************}


(*
[inherit(    'lib:cpas_b__src',         { Use Basiclib Definitions }
             'lib:pas_env')]            { Use tree definitions }

*)

module PAS_INSYMBOL( Input, Output );   { input and output for user terminal }


{ *** Include basic compiler environment *** }
%include 'PASSRC:pcmp_env';


procedure BACK_TRACING( var f: text ); external 'PAS__BACK_TRACING';



     {********************************************}
     { local variable used for macros expressions }
     {********************************************}

type
  mfnc_symb = ( mfnc_def,               { Defined Macro function }
                mfnc_sub,               { Substr Macro function }
                mfnc_idx,               { Index macro function }
                mfnc_len,               { Length Macro function }
                mfnc_undef              { Undefined macro function }
              );

  mfnc_prec = record
    name:              id_name;         { name of macro function }
    narg:              integer          { Number of argument(s) }
  end;

  mfnc_tbty = array[mfnc_symb] of mfnc_prec;    { Macro function name table type }

  cnd_stat = (                          { Define the condition parsing status values. }
               cnd_if,                  { we are in %if condition part. }
               cnd_ei,                  { we are in %elif condition part. }
               cnd_el,                  { we are in %else condition part. }
               cnd_en                   { we are in %endif condition part. }
             );
const
  gmnam     =   'GBSY';                 { Global module name }

  max_stkp  =       32;                         { Should be very large }

  mfnc_tab = mfnc_tbty[  [ [ 7,  'defined' ],  1 ], [ [ 6,   'substr' ],-3 ],
                         [ [ 5,    'index' ],  2 ], [ [ 5,   'length' ], 1 ],
                         [ [ 8, '********' ],  0 ]
                      ];

  mvl_null = mval_rec[ mval_nul, 0, nil ];


var
  mastk: array[1..max_stkp] of mval_rec;        { Stack for macro expression }
  cnstk: array[1..max_stkp] of cnd_stat;        { Stack for condition status nesting }

  cnlvl,                                        { Source macro Condition level. }
  stkp:                   integer := 0;         { Macro Stack pointer }




     {*********************************}
     {*  Forward procedure declation  *}
     {*********************************}

procedure GET_MACRO_VALUE( var v: mval_rec ); forward;

procedure M_EXPR_INSTK; forward;

procedure MEXP_IDENT; forward;




     {*********************************}
     { Source file specification saver }
     {*********************************}

[global]
procedure SET_SRCFILESPC( icd: integer );
{ Routine to build the source file record used to manage the source back tracing context.
  This information is used for listing file edition and by error message edition when the
  back-tracing is enabled.
  icd =  1 : To create the source context (used by %include directive).
  ICD =  0 : To update the source context (used by %chaine directive).
  ICD = -1 : TO restore the previous source context (used by  %eof directive or the real source EOF.
}
var
  sp:  srf_ptr;
  str: string( 255 );

begin
  if icd >= 0 then
  begin
    str := FILE_SPECIFICATION( src_control^.src_file );
    if str.length = 0 then str := 'TT:';
    if (icd > 0) or (srf_list = nil) then
    begin { Include Mode, create a new context }
      if srf_free = nil then
        NEW( sp ) { when no free record exist, create one source ref. record ... }
      else { ... else take it from the free list. }
      begin  sp := srf_free; srf_free := sp^.srf_previous  end;
      with sp^ do
      begin
        srf_previous := srf_list;
        srf_value    :=      nil;
        VAL_NEW( srf_value, nil )
      end;
      srf_list := sp
    end
    else { Chaine Mode, update context }
      with srf_list^ do
      begin
        VAL_FREE( srf_value );
        VAL_NEW( srf_value, nil )
      end;
    with srf_list^.srf_value^ do
    begin { Fill context withe the new source file specification }
      val_kind := form_string;
      val_size := str.length;
      NEW( val_str, str.length );
      val_str^ := str
    end
  end
  else
  begin { eof Mode, Free the current context }
    sp := srf_list;
    with sp^ do
    begin
      srf_list := srf_previous;
      VAL_FREE( srf_value );
      srf_previous := srf_free
    end;
    srf_free := sp
  end;
  cntx_srinf := true
end SET_SRCFILESPC;





     {************************************************}
     { Title and Subtitle Listing of source managment }
     {************************************************}


[global]
procedure COMPILE_SET_TITLE;
{ Update the listing file title when the source file is changed.
}
var
  st: string( 255 );

begin
  { Create a title string - temporary }
  with lst_current^ do
  begin
    if lst_title <> nil then DISPOSE( lst_title );
    st := FILE_SPECIFICATION( src_control^.src_file );
    if st.length = 0 then st := 'TT:';
    st := def_title||st||'".';
    NEW( lst_title, st.length );
    lst_title^ := st
  end
end COMPILE_SET_TITLE;



[global]
procedure COMPILE_SET_SBTTL( prc: pro_ptr );
{ Update the listing file sub-title when the compiled module
  or entry (procedure or function) is changed.
}
var
  st: string( 255 );

begin
  { Create a title string - temporary }
  with lst_current^ do
  begin
    if lst_sbttl <> nil then DISPOSE( lst_sbttl );
    if prc^.pro_stdname <> nil then
      with prc^.pro_stdname^ do
      begin
        st := SUBSTR( s, 1, l );
        st := def_sbttl||st||'".';
        NEW( lst_sbttl, st.length );
        lst_sbttl^ := st
      end
  end
end COMPILE_SET_SBTTL;





{ ********************************************* }
{ *  INSYMBOL group functions and procedures  * }
{ ********************************************* }



procedure SET_KEYWORD_TREE;
{ To create the Keyword identifier tree.
}

  procedure SETKEYWORD( in_var kname: string; skw: symbol; opkw: operator );
  var
    m, i: integer;
    p, p1, p2: keyword_ptr;


  begin { SETKEYWORD }
    NEW(p);
    with p^ do
    begin
      name.l := kname.length;
      for i := 1 to name.l do  name.s[i] := kname.body[i];
      leftp := nil; rightp := nil;
      symb.sy := skw; symb.op := opkw
    end;
    p1 := keyword_tree;
    if p1 <> nil then
    begin
      repeat
        p2 := p1 { keep the memory of the last p1 };
        with p1^ do
        begin
          m := MATCH( p^.name, name );
          if m > 0 then p1 := rightp
                   else p1 := leftp
        end;
      until p1 = nil;
      if m > 0 then p2^.rightp := p
               else p2^.leftp := p
    end
    else keyword_tree := p
  end SETKEYWORD;

begin { SET_KEYWORD_TREE }
        {**** set keywords symbol operators ****}

  SETKEYWORD(        'or',     lgorop,     or_op );
  SETKEYWORD(       'xor',     lgorop,    xor_op );
  SETKEYWORD(       'and',    lgandop,    and_op );
  SETKEYWORD(       'not',      unaop,    not_op );
  SETKEYWORD(       'div',      mulop,   idiv_op );
  SETKEYWORD(       'mod',      mulop,   imod_op );
  SETKEYWORD(       'rem',      mulop,   irem_op );
  SETKEYWORD(        'in',      relop,     in_op );

        {**** set keywords symbol ****}

  SETKEYWORD(       'nil',      nilsy,     no_op );
  SETKEYWORD(     'label', labeldclsy,     no_op );
  SETKEYWORD(      'goto',     gotosy,     no_op );
  SETKEYWORD(      'type',     typesy,     no_op );
  SETKEYWORD(     'const',    constsy,     no_op );
  SETKEYWORD( 'user_statement',
                          statementsy,     no_op );
  SETKEYWORD(       'var',      varsy,     no_op );
  SETKEYWORD(    'in_var',    invarsy,     no_op );
  SETKEYWORD(   'out_var',   outvarsy,     no_op );
  SETKEYWORD(   'private',  privatesy,     no_op );
  SETKEYWORD(    'access',  pointersy,     no_op ); {tmp?}
  SETKEYWORD(      'file',     filesy,     no_op );
  SETKEYWORD(     'array',    arraysy,     no_op );
  SETKEYWORD(    'record',   recordsy,     no_op );
  SETKEYWORD(       'set',      setsy,     no_op );
  SETKEYWORD(       'use',      usesy,     no_op );
  SETKEYWORD(       'new',      newsy,     no_op );
  SETKEYWORD(      'newh',     newhsy,     no_op );
  SETKEYWORD(   'dispose',  disposesy,     no_op );
  SETKEYWORD(      'open',     opensy,     no_op );
  SETKEYWORD(     'reset',    resetsy,     no_op );
  SETKEYWORD(   'rewrite',  rewritesy,     no_op );
  SETKEYWORD(    'append',   appendsy,     no_op );

  SETKEYWORD(      'read',     readsy,     no_op );
  SETKEYWORD(    'readln',   readlnsy,     no_op );
  SETKEYWORD(     'readv',    readvsy,     no_op );
  SETKEYWORD(     'write',    writesy,     no_op );
  SETKEYWORD(   'writeln',  writelnsy,     no_op );
  SETKEYWORD(    'writev',   writevsy,     no_op );

  SETKEYWORD(   'program',   programsy,    no_op );
  SETKEYWORD(    'module',    modulesy,    no_op );
  SETKEYWORD( 'procedure', proceduresy,    no_op );
  SETKEYWORD(  'function',  functionsy,    no_op );
  SETKEYWORD(    'return',    returnsy,    no_op );
  SETKEYWORD(   'forward',   forwardsy,    no_op );
  SETKEYWORD(    'static',    staticsy,    no_op );
  SETKEYWORD(    'global',    globalsy,    no_op );
  SETKEYWORD(    'packed',    packedsy,    no_op );
  SETKEYWORD(  'external',  externalsy,    no_op );
  SETKEYWORD(  'standard',  standardsy,    no_op );
  SETKEYWORD(    'inline',    inlinesy,    no_op );
  SETKEYWORD(     'begin',     beginsy,    no_op );
  SETKEYWORD(        'if',        ifsy,    no_op );
  SETKEYWORD(      'then',      thensy,    no_op );
  SETKEYWORD(      'else',      elsesy,    no_op );
  SETKEYWORD(     'while',     whilesy,    no_op );
  SETKEYWORD(    'repeat',    repeatsy,    no_op );
  SETKEYWORD(     'until',     untilsy,    no_op );
  SETKEYWORD(       'for',       forsy,    no_op );
  SETKEYWORD(      'loop',      loopsy,    no_op );
  SETKEYWORD(      'exit',      exitsy,    no_op );
  SETKEYWORD(      'case',      casesy,    no_op );
  SETKEYWORD(      'with',      withsy,    no_op );
  SETKEYWORD(       'end',       endsy,    no_op );
  SETKEYWORD( 'otherwise',     othersy,    no_op );
  SETKEYWORD(        'of',        ofsy,    no_op );
  SETKEYWORD(        'do',        dosy,    no_op );
  SETKEYWORD(        'to',        tosy,    no_op );
  SETKEYWORD(    'downto',    downtosy,    no_op );

  SETKEYWORD(   '%define',   mdefinesy,    no_op );
  SETKEYWORD( '%undefine',    mundefsy,    no_op );
  SETKEYWORD(      '%let',      mletsy,    no_op );
  SETKEYWORD(       '%if',       mifsy,    no_op );
  SETKEYWORD(    '%ifdef',    mifdefsy,    no_op );
  SETKEYWORD(  '%ifundef',  mifundefsy,    no_op );
  SETKEYWORD(     '%then',     mthensy,    no_op );
  SETKEYWORD(     '%else',     melsesy,    no_op );
  SETKEYWORD(     '%elif',     melifsy,    no_op );
  SETKEYWORD(    '%endif',    mendifsy,    no_op );

  SETKEYWORD(  '%message',   messagesy,    no_op );
  SETKEYWORD(   '%chaine',    chainesy,    no_op );
  SETKEYWORD(  '%include',   includesy,    no_op );
  SETKEYWORD(   '%pragma',    pragmasy,    no_op );


        {**** set micellious symbol ****}

  SETKEYWORD(      '%eof',      peofsy,    no_op );

	{**** end of the keyword list ****}

end SET_KEYWORD_TREE;



[global]
function  CREATE_MAC_IDENT( var id: id_name; cmdf: boolean := false ): macsymb_ptr;
{ To create macro identifier.
}
const
  mdnam = 'CREM';

var
  p, p1, p2: macsymb_ptr;
  m:             integer;

begin
  p1 := macsymb_tree;
  p2 := nil;
  m := 1;
  while p1 <> nil and (m <> 0) do
  begin
    p2 := p1;
    with p1^ do
    begin
      m := MATCH( id, name );
      if m <> 0 then
        if m > 0 then p1 := rightp
                 else p1 := leftp
    end
  end;
  if m = 0 then { Macro symbol already exist }
    if cmdf then p := p1 { Re-Definition from command No error, the value can be changed }
            else SRC_ERROR_S( mdnam, 921, e_error, id )
  else
  begin { Create a new macro symbol }
    NEW( p );
    with p^ do
    begin
      name     :=       id;
      nextp    :=      nil;
      leftp    :=      nil;
      rightp   :=      nil;
      macti    :=    false;
      value    := mvl_null
    end;
    if p2 = nil then { First macro symbol to create }
      macsymb_tree := p
    else
      if m > 0 then p2^.rightp := p
               else p2^.leftp := p
  end;
  CREATE_MAC_IDENT := p
end CREATE_MAC_IDENT;



[global]
procedure REBUILD_MCTREE;
{ Rebuild the Macro symbol tree:
  Build the sorted symbol list,
  free the remove flagged symbol and
  rebuild the tree with optimization.
}
type
  mctabty( sz: integer ) = array[1..sz] of macsymb_ptr;

var
  nmc:         integer;
  tbp:        ^mctabty;
  lst:     macsymb_ptr;


  procedure SORTMC1( p: macsymb_ptr );
  { Recurssive procedure to sort the symbol }
  var
    rmf:  boolean;

  begin
    with p^ do
    begin
      if leftp  <> nil then SORTMC1( leftp );
      rmf := (name.l = 0);
      if not rmf then { We must skip the symbole to remove }
      begin
        if macsymb_frst = nil then macsymb_frst := p
                              else lst^.nextp := p;
        lst := p;
        nextp := nil;
        nmc := nmc + 1
      end;
      if rightp <> nil then SORTMC1( rightp )
    end;
    if rmf then
    begin { Free the memory of removed macro symbol }
      with p^, value do
        if mvl_str <> nil then DISPOSE( mvl_str );
      DISPOSE( p )
    end
  end SORTMC1;



  function  REBUILD_MCTREE1( n, m: integer ): macsymb_ptr;
  { Recursive function to rebuild an optimized symbol tree }
  var
    ii: integer;
    pp: macsymb_ptr;

  begin
    ii := (n + m) div 2;
    pp := tbp^[ii];
    with pp^ do
    begin
      if ii = n then leftp  := nil
                else leftp  := REBUILD_MCTREE1( n, ii-1 );
      if ii = m then rightp := nil
                else rightp := REBUILD_MCTREE1( ii+1, m )
    end;
    return pp
  end REBUILD_MCTREE1;



begin { REBUILD_MCTREE }
  nmc := 0;                                     { Initialize the count and the symbol list to empty }
  macsymb_frst := nil;
  if macsymb_tree <> nil then                   { Work only when he have some symbol }
  begin
    SORTMC1( macsymb_tree );                    { Build the sorted symbol list }
    if nmc > 0 then
    begin                                       { Build the sorted table of symbol pointer ... }
      lst := macsymb_frst;                      { ... to prepare the re-build of the symbol tree }
      NEW( tbp, nmc );
      for i := 1 to nmc do
      begin  tbp^[i] := lst; lst := lst^.nextp  end;
      { Rebuilt an optimized tree }
      macsymb_tree  := REBUILD_MCTREE1( 1, nmc );
    end
    else
    begin
      macsymb_tree := nil;
      macsymb_frst := nil
    end
  end
end REBUILD_MCTREE;



[global]
procedure INSYMBOL_SRC;
{ Main Input Symbol procedure:
  Perform the basic lexical parsing.
}
const
  mdnam = 'INSY';
  iten  =     10;
  ten   =   10.0;
  one   =    1.0;

  { Some ASCII Definitions for special character(s) in the strings }
  A_NUL = CHR(  0 );    { \z }
  A_BEL = CHR(  7 );    { \a }
  A_BS  = CHR(  8 );    { \b }
  A_HT  = CHR(  9 );    { \t }
  A_LF  = CHR( 10 );    { \n }
  A_VT  = CHR( 11 );    { \v }
  A_FF  = CHR( 12 );    { \f }
  A_CR  = CHR( 13 );    { \r }
  A_ESC = CHR( 27 );    { \e }


type
  chartype = (  nul,  eol,  eos, ctrl,  oth,  orc, quot, { Special or multi-use characters }
                let,  dig, sdot, mulc,  ltc, equc,  gtc,
               coln, andc, lpar, rpar, addc, coma, subc,
               divc, semc, lbra, notc, rbra, indc
             );
  chartabtype = array[CHR(0)..CHR(127)] of chartype;

  charsymbol  = array[andc..indc] of sym_rec;

var
  chartyp: [static] chartabtype := (
         {  0     1     2     3     4     5     6     7 }
  { 000 }  nul,  eol, ctrl, ctrl,  eos, ctrl, ctrl, ctrl,  {*NUL,*SOH, STX, ETX, EOT, ENQ, ACK, BEL  } { *EOT (eos) used as End of Stream }
  { 010 } ctrl, ctrl, ctrl, ctrl, ctrl, ctrl, ctrl, ctrl,  {  BS, TAB,  LF,  VT,  FF,  CR,  SO,  SI  } { *SOH (eol) used as End of Line }
  { 020 } ctrl, ctrl, ctrl, ctrl, ctrl, ctrl, ctrl, ctrl,  { DLE, DC1, DC2, DC3, DC4, NAK, SYN, ETB  } { *NUL (nul) always skipped }
  { 030 } ctrl, ctrl, ctrl, ctrl, ctrl, ctrl, ctrl, ctrl,  { CAN,  EM, SUB, ESC,  FS,  GS,  RS,  US  }
  { 040 }  oth,  orc, quot,  let,  let,  let, andc, quot,  { space  !    "    #    $    %    &    '  }
  { 050 } lpar, rpar, mulc, addc, coma, subc, sdot, divc,  {   (    )    *    +    ,    -    .    /  }
  { 060 }  dig,  dig,  dig,  dig,  dig,  dig,  dig,  dig,  {   0    1    2    3    4    5    6    7  }
  { 070 }  dig,  dig, coln, semc,  ltc, equc,  gtc,  oth,  {   8    9    :    ;    <    =    >    ?  }
  { 100 }  let,  let,  let,  let,  let,  let,  let,  let,  {   @    A    B    C    D    E    F    G  }
  { 110 }  let,  let,  let,  let,  let,  let,  let,  let,  {   H    I    J    K    L    M    N    O  }
  { 120 }  let,  let,  let,  let,  let,  let,  let,  let,  {   P    Q    R    S    T    U    V    W  }
  { 130 }  let,  let,  let, lbra, notc, rbra, indc,  let,  {   X    Y    Z    [    \    ]    ^    _  }
  { 140 }  oth,  let,  let,  let,  let,  let,  let,  let,  {   `    a    b    c    d    e    f    g  }
  { 150 }  let,  let,  let,  let,  let,  let,  let,  let,  {   h    i    j    k    l    m    n    o  }
  { 160 }  let,  let,  let,  let,  let,  let,  let,  let,  {   p    q    r    s    t    u    v    w  }
  { 170 }  let,  let,  let,  oth,  orc,  oth,  oth, ctrl); (*  x    y    z    {    |    }    ~  DEL *)

  charsymb: [static] charsymbol := (
    (lgandop,   and_op), { andc: '&' }
    (lparen,    no_op ), { lpar: '(' }
    (rparen,    no_op ), { rpar: ')' }
    (addop,     add_op), { addc: '+' }
    (comma,     no_op ), { coma: ',' }
    (addop,     sub_op), { subc: '-' }
    (mulop,     div_op), { divc: '/' }
    (semicolon, no_op ), { semc: ';' }
    (lbrack,    no_op ), { lbra: '[' }
    (unaop,     not_op), { notc: '\' }
    (rbrack,    no_op ), { rbra: ']' }
    (indirsign, no_op )  { indc: '^' }
    );


var
  i, idig, i_prec, i_radix, ivl, j, k, n: integer;
  radix, rexp, rfac, rval: double;
  pch: char;
  getnuchar, maxstr, found, sign, bint, bbase, bline_enabled, battr, ovf_error: boolean;
  pkw: keyword_ptr;
  pmc: macsymb_ptr;


procedure NEXTCH;
begin { NEXTCH }
  sy_ch := SRC_INCHAR;
  sy_cmin := sy_ch;
  if (sy_ch >= 'A') and (sy_ch <= 'Z') then sy_cmin := CHR( ORD( sy_ch ) + 32);
  if sy_cmin > CHR( 127) then sy_cmin := '?'
end NEXTCH;


procedure PUT_CHAR;
begin { PUT_CHAR }
  if not maxstr then
  if k < sy_string.capacity then
  begin
    sy_string.body[k] := sy_ch;
    k := SUCC(k)
  end
  else
  begin
    SRC_ERROR( mdnam, 12, e_error );
    maxstr := true
  end
end PUT_CHAR;


function TEST_FIGURES( ch: char ): integer;
var
  va: integer;

begin
  if (ch >= '0') and (ch <= '9') then va := ORD( ch ) - ORD( '0' )
  else
    if bbase and (ch >= 'a') and (ch <= 'f') then va := 10 + ORD( ch ) - ORD( 'a' )
    else va := -1;
  if va >= i_radix then va := -2;
  TEST_FIGURES := va
end TEST_FIGURES;



function FIGURES( ch: char ): integer;
var
  va: integer;
  be: boolean

begin
  va := TEST_FIGURES( ch );
  if va = -2 then
  begin
    va := 0; SRC_ERROR( 'FIGU', 16, e_error )
  end;
  FIGURES := va
end FIGURES;



function OVF_HANDLER( ierr: cc__int ): cc__int;
var
 ner: [static] integer := 0;

begin
  ner := ner + 1;
  if ner > 10 then begin WRITELN( 'Too OVH ERR' ); PASCAL_EXIT( 2 ); end;
  if (ierr >= 20) and (ierr <= 25) then                 { For floatting Overflow error }
  begin
    if not ovf_error then
    begin
      SRC_ERROR( mdnam, 11, e_error );                  { Send the error message }
      rexp := 1.0; rfac := 1.0;
      ovf_error := true                                 { Signal the error }
    end;
    OVF_HANDLER := 1                                    { Does not propagate the error }
  end
  else
    OVF_HANDLER := 0                                    { Propagate any other error }
end OVF_HANDLER;



begin { INSYMBOL_SRC }
  while (sy_ch = ' ') or (sy_ch = src_ch_null) do NEXTCH;
  getnuchar := true;
  battr := sy_attrflg;                                  { Make a local copy of the attribute flag to clear it }
  sy_attrflg := false;
  with src_control^ do  src_wchpt := src_chidx;
  with sy_sym do
  case chartyp[sy_cmin] of
    dig, sdot: { number can be begin by digit or period }
      begin
        ESTABLISH( OVF_HANDLER );                       { Install the overflow handler error }
        ovf_error := false;                             { Set as no OVF error until shown otherwise }
        sy := period; { Assume '.' until shown otherwise}
        op := no_op;
        rval    :=   0.0;
        i_radix :=  iten;
        i_prec  :=     0;
        bbase   := false;
        while chartyp[sy_cmin] = dig do
        begin
          sy := intconst; { it is a number }
          idig := ORD( sy_cmin ) - ORD( '0' );
          if (i_prec > 0) or (idig > 0) then i_prec := i_prec + 1;
          repeat NEXTCH until sy_cmin <> '_';           { Ignore any "_" in a number value }
          if not ovf_error then rval := rval*iten + idig
        end;

        if (sy = intconst) and (sy_ch = '#') then
        begin { rval was the base to use for the numeric cte. }
          i_prec := 0;
          i_radix := ROUND( rval );
          if (i_radix < 2) or (i_radix > 16) then
          begin
            SRC_ERROR( mdnam, 14, e_error );
            i_radix := 10
          end;
          bbase := true;
          rval  :=  0.0;
          loop
            repeat NEXTCH until sy_cmin <> '_';         { Ignore any "_" in a number value }
            idig := FIGURES( sy_cmin );
          exit if idig < 0.0;
            if not ovf_error then rval := rval*i_radix + idig;
            if (i_prec > 0) or (idig > 0) then i_prec := i_prec + 1;
          end
        end;

        { To prepare the read of a possible fractional part or exponnant }
        radix := i_radix; rexp := radix;

        if sy_ch = '.' then
        begin
          if SRC_NEXT_CHAR = '.' then { ".." is following }
            if sy = period then
            begin { Our syntax unit }
              NEXTCH;
              sy := twodot
            end
            else getnuchar := false { For the next syntax unit }
          else
          begin { Decimal/Fractional Period }
            repeat NEXTCH until sy_cmin <> '_';         { Ignore any "_" in a number value }
            radix := i_radix; rexp := radix; rfac := 1.0;
            if TEST_FIGURES( sy_cmin ) < 0 then
              getnuchar := false
            else
            begin
              sy := doubleconst;
              loop
                if rfac <> 1.0 then
                  repeat NEXTCH until sy_cmin <> '_';   { Ignore any "_" in a number value }
                idig := FIGURES( sy_cmin );
              exit if idig < 0;
                rfac := rfac / radix;
                rval := rval + rfac*idig;
                i_prec := i_prec + 1
              end
            end
          end
        end;

        if sy_cmin = '#' then NEXTCH;                   { To be compatible with ADA notation for based cte number }

        if (sy <> period) and (sy <> twodot) then
        begin
          if sy_cmin = 'e' then
          begin
            sy := doubleconst;
            NEXTCH;
            if (sy_ch = '+') or (sy_ch = '-') then
            begin
              if sy_ch = '-' then rexp := one/rexp;
              NEXTCH
            end;
            ivl := 0;

            while chartyp[sy_cmin] = dig do
            begin
              ivl := ivl*10 + (ORD( sy_ch ) - ORD( '0' ));
              NEXTCH
            end;

            { Compute the exponential part (use the current base) }
            rfac := one;
            while (ivl <> 0) and not ovf_error do
            begin
              if ODD( ivl ) then
              begin
                ivl := ivl - 1;
                rfac := rfac*rexp
              end
              else
              begin
                ivl := ivl div 2;
                rexp := SQR( rexp )
              end
            end;
            if not ovf_error then rval := rval*rfac
          end;

          if ovf_error then rval := 0.0;        { Set the Null value for any too large value }

          if sy = intconst then
          begin
            if (rval <= unsmax) and (rval >= intmin) then       { Is integer or unsigned }
            begin                               { Set unsigned value in integer equivalent }
              if rval > intmax then sy_ival := TRUNC( unsmax - rval ) + 1
                               else sy_ival := TRUNC( rval );
              sy_rval := sy_ival;
              if sy_label_flag and (chartyp[sy_cmin] = coln) and
                 (SRC_NEXT_CHAR <> '=') then    { a single ":" is following }
              begin
                NEXTCH;                         { Skip the colon character }
                IDE_INT_LABEL;
                sy := labelsy
              end
            end
            else sy := doubleconst
          end;

          if sy = doubleconst then
          begin
            sy_ival := 0;
            if rval > single"large then sy := doubleconst
            else
            begin
              { i_prec is the number of specified signifiant figures -> we must deduce the conveniant number binary precision }
              if i_radix > 2 then i_prec := i_prec*TRUNC( LN( radix )/LN( 2.0 ) ) + 2
                             else i_prec := i_prec + 2;
              { Select the type to assume a correct accuracy }
              if (i_prec > single"mantissa) then sy := doubleconst
                                            else sy := singleconst
            end
          end;
          sy_rval := rval;
          getnuchar := false
        end;
        REVERT                                  { Uninstall the overflow handler error }
      end;

    let: { keyword or identifier }
      with sy_ident do
      begin
        l := 0;
        repeat
          if l < id_maxsize then
          begin
            { Map to lower case in keywords and identifiers }
            l := SUCC( l );
            s[l] := sy_cmin
          end;
          NEXTCH
        until (chartyp[sy_cmin] <> let) and (chartyp[sy_cmin] <> dig);
        { Now search for known keyword }
        pkw := keyword_tree;
        repeat
          with pkw^ do
          begin
            i := MATCH( sy_ident, name );
            if i <> 0 then
              if i > 0 then pkw := rightp else pkw := leftp
          end
        until (i = 0) or (pkw = nil);
        if pkw <> nil then
          sy_sym := pkw^.symb
        else
        begin{ It is an identifier }
          sy_macsymb := macsymb_tree;
          { Check if it is a macro identifier }
          sy := midentsy;
          i := 1;
          while (i <> 0) and (sy_macsymb <> nil) do
            with sy_macsymb^ do
            begin
              i := MATCH( sy_ident, name );
              if i <> 0 then
                if i > 0 then sy_macsymb := rightp else sy_macsymb := leftp
            end;
          if i = 0 then sy := midentsy
          else
          begin { It is a Standard Pascal identifier }
            if sy_label_flag and (chartyp[sy_cmin] = coln) and
              (SRC_NEXT_CHAR <> '=') then       { A single ":" is following }
            begin
              NEXTCH; sy := labelsy             { Skip the colon character }
            end
            else
            begin
              if chartyp[sy_cmin] = quot then   { An attribut quote was seen }
                sy_attrflg := true;             { Set the flag for attribut quote readden }
              sy := identsy
            end;
          end;
          op := no_op
        end;
        getnuchar := false
      end;

    quot: { quote }
      if battr then
      begin
        sy := attrsign; op := no_op;
        battr := false
      end
      else
      begin { Litteral string }
        op := no_op;
        { Set line mode to ignore any end of line or comment mark }
        with src_control^ do
        begin
          bline_enabled := (src_linemode in src_flags);
          src_commentty := src_nocomment;
          src_flags := src_flags + [src_linemode]
        end;
        pch := sy_cmin;                                 { Keep the quote character }
        sy := stringconst;
        k := 1; maxstr := false;
        bint := true;
        while bint do
        begin
          NEXTCH;
          while chartyp[sy_cmin] = eol do NEXTCH;       { Skip any end of line }
          if chartyp[sy_cmin] = eos then bint := false;
          if sy_ch = pch then
          begin
            if pch = SRC_NEXT_CHAR then
            begin
              NEXTCH; PUT_CHAR
            end
            else
            begin
              src_control^.src_commentty := src_pascomment;
              bint := false
            end
          end
          else
          if (pch = '"') and (sy_ch = '\') then
          begin
            NEXTCH;
            case sy_cmin of
              '\', '''', '"': ;
              'a': sy_ch := A_BEL;      { Bell Alarm }
              'b': sy_ch :=  A_BS;      { Back-Space ~ return from one character }
              'e': sy_ch := A_ESC;      { Escape }
              'f': sy_ch :=  A_FF;      { Form Feed ~ Skip to the next page }
              'n': sy_ch :=  A_LF;      { Line Feed ~ Skip to the next line }
              'r': sy_ch :=  A_CR;      { Carriage return ~ Return to the begin of line }
              't': sy_ch :=  A_HT;      { Horizontal tabulation }
              'v': sy_ch :=  A_VT;      { Vertical tabulation }
              'z': sy_ch := A_NUL;      { The null character (also used by C codes }
              '0'..'9':
                begin { Character code in decimal }
                  i := ORD( sy_cmin ) - ORD( '0' ); NEXTCH;
                  j := 3;
                  while (j > 0) and (sy_cmin >= '0') and (sy_cmin <= '9') do
                  begin
                    i := i*10 + ORD( sy_cmin ) - ORD( '0' );
                    NEXTCH
                  end;
                  if i <= ORD( char"last ) then sy_ch := CHR( i )
                                           else begin  sy_ch := ' '; SRC_ERROR( mdnam, 17, e_error )  end
                end;
            otherwise
              sy_ch := ' '; SRC_ERROR( mdnam, 18, e_error )
            end;
            PUT_CHAR
          end
          else PUT_CHAR
        end;
        sy_string.length := PRED( k );

        with src_control^ do
          if not bline_enabled then
          begin
            src_flags := src_flags - [src_linemode];
            if chartyp[sy_cmin] = eol then sy_ch := ' '
          end
      end;

    coln: {':' or ':='}
      begin
        NEXTCH;
        op := no_op;
        if sy_ch = '=' then sy := becomes
                       else begin  sy := colon; getnuchar := false  end
      end;

    ltc: {'<' or '<=' or '<>'}
      begin
        NEXTCH;
        sy := relop;
        if sy_ch = '=' then op := le_op
        else
        if sy_ch = '>' then op := ne_op
                       else begin  op := lt_op; getnuchar := false  end
      end;

    gtc: {'>' or '>='}
      begin
        NEXTCH;
        sy := relop;
        if sy_ch = '=' then op := ge_op
                       else begin  op := gt_op; getnuchar := false  end
      end;

    orc: { '!','|' = "logical or" or '!!','||' = "concatenation" }
      begin
        pch := sy_ch;
        NEXTCH;
        if sy_ch = pch then begin  sy := addop; op := concat_op  end
                       else begin  sy := lgorop; op := or_op; getnuchar := false  end
      end;

    andc, lpar, addc, coma, subc, divc, semc, lbra, notc:
      { '&', '(', '+', ',', '-', '/', ';', '[', '\' }
      sy_sym := charsymb[chartyp[sy_cmin]];

    rpar, rbra, indc:
      begin { Can be design an objet that can be have an attribute }
        sy_sym := charsymb[chartyp[sy_cmin]];
        NEXTCH;
        if chartyp[sy_cmin] = quot then sy_attrflg := true;     { Set the flag for attribut quote readden }
        getnuchar  := false                     { Mark the character as alreafdy read }
      end;

    mulc: { '*' look at power '**' }
      begin
        NEXTCH;
        if sy_ch = '*' then begin  sy := powop; op := pow_op  end
                       else begin  sy := mulop; op := mul_op; getnuchar := false  end
      end;

    equc: { '=>' = "implicate" }
      begin
        NEXTCH;
        if sy_ch = '>' then begin  sy := implic; op := no_op  end
                       else begin  sy := relop; op := eq_op; getnuchar := false  end
      end;

    eos: { end_of_file }
      begin  sy := peofsy; op := no_op; sy_ch := ' ' ; getnuchar := false  end;

    eol: { end_of_line }
      begin  sy := eolnsy; op := no_op; getnuchar := false; sy_ch := ' '  end;

  otherwise
      sy := nothing;
      op := no_op;
      getnuchar := false;
      if sy_ch <> ' ' then SRC_ERROR( mdnam, 13, e_error );
      sy_ch := ' '

  end { case chartab };
  if getnuchar then NEXTCH
end INSYMBOL_SRC;



procedure SKIP_MCSYMBOL(  in_var smb: setsymb );
begin;
  with sy_sym do
    if not (sy in smb) then
    begin
      while not ((sy in smb) or (sy = eofsy) or (sy = peofsy)) do
      begin
        INSYMBOL_SRC;
        case sy of
          mifsy, mifdefsy, mifundefsy:
            while sy in [mifsy,mifdefsy,mifundefsy] do
            begin
              SKIP_MCSYMBOL( [mendifsy] );        { Skip to the corresponding %endif }
              INSYMBOL_SRC                        { Gobble up the internal %endif }
            end;
          (*
          lbrack:
            begin
              SKIP_SYMBOL( rbrack ); INSYMBOL_SRC
            end;
          repeatsy:
            begin
              SKIP_SYMBOL( untilsy ); INSYMBOL_SRC
            end;
          *)
          peofsy:
            SRC_ERROR( 'MCSK', 932, e_severe ); { Not ended %if* condition in the file. }
        otherwise
          end
      end
    end
end SKIP_MCSYMBOL;



function CB_MACRO_MANAGER( mcp: macsymb_ptr ): str_ptr;
begin
  with mcp^ do
  begin
    macti := false              { Set macro flag to inactive to allow a next call. }
  end;
  CB_MACRO_MANAGER := nil
end CB_MACRO_MANAGER;



[global]
procedure MACRO_REPLACE( mcp: macsymb_ptr );
{ Activate a macro code }
const
  mdnam = 'MACT';

var
  stp:   str_ptr := nil;
  src_p: src_ptr := nil;
  ch: char;

begin
  with mcp^, value do
    if not macti then
    begin
      { Make a string copy of the macro symbol value }
      case mvl_typ of
        mval_int:
          begin
            NEW( stp, 22 );
            WRITEV( stp^, mvl_int:0 )
          end;
        mval_str:
          begin
            NEW( stp, mvl_str^.length );
            stp^ := mvl_str^
          end;
      otherwise
      end;
      macti := true;                            { Set anti recurssive call flag }
      { Install the CallBack routine and start the macro interpretation }
      src_p := SRC_MACRO_OPEN( stp, src_callback[CB_MACRO_MANAGER], mcp );
      sy_ch := src_ch_null;                     { Erase the old  end of identifier character }
    end
    else                                        { The recursive macro call are not allowed }
      SRC_ERROR_S( mdnam, 930, e_error, mcp^.name )
end MACRO_REPLACE;



procedure DEFINE_MACRO_SYMBOL;
{ To define or re-define a macro symbol }
const
  mdnam = 'CRMS';

var
  p: macsymb_ptr;

begin
  with sy_sym do
  begin
    INSYMBOL_SRC;                               { Gobble up the %define symbol }
    loop { Loop for multi-macro-definitions }
      if sy = midentsy then                     { when the macro symbol already exist, signal ... }
      begin                                     { ... the error and skip to the next separator }
        SRC_ERROR_S( mdnam, 921, e_error, sy_ident );
        SKIP_MSYMBOL( [comma,semicolon] )
      end
      else
      if sy <> identsy then                     { When the syntax unit is not an identifier, signal error }
      begin
        SRC_ERROR( mdnam, 922, e_error );       { Error message: "Macro declaration: An identifier was expected" }
        SKIP_MSYMBOL( [comma,semicolon] )
      end
      else
      begin
        p := CREATE_MAC_IDENT( sy_ident );      { Create the macro symbol }
        INSYMBOL_SRC;                           { Gobble up the macro symbol }
        if sy = becomes then                    { A symbol value is specified by := <string> }
        with p^ do
        begin
          INSYMBOL_SRC;
          GET_MACRO_VALUE( value )              { Get the value as a macro expression, sy is separator }
        end
      end;
(*
with p^, value do
begin
WRITE( ' Define macro symbol "', name.s:name.l, '" with the value ' );
if mvl_typ = mval_int then WRITELN( mvl_int:0 )
else if mvl_str <> nil then WRITELN( '''', mvl_str^, '''' )
else WRITELN( '''''' )
end;
*)
    exit if sy <> comma;
      INSYMBOL_SRC                              { Gobble up the comma }
    end;
  if sy <> semicolon then SRC_ERROR( mdnam, 21, e_error )
                     else INSYMBOL_SRC
  end
end DEFINE_MACRO_SYMBOL;



procedure UNDEFINE_MACRO_SYMBOL;
{ To purge/remove macro symbol(s) }
const
  mdnam = 'RMMS';

begin
  with sy_sym do
  begin
    INSYMBOL_SRC;                               { Gobble up the %undefine symbol }
    loop
      if sy <> midentsy then
      begin
        SRC_ERROR( mdnam, 922, e_error );       { A macro symbol was expected }
        SKIP_MSYMBOL( [comma,semicolon] )
      end
      else
      begin
        if sy_macsymb = nil then                { When the symbol is not defined, emit an error message "Undefined macro symbol" ... }
          SRC_ERROR_S( mdnam, 923, e_error, sy_ident )
        else
        with sy_macsymb^ do
        begin
          name.l := 0;                          { ... else marks it to be removed and ... }
          if value.mvl_str <> nil then          { ... free any string value allocation. }
            DISPOSE( value.mvl_str )
        end;
        INSYMBOL_SRC                            { Take the separator  }
      end;
    exit if sy <> comma;
      INSYMBOL_SRC                              { Gobble up the comma }
    end;
    if sy <> semicolon then SRC_ERROR( mdnam, 21, e_error )
                       else INSYMBOL_SRC
  end;
  REBUILD_MCTREE                                { Rebuild the macro symbol tree }
end UNDEFINE_MACRO_SYMBOL;



procedure MACRO_ASSIGN_SYMBOL;
const
  mdnam = 'ASMS';

var
  p:       macsymb_ptr;

begin
  with sy_sym do
  begin
    INSYMBOL_SRC;                               { Gobble up the %let keyword }
    if sy <> midentsy then
    begin
      SRC_ERROR( mdnam, 922, e_error );         { A macro symbol was expected }
      SKIP_MSYMBOL( [comma,semicolon,endsy,mendifsy] )
    end
    else
    begin
      p := sy_macsymb;
      INSYMBOL_SRC;                             { Gobble up the macro identifier }
      if sy <> becomes then SRC_ERROR( mdnam, 32, e_error )     { Error: Expected ":=" }
                       else INSYMBOL_SRC;
      with p^ do
      begin
        if value.mvl_str <> nil then DISPOSE( value.mvl_str );
        GET_MACRO_VALUE( value )
      end
    end;
    if sy <> semicolon then SRC_ERROR( mdnam, 21, e_error )
                       else INSYMBOL_SRC
  end
end MACRO_ASSIGN_SYMBOL;



procedure SHOW_MVAL( var vl: mval_rec );
begin
  WRITE( ' EXP_IDENT -> [', vl.mvl_typ, ', i=', vl.mvl_int:0, ', s="' );
  if (vl.mvl_str <> nil) and (vl.mvl_typ = mval_str) then WRITE( vl.mvl_str^ );
  WRITELN( '"]' )
end SHOW_MVAL;

procedure SHOW_MSTACK( in_var s: string );
begin
  WRITELN( ' ', s, ' stack: stkp = ', stkp:-2 );
  for ii := stkp downto 1 do
  begin
    WRITE( ' [', ii:-2, '] = ' ); SHOW_MVAL( mastk[ii] )
  end
end SHOW_MSTACK;



procedure MVAL_COP( in_var src: mval_rec; var dst: mval_rec );
begin
  dst.mvl_typ := src.mvl_typ;
  dst.mvl_int := src.mvl_int;
  if src.mvl_str <> nil then
  begin
   NEW( dst.mvl_str, src.mvl_str^.length );
   dst.mvl_str^ := src.mvl_str^
  end
  else dst.mvl_str := nil
end MVAL_COP;



procedure MVAL_FREE( var vl: mval_rec );
begin
  if vl.mvl_str <> nil then DISPOSE( vl.mvl_str );
  vl := mvl_null
end MVAL_FREE;



procedure MPUSH( in_var vl: mval_rec );
const
  mdnam = 'MPUS';

begin
  if stkp < max_stkp then stkp := stkp + 1
                     else SRC_ERROR( mdnam, 911, e_severe );    { Macro expression stack Overflow }
  MVAL_COP( vl, mastk[stkp] )
end MPUSH;



procedure MPOP( var vl: mval_rec );
const
  mdnam = 'MPOP';

begin
  if stkp > 0 then
  begin
    vl := mastk[stkp];
    stkp := stkp - 1
  end
  else
  begin
    SRC_ERROR( mdnam, 912, e_severe );                          { Macro expression stack Underflow }
    vl := mvl_null
  end
(*
  ;with vl do
  case mvl_typ of
    mval_int: WRITELN( ' V type Int:', mvl_int:0 );
  otherwise
    WRITE( ' V type Str:' );
    if mvl_str = nil then WRITELN( '""' )
    else WRITELN( '"', mvl_str^, '"' )
  end
*)
end MPOP;



procedure GET_MACRO_VALUE( var vl: mval_rec );
begin
  M_EXPR_INSTK;
  MPOP( vl )
end GET_MACRO_VALUE;



function MEXP_INTV: integer;
var
  ch:             char;
  ii, iv, ln:  integer;
  vl:         mval_rec;

begin
  MPOP( vl );
  with vl do
    if mvl_typ = mval_int then iv := mvl_int
    else if mvl_str = nil then iv := 0
    else
    begin
      with mvl_str^ do
      begin
        iv := 0; ii := 0; ln := length;
        while (ii < ln) and (body[ii] <= ' ') do ii := ii + 1;
        while ii < ln do
        begin
          ch := body[ii];
        exit if ((ch < '0') or (ch > '9')) and (ch > ' ');
          if ch > ' ' then iv := iv*10 + (ORD( ch ) - ORD( '0' ));
          ii := ii + 1
        end
      end;
      DISPOSE( mvl_str )
    end;
  MEXP_INTV := iv
end MEXP_INTV;




procedure MEXP_VL_STR( var vl: mval_rec; var str: string );
begin
  with vl do
    if mvl_typ = mval_int then WRITEV( str, mvl_int:0 )
    else
    begin
      str := mvl_str^;
      DISPOSE( mvl_str );
      mvl_str := nil
    end
end MEXP_VL_STR;



procedure MEXP_STRV( var str: string );
var
  vl: mval_rec;

begin
  MPOP( vl );
  MEXP_VL_STR( vl, str )
end MEXP_STRV;



procedure MEXP_CALL( fn: mfnc_symb; narg: integer );
const
  mdnam = 'MCAL';

var
  na, iv, jv:          integer;
  re:     mval_rec := mvl_null;
  s1, s2:               string;

begin
(*  WRITELN( ' MEXP_CALL' ); *)
  with sy_sym do
  begin
    if fn = mfnc_def then
    begin
      if sy <> lparen then SRC_ERROR( mdnam, 22, e_error )      { Error: Expected "(" }
                      else INSYMBOL_SRC;
      if (sy = midentsy) and (sy_macsymb <> nil) then re.mvl_int := 1
                                                 else re.mvl_int := 0;
      re.mvl_typ := mval_int;
(* WRITELN( ' Defined(', sy_ident.s:sy_ident.l, ') -> ', re.mvl_int:0 ); *)
      INSYMBOL_SRC;
      if sy <> rparen then SRC_ERROR( mdnam, 23, e_error );     { Error: Expected ")" }
      MPUSH( re )
    end
    else
    begin { * Get all effective arguments * }
      na := 0;
      if sy = lparen then
      begin
        sy := comma;
        while sy = comma do
        begin
          INSYMBOL_SRC;
          M_EXPR_INSTK;
          na := na + 1;
        end;
        if sy <> rparen then SRC_ERROR( mdnam, 23, e_error )    { Error: Expected ")" }
      end;
      if ((narg > 0) and (na <> narg)) or
         ((narg < 0) and (na > -narg)) then
      begin { Illegal number of argument(s) for a macro call function }
        SRC_ERROR_S( mdnam, 925, e_error, mfnc_tab[fn].name );
        for ii := 1 to na do MPOP( re );
        MPUSH( mvl_null )
      end
      else
      begin
(* SHOW_MSTACK( ' Before Function' ); *)
        case fn of
          mfnc_len:
            begin
              MEXP_STRV( s1 );
              re.mvl_int := s1.length
(* ;WRITELN( ' Length("', s1, '") -> ', re.mvl_int:0 ); *)
            end;

          mfnc_sub:
            begin
              if narg = 3 then jv := MEXP_INTV else jv := 0;
              if narg > 1 then iv := MEXP_INTV else iv := 1;
              if narg > 0 then MEXP_STRV( s1 ) else s1.length := 0;
              re.mvl_typ := mval_str;
              re.mvl_int := 0;
              s2 := SUBSTR( s1, iv, jv );
              if s2.length > 0 then
              begin
                NEW( re.mvl_str, s2.length );
                re.mvl_str^ := s2
              end
(* ;WRITELN( ' Substr("', s1, '", ', iv:0, ', ', jv:0, ' ) -> "', s2, '"' ); *)
            end;

          mfnc_idx:
            begin
              if narg = 3 then iv := MEXP_INTV else iv := 0;
              if narg < 2 then s2.length := 0 else MEXP_STRV( s2 );
              if narg < 1 then s1.length := 0 else MEXP_STRV( s1 );
              re.mvl_int := INDEX( s1, s2, iv )
(* ;WRITELN( ' Index("', s1, '","', s2, '") -> ', re.mvl_int:0 ); *)
            end;

        otherwise
          SRC_ERROR( mdnam, 929, e_error );                     { Unimplemented macro function }
        end;
        MPUSH( re )
      end
    end
  end
end MEXP_CALL;



procedure MEXP_IDENT; { was forward }
const
  mdnam = 'MIDE';

var
  fn:        mfnc_symb;
  vl:         mval_rec;
  rd, ph:      boolean;

begin
  rd := true;
  ph := true;
  with sy_sym, vl do
  begin
    case sy of
      identsy:
        begin { Search for macro function }
          fn := mfnc_symb"first;
          while fn < mfnc_symb"last do
          begin
          exit if MATCH( sy_ident, mfnc_tab[fn].name ) = 0;
            fn := SUCC( fn )
          end;
          if fn >= mfnc_undef then
          begin
            SRC_ERROR_S( mdnam, 928, e_severe, sy_ident );      { Unknown macro function }
            SKIP_MSYMBOL( [semicolon] )
          end
          else
          begin
            INSYMBOL_SRC;
            MEXP_CALL( fn, mfnc_tab[fn].narg );
            ph := false
          end
        end;

      midentsy:
        if sy_macsymb <> nil then
        begin
          ph := false;
          MPUSH( sy_macsymb^.value )
        end
        else
        begin
          SRC_ERROR_S( mdnam, 923, e_error, sy_ident )  { Undefined macro symbol }
        end;

      stringconst:
        begin
          vl.mvl_typ := mval_str;
          vl.mvl_int :=        0;
          if sy_string.length > 0 then
          begin
            NEW( vl.mvl_str, sy_string.length );
            vl.mvl_str^ := sy_string
          end
        end;

      intconst:
        begin
          vl.mvl_typ := mval_int;
          vl.mvl_int :=  sy_ival;
          vl.mvl_str :=      nil
        end;

      comma,
      semicolon:
        begin
          vl := mvl_null;
          rd :=    false
        end;

    otherwise
      vl := mvl_null;
      rd :=    false;
      SRC_ERROR( mdnam, 926, e_error );                        { Illegal macro expression element }
    end;
    if ph then MPUSH( vl );
    if rd then INSYMBOL_SRC
  end
(* ;SHOW_MSTACK( 'After ExpIdent' ); *)
end MEXP_IDENT;



procedure MEXP_BINOP( ope: operator );
const
  mdnam = 'MBIN';

var
  iv, jv:          integer;
  s1, s2:           string;
  tm:             mval_rec;
  re: mval_rec := mvl_null;

begin
(* SHOW_MSTACK( 'Bin Op' );;*)
  case ope of
    mul_op:
      begin
        jv := MEXP_INTV;
        re.mvl_typ := mval_int;
        re.mvl_int := MEXP_INTV * jv
      end;
    div_op:
      begin
        jv := MEXP_INTV;
        re.mvl_typ := mval_int;
        if jv <> 0 then re.mvl_int := MEXP_INTV / jv
                   else re.mvl_int := 0
      end;
    add_op:
      begin
        jv := MEXP_INTV;
        re.mvl_typ := mval_int;
        re.mvl_int := MEXP_INTV + jv
      end;
    sub_op:
      begin
        jv := MEXP_INTV;
        re.mvl_typ := mval_int;
        re.mvl_int := MEXP_INTV - jv
      end;
    concat_op:
      begin
        MEXP_STRV( s2 );
        MEXP_STRV( s1 );
        s1 := s1||s2;
        re.mvl_typ := mval_str;
        re.mvl_int := 0;
        NEW( re.mvl_str, s1.length );
        re.mvl_str^ := s1
      end;
    lt_op, le_op, ge_op, gt_op, ne_op, eq_op:
      begin
        MPOP( tm );
        MPOP( re );
        re.mvl_typ := mval_int;
        if (tm.mvl_typ = mval_str) or (re.mvl_typ = mval_str) then
        begin
          MEXP_VL_STR( tm, s2 );
          MEXP_VL_STR( re, s1 );
          iv := STR_MATCH( s1, s2 );
          re.mvl_str := nil
        end
        else iv := re.mvl_int - tm.mvl_int;
        case ope of
          lt_op: re.mvl_int := ORD( iv <  0 );
          le_op: re.mvl_int := ORD( iv <= 0 );
          ge_op: re.mvl_int := ORD( iv >= 0 );
          gt_op: re.mvl_int := ORD( iv >  0 );
          ne_op: re.mvl_int := ORD( iv <> 0 );
          eq_op: re.mvl_int := ORD( iv =  0 )
        end
      end;
    and_op:
      begin
        jv := MEXP_INTV;
        iv := MEXP_INTV;
        re.mvl_typ := mval_int;
        re.mvl_int := ORD( (jv > 0) and (iv > 0) )
(* ;WRITELN( ' ', iv:0, ' and ', jv:0, ' -> ', re.mvl_int:0 ) *)
      end;
    or_op:
      begin
        jv := MEXP_INTV;
        iv := MEXP_INTV;
        re.mvl_typ := mval_int;
        re.mvl_int := ORD( (jv > 0) or (iv > 0) )
(* ;WRITELN( ' ', iv:0, ' or ', jv:0, ' -> ', re.mvl_int:0 ) *)
      end;
    xor_op:
      begin
        jv := MEXP_INTV;
        iv := MEXP_INTV;
        re.mvl_typ := mval_int;
        re.mvl_int := ORD( ((jv > 0) and (iv <= 0)) or
                           ((jv <= 0) and (iv > 0)) )
(* ;WRITELN( ' ', iv:0, ' xor ', jv:0, ' -> ', re.mvl_int:0 ) *)
      end;

  otherwise
    SRC_ERROR( mdnam, 927, e_severe )
  end;
  MPUSH( re )
(* ;SHOW_MSTACK( ' End Bin Op' ) *)
end MEXP_BINOP;



procedure MEXP_UNAOP( ope: operator );
const
  mdnam = 'MUNA';

var
  iv, jv:              integer;
  re:     mval_rec := mvl_null;

begin
(* SHOW_MSTACK( 'Una Op' ); *)
  case ope of
    not_op:
      begin
        re.mvl_typ := mval_int;
        re.mvl_str := nil;
        re.mvl_int := ORD( MEXP_INTV <= 0 )
      end;
    sub_op:
      begin
        re.mvl_typ := mval_int;
        re.mvl_str := nil;
        re.mvl_int := -MEXP_INTV
      end;

  otherwise
    SRC_ERROR( mdnam, 927, e_severe )
  end;
(* WRITELN( ' not -> ', re.mvl_int:0 ); *)
  MPUSH( re )
end MEXP_UNAOP;



procedure M_EXPR_INSTK;
const
  mdnam = 'MEXP';

var
  p1, p2:     mval_rec;
  oper:       operator;


  procedure MEXP_AND;
  var
    oper:     operator;

    procedure MEXP_REL;
    var
      oper:         operator;

      procedure MEXP_ADD;
      var
        oper:       operator;

        procedure MEXP_MUL;
        var
          oper:     operator;

          procedure MEXP_UNA;
          var
            oper:   operator;

          begin
            with sy_sym do
            begin
              if (sy = addop) and ((op = sub_op) or (op = add_op)) then
                sy := unaop;
              if sy = unaop then
              begin
                if op <> add_op then oper := op
                                else oper := no_op;
                INSYMBOL_SRC;
                MEXP_UNA;
                if oper <> no_op then MEXP_UNAOP( oper )
              end
              else
              if sy = lparen then
              begin
                INSYMBOL_SRC;
                M_EXPR_INSTK;
                if sy = rparen then INSYMBOL_SRC
                               else SRC_ERROR( mdnam, 23, e_severe )
              end
              else MEXP_IDENT
            end
          end MEXP_UNA;

        begin { MEXP_MUL }
          MEXP_UNA;
          with sy_sym do
            while sy = mulop do
            begin
              oper := op;
              INSYMBOL_SRC;  { Gobble up the operator }
              MEXP_UNA;
              MEXP_BINOP( oper )
            end
        end MEXP_MUL;

      begin { MEXP_ADD }
        MEXP_MUL;
        with sy_sym do
          while sy = addop do
          begin
            oper := op;
            INSYMBOL_SRC;     { Gobble up the operator }
            MEXP_MUL;
            MEXP_BINOP( oper )
          end
      end MEXP_ADD;

    begin { MEXP_REL }
      MEXP_ADD;
      with sy_sym do
        while sy = relop do
        begin
          oper := op;
          INSYMBOL_SRC;     { Gobble up the operator }
          MEXP_ADD;
          MEXP_BINOP( oper )
        end
    end MEXP_REL;

  begin { MEXP_AND }
    MEXP_REL;
    with sy_sym do
      while sy = lgandop do
      begin
        oper := op;
        INSYMBOL_SRC;   { Gobble up the operator }
        MEXP_REL;
        MEXP_BINOP( oper )
      end
  end MEXP_AND;

begin { M_EXPR_INSTK }
  MEXP_AND;
  with sy_sym do
    while sy = lgorop do
    begin
      oper := op;
      INSYMBOL_SRC;     { Gobble up the operator }
      MEXP_AND;
      MEXP_BINOP( oper )
    end
end M_EXPR_INSTK;



procedure MEXPR_SVAL( var str: string );
begin
  M_EXPR_INSTK;
  MEXP_STRV( str )
end MEXPR_SVAL;



function MEXPR_IVAL: integer;
begin
  M_EXPR_INSTK;
  MEXPR_IVAL := MEXP_INTV
end MEXPR_IVAL;



function MEXPR_BVAL: boolean;
begin
  M_EXPR_INSTK;
  MEXPR_BVAL := (MEXP_INTV > 0)
end MEXPR_BVAL;




procedure PRAGMA_MSTATE( p_src: src_ptr );
const
  mdnam = 'PRAG';
  nopt  =     26;

  mod_flgs = [src_becho,src_blist,src_bmacroex];

type
  pragmatyps = ( { * Definitions of pragma options }
                 prgm_list_on,          { Enable/Disable listing }
                 prgm_list_off,
                 prgm_lmac_on,          { Enable/Disable macro listing }
                 prgm_lmac_off,
                 prgm_echo_on,          { Enable/Disable input echo }
                 prgm_echo_off,
                 prgm_listlvl,          { Change the Source Listing Level }
                 prgm_format,           { Change the line source format }
                 prgm_cp_list_on,       { Enable/Disable Tree P Code listing }
                 prgm_cp_list_off,
                 prgm_cp_bin_exe_on,    { Create an executable file }
                 prgm_cp_bin_exe_off,   { Does not create an executable file }
                 prgm_cp_bin_on,        { Enable/Disable object file creation }
                 prgm_cp_bin_off,
                 prgm_cp_exe_on,        { Enable/disable executable file creation }
                 prgm_cp_exe_off,
                 prgm_trace,            { Modify the trace mode/accuracy }
                 prgm_debug_on,         { Enable/Disable the debug mode }
                 prgm_debug_off,
                 prgm_range_on,         { Enable/Disable Pascal Range check }
                 prgm_range_off,
                 prgm_code_option,      { C code option }
                 prgm_pcmpdeb_on,       { Enable/Disable the compiler debug mode }
                 prgm_pcmpdeb_off
               );

var
  { warning this table must be modified when the identifier size is changed }
  optnam: [static] array[pragmatyps] of id_name := (
  ( 7,'list_on        '),       { list_on        --list on }
  ( 8,'list_off       '),       { list_off       --list off }
  (13,'macro_list_on  '),       { lmac_on        --Macro list on }
  (14,'macro_list_off '),       { lmac_off       --Macro list off }
  ( 7,'echo_on        '),       { echo_on        --echo on }
  ( 8,'echo_off       '),       { echo_off       --no echo }
  ( 7,'listlvl        '),       { listlvl        --listing source level (-L<n>) }
  ( 6,'format         '),       { format         --listing source format }
  (10,'cp_list_on     '),       { cp_list_on     --listing object P code (-C) }
  (11,'cp_list_off    '),       { cp_list_off    --no listing object P code }
  (13,'cp_bin_exe_on  '),       { cp_bin_exe_on  --object P code bin. file et/ou exec. tree  }
  (14,'cp_bin_exe_off '),       { cp_bin_exe_off --no object P code bin. file et/ou exec. tree }
  ( 9,'cp_bin_on      '),       { cp_bin_on,     --object P code bin. file }
  (10,'cp_bin_off     '),       { cp_bin_off     --no object P code bin. file (-N) }
  ( 9,'cp_exe_on      '),       { cp_exe_on      --create an executable file (no-effect) }
  (10,'cp_exe_off     '),       { cp_exe_off     --no executable file created (no-effect) }
  ( 5,'trace          '),       { trace          --Pascal error tracing (-T<n>) }
  ( 8,'debug_on       '),       { debug_on       --Pascal debuging on (-D) }
  ( 9,'debug_off      '),       { debug_off      --Pascal debuging off }
  ( 8,'range_on       '),       { range_on       --Pascal range check on (-R) }
  ( 9,'range_off      '),       { range_off      --Pascal range check off }
  (11,'code_option    '),       { code_option    --Code generator pragma }
  (10,'pcmpdeb_on     '),       { pcmpdeb_on     --Compiler Debug activation }
  (11,'pcmpdeb_off    ')        { pcmpdeb_off    --Compiler Debug unactivation }
  );

  opt:                              pragmatyps;
  i, j, k, ierr:                       integer;
  s1, s2, s3, fname:    string( 255 )   :=  '';
  plist:                lst_ptr :=         nil;
  save_status:                       src_flagw;
  sav_label_flg, swflg, bf:            boolean;
  vl:                     mval_rec := mvl_null;

begin { PRAGMA_MSTATE }
  INSYMBOL_SRC;                                         { Gobble up the %pragma symbol }
  sav_label_flg := sy_label_flag;
  sy_label_flag := false;
  with sy_sym, p_src^ do
  begin
    save_status := mod_flgs*src_flags;
    while sy = identsy do
    begin
      bf := false;
      opt := pragmatyps"first;
      loop
        bf := MATCH( sy_ident, optnam[opt] ) = 0;
      exit if bf or opt = pragmatyps"last;
        opt := SUCC( opt )
      end;
      INSYMBOL_SRC;                                     { Gobble up the option name }
      case opt of
        prgm_list_on:
          begin                                         { Enable listing (with optional listing file specification) }
            GET_MACRO_VALUE( vl );                      { Get the listing file specification as a macro expression }
            if vl.mvl_typ = mval_str then
            begin
              MEXP_VL_STR( vl, fname );                 { Get the string value }
              if fname.length > 0 then
              with lst_current^ do
              begin
                { Flush output if some output line is begining }
                if lst_currline^.length > 0 then LST_EOLN;

                if lst_heading <> nil then s1 := lst_heading^;
                if lst_title   <> nil then s2 := lst_title^;
                if lst_sbttl   <> nil then s3 := lst_sbttl^;

                LST_OPEN( lst_current,
                          fname,
                          s1,
                          s2,
                          s3,
                          132,                          { 132 char. / line }
                          -1,                           { tty/no => dis./ena. page managment }
                          false,                        { New version mode }
                          false,                        { No print on close time }
                          false,                        { No append }
                          ierr );

                if ierr <> 0 then
                  if not sy_init_mod then
                    { If error lst -> terminal }
                    SRC_ERROR( mdnam, 905, e_error )
                  else
                    save_status := save_status - [src_blist]
                else
                begin
                  if src_lstmxlev < src_level then src_lstmxlev := src_level;
                  save_status := save_status + [src_blist]
                end
              end
              else                                      { Null file specification => no listing }
                save_status := save_status - [src_blist]
            end;
            if src_lstmxlev < src_level then src_lstmxlev := src_level;
            save_status := save_status + [src_blist]
          end;

        prgm_list_off: save_status := save_status - [src_blist];        { Disable the source listing }

        prgm_lmac_on:  save_status := save_status + [src_bmacroex];     { Enable/disable the Macro listing }
        prgm_lmac_off: save_status := save_status - [src_bmacroex];

        prgm_echo_on:  save_status := save_status + [src_becho];        { Enable the echo mode }
        prgm_echo_off: save_status := save_status - [src_becho];        { Disable the echo mode }

        prgm_listlvl:
           begin                                        { Change the Source Listing Level (same effect that the -l<n> option) }
             if sy = colon then INSYMBOL_SRC;
             if sy = intconst then
             begin
               src_lstmxlev := sy_ival;
               INSYMBOL_SRC
             end
             else src_lstmxlev := 1
           end;

        prgm_format:
           begin                                        { Change the line source format }
             if sy = colon then INSYMBOL_SRC;
             if sy = intconst then
             begin
               src_frspos := sy_ival;
               INSYMBOL_SRC
             end
             else src_frspos := 1;
             if sy = colon then INSYMBOL_SRC;
             if sy = intconst then
             begin  src_lstpos := sy_ival; INSYMBOL_SRC  end
             else src_lstpos := 255
           end;

        prgm_cp_list_on:     cmp_macf :=  true;         { Enable the P code tree Listing (same effect that the -c option) } {  9 }
        prgm_cp_list_off:    cmp_macf := false;         { Disable the P code tree Listing } { 10 }

        prgm_cp_bin_exe_on:  cmp_cobj :=  true;         { Enable the creation of executable (presently no-effect) } { 11 }
        prgm_cp_bin_exe_off: cmp_cobj := false;         { Disable the creation of executable (presently no-effect) } { 12 }

        prgm_cp_bin_on:                                 { Enable object file Creation } { 13 }
          if sy = stringconst then
            if sy_ident.l > 0 then
            begin
              if pas_obj <> nil then BINARY_OBJF_CLOSE;
              NEW( pas_obj, sy_string.length );
              pas_obj^ := sy_string;
              INSYMBOL_SRC;  cmp_objf := true           { Gobble up the object file name }
            end
            else SRC_ERROR( mdnam, 904, e_fatal )
          else
          begin
            if cmp_objf then BINARY_OBJF_CLOSE; cmp_objf := false
          end;

        prgm_cp_bin_off:                                { Disable object file Creation (same effect that the -N option)} { 14 }
          if cmp_objf then
          begin
            BINARY_OBJF_CLOSE; cmp_objf := false
          end;

        prgm_cp_exe_on:      cmp_bltt :=  true;         { Enable the in-line build tree (presently no-effect) } { 15 }
        prgm_cp_exe_off:     cmp_bltt := false;         { Disable the in-line build tree (presently no-effect) } { 16 }

        prgm_trace:
          if sy = intconst then
          begin                                         { PASCAL line tracing  accuracy  (same effect that the -T<n> option) }
            { cmp_traceopt > 0 => trace precision cmp_line }
            { cmp_traceopt = 0 => No PASCAL RTL ERROR tracing }
            cmp_traceopt := sy_ival;
            INSYMBOL_SRC;                               { Gobble up the trace value }
          end
          else SRC_ERROR( mdnam, 906, e_severe );

        prgm_debug_on:   cmp_debugopt :=  true;         { Enable the Debug mode (same effect that the -D option) }
        prgm_debug_off:  cmp_debugopt := false;         { Disable the debug mode }

        prgm_range_on:   cmp_range    :=  true;         { Enable Pascal Range Check (same effect that the -R option) }
        prgm_range_off:  cmp_range    := false;         { Disable Pascal Range Check }

        prgm_code_option:    GENERATION_PRAGMA;         { Code generator options }

        prgm_pcmpdeb_on:   cmp_cmpdbg :=  true;         { Enable compiler debug mode - for compiler dev. }
        prgm_pcmpdeb_off:  cmp_cmpdbg := false;         { Disable compiler debug mode - for compiler dev. }

      otherwise { No legal option warning }
        SRC_ERROR_S( mdnam, 901, e_warning, sy_ident )
      end { * case };
      if sy = comma then INSYMBOL_SRC
    end;
    if sy <> semicolon then SRC_ERROR( mdnam, 21, e_error )
                       else INSYMBOL_SRC;
    src_flags := (src_flags - mod_flgs) + save_status
  end;
  sy_label_flag := sav_label_flg
end PRAGMA_MSTATE;



procedure INCLUDE_MSTATE( bnsferr: boolean );
const
  mdnam = 'INCL';

var
  src_p:       src_ptr;
  fext:        str_ptr;
  i, ierr:     integer;
  txtline,
  fname:   string(254);
  bincl:       boolean;

begin { INCLUDE_MSTATE }
  with sy_sym do
  begin
    bincl := (sy = includesy);                  { Keep memory of Include or Chaine directive }
    INSYMBOL_SRC;                               { Gobble up Include or chain symbol }
    MEXPR_SVAL( fname );                        { Get a string macro expression as source file specification }
    src_p := nil;                               { To force allocation by src_open }
    i := INDEX( fname, '.', -1 );
    if i = 0 then fname := fname||'.pas';
(*  INSYMBOL_SRC; *)
    src_p := SRC_FILE_OPEN( fname );            { Open this file }
    if src_openerr = 0 { no open error } then
    begin
      with src_p^ do
      begin
        src_previous := src_control;            { Link the file to the previous src. }
        src_level  := src_control^.src_level;
        if bincl then src_level := src_level + 1;
        src_lstmxlev := src_control^.src_lstmxlev;
        src_insnb  := src_control^.src_insnb;
        src_frspos := src_control^.src_frspos;
        src_lstpos := src_control^.src_lstpos;
        src_flags  := src_flags +
          src_control^.src_flags * [src_blist,src_bmacroex,src_bphys,
                                    src_becho,src_echerr]
      end;
      if sy = comma then                        { Look for %PRAGMA options }
      begin
        INSYMBOL_SRC;
        PRAGMA_MSTATE( src_p )
      end
      else
        if sy <> semicolon then SRC_ERROR( mdnam, 21, e_error );
      if not bincl then SRC_END_OF_LINE;
      src_control := src_p;                     { Switch to new source file }
      if cmp_trace > 0 then cmp_tracecount := 0;{ Force the backtracing }
      COMPILE_SET_TITLE;
      SET_SRCFILESPC( ORD( bincl ) );
      sy_ch := ' ';                             { Init the read character }
      sy := chainesy;                           { Force to unsignificant value }
      if not bincl then                         { Chaine macro STATEMENT }
      begin
        with src_control^ do
        begin
          src_p := src_previous;
          src_previous := src_p^.src_previous   { Get the true previous src}
        end;
        SRC_CLOSE( src_p, true )                { Elliminate the old source file context }
      end;
      with src_control^ do
        if src_blist in src_flags then
        begin
          LST_NEWLINE;
          WRITEV( txtline, ' * The current source file is now "', fname,'"');
          LST_PUT_STRING( txtline )
        end
    end else
    begin
      LST_NEWLINE;
      WRITEV( txtline, ' Open Error (RTL) # ', src_openerr:3, ' for the Source file :' );
      LST_PUT_STRING( txtline );
      LST_NEWLINE; WRITEV( txtline, fname ); LST_PUT_STRING( txtline );
      GET_RTL_MESSAGE( src_openerr, fname );
      LST_NEWLINE; WRITEV( txtline, ' RTL message : ', fname ); LST_EOLN;
(*
      WRITELN( ' Open Error (RTL) # ', src_openerr:3 );
      LST_NEWLINE;
      WRITELN( ' for the Source file "', fname, '".' );
      GET_RTL_MESSAGE( src_openerr, fname );
      LST_NEWLINE;
      WRITELN( ' RTL message : ', fname );
*)
      while (sy <> semicolon)
        and (sy <> eofsy)
        and (sy <> peofsy) do INSYMBOL_SRC;
      if bnsferr then
        SRC_ERROR( mdnam, 902, e_fatal )        { Cannot open the file }
      else
        SRC_ERROR( mdnam, 902, e_severe )
    end
  end;
  INSYMBOL_SRC
end INCLUDE_MSTATE;



procedure ENDFILE_MSTATE;
var
  src_p:       src_ptr;
  mcs_p:   macsymb_ptr;
  txtline: string( 254 );

begin { ENDFILE_MSTATE }
  with src_control^ do
  begin
    if src_previous = nil then sy_sym.sy := eofsy
    else                                        { Return to a previous source file }
    begin
      sy_ch := SRC_RETURN( true );              { Perform the close, free the src record, and return the control to the father source. }
      sy_sym.sy := nothing;
      sy_ch := ' ';
      SET_SRCFILESPC( -1 );                     { Restore the back tracing source file info and restore ... }
      COMPILE_SET_TITLE                         { ... the parent source file specification for the the page heads. }
    end;
    with src_control^ do
      if src_blist in src_flags then
      begin
        LST_NEWLINE;
        WRITEV( txtline, ' * The current source file is now "',
                         FILE_SPECIFICATION( src_file ), '"' );
        LST_PUT_STRING( txtline )
      end
  end;
  if cmp_trace > 0 then cmp_tracecount := 0;    { Force the backtracing }
  if sy_sym.sy = eofsy then pas_compile := false{ For definitive EOF we stop, ... }
                       else INSYMBOL_SRC;       { ... else we continue to read from parent. }

end ENDFILE_MSTATE;



function CND_PUSH( cds: cnd_stat ): boolean;
var b: boolean := true;
begin
  if cnlvl < max_stkp then
  begin
    cnlvl := cnlvl + 1;
    cnstk[cnlvl] := cds;
  end
  else
  begin
    SRC_ERROR( gmnam, 950, e_fatal );
    b := false
  end;
  CND_PUSH := b
end CND_PUSH;


function CND_POP: boolean;
var b: boolean := true;
begin
  if cnlvl <= 0 then
  begin
    SRC_ERROR( gmnam, 951, e_fatal );
    b := false;
  end
  else cnlvl := cnlvl - 1;
  CND_POP := b
end CND_POP;



function CND_CHECK( cds: cnd_stat ): boolean;
var b: boolean := true;
begin
  if cnlvl > 0 then
  begin
    if cds >= cnstk[cnlvl] then cnstk[cnlvl] := cds
                           else b := false
  end else b := false;

  if not b then SRC_ERROR( gmnam, 952, e_fatal );
  CND_CHECK := b
end CND_CHECK;



function MIF_MSTATE: boolean;
{ To manage mifsy, melifsy, mifdefsy, mifundefsy copnditionnal compilation }
const
  mdnam = 'MCIF';

var
  kw:      symbol;
  btrue:  boolean;

begin
  with sy_sym do
  begin
    kw := sy;                           { Keep the memory of %if* directive }
    INSYMBOL_SRC;                       { Gobble up the %if* (or %elif) keyword directive }
    case kw of
      mifdefsy,                         { %ifdef <macro_symbol> %then  ... [%elif ... ] [ %else ...] %endif }
      mifundefsy:                       { %ifundef <macro_symbol> %then  ... [%elif ... ] [ %else ...] %endif }
        begin
          btrue := (sy_macsymb <> nil) xor (kw = mifundefsy);
          INSYMBOL_SRC;                 { Get the %then Keyword and ... }
          CND_PUSH( cnd_if )            { ...  evaluation, establish the if nesting level }
        end;

      mifsy:                            { %if <Macro_expression> %then ... [ %elif ... ] [ %else ...] %endif }
        begin
          btrue := MEXPR_BVAL;          { Get the Macro expression, the %then keyword stop the expression ... }
          CND_PUSH( cnd_if )            { ...  evaluation, establish the if nesting level }
        end;

      melifsy:                          { %if <Macro_expression> %then ... [ %else ...] %endif }
        begin
          btrue := MEXPR_BVAL;          { Get the Macro expression, the %then keyword stop the expression Evaluation }
          CND_CHECK( cnd_ei )           { %elif must in %if macro statement after %if end before %Else or %endif }
        end;

    otherwise
    end;
    if sy <> mthensy then SRC_ERROR( mdnam, 932, e_error )
  end;
  MIF_MSTATE := btrue
end MIF_MSTATE;



[global]
procedure INSYMBOL;

{ Handle directly all the following STATEMENTs :
        includesy,      - %include : include a source file,
        chainesy,       - %chaine : chaine a source file,
        pragmasy,       - %pragma : pragma to compiler option setting,
        peofsy          - %eof : end of file seen or %endfile symbol,
        mdefinesy       - %define : Macro symbol definition statement.
        mundefsy,       - %undefine : Macro symbol remove statement.
        mletsy,         - %let : macro symbol value change statement.
        mifsy,          - %if, %ifdef, %ifundef : Three kinds of begin conditional compilation ...
        mifdefsy,          when macro symbol is defined, when macro symbol is undefined and ...
        mifundefsy,        when macro expression is true.
        melse,          - Else reverse macro if condition.
        mendif          - End of conditional compilation section.
}

const
  mdnam = 'MSYM';

var
  bcmp, bmvl:           boolean;
  sch:                     char;
  smsg:  [static] string( 254 );

begin { INSYMBOL }
  INSYMBOL_SRC;
  with sy_sym, src_control^ do
  while sy >= mdefinesy do
  begin
    case sy of
      includesy,chainesy:
        INCLUDE_MSTATE( sy_init_mod );

      pragmasy:
        PRAGMA_MSTATE(src_control);

      mdefinesy:
        DEFINE_MACRO_SYMBOL;

      mundefsy:
        UNDEFINE_MACRO_SYMBOL;

      mletsy:
        MACRO_ASSIGN_SYMBOL;

      mifsy, mifdefsy, mifundefsy:              { ** Conditional compilation ** }
        begin
          sch := src_fchcd;                     { Save the current listing line flag.}
          repeat
            bcmp := MIF_MSTATE;                 { Get the compilation condition and when the condition is verify ... }
          exit if bcmp;                         { We stop loop when %if or an %elsesy has a true condition. }
            { ... we skip the next code until we find %else or %endif keyword. }
            src_fchcd := 'S';                   { Set the suppress line Flag }
            SKIP_MCSYMBOL( [melifsy,melsesy,mendifsy] );
            src_fchcd := sch
          until sy <> melifsy;
          if bcmp or sy = melsesy then
          begin
            src_fchcd := sch;                   { Restore the normal listing line flag, ... }
            CND_CHECK( cnd_el );                { ... mark the condition success ... }
            INSYMBOL_SRC                        { ... and gobble up %then of %else. }
          end
          else
          begin                                 { Any conditions are verified then ... }
            SKIP_MCSYMBOL( [mendifsy] );        { ... we skip to the %endif }
            CND_POP;                            { end go out of this %if sequence. }
            src_fchcd := sch;                   { We restore the listing line flag. }
            if sy = mendifsy then INSYMBOL_SRC
            else SRC_ERROR( mdnam, 953, e_fatal )
          end
        end;

      melifsy, melsesy, mendifsy:               { Here only when one of previous cond.ition is set ... }
        begin
        sch := src_fchcd; src_fchcd := 'S';   { Flag skipped line for source listing, ... }
          SKIP_MCSYMBOL( [mendifsy] );          { ... skip all the source lines until the %endif keyword ... }
          INSYMBOL_SRC;
          src_fchcd := sch;                     { ... restore the current listing line flag ... }
          CND_POP                               { ... and go out of %if sequence. }
        end;

      midentsy:
        if sy_macsymb <> nil then
        begin
          MACRO_REPLACE( sy_macsymb );          { Start Macro interpretation }
          INSYMBOL_SRC
        end
        else
          SRC_ERROR_S( mdnam, 995, e_severe, sy_ident );        { Undefined Macro symbol }

      messagesy:
        begin
          INSYMBOl_SRC;
          if sy = lparen then INSYMBOL_SRC else SRC_ERROR( mdnam, 955, e_error );
          WRITEV( smsg, '*** Message ***' );
          repeat
            INSYMBOL_SRC;
            case sy of
              stringconst:
                WRITEV( smsg:false, ' ', sy_string );

              midentsy:
                begin
                  bmvl := true;
                  if sy_macsymb <> nil then with sy_macsymb^.value do
                    case mvl_typ of
                      mval_int: WRITEV( smsg:false, ' ', mvl_int:0 );
                      mval_str: WRITEV( smsg:false, ' ', mvl_str^ );
                    otherwise
                      bmvl := false
                    end
                  else bmvl := false;
                  if bmvl then WRITEV( smsg:false, ' NoVal for mac symb "', sy_ident.s:sy_ident.l, '"' );
                end;

              identsy:
                WRITEV( smsg:false, ' ', sy_ident.s:sy_ident.l );

              intconst:
                WRITEV( smsg:false, ' ', sy_ival );
              singleconst:
                WRITEV( smsg:false, ' ', sy_rval, 'F' );
              doubleconst:
                WRITEV( smsg:false, ' ', sy_rval, 'D' );

            otherwise
              SRC_ERROR( mdnam, 956, e_error )

            end;
            INSYMBOL_SRC
          until sy <> comma;
          LST_NEWLINE; LST_PUT_STRING( smsg ); LST_EOLN;
          if sy = rparen then INSYMBOL_SRC
          else SRC_ERROR( mdnam, 957, e_error );

        end;

      peofsy:
        begin
          ENDFILE_MSTATE;                       { If it is the last file => eofsy, else ready to get ... }
                                                { ... the next syntax unit in the parent source file. }
        end;

    otherwise
      SRC_ERROR( mdnam, 997, e_severe );
      INSYMBOL_SRC
    end;
    if cnlvl < 0 then
    begin
      SRC_ERROR( mdnam, 996, e_error );
      cnlvl := 0
    end
  end { while sy >= mdefinesy do }
end INSYMBOL;


[global]
procedure SKIP_SYMBOL( tosymbol: symbol;
                       errflg: boolean := false; rec_flg: boolean := false );
begin { * SKIP_SYMBOL }
  with sy_sym do
  while (sy <> tosymbol) and (sy <> eofsy) do
  begin
    INSYMBOL;
    case sy of
      casesy:
        if not rec_flg then
        begin
          SKIP_SYMBOL( endsy ); INSYMBOL
        end;
      recordsy:
        begin
          SKIP_SYMBOL( endsy, true ); INSYMBOL
        end;
      beginsy, loopsy:
        begin
          SKIP_SYMBOL( endsy ); INSYMBOL
        end;
      lparen:
        begin
          SKIP_SYMBOL( rparen ); INSYMBOL
        end;
      lbrack:
        begin
          SKIP_SYMBOL( rbrack ); INSYMBOL
        end;
      repeatsy:
        begin
          SKIP_SYMBOL( untilsy ); INSYMBOL
        end;
      endsy:
        exit if (tosymbol = elsesy) or errflg;
    otherwise
    end
  end
end SKIP_SYMBOL;


[global]
procedure SKIP_MSYMBOL( in_var smb: setsymb; rec_flg: boolean := false );
begin
  with sy_sym do
  while not ((sy in smb) or (sy = eofsy)) do
  begin
    INSYMBOL;
    case sy of
      casesy:
        if not rec_flg then
        begin
          SKIP_MSYMBOL( [endsy] ); INSYMBOL
        end;

      recordsy:
        begin
          SKIP_MSYMBOL( [endsy], true ); INSYMBOL
        end;

      beginsy, loopsy:
        begin
          SKIP_MSYMBOL( [endsy] ); INSYMBOL
        end;

      lparen:
        begin
          SKIP_MSYMBOL( [rparen] ); INSYMBOL
        end;

      lbrack:
        begin
          SKIP_MSYMBOL( [rbrack] ); INSYMBOL
        end;

      repeatsy:
        begin
          SKIP_MSYMBOL( [untilsy] ); INSYMBOL
        end;
    otherwise
    end
  end
end SKIP_MSYMBOL;



[global]
procedure INSY_INIT;
var
  ierr: integer;

begin
  { Initialize string, file and listing sub-systems }
  LST_G_INIT( pas_deflst, pas_page_head, ierr );

  if ierr = 0 then
  begin
    { Initialize the error message system }
    ERR_INIT( cpas_ini_dir||'cpas_cmp.err' );

    { Initialize source input system on the target system environment file }
    SRC_INIT( pas_prompt, trg_env_fspc );
    if cmp_macroexp then
      SRC_SET_FLAGS( [src_bmacroex] );

    if src_control = nil then
    begin
      error_result := e_fatal; emergency_stop := true;
      WRITELN( ' *** PCMP-CPAS-COMPILER ERROR: Cannot open the "', trg_env_fspc, '" initial file.' );
      WRITELN( '                               for the target system "', trg_sys_name, '"' )
    end
    else
    begin
      { We can set standard definition source listing mode for debug }
      { with src_control^ do  src_flags := src_flags + [src_blist]; }

      { Built the keyword tree }
      SET_KEYWORD_TREE;

      proc_seq    :=   0;                       { Initialize the procedure number }
      srf_free    := nil;
      srf_list    := nil;
      cnlvl       :=   0;                       { Initialize the cond. nesting level }


      { Init the standard identifier environment }
      IDE_INIT;

      { Initialize the Source file info for back-tracing }
      SET_SRCFILESPC( 1 );

      { Init sy_ch for INSYMBOL }
      sy_attrflg := false;
      sy_ch      :=   ' ';

      { Init label flag knowledge to disable state }
      sy_label_flag := false
    end
  end
  else
  begin
    error_result := e_fatal; emergency_stop := true;
    WRITELN( ' PAS CANNOT OPEN THE "', pas_deflst, '" OUTPUT MSG/LIST FILE.' )
  end
end INSY_INIT;

end PAS_INSYMBOL.
