//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers Software
//

//
// C/C++ Document integrator program V1.0
// 10-Feb-2022
//


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>



//
// Help text.
//

const char * HelpText[] = {
  "\n C/C++ Data Ressource Integrator V1.0 Help\n P. Wolfers Software - 10-Feb-2022\n\n",
  " Order format :\n\n",
  "     ResIntegr <Output_File> [Options ...] [Entry_Options ...] <Data_Entry> ... [Entry_Options ...] <Data_Entry> ...\n\n",
  " Each <Data_Entry> has the format <C/C++_Identifier>:<Data_File_Path> where the identifier refers\n",
  " to the name Data array and the <Data_file_Path> specify the data file used to fill this array.",
  " The <output_File> is a C/C++ file where each data block will be written.\n\n",
  "   For Exemple, the following command line :\n\n",
  "     ResIntegr MyData.c -S -C MyHouse:House_Image.png\n\n",
  " will create the C source file Mydata.c with the C data block declarations as below :\n\n\t",
  "static const int MyHouse_size = 26400;\n\tstatic const unsigned char MyHouse[] = {\n\n\t",
  "                                     . . . . . . .\n\n\t",
  "The hexadecimal values of all 26400 bytes of the House_Image.png file (for exemple : 0xFE, 0xA3, 0x05 ...)\n\n\t",
  "                                     . . . . . . .\n\n\t",
  "};\n\n\t",
  "Remark: the data size specified by the compound identifier <C/C++_Identifier>_size\n\t",
  "        is always the length in byte of the <Data_File> file.\n\n",
  " The Data_Entry Options are used to modify the Block data characteristics :\n\n\t",
  "-S             sets the static class option for the next data block entries and -s cancels -S effect.\n\n\t",
  "-C             sets the const class value option and -c cancels -C effect.\n\n\t",
  "-I             sets the block in as integer as first element of the data block (it is the default)\n\t",
  "               and -i cancels the -I effect.\n\n\t",
  "-a  or -A      sets the address mode generation table (void * array - 32 or 64 bits processor dependant).\n\n\t",
  "-1  or -B, -b  sets the table of byte (unsigned char array - 8 bits) generation.\n\n\t",
  "-2  or -W, -w  sets the table of word (unsigned short array - 16bits) generation.\n\n\t",
  "-4  or -L  -l  sets the table of long (unsigned int array - 32 bits) generation.\n\n\t",
  "-8  or -Q, -q  sets the table of long long (unsigned long long int array - 64 bits) generation.\n\n\t",
  "-tn  or -Tn    sets the total table size as a multiple of 1, 2, 4 or 8 for n = 1, 2, 4 or 8.\n\t",
  "               The <ident>_size value don't take in account the trailing bytes that can be append to the data block.\n\n",
  " The Command Line Options include all Entry Options and :\n\n\t",
  "-f <EFile>\n\t",
  "-F <EFile>     Reads the specified <EFile> to construct there Data Blocks as specified in each file line.\n\t",
  "               The syntax of each line will be :\n\t",
  "               [<Entry_Option> ... ] <Data_Entry> [Entry_Options ...] <Data_Entry> ...\n\n\t",
  "-h  or -H      Display this help text.\n\n",
  " The entry options can be collected in an option group : -SC2 is equivalent to -S -C -2. The effects of Data_Entry\n",
  " block options are retained until other options change them (exemple : -c will cancel the effect of -C for the next\n",
  " Data_Entry specifications, then a following Data_Entry will not have the \"const\" attribut)\n",
  " This memory effect is transmitted to the Data_Entry(ies) specified in the files included by -F (or -f) option,\n",
  " but the Entry_Option changes in the file are lost when the file is closed.\n\n",
  " In a file open by the option -F (or -f), each part of <Data_Entry> must be on the same line. You cannot separe\n", " the <C/C++_Identifier>, the separator ':' and the <Data_file_Path> by a line change.\n",
  " In these indirect files, any line begining by the '#' character is ignored (as comment line).\n\n",
  NULL
};


//
// Define the Data Block class flags.
//
#define TYPE_IDM      0xF
#define STATIC_F    0x100
#define CONSTA_F    0x200
#define INSIZE_F    0x400



// Define the Data Block characters for each data file.
typedef struct FileRef_rec * FileRefPtr;

typedef struct FileRef_rec {
  FileRefPtr   next; // Pointer to the next data file.
  char       * fspc, // The data file path.
             * type, // The C/C++ type to use for Dta block element.
             * name; // The C/C++ data block identifier.
  int          alns, // the alignement for total size.
               flgs; // The allocation class flags.
} FilerefRec;


typedef struct {
  unsigned char typ_siz;
  char      typ_ide[24];
} TypDescr;


static const TypDescr SizeTab[] = {
  { sizeof( char ),          "unsigned char" },         // 0: Byte of 8 bits,
  { sizeof( short ),         "unsigned short" },        // 1: Should be 16 bits Word,
  { sizeof( int ),           "unsigned int" },          // 2: Should be 32 bits Word,
  { sizeof( long int ),      "unsigned long int" },     // 3: Should be 32 or 64 bits Word,
  { sizeof( long long int ), "unsigned long long int" },// 4: Should be 64 bits word.
  { sizeof( void * ), "void *" },                       // 5: Shouls be as void* word (32 or 64 bits.
  { 0, "" }
};


static int    ElemSize    =          1, // Default is the byte element size ...
              ElemIdex    =          0, // ... the first type ident ...
              ElemFlag    =   INSIZE_F, // Element flags ...
              TabAlign    =          0; // ... without alignement.

static FileRefPtr FrstRef = NULL, // First and last data file reference ...
                  LastRef = NULL; // ... to form a queue of data files.

static char * DataCFile   = NULL, // Resulting C/C++ data file.
            * ElemType    = NULL; // The selected Data block elelement C/C++ type identifier.

static FILE               * fout; // Output file (some Data Blocks in C/C++.


//
// The used routines
//



int  TypeSelect( int size )
// Routine to select the first defined type with the required type size and ...
// ... return it in typide argument, and set the required total size alignement in aln. 
//
{
  int   ii = 0;

  while (SizeTab[ii].typ_ide) {
    if (SizeTab[ii].typ_siz == size) {
      ElemType = strdup( SizeTab[ii].typ_ide );
      TabAlign = size - 1;
      return ii;
    }
    ii++;
  }
  fprintf( stderr, " Ressource Integrator Error:\n\t%s\n",
           "Unknown defined C/C++ type for the specified DataBlock element." ); 
  return -1;
} // int TypeSelect( int size ).



char * StrExtr( const char * str, int sf, int sz )
{ char     * p;
  int    i,  j;

  p = new char[sz - sf + 1];
  j = 0;
  for(i = sf; i < sz; i++) p[j++] = str[i];
  p[j] = 0;
  return p;
} // char * StrExtr( const char str, int sf, int sz ).



int  GetFileEntry( const char * str, int ij )
//
//
{
  const char        * id, * pa;
  int   nf, idf, idz, paf, paz;
  char                      ch;
  FileRefPtr                 p;

  ch = str[ij];
  while (ch&&(ch <= ' ')) ch = str[++ij];       // Skip any space or control character.
  // Get the C/C++ identifier character.
  nf  =  0;
  idf = ij;
  while (((ch >= 'A')&&(ch <= 'Z'))||
         ((ch >= 'a')&&(ch <= 'z'))||(ch == '_')||
         (nf&&((ch >= '0')&&(ch <= '9')))) { ch = str[++ij]; nf++; }
  idz = ij;                                     // Keep the length of identifier.
  while (ch&&(ch <= ' ')) ch = str[++ij];       // Skip any space or control character.
  if (ch != ':') ij = -1;                       // No colon separatot => Syntaxe error. 
  else {
    ch = str[++ij];                                       // Skip to data file path characters.
    while (ch&&(ch <= ' ')) ch = str[++ij];     // Skip any space or control character.
    paf = ij;
    while (ch > ' ') ch = str[++ij];            // Size the file path and ...
    paz = ij;                                   // ... keep its end position.
  }

  if ((ij > 0)&&(paz > paf)&&(idz > idf)) {
    // Now we can create the Data Block entry.
    p = new FilerefRec;
    p->next = NULL;                     // Init the nextr request pointer.
    p->fspc = StrExtr( str, paf, paz ); // Set the Data File path.
    p->type = strdup( ElemType );       // Set the C/C++ Element Type identifier.
    p->name = StrExtr( str, idf, idz ); // Set the C/C++ Data Block identifier.
    p->alns = TabAlign;                 // Set the requested Alignement for total size.
    p->flgs = ElemFlag + ElemIdex;      // Set the Data Block Allocation class.
    // ... and insert it in the Request Queue.
    if (LastRef) LastRef->next = p;
            else FrstRef = p;
    LastRef = p;
  } else ij = -1;
  if (ij < 0) {
    fprintf( stderr, " Ressource Integrator Error: illegal Data Block entry :\n\t\"%*s\".\n", 
                     ij-idf, str+idf );
  }
  return ij;
} // int  GetFileEntry( const char * str, int idx ).



int  SetEntryOption( const char * str, int ij )
//
// Manage one group of entry option in the string str.
// ij is the index of first not readden character in str.
// The returned value is the new chracter index in str on success ...
// ... and -1 for syntax error.
//
{
  int al  = 0;
  char     ch;

  while ((ch = str[++ij]) > ' ') {
    switch (ch) {
      case 's':
        ElemFlag &= ~STATIC_F;
      break;
      case 'S':
        ElemFlag |=  STATIC_F;
      break;

      case 'c':
        ElemFlag &= ~CONSTA_F;
      break;
      case 'C':
        ElemFlag |=  CONSTA_F;
      break;

      case 'i': // Disable Block data size in block.
        ElemFlag &= ~INSIZE_F;
      break;

      case 'I': // Enable Block data size in block.
        ElemFlag |=  INSIZE_F;
      break;

      case 'A': case 'a':
        ElemSize = sizeof( void * );
        ElemIdex = 5;
        ElemType = strdup( SizeTab[ElemIdex].typ_ide );
        TabAlign = ElemSize - 1;
      break;

      case '1': case 'B': case 'b':
        ElemSize = 1; ElemIdex = TypeSelect( ElemSize );
      break;

      case '2': case 'W': case 'w':
        ElemSize = 2; ElemIdex = TypeSelect( ElemSize );
      break;

      case '4': case 'L': case 'l':
        ElemSize = 4; ElemIdex = TypeSelect( ElemSize );
      break;

      case '8': case 'Q': case 'q':
        ElemSize = 8; ElemIdex = TypeSelect( ElemSize );
      break;

      case 'T': case 't':
        switch (str[++ij]) {
          case '1': al = 0; break;
          case '2': al = 1; break;
          case '3':
          case '4': al = 3; break;
          case '5':
          case '6':
          case '7':
          case '8': al = 7; break;
        default:
          al = 0; // Ignored size aligne option.
        }
        // Take in only when the alignement is more large ...
        // ... than the native type alignement.
        if (al > TabAlign) TabAlign = al;
      break;

    default:
      fprintf( stderr, " Ressource Integrator Error: Unkown Entry Option \"-%c\".\n", str[0] );
      ij = -1;
    }
  }
  return ij;
} // SetEntryOption( char * str ).



int  SetupFList( char * fname )
{
  FILE            * fp;
  char   * line = NULL;
  char           * str;
  char              ch;
  size_t    len =    0;
  ssize_t        nread;
  int    bst, idx, ier;
  int       SzSv, IdSv,
            FlSv, AlSv;

  SzSv = ElemSize; // Save current element options.
  IdSv = ElemIdex;
  FlSv = ElemFlag;
  AlSv = TabAlign;

  ier = 0;

  if (fp = fopen( fname, "r")) {
    // Read each line of the file.
    while ((nread = getline( &line, &len, fp )) != -1) {
      // Remove any control or space character at the end of line.
      if (line[0] != '#') { // Ignore any comment line.
        while (line[nread-1] <= ' ') {
          line[nread-1] = 0; nread--;
        }
        if (nread) {
          idx = -1;
          do {
            // Skip any space or control chraracters.
            do ch = line[++idx]; while (ch <= ' ');
            if (ch == '-') idx = SetEntryOption( line, idx );      // Manage one group of entry options.
                      else idx = GetFileEntry( line, idx );        // Manage one Data Block entry.
            ch = line[idx];
          } while (ch >= ' ');
        }
      }
    } while (!feof( fp ));
    fclose( fp );
  } else {
    fprintf( stderr, " Ressource Integrator Error: Cannot open file :\n\t%s\n\tError : %s.\n",
             fname, strerror( errno ) );
    ier = -1;
  }

  ElemSize = SzSv; // Restore current element options.
  ElemIdex = IdSv;
  ElemFlag = FlSv;
  TabAlign = AlSv;

  return ier;
} // int  SetupFList( char * fname ).


void ShowHelp()
{
  const char    * line;
  int           ii = 0;

  while (line = HelpText[ii++]) printf( line );
  exit( 0 );
} // void ShowHelp().



int  SetArguments( int argc, char ** argv )
// Manage all program argument to create the Data Block Request Queue.
//
{
  int nf   = 0,
      ii   = 0,
      ij   = 0,
      flg  = 0,
      hlf  = 0,
      aln  = 0,
      ier  = 0;
  char * ident,
       *  fspc,
       *   arg;
  FileRefPtr p;

  ElemIdex  =   TypeSelect( ElemSize ); // Select the default element type (unsigned char).
  while ((++ii < argc)&&!hlf) {
    arg = argv[ii];
    if (arg&&arg[0]) {
      if (arg[0] == '-') {
        switch (arg[1]) {
          case 'f':
          case 'F': // Indirect file option.
            ier = SetupFList( argv[++ii] );
            if (ier < 0) fprintf( stderr, " Stop on error in file :\n\t\"%s\"", argv[ii] );
          break;

          case 'h':
          case 'H': // Help output option.
            hlf = 1;
          break;

        default:
          ier = SetEntryOption( arg, 0 );
        }
        if (ElemIdex < 0) { ElemIdex = 0; ier = -1; }
      } else {
        if (nf) {
          ier = GetFileEntry( arg, 0 );
          if (ier < 0) exit( 2 );          
        } else {
          DataCFile = strdup( arg );
        }
        nf++;
      }
    }
    if (ier < 0) break;
    if (hlf) {
      ShowHelp();
      return 0;
    }
  }
  if (!(DataCFile&&FrstRef)) {
    fprintf( stderr, " Ressource Integrator Error: no target C/C++ file or Ressource list empty\n" );
    return -1;
  }
  return 0;
} // int SetArguments( int argc, char ** argv ).





int  CreateDataBlock( FileRefPtr p )
{
  typedef union {
    void *                 ad[8/sizeof( void *)];
    unsigned char          ub[8];
    unsigned short int     uw[8/sizeof( short int )];
    unsigned int           ui[8/sizeof( int )];
    unsigned long int      ul[8/sizeof( long int )];
    unsigned long long int uq[1];
  } EquType;

  static const int intsz = sizeof( int );

  FILE                    * finp;
  unsigned int  bsiz, fsiz, wsiz;
  int     tidx, ncol, esiz, idby,
                ierr, nclm, ncnt;
  unsigned char         * buffer,
                        *   data;
  struct stat              entbf;
  EquType                   equv;

  ierr = stat( p->fspc, &entbf );               // Get information from file inode ...
  if (!ierr) {                                  // ... to get the file size.
    fsiz = entbf.st_size;                       //
    if (finp = fopen( p->fspc, "r" )) {
      bsiz = (fsiz+p->alns)&~(p->alns);         // Compute the aligned buffer size.
      buffer = new unsigned char[bsiz+intsz];   // Allocate the buffer including the size of source file (unsigned integer).
      data = buffer;
      // Insert the data block size at the begin of block.
      equv.ui[0] = fsiz;
      for(idby=0; idby<intsz; idby++) *(data++) = equv.ub[idby];
      // Read the whole of data file ...
      wsiz = fread( data, sizeof( char ), bsiz, finp );
      fclose( finp );                           // ... and close it.
      if (wsiz != fsiz) {                       // we must read exactly the size of file.
        ierr = -1;                              // else we have a read error.
        fprintf( stderr, " Ressource Integrator Error:  Read error on file \n\t\"%s\"\n\t%s\n",
                         p->fspc, strerror( errno ) );
      } else {
        // When the read is ok, we complete for required size alignement.
        while (wsiz < bsiz) buffer[wsiz++] = 0;
        ierr = 0;
      }
    } else ierr = -1;
  }
  if (ierr) {
    fprintf( stderr, " Ressource Integrator Error:  Cannot create the file\n\t\"%s\"\n\t%s\n",
                     p->fspc, strerror( errno ) );
    return -1;
  }

  wsiz = 0;                     // Initialize the data byte count.
  ncol = 0;                     // Initialize the column count.
  tidx = TYPE_IDM&p->flgs;      // Get the C/C++ type index in SizeTab[].
  esiz = SizeTab[tidx].typ_siz; // Get the table element size.

  switch (esiz) {
    case 1: nclm = 16; break;
    case 2: nclm =  8; break;
    case 4: nclm =  8; break;
    case 8: nclm =  4; break;
  default:
    nclm = 4;
  }

  fprintf( fout, "/* Data block built from the file \"%s\" */\n", p->fspc );

  if (p->flgs&INSIZE_F) {       // When the source file size is included in the data block.
    data = buffer;      // Include the block size in the data  ...
    fsiz += intsz;      // ... and set the correct size to write.
  } else {
    // Create a specific unsigned integer for data data block size.
    if (p->flgs&STATIC_F) fprintf( fout, "static " );
    fprintf( fout, "const int %s_size = %d;\n", p->name, fsiz );
  }

  if (p->flgs&STATIC_F) fprintf( fout, "static " );
  if (p->flgs&CONSTA_F) fprintf( fout, "const " );

  fprintf( fout, "%s %s[] = {\n", p->type, p->name );

  while (wsiz < fsiz) {         // Loop to write the Data block.
    if (!ncol) fprintf( fout, "  " );
          else fprintf( fout, ", " );
    if (esiz == 1)
      fprintf( fout, "0x%02X", data[wsiz++] );
    else {
      for(idby = 0; idby < esiz; idby++)
        equv.ub[idby] = data[wsiz++];
      switch (esiz) {
        case 2: // unsigned short word (16 bits).
          fprintf( fout, "0x%04X", equv.uw[0] );
        break;

        case 4: // unsigned long word (32 bits).
          fprintf( fout, "0x%08X", equv.ul[0] );
        break;

        case 8: // unsigned quad word (64 bits).
          fprintf( fout, "0x%016llX", equv.uq[0] );
        break;

        default: ;
      }
    }
    ncol++;
    if (wsiz >= fsiz) {
      fprintf( fout, "\n" );
      ncol = 0;
    } else {
      if (ncol >= nclm) { fprintf( fout, ",\n" ); ncol = 0; }
    }
  }
  fprintf( fout, "};\n\n" );
  delete[] buffer;
  return 0;
} // void CreateDataBlock( FileRefPtr p ).



int main( int argc, char ** argv )
{
  FileRefPtr p;
  int  ier = 0;

  ier = SetArguments( argc, argv );
  if (!ier) {
    if (fout = fopen( DataCFile, "w" )) {
      fprintf( fout, "/* Block Data file Created by Ressource Integrator program V1.0 */\n\n" );
      p = FrstRef;
      while (p) {
        if (CreateDataBlock( p )) exit( 2 );
        p = p->next;
      }
      fclose( fout );
    } else {
      fprintf( stderr, " Ressource Integrator Error:  Cannot create the target file\n\t\"%s\"\n\t%s",
               DataCFile, strerror( errno ) );
      exit( 2 );
    }
  }

  return ier;
} // Main program.



//
// end of <PW-Id>.
//

