//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module environement to Allocate and store a large number of user Record of various sizes.
//
//

//
//
// General notes for RecStk and Any_RecStk<User_Record_type> and
//               for Re8Stk and Any_Re8Stk<User_Record_type> modules users.
//
// These two modules are used by these inclusion of these C++ statments :
//
//      #include "Service_RECSTK.h"
// or
//      #include "Service_Re8Stk.h"
//
// These instructions may change depending on the location of the header files.
//
// These class elements are used to manage stacks of different types of user records.
//      In the same stack, different records can have different sizes.
//
// For example, if the user defines the following type:
//
//      typedef struct { SInt32 u, v; float r; char name[256]; } Rec; // To use with RecSTK or Any_RecStk.
// or
//      typedef struct { SInt16 u, v; char name[128]; } Rec;          // To use with Re8Stk or Any_Re8Stk.
//
// The differences between the two types of record stacks are:
//
// With RecStk/ANy_RecStk class you can include the base objects (SInt32/UInt32,SInt16/UInt16,
// SInt_8/UInt_8), float reals or SChar/UChar/char characters), i.e. all types with a maximum size
// of 4 bytes without requiring 8-bytes alignment.
//      If the pointers are 4 bytes long, they can be there too, but not on 64-bits systems (8 bytes).
// Types that are not in this list can be present in the record but cannot be used without first being
// copied into normal variables because their alignment is not guaranteed.
//      For the class Re8Stk/Any_Re8Stk the list of base type is restricted to (SInt16/UInt16,
// SInt_8/UInt_8, SChar/UChar/char) because alignment can only be guaranteed for these types.
//      Another difference is the size limits between the two kinds of item stacks. The
// RecSTK/Any_RecStk<type> can accept large size recordings limited only by the size of the internal
// memory areas (private Area type of a capacity of 16384 bytes = 4096 SInt32). Knowing that each
// record costs for 2 UInt16 this allows, for example, to store 256 records of 60 bytes (15 SInt32) each.
// This corresponds to a storage efficiency of ~93% if we take into account the variables specific to
// each Area zone and that specific to the RecStk/Any_RecStk class.
//      With the Re8Stk/Any_Re8Stk<type> record stacks, the size of the records is limited by the use of
// UInt_8 in the header of each record so the maximum size of a record cannot be greater than 508 bytes.
//      Base addresses and stack record sizes are always aligned to multiples of 4 (UInt32) for RecStk and
// 2(Uint16) for Re8Stk.
//
// The Any_RecStk and Any_Re8Stk templates are only designed to simplify the use of the RecStk and Re8Stk
// classes respectively. In fact, the direct use of RecStk and Re8Stk classes actually requires type
// conversions to use most of the Members of these classes.
//
// To declare a stack of records you can use one of the following four forms (if your Record type is Re):
//
// Any_RecStk<Re> * p = new ANy_RecStk<Re> MyStack;   // Dynamic allocation.
//
// Any_RecStk<Re> MyStack;                            // Automatic allocation.
//
// static Any_RecStk<Re> MyStack;                     // Static allocation (space allocated by compiler.) or ...
//
// Any_RecStk<Re> MyStack;                            // ... Global variable also alloocated by compiler.
//
//
// Calls to the public methods of these classes will take two forms (r is a record of type Re):
//
//      Re * ar = MyStack->Push( r, 25 );             // Dynamic form or ...
//
//      Re * ar = Mystack.Push( r, 25 );              // Automatic/Static/Global/External form.
//
// In these two cases, ar will be the address of the Record in the stack. The size can be the value of a ..
// variable or the result of a computing.
//
//      The call of Member int MyStack.Size() gives the size in bytes of the Record on the top of stack
// and the Member Re * MyStack.Get() will return the address of the stack record. The Pop Member will
// copy the record at the top of the stack into the record whose address is provided. The user is expected
// to provide a record of sufficient size (which can be obtained by calling the Size() member).
//
// The "void Remove( Re * r = 0 )" member can be used to remove one (if no address is provided) or multiple
// items from the stack, the last being the one whose address is provided. If the latter is not found,
// the stack will be completely empty.
// The specific call of "Re * RecNb( int )" can be used to get to the address of any member in the stack
// if the argument is >=0 it will be the top of the stack as with Get, if it is negative, it will allow
// access to all the elements of the stack. It should however be pointed out that this call does a downward
// scan of the stack which may not be extremely fast.
//
//      The Member Record * NextUp() and Record * NextDown() of these class allow to change of record between
// the top of Stack to its Bottom (with) NextDown() and in revers direction with NextUp(). After the last Push(),
// execute RecNb(-2) has the same effect which two NextDown() call, and after two NextUp() we return to the top
// of stack. The members method Size( 1 ) and Get( 1 ) allows to known the size or to get current record after
// any operations with the three meber methods RecNb, NextUp and NextDown.
//
// The form call are :
//
//      Any_RecStk<Ty>();        // The Constructor.
//      ~Any_RecStk();           // The Destructor.
//
//      void Zero();             // To Free all Area,and Reset to Init state the class (stack empty).
//
//      Ty * Push( Ty * record, int size );     // To push a record sz bytes in the stack.
//      Ty * Get( int fl = 0 );                 // if (!fl) Get the top stack record (), else ...
//                                              // ... Get the record selected by the previous call of ...
//                                              // ... RecNb, NextUp or NextDown.
//      int Size( int fl = 0 );                 // Get the size of top or Selected Record as flagged ...
//                                              // ... by fl. fl 0/1 for Top/Selected.
//                                              // When RecNb, NextUP and NextDown are not used, the ...
//                                              // ... The selected record is the top record.
//      Ty * Remove( int n = 1 );               // Delete n Records from the stack and return the Address ...
//                                              // ... of new top of stack Record.
//      Ty * Remove( Ty * record = 0 );         // Like the previous call, we delete the Records from the ...
//                                              // ... stack, the last one being the one provided as ...
//                                              // ... an argument. Thes two calls.
//                                              //
//      int Pop( Ty * record );                 // Remove/delete the top record after give its copy in ...
//                                              // ... the user specified record. Warning: the user ...
//                                              // ... record must be enough large to receive the record data.
//
//      Ty * RecNb( int id );                   // Select and get address of id'th record in the stack.
//      Ty * NextUp();                          // Select and get address of record just above the ...
//                                              // ... previously selected one.
//      Ty * NextDown();                        // As NextUp except that the selected record is just below ...
//                                              // ... the previously one.
//                                              // As a side effect, calls to the Push and Remove member ...
//                                              // ... methods have the effect of selecting the record ...
//                                              // ... at the top of the stack.
//
//      Member calls of the Re8Stk and Any_Re8Stk classes are identical to those of the
//      RecStk and Any_RecStk classes.
//
//

#ifndef SERVICE_RECSTK_H
#define SERVICE_RECSTK_H


#include "Service_ENV.h"


#define AREA_SIZE_B 16384
#define AREA_SIZE_W  8192
#define AREA_SIZE_L  4096

// Define the record to store.


class RecStk {
public:
    // Define the record to store.
    typedef union {
        float      fl[128];
        SInt32     sl[128];
        UInt32     ul[128];
        SInt16     sw[256];
        UInt16     uw[256];
        SInt_8     sb[512];
        UInt_8     ub[512];
        char       ch[512];
    } Record;

private:
    typedef struct AREA {
        AREA            * nxt_, // Pointer to next area.
                        * prv_; // Pointer to previous area.
        UInt32            thd_, // Position of top HeadRec.
                          top_; // Area used size.
        SInt32 tab_[AREA_SIZE_L];// Area memory.
    } Area;

    typedef struct {
        UInt16   link_, // The Index distance with previous RecHeader ...
                 size_; // ... and current (Record+Header) size (in Uint32 units)
    } Header;

    typedef union {
        Header   he_;
        UInt32   ll_;   // Is here just to force alignement on 32 bits words.
    } HeadRec;

    Area                 area_; // The internal area (always allocated) is the first ...
    Area       * fst_ = &area_, // ... in the used Area and cannot be free except when ...
               * lst_ = &area_, // ... the Varobj object is destroye.
               * cur_ = &area_, // Pointer to the current Area for allocation.
               * own_ = &area_; // Pointer to last area  wher is find a Record (used by
                                // RecNb, NextUp and NextDown).

    UInt16       fre_ =      0; // Pointer to the last found element (used by RecNb, NextUp
                                // and NextDown)

    void     URecCopy( Record *, Record *, int );

    Area *   NewArea();         // To allocate a new Area.
    void     DelArea( Area * ); // To free an Area.
    void     Init();            // Initialize the class.

public:
    void Zero();

    RecStk() { Init(); }
    ~RecStk() { Zero(); }

    Record * Push( Record *, int );
    Record * Get( int = 0 );

    int      Size( int = 0 );

    Record * Remove( Record * = 0 );
    Record * Remove( int = 1 );

    int      Pop( Record * );

    Record * RecNb( int );
    Record * NextUp();
    Record * NextDown();

    void    Statistic(int &, int &, int & );
};



//
// Small template class to use with various type of record (at varible sizes).
//


template <class Ty>
class Any_RecStk : public RecStk {
public:
    Any_RecStk <Ty> () : RecStk() { }
    Ty * Push( Ty * r, int sz ) { return (Ty *) RecStk::Push( (Record*) &r, sz ); }
    Ty * Get() { return (Ty *) RecStk::Get(); }
    Ty * Remove( int n = 0 ) { return (Ty *) RecStk::Remove( n ); }
    Ty * Remove( Ty * r = 0 ) { return (Ty *) RecStk::Remove( (Record*)r ); }
    int Pop( Ty * r ) { return RecStk::Pop( (Record*) r ); }
    Ty * RecNb( int n ) { return (Ty *)RecNb( n ); }
    Ty * NextUp() { return (Ty *)NextUp(); }
    Ty * NextDown() { return (Ty *)NextDown(); }
};


#endif


//
// end of <PW-Id>.
//
