//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers Software
//

//
// DiaViewer Ranking context and map module.
//
//

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>

#include <FL/Fl.H>
#include <FL/fl_ask.H>
#include <FL/fl_utf8.h>
#include <FL/fl_message.H>


#include "DiaViewer_RKGCN.h"

#include "Text_Display.h"



//
RkgTabCnt::RkgTabCnt( DiaViewer_GL * win )
{
    Frs_Flm = Flm_Siz = Flm_Esz = Sli_Siz = Sli_Tsz = Sta_Tli = 0;
    SliTable = NULL; FlmTable = NULL;
    FlmIdeNb = FlmRowNb = FlmRowSz = NULL;
    DspTxt = NULL;
    SrcWin = win;
    SrcVol = win->Curr_Vol;
    SrcDir = SrcVol->VRoot();
    Active = TmpMap = 0;
} // RkgTabCnt::RkgTabCnt( DiaViewer_GL * win ).


RkgTabCnt::~RkgTabCnt()
{
    if (DspTxt) {
        rktb_wid = DspTxt->DspW();
        rktb_hig = DspTxt->DspH();
//printf( " map size %d , %d\n", rktb_wid, rktb_hig );
    }
    Clear();
    SrcWin = NULL; SrcVol = NULL; SrcDir = NULL;
} // RkgTabCnt::~RkgTabCnt().



void RkgTabCnt::Clear()
{
    Frs_Flm = Flm_Siz = Flm_Esz = Sli_Siz = Sli_Tsz = Sta_Tli = 0;
    if (SliTable) { delete[] SliTable; SliTable = NULL; }
    if (FlmTable) { delete[] FlmTable; FlmTable = NULL; }
    if (FlmIdeNb) { delete[] FlmIdeNb; FlmIdeNb = NULL; }
    if (FlmRowNb) { delete[] FlmRowNb; FlmRowNb = NULL; }
    if (FlmRowSz) { delete[] FlmRowSz; FlmRowSz = NULL; }
} // void RkgTabCnt::Clear().




int  RkgTabCnt::FlmSliCount_( DirEntry * dir )
// Recursive function to compute the total number of ...
// ... numbered slides without the Lock Ranking flag set.
//
{
    int    nd   =  dir->NDir(),
           ns   =  dir->NSli(),
           efs  =            0,
                flid,     slid;
    SliEntry             * sli;

    for (int i=1; i<=nd; i++)
        efs += FlmSliCount_( dir->SbDir( i ) );

    for (int i=1; i<=ns; i++) {
        sli = dir->Slide( i );
        flid = sli->Film(); slid = sli->Numb();
        if (flid&&slid&&(!sli->FlgTst( Context_LckRank ))) {
            // Set Frs_Flm as the first ranking enable film number, and ...
            // ... Flm_Siz as the related last film number. Set also, ...
            // ... Sli_Siz as the maximum slides index in any ranking valid film ...
            // ... or partial film (in an incremental mode). The previously loaded slides ...
            // ... in the new volume are indicated by the Context_LckRank flag.
            if (!Frs_Flm) Frs_Flm = flid;
                     else if (Frs_Flm > flid) Frs_Flm = flid;
            if (Flm_Siz < flid) Flm_Siz = flid;
            if (Sli_Siz < slid) Sli_Siz = slid;
            efs++;
        }
    }
    return efs;
} // int  RkgTabCnt::FlmSliCount( DirEntry * dir ).



int  RkgTabCnt::FlmSliCount()
{
    int esli;

    Frs_Flm = Flm_Siz = Sli_Siz = 0;
    esli = FlmSliCount_( SrcDir );      // Get the sizes for table to build.
    Flm_Siz = Flm_Siz - Frs_Flm + 1;    // Now Flm_Siz is the number of film used for ranking.
    return esli;
} // int  RkgTabCnt::FlmSliCount().



void RkgTabCnt::CountFillFilm( DirEntry * dir )
//
// Recursive routine to set the film index table.
//
{
    int nd = dir->NDir(),
        ns = dir->NSli(),
                     cnf;
    SliEntry       * sli;

    for(int i=1; i<=nd; i++) CountFillFilm( dir->SbDir( i ) );

    for(int i=1; i<=ns; i++) {
        sli = dir->Slide( i );
        cnf = sli->Film();
        if (cnf&&sli->Numb()&&(!sli->FlgTst(Context_LckRank))) FlmIdeNb[cnf-1]  = cnf;
    }
} // void RkgTabCnt::CountFillFilm( DirEntry * dir ).







void RkgTabCnt::RequiredSliEval( int nsli, int &flmdg, int &slidg )
//
// Check the fit of ranking parameters with the source slide volume.
//
// nsli is the total possible number of slides to keep.
// flmdg is the number of figures for film #.
// slidg is the number of figures for slide #.
//
{
    int     qflm,  qsli,  qusl,
            nfld,  nsld,  esli,
                   nequ,   res;

    nfld = flmdg; qflm = 10;    // Compute the total capacity for the number of films.
    while (--nfld) qflm *= 10;
    qflm--;                     // A zero film number is not allowed.

    nsld = slidg; qsli = 10;    // Compute the total capacity for the number of slides/film.
    while (--nsld) qsli *= 10;
    qsli--;                     // A zero slide number is not allowed.

    // Compute the effective number of ranked slides in the source volume.
    esli = FlmSliCount();

    // When the total number of slide in the source volume is greater than ...
    // ... the global capacity of films and slides/film, it is better to ...
    // ... change the figures numbers flmdg and/or slidg.
    // In this case, we suggest to increase the films number capacity.
    // When the number of effective numbered of slides is less than this capacity, ...
    // ... we modify the film capacity and stop the ranking process for agreement.
    //
    nequ = 10;
    qusl = qsli*qflm;
    if (qusl < nsli) {  // The posiible slide number is too large =>warning or error.
        qflm = (esli + qsli - 1)/qsli;          // Now qflm is the minimum capacity and we ...
        res = 1;
        while (nequ <= qflm) {nequ*=10;res++;}  // ... return a new value (to flag warning).
    } else res = 0;

    // Send diagnostic to user.
    if (res) {
        if (qusl < esli) {
            fl_alert(   "Error :\n\tFor a total of %d slides with %d numbered slides,\n\t%s\n\t%s%d%s",
                        nsli, esli,
                        "you must change the film or slide figures number(s) as (for example) :",
                        "set figures film number at ", res, " to keep all source volume slides." );
        } else {
            fl_message( "Warning :\n\tFor a total number of %d slides, you should change\n\t%s%d.\n\t%s",
                        nsli, "(for example) the film figures number at ", res,
                        "The actual setting do not authorize to keep all possible source volume slide." );
        }
    }
} // void RkgTabCnt::RequiredSliEval( int nsli, int &flmdg, int &slidg ).




void RkgTabCnt::MouseCallBack( int bt, int row, int col )
{
    int  flmnb = 0,
         slinb = 0;

//printf( " Position = %d, %d\n", row, col );
    if ((row >= 0) &&(row < Flm_Siz)) {
        flmnb = FlmRowNb[row];
        slinb = (col - 12 + 3)/3;
//printf( " f, s = %d, %d\n", flmnb, slinb );
        if ((flmnb > 0)&&(slinb > 0)&&(slinb <= FlmRowSz[row])) {
            SrcWin->Search_Init( flmnb, slinb );
            SrcWin->Search_Process( 0 );
        }
        else flmnb = slinb = 0;
    }
} // void RkgTabCnt::MouseCallBack( int bt, int row, int col )




static void MouseCallBack_fnc( int bt, int row, int col, void * cntx )
{
    ((RkgTabCnt*)cntx)->MouseCallBack( bt, row, col );
} // static void MouseCallBack_fnc( int, bt, int row, int col, void * cntx ).



void RkgTabCnt::ExitCallBack()
{
    Active = 0;
    SrcVol->FlgClr( Context_VRkMap );
    if (TmpMap) {
        TextExit();
        SrcWin->SetRkgMap( NULL );
        delete this;
    }
} // void RkgTabCnt::ExitCallBack().




static void ExitCallBack_fnc( void * cntx )
{
    ((RkgTabCnt*)cntx)->ExitCallBack();
} // static void ExitCallBack_fnc( void * cntx ).



int  RkgTabCnt::AnalyseFlmSli( DirEntry * dir )
//
// Routine to create a Chart/Map of defined films and slides numbers.
//
{
    int nd = dir->NDir(),
        ns = dir->NSli();
    int  cnf,  cns,  ner;
    SliEntry       * sli;

    ner = 0;
    for(int i=1; i<=nd; i++) ner += AnalyseFlmSli( dir->SbDir( i ) );

    for(int i=1; i<=ns; i++) {
        sli = dir->Slide( i );
        cnf = sli->Film(); cns = sli->Numb();
        if (!(sli->FlgTst( Context_LckRank ))&&
            cnf&&(cnf <= Flm_Siz)&&(cns <= Sli_Siz))
        {
            if (FlmTable[cnf-1][cns-1]) {
                FlmTable[cnf-1][cns-1] = (SliEntry*)8;
                ner++;
            } else FlmTable[cnf-1][cns-1] = sli;
        }
    }
    return ner;
} // int  RkgTabCnt::AnalyseFlmSli( DirEntry * dir ).



int  RkgTabCnt::FlmSliChart( int tmpflg )
// To Create a table of films and slide numbers.
// Routine limited to 64 slide/film(> common length of slide silver-film),
// and 64 films.
//
// tmpflg  true/1: for direct call from DiaViewer Menu (temporary mode).
//        false/0: Call during the validation ranking process.
//
{
    int   esli,   cflm,    lflm,  nerr,
                wlrg,    nrow,    ncol,
          scr_wmax,   scr_wmin,   widx;
    SliEntry                      ** p;
    char                   txtbuf[256],
                           txtitle[40];
//  Text_Display              * DspTxt;

    if (!SrcVol) return 0;

    scr_wmax = (screen_w*8)/10,  scr_wmin = 550; //screen_w/5
/// scr_hmax = (screen_h*8)/10, scr_hmin = screen_h/5;

    // Free all previous Ranking/Sorting slides table.
    if (SliTable) { delete[] SliTable; SliTable = NULL; }
    if (FlmTable) { delete[] FlmTable; FlmTable = NULL; }
    if (FlmIdeNb) { delete[] FlmIdeNb; FlmIdeNb = NULL; }
    if (FlmRowNb) { delete[] FlmRowNb; FlmRowNb = NULL; }
    if (FlmRowSz) { delete[] FlmRowSz; FlmRowSz = NULL; }

    esli = FlmSliCount();               // Get the sizes (in slides) for table to build.

    FlmIdeNb = new UInt16[Flm_Siz];
    for (int i = 0; i < Flm_Siz; i++) FlmIdeNb[i] = 0;
    CountFillFilm( SrcDir );

    // Now counts of really present films.
    Flm_Esz = 0;
    for (int i=0; i<Flm_Siz; i++) if (FlmIdeNb[i]) Flm_Esz++;

    Sli_Tsz = Flm_Esz*Sli_Siz;
    SliTable = new SliEntry*[Sli_Tsz];  // Allocate  and initialyze the new sorting table.
    for (int i=0;i<Sli_Tsz;i++) SliTable[i] = NULL;

    FlmTable = new SliEntry**[Flm_Siz]; // Allocate the direct films access slide list table ...
    p = SliTable;                       // ... and built it.
    for(int i=0;i<Flm_Siz;i++) {
        if (FlmIdeNb[i]) { FlmTable[i] = p; p += Sli_Siz; }
                    else FlmTable[i] = NULL;
    }

    FlmRowNb = new UInt16[2*Flm_Siz];
    FlmRowSz = new UInt16[2*Flm_Siz];
    for(int r=0; r<2*Flm_Siz; r++) { FlmRowNb[r] = 0; FlmRowSz[r] = 0; }

    nerr = AnalyseFlmSli( SrcDir );

    // New section to support new version of Text_Display module.

    widx = SrcWin->ViewIndex() + 1;
    snprintf( txtitle, 38, "DiaViewer Ranking Map from window %d", widx );
    DspTxt = Text_Display::Init( rktb_wid, rktb_hig,
                                 txtitle, "DiaViewer_Rank_Check.txt", this );
    Active = 1;                 // Flag the active Text Display.

    SrcVol->FlgSet( Context_VRkMap );

    DspTxt->SetTitleFont( FL_HELVETICA, rktb_tsz );
    snprintf( txtbuf, 255, "Ranking volume table %s for window # %d", SrcDir->FName(), widx );
    DspTxt->SetTitle( strdup( txtbuf ) );

    DspTxt->SetFont( FL_COURIER, rktb_esz );

    DspTxt->SetCallBackMouse( (TxtDsp_Mouse_cb_t)MouseCallBack_fnc );
    TmpMap = tmpflg? 1 : 0;
    DspTxt->SetCallBackExit( (TxtDsp_cb_t)ExitCallBack_fnc );

    wlrg = (int)(18.25*Sli_Siz + 135.25); // Formulae used to determine for courrier font of fixed size 10.
    if (wlrg < scr_wmin) wlrg = scr_wmin;
                    else if (wlrg > scr_wmax) wlrg = scr_wmax;


    DspTxt->Append( " Source volume %s contains a total of %d slides\n",
                    SrcVol->VName(), SrcVol->TnSli() );
    DspTxt->Append( " and %d slides are numbered to be insert in a new volume to create.\n", esli );
    DspTxt->Append( "\n There are %d films with a maximum of %d slides\n", Flm_Esz, Sli_Siz );
    DspTxt->Append( " The slide above the maximum number of 64 are not take in account in the following map.\n\n" );
    if (nerr) {
        DspTxt->Append( "\t *** Ranking Error : %d %s ***\n\t\t%s\n\t\t%s\n\n", nerr,
                        "\t  *  Some slides share the same numbers (of film and slide).",
                        "\t  *  You must correct this to be allowed to perform the ranking.",
                        "\t  *  These error are flagged in the following map by ** in the films." );

    } else  DspTxt->Append( " You are ready to perform the ranking.\n\n" );


    Sta_Tli = DspTxt->SetOrigine();
//printf( " Sta_Tli = %d, text_pos = %d\n", Sta_Tli, DspTxt->TextPosition() );
    //
    // Sequence to create the text block to display the ranking table.
    //
    lflm = 0; nrow = 0;
    for (int ii=0; ii<Flm_Siz; ii++) {
        p = FlmTable[ii];
        if (p) {
            cflm = FlmIdeNb[ii];
            // Skip one line when the films # are not consecutive.
            if (lflm&&(cflm-lflm > 1)) {
                DspTxt->append( "\n" );
                nrow++;
            }
            lflm = cflm;
            FlmRowNb[nrow] = cflm;
            FlmRowSz[nrow] =    1;
            ncol = 0;
            snprintf( txtbuf, 255, " Film# %3d /", cflm );
            DspTxt->append( txtbuf );
            for (int jj = 0; jj < Sli_Siz; jj++) {
                if (p[jj]) {
                    if (p[jj] != ((SliEntry*)8)) sprintf( txtbuf, " %2d", jj+1 );
                                            else if (p[jj]) sprintf( txtbuf, " **" );
                    ncol = jj;
                } else sprintf( txtbuf, " .." );
                DspTxt->append( txtbuf );
            }
            DspTxt->append( "\n" );
            FlmRowSz[nrow] = ncol + 1;
//printf( " nrow = %d, Film # = %d, Film Size = %d\n", nrow, FlmRowNb[nrow], FlmRowSz[nrow] );
            nrow++;
        }
    }
    DspTxt->append( "\n" );
//printf( " End Table build : Sta_Tli = %d\n", Sta_Tli );
    DspTxt->Show();

    return (nerr)? 0 : 1;
} // int RkgTabCnt::FlmSliChart(  DiaViewer_GL * wgl ).



void RkgTabCnt::SlideScan( Slide_Process_t Scan )
{
    int flcnt, sliid;
    SliEntry   * sli;

    flcnt = 0;
    while (flcnt < Flm_Siz) {                           // Loop one all Films.
        if (FlmTable[flcnt]) {                          // When this Film exist.
            sliid = 0;
            while (sliid < Sli_Siz) {                   // Loop on all slide of the same film.
                if (sli = FlmTable[flcnt][sliid])       // Get the SliEntry reference.
                    if (!(sli->FlgTst( Context_LckRank )))
                        Scan( sli );                    // Copy all existing slides ...
                                                        // ... in new volume tree.
                sliid++;
            }
        }
        flcnt++;
    }
} // void RkgTabCnt::SlideScan().



int RkgTabCnt::TmpChartMap( DiaViewer_GL * win )
{
    int ierr;

    if (!(win->GetRkgMap()||win->Curr_Vol->FlgTst( Context_VRkMap ))) {
        RkgTabCnt * cntx = new RkgTabCnt( win );
        ierr = cntx->FlmSliChart( 1 );  // Not called during a ranking process.
        win->SetRkgMap( cntx );         // Set W_RkgMap to can clean related sub-window.
    }
    return ierr;
} // int RkgTabCnt::TmpChartMap( DiaViewer_GL * win ).



void RkgTabCnt::TextExit()
{
    if (Active&&DspTxt) DspTxt->Exit();
} // void RkgTabCnt::TextExit().



//
// end of <PW-Id>.
//
