//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


//
// Module to manage the Image file/Pixel Map for OpenGL.
//

//
// FLTK software part modified by P.Wolfers
// for DiaViewer Software using OPENGL image management.
//
//

//
// "$Id$"
//
// Fl_PNG_Image routines.
//
// Copyright 1997-2012 by Easy Software Products.
// Image support by Matthias Melcher, Copyright 2000-2009.
//
// This library is free software. Distribution and use rights are outlined in
// the file "COPYING" which should have been included with this file.  If this
// file is missing or damaged, see the license at:
//
//     http://www.fltk.org/COPYING.php
//
// Please report all bugs and problems on the following page:
//
//     http://www.fltk.org/str.php
//
// Contents:
//
//   Fl_PNG_Image::Fl_PNG_Image() - Load a PNG image file.
//

//
// Include necessary header files...
//

#include <stdio.h>
#include <stdlib.h>

#include <FL/Fl.H>
#include <FL/fl_utf8.h>

#include "ImageMan_MAP.h"


extern "C"
{
#  include <zlib.h>
#  ifdef HAVE_PNG_H
#    include <png.h>
#  else
#    include <libpng/png.h>
#  endif // HAVE_PNG_H
}

typedef struct  {
    png_structp             pp;
    const uchar       *current;
    const uchar          *last;
} fl_png_memory;



//
//  To save a copy of an image in BMP format.
//
void Image_MAP::Write_PNG( const char * name, int pf )
{

} // void Image_MAP::Write_PNG( const char * filename ).





extern "C" {
    static void png_read_data_from_mem(
        png_structp png_ptr,    //pointer to our data
        png_bytep data,         // where to copy the image data for libpng computing
        png_size_t length       // length of data to copy
    )
    {
        fl_png_memory *png_mem_data =
            (fl_png_memory*)png_get_io_ptr( png_ptr );  // get the pointer to our struct
        if (png_mem_data->current + length > png_mem_data->last) {
            png_error( png_mem_data->pp, "Invalid attempt to read row data" );
            return;
        }
        /* copy data from image buffer */
        memcpy ( data, png_mem_data->current, length );
        /* advance in the memory data */
        png_mem_data->current += length;
    }
} // extern "C"



/**
 The constructor loads the named PNG image from the given png filename.

 The destructor frees all memory and server resources that are used by
 the image.

 Use Fl_Image::fail() to check if Fl_PNG_Image failed to load. fail() returns
 ERR_FILE_ACCESS if the file could not be opened or read, ERR_FORMAT if the
 PNG format could not be decoded, and ERR_NO_IMAGE if the image could not
 be loaded for another reason.

 \param[in] filename	Name of PNG file to read
 */
void Image_MAP::Get_PNG ( const char *filename )
{
    Load_PNG_( filename, NULL, 0 );
}


/** 
 \brief Constructor that reads a PNG image from memory.

 Construct an image from a block of memory inside the application.
 
 \param name_png  A name given to this image or NULL
 \param buffer	  Pointer to the start of the PNG image in memory
 \param maxsize   Size in bytes of the memory buffer containing the PNG image
 */
void Image_MAP::Get_PNG( const unsigned char *buffer, int maxsize )
{
    Load_PNG_( NULL, buffer, maxsize );
}


void Image_MAP::Load_PNG_( const char    *name_png, 
                           const uchar *buffer_png,
                           int maxsize )
{
    int                      i;	// Looping var
    FILE            *fp = NULL;	// File pointer
    int               channels;	// Number of color channels
    png_structp             pp; // PNG read pointer
    png_infop         info = 0; // PNG info pointers
    png_bytep            *rows; // PNG row pointers
    fl_png_memory png_mem_data;
    GLubyte               *map; // The temporary OpenGL 
    GLint          ww, hh, nsz; // Used to optimize Pixel map allocation.
    
    int from_memory = (buffer_png != NULL);     // true if reading image from memory

    if (!from_memory) {
        if ((fp = fl_fopen( name_png, "rb") ) == NULL) {
            E(ERR_FILE_ACCESS);
            return;
        }
    }
    const char *display_name = (name_png ? name_png : "In-memory PNG data");

    // Setup the PNG data structures...
    pp = png_create_read_struct( PNG_LIBPNG_VER_STRING, NULL, NULL, NULL );
    if (pp) info = png_create_info_struct( pp );
    if (!pp || !info) {
        if (pp) png_destroy_read_struct( &pp, NULL, NULL );
        if (!from_memory) fclose( fp );
        Fl::warning( "Cannot allocate memory to read PNG file or data \"%s\".\n",
                     display_name );
        E( ERR_FORMAT );
        return;
    }
  
    if (setjmp( png_jmpbuf( pp ) ) ) {
        png_destroy_read_struct( &pp, &info, NULL );
        if (!from_memory) fclose( fp );
        Fl::warning("PNG file or data \"%s\" is too large or contains errors!\n",
                    display_name );
        E( ERR_FORMAT );
        return;
    }

    if (from_memory) {
        png_mem_data.current = buffer_png;
        png_mem_data.last = buffer_png + maxsize;
        png_mem_data.pp = pp;
        // Initialize the function pointer to the PNG read "engine"...
        png_set_read_fn( pp, (png_voidp) &png_mem_data, png_read_data_from_mem );
    } else {
        png_init_io( pp, fp ); // Initialize the PNG file read "engine"...
    }  

    // Get the image dimensions and convert to grayscale or RGB...
    png_read_info( pp, info );

    if (png_get_color_type( pp, info ) == PNG_COLOR_TYPE_PALETTE)
        png_set_expand( pp );

    if (png_get_color_type( pp, info ) & PNG_COLOR_MASK_COLOR)
        channels = 3;
    else
        channels = 1;

    int num_trans = 0;
    png_get_tRNS( pp, info, 0, &num_trans, 0 );
    if ((png_get_color_type( pp, info ) & PNG_COLOR_MASK_ALPHA) || (num_trans != 0))
        channels ++;

    ww = (GLint)(png_get_image_width( pp, info ));
    hh = (GLint)(png_get_image_height(pp, info));

    if (png_get_bit_depth(pp, info) < 8) {
        png_set_packing( pp );
        png_set_expand( pp );
    }
    else if (png_get_bit_depth( pp, info ) == 16)
        png_set_strip_16( pp );

#  if defined( HAVE_PNG_GET_VALID ) && defined( HAVE_PNG_SET_TRNS_TO_ALPHA )
    // Handle transparency...
    if (png_get_valid( pp, info, PNG_INFO_tRNS )) png_set_tRNS_to_alpha( pp );
#  endif // HAVE_PNG_GET_VALID && HAVE_PNG_SET_TRNS_TO_ALPHA

    if ((nsz = ww*hh*channels) > Max_Size ) longjmp( png_jmpbuf( pp ), 1 );
    
    map = New_Map( nsz );
    W( ww );
    H( hh );
    P( channels );

    // Allocate pointers...
    rows = new png_bytep[hh];

    // Set the row pointers in from Bottom to UP for OpenGL.
    for (i = 0; i < hh; i ++)
        rows[i] = (png_bytep)(map + (hh -1 - i)*ww*channels);

    // Read the image, handling interlacing as needed...
    for (i = png_set_interlace_handling( pp ); i > 0; i --)
        png_read_rows( pp, rows, NULL, hh );

#ifdef WIN32
    // Some Windows graphics drivers don't honor transparency when RGB == white
    if (channels == 4) { // Convert RGB to 0 when alpha == 0 ...
        uchar *ptr = (uchar *)map;
        for (i = ww * hh; i > 0; i --, ptr += 4)
            if (!ptr[3]) ptr[0] = ptr[1] = ptr[2] = 0;
    }
#endif // WIN32

    // Free memory and return...
    delete[] rows;
    
    map = NULL; // Already copied in map_ of class Image_MAP.

    png_read_end( pp, info );
    png_destroy_read_struct( &pp, &info, NULL );

    if (!from_memory) fclose( fp );
}


//
// End of "$Id$".
//

//
// end of <PW-Id>.
//
