//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


//
// Module to manage the Image file/Pixel Map for OpenGL.
//

//
// FLTK software part modified by P.Wolfers
// for DiaViewer Software using OPENGL image management.
//
//

//
// "$Id$"
//
// Fl_BMP_Image routines.
//
// Copyright 1997-2010 by Easy Software Products.
// Image support by Matthias Melcher, Copyright 2000-2009.
//
// This library is free software. Distribution and use rights are outlined in
// the file "COPYING" which should have been included with this file.  If this
// file is missing or damaged, see the license at:
//
//     http://www.fltk.org/COPYING.php
//
// Please report all bugs and problems on the following page:
//
//     http://www.fltk.org/str.php
//
// Contents:
//
//   Image_MAP::Get_BMP() - Load a BMP image file.
//

//
// Include necessary header files...
//


#include <FL/fl_utf8.h>
#include <FL/Fl.H>

#include <stdio.h>
#include <stdlib.h>

#include "ImageMan_MAP.h"



//
// BMP definitions...
//

#ifndef BI_RGB
#  define BI_RGB       0             // No compression - straight BGR data
#  define BI_RLE8      1             // 8-bit run-length compression
#  define BI_RLE4      2             // 4-bit run-length compression
#  define BI_BITFIELDS 3             // RGB bitmap with RGB masks
#endif // !BI_RGB


// The common FILE variable.

static FILE*  fp;


//
// Local functions...
//


static void wrInt16( int iv );
static void wrInt32( int iv );



static int		read_long();
static unsigned short	read_word();
static unsigned int	read_dword();


//
//  To save a copy of an image in BMP format.
//
void Image_MAP::Write_BMP( const char * name, int prtflg )
{ // Routine to Create a BMP File from the Pixel_buffer Image.

    int ww, hh, pp, rwsz, bmsz;

    GLubyte * map, *p;

    float d_xscale = 100.0,
          d_yscale = 100.0;

    int   Hres, npad, Vres, nCol, i, j;

    char  com[255];

    if (fp = fopen( name, "wb" )) {     // Open is a success

        ww = W(); hh = H();             // Get the size of  image in pixels.
        pp = P();                       // Get the size of pixel.
        map = Map();
        rwsz = pp*ww;
        npad = (rwsz*3)&3;
        if (npad) npad = 4 - npad;      // Compute BMP Round UP
        rwsz = rwsz + npad;             // Round up to multipl of 4.
        bmsz = hh*rwsz;                 // Bit Map Size (with Round up to a multiple of 4) ///

        Hres = (int) (d_xscale*100.0 + 0.5);
        Vres = (int) (d_yscale*100.0 + 0.5);

        nCol = 0x1000000;               //   : Set to 2**24 Colors.

        // The 14 First Bytes Are Write Now :
        putc( 'B', fp );       // Generate the BM BitMap type
        putc( 'M', fp );
        wrInt32( bmsz + 54 );  // Bit Map file Size
        wrInt16( 0 );          // Reserved Dword
        wrInt16( 0 );          // Reserved Dword
        wrInt32( 54 );         // Offset of Beginning of data (no Color Map).

        // The 40 Bytes of Image are Write Now
        wrInt32( 0x28 );       // Write bitmap Info Header Size
        wrInt32( W() );        // Write Image Width
        wrInt32( H() );        // Write Image High
        wrInt16( 1 );          // Write the Number of planes
        wrInt16( 24 );         // Write the number of bits by Pixels (always 24 Here)
        wrInt32( 0 );          // Specify NO Compression
        wrInt32( bmsz );       // Specify the total size of the bitmap data
        wrInt32( Hres );       // Specify the Horizontal resolution (pixels/meter)
        wrInt32( Vres );       // Specify the Vertical resolution (pixels/meter)
        wrInt32( nCol );       // Specify the total number of used Color
        wrInt32( 0 );          // Specify the total number of Important Color

         // We do not generate a Color map.

        // Now Write the RGB BitMap :
        p = map;
        for (j = 0; j < hh; j++) {
            p = map + j*rwsz;
            for (i = 0; i < rwsz; i++) {
                putc( p[2], fp );
                putc( p[1], fp );
                putc( p[0], fp );
                p += 3;
            }
            // Write pad byte :
            for (i = 0; i < npad; i++) putc( 0, fp );
        }

        fclose( fp );           // Close the BMP File


//      status = fork();        // Create the Client process.
//
//      if (status == 0)        // Select the process
//          if ((status = execl( "mspaint", "mspaint", "Draw_Image.bmp" )) == -1) // ... to start it.
//          ; //  exit( 3 );         /* Stop with message on Exec ERROR. */

        if (prtflg) {          // When a BitMap print is required
            sprintf( com, "mspaint /P %s\n", name );
//          fprintf( fmsg, " Execute Command com %s\n", com );
            system( com );
        }
    }

//  fprintf( fmsg, " BMP_Image End on Success.\n" );
//  fflush( fmsg );

} // void Image_MAP::Write_BMP( const char * filename ).




/**
 The constructor loads the named BMP image from the given bmp filename.

 The destructor frees all memory and server resources that are used by
 the image.

 Use Fl_Image::fail() to check if Fl_BMP_Image failed to load. fail() returns
 ERR_FILE_ACCESS if the file could not be opened or read, ERR_FORMAT if the
 BMP format could not be decoded, and ERR_NO_IMAGE if the image could not
 be loaded for another reason.
 */



//=============================>>> wrInt16 <<<=============================
  static void wrInt16( int iv )
  {
    union { char  ct[2];
            short iv;
          } equ;

//    int c0, c1;

//  c0 = iv & 255; c1 = (iv >> 8) & 255;
//  putc( c0, fimg );  putc( c1, fimg );

    equ.iv = iv;
    putc( equ.ct[0], fp ); putc( equ.ct[1], fp );
  }

//=============================>>> wrInt32 <<<=============================
  static void wrInt32( int iv )
  {
    union { char  ct[4];
            int   iv;
          } equ;

    equ.iv = iv;
    putc( equ.ct[0], fp ); putc( equ.ct[1], fp );
    putc( equ.ct[2], fp ); putc( equ.ct[3], fp );
  }


//=============================>>> wrInt32 <<<=============================

#if 0
  static void wrFlt32( float v )
  {
    union { float    v;
            char ch[4];
          } equ;

    equ.v = v;
    putc( equ.ch[0], fp ); putc( equ.ch[1], fp );
    putc( equ.ch[2], fp ); putc( equ.ch[3], fp );
  }
#endif









void Image_MAP::Get_BMP( const char *bmp ) // I - File to read
{

    int		 info_size, // Size of info header
                 depth, // Depth of image (bits)
		    bDepth = 3, // Depth of image (bytes)
		   compression, // Type of compression
		   colors_used, // Number of colors used
		          x, y,	// Looping vars
                 color, // Color of RLE pixel
		      repcount, // Number of times to repeat
		          temp, // Temporary color
		         align, // Alignment bytes
		      dataSize, // number of bytes in image data set
		     row_order, // 1 = normal;  -1 = flipped row order
		       start_y, // Beginning Y
		         end_y;	// Ending Y
    long	   offbits; // Offset to image data
    uchar		   bit,	// Bit in image
		          byte;	// Byte in image
    uchar		  *ptr;	// Pointer into pixels

    uchar colormap[256][3]; // Colormap
    uchar	      havemask;	// Single bit mask follows image data
    int		     use_5_6_5;	// Use 5:6:5 for R:G:B channels in 16 bit images
    GLubyte          * map; // The Pixel map to build
    GLint  ww, hh, pp, nsz; // To optimise the Pixel Map allocation.

    // Open the file...
    if ((fp = fl_fopen( bmp, "rb" )) == NULL) {
        E( ERR_FILE_ACCESS );
        return;
    }

    // Get the header...
    byte = (uchar)getc( fp );	// Check "BM" sync chars
    bit  = (uchar)getc( fp );
    if (byte != 'B' || bit != 'M') {
        fclose( fp );
        E( ERR_FORMAT );
        return;
    }

    read_dword();		// Skip size
    read_word();		// Skip reserved stuff
    read_word();
    offbits = (long)read_dword();   // Read offset to image data

    // Then the bitmap information...
    info_size = read_dword();

    //  printf("offbits = %ld, info_size = %d\n", offbits, info_size);

    havemask  =  0;
    row_order = -1;
    use_5_6_5 =  0;

    if (info_size < 40) { // Old Windows/OS2 BMP header...
        ww = read_word();
        hh = read_word();
        read_word();
        depth = read_word();
        compression = BI_RGB;
        colors_used = 0;

        repcount = info_size - 12;
    } else {
        // New BMP header...
        ww = read_long();
        // If the height is negative, the row order is flipped
        temp = read_long();
        if (temp < 0) row_order = 1;
        hh = abs( temp );
        read_word();
        depth = read_word();
        compression = read_dword();
        dataSize = read_dword();
        read_long();
        read_long();
        colors_used = read_dword();
        read_dword();

        repcount = info_size - 40;

        if (!compression && depth>=8 && ww>32/depth) {
            int Bpp = depth/8;
            int maskSize = (((ww*Bpp+3)&~3)*hh) + (((((ww+7)/8)+3)&~3)*hh);
            if (maskSize==2*dataSize) {
                havemask = 1;
                hh /= 2;
                bDepth = 4;
            }
        }
    }

    //  printf("w() = %d, h() = %d, depth = %d, compression = %d, colors_used = %d, repcount = %d\n",
    //         w(), h(), depth, compression, colors_used, repcount);

    // Skip remaining header bytes...
    while (repcount > 0) {
        getc( fp );
        repcount --;
    }

    // Check header data...
    if (!ww || !hh || !depth) {
        fclose( fp );
        E( ERR_FORMAT );
        return;
    }

    // Get colormap...
    if (colors_used == 0 && depth <= 8)
        colors_used = 1 << depth;

    for (repcount = 0; repcount < colors_used; repcount ++) {
        // Read BGR color...
        if (fread( colormap[repcount], 1, 3, fp )==0) { /* ignore */ }

        // Skip pad byte for new BMP files...
        if (info_size > 12) getc( fp );
    }

    // Read first dword of colormap. It tells us if 5:5:5 or 5:6:5 for 16 bit
    if (depth == 16)
        use_5_6_5 = (read_dword() == 0xf800);

    // Set byte depth for RGBA images
    if (depth == 32)
        bDepth=4;

    // Setup image and buffers...
    pp = bDepth;
    if (offbits) fseek( fp, offbits, SEEK_SET );

    if ((nsz = ww*hh*bDepth) > Max_Size ) {
        Fl::warning( "BMP file \"%s\" is too large!\n", bmp );
        fclose( fp );
        E( ERR_FORMAT );
        return;
    }
    map = New_Map( nsz );
    W( ww );
    H( hh );
    P( pp );

    // Read the image data...
    color = 0;
    repcount = 0;
    align = 0;
    byte  = 0;
    temp  = 0;

    // Modified (from FLTK version) to give Bottom ...
    // ... to UP image compatible with OpenGL.
    if (row_order > 0) {
        start_y = hh - 1;
        end_y   = -1;
    } else {
        start_y = 0;
        end_y   = hh;
    }

    for (y = start_y; y != end_y; y -= row_order) {
        ptr = (uchar *)map + y * ww * pp;

        switch (depth) {
            case 1 : // Bitmap
                for (x = ww, bit = 128; x > 0; x --) {
                    if (bit == 128) byte = (uchar)getc( fp );

                    if (byte & bit) {
                        *ptr++ = colormap[1][2];
                        *ptr++ = colormap[1][1];
                        *ptr++ = colormap[1][0];
                    } else {
                        *ptr++ = colormap[0][2];
                        *ptr++ = colormap[0][1];
                        *ptr++ = colormap[0][0];
                    }

                    if (bit > 1) bit >>= 1; else bit = 128;
                }

                // Read remaining bytes to align to 32 bits...
                for (temp = (ww + 7) / 8; temp & 3; temp ++) {
                    getc( fp );
                }
            break;

            case 4 : // 16-color
                for (x = ww, bit = 0xf0; x > 0; x --) {
                    // Get a new repcount as needed...
                    if (repcount == 0) {
                        if (compression != BI_RLE4) {
                            repcount = 2;
                            color = -1;
                        } else {
                            while (align > 0) {
                                align --;
                                getc( fp );
                            }

                            if ((repcount = getc( fp )) == 0) {
                                if ((repcount = getc( fp )) == 0) {
                                    // End of line...
                                    x ++;
                                    continue;
                                } else if (repcount == 1) {
                                    // End of image...
                                    break;
                                } else if (repcount == 2) {
                                    // Delta...
                                    repcount = getc( fp ) * getc( fp ) * W();
                                    color = 0;
                                } else {
                                    // Absolute...
                                    color = -1;
                                    align = ((4 - (repcount & 3)) / 2) & 1;
                                }
                            } else {
                                color = getc(fp);
                            }
                        }
                    }

                    // Get a new color as needed...
                    repcount --;

                    // Extract the next pixel...
                    if (bit == 0xf0) {
                        // Get the next color byte as needed...
                        if (color < 0) temp = getc( fp );
                                  else temp = color;

                        // Copy the color value...
                        *ptr++ = colormap[(temp >> 4) & 15][2];
                        *ptr++ = colormap[(temp >> 4) & 15][1];
                        *ptr++ = colormap[(temp >> 4) & 15][0];

                        bit  = 0x0f;
                    } else {
                        bit  = 0xf0;

                        // Copy the color value...
                        *ptr++ = colormap[temp & 15][2];
                        *ptr++ = colormap[temp & 15][1];
                        *ptr++ = colormap[temp & 15][0];
                    }
                }

                if (!compression) {
                    // Read remaining bytes to align to 32 bits...
                    for (temp = (ww + 1) / 2; temp & 3; temp ++) {
                        getc( fp );
                    }
                }
            break;

            case 8 : // 256-color
                for (x = ww; x > 0; x --) {
                    // Get a new repcount as needed...
                    if (compression != BI_RLE8) {
                        repcount = 1;
                        color = -1;
                    }

                    if (repcount == 0) {
                        while (align > 0) {
                            align --;
                            getc( fp );
                        }

                        if ((repcount = getc( fp )) == 0) {
                            if ((repcount = getc( fp )) == 0) {
                                // End of line...
                                x ++;
                                continue;
                            } else if (repcount == 1) {
                                // End of image...
                                break;
                            } else if (repcount == 2) {
                                // Delta...
                                repcount = getc( fp ) * getc( fp ) * W();
                                color = 0;
                            } else {
                                // Absolute...
                                color = -1;
                                align = (2 - (repcount & 1)) & 1;
                            }
                        } else {
                            color = getc(fp);
                        }
                    }

                    // Get a new color as needed...
                    if (color < 0) temp = getc( fp );
                              else temp = color;
    
                    repcount --;

                    // Copy the color value...
                    *ptr++ = colormap[temp][2];
                    *ptr++ = colormap[temp][1];
                    *ptr++ = colormap[temp][0];
                    if (havemask) ptr++;
                }

                if (!compression) {
                    // Read remaining bytes to align to 32 bits...
                    for (temp = ww; temp & 3; temp ++) {
                        getc( fp );
                    }
                }
            break;

            case 16 : // 16-bit 5:5:5 or 5:6:5 RGB
                for (x = ww; x > 0; x --, ptr += bDepth) {
                    uchar b = getc( fp ), a = getc( fp ) ;
                    if (use_5_6_5) {
                        ptr[2] = (uchar)(( b << 3 ) & 0xf8);
                        ptr[1] = (uchar)(((a << 5) & 0xe0) | ((b >> 3) & 0x1c));
                        ptr[0] = (uchar)(a & 0xf8);
                    } else {
                        ptr[2] = (uchar)((b << 3) & 0xf8);
                        ptr[1] = (uchar)(((a << 6) & 0xc0) | ((b >> 2) & 0x38));
                        ptr[0] = (uchar)((a<<1) & 0xf8);
                    }
                }

                // Read remaining bytes to align to 32 bits...
                for (temp = ww * 2; temp & 3; temp ++) {
                    getc( fp );
                }
            break;

            case 24 : // 24-bit RGB
                for (x = ww; x > 0; x --, ptr += bDepth) {
                    ptr[2] = (uchar)getc( fp );
                    ptr[1] = (uchar)getc( fp );
                    ptr[0] = (uchar)getc( fp );
                }

                // Read remaining bytes to align to 32 bits...
                for (temp = ww * 3; temp & 3; temp ++) {
                    getc( fp );
                }
            break;
		  
            case 32 : // 32-bit RGBA
                for (x = ww; x > 0; x --, ptr += bDepth) {
                    ptr[2] = (uchar)getc( fp );
                    ptr[1] = (uchar)getc( fp );
                    ptr[0] = (uchar)getc( fp );
                    ptr[3] = (uchar)getc( fp );
                }
            break;
        }
    }
  
    if (havemask) {
        for (y = hh - 1; y >= 0; y --) {
            ptr = (uchar *)map + y * ww * bDepth + 3;
            for (x = ww, bit = 128; x > 0; x --, ptr+=bDepth) {
                if (bit == 128) byte = (uchar)getc( fp );
                if (byte & bit)
                    *ptr = 0;
                else
                    *ptr = 255;
                if (bit > 1)
                    bit >>= 1;
                else
                    bit = 128;
            }
            // Read remaining bytes to align to 32 bits...
            for (temp = (ww + 7) / 8; temp & 3; temp ++)
                getc( fp );
        }
    }

    // Close the file and return...
    fclose( fp );
    map = NULL; // Already copied in map_ of class Image_MAP.
}






//
// 'read_word()' - Read a 16-bit unsigned integer.
//

static unsigned short read_word()
// O - 16-bit unsigned integer
// I - File to read from
{
    unsigned char b0, b1;	// Bytes from file

    b0 = (uchar)getc( fp );
    b1 = (uchar)getc( fp );

    return ((b1 << 8) | b0);
}


//
// 'read_dword()' - Read a 32-bit unsigned integer.
//

static unsigned int read_dword()
// O - 32-bit unsigned integer
// I - File to read from
{
    unsigned char b0, b1, b2, b3;	// Bytes from file

    b0 = (uchar)getc( fp );
    b1 = (uchar)getc( fp );
    b2 = (uchar)getc( fp );
    b3 = (uchar)getc( fp );

    return ((((((b3 << 8) | b2) << 8) | b1) << 8) | b0);
}


//
// 'read_long()' - Read a 32-bit signed integer.
//

static int read_long()
// O - 32-bit signed integer
// I - File to read from
{
    unsigned char b0, b1, b2, b3;	// Bytes from file

    b0 = (uchar)getc( fp );
    b1 = (uchar)getc( fp );
    b2 = (uchar)getc( fp );
    b3 = (uchar)getc( fp );

    return ((int)(((((b3 << 8) | b2) << 8) | b1) << 8) | b0);
}


//
// End of "$Id$".
//

//
// end of <PW-Id>.
//
