//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module to manage a slide collection
// DCM = DiaViewer Collection Management.
//

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <errno.h>
#include <string.h>

#include <setjmp.h>

#include <FL/Fl.H>
#include <FL/fl_ask.H>
#include <FL/fl_utf8.h>         // To manipulate utf8 strings.

#include "SetUpMan.h"

#include "Service_Util.h"
#include "Service_IO.h"

#include "DiaViewer_GBL.h"
#include "DiaViewer_DIA.h"
#include "DiaViewer_DCM.h"
#include "DiaViewer_PRG.h"
#include "DiaViewer_MVL.h"
#include "DiaViewer_VRFL.h"
#include "DiaViewer_V_MN.h"



// Define the increment(step) to allocate (or extend) a dynamic table of item.
#define TABLE_ALLOC_STEP     16

// Define the limit table length to select the quick sort process.
#define QSORT_LIMIT           8




//
// Each diapositive/slide is identified by a unique IIde_t integer.
// This identifier include three part. The volume identifier number,
// the directory identifier number (in the volume) and the slide identifier
// unique in the volume. The three field in the IIde_t identifier can be 
// defined by a definition as :
// #define LARGE_UNI_IDENT
// to have a 64 bits identifier type (the default is 32 bits), and by
// #define UNIQUE_IDE_FIELDS {<sld>,<dir>}
// For example, for 32 bits identifiers:
// #define  UNIQUE_IDE_FIELDS {16,10}
// while by used to get a slide field of 16 bits and a directories field of 10 bits,
// that keep 6 bits (32 - 16 - 10 = 6) for the volume. With this, you can have a
// maximum of 2**16-1 = 65535 slides in each directory, 2**10-1 = 1023 directories
// in each volume and 2**(32-16-10)-1 = 64 volumes. The default 32 bits IIde Identifier
// is defined at the compilation time but the UNIQUE_IDE_FIELDS is keep in the user's
// local data context and can be changed when the DiaViewer software is restarted and
// it should be possible to change these field sizes by using the setup menu in some
// future versions.
// The default fields size values are :
//    #define UNIQUE_IDE_FIELDS {14,10} in 32 bits that allows
//           16,383 slides in each directory,
//             1023 directories in each volume and
//              255 opened volumes.
//
//    #define UNIQUE_IDE_FIELDS {32,16} in 64 bits that allows
//    4,294,967,297 slides in each directory,
//           65,535 directories in each volume and
//           65,535 opened volumes.
//
// The default DiaViewer software generation is with Unique Identifiers of 32 bits.
//
//




extern   VolRefList    VolList; // The Known slide Volume list (memory and user env. file).


static VarString         V1Str, // Variable Strings used as temporary strings.
                         V2Str;



//
// Volume List file input/output Read and Write.
//



static sigjmp_buf       JmpBuf; // Buffer for long jump on I/O context error.



//
// Our Task'Global Variables Definitions (public).
//
const char  DirSep  =   DIRSEP; // Separator to use between files and directories for our OS.


//
// Global variables.
//


VolEntry  * IniSelVolum = NULL; // Selected volume pointer.
SliEntry  * IniSelSlide = NULL; // Initial selected Slide pointer.


//
// Our module specific Variables Definitions (private).
//

static char        ErrMsg[512]; // To keep the last Io Error Message.
static int        ErrMsglength; //

static FilePath       CurrPath; // To keep the current path During the construction of volume tree.

/*
static int    max_nbdir =    0, // Maximul number of Sub-Directories in a Directory.
              max_nbsli =    0; // Maximum number of slides in a Directory.
*/

static int    curr_deep =    0, // Current directory deep from the base directory.
              curr_tsli =    0, // Current total numbers of slides and subdirectories.
              curr_tdir =    0,
              maxi_deep =    0, // Maximum directory deep.
              deep_error   = 0; // Slide Volume overlay error flag.

/// static char * CFilePath = NULL; // Config File Path (when found).

/// static FILE *             cfgf; // Configuration file.



#define PRT( f, v ) printf( f, v ); fflush( stdout )

//
// Entry type defined to directory entries.
//

typedef void * Entry;

typedef struct { // Define the element of Entry table (for sorting).
    int    use,  // Used size.
           all;  // Allocated size.
    Entry *tab;  // The table of name.
} EntryTable;

static int DirFlg = 0; // for sorting directories/slides entries.



//
// Some routines.
//

static void ErrorCatch( int ierr )
{
    siglongjmp( JmpBuf, ierr );
} // void ErrorCatch( int ierr ).



static EntryTable * NewEntry( Entry en, EntryTable *tb = NULL )
{  // Allocate a enough large Sort table.
  Entry    *tm;
  int     size;

  if (!tb) {
    tb = new EntryTable;
    tb->use =    0;
    tb->all =    0;
    tb->tab = NULL;
    size    =    0;
  }
  size = tb->use + 1;
  if (size > tb->all) {
    size = (size + TABLE_ALLOC_STEP)&~(TABLE_ALLOC_STEP - 1);
    tm = new Entry[size];
    if (tb->use)
      for(int ii=0;ii<tb->use;ii++) tm[ii] = tb->tab[ii];
    if (tb->tab) delete[] tb->tab;
    tb->tab =   tm;
    tb->all = size;
  }
  tb->tab[tb->use++] = en;
  return tb;
} // static EntryTable * NewEntry( Entry en, EntryTable *tb = NULL ).



static void FreeEntryTable( EntryTable * etb )
{
  etb->use = etb->all = 0;
  if (etb->tab) delete[] etb->tab;
  delete etb;
} // static void FreeEntryTable( EntryTable * etb ).



static int CmpItemName( Entry *tb, int jp, Entry id )
{ // Compare the jp'th entry name in the tb entry table with the string sp. 
  const char * sj = (DirFlg)? ((DirEntry*)tb[jp])->FName() : ((SliEntry*)tb[jp])->FName(),
             * sp = (DirFlg)? ((DirEntry*)id)->FName() : ((SliEntry*)id)->FName();

//#if defined(WIN32) || defined(__EMX__)
  return fl_utf_strcasecmp( sj, sp );
//#else
//return fl_utf_strcasecmp( sj, id, nnnn );
//#endif
} // static int CmpItemName( Entry *tb, int jp, Entry id ).



static void ItemSimplSort( Entry *tb, int min, int max )
{
  int       jj;
  Entry     id;
    
  for(int ii=min+1; ii<=max; ii++) {
    id = tb[ii];
    jj = ii - 1;
    while (jj>=min&&CmpItemName( tb, jj, id )>=0) {
      tb[jj+1] = tb[jj];
      jj--;
    }
    tb[jj+1] = id;
  }
} // static void ItemSimplSort( Entry *tb, int min, int max ).



static void ItemQuickSort( Entry *tb, int min, int max )
// Perform a Quick sort.
{
  int   ic, ip;
  Entry pi, tm;
    
  if (min>=max) return;
    
  ip = (min + max)/2;   // Get the middle Sort table index ...
  pi =        tb[ip];   // ... as the pivot.

  tb[ip] = tb[min];     // Put the pivot item in first position.
  tb[min] = pi;         //

  ic = min;             //
  for(int ii = min; ii <= max; ii++) {
    // Loop on all entries.
    if (CmpItemName( tb, ii, pi )<0) {
      if (ii != ++ic) {
        // When the entry are differents, we exchange them.
        tm = tb[ii]; tb[ii] = tb[ic]; tb[ic] = tm;
      }
    }
  }
  // Put the middle entry to its right place.
  tm = tb[ic]; tb[ic] = tb[min]; tb[min] = tm;

  if (ic-min<QSORT_LIMIT) ItemSimplSort( tb, min, ic-1 );
                     else ItemQuickSort( tb, min, ic-1 );
  if (max-ic<QSORT_LIMIT) ItemSimplSort( tb, ic+1, max );
                     else ItemQuickSort( tb, ic+1, max );
} // static void ItemQuickSort( Entry *tb, int min, int max ).



static void EntrySort( Entry *tb, int size )
{
  if (size<QSORT_LIMIT) ItemSimplSort( tb, 0, size-1 );
                   else ItemQuickSort( tb, 0, size-1 );
} // static void EntrySort( Entry *tb, int size ).



static SliEntry * SliInList( char * name )
{
  SliEntry     * nwsli;
  int     ii,  jj,  ll;
  char          ext[8];

  const char extab[] = "jpg ,jpeg,JPG ,JPEG,png ,PNG ,bmp ,BMP ,0000";
  const int  n_extab = 8;
    
  // We must check for supported image file.
  ii = strlen( name ); ll = 0;
  while (name[--ii] != '.' && ll < 5) ll++;
  if (ll>4||ll<3) return NULL;  // Unknown file extension -> skip file.
    
  ext[0] = name[++ii];          // Get the founded file extention.
  ext[1] = name[++ii];
  ext[2] = name[++ii];
  ext[3] = (ll==4)?name[++ii]:' ';
  ext[4] = 0;
  jj=0;
  for (ii=0;ii<n_extab;ii++) {  // Loop to find when it a supported file type.
    if (!memcmp( ext, extab+jj, 4 )) break;
    jj += 5;
  }
  if (ii>=n_extab) return NULL; // Unknown file extension -> skip file.
    
  // Now we can include this image in the current directory image list.
  nwsli = new SliEntry( name, 0 );
    
/// printf( " Enter the image \"%s\"\n", dia->diafnam );
    
  return nwsli;
} // static SliEntry * SliInList( char * name ).



// To check for file/dir access with req = 0:directory, 1:r, 2:rw.
int CheckFilePath( char ** path, int req )
{
    int mod, res, ret;

    ret = 0;
    switch (req) {
        case 0: // Directory check.
          if (*path) {
            res = fl_access( *path, R_OK );
            if (res) delete[] (*path);
          } else { res = 1; errno = ENOENT; }
          if (res) {
            *path = strdup( "./" );
            ret = (errno == ENOENT)? 1 : -1;
          }
          return ret;
        case 1:
            res = fl_access( *path, R_OK );
        break;
        default:
            res = fl_access( *path, R_OK|W_OK );
        break;
    }
    if (res) ret = errno == ENOENT ? 1 : -1;
    return ret;
} // int CheckFilePath( const char * ).



/*
static int OvrlTreeTest()
{
  int pthl, ierr;

  pthl = CurrPath.Length();
  CurrPath.Set( CNTX_FNAME, 0 );
  ierr = CntxOpen( CurrPath.Path(), 0 );
  if (ierr > 0) {
    // The Context file exist.
        
    CntxClose();
  }
  CurrPath.Rem( pthl );

  return ierr;
} // static int OvrlTreeTest().
*/


static void TreeUpdateDeep( DirEntry * dir, int deep )
{
    int ii, nd;

    dir->LDeep( deep );
    nd = dir->NDir();
    for(ii = 1; ii <= nd; ii++)
        TreeUpdateDeep( dir->SbDir( ii ), deep + 1 );
} // static void TreeUpdateDeep( DirEntry * dir, int deep ).



static DirEntry * ScanFoundDir( const char * name ) {
  // pdir is the parent directory descriptor,
  // path is the complete path where is founded the new directory entry,
  // name is the name of the new directory entry.
  //

  DIR            * dir; // Directory file descriptor used by opendir(),
                        // readir() and closedir().
  struct dirent  * dsc; // Directory entry descriptor - from readir().
  struct stat    entbf; // File or Sub-directory descriptor - from stat().
  DirEntry     * nwdir; // Pointer to the current directory descriptor,
  SliEntry     * nwsli; // Pointer to the current diapositive descriptor.
  EntryTable   * lidir, // Sub directory pointers list.
               * lisli; // Slides/Diapositives pointers list.
  int        ncdir = 0, // Cumullated number of sub-directories found.
             ncsli = 0, // Cumullated number of diapositive found.
             ndir  = 0, // Local number of sub-directories found.
             nsli  = 0, // Local number of diapositives found.
                 ideep; // Last deep.

//int  ideep,  basplen; // Last deep ,cheeck, Keep the base Path length.

  CurrPath.Set( NULL, 0 );      // Open Directory without final DirSep.

#ifdef _HAVE_FL_UTF8_HDR_
  if (!(dir=opendir( CurrPath.Path() ))) {
#else
  if (!(dir=opendir( CurrPath.Path() ))) {
#endif

    ErrMsglength = snprintf( ErrMsg, 512, "\n  * DiaViewer Cannot open the directory :\n %s \n * %s => Scan Abort\n",
                             CurrPath.Path(), /* PathString, */ strerror( errno ) );

    LogF.Write( "%s", ErrMsg );

//  fprintf( stderr, "\n  * DiaViewer Cannot open the directory :\n %s \n * %s => Scan Abort\n",
//           CurrPath.Path(), /* PathString, */ strerror( errno ) );

    CurrPath.Set( NULL, 1 );    // Restore the Directory separator.
    return NULL;
  }

  CurrPath.Set( NULL, 1 );      // Re-append the final "/" and keep length to restore original path...
//basplen = CurrPath.Length();

//printf( " Our Path = (%d)\"%s\"\n", CurrPath.Length(), CurrPath.Path() );

  lidir = NULL;                 // Initialize the sub-directories and ...
  lisli = NULL;                 // ... diapositives list.


// Scan this directory.
  while (dsc=readdir(dir)) {    // Read a directory entry.
    if (dsc->d_name[0]!='.') {  // Ignore all entry as ., .. and special dir/files.
      // Complete the path to get access to the entry (File or Directory).
      CurrPath.Set( dsc->d_name, 0 );

//printf( " New path <%s>\n", CurrPath.Path() );

            // Check for directory entry category (skip when no file or directory).
#if defined( _WIN32 ) && !defined( _CYGWIN_ )
      if (stat( CurrPath.Path(), &entbf ))
#else
      if (lstat( CurrPath.Path(), &entbf ))
#endif
        // One entry access error.
        LogF.Write( " (l)stat error %s\n on entry \"%s\"\n", strerror( errno ), CurrPath.Path() );

      else {    // Entry Access is OK.
        switch (entbf.st_mode & S_IFMT) {
          case S_IFDIR: { // For directory.
            CurrPath.Set( NULL, 1 );
            curr_deep++;
            if (nwdir = ScanFoundDir( dsc->d_name )) {
              // Append this directory in the directory list.
              lidir = NewEntry( (void*)nwdir, lidir );
              // Update the diapositive and directory counts.
              ncdir += nwdir->NDir() + 1;
              ncsli += nwdir->CnSli();
              ndir++;
            }
            curr_deep--;
          }
          break;

          case S_IFREG: { // For regular file.
            if (nwsli = SliInList( dsc->d_name )) {
              // Put the new diapositive in the directory diapositive list.
              lisli = NewEntry( (void*)nwsli, lisli );
              nsli++;
              ncsli++;
            }
          }
          break;

          default: ;   // Ignored entry.
        }
      }
      CurrPath.Rem();  // Restore Scanned directory path with its "/".
//    printf( " New path <%s>\n", CurrPath.Path() );
    }
  }

//printf( " fin dir scan\n" ); fflush( stdout );
  closedir( dir );

  // Do not keep any directory without diapositives inside (can be in subdirectories).
#ifdef UNDEF
  if (ncsli && ndir == 1 && !nsli) {
      // The current directory include a unique sub-directory but no slide.
      nwdir = (DirEntry*)lidir[0];
      CurrPath.Set( nwdir->FName(), 1 );
      delete[] nwdir->FName();
      FName( CurrPath.Path() );
      // The owner will be the owner of this current. (for RootDir the VolEntry).
      TreeUpdateDeep( nwdir, curr_deep );
      FreeEntryTable( lidir );
  }
  else
#endif


  if (ncsli) { // When some slides was found :
    DirFlg = 1; // To a directory sort.
    if (ndir>1) EntrySort( lidir->tab, ndir ); // Directories sorting.
    DirFlg = 0; // To a slides sort.
    if (nsli>1) EntrySort( lisli->tab, nsli ); // Slides sorting.

    nwdir = new DirEntry( name, 0 );

    if (lidir) {
      // The SdDir() call set the sub-directory table and for each set the Owner_ link.
      nwdir->SbDir( lidir->use, (DirEntry**)(lidir->tab) );
      FreeEntryTable( lidir );
    }
    if (lisli) {
      nwdir->Slide( lisli->use, (SliEntry**)(lisli->tab) );
      FreeEntryTable( lisli );
    }
        
    nwdir->LDeep( curr_deep );
    nwdir->CnSli( ncsli );

    if (!curr_deep) {
      curr_tdir = ncdir + 1;
      curr_tsli = ncsli;
    } else {
      if (curr_deep > maxi_deep) maxi_deep = curr_deep;
    }

    // Try to read a set all directory and slide context informations (old version file).
    CurrPath.Set( CNTX_FNAME, 0 );
//printf( " Try to get context\n" ); fflush( stdout );
    ideep = nwdir->ReadCntxFile( CurrPath.Path() );
    if (!deep_error) deep_error = ideep;

    CurrPath.Rem();
//printf( " Read context file done for \"%s\".\n", CurrPath.Path() ); fflush( stdout );
  } else nwdir = NULL;
//printf( " ZZZ\n" ); fflush( stdout );
//PRT( " v Fin %d\n", curr_deep );
  return nwdir;
} // static DirEntry * ScanFoundDir( const char * name ).



static VolEntry * CreateVolume( const char    * path,
                                DirEntry      * root,
                                int           newdir,
                                VolRef * vref = NULL
                               )

// Function to create a new Volume of slides (diapositives).
// The name argument is the volume root directory.
// If <newdir> is true (!=0), a new volume directory is created
// with the path <name>. else the directory access is checked
// for read access.
//
{
  int            flg = 1;
  VolEntry  * vol = NULL;
	
  if (newdir) { // We create a new slides volume directory.
    flg = fl_mkdir( path, 0 );
  } else {      // We use an existing slides volume directory.
    flg = fl_access( path, R_OK );
  }
		
  if (!flg) {
    // The creator of VolEntry put the Volume reference in the table OpenVolTable.
    vol = new VolEntry( path, root, vref );     // Allocate the Volume Entry.
    // Now, complete the volume initialization.
    vol->VRoot( root );
    vol->TDeep( maxi_deep );
    vol->TnDir( curr_tdir );
    vol->TnSli( curr_tsli );
  }
  maxi_deep = 0;        // Reset the deep maximum for a next volume creation.
  curr_tsli = 0;
  curr_tdir = 0;
  return vol;
} // CreateVolume( const char * name, DirEntry * root, VolRef * vref = NULL ).



static void SetVolumeIdents( VolEntry *vol )
// Set the Unique integer identifier for a complete volume.
{
  if (vol) {
    vol->SetVolIdents();    // Setup the volume identifier.
    if (IIdeOverflow)  {
      if (IIdeOverflow&1) fl_alert( "\n  * DiaViewer : Too many slides.\n" );
      if (IIdeOverflow&2) fl_alert( "\n  * DiaViewer : Too many slides'directories.\n" );
      if (IIdeOverflow&4) fl_alert( "\n  * DiaViewer : Too many slides'volumes.\n" );
      // Actions correctrices !!!

    }
  }
} // static void SetVolumeIdents( VolEntry *vol ).



int CreateVolumeTree( VolRef * vref )
// Scan and create the diapositives (Slides) tree list.
// The total number of found diapositives is returned (<0 on error or 0 for empty slide volume).
// The procedure return this OpenVolTable number index of the volume or -1 on fatal error.
//
{
    DirEntry            * root;
    VolEntry            *  vol;
    const char       * dirfnam;

    dirfnam = vref->path;
    CurrPath.Clear();           // Force new path to be built.
    CurrPath.Set( dirfnam, 1 ); // Set PathString initial value for a Directory.
    maxi_deep              = 0; // Initialize the directory maximum deep ...
    deep_error             = 0; // ... and deep error flag.

    root = ScanFoundDir( dirfnam );     // Scan the directory.
    // Here the PathString is restored at dirfnam initial value.

//  printf( " DCM fdir Root? %d\n", root? 1 : 0 );

    if (root) {
        vol = CreateVolume( dirfnam, root, 0, vref );
        vol->SetVolContext();           // Set the Volume specific data.
        SetVolumeIdents( vol );         // Set all vol/dir/slide integer identifiers.
        SlidePrg * prg = new SlidePrg( vol );

        prg->Display();

        vol->RelPrg( (void*)prg );
    } else vol = NULL;

    CurrPath.Clear();                   // Clear the Path.

    if (deep_error) {
        // Positive value : The volume include an other one..
        // Negative value : The volume is include in an other one.
        fl_alert(
            " *** DiaViewer Overlay Error on the volume :\n Path = \"%s\"\n Read only use recommanded",
            vol->VRoot()->FName()
        );
    }
    LogF.Write( " Open Volume %s : ID %3d, Path = %s\n",
                (vref->name)?vref->name:"?", vref->iide, (vref->path)?vref->path:"<>" );

    return (root)? OpenVolTable.TabUse(): -1;
} // static int CreateVolumeTree( VolRef * vref ).



static void CleanWindowsTable( int * idt )
// To suppress the window of volume reference # idt.
// idt is the OpenVolTable indexies table for each VolToOpen table element.
//
{
    int   ip,   ir,   iv,  iw,   nv;
    int * jdt;

    nv = OpenVolTable.TabUse();                 // Get the number of opened slides volumes.
    // Create an init the volume table use.
    jdt = new int[nv];
    for(iv = 0; iv < nv; iv++) jdt[iv] = 0;

//printf( " Clr A idt[] : %d, %d\n", idt[0], idt[1] );
    if (WinTable.size) {
        // Loop for examine the View(s) to create table and ...
        // ... flags each window without opened volume reference.
        for( iw = 0; iw < WinTable.size; iw += 4) {
            ip = WinTable.tab.i[iw] - 1;        // Get the original VolToOpen table index of this View ...
            iv = idt[ip];                       // ... and the related OpenVolTable index (or -1).
//printf( " Clr B L iw=%d, ip=%d, iv=%d, jdt[iv]=%d\n", iw, ip, iv, jdt[iv] );
            if (iv < 0) {                       // When we have a volume Open Error ...
                iv = -iv - 1;                   // Restore the Opened Volume index.
                if (iv < 1 && nv > 0) iv = 1;   // Select an opened volume when possible.
                if (jdt[iv] > 1)                // We should supress this View.
                    WinTable.tab.i[iw]   = -1;  // Mark window to suppress.
                else {
                    WinTable.tab.i[iw+1] =  0;
                    WinTable.tab.i[iw+2] = -1;  // Force the initial position to be ...
                    WinTable.tab.i[iw+3] = -1;  // ... the position saved in Volume context.
                }
            } else {
                jdt[iv]++;                      // Flag for normal View Volume ref.
            }
            WinTable.tab.i[iw+0]   = iv;        // Set the OpenVolume index volume.
//printf( " Clr M L iw=%d, iv=%d, jdt[iv]=%d\n", iw, iv, jdt[iv] );
//printf( " Clr M LW Win : %d, %d, %d, %d\n", WinTable.tab.i[iw+0],
//          WinTable.tab.i[iw+1], WinTable.tab.i[iw+2], WinTable.tab.i[iw+3] );
        }
    }

//printf( " Clr T\n" );
    // View scan to compress window table.
    ir = 0;
    for(ip = 0; ip < WinTable.size; ip) {
        if (!(ip % 4) && (WinTable.tab.i[ir] < 0)) ir += 4;
        if (ir != ip) WinTable.tab.i[ip] = WinTable.tab.i[ir];
        ip++; ir++;
    }
    WinTable.size = ip;                         // Adjust the Window view table size.
//printf( " Clr Z\n" );
} // static void CleanWindowsTable( int * idt ).



void DisplayAllImageTree();



static void VolVersionTransfert()
// Manage the transition with the verions 1.2. where the volume path list is ...
// ... keep in the specific file ~/.DiaViewer/DVW_Vol_List.Dvl .
{
    int        ij,  ll;
    char          * vp;

    ij = 0;
    ll = VolP2Open.size;                // Get the number of volume to open.
    VolToOpen.tab.i = new int[ll];      // Allocate the VolToOpen table.
    VolToOpen.size = ll;
    for(int ii = 0; ii < ll; ii++) {    // For each volume to open ...
        vp = VolP2Open.tab.s[ii];       // Get the Volume path.
        if (vp) {                       // Protection for empty !
            VolP2Open.tab.s[ii] = NULL; // To previous a string delete on VolP2Open delete.
            VolList.Append( vp );       // Allocate a volume reference and set it..
            VolToOpen.tab.i[ij++]=ii+1; // Set this volume for automatic open.
        }
    }
    delete[] VolP2Open.tab.i;           // Delete the old standard volume to open list.
    VolP2Open.size = 0;
} // static void VolVersionTransfert().



void OpenVolumeQuList()
// Scan the volume to open table VolToOpen and open each ones.
//
{
    int    nvol, ide, idn, idv;
    VolEntry     * svol = NULL;
    const char  * path, * vnam;
    int                  * idt;
    int     ie,  ii, oldfl = 0;
    VolRef               * rec;
    VolEntry           ** vlst;

    svol = MemVolume();                 // Create the memory volume.

    if (VolP2Open.size) {
        oldfl = 1;
        VolVersionTransfert();          // When old version setup we build the new VolList ...
    }
    else {
        ie = VolList.Read();            // ... else we get the actual Volume list.
        if (!VolList.ListSize()) {      // When the list of volume is empty ...
            VolMN_StartDef();           // Request the User to define a first Slides Volume.
            if (VolList.ListSize()) {   // If it is OK, we flag it for open.
                VolToOpen.tab.i = new int[1];
                VolToOpen.size = 1;
                VolToOpen.tab.i[0] = 1;
            }
        }
    }

    // Here, the list of known slides volume (reference) is in the memory list VolList.

    if (VolList.ListSize()) {           // For a minimum of One Volume is defined ...
        if (!VolToOpen.size) {          // Some volume are defined but no Volume selected to be open.
            VolToOpen.tab.i=new int[1]; // We take the first one.
            VolToOpen.size = 1;
            // Get the integer identifier from the first Slide Volume Reference.
            VolToOpen.tab.i[0] = VolList.IdxVolRef( 0 )->iide;
        }
        if (!WinTable.size) {           // When we have no window(s) table, we create one ...
            WinTable.tab.i=new int[4];  // ... for a first window to display the first opened volume.
            WinTable.size = 4;
            WinTable.tab.i[0] = 1;      // Here, we give the index in the VolToOpen table.
            for(ii = 1; ii < 4; ii++)  WinTable.tab.i[ii] = 0;
        }
    }

    // Now, the table VolList is the Known Volume reference list, ...
    // ... and the VolToOpen is the list of volume to open index in VolList.

    idt = new int[VolToOpen.size + 1];  // Table of OpenVolTable from VolToOpen table. Maximum possible !
    for(ii = 0; ii <= VolToOpen.size; ii++) idt[ii] = 0;
    idn = 0;                            // Init the count for open volume error(s).

    if (nvol = VolToOpen.size) {        // Get the number of Volume to open.
        for(ii = 0; ii < nvol; ii++) {
            ide = VolToOpen.tab.i[ii];  // Get a Volume integer identifier and then locate ...
            rec = VolList.Locate( ide );        // ... and get its volume reference record.
            if (rec) {
                idv = CreateVolumeTree( rec ) - 1;
                // idv is the index (in OpenVolTable) of the new slide volume ...
                // ... or -1 when an open error occurs.
                if (idv < 0) { // The Slides Volume loading is not a success.
                    // Send an Alert message and remove the Volume.
                    fl_alert( "Open Slide Volume Error %d :\n%s", idv, ErrMsg );
//                  fl_alert( "Open Slide Volume Error %d : \n path = %s with message\n *** %s\n",
//                            -idv, path );
                    // Select the previous opened volume or memory volume.
                    idv = -(OpenVolTable.TabUse());
                    idn++;              // To signal some volume open error.
                }
            }
            idt[ii] = idv;              // Keep corresponding index in OpenvolTab.
        }
    }
    if (idn) {
      ie = fl_choice( " *** DiaViewer cannot get access to some defined slides volume(s) :\n *** %s\n *** %s",
                      "Try to continue", "Stop DiaViewer", 0,
                      "You can stop DiaViewer (to take some preventive actions (ex: mount a device),",
                      "or try to continue the DiaViewer execution (but some parts of setup can be lost.) :");
      if (ie) exit( 1 );
    }
    delete[] VolToOpen.tab.i;  // Now we can free the Volume to open table and we protect ...
    VolToOpen.tab.i = NULL;    // ... ourselves from possible deallocation from a garbage collector.
//printf( " M idt[] %d : %d, %d\n", VolToOpen.size, idt[0], idt[1] );

    if (idn) CleanWindowsTable( idt );  // When some open error was detected.
//printf( " R\n" );

  // When we use the old DiaViewer env., we create the first Volume list.
    if (oldfl) {
        VolList.CheckAllNames();
/*      nvol = OpenVolTable.TabUse();
        vlst = (VolEntry**)OpenVolTable.GetObjs();
        for(ii = 1; ii < nvol; ii++) {
            vnam = vlst[ii]->VName();
            if (!vnam) vnam = VolList.DefName();
            vlst[ii]->VRefer()->name = strdup( vnam );
        }
*/
        VolList.Write();
    }
    delete[] idt;
    delete[] VolToOpen.tab.i;

    for(ii = 1; ii <= nvol; ii++) {
        rec = VolList.Locate( ii );
        LogF.Write( " Find Volume <%s> : %02d, \"%s\"",
                    (rec->name) ? rec->name : "*", rec->iide,
                    (rec->path) ? rec->path : "*" );
    }
    IniSelSlide = NULL;
} // void OpenVolumeQuList().





//
// **********************************************************************************
//


static void PrtSpace( int n ) {
    for (int ii=0; ii<n; ii++) putchar( ' ' );
} // static void PrtSpace( int n ).



void DisplayDirectory( DirEntry * dir, int nsp )
{
    DirEntry               * sdi;
    SliEntry               * sli;
    int    ndir,  nsli,  nbr = 0;

    if (dir) {
        ndir = dir->NDir();
        nsli = dir->NSli();
    
        PrtSpace( nsp ); printf( "Directory %8xH/ \"%s\" (lvl=%d) :\n",
                                 dir->Ident(), dir->FName(), dir->LDeep() );
        if (dir->SText()) { PrtSpace( nsp ); printf( "* title \"%s\"\n", dir->SText() ); }
        PrtSpace( nsp );
        printf( "* with %d subdirectorie(s) and %d slide(s) or a total number of %d slide(s) \n",
                ndir, nsli, dir->CnSli() );
        // Edit the list of diapositives.
        nsp += 2;

        for(int i=1;i<=ndir;i++) {
            sdi = dir->SbDir( i );
            if (sdi) DisplayDirectory( sdi, nsp );
        }
        for(int i=1;i<=nsli;i++) {
            sli = dir->Slide( i );
            if (sli) {
                PrtSpace( nsp );
                printf( "%3d:%8xH/ %s\n", nbr++, sli->Ident(), sli->FName() );
                printf( " * fl %d, or1 %d, or2 %d, sn %d, fn %d, cn %d, no %d\n",
                        sli->FlgVal(), sli->OriOpe(), sli->CurOpe(), sli->Numb(),
                        sli->Film(), sli->Chap(), sli->Note() );
            }
        }
        PrtSpace( nsp-2 ); printf( "****\n" );
    }
    else
        printf( "/n *** error: The directory pointer is NULL ***\n" );
} // void DisplayDirectory( DirEntry * dir, int nsp ).



void DisplayImageTree( VolEntry* vol )
{
    printf( "\n List of Diapositives Tree \"%s\":\n\n", CurrPath.Path() );

    if (vol) {
        const char * vnam = vol->VName();

        if (vnam) printf( " Volume Name \"%s\"\n", vnam );

        DisplayDirectory( vol->VRoot(), 1 );
		
        printf( "\n There are a total of %d slides in %d directories\n\n", vol->TnSli(), vol->TnDir() );

        printf( "***\n\n" );
    } else printf( "\n Slide Volume was not existing.\n" );
} // void DisplayImageTree( VolEntry* vol ).



void DisplayAllImageTree()
{
    for(int i = 0; i < OpenVolTable.TabUse(); i++) {
        DisplayImageTree( (VolEntry*)(OpenVolTable.GetObj( i )) );
    }
} // void DisplayAllImageTree().




//
// end of <PW-Id>.
//
