//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module  to scan a directory tree.
//

//#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdarg.h>

#include "Service_DIR.h"


static const int MiMa = 'a' -'A';

//
// Scan Error Manager.
// When no user catch error is defined the message is output to stdout.
//
int  SrvDir::Error_Manager( int ie, const char * frm, ... )
{
    va_list                 ap;
    int                      n;
    char   buf[512],  fmt[128];
    const char * mne = "SrvDir : To Many (%d) Error => Stop scan on Fatal errors.";

    snprintf( fmt, 127, " *** SrvDir%sError %3d : %s", (ie < 0)? " Fatal ": " ", ie, frm );

    va_start( ap, frm );
    ::vsnprintf( buf, 511, fmt, ap );
    va_end( ap );

    if (ehdl_) ie = ehdl_( ie, buf, data_ ); else fprintf( stderr, "%s\n", buf );

    if (ie > 0 && ++nerr_ > MAX_ERROR) {
        snprintf( buf, 511, mne, MAX_ERROR );
        if (ehdl_) ie = ehdl_( -1, buf, data_ ); else fprintf( stderr, " %s\n\n", buf );
    }
    ierr_ = ie;
    if (ie < 0) Clean_Stack();
           else ierr_ = 0;      // Does not stop process on not fatal error.
    return ierr_;
} // int  SrvDir::Error_Manager( int ie, const char * frm, ... ).



// Small routine to reject any directory entries as "." or ".." and hidden file (if not specified).
int  SrvDir::Check_FName( const char * nm ) {
    int ok = 1;

    if (nm && nm[0]) {
        if (nm[0] == '.') {
            if (!nm[1] || (nm[1] == '.' && !nm[2])) ok = 0; // "." or ".."
            else ok = (flgs_&SDir_Hidd) ? 1 : 0; // Hidden files.
        }
    } else ok = 0;
    return ok;
} // int  SrvDir::Check_FName( const char * nm ).



int  SrvDir::Open_Dir( const char * name )
{
    DIR          * dir;

    if (llvl_ >= MAX_LEX_DIR-1) {
        ierr_ = Error_Manager( -2, "Directory Stack overflow for path :\n\tpath = \"%s\"\n", Path() );
        stat_ = Stat_Stop;
        return 0;
    }
//  if (llvl_ >= 0) Set( name, 0 );
    if (dir = opendir( Path())) {       // Open the directory.
        dirf_[++llvl_]  = dir;          // On success, set file DIr channel in the stack ...
        Set( 0, 1 );                    // ... and Set a directory separator.
        if (ulvl_) *ulvl_ = llvl_;
        return llvl_;
    } else {
        if (flgs_&SDir_PrDi) {
            ierr_ = Error_Manager( -3, "Cannot open the directory :\n\tpath = \"%s\",\n\tSysMsg : %s\n",
                                       Path(), strerror( errno ) );
            stat_ = Stat_Stop;
            return 0;
        } else return llvl_;
    }
} // int  SrvDir::Open_Dir( const char * name ).



int  SrvDir::Close_Dir()
{
    if (llvl_ >= 0) {
        closedir( dirf_[llvl_--] );
        return -1;
    }
    return -2;
} // int  SrvDir::Close_Dir().



void SrvDir::Clean_Stack()
{
    while (llvl_ >=0) Close_Dir();
    if (ulvl_) *ulvl_ = llvl_;
} // void SrvDir::Clean_Stack().



int  SrvDir::Get_Entry()
// Read a valid (conform with filter) directory entry :
// return Fnd_File for a regular file entry,
//        Fnd_Dire or Fnd_Dirt for a directory entry,
//        Fnd_Eodi for a directory EOF,
//        Fnd_Erro for the end of scan.
//
// After call, *name_ is the entry filename.
//
{
    static struct dirent * dsc;
    static struct stat   entbf;
    int          re = Fnd_Null; // Fnd_Null == 0
    int       ie = 0,   ok = 0;

    while (re == Fnd_Null) {
        dsc = readdir( dirf_[llvl_] );          // Read a directory entry.
// if (dsc) printf( " Seen \"%s\"\n", dsc->d_name );
        if (!dsc) break;                        // Stop if End Of Directory (EOD).
        if (name_) delete[] name_;              // Set or Update the terminam filename.
        name_ = strdup( dsc->d_name );
        if (Check_FName( name_ )) {             // Perform check to skip "." and "..".
            Set( name_, 0 );                    // Append the entry name to path.

#if defined( _WIN32 ) && !defined( _CYGWIN_ )
            if (ie = stat( Path(), &entbf ))    // Get the entry kind.
#else
            if (TstFlg( SDir_SLnk )) ie = stat( Path(), &entbf );
                                else ie = lstat( Path(), &entbf );
#endif
            if (ie) { // in stat/lstat error.
                ierr_ = Error_Manager( -4, "%s\n\ttpath = \"%s\", \n\tSysMsg : %s\n",
                                           "(l)stat error on directory entry :",
                                           Path(), strerror( errno ) );
                re = Fnd_Erro;
            } else { // Ok for this entry.
                switch(entbf.st_mode & S_IFMT) {
                    // select the kind of directory entry.
                    case S_IFDIR: {
                        ok = Filter( 2, name_ );
                        if (ok) re = (ok == 1) ? Fnd_Dire : Fnd_Dirt;
                        else Rem();
                        break;
                    }
                    case S_IFREG: { // For regular file.
                        ok = Filter( 1, name_ );
                        if (ok) re = Fnd_File;
                        else Rem();
                        break;
                    }
                    default: ;                  // Ignored entry.
                } // switch ...
            } // if (ierr_) ... else
        } // if (Check_FName(dsc->d_name) ...
    } // while ...

    if (ierr_) return Fnd_Erro;
    if (dsc) return re;                         // Success on an entry.
        else return Fnd_Eodi;
} // int  SrvDir::Get_Entry().



void SrvDir::Init( const char * root, int flgs )
{
    if (root && root[0]) {
        Set( root, 0 );
        stat_ = Stat_Init; llvl_ = -1;
        if (ulvl_) *ulvl_ = llvl_;
        name_ = strdup( Path() );
        flgs_ = flgs & (SDir_NCEx | SDir_NCas | SDir_Hidd | SDir_SLnk | SDir_PrDi);
    }
} // void SrvDir::Init( const char * root, int flgs ).



int  SrvDir::Scan()
// Perform a scan until one file/Directory is selected by calling the current filter.
// Return 1 for a file, O for a directory and -1 at the end of scan.
// When an access is not possible (for file or directory) the Error Handler is called.
//
{
    olds_ = news_;
    switch (stat_) {
        case Stat_Init:
            olds_ = Scan_Null;
            Open_Dir( Path() );
            stat_ = Stat_Rdir; news_ = Scan_IDir;
        break;

        case Stat_Rdir: // When we start the scan.
            if (olds_ == Scan_File ||
                olds_ == Scan_DiRf) Rem();
            if (olds_ == Scan_Eodi && ulvl_) { Rem(); *ulvl_ = llvl_; }
            switch (Get_Entry()) {
                case Fnd_File:
                    news_ = Scan_File;
                    break;
                case Fnd_Dire:
                    Open_Dir( Path() );
                    news_ = Scan_IDir;
                    break;
                case Fnd_Dirt:
                    news_ = Scan_DiRf;
                    break;
                case Fnd_Eodi:
                    Close_Dir();
                    if (llvl_ < 0) stat_ = Stat_Stop;
                    news_ = Scan_Eodi;
                    break;
                case Fnd_Erro:
                default:
                    stat_ = Stat_Stop;
                    news_ = Scan_Erro;
                    break;
            }
        break;

        case Stat_Stop:
        default: news_ = Scan_Stop;
        break;
    }
    return news_;
} // int  SrvDir::Scan().



void SrvDir::SetFltExt( const FTypLst extb ) {

    char         * str;
    char            ch;
    int   ii,  jj,  ln;


    while ((extb[ln])) ln++;  // Evaluate the size of extensions size.
    if (typls_) {
        while ((str = typls_[ii++])) { if (str) delete[] str; }
        delete[] typls_;
    }
    typls_ = new char *[ln+1];
    for(ii = 0; ii < ln; ii++) {
        str = strdup( extb[ii]);
        if (TstFlg( SDir_NCEx )) {
            jj = 0;
            while ((ch = str[jj])) {
                if ((ch >= 'a') && (ch <= 'z')) ch -= MiMa;
                str[jj] = ch;
            }
        }
        typls_[ii] = str;
    }
    typls_[ln] = 0;
} // void SrvDir::SetFltExt( DirFiltre_t flt ).



// Small Filter member to check matching of filename with accepted filetypes.
int  SrvDir::Filter( int kn, const char * nm ) {
    const char * p = nm + strlen( nm );
    char  ch, buf[16];
    int ii = 0;

    if (kn != 1) return 1;              // No filter only for file.

    while (p != nm && (*p) != '.') p--; // Loop to find the last period of nm.
    if (p == nm) p = " " ;              // " " is used to design a null extension.
    else if (p[1]) p++;                 // for filename end by '.' it is the extension ".".
    ii = 0;
    while ((ch = *(p++)) && ii < 15) {
        if (TstFlg( SDir_NCEx )) { if ((ch >= 'a') && (ch <= 'z')) ch -= MiMa; }
        buf[ii++] = ch;
    }
    buf[ii] = 0;
    ii = 0;
    // Loop to check for each file extension.
    while (typls_[ii]) {
        if (!strcmp( typls_[ii], buf )) break;
        ii++;
    }
    if (typls_[ii]) { exti_ = ii; return 1; }
               else { exti_ = -1; return 0; }
} // int SrvDir::Filter( int kn, const char * nm ).



// ------------------------------------------------------------------------------------------------

// #include "Service_DIR.h"

#ifdef DIR_DEBUG

typedef struct {
    int nbd, nbf;
} Stat_Rec;

Stat_Rec stat_tb[MAX_LEX_DIR];  // The size is equal to maximum authorized directory deep.



static int     ncol =   1;
static int dir_nl, fil_nl,
           dir_nt, fil_nt;




static void PrtSpace( int n ) {
    for (int ii=0; ii<n; ii++) printf( "   " );
} // static void PrtSpace( int n ).



//
// Our specific routines called by sca_dir.Scan() :
//
static int Err_Handler( int ie, const char * msg, void * d ) {

    printf( "\n\n *** Our handler called with the following message\n" );
    printf( msg ); printf( "\n\n" );
    return ie;
} // static int Err_Handler( int ie, const char * msg, void * d ).






// g++ -DDIR_DEBUG -o dirscan Service_DIR.cxx Service_Uti.cxx
//
// use with the command  "./dirscan [-f<flags>] <base_dir> <t1> <t2> t3> ...
//
// were flags 0 the default, 1 = Dir_MLink, 2 = Dir_SLink, 3 = Dir_MLink|Dir_SLink
// <t1>, <t2> ... the different selected file types.
//

int main( int argc, char ** argv ) {
    // Default we scan to look for these file extensions :
    const char * dxtb[] = { "jpeg", "JPEG", "jpg", "JPG", "png", "PNG", "bmp", "BMP", 0 };
    const char * dxtc[] = { "JPEG", "JPG", "PNG", "BMP", 0 };

    SrvDir                   sca;
    int       ii = 0,     ir = 0;
    char           * arg  =    0,
         * root = strdup( "./" );
    const   char *     tbext[65];
    int  hlp= 0, flg= 0, nar= -1,
                         mod = 0;

    int           lvl; // Local copy of sca.Level()
    const char * path, // Local copy of sca.Path() ...
               * name; // ... and sca.Name().

    int nd, nf;

    tbext[0] = 0;
    for(ii = 1; ii < argc; ii++) {
        arg = argv[ii];
        if (arg && arg[0]) {
            if (arg[0] == '-') {
                switch (arg[1]) {
                    case 'e': case 'E': flg |= SrvDir::SDir_NCEx; break;
                    case 'c': case 'C': flg |= SrvDir::SDir_NCas | SrvDir::SDir_NCEx; break;
                    case 'm': case 'M': flg |= SrvDir::SDir_Hidd; break;
                    case 'l': case 'L': flg |= SrvDir::SDir_SLnk; break;
                    case 'p': case 'P': flg |= SrvDir::SDir_PrDi; break;

                    case 'H': case 'h': case '-': hlp = 1; break;
                    default: ir = 1; // Unkown option.
                }
            } else {
                if (nar < 0) {
                    root = arg;
                    nar++;
                } else tbext[nar++] = arg;
            }
        }
    }

    if (ir || hlp) {
        printf( "    Use Service_DIR program by the command :\n\t %s %s\n\n",
                argv[0], "[-f<flags>] <base_dir> <t1> <t2> ..." );
        printf( " Where :\n\t<flags> is 0, 1 or 2 for nothing, SDir_NCas, SSDir_SLink\n" );
        printf( "\t<base_dir> is the root directory of the files tree to examine\n" );
        printf( "\toptions                   meaning\n");
        printf( "\t -e  -E\tCase not taken into account for file extension\n" );
        printf( "\t -c  -C\tCase not taken into account for filename\n" );
        printf( "\t -m  -M\tHidden file/directory as take into account\n" );
        printf( "\t -l  -L\tTo Follow the symbolic links\n" );
        printf( "\t -p  -P\tNot authorized access error ignored.\n" );
        printf( " As result %s will print the list of path selected files\n\n", argv[0] );
        printf( " The option -h will print this text.\n\n" );

        return 0;
    }

    if (nar <= 0) {
        nar = 0;
        if (flg&SrvDir::SDir_NCEx) while (dxtb[nar]) { tbext[nar] = dxtc[nar]; nar++; }
                              else while (dxtb[nar]) { tbext[nar] = dxtb[nar]; nar++; }
    }
    tbext[nar] = 0;

    printf( " We look for files with extensions in  %s :\n", root );
    ii = 0;
    while (tbext[ii]) {
        printf( "  %2d / %s\n", ii+1, tbext[ii] );
        ii++;
    }
    printf( "\n\n" );

    dir_nt = fil_nt = 0;

    // Now we can work !
    if (!ir) {
        sca.ErrHdl( Err_Handler );
        sca.SetFltExt( (SrvDir::FTypLst)tbext );
        sca.UsrLevel( lvl );
        sca.Init( root, flg );

        ir = 1;
        do {
            switch (sca.Scan()) {
                case SrvDir::Scan_IDir: { // Begin of a directory or sub-directory.
                    stat_tb[lvl].nbd = stat_tb[lvl].nbf = 0; // init file and dir count of this directory..

                    printf( "%3d    ", lvl ); PrtSpace( ncol );
                    printf( "*BD* We enter in the directory %s\n", sca.Name() );
                    printf( "       " ); PrtSpace( ncol );
                    printf( "with path is now \"%s\"\n", sca.Path() );
                    ncol += 2;
                }
                break;

                case SrvDir::Scan_File: { // Regular file.
                    nf = ++(stat_tb[lvl].nbf);  // Incrment the count of selected file.

                    printf( "   %3d", nf ); PrtSpace( ncol );
                    printf( " *F* ->  we select the file \"%s\"\n", sca.Path() );
                }
                break;

                case SrvDir::Scan_DiRf: { // Sub-Directory found.
                                          // For non scanned directory, we count this as a reference.
                    nd = ++(stat_tb[lvl].nbd);

                    printf( "%3d   ", lvl ); PrtSpace( ncol );
                    printf( " Sub directory \"%s\"\n", sca.Path() );
                }
                break;

                case SrvDir::Scan_Eodi: { // End of (sub-)directory.
                    // Get the count of new selected directories and files.
                    dir_nl = stat_tb[lvl].nbd; fil_nl = stat_tb[lvl].nbf;

                    printf( "%3d%3d", lvl, fil_nl ); PrtSpace( ncol );
                    printf( " *ED* We have %d directories and %d files was selected\n", dir_nl, fil_nl );
                    printf( "      -> " ); PrtSpace( ncol );
                    printf ( "    We return to previous directory at the deep %d\n", lvl-1 );

                    // When these count are null, we don't count the current directory.
                    if (fil_nl > 0 && lvl > 0) stat_tb[lvl-1].nbd++;
                    // Update the total count of files and directories.
                    dir_nt += dir_nl; fil_nt += fil_nl;
                    ncol -= 2;
                }
                break;

                case SrvDir::Scan_Erro: printf( "\n\n *** Fatal error was found.\n\n" );

                case SrvDir::Scan_Stop:
                default:  ir = 0;
            }
        } while (ir );

        printf( "\n\n Complete statistics : We have found %d directories and %d files.\n\n", dir_nt, fil_nt );
    }
    return 0;
} // int main( int argc, char ** argv ).




#endif

//
// end of <PW-Id>.
//

