//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers Software
//

//
// Module to scan slides in a directory
// Last Revision 2022-07-21
//

#include <string.h>
#include <dirent.h>

#include <FL/Fl.H>
#include <FL/fl_ask.H>

#include "SetUpMan.h"
#include "Service_Util.h"
#include "Service_IO.h"
#include "Service_FLOCK.h"
#include "Service_TRAP.h"

#include "DiaViewer_VRFL.h"
#include "DiaViewer_DCM.h"
#include "DiaViewer_GL.h"


//
// Main SetUp file variables.
//

// File name of the known slide volumes list.
#define VolListFName "DVW_Vol_List.Dvl"

// The Log file must be global.
LogFile LogF( "DiaViewer.log", 1 ); // The Log File of DiaViewer is masked.

char              MsgBuf[1024]; // Buffer for any message.


int win_dx = 860, win_dy = 715; // Default Window size.

int                     svargc; // To keep the task arguments number ...
char                 ** svargv; // ... and the task arguments list.


SetUpDynTab
    VolToOpen    = { 0, NULL }, // List of index of volume to open (integers).
    VolP2Open    = { 0, NULL }, // To keep the volume list to open (strings), temporary for compatibility.
    WinTable     = { 0, NULL }; // To keep all windows (to create at start time) data.

ObjTable          OpenVolTable; // The table of all opened slides volumes, ...

// Define the Known Volume List (in memory and user env file.
VolRefList    VolList( "DiaViewer Known Slide Volume list" );


int       ScanNoLim  =       1, // Scan no limited to sub-directories limits (default yes).
          SlidePrior =       0, // Flag for scan priority for directories ...
                                // ... (for 0) or for slides if not 0.
          NativeOFil =       0; // Flag to use native openfile and directory.


FilePath        * CPath = NULL; // Define the Global Current dynamic Path.
char         * SetUpDir = NULL, // User environment directory path.
             * SetUpFil = NULL, // User Setup File path.
             * User_Dir = NULL; // The User home directory.

char      * OpenVolPath = NULL, // Default Open Volume Path.
          * UserHelpUri = NULL, // User Help HTML URI.
         * SlideEditPrg = NULL; // Slide Editor program.



//
// Local static Variables.
//

static const char  SetUpFilName[] =   "DiaViewer.conf",
                   SetUpOldName[] =  ".DiaViewer.conf";

static char *VolLstPath = NULL; // To keep the path of file of known Volume paths.


// The Global Lock file to  cancel all second DiViewer run for the same user.
static FileLock   No2Pr( "DiaViewer" );


//
// Main SetUp file definitions
//

static
    const SetUpDsc Rank_SliFrm[] = {    // RKG: Slide Filename format parameters.
        { SetUpStr,   1,  AD( slid_hde ) },     // Slide filename header string.
        { SetUpInt,   1,  AD( film_dig ) },     // Number of figures for film #.
        { SetUpStr,   1,  AD( slid_sep ) },     // Separator.
        { SetUpInt,   1,  AD( slid_dig ) },     // Number of figures for slide # in the film.
        { SetUpEof,   0,               0 }
    };

static
    const SetUpDsc Rank_MapSiz[] = {    // RKG: Slide Map parameters.
        { SetUpInt,   1,  AD( rktb_wid ) },     // Window width ...
        { SetUpInt,   1,  AD( rktb_hig ) },     // ... and high.
        { SetUpInt,   1,  AD( rktb_tsz ) },     // Font size for the text title ...
        { SetUpInt,   1,  AD( rktb_esz ) },     // ... and table elements.
        { SetUpEof,   0,               0 }
    };

static
    const SetUpDsc Volu_TxtSiz[] = {    // VMN: Volume list Display arguments.
        { SetUpInt,   1,  AD( vltb_wid ) },     // Window width ...
        { SetUpInt,   1,  AD( vltb_hig ) },     // ... and high.
        { SetUpInt,   1,  AD( vltb_tsz ) },     // Font size for the text title ...
        { SetUpInt,   1,  AD( vltb_esz ) },     // ... and table elements.
        { SetUpEof,   0,               0 }
    };

static
    const SetUpDsc Rank_DirFrm[] = {    // RKG: Folder Filename format parameters.
        { SetUpStr,   1,  AD( fold_hde ) },     // Folder/Directory filename header string.
        { SetUpInt,   1,  AD( fold_dig ) },     // Number of figures for Directory/folder #,
        { SetUpStr,   1,  AD( fold_tra ) },     // Folder/Directory filename trailer string.
        { SetUpEof,   0,               0 }
    };

static
    const SetUpDsc Rank_Sizes[] = {     // RKG: Target max. numbers of film and slides /folder.
        { SetUpInt,   1,  AD( dirslisz ) },     // Max. # of slides/folder in the target volume,
        { SetUpInt,   1,  AD( dirdirsz ) },     // Max. # of Folder/folder in the target volume.
        { SetUpEof,   0,               0 }
    };

static
    const SetUpDsc SlNb_Maxs[] = {      // RKG: maximum size in 
        { SetUpInt,   1,  AD( flmn_max ) },     // RKG: Maximum value of film number (#),
        { SetUpInt,   1,  AD( slin_max ) },     // RKG: Maximum value of slide number (#).
        { SetUpEof,   0,               0 }
    };

// List od defined Setup file identifier :
// In this list, when the field adress (vaddr) is NULL the object ...
// ... is ignored in input and never write in output.
static
    const SetUpIde MainSetUp[] = {
        {   "SCREEN_SIZES", { SetUpRec,    2,               NULL } }, // Memory of windows size.
        { "SLIDE_PRIORITY", { SetUpInt,    1,   AD( SlidePrior ) } }, // Slide scan folder priority.
        {    "SCAN_NO_LIM", { SetUpInt,    1,    AD( ScanNoLim ) } }, // Flag to skip folders limits.
        {  "PRINT_FROM_GL", { SetUpInt,    1,    AD( PrtFromGL ) } }, // Flag for Direct Screen print.
        {    "NATIVE_OPEN", { SetUpInt,    1,   AD( NativeOFil ) } }, // Flag for System Native Open use.
        {     "CACHE_SIZE", { SetUpInt,    1, AD( SliCacheSize ) } }, // Slide Cache size (=2**n).
        {  "INT_IDENT_MAP", { SetUpInt,    2,      AD( IIdeFld ) } }, // Internal Identifiers Mapping.
        { "SLIDES_NUMBERS", { SetUpRec,    2,    AD( SlNb_Maxs ) } }, // RKG: Max sizes of target Vol.
        { "SLIDE_RANK_FRM", { SetUpRec,    4,  AD( Rank_SliFrm ) } }, // RKG: Slide name Format arg.
        {   "DIR_RANK_FRM", { SetUpRec,    3,  AD( Rank_DirFrm ) } }, // RKG: Folder name Format arg.
        {    "SLIDIR_RANK", { SetUpRec,    2,   AD( Rank_Sizes ) } }, // RKG: Max sizes / folder.
        {   "OPENVOL_PATH", { SetUpStr,    1,  AD( OpenVolPath ) } }, // Last opened slides volume path.
        {    "TARGET_PATH", { SetUpStr,    1,     AD( tvol_pth ) } }, // RKG: Path to target volume.
        {    "TARGET_NAME", { SetUpStr,    1,     AD( tvol_nam ) } }, // RKG: Name of target Volume.
        {   "RANK_MAPSIZE", { SetUpRec,    4,  AD( Rank_MapSiz ) } }, // RKG: Map Sizes Arguments.
        {   "VOLM_TXTSIZE", { SetUpRec,    4,  AD( Volu_TxtSiz ) } }, // VMN: Volume List Display Arguments.
//      { "SAVE_CNTX_PERI", { SetUpInt,    1,               NULL } },
        { "VOLUMES_SELECT", { SetUpDSt,    1,    AD( VolP2Open ) } }, // Table of Slides Vol. to open (old versions).
        {    "VOL_TO_OPEN", { SetUpDIn,    1,    AD( VolToOpen ) } }, // Table of vol.s to open (index of VolKnwLst).
        {  "WINDOWS_TABLE", { SetUpDIn,    1,     AD( WinTable ) } }, // Table of Views to open.
        {  "USER_URI_HELP", { SetUpStr,    1,  AD( UserHelpUri ) } }, // User Help Uri address.
        {   "SLIDE_EDITOR", { SetUpStr,    1, AD( SlideEditPrg ) } }, // Slide Editor Program path.
//      { "WINDOWS_VOLUME", { SetUpDIn,    1,               NULL } }, // Temporary.
//      { "WINSLID_SELECT", { SetUpDIn,    1,               NULL } }, // Temporary.
        {                0, { SetUpEof,    0,                  0 } }
    };


static const char * msgtab[] = { // Error message table for the Main.
    "DiaViewer Error : Cannot find the user environment => Stop.",              // 0
    "DiaViewer Error : Cannot create directory \"%s\":\n\t%s",                  // 1
    "DiaViewer can't run twice at the same time for the same user => Stop",     // 2
    "DiaViewer Error : Cannot read the user lock file => stop.",                // 3
    "DiaViewer Error : Cannot Manage the Known Volume List \"%s\":\n\t%s",      // 4
};




void ErrorMessage( const char *msg )
// Graphic Error message routine.
//
{
    LogF.Write( msg );
    fl_alert( msg );
}




void SaveAllApplContext( VolEntry * cvol )
// Should be called by exit processing.
{
    // Write or Update the SetUp Base File.
    SetUpWrite( SetUpFil,"DiaViewer",MainSetUp );
} // SaveAllApplContext.



int main( int argc, char ** argv )
{
    int ic = 1;

    svargc = argc; svargv = argv;               // Keep task argument list for all uses.

    FSvSel.Native( &NativeOFil );               // Connect the File/dir dialog to current option.

    CPath = new FilePath();                     // Stack enough large for any event.

    CPath->PathSearch( USER_DIR );              // Get the user home directory.
    User_Dir = strdup( CPath->Path() );
    OpenVolPath = strdup( User_Dir );

    LogF.SetUp( User_Dir, 1 );                  // Init the Log file

    SetUpErrDisplayInstall( ErrorMessage );     // Install the SetMan Error Display.

    CPath->Set( NULL, 1 );
    ic = CPath->PathSearch( USER_ENV );         // Get User Environment directory.
    if (!ic) {
        ErrorMessage( msgtab[0] );
        exit( 2 );
    }

    CPath->Set( 0, 1 );                         // Force a directory separator.
    if (ic == 2) {
        CPath->Set( SetUpOldName, 0 );          // Old Configuration file ".Dia..." !!
        ic = SetUpRead( CPath->Path(), MainSetUp );  // Read the SetUp File ic = 1 si read ok.
        CPath->Rem();
        CPath->Set( ".DiaViewer", 1 );          //
        if (MkDir( CPath->Path() )) {           // Create the new SetUp/config Directory.
            snprintf( MsgBuf, 1024, msgtab[1], CPath->Path(), strerror( errno ) );
            ErrorMessage( MsgBuf );
            exit( 2 );
        }
        SetUpDir = strdup( CPath->Path() );     // Keep its complete path.
        CPath->Set( SetUpFilName, 0 );          // Set the new SetUP Config Filename ...
        SetUpFil = strdup( CPath->Path() );     // ... and keep its complete path.
        // ReWrite the SetUp File in new directory .DiaViewer when it was existing.
        if (ic) SetUpWrite( SetUpFil, "DiaViewer", MainSetUp );
    } else { // New Configuration file "Dia...".
        SetUpDir = strdup( CPath->Path() );     // Keep the path of environment directory.
        CPath->Set( SetUpFilName, 0 );          // New Configuration file "Dia...".
        SetUpFil = strdup( CPath->Path() );     // Keep
        SetUpRead( SetUpFil, MainSetUp );       // Read the SetUp File.
    }
    CPath->Rem();
    CPath->Set( VolListFName, 0 );
    VolLstPath = strdup( CPath->Path() );       // Set the Known slide volume list file path.
    CPath->Rem();

    if (CheckFilePath( &VolLstPath, 3 ) < 0) {  // Read-Write-Create access or creation ...
        snprintf( MsgBuf, 1024, msgtab[4], VolLstPath, strerror( errno ) );
        ErrorMessage( MsgBuf );
        exit( 2 );
    }
    VolList.FileSet( VolLstPath );              // Set the path of Known volume list.


    CPath->Set( "DiaProcId.dat", 0 );
    ic = No2Pr.OpenL( CPath->Path() );
    if (ic) {
        if (ic > 0) ErrorMessage( msgtab[2] );
               else ErrorMessage( msgtab[3] );
        exit( 2 );
    }
    CPath->Clear();

    CheckFilePath( &OpenVolPath, 0 );           // Check access to default volume path.
    CheckFilePath( &tvol_pth, 0 );              // Check access to target ranking volume path.

    LogF.Write( " User-Env = \"%s\"\n SetUpDir = \"%s\"", User_Dir, SetUpDir );

    OpenVolumeQuList();                         // Open all volume in VolQueue.

    if (!WinTable.size) {                       // When we have no config file (or too old) ...
        WinTable.size = 4;                      // ... we create it for one window.
        WinTable.tab.i = new int[4];
        for(int i = 0; i < 4; i++) WinTable.tab.i[i] = 0;
    }

    DiaViewer_GL::Init( 0, 1 );                 // Init the windowing system (display 0).

    return Fl::run();

    No2Pr.CloseL();
}



//
// end of <PW-Id>.
//
