//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module environement to manage the Slide Context Input/output Files
//
//


#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdarg.h>


#include <FL/Fl.H>
#include <FL/fl_ask.H>
#include <FL/fl_utf8.h>         // To manipulate utf8 strings.
#include <FL/Fl_File_Chooser.H>
#include <FL/Fl_Native_File_Chooser.H>

#include "Service_IO.h"



// Create a Global FSelect object to be used by the application.
// The main of application just must include :
//  #include "Service_IO.h"
//
// and in the main function the statement: FSvSel.Native( &flag );
// Then depending of flag :
// flag != 0 => FSvSel.FSdir() and FSvSel.FSFile() use the Native open file/dir dialog
//                                      else       use the FLTK open file/dir dialog.

FSelect FSvSel( NULL );



const char * FSelect::Dir( const char * lab,     // Label for window,
                           const char * fnm )    // Default path.
{
    static      char   path[512];
    char                 * npath;
    int                       ie;

    if (SysFlg()) {  // Native directory chooser.
        Fl_Native_File_Chooser fnfc;
        fnfc.title( lab );
        fnfc.type( Fl_Native_File_Chooser::BROWSE_DIRECTORY );
        fnfc.directory( fnm );
        switch (ie = fnfc.show() ) {
            case -1:
                fl_alert( "Error: %s", fnfc.errmsg() );
            case  1:
                return NULL;
            default:
                strncpy( path, fnfc.filename(), 511 );
        }
    } else {    // FLTK Directory file chooser.
        npath = fl_dir_chooser( lab, fnm, 0 /*Absolu path*/ );
        if (npath&&npath[0]) {
            strncpy( path, npath, 511 );
            return path;
        } else return NULL;
    }
    return path;
} // char * FSelect::Dir( const char * msg, const char * fnm ).



const char * FSelect::File( const char * lab,     // Label for window,
                            const char * fnm,     // Default path,
                            const char * flt,     // File of displayed files,
                            int          opt )    // Create mode = 1.
{
    static     char  path[512];
    char               * npath;
    int                     ie;

    if (!flt) flt = "*";
    if (SysFlg()) { // The native mode.
        Fl_Native_File_Chooser fnfc;
        fnfc.title( lab );
        fnfc.filter( flt );
        fnfc.preset_file( fnm );
        if (opt) fnfc.type( Fl_Native_File_Chooser::BROWSE_SAVE_FILE );
            else fnfc.type( Fl_Native_File_Chooser::BROWSE_FILE );
        switch (ie = fnfc.show() ) {
            case -1:
                fl_alert( "Error: %s", fnfc.errmsg() );
            case  1:
                return NULL;
            default:
                strncpy( path, fnfc.filename(), 511 );
        }
    } else {
        npath = fl_file_chooser( lab, flt, fnm, 0 /*Absolu path*/ );
        if (npath&&npath[0]) strncpy( path, npath, 511 );
                       else return NULL;
    }
    return path;
} // const char * FSelect::File( const char * lab,
  //                             const char * fnm,
  //                             const char * flt,
  //                             int          opt )

// ----------------------------------------------------------------------------



LogFile::LogFile( const char * lgn, int fl )
{
    logn_ = lgn ? strdup( lgn ) : NULL;
    path_ =          NULL;
    mskf_ =            fl;
} // LogFile::LogFile( const char * src, int fl ).



LogFile::~LogFile()
{
    if (logn_) { delete[] logn_; logn_ = NULL; }
    if (path_) { delete[] path_; path_ = NULL; }
} // LogFile::~LogFile().



void LogFile::ErrorMsg()
{
    fl_alert( " %s : Log File \"%s\" Cannot be Created/Appended\n\t%s\n\n",
              logn_, path_, strerror( errno ) );
    exit( 2 );
} // void LogFile::ErrorMsg().



void LogFile::IniWrite( const char * msg )
{
    FILE          * fp;

    fp = fopen( path_, "w" );
    if (fp) {
        fprintf( fp, "# *** %s : Automatic log file creation ", logn_ );
        if (msg) fprintf( fp, "%s. ***\n\n", msg );
            else fprintf( fp, "at each new execution. ***\n\n" );
        fclose( fp );
    } else ErrorMsg();
} // void LogFile::IniWrite( const char * msg ).



void LogFile::SetUp( const char * path, int fl, const char * msg, const char * lgn )
{
    char      buf[256];
    char   * pos = buf;
    int             nc;

    if (lgn) logn_ = strdup( lgn );  
    nc = snprintf( buf, 256, "%s", path );
    pos = buf + nc;
    if (*pos != DIR_SEPAR) { *(pos++) = DIR_SEPAR; nc++; }
    if (mskf_) { *(pos++) = '.'; nc++; }
    snprintf( pos, 256-nc, "%s", logn_ );
    path_ = strdup( buf );
    IniWrite( msg );
} // void LogFile::SetUp( char * path, int fl = 0, const char * msg = NULL ).



void LogFile::Write( const char * fmt, ... )
{
    FILE  * fp;
    va_list ap;
    char buffer[1024];

    va_start( ap, fmt );
    ::vsnprintf( buffer, 1024, fmt, ap );
    va_end( ap );
    fp = fopen( path_, "a" );
    if (fp) {
        fprintf( fp, "%s\n", buffer );
        fclose( fp );
    } else ErrorMsg();
} // void LogFile::Write( const char * frm, ... ).



// --------------------------------------------------------------


void SrvFile::IOError( int cod, const char *msg )
//
// IO Error routine.
//
// Used Routine to generate all file context attached error message.
//
{
    int ns;

    erl_ = snprintf( err_, 512, " * %s %s # %d : %s\n * file\" %s\"", mnam_, (cod>0)? "Warning":"Error",
                                cod, msg, path_ );
    if (cod > 0) {
        if (smg_) fl_message( "%s", err_ );
    } else {
        if (smg_) fl_alert( "%s", err_ );
        Close();
        if (ehdl_) ehdl_( cod );        // Call user catcher to finish the current action.
    }
/*
    if (cod > 0)
        snprintf( " * %s Warning # %d : %s\n * file\" %s\"", mnam_, cod, msg, path_  );
    else
        snprintf( " * %s Error # %d : %s\n * file \"%s\"", mnam_, -cod, msg, path_ )
        fl_message( " * %s Warning # %d : %s\n * file\" %s\"", mnam_, cod, msg, path_ );
    else {
        fl_alert( " * %s Error # %d : %s \n * file \"%s\"", mnam_, -cod, msg, path_ );
        Close();
        if (ehdl_) ehdl_( cod );        // Call user catcher to finish the current action.
    }
*/
} // (private) void SrvFile::IOError( int cod, const char *msg ).




void SrvFile::BugError( int ie )
{
    typedef char * msgty;

    const char * m0 = "%s : File  \"%s\" not open to read\n",
               * m1 = "%s : File  \"%s\" not open to write\n";
    const char *mtb[] = { m0, m1 };

    fl_alert( mtb[ie], " *** Service_IO Fatal Bug : ", path_ );
    exit( 3 );
} // static void BugError( int ie ).



void SrvFile::Close()
//
// This function must be used to close a service file.
//
{
    if (path_ && wr_&2) fclose( fp_ );
    fp_   = NULL;
    path_ = NULL;
    wr_ = 0;
} // void SrvFile::Close().



int  SrvFile::Open( const char * fname,
                    int wrt, const Header * fhl,
                    const char * md )
//
// This function must be used to open a slide context file.
// Return >0 (= )idex+1 in format list) on success, 0 on (when no such file ...
// ... and -1 on open error or unknown format.
//
{
    Header  head, rdhd;
    int         ie = 1, // Assume Success.
              sz0, sz1;

    if (wr_ && fp_) { fclose( fp_ ); delete[] path_; }
    wr_ = wrt? 1 : 0;
    if (fhl) hl_ = fhl;
    if (md) mnam_ = md;
    fp_ = fopen( fname, wr_ ? "w" : "r" );
    if (!fp_) {
        if (wr_) {     // Error : cannot create the new file.
            erl_ = snprintf( err_, 512, " * %s : Cannot create file \"%s\"\n * Error : %s",
                                        mnam_, fname, strerror( errno ) );
            if (smg_) fl_alert( "%s", err_ );
            ie = -1;
        } else {
            if (errno !=ENOENT) { // if file exist with Open error, signal it.
                erl_ = snprintf( err_, 512, " * %s : Cannot Open file \"%s\"\n * Error : %s",
                                            mnam_, fname, strerror( errno ) );
                if (smg_) fl_alert( "%s", err_ );
                ie = -1;
            } else ie = 0;      // return 0 without error message for file not found.
        }
    } else { // fopen success.
        path_ = strdup( fname ); wr_ |= 2; // Set at opened file.
        // The next string must be exactly 15 char.
//      sz0 = sprintf( head, DIAVIEWER_VERFORMAT,
//                     DIAVIEWER_VERSION, DIAVIEWER_SUB_VERSION, DIAVIEWER_RELEASE ) + 1;
        if (hl_) {
            if (wr_&1) {
                fprintf( fp_, "%s\n", hl_[0] );     // To Create/Supershed a new service file.
                ie = 1;
            } else {
                sz1 = fscanf( fp_,"%[^\n]",rdhd );  // To read a service file.
                // Loop to search if the format is known and return the format index when known.
                ie = 0;
                rdhd[15] = '\0';                    // The final character is not take in account.
                while (hl_[ie][0] && strcmp( rdhd, hl_[ie] )) ie++;
                if (!hl_[ie][0]) {
                    erl_ = snprintf( err_, 512, " * %s Error: Bad format \"%s\" in file \"%s\"\n",
                                    mnam_, rdhd, fname );
                    if (smg_) fl_alert( "%s", err_ );
                    Close();
                    ie = -1;
                } else ie++; // return the format index (starting from 1).
            }
        }
    }
    return ie;
} // int  SrvFile::Open( const char * fname, int wrt ).



//
// **** Routines to read context files of a volume.
//


char * SrvFile::GetText( VarString &vs )
//
// Read any string of text..
//
{
    int   ch, ii, ie, siz, slu;
    char                frm[8];
    char          * str = NULL;

    ie = 1;                                     // Assume a sucess.
    if (wr_ != 2) BugError( 1 );
    slu = fscanf( fp_, "%d", &siz );          // Get the size of string,
    if (slu<0) { siz = 0; ie = -1; }
    if (slu>0) {
        if (siz>0) {
            str = vs.Size( siz+1 );             // Get temporary memory.
            ii = 0; ch = fgetc( fp_ );        // Init loop and skip one space.
//printf( " Read Cntx str l = %d\n", siz );
            while ((ii < siz)&&(ch > 0) ) {
                ch = fgetc( fp_ );
//printf( " %d / %c %d\n", ii, ch, ch );
                if (ch != 13) str[ii++] = ch;   // Keep any character except CR ascii(13).
            }
            siz = ii;
//          sprintf( frm, " %%%dc", siz );      // Make the adapted read format.
//          slu = fscanf( fcntx, frm, str );    // Read the text string.
//          siz = j;
            str[siz] = 0;                       // Append the final null character.
//printf( " str = \"%s\"\n", str );            
            if (slu <= 0) ie = -1;
        } else str = NULL;
    }
    if (ie<0) {
        str = NULL;
        if (ferror( fp_ )) IOError( -3, "Text Input file error" );
                      else if (ehdl_) ehdl_( 1 );
    }
    return str;
} // char * SrvFile::GetText().



int  SrvFile::GetInt()
{
    int slu, val;

    if (wr_ != 2) BugError( 0 );
    slu = fscanf( fp_, "%d", &val );
    if (slu<=0) {
        if (ferror( fp_ )) IOError( -7, "Integer Input file error" );
                      else if (ehdl_) ehdl_( 1 );
    }
    return val;
} // int SrvFile::GetInt( short int &val ).



float SrvFile::GetFlt()
{
    int slu, val;

    if (wr_ != 2) BugError( 0 );
    slu = fscanf( fp_, "%f", &val );
    if (slu<=0) {
        if (ferror( fp_ )) IOError( -8, "Float Input file error" );
        else if (ehdl_) ehdl_( 1 );
    }
    return val;
} // float SrvFile::GetFlt( float &val ).



void SrvFile::PutText( const char * str )
//
// Write a text.
//
{
    int len, swr;

    if (wr_ != 3) BugError( 1 );
    if (str&&str[0]) {
        len = strlen( str );
        swr = fprintf( fp_, " %4d %s\n", len, str );
    } else swr = fprintf( fp_, " %4d\n", 0 );
    if (swr<=0) {
        if (ferror( fp_ )) IOError( -11, "Text Output file error" );
                      else if (ehdl_) ehdl_( 1 );
    }
} // int SrvFile::PutText( const char * str ).



void SrvFile::PutEoln()
{
    if (wr_ != 3) BugError( 1 );
    fprintf( fp_, "\n" );
} // void SrvFile::PutEoln().



void SrvFile::PutInt( int val, int fld )
//
// Write a number.
//
{
    int     swr;
    char frm[8];
    
    if (wr_ != 3) BugError( 1 );
    sprintf( frm, " %%%dd", fld );
    swr = fprintf( fp_, frm, val );
    if (swr<=0) {
        if (ferror( fp_ )) IOError( -12, "Integer Output file error" );
                      else if (ehdl_) ehdl_( 1 );
    }
} // void SrvFile::PutInt( const char * str, int fld ).



void SrvFile::PutFlt( float val, int f, int d )
//
// Write a number.
//
{
    int     swr;
    char frm[8];

    if (wr_ != 3) BugError( 1 );
    sprintf( frm, " %%%d.%df", f, d );
    swr = fprintf( fp_, frm, val );
    if (swr<=0) {
        if (ferror( fp_ )) IOError( -13, "Integer Output file error" );
        else if (ehdl_) ehdl_( 1 );
    }
} // void SrvFile::PutFlt( const char * str, int fld ).



//
// end of <PW-Id>.
//
