//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers Software
//

//
// Environment of Service Stack of any objects.
//
//
// Object Stack class routines to manage tables of undefined objects.
// The size of object are free but, all in the table must have the same size.
//
// Small description :
// -------------------
//
// The class ObjStack is the basic class that gives all objects stack
// management. The function members are :
//
//  Three class creators :
//      ObjStack();
//      ObjStack( int els, UInt_8 ** sp = 0 );
//      ObjStack( int els, int tbs, UInt_8 * tab, UInt_8 ** sp = 0 );
//
//  The first is to use inside an other class (because you can't use sizeof() function).
//  In the second one, <els> must be the size (in bytes) of your object to store in
//  the stack, and <ps> is a stack pointer variable that you can use to get access
//  at each element in the stack. This stack pointer is automaticaly updated when
//  the stack contents change. This stack pointer give access at each object ins the stack
//  For example ps[0] or *ps you give access to object at the top of stack, and ps[-1] at the
//  previous one.
//  The third form is to use with a fixed size array allocated by the user as any C/C++ array.
//  the argument <tbs> is the size of array in bytes, and <tab> the used table.
//
//  to get an easy use of this ObjStack class. Two children template class are defined :
//      Any_STK<Ty> where Ty is the type (of the stack element) with two forms of creator :
//          Any_STK<Ty>();
//          Any_STK<Ty>( size_t els, Ty ** sp);
//  and
//      Any_STK_F<Ty>( size_t els, size_t tbs, Ty * tab, Ty ** sp )
//  used to create a stack on a user allocated table with a fixed size.
//
//  To these three forms of stack creation are completed by three macro :
//
//      Any_STK_L( Ty, name, sp )  -->  Any_STK<Ty> name( sizeof( Ty ), & sp )
//
//      Any_STK_D( Ty, name, sp )  -->  Any_STK * name = new Any_STK<Ty>( sizeof( Ty ), & sp )
//
//      Any_STK_F( Ty, name, tab, sp ) -->
//              Any_STK_F<Ty> name( sizeof( Ty ), sizeof( tab ), Ty * tb, & sp )
//
//  The forms of creator without arguments must be completed by a call of function member :
//
//      typedef ... Ty;
//      Ty  ** sp;
//      ...
//      Any_STK<Ty> mystack;
//      ...
//      mystack.IniStk( sizeof( Ty ), sp, inc = 16 );
//       // or to attach a user array (as Ty tab[1024]) :
//      mystack.Fix_Init( sizeof( Ty ), sizeof( tab ), tab, sp );
//
//  In the use of IniStk call, the third argument is the increment in number of elements, this
//  increment is used to automatically increase the stack size on an overflow.
//
//  The user can define a virtual function (in a sub class of the Stack class) to trap any
//  stack overflow.
//
//      int ErrHandler( int siz )
//
//  It is also possible to trap the stack overfloaw by a single function of model :
//
//       typedef int (*OvFlError_t)( int );
//  that to be installed by the call of the member function :
//
//      void SetOvFlError( OvFlError_t fovf )
//
// In this two cases: to continue the process, the trap function must return 0 and the stack size
// will be increased (except for the stacks of fixed forms). For the underflow (with Pop call), no
// trap occurs and the function member result is always null pointer.
//
// For not fixe allocation stack on a user array, you have the possibility to get a first stack
// allocation by the call of the member function "void IniAlloc( int size )" where size is the
// first size allocation in bytes.
//
// Except fot the Fixed stack on a user allocated array, on an overflow, the stack size is
// automatically increased by memory increment. This increment can be changed by call of member
// function "void Increment( int incr )" where incr is the incrment in number of element of the
// stack.
//
// The main Service_STACK member function are :
//
//  If you have the declarations:
//
//      Ty * clm, el1, el2, el3;        // Two object of type Ty.
//      Ty           ** sp;             // Our stack pointer.
//         ...
//      Any_STK_L( Ty, mystk, sp );     // Declare our Stack of objects of type Ty.
//         ...
//      clm = mystk.Push( el1 );        // Now *clm and sp[0] are the content of the top of stack (copy of el1).
//      clm = mystk.Push( el2 );        // Now *clm and sp[0] are the content of the top of stack (copy of el2).
//      el3 = sp[-1];                   // Now el3 is a copy of el1.
//      el1 = sp[0];                    // value of el2 (top of stack) is copied in el1 ...
//      el2 = sp[-1];                   // ... and the old value of el1 is copied in el2.
//                                      // el1 and el2 have been swapped.
//      clm = mystack.Pop( el3 );       // el3 take the old value of el2 and ...
//                                      // ... *clm = sp[0] are the old value of el1.
//      clm = (mystack.Pop( el3 );      // Now, el3 take the value old value of el1 and ...
//                                      // ... and sp = clm = 0 because ste stack is empty.
//                                      // Any other Pop call return a null pointer on underflow.
//
//
//

#ifndef SERVICE_STACK_H
#define SERVICE_STACK_H

#include <unistd.h>


#include "Service_ENV.h"



class ObjStack {
public:

    typedef int (*OvFlError_t)( int );  // Address Catch overflow error routine.


private:
    typedef union {
#ifdef Adr32
        void *       ad[16384];
#else
        void *        ad[8192];
#endif
        double        db[8192];
        UInt64        uq[8192];
        SInt64        sq[8192];
        float        fl[16384];
        SInt32       sl[16384];
        UInt32       ul[16384];
        SInt16       sw[32768];
        UInt16       uw[32768];
        SInt_8       sb[65536];
        UInt_8       ub[65536];
        char         ch[65536];
    } TableAcc;

    int           els_, // Stack Element size in byte.
                  nal_, // Object with alignement (if not null).
                  inc_, // Increment of allocation.
                  use_, // Used size.
                  all_; // Allocated size.

    UInt_8      * tab_, // The table of object pointer.
                * stp_, // The internal stack pointer.
               ** usp_; // The address of the user copy of the stack pointer.
    OvFlError_t   ovf_; // User function to trap stack overflow error.

    void MemCpy( UInt_8 *p, UInt_8 *q, int s ) { for(int ii = 0; ii < s; ii++) *(p++) = *(q++); }
    void MemCpy( UInt16 *p, UInt16 *q, int s ) { for(int ii = 0; ii < s; ii++) *(p++) = *(q++); }
    void MemCpy( UInt32 *p, UInt32 *q, int s ) { for(int ii = 0; ii < s; ii++) *(p++) = *(q++); }
    void MemCpy( UInt64 *p, UInt64 *q, int s ) { for(int ii = 0; ii < s; ii++) *(p++) = *(q++); }

protected:
    // Routine to copy a stack object (cannot be used out of the Stack class).
    void ElmCpy( UInt_8 *p, UInt_8 *q, int = 1 );

    // To trap the overflow stack error the user must redefine this function.
    // If the return 0, the process continue on overflow and the stack size will be
    // increased. For underflow, the Pop function return a null pointer because ...
    // ... A stack underflow is always a bug in the user software.
    virtual int ErrHandler( int siz ) { if (!inc_) return ovf_( siz ); else return 0; }

public:
    //
    // The creator arguments is the size of stack element.
    //
    void InitStk( size_t sls_, UInt_8 ** sp, int = 16 );
    void Increment( int = 16 );
    void IniAlloc( int );
    ObjStack();
    ObjStack( int, UInt_8 ** = 0 );
    ObjStack( int, int, UInt_8 *, UInt_8 ** = 0 );
    ~ObjStack() { if (tab_ && inc_) delete[] tab_; tab_ = 0; usp_ = 0; }

    void SetOvFlError( OvFlError_t fovf ) { ovf_ = fovf; }

    void Fix_Init( int, int, UInt_8 *, UInt_8 **);

    void Clear() { use_ = 0; stp_ = tab_ - els_; }

    UInt_8 * Push( UInt_8 * );
    UInt_8 *  Pop( UInt_8 * );

    int  Index() { return (stp_ - tab_)/els_; }

    int  Empty() { return (stp_ < tab_)? 1 : 0; }

}; // class ObjStack.



//
// Easy to use Stack of any object definition :
//
//  Use with calls :
//
//  Declaration  for any object of type X :
//
//     Any_Stk<X> my_stack( sizeof( X ));   // Declaration extern or automatic.
//
//
//
//  The basic call are :
//     my_stack.Setup( int size, int ovf ); //
//
//     X * p = my_stack.Push( X * o );      // To put the object o at the top of stack and
//     X * p = mu_stack.Pop(  X * o );      // Pop the object o from the top of stack.
//                                          // the returned pointer p is the address of the stack top ...
//                                          // ... but this pointer is clean to 0 if the stack is ...
//                                          // reallocated.
//
//
//
//



template <class Ty>
class Any_STK : public ObjStack {
public:
    Any_STK<Ty>() : ObjStack() { }
    Any_STK <Ty> ( size_t s, Ty ** psp = 0 ) : ObjStack( s, (UInt_8**) psp ) { }
    Ty * Push( Ty & o ) { return (Ty *) ObjStack::Push( (UInt_8 *) &o ); }
    Ty * Pop(  Ty & o ) { return (Ty *) ObjStack::Pop(  (UInt_8 *) &o ); }
    Ty * PushD( Ty s ) { return (Ty *) ObjStack::Push( (UInt_8 *) (&s)); }
};

#define Any_STK_L( Ty, nam, sp )    Any_STK< Ty > nam ( sizeof( Ty ), & sp )
#define Any_STK_D( Ty, nam, sp )    Any_STK * nam = new Any_STK< Ty > ( sizeof( Ty ), & sp )


template <class Ty>
class Any_Stack_F : public ObjStack {
public:
    Any_Stack_F<Ty> ( size_t es, size_t ts, Ty * tb, Ty ** psp = 0 ):
                ObjStack( es, ts, (UInt_8 *) tb, (UInt_8**) psp ) { }
    Ty * Push( Ty & o ) { return (Ty *) ObjStack::Push( (UInt_8 *) &o ); }
    Ty * Pop(  Ty & o ) { return (Ty *) ObjStack::Pop(  (UInt_8 *) &o ); }
};

#define Any_STK_F( Ty, nm, tb, sp ) \
           Any_Stack_F< Ty > nm( sizeof( Ty ), sizeof( tb), (Ty *) tb, & sp )

//

#endif

//
// end of <PW-Id>.
//
