//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// DiaViewer Slide Volume List Managment module.
//


#include <string.h>
#include <setjmp.h>
#include <FL/fl_utf8.h>

#include "DiaViewer_VRFL.h"



//
// Context input/output Read and Write.
//

//
// Set the necessary I/O error trap definitions.
//
static sigjmp_buf       JmpBuf; // Buffer for long jump on I/O context error.

//
// Define the supported format of the List of slide volumes file...
// ... and the File object used to manage it.
//
static SrvFile::Header VolLFrm[] = {
    "DVWVL 1.000-A00", // We first format version of this file.
    0
};


static VarString         V1Str, // Variable Strings used as temporary strings.
                         V2Str;


static void ErrorCatch( int ierr )
{
    siglongjmp( JmpBuf, ierr );
} // void ErrorCatch( int ierr ).



VolRefList::VolRefList( const char * name ) :
    SrvFile( VolLFrm, name ){ upd_ = osz_ = 0; svl_ = NULL; }


int  VolRefList::Read( const char * pa )
// Read the Slides Volumess List.
{
    int   id,  ie,  sz;
    VolRef        * rc;
    if (pa) pat_ =  pa;
    ie = -1;                                    // Assume error until show otherwise.
    InsErrCatcher( ErrorCatch );                // Install the File I/O error catcher.
    if (Open( pat_, 0 ) > 0) {                  // Open to write Success.
        if (!sigsetjmp( JmpBuf, 1 )) {          // To stop on I/O error.
            sz = GetInt();                      // Get the maximum size (used for allocation).
            for(int ii = 0; ii < sz; ii++) {    // For each list Volume.
                id = GetInt();                  //    Get the Volume index, ...
                if (id > 0) {                   // ... If it is legal  ...
                    rc = new VolRef;            // Create the Volume Reference.
                    rc->iide = id;              // ... and set the n° (idex) of Volume, ...
                    rc->path = strdup( GetText( V1Str ) );//   ... the Volume path, ...
                    rc->name = strdup( GetText( V2Str ) );//   ... the Volume name (if defined), ...
                    rc->volm = NULL;            //   ... not opened => No volume record and ...
                    ols_.AddObj(rc);            //   ... put the reference record in the KwonVol Table.
               } else break;                    // ... else we stop the loop.
            }
            ie = 0;
        }
        Close();
    }
    return ie;
} // void void VolRefList::Read().



int  VolRefList::Write( const char * pa )
// Rewrite or ReWrite(to update) the Slides Volumes list.
{
    int   ii,  ie,  sz;
    VolRef        * rc;

    if (pa) pat_ = pa;
    ie = -1;                                    // Assume error until show otherwise.
    sz = ListSize();                            // Get the number of volume to open.
    InsErrCatcher( ErrorCatch );                // Install the File I/O error catcher.
    if (Open( pat_, 1 ) > 0) {                  // Open to write Success.
        if (!sigsetjmp( JmpBuf, 1 )) {          // To stop on I/O error.
            PutInt( sz, 4 ); PutEoln();         // Write the size of Volume table.
            for( ii = 0; ii < sz; ii++) {       // For each list Volume.
                rc = (VolRef*)ols_.GetObj(ii);  // Get the Volume reference block.
                PutInt( rc->iide, 4 );
                PutText( rc->path );
                PutText(rc->name );
                rc = NULL;
            }
        }
        Close();
        ie   = 0;
        upd_ = 0;
    }
    return ie;
} // void void VolRefList::Write().



VolRef * VolRefList::Append( const char * pa, const char * nm, int upd )
// Append a new Slides Volume definition in the defined Slides Volume list.
{
    VolRef * ref =                  new VolRef;
    int      sz  =   ols_.AddObj( ref );
    VolRef ** VL = RefList();
    if (sz > 1) ref->iide = ((VolRef*)(ols_.GetObj(sz-2)))->iide + 1;
    else ref->iide = 1;
    ref->path = (pa) ? strdup( pa ) : NULL;
    ref->name = (nm) ? strdup( nm ) : NULL;
    ref->volm = NULL;
    if (upd) Write();
    VL = NULL;
    return ref;
} // VolRef * VolRefList::Append( const char * path, const char * name ).



VolRef * VolRefList::Insert( const char * pa, const char * nm )
// To insert a new volume to the volume list.
// The New Volume receive an unique integer identifier
//
{
    VolRef               ** VL;
    VolRef        * ref = NULL;
    int   id0, id1, ii, jj, sz;

    sz = ols_.TabUse();         // Get original table size ...
    ols_.AddObj( 0 );           // ... and append an empty VolRef.
    VL = RefList();

    ii = id0 = 0;
    if (sz) {                   // The original table was not empty.
        id1 = VL[0]->iide;      // Get first iide value (increasing ordering).
        while (id1 - id0 <= 1) {// Loop to find the first free value for iide.
            id0 = id1;
            if (++ii < sz) id1 = VL[ii]->iide;
            else break;
        }
        if (ii < sz) {          // When a free value  was found befaore the last one ...
            jj = sz;            // ... we shift the upper VolRef of one (index ->index + 1).
            do VL[jj] = VL[jj-1]; while (ii < --jj);
        }
    }
    VL[ii] = ref = new VolRef;  // Allocatand fill the new VolRef.
    ref->iide = id0 + 1;
    ref->path = (pa) ? strdup( pa ) : NULL;
    ref->name = (nm) ? strdup( nm ) : NULL;
    ref->volm = NULL;
    Write();
    VL = NULL;
    return ref;
} // int  VolRefList::Insert( const char * pa, const char * nm )



VolRef * VolRefList::Locate( int id )
// Find a Slides Volume from its integer identifier.
{
    VolRef      **  VL;
    VolRef       * ref;
    int        ii,  sz;

    sz = ols_.TabUse();
    if (!sz) return 0;
    VL = RefList();
    ii = 0;
    while ((VL[ii]->iide != id) && (ii < sz)) ii++;
    if (ii >= sz) return 0;
    else return VL[ii];
    VL = 0;
} // VolRef * VolRefList::Locate( int id ).



VolRef * VolRefList::Locate( const char * nm )
// Find the Slides Volume list with the specified name.
{
    VolRef ** VL, * rf;
    int         ii, sz;

    sz = ols_.TabUse();
    if (!(sz&&nm&&nm[0])) return 0;

    VL = RefList();
    ii = 0; rf = VL[0];
    while (ii < sz && fl_utf_strcasecmp( nm, rf->name )) ii++;

    if (ii < sz) return rf;
    return 0;
} // VolRef * VolRefList::Locate( const char * nm ).



void VolRefList::Remove( VolRef * &ref )
// Remove a slide volume from the Volume list and ...
// ... free its iteger identifier.
{
    VolRef      **  VL;
    int        ii,  sz;

    if (ref) {
        sz = ListSize();
        VL = RefList();
        ii = 0;
        while ((VL[ii] != ref) && (ii < sz)) ii++;
        delete ref; ref = 0;
        if (ii < sz) { VL[ii] = 0; ols_.Tamp( ii ); }
        Write();
        VL = 0;
    }
} // int  VolRefList::Remove( VolEntry * vol ).



int  VolRefList::CheckPath( const char * path )
// Check for not overlay on the proposed Slides Volume path.
{
    int ii, jj, l1, l2, lc, sz;
    VolRef               ** VL;
    char                  * s2;

    sz = ListSize();
    if (sz) {
        VL = RefList();
        l1 = strlen( path );
        for(ii = 0; ii < sz; ii++) {
            s2 = VL[ii]->path;
            l2 = strlen( s2 );
            lc = l1;
            if (l2 < l1) lc = l2;
            jj = 0;
            while ((jj < lc) && (path[jj] == s2[jj])) jj++;
            // If the shortest string is the first part of the other one ...
            // ... then the two path are same, else one path in include ...
            // ... in the other one.
            if (jj == lc) return 1;
        }
    }
    return 0;
} // int  VolRefList::CheckPath( const char * path ).



int  VolRefList::CheckName( const char * pname )
//
// Check if the Volume name is unique, if true return 0,
// else return the first other  Reference with the same name.
//
{
    int ii = 0, sz = ListSize();
    VolRef ** VL   =  RefList();

    while (ii < sz && strcmp( VL[ii]->name, pname )) ii++;
    VL = NULL;
    return (ii == sz) ? 0 : ii;
} // int  VolRefList::SetName( const char * pname ).



int VolRefList::GetFreeIde()
// Get a unique slides Volume integer identifier.
{
    int i0, i1, ii, sz;
    VolRef       ** VL;
    VolRef        * rf;

    sz = ListSize();
    i0 = 0;
    if (sz) {
        VL = RefList();
        ii = 0;
        i1 = VL[0]->iide;
        while (i1 - i0 <= 1) {
            i0 = i1;
            if (++ii < sz) i1 = VL[ii]->iide;
            else break;
        }
        VL = NULL;
    }
    return i0 + 1;
} // int VolRefList::GetFreeIde().



int  VolRefList::NewReferEl( const char * pat, char *&nna )
//
// We don't check for index field overflow (when iide > 99).
//
{
    int ii, id, ll, sz;
    char                 * tna;
    char       buf[32];

    if (CheckPath( pat )) return -1;

    if ((ii = CheckName( nna ))) {
        id = GetFreeIde();
        if (nna&&nna[0]) {
            ll = strlen( nna );
            if (26 < ll) nna[26] = 0;
            snprintf( buf, 31, "%s%%%%_%02d", nna, id );
        } else
            snprintf( buf, 31, "%%%%_%02d", id );
        nna = strdup( buf );
        return 1;
    }
    return 0;
} // const char * VolRefList::NewReferEl().



void VolRefList::MakeUnique( char * vname, int all )
// To create an unique volume name (based on the unique integer  Volume identifier).
{
    int id, ln;

    id = GetFreeIde();
    ln = strlen( vname );
    if (ln > all - 6) ln = all - 6;
    snprintf( vname+ln, all-ln, "%%%%_%02d", id );
} // void VolRefList::MakeUnique( char * vname );



int  VolRefList::CheckAllNames()
// Check for unique names for each defined Slides Volume.
{
    int nc, id, ii, jj, ln, sz;
    VolRef               ** VL;
    VolRef                * rf;
    char                  * nm;
    char               buf[32];

    sz =            ListSize();
    VL =             RefList();

    nc = 0;
    for(ii = 0; ii < sz; ii++) {
        rf = VL[ii];
        nm = rf->name; id = rf->iide;
        if (!(nm)) { // No name !
            snprintf( buf, 31, "%%%%_%02d", id );
            rf->name = strdup( buf );
            nc++;
        } else {
            if (ii > 0) {
                for(jj = 0; jj < ii; jj++) {
                    if (!strcmp( VL[jj]->name, nm )) {
                        // This name was already used.
                        ln = strlen( nm );
                        if (31 - ln < 5) nm[26] = 0;
                        snprintf( buf, 31, "%s%%%%_%02d", rf->name, rf->iide );
                        delete[] nm;
                        rf->name = strdup( buf );
                        nc++;
                        break;
                    }
                }
            }
        }
    }
    VL = NULL;

    return nc;
} // int  VolRefList::CheckAllNames().



static VolRef * OpenVolRef( int i ) { return ((VolEntry*)OpenVolTable.GetObj( i ))->VRefer(); }


static int StrNChar( const char * str )
{
    return fl_utf_nb_char( (const unsigned char *) str, strlen( str ) );
} // static int StrNChar( const char * str ).



VolRef ** VolRefList::SortVList( int &su, int &ns, int &np,  VolMnf_t md )
// Build an alphanumeric sorted slides volume list.
//
{
    int     ii, jj, ln, sn, sp, ts, sz;
    VolRef           * rf,       ** VL;

    sz = ListSize();
    if (osz_ != sz) {
        if (svl_) delete[] svl_;        // Free any old Sorted Vol List allocation ...
        svl_ = new VolRef * [sz+1];     // ... and allocate a new one (+1 when all are opened).
        osz_ = sz;                      // Keep the new size of the sorted list.
    }
    ii = jj = sn = sp = ts = 0;
    switch (md) {
        case VolMnf_Open:
            ii = 1;                     // To skip the memory Volume.
        case VolMnf_SOpen:
            jj = OpenVolTable.TabUse();
            while (ii < jj) {
                svl_[ts++] = rf = OpenVolRef( ii++ );
                ln = StrNChar( rf->name ); if (sn < ln) sn = ln;
                ln = StrNChar( rf->path ); if (sp < ln) sp = ln;
            }
        break;

        case VolMnf_VList:
        case VolMnf_Rename:
        case VolMnf_ToOpen:
            VL = GetRefs();
            while(ii < sz) {
                rf = VL[ii++];
                if (md != VolMnf_ToOpen || !rf->volm) {
                    svl_[ts++] = rf;
                    ln = StrNChar( rf->name ); if (sn < ln) sn = ln;
                    ln = StrNChar( rf->path ); if (sp < ln) sp = ln;
                }
            }
        break;
    }
    su = ts;
    ns = sn; np = sp;
    if (!ts) return NULL;

    if (ts > 2 || (md != VolMnf_SOpen && ts > 1)) {
        // We must perform the sort by volume name.
        ln = (md == VolMnf_SOpen) ? 1 : 0;
        ii = ln + 1;
        while (ii < ts) {
            rf = svl_[ii];
            jj = ii - 1;
            while (jj >= ln && fl_utf_strcasecmp( svl_[jj]->name, rf->name ) >= 0) {
                svl_[jj+1] = svl_[jj];
                jj--;
            }
            svl_[jj+1] = rf;
            ii++;
        }
    }
    return svl_;
} // VolRef ** VolRefList::SortVList().




//
// end of <PW-Id>.
//
