
//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// Module to manage the Image file/Pixel Map for OpenGL.
//

//
// FLTK software part modified by P.Wolfers
// for DiaViewer Software using OPENGL image management.
//
//

//
// "$Id$"
//
// Fl_JPEG_Image routines.
//
// Copyright 1997-2011 by Easy Software Products.
// Image support by Matthias Melcher, Copyright 2000-2009.
//
// This library is free software. Distribution and use rights are outlined in
// the file "COPYING" which should have been included with this file.  If this
// file is missing or damaged, see the license at:
//
//     http://www.fltk.org/COPYING.php
//
// Please report all bugs and problems on the following page:
//
//     http://www.fltk.org/str.php
//
// Contents:
//
//   Fl_JPEG_Image::Fl_JPEG_Image() - Load a JPEG image file.
//

//
// Include necessary header files...
//

#include <FL/fl_utf8.h>
#include <FL/Fl.H>

#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>

#include "ImageMan_MAP.h"



// Some releases of the Cygwin JPEG libraries don't have a correctly
// updated header file for the INT32 data type; the following define
// from Shane Hill seems to be a usable workaround...

#if defined(WIN32) && defined(__CYGWIN__)
#  define XMD_H
#endif // WIN32 && __CYGWIN__


extern "C"
{
//#ifdef HAVE_LIBJPEG
#  include <jpeglib.h>
//#endif // HAVE_LIBJPEG
}



//
// Custom JPEG error handling structure...
//

struct fl_jpeg_error_mgr {
  jpeg_error_mgr   pub_; // Destination manager...
  jmp_buf      errhand_; // Error handler
};



//
// Error handler for JPEG files...
//

extern "C" {
  static void
  fl_jpeg_error_handler( j_common_ptr dinfo ) {	// I - Decompressor info
    longjmp( ((fl_jpeg_error_mgr *)( dinfo->err ))->errhand_, 1 );
  }

  static void
  fl_jpeg_output_handler( j_common_ptr ) {	// I - Decompressor info (not used)
  }
}



/**
 \brief The constructor loads the JPEG image from the given jpeg filename.
 
 The inherited destructor frees all memory and server resources that are used 
 by the image.
 
 Use Fl_Image::fail() to check if Fl_JPEG_Image failed to load. fail() returns
 ERR_FILE_ACCESS if the file could not be opened or read, ERR_FORMAT if the
 JPEG format could not be decoded, and ERR_NO_IMAGE if the image could not
 be loaded for another reason. If the image has loaded correctly,
 ww(), hh(), and pp() should return values greater than zero.
 
 \param[in] filename a full path and name pointing to a valid jpeg file.
 */

void Image_MAP::Get_JPG( const char * filename )
{
    FILE			   *fp;	// File pointer
    jpeg_decompress_struct	 dinfo;	// Decompressor info
    fl_jpeg_error_mgr		  jerr;	// Error handler info
    JSAMPROW			   row;	// Sample row pointer
    GLubyte *               map = NULL; // Pointer of Pixel Map to create.
    GLint              ww, hh, pp, nsz; // For optimize Map allocation.

    // the following variables are pointers allocating some private space that
    // is not reset by 'setjmp()'
    char*    max_finish_decompress_err; // Count errors and give up after a while ...
    char*   max_destroy_decompress_err; // ... to avoid recusion and deadlock
  
    // Open the image file...
    if ((fp = fl_fopen( filename, "rb" )) == NULL) {
        E( ERR_FILE_ACCESS );
        return;
    }
  
    // Setup the decompressor info and read the header...
    dinfo.err                = jpeg_std_error( (jpeg_error_mgr *)&jerr );
    jerr.pub_.error_exit     = fl_jpeg_error_handler;
    jerr.pub_.output_message = fl_jpeg_output_handler;
  
    // Setup error loop variables
    max_finish_decompress_err  = (char*)malloc( 1 );    // allocate space on the frame for error counters
    max_destroy_decompress_err = (char*)malloc( 1 );    // otherwise, the variables are reset on the longjmp
    *max_finish_decompress_err = 10;
    *max_destroy_decompress_err= 10;
  
    if (setjmp( jerr.errhand_ )) {
        // JPEG error handling...
        Fl::warning( "JPEG file \"%s\" is too large or contains errors!\n", filename );
        // if any of the cleanup routines hits another error, we would end up 
        // in a loop. So instead, we decrement max_err for some upper cleanup limit.
        if ( ((*max_finish_decompress_err)-- > 0) && map)
            jpeg_finish_decompress( &dinfo );
        if ( (*max_destroy_decompress_err)-- > 0)
            jpeg_destroy_decompress( &dinfo );
    
        fclose( fp );
    
        free( max_destroy_decompress_err );
        free( max_finish_decompress_err );
    
        E( ERR_FORMAT );
        return;
    }
  
    jpeg_create_decompress( &dinfo );
    jpeg_stdio_src( &dinfo, fp );
    jpeg_read_header( &dinfo, TRUE );
  
    dinfo.quantize_colors      = (boolean)FALSE;
    dinfo.out_color_space      = JCS_RGB;
    dinfo.out_color_components = 3;
    dinfo.output_components    = 3;
  
    jpeg_calc_output_dimensions( &dinfo );
  
//  ww = W(); hh = H(); pp = P();  // Get previous Map dimensions.

    ww = dinfo.output_width; 
    hh = dinfo.output_height;
    pp = dinfo.output_components;

    // Compute new Pixel Map size ...
    if ((nsz = ww*hh*pp) > Max_Size ) longjmp( jerr.errhand_, 1 );

    // ... and allocate it.
    map = New_Map( nsz );
    W( ww );    H( hh );
    P( pp );

    jpeg_start_decompress( &dinfo );
  
    // Store the image row from low to high ordering for OpenGL.
  
    row = map + H()*W()*P(); // Store the image line from Bottom to UP.
    while (dinfo.output_scanline < dinfo.output_height) {
        row -= W()*P();
//      row = (JSAMPROW)(map +
//                   dinfo.output_scanline * dinfo.output_width *
//                   dinfo.output_components);
        jpeg_read_scanlines( &dinfo, &row, (JDIMENSION)1 );
    }
    
    map = NULL; // Already copied in map_ of class Image_MAP.
  
    jpeg_finish_decompress( &dinfo );
    jpeg_destroy_decompress( &dinfo );
  
    free( max_destroy_decompress_err );
    free( max_finish_decompress_err );
  
    fclose( fp );
} // FlGL_JPG_IMAGE alias Image_MAP::Get_JPG.




// data source manager for reading jpegs from memory
// init_source (j_decompress_ptr cinfo)
// fill_input_buffer (j_decompress_ptr cinfo)
// skip_input_data (j_decompress_ptr cinfo, long num_bytes)
// resync_to_restart (j_decompress_ptr cinfo, int desired)
// term_source (j_decompress_ptr cinfo)
//         JOCTET * next_output_byte;  /* => next byte to write in buffer */
//         size_t free_in_buffer;      /* # of byte spaces remaining in buffer */


typedef struct {
    struct jpeg_source_mgr pub;
    const unsigned char *data, *s;
    // JOCTET * buffer;              /* start of buffer */
    // boolean start_of_file;        /* have we gotten any data yet? */
} my_source_mgr;

typedef my_source_mgr *my_src_ptr;


extern "C" {

    static void init_source( j_decompress_ptr cinfo ) {
        my_src_ptr src = (my_src_ptr)cinfo->src;
        src->s = src->data;
    }

    static boolean fill_input_buffer(   j_decompress_ptr cinfo ) {
        my_src_ptr src = (my_src_ptr)cinfo->src;
        size_t nbytes = 4096;
        src->pub.next_input_byte = src->s;
        src->pub.bytes_in_buffer = nbytes;
        src->s += nbytes;
        return TRUE;
    }

    static void term_source( j_decompress_ptr cinfo )
    {
    }

    static void skip_input_data(j_decompress_ptr cinfo, long num_bytes) {
        my_src_ptr src = (my_src_ptr)cinfo->src;
        if (num_bytes > 0) {
            while (num_bytes > (long)src->pub.bytes_in_buffer) {
                num_bytes -= (long)src->pub.bytes_in_buffer;
                fill_input_buffer(  cinfo   );
            }
            src->pub.next_input_byte += (size_t) num_bytes;
            src->pub.bytes_in_buffer -= (size_t) num_bytes;
        }
    }

} // extern "C"

static void jpeg_mem_src(   j_decompress_ptr cinfo, const unsigned char *data   )
{
  my_src_ptr src;
  cinfo->src = (struct jpeg_source_mgr *)malloc( sizeof( my_source_mgr ) );
  src = (my_src_ptr)cinfo->src;
  src->pub.init_source = init_source;
  src->pub.fill_input_buffer = fill_input_buffer;
  src->pub.skip_input_data = skip_input_data;
  src->pub.resync_to_restart = jpeg_resync_to_restart;
  src->pub.term_source = term_source;
  src->pub.bytes_in_buffer = 0; /* forces fill_input_buffer on first read */
  src->pub.next_input_byte = NULL; /* until buffer loaded */
  src->data = data;
  src->s = data;
}



/**
 \brief The constructor loads the JPEG image from memory.

 Construct an image from a block of memory inside the application.

 The inherited destructor frees all memory and server resources that are used 
 by the image.

 Use Fl_Image::fail() to check if ImageMan_JPEG failed to load. fail() returns
 ERR_FILE_ACCESS if the file could not be opened or read, ERR_FORMAT if the
 JPEG format could not be decoded, and ERR_NO_IMAGE if the image could not
 be loaded for another reason. If the image has loaded correctly,
 ww(), hh(), and pp() should return values greater than zero.

 \param name A unique name or NULL
 \param data A pointer to the memory location of the JPEG image
 */
void Image_MAP::Get_JPG_MEM( const uchar *data )
{
    jpeg_decompress_struct	 dinfo;	// Decompressor info
    fl_jpeg_error_mgr		  jerr;	// Error handler info
    JSAMPROW			   row;	// Sample row pointer
    GLubyte *               map = NULL; // Pointer of Pixel Map to create.
    GLint              ww, hh, pp, nsz; // For optimize Map allocation.
  
    // the following variables are pointers allocating some private space that
    // is not reset by 'setjmp()'
    char*    max_finish_decompress_err; // count errors and give up afer a while
    char*   max_destroy_decompress_err; // to avoid recusion and deadlock
  
    // Setup the decompressor info and read the header...
    dinfo.err                = jpeg_std_error( (jpeg_error_mgr *)&jerr );
    jerr.pub_.error_exit     = fl_jpeg_error_handler;
    jerr.pub_.output_message = fl_jpeg_output_handler;
  
    // Setup error loop variables
    max_finish_decompress_err   = (char*)malloc( 1 );   // allocate space on the frame for error counters
    max_destroy_decompress_err  = (char*)malloc( 1 );   // otherwise, the variables are reset on the longjmp
    *max_finish_decompress_err  = 10;
    *max_destroy_decompress_err = 10;
  
    if (setjmp( jerr.errhand_ )) {
        // JPEG error handling...
        Fl::warning( "JPEG data is too large or contains errors!\n" );
        // if any of the cleanup routines hits another error, we would end up 
        // in a loop. So instead, we decrement max_err for some upper cleanup limit.
        if ( ((*max_finish_decompress_err)-- > 0) && map)
            jpeg_finish_decompress( &dinfo );
        if ( (*max_destroy_decompress_err)-- > 0)
            jpeg_destroy_decompress( &dinfo );
    
        free( max_destroy_decompress_err );
        free( max_finish_decompress_err );
    
        return;
    }
  
    jpeg_create_decompress( &dinfo );
    jpeg_mem_src( &dinfo, data );
    jpeg_read_header( &dinfo, TRUE );
  
    dinfo.quantize_colors      = (boolean)FALSE;
    dinfo.out_color_space      = JCS_RGB;
    dinfo.out_color_components = 3;
    dinfo.output_components    = 3;
  
    jpeg_calc_output_dimensions( &dinfo );
  
    ww = dinfo.output_width; 
    hh = dinfo.output_height;
    pp = dinfo.output_components;
  
    // Compute new Pixel Map size ...
    if ((nsz = ww*hh*pp) > Max_Size ) longjmp( jerr.errhand_, 1 );

    // ... and allocate it.
    map = New_Map( nsz );
    W( ww );    H( hh );
    P( pp );

    jpeg_start_decompress( &dinfo );

    row = map + H()*W()*P();
    while (dinfo.output_scanline < dinfo.output_height) {
        row -= W()*P();
//      row = (JSAMPROW)(map +
//                   dinfo.output_scanline * dinfo.output_width *
//                   dinfo.output_components);
        jpeg_read_scanlines( &dinfo, &row, (JDIMENSION)1 );
    }
    
    map = NULL;     // Set the New map in the Image Cache.
  
    jpeg_finish_decompress( &dinfo );
    jpeg_destroy_decompress( &dinfo );
  
    free( max_destroy_decompress_err );
    free( max_finish_decompress_err );
} // Alias Image_MAP::Get_JPG_MEM



//
// End of "$Id$".
//

//
// end of <PW-Id>.
//
