//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers Software
//

//
// DiaViewer GL picture diaporama interpretor.
//
//


#include <string.h>

#include "DiaViewer_GBL.h"
#include "DiaViewer_PRG.h"


// Must be a power of 2.
#define  PROG_INCR 256


#ifndef DIRSEP
#   if defined(WIN32) || defined(__EMX__) && !defined(__CYGWIN__)
#       define DIRSEP '\\'
#   else
#       define DIRSEP '/'
#   endif
#endif




//
// Class SlidePrg to create (and display) some diaporama programs.
//
//

// This variable are used to build a Slide program.
// As a consequence, the Build process of a Slide program is not ...
// ... reentrant and cannot be used by several user in the same time.
static int   PrgAlloc   =    0; // Allocated size of codes'array.
static char    * CPrg   = NULL; // Pointer to array of codes.


//
// Private methods.
//

void SlidePrg::AddCode( char ch )
{
    if (siz_ >= PrgAlloc) {
        PrgAlloc = PrgAlloc + PROG_INCR;
        char *tmp = new char[PrgAlloc];
        if (siz_ > 0) {
            memcpy( tmp, CPrg, siz_ );
            delete[] CPrg;
        }
        CPrg = tmp;
    }
    CPrg[siz_++] = ch;
} // void SlidePrg::AddCode( char ch ).



void SlidePrg::AddInt( int iv )
{
    if (iv==0) AddCode( '0' );
    else {
        if (iv<0) { AddCode( '-' ); iv = -iv; }
        while (iv>0){
            AddCode( (iv % 10)+ '0' );
            iv /= 10;
        }
    }
} // void SlidePrg::AddInt( int iv ).



void SlidePrg::Build( DirEntry * dir, int slv )
{
    int   nd = dir->NDir(),
          ns = dir->NSli();

    if (!slv) AddCode( '>' );
    if (nd) {
      if (ns) AddCode( 'S' );
      if (nd > 0) {
        for(int i=1;i<=nd;i++) {
          DirEntry *sbd = dir->SbDir( i );
          if (i>1) AddCode( '+' );
          AddCode( 'U' );
          Build( sbd, slv+1 );
          AddCode( 'D' );
        }
      }
      if (ns > 0) AddCode( 's' );
    } else AddCode( '*' );
    if (!slv) AddCode( '<' );
} // void SlidePrg::Build( int iv ).



//
// Class Slide_PRG public methodes.
//

SlidePrg::SlidePrg( VolEntry * vol )
{
    DirEntry * dir = vol->VRoot();;

    siz_  =    0;
    ptb_  = NULL;
    if (dir) {
        Build( dir, 0 );
        if (siz_ > 0) {
            ptb_ = new char[siz_+1];
            memcpy( ptb_, CPrg, siz_ );
            ptb_[siz_] = 0;
            CPrg[0] = 0;
        }
    } else ptb_ = NULL;
} // SlidePrg::SlidePrg( VolEntry * vol ).



SlidePrg::~SlidePrg()
{
    if (ptb_) delete[] ptb_;
    siz_  =     0;
    ptb_  =  NULL;
} // SlidePrg::~SlidePrg().



void SlidePrg::Display()
{
    char line[129];
    int     ad, ic;
    
    LogF.Write( "\n  *** Navigation programme code ***\n" );
    
    ic = 0; ad = 0;
    for(int ii=0; ii<siz_; ii++) {
        line[ic++] = ptb_[ii];
        if (ic>=128) {
            line[128] = 0;
            LogF.Write( " %4d / %s", ad, line );
            ic = 0; ad += 128;
        }
    }
    if (ic>0) {
        line[ic] = 0;
        LogF.Write( " %4d / %s", ad, line );
    }
    LogF.Write( "\n\n");
} // void SlidePrg::Display().




//
// Class SlideExec to execute the slides programs build by SlidePrg.
//

SlideExec::SlideExec( VolEntry *vol, SlidePrg * prg = NULL )
//
// The SlideExec creator, create a program context that allows
// to be executed for several use in same time.
// Eachprogram is specialy adaptated to a specific slide volume,
// but it is possible to other several different programs for the
// same slide volume.
//
{
    if (vol) {
        vol_    = vol;
        dir_    = vol->VRoot();
        sli_    = NULL;

        if (!prg) prg = (SlidePrg*)(vol->RelPrg());
        if (prg) {
            prg_ = prg->Program();
            sz_  = prg->ProgSize();
        } else { prg_ = NULL; sz_ = 0; }

        stks_   =       vol_->TDeep() + 1;
        stack_  =       new int[stks_]; // Create the specific stack.
        sp_     =                   -1; // Initialize the stack pointer, ...
        pc_     =                    0; // ... the ordinal counter, ...
        st_     =                    0; // ... the status register, ...
        idx_    =                    0; // ... and the index register.
        spc_    =                   -1; // No active  search.
        sid_    =                    0;
        sdi_    =                    0;
        if (!(vol->FlgTst( Context_MemImg )))
          path_ = new FilePath(stks_ ); // Create the specific dynamic path.
        else
          path_ = NULL;
    }
} // SlideExec::SlideExec( VolEntry *vol, SlidePrg * prg = NULL ).



SlideExec::~SlideExec()
{
    if (stack_) delete[] stack_;
    if (path_)  delete[]  path_;
    dir_ = NULL;
    prg_ = NULL;
} // SlideExec::~SlideExec.



int SlideExec::DirPush( int id )
// Install the sub-directory # -i, and set its scan begin
// in agreement with direction id (reverse direction if id>0).
{    
    int ie = -1;

    if (dir_) {
        if (sp_ < stks_) {
            stack_[++sp_] = idx_;
            dir_ = dir_->SbDir( -idx_ );
//          if (dir_->ndir_ == 1) dir_ = (DirEntry*)(dir_->dlst_);
//                          else dir_ = dir_->dlst_[-i-1];
            if (path_) path_->Set( dir_->FName(), 1 );
            if (id<0) idx_ = - dir_->NDir();
                 else idx_ = - 1;
            ie = 0;
        } else ie = -2; // Stack overflow!
    }
    return ie;
} // int SlideExec::DirPush( int i, int id ).



int SlideExec::DirPop()
//  Restore the context when quit a sub-directory for its father.
{
    int ie = -1;

//printf( " POP  A\n" ); fflush( stdout );
    DirEntry * rdi = dir_->Owner();
    if (rdi) {
        dir_ = rdi;
        if (path_) path_->Rem();
        ie = 0;
        if (sp_ >= 0)
            idx_ = stack_[sp_--];
        else ie = -3; // Stack underflow!
    }
    return ie;
} // int SlideExec::DirPop().



int SlideExec::Start( SlidePrg * prg, int lstk, int id, int pr )
// Establish and start the program *prg  with a stack deep if lstk
// (if specified !=0) and in the id, pr direction (default at 0, 0).
// Routine to use to start a custome slide program.
//
{
    int ie = -1;
    
    if (vol_&&prg) {
        dir_    =        vol_->VRoot();
        sli_    =                 NULL;
        prg_    =       prg->Program();
        sz_     =      prg->ProgSize();
        sp_     =                   -1; // Initialize the stack pointer.
        if (id<0) {
            st_ =                    1; 
            pc_ =              sz_ - 1;
            idx_    =   - dir_->NDir(); // ... the directory index register for reverse direction.
        } else {
            st_ =                    0;
            pc_ =                    0;            
            idx_    =               -1; // ... the directory index register for right direction.
        }

        if (lstk==0) stks_ = lstk;
        else if (stks_ < lstk) {
            stks_ = lstk;
            if (stack_) {
                delete[] stack_;
                stack_ = new int[stks_];
            }
        }
        PrvRkgClear();                  // Initialize the previous slide ranking information.
        return Continue( id, pr );
    }
    return ie;
} // int SlideExec::Start( SlidePrg * prg, int lstk, int id, int pr ).



int SlideExec::Start( int id, int pr )
// Start the slide program from begin (id>=0) or from its end (id<0).
// The pr argument is the priority choice for the slides any directory
// having also some sub-directories.
{
    // Set the base volume path.
    dir_ = vol_->VRoot();       // Restart from the root of the volume.
    if (path_) {
      path_->Clear();
      path_->Set( dir_->FName(), 1 );
    }
    sli_ = NULL;
    
    sp_  =   -1;                // Initialize the stack pointer.
    
    if (id < 0) {               // Initialize a code in reverse (left) direction.
        st_  =     1;
        pc_  =   sz_ - 1;
        idx_ = - dir_->NDir();
    } else {                    // Initialize a code in natural (right) direction.
        st_  =     0;
        pc_  =     0;
        idx_ =    -1;
    }
    PrvRkgClear();              // Initialize the previous slide ranking information.
    // Start the execution.
    return Continue( id, pr );
} // int SlideExec::Start( int id, int pr ).



int SlideExec::Abort()
// The program can be stopped/Aborted at each pause.
{
    path_->Clear();
    Start( 0, SlidePrior );
    return 0;
} // int SlideExec::Abort().



int SlideExec::Continue( int id, int pr )
// We can change the direction or the slide priority ...
// ... during each pause.
// Return 0 on success, 1 when volume limits are reached, ...
// ... and a negative value on error.
{
    char                 cd;
    int       ier, exe, sid;

    ier =  0;
    exe =  1;           // Ready to run until next pause (code '*', 'S' or 's') or end.
    sid = id;           // Save original direction.

    if (idx_ > 0)       // We was in a slide scan.Slide
        if (sp_ >= 0) idx_ = stack_[sp_--]; // Restore the index of directories.
                else return -3; // Stack underflow.

    // We do not re-execute the same instruction when the direction is changed.
    if (id<0) { if (!(st_&1)) { pc_-=2; st_|=1; } }
         else { if (st_&1) { pc_+=2; st_^=1; } }

    do { // Loop on each program statement.
        cd = prg_[pc_]; if (id<0) pc_--; else pc_++;
//printf ("Cont : pc = %d, code = %c, idx = %d\n", pc_, cd, idx_ ); fflush( stdout );
        switch (cd) {
            case '>': if (id<0)  { id =  0; pc_++; st_ ^= 1; ier = 1; } break;
            case '<': if (id>=0) { id = -1; pc_--; st_ |= 1; ier = 1; } break;
            case '*':           exe = 0; break;
            case 'S': if (pr)   exe = 0; break;
            case 's': if (!pr)  exe = 0; break;
            case 'U':
            case 'D': {
                if (((cd=='U')&&(id<0))||((cd=='D')&&(id>=0)))
                    ier = DirPop();
                else
                    ier = DirPush( id );
                break;
            }
            case '-':
                if (id<0) { if (idx_ > dir_->NDir()) idx_--; }
                     else { if (idx_ > 1) idx_++; }
            break;

            case '+':
                if (id<0) { if (idx_ < -1) idx_++; }
                     else { if (idx_ > - dir_->NDir()) idx_--; }
            break;
            
            default:  ier = -1; // Bad Code.
        }
        if (ier < 0) exe = 0; // Stop on error.
    } while ((exe>0)&&(pc_ >= 0)&&(pc_ < sz_));

    if (exe==0) { // Pause to scan some slides.
        int ll = dir_->NSli();
        if (ll > 0) {
            if (sp_ < stks_) { stack_[++sp_] = idx_; }
                        else ier = -2; // Stack Overflow.
            if (id<0) idx_ = ll; else idx_ = 1;
            sli_ = dir_->Slide( idx_ );
        } else ier = -4; // DirEnt without Slide.
        if (id != sid) ier = 1; // To signal when the volume limits are reached.
    }
    return ier;
} // int SlideExec::Continue( int id, int pr ).



int SlideExec::SlideScan( int id )
// This routine is used to select a slide in the current directory.
// The id argument is the slide number in the directory.
// To give some usefull facilities the special id values 0, -1 or -2
// can be used. On the first call after a DirPush call
// id = 0 select the first slide (equivalent to id = 1), and
// id = -2 select the last slide of the directory. For all next
// SlideSetup calls when id = 0 the next slide is selected and the
// previous one when id = -1.
// The return value -1 is the indicator of a not opened directory
// or volume, The returned value of 1 flags any tentative to get
// a slide with an index out of the range and the 0 value is
// returned when no error occure.
//
{
    int     len;
    int ie = -1;

    SliEntry * psli = sli_;

    if (dir_&&(len = dir_->NSli())) {
        // dir_-> is a directory descriptor with slide(s).
        if (len) { // This directory must have some slides.
            ie = 0;
            switch (id) {
                case -2: idx_ = len; break;
                case -1: idx_--; break;
                case  0: idx_++; break;
                default: idx_ = id; break;
            }
            // We check for index limits.
            if (id < 0) {
                if (idx_ < 1) { idx_ = 1; ie = 1; }
            } else {
                if (idx_ > len) { idx_ = len; ie = 1; }
            }
            sli_ = dir_->Slide( idx_ );
        }
    }
    // Save previous slide ranking numbers only when we scan to right direction (to next slide).
    if (psli&&(!id)) psli->SliRkgGet( prsn_, prfl_, prpa_, prch_, len );
                else PrvRkgClear();
    return ie;
} // int SlideExec::SlideScan( int id ).



int SlideExec::SetPosition( int pc, int idx )
{
    int ier = 0;

    if (idx>0) {
        while (!ier&&(pc_ < pc)) ier = Continue( 0, SlidePrior );
        if (idx < 1) idx = 1;
                else if (idx > dir_->NSli()) idx = dir_->NSli();
        sli_ = dir_->Slide( idx );
        idx_ = idx;
    }
    return ier;
} // void SlideExec::SetPosition( int pc, int idx ).



int SlideExec::SlideMatch( int ss, int sf, int sp, int sc, const char *st )
{
    int s = sli_->Numb(), f = sli_->Film(),
        p = sli_->Para(), c = sli_->Chap();
    int nfn = 0;  // Assume as found.
    char    * t;
    if (st&&st[0]) t = sli_->Text();
              else t = NULL;
    
    if (!(s|f|p|c)) return 1;
    if (ss&&(ss != sli_->Numb())) nfn = 1;
    if (sf&&(sf != sli_->Film())) nfn = 1;
    if (sp&&(sp != sli_->Para())) nfn = 1;
    if (sc&&(sc != sli_->Chap())) nfn = 1;
    if (st&&st[0]) {
        if (t&&t[0]) nfn = (strstr( t, st ))? 0 : 1;
                else nfn = 1;
    }
    return nfn;
} // int SlideExec::SearchInit( int &i1, int &i2, const char *s ).



int SlideExec::Search( int id, int sc, int sp, int sf, int ss,
                       const char * st, int lpflg )
// Slide Search function:
// sf = 0, ss = 0, scom = NULL to accept any value of this field.
// For scom the character * can be used to accept a partial match string (in future!).
{
    int     ier = 0,   nfn = 0,
                pc2,       id2;
    SliEntry      * osl = sli_;

    if (st&&st[0]&&(st[0] == '*')&&!st[1]) st = NULL;
    if (!(sc||sp||sf||ss||st)) return 0; 
    if (!osl) return 0;

    if (spc_ < 0) {     // Initialize the search.
        spc_ =  pc_;    // Save actual pc_ and idx_.
        sid_ = idx_;
    }
    pc2 = pc_; id2 = idx_;                      // Save current position.

    sdi_ = (id < 0)? -1: 0;
    lpf_ = 0;
    if (id == 0) lpflg = 1;                     // To orce a temporary Loop mode.
    do { // Search Loop.
        ier = SlideScan( sdi_ );                // Skip to next/previous slide.
        if (ier > 0) {                          // When the directory limits are reached ...
            ier = Continue( sdi_, SlidePrior ); // ... we try to skip to next/previous One.
                                                // ... skip to next/previous directory.
            if (lpflg && ier > 0) {
                ier = Start( id, SlidePrior );  // Restart scan from end or begin.
                lpf_ = 1;                       // Set the Loop flag.
            }
        }
        if (osl == sli_) break;                 // Stop on loop when not found.
        nfn = SlideMatch( ss, sf, sp, sc, st );
    } while (nfn&&!ier);
    if (nfn) {
        Abort();
        SetPosition( pc2, id2 );
        ier = 1;
    }
    return ier;
} // int SlideExec::Search(  ).



int SlideExec::SearchRestore( int fl )
// To quit the search mode.
{
    if (fl&&(spc_ >= 0)) {              // When the origianl slide return is required.
        Abort();                        // Restart the directories scan sequence.
        SetPosition( spc_, sid_ );      //  Go to original slide (before search).
    }
    spc_ = -1;                          // Clear the original slide memory.
    return 0;
} // int SlideExec::SearchRestore().



//
// end of <PW-Id>.
//
