//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module to define and manage the Slide object
// by using the image manager for OpenGL ImageMan_REF.
//


#ifndef DIAVIEWER_DIA
#define DIAVIEWER_DIA

#include "ImageMan_REF.h"
#include "Service_Util.h"

#include "DiaViewer_GBL.h"



//
// Define the Unique Slides Identifiers field (power of 2 -1).
//
// Define the IIde sharing of bits between slides, Directories and volumes.
#ifndef UNIQUE_IDE_FIELDS
#   ifdef L64IDENT
// For (2**32-1) slides, (2**16-1) foolders and (2**16-1) volumes.
#       define IIDE_MASK ~((IIde_t)0);
#       define UNIQUE_IDE_FIELDS {32,16}
#   else
// For (2**14-1) slides, (2**10-1) foolders and (2**8-1) volumes.
#       define UNIQUE_IDE_FIELDS {14,10}
#   endif
#endif


// Define the context file name. (one for each volume directory).
#define  CNTX_FNAME  ".DiaViewer.cntxdat"


//
// Define the various flags for the volume and directory class descriptors (flags on 16 bits).
//
typedef enum {
    Context_Update    =      1, // The Updated Flag ...
    Context_SubDir    =      2, // Some subdiretory updated.

    Context_RkgEna    =      4, // Ranking in progress.
    Context_RkgCpl    =      8, // Ranking completed.

    Context_Slinbr    =   0x10, // Not null slide numbers (to keep).
    Context_SText     =   0x20, // A text is present (to keep).
    Context_Orient    =   0x40, // Slide orientation specified (to keep).
    Context_PrgPos    =   0x80, // Program Slide Position sepcified.

    Context_ProInb    =  0x100, // Write-Delete Protect for slide numbers.
    Context_ProTxt    =  0x200, // Write-Delete Protect for Text comment.
    Context_ProOri    =  0x400, // Write-Delete Protect for slide orientation.

    Context_RkgAuto   = 0x1000, // Viewer flag for Auto Number Generation (for Viewer).
    Context_RkgEdit   = 0x2000, // Viewer flag for Ranking Data edition (for Viewer).
    Context_VnaEdit   = 0x4000, // Viewer flag for slide volume name edition enabled.

    Context_LckRank   = 0x8000, // To lock data for ranking (cannot change ranking data if set).
    Context_SeaLoop  = 0x10000, // Loop flag for Slide Search.

    Context_ChNoLnk = 0x400000, // Set to unlink Chapter and Paragraph numbers from Film and slide numbers.

    Context_VDiRam = 0x1000000, // Flag of Slide Show Volume.
    Context_VRkMap = 0x2000000, // Flag of Ranking Volume.
    Context_RankOn = 0x4000000, // Flag of volume in ranking.

    Context_MemImg = 0x8000000, // To Flag a memory Volume (ReadOnly Context).

    // Mask for slide context data.
    Context_Data      =  Context_Orient|Context_Slinbr|Context_SText, 
    Context_Protect   =  Context_ProInb|Context_ProTxt|Context_ProOri|Context_MemImg,
    Context_Viewer    =  Context_RkgAuto|Context_RkgEdit|Context_VnaEdit|Context_MemImg

} Context_Flags;



// Function to change the IIde_t Fields setting.
void SetIIdeFields( int, int );

class DirEntry;
class VolEntry;


typedef enum {          // Defines the modes of slides volumes list.
    VolMnf_VList   = 0, // List of all defined slide volumes.
    VolMnf_Rename,      // List of slide volumes all defined that can be rename.
    VolMnf_ToOpen,      // List of all defined not open slides volumes.
    VolMnf_Open,        // List of open slide volumes (excluding memory ones).
    VolMnf_SOpen        // List of open slide volumes including memory ones.
} VolMnf_t;


typedef struct {        // Define the small record of any Known volume.
    VolEntry    * volm; // Pointer to Volume record.
    char        * path; // The Volume path.
    char        * name; // The volume name.
    int           iide; // The Volume integer identifier.
} VolRef;


// Define the Slide/Diapositive Class.
class SliEntry : public Image_REF {

public:
    
    char      * stext_; // Slide (diapositive) attached comment.
    DirEntry  * owner_; // Pointer to the owner.
    UInt32      uflgs_; // The slides flags.
    UInt16     slid_n_, // The slide index and the film index used ...
               film_n_, // ... to define the new slide filename (can be tfilm #).
               para_n_, // Paragraph index (user defined for thematic classement).
               chap_n_; // Chapter index (user defined for thematic classement).

    SliEntry( const char *, IIde_t = 0 );
    SliEntry( const uchar *, int, IIde_t = 0, char * = NULL );

    ~SliEntry();

    int   SetSliIIde(); // To get an identifier for a slide.

    int    FlgTst( int fl ) { return uflgs_ & fl; }
    int    FlgTvl( int fl ) { return (uflgs_ & ~fl) == fl; }
    void   FlgSet( int fl ) { uflgs_ |=  fl; }
    void   FlgClr( int fl ) { uflgs_ &= ~fl; }

    int    FlgVal() { return uflgs_; }         
    void   FlgVal( int fl ) { uflgs_ = fl; }

    void   Text( char * text ) { stext_ = text; }
    void   Owner( DirEntry * dir ) { owner_ = dir; }
    void   Numb( int slnb )  { slid_n_ = slnb; }
    void   Film( int film )  { film_n_ = film; }
    void   Para( int para )  { para_n_ = para; }
    void   Chap( int chap )  { chap_n_ = chap; }

    
    char * Text()  { return stext_; }
    DirEntry * Owner() { return owner_; }
    int    Numb()  { return slid_n_; }
    int    Film()  { return film_n_; }
    int    Para()  { return para_n_; }
    int    Chap()  { return chap_n_; }
    void   Rkg_SliProt( int, int );

    void   SliRkgClear() { slid_n_ = film_n_ = para_n_ = chap_n_ = 0; }
    void   SliRkgSet( int sl, int fl, int pa, int ch, int pr )
                    { slid_n_ = sl; film_n_ = fl; para_n_ = pa; chap_n_ = ch; FlgVal( pr ); }
    void   SliRkgGet( int &sl, int &fl, int &pa, int &ch, int &pr )
                    { sl = slid_n_; fl = film_n_; pa = para_n_; ch = chap_n_;  pr = FlgTst( Context_Protect ); }
    int    SliRkgTest() { return (slid_n_&&film_n_&&para_n_&&chap_n_&&FlgTst( Context_Protect )); }

    void   SliCntxFlags();

    void   RkgClear();

    const char * BltPath();

};



// Define the Directory of Slide/Diapositive Class.
class DirEntry { // Define the directory of Slides class.
    friend SliEntry;

private:

    const char *fname_, // Filename of the directory.
               *stext_; // Attached comment.
    DirEntry  * owner_, // Directory Owner (NULL for a Volume root directory).
              ** dlst_; // Directory table.
    SliEntry  ** slst_; // Slide Table.
    IIde_t      ident_; // The directory identifier.
    UInt32      flags_; // Directory flags word.
    int          deep_, // Deep of the directory.
                udeep_, // User deep to use for classement.
                 ndir_, // Number of sub-directories.
                 nsli_, // Number of Slides.
                cnsli_; // Cumulative number of slides.

    int   SetDirIIde(); // To get an identifier for a directory.

    void     BltPath(); // Optimized reverse build of slide path, a sub-routine ...
                        // ... of SliEntry::BltPath().

public:

    DirEntry( const char *name, IIde_t = 0 );
    
    ~DirEntry();
    
    const char * FName() { return fname_; }
    const char * SText() { return stext_; }
    IIde_t       Ident() { return ident_; }
    DirEntry   * Owner() { return owner_; }
    int          LDeep() { return  deep_; }
    int          UDeep() { return udeep_; }
    int          NDir()  { return  ndir_; }
    int          NSli()  { return  nsli_; }
    int          CnSli() { return cnsli_; }

    int          FlgTst( int fl ) { return flags_ & fl; }
    void         FlgSet( int fl ) { flags_ |=  fl; }
    void         FlgClr( int fl ) { flags_ &= ~fl; }

    int          FlgVal() { return flags_; }         
    void         FlgVal( int fl ) { flags_ = fl; }

    DirEntry   * SbDir( int idx = 0 );
    SliEntry   * Slide( int idx = 0 );

    void   FName( const char * fname ) { fname_ = fname; }
    void   SText( const char * stext ) { stext_ = stext; }
    void   Ident( IIde_t ide ) { ident_ = ide; }
    void   Owner( DirEntry * own ) { owner_ = own; }
    void   LDeep( int dp ) { deep_ = dp; }
    void   UDeep( int dp ) { udeep_ = dp; }
    void   CnSli( int cns ) { cnsli_ = cns; }

    void   SbDir( int, DirEntry** );
    void   Slide( int, SliEntry** );

    int    ReadCntxFile( const char * );
    void   WriteCntxFile( const char * );
    void   WriteDirCntx();

    void   SetDirIdents();

    SliEntry * SliLocate( const char * );

    void   RkgClear();

};



// Define the DirReference class to define arbitrary slide order scan.
class DirRefer {

private:

    typedef struct Sli_Refer {
        struct Sli_Refer     * rfs_nxt_; // Pointer to the next slide reference.
        char                 * rfs_nam_; // File name of the new slide file.
        SliEntry             * rfs_sli_; // Slide descriptor pointer in the old slide volume.
    } SliRefer;


    DirRefer       * rfd_own_; // Owner directory reference (Null for the root volume).
    SliRefer       * rfd_frs_, // Pointer to the first and last slide references,
                   * rfd_lst_;
    int              rfd_nsl_; // Number of slide reference in the directory reference.

public:

    DirRefer( DirRefer * );                     //
    DirRefer * DelRef( DirRefer * );
    void AddSliRef( const char *, SliEntry * );
    void WriteDirCntxFile( int, const char * );
};



// Define the Volume of Slide/Diapositive Class.
class VolEntry { // Define the Volume of Slides class.
    friend SliEntry;
    friend DirEntry;

private:
    VolRef    * volrf_; // Volume reference pointer.
    DirEntry  * rootd_; // Volume root directory descriptor pointer.
    void      * scprg_; // Pointer to related scan program object.
    IIde_t      ident_; // Integer volume identifier.
    UInt16      tdeep_, // Deep of the directory.
                nwind_; // Count of window open that display the volume.
    UInt32      flags_; // Volume flags.
    int         tndir_, // Total number of directory.
                tnsli_, // total number of slides.
                index_, // Volume setup index.
                prcnt_, // Saved program count ...
                pridx_; // ... and slide index.

    int   SetVolIIde(); // To get an identifier to a slide.

    void RevPath();

public:

    VolEntry( const char *name, DirEntry * root = NULL, VolRef * = NULL );

    ~VolEntry();
    
    const char   * VName()  { return volrf_->name; }
    DirEntry     * VRoot()  { return rootd_; }
    void         * RelPrg() { return scprg_; }
    IIde_t         Ident()  { return ident_; }
    int            TDeep()  { return tdeep_; }
    int            TnDir()  { return tndir_; }
    int            TnSli()  { return tnsli_; }
    VolRef       * VRefer() { return volrf_; }

    int          FlgTst( int fl ) { return flags_ & fl; }
    void         FlgSet( int fl ) { flags_ |=  fl; }
    void         FlgClr( int fl ) { flags_ &= ~fl; }

    void         FlgVal( int fl ) { flags_ = fl; }
    int          FlgVal() { return flags_; }
/*
    int          VflTst( int fl ) { return volfl_ & fl; }
    void         VflSet( int fl ) { volfl_ |= fl; }
    void         VflClr( int fl ) { volfl_ &= ~fl; }
*/
    int            PrgPc()  { return prcnt_; }
    int            PrgId()  { return pridx_; }
    void           PrgPc( int pc ) { prcnt_ = pc; }
    void           PrgId( int idx ) { pridx_ = idx; }
    int            NWind() { return nwind_; }
    void           WinSel() { nwind_++; }
    int            WinUnSel() { return (--nwind_)? 0: 1; }

    void    VRefer( VolRef * vref ) { volrf_ = vref; }
    void    VName( const char * name );
    void    VRoot( DirEntry * dir ) { rootd_ = dir; }
    void    RelPrg( void* prg ) { scprg_ = prg; }
    void    Ident( IIde_t ide ) { ident_ = ide; }
    void    TDeep( int tde ) { tdeep_ = tde; }
    void    TnDir( int tnd ) { tndir_ = tnd; }
    void    TnSli( int tns ) { tnsli_ = tns; }

    void    SetVolContext();
    void    WriteVolContext();

    void    SetVolIdents();

};



//
// Some global definitions.
//
//

//
// Global variables attached to the DIA module.
//
extern int           OpenVolNb; // Number of opened volume(s).
extern VolEntry  * IniSelVolum; // Pointer of volume selected at DiaViewer init (by DCM module) time.
extern SliEntry  * IniSelSlide; // Pointer to the selected slide.


//
// Functions defined in the Main module.
//
void SaveAllApplContext( VolEntry * );  // Save Application context before exit.



//
// Functions defined in DiaViewer_DLG module
//
int SlideVolSelect( const char * title, VolEntry * &vol, int = 0 );


#endif

//
// end of <PW-Id>.
//
