{
*************************************************************************
*                                                                       *
*                                                                       *
*        V I E W - 3 D  (View Powder Diffraction Data in 3D)            *
*                                                                       *
*               ( ILL Data Base Manager Source File )                   *
*                                                                       *
*                  Version  1.0-A  - - 30-Nov-2009                      *
*                                                                       *
*                                by                                     *
*                                                                       *
*                   Pierre Wolfers, Institut Neel                       *
*                                                                       *
*            CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166            *
*                                                                       *
*                       F 38042 GRENOBLE CEDEX 9                        *
*                                                                       *
*                             F R A N C E                               *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}

module VIEW3D_DATA_INP;


%include 'RPWSRC:view3d_env.pas' {, list_on};

const
  mimast       = 1.0E+10;                       { Value to init the minimum and maximum }



procedure INIT_PATTERN( var r: pat_rec );
begin
  with r, mai_parm do
  begin
    next       :=          nil;                 { Default is to have no successor }
    prev       :=     pat_last;                 { The predecessor is always the last created pattern }
    lab_lst    :=          nil;                 { No label record }
    idcnt      :=            0;                 { Set pattern index to null index value }
    numor      :=           -1;                 { No numor }
    dateh.length        :=   0;                 { Init the pattern identification to empty }
    sample.length       :=   0;
    comment.length      :=   0;
    monitor    :=          0.0;                 { No monitor never counting time }
    cnttime    :=          0.0;

    dtheta     :=        th0_r;                 { Get initial 2 theta angle }
    omega      :=          0.0;                 { Set Eulerian Angles to 0.0 }
    chi        :=          0.0;
    phi        :=          0.0;
    tr1        :=          0.0;
    tr2        :=          0.0;
    step       :=        ths_r;                 { Set the fixed theta step }
    t_set      :=          0.0;                 { Set all temperature to 0.0 (no meaning) }
    t_reg      :=          0.0;
    t_sample   :=          0.0;
    lambda1    :=          0.0;                 { No defined Wave length }
    lambda2    :=          0.0;
    rwave2     :=          1.0;
    rvp2       :=          0.0;                 { Init to zero all Free parameter fields }
    rvp3       :=          0.0;
    rvp4       :=          0.0;
    rvp5       :=          0.0;
    rvp6       :=          0.0;
    rvp7       :=          0.0;
    rvp8       :=          0.0;
    rvp9       :=          0.0;
    ista       :=            0;
    iend       :=            0;
    ivp0       :=            0;
    ivp1       :=            0;
    amin       :=        1e+10;
    amax       :=        -amin;
    min        :=        1e+10;
    max        :=        - min
  end
end INIT_PATTERN;



procedure PATTERN_INFO_COPY( var src, dst: pat_rec );
begin
  with dst do
  begin
    next       :=          nil;                 { Default is to have no successor }
    prev       :=     pat_last;                 { The predecessor is always the last created pattern }
    lab_lst    :=  src.lab_lst;                 { Label records list }
    idcnt      :=    src.idcnt;                 { Pattern index }
    numor      :=    src.numor;                 { Numor }
    dateh      :=    src.dateh;                 { Init the pattern identification to empty }
    sample     :=   src.sample;
    comment    :=  src.comment;
    monitor    :=  src.monitor;                 { Monitor }
    cnttime    :=  src.cnttime;

    dtheta     :=   src.dtheta;                 { Copy initial 2 theta and ... }
    omega      :=    src.omega;                 { ... Eulerian Angles }
    chi        :=      src.chi;
    phi        :=      src.phi;
    tr1        :=      src.tr1;
    tr2        :=      src.tr2;
    step       :=     src.step;                 { Set the fixed theta step }
    t_set      :=    src.t_set;                 { Set all temperature to 0.0 (no meaning) }
    t_reg      :=    src.t_reg;
    t_sample   := src.t_sample;
    lambda1    :=  src.lambda1;                 { No defined Wave length }
    lambda2    :=  src.lambda2;
    rwave2     :=   src.rwave2;
    rvp2       :=     src.rvp2;                 { Init to zero all Free parameter fields }
    rvp3       :=     src.rvp3;
    rvp4       :=     src.rvp4;
    rvp5       :=     src.rvp5;
    rvp6       :=     src.rvp6;
    rvp7       :=     src.rvp7;
    rvp8       :=     src.rvp8;
    rvp9       :=     src.rvp9;
    ista       :=     src.ista;
    iend       :=     src.iend;
    ivp0       :=     src.ivp0;
    ivp1       :=     src.ivp1;
    amin       :=     src.amin;
    amax       :=     src.amax;
    min        :=      src.min;
    max        :=      src.max
  end
end PATTERN_INFO_COPY;



[global]
procedure INIT_LOAD;
var
  p: pat_ptr;

begin
  { Init the selection table }
  for i := 1 to npat_max do  tbpat[i] := nil;
  npat := 0;                                    { No pattern in memory }

  tmin  := mimast; tmax  := - tmin;             { Initialize all minimaxi }
  thmin := mimast; thmax := -thmin;
  gmin  := mimast; gmax  := - gmin;
  smin  := mimast; smax  := - smin;

  { Free all existing patern allocations }
  while pat_first <> nil do
  begin
    p := pat_first;
    pat_first := p^.next;
    DISPOSE( p )
  end;
  pat_last := nil;
  sel_pat  := nil
end INIT_LOAD;



[global]
procedure SET_PATTERN_SELECTION;
{ Set the pattern display/integration selection }
var
  p:           pat_ptr;
  ipat, nda:   integer;
  bs:          boolean;

begin
  smin := mimast; smax := - smin;
  nda := 0;
  p := pat_first;
  ipat :=      1;
  while p <> nil do
  begin { Loop on all loaded patterns }
    bs := false;
    if not bs then {See for temperature selection }
      for ij := 1 to ndiat do
        if fc_celsius in funcs then
        begin
          if ABS( p^.t_set - tbtem[ij] - Ice_Temp ) < 1.0E-3 then bs := true
        end
        else if ABS( p^.t_set - tbtem[ij] ) < 1.0E-3 then bs := true;
    if bs then
    begin { When this pattern must be selected }
      if smin > p^.min then smin := p^.min;     { Update the display/integration mini-maxi ... }
      if smax < p^.max then smax := p^.max;
      nda := nda + 1;                           { ... and set the pattern selection }
      tbpat[nda] := p
    end;
    ipat := ipat + 1;
    p := p^.next
  end;
  for ij := nda + 1 to 2*npat_max do tbpat[ij] := nil
end SET_PATTERN_SELECTION;



[global]
procedure GET_PATTERNS;
{ Procedure to read all the pattern in the input data file (D1B/D20 format) }
var
  nda:               integer;
  rfrm:         string( 14 );
  ermsg:              string;
  err_of_read:       boolean;

label
  ERR_RECOVER;


  procedure RPWF_ERROR( in_var msg: string );
  begin
    WRITEV( ermsg, 'Read Pattern format"', rfrm, '" error: ', msg );
    WRITELN( 'VIEW-3D-DATA: ', ermsg );
    goto ERR_RECOVER
  end RPWF_ERROR;



  procedure SKIP_INP_CHAR( nn: integer );
  var
    ch: char;

  begin
    for ii := 1 to nn do READ( inp, ch )
  end SKIP_INP_CHAR;



  procedure ELLIMINATE_CR( var st: string );
  const
    CR      = CHR( 13 );                        { ASCII Character 13 to elliminate }

  begin
    with st do
      if length > 0 then
        if body[length] = CR then length := length - 1
  end ELLIMINATE_CR;



  procedure GET_RPWSTD_FORMAT;
  var
    p:                                  pat_ptr;            { Pointer of new pattern record }
    llab, clab:                       label_ptr;            { Pointer to the last label }
    line:                                string;            { The current line }
    word:                          string( 32 );            { The current word }
    ip, ij, len, npa:                   integer;            { Data size in pattern points and service integers }
    mi, ma:                                real;            { Pattern min-maxi }
    pat_model:                     pat_rec( 1 );            { Model pattern }
    bnrd, beof:                         boolean;


  procedure SKIP_SPACE;
  begin
    while (ip > 0) and (line[ip] <= ' ') do ip := ip + 1
  end SKIP_SPACE;


  procedure GET_NEW_LINE;
  begin
    line.length := 0;
    while not EOF( inp ) and (line.length = 0) do
    begin
      READLN( inp, line );
      if line.length > 0 then
        if (line[1] <> '#') and (line[1] <> '!') then
        begin
          while (line.length > 0) and (line[line.length] <= ' ') do line.length := line.length - 1;
          ip := 1
        end
        else line.length := 0
      else
        if EOF( inp ) then beof := true
    end
  end GET_NEW_LINE;



  procedure GET_ONE_WORD( var word: string; bwf: boolean := true );
  begin
    word.length := 0;
    while (ip >= 0) and (word.length = 0) do                { When the current line is not finished ... }
      READV( line:ip, word:word.capacity:true );            { ... we read a new word }
    if (word.length > 0) and bwf then
    begin
      SET_CASE( word, true );                               { Force the word to be in Major Case }
      if word.length > 16 then word.length := 16            { Limit the word for Keyword comparaizon }
    end
  end GET_ONE_WORD;



  begin { GET_RPWSTD_FORMAT }
    rfrm := 'FPD_DATA';
    beof := false;
    npa := 0;
    READLN( inp, word:word.capacity:true );     { Get the first line with the FileType word name "SPDF_DATA" }
    SET_CASE( word );
    if word <> rfrm then RPWF_ERROR( 'Bad data file format' );

    GET_NEW_LINE;                               { Get a first pattern line (of the first pattern in the file) }
    GET_ONE_WORD( word );                       { Get the related Directive name }
    while not EOF( inp ) and (word <> 'EOF') do { Loop on all pattern(s) in this file until End Of File }
    begin
      if word <> 'PATTERN' then RPWF_ERROR( 'Format error : each pattern must begin by the "PATTERN" word' );

      INIT_PATTERN( pat_model );                { Init the Pattern model }

      npa := npa + 1;

      bnrd := false;

      with pat_model do
      begin
        p := nil;
        SKIP_SPACE; READV( line:ip, comment );  { Get the pattern comment }
        repeat
          if bnrd then bnrd := false
          else
            repeat
              GET_NEW_LINE;                     { Get a new data line }
              if beof then RPWF_ERROR( 'Incomplete Pattern, EOF reached' );
              GET_ONE_WORD( word )              { Get the directive name }
            until word.length > 0;
          { Dispatch following the directive }
          if word = 'LABEL'       then begin
                                         SKIP_SPACE;
                                         NEW( clab, line.length - ip + 1 );
                                         READV( line:ip, clab^.txt );
                                         clab^.nxt := nil;
                                         if lab_lst = nil then lab_lst := clab
                                                          else llab^.nxt := clab;
                                         llab := clab
                                       end
          else
          if word = 'IDNUMBER'    then READV( line:ip, idcnt )
          else
          if word = 'NUMOR'       then READV( line:ip, numor )
          else
          if word = 'DATEH'       then GET_ONE_WORD( dateh )
          else
          if word = 'SAMPLE'      then begin  SKIP_SPACE; READV( line:ip, sample )  end
          else
          if word = 'LAMBDA'      then READV( line:ip, lambda1, lambda2, rwave2 )
          else
          if word = 'TEMP'        then READV( line:ip, t_set, t_reg, t_sample )
          else
          if word = 'MONTIME'     then READV( line:ip, monitor, cnttime )
          else
          if word = 'TRANSLATION' then READV( line:ip, tr1, tr2 )
          else
          if word = 'PARAMETER'   then begin
                                         READV( line:ip, ij );
                                         case ij of
                                           0: READV( line:ip, ivp0 );
                                           1: READV( line:ip, ivp1 );
                                           2: READV( line:ip, rvp2 );
                                           3: READV( line:ip, rvp3 );
                                           4: READV( line:ip, rvp4 );
                                           5: READV( line:ip, rvp5 );
                                           6: READV( line:ip, rvp6 );
                                           7: READV( line:ip, rvp7 );
                                           8: READV( line:ip, rvp8 );
                                           9: READV( line:ip, rvp9 );
                                         otherwise
                                         end
                                       end
          else
          if word = 'ORIENTATION' then READV( line:ip, omega, chi, phi )
          else
          if word = 'DATA' then                 { Directive for powder data }
          begin
            if p <> nil then RPWF_ERROR( 'Format error : Two DATA BLOCK directives for the same pattern' );
            READV( line:ip, len );              { Get the size of data table }
            NEW( p, len );                      { Allocate the Pattern record with the necessary data table size }
            amin := mimast; amax := - amin;     { Set the minimum and maximum of 2*theta }
            mi := mimast; ma := - mi;           { Prepare for mini-maxi computing of counts }

            for ii := 1 to len do               { Loop on all points of the pattern }
              with p^.dat[ii] do
              begin
                GET_NEW_LINE;
                READV( line:ip, flag, theta, int );
                if ip > 0 then READV( line:ip, sig )
                          else sig := SQRT( int );
                if mi > int then mi := int;
                if ma < int then ma := int;
                if amax < theta then amax := theta;
                if amin > theta then amin := theta;
                mflg := []
              end;

            GET_NEW_LINE;
            GET_ONE_WORD( word );
            if word <> 'EOD' then bnrd := true
          end
          { Ignore any exotic directive }
        until beof or (word = 'PATTERN') or (word = 'EOF');

        { Update the common pattern and global data }

        dtheta := amin;
        min := mi; max := ma;
        if t_set <> 0.0 then
        begin
          if tmin > t_set then tmin := t_set;
          if tmax < t_set then tmax := t_set
        end;
        if thmin > amin then thmin := amin;
        if thmax < amax then thmax := amax;
        if gmin > mi then gmin := mi;
        if gmax < ma then gmax := ma;

        if dat_celsius or (t_sample <= 0.0) then
        begin                                   { Any null or negative temperature value is ... }
          t_set    := t_set    + Ice_Temp;      { ... taken as provided in Celsius degrees }
          t_reg    := t_reg    + Ice_Temp;
          t_sample := t_sample + Ice_Temp
        end
      end { with ... };

      PATTERN_INFO_COPY( pat_model, p^ );       { Copy the Pattern specification in the resulting pattern record }

      { Set the pattern in the pattern list }
      if pat_first = nil then pat_first := p
                         else pat_last^.next := p;
      pat_last := p;
      { Set this pattern in selection table if required }
      npat := npat + 1;
    end { while ... loop on all file's patterns }
  end GET_RPWSTD_FORMAT;



  procedure GET_D1B_FORMAT;
  var
    p:                                 pat_ptr; { Pointer to the new pattern record }
    line1, line2, line3, line4:         string; { String to keep the first four lines }
    i, len, nlupat:                    integer;
    mi, ma, trash:                        real;
    bs:                                boolean;
    sp:                                   char; { Space character to read }

  begin
    rfrm := 'D1B';
    READLN( inp, nlupat );                      { Ignore the first line (where the number of pattern can be specified) }
    while not EOF( inp ) do                     { Loop on all pattern(s) }
    begin
      READLN( inp, line1 );
    exit if EOF( inp );
      READV( line1, line2:0:true );             { Try to find the "-10000" of the end of data file }
    exit if line2 = '-10000';
      READLN( inp, line2 );
      READLN( inp, line3 ); READLN( inp, line4 );
      READV( line4, len );                      { Get the size of pattern }

      NEW( p, len );                            { Create the pattern record }
      INIT_PATTERN( p^ );
      with p^ do
      begin
        READV( line1, dateh:20, sp, sample:10, sp, comment );   { Get the date, sample name, and comment }
        READV( line2, idcnt, numor );           { Get the pattern index and NUMOR }
        if line4[11] = '.' then
        begin { CNRS D1B file type }
          READV( line3, monitor, cnttime, dtheta,       { Get the monitor and counting time, ... }
                        omega,   chi,     phi,          { ... Possible Eulerian angles, ... }
                        tr1,     tr2,     step,         { ... X,Y sample position and 2theta step, ... }
                        t_set,   t_reg,   t_sample );   { ... Temperatures setting, on regulation and on sample. }
          READV( line4, len,     lambda1, rvp2,         { Get the Size of pattern, wave Length and all free parameters }
                        rvp3,    rvp4,    rvp5,
                        rvp6,    rvp7,    rvp8,
                        rvp9,    ivp0,    ivp1 )
        end
        else
        begin
          READV( line3, monitor, cnttime, dtheta,       { Get the monitor and counting time, ... }
                        omega,   chi,     phi,          { ... Possible Eulerian angles, ... }
                        trash,   lambda1, step,         { ... X,Y sample position and 2theta step, ... }
                        t_set,   t_reg,   t_sample );   { ... Temperatures setting, on regulation and on sample. }
          READV( line4, len,     rvp2,    rvp3,         { Get the Size of pattern, wave Length and all free parameters }
                        rvp4,    rvp5,    rvp6 )
        end;
        if dat_celsius or (t_sample <= 0.0) then
        begin                                   { Any null or negative temperature value is ... }
          t_set    := t_set    + Ice_Temp;      { ... taken as provided in Celsius degrees }
          t_reg    := t_reg    + Ice_Temp;
          t_sample := t_sample + Ice_Temp
        end;
        lambda2 := 0.0;                         { Always for D1B }
        amin := dtheta;                         { Set the minimum and maximum of 2*theta }
        amax := dtheta + (dim - 1)*step;
        mi := mimast; ma := - mi;

        i := 0;
        ista := 1; iend := dim;                 { Set the default using }
        while i < dim do
        begin
          with dat[i + 1] do
          begin
            theta := dtheta + i*step;
            READ( inp, flag, int );             { Read each point of the diagram }
            sig := SQRT( ABS( int ) );
            mflg := [];                         { Set the marker flag to empty }
            if flag <= 0 then mflg := [mk_invalid];
            { Set the diagram minimum and maximum }
            if mi > int then mi := int;
            if ma < int then ma := int
          end;
          i := i + 1
        end;
        { Set pattern minimaxi }
        min := mi; max := ma;
        { Update the general minimaxi }
        if t_set <> 0.0 then
        begin
          if tmin > t_set then tmin := t_set;
          if tmax < t_set then tmax := t_set
        end;
        if thmin > amin then thmin := amin;
        if thmax < amax then thmax := amax;
        if gmin > mi then gmin := mi;
        if gmax < ma then gmax := ma
      end;
      { Set the pattern in the pattern list }
      if pat_first = nil then pat_first := p
                         else pat_last^.next := p;
      pat_last := p;
      { Set this pattern in selection table if required }
      npat := npat + 1;
      READLN( inp );                            { Skip to end of current count line }
      READLN( inp, nlupat )                     { Read the End Pattern mark (should be -1000) }
    end
  end GET_D1B_FORMAT;



  procedure GET_XYD_FORMAT;
  const
    pat_msize = 1000000;                        { Maximum pattern size }

  var
    p:                                 pat_ptr; { Pointer to the new pattern record }
    nc:                                integer; { Pattern size }
    word:               [static]  string( 32 );
    line:               [static] string( 255 ); { Current line }
    xfc, yfc, stpin,                            { Internal X and Y multiplicators, and stepin }
    xmi, ymi, xma, yma, xx, yy, sg:       real;
    ch: char;
    ip, np, interp:                    integer;
    hd, bt: boolean;

  begin
    rfrm := 'XYDATA';
    { Size the Pattern }
    nc := -6;                                   { Initialize the pattern size count (with six lines for the header) }
    while not EOF( inp ) do
    begin
      READLN( inp, line );
      if line.length > 0 then
        if (nc < 0) or 
           ((nc >= 0) and
           (line[1] <> '!') and (line[1] <> '#')) then
          nc := nc + 1                          { Do not count the comment lines in the pattern }
    end;
    CLOSE( inp );
    RESET( inp, inp_name );                     { Re-Open the data file to simulate a rewind }

    READLN( inp, line );                        { Read the first XYDATA Line }
    while (line.length > 0) and (line[line.length] < ' ') do line.length := line.length - 1;
    ip := 1;
    word.length := 0;
    while (ip < line.length) and (word.length = 0) do   { Look for the word "XYDATA" }
    begin
      READV( line:ip, word:16:true );
      if word = '#' then word.length := 0       { Ignore initial "#" character }
    end;
    SET_CASE( word, true );                     { Put the word in Capital characters }
    hd := word = 'XYDATA';                      { Select the header mode }
    NEW( p, nc );                               { Allocate the pattern record }
    INIT_PATTERN( p^ );
    with p^, mai_parm do
    begin
      xfc := 1.0; yfc := 1.0;
      stpin := 0.0001; interp := 0;
      step   := 0.001;
      if hd then
      begin
        READV( line:ip, ch, comment );          { Set the comment }
        for ii := 1 to 5 do                     { Loop on all others header lines }
        begin
          READLN( inp, line );
          while (line.length > 0) and (line[line.length] < ' ') do line.length := line.length - 1;
          ip := 1;
          word.length := 0;
          while (ip < line.length) and (word.length = 0) do     { Look for the word "XYDATA" }
          begin
            READV( line:ip, word:16:true );             { Get a word }
            if word = '#' then word.length := 0         { Ignore initial "#" character }
          end;
          SET_CASE( word );                             { Put the word in Capital characters }
          if word = 'INTER' then
            READV( line:ip, xfc, yfc, interp, stpin )
          else
          if (word = 'TEMP') or (word = 'TEMPERATURE') then
          begin
            READV( line:ip, t_sample );
            if dat_celsius or (t_sample <= 0.0) then    { Any null or negative temperature value is ... }
              t_sample := t_sample + Ice_Temp;          { ... taken as provided in Celsius degrees }
            t_reg := t_sample; t_set := t_reg
          end
          else
          if SUBSTR( word, 1, 7 )  = 'MONITOR' then
          begin
            bt := false;
            while (ip < line.length) and (line[ip] <> ':') do ip := ip + 1;
            ip := ip + 1;
            word.length := 0;
            if ip < line.length then
              READV( line:ip, monitor, word:0:true );
            if word.length > 0 then READV( word, cnttime )
          end
          else
          if word = 'LAMBDA' then READV( line:ip, lambda1 )
          else
          if word = 'LAMBDA2' then READV( line:ip, lambda2 )
          else
          if word = 'LAMBDAR' then READV( line:ip, rwave2 )
          else
          if word = 'SAMPLE' then READV( line:ip, ch, sample )
          else
          if word = 'DATEH'  then READV( line:ip, ch, dateh )
          else
          if word = 'ORIENT' then READV( line:ip, ch, omega, chi, phi )
        end
      end
      else
      begin
        comment := line;
        for ii := 1 to 5 do READLN( inp )       { Ignore all not header lines }
      end;

      np := 1;
      while np <= nc do
      with dat[np] do
      begin
        READLN( inp, line );
        if line[line.length] < ' ' then line.length := line.length - 1;
        if line.length > 0 then
          if (line[1] <> '#') and (line[1] <> '!') then
          begin
            ip := 1;
            READV( line:ip, xx, yy, word:0:true );
            if word.length > 0 then READV( word, sg )
                               else sg := SQRT( ABS( yy ) );
            xx := xx*xfc; yy := yy*yfc; sg := yfc*sg;
            if np = 1 then { * For the first point }
            begin
              xmi := xx; xma := xx;
              ymi := yy; yma := yy;
              dtheta := xx*xfc
            end
            else
            begin
              if xmi > xx then xmi := xx
                          else if xma < xx then xma := xx;
              if ymi > yy then ymi := yy
                          else if yma < yy then yma := yy;
            end;
            flag := 1; theta := xx; int := yy; sig := sg;
            mflg := [];                         { Set the marker flag to empty }
            np := np + 1
          end;
      end;
      amin  :=      xmi; amax  :=      xma;
      min   :=      ymi; max   :=      yma;
      { Set the global mini maxi for this unique pattern }
      smin  :=      ymi; smax  :=      yma;
      tmin  := t_sample; tmax  := t_sample;
      thmin :=      xmi; thmax :=      xma;
      gmin  :=      ymi; gmax  :=      yma
    end;
    { Set the pattern with this unique pattern }
    npat := npat + 1;
    nda := 1; tbpat[1] := p;
    if pat_first = nil then pat_first := p
                       else pat_last^.next := p;
    pat_sel   := p;
    pat_last  := p
  end GET_XYD_FORMAT;



  procedure GET_INLRX_FORMAT;
  var
    p:                         pat_ptr;
    line:                       string;
    nc, i:                     integer;
    th, mi, ma:                   real;
    bs:                        boolean;
    ch:                           char;

  begin
    rfrm := 'INELRX';
    repeat                                      { Loop on all pattern(s) }
      nc := 0;                                  { Initialize the number of point in one pattern }
      { Look for "RAW" word }
      repeat
        READLN( inp, line );
        if EOF( inp ) then nc := -1;
      until (nc < 0) or ((line[1] = 'R') and (line[2] = 'A') and (line[3] = 'W'));
    exit if nc < 0;
      READLN( inp, nc );                        { Read the size (in point) of the pattern }
      NEW( p, nc );                             { Allocate  a pattern record for nc points }
      INIT_PATTERN( p^ );
      with p^, mai_parm do
      begin                                     { Now we are filling the pattern record }
        READLN( inp, cnttime );                 { Read the counting time }
        for ii := 1 to 7 do READLN( inp );      { Skip the next seven lines }
        READLN( inp, comment );                 { Read the Pattern comment }
        ELLIMINATE_CR( comment );
        READLN( inp, lambda1 );                 { Read the Wave length for alpha1 and alpha 2 }
        READLN( inp, lambda2 );
        { if gw1_r > 0.0 then lambda1 := gw1_r; { Force the Lambda when it is externaly specified }
        { if gw2_r > 0.0 then lambda2 := gw2_r; { Force the Lambda when it is externaly specified }
(*
        READLN( inp, dateh );                   { Get the date and hour of Measurement >>> Actually ignored }
        SKIP_INP_CHAR( 12 );
*)
        READLN( inp, ch, dateh );
        ELLIMINATE_CR( dateh );
        numor := npat + 1;                      { Set the integer identifier }
        SKIP_INP_CHAR( 12 );
        READLN( inp, sample );                  { Get sample field }
        ELLIMINATE_CR( sample );
        SKIP_INP_CHAR( 15 );                    { Skip 15 characters }
        READLN( inp, t_sample );                { Get the temperature }
        t_sample := t_sample + Ice_Temp;        { Convert it in Kelvin (always in Celsius) }
        t_set    := t_sample;                   { Default assignements }
        t_reg    := t_sample;
        READLN( inp );                          { Skip one line }
        READLN( inp );                          { Skip one line }
        repeat
          READLN( inp, line );
        until line[1] = '0';                    { Look for the begin of pattern Counts }
        i := 0;
        amin := dtheta;                         { Set the minimum and maximum of 2*theta }
        amax := dtheta + nc*step;
        mi := mimast; ma := - mi;
        ista := 1; iend := nc;
        while i < nc do
        begin
          with dat[i + 1] do
          begin
            flag := 1;                          { Set as a valid point }
            theta := dtheta + i*step;
            READLN( inp, int );                 { Read each point of the diagram }
            sig := SQRT( ABS( int ) );          { Set the sigma field }
            mflg := [];                         { Set the marker flag to empty }
            { Set the diagram minimum and maximum }
            if mi > int then mi := int;
            if ma < int then ma := int
          end;
          i := i + 1
        end;
        { Set pattern minimaxi }
        min := mi; max := ma;
        { Update the general minimaxi }
        if tmin > t_set then tmin := t_set;
        if tmax < t_set then tmax := t_set;
        if thmin > amin then thmin := amin;
        if thmax < amax then thmax := amax;
        if gmin > mi then gmin := mi;
        if gmax < ma then gmax := ma
      end;
      { Set the pattern in the pattern list }
      if pat_first = nil then pat_first := p
                         else pat_last^.next := p;
      pat_last := p;
      { Set this pattern in selection table if required }
      npat := npat + 1;
    until EOF( inp )
  end GET_INLRX_FORMAT;



  procedure GET_SRAW_FORMAT;
  var
    len, pid:                  integer;
    mi, ma, th0, stp, thnd:     double;
    p:                         pat_ptr;
    bs:                        boolean;

  begin
    pid := 0;                                   { Init the pattern index }
    while not EOF( inp ) do                     { Loop  for each pattern to get }
    begin
      READ( inp, th0 );                         { Get the start, step and end angles (2*theta) }
    exit if EOF( inp );
      READ( inp, stp );
    exit if EOF( inp );
      READLN( inp, thnd );
    exit if EOF( inp );
      pid := pid + 1;
      len := ROUND( (thnd - th0)/stp ) + 1;     { Compute the size of pattern }
      thnd := (len + 0.5)*stp + th0;            { Adjust the end angle }
      NEW( p, len );                            { Allocate a new pattern }
      INIT_PATTERN( p^ );                       { Initialize all pattern fields }
      with p^, mai_parm do
      begin
        idcnt  :=  pid;                         { Set the local identifier }
        if ABS( th0 ) < 180.0 then              { Set specific 2*theta origine and step ...when legal value }
        begin  dtheta :=  th0; step   :=  stp  end;     { ... when the th0 value is legal }
        amin := dtheta;                         { Set the 2*theta mini-maxi }
        amax := dtheta + (len - 1)*step;
        mi := min; ma := max;                   { Get the initial minmaxi (from INIT_PATTERN) }
        for ii := 1 to len do
        with dat[ii] do
        begin
          theta := (ii - 1)*stp + th0;          { Set the 2*theta value }
          int   := 0.0;                         { Init and read the intensity }
          if not EOF( inp ) then READ( inp, int );
          sig := SQRT( ABS( int ) );            { Set a default sigma }
          mflg := [];                           { Set the marker flag to empty }
          if mi > int then mi := int;           { set the intensity Mini-Maxi }
          if ma < int then ma := int
        end;
        { Set pattern minimaxi }
        min := mi; max := ma;
        { Update the general minimaxi }
        if tmin > t_set then tmin := t_set;
        if tmax < t_set then tmax := t_set;
        if thmin > amin then thmin := amin;
        if thmax < amax then thmax := amax;
        if gmin > mi then gmin := mi;
        if gmax < ma then gmax := ma
      end;
      { Set the pattern in the pattern list }
      if pat_first = nil then pat_first := p
                         else pat_last^.next := p;
      pat_last := p;
      { Set this pattern in selection table if required }
      npat := npat + 1;
    end
  end GET_SRAW_FORMAT;



begin { GET_PATTERNS }
  nda  := 0;                                    { No selection }
  OPEN( inp, inp_name, [read_file,error_file] );{ Open the data file }
  if iostatus <> 0 then
  begin
    WRITEV( str_msg, ' *** VIEW-3D-DATA Error: Cannot open the file "', inp_name, '". ***' );
    return
  end;

  case inp_frm of

    pf_standard:   GET_RPWSTD_FORMAT;

    pf_d1b:        GET_D1B_FORMAT;

    pf_xydata:     GET_XYD_FORMAT;

    pf_macinelrx:  GET_INLRX_FORMAT;

    pf_singleraw:  GET_SRAW_FORMAT;

  otherwise
    str_msg := ' *** VIEW-3D-DATA Error: Unsupported or unimplemented input format. ***';
    return
  end;

  SET_PATTERN_SELECTION;

ERR_RECOVER:

  CLOSE( inp );                                 { Close the data file }

  WRITELN( ' VIEW-3D-DATA Found ', npat:0, ' patterns in the file "', inp_name, '".' );
  if npat = 0 then
  begin
    str_msg := ' *** VIEW-3D-DATA Error: Cannot work without Pattern and exit.';
    return
  end
  else
  begin
    WRITELN( ' ':24, 'Global Statistics' );
    WRITELN( ' ':6, 'Minimum and maximum of 2*Theta Angle: ', thmin:8:3, ', ', thmax:8:3, ',' );
    WRITELN( ' ':6, 'Minimum and maximum of Temperature:   ', tmin:8:2,  ', ', tmax:8:2,  ',' );
    WRITELN( ' ':6, 'Minimum and maximum of intensities:   ', gmin:8:3,  ', ', gmax:8:3,  '.' );
  end
end GET_PATTERNS;



end VIEW3D_DATA_INP.
