{
*************************************************************************
*                                                                       *
*                                                                       *
*        V I E W - 3 D  (View Powder Diffraction Data in 3D)            *
*                                                                       *
*               ( ILL Data Base Manager Source File )                   *
*                                                                       *
*                  Version  1.0-A  - - 30-Nov-2009                      *
*                                                                       *
*                                by                                     *
*                                                                       *
*                   Pierre Wolfers, Institut Neel                       *
*                                                                       *
*            CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166            *
*                                                                       *
*                       F 38042 GRENOBLE CEDEX 9                        *
*                                                                       *
*                             F R A N C E                               *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}
module DRAW_3D;

%include 'RPWSRC:view3d_env' {, list_on};


const
  { 3D Keys Menu definitions }
  Clic_Omega_Rotate            =     1;         { Omega Rotate action }
  Clic_Chi_Rotate              =     2;         { Chi   Rotate action }
  Clic_Phi_Rotate              =     3;         { Phi   Rotate action }
  Clic_Dist_Rotate             =     4;         { Distance Change action }
  Clic_Light0_On_Off           =     5;         { On/Off Light0 switch }
  Clic_Light1_On_Off           =     6;         { On/Off Light1 switch }

  Clic_Mai_Setup               =     7;         { Display of thje Main setup }
  Clic_Drw_setup               =     8;         { Display the Graphic setup }

  Light_Seg                    =     2;         { Light Segment # }
  Angle_Seg                    =     3;         { String Angle Segment # }
  Box_Seg                      =     4;         { Axis Box Segment # }


var
  unt: unit_type;                               { Unit type of pixel }
  xs, ys: Dfloat;                               { Paper size in unt units }

  dstatus,                                      { Return status for DRAW Graphic Library }
  irep:                            Dint;        { Return value for DRAW$DIALOG Procedure }




{**********************************************************************************
*                                                                                 *
*                               Drawing Procedures                                *
*                                                                                 *
**********************************************************************************}



procedure INIT_PLOT( blight: Dbool );
{****************************************************}
{          INITIALYSE THE DRAWING SYSTEM             }
{****************************************************}
{ The external procedure

    DRAW$DEF_MENU( code: Dint; in_var name: [optional] string );

                is a procedure to modify by insertion the DRAW$DIALOG
  standard menu setting.

  The code are :
    Menu_Close             Close the currently opened menu.
    Menu_Open              Open the next Standard Menu.
    Menu_Skip              Skip_the next standard (Menu or Entry).
    Menu_New    <Name>     Create and Open a new user menu of specified name.
    Menu_Entry  <Name>     Create a menu entry of specified name.
    Menu_Separ             Create a Separator menu Entry.

  The directives Menu_Entry and Menu_Separ allocate an integer identifier
  to the new menu elements beginning from 1. This identifier is returned by
  the procedure DRAW$DIALOG when the corresponding entry is selected by
  the mouse.
  The Menu_skip can be used with the form n*16+Menu_Skip where n is a skip number
  with the following specificities:
    If n < 0 the skip go to the end of current standard menu else,
    if n > 0 the result is to skip n entry in the current standard menu.
 
}
begin
  Draw_Server_Mode := DrwSrv_Disable_IniErr;
  if blight then
    Draw_Server_Mode := Draw_Server_Mode +
                        DrwSrv_Do_Lighting +                    { Require the Lighting mode }
                        DrwSrv_Enable_Light0 +                  { with two lamps }
                        DrwSrv_Enable_Light1;

  DRAW$SET_MENU_SPC( Menu_Skip );                               { Skip the standard menu "File" }
  DRAW$SET_MENU_SPC( Menu_New, '&Plot Edit' );                  { Create a New menu "Plot edit" }
  DRAW$SET_MENU_SPC( Menu_Entry, '&Main Setup', Clic_Mai_Setup, ORD( 'M' ) );       { Display Main Setup }
  DRAW$SET_MENU_SPC( Menu_Entry, '&Graphic Setup', Clic_Drw_Setup, ORD( 'G' ) );    { Display Graphic Setup }
  if blight then
  begin
    DRAW$SET_MENU_SPC( Menu_Entry, 'L0 On/off', Clic_Light0_On_Off );   { Create a New menu entry "Light 0 On/off" }
    DRAW$SET_MENU_SPC( Menu_Entry, 'L1 On/off', Clic_Light1_On_Off );   { Create a New menu entry "Light 1 On/Off" }
  end;
  DRAW$SET_MENU_SPC( Menu_Entry, '&Omega', Clic_Omega_Rotate ); { Create a New menu entry "Omega" }
  DRAW$SET_MENU_SPC( Menu_Entry, '&Chi',   Clic_Chi_Rotate );   { Create a New menu entry "Chi" }
  DRAW$SET_MENU_SPC( Menu_Entry, '&Phi',   Clic_Phi_Rotate );   { Create a New menu entry "Phi" }
  DRAW$SET_MENU_SPC( Menu_Entry, '&Dist',  Clic_Dist_Rotate );  { Create a New menu entry "Dist" }
  DRAW$SET_MENU_SPC( Menu_Close );                              { Close the user menu "Rotate" }
  DRAW$SET_MENU_SPC( Menu_Close );                              { Close the main standard menu (barre menu) }

  DRAW$INIT( xs, ys, unt, 'RPWData 3D' );
  if Draw_Server_Flags <> 0 then                                { When the DRAW$INIT is a success ... }
  begin
    DRAW$PIC_VIEW( 1 );                                         { ... set to graphic update as soon as mode and ...  }
    drw_ok := true                                              { ... set on the DRAW Init flag }
  end
end INIT_PLOT;



[global]
procedure MAKE_3D;
const
  light0_amb_color =  Dcolor[ 0.6, 0.6, 0.6, 1.0];      { Color for Ambiant light }
  light0_dif_color =  Dcolor[ 1.0, 1.0, 1.0, 1.0];      { Color for Diffuse light }
  light0_spe_color =  Dcolor[ 1.0, 1.0, 1.0, 1.0];      { Color for Specular light }
(*
  grid_color       =  Dcolor[ 0.9, 0.8, 0.6, 1.0];      { Color of lines use when lighting is disable }
  fill_color       =  Dcolor[ 0.3, 0.3, 0.7, 1.0];      { Filling color }
*)
type
  tbraw( sz: integer ) = record
                           tb: array[1..sz] of Dfloat
                         end;

var
  dview,                                                { View distance (cm) }
  ome, chi, phi,                                        { Eulerian Angles }
  xp, yp, zp:                   Dfloat;                 { 3D Picture size }

  bstatus:                       Dbool;                 { Status return word }

  bstlght0,
  bstlght1:              Dbool := true;

  light0_position,                                      { Position of Lights }
  light1_position,
  box_position:                Dpoint3;                 { Plot Origine }

  lseg, iseg, aseg,                                     { Segments # for Boxe and Angles }
  gdim:                 Dint    :=   0;

  untx, unty, untz,                                     { Axis Unit strings }
  name,
  angmsg:          string( 255 ) := '';                 { Angle Message }

  raw:                          ^tbraw;                 { Raw array to send points }

  picture_size:                 Dfloat;                 { Size of picture to use }




  procedure DISPLAY_SIZE;
  const
    eps = 0.5E-5;

  var
    p:                              pat_ptr;
    minth0, maxth0, minth1, maxth1:  Dfloat;
    np0, np1, npa, ndg:             integer;

  begin
    with mai_parm do
    begin
      if tmi_r <> 1100.0 then e_thmin := tmi_r*(1.0 - eps)
                         else e_thmin := thmin;
      if tma_r <> 1100.0 then e_thmax := tma_r*(1.0 + eps)
                         else e_thmax :=   thmax;
    end;
    minth0 := thmax;
    maxth0 := thmin;
    npa := 0;
    ndg := 0;
    p := pat_first;
    while p <> nil do
    with p^ do
    begin
      np1 := 0;
      ndg := ndg + 1;
      for ij := 1 to dim do
        with dat[ij] do
          if (theta >= e_thmin) and (theta <= e_thmax) then
            if np1 = 0 then
            begin
              np1  :=  1;
              ista := ij;
              iend := ij;
              minth1 := dat[1].theta;
              maxth1 := minth1
            end
            else
            begin
               np1 := np1 + 1;
                iend := ij;
                if theta > maxth1 then maxth1 := theta
                                  else if theta < minth1 then minth1 := theta
            end;

        if npa = 0 then
        begin
          npa := 1; np0 := np1; minth0 := minth1; maxth0 := maxth1
        end
        else
          if (ABS( minth1 - minth0 ) > ABS( step )/4.0) or
             (ABS( maxth1 - maxth0 ) > ABS( step )/4.0) or
             (np1 <> np1) then
          begin
            WRITELN( ' *** PLOT_PATTERN error : Not consistent pattern list for a 3D plot. ***' );
            WRITELN( ' ':26, 'The 2theta limits of each pattern must be close to the pattern step/4 and' );
            WRITELN( ' ':26, 'the number of point must be equals.' );
            DRAW$END;
            PASCAL_EXIT( 2 )
          end;
        p := next
    end;
    if np0 < 10 then
    begin
      WRITELN( ' *** PLOT_PATTERN error : Too small Pattern for a 3D plot (', np0:0, ' dots: less long than 10 dots) . ***' );
      if np0 < 2 then
      begin
        WRITELN( ' ':26, 'Stop because can plot it with less than 2point/pattern.' );
        DRAW$END;
        PASCAL_EXIT( 2 )
      end
    end;
    NEW( raw, np0*3 );
    gdim := np0
  end DISPLAY_SIZE;



  procedure DISPLAY_ANGLES;
  begin
    with drw_parm do
    begin
      DRAW$UPDATE_SEG( aseg {, 0} );
      DRAW$COLOR( 1 );
      WRITEV( angmsg, ' Ome = ', ome_r:7:2, ', Chi = ', chi_r:7:2, ', Phi = ', phi:7:2, ', Dist = ', dis_r:6:1  );
      DRAW$COLOR( 1 );
      DRAW$TEXT_ATTR( 3, 4, 1, 1.0, 0.0 );        { H:Centered, V:Middle, Left_to_Right, Exp_Factor, Space_Factor }
      DRAW$STRING( 0.0, 2.0 - ysz_r, 0.0, 1.0, angmsg );
      DRAW$SEG_END
    end
  end DISPLAY_ANGLES;



  procedure DISPLAY_SURFACE;
  { Procedure to display the 3D image }
  const
    mi2th =    1.0;                             { Minimum 2*Theta Spacing }
    shz   =  0.001;                             { Shift for Z lines }

  var
    x_axe, y_axe, z_axe, box, patsize: [static] Dint;
    bx_dir:             [static] array[1..5] of Dint;
    zsh, cmin, cmax, xx, yy, zz:              Dfloat;
    i, j, k, l, m:                           integer;
    p1, p2:                                  pat_ptr;
    ste:                                string( 16 );


    procedure DISPLAY_3D_0;
    { Display the surface as successive block in one surface }
    var
      p:       pat_ptr;
      sz:      integer;
      ver:     Dpoint3;
      i, nr:   integer;

    begin
      DRAW$FILL_ATTR( 2, 1 );                   { Set the Filling attribute mode }
      DRAW$OUT_MODE( 3 );                       { Set the Filling mode }
      p := pat_first;
      if p <> nil then DRAW$SURFACE( npat, gdim, 16 );
      nr := 0;
      while p <> nil do
      with p^ do
      begin
        i := 0;
        with raw^ do
        for ij := ista to iend do
        with dat[ij] do
        begin
          i := i + 1; tb[i] := theta;
          i := i + 1; if fc_celsius in funcs then tb[i] := t_sample - Ice_Temp
                                             else tb[i] := t_sample;
          i := i + 1; tb[i] :=   int;
        end;
        nr := nr + 1;
        DRAW$SEND_BLOCK( raw^.tb );
        p := next
      end;
      DRAW$COLOR( 1 )
    end  DISPLAY_3D_0;


    procedure DISPLAY_3D_1;
    { Display the surface as successive triangle_strips }
    var
      p1, p2:  pat_ptr;
      sz:      integer;
      ta, tb:   Dfloat;

    begin
      DRAW$FILL_ATTR( 2, 1 );                   { Set the Filling mode }
      DRAW$OUT_MODE( -5 {GL_TRIANGLE_STRIP} );  { Set the Triangle Strip mode }
      p1 := pat_first;                          { Start from first pattern }
      p2 := p1^.next;                           { Get the second pattern address }
      while p2 <> nil do
      begin { Loop on the patterns }
        if fc_celsius in funcs then
        begin
          ta := p1^.t_sample - Ice_Temp;
          tb := p2^.t_sample - Ice_Temp
        end
        else
        begin
          ta := p1^.t_sample;
          tb := p2^.t_sample
        end;

        { Output the first point }
        with p1^, dat[ista] do DRAW$PLOT3( theta, ta, int, false );
        for i := p1^.ista to p1^.iend - 1 do    { Loop on the pattern points }
        begin
        with p2^, dat[i] do DRAW$PLOT3( theta, tb, int, true );
        with p1^, dat[i + 1] do DRAW$PLOT3( theta, ta, int, true )
      end;
      with p2^, dat[iend] do DRAW$PLOT3( theta, tb, int, true );
      DRAW$PLOT3( 0.0, 0.0, 0.0, false );       { Finish the Triangle Strip block }
      p1 := p2;                                 { Skip to the next patterns }
      p2 := p1^.next
    end
  end DISPLAY_3D_1;


  procedure DISPLAY_3D_2;
  { Display the surface as network of line_strips }
  var
    p:          pat_ptr;
    ts, te:     real;
    bs:         boolean;
    sz, ii, st: integer;

  begin
    DRAW$COLOR( 0.9, 0.8, 0.6 );
(*  DRAW$COLOR( grid_color ); *)
    DRAW$FILL_ATTR( 1, 1 );                     { Set the Hollow mode }
    { Plot each Pattern }
    p := pat_first;                             { Start from first pattern }
    with p^ do
      ts := (dat[dim].theta - dat[1].theta)/(dim - 1);  { Compute the 2*theta step }
    st := TRUNC( mi2th/ts );
    if st*ts < mi2th then st := st + 1;         { Get the 2*theta spacing }
    while p <> nil do
    with p^ do
    begin { Loop on the patterns }
      if fc_celsius in funcs then te := t_sample - Ice_Temp
                             else te := t_sample;
      with dat[ista] do
        DRAW$PLOT3( theta, te, int + zsh, false );
      for ij := ista + 1 to iend do
        with dat[ij] do
          DRAW$PLOT3( theta, te, int + zsh, true );
      p := next
    end;
    { Plot for each theta the variation with temperature }
    ii := 0;
    while ii < gdim do
    begin
      bs :=    false;
      p := pat_first;
      while p <> nil do
      with p^, dat[ista + ii] do
      begin
        if fc_celsius in funcs then te := t_sample - Ice_Temp
                               else te := t_sample;
        DRAW$PLOT3( theta, te, int + zsh, bs );
        bs := true;
        p := next
      end;
      ii := ii + st
    end;
    { Finish the Last curve }
    DRAW$PLOT3( 0.0, 0.0, 0.0, false )
  end DISPLAY_3D_2;


  begin { DISPLAY_SURFACE }
    with drw_parm do
    begin
      if fc_light in funcs then                         { When the lighting mode is used }
      begin                                             { Define the Lights }
        if lseg = 0 then lseg := DRAW$NEW_SEG( Light_Seg )
                    else DRAW$UPDATE_SEG( lseg );       { Open/Create the light segement }
(*
        DRAW$LIGHT_ON_OFF( 1, 0 );                      { Set Off the Light # 1 }
*)
        bstlght0 := true; bstlght1 := true;

        DRAW$LIGHT_ON_OFF( 0, 1 );                      { Set On the Light # 0 }

        DRAW$LIGHT_DEFINE( 0, LIGHT_POSITION,       light0_position  );
        DRAW$LIGHT_DEFINE( 0, LIGHT_AMBIENT_COLOR,  light0_amb_color );
        DRAW$LIGHT_DEFINE( 0, LIGHT_DIFFUSE_COLOR,  light0_dif_color );
        DRAW$LIGHT_DEFINE( 0, LIGHT_SPECULAR_COLOR, light0_spe_color );
        DRAW$LIGHT_ON_OFF( 1, 0 );                      { Set off the # 1 light }

        DRAW$LIGHT_DEFINE( 1, LIGHT_POSITION,       light1_position );
        DRAW$LIGHT_DEFINE( 1, LIGHT_AMBIENT_COLOR,  Dcolor[ 0.1, 0.1, 0.1, 1.0] );
        DRAW$LIGHT_DEFINE( 1, LIGHT_DIFFUSE_COLOR,  Dcolor[ 0.7, 0.7, 0.4, 1.0] );
        DRAW$LIGHT_DEFINE( 1, LIGHT_SPECULAR_COLOR, Dcolor[ 0.8, 0.8, 0.5, 1.0] );


        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_SPECULAR_COLOR, Dcolor[1.0, 1.0, 1.0, 0.55] );
        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_SHININESS,      Dvalue[100.0] );
        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_EMISSION_COLOR, Dcolor[0.0, 0.0, 0.0, 1.0] );
        DRAW$MAT_LIGHT_PROP( MAT_FRONT_FACE, MAT_DIFFUSE_COLOR,  Dcolor[1.0, 0.9, 0.2, 1.0] );

        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE,  MAT_SPECULAR_COLOR, Dcolor[1.0, 1.0, 1.0, 0.55] );
        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE,  MAT_SHININESS,      Dvalue[100.0] );
        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE,  MAT_EMISSION_COLOR, Dcolor[0.0, 0.0, 0.0, 1.0] );
        DRAW$MAT_LIGHT_PROP( MAT_BACK_FACE,  MAT_DIFFUSE_COLOR,  Dcolor[1.0, 0.9, 0.2, 1.0] );

        DRAW$COLOR( 1.0, 1.0, 0.5, 1.0 );
        DRAW$SPHERE( light0_position[1], light0_position[2], light0_position[3], 2.0, 32, 32, 4 );

        DRAW$SEG_END
      end;

      if aseg = 0 then aseg := DRAW$NEW_SEG( Angle_seg) { Open/Create the Angle segment }
                  else DRAW$UPDATE_SEG( aseg );

      DISPLAY_ANGLES;

      if iseg = 0 then iseg := DRAW$NEW_SEG( Box_Seg )  { Open/Create the Box/Plot Segment }
                  else DRAW$UPDATE_SEG( iseg );


      { Create the Orientation Matrix of the Box/Plot }

      DRAW$ROTATE_SEG3( iseg, 0.0, 0.0, 0.0, ome_r, chi_r, phi_r );
      DRAW$SEG_ORIENTATION( iseg, 0.0, 0.0, 0.0 );

      { The flag parameter (last one) must be flag the axis characteristics as this :
        the value must be a set of this following constantes :

        *****   Axis Flags   *****

        Axis_Log       =       1; (* Axis is in Log10 Coordinate *)
        Axis_Left      =       2; (* Ticks and displayed value at left position ( else at right ) *)
        Axis_VOther    =       4; (* Value are in other side (than ticks) *)
        Axis_NFrs      =       8; (* Does not display the first Ticks *)
        Axis_NFVl      =      16; (* Does not display the first Value *)
        Axis_NLsT      =      32; (* Does not display the last Ticks *)
        Axis_NLVl      =      64; (* Does not display the last Value *)
        Axis_NZer      =     128; (* Does not display the Zero Ticks *)
        Axis_NZVl      =     256; (* Does not display the Zero Value *)
        Axis_Arrow     =     512; (* Put an arrow at the end of axis *)
        Axis_ArroF     =    1024; (* Fill mode for Axis_Arrow *)
        Axis_Tilte     =    2048; (* Axis character are tilted with ticks *)
        Axis_VPath     =    4096; (* String Path (2 bits) for Ticks Value *)
        Axis_Upath     =   16384; (* String Path (2 bits) for Axis Unit *)
        Axis_SpLim     =   65536; (* Axis Limit auto-adaptation when set *)
        Axis_VLeft     =  131072; (* Left Alignement for Unit String *)
        Axis_VRight    =  262144; (* Right Alignement for Unit String *)
        Axis_ULeft     =  524288; (* Left Alignement for Unit String *)
        Axis_URight    = 1048576; (* Right Alignement for Unit String *)
        Axis_UMiddle   = 2097152; (* Set Unit String at the Middle of Axis *)


        *****   Axis Plot Directives and Flags for Draw_Plot   *****

        Axis_Plot_X    =     1; (* Plot an X axis *)
        Axis_Plot_Y    =     2; (* Plot an Y axis *)
        Axis_Plot_Z    =     3; (* Plot a  Z axis *)
        Axis_PFlg_TVl  =     4; (* Ticks values and Unit String will be Displayed *)
        Axis_PFlg_TPrm =     8; (* The ticks left and right sides are permutted *)
        Axis_PFlg_TSym =    16; (* The Ticks must be symmetrized *)
        Axis_PFlg_XEnd =    32; (* Plot Axis with a shift of x axis size *)
        Axis_PFlg_YEnd =    64; (* Plot Axis with a shift of Y axis size *)
        Axis_PFlg_ZEnd =   128; (* Plot Axis with a shift of Z axis size *)
        Axis_PFlg_XPos =   256; (* Plot Axis at specified X shift *)
        Axis_PFlg_YPos =   512; (* Plot Axis at specified Y shift *)bs := (
        Axis_PFlg_ZPos =  1024; (* Plot Axis at specified Z shift *)
        Axis_Pflg_SymU =  2048; (* Apply a mU (mirror u) to tick direction *)
        Axis_Pflg_SymV =  4096; (* Apply a mV (mirror v) to tick direction *)
        Axis_Pflg_SymW =  8192; (* Apply a mW (mirror w) to tick direction *)
        Axis_Pflg_Rotv = 16384; (* Apply an additional rotation of 180 to value and unit strings *)
      }

      cmin := tmin; cmax := tmax;
      if fc_celsius in funcs then
      begin
        cmin := cmin - Ice_Temp;
        cmax := cmax - Ice_Temp;
        ste:= 'C'
      end
      else ste := 'K';
      ste := ste||' (Temperature)';

      zsh := (gmax - gmin)*shz;

      x_axe := DRAW$NEW_AXIS( Dpoint3[ 1.0, 0.0, 0.0],
                              Dpoint3[ 0.0, 1.0, 1.0],
                              xsz_r, e_thmin,e_thmax, -3,                                Axis_UMiddle );
      y_axe := DRAW$NEW_AXIS( Dpoint3[ 0.0, 1.0, 0.0],
                              Dpoint3[-1.0, 0.0, 1.0],
                              ysz_r,    cmin,   cmax, -3,  Axis_UMiddle + Axis_VPath*0 + Axis_UPath*0 );
      z_axe := DRAW$NEW_AXIS( Dpoint3[ 0.0, 0.0, 1.0],
                              Dpoint3[ 1.0, 1.0, 0.0],
                              zsz_r,    gmin,   gmax, -3,  Axis_Arrow   + Axis_ArroF  +  Axis_UMiddle );


      Draw$Set_Axis_Unit( x_axe, ' (2*Theta)' );
      Draw$Set_Axis_Unit( y_axe, ste );
      Draw$Set_Axis_Unit( z_axe, 'Cps (Intensity)' );

      Draw$Set_Axis_Value( x_axe, 2, 112,,,,, 4,  5, 2 );       { Set Value Font, Field and decimal }
      Draw$Set_Axis_Value( y_axe, 2, 112,,,,, 4,  5, 2 );       { Set Angle, Value Font, Field and decimal }
      Draw$Set_Axis_Value( y_axe, 4,   2,, 0.0 );               { Set Angle, fbs := (or Unit String }
      Draw$Set_Axis_Value( z_axe, 2, 112,,,,, 4,  5, 2 );       { Set Value Font, Field and decimal }

      bx_dir[1] := Axis_PFlg_TVl  +                     { Tick Values, }
                   Axis_Plot_X;                         { Plot X axis }
      bx_dir[2] := Axis_PFlg_TVl  +                     { Tick Values, }
                   Axis_PFlg_XEnd + Axis_Plot_Y;        { at end'X axis, Plot Y axis }
      bx_dir[3] := Axis_PFlg_TVl  + Axis_Pflg_Rotv +    { Tick Values, 180 String rotate, }
                   Axis_PFlg_SymW + Axis_PFlg_TPrm +    { mw symetry Ope., Left/right Permut, }
                   Axis_Plot_Y;                         { Plot Y axis }
      bx_dir[4] := Axis_PFlg_TVl  + Axis_Pflg_Rotv +    { Tick Values, 180 String rotate, }
                   Axis_PFlg_SymW + Axis_PFlg_TPrm +    { mw symetry Ope., Left/right Permut, }
                   Axis_PFlg_YEnd + Axis_Plot_X;        { at end'Y axis, Plot X axis }

      bx_dir[5] := Axis_PFlg_TVl  + Axis_Plot_Z;        { Tick Values, Plot Y axis }

      DRAW$COLOR( 1 );

      { Create the Axis Box }
      box := DRAW$NEW_BOX( box_position, x_axe, y_axe, z_axe, bx_dir );

      DRAW$LINE_ATTR( 1, 2.0 );                 { Set continue line type }
      DRAW$PLOT_BOX( box );                     { Draw the Axis box }

      DRAW$COLOR( 1 );
      DRAW$OPEN_BOX( box );                     { Open the box to plot the curve }

      if fc_light in funcs then
      begin
        DRAW$COLOR( 0.55, 0.55, 0.70, 1.0 );
        DISPLAY_3D_0
      end
      else
      begin
        DRAW$COLOR( 0.3, 0.3, 0.7 );
(*      DRAW$COLOR( fill_color );    *)         { Set the surface color }
        DISPLAY_3D_1                            { Form the Triangle fillings }
      end;

      if not (fc_light in funcs) then DISPLAY_3D_2;     { Form a grid of line }

      DRAW$CLOSE_BOX;                           { Close the Axis box }
      DRAW$SEG_END;                             { Close the boxe/plot segment }
      DRAW$VIEW( dview )
    end
  end DISPLAY_SURFACE;


  procedure ROTATE_PLOT;
  var
    pmat, mat: array[1..3,1..3] of Dfloat;

  begin
    with drw_parm do
    begin
(*
      DRAW$UPDATE_SEG( aseg {, 0} );
      DRAW$COLOR( 1 );
      WRITEV( angmsg, ' Ome = ', ome_r:8:2, ', Chi = ', chi_r:8:3, ', Phi = ', phi_r:8:3, ', Dist = ', dis_r:7:1 );
      DRAW$COLOR( 1 );
      DRAW$TEXT_ATTR( 3, 4, 1, 1.0, 0.0 );     { H:Centered, V:Middle, Left_to_Right, Exp_Factor, Space_Factor }
      DRAW$STRING( 0.0, 2.0 - ysz_r, 0.0, 1.0, angmsg );

      DRAW$SEG_END;
*)
      DISPLAY_ANGLES;

      DRAW$VIEW( dis_r );
      DRAW$SEG_UNIT_TRANSF( iseg );
      DRAW$ROTATE_SEG3( iseg,  0.0,  0.0,  0.0, ome_r, chi_r, phi_r )
    end
  end ROTATE_PLOT;


procedure UPDATE_ANGLES;
var
  ir:                          integer;
  mat:                         Dmatrix;
  cc, sc, co, so, cp, sp:       Dfloat;

(*
  ome = inrd*ome; chi = inrd*chi; psi = inrd*psi;
  co = cos( ome ); cc = cos( chi ); cp = cos( psi );
  so = sin( ome ); sc = sin( chi ); sp = sin( psi );
  m[1,1] =  co*cc*cp - so*sp;                     /* m11 */
  m[1,2] = -co*cc*sp - so*cp;                     /* m12 */
  m[1,3] =  co*sc;                                /* m13 */
  m[1,4] =  px - m[ 0]*px - m[ 4]*py - m[ 8]*pz;  /* m14 */
  m[2,1] =  so*cc*cp + co*sp;                     /* m21 */
  m[2,2] = -so*cc*sp + co*cp;                     /* m22 */
  m[2,3] =  so*sc;                                /* m23 */
  m[2,4] =  py - m[ 1]*px - m[ 5]*py - m[ 9]*pz;  /* m24 */
  m[3,1] = -sc*cp;                                /* m31 */
  m[3,2] =  sc*sp;                                /* m32 */
  m[3,3] =  cc;                                   /* m33 */
  m[3,4] =  pz - m[ 2]*px - m[ 6]*py - m[10]*pz;  /* m34 */
*)

begin
  ir := DRAW$GET_SEG_MAT( iseg, mat );
  if ir > 0 then
  with drw_parm do
  begin { we have the segment matrix }
(*  WRITELN( ' Matrix is ', mat[1,1]:8:4, ', ', mat[1,2]:8:4, ', ', mat[1,3]:8:4, ', ', mat[1,4]:8:4 );
    WRITELN( '           ', mat[2,1]:8:4, ', ', mat[2,2]:8:4, ', ', mat[2,3]:8:4, ', ', mat[2,4]:8:4 );
    WRITELN( '           ', mat[3,1]:8:4, ', ', mat[3,2]:8:4, ', ', mat[3,3]:8:4, ', ', mat[3,4]:8:4 );
*)
    cc := mat[3,3];
    sc := SQRT( 1.0 - SQR( cc ) );
    if ABS( sc ) < 1.0e-5 then
    begin
      ome_r := 0.0;
      if cc > 0 then chi_r :=   0.0
                else chi_r := 180.0;
      phi_r := ARCTAN( mat[1,2]/cc, -mat[1,1]/cc )/inrd;
    end
    else
    begin
      cp := - mat[3,1]/sc;
      sp :=   mat[3,2]/sc;
      co :=   mat[1,3]/sc;
      so :=   mat[2,3]/sc;
      ome_r := ARCTAN( so, co )/inrd;
      phi_r := ARCTAN( sp, cp )/inrd;
      chi_r := ARCCOS( cc )/inrd
    end;
    DISPLAY_ANGLES
  end
end UPDATE_ANGLES;



begin { MAKE_3D }
  with drw_parm do
  begin
    INIT_PLOT( fc_light in funcs );             { Initialize the DRAW System }
    if not drw_ok then return;

    if setup_tab[MAI_IDX] <> nil then BUILD_PANNEL( setup_tab[MAI_IDX]^ );
    if setup_tab[DRW_IDX] <> nil then BUILD_PANNEL( setup_tab[DRW_IDX]^ );


    DISPLAY_SIZE;                               { Sizes the surface to plot }

    { DRAW$LIGHT_ON_OFF( 0, 1 );                { Set On the Light # 0 }
    DRAW$LIGHT_ON_OFF( 1, 1 );                  { Set On the Light # 1 }

ET_3D_RESET:

    { Set the axis sizes when not user specified }
    if (xsz_r < 5.0) or (xsz_r > 300.0) then xsz_r := 40.0;
    if (ysz_r < 5.0) or (ysz_r > 300.0) then ysz_r := 40.0;
    if (zsz_r < 5.0) or (zsz_r > 300.0) then zsz_r := 28.0;

    box_position[1]    :=  -0.5*xsz_r;
    box_position[2]    :=  -0.5*ysz_r;
    box_position[3]    :=  -0.5*zsz_r;

    light0_position[1] :=  0.35*xsz_r;            { Position of Lights }
    light0_position[2] :=  0.55*ysz_r;
    light0_position[3] :=  1.40*zsz_r;

    light1_position[1] := -1.10*xsz_r;
    light1_position[2] :=  0.10*ysz_r;
    light1_position[3] :=  1.00*zsz_r;

    picture_size := xsz_r;
    if ysz_r > picture_size then picture_size := ysz_r;
    if zsz_r > picture_size then picture_size := zsz_r;
    picture_size := 2.0*picture_size;
    bstatus := DRAW$PICTURE3( inp_name, picture_size, picture_size, picture_size, false, true );

    DISPLAY_SURFACE;

    repeat
      irep := DRAW$DIALOG( 1 );

      if Draw_Pannel_Ident > 0 then                     { When the Graphic pannel is modified ... }
      begin
        DRAW$PANNEL_GET( Draw_Pannel_Ident );           { ... take it the modifications }
        DRAW$CLOSE_BOX;
        DRAW$FREE_BOX_LIST;                             { Delete the pattern box and the markers box }
        goto ET_3D_RESET
      end;

      if irep = 100000 then UPDATE_ANGLES
      else
      case irep of
        Clic_Omega_Rotate: { Euler Angle Omega Function }
          begin
            dstatus := DRAW$GET_VALUE( 'Angle Omega', ome_r, -180.0, 180.0 );
            if dstatus <> 1 then ome :=  90.0;
            ROTATE_PLOT
          end;

        Clic_Chi_Rotate: { Euler Angle chi Function }
          begin
            dstatus := DRAW$GET_VALUE( 'Angle Chi', chi_r, -180.0, 180.0 );
            if dstatus <> 1 then chi :=  80.0;
            ROTATE_PLOT
          end;

        Clic_Phi_Rotate: { Euler Angle Phi Function }
          begin
            dstatus := DRAW$GET_VALUE( 'Angle Phi', phi_r, -180.0, 180.0 );
            if dstatus <> 1 then phi := -40.0;
            ROTATE_PLOT
          end;

        Clic_Dist_Rotate: { View Distance }
          begin
            dstatus := DRAW$GET_VALUE( 'Distance', dis_r, 0.0, 200.0 );
            if dstatus <> 1 then dview := 0.0;
            ROTATE_PLOT
          end;

        Clic_Light0_On_Off: { Light 0 On/off }
          begin  bstlght0 := not  bstlght0; DRAW$LIGHT_ON_OFF( 0, ORD( bstlght0 ) )  end;

        Clic_Light1_On_Off: { Light 1 On/off }
          begin  bstlght1 := not  bstlght1; DRAW$LIGHT_ON_OFF( 1, ORD( bstlght1 ) )  end;

        Clic_Mai_setup: { Display the Main setup Pannel }
          DRAW$PANNEL_OPEN( setup_tab[MAI_IDX]^.id );

        Clic_Drw_setup: { Display the Graphic setup Pannel }
          DRAW$PANNEL_OPEN( setup_tab[DRW_IDX]^.id );

      otherwise
      end { case }
    until irep <= 0;

ET_Stop:
    { End of Graphic Session }
    DRAW$END;
    DISPOSE( raw )
  end
end MAKE_3D;



end DRAW_3D.
